/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
*/

/*
 *  This file contains all of the code that deals with drawing
 *  arcs on the graph.  This was seperated from xdraw.c on 2/3/92
 *  in an attempt to isolate related code from unrelated code.
*/
#include "config.h"
#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "graph.h"
#include "extern.h"
#include "edge.h"

/*
 *  THRESHOLD is defined to be maximum distance that a mouse click can be
 *  from a given node and still work on that node
*/
#define THRESHOLD 15

/*
 * ARROW_HEAD_LEN is the length (from 0 to 1) of the lines making up the
 * arrow head.
*/
#define ARROW_HEAD_LEN .02

/*
 * ARROW_ANGLE is the angle that the arrow heads flair out from the
 * line.  This should be in radians.  This is currently 15 degrees
*/
#define ARROW_ANGLE .26179939

/*
 * NUMPOINTS defines the number of points to use when computing the
 * spline that goes from one node to the next.
*/
#define NUMPOINTS 11

/*
 * NEXTBEZIER is designed to do the appropriate things to move along
 * to the next Bezier spline point.
*/
#define NEXTBEZIER() {\
    fxdiff0 += fxdiff1;\
    fydiff0 += fydiff1;\
    fxdiff1 += fxdiff2;\
    fydiff1 += fydiff2;\
    fxdiff2 += fxdiff3;\
    fydiff2 += fydiff3;\
    index++; }
   
/*
 * CALCPOINT is designed to take the x and y locations and multiply
 * them by graph coordinates to get x and y display values.  It stores
 * them in the points array and increments the counter.
*/
#define CALCPOINT(x_value, y_value) {\
    save_x[PointCtr] = x_value;\
    save_y[PointCtr] = y_value;\
    points[PointCtr].x = (x_value) * graph->viewing_area.width;\
    points[PointCtr].y = (y_value) * graph->viewing_area.height;\
    PointCtr++; }

/*
   rotate was created by Doug Bogia on Aug. 26, 1991
   Description:
     It is designed to take the given coordinates, rotate them according
     to sin_rot and cos_rot, and then add on the x and y offset given
     (probably a center around which things are being rotated).
   Entry:
     orig_x----------> The original x value that is being rotated
     orig_y----------> The original y value that is being rotated
     sin_rot---------> The sin of the rotation.  The sine and cosine are
       passed in instead of the theta so if you are doing lots of rotations
       by the same amount, sine and cosine are only computed once.
     cos_rot---------> Cosine of the rotation.
     xc--------------> This is the center that should be added to the points
     yc--------------> This is the center that should be added to the points
     new_x-----------> None.
     new_y-----------> None.
   Exit:
     orig_x----------> None.
     orig_y----------> None.
     sin_rot---------> None.
     cos_rot---------> None.
     xc--------------> None.
     yc--------------> None.
     new_x-----------> This is the newly rotated x value
     new_y-----------> This is the newly rotated y value
   Returns:
     None.
*/
void rotate (orig_x, orig_y, sin_rot, cos_rot, xc, yc, new_x, new_y)
  coord_t            orig_x;
  coord_t            orig_y;
  coord_t            sin_rot;
  coord_t            cos_rot;
  coord_t            xc;
  coord_t            yc;
  coord_t *          new_x;
  coord_t *          new_y;
{
  /* Start of variables section */
  /* End of variables section */
  *new_x = (cos_rot * orig_x - sin_rot * orig_y) + xc;
  *new_y = (sin_rot * orig_x + cos_rot * orig_y) + yc;
} /* End of rotate */

/*
   InBox was created by Doug Bogia on Aug. 26, 1991
   Description:
     This is designed to tell if the given point is inside the box
     bounded by xmin, ymin and xmax, ymax
   Entry:
     x---------------> X value of point being checked
     y---------------> Y value of point being checked
     xmin------------> This is the minimum x value of the bounding box
     ymin------------> This is the minimum y value of the bounding box
     xmax------------> This is the maximum x value of the bounding box
     ymax------------> This is the maximum y value of the bounding box
   Exit:
     x---------------> None.
     y---------------> None.
     xmin------------> None.
     ymin------------> None.
     xmax------------> None.
     ymax------------> None.
   Returns:
     true if the point is in the box false otherwise.
*/
int InBox (x, y, xmin, ymin, xmax, ymax)
  coord_t            x;
  coord_t            y;
  coord_t            xmin;
  coord_t            ymin;
  coord_t            xmax;
  coord_t            ymax;
{
  /* Start of variables section */
  /* End of variables section */
  return (x >= xmin &&
	  x <= xmax &&
	  y >= ymin &&
	  y <= ymax);
} /* End of InBox */

/*
   BoxIntersect was created by Doug Bogia on Aug. 26, 1991
   Description:
     This computes the intersection of a line and a box.  It is assumed that
     the one point given is inside the box and the other is outside of
     the box.  If this is not the case, results are going to be weird.
   Entry:
     x1--------------> This is the x location of one of the line points
     y1--------------> This is the y location of one of the line points
     x2--------------> This is the x location of the other of the line points
     y2--------------> This is the x location of the other of the line points
     xmin------------> This is the minimum x of the box
     ymin------------> This is the minimum y of the box
     xmax------------> This is the maximum x of the box
     ymax------------> This is the maximum y of the box
     x---------------> None.
     y---------------> None.
   Exit:
     x1--------------> None.
     y1--------------> None.
     x2--------------> None.
     y2--------------> None.
     xmin------------> None.
     ymin------------> None.
     xmax------------> None.
     ymax------------> None.
     x---------------> The computed intersection point of the line and the box
     y---------------> The computed intersection point of the line and the box
   Returns:
     None.
*/
void BoxIntersect (x1, y1, x2, y2, xmin, ymin, xmax, ymax, x, y)
  coord_t            x1;
  coord_t            y1;
  coord_t            x2;
  coord_t            y2;
  coord_t            xmin;
  coord_t            ymin;
  coord_t            xmax;
  coord_t            ymax;
  coord_t *          x;
  coord_t *          y;
{
  /* Start of variables section */
  coord_t            lxmax;	/* line x maximum for bounding box */
  coord_t            lxmin;	/* line x minimum for bounding box */
  coord_t            lymax;	/* line y maximum for bounding box */
  coord_t            lymin;	/* line y minimum for bounding box */
  coord_t            slope;	/* Slope of the line */
  coord_t            xdiff;	/* This is the difference in x's */
  coord_t            ydiff;	/* This is the difference in the y's */
  /* End of variables section */

  xdiff = x1 - x2;
  ydiff = y1 - y2;
  if (x1 < x2)
  {
    lxmin = x1;
    lxmax = x2;
  }
  else
  {
    lxmin = x2;
    lxmax = x1;
  }
  if (y1 < y2)
  {
    lymin = y1;
    lymax = y2;
  }
  else
  {
    lymin = y2;
    lymax = y1;
  }
  if (xdiff == 0.0)		/* Vertical line */
  {
    *x = x1;
    if (InBox (x1, ymin, lxmin, lymin, lxmax, lymax))
    {
      *y = ymin;
    }
    else
    {
      *y = ymax;
    }
  }
  else if (ydiff == 0.0)	/* Horizontal line */
  {
    *y = y1;
    if (InBox (xmin, y1, lxmin, lymin, lxmax, lymax))
    {
      *x = xmin;
    }
    else
    {
      *x = xmax;
    }
  }
  else
  {
    /* Check for intersection with 2 horizontal lines first. */
    slope = xdiff / ydiff;	/* Actually 1/m, but we know y, not x */
    *y = ymin;
    *x = slope * (*y - y1) + x1;
    if (InBox (*x, *y, lxmin, lymin, lxmax, lymax) &&
	InBox (*x, *y, xmin, ymin, xmax, ymin))
    {
      return;
    }
    *y = ymax;
    *x = slope * (*y - y1) + x1;
    if (InBox (*x, *y, lxmin, lymin, lxmax, lymax) &&
	InBox (*x, *y, xmin, ymax, xmax, ymax))
    {
      return;
    }
    /* Check vertical lines now */
    slope = 1.0 / slope;
    *x = xmin;
    *y = slope * (*x - x1) + y1;
    if (InBox (*x, *y, lxmin, lymin, lxmax, lymax) &&
	InBox (*x, *y, xmin, ymin, xmin, ymax))
    {
      return;
    }
    /* If it doesn't intersect the other lines, must intersect the
     * last one.
     */
    *x = xmax;
    *y = slope * (*x - x1) + y1;
    return;
  }
} /* End of BoxIntersect */

/*
   draw_arrow was created by Doug Bogia on Aug. 26, 1991
   Description:
     This routine takes x1, y1 and x2, y2 and tries to draw an arrow
     head at the location x1, y1.  x2, y2 is needed to be able to compute
     the angels for the arrow head lines.
   Entry:
     x1--------------> This is the x location to draw the arrow head
     y1--------------> This is the y location to draw the arrow head
     x2--------------> This is used to get a direction on the line
     y2--------------> This is used to get a direction on the line
   Exit:
     x1--------------> None.
     y1--------------> None.
     x2--------------> None.
     y2--------------> None.
   Returns:
     1 if x1, y1 and x2, y2 are the same point (can't get a direction for
     the arrow head from that); otherwise, 0 is returned
*/
int draw_arrow (x1, y1, x2, y2)
  coord_t            x1;
  coord_t            y1;
  coord_t            x2;
  coord_t            y2;
{
  /* Start of variables section */
  XPoint           points[4];	/* These are the points on the arrow head */
  coord_t            arrow_x;     /* This is the x location of the arrow */
  coord_t            arrow_y;     /* This is the y location of the arrow */
  coord_t            cos_rot;     /* This is the cosine of the rotation angle */
  coord_t            sin_rot;     /* This is the sine of the rotation angle */
  coord_t            theta;       /* This is the angle of the line */
  /* End of variables section */
  if (calc_angle (x2, y2, x1, y1, &theta))
  {
    return (1);
  }
  /* Ok, so now that we have theta we want one side of the arrow head
   * at theta + 30 degrees and the other at theta - 30 degrees.
   */
  sin_rot = sin(theta + ARROW_ANGLE);
  cos_rot = cos(theta + ARROW_ANGLE);
  rotate (ARROW_HEAD_LEN, 0.0, sin_rot, cos_rot, x1, y1, &arrow_x, &arrow_y);
  points[0].x = arrow_x * graph->viewing_area.width;
  points[0].y = arrow_y * graph->viewing_area.height;
  points[1].x = x1 * graph->viewing_area.width;
  points[1].y = y1 * graph->viewing_area.height;
  sin_rot = sin(theta - ARROW_ANGLE);
  cos_rot = cos(theta - ARROW_ANGLE);
  rotate (ARROW_HEAD_LEN, 0.0, sin_rot, cos_rot, x1, y1, &arrow_x, &arrow_y);
  points[2].x = arrow_x * graph->viewing_area.width;
  points[2].y = arrow_y * graph->viewing_area.height;
  points[3].x = points[0].x;
  points[3].y = points[0].y;
  XDrawLines(XtDisplay(graph->viewing_area.widget),
	     XtWindow(graph->viewing_area.widget),
	     graph->graph_gc,
	     points, 4, CoordModeOrigin);
} /* End of draw_arrow */

/*
   draw_arc was created by Doug Bogia on Aug. 26, 1991
   Description:
     This function is designed to draw an arc from x1, y1 to x2, y2.  If
     x1min is >= 0, then it is assumed that there is a bounding box of
     x1min, y1min to x1max, y1max that surrounds x1, y1.  Likewise if
     x2min is >= 0, there is a bounding box around x2, y2.  These are the
     boxes that is composed of text.  The reason for not requiring the
     bounding box is that hyperedges will have a single point with
     no bounding box.
   Entry:
     x1--------------> This is the x location of the first point
     y1--------------> This is the y location of the first point
     x1min-----------> This is minimum x of the bounding box around x1, y1
     y1min-----------> This is the minimum y of the bounding box around x1, y1
     x1max-----------> This is the maximum x of the bounding box around x1, y1
       If it is < 0 then there is no bounding box
     y1max-----------> This is the maximum y of the bounding box around x1, y1
     x2--------------> This is the x location of the second point
     y2--------------> This is the y location of the second point
     x2min-----------> This is minimum x of the bounding box around x2, y2
     y2min-----------> This is the minimum y of the bounding box around x2, y2
     x2max-----------> This is the maximum x of the bounding box around x2, y2
       If it is < 0 then there is no bounding box
     y2max-----------> This is the maximum y of the bounding box around x2, y2
     offset----------> This is the offset from the x1, y1 to x2, y2 line
       segment that we want our arc drawn.  If offset is positive it is
       above the line.  Above and below is computed assuming that x2, y2 lies
       at the 3 o'clock position from x1, y1.  Thus if x2, y2 is at the
       9 o'clock position the arc is drawn below the line.
     arrows----------> This defines whether there should be arrow heads
       drawn on the arc.  There are four arrow heads that can be drawn.
       INARROW1 draws an arrow incoming to x1, y1
       OUTARROW1 draws an arrow outgoing from x1, y1.
       INARROW2 draws an arrow incoming to x2, y2
       OUTARROW2 draws an arrow outgoing from x2, y2
       These can be ORed together to get multiple arrow heads.
   Exit:
     x1--------------> None.
     y1--------------> None.
     x1min-----------> None.
     y1min-----------> None.
     x1max-----------> None.
     y1max-----------> None.
     x2--------------> None.
     y2--------------> None.
     x2min-----------> None.
     y2min-----------> None.
     x2max-----------> None.
     y2max-----------> None.
     offset----------> None.
     arrows----------> None.
   Returns:
     None.
*/
void draw_arc (x1, y1, x1min, y1min, x1max, y1max, x2, y2,
	       x2min, y2min, x2max, y2max, offset, arrows)
  coord_t            x1;
  coord_t            y1;
  coord_t            x1min;
  coord_t            y1min;
  coord_t            x1max;
  coord_t            y1max;
  coord_t            x2;
  coord_t            y2;
  coord_t            x2min;
  coord_t            y2min;
  coord_t            x2max;
  coord_t            y2max;
  coord_t            offset;
  int                     arrows;
{
  /* Start of variables section */
  int              index, numpoints;	/* Count of Bezier points computed */
  XPoint           points[NUMPOINTS+2];	/* Points making up spline */
  coord_t            Lastx;	/* Last x location either in or out of a box */
  coord_t            Lasty;	/* Last y location either in or out of a box */
  coord_t            ax0;		/* This is zero coefficient x value */
  coord_t            ax1;		/* This is zero coefficient x value */
  coord_t            ax2;		/* This is zero coefficient x value */
  coord_t            ay0;		/* This is zero coefficient y value */
  coord_t            ay1;		/* This is zero coefficient y value */
  coord_t            ay2;		/* This is zero coefficient y value */
  coord_t            cos_rot;	/* cosine of the rotation angle */
  coord_t            fxdiff0;	/* Forward x difference 0 */
  coord_t            fxdiff1;	/* Forward x difference 1 */
  coord_t            fxdiff2;	/* Forward x difference 2 */
  coord_t            fxdiff3;	/* Forward x difference 3 */
  coord_t            fydiff0;	/* Forward y difference 0 */
  coord_t            fydiff1;	/* Forward y difference 1 */
  coord_t            fydiff2;	/* Forward y difference 2 */
  coord_t            fydiff3;	/* Forward y difference 3 */
  coord_t            save_x[NUMPOINTS+2];	/* Save the x values for drawing arrow heads */
  coord_t            save_y[NUMPOINTS+2];	/* Save up the y values for arrows */
  coord_t            sin_rot;	/* Sine of the rotation angle */
  coord_t            tempx1;	/* Temp x value to save on multiplies */
  coord_t            tempx2;	/* Temp x value to save on multiplies */
  coord_t            tempy1;	/* Temp y value to save on multiplies */
  coord_t            tempy2;	/* Temp y value to save on multiplies */
  coord_t            theta;	/* This is an angle used for computations */
  coord_t            x;		/* Gets a new x value */
  coord_t            xc;		/* Center of the line from x1, y1 to x2, y2 */
  coord_t            xmid1;	/* This spline's first mid vertex's x value */
  coord_t            xmid2;	/* This spline's second mid vertex's x value */
  coord_t            y;		/* Gets a new y value */
  coord_t            yc;		/* Center of the line from x1, y1 to x2, y2 */
  coord_t            ymid1;	/* This spline's first mid vertex's y value */
  coord_t            ymid2;	/* This spline's second mid vertex's y value */
  int              PointCtr;	/* Count of useful points computed */
  static coord_t     delta;	/* step changes along Bezier spline */
  static coord_t     deltacubed;	/* delta cubed for Bezier spline */
  static coord_t     deltasqrd;	/* delta squared for Bezier spline */
  static short     first_time = 1; /* This tells if it is the first entry */

  /* End of variables section */
  /* For speed purposes you could look to see if the offset is 0.0 here
   * and if it is, just draw a line.  The reason I didn't do that in this
   * first pass is that we would have to compute all the arrows here also.
   * So for now, I will just compute the line as another form of an arc.
   */
  if (first_time)
  {
    /* To save on CPU time, we will compute things that will never
     * change here.
     */
    first_time = 0;
    delta = 1.0 / NUMPOINTS;
    deltasqrd = delta * delta;
    deltacubed = deltasqrd * delta;
  }
  /* First we want to compute the angle of the line between x1, y1 and
   * x2, y2.  We do this so we can figure out the two other control
   * points for the Bezier spline.
   */
  if (calc_angle (x1, y1, x2, y2, &theta))
  {
    /* No need to go on, x1, y1 is the same point as x2, y2.  Can't get
     * much of an arc.
     */
    return;
  }
  sin_rot = sin(theta);
  cos_rot = cos(theta);
  xc = (x1 + x2) / 2.0;
  yc = (y1 + y2) / 2.0;
  rotate (0.0, offset, sin_rot, cos_rot, xc, yc, &xmid1, &ymid1);
  xmid2 = xmid1;
  ymid2 = ymid1;
  /* Compute the coefficients of the spline function.  See page 204 - 205
   * of Computer Graphics by Donald Hearn and M. Pauline Baker for info
   * on Bezier splines and forward differences.
   */
  ax0 = 3 * (xmid1 - xmid2) + x2 - x1;
  ay0 = 3 * (ymid1 - ymid2) + y2 - y1;
  ax1 = 3 * (x1 - 2 * xmid1 + xmid2);
  ay1 = 3 * (y1 - 2 * ymid1 + ymid2);
  ax2 = 3 * (xmid1 - x1);
  ay2 = 3 * (ymid1 - y1);
  /* Compute the forward differences. */
  tempx1 = ax0 * deltacubed;
  tempy1 = ay0 * deltacubed;
  tempx2 = ax1 * deltasqrd;
  tempy2 = ay1 * deltasqrd;
  fxdiff3 = 6 * tempx1;
  fydiff3 = 6 * tempy1;
  fxdiff2 = fxdiff3 + 2 * tempx2;
  fydiff2 = fydiff3 + 2 * tempy2;
  fxdiff1 = tempx1 + tempx2 + ax2 * delta;
  fydiff1 = tempy1 + tempy2 + ay2 * delta;
  fxdiff0 = x1;
  fydiff0 = y1;
  index = 0;
  PointCtr = 0;
  /* Now if we wanted we could put the test for x1min being less than 0
   * in the InBox code.  This might make this code shorter; however, doing
   * it this way allows us to be more efficient since we aren't testing
   * x1min all the time.
   */
  if (x1max < 0.0) 
  {
    CALCPOINT (x1, y1);
  }
  else
  {
    Lastx = x1;
    Lasty = y1;
    NEXTBEZIER();
    while (InBox (fxdiff0, fydiff0, x1min, y1min, x1max, y1max) &&
	   index < NUMPOINTS + 1)
    {
      Lastx = fxdiff0;
      Lasty = fydiff0;
      NEXTBEZIER();
    }
    if (index >= NUMPOINTS+1)
    {
      return;			/* Never found a point outside bounding box */
    }
    BoxIntersect (Lastx, Lasty, fxdiff0, fydiff0, x1min, y1min, x1max, y1max,
		  &x, &y);
    if (InBox (x, y, x2min, y2min, x2max, y2max))
    {
      return;			/* Point is inside of 2nd bounding box */
    }
    CALCPOINT (x, y);
  }
  NEXTBEZIER();
  if (x2max < 0.0)
  {
    do 
    {
      CALCPOINT(fxdiff0, fydiff0);
      NEXTBEZIER();
    } while (index < NUMPOINTS+1);
  }
  else
  {
    Lastx = x;
    Lasty = y;
    while (!InBox(fxdiff0, fydiff0, x2min, y2min, x2max, y2max)
	   && index < NUMPOINTS+1)
    {
      CALCPOINT (fxdiff0, fydiff0);
      Lastx = fxdiff0;
      Lasty = fydiff0;
      NEXTBEZIER();
    }
    if (index < NUMPOINTS+1)
    {
      BoxIntersect (Lastx, Lasty, fxdiff0, fydiff0,
		    x2min, y2min, x2max, y2max, &x, &y);
      CALCPOINT (x, y);
    }
  }
  XDrawLines(XtDisplay(graph->viewing_area.widget), XtWindow(graph->viewing_area.widget),
	     graph->graph_gc, points, PointCtr, CoordModeOrigin);
  if (arrows & INARROW1 && PointCtr > 1)
  {
    draw_arrow (save_x[0], save_y[0], save_x[1], save_y[1]);
  }
  if (arrows & OUTARROW1 && PointCtr > 3)
  {
    index = PointCtr / 1.5;
    draw_arrow (save_x[index], save_y[index],
		save_x[index-1], save_y[index-1]);
  }
  if (arrows & INARROW2 && PointCtr > 1)
  {
    draw_arrow (save_x[PointCtr-1], save_y[PointCtr-1],
		save_x[PointCtr-2], save_y[PointCtr-2]);
  }
  if (arrows & OUTARROW2 && PointCtr > 3)
  {
    index = PointCtr - PointCtr / 1.5;
    draw_arrow (save_x[index], save_y[index],
		save_x[index+1], save_y[index+1]);
  }
} /* End of draw_arc */


