/* This file is part of the
 *
 *      Delta Project  (ConversationBuilder)
 *      Human-Computer Interaction Laboratory
 *      University of Illinois at Urbana-Champaign
 *      Department of Computer Science
 *      1304 W. Springfield Avenue
 *      Urbana, Illinois 61801
 *      USA
 *
 *      c 1989,1990,1991 Board of Trustees
 *              University of Illinois
 *              All Rights Reserved
 *
 *      This file is distributed under license and is confidential
 *
 *      File title and purpose
 *      Author:  Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *      Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *      Direct enquiries to the project leader please.
 */

#include "extern.h"
#include "graph.h"
/*
 * SelectedNodes returns a list of nodes that are selected.  The
 * routine FreeNodeList should be called to free this space.
 */
NodeList *
SelectedNodes()
{
  NodeList *head = NULL;
  Vertex *node;
  for (node = graph->v; node; node = node->next)
  {
    if (NodeSelected(node))
	AddNodetoList(&head, node);
  }
  return head;
}  

/*
 * SweptNodes returns a list of nodes that are in the swept region.  The
 * routine FreeNodeList should be called to free this space.
 */
NodeList *
SweptNodes(CallbackInfo)
  struct CallbackInfo *CallbackInfo;
{
  coord_t minx, miny, maxx, maxy;
  NodeList *head = NULL;
  Vertex *node;

  if (CallbackInfo->start_x > CallbackInfo->end_x)
  {
    minx = VIEWX_INTX(CallbackInfo->end_x);
    maxx = VIEWX_INTX(CallbackInfo->start_x);
  }
  else
  {
    minx = VIEWX_INTX(CallbackInfo->start_x);
    maxx = VIEWX_INTX(CallbackInfo->end_x);
  }
  if (CallbackInfo->start_y > CallbackInfo->end_y)
  {
    miny = VIEWY_INTY(CallbackInfo->end_y);
    maxy = VIEWY_INTY(CallbackInfo->start_y);
  }
  else
  {
    miny = VIEWY_INTY(CallbackInfo->start_y);
    maxy = VIEWY_INTY(CallbackInfo->end_y);
  }
  for (node = graph->v; node; node = node->next)
      if (InBox(node->x, node->y, minx, miny, maxx, maxy))
	  AddNodetoList(&head, node);
  return head;
}  

/*
 * SelectedEdges returns a list of edges that are selected.  The
 * routine FreeEdgeList should be called to free this space.
 */
EdgeList *
SelectedEdges()
{
  EdgeList *head = NULL;
  Edge *edge;
  for (edge = graph->edges; edge; edge = edge->next)
  {
    if (EdgeSelected(edge))
	AddEdgetoList(&head, edge);
  }
  return head;
}  

void
AssignMin (min, new)
  coord_t *min, new;
{
  if (*min > new)
      *min = new;
}

void
AssignMax (max, new)
  coord_t *max, new;
{
  if (*max < new)
      *max = new;
}

/* Given a node list, this routine goes out and computes the bounding
 * box that contains all the nodes.  It returns TRUE if there was something
 * in the node list and FALSE otherwise.
 */
int
BoundingBox (head, minx, miny, maxx, maxy)
  NodeList *head;
  coord_t *minx, *miny, *maxx, *maxy;
{
  if (head)
  {
    *minx = MINX(head->node);
    *miny = MINY(head->node);
    *maxx = MAXX(head->node);
    *maxy = MAXY(head->node);
    while (head = head->next)
    {
      AssignMin(minx, MINX(head->node));
      AssignMin(miny, MINY(head->node));
      AssignMax(maxx, MAXX(head->node));
      AssignMax(maxy, MAXY(head->node));
    }      
    return TRUE;
  }
  else
      return FALSE;
}

/*
 * Here we want to draw a box in both the virtual area and in the viewing
 * area since the selected nodes may be outside the viewing area.
 */
void
DrawSelectedBox (gc, minx, miny, maxx, maxy)
  GC gc;
  coord_t minx, miny, maxx, maxy;
{
  int iminx, iminy, imaxx, imaxy, ilenx, ileny;
  coord_t tmp;

  Display *dpy;
  Window win;

  if (minx > maxx)
  {
    tmp = minx;
    minx = maxx;
    maxx = tmp;
  }
  if (miny > maxy)
  {
    tmp = miny;
    miny = maxy;
    maxy = tmp;
  }

  iminx = INTX_VIEWX(minx);
  iminy = INTY_VIEWY(miny);
  imaxx = INTX_VIEWX(maxx);
  imaxy = INTY_VIEWY(maxy);
  
  ilenx = imaxx - iminx;
  ileny = imaxy - iminy;
  
  dpy = XtDisplay(graph->viewing_area.widget);
  win = XtWindow(graph->viewing_area.widget);

  XDrawRectangle(dpy, win, gc, iminx, iminy, ilenx, ileny);
  
  dpy = XtDisplay(graph->virtual_area.widget);
  win = XtWindow(graph->virtual_area.widget);

  iminx = INTX_VIRTX(minx);
  iminy = INTY_VIRTY(miny);
  imaxx = INTX_VIRTX(maxx);
  imaxy = INTY_VIRTY(maxy);
  
  ilenx = imaxx - iminx;
  ileny = imaxy - iminy;
  
  XDrawRectangle(dpy, win, gc, iminx, iminy, ilenx, ileny);
}  

/*
 * Here we want to draw a box in the viewing area.
 */
void
DrawSweptBox (gc, minx, miny, maxx, maxy)
  GC gc;
  int minx, miny, maxx, maxy;
{
  int ilenx, ileny, tmp;

  Display *dpy;
  Window win;

  if (minx > maxx)
  {
    tmp = minx;
    minx = maxx;
    maxx = tmp;
  }
  if (miny > maxy)
  {
    tmp = miny;
    miny = maxy;
    maxy = tmp;
  }

  ilenx = maxx - minx;
  ileny = maxy - miny;
  
  dpy = XtDisplay(graph->viewing_area.widget);
  win = XtWindow(graph->viewing_area.widget);

  XDrawRectangle(dpy, win, gc, minx, miny, ilenx, ileny);
}  
