/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/*
 * Twiddled by Doug Bogia on 02/13/92. 
 * An option to ignore interrupt (SIGINT) was added so the interrupt
 * to the common lisp process wouldn't kill the underlying connect.
 */

/* Provide a direct tty connection to the message bus */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/commands/RCS/connect.c,v $ */

static char rcsid[] = "$Revision: 2.1.1.2 $ $Date: 91/11/15 13:37:56 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include "header.h"

#include <netdb.h>
#include <sys/signal.h>

#if POLL
#include <poll.h>
#include <stropts.h>
#endif
#include "getopt.h"
#include "cb-defs.h"
/* ------------------------------------------------------------------------- */
static int trace_fd = -1;
static int port = DEF_MBUS_PORT;
static char *host = NULL;
static int noint = 0;

static char print_hello = 0;
static char hello_message[] = { "\n(\"connect\" \"bus\")\n" };
/* ------------------------------------------------------------------------- */
void
bomb(s) char *s;
{
  perror(s);
  exit(1);
}
/* ------------------------------------------------------------------------- */
int
ConnectSocket(host,port)
     char *host;
     int port;
{
  int sock;
  int flags;
  struct sockaddr_in saddr;
  struct hostent *h;
  char hostname[64];

  if (port <= 0) port = 0x956;		/* magic number! */
  if (NULL == host || 0 == *host)
    {
      host = hostname;
      if (0 > gethostname(host,64)) bomb("Can't get host name");
    }

  sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (sock < 0) bomb("Cannot open socket");

  h = gethostbyname(host);

  if (NULL == h) bomb("Cannot find host");
  if (h->h_addr_list == 0) bomb("No addresses for host");

  saddr.sin_family = AF_INET;
  saddr.sin_addr.s_addr = INADDR_ANY;
  saddr.sin_port = 0;			/* anywhere */

  saddr.sin_family = AF_INET;
  memcpy((char *) &saddr.sin_addr, h->h_addr_list[0], h->h_length);
  saddr.sin_port = htons(port);

  if (connect(sock, (struct sockaddr *) &saddr, sizeof(saddr)))
    bomb("Couldn't connect to Mbus");

  return sock;
}
/* ------------------------------------------------------------------------- */
#define BUFFSIZE 32768
#define SET_NDELAY(f)  { int ff; fcntl(f, F_GETFL, &ff); ff |= O_NDELAY; fcntl(f, F_SETFL, ff); }
#define RESET_NDELAY(f)  { int ff; fcntl(f, F_GETFL, &ff); ff &= ~O_NDELAY; fcntl(f, F_SETFL, ff); }

#if POLL
#define SOCKET_READ_OK	(p_fd[2].revents & POLLIN)
#define SOCKET_WRITE_OK (p_fd[2].revents & POLLOUT)
#define CLIENT_READ_OK  (p_fd[0].revents & POLLIN)
#define CLIENT_WRITE_OK (p_fd[1].revents & POLLOUT)
#else
#define SOCKET_READ_OK  FD_ISSET(s, &read_fds)
#define SOCKET_WRITE_OK FD_ISSET(s, &write_fds)
#define CLIENT_READ_OK  FD_ISSET(fileno(stdin), &read_fds)
#define CLIENT_WRITE_OK FD_ISSET(fileno(stdout), &write_fds)
#endif
/* ------------------------------------------------------------------------- */
void
CleanUpAndExit(val)
     int val;
{
  RESET_NDELAY(fileno(stdin));
  RESET_NDELAY(fileno(stdout));
  exit(val);
}

#ifdef HPUX
void
#else  
int
#endif
SignalHandler(sig)
     int sig;
{
  switch (sig)
    {
    default: CleanUpAndExit(2); break;
    }
}

void
Run(s) int s;
{
#if POLL
  struct pollfd p_fd[3];
#else
  fd_set read_fds, write_fds;
#endif

  int ff;				/* file flags var */
  int n;				/* n of ready fd's */
  int count;				/* result of read/write */
  struct
    {
      int head,tail;
      int count;
      char buff[BUFFSIZE];
    } client, socket;

#if POLL
  p_fd[0].fd = fileno(stdin);
  p_fd[1].fd = fileno(stdout);
  p_fd[2].fd = s;
#endif

  client.head = client.tail = socket.head = socket.tail = 0;
  client.count = socket.count = 0;

  /* write out initial message */
  if (print_hello)
    {
      memcpy(client.buff+client.head,hello_message,sizeof(hello_message)-1);
      client.count += sizeof(hello_message);
      client.head += sizeof(hello_message);
    }

  /* put in some start up stuff */

  if (noint) signal(SIGINT, SIG_IGN);
  else signal(SIGINT, SignalHandler);

  signal(SIGHUP, SignalHandler);
  signal(SIGTERM, SignalHandler);
  signal(SIGQUIT, SignalHandler);

  SET_NDELAY(s);
  SET_NDELAY(fileno(stdin));
  SET_NDELAY(fileno(stdout));

  while (1)
    {
#if POLL
      p_fd[2].events = (client.count < BUFFSIZE ? POLLIN : 0)
	| (socket.count ? POLLOUT : 0);
      p_fd[0].events = socket.count < BUFFSIZE ? POLLIN : 0;
      p_fd[1].events = client.count ? POLLOUT : 0;
      n = poll(p_fd, 3, -1);
#else
      FD_ZERO(&read_fds); FD_ZERO(&write_fds);
      if (socket.count < BUFFSIZE) FD_SET(fileno(stdin), &read_fds);
      if (client.count) FD_SET(fileno(stdout), &write_fds);
      if (client.count < BUFFSIZE) FD_SET(s, &read_fds);
      if (socket.count) FD_SET(s, &write_fds);
      n = select(16, &read_fds, &write_fds, 0, 0);
#endif

      if (n <= 0) continue;

      /* deal with the write's first */

      if (SOCKET_WRITE_OK)
	{
	  count = socket.head - socket.tail;
	  if (count <= 0) count = BUFFSIZE - socket.tail;
	  count = write(s, socket.buff + socket.tail, count);
	  if (count > 0)
	    {
	      socket.tail = (socket.tail + count) % BUFFSIZE;
	      socket.count -= count;
	    }
	}
      if (CLIENT_WRITE_OK)
	{
	  count = client.head - client.tail;
	  if (count <= 0) count = BUFFSIZE - client.tail;
	  count = write(fileno(stdout), client.buff + client.tail, count);
	  if (count > 0)
	    {
	      client.tail = (client.tail + count) % BUFFSIZE;
	      client.count -= count;
	    }
	}

      /* now, read */
      if (CLIENT_READ_OK)
	{
	  count = socket.tail - socket.head;
	  if (count <= 0) count = BUFFSIZE - socket.head;
	  count = read(fileno(stdin), socket.buff + socket.head, count);
	  if (count > 0)
	    {
	      if (trace_fd > 0) write(trace_fd,socket.buff+socket.head,count);
	      socket.head = (socket.head + count) % BUFFSIZE;
	      socket.count += count;
	    }
	  else if (count == 0)
	    {
	      fprintf(stderr, "Lost connection to terminal\n");
	      close(s);
	      CleanUpAndExit(3);
	    }
	}
      if (SOCKET_READ_OK)
	{
	  count = client.tail - client.head;
	  if (count <= 0) count = BUFFSIZE - client.head;
	  count = read(s, client.buff + client.head, count);
	  if (count > 0)
	    {
	      if (trace_fd > 0) write(trace_fd,client.buff+client.head,count);
	      client.head = (client.head + count) % BUFFSIZE;
	      client.count += count;
	    }
	  else if (count == 0)
	    {
	      fprintf(stderr, "Lost connection to Mbus\n");
	      CleanUpAndExit(4);
	    }
	}
    }
}
/* ------------------------------------------------------------------------- */
int DealWithCommandLine(argc,argv)
     int argc;
     char **argv;
{
  int opt;
  char *tmp;
  char useage = 0;
  extern char *getenv();

  /* environment variable checks */
  host = getenv(ENV_MBUS_HOST);
  if (NULL != (tmp = getenv(ENV_MBUS_PORT))) port = strtol(tmp, NULL, 0);

  while ( (opt = getopt(argc,argv,"iIlLt;h:p:")) != EOF)
    switch (opt)
      {
      case '?' :
	fprintf(stderr,"%c\tBad option\n",opt_err_char);
	useage = 1;
	break;
      case 'l' : print_hello = 1; break;
      case 'L' : print_hello = 0; break;
      case 'i' : noint = 1; break;
      case 'I' : noint = 0; break;
      case 'h' : host = optarg; break;
      case 'p' : port = strtol(optarg, NULL, 0); break;
      case 't' :
	if (optarg == NULL || !strcmp(optarg,"=")) trace_fd = fileno(stdout);
	else
	  {
	    /* some systems don't have O_SYNC (like the NeXT) */
#ifdef O_SYNC
	    trace_fd = open(optarg, O_WRONLY|O_SYNC|O_APPEND|O_CREAT, 0666);
#else
	    trace_fd = open(optarg, O_WRONLY|O_APPEND|O_CREAT, 0666);
#endif
	    if (trace_fd < 0)
	      {
		fprintf(stderr,"Trace file %s : " , optarg);
		perror("open");
	      }
	  }
	break;
      }

  if (useage)
    {
      printf("Useage: %s -h host -p port\n", argv[0]);
      exit(5);
    }
  return optind;
}
/* ------------------------------------------------------------------------- */
main(argc,argv)
     int argc;
     char *argv[];
{
  int skip;
  int s;

  skip = DealWithCommandLine(argc,argv);

  /* this succeeds or doesn't return */
  s = ConnectSocket(host,port);
  Run(s);
}
/* ------------------------------------------------------------------------- */
