;;; analysis.lsp 2.2 7/11/88 16:35:00
;;; Copyright (c) 1987, Benjamin G. Zorn
;;;

;;; MISCELLANEOUS

(defvar *debug* nil)

(defvar *language* :normal)		;; settings :terse, :normal, :verbose
(defvar *leak-level* :show)		; settings :none, :show, :offsets

(defun logical-or (x y)
  (or x y))


(defun logical-and (x y)
  (and x y))



;;; EDGE structure
;;;


(defstruct (edge (:print-function print-edge))
  (from nil)
  (to nil)
  (other nil)
  (save nil))


(eval-when (compile load eval)
	   
(defun define-edge-slot (slot-name)
  (setf (macro-function (intern (string-upcase
				 (format nil "EDGE-~a" slot-name))))
	#'(lambda (f e)
	    (declare (ignore e))
	    `(getf (edge-other ,(second f)) ',slot-name))))

;; for allocation information
(define-edge-slot 'count)

;; for strongly connected component detection
(define-edge-slot 'mark)

) ;;; end eval-when


(defun print-edge (e stream depth)
  (declare (ignore depth))
  (format stream "#e(~d ~d)"
	  (symbol-name (vertex-name (edge-from e)))
	  (symbol-name (vertex-name (edge-to e)))
	  #+all (edge-count e)))


(defun save-edge-state (e)
  (push (edge-other e) (edge-save e))
  (setf (edge-other e) nil))


(defun restore-edge-state (e)
  (setf (edge-other e) (pop (edge-save e))))


(defun e-to-names (e)
  (values (symbol-name (vertex-name (edge-from e)))
	  (symbol-name (vertex-name (edge-to e)))))

	   
(defun edge-less-p (e1 e2)
  (multiple-value-bind (from-name1 to-name1) (e-to-names e1)
    (multiple-value-bind (from-name2 to-name2) (e-to-names e2)
      (cond ((string= from-name1 from-name2)
	     (string< to-name1 to-name2))
	    ((string< from-name1 from-name2))))))


(defun kill-edge (e)
  (setf (edge-from e) nil)
  (setf (edge-to e) nil))


(defun null-edge-p (e)
  (and (null (edge-from e))
       (null (edge-to e))))


;;; VERTEX structure
;;;


(defstruct (vertex (:print-function print-vertex))
  (name)
  (number nil)
  (edges nil)
  (backedges nil)
  (other nil)
  (save nil))


(eval-when (compile load eval)

(defun define-vertex-slot (slot-name)
  (setf (macro-function (intern (string-upcase
				 (format nil "VERTEX-~a" slot-name))))
	#'(lambda (f e)
	    (declare (ignore e))
	    `(getf (vertex-other ,(second f)) ',slot-name))))

;; for random things
(define-vertex-slot 'scratch)

;; for allocation information
(define-vertex-slot 'srefs)
(define-vertex-slot 'lcount)
(define-vertex-slot 'members)

;; for strongly connected components detection
(define-vertex-slot 'k)
(define-vertex-slot 'L)
(define-vertex-slot 'father)
(define-vertex-slot 'on-S)
(define-vertex-slot 'in-cycle)

;; for copying
(define-vertex-slot 'save-copy)

;; for listing
(define-vertex-slot 'index)
(define-vertex-slot 'indices)

) ;;; end eval-when


(defun print-vertex (v stream depth)
  (declare (ignore depth))
  (format stream "#v(~a :out ~a :in ~a)"
	  (vertex-name v)
	  (mapcar #'(lambda (e) (vertex-name (edge-to e)))
		  (vertex-edges v))
	  (mapcar #'(lambda (e) (vertex-name (edge-from e)))
		  (vertex-backedges v))
	  #+all (vertex-lcount v)))


(defun save-vertex-state (e)
  (push (vertex-other e) (vertex-save e))
  (setf (vertex-other e) nil))


(defun restore-vertex-state (e)
  (setf (vertex-other e) (pop (vertex-save e))))



;;; GRAPH structure
;;;

(defstruct graph 
  (vsize 0)
  (vset nil)
  (esize 0)
  (eset nil)
  (derived nil))


(defun copy-graph (g)
  (let ((new-v nil)
	(new-e nil)
	(new-vset nil)
	(new-eset nil))
    
    (dolist (v (graph-vset g))
      (setq new-v (copy-vertex v))
      (setf (vertex-save-copy v) new-v)
      (setf (vertex-other new-v) (copy-list (vertex-other new-v)))
      (setf (vertex-save new-v) (copy-list (vertex-save new-v)))
      (setf (vertex-edges new-v) nil)
      (setf (vertex-backedges new-v) nil)
      (push new-v new-vset))

    (dolist (e (graph-eset g))
      (setq new-e (copy-edge e))
      (setf (edge-other new-e) (copy-list (edge-other e)))
      (setf (edge-save new-e) (copy-list (edge-save e)))
      (setf (edge-to new-e) (vertex-save-copy (edge-to new-e)))
      (setf (edge-from new-e) (vertex-save-copy (edge-from new-e)))
      (push new-e (vertex-edges (edge-from new-e)))
      (push new-e (vertex-backedges (edge-to new-e)))
      (push new-e new-eset))

    (dolist (v (graph-vset g))
      (remf (vertex-other v) 'save-copy))
	
    (make-graph :vsize (graph-vsize g)
		:vset (nreverse new-vset)
		:esize (graph-esize g)
		:eset (nreverse new-eset))))


(defun extract-eset (vset)
  (sort	(apply #'append (mapcar #'(lambda (v) (vertex-edges v))
				vset))
	#'edge-less-p))


;;; DATA structure
;;;


(defstruct (data (:print-function print-data))
  (d1-b 0) (d2-b 0) (d3-b 0) (d4-b 0)
  (d1-n 0) (d2-n 0) (d3-n 0) (d4-n 0)
  (d1-d 0) (d2-d 0) (d3-d 0) (d4-d 0))


(defun print-data (d stream depth)
  (declare (ignore depth))
  (format stream "#d(~d/~d/~d ~d/~d/~d ~d/~d/~d ~d/~d/~d)"
	  (data-d1-b d) (data-d1-n d) (data-d1-d d)
	  (data-d2-b d) (data-d2-n d) (data-d2-d d)
	  (data-d3-b d) (data-d3-n d) (data-d3-d d)
	  (data-d4-b d) (data-d4-n d) (data-d4-d d)))


(defun list-to-data (l)
  (make-data :d1-b (first l)
	     :d2-b (second l)
	     :d3-b (third l)
	     :d4-b (fourth l)
	     :d1-n (fifth l)
	     :d2-n (sixth l)
	     :d3-n (seventh l)
	     :d4-n (eighth l)
	     :d1-d (ninth l)
	     :d2-d (tenth l)
	     :d3-d (nth 10 l)
	     :d4-d (nth 11 l)))

   
(defun add-data (d1 d2)
    (make-data 
     	       :d1-b (+ (data-d1-b d1) (data-d1-b d2))
	       :d2-b (+ (data-d2-b d1) (data-d2-b d2))
	       :d3-b (+ (data-d3-b d1) (data-d3-b d2))
	       :d4-b (+ (data-d4-b d1) (data-d4-b d2))

     	       :d1-n (+ (data-d1-n d1) (data-d1-n d2))
	       :d2-n (+ (data-d2-n d1) (data-d2-n d2))
	       :d3-n (+ (data-d3-n d1) (data-d3-n d2))
	       :d4-n (+ (data-d4-n d1) (data-d4-n d2))

     	       :d1-d (+ (data-d1-d d1) (data-d1-d d2))
	       :d2-d (+ (data-d2-d d1) (data-d2-d d2))
	       :d3-d (+ (data-d3-d d1) (data-d3-d d2))
	       :d4-d (+ (data-d4-d d1) (data-d4-d d2))))


(defvar zero-data (make-data))


(defun add-data-over (list)
  (reduce #'add-data list :initial-value zero-data))



;;; READING A RAW DATA FILE
;;;


(defvar *psymhash* (make-hash-table :size 3000 :test #'equal))
		  
(defun print-symbol (nameaddr)
  (if (not (search "@@@" nameaddr))
      nameaddr
      (let ((addr (read-from-string
		   (subseq nameaddr (+ (search "@@@" nameaddr) 3))))
	    (name (subseq nameaddr 0 (search "@@@" nameaddr))))

	(do* ((i 0 (+ i 1))
	      (pname name (concatenate 'string name "%" (format nil "~d" i)))
	      (hashval (gethash pname *psymhash*)
		       (gethash pname *psymhash*)))
	    ((null hashval)
	     ;; couldn't find this name in the table -- enter it
	     (setf (gethash pname *psymhash*) addr)
	     pname)
	  ;; if the names and addresses correspond - return pname
	  (if (= hashval addr)
	      (return pname))))))
  

(defun create-symbol (pair)
  (if (consp pair)
      (let ((pack (find-package (first pair)))
	    (sym (print-symbol (second pair))))
	(if pack
	    (intern sym pack)
	    (intern sym (make-package (first pair)))))
      (intern pair)))


(defun read-graph (instream)
  (let ((vset nil)
	(eset nil))

    ;; read in the vertices
    
    (do ((vdata (read instream nil nil) (read instream nil nil))
	 (i 0 (+ i 1)))
	((null vdata)
	 (setq vset (nreverse vset)))
      (let ((v (make-vertex :name (create-symbol (first vdata))
			    :number i)))
	(setf (vertex-lcount v) (list-to-data (second vdata)))
	(setf (vertex-scratch v) (third vdata))
	(setf (vertex-srefs v) 0)
	(push v vset)
	(setf (get (vertex-name v) 'vertex) v)))

    ;; create the backedges (must be done after reading in all vertices)
    
    (dolist (v vset)
      
      ;; make the backward edge list
      
      (do* ((parent-list (vertex-scratch v) (cdr parent-list))
	    (vto v)
	    (parent-name nil)
	    (count nil)
	    (elist nil))
	    
	  ((null parent-list)
	   (setf (vertex-backedges v) elist))
	
	(setq parent-name (create-symbol (first (car parent-list))))
	(setq count (second (car parent-list)))

	(let* ((vfrom (get parent-name 'vertex))
	       (e nil))
	  
	  ;; remove self-cycles
	  ;;
	  (if (equal vfrom vto)
	      (incf (vertex-srefs v))
	      (progn
		(setq e (make-edge :from vfrom :to vto))
		(setf (edge-count e) (list-to-data count))
		(push e eset)
		(push e elist))))))
    
    ;; make forward edge lists from the backward edge lists
    (dolist (v vset)
      (dolist (e (vertex-backedges v))
	(push e (vertex-edges (edge-from e)))))

    (make-graph :vsize (length vset)
		:vset vset
		:esize (length eset)
		:eset (extract-eset vset))))


;;; FINDING THE STRONGLY CONNECTED COMPONENTS
;;;


(defun no-unused-incident-edges-from (v)
  (reduce #'logical-and (mapcar #'(lambda (e)
				    (eql (edge-mark e) :used))
				(vertex-edges v))
	  :initial-value t))


(defun first-unused-edge (edges)
  (dolist (e edges (error "no first-unused-edge"))
    (if (eql (edge-mark e) :unused)
	(return e))))


(defun find-vertex-with-k=0 (vset)
  (dolist (v vset nil)
    (if (= (vertex-k v) 0)
	(return v))))

  
(defun scc (g)
  (let* ((vset (graph-vset g))
	 (eset (graph-eset g))
	 (scc-vset nil)
	 (i 0)
	 (v (first vset))
	 (u nil)
	 (e nil)
	 (S nil))			; stack of vertex numbers

    ;; save the old extra state
    
    (dolist (v vset)
      (save-vertex-state v))
    (dolist (e eset)
      (save-edge-state e))

    (dolist (e eset)
      (setf (edge-mark e) :unused))
    (dolist (v vset)
      (setf (vertex-father v) :undefined)
      (setf (vertex-k v) 0))
    
    #+bugs (mapcan #'(lambda (e) (setf (edge-mark e) :unused)) eset)
    #+bugs (mapcan #'(lambda (v)
		(setf (vertex-father v) :undefined)
		(setf (vertex-k v) 0))
	    vset)
    (prog ()
	  step2	  
	  (setq i (+ i 1))
	  (setf (vertex-k v) i)
	  (setf (vertex-L v) i)
	  (push v S)
	  (setf (vertex-on-S v) t)
	  step3
 (if *debug* (break "3"))	  
	  (if (no-unused-incident-edges-from v)
	      (go step7))
	  step4
 (if *debug* (break "4"))	  
	  (setq e (first-unused-edge (vertex-edges v)))
	  (setq u (edge-to e))
	  (setf (edge-mark e) :used)
	  (when (= (vertex-k u) 0)
	    (setf (vertex-father u) v)
	    (setq v u)
	    (go step2))
	  step5
 (if *debug* (break "5"))	  
	  (if (> (vertex-k u) (vertex-k v))
	    (go step3))
	  (if (not (vertex-on-S u))
	    (go step3))
	  step6
	  (assert (and (< (vertex-k u) (vertex-k v))
		       (vertex-on-S u)))
 (if *debug* (break "6"))	  
	  (setf (vertex-L v) (min (vertex-L v) (vertex-k u)))
	  (go step3)
	  step7
 (if *debug* (break "7"))
	  (when (= (vertex-L v) (vertex-k v))
	    (do ((comp nil))
		((eql (car S) v)
		 (push (pop S) comp)
		 (setf (vertex-on-S (car comp)) nil)
		 (push comp scc-vset))
	      (push (pop S) comp)
	      (setf (vertex-on-S (car comp)) nil)))
	  step8
	  (when (not (eql (vertex-father v) :undefined))
 (if *debug* (break "8"))	  
	    (setf (vertex-L (vertex-father v))
		  (min (vertex-L (vertex-father v))
		       (vertex-L v)))
	    (setq v (vertex-father v))
	    (go step3))
	  step9
	  (assert (eql (vertex-father v) :undefined))
	  (let ((u (find-vertex-with-k=0 vset)))
	    (when u
	      (setq v u)
	      (go step2))
	    (if S
		(push S scc-vset))
	    
	    ;; restore the extra state
	    ;; 
    
	    (dolist (v vset)
	      (restore-vertex-state v))
	    (dolist (e eset)
	      (restore-edge-state e))
	    
	    (return scc-vset)))))


;;; MERGING THE STRONGLY CONNECTED COMPONENTS
;;;


(defun same-edge (from-v to-v elist)
  (dolist (e elist nil)
    (if (and (eql (edge-to e) to-v)
	     (eql (edge-from e) from-v))
	(return e))))

   
(defun merge-scc (g)
  (let* ((newg (copy-graph g))
	 (vset nil)
	 (eset nil)
	 (added-eset nil)
	 (cycle-count 0))
    (setf (graph-derived newg) g)

    ;; create vertices correspondind to each of the cycles
    
    (dolist (cc (scc newg))
      (if (/= (length cc) 1)
	  (let* ((newv (make-vertex
			:name (intern
			       (format nil "<cycle ~d>" (incf cycle-count)))
			:number (+ cycle-count (graph-vsize g) -1)
			:edges nil
			:backedges nil)))
	    (push newv vset)
	    (setf (vertex-lcount newv) (make-data))
	    (setf (vertex-members newv) cc)
	    (setf (vertex-srefs newv) 0)
	    (dolist (v cc)
	      (push v vset)
	      (setf (vertex-in-cycle v) newv)
	      (setf (vertex-lcount newv)
		    (add-data (vertex-lcount newv) (vertex-lcount v)))))
	  (push (car cc) vset)))
    
    ;; look at all the edges and create a new edge set (add edges to cycle)

    (dolist (e (graph-eset newg))

      ;; cases --
      ;; 1. either vertex is part of a cycle
      ;; 2. neither vertex is in a cycle

      (let ((fv (vertex-in-cycle (edge-from e)))
	    (tv (vertex-in-cycle (edge-to e))))
	(when (or fv tv)
	  (when (not fv) (setq fv (edge-from e)))
	  (when (not tv) (setq tv (edge-to e)))
	  (if (not (eql fv tv))
	    (let ((newe (same-edge fv tv added-eset)))
	      (if newe
		  (setf (edge-count newe)
			(add-data (edge-count newe) (edge-count e)))
		  (progn
		    (setq newe (make-edge :from fv :to tv))
		    (setf (edge-count newe) (edge-count e))
		    (push newe added-eset))))
	    (incf (vertex-srefs fv))))
	(push e eset)))

    (setq eset (sort (append eset added-eset) #'edge-less-p))
    
    (dolist (v vset)
      (setf (vertex-edges v) nil)
      (setf (vertex-backedges v) nil))

    (dolist (e eset)
      (let ((vto (edge-to e))
	    (vfrom (edge-from e)))
	(push e (vertex-edges vfrom))
	(push e (vertex-backedges vto))))
	  
    (setf (graph-vsize newg) (length vset))
    (setf (graph-vset newg) (nreverse vset))
    (setf (graph-eset newg) eset)
    (setf (graph-esize newg) (length eset))
    newg))
    

;;; TOPOLOGICAL SORT OF A DAG
;;;

(defun tsort (g)
  (let ((vset (graph-vset g))
	(leaves nil))
    (dolist (v vset)
      (setf (vertex-scratch v) (length (vertex-edges v)))
      (if (= (vertex-scratch v) 0)
	  (push v leaves)))
    (do* ((l leaves)
	  (leaf (pop l) (pop l))
	  (g-sort nil))
	((null leaf)
	 (setf (graph-vset g) g-sort)
	 g)
      (push leaf g-sort)
      (dolist (e (vertex-backedges leaf))
	(let ((v (edge-from e)))
	  (setf (vertex-scratch v) (- (vertex-scratch v) 1))
	  (if (= (vertex-scratch v) 0)
	      (push v l)))))))
	  

;;; PRINTING THE MPROF DATA
;;;


(defun cycle-p (v)
  (vertex-members v))


(defun augment-graph-data (g)
  (dolist (v (graph-vset g))
    (augment-data (vertex-lcount v)))
  (dolist (e (graph-eset g))
    (augment-data (edge-count e)))
  g)


(defconstant CONS-SIZE 8)
(defconstant FLOAT-SIZE 8)


(defun augment-data (d)
  ;; d1-b for Lisp data is the number of CONS bytes
  (when (= (data-d1-b d) 0)
    (setf (data-d1-b d) (* CONS-SIZE (data-d1-n d))))

  ;; d2-b for Lisp data is the number of FLOAT bytes
  (when (= (data-d2-b d) 0)
    (setf (data-d2-b d) (* FLOAT-SIZE (data-d2-n d)))))

	
(defun sum-calls (d)
  (+ (data-d1-n d) (data-d2-n d) (data-d3-n d) (data-d4-n d)))


(defun sum-data (d)
  (+ (data-d1-b d) (data-d2-b d) (data-d3-b d) (data-d4-b d)))


(defun sum-kept (d)
  (+ (data-d1-d d) (data-d2-d d) (data-d3-d d) (data-d4-d d)))


(defun filter-cycle-edges (elist)
  (remove-if #'(lambda (e)
		 (or (cycle-p (edge-to e))
		     (cycle-p (edge-from e))))
	     elist))

(defun filter-in-cycle-edges (elist)
  (remove-if #'(lambda (e)
		 (or (vertex-in-cycle (edge-to e))
		     (vertex-in-cycle (edge-from e))))
	     elist))

  
(defun sum-calls-over-edges (elist)
  (let ((sum 0))
    (dolist (e (filter-in-cycle-edges elist) sum)
      (incf sum (sum-calls (edge-count e))))))


(defun sum-data-over-edges (elist)
  (let ((sum 0))
    (dolist (e (filter-in-cycle-edges elist) sum)
      (incf sum (sum-data (edge-count e))))))


(defun sum-vertex-bytes (v)
  (if (not (vertex-in-cycle v))
      (+ (sum-data (vertex-lcount v))
	 (sum-data-over-edges (vertex-edges v)))
      (sum-vertex-bytes (vertex-in-cycle v))))


(defun sum-vertex-bytes1 (v)
  (if (not (vertex-in-cycle v))
      (+ (sum-data (vertex-lcount v))
	 (sum-data-over-edges (vertex-edges v)))
      (sum-data (vertex-lcount v))))


(defun sum-vertex-incoming-calls (v)
  (if (not (vertex-in-cycle v))
      (sum-calls-over-edges (vertex-backedges v))
      (sum-vertex-incoming-calls (vertex-in-cycle v))))


(defun sum-vertex-outgoing-calls (v)
  (if (not (vertex-in-cycle v))
      (sum-calls-over-edges (vertex-edges v))
      (sum-vertex-outgoing-calls (vertex-in-cycle v))))


(defun printable-rank (r value)
  (if (and (< r 6) (> value 0))
      r
      " "))


(defun percent (x y)
  (if (= y 0)
      0
      (/ (* 100 x) y)))


(defun truncate-or-blank (n)
  (multiple-value-bind (intpart fraction)
      		       (truncate n)
    (cond ((and (= intpart 0)
		(= fraction 0))
	   "")
	  ((= intpart 0)
	   ".")
	  ((= intpart 100)
	   "**")
	  (t
	   (format nil "~2d" intpart)))))


(defun data-kept-string (data divisor)
  (if (= divisor 0)
      ""
      (format nil " ~2@a ~2@a ~2@a ~2@a"
	      (truncate-or-blank (percent (data-d1-d data) divisor))
	      (truncate-or-blank (percent (data-d2-d data) divisor))
	      (truncate-or-blank (percent (data-d3-d data) divisor))
	      (truncate-or-blank (percent (data-d4-d data) divisor)))))


(defun relative-kept-string (num den)
  (data-kept-string num (sum-kept den)))


(defun kept-fraction-string (data)
  (data-kept-string data (sum-kept data)))

  
(defun data-type-string (data divisor)
  (if (= divisor 0)
      ""
      (format nil " ~2@a ~2@a ~2@a ~2@a"
	      (truncate-or-blank (percent (data-d1-b data) divisor))
	      (truncate-or-blank (percent (data-d2-b data) divisor))
	      (truncate-or-blank (percent (data-d3-b data) divisor))
	      (truncate-or-blank (percent (data-d4-b data) divisor)))))

  
(defun relative-type-string (num den)
  (data-type-string num (sum-data den)))


(defun type-fraction-string (data)
  (data-type-string data (sum-data data)))


(defun vertex-name-string (v)
  (format nil "~a [~d]~a"
	  (vertex-name v)
	  (vertex-index v)
	  (if (vertex-in-cycle v)
	      (format nil " in ~a" (vertex-name (vertex-in-cycle v)))
	      "")))



;;; PRINTING 



(defconstant c-template0
  "~%~25,,,'-a--s--m--l--x--~15,,,'-a--s--m--l--x~15,,,'-a~%~%")
(defconstant c-template1
  "  ~8@a ~12@a |~12a | ~12@a |~12a | ~12@a   ~a~%")

(defconstant lisp-template0
  "~%~25,,,'-a--c--f--s--o--~30,,,'-a~%~%")
(defconstant lisp-template1
  "  ~8@a ~12@a |~12a | ~12@a   ~a~%")

(defconstant template2
  "~5a~7@a~9@a ~5a |~12a |~12a |~8@a~9a~a~%")

(defconstant lisp-template3
  "~%~28,,,'-a---c--f--s--o----c--f--s--o--~8,,,'-a~%~%")
(defconstant c-template3
  "~%~28,,,'-a---s--m--l--x----s--m--l--x--~8,,,'-a~%~%")


(defvar template3)


(defun separator (outstream)
  (format outstream template3 "" ""))



;;; PROLOG AND ALLOCATION BIN TABLE
;;;


(defun mprof-prologue (outstream data-file)
  (let* ((instream (open data-file :direction :input))
	 (stats (read instream)))
    (format outstream
"--c~2,'0d+--v2.0+--m~d+--+--+--+--+--+--+--+--+ MPROF +--+--+--+--+--+--+--s~d+--f~d+--d~d+--l~d+~%~%~%"
	    (truncate (percent (nth 3 stats) (nth 4 stats)))
	    (nth 2 stats)
            (nth 8 stats)
            (nth 5 stats)
            (nth 6 stats)
            (nth 7 stats))
    instream))


(defconstant MP-NUM-BINS 1025)


(defun type< (t1 t2)
  (cond ((< (second t1) (second t2))
	 t)
	((> (second t1) (second t2))
	 nil)
	(t
	 (string< (first t1) (first t2)))))


(defun print-type-list (outstream types compar-f binsize)
  (do ((l types (cdr l))
       (i 0))
      ((or (null l)
	   (> (second (car l)) binsize))
       l)
    (when (funcall compar-f (second (car l)) binsize)
      (if (and (> i 0)
	       (= (mod i 3) 0))
	  (format outstream "~%~64a" ""))
      (format outstream "~12a " (first (car l)))
      (incf i))))


(defconstant abin-template "~11@a~10d~10d ~4a~10d~10d ~4a   ")
(defconstant abin-titles-template "~11@a~10@a~10@a ~4a~10@a~10@a ~4a    ~8a~%")


(defun percent-string (i n)
  (multiple-value-bind (intpart fraction)
      		       (truncate (percent i n))
    (cond ((and (= intpart 0)
		(= fraction 0))
	   "")
	  ((= intpart 0)
	   "(.)")
	  ((= intpart 100)
	   "(**)")
	  (t
	   (format nil "(~d)" intpart)))))


(defun bin-table (instream outstream)
  (let* ((types (read instream))
	 (alloc-bins (read instream))
	 (free-bins (read instream))
	 (big-alloc-count (nth (- MP-NUM-BINS 2) alloc-bins))
	 (big-free-count (nth (- MP-NUM-BINS 2) free-bins))
	 (big-alloc-bytes (nth (- MP-NUM-BINS 1) alloc-bins))
	 (big-free-bytes (nth (- MP-NUM-BINS 1) free-bins))
	 (alloc-count 0)
	 (alloc-bytes 0)
	 (free-count 0)
	 (free-bytes 0)
	 (other-alloc-count 0)
	 (other-alloc-bytes 0)
	 (other-free-count 0)
	 (other-free-bytes 0))

    (setf (cdr (nthcdr (- MP-NUM-BINS 3) alloc-bins)) nil)
    (setf (cdr (nthcdr (- MP-NUM-BINS 3) free-bins)) nil)
    (setq types (sort types #'type<))
    
    (setq  alloc-count (max (+ big-alloc-count (reduce #'+ alloc-bins)) 1))
    (setq  free-count (+ big-free-count (reduce #'+ free-bins)))
    
    (do* ((abins alloc-bins (cdr abins))
	  (fbins free-bins (cdr fbins))
	  (abin (car abins) (car abins))
	  (fbin (car fbins) (car fbins))
	  (i 0 (+ i 1)))
	((null abin))
      (incf alloc-bytes (* i abin))
      (incf free-bytes (* i fbin)))

    (incf alloc-bytes big-alloc-bytes)
    (incf free-bytes big-free-bytes)
	 
    (format outstream "--------- Allocation Bins with possible Types ------------~%~%")
    (format outstream abin-titles-template
	    "size:" "allocs" "bytes" "(%)" "frees" "kept" "(%)" "types")
    (format outstream "~%")
    
    (do* ((abins alloc-bins (cdr abins))
	  (fbins free-bins (cdr fbins))
	  (abin (car abins) (car abins))
	  (fbin (car fbins) (car fbins))
	  (i 0 (+ i 1))
	  (types types))
	((null abin)
	 (format outstream abin-template
		 "> 1024"
		 big-alloc-count
		 big-alloc-bytes
		 (percent-string big-alloc-bytes alloc-bytes)
		 big-free-count
		 (- big-alloc-bytes big-free-bytes)
		 (percent-string (- big-alloc-bytes big-free-bytes)
				(- alloc-bytes free-bytes)))
	 (print-type-list outstream types #'> i)
	 (format outstream "~%~%")
	 (when (or (> other-alloc-count 0)
		   (> other-free-count 0))
	   (format outstream abin-template
		   "other bins"
		   other-alloc-count
		   other-alloc-bytes
		   (percent-string other-alloc-bytes alloc-bytes)
		   other-free-count
		   (- other-alloc-bytes other-free-bytes)
		   (percent-string (- other-alloc-bytes other-free-bytes)
				   (- alloc-bytes free-bytes)))
	   (format outstream "~%"))
	 (format outstream abin-template
		 "<TOTAL>"
		 alloc-count
		 alloc-bytes
		 ""
		 free-count
		 (- alloc-bytes free-bytes)
		 "")
	 (format outstream "~%"))
      (if (or (and (eql *language* :verbose)
		   (or (> abin 0)
		       (> fbin 0)
		       (and types (= (second (car types)) i))))
	      (and (eql *language* :normal)
		   (or (and (> abin 0)
			    (> i 100))
		       (> (/ abin alloc-count) 1/500)
		       (and types
			    (> (/ abin alloc-count) 1/500)
			    (= (second (car types)) i))))
	      (and (eql *language* :terse)
		   (or (> (/ abin alloc-count) 1/50)
		       (and types
			    (> (/ abin alloc-count) 1/100)
			    (= (second (car types)) i)))))
	  
	  ;; print the bin out
	  (progn 
	    (format outstream
		    abin-template
		    i
		    abin
		    (* i abin)
		    (percent-string (* i abin) alloc-bytes)
		    fbin
		    (- (* i abin) (* i fbin))
		    (percent-string (- (* i abin) (* i fbin)) (- alloc-bytes free-bytes)))
	    (setq types (print-type-list outstream types #'= i))
	    (format outstream "~%"))
	  
	  ;; else, just add the bin to the non-printed ones
	  (progn
	    (incf other-alloc-count abin)
	    (incf other-free-count fbin)
	    (incf other-alloc-bytes (* abin i))
	    (incf other-free-bytes (* fbin i)))
	  ))
    (format outstream "~|~%~%")))


;;; LEAK TABLE

(defconstant leak-titles-template1 "~10@a ~4a~10@a~10@a ~4a~10@a~10@a ~4a    ~8a~%")
(defconstant leak-titles-template2 "~10@a ~4a~10@a~10@a ~4a    ~8a~%")
(defconstant leak-template1 "~10d ~4a~10d~10d ~4a~10d~10d ~4a   ")
(defconstant leak-template2 "~10d ~4a~10d~10d ~4a   ")

(defun seq-string< (s1 s2)
  (dotimes (i (length s1) nil)
    (cond ((string< (elt s1 i) (elt s2 i))
	   (return i))
	  ((string> (elt s1 i) (elt s2 i))
	   (return nil)))))

  
(defun leak-table (instream outstream)
  (let ((leaks (read instream))
	(total-allocs 0)
	(bytes-alloced 0)
	(total-frees 0)
	(bytes-freed 0))
    (when (and leaks (not (eql *leak-level* :none)))
      (format outstream "--------- Partial Dynamic Call Paths for Memory Leaks ------------~%~%")

      
      (dolist (l leaks)
	(let* ((counts (second l)))
	  (incf total-allocs (first counts))
	  (incf bytes-alloced (second counts))
	  (incf total-frees (third counts))
	  (incf bytes-freed (fourth counts))))

      (format outstream "Total bytes not freed: ~d~%~%"
	      (- bytes-alloced bytes-freed))
      
      (if (> total-frees 0)
	  (format outstream leak-titles-template1
		  "kept bytes" "(%)" "allocs" "bytes" "(%)" "frees" "bytes" "(%)" "path")
	  (format outstream leak-titles-template2
		  "kept bytes" "(%)" "allocs" "bytes" "(%)" "path"))

      (format outstream "~%")

      ;; either show leaks with offsets or merge all offsets into one
      ;;
      (case *leak-level*
	(:show
	 ;; merge functions
	 (progn
	   (setq leaks (mapcar #'(lambda (l) (list (reverse (mapcar #'car (first l)))
							    (second l)))
			       leaks))

	   (setq leaks (sort leaks #'seq-string< :key #'first))
	   (let ((new-leaks nil)
		 (hold '(0 0 0 0)))
	     (if (> (length leaks) 1)
		 (dotimes (i (- (length leaks) 1))
		   (if (equal (first (elt leaks i))
			      (first (elt leaks (+ i 1))))
		       ;; if same, just update the stored value
		       (setq hold (mapcar #'+ hold (second (elt leaks i))))
		       ;; if different, push the current hold and update hold
		       (progn
			 (setq hold (mapcar #'+ hold (second (elt leaks i))))
			 (push (list (first (elt leaks i)) hold) new-leaks)
			 (setq hold '(0 0 0 0))))))
	     ;; add in the data from the last element
	     (setq hold (mapcar #'+ hold (second (car (last leaks)))))
	     (push (list (first (car (last leaks))) hold) new-leaks)
	     (setq leaks new-leaks))))
	(:offsets
	 ;; reverse the pathnames
	 (setq leaks (mapcar #'(lambda (l) (list (reverse (first l))
						 (second l)))
			     leaks))))

      (setq leaks (sort leaks #'>
			:key #'(lambda (d)
				 (let* ((data (second d))
					(alloced-bytes (second data))
					(freed-bytes (fourth data)))
				   (- alloced-bytes freed-bytes)))))
      (dolist (l leaks)
	(let* ((funs (first l))
	       (counts (second l)))
	  (case *language*
	    (:terse
	     (if (< (/ (second counts) bytes-alloced) 0.01)
		 (return)))
	    (:normal
	     (if (< (/ (second counts) bytes-alloced) 0.005)
		 (return))))
	  (if (> total-frees 0)
	      (format outstream leak-template1
		      (- (second counts) (fourth counts))
		      (percent-string (- (second counts) (fourth counts))
				      (- bytes-alloced bytes-freed))
		      (first counts)
		      (second counts)
		      (percent-string (second counts) bytes-alloced)
		      (third counts)
		      (fourth counts)
		      (percent-string (fourth counts) bytes-freed))
	      (format outstream leak-template2
		      (- (second counts) (fourth counts))
		      (percent-string (- (second counts) (fourth counts))
				      (- bytes-alloced bytes-freed))
		      (first counts)
		      (second counts)
		      (percent-string (second counts) bytes-alloced)))
	  (case *leak-level*
	    (:show
	     (if (or (string= (first funs) "")
		     (string= (first funs) "main"))
		 (format outstream "|| ")
		 (format outstream "..."))
	     (dolist (f funs)
	       (if (string/= f "")
		   (format outstream "> ~a " f))))
	    (:offsets
	     (if (or (string= (first (first funs)) "")
		     (string= (first (first funs)) "main"))
		 (format outstream "|| ")
		 (format outstream "..."))
	     (dolist (f funs)
	       (if (string/= (first f) "")
		   (format outstream "> ~a+~d " (first f) (second f))))))
	  (format outstream "~%")))
      (format outstream "~|~%~%"))))
      

;;; PRINT STATIC AND DYNAMIC CALL GRAPHS
;;;


(defun mprof (data-file &key outfile (type :c))
  (let* ((outstream (if outfile
			(open outfile
			      :direction :output
			      :if-exists :supersede)
			t))
	 (g (augment-graph-data
	     (let* ((instream (mprof-prologue outstream data-file))
		    (g nil))
	       (if (eql type :c)
		   (bin-table instream outstream))
	       (if (eql type :c)
		   (leak-table instream outstream))
	       (setq g (read-graph instream))
	       (close instream)
	       g)))
	 
	 (newg (merge-scc g))
	 (cumul-bytes 0)
	 (total-bytes 0)
	 (total-calls 0)
	 (total-kept 0)
	 (total-data (make-data))
	 (old-vset (graph-vset g))
	 (vset (graph-vset newg))
	 (ordered-vset nil)
	 (template3 (if (eql type :c)
			c-template3
			lisp-template3))
	 (old-ordered-vset (sort (copy-list old-vset)
				 #'(lambda (x y)
				     (> (sum-data (vertex-lcount x))
					(sum-data (vertex-lcount y)))))))

    (dolist (v old-vset)
      (setq total-data (add-data total-data (vertex-lcount v))))

    (setq total-bytes (sum-data total-data))
    (setq total-calls (sum-calls total-data))
    (setq total-kept (sum-kept total-data))
    
    (format outstream "---------  Direct Allocation Table ------------~%~%")

    (cond ((eql type :c)
	   (format outstream c-template1
		   " % mem" "bytes" " % mem(size)" "bytes kept" "  % all kept" "calls" "name")
	   (format outstream c-template0 "" "" "")
	   (format outstream c-template1
		   "-----"
		   total-bytes
		   (type-fraction-string total-data)
		   (format nil "~d" total-kept)
		   (format nil "~a" (kept-fraction-string total-data))
		   total-calls
		   "<TOTAL>"))
	  (t
	   (format outstream lisp-template1
		   " % mem" "bytes" " % mem(size)" "calls" "name")
	   (format outstream lisp-template0 "" "")
	   (format outstream lisp-template1
		   "-----"
		   total-bytes
		   (type-fraction-string total-data)
		   total-calls
		   "<TOTAL>")))

    (format outstream "~%")
    
    ;; print out the allocation recorded in the leaf nodes
    
    (dolist (v old-ordered-vset)
      (let* ((vdata (vertex-lcount v))
	     (nbytes (sum-data vdata)))
	(when (= nbytes 0)
	  (return))
	(incf cumul-bytes nbytes)
	(if (eql type :c)
	    (format outstream c-template1
		    (format nil "~8,1f" (percent nbytes total-bytes))
		    (format nil "~d" nbytes)
		    (relative-type-string vdata total-data)
		    (format nil "~d" (sum-kept vdata))
		    (format nil "~a" (data-kept-string vdata total-kept))
		    (format nil "~d" (sum-calls vdata))
		    (format nil "~a" (vertex-name v)))
	    (format outstream lisp-template1
		    (format nil "~8,1f" (percent nbytes total-bytes))
		    (format nil "~d" nbytes)
		    (relative-type-string vdata total-data)
		    (format nil "~d" (sum-calls vdata))
		    (format nil "~a" (vertex-name v)))
	    )))

    ;; check allocation consistency from children to parents
    
    (dolist (v (reverse vset))
      ;; because the vertex list is sorted, we always encounter a node
      ;; before any of the callers of that node and after all the nodes
      ;; it calls
      (let* ((clist (vertex-edges v))
	     (plist (vertex-backedges v))
	     (csum (sum-data-over-edges clist))
	     (psum (sum-data-over-edges plist)))

	(when (and (not (vertex-in-cycle v))
		   (vertex-backedges v)
		   (/= psum (+ csum (sum-data (vertex-lcount v)))))
	  #+mpdebug (cerror "" "parent and children do not agree on data allocation")
	  (format outstream "~a --> ~d != ~d+~d (~d)~%"
		  (vertex-name v)
		  psum
		  csum
		  (sum-data (vertex-lcount v))
		  (+ csum (sum-data (vertex-lcount v))))
	  )
	#+mpdebug(format outstream "~a --> ~d ?= ~d+~d (~d)~%"
			 (vertex-name v)
			 psum
			 csum
			 (sum-data (vertex-lcount v))
			 (+ csum (sum-data (vertex-lcount v))))))


    (setq ordered-vset (sort (copy-list vset)
			     #'(lambda (x y) (> (sum-vertex-bytes1 x)
						(sum-vertex-bytes1 y)))))
    (let ((i 0))
      (dolist (v ordered-vset)
	(setf (vertex-index v) i)
	(incf i)))

    (print-dynamic-header outstream)

    (dolist (v ordered-vset)
      (cond ((cycle-p v)
	     (print-cycle-information v outstream total-bytes)
	     (separator outstream)
	     (print-normal-vertex v outstream total-bytes))
	    ((vertex-in-cycle v)
	     (print-vertex-in-cycle v outstream total-bytes))
	    (t
	     (print-normal-vertex v outstream total-bytes)))
      (separator outstream))
    (if (not (eql outstream t))
	(close outstream))
    ))

(defun print-dynamic-header (outstream)
  ;; print out the memory profile
  (format outstream "~|~%~%")
  (format outstream template2
	  "" "self" "" "" "     /ances" "     /ances" "called" "/total "
	  "    ancestors")
  (format outstream template2
	  "index" "+  " "self" "(%)" " size-func" " frac" "called" "/recur"
	  "name [index]")
  (format outstream template2
	  "" "desc" "" "" "     \\desc" "     \\desc" "called" "/total"
	  "    descendents")
    
  (separator outstream))


(defun print-self-line (outstream v total-bytes cyclep)

  (let* ((plist (vertex-backedges v))
	 (clist (vertex-edges v))
	 (pcallsum (sum-calls-over-edges plist))
	 (cbytesum (sum-data-over-edges clist))
	 (self-bytesum (sum-data (vertex-lcount v)))
	 (fbytesum (+ self-bytesum cbytesum)))

    (format outstream template2
	    (format nil "[~d]" (vertex-index v))
	    (format nil "~7,1f"(percent fbytesum total-bytes))
	    self-bytesum
	    (percent-string self-bytesum fbytesum)
	    (type-fraction-string (vertex-lcount v))
	    " -----------"
	    pcallsum
	    (if (or (null (vertex-srefs v))
		    (= (vertex-srefs v) 0))
		""
		(format nil "+~d" (vertex-srefs v)))
	    (if cyclep
		(format nil "~a as a whole" (vertex-name-string v))
		(vertex-name-string v)))))


(defun print-normal-vertex (v outstream total-bytes)
  
  (let* ((plist (vertex-backedges v))
	 (filt-plist (if (cycle-p v)
			 plist
			 (filter-cycle-edges (vertex-backedges v))))
	 (clist (vertex-edges v))
	 (filt-clist (if (cycle-p v)
			 clist
			 (filter-cycle-edges (vertex-edges v))))
	 (pbytesum (sum-data-over-edges plist))
	 (cbytesum (sum-data-over-edges clist))
	 (all-pdata (add-data-over (mapcar #'(lambda (e) (edge-count e))
					   filt-plist)))
	 (all-cdata (add-data-over (mapcar #'(lambda (e) (edge-count e))
					   filt-clist))))
    
    ;; parent-listings
	
    (when (> (length filt-plist) 1)
      (format outstream template2
	      "" "all"
	      (sum-data all-pdata)
	      ""
	      (type-fraction-string all-pdata)
	      "" "" "" ""))
									      
    (dolist (e filt-plist)
      (let* ((edata (edge-count e))
	     (pcallsum (sum-vertex-outgoing-calls (edge-from e)))
	     (ecallsum (sum-calls edata))
	     (ebytesum (sum-data edata)))
	(format outstream template2
		"" ""
		ebytesum
		(percent-string ebytesum pbytesum)
		(type-fraction-string edata)
		(relative-type-string edata all-pdata)
		ecallsum
		(format nil "/~d" pcallsum)
		(format nil "    ~a" (vertex-name-string (edge-from e))))))
	
    ;; self listing

    (print-self-line outstream v total-bytes (not :cyclep))


    ;; children listings
	
    (dolist (e filt-clist)
      (let* ((edata (edge-count e))
	     (ccallsum (sum-vertex-incoming-calls (edge-to e)))
	     (ecallsum (sum-calls edata))
	     (ebytesum (sum-data edata)))
	    
	(format outstream template2
		"" ""
		ebytesum
		(percent-string ebytesum cbytesum)
		(type-fraction-string edata)
		(relative-type-string edata all-cdata)
		ecallsum
		(format nil "/~d" ccallsum)
		(format nil "    ~a" (vertex-name-string (edge-to e))))))

    (when (> (length filt-clist) 1)
      (format outstream template2
	      "" "all"
	      (sum-data all-cdata)
	      ""
	      (type-fraction-string all-cdata)
	      "" "" "" ""))))


(defun print-cycle-information (v outstream total-bytes)
  (let* ((members (vertex-members v))
	 (mdata (vertex-lcount v))
	 (mbytesum (sum-data mdata)))
    
    ;; self listing

    (print-self-line outstream v total-bytes :cyclep)
    
    ;; members listing

    (dolist (v members)
      (let* ((vdata (vertex-lcount v))
	     (vbytesum (sum-data vdata)))
	    
	(format outstream template2
		""
		""
		vbytesum
		(percent-string vbytesum (if (= 0 mbytesum)
					     1
					     mbytesum))
		(type-fraction-string vdata)
		(relative-type-string vdata mdata)
		""
		""
		(format nil "    ~a" (vertex-name-string v)))))))


(defun print-vertex-in-cycle (v outstream total-bytes)
  
  (let* ((clist (filter-cycle-edges (vertex-edges v)))
	 (plist (filter-cycle-edges (vertex-backedges v)))
	 (pcallsum 0)
	 (self-bytesum (sum-data (vertex-lcount v))))

    ;; parent-listings
	
    (dolist (e plist)
      (let* ((edata (edge-count e))
		 (ecallsum (sum-calls edata)))
	    (incf pcallsum ecallsum)
	    (format outstream template2
		    "" "" "" "" "" ""
		    ecallsum
		    ""
		    (format nil "    ~a"
			    (vertex-name-string (edge-from e))))))
      
	
    ;; self listing
       
    (format outstream template2
	    (format nil "[~d]" (vertex-index v))
	    (format nil "~7,1f"(percent self-bytesum total-bytes))
	    self-bytesum
	    ""
	    (type-fraction-string (vertex-lcount v))
	    " -----------"
	    pcallsum
	    (if (or (null (vertex-srefs v))
		    (= (vertex-srefs v) 0))
		""
		(format nil "+~d" (vertex-srefs v)))
	    (vertex-name-string v))

    ;; children listings
	
    (dolist (e clist)
      (let* ((ecallsum (sum-calls (edge-count e))))
	    
	(format outstream template2
		"" "" "" "" "" ""
		ecallsum
		""
		(format nil "    ~a"
			(vertex-name-string (edge-to e))))))))
