/*-
 * See the file LICENSE for redistribution information.
PGNO_INVALID  *
 * Copyright (c) 2000
 *	Sleepycat Software.  All rights reserved.
 */

#include "db_config.h"

#ifndef lint
static const char revid[] = "$Id: db_cam.c,v 1.1.1.1.2.2 2000/02/08 00:43:46 noriko Exp $";
#endif /* not lint */

#ifndef NO_SYSTEM_INCLUDES
#include <sys/types.h>

#include <errno.h>
#endif

#include "db_int.h"
#include "db_page.h"
#include "db_shash.h"
#include "lock.h"
#include "btree.h"
#include "hash.h"
#include "qam.h"
#include "db_ext.h"

static int __db_c_cleanup __P((DBC *, DBC *, int));
static int __db_c_idup __P((DBC *, DBC **, u_int32_t));

#define	CDB_LOCKING_INIT(dbp, dbc)					\
	/*								\
	 * If we are running CDB, this had better be either a write	\
	 * cursor or an immediate writer.  If it's a regular writer,	\
	 * that means we have an IWRITE lock and we need to upgrade	\
	 * it to a write lock.						\
	 */								\
	if (CDB_LOCKING((dbp)->dbenv)) {				\
		if (!F_ISSET(dbc, DBC_WRITECURSOR | DBC_WRITER)) {	\
			DB_ASSERT(0);					\
			return (EINVAL);				\
		}							\
		if (F_ISSET(dbc, DBC_WRITECURSOR) &&			\
		    (ret = lock_get((dbp)->dbenv, (dbc)->locker,	\
		    DB_LOCK_UPGRADE, &(dbc)->lock_dbt, DB_LOCK_WRITE,	\
		    &(dbc)->mylock)) != 0)				\
			return (ret);					\
	}
#define	CDB_LOCKING_DONE(dbp, dbc)					\
	/* Release the upgraded lock. */				\
	if (F_ISSET(dbc, DBC_WRITECURSOR))				\
		(void)__lock_downgrade(					\
		    (dbp)->dbenv, &(dbc)->mylock, DB_LOCK_IWRITE, 0);

/* 
 * Queue doesn't preserve pgno across cursor gets--the important
 * piece of state is the recno.
 * 
 *  ((QUEUE_CURSOR *)(dbc)->internal)->recno != RECNO_OOB :		\ 
 * 
 */
#define	IS_INITIALIZED(dbc)	((dbc)->dbtype == DB_QUEUE ? 		\
    ((QUEUE_CURSOR *)((dbc)->internal))->recno != RECNO_OOB :	\
    (dbc)->pgno != PGNO_INVALID)

/*
 * __db_c_close --
 *	Close the cursor.
 *
 * PUBLIC: int __db_c_close __P((DBC *));
 */
int
__db_c_close(dbc)
	DBC *dbc;
{
	DB *dbp;
	int ret, t_ret;

	dbp = dbc->dbp;
	ret = 0;

	PANIC_CHECK(dbp->dbenv);

	/*
	 * If the cursor is already closed we have a serious problem, and we
	 * assume that the cursor isn't on the active queue.  Don't do any of
	 * the remaining cursor close processing.
	 */
	if (!F_ISSET(dbc, DBC_ACTIVE)) {
		DB_ASSERT(0);
		return (EINVAL);
	}

	/*
	 * Remove the cursor(s) from the active queue.  We may be closing two
	 * cursors at once here, a top-level one and a lower-level, off-page
	 * duplicate one.  The acess-method specific cursor close routine must
	 * close both of them in a single call.
	 *
	 * !!!
	 * Cursors must be removed from the active queue before calling the
	 * access specific cursor close routine, btree depends on having that
	 * order of operations.  It must also happen before any action that
	 * can fail and cause __db_c_close to return an error, or else calls
	 * here from __db_close may loop indefinitely.
	 */
	MUTEX_THREAD_LOCK(dbp->mutexp);

	if (dbc->opd != NULL) {
		F_CLR(dbc->opd, DBC_ACTIVE);
		TAILQ_REMOVE(&dbp->active_queue, dbc->opd, links);
	}
	F_CLR(dbc, DBC_ACTIVE);
	TAILQ_REMOVE(&dbp->active_queue, dbc, links);

	MUTEX_THREAD_UNLOCK(dbp->mutexp);

	/* Call the access specific cursor close routine. */
	if ((t_ret =
	    dbc->c_am_close(dbc, PGNO_INVALID, NULL)) != 0 && ret == 0)
		ret = t_ret;

	/*
	 * Release the lock after calling the access method specific close
	 * routine, a Btree cursor may have had pending deletes.
	 */
	if (CDB_LOCKING(dbc->dbp->dbenv) && dbc->mylock.off != LOCK_INVALID) {
		if ((t_ret =
		    lock_put(dbc->dbp->dbenv, &dbc->mylock)) != 0 && ret == 0)
			ret = t_ret;
		dbc->mylock.off = LOCK_INVALID;
	}

	/* Move the cursor(s) to the free queue. */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	if (dbc->opd != NULL) {
		TAILQ_INSERT_TAIL(&dbp->free_queue, dbc->opd, links);
		dbc->opd = NULL;
	}
	TAILQ_INSERT_TAIL(&dbp->free_queue, dbc, links);
	MUTEX_THREAD_UNLOCK(dbp->mutexp);

	return (ret);
}

/*
 * __db_c_destroy --
 *	Destroy the cursor, called after DBC->c_close.
 *
 * PUBLIC: int __db_c_destroy __P((DBC *));
 */
int
__db_c_destroy(dbc)
	DBC *dbc;
{
	DB *dbp;
	int ret;

	dbp = dbc->dbp;

	/* Remove the cursor from the free queue. */
	MUTEX_THREAD_LOCK(dbp->mutexp);
	TAILQ_REMOVE(&dbp->free_queue, dbc, links);
	MUTEX_THREAD_UNLOCK(dbp->mutexp);

	/* Call the access specific cursor destroy routine. */
	ret = dbc->c_am_destroy == NULL ? 0 : dbc->c_am_destroy(dbc);

	/* Free up allocated memory. */
	if (dbc->rkey.data != NULL)
		__os_free(dbc->rkey.data, dbc->rkey.ulen);
	if (dbc->rdata.data != NULL)
		__os_free(dbc->rdata.data, dbc->rdata.ulen);
	__os_free(dbc, sizeof(*dbc));

	return (ret);
}

/*
 * __db_c_count --
 *	Return a count of duplicate data items.
 *
 * PUBLIC: int __db_c_count __P((DBC *, db_recno_t *, u_int32_t));
 */
int
__db_c_count(dbc, recnop, flags)
	DBC *dbc;
	db_recno_t *recnop;
	u_int32_t flags;
{
	DB *dbp;
	int ret;

	/*
	 * Cursor Cleanup Note:
	 * All of the cursors passed to the underlying access methods by this
	 * routine are not duplicated and will not be cleaned up on return.
	 * So, pages/locks that the cursor references must be resolved by the
	 * underlying functions.
	 */
	dbp = dbc->dbp;

	PANIC_CHECK(dbp->dbenv);

	/* Check for invalid flags. */
	if ((ret = __db_ccountchk(dbp, flags, IS_INITIALIZED(dbc))) != 0)
		return (ret);

	switch (dbc->dbtype) {
	case DB_QUEUE:
	case DB_RECNO:
		*recnop = 1;
		break;
	case DB_HASH:
		if (dbc->opd == NULL) {
			if ((ret = __ham_c_count(dbc, recnop)) != 0)
				return (ret);
			break;
		}
		/* FALLTHROUGH */
	case DB_BTREE:
		if ((ret = __bam_c_count(dbc, recnop)) != 0)
			return (ret);
		break;
	default:
		DB_ASSERT(0);
		return (EINVAL);
	}
	return (0);
}

/*
 * __db_c_del --
 *	Delete using a cursor.
 *
 * PUBLIC: int __db_c_del __P((DBC *, u_int32_t));
 */
int
__db_c_del(dbc, flags)
	DBC *dbc;
	u_int32_t flags;
{
	DB *dbp;
	int ret;

	/*
	 * Cursor Cleanup Note:
	 * All of the cursors passed to the underlying access methods by this
	 * routine are not duplicated and will not be cleaned up on return.
	 * So, pages/locks that the cursor references must be resolved by the
	 * underlying functions.
	 */
	dbp = dbc->dbp;

	PANIC_CHECK(dbp->dbenv);

	/* Check for invalid flags. */
	if ((ret = __db_cdelchk(dbp, flags,
	    F_ISSET(dbp, DB_AM_RDONLY), IS_INITIALIZED(dbc))) != 0)
		return (ret);

	DEBUG_LWRITE(dbc, dbc->txn, "db_c_del", NULL, NULL, flags);

	CDB_LOCKING_INIT(dbp, dbc);

	/*
	 * Off-page duplicate trees are locked in the primary tree, that is,
	 * we acquire a write lock in the primary tree and no locks in the
	 * off-page dup tree.  If the del operation is done in an off-page
	 * duplicate tree, call the primary cursor's upgrade routine first.
	 */
	if (dbc->opd == NULL)
		ret = dbc->c_am_del(dbc);
	else
		if ((ret = dbc->c_am_writelock(dbc)) == 0)
			ret = dbc->opd->c_am_del(dbc->opd);

	CDB_LOCKING_DONE(dbp, dbc);

	return (ret);
}

/*
 * __db_c_dup --
 *	Duplicate a cursor
 *
 * PUBLIC: int __db_c_dup __P((DBC *, DBC **, u_int32_t));
 */
int
__db_c_dup(dbc_orig, dbcp, flags)
	DBC *dbc_orig;
	DBC **dbcp;
	u_int32_t flags;
{
	DB *dbp;
	DBC *dbc_n, *dbc_nopd;
	int ret;

	dbp = dbc_orig->dbp;
	dbc_n = dbc_nopd = NULL;

	PANIC_CHECK(dbp->dbenv);

	/*
	 * We can never have two write cursors open in CDB, so do not
	 * allow duplication of a write cursor.
	 */
	if (!F_ISSET(dbc_orig, DBC_IDUP) && flags != DB_POSITIONI &&
	    F_ISSET(dbc_orig, DBC_WRITER | DBC_WRITECURSOR))
		return (EINVAL);
	F_CLR(dbc_orig, DBC_IDUP);

	/* Allocate a new cursor and initialize it. */
	if ((ret = __db_c_idup(dbc_orig, &dbc_n, flags)) != 0)
		goto err;
	*dbcp = dbc_n;

	/*
	 * If the cursor references an off-page duplicate tree, allocate a
	 * new cursor for that tree and initialize it.
	 */
	if (dbc_orig->opd != NULL) {
		if ((ret = __db_c_idup(dbc_orig->opd, &dbc_nopd, flags)) != 0)
			goto err;
		dbc_n->opd = dbc_nopd;
	}

	return (0);

err:	if (dbc_n != NULL)
		(void)dbc_n->c_close(dbc_n);
	if (dbc_nopd != NULL)
		(void)dbc_nopd->c_close(dbc_nopd);

	return (ret);
}

/*
 * __db_c_idup --
 *	Internal version of __db_c_dup.
 */
static int
__db_c_idup(dbc_orig, dbcp, flags)
	DBC *dbc_orig, **dbcp;
	u_int32_t flags;
{
	DB *dbp;
	DBC *dbc_n;
	int ret;

	dbp = dbc_orig->dbp;

	if ((ret = __db_icursor(dbp, dbc_orig->txn, dbc_orig->dbtype,
	    dbc_orig->root, F_ISSET(dbc_orig, DBC_OPD), &dbc_n)) != 0)
		return (ret);

	dbc_n->locker = dbc_orig->locker;

	/* If the user wants the cursor positioned, do it here.  */
	if (flags == DB_POSITION || flags == DB_POSITIONI) {
		dbc_n->root = dbc_orig->root;
		dbc_n->pgno = dbc_orig->pgno;
		dbc_n->indx = dbc_orig->indx;
		dbc_n->flags = dbc_orig->flags;

		switch (dbc_orig->dbtype) {
		case DB_QUEUE:
			if ((ret = __qam_c_dup(dbc_orig, dbc_n)) != 0)
				goto err;
			break;
		case DB_BTREE:
		case DB_RECNO:
			if ((ret = __bam_c_dup(dbc_orig, dbc_n)) != 0)
				goto err;
			break;
		case DB_HASH:
			if ((ret = __ham_c_dup(dbc_orig, dbc_n)) != 0)
				goto err;
			break;
		default:
			ret = EINVAL;
			goto err;
		}
	}

	*dbcp = dbc_n;
	return (0);

err:	(void)dbc_n->c_close(dbc_n);
	return (ret);
}

/*
 * __db_c_get --
 *	Get using a cursor.
 *
 * PUBLIC: int __db_c_get __P((DBC *, DBT *, DBT *, u_int32_t));
 */
int
__db_c_get(dbc_arg, key, data, flags)
	DBC *dbc_arg;
	DBT *key, *data;
	u_int32_t flags;
{
	DB *dbp;
	DBC *dbc, *dbc_n;
	db_pgno_t pgno;
	u_int32_t tmp_flags, tmp_rmw;
	u_int8_t type;
	int ret, t_ret;

	/*
	 * Cursor Cleanup Note:
	 * All of the cursors passed to the underlying access methods by this
	 * routine are duplicated cursors.  On return, any referenced pages
	 * will be discarded, and, if the cursor is not intended to be used
	 * again, the close function will be called.  So, pages/locks that
	 * the cursor references do not need to be resolved by the underlying
	 * functions.
	 */
	dbp = dbc_arg->dbp;
	dbc_n = NULL;

	PANIC_CHECK(dbp->dbenv);

	/* Check for invalid flags. */
	if ((ret =
	    __db_cgetchk(dbp, key, data, flags, IS_INITIALIZED(dbc_arg))) != 0)
		return (ret);

	/* Clear OR'd in additional bits so we can check for flag equality. */
	tmp_rmw = LF_ISSET(DB_RMW);
	LF_CLR(DB_RMW);

	DEBUG_LREAD(dbc_arg, dbc_arg->txn, "db_c_get",
	    flags == DB_SET || flags == DB_SET_RANGE ? key : NULL, NULL, flags);

	/*
	 * Return a cursor's record number.  It has nothing to do with the
	 * cursor get code except that it was put into the interface.
	 */
	if (flags == DB_GET_RECNO)
		return (__bam_c_rget(dbc_arg, data, flags | tmp_rmw));

	/*
	 * If we have an off-page duplicates cursor, and the operation applies
	 * to it, perform the operation.  Duplicate the cursor and call the
	 * underlying function.
	 *
	 * Off-page duplicate trees are locked in the primary tree, that is,
	 * we acquire a write lock in the primary tree and no locks in the
	 * off-page dup tree.  If the DB_RMW flag was specified and the get
	 * operation is done in an off-page duplicate tree, call the primary
	 * cursor's upgrade routine first.
	 */
	if (dbc_arg->opd != NULL &&
	    (flags == DB_CURRENT || flags == DB_GET_BOTHC || 
	    flags == DB_NEXT || flags == DB_NEXT_DUP || flags == DB_PREV)) {
		if (tmp_rmw && (ret = dbc_arg->c_am_writelock(dbc_arg)) != 0)
			return (ret);
		if ((ret = __db_c_dup(dbc_arg, &dbc_n, DB_POSITIONI)) != 0)
			return (ret);
		switch (ret = dbc_n->opd->c_am_get(
		    dbc_arg->opd, dbc_n->opd, key, data, flags, NULL)) {
		case 0:
			goto done;
		case DB_NOTFOUND:
			/*
			 * Translate DB_NOTFOUND failures for the DB_NEXT and
			 * DB_PREV operations into a subsequent operation on
			 * the parent cursor.
			 */
			if (flags == DB_NEXT || flags == DB_PREV) {
				if ((ret = dbc_n->c_close(dbc_n)) != 0)
					goto err;
				dbc_n = NULL;
				break;
			}
			goto err;
		default:
			goto err;
		}
	}

	/*
	 * Perform an operation on the main cursor.  Duplicate the cursor,
	 * upgrade the lock as required, and call the underlying function.
	 */
	tmp_flags = flags == DB_CURRENT || flags == DB_GET_BOTHC || 
	    flags == DB_NEXT || flags == DB_NEXT_DUP || 
	    flags == DB_NEXT_NODUP || flags == DB_PREV ? DB_POSITIONI : 0;
	if ((ret = __db_c_idup(dbc_arg, &dbc_n, tmp_flags)) != 0)
		goto err;
	if (tmp_rmw)
		F_SET(dbc_n, DBC_RMW);
	pgno = PGNO_INVALID;
	ret = dbc_n->c_am_get(dbc_arg, dbc_n, key, data, flags, &pgno);
	if (tmp_rmw)
		F_CLR(dbc_n, DBC_RMW);
	if (ret != 0)
		goto err;

	/*
	 * We may be referencing a new off-page duplicates tree.  Acquire
	 * a new cursor and call the underlying function.
	 */
	if (pgno != PGNO_INVALID) {
		if ((ret = __db_icursor(dbp, dbc_arg->txn,
		    dbp->dup_compare == NULL ? DB_RECNO : DB_BTREE,
		    pgno, 1, &dbc_n->opd)) != 0)
			goto err;

		switch (flags) {
		case DB_FIRST:
		case DB_NEXT:
		case DB_NEXT_NODUP:
		case DB_SET:
		case DB_SET_RECNO:
		case DB_SET_RANGE:
			tmp_flags = DB_FIRST;
			break;
		case DB_LAST:
		case DB_PREV:
			tmp_flags = DB_LAST;
			break;
		case DB_GET_BOTH:
			tmp_flags = DB_GET_BOTH;
			break;
		case DB_GET_BOTHC:
			tmp_flags = DB_GET_BOTHC;
			break;
		default:
			DB_ASSERT(0);

			ret = EINVAL;
			goto err;
		}
		if ((ret = dbc_n->opd->c_am_get(
		    NULL, dbc_n->opd, key, data, tmp_flags, NULL)) != 0)
			goto err;
	}

done:	/*
	 * Return a key/data item.  The only exception is that we don't return
	 * a key if the user already gave us one, that is, if the DB_SET flag
	 * was set.  The DB_SET flag is necessary.  In a Btree, the user's key
	 * doesn't have to be the same as the key stored the tree, depending on
	 * the magic performed by the comparison function.  As we may not have
	 * done any key-oriented operation here, the page reference may not be
	 * valid.  Fill it in as necessary.  We don't have to worry about any
	 * locks, the cursor must already be holding appropriate locks.
	 *
	 * XXX
	 * If not a Btree and DB_SET_RANGE is set, we shouldn't return a key
	 * either, should we?
	 */
	dbc = dbc_n;
	if (dbc->dbtype != DB_RECNO && flags != DB_SET) {
		if (dbc->page == NULL && (ret =
		    memp_fget(dbp->mpf, &dbc->pgno, 0, &dbc->page)) != 0)
			goto err;

		if ((ret = __db_ret(dbp, dbc->page, dbc->indx,
		    key, &dbc_n->rkey.data, &dbc_n->rkey.ulen)) != 0)
			goto err;
	}
	dbc = dbc->opd != NULL ? dbc->opd : dbc;
	type = TYPE(dbc->page);
	if (!F_ISSET(data, DB_DBT_ISSET)) 
		ret = __db_ret(dbp, dbc->page, dbc->indx + 
		    (type == P_LBTREE || type == P_HASH ? O_INDX : 0),
		    data, &dbc_arg->rdata.data, &dbc_arg->rdata.ulen);

err:	/* Don't pass DB_DBT_ISSET back to application level, error or no. */
	F_CLR(data, DB_DBT_ISSET);

	/* Cleanup and cursor resolution. */
	if ((t_ret = __db_c_cleanup(dbc_arg, dbc_n, ret)) != 0 && ret == 0)
		ret = t_ret;
	
	return (ret);
}

/*
 * __db_c_put --
 *	Put using a cursor.
 *
 * PUBLIC: int __db_c_put __P((DBC *, DBT *, DBT *, u_int32_t));
 */
int
__db_c_put(dbc_arg, key, data, flags)
	DBC *dbc_arg;
	DBT *key, *data;
	u_int32_t flags;
{
	DB *dbp;
	DBC *dbc_n;
	db_pgno_t pgno;
	u_int32_t tmp_flags;
	int ret, t_ret;

	/*
	 * Cursor Cleanup Note:
	 * All of the cursors passed to the underlying access methods by this
	 * routine are duplicated cursors.  On return, any referenced pages
	 * will be discarded, and, if the cursor is not intended to be used
	 * again, the close function will be called.  So, pages/locks that
	 * the cursor references do not need to be resolved by the underlying
	 * functions.
	 */
	dbp = dbc_arg->dbp;
	dbc_n = NULL;

	PANIC_CHECK(dbp->dbenv);

	/* Check for invalid flags. */
	if ((ret = __db_cputchk(dbp, key, data, flags,
	    F_ISSET(dbp, DB_AM_RDONLY), IS_INITIALIZED(dbc_arg))) != 0)
		return (ret);

	DEBUG_LWRITE(dbc_arg, dbc_arg->txn, "db_c_put",
	    flags == DB_KEYFIRST || flags == DB_KEYLAST ? key : NULL,
	    data, flags);

	CDB_LOCKING_INIT(dbp, dbc_arg);

	/*
	 * If we have an off-page duplicates cursor, and the operation applies
	 * to it, perform the operation.  Duplicate the cursor and call the
	 * underlying function.
	 *
	 * Off-page duplicate trees are locked in the primary tree, that is,
	 * we acquire a write lock in the primary tree and no locks in the
	 * off-page dup tree.  If the put operation is done in an off-page
	 * duplicate tree, call the primary cursor's upgrade routine first.
	 */
	if (dbc_arg->opd != NULL &&
	    (flags == DB_AFTER || flags == DB_BEFORE || flags == DB_CURRENT)) {
		if ((ret = dbc_arg->c_am_writelock(dbc_arg)) != 0)
			return (ret);
		if ((ret = __db_c_dup(dbc_arg, &dbc_n, DB_POSITIONI)) != 0)
			goto err;
		if ((ret = dbc_n->opd->c_am_put(
		    dbc_n->opd, key, data, flags, NULL)) != 0)
			goto err;
		goto done;
	}

	/*
	 * Perform an operation on the main cursor.  Duplicate the cursor,
	 * and call the underlying function.
	 *
	 * XXX: MARGO
	 *
	tmp_flags = flags == DB_AFTER ||
	    flags == DB_BEFORE || flags == DB_CURRENT ? DB_POSITIONI : 0;
	 */
	tmp_flags = DB_POSITIONI;
	if ((ret = __db_c_idup(dbc_arg, &dbc_n, tmp_flags)) != 0)
		goto err;
	pgno = PGNO_INVALID;
	if ((ret = dbc_n->c_am_put(dbc_n, key, data, flags, &pgno)) != 0)
		goto err;

	/*
	 * We may be referencing a new off-page duplicates tree.  Acquire
	 * a new cursor and call the underlying function.
	 */
	if (pgno != PGNO_INVALID) {
		if ((ret = __db_icursor(dbp, dbc_arg->txn,
		    dbp->dup_compare == NULL ? DB_RECNO : DB_BTREE,
		    pgno, 1, &dbc_n->opd)) != 0)
			goto err;

		if ((ret = dbc_n->opd->c_am_put(
		    dbc_n->opd, key, data, flags, NULL)) != 0)
			goto err;
	}

done:
err:	/* Cleanup and cursor resolution. */
	if ((t_ret = __db_c_cleanup(dbc_arg, dbc_n, ret)) != 0 && ret == 0)
		ret = t_ret;

	CDB_LOCKING_DONE(dbp, dbc_arg);

	return (ret);
}

/*
 * __db_c_cleanup --
 *	Clean up duplicate cursors.
 */
static int
__db_c_cleanup(dbc, dbc_n, failed)
	DBC *dbc, *dbc_n;
	int failed;
{
	DB *dbp;
	DBC t_dbc;
	int ret, t_ret;

	dbp = dbc->dbp;
	ret = 0;

	/* Discard any pages we're holding. */
	if (dbc->page != NULL) {
		if ((t_ret =
		    memp_fput(dbp->mpf, dbc->page, 0)) != 0 && ret == 0)
			ret = t_ret;
		dbc->page = NULL;
	}
	if (dbc->opd != NULL && dbc->opd->page != NULL) {
		if ((t_ret =
		    memp_fput(dbp->mpf, dbc->opd->page, 0)) != 0 && ret == 0)
			ret = t_ret;
		 dbc->opd->page = NULL;
	}
	if (dbc_n->page != NULL) {
		if ((t_ret =
		    memp_fput(dbp->mpf, dbc_n->page, 0)) != 0 && ret == 0)
			ret = t_ret;
		dbc_n->page = NULL;
	}
	if (dbc_n->opd != NULL && dbc_n->opd->page != NULL) {
		if ((t_ret =
		    memp_fput(dbp->mpf, dbc_n->opd->page, 0)) != 0 && ret == 0)
			ret = t_ret;
		dbc_n->opd->page = NULL;
	}

	/*
	 * If we didn't failed originally or just now, swap the interesting
	 * contents of the old and new cursors.  Close the cursor we don't
	 * care about anymore.
	 *
	 * The following close can fail, but we only expect DB_LOCK_DEADLOCK
	 * failures.  This violates our "the cursor is unchanged on error"
	 * semantics, but since all you can do with a DB_LOCK_DEADLOCK failure
	 * is close the cursor, I believe that's OK.
	 *
	 * XXX
	 * There's no way to recover from failure to close the old cursor.
	 * All we can do is move to the new position and return an error.
	 *
	 * XXX
	 * We might want to consider adding a flag to the cursor, so that any
	 * subsequent operations other than close just return an error?
	 */
	if (!failed && ret == 0) {
		t_dbc.opd = dbc->opd;
		t_dbc.root = dbc->root;
		t_dbc.pgno = dbc->pgno;
		t_dbc.indx = dbc->indx;
		t_dbc.rkey = dbc->rkey;
		t_dbc.rdata = dbc->rdata;
		t_dbc.internal = dbc->internal;

		dbc->opd = dbc_n->opd;
		dbc->root = dbc_n->root;
		dbc->pgno = dbc_n->pgno;
		dbc->indx = dbc_n->indx;
		dbc->rkey = dbc_n->rkey;
		dbc->rdata = dbc_n->rdata;
		dbc->internal = dbc_n->internal;

		dbc_n->opd = t_dbc.opd;
		dbc_n->root = t_dbc.root;
		dbc_n->pgno = t_dbc.pgno;
		dbc_n->indx = t_dbc.indx;
		dbc_n->rkey = t_dbc.rkey;
		dbc_n->rdata = t_dbc.rdata;
		dbc_n->internal = t_dbc.internal;
	}

	if ((t_ret = dbc_n->c_close(dbc_n)) != 0 && ret == 0)
		ret = t_ret;

	return (ret);
}
