use 5.14.0;
use strict;
use warnings;

package Dist::Zilla::Plugin::Stenciller::HtmlExamples {

    our $VERSION = '0.0100'; # VERSION
    # ABSTRACT: Create Html example files from text files parsed with Stenciller

    use Moose;
    with 'Dist::Zilla::Role::FileGatherer';
    use Stenciller;
    use Types::Stenciller -types;
    use Path::Tiny;
    use Dist::Zilla::File::InMemory;

    has source_directory => (
        is => 'ro',
        isa => Dir,
        coerce => 1,
        default => 'examples/source',
    );
    has file_pattern => (
        is => 'ro',
        isa => Str,
        default => '.+\.stencil',
    );
    has output_directory => (
        is => 'ro',
        isa => Path,
        coerce => 1,
        default => 'examples',
    );
    has template_file => (
        is => 'ro',
        isa => AbsFile,
        lazy => 1,
        coerce => 1,
        default => sub { shift->source_directory->child('template.html')->absolute },
    );

    sub gather_files {
        my $self = shift;

        my $template = $self->template_file->slurp_utf8;
        my @source_files = $self->source_directory->children(qr{^@{[ $self->file_pattern ]}$});

        $self->log('Generating from stencils');

        foreach my $file (@source_files) {
            my $contents = Stenciller->new(filepath => $file->stringify)->transform(plugin_name => 'ToHtmlPreBlock', transform_args => { require_in_extra => { key => 'is_html_example', value => 1, default => 1 }});
            my $all_contents = $template;
            $all_contents =~ s{\[STENCILS\]}{$contents};
            my $new_filename = $file->basename(qr/\.[^.]+$/) . '.html';
            $self->log("Generated $new_filename");

            my $generated_file = Dist::Zilla::File::InMemory->new(
                name => path($self->output_directory, $new_filename)->stringify,
                content => $all_contents,
            );
            $self->add_file($generated_file);

        }
    }

}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Stenciller::HtmlExamples - Create Html example files from text files parsed with Stenciller



=begin HTML

<p><img src="https://img.shields.io/badge/perl-5.14-brightgreen.svg" alt="Requires Perl 5.14" /> <a href="https://travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Stenciller-HtmlExamples"><img src="https://api.travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Stenciller-HtmlExamples.svg?branch=master" alt="Travis status" /></a></p>

=end HTML


=begin markdown

![Requires Perl 5.14](https://img.shields.io/badge/perl-5.14-brightgreen.svg) [![Travis status](https://api.travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Stenciller-HtmlExamples.svg?branch=master)](https://travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Stenciller-HtmlExamples)

=end markdown

=head1 VERSION

Version 0.0100, released 2015-11-23.

=head1 SYNOPSIS

    ; in dist.ini
    ; these are the defaults
    [Stenciller::HtmlExamples]
    source_directory = examples/source
    output_directory = examples
    template_file = examples/source/template.html
    file_pattern = .+\.stencil

=head1 DESCRIPTION

Dist::Zilla::Plugin::Stenciller::HtmlExamples uses L<Stenciller> and L<Stenciller::Plugin::ToHtmlPreBlock> to turn
stencil files in C<source_directory> (that matches the C<file_pattern>) into
html example files in C<output_directory> by applying the C<template_file>.

This L<Dist::Zilla> plugin does the C<FileGatherer> role.

=head1 ATTRIBUTES

=head2 source_directory

Path to where the stencil files are.

=head2 output_directory

Path to where the generated files are saved.

=head2 file_pattern

This is put inside a regular expression (with start and end anchors) to find stencil files in the C<source_directory>. The output files
will have the same basename, but the suffix is replaced by C<html>. 

=head2 template_file

The template file is an ordinary html file, with one exception: The first occurence of C<[STENCILS]> will be replaced with the
string returned from L<Stenciller::Plugin::ToHtmlPreBlock>. The template file is applied to each stencil file, so the number of generated example files is equal
to the number of stencil files.

=head1 SEE ALSO

=over 4

=item *

L<Stenciller>

=item *

L<Stenciller::Plugin::ToHtmlPreBlock>

=back

=head1 SOURCE

L<https://github.com/Csson/p5-Dist-Zilla-Plugin-Stenciller-HtmlExamples>

=head1 HOMEPAGE

L<https://metacpan.org/release/Dist-Zilla-Plugin-Stenciller-HtmlExamples>

=head1 AUTHOR

Erik Carlsson <info@code301.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Erik Carlsson.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
