package Dist::Zilla::Plugin::Test::Version;
use 5.006;
use strict;
use warnings;
use namespace::autoclean;

our $VERSION = '1.06'; # VERSION

use Moose;
extends 'Dist::Zilla::Plugin::InlineFiles';
with
  'Dist::Zilla::Role::TextTemplate',
  'Dist::Zilla::Role::PrereqSource',
  'Dist::Zilla::Role::FileFinderUser' => {
    method           => 'found_files',
    finder_arg_names => [ 'finder' ],
    default_finders  => [],
  },
;


around add_file => sub {
  my ( $orig, $self, $file ) = @_;

  my @filenames;
  my $use_finder = 0;

  if(@{ $self->finder } > 0) {
    @filenames = map { Path::Class::File->new($_->name)->relative('.')->stringify }
      grep { not ($_->can('is_bytes') and $_->is_bytes) }
      @{ $self->found_files };
    $use_finder = 1;
  }

  $self->$orig(
    Dist::Zilla::File::InMemory->new({
      name    => $file->name,
      content => $self->fill_in_string(
        $file->content,
        {
          name           => __PACKAGE__,
          version        => __PACKAGE__->VERSION
            || 'bootstrapped version'
            ,
          is_strict      => \$self->is_strict,
          has_version    => \$self->has_version,
          multiple       => \$self->multiple,
          filename_match => join(", ", @{ $self->filename_match }),
          filenames      => [ sort @filenames ],
          use_finder     => $use_finder,
        },
      ),
    })
  );
};

sub register_prereqs {
  my $self = shift;
  $self->zilla->register_prereqs({
      type  => 'requires',
      phase => 'develop',
    },
    'Test::More'    => 0,
    'Test::Version' => @{ $self->filename_match } > 0 ? '2.00' : '1',
  );
  return;
}

has is_strict => (
  is => 'ro',
  isa => 'Bool',
  lazy => 1,
  default => sub { 0 },
);

has has_version => (
  is => 'ro',
  isa => 'Bool',
  lazy => 1,
  default => sub { 1 },
);

has multiple => (
  is      => 'ro',
  isa     => 'Bool',
  lazy    => 1,
  default => sub { 0 },
);

has filename_match => (
  is      => 'ro',
  isa     => 'ArrayRef',
  lazy    => 1,
  default => sub { [] },
);

around mvp_multivalue_args => sub {
  my($orig, $self) = @_;
  return ($self->$orig, 'filename_match');
};

__PACKAGE__->meta->make_immutable;
1;

# ABSTRACT: Author Test::Version tests

=pod

=head1 NAME

Dist::Zilla::Plugin::Test::Version - Author Test::Version tests

=head1 VERSION

version 1.06

=head1 SYNOPSIS

in C<dist.ini>

  [Test::Version]
  is_strict   = 0
  has_version = 1

=head1 DESCRIPTION

This module will add a L<Test::Version> test as a author test to your module.

=head1 ATTRIBUTES

=head2 is_strict

set L<Test::Version is_strict|Test::Version/is_strict>

=head2 has_version

set L<Test::Version has_version|Test::Version/has_version>

=head2 filename_match

set L<Test::Version filename_match|Test::Version/filename_match>

=head2 multiple

set L<Test::Version multiple|Test::Version/multiple>

=head2 finder

This is the name of a L<Dist::Zilla::Role::FileFinder> for finding files to check.
If this is specified then C<version_ok> will be used for each file that matches,
otherwise C<version_all_ok> will be used, and the file discovery will be handled
by L<Test::Version>.

=head1 METHODS

=head2 register_prereqs

Register L<Test::Version> as an a development prerequisite.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website
https://github.com/plicease/dist-zilla-plugin-test-version/issues

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHORS

=over 4

=item *

Graham Ollis <plicease@cpan.org>

=item *

Caleb Cushing <xenoterracide@gmail.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Caleb Cushing <xenoterracide@gmail.com>.

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut

__DATA__
__[ xt/author/test-version.t ]__
use strict;
use warnings;
use Test::More;

# generated by {{ $name }} {{ $version }}
use Test::Version;

my @imports = qw( {{ $use_finder ? 'version_ok' : 'version_all_ok' }} );

my $params = {
    is_strict      => {{ $is_strict }},
    has_version    => {{ $has_version }},
    multiple       => {{ $multiple }},
{{ $filename_match ? "    filename_match => [ $filename_match ]," : '' }}
};

push @imports, $params
    if version->parse( $Test::Version::VERSION ) >= version->parse('1.002');

Test::Version->import(@imports);

{{
  $use_finder
  ? @filenames > 0 ? join("\n", map { s/'/\\'/g; "version_ok('$_');" } @filenames) : '# finder found no files'
  : 'version_all_ok;';
}}
done_testing;
