// fingerd.c -- Finger daemon service for Windows NT
// Copyright 1995 Dean Troyer (troyer@indirect.com)
// All rights reserved.
//
// Implements a finger server as a Windows NT service.
//
// This software has been developed by Dean Troyer and other contributors
// ('The Authors') with the intention that it's source code be available 
// to the user/developer community, not only to maintain it's correct 
// operation, but to allow for enhancements by others to contribute to the 
// overall improvement of the software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. All advertising materials mentioning features or use of this software
//    must display the following acknowledgement:
//      This product includes software developed by Dean Troyer and
//      other contributors.
// 4. The names of the Authors may not be used to endorse or promote 
//    products derived from this software without specific prior written 
//    permission.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//
// Contributors:
//   Winsock & finger protocol inspired by TXTSRV.C by Lee Murach
//
// 0.4  02Feb95     dlt     Initial release

#include "config.h"
#include "fingerd.h"
#include "process.h"

// Global data

extern char LMDomain[];                 // LanMan Domain Name
SOCKET skListen;                        // Wait for connection requests
short FingerPort;                       // Finger service port number
char RequestBuf[REQLEN + 1];            // Buffer for client request string

// NT handles

HANDLE hLT = INVALID_HANDLE_VALUE;      // ListenThread handle
HANDLE hServiceEvent = NULL;

// Local prototypes

void Blocking(SOCKET s);
char *WSErrorString(UINT err);

// ServiceInstall - Perform service installation tasks
//
// Parameters:
//   none
//
// Return:
//   none
//
// Comments:
//   Install the required registry keys with reasonable default values;
//   reasonable being whatever is defined in config.h

VOID ServiceInstall()
{
    HKEY    RegKey;
    DWORD   Result;
    DWORD   EntryType;
    char    szBuf[32];
    
    // Set up registry entries
    RegCreateKey(HKEY_LOCAL_MACHINE, Finger_Reg_Key, &RegKey);

    // Lan Manager/NT Domain
    Result = RegQueryValueEx(RegKey, Reg_LMDomain, NULL, &EntryType, NULL, NULL);
    if (Result != ERROR_SUCCESS || 
       ((EntryType != REG_SZ) && (EntryType != REG_EXPAND_SZ))) {
        strcpy(szBuf, Reg_LMDomain_Def);
        RegSetValueEx(RegKey, Reg_LMDomain, 0, REG_SZ, (CONST BYTE *)szBuf, 
                      strlen(szBuf) + 1);
    }

    RegCloseKey(RegKey);
}  // ServiceInstall


// ServiceRemove - Perform service removal tasks
//
// Parameters:
//   none
//
// Return value:
//   none
//
// Comments:
//   It might be useful to provide for an option to remove the registry 
//   entries, and anything else that might be left behind.

VOID ServiceRemove()
{
    // Only delete registry keys if we're nuking the service
}  // ServiceRemove


// ServiceStart - Initialize and start the service functions
//
// Parameters:
//   dwArgc   - number of command line arguments
//   lpszArgv - array of command line arguments
//
// Return value:
//   none
//
// Comments:
//   Initializes the service data space, and spawns a thread to listen 
//   for connections.  Waits for ServiceStop to signal shutdown.

VOID ServiceStart (DWORD dwArgc, LPTSTR *lpszArgv)
{
    WSADATA WSAData;                    // Windows Sockets info
    int iWSAErr = -1;
    HKEY RegKey;
    unsigned long BufLen;
    char szBuf[32];
    DWORD EntryType, Result;
    DWORD dwThreadId, dwParam = 1;
    DWORD WaitResult;
    
    // Service initialization

    _tzset();                           // Set up our timezone

    // Startup WinSock
    if (iWSAErr = WSAStartup(WSVERSION, &WSAData)) {
        LogWSError();
        goto cleanup;
    }

    // report the status to the service control manager.
    if (!ReportStatusToSCMgr(
        SERVICE_START_PENDING,          // service state
        NO_ERROR,                       // exit code
        3000))                          // wait hint
        goto cleanup;

    // Create the ConnectEvent object. The process thread signals
    // this event when it has connected the incomming request.
    hServiceEvent = CreateEvent(
        NULL,                           // no security attributes
        TRUE,                           // manual reset event
        FALSE,                          // not-signalled
        NULL);                          // no name

    if ( hServiceEvent == NULL)
        goto cleanup;

    // Get finger service port number
//    if ((se = getservbyname(FINGER_SERVICE, NULL)) == NULL)
        FingerPort = DEF_FINGER_PORT;
//    else
//        FingerPort = se->s_port;

    // Read registry data
    
    RegCreateKey(HKEY_LOCAL_MACHINE, Finger_Reg_Key, &RegKey);

    BufLen = sizeof(szBuf);
    strcpy(LMDomain, Reg_LMDomain_Def);
    Result = RegQueryValueEx(RegKey, Reg_LMDomain, 0, &EntryType, szBuf, &BufLen);
    if (Result == ERROR_SUCCESS && ((EntryType == REG_SZ) || (EntryType == REG_EXPAND_SZ))) {
        strcpy(LMDomain, szBuf);
    }

    RegCloseKey(RegKey);

    // report the status to the service control manager.
    if (!ReportStatusToSCMgr(
        SERVICE_RUNNING,                // service state
        NO_ERROR,                       // exit code
        0))                             // wait hint
        goto cleanup;

    // Service is now running, perform work until shutdown

    // Create listen thread
    hLT = CreateThread(NULL,                        // no security attributes
                       0,                           // use default stack size
                       (LPTHREAD_START_ROUTINE)ListenThreadProc, // thread function
                       &dwParam,                    // argument to thread function
                       0,                           // use default creation flags
                       &dwThreadId                  // returns the thread identifier
                       );
    WaitResult = WaitForSingleObject(hServiceEvent, INFINITE);

    // Service shutdown here    
cleanup:
    if (iWSAErr == 0)
        WSACleanup();
    if (hServiceEvent)
        CloseHandle(hServiceEvent);
}  // ServiceStart


// ServiceStop - Stop the service
//
// Parameters:
//   none
//
// Return value:
//   none
//
// Comments:
//   Terminate the listen thread and signal the main thread to exit
    
VOID ServiceStop()
{
    if (hLT != INVALID_HANDLE_VALUE) {
        TerminateThread(hLT, 0);            // Stop the listen thread
    }
    SetEvent(hServiceEvent);                // Signal the service thread to exit
}  // ServiceStop


// ListenThreadProc - Thread listens for connections
//
// Parameters:
//   none
//
// Return value:
//   none
//
// Comments:
//   Blocks waiting for a connection, and spawns a thread in ProcessThread
//   for each connection request.

void ListenThreadProc()
{
    fd_set fds;
    DWORD dwThreadId, dwParam = 1;
    HANDLE hTh;
    
    if (!SetupListen()) {
        ExitThread(0);              // Error, shutdown
    }
    
    FD_ZERO(&fds);
    FD_SET(skListen, &fds);
    while (select(0, &fds, NULL, NULL, NULL) != 0) {
        // Create process thread
        hTh = CreateThread(
                NULL,                        // no security attributes
                0,                           // use default stack size
                (LPTHREAD_START_ROUTINE)ProcessThreadProc, // thread function
                &dwParam,                    // argument to thread function
                0,                           // use default creation flags
                &dwThreadId                  // returns the thread identifier
                );
        FD_ZERO(&fds);
        FD_SET(skListen, &fds);
     }  // while
    return;
}  // ListenThreadProc


// SetupListen - set a socket to listen for client connection requests

BOOL SetupListen()
{
    SOCKADDR_IN sin;
   
    skListen = INVALID_SOCKET;
    if ((skListen = socket(AF_INET, SOCK_STREAM, 0)) == INVALID_SOCKET) {
        LogWSError();
        return(FALSE);
    }

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
    sin.sin_port = htons(FingerPort);

    if (bind(skListen, (LPSOCKADDR) &sin, sizeof(sin))) {
        LogWSError();
        return(FALSE);
    }

    if (listen(skListen, 5)) {
        LogWSError();
        return(FALSE);
    }
    return(TRUE);
}  // SetupListen


// ProcessThreadProc - Thread processes finger requests
//
// Parameters:
//   lpdwParam - ???
//
// Return value:
//   none
//
// Comments:
//   ProcessThreadProc is spawned by StartService when a connection request is 
//   received to process the request.

DWORD ProcessThreadProc(LPDWORD lpdwParam)
{
    SOCKET skOut;            // Send output to client

    skOut = INVALID_SOCKET;
    AcceptConnection(&skOut);
    if (ReceiveRequest(&skOut, RequestBuf))
        ParseRequest(&skOut, RequestBuf);
    if (skOut != INVALID_SOCKET);
        closesocket(skOut);
    return(1);
}  // ProcessThread


// AcceptConnection opens a connection to the client
//
// Parameters:
//   skOut - an open socket
//
// Return value:
//   TRUE upon successful accept, FALSE otherwise

BOOL AcceptConnection(SOCKET *skOut)
{
    struct sockaddr sad;
    int len = sizeof(sad);
    fd_set fds;

    // Get connection request
    *skOut = accept(skListen, &sad, &len);
    if (*skOut == INVALID_SOCKET) {
        LogWSError();
        return(FALSE);
    }

    // Wait for the data to arrive
    FD_ZERO(&fds);
    FD_SET(*skOut, &fds);
    if (select(0, &fds, NULL, NULL, NULL) == SOCKET_ERROR) {
        LogWSError();
        return(FALSE);
    } else
        return(TRUE);
}  // AcceptConnection


// ReceiveRequest receives the text from the client and puts it in
// the request buffer.  We expect <CR><LF> terminator, but will accept
// end-of-stream (zero recv return).

BOOL ReceiveRequest(SOCKET *skOutput, LPSTR ReqStr)
{
    char Buf[XFERBUFLEN];               // transfer buffer holds in/outbound text
    int ReqLen;                         // length of request string
    char *p;
    int nchars;
    fd_set fds;
   
    ReqStr[0] = ReqLen = 0;
    Blocking(*skOutput);                // going synchronous now

    do {                                // until we've received the whole request
        nchars = recv(*skOutput, (LPSTR) Buf, sizeof(Buf), 0);
        if (nchars == SOCKET_ERROR) {
            LogWSError();
            return(FALSE);
        }

        Buf[nchars] = 0;                        // null terminate the buffer
        if ((ReqLen += nchars) > REQLEN)  {     // prevent request overflow
            SendError(*skOutput, "query too long");    // tell client
            return(FALSE);                      // request overflow, bail
        }

        strcat(ReqStr, Buf);                    // append to request
        p = strstr(ReqStr, "\r\n");             // end-of-request?
    }
    while (nchars && !p);                       // stop for either end-of-stream
                                                // or <CR><LF> terminator
    if (p) *p = 0;                              // chop the terminator
   
    // Wait for the reply side to be ready
    FD_ZERO(&fds);
    FD_SET(*skOutput, &fds);
    if (select(0, NULL, &fds, NULL, NULL) == SOCKET_ERROR) {
        LogWSError();
        return(FALSE);
    } else
        return(TRUE);
}  // ReceiveRequest


// Blocking -- sets socket to block.
void Blocking(SOCKET s)
{
   u_long nonblock = FALSE;               // yes, we have no bananas

//   WSAAsyncSelect(s, hFrame, 0, 0);       // turn off message notifications
   ioctlsocket(s, FIONBIO, &nonblock);    // set socket to blocking
}  // Blocking


// ParseRequest - Determine the user's command line request
//
// Parameters:
//   skOut  - an open socket
//   ReqStr - the user command line
//
// Return value:
//   none

void ParseRequest(SOCKET *skOut, LPSTR ReqStr)
{
    static char *hello = "hello you\r\n";
        
    if (strlen(ReqStr) == 0) {
        SendFingerAll(*skOut);
    } else
    if (!stricmp(ReqStr, "hello")) {
        SendLine(*skOut, hello);
    } else 
    if (!stricmp(ReqStr, "id")) {
        SendID(*skOut);
    } else
    {
        SendFinger(*skOut, ReqStr);
    }
}  // ParseRequest

