// process.c -- Finger daemon processors for special cases
// Copyright 1995 Dean Troyer (troyer@indirect.com)
// All rights reserved.
//
// See fingerd.txt for license information
//
// Here's where the 'action' of the finger server is implemented
//
// This module is compiled with Unicode turned on to support
// the Win32 APIs that require Unicode
//
// 0.4  02Feb95      dlt     Initial release

#include <windows.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <time.h>
#include "config.h"
#include "process.h"

#define UNICODE
#define _UNICODE
#include <lm.h>

#define BUFSIZE 512

// Error reporting messages
typedef struct {                    // associates an error code with text
   UINT err;
   char *sztext;
} ERRENTRY;

ERRENTRY WSErrors[] =               // error text for windows sockets errors
{
   WSAVERNOTSUPPORTED,  "This version of Windows Sockets is not supported",
   WSASYSNOTREADY,      "Windows Sockets is not present or is not responding",
};


// Global Data
char LMDomain[DNLEN+2];


// Misc macros
#define dim(x) (sizeof(x) / sizeof(x[0]))


BOOL WINAPI UnicodeToANSI(LPTSTR lpInputString, 
                          LPSTR lpszOutputString, 
                          int nOutStringLen);
VOID WINAPI ANSIToUnicode(LPSTR  lpInputString, 
                          LPTSTR lpszOutputString, 
                          int nOutStringLen);


// SendID - Sends the service ID information
//
// Parameters:
//   skOut  - an open socket
//
// Return value:
//   none

void SendID(SOCKET skOut)
{
//    TCHAR Buf[BUFSIZE];

//    gethostname(Buf, 99);
//    send(*skOutput, Buf, strlen(Buf), 0);
    
    SendLine(skOut, "%s %s - %s\n", INTERNAL_NAME, VERSION, DISPLAY_NAME);
}  // SendID


// SendFinger - Sends the information about the requested userid
//
// Parameters:
//   skOut  - an open socket
//   UserID - the userid to query & respond
//
// Return value:
//   none

void SendFinger(SOCKET skOut, LPSTR UserID)
{
    NET_API_STATUS dwRetCode;
    LPUSER_INFO_3 lpInfo;
    WCHAR User[UNLEN+1];            // User name
    WCHAR Dom[DNLEN+1];             // Domain name
    LPWSTR lpDom;
    struct tm *t;
    LPSTR p;
    char Buf[BUFSIZE];
        
    if (p = strchr(UserID, '\\')) {
        *p = '\0';                  // Split domain & username
        p++;
        ANSIToUnicode(UserID, (LPTSTR)Dom, sizeof(Dom));
        /* Try to find a DC for the domain */
        if ((dwRetCode = NetGetDCName(NULL, (LPWSTR)&Dom, &lpDom)) 
             != NERR_Success) {
            /* Can't find a domain controller, fail */
            SendLineW(skOut, 
                TEXT("Error: can't find domain controller for %s\n"), 
                lpDom);
            return;
        }
        ANSIToUnicode(p, (LPTSTR)User, sizeof(User));
    } else {
        // Determine domain/machine name & convert to Unicode
        if (strlen(LMDomain)) {
            ANSIToUnicode(LMDomain, (LPTSTR)Dom, sizeof(Dom));
            if (LMDomain[0] == '\\' && LMDomain[1] == '\\') { 
                /* Assume it's a machine name */
                lpDom = (LPWSTR)&Dom;
            } else {
                /* Try to find a DC for the domain */
                if ((dwRetCode = NetGetDCName(NULL, (LPWSTR)&Dom, &lpDom)) 
                     != NERR_Success) {
                    /* Can't find a domain controller, fail */
                    SendLineW(skOut, 
                        TEXT("Error: can't find domain controller for %s\n"), 
                        lpDom);
                    return;
                }
            }
        } else {
            char Buf[32];

            // Default to the local machine
            gethostname(Buf, 32);
            ANSIToUnicode(Buf, (LPTSTR)Dom, sizeof(Dom));
            /* No domain or machine specified */
            lpDom = NULL;
        }
        ANSIToUnicode(UserID, (LPTSTR)User, sizeof(User));
    }
    
    if ((dwRetCode = NetUserGetInfo(lpDom,      /* Domain */
            User,                               /* Username */
            3,                                  // Level
            (LPBYTE *) &lpInfo))                // Return buffer
             == NERR_Success) {
        SendLineW(skOut, 
            TEXT("Login name: %ws\\%ws\t\t\tIn real life: %ws\nDirectory: %ws\n"), 
            Dom,
            lpInfo->usri3_name,
            lpInfo->usri3_full_name,
            lpInfo->usri3_home_dir);
        t = localtime(&lpInfo->usri3_last_logon);
        SendLine(skOut, "Last login %s\n", asctime(localtime(&lpInfo->usri3_last_logon)));
        UnicodeToANSI(lpInfo->usri3_home_dir, Buf, sizeof(Buf));
        SendPF(skOut, Buf, "Project");
        UnicodeToANSI(lpInfo->usri3_home_dir, Buf, sizeof(Buf));
        SendPF(skOut, Buf, "Plan");
        NetApiBufferFree(lpInfo);
    } else {  // if
        switch (dwRetCode) {
            case NERR_UserNotFound:
                SendLine(skOut, "unknown userid.\n");
                break;
            default:
                SendLine(skOut, "error: %ld\n", dwRetCode);
        }  // switch
    }  // if !
}  // SendFinger


// SendFingerAll - Sends the list of those logged on to skOut
//
// Parameters:
//   skOut  - an open socket
//
// Return value:
//   none
//
// Comments:
//   Under NT, how does one define 'logged on' in the absence
//   of a local logon context?  Let's list those with connections...

void SendFingerAll(SOCKET skOut)
{
    NET_API_STATUS dwRetCode;
    LPWKSTA_USER_INFO_1 lpInfo;
    DWORD dwEntries, dwTotal;
    DWORD hResume;
    int i;
    
    static char *NoLogged = "No one logged on.\n";
    static TCHAR *Line1 = "FingerD: enumerate users\n";

    SendLine(skOut, Line1);
    
    hResume = 0;
    // NetWkstaUserEnum to get all users?
    if ((dwRetCode = NetWkstaUserEnum(NULL,     // Do the local machine
            1,                                  // Level 1?
            (LPBYTE *) &lpInfo,                 // Return buffer
            65536,                              // Max return data len
            &dwEntries,                         // Number of entries returned
            &dwTotal,                           // Number of entries possible from here
            &hResume)) == NERR_Success) {       // Resume handle
        SendLine(skOut, "Total number users: %d\tNumber users: %d\n", 
            dwTotal, dwEntries);
        if (dwEntries == 0) {
            SendLine(skOut, NoLogged);
        } else
            for (i=0; i<dwEntries; ++i) {
                SendLineW(skOut, 
                    TEXT("User: %ws  server: %ws\n"), 
                    lpInfo[i].wkui1_username,
                    lpInfo[i].wkui1_logon_server);
            }  // for
    }  // if
    NetApiBufferFree(lpInfo);
}  // SendFingerAll


// SendPF - Sends the user's plan or project file
//
// Parameters:
//   skOut   - an open socket
//   HomeDir - users home directory
//
// Return value:
//   TRUE if successful
//
// Comments:
//   Assumes that HomeDir is big enough to accept .plan - should fix this

BOOL SendPF(SOCKET skOut, LPSTR HomeDir, LPSTR PP)
{
    FILE *infile;
    BOOL Result = TRUE;
    
    SendLine(skOut, "%s:\n", PP);
    strcat(HomeDir, "\\.");
    strcat(HomeDir, PP);
    if (infile = fopen(HomeDir, "rb")) {       // Open for read
        SendFilePrim(infile, skOut);
        fclose(infile);
    } else {
        SendLine(skOut, "no %s.\n", PP);
        Result = FALSE;
    }
    return(Result);
}  // SendPF


// SendFile - Sends the named file to the client
//
// Parameters:
//   skOut    - an open socket
//   FileName - the file to send
//
// Return value:
//   TRUE if successful

BOOL SendFile(SOCKET *skOut, LPSTR FileName)
{
    FILE *infile;
    BOOL Result = TRUE;
    
//    Blocking(*skOut);                        // going synchronous now
    if (infile = fopen(FileName, "rb")) {       // Open for read
        SendFilePrim(infile, *skOut);
        fclose(infile);
    } else {
        SendError(*skOut, _strerror(NULL));  // file access error text
        Result = FALSE;
        AddToMessageLog("SendFile: file not found");
    }
    return(Result);
}  // SendFile


// SendFilePrim - Sends the open infile down the skOutput stream
//
// Parameters:
//   infile - an open file descriptor
//   skOut  - an open socket
//
// Return value:
//   none

void SendFilePrim(FILE *infile, SOCKET skOut)
{
    char Buf[XFERBUFLEN];                       // Send buffer
    int nchars;

    do {
        nchars = fread(Buf, 1, sizeof(Buf), infile);
        if (ferror(infile)) {
            SendError(skOut, _strerror(NULL));
            return;
        }

        if ((nchars > 0) && (send(skOut, Buf, nchars, 0) == SOCKET_ERROR)) {
            LogWSError();
            return;
        }
    } while (!feof(infile));
}  // SendFilePrim


// SendLine - Send a text buffer on a socket
//
// Parameters:
//   skOut  - an open socket
//   fmtstr - format string
//   <args> - arguments for the format string
//
// Return value:
//   Returns 0 on success, -1 on SOCKET_ERROR
 
int SendLine(SOCKET skOut, LPSTR fmtstr, ...)
{
    va_list args;
    char Buf[XFERBUFLEN];

    va_start(args, fmtstr);
    wvsprintfA(Buf, fmtstr, args);
//    vsprintf(Buf, fmtstr, args);
    va_end(args);

    if (send(skOut, Buf, strlen(Buf), 0) == SOCKET_ERROR) {
        LogWSError();
        return(-1);
    }
    return(0);
}  // SendLine


// SendLineW - Send a text buffer on a socket
//
// Parameters:
//   skOut  - an open socket
//   fmtstr - format string
//   <args> - arguments for the format string
//
// Return value:
//   Returns 0 on success, -1 on SOCKET_ERROR
//
// Comments:
//   SendLineW sends a Unicode buffer

int SendLineW(SOCKET skOut, LPTSTR fmtstr, ...)
{
    va_list args;
    CHAR  BufA[XFERBUFLEN];
    TCHAR BufW[XFERBUFLEN];

    va_start(args, fmtstr);
    wvsprintf(BufW, fmtstr, args);
    va_end(args);

    UnicodeToANSI(BufW, BufA, BUFSIZE);
    if (send(skOut, BufA, strlen(BufA), 0) == SOCKET_ERROR) {
        LogWSError();
        return(-1);
    }
    return(0);
}  // SendLineW


// SendError - Sends an error string to client.
//
// Parameters:
//   skOut  - an open socket
//   errstr - the error text
//
// Return value:
//   none

void SendError(SOCKET skOut, char *errstr)
{
   static char *prefix = "\r\nserver error: ";

   SendLine(skOut, prefix);
   SendLine(skOut, errstr);
}  // SendError

// LogWSError - Logs the winsock error in the Application Event Log
void LogWSError()
{
   AddToMessageLog(WSErrorString(WSAGetLastError()));
}  // LogWSError

// WSErrorString - Translates winsock error to appropriate string.
char *WSErrorString(UINT err)
{
   int i;
   static char szerr[80];

   for (i = 0; i < dim(WSErrors); i++)
      if (err == WSErrors[i].err)
         return(WSErrors[i].sztext);

   wsprintfA(szerr, "Windows Sockets reports error %04x (%d)", err, err);
   return(szerr);
}  // WSErrorString

// Taken from Ralph Davis' \NTNET\CODE\WNETMISC.CPP

BOOL WINAPI UnicodeToANSI(LPTSTR lpInputString, 
                          LPSTR lpszOutputString, 
                          int nOutStringLen)
{         
#ifndef WIN32S
   CPINFO CodePageInfo;

   GetCPInfo(CP_ACP, &CodePageInfo);

   if (CodePageInfo.MaxCharSize > 1)
      // Only supporting non-Unicode strings
      return FALSE; 
   else if (((LPBYTE) lpInputString)[1] == '\0')
      {
      // Looks like unicode, better translate it
      WideCharToMultiByte(CP_ACP, 0, (LPCWSTR) lpInputString, -1,
         lpszOutputString, nOutStringLen, NULL, NULL);
      }
   else
      lstrcpyA(lpszOutputString, (LPSTR) lpInputString);
#else
      lstrcpy(lpszOutputString, (LPSTR) lpInputString);
#endif
   return TRUE;
}  // UnicodeToANSI

VOID WINAPI ANSIToUnicode(LPSTR  lpInputString, 
                          LPTSTR lpszOutputString, 
                          int nOutStringLen)
{         

#ifndef WIN32S
   CPINFO CodePageInfo;

   lstrcpy(lpszOutputString, (LPTSTR) lpInputString);

   GetCPInfo(CP_ACP, &CodePageInfo);

   if (CodePageInfo.MaxCharSize > 1)
      // It must already be a Unicode string
      return;
   else if (((LPBYTE) lpInputString)[1] != '\0')
      {
      // Looks like ANSI, better translate it
      MultiByteToWideChar(CP_ACP, 0, (LPCSTR) lpInputString, -1,
         (LPWSTR) lpszOutputString, nOutStringLen);
      }
else
   lstrcpy(lpszOutputString, (LPTSTR) lpInputString);
#endif
}  // ANSIToUnicode


