# NAME

Error::Show - Locate and Diagnose Errors/Exceptions in Perl programs

# SYNOPSIS

## String EVAL

Consider the string/program to be evaluated that has a illegal divide by zero ( or could be a syntax error );

```perl
use Error::Show;

my $prog='

sub call_me {
  $a+1/0;
}

say "hello there";
call_me;

';
```

With normal string eval and exception checking, we get:

```perl
my $result =eval $prog; 
if($@){
  say $@;
}
#======== OUTPUT

hello there
Illegal division by zero at (eval 1) line 4.
```

Using normal string eval with `context` we get:

```perl
my $result =eval $prog; 
if($@){
  say "ERROR is $@".Error::Show::context $@;
}

#====== OUTPUT

(eval 2)
Illegal division by zero at (eval 2) 0
    examples/eval.pl
    30
    31   {
    32     say "================== Normal string with context =====================";
    33     my $result =eval $prog;
    34     if($@){
    35=>     say Error::Show::context $@;
    36     }
    37
    38   }
    39
    40
```

Using `streval` with `context` we get:

```perl
my $result =eval {streval $prog};
if($@){
  say Error::Show::context $@;
}

#======= OUTPUT

hello there
(eval 4)
1
2
3     sub call_me {
4=>     $a+1/0;
5     }
6
7     say "hello there";
8     call_me;
9
Illegal division by zero at (eval 4) 4
    examples/eval.pl
    41
    42   say "";
    43
    44   {
    45     say "================== eval streval =====================";
    46=>   my $result =eval {streval $prog};
    47     if($@){
    48       say Error::Show::context $@;
    49     }
    50
    51
```

## Command Line Syntax Checking

Consider the following program (at examples/synopsis.pl in this distribution).
It has a syntax error on line 13, and uses an experimental feature on line 7.

```perl
use strict;
use warnings;
use Time::HiRes;

use feature "refaliasing";

\my $a=\"hello";
my $time=time;
for(1..1000){
  print "$_\n";
}

my $crazy-var=2;

use Socket;

print "this will never work";
```

Attempting to run this program with Perl normally gives this error output:

```perl
->perl examples/synopsis.pl       
Aliasing via reference is experimental at examples/synopsis.pl line 7.
Can't modify subtraction (-) in scalar assignment at examples/synopsis.pl line 13, near "2;"
BEGIN not safe after errors--compilation aborted at examples/synopsis.pl line 15.
```

With `Error::Show` enabled with the `-M` switch, this instead looks like this:

```perl
->perl -MError::Show=warn  examples/synopsis.pl
examples/synopsis.pl
 3   use Time::HiRes;
 4
 5   use feature "refaliasing";
 6
 7=> \my $a=\"hello";
 8   my $time=time;
 9   for(1..1000){
10    print "$_\n";
11   }
12
Aliasing via reference is experimental at examples/synopsis.pl line 7.
examples/synopsis.pl
 9   for(1..1000){
10    print "$_\n";
11   }
12
13=> my $crazy-var=2;
14
15   use Socket;
16
17   print "this will never work";
Can't modify subtraction (-) in scalar assignment at examples/synopsis.pl line 13, near "2;"
examples/synopsis.pl
11   }
12
13   my $crazy-var=2;
14
15=> use Socket;
16
17   print "this will never work";
BEGIN not safe after errors--compilation aborted at examples/synopsis.pl line 15.
```

## Command Line Exception Call Stack Dump (from v0.4.0)

Consider the following short program (examples/synopsis2.pl

```perl
use v5.36;

sub test {

  my $a=1/0;
}


test;
```

Running this program with Perl normally gives this error output:

```
perl examples/synopsis2.pl  
Illegal division by zero at examples/synopsis2.pl line 5.
```

With `Error::Show` enabled with the `-M` switch, this instead looks like this:

```perl
perl -MError::Show examples/synopsis2.pl
examples/synopsis2.pl
1   use v5.36;
2
3   sub test {
4
5=>   my $a=1/0;
6   }
7
8
9   test;
Illegal division by zero at examples/synopsis2.pl 5

    examples/synopsis2.pl
    4
    5     my $a=1/0;
    6   }
    7
    8
    9=> test;
Illegal division by zero at examples/synopsis2.pl 5
```

## In Program

Use at runtime to supplement exception handling without signal handler modification:

```perl
use Error::Show;

#an die caught in a try/eval triggers an exception

# No argument uses $@ as error
#
eval { exceptional_code };
say STDERR context if $@;


# or a single exception argument of your choosing
#
use v5.36;
try { 
  exceptional_code
}
catch($e) {
  say STDERR context $e;
}

# Show context down a stack

try {

  Some_execption_class->throw("Bad things");

}
catch($e){
  say STDERR context $e;
}

```

# DESCRIPTION

This module provides three tools/modes to help locate and diagnose errors in
your Perl programs.

## Command Line

From the command line this module transparently executes your syntactically
correct program. No code changes are required. However in the case of syntax
errors (or warnings if desired), it extracts context (lines of code)
surrounding them. The lines are prefixed with numbers  and the nicely formatted
context is dumped on STDERR for you to see the error or your ways.

The resulting output is optionally filtered seamlessly through the **splain**
program (see [diagnostics](https://metacpan.org/pod/diagnostics)), giving more information on why the reported
syntax errors and warnings might have occurred. 

## In Program Exception Context

From within a program, this module can be used to give formatted code context
around the source of an exception and the context of each of the stack frames
captured when the exception was raised.

It supports Perl string exceptions and warnings directly and also provides the
ability to integrate third party CPAN exception objects and traces with minimal
effort. Please see examples in this document or in the examples directory of
the distribution showing use with [Mojo::Exception](https://metacpan.org/pod/Mojo%3A%3AException), [Exception::Base](https://metacpan.org/pod/Exception%3A%3ABase),
[Exception::Class::Base](https://metacpan.org/pod/Exception%3A%3AClass%3A%3ABase) and [Class::Throwable](https://metacpan.org/pod/Class%3A%3AThrowable).

From `v0.5.0`  a `throw` routine is exported which, generates a very basic
exception object, with stack frame capture, in case using an larger exception
object/class is unneeded

## String evals with Execption/Syntax Error Context

From `v0.5.0` the `streval` subroutine has been added to allow context
information to be generated for errors originating from code dynamically
created from string evaluations.

# Changes and Options

A handful of options are available for basic configuration of how many lines of
code to print before and after the error line, indenting of stack trace
context, etc.

**v0.5.0** Has alot of changes which might break compatibility. THe up side is
the codes is simpler, has less errors, and easier to use. This is especial the
case for third party exceptions objects. It also gives the intended purpose of
handling errors and exceptions from  string evals like any other.

That being said, an earlier version might need to be used if you want the use
the `$@` variable implicitly in a call to `context`. This subroutine now
requires an explicated error argument

Sub routines `context`, `throw` and `streval` are exported by default.

**From v0.3.0:** `context` subroutine is now exported by default. To prevent
this, import with an empty list, ie `use Error::Show ()`.

`From v0.2.0:`, Added 'advanced string eval' support has been added for better
context reporting of dynamically generated code.

# USAGE

## Command Line Usage (Syntax check and Exception Catching)

```
    perl -MError::Show  [options] file.pl 
```

When included in a command line switch to Perl, `-MError::Show` syntax checks
the input program. If the syntax is OK, normal execution continues in a
transparent fashion.  Otherwise, detailed code context surrounding the source
of the error is generated and printed on STDERR.

**From v0.4.0** a global \_\_DIE\_\_ handler is also installed, which will catch any
stray exceptions during execution and present a line numbered summary stack
trace. Programs a free to overload this handler. However the features of this
module will be lost.

**NOTE:** It is important that it's the first `-M` switch for this module to
operate correctly and to prevent any incompatibilities withe global signal
handlers.

If the **-c** flag is specified, only a syntax check will be performed,
mimicking normal Perl behaviour.

Additional `@INC` directories using the **-I** switch are supported as are
additional modules via the **-M** switch.

### CLI Usage Options

The following options can be used in isolation or together:

#### clean

If you prefer just the code context without the Perl error, add the clean
option:

```
perl -MError::Show=clean file.pl
```

#### warn

This options enables processing of warnings as well as errors.

```
perl -MError::Show=warn file.pl
```

#### splain

Runs the output through the splain program (see [diagnostics](https://metacpan.org/pod/diagnostics)), giving
probable reasons behind the error or warning

```
perl -MError::Show=splain file.pl
```

#### no\_handler (from v0.4.0)

```
perl -MError::Show=no_handler file.pl
```

Prevents the global DIE handler from being installed.

### Return code

When in check only mode (-c), the main process is exited, just has Perl
normally would have done. The return code is a replica of what Perl would have
reported for success/failure of a syntax check.

## In Program (Exception) Usage

Simply bring [Error::Show](https://metacpan.org/pod/Error%3A%3AShow) into your program with a use statement:

```perl
use Error::Show;
```

It provides a single subroutine for processing errors and exceptions.

### Error::Show::context

```perl
my $context=Error::Show::context $error , options_pairs, ...;
```

Takes an error string, or exception object `$error` and extracts the code
surrounding the source of the error. The code lines are prefixed with line
numbers and the error line marked with a fat arrow.

The expected types of error are 

- Error string, as per `die` and `warn`

    This is expected to contain file and line number. These are extracted from the
    string error to locate context.

- Supported Execption Object

    Several common exceptions classes on CPAN are supported:

    - Error::Show::Exception
    - Exception::Class::Base
    - Exception::Base
    - Class::Throwable
    - Mojo::Exception

    Other classes will likely work as long as the stringify to resemble a perl
    error string  (with the filename and line number formatted correctly). The
    stack frame capture would work however)

The return value is the formatted context, followed by the original Perl error
strings, or stringified exception objects/messages:

```perl
filename.pl 
10  #code before 
11  #code before 
12=>#this line caused the error
13  #code after
14  #code after

... error... at filename.pl line 12 ...
```

The `context` subroutine (5) is exported by default, so does not need a fully
qualified name

`context` Can also be called with package arrow notation (6) ie
`Error::Show->context(...)` if prefered, with the same argument handling
as the other forms.

**Options include:**

#### limit 

```perl
limit=>$int
```

**From v0.2.0:** Limits the number of errors to extract and generate context
for. Default is 100. If <=0,  no limiting is applied all all errors are
processed.

#### reverse

```perl
reverse=>$bool
```

**From v0.2.0:** Reverses the order of error processing. 

Perl type string errors are sorted and processing in ascending line number
order. When this option is used, the lines are processed by descending line
number first. Does not change order of files processed.

If frames are used instead, they are processed in reverse order to how they
where supplied when this option is in effect.

#### pre\_lines

```perl
pre_lines=>value
```

Specific the maximum lines of code to display before the error line. Default is
5.

#### post\_lines

```perl
post_lines=>value
```

Specific the maximum lines of code to display after the error line. Default is
5.

#### clean

```perl
clean=>bool
```

When true, the normal Perl error string is not included in the context
information, for a cleaner look.

#### indent

```perl
indent=>string
```

The string to use for each level of indent when printing multiple stack frames.
Defaults to 4 spaces.

#### splain

```perl
splain=>1
```

The resulting output will be filtered through the [splain](https://metacpan.org/pod/splain) program.  

#### program

```perl
program=>$prog
```

The **program** option is used to specify the program text to process when
there is no actual file. This is needed when to show syntax errors in a string
`eval`:

```perl
my $prog='my $a="This will Fail"+b';
eval $prog;
if($@){
  say Error::Show::context error=>$@, program=>prog;
}
```

For advanced string eval processing options please see the **ADVANCED STRING
EVAL** section in this document.

### streval

```perl
local $@;

my $result=eval { streval "program";

if($@){
  print STDERR context $@;
}


##### or ####


try {
  my $result=strval "program";
}
catch($e){
  print STDERR context $e;
}
```

`streval` makes it possible to debug code generated from string eval.
Internally it caches program code, to allow a context to be generated in the
case of an exception.

**NOTE:** It also throws an exception on syntax error, which also then can have
a context generated for it.

### throw

```
throw "My error message";
```

Raises an exception with a basic exception object with the provided message.
Captures the call stack frames at the point of call.

Use this if you don't want to use other exception classes/modules

# ADVANCED STRING EVAL 

**From v0.2.0** features to support advanced string evaluation are available.
This was added to support the error reporting needs of the [Template::Plex](https://metacpan.org/pod/Template%3A%3APlex)
module.

Consider the following example. The 'meat' is the sub  in the middle of the
string. Any errors/context reported should be relative to this, not the start
of the overall eval string. With some help from comment markers "##\_PREAMBLE"
and "##\_POSTAMBLE",  when can search for the middle and rebase the error line
numbering.

```perl
eval {
  "my $some_prep_work=1;
  call_somthing();
  #comments...
  
  ##_PREAMBLE

  sub {
     #code generated from user input...
     say "My really cool dynamically created code"
  }

  ##_POSTAMBLE

  #More code 
  #Cleanup stuff.
  "
}
```

Additional configuration options can be provided  to search for the relevant
code lines and offset the error line numbers.

**NOTE** if these options are used, the **message** field is modified with
updated line numbers if its in the form of a normal Perl errors ie 'error line
123 at file'.

### start\_mark

```perl
start_mark=>$regexp
```

If specified, is a used as a regexp to match against source code lines. The line
after a successful match is now the first line.

This allows inserting a special marker to indicate the start of 'code of
interest' with out knowing the exact line number in the resulting code.

This is undefined and unused by default.

### end\_mark

```perl
end_mark=>$regexp
```

If specified, is used as a regexp to match against source lines, in reverse
order. The line after a successful match is now the last line

This allows inserting a special marker to indicate the end of 'code of
interest'.

This is undefined unused by default.

### start\_offset

```perl
start_offset=>$int
```

A static offset to add to the start line (which may have been modified by the
**start\_mark** option). The result will be classed as the minimum line number of
the file/string.

This is useful to prevent any preamble after the start\_mark line in your string
eval showing up in the user program context.

### end\_offset

A static offset to subtract to the end line (which may have been modified by
the **end\_mark** option). The result will be classed as the maximum line number
of the file.

This is useful to prevent any postamble before the end\_mark in your string eval
showing up in the user program context.

# FUTURE WORK/TODO

- Possible just use the DIE and WARN signal handler instead of forking processes?
- Make usable from a Language Server?
- Colour terminal output?
- JSON output?

# KNOWN ISSUES/GOTCHAS

Checking/running  programs via STDIN, -e and -E switches is not supported and
will die with an error message.

More data then needed is pushed through the splain program when splain option
is used, which isn't ideal.

# SEE ALSO

[Perl::Syntax](https://metacpan.org/pod/Perl%3A%3ASyntax) provides syntax checking from the command line. However it
doesn't show any errors by design (only interested in process return code)

[Syntax::Check](https://metacpan.org/pod/Syntax%3A%3ACheck) provides programmatic syntax checking of files.

[Perl::Critic](https://metacpan.org/pod/Perl%3A%3ACritic) gives actual Perl linting, but not great for syntax errors.

[diagnostics](https://metacpan.org/pod/diagnostics)  and the `splain` program give some very useful explanations
about the otherwise terse error strings normally output. It is part of the Perl
distribution

# AUTHOR

Ruben Westerberg, <drclaw@mac.com>

# REPOSITORTY and BUGS

Please report any bugs via git hub:
[https://github.com/drclaw1394/perl-error-show](https://github.com/drclaw1394/perl-error-show)

# COPYRIGHT AND LICENSE

Copyright (C) 2025 by Ruben Westerberg

This library is free software; you can redistribute it and/or modify it under
the same terms as Perl or the MIT license.

# DISCLAIMER OF WARRANTIES

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE.
