/*
 * Copyright (c) 2000 Ben Woodard
 * Copyright (c) 1983 Regents of the University of California.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licence
 * as published by the Free Software Foundation; either version 2
 * of the Licence, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRENTY; without even the implied warrenty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public Licence in the COPYING file for more
 * details.
 *
 * You should have received a copy of the GNU Library General 
 * Public License along with the GNU C Library; see the file 
 * COPYING.LIB.  If not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 *
 * Some portions of this work fall under the following license:
 *
 * Redistribution and use in source and binary forms, with or 
 * without modification, are permitted provided that the 
 * following conditions are met:
 * 1. Redistributions of source code must retain the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer.
 * 2. Redistributions in binary form must reproduce the above 
 *    copyright notice, this list of conditions and the following 
 *    disclaimer in the documentation and/or other materials 
 *    provided with the distribution.
 * 3. Neither the name of the University nor the names of its 
 *    contributors may be used to endorse or promote products 
 *    derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND 
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT 
 * SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF 
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __USE_GNU
#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <syslog.h>
#include <unistd.h>

#include <glib.h>

#include "printsys.h"
#include "printcap.h"

#define MAXHOP	32		/* max number of tc= indirections */
#define MAXBUF 1024
/* This parsing of printcap entries inside of every function has 
   got to stop. Borrowed this code from the gnu-lpr v2.0. source 
   tree hopefully by embedding it in to gnu-lpr v1.0 we can 
   improve some things over here a little bit */
 /* 
  * <ben> Well we had to do our comp sci project for hash tables 
  *     but I didn't like hash tables and I'm a bit of a rebel 
  *     see.
  * <ben> We were allowed to use any method we wanted to deal 
  *     with hash collisons. 
  * <ben> They expected us to do linked lists, ajacent element 
  *     scan etc..
  * <ben> I decided to use trees.
  * <ben> However, I cheated on my program. I #define HASH_SIZE 
  *     to be 1
  * <ben> So basically I had a 1 element hash table with a tree 
  *     underneath it.
  * <ben> I figured that if the grader asked me I would say "oops 
  *     that was a mistake from when I was testing the hash 
  *     collision code. Just make that 1000 or something"
  * <ben> Well it turned out my program was one of the fastest 
  *     and most reliable programs in the class and the TA never 
  *     noticed that I only had a 1 element hash table.
  *
  *                     That said, I'm using a hash table.  -NickM
  */

/* That said, you suck nick. -ben */

/* BTW the reason I used trees was because I already had working 
   tree code from a previous assignment. -ben */

struct printcap_data {
  GHashTable *name_lookup;
  GList *printers;
  time_t last_change;
  GList *last_printer;
  char *printcapfile;
};

static guint wstr_hash (gconstpointer key){
  const wchar_t *p = key;
  guint h = *p;
  
  if (h)
    for (p += 1; *p != '\0'; p++)
      h = (h << 5) - h + *p;
  
  return h;
}

static gint wstr_equal (gconstpointer v, gconstpointer v2){
  return wcscmp ((const wchar_t*) v, (const wchar_t*)v2) == 0;
}


/*----------------------------------------------------------------
----------------------------------------------------------------*/
static int uptodate(LPS_System_t *sys){
  struct stat buf;
  struct printcap_data *local_data=(struct printcap_data*)sys->data;

  if (local_data->printers == NULL) {
    printcap_init(sys,local_data->printcapfile);
    return 0;
  }

  return 1; /* FIXME: You can't just re-read the printcap file at this point
             * James */
  
  stat(local_data->printcapfile, &buf);
  if (buf.st_mtime != local_data->last_change) {
    printcap_end(sys);
    printcap_init(sys,local_data->printcapfile);
    return 0;
  }
  return 1;
}

#ifdef NO_COMPILE
/*----------------------------------------------------------------
  This is here to help in debugging. 
----------------------------------------------------------------*/
static void list_printers(GList * top){

  GList *curr;
  return;
  for (curr = top; curr; curr = g_list_next(curr)) {

    LPS_Printer_t *curr_printer = (LPS_PrinterType *) curr->data;

    printf("PRINTER: %s curr_printer = %p [num_fields = %u]\n",
	   curr_printer->names[0], curr_printer,
	   lps_pr_num_fields(curr_printer));
    printf("FIRST FIELD: %p (%p)\n", curr_printer->fields[0].key,
	   curr_printer->fields[0].value);
    printf("COMMENTS: %s\n==========================\n",
	   curr_printer->comments);
  }
}
#endif

/*----------------------------------------------------------------
----------------------------------------------------------------*/
int printcap_init(LPS_System_t *sys, const char *filename){
  int pc_fd = -1;
  GString *curbuf = NULL;
  char buf[MAXBUF];
  ssize_t bytes;
  struct stat stbuf;
  GList *cur = NULL;
  struct printcap_data *data=malloc(sizeof(struct printcap_data));

  assert(data);
  sys->data=data;
  data->name_lookup=g_hash_table_new(wstr_hash, wstr_equal);
  data->printers=NULL;
  data->last_change=0;
  data->last_printer=NULL;
  data->printcapfile=strdup(filename!=NULL?filename:"/etc/printcap");
  assert(data->printcapfile);

  pc_fd = open(data->printcapfile, O_RDONLY);
  if (pc_fd == -1) {
    sys->lps_errno=LPS_NOCONFIG;
    return 0;
  }

  if (fstat(pc_fd, &stbuf)) {
    g_hash_table_destroy(data->name_lookup);
    data->name_lookup=NULL;
    free(data->printcapfile);
    free(data);
    sys->lps_errno=LPS_INSANE;
    /* don't know what to do here */
    return 0;
  }
  data->last_change = stbuf.st_mtime;

  curbuf = g_string_new("");
  buf[1023] = 0;
  buf[0] = 0;
  for (bytes=read(pc_fd,buf,1023);bytes!=0;bytes=read(pc_fd,buf,1023)) {
    char *curpos;
    char *line_end;
    char *cpy = NULL;
    
    //    printf("read\n"); //debug
    if (bytes == -1){
      //      fprintf(stderr,"bad read\n");//DEBUG
      break;
    }

    buf[bytes] = 0;  /* null-terminate so that g_string_append works 
			properly */

    g_string_append(curbuf,buf);
    
    /* ok to '\' substitution ...
       "\\\n" == ""
       "\\\\\n" == "\\\n"
       ...that's it. Hope that's good enough.
    */
    /* move or copy forward by set ammount */
#define MVCP_FWD(l) do { \
  if (curpos != cpy) memmove(cpy, curpos, (l)); \
  curpos += (l); \
  cpy += (l); \
 } while (FALSE)

    cpy = curpos = curbuf->str;
    while (*curpos)
    {
      size_t tmp = 0;
      
      if (*curpos == '#')
      {
        tmp = strcspn(curpos, "\n");
        MVCP_FWD(tmp);
        
        if (*curpos == '\n')
          MVCP_FWD(1);
      }
      else
      {
        while (*curpos)
        {
          tmp = strcspn(curpos, "\\\n");
          MVCP_FWD(tmp);
          if (*curpos == '\n')
          {
            tmp = strspn(curpos, "\n");
            MVCP_FWD(tmp);
            break;
          }
          
          if ((curpos[0] == '\\') && (curpos[1] == '\n'))
            curpos += 2;
          else if (*curpos == '\\')
            MVCP_FWD(1);
        }
      }
    }
    MVCP_FWD(1);
#undef MVCP_FWD
    
    //    printf("merged:\n%s\n----\n",curbuf->str); //debug
    // fflush(stdout);

    while ((line_end=strchr(curbuf->str, '\n'))!=NULL) {
      char *tmpstr=strdup(curbuf->str);
      LPS_Printer_t *newone=_lps_process_entry(tmpstr);
      wchar_t **curname;

      // printf("process:\n%s\n---\n",tmpstr); //debug
      if(newone==NULL){
	g_string_assign(curbuf,tmpstr);
	free(tmpstr);
	continue;
      }

      // make sure that the name is unique
      // bomb out if it isn't come up with something better later
      for(curname=newone->names;*curname!=NULL;curname++)
	assert(g_hash_table_lookup(data->name_lookup,*curname)==NULL);

      //      printf("%S\n",newone->names[0]); // debug
      newone->system=sys;
      // stick the names into the lookup tree
      for (curname = newone->names; *curname; curname++)
	g_hash_table_insert(data->name_lookup, *curname, newone);
      // put it on list of printers
      data->printers = g_list_append(data->printers, newone);

      // remove processed line
      // printf("\n***\n%s\n***\n",tmpstr);
      g_string_assign(curbuf,tmpstr);
      free(tmpstr);
    }			    // end of while we have at least one line
  }			    // end of for - we have more file to read}
  close(pc_fd);

  // Now take care of the TC entries.
  /* Through a strange twist of fate (see above) the list of 
     fields ends up being in the same order as it was in the 
     printcap file. Therefore the TC entry should be at the end 
     which makes it easy to just extend the vector. */
  for (cur = data->printers; cur != NULL; cur = g_list_next(cur)) {

    LPS_Printer_t *cur_printer = (LPS_Printer_t *) cur->data;
    int depth;

    for (depth = 0; depth < MAXHOP; depth++) {
      LPS_Printer_t *template_printer;
      unsigned int num_entries = lps_pr_num_fields(cur_printer);
      unsigned int tot_entries;
      LPS_Pair_t *fields = cur_printer->fields;
      LPS_Pair_t *cur_field;
      wchar_t *cmpfld=lps_promote("tc");

      if (!num_entries)
      {
  	syslog(LOG_WARNING,
	       "printer %S misconfigured (no fields).",
	       cur_printer->names[0]);
        free(cmpfld);
        break;
      }
      
      /* Check the last entry. If this isn't a tc entry, break 
	 out of the loop and skip to next printer. */
      if (wcscmp(fields[num_entries - 1].key, cmpfld)){
	free(cmpfld);
	break;
      }

      /* Report problem and skip to next printer entry if we 
	 can't match the tc entry. */
      template_printer = g_hash_table_lookup(data->name_lookup,
					     fields[num_entries - 1].value);
      if ( template_printer == NULL) {
	syslog(LOG_WARNING,
	       "printer %S misconfigured. template printer %S can't be found.",
	       cur_printer->names[0], fields[num_entries - 1].value);
	free(cmpfld);
	break;			// break out to next printer
      }

      tot_entries = num_entries + lps_pr_num_fields(template_printer);
      cur_printer->fields = fields =
	g_realloc(fields, tot_entries * sizeof(LPS_Pair_t));

      fields[tot_entries - 1].key = fields[tot_entries - 1].value = NULL;

      /* copy the entries over -- notice that this overwrites the 
	 tc entry */
      for (fields = &fields[num_entries - 1], cur_field =
	   template_printer->fields; cur_field->key != NULL;
	   fields++, cur_field++) {
	fields->key=(wchar_t*) malloc(sizeof(wchar_t)*
				      (wcslen(cur_field->key)+1));
	wcscpy(fields->key,cur_field->key);
	fields->value = (wchar_t*) malloc(sizeof(wchar_t)*
				    (wcslen(cur_field->value)+1));
	wcscpy(fields->value,cur_field->value);
      }
      free(cmpfld);
    }				// for depth

    if (depth == MAXHOP)
      syslog(LOG_WARNING,
	     "printer %S misconfigured. Too many template printers.",
	     cur_printer->names[0]);
  } 			      // for each printer
  return 1;
}

/*----------------------------------------------------------------
  FIXME: Presumably this will return something on failure? -NickM
----------------------------------------------------------------*/
int printcap_end(LPS_System_t *sys)
{
  struct printcap_data *data=(struct printcap_data*)sys->data;
  // XXX this is probably a mem leak
  g_hash_table_destroy(data->name_lookup); 
  data->name_lookup = NULL;
  while (data->printers) {
    GList *save = data->printers->next;
    lps_free_printer(data->printers->data);
    g_list_free_1(data->printers);
    data->printers = save;
  }
  data->last_change = 0;
  return 1;
}

/*----------------------------------------------------------------
----------------------------------------------------------------*/
const wchar_t **printcap_get_printer_list(LPS_System_t *sys)
{
  guint numprs;
  wchar_t **retval;
  GList *cur;
  struct printcap_data *data=(struct printcap_data*)sys->data;
  
  uptodate(sys);

  numprs = g_list_length(data->printers);
  retval = (wchar_t**)malloc(sizeof(wchar_t**)*(numprs+1));
  retval[numprs] = NULL;

  for (cur = data->printers; cur != NULL; cur = g_list_next(cur)) {
    retval[--numprs] = ((LPS_Printer_t *) cur->data)->names[0];
  }
  return (const wchar_t**) retval;
}


/*----------------------------------------------------------------
----------------------------------------------------------------*/
LPS_Printer_t *printcap_get_printer(LPS_System_t *sys,
				      const wchar_t *prname){
  struct printcap_data *data=(struct printcap_data*)sys->data;
  uptodate(sys);
  return g_hash_table_lookup(data->name_lookup, prname);
}


static int write_printcap(struct printcap_data *data, int fd){
  GList *cur;
  int retval;

  retval=write(fd,"#\n# This printcap is being created by "
	       "libprintsys\n# Any changes made here manually may be"
	       " lost if printtool\n# is run later on.\n# The presence"
	       " of this header means that no printcap\n# existed when"
	       " printtool was run.\n#\n\n",217);
  if(retval==-1)
    return 0;

  for (cur = data->printers; cur != NULL; cur = g_list_next(cur)) {
    char *str;
    str=_lps_write_printcap_entry((LPS_Printer_t*)cur->data);
    retval=write(fd,str,strlen(str));
    if(retval==-1)
      return 0;
    free(str);
  }
  return 1;
}

struct del_struct{
  GHashTable *name_lookup;
  LPS_Printer_t *printer;
};

static void del_old_names(gpointer key,gpointer value,gpointer user_data){
  LPS_Printer_t *printer=((struct del_struct*)user_data)->printer;
  LPS_Printer_t *test_printer=value;

  GHashTable *name_lookup;
  wchar_t *test_name;
  wchar_t **curname;

  // leave if this isn't the same printer
  if(printer!=test_printer)
    return;

  // leave if this is a currently valid name
  test_name=key;
  for(curname=printer->names;*curname!=NULL;curname++)
    if(!wcscmp(*curname,test_name))
      return;

  name_lookup=((struct del_struct*)user_data)->name_lookup;
  g_hash_table_remove(name_lookup,test_name);
}

LPS_Error_t printcap_commit_printer(LPS_Printer_t *printer){
  struct printcap_data *data=(struct printcap_data*)printer->system->data;
  wchar_t **curname;
  int fd=open(data->printcapfile,O_WRONLY|O_CREAT|O_TRUNC,0644);
  struct del_struct ds;

  if(fd==-1)
    return LPS_PRINTCAP;
  flock(fd,LOCK_EX);

  /* FIXME: uptodate can't work here can it ... ?
     certainly we've changed the mtime of the file with the open()
     so if the mtime checks go back this has to change here */
  uptodate(printer->system);
  
  // stick the names into the lookup tree
  for (curname = printer->names; *curname; curname++)
    if(g_hash_table_lookup(data->name_lookup,*curname)==NULL)
      g_hash_table_insert(data->name_lookup, *curname, printer);

  ds.name_lookup=data->name_lookup;
  ds.printer=printer;
  g_hash_table_foreach(data->name_lookup,del_old_names,&ds);

  // remove it from the list if it is already there
  data->printers=g_list_remove(data->printers,printer);
  // put it on list of printers
  data->printers = g_list_prepend(data->printers, printer);

  if(write_printcap(data,fd)!=1)
    return LPS_PRINTCAP;
  flock(fd,LOCK_UN);
  close(fd);
  return LPS_OK;
}

LPS_Error_t printcap_destroy_printer(LPS_Printer_t *printer){
  struct printcap_data *data=(struct printcap_data*)printer->system->data;
  int fd=open(data->printcapfile,O_WRONLY|O_CREAT|O_TRUNC,0644);
  wchar_t **curname;

  if(fd==-1)
    return LPS_PRINTCAP;
  flock(fd,LOCK_EX);
  
  uptodate(printer->system);
  data->printers=g_list_remove(data->printers,printer);

  for (curname = printer->names; *curname; curname++)
    g_hash_table_remove(data->name_lookup, *curname);
  
  if(write_printcap(data,fd)!=1)
    return LPS_PRINTCAP;
  flock(fd,LOCK_UN);
  close(fd);
  return LPS_OK;
}

LPS_Error_t printcap_write(LPS_System_t *sys,int fd){
  struct printcap_data *data=(struct printcap_data*)sys->data;
  return write_printcap(data,fd)!=0?LPS_OK:LPS_PRINTCAP;
}
