/*
 * @(#)FileSingleSourceLoggerFactory.java
 *
 * Copyright (C) 2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.logger;

import java.io.File;
import java.io.FileWriter;
import java.io.Writer;
import java.io.IOException;

import java.util.Properties;


/**
 * Output the shared marks to a file.
 * <P>
 * Since this outputs to a single file, this has a potential issue
 * when multiple loggers are trying to access the same file.  This
 * could happen if the CoverageLogger class is loaded by different class
 * loaders; as a result, multiple CoverageLogger instances will be created,
 * allowing for multiple channel loggers to write to the same area.
 * <P>
 * We resolve this issue by adding a pseudo-random value to the end of
 * the file name.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/07/07 09:39:10 $
 * @since     April 16, 2004
 */
public class FileSingleSourceLoggerFactory extends AbstractSingleSourceLoggerFactory
{
    public static final String DIRECTORY_PROPERTY = "dir";
    public static final String FILENAME_PROPERTY = "file";
    private static final String DEFAULT_FILENAME_PREFIX = "single.";
    private static final String FILENAME_SUFFIX = ".log";
    
    private File outfile;
    
    /**
     * Setup the source writer.  This can return <tt>null</tt>.  Its
     * actions will be synchronized for you.
     */
    protected Writer setupSource()
    {
        setReloadSourceAfterError( false );
        
        try
        {
//System.out.println(":: Creating FileWriter for "+outfile.getAbsolutePath());
            return new FileWriter( this.outfile );
        }
        catch (IOException e)
        {
            e.printStackTrace();
            return null;
        }
    }
    
    
    /**
     * Extend this method to setup your own properties.  Be sure to
     * call the super's setupProps, though.  Be sure to keep the
     * properties around for how to connect to the source.
     */
    protected void setupProps( String prefix, Properties props )
    {
        super.setupProps( prefix, props );
        
        // fix the filename issue by adding a random value to the
        // name
        String pseudoRandom = Long.toString(
            System.currentTimeMillis() ) + Long.toString(
            Math.round( Math.random() * 1000.0 ) ) +
            FILENAME_SUFFIX;
        
        // the filename property should only really be used for debugging
        // purposes, as it invalidates the "single." prefix that the
        // report filter looks for.
        String filename = props.getProperty( prefix + FILENAME_PROPERTY );
        if (filename != null)
        {
            this.outfile = new File( filename + pseudoRandom );
        }
        else
        {
            String dir = props.getProperty( prefix + DIRECTORY_PROPERTY );
            if (dir != null)
            {
                this.outfile = new File( dir,
                    DEFAULT_FILENAME_PREFIX + pseudoRandom );
            }
        }
        
        if (this.outfile == null)
        {
            this.outfile = new File( DEFAULT_FILENAME_PREFIX + pseudoRandom );
        }
        
        File parent = this.outfile.getParentFile();
        if (!parent.exists())
        {
            parent.mkdirs();
        }
    }
}

