
/*****************************************************************************
*
* Copyright (c) 2003-2020 by The University of Queensland
* http://www.uq.edu.au
*
* Primary Business: Queensland, Australia
* Licensed under the Apache License, version 2.0
* http://www.apache.org/licenses/LICENSE-2.0
*
* Development until 2012 by Earth Systems Science Computational Center (ESSCC)
* Development 2012-2013 by School of Earth Sciences
* Development from 2014-2017 by Centre for Geoscience Computing (GeoComp)
* Development from 2019 by School of Earth and Environmental Sciences
**
*****************************************************************************/

#include <ripley/WaveAssembler3D.h>
#include <ripley/domainhelpers.h>

#include <escript/index.h>

using escript::Data;

namespace ripley {

WaveAssembler3D::WaveAssembler3D(escript::const_Domain_ptr dom,
                                 const double *dx, const dim_t *NE,
                                 const dim_t *NN, const DataMap& c)
    : AbstractAssembler(),
    m_dx(dx),
    m_NE(NE),
    m_NN(NN)
{
    domain = REFCOUNTNS::static_pointer_cast<const Brick>(dom);
    isHTI = isVTI = false;
    DataMap::const_iterator a = c.find("c12"), b = c.find("c23");
    if (c.find("c11") == c.end()
                || c.find("c13") == c.end() || c.find("c33") == c.end()
                || c.find("c44") == c.end() || c.find("c66") == c.end()
                || (a == c.end() && b == c.end()))
        throw escript::ValueError("required constants missing for WaveAssembler");

    if (a != c.end() && b != c.end()) {
        throw escript::NotImplementedError("WaveAssembler3D() doesn't support "
                                           "general form waves (yet)");
    } else if (a == c.end()) {
        c23 = b->second;
        isHTI = true;
    } else if (b == c.end()) {
        c12 = a->second;
        isVTI = true;
    } // final else case taken care of with the missing constants above
    c11 = c.find("c11")->second;
    c13 = c.find("c13")->second;
    c33 = c.find("c33")->second,
    c44 = c.find("c44")->second;
    c66 = c.find("c66")->second;

    int fs = c11.getFunctionSpace().getTypeCode();

    if (fs != c13.getFunctionSpace().getTypeCode()
            || fs != c33.getFunctionSpace().getTypeCode()
            || fs != c44.getFunctionSpace().getTypeCode()
            || fs != c66.getFunctionSpace().getTypeCode()) {
        throw escript::ValueError("C tensor elements are in mismatching function spaces");
    }
}

void WaveAssembler3D::collateFunctionSpaceTypes(std::vector<int>& fsTypes,
                                                const DataMap& coefs) const
{
    if (isNotEmpty("A", coefs))
        fsTypes.push_back(coefs.find("A")->second.getFunctionSpace().getTypeCode());
    if (isNotEmpty("B", coefs))
        fsTypes.push_back(coefs.find("B")->second.getFunctionSpace().getTypeCode());
    if (isNotEmpty("C", coefs))
        fsTypes.push_back(coefs.find("C")->second.getFunctionSpace().getTypeCode());
    if (isNotEmpty("D", coefs))
        fsTypes.push_back(coefs.find("D")->second.getFunctionSpace().getTypeCode());
    if (isNotEmpty("du", coefs))
        fsTypes.push_back(coefs.find("du")->second.getFunctionSpace().getTypeCode());
    if (isNotEmpty("Y", coefs))
        fsTypes.push_back(coefs.find("Y")->second.getFunctionSpace().getTypeCode());
}


void WaveAssembler3D::assemblePDESystem(escript::AbstractSystemMatrix* mat,
                               escript::Data& rhs, const DataMap& coefs) const
{
    if (isNotEmpty("X", coefs))
        throw RipleyException("Coefficient X was given to WaveAssembler "
                "unexpectedly. Specialised domains can't be used for general "
                "assemblage.");
    const Data& A = unpackData("A", coefs);
    const Data& B = unpackData("B", coefs);
    const Data& C = unpackData("C", coefs);
    const Data& D = unpackData("D", coefs);
    const Data& Y = unpackData("Y", coefs);
    const Data& du = unpackData("du", coefs);

    if ((!du.isEmpty()) && du.getFunctionSpace().getTypeCode() != c11.getFunctionSpace().getTypeCode()) {
        throw escript::ValueError("WaveAssembler3D: du and C tensor in mismatching function spaces");
    }

    dim_t numEq, numComp;
    if (!mat)
        numEq=numComp=(rhs.isEmpty() ? 1 : rhs.getDataPointSize());
    else {
        numEq=mat->getRowBlockSize();
        numComp=mat->getColumnBlockSize();
    }

    const double SQRT3 = 1.73205080756887719318;
    const double w10 = -m_dx[0]/288;
    const double w12 = w10*(-SQRT3 - 2);
    const double w6 = w10*(SQRT3 - 2);
    const double w18 = w10*(-4*SQRT3 - 7);
    const double w4 = w10*(-4*SQRT3 + 7);
    const double w11 = m_dx[1]/288;
    const double w15 = w11*(SQRT3 + 2);
    const double w5 = w11*(-SQRT3 + 2);
    const double w2 = w11*(4*SQRT3 - 7);
    const double w17 = w11*(4*SQRT3 + 7);
    const double w8 = m_dx[2]/288;
    const double w16 = w8*(SQRT3 + 2);
    const double w1 = w8*(-SQRT3 + 2);
    const double w20 = w8*(4*SQRT3 - 7);
    const double w21 = w8*(-4*SQRT3 - 7);
    const double w54 = -m_dx[0]*m_dx[1]/72;
    const double w68 = -m_dx[0]*m_dx[1]/48;
    const double w38 = w68*(-SQRT3 - 3)/36;
    const double w45 = w68*(SQRT3 - 3)/36;
    const double w35 = w68*(5*SQRT3 - 9)/36;
    const double w46 = w68*(-5*SQRT3 - 9)/36;
    const double w43 = w68*(-19*SQRT3 - 33)/36;
    const double w44 = w68*(19*SQRT3 - 33)/36;
    const double w66 = w68*(SQRT3 + 2);
    const double w70 = w68*(-SQRT3 + 2);
    const double w56 = -m_dx[0]*m_dx[2]/72;
    const double w67 = -m_dx[0]*m_dx[2]/48;
    const double w37 = w67*(-SQRT3 - 3)/36;
    const double w40 = w67*(SQRT3 - 3)/36;
    const double w34 = w67*(5*SQRT3 - 9)/36;
    const double w42 = w67*(-5*SQRT3 - 9)/36;
    const double w47 = w67*(19*SQRT3 + 33)/36;
    const double w48 = w67*(-19*SQRT3 + 33)/36;
    const double w65 = w67*(SQRT3 + 2);
    const double w71 = w67*(-SQRT3 + 2);
    const double w55 = -m_dx[1]*m_dx[2]/72;
    const double w69 = -m_dx[1]*m_dx[2]/48;
    const double w36 = w69*(SQRT3 - 3)/36;
    const double w39 = w69*(-SQRT3 - 3)/36;
    const double w33 = w69*(5*SQRT3 - 9)/36;
    const double w41 = w69*(-5*SQRT3 - 9)/36;
    const double w49 = w69*(19*SQRT3 - 33)/36;
    const double w50 = w69*(-19*SQRT3 - 33)/36;
    const double w64 = w69*(SQRT3 + 2);
    const double w72 = w69*(-SQRT3 + 2);
    const double w58 = m_dx[0]*m_dx[1]*m_dx[2]/1728;
    const double w60 = w58*(-SQRT3 + 2);
    const double w61 = w58*(SQRT3 + 2);
    const double w57 = w58*(-4*SQRT3 + 7);
    const double w59 = w58*(4*SQRT3 + 7);
    const double w62 = w58*(15*SQRT3 + 26);
    const double w63 = w58*(-15*SQRT3 + 26);
    const double w75 = w58*6*(SQRT3 + 3);
    const double w76 = w58*6*(-SQRT3 + 3);
    const double w74 = w58*6*(5*SQRT3 + 9);
    const double w77 = w58*6*(-5*SQRT3 + 9);
    const double w13 = -m_dx[0]*m_dx[1]/(288*m_dx[2]);
    const double w19 = w13*(4*SQRT3 + 7);
    const double w7 = w13*(-4*SQRT3 + 7);
    const double w23 = w13*(+SQRT3 - 2);
    const double w25 = w13*(-SQRT3 - 2);
    const double w22 = -m_dx[0]*m_dx[2]/(288*m_dx[1]);
    const double w3 = w22*(SQRT3 - 2);
    const double w9 = w22*(-SQRT3 - 2);
    const double w24 = w22*(4*SQRT3 + 7);
    const double w26 = w22*(-4*SQRT3 + 7);
    const double w27 = -m_dx[1]*m_dx[2]/(288*m_dx[0]);
    const double w0 = w27*(SQRT3 - 2);
    const double w14 = w27*(-SQRT3 - 2);
    const double w28 = w27*(-4*SQRT3 + 7);
    const double w29 = w27*(4*SQRT3 + 7);
    const int NE0 = m_NE[0];
    const int NE1 = m_NE[1];
    const int NE2 = m_NE[2];
    const bool add_EM_S = (!A.isEmpty() || !B.isEmpty() || !C.isEmpty() || !D.isEmpty());
    const bool add_EM_F = (!du.isEmpty() || !Y.isEmpty());
    rhs.requireWrite();

#pragma omp parallel
    {
        std::vector<double> EM_S(8*8*numEq*numComp, 0);
        std::vector<double> EM_F(8*numEq, 0);

        for (index_t k2_0=0; k2_0<2; k2_0++) { // colouring
#pragma omp for
            for (index_t k2=k2_0; k2<NE2; k2+=2) {
                for (index_t k1=0; k1<NE1; ++k1) {
                    for (index_t k0=0; k0<NE0; ++k0)  {
                        const index_t e = k0 + NE0*k1 + NE0*NE1*k2;
                        if (add_EM_S)
                            fill(EM_S.begin(), EM_S.end(), 0);
                        if (add_EM_F)
                            fill(EM_F.begin(), EM_F.end(), 0);

                        ///////////////
                        // process A //
                        ///////////////
                        if (!A.isEmpty()) {
                            const double* A_p = A.getSampleDataRO(e);
                            if (A.actsExpanded()) {
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double A_00_0 = A_p[INDEX5(k,0,m,0,0,numEq,3,numComp,3)];
                                        const double A_01_0 = A_p[INDEX5(k,0,m,1,0,numEq,3,numComp,3)];
                                        const double A_02_0 = A_p[INDEX5(k,0,m,2,0,numEq,3,numComp,3)];
                                        const double A_10_0 = A_p[INDEX5(k,1,m,0,0,numEq,3,numComp,3)];
                                        const double A_11_0 = A_p[INDEX5(k,1,m,1,0,numEq,3,numComp,3)];
                                        const double A_12_0 = A_p[INDEX5(k,1,m,2,0,numEq,3,numComp,3)];
                                        const double A_20_0 = A_p[INDEX5(k,2,m,0,0,numEq,3,numComp,3)];
                                        const double A_21_0 = A_p[INDEX5(k,2,m,1,0,numEq,3,numComp,3)];
                                        const double A_22_0 = A_p[INDEX5(k,2,m,2,0,numEq,3,numComp,3)];
                                        const double A_00_1 = A_p[INDEX5(k,0,m,0,1,numEq,3,numComp,3)];
                                        const double A_01_1 = A_p[INDEX5(k,0,m,1,1,numEq,3,numComp,3)];
                                        const double A_02_1 = A_p[INDEX5(k,0,m,2,1,numEq,3,numComp,3)];
                                        const double A_10_1 = A_p[INDEX5(k,1,m,0,1,numEq,3,numComp,3)];
                                        const double A_11_1 = A_p[INDEX5(k,1,m,1,1,numEq,3,numComp,3)];
                                        const double A_12_1 = A_p[INDEX5(k,1,m,2,1,numEq,3,numComp,3)];
                                        const double A_20_1 = A_p[INDEX5(k,2,m,0,1,numEq,3,numComp,3)];
                                        const double A_21_1 = A_p[INDEX5(k,2,m,1,1,numEq,3,numComp,3)];
                                        const double A_22_1 = A_p[INDEX5(k,2,m,2,1,numEq,3,numComp,3)];
                                        const double A_00_2 = A_p[INDEX5(k,0,m,0,2,numEq,3,numComp,3)];
                                        const double A_01_2 = A_p[INDEX5(k,0,m,1,2,numEq,3,numComp,3)];
                                        const double A_02_2 = A_p[INDEX5(k,0,m,2,2,numEq,3,numComp,3)];
                                        const double A_10_2 = A_p[INDEX5(k,1,m,0,2,numEq,3,numComp,3)];
                                        const double A_11_2 = A_p[INDEX5(k,1,m,1,2,numEq,3,numComp,3)];
                                        const double A_12_2 = A_p[INDEX5(k,1,m,2,2,numEq,3,numComp,3)];
                                        const double A_20_2 = A_p[INDEX5(k,2,m,0,2,numEq,3,numComp,3)];
                                        const double A_21_2 = A_p[INDEX5(k,2,m,1,2,numEq,3,numComp,3)];
                                        const double A_22_2 = A_p[INDEX5(k,2,m,2,2,numEq,3,numComp,3)];
                                        const double A_00_3 = A_p[INDEX5(k,0,m,0,3,numEq,3,numComp,3)];
                                        const double A_01_3 = A_p[INDEX5(k,0,m,1,3,numEq,3,numComp,3)];
                                        const double A_02_3 = A_p[INDEX5(k,0,m,2,3,numEq,3,numComp,3)];
                                        const double A_10_3 = A_p[INDEX5(k,1,m,0,3,numEq,3,numComp,3)];
                                        const double A_11_3 = A_p[INDEX5(k,1,m,1,3,numEq,3,numComp,3)];
                                        const double A_12_3 = A_p[INDEX5(k,1,m,2,3,numEq,3,numComp,3)];
                                        const double A_20_3 = A_p[INDEX5(k,2,m,0,3,numEq,3,numComp,3)];
                                        const double A_21_3 = A_p[INDEX5(k,2,m,1,3,numEq,3,numComp,3)];
                                        const double A_22_3 = A_p[INDEX5(k,2,m,2,3,numEq,3,numComp,3)];
                                        const double A_00_4 = A_p[INDEX5(k,0,m,0,4,numEq,3,numComp,3)];
                                        const double A_01_4 = A_p[INDEX5(k,0,m,1,4,numEq,3,numComp,3)];
                                        const double A_02_4 = A_p[INDEX5(k,0,m,2,4,numEq,3,numComp,3)];
                                        const double A_10_4 = A_p[INDEX5(k,1,m,0,4,numEq,3,numComp,3)];
                                        const double A_11_4 = A_p[INDEX5(k,1,m,1,4,numEq,3,numComp,3)];
                                        const double A_12_4 = A_p[INDEX5(k,1,m,2,4,numEq,3,numComp,3)];
                                        const double A_20_4 = A_p[INDEX5(k,2,m,0,4,numEq,3,numComp,3)];
                                        const double A_21_4 = A_p[INDEX5(k,2,m,1,4,numEq,3,numComp,3)];
                                        const double A_22_4 = A_p[INDEX5(k,2,m,2,4,numEq,3,numComp,3)];
                                        const double A_00_5 = A_p[INDEX5(k,0,m,0,5,numEq,3,numComp,3)];
                                        const double A_01_5 = A_p[INDEX5(k,0,m,1,5,numEq,3,numComp,3)];
                                        const double A_02_5 = A_p[INDEX5(k,0,m,2,5,numEq,3,numComp,3)];
                                        const double A_10_5 = A_p[INDEX5(k,1,m,0,5,numEq,3,numComp,3)];
                                        const double A_11_5 = A_p[INDEX5(k,1,m,1,5,numEq,3,numComp,3)];
                                        const double A_12_5 = A_p[INDEX5(k,1,m,2,5,numEq,3,numComp,3)];
                                        const double A_20_5 = A_p[INDEX5(k,2,m,0,5,numEq,3,numComp,3)];
                                        const double A_21_5 = A_p[INDEX5(k,2,m,1,5,numEq,3,numComp,3)];
                                        const double A_22_5 = A_p[INDEX5(k,2,m,2,5,numEq,3,numComp,3)];
                                        const double A_00_6 = A_p[INDEX5(k,0,m,0,6,numEq,3,numComp,3)];
                                        const double A_01_6 = A_p[INDEX5(k,0,m,1,6,numEq,3,numComp,3)];
                                        const double A_02_6 = A_p[INDEX5(k,0,m,2,6,numEq,3,numComp,3)];
                                        const double A_10_6 = A_p[INDEX5(k,1,m,0,6,numEq,3,numComp,3)];
                                        const double A_11_6 = A_p[INDEX5(k,1,m,1,6,numEq,3,numComp,3)];
                                        const double A_12_6 = A_p[INDEX5(k,1,m,2,6,numEq,3,numComp,3)];
                                        const double A_20_6 = A_p[INDEX5(k,2,m,0,6,numEq,3,numComp,3)];
                                        const double A_21_6 = A_p[INDEX5(k,2,m,1,6,numEq,3,numComp,3)];
                                        const double A_22_6 = A_p[INDEX5(k,2,m,2,6,numEq,3,numComp,3)];
                                        const double A_00_7 = A_p[INDEX5(k,0,m,0,7,numEq,3,numComp,3)];
                                        const double A_01_7 = A_p[INDEX5(k,0,m,1,7,numEq,3,numComp,3)];
                                        const double A_02_7 = A_p[INDEX5(k,0,m,2,7,numEq,3,numComp,3)];
                                        const double A_10_7 = A_p[INDEX5(k,1,m,0,7,numEq,3,numComp,3)];
                                        const double A_11_7 = A_p[INDEX5(k,1,m,1,7,numEq,3,numComp,3)];
                                        const double A_12_7 = A_p[INDEX5(k,1,m,2,7,numEq,3,numComp,3)];
                                        const double A_20_7 = A_p[INDEX5(k,2,m,0,7,numEq,3,numComp,3)];
                                        const double A_21_7 = A_p[INDEX5(k,2,m,1,7,numEq,3,numComp,3)];
                                        const double A_22_7 = A_p[INDEX5(k,2,m,2,7,numEq,3,numComp,3)];
                                        const double tmp0 = w18*(-A_12_7 + A_21_3);
                                        const double tmp1 = w13*(A_22_1 + A_22_2 + A_22_5 + A_22_6);
                                        const double tmp2 = w11*(-A_02_2 - A_02_5 + A_20_1 + A_20_6);
                                        const double tmp3 = w14*(A_00_2 + A_00_3 + A_00_6 + A_00_7);
                                        const double tmp4 = w7*(A_22_0 + A_22_4);
                                        const double tmp5 = w10*(A_12_1 + A_12_6 - A_21_2 - A_21_5);
                                        const double tmp6 = w3*(A_11_0 + A_11_2 + A_11_4 + A_11_6);
                                        const double tmp7 = w1*(A_01_0 + A_01_4 + A_10_0 + A_10_4);
                                        const double tmp8 = w4*(A_12_0 - A_21_4);
                                        const double tmp9 = w15*(-A_02_3 - A_02_6 + A_20_2 + A_20_7);
                                        const double tmp10 = w0*(A_00_0 + A_00_1 + A_00_4 + A_00_5);
                                        const double tmp11 = w16*(A_01_3 + A_01_7 + A_10_3 + A_10_7);
                                        const double tmp12 = w9*(A_11_1 + A_11_3 + A_11_5 + A_11_7);
                                        const double tmp13 = w12*(-A_12_3 - A_12_5 + A_21_1 + A_21_7);
                                        const double tmp14 = w5*(-A_02_1 - A_02_4 + A_20_0 + A_20_5);
                                        const double tmp15 = w8*(A_01_1 + A_01_2 + A_01_5 + A_01_6 + A_10_1 + A_10_2 + A_10_5 + A_10_6);
                                        const double tmp16 = w6*(-A_12_2 - A_12_4 + A_21_0 + A_21_6);
                                        const double tmp17 = w19*(A_22_3 + A_22_7);
                                        const double tmp18 = w17*(-A_02_7 + A_20_3);
                                        const double tmp19 = w2*(A_02_0 - A_20_4);
                                        const double tmp20 = w13*(-A_22_0 - A_22_1 - A_22_2 - A_22_3 - A_22_4 - A_22_5 - A_22_6 - A_22_7);
                                        const double tmp21 = w11*(-A_02_1 - A_02_3 - A_02_4 - A_02_6 + A_20_0 + A_20_2 + A_20_5 + A_20_7);
                                        const double tmp22 = w14*(-A_00_4 - A_00_5 - A_00_6 - A_00_7);
                                        const double tmp23 = w20*(A_01_2 + A_10_1);
                                        const double tmp24 = w10*(A_12_2 + A_12_3 + A_12_4 + A_12_5 - A_21_0 - A_21_1 - A_21_6 - A_21_7);
                                        const double tmp25 = w3*(-A_11_0 - A_11_1 - A_11_2 - A_11_3);
                                        const double tmp26 = w1*(-A_01_0 - A_01_3 - A_10_0 - A_10_3);
                                        const double tmp27 = w15*(-A_02_5 - A_02_7 + A_20_4 + A_20_6);
                                        const double tmp28 = w0*(-A_00_0 - A_00_1 - A_00_2 - A_00_3);
                                        const double tmp29 = w16*(-A_01_4 - A_01_7 - A_10_4 - A_10_7);
                                        const double tmp30 = w9*(-A_11_4 - A_11_5 - A_11_6 - A_11_7);
                                        const double tmp31 = w21*(A_01_5 + A_10_6);
                                        const double tmp32 = w12*(-A_12_6 - A_12_7 + A_21_4 + A_21_5);
                                        const double tmp33 = w5*(-A_02_0 - A_02_2 + A_20_1 + A_20_3);
                                        const double tmp34 = w8*(-A_01_1 - A_01_6 - A_10_2 - A_10_5);
                                        const double tmp35 = w6*(-A_12_0 - A_12_1 + A_21_2 + A_21_3);
                                        const double tmp36 = w20*(-A_01_6 + A_10_4);
                                        const double tmp37 = w18*(A_12_3 - A_21_1);
                                        const double tmp38 = w11*(-A_02_0 - A_02_2 - A_02_5 - A_02_7 - A_20_0 - A_20_2 - A_20_5 - A_20_7);
                                        const double tmp39 = w14*(A_00_0 + A_00_1 + A_00_2 + A_00_3);
                                        const double tmp40 = w26*(A_11_4 + A_11_6);
                                        const double tmp41 = w0*(A_00_4 + A_00_5 + A_00_6 + A_00_7);
                                        const double tmp42 = w10*(-A_12_2 - A_12_5 + A_21_0 + A_21_7);
                                        const double tmp43 = w22*(A_11_0 + A_11_2 + A_11_5 + A_11_7);
                                        const double tmp44 = w1*(A_01_4 + A_01_7 - A_10_5 - A_10_6);
                                        const double tmp45 = w25*(A_22_1 + A_22_3 + A_22_5 + A_22_7);
                                        const double tmp46 = w4*(-A_12_4 + A_21_6);
                                        const double tmp47 = w15*(-A_02_1 - A_02_3 - A_20_1 - A_20_3);
                                        const double tmp48 = w21*(-A_01_1 + A_10_3);
                                        const double tmp49 = w16*(A_01_0 + A_01_3 - A_10_1 - A_10_2);
                                        const double tmp50 = w5*(-A_02_4 - A_02_6 - A_20_4 - A_20_6);
                                        const double tmp51 = w12*(A_12_1 + A_12_7 - A_21_3 - A_21_5);
                                        const double tmp52 = w24*(A_11_1 + A_11_3);
                                        const double tmp53 = w8*(A_01_2 + A_01_5 - A_10_0 - A_10_7);
                                        const double tmp54 = w6*(A_12_0 + A_12_6 - A_21_2 - A_21_4);
                                        const double tmp55 = w23*(A_22_0 + A_22_2 + A_22_4 + A_22_6);
                                        const double tmp56 = w18*(A_12_4 - A_21_6);
                                        const double tmp57 = w14*(A_00_4 + A_00_5 + A_00_6 + A_00_7);
                                        const double tmp58 = w26*(A_11_1 + A_11_3);
                                        const double tmp59 = w20*(-A_01_1 + A_10_3);
                                        const double tmp60 = w1*(A_01_0 + A_01_3 - A_10_1 - A_10_2);
                                        const double tmp61 = w25*(A_22_0 + A_22_2 + A_22_4 + A_22_6);
                                        const double tmp62 = w4*(-A_12_3 + A_21_1);
                                        const double tmp63 = w15*(-A_02_4 - A_02_6 - A_20_4 - A_20_6);
                                        const double tmp64 = w0*(A_00_0 + A_00_1 + A_00_2 + A_00_3);
                                        const double tmp65 = w16*(A_01_4 + A_01_7 - A_10_5 - A_10_6);
                                        const double tmp66 = w24*(A_11_4 + A_11_6);
                                        const double tmp67 = w21*(-A_01_6 + A_10_4);
                                        const double tmp68 = w12*(A_12_0 + A_12_6 - A_21_2 - A_21_4);
                                        const double tmp69 = w5*(-A_02_1 - A_02_3 - A_20_1 - A_20_3);
                                        const double tmp70 = w6*(A_12_1 + A_12_7 - A_21_3 - A_21_5);
                                        const double tmp71 = w23*(A_22_1 + A_22_3 + A_22_5 + A_22_7);
                                        const double tmp72 = w20*(A_01_5 + A_10_6);
                                        const double tmp73 = w14*(-A_00_0 - A_00_1 - A_00_2 - A_00_3);
                                        const double tmp74 = w0*(-A_00_4 - A_00_5 - A_00_6 - A_00_7);
                                        const double tmp75 = w3*(-A_11_4 - A_11_5 - A_11_6 - A_11_7);
                                        const double tmp76 = w1*(-A_01_4 - A_01_7 - A_10_4 - A_10_7);
                                        const double tmp77 = w15*(-A_02_0 - A_02_2 + A_20_1 + A_20_3);
                                        const double tmp78 = w21*(A_01_2 + A_10_1);
                                        const double tmp79 = w16*(-A_01_0 - A_01_3 - A_10_0 - A_10_3);
                                        const double tmp80 = w9*(-A_11_0 - A_11_1 - A_11_2 - A_11_3);
                                        const double tmp81 = w12*(-A_12_0 - A_12_1 + A_21_2 + A_21_3);
                                        const double tmp82 = w5*(-A_02_5 - A_02_7 + A_20_4 + A_20_6);
                                        const double tmp83 = w6*(-A_12_6 - A_12_7 + A_21_4 + A_21_5);
                                        const double tmp84 = w6*(-A_12_2 - A_12_3 - A_21_2 - A_21_3);
                                        const double tmp85 = w11*(A_02_1 + A_02_6 - A_20_0 - A_20_7);
                                        const double tmp86 = w20*(A_01_3 - A_10_2);
                                        const double tmp87 = w10*(A_12_0 + A_12_1 + A_12_6 + A_12_7 + A_21_0 + A_21_1 + A_21_6 + A_21_7);
                                        const double tmp88 = w3*(A_11_0 + A_11_1 + A_11_2 + A_11_3);
                                        const double tmp89 = w23*(A_22_2 + A_22_3 + A_22_6 + A_22_7);
                                        const double tmp90 = w1*(-A_01_1 - A_01_2 + A_10_0 + A_10_3);
                                        const double tmp91 = w25*(A_22_0 + A_22_1 + A_22_4 + A_22_5);
                                        const double tmp92 = w15*(A_02_0 + A_02_5 - A_20_1 - A_20_4);
                                        const double tmp93 = w21*(A_01_4 - A_10_5);
                                        const double tmp94 = w16*(-A_01_5 - A_01_6 + A_10_4 + A_10_7);
                                        const double tmp95 = w28*(A_00_2 + A_00_3);
                                        const double tmp96 = w12*(-A_12_4 - A_12_5 - A_21_4 - A_21_5);
                                        const double tmp97 = w29*(A_00_4 + A_00_5);
                                        const double tmp98 = w5*(A_02_2 + A_02_7 - A_20_3 - A_20_6);
                                        const double tmp99 = w8*(-A_01_0 - A_01_7 + A_10_1 + A_10_6);
                                        const double tmp100 = w9*(A_11_4 + A_11_5 + A_11_6 + A_11_7);
                                        const double tmp101 = w27*(A_00_0 + A_00_1 + A_00_6 + A_00_7);
                                        const double tmp102 = w17*(A_02_4 - A_20_5);
                                        const double tmp103 = w2*(-A_02_3 + A_20_2);
                                        const double tmp104 = w13*(A_22_0 + A_22_1 + A_22_2 + A_22_3 + A_22_4 + A_22_5 + A_22_6 + A_22_7);
                                        const double tmp105 = w6*(-A_12_4 - A_12_5 - A_21_2 - A_21_3);
                                        const double tmp106 = w22*(A_11_0 + A_11_1 + A_11_2 + A_11_3 + A_11_4 + A_11_5 + A_11_6 + A_11_7);
                                        const double tmp107 = w1*(-A_01_2 - A_01_6 - A_10_1 - A_10_5);
                                        const double tmp108 = w15*(-A_02_1 - A_02_3 - A_20_4 - A_20_6);
                                        const double tmp109 = w16*(-A_01_1 - A_01_5 - A_10_2 - A_10_6);
                                        const double tmp110 = w12*(-A_12_2 - A_12_3 - A_21_4 - A_21_5);
                                        const double tmp111 = w5*(-A_02_4 - A_02_6 - A_20_1 - A_20_3);
                                        const double tmp112 = w8*(-A_01_0 - A_01_3 - A_01_4 - A_01_7 - A_10_0 - A_10_3 - A_10_4 - A_10_7);
                                        const double tmp113 = w27*(A_00_0 + A_00_1 + A_00_2 + A_00_3 + A_00_4 + A_00_5 + A_00_6 + A_00_7);
                                        const double tmp114 = w11*(A_02_0 + A_02_2 + A_02_5 + A_02_7 - A_20_1 - A_20_3 - A_20_4 - A_20_6);
                                        const double tmp115 = w21*(-A_01_4 - A_10_7);
                                        const double tmp116 = w20*(-A_01_3 - A_10_0);
                                        const double tmp117 = w15*(A_02_4 + A_02_6 - A_20_5 - A_20_7);
                                        const double tmp118 = w16*(A_01_5 + A_01_6 + A_10_5 + A_10_6);
                                        const double tmp119 = w5*(A_02_1 + A_02_3 - A_20_0 - A_20_2);
                                        const double tmp120 = w8*(A_01_0 + A_01_7 + A_10_3 + A_10_4);
                                        const double tmp121 = w1*(A_01_1 + A_01_2 + A_10_1 + A_10_2);
                                        const double tmp122 = w18*(A_12_2 - A_21_6);
                                        const double tmp123 = w13*(A_22_0 + A_22_3 + A_22_4 + A_22_7);
                                        const double tmp124 = w11*(-A_02_0 - A_02_7 + A_20_3 + A_20_4);
                                        const double tmp125 = w7*(A_22_1 + A_22_5);
                                        const double tmp126 = w10*(-A_12_3 - A_12_4 + A_21_0 + A_21_7);
                                        const double tmp127 = w3*(A_11_1 + A_11_3 + A_11_5 + A_11_7);
                                        const double tmp128 = w1*(-A_01_1 - A_01_5 - A_10_1 - A_10_5);
                                        const double tmp129 = w4*(-A_12_5 + A_21_1);
                                        const double tmp130 = w16*(-A_01_2 - A_01_6 - A_10_2 - A_10_6);
                                        const double tmp131 = w9*(A_11_0 + A_11_2 + A_11_4 + A_11_6);
                                        const double tmp132 = w19*(A_22_2 + A_22_6);
                                        const double tmp133 = w17*(-A_02_2 + A_20_6);
                                        const double tmp134 = w2*(A_02_5 - A_20_1);
                                        const double tmp135 = w11*(A_02_1 + A_02_3 + A_02_4 + A_02_6 + A_20_1 + A_20_3 + A_20_4 + A_20_6);
                                        const double tmp136 = w1*(A_01_3 + A_01_7 + A_10_0 + A_10_4);
                                        const double tmp137 = w15*(A_02_0 + A_02_2 + A_20_5 + A_20_7);
                                        const double tmp138 = w16*(A_01_0 + A_01_4 + A_10_3 + A_10_7);
                                        const double tmp139 = w5*(A_02_5 + A_02_7 + A_20_0 + A_20_2);
                                        const double tmp140 = w18*(A_12_5 - A_21_1);
                                        const double tmp141 = w14*(A_00_0 + A_00_1 + A_00_4 + A_00_5);
                                        const double tmp142 = w7*(A_22_2 + A_22_6);
                                        const double tmp143 = w1*(-A_01_2 - A_01_6 - A_10_2 - A_10_6);
                                        const double tmp144 = w4*(-A_12_2 + A_21_6);
                                        const double tmp145 = w15*(-A_02_1 - A_02_4 + A_20_0 + A_20_5);
                                        const double tmp146 = w0*(A_00_2 + A_00_3 + A_00_6 + A_00_7);
                                        const double tmp147 = w16*(-A_01_1 - A_01_5 - A_10_1 - A_10_5);
                                        const double tmp148 = w5*(-A_02_3 - A_02_6 + A_20_2 + A_20_7);
                                        const double tmp149 = w19*(A_22_1 + A_22_5);
                                        const double tmp150 = w17*(-A_02_5 + A_20_1);
                                        const double tmp151 = w2*(A_02_2 - A_20_6);
                                        const double tmp152 = w18*(A_12_3 - A_21_7);
                                        const double tmp153 = w11*(A_02_1 + A_02_6 - A_20_2 - A_20_5);
                                        const double tmp154 = w10*(-A_12_2 - A_12_5 + A_21_1 + A_21_6);
                                        const double tmp155 = w4*(-A_12_4 + A_21_0);
                                        const double tmp156 = w15*(A_02_2 + A_02_7 - A_20_3 - A_20_6);
                                        const double tmp157 = w5*(A_02_0 + A_02_5 - A_20_1 - A_20_4);
                                        const double tmp158 = w17*(A_02_3 - A_20_7);
                                        const double tmp159 = w2*(-A_02_4 + A_20_0);
                                        const double tmp160 = w6*(A_12_6 + A_12_7 + A_21_0 + A_21_1);
                                        const double tmp161 = w10*(-A_12_2 - A_12_3 - A_12_4 - A_12_5 - A_21_2 - A_21_3 - A_21_4 - A_21_5);
                                        const double tmp162 = w1*(A_01_0 + A_01_4 + A_10_3 + A_10_7);
                                        const double tmp163 = w16*(A_01_3 + A_01_7 + A_10_0 + A_10_4);
                                        const double tmp164 = w12*(A_12_0 + A_12_1 + A_21_6 + A_21_7);
                                        const double tmp165 = w20*(A_01_6 + A_10_5);
                                        const double tmp166 = w10*(-A_12_0 - A_12_1 - A_12_6 - A_12_7 + A_21_2 + A_21_3 + A_21_4 + A_21_5);
                                        const double tmp167 = w15*(A_02_1 + A_02_3 - A_20_0 - A_20_2);
                                        const double tmp168 = w21*(A_01_1 + A_10_2);
                                        const double tmp169 = w12*(A_12_2 + A_12_3 - A_21_0 - A_21_1);
                                        const double tmp170 = w5*(A_02_4 + A_02_6 - A_20_5 - A_20_7);
                                        const double tmp171 = w8*(-A_01_2 - A_01_5 - A_10_1 - A_10_6);
                                        const double tmp172 = w6*(A_12_4 + A_12_5 - A_21_6 - A_21_7);
                                        const double tmp173 = w2*(A_02_1 + A_20_4);
                                        const double tmp174 = w11*(-A_02_3 - A_02_4 - A_20_1 - A_20_6);
                                        const double tmp175 = w14*(-A_00_2 - A_00_3 - A_00_6 - A_00_7);
                                        const double tmp176 = w22*(-A_11_0 - A_11_1 - A_11_2 - A_11_3 - A_11_4 - A_11_5 - A_11_6 - A_11_7);
                                        const double tmp177 = w1*(A_01_1 + A_01_5 - A_10_0 - A_10_4);
                                        const double tmp178 = w25*(-A_22_2 - A_22_3 - A_22_6 - A_22_7);
                                        const double tmp179 = w15*(-A_02_2 - A_02_7 - A_20_2 - A_20_7);
                                        const double tmp180 = w0*(-A_00_0 - A_00_1 - A_00_4 - A_00_5);
                                        const double tmp181 = w16*(A_01_2 + A_01_6 - A_10_3 - A_10_7);
                                        const double tmp182 = w12*(-A_12_6 - A_12_7 + A_21_2 + A_21_3);
                                        const double tmp183 = w5*(-A_02_0 - A_02_5 - A_20_0 - A_20_5);
                                        const double tmp184 = w8*(A_01_0 + A_01_3 + A_01_4 + A_01_7 - A_10_1 - A_10_2 - A_10_5 - A_10_6);
                                        const double tmp185 = w6*(-A_12_0 - A_12_1 + A_21_4 + A_21_5);
                                        const double tmp186 = w17*(-A_02_6 - A_20_3);
                                        const double tmp187 = w23*(-A_22_0 - A_22_1 - A_22_4 - A_22_5);
                                        const double tmp188 = w18*(A_12_4 - A_21_0);
                                        const double tmp189 = w7*(A_22_3 + A_22_7);
                                        const double tmp190 = w1*(A_01_3 + A_01_7 + A_10_3 + A_10_7);
                                        const double tmp191 = w4*(-A_12_3 + A_21_7);
                                        const double tmp192 = w16*(A_01_0 + A_01_4 + A_10_0 + A_10_4);
                                        const double tmp193 = w19*(A_22_0 + A_22_4);
                                        const double tmp194 = w17*(A_02_4 - A_20_0);
                                        const double tmp195 = w2*(-A_02_3 + A_20_7);
                                        const double tmp196 = w20*(-A_01_7 - A_10_4);
                                        const double tmp197 = w21*(-A_01_0 - A_10_3);
                                        const double tmp198 = w16*(A_01_1 + A_01_2 + A_10_1 + A_10_2);
                                        const double tmp199 = w8*(A_01_3 + A_01_4 + A_10_0 + A_10_7);
                                        const double tmp200 = w1*(A_01_5 + A_01_6 + A_10_5 + A_10_6);
                                        const double tmp201 = w27*(A_00_2 + A_00_3 + A_00_4 + A_00_5);
                                        const double tmp202 = w11*(-A_02_2 - A_02_5 + A_20_3 + A_20_4);
                                        const double tmp203 = w20*(A_01_0 - A_10_1);
                                        const double tmp204 = w23*(A_22_0 + A_22_1 + A_22_4 + A_22_5);
                                        const double tmp205 = w25*(A_22_2 + A_22_3 + A_22_6 + A_22_7);
                                        const double tmp206 = w21*(A_01_7 - A_10_6);
                                        const double tmp207 = w12*(A_12_6 + A_12_7 + A_21_6 + A_21_7);
                                        const double tmp208 = w28*(A_00_0 + A_00_1);
                                        const double tmp209 = w29*(A_00_6 + A_00_7);
                                        const double tmp210 = w8*(-A_01_3 - A_01_4 + A_10_2 + A_10_5);
                                        const double tmp211 = w6*(A_12_0 + A_12_1 + A_21_0 + A_21_1);
                                        const double tmp212 = w17*(-A_02_7 + A_20_6);
                                        const double tmp213 = w2*(A_02_0 - A_20_1);
                                        const double tmp214 = w13*(-A_22_1 - A_22_2 - A_22_5 - A_22_6);
                                        const double tmp215 = w22*(-A_11_0 - A_11_2 - A_11_5 - A_11_7);
                                        const double tmp216 = w8*(A_01_0 + A_01_7 + A_10_0 + A_10_7);
                                        const double tmp217 = w27*(-A_00_0 - A_00_1 - A_00_6 - A_00_7);
                                        const double tmp218 = w17*(-A_02_3 - A_20_3);
                                        const double tmp219 = w2*(A_02_4 + A_20_4);
                                        const double tmp220 = w11*(-A_02_1 - A_02_6 - A_20_1 - A_20_6);
                                        const double tmp221 = w26*(-A_11_4 - A_11_6);
                                        const double tmp222 = w10*(A_12_2 + A_12_5 + A_21_2 + A_21_5);
                                        const double tmp223 = w20*(-A_01_4 - A_10_4);
                                        const double tmp224 = w21*(-A_01_3 - A_10_3);
                                        const double tmp225 = w6*(-A_12_0 - A_12_6 - A_21_0 - A_21_6);
                                        const double tmp226 = w7*(-A_22_0 - A_22_4);
                                        const double tmp227 = w24*(-A_11_1 - A_11_3);
                                        const double tmp228 = w19*(-A_22_3 - A_22_7);
                                        const double tmp229 = w18*(-A_12_3 - A_21_3);
                                        const double tmp230 = w4*(A_12_4 + A_21_4);
                                        const double tmp231 = w28*(-A_00_4 - A_00_5);
                                        const double tmp232 = w12*(-A_12_1 - A_12_7 - A_21_1 - A_21_7);
                                        const double tmp233 = w29*(-A_00_2 - A_00_3);
                                        const double tmp234 = w20*(-A_01_5 + A_10_7);
                                        const double tmp235 = w18*(-A_12_0 + A_21_2);
                                        const double tmp236 = w26*(A_11_5 + A_11_7);
                                        const double tmp237 = w10*(A_12_1 + A_12_6 - A_21_3 - A_21_4);
                                        const double tmp238 = w22*(A_11_1 + A_11_3 + A_11_4 + A_11_6);
                                        const double tmp239 = w4*(A_12_7 - A_21_5);
                                        const double tmp240 = w15*(A_02_0 + A_02_2 + A_20_0 + A_20_2);
                                        const double tmp241 = w21*(-A_01_2 + A_10_0);
                                        const double tmp242 = w5*(A_02_5 + A_02_7 + A_20_5 + A_20_7);
                                        const double tmp243 = w12*(-A_12_2 - A_12_4 + A_21_0 + A_21_6);
                                        const double tmp244 = w24*(A_11_0 + A_11_2);
                                        const double tmp245 = w8*(A_01_1 + A_01_6 - A_10_3 - A_10_4);
                                        const double tmp246 = w6*(-A_12_3 - A_12_5 + A_21_1 + A_21_7);
                                        const double tmp247 = w11*(A_02_3 + A_02_4 - A_20_2 - A_20_5);
                                        const double tmp248 = w20*(-A_01_1 + A_10_0);
                                        const double tmp249 = w21*(-A_01_6 + A_10_7);
                                        const double tmp250 = w8*(A_01_2 + A_01_5 - A_10_3 - A_10_4);
                                        const double tmp251 = w17*(A_02_6 - A_20_7);
                                        const double tmp252 = w2*(-A_02_1 + A_20_0);
                                        const double tmp253 = w17*(-A_02_4 - A_20_4);
                                        const double tmp254 = w2*(A_02_3 + A_20_3);
                                        const double tmp255 = w26*(-A_11_1 - A_11_3);
                                        const double tmp256 = w20*(-A_01_3 - A_10_3);
                                        const double tmp257 = w21*(-A_01_4 - A_10_4);
                                        const double tmp258 = w6*(-A_12_1 - A_12_7 - A_21_1 - A_21_7);
                                        const double tmp259 = w7*(-A_22_3 - A_22_7);
                                        const double tmp260 = w15*(-A_02_0 - A_02_5 - A_20_0 - A_20_5);
                                        const double tmp261 = w24*(-A_11_4 - A_11_6);
                                        const double tmp262 = w19*(-A_22_0 - A_22_4);
                                        const double tmp263 = w18*(-A_12_4 - A_21_4);
                                        const double tmp264 = w4*(A_12_3 + A_21_3);
                                        const double tmp265 = w28*(-A_00_2 - A_00_3);
                                        const double tmp266 = w12*(-A_12_0 - A_12_6 - A_21_0 - A_21_6);
                                        const double tmp267 = w5*(-A_02_2 - A_02_7 - A_20_2 - A_20_7);
                                        const double tmp268 = w29*(-A_00_4 - A_00_5);
                                        const double tmp269 = w11*(A_02_2 + A_02_5 + A_20_0 + A_20_7);
                                        const double tmp270 = w1*(-A_01_0 - A_01_4 + A_10_1 + A_10_5);
                                        const double tmp271 = w15*(A_02_3 + A_02_6 + A_20_3 + A_20_6);
                                        const double tmp272 = w16*(-A_01_3 - A_01_7 + A_10_2 + A_10_6);
                                        const double tmp273 = w5*(A_02_1 + A_02_4 + A_20_1 + A_20_4);
                                        const double tmp274 = w8*(-A_01_1 - A_01_2 - A_01_5 - A_01_6 + A_10_0 + A_10_3 + A_10_4 + A_10_7);
                                        const double tmp275 = w17*(A_02_7 + A_20_2);
                                        const double tmp276 = w2*(-A_02_0 - A_20_5);
                                        const double tmp277 = w18*(-A_12_1 + A_21_5);
                                        const double tmp278 = w11*(A_02_3 + A_02_4 - A_20_0 - A_20_7);
                                        const double tmp279 = w10*(A_12_0 + A_12_7 - A_21_3 - A_21_4);
                                        const double tmp280 = w4*(A_12_6 - A_21_2);
                                        const double tmp281 = w17*(A_02_1 - A_20_5);
                                        const double tmp282 = w2*(-A_02_6 + A_20_2);
                                        const double tmp283 = w11*(A_02_0 + A_02_7 + A_20_2 + A_20_5);
                                        const double tmp284 = w12*(A_12_2 + A_12_3 - A_21_6 - A_21_7);
                                        const double tmp285 = w6*(A_12_4 + A_12_5 - A_21_0 - A_21_1);
                                        const double tmp286 = w17*(A_02_2 + A_20_7);
                                        const double tmp287 = w2*(-A_02_5 - A_20_0);
                                        const double tmp288 = w13*(-A_22_0 - A_22_3 - A_22_4 - A_22_7);
                                        const double tmp289 = w22*(-A_11_1 - A_11_3 - A_11_4 - A_11_6);
                                        const double tmp290 = w8*(-A_01_1 - A_01_6 - A_10_1 - A_10_6);
                                        const double tmp291 = w17*(A_02_2 + A_20_2);
                                        const double tmp292 = w2*(-A_02_5 - A_20_5);
                                        const double tmp293 = w11*(A_02_0 + A_02_7 + A_20_0 + A_20_7);
                                        const double tmp294 = w26*(-A_11_5 - A_11_7);
                                        const double tmp295 = w10*(A_12_3 + A_12_4 + A_21_3 + A_21_4);
                                        const double tmp296 = w20*(A_01_5 + A_10_5);
                                        const double tmp297 = w21*(A_01_2 + A_10_2);
                                        const double tmp298 = w7*(-A_22_1 - A_22_5);
                                        const double tmp299 = w24*(-A_11_0 - A_11_2);
                                        const double tmp300 = w19*(-A_22_2 - A_22_6);
                                        const double tmp301 = w18*(-A_12_2 - A_21_2);
                                        const double tmp302 = w4*(A_12_5 + A_21_5);
                                        const double tmp303 = w8*(A_01_3 + A_01_4 + A_10_3 + A_10_4);
                                        const double tmp304 = w27*(-A_00_2 - A_00_3 - A_00_4 - A_00_5);
                                        const double tmp305 = w17*(A_02_7 + A_20_7);
                                        const double tmp306 = w2*(-A_02_0 - A_20_0);
                                        const double tmp307 = w11*(A_02_2 + A_02_5 + A_20_2 + A_20_5);
                                        const double tmp308 = w26*(-A_11_0 - A_11_2);
                                        const double tmp309 = w10*(-A_12_1 - A_12_6 - A_21_1 - A_21_6);
                                        const double tmp310 = w20*(-A_01_0 - A_10_0);
                                        const double tmp311 = w21*(-A_01_7 - A_10_7);
                                        const double tmp312 = w6*(A_12_2 + A_12_4 + A_21_2 + A_21_4);
                                        const double tmp313 = w24*(-A_11_5 - A_11_7);
                                        const double tmp314 = w18*(A_12_7 + A_21_7);
                                        const double tmp315 = w4*(-A_12_0 - A_21_0);
                                        const double tmp316 = w28*(-A_00_0 - A_00_1);
                                        const double tmp317 = w12*(A_12_3 + A_12_5 + A_21_3 + A_21_5);
                                        const double tmp318 = w29*(-A_00_6 - A_00_7);
                                        const double tmp319 = w18*(-A_12_7 + A_21_5);
                                        const double tmp320 = w26*(A_11_0 + A_11_2);
                                        const double tmp321 = w21*(-A_01_5 + A_10_7);
                                        const double tmp322 = w20*(-A_01_2 + A_10_0);
                                        const double tmp323 = w4*(A_12_0 - A_21_2);
                                        const double tmp324 = w15*(A_02_5 + A_02_7 + A_20_5 + A_20_7);
                                        const double tmp325 = w24*(A_11_5 + A_11_7);
                                        const double tmp326 = w5*(A_02_0 + A_02_2 + A_20_0 + A_20_2);
                                        const double tmp327 = w18*(A_12_7 + A_21_1);
                                        const double tmp328 = w10*(-A_12_1 - A_12_6 - A_21_0 - A_21_7);
                                        const double tmp329 = w3*(-A_11_0 - A_11_2 - A_11_4 - A_11_6);
                                        const double tmp330 = w1*(A_01_2 + A_01_6 - A_10_0 - A_10_4);
                                        const double tmp331 = w4*(-A_12_0 - A_21_6);
                                        const double tmp332 = w25*(-A_22_1 - A_22_3 - A_22_5 - A_22_7);
                                        const double tmp333 = w15*(-A_02_5 - A_02_7 + A_20_1 + A_20_3);
                                        const double tmp334 = w16*(A_01_1 + A_01_5 - A_10_3 - A_10_7);
                                        const double tmp335 = w9*(-A_11_1 - A_11_3 - A_11_5 - A_11_7);
                                        const double tmp336 = w5*(-A_02_0 - A_02_2 + A_20_4 + A_20_6);
                                        const double tmp337 = w27*(-A_00_0 - A_00_1 - A_00_2 - A_00_3 - A_00_4 - A_00_5 - A_00_6 - A_00_7);
                                        const double tmp338 = w23*(-A_22_0 - A_22_2 - A_22_4 - A_22_6);
                                        const double tmp339 = w14*(-A_00_0 - A_00_1 - A_00_4 - A_00_5);
                                        const double tmp340 = w23*(-A_22_2 - A_22_3 - A_22_6 - A_22_7);
                                        const double tmp341 = w1*(A_01_2 + A_01_6 - A_10_3 - A_10_7);
                                        const double tmp342 = w25*(-A_22_0 - A_22_1 - A_22_4 - A_22_5);
                                        const double tmp343 = w15*(A_02_1 + A_02_4 + A_20_1 + A_20_4);
                                        const double tmp344 = w0*(-A_00_2 - A_00_3 - A_00_6 - A_00_7);
                                        const double tmp345 = w16*(A_01_1 + A_01_5 - A_10_0 - A_10_4);
                                        const double tmp346 = w12*(A_12_4 + A_12_5 - A_21_0 - A_21_1);
                                        const double tmp347 = w5*(A_02_3 + A_02_6 + A_20_3 + A_20_6);
                                        const double tmp348 = w6*(A_12_2 + A_12_3 - A_21_6 - A_21_7);
                                        const double tmp349 = w17*(A_02_5 + A_20_0);
                                        const double tmp350 = w2*(-A_02_2 - A_20_7);
                                        const double tmp351 = w8*(-A_01_2 - A_01_5 - A_10_2 - A_10_5);
                                        const double tmp352 = w17*(-A_02_1 - A_20_1);
                                        const double tmp353 = w2*(A_02_6 + A_20_6);
                                        const double tmp354 = w11*(-A_02_3 - A_02_4 - A_20_3 - A_20_4);
                                        const double tmp355 = w10*(-A_12_0 - A_12_7 - A_21_0 - A_21_7);
                                        const double tmp356 = w20*(A_01_6 + A_10_6);
                                        const double tmp357 = w21*(A_01_1 + A_10_1);
                                        const double tmp358 = w7*(-A_22_2 - A_22_6);
                                        const double tmp359 = w19*(-A_22_1 - A_22_5);
                                        const double tmp360 = w18*(A_12_1 + A_21_1);
                                        const double tmp361 = w4*(-A_12_6 - A_21_6);
                                        const double tmp362 = w28*(-A_00_6 - A_00_7);
                                        const double tmp363 = w29*(-A_00_0 - A_00_1);
                                        const double tmp364 = w2*(A_02_4 + A_20_1);
                                        const double tmp365 = w11*(-A_02_1 - A_02_6 - A_20_3 - A_20_4);
                                        const double tmp366 = w17*(-A_02_3 - A_20_6);
                                        const double tmp367 = w2*(A_02_5 - A_20_4);
                                        const double tmp368 = w6*(-A_12_4 - A_12_5 - A_21_4 - A_21_5);
                                        const double tmp369 = w11*(-A_02_0 - A_02_7 + A_20_1 + A_20_6);
                                        const double tmp370 = w20*(-A_01_5 + A_10_4);
                                        const double tmp371 = w3*(A_11_4 + A_11_5 + A_11_6 + A_11_7);
                                        const double tmp372 = w12*(-A_12_2 - A_12_3 - A_21_2 - A_21_3);
                                        const double tmp373 = w21*(-A_01_2 + A_10_3);
                                        const double tmp374 = w9*(A_11_0 + A_11_1 + A_11_2 + A_11_3);
                                        const double tmp375 = w29*(A_00_2 + A_00_3);
                                        const double tmp376 = w8*(A_01_1 + A_01_6 - A_10_0 - A_10_7);
                                        const double tmp377 = w28*(A_00_4 + A_00_5);
                                        const double tmp378 = w17*(-A_02_2 + A_20_3);
                                        const double tmp379 = w17*(A_02_0 + A_20_0);
                                        const double tmp380 = w2*(-A_02_7 - A_20_7);
                                        const double tmp381 = w20*(-A_01_7 - A_10_7);
                                        const double tmp382 = w21*(-A_01_0 - A_10_0);
                                        const double tmp383 = w6*(A_12_3 + A_12_5 + A_21_3 + A_21_5);
                                        const double tmp384 = w18*(A_12_0 + A_21_0);
                                        const double tmp385 = w4*(-A_12_7 - A_21_7);
                                        const double tmp386 = w12*(A_12_2 + A_12_4 + A_21_2 + A_21_4);
                                        const double tmp387 = w17*(-A_02_6 - A_20_6);
                                        const double tmp388 = w2*(A_02_1 + A_20_1);
                                        const double tmp389 = w20*(A_01_1 + A_10_1);
                                        const double tmp390 = w21*(A_01_6 + A_10_6);
                                        const double tmp391 = w18*(A_12_6 + A_21_6);
                                        const double tmp392 = w4*(-A_12_1 - A_21_1);
                                        const double tmp393 = w2*(A_02_3 + A_20_6);
                                        const double tmp394 = w1*(-A_01_3 - A_01_7 + A_10_2 + A_10_6);
                                        const double tmp395 = w16*(-A_01_0 - A_01_4 + A_10_1 + A_10_5);
                                        const double tmp396 = w17*(-A_02_4 - A_20_1);
                                        const double tmp397 = w18*(-A_12_5 - A_21_3);
                                        const double tmp398 = w10*(A_12_3 + A_12_4 + A_21_2 + A_21_5);
                                        const double tmp399 = w1*(-A_01_0 - A_01_4 + A_10_2 + A_10_6);
                                        const double tmp400 = w4*(A_12_2 + A_21_4);
                                        const double tmp401 = w16*(-A_01_3 - A_01_7 + A_10_1 + A_10_5);
                                        const double tmp402 = w20*(-A_01_2 + A_10_3);
                                        const double tmp403 = w21*(-A_01_5 + A_10_4);
                                        const double tmp404 = w17*(-A_02_5 + A_20_4);
                                        const double tmp405 = w2*(A_02_2 - A_20_3);
                                        const double tmp406 = w18*(-A_12_0 + A_21_4);
                                        const double tmp407 = w4*(A_12_7 - A_21_3);
                                        const double tmp408 = w17*(-A_02_0 + A_20_4);
                                        const double tmp409 = w2*(A_02_7 - A_20_3);
                                        const double tmp410 = w17*(A_02_5 + A_20_5);
                                        const double tmp411 = w2*(-A_02_2 - A_20_2);
                                        const double tmp412 = w20*(A_01_2 + A_10_2);
                                        const double tmp413 = w21*(A_01_5 + A_10_5);
                                        const double tmp414 = w18*(-A_12_5 - A_21_5);
                                        const double tmp415 = w4*(A_12_2 + A_21_2);
                                        const double tmp416 = w12*(-A_12_0 - A_12_1 + A_21_4 + A_21_5);
                                        const double tmp417 = w6*(-A_12_6 - A_12_7 + A_21_2 + A_21_3);
                                        const double tmp418 = w17*(A_02_0 + A_20_5);
                                        const double tmp419 = w2*(-A_02_7 - A_20_2);
                                        const double tmp420 = w18*(-A_12_4 - A_21_2);
                                        const double tmp421 = w10*(A_12_2 + A_12_5 + A_21_3 + A_21_4);
                                        const double tmp422 = w3*(-A_11_1 - A_11_3 - A_11_5 - A_11_7);
                                        const double tmp423 = w1*(A_01_1 + A_01_5 - A_10_3 - A_10_7);
                                        const double tmp424 = w25*(-A_22_0 - A_22_2 - A_22_4 - A_22_6);
                                        const double tmp425 = w4*(A_12_3 + A_21_5);
                                        const double tmp426 = w15*(A_02_4 + A_02_6 - A_20_0 - A_20_2);
                                        const double tmp427 = w16*(A_01_2 + A_01_6 - A_10_0 - A_10_4);
                                        const double tmp428 = w9*(-A_11_0 - A_11_2 - A_11_4 - A_11_6);
                                        const double tmp429 = w5*(A_02_1 + A_02_3 - A_20_5 - A_20_7);
                                        const double tmp430 = w23*(-A_22_1 - A_22_3 - A_22_5 - A_22_7);
                                        const double tmp431 = w18*(A_12_5 - A_21_7);
                                        const double tmp432 = w10*(-A_12_3 - A_12_4 + A_21_1 + A_21_6);
                                        const double tmp433 = w21*(A_01_7 - A_10_5);
                                        const double tmp434 = w20*(A_01_0 - A_10_2);
                                        const double tmp435 = w4*(-A_12_2 + A_21_0);
                                        const double tmp436 = w8*(-A_01_3 - A_01_4 + A_10_1 + A_10_6);
                                        const double tmp437 = w2*(-A_02_4 + A_20_5);
                                        const double tmp438 = w20*(A_01_4 - A_10_5);
                                        const double tmp439 = w21*(A_01_3 - A_10_2);
                                        const double tmp440 = w16*(-A_01_1 - A_01_2 + A_10_0 + A_10_3);
                                        const double tmp441 = w1*(-A_01_5 - A_01_6 + A_10_4 + A_10_7);
                                        const double tmp442 = w17*(A_02_3 - A_20_2);
                                        const double tmp443 = w20*(-A_01_4 - A_10_7);
                                        const double tmp444 = w21*(-A_01_3 - A_10_0);
                                        const double tmp445 = w18*(A_12_6 + A_21_0);
                                        const double tmp446 = w10*(-A_12_0 - A_12_7 - A_21_1 - A_21_6);
                                        const double tmp447 = w1*(-A_01_3 - A_01_7 + A_10_1 + A_10_5);
                                        const double tmp448 = w4*(-A_12_1 - A_21_7);
                                        const double tmp449 = w16*(-A_01_0 - A_01_4 + A_10_2 + A_10_6);
                                        const double tmp450 = w2*(A_02_7 - A_20_6);
                                        const double tmp451 = w6*(A_12_6 + A_12_7 + A_21_6 + A_21_7);
                                        const double tmp452 = w20*(A_01_7 - A_10_6);
                                        const double tmp453 = w21*(A_01_0 - A_10_1);
                                        const double tmp454 = w12*(A_12_0 + A_12_1 + A_21_0 + A_21_1);
                                        const double tmp455 = w29*(A_00_0 + A_00_1);
                                        const double tmp456 = w28*(A_00_6 + A_00_7);
                                        const double tmp457 = w17*(-A_02_0 + A_20_1);
                                        const double tmp458 = w21*(-A_01_7 - A_10_4);
                                        const double tmp459 = w20*(-A_01_0 - A_10_3);
                                        const double tmp460 = w12*(A_12_4 + A_12_5 - A_21_6 - A_21_7);
                                        const double tmp461 = w6*(A_12_2 + A_12_3 - A_21_0 - A_21_1);
                                        const double tmp462 = w18*(A_12_1 + A_21_7);
                                        const double tmp463 = w4*(-A_12_6 - A_21_0);
                                        const double tmp464 = w15*(A_02_1 + A_02_3 - A_20_5 - A_20_7);
                                        const double tmp465 = w5*(A_02_4 + A_02_6 - A_20_0 - A_20_2);
                                        const double tmp466 = w2*(-A_02_6 + A_20_7);
                                        const double tmp467 = w20*(-A_01_6 + A_10_7);
                                        const double tmp468 = w21*(-A_01_1 + A_10_0);
                                        const double tmp469 = w17*(A_02_1 - A_20_0);
                                        const double tmp470 = w6*(-A_12_2 - A_12_3 - A_21_4 - A_21_5);
                                        const double tmp471 = w1*(-A_01_1 - A_01_5 - A_10_2 - A_10_6);
                                        const double tmp472 = w15*(-A_02_4 - A_02_6 - A_20_1 - A_20_3);
                                        const double tmp473 = w16*(-A_01_2 - A_01_6 - A_10_1 - A_10_5);
                                        const double tmp474 = w12*(-A_12_4 - A_12_5 - A_21_2 - A_21_3);
                                        const double tmp475 = w5*(-A_02_1 - A_02_3 - A_20_4 - A_20_6);
                                        const double tmp476 = w18*(-A_12_6 + A_21_4);
                                        const double tmp477 = w20*(A_01_3 - A_10_1);
                                        const double tmp478 = w10*(A_12_0 + A_12_7 - A_21_2 - A_21_5);
                                        const double tmp479 = w4*(A_12_1 - A_21_3);
                                        const double tmp480 = w21*(A_01_4 - A_10_6);
                                        const double tmp481 = w8*(-A_01_0 - A_01_7 + A_10_2 + A_10_5);
                                        const double tmp482 = w6*(A_12_0 + A_12_1 + A_21_6 + A_21_7);
                                        const double tmp483 = w12*(A_12_6 + A_12_7 + A_21_0 + A_21_1);
                                        const double tmp484 = w15*(A_02_5 + A_02_7 + A_20_0 + A_20_2);
                                        const double tmp485 = w5*(A_02_0 + A_02_2 + A_20_5 + A_20_7);
                                        const double tmp486 = w18*(-A_12_1 + A_21_3);
                                        const double tmp487 = w20*(A_01_4 - A_10_6);
                                        const double tmp488 = w4*(A_12_6 - A_21_4);
                                        const double tmp489 = w21*(A_01_3 - A_10_1);
                                        const double tmp490 = w20*(A_01_7 - A_10_5);
                                        const double tmp491 = w18*(A_12_2 - A_21_0);
                                        const double tmp492 = w4*(-A_12_5 + A_21_7);
                                        const double tmp493 = w21*(A_01_0 - A_10_2);
                                        const double tmp494 = w20*(A_01_1 + A_10_2);
                                        const double tmp495 = w21*(A_01_6 + A_10_5);
                                        const double tmp496 = w18*(-A_12_2 - A_21_4);
                                        const double tmp497 = w4*(A_12_5 + A_21_3);
                                        const double tmp498 = w15*(-A_02_0 - A_02_2 + A_20_4 + A_20_6);
                                        const double tmp499 = w5*(-A_02_5 - A_02_7 + A_20_1 + A_20_3);
                                        const double tmp500 = w18*(-A_12_6 + A_21_2);
                                        const double tmp501 = w4*(A_12_1 - A_21_5);
                                        const double tmp502 = w17*(A_02_6 - A_20_2);
                                        const double tmp503 = w2*(-A_02_1 + A_20_5);
                                        const double tmp504 = w18*(-A_12_3 - A_21_5);
                                        const double tmp505 = w4*(A_12_4 + A_21_2);
                                        const double tmp506 = w2*(A_02_6 + A_20_3);
                                        const double tmp507 = w17*(-A_02_1 - A_20_4);
                                        const double tmp508 = w18*(A_12_0 + A_21_6);
                                        const double tmp509 = w4*(-A_12_7 - A_21_1);
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=tmp198 + tmp200 + tmp214 + tmp259 + tmp262 + tmp289 + tmp294 + tmp299 + tmp303 + tmp304 + tmp307 + tmp309 + tmp343 + tmp347 + tmp362 + tmp363 + tmp379 + tmp380 + tmp381 + tmp382 + tmp383 + tmp384 + tmp385 + tmp386;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=tmp161 + tmp201 + tmp247 + tmp250 + tmp371 + tmp374 + tmp44 + tmp451 + tmp454 + tmp455 + tmp456 + tmp466 + tmp467 + tmp468 + tmp469 + tmp49 + tmp89 + tmp91 + tmp92 + tmp98;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=tmp135 + tmp236 + tmp238 + tmp240 + tmp242 + tmp244 + tmp39 + tmp41 + tmp432 + tmp436 + tmp440 + tmp441 + tmp490 + tmp491 + tmp492 + tmp493 + tmp61 + tmp68 + tmp70 + tmp71;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=tmp114 + tmp165 + tmp166 + tmp167 + tmp168 + tmp169 + tmp170 + tmp171 + tmp172 + tmp20 + tmp73 + tmp74 + tmp75 + tmp76 + tmp79 + tmp80;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=tmp1 + tmp127 + tmp131 + tmp141 + tmp145 + tmp146 + tmp148 + tmp15 + tmp189 + tmp190 + tmp192 + tmp193 + tmp2 + tmp243 + tmp246 + tmp406 + tmp407 + tmp408 + tmp409 + tmp5;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=tmp174 + tmp176 + tmp184 + tmp24 + tmp260 + tmp267 + tmp339 + tmp340 + tmp341 + tmp342 + tmp344 + tmp345 + tmp416 + tmp417 + tmp506 + tmp507;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=tmp21 + tmp258 + tmp266 + tmp274 + tmp337 + tmp398 + tmp422 + tmp424 + tmp428 + tmp430 + tmp447 + tmp449 + tmp496 + tmp497 + tmp498 + tmp499;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=tmp104 + tmp105 + tmp106 + tmp107 + tmp108 + tmp109 + tmp110 + tmp111 + tmp112 + tmp113 + tmp38 + tmp87;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=tmp145 + tmp148 + tmp161 + tmp201 + tmp202 + tmp210 + tmp371 + tmp374 + tmp440 + tmp441 + tmp450 + tmp451 + tmp452 + tmp453 + tmp454 + tmp455 + tmp456 + tmp457 + tmp89 + tmp91;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=tmp215 + tmp221 + tmp227 + tmp260 + tmp267 + tmp288 + tmp304 + tmp312 + tmp317 + tmp351 + tmp352 + tmp353 + tmp354 + tmp355 + tmp356 + tmp357 + tmp358 + tmp359 + tmp360 + tmp361 + tmp362 + tmp363 + tmp76 + tmp79;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=tmp166 + tmp169 + tmp172 + tmp196 + tmp197 + tmp198 + tmp199 + tmp20 + tmp200 + tmp21 + tmp73 + tmp74 + tmp75 + tmp77 + tmp80 + tmp82;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=tmp36 + tmp37 + tmp38 + tmp39 + tmp40 + tmp41 + tmp42 + tmp43 + tmp44 + tmp45 + tmp46 + tmp47 + tmp48 + tmp49 + tmp50 + tmp51 + tmp52 + tmp53 + tmp54 + tmp55;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=tmp176 + tmp24 + tmp269 + tmp274 + tmp339 + tmp340 + tmp342 + tmp343 + tmp344 + tmp347 + tmp394 + tmp395 + tmp416 + tmp417 + tmp418 + tmp419;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=tmp112 + tmp12 + tmp123 + tmp13 + tmp141 + tmp142 + tmp143 + tmp146 + tmp147 + tmp149 + tmp16 + tmp277 + tmp278 + tmp279 + tmp280 + tmp281 + tmp282 + tmp6 + tmp92 + tmp98;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=tmp104 + tmp105 + tmp106 + tmp110 + tmp113 + tmp135 + tmp136 + tmp137 + tmp138 + tmp139 + tmp15 + tmp87;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=tmp114 + tmp184 + tmp225 + tmp232 + tmp329 + tmp330 + tmp332 + tmp334 + tmp335 + tmp337 + tmp338 + tmp421 + tmp464 + tmp465 + tmp504 + tmp505;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=tmp135 + tmp234 + tmp235 + tmp236 + tmp237 + tmp238 + tmp239 + tmp240 + tmp241 + tmp242 + tmp243 + tmp244 + tmp245 + tmp246 + tmp39 + tmp41 + tmp44 + tmp49 + tmp61 + tmp71;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=tmp114 + tmp120 + tmp167 + tmp170 + tmp198 + tmp20 + tmp200 + tmp24 + tmp443 + tmp444 + tmp73 + tmp74 + tmp75 + tmp80 + tmp81 + tmp83;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=tmp217 + tmp231 + tmp233 + tmp258 + tmp266 + tmp271 + tmp273 + tmp288 + tmp289 + tmp290 + tmp291 + tmp292 + tmp293 + tmp294 + tmp295 + tmp296 + tmp297 + tmp298 + tmp299 + tmp300 + tmp301 + tmp302 + tmp76 + tmp79;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=tmp101 + tmp156 + tmp157 + tmp204 + tmp205 + tmp368 + tmp371 + tmp372 + tmp374 + tmp375 + tmp377 + tmp437 + tmp438 + tmp439 + tmp440 + tmp441 + tmp442 + tmp85 + tmp87 + tmp99;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=tmp184 + tmp21 + tmp328 + tmp337 + tmp383 + tmp386 + tmp422 + tmp423 + tmp424 + tmp427 + tmp428 + tmp430 + tmp498 + tmp499 + tmp508 + tmp509;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=tmp104 + tmp106 + tmp108 + tmp111 + tmp113 + tmp15 + tmp160 + tmp161 + tmp162 + tmp163 + tmp164 + tmp38;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=tmp10 + tmp112 + tmp122 + tmp123 + tmp124 + tmp125 + tmp126 + tmp127 + tmp128 + tmp129 + tmp130 + tmp131 + tmp132 + tmp133 + tmp134 + tmp14 + tmp3 + tmp68 + tmp70 + tmp9;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=tmp166 + tmp175 + tmp176 + tmp178 + tmp179 + tmp180 + tmp183 + tmp187 + tmp270 + tmp272 + tmp274 + tmp284 + tmp285 + tmp364 + tmp365 + tmp366;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=tmp20 + tmp21 + tmp24 + tmp34 + tmp72 + tmp73 + tmp74 + tmp75 + tmp76 + tmp77 + tmp78 + tmp79 + tmp80 + tmp81 + tmp82 + tmp83;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=tmp13 + tmp16 + tmp38 + tmp39 + tmp40 + tmp41 + tmp43 + tmp440 + tmp441 + tmp45 + tmp47 + tmp478 + tmp481 + tmp486 + tmp487 + tmp488 + tmp489 + tmp50 + tmp52 + tmp55;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=tmp101 + tmp14 + tmp204 + tmp205 + tmp367 + tmp368 + tmp369 + tmp370 + tmp371 + tmp372 + tmp373 + tmp374 + tmp375 + tmp376 + tmp377 + tmp378 + tmp44 + tmp49 + tmp87 + tmp9;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=tmp179 + tmp183 + tmp198 + tmp200 + tmp214 + tmp215 + tmp216 + tmp217 + tmp218 + tmp219 + tmp220 + tmp221 + tmp222 + tmp223 + tmp224 + tmp225 + tmp226 + tmp227 + tmp228 + tmp229 + tmp230 + tmp231 + tmp232 + tmp233;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=tmp104 + tmp106 + tmp112 + tmp113 + tmp135 + tmp137 + tmp139 + tmp160 + tmp161 + tmp164 + tmp471 + tmp473;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=tmp114 + tmp274 + tmp312 + tmp317 + tmp329 + tmp332 + tmp335 + tmp337 + tmp338 + tmp399 + tmp401 + tmp446 + tmp462 + tmp463 + tmp464 + tmp465;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=tmp166 + tmp175 + tmp176 + tmp177 + tmp178 + tmp180 + tmp181 + tmp184 + tmp187 + tmp271 + tmp273 + tmp283 + tmp284 + tmp285 + tmp286 + tmp287;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=tmp1 + tmp10 + tmp11 + tmp12 + tmp15 + tmp152 + tmp153 + tmp154 + tmp155 + tmp156 + tmp157 + tmp158 + tmp159 + tmp17 + tmp3 + tmp4 + tmp51 + tmp54 + tmp6 + tmp7;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=tmp1 + tmp127 + tmp131 + tmp141 + tmp146 + tmp15 + tmp153 + tmp154 + tmp188 + tmp189 + tmp190 + tmp191 + tmp192 + tmp193 + tmp194 + tmp195 + tmp68 + tmp70 + tmp92 + tmp98;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=tmp166 + tmp176 + tmp184 + tmp283 + tmp339 + tmp340 + tmp341 + tmp342 + tmp343 + tmp344 + tmp345 + tmp346 + tmp347 + tmp348 + tmp349 + tmp350;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=tmp114 + tmp274 + tmp337 + tmp383 + tmp386 + tmp422 + tmp424 + tmp426 + tmp428 + tmp429 + tmp430 + tmp445 + tmp446 + tmp447 + tmp448 + tmp449;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=tmp104 + tmp106 + tmp107 + tmp109 + tmp112 + tmp113 + tmp135 + tmp161 + tmp482 + tmp483 + tmp484 + tmp485;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=tmp118 + tmp121 + tmp214 + tmp215 + tmp216 + tmp217 + tmp220 + tmp222 + tmp253 + tmp254 + tmp255 + tmp256 + tmp257 + tmp258 + tmp259 + tmp260 + tmp261 + tmp262 + tmp263 + tmp264 + tmp265 + tmp266 + tmp267 + tmp268;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=tmp100 + tmp101 + tmp145 + tmp148 + tmp369 + tmp376 + tmp402 + tmp403 + tmp404 + tmp405 + tmp60 + tmp65 + tmp84 + tmp87 + tmp88 + tmp89 + tmp91 + tmp95 + tmp96 + tmp97;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=tmp243 + tmp246 + tmp38 + tmp43 + tmp476 + tmp477 + tmp478 + tmp479 + tmp480 + tmp481 + tmp57 + tmp58 + tmp61 + tmp63 + tmp64 + tmp66 + tmp69 + tmp71 + tmp90 + tmp94;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=tmp20 + tmp21 + tmp22 + tmp23 + tmp24 + tmp25 + tmp26 + tmp27 + tmp28 + tmp29 + tmp30 + tmp31 + tmp32 + tmp33 + tmp34 + tmp35;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=tmp166 + tmp176 + tmp260 + tmp267 + tmp274 + tmp339 + tmp340 + tmp342 + tmp344 + tmp346 + tmp348 + tmp365 + tmp393 + tmp394 + tmp395 + tmp396;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=tmp112 + tmp12 + tmp123 + tmp124 + tmp126 + tmp140 + tmp141 + tmp142 + tmp143 + tmp144 + tmp145 + tmp146 + tmp147 + tmp148 + tmp149 + tmp150 + tmp151 + tmp51 + tmp54 + tmp6;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=tmp104 + tmp106 + tmp113 + tmp136 + tmp138 + tmp15 + tmp161 + tmp38 + tmp472 + tmp475 + tmp482 + tmp483;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=tmp184 + tmp21 + tmp312 + tmp317 + tmp327 + tmp328 + tmp329 + tmp330 + tmp331 + tmp332 + tmp333 + tmp334 + tmp335 + tmp336 + tmp337 + tmp338;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=tmp100 + tmp101 + tmp102 + tmp103 + tmp84 + tmp85 + tmp86 + tmp87 + tmp88 + tmp89 + tmp90 + tmp91 + tmp92 + tmp93 + tmp94 + tmp95 + tmp96 + tmp97 + tmp98 + tmp99;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=tmp217 + tmp225 + tmp232 + tmp26 + tmp265 + tmp268 + tmp288 + tmp289 + tmp29 + tmp290 + tmp293 + tmp295 + tmp308 + tmp313 + tmp343 + tmp347 + tmp358 + tmp359 + tmp410 + tmp411 + tmp412 + tmp413 + tmp414 + tmp415;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=tmp114 + tmp115 + tmp116 + tmp117 + tmp118 + tmp119 + tmp120 + tmp121 + tmp20 + tmp22 + tmp24 + tmp25 + tmp28 + tmp30 + tmp32 + tmp35;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=tmp13 + tmp135 + tmp16 + tmp237 + tmp238 + tmp245 + tmp319 + tmp320 + tmp321 + tmp322 + tmp323 + tmp324 + tmp325 + tmp326 + tmp45 + tmp55 + tmp57 + tmp60 + tmp64 + tmp65;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=tmp114 + tmp184 + tmp258 + tmp266 + tmp337 + tmp420 + tmp421 + tmp422 + tmp423 + tmp424 + tmp425 + tmp426 + tmp427 + tmp428 + tmp429 + tmp430;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=tmp104 + tmp106 + tmp113 + tmp135 + tmp15 + tmp162 + tmp163 + tmp470 + tmp474 + tmp484 + tmp485 + tmp87;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=tmp10 + tmp112 + tmp123 + tmp125 + tmp127 + tmp128 + tmp130 + tmp131 + tmp132 + tmp156 + tmp157 + tmp243 + tmp246 + tmp278 + tmp279 + tmp3 + tmp500 + tmp501 + tmp502 + tmp503;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=tmp175 + tmp176 + tmp178 + tmp180 + tmp182 + tmp185 + tmp187 + tmp24 + tmp269 + tmp270 + tmp271 + tmp272 + tmp273 + tmp274 + tmp275 + tmp276;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=tmp38 + tmp42 + tmp43 + tmp53 + tmp56 + tmp57 + tmp58 + tmp59 + tmp60 + tmp61 + tmp62 + tmp63 + tmp64 + tmp65 + tmp66 + tmp67 + tmp68 + tmp69 + tmp70 + tmp71;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=tmp118 + tmp121 + tmp166 + tmp199 + tmp20 + tmp21 + tmp22 + tmp25 + tmp27 + tmp28 + tmp30 + tmp33 + tmp458 + tmp459 + tmp460 + tmp461;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=tmp179 + tmp183 + tmp215 + tmp255 + tmp26 + tmp261 + tmp288 + tmp29 + tmp298 + tmp300 + tmp304 + tmp316 + tmp318 + tmp351 + tmp354 + tmp355 + tmp383 + tmp386 + tmp387 + tmp388 + tmp389 + tmp390 + tmp391 + tmp392;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=tmp100 + tmp14 + tmp161 + tmp201 + tmp202 + tmp203 + tmp204 + tmp205 + tmp206 + tmp207 + tmp208 + tmp209 + tmp210 + tmp211 + tmp212 + tmp213 + tmp88 + tmp9 + tmp90 + tmp94;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=tmp104 + tmp106 + tmp112 + tmp113 + tmp38 + tmp470 + tmp471 + tmp472 + tmp473 + tmp474 + tmp475 + tmp87;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=tmp21 + tmp225 + tmp232 + tmp274 + tmp329 + tmp332 + tmp333 + tmp335 + tmp336 + tmp337 + tmp338 + tmp397 + tmp398 + tmp399 + tmp400 + tmp401;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=tmp173 + tmp174 + tmp175 + tmp176 + tmp177 + tmp178 + tmp179 + tmp180 + tmp181 + tmp182 + tmp183 + tmp184 + tmp185 + tmp186 + tmp187 + tmp24;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=tmp0 + tmp1 + tmp10 + tmp11 + tmp12 + tmp13 + tmp14 + tmp15 + tmp16 + tmp17 + tmp18 + tmp19 + tmp2 + tmp3 + tmp4 + tmp5 + tmp6 + tmp7 + tmp8 + tmp9;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=tmp114 + tmp117 + tmp119 + tmp166 + tmp171 + tmp20 + tmp22 + tmp25 + tmp26 + tmp28 + tmp29 + tmp30 + tmp460 + tmp461 + tmp494 + tmp495;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=tmp135 + tmp238 + tmp320 + tmp324 + tmp325 + tmp326 + tmp431 + tmp432 + tmp433 + tmp434 + tmp435 + tmp436 + tmp45 + tmp51 + tmp54 + tmp55 + tmp57 + tmp64 + tmp90 + tmp94;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=tmp100 + tmp156 + tmp157 + tmp161 + tmp201 + tmp204 + tmp205 + tmp207 + tmp208 + tmp209 + tmp211 + tmp247 + tmp248 + tmp249 + tmp250 + tmp251 + tmp252 + tmp60 + tmp65 + tmp88;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=tmp118 + tmp121 + tmp214 + tmp226 + tmp228 + tmp271 + tmp273 + tmp289 + tmp303 + tmp304 + tmp305 + tmp306 + tmp307 + tmp308 + tmp309 + tmp310 + tmp311 + tmp312 + tmp313 + tmp314 + tmp315 + tmp316 + tmp317 + tmp318;
                                    }
                                }
                            } else { // constant data
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double Aw00 = 8*A_p[INDEX4(k,0,m,0, numEq,3, numComp)]*w27;
                                        const double Aw01 = 12*A_p[INDEX4(k,0,m,1, numEq,3, numComp)]*w8;
                                        const double Aw02 = 12*A_p[INDEX4(k,0,m,2, numEq,3, numComp)]*w11;
                                        const double Aw10 = 12*A_p[INDEX4(k,1,m,0, numEq,3, numComp)]*w8;
                                        const double Aw11 = 8*A_p[INDEX4(k,1,m,1, numEq,3, numComp)]*w22;
                                        const double Aw12 = 12*A_p[INDEX4(k,1,m,2, numEq,3, numComp)]*w10;
                                        const double Aw20 = 12*A_p[INDEX4(k,2,m,0, numEq,3, numComp)]*w11;
                                        const double Aw21 = 12*A_p[INDEX4(k,2,m,1, numEq,3, numComp)]*w10;
                                        const double Aw22 = 8*A_p[INDEX4(k,2,m,2, numEq,3, numComp)]*w13;
                                        const double tmp0 = Aw01 + Aw10;
                                        const double tmp1 = Aw01 - Aw10;
                                        const double tmp2 = Aw02 + Aw20;
                                        const double tmp3 = Aw02 - Aw20;
                                        const double tmp4 = Aw12 + Aw21;
                                        const double tmp5 = Aw12 - Aw21;
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp0 + 2*tmp2 - 2*tmp4;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - tmp4 + 2*tmp1 + 2*tmp3;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 - 2*tmp1 + tmp2 - 2*tmp5;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 - 2*tmp0 + tmp3 - tmp5;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 - 2*tmp3 + 2*tmp5 + tmp0;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - 2*tmp2 + tmp1 + tmp5;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + 2*tmp4 - tmp1 - tmp3;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 + tmp4 - tmp0 - tmp2;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - 2*tmp3 - 2*tmp1 - tmp4;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 - 2*tmp2 - 2*tmp4 - 2*tmp0;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 + 2*tmp0 - tmp5 - tmp3;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 - tmp2 - 2*tmp5 + 2*tmp1;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 + 2*tmp2 - tmp1 + tmp5;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 + 2*tmp5 - tmp0 + 2*tmp3;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 + tmp4 + tmp2 + tmp0;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp3 + tmp1 + 2*tmp4;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 + 2*tmp5 + tmp2 + 2*tmp1;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 + tmp3 + 2*tmp0 + tmp5;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp4 + 2*tmp2 - 2*tmp0;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 + tmp4 - 2*tmp1 + 2*tmp3;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp1 - 2*tmp4 - tmp3;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 - tmp4 + tmp0 - tmp2;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 - 2*tmp3 - tmp0 - 2*tmp5;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - tmp5 - 2*tmp2 - tmp1;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 - tmp3 + tmp5 - 2*tmp0;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 + 2*tmp5 - 2*tmp1 - tmp2;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - 2*tmp3 + tmp4 + 2*tmp1;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp0 - 2*tmp2 + 2*tmp4;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 - tmp0 + tmp2 - tmp4;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp3 - tmp1 - 2*tmp4;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - tmp5 + tmp1 + 2*tmp2;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 + tmp0 - 2*tmp5 + 2*tmp3;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 + tmp0 - 2*tmp5 + 2*tmp3;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - tmp5 + tmp1 + 2*tmp2;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp3 - tmp1 - 2*tmp4;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 - tmp0 + tmp2 - tmp4;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp0 - 2*tmp2 + 2*tmp4;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - 2*tmp3 + tmp4 + 2*tmp1;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 + 2*tmp5 - 2*tmp1 - tmp2;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 - tmp3 + tmp5 - 2*tmp0;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - tmp5 - 2*tmp2 - tmp1;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 - 2*tmp3 - tmp0 - 2*tmp5;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 - tmp4 + tmp0 - tmp2;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp1 - 2*tmp4 - tmp3;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 + tmp4 - 2*tmp1 + 2*tmp3;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp4 + 2*tmp2 - 2*tmp0;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 + tmp3 + 2*tmp0 + tmp5;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 + 2*tmp5 + tmp2 + 2*tmp1;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + tmp3 + tmp1 + 2*tmp4;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 + tmp4 + tmp2 + tmp0;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 + 2*tmp5 - tmp0 + 2*tmp3;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 + 2*tmp2 - tmp1 + tmp5;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 - tmp2 - 2*tmp5 + 2*tmp1;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 + 2*tmp0 - tmp5 - tmp3;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 - 2*tmp2 - 2*tmp4 - 2*tmp0;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - 2*tmp3 - 2*tmp1 - tmp4;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=   Aw00 +   Aw11 +   Aw22 + tmp4 - tmp0 - tmp2;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=  -Aw00 + 2*Aw11 + 2*Aw22 + 2*tmp4 - tmp1 - tmp3;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+= 2*Aw00 -   Aw11 + 2*Aw22 - 2*tmp2 + tmp1 + tmp5;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=-2*Aw00 - 2*Aw11 + 4*Aw22 - 2*tmp3 + 2*tmp5 + tmp0;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+= 2*Aw00 + 2*Aw11 -   Aw22 + tmp3 - tmp5 - 2*tmp0;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=-2*Aw00 + 4*Aw11 - 2*Aw22 - 2*tmp1 + tmp2 - 2*tmp5;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+= 4*Aw00 - 2*Aw11 - 2*Aw22 - tmp4 + 2*tmp1 + 2*tmp3;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=-4*Aw00 - 4*Aw11 - 4*Aw22 + 2*tmp0 + 2*tmp2 - 2*tmp4;
                                    }
                                }
                            }
                        }
                        ///////////////
                        // process B //
                        ///////////////
                        if (!B.isEmpty()) {
                            const double* B_p=B.getSampleDataRO(e);
                            if (B.actsExpanded()) {
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double B_0_0 = B_p[INDEX4(k,0,m,0, numEq,3,numComp)];
                                        const double B_1_0 = B_p[INDEX4(k,1,m,0, numEq,3,numComp)];
                                        const double B_2_0 = B_p[INDEX4(k,2,m,0, numEq,3,numComp)];
                                        const double B_0_1 = B_p[INDEX4(k,0,m,1, numEq,3,numComp)];
                                        const double B_1_1 = B_p[INDEX4(k,1,m,1, numEq,3,numComp)];
                                        const double B_2_1 = B_p[INDEX4(k,2,m,1, numEq,3,numComp)];
                                        const double B_0_2 = B_p[INDEX4(k,0,m,2, numEq,3,numComp)];
                                        const double B_1_2 = B_p[INDEX4(k,1,m,2, numEq,3,numComp)];
                                        const double B_2_2 = B_p[INDEX4(k,2,m,2, numEq,3,numComp)];
                                        const double B_0_3 = B_p[INDEX4(k,0,m,3, numEq,3,numComp)];
                                        const double B_1_3 = B_p[INDEX4(k,1,m,3, numEq,3,numComp)];
                                        const double B_2_3 = B_p[INDEX4(k,2,m,3, numEq,3,numComp)];
                                        const double B_0_4 = B_p[INDEX4(k,0,m,4, numEq,3,numComp)];
                                        const double B_1_4 = B_p[INDEX4(k,1,m,4, numEq,3,numComp)];
                                        const double B_2_4 = B_p[INDEX4(k,2,m,4, numEq,3,numComp)];
                                        const double B_0_5 = B_p[INDEX4(k,0,m,5, numEq,3,numComp)];
                                        const double B_1_5 = B_p[INDEX4(k,1,m,5, numEq,3,numComp)];
                                        const double B_2_5 = B_p[INDEX4(k,2,m,5, numEq,3,numComp)];
                                        const double B_0_6 = B_p[INDEX4(k,0,m,6, numEq,3,numComp)];
                                        const double B_1_6 = B_p[INDEX4(k,1,m,6, numEq,3,numComp)];
                                        const double B_2_6 = B_p[INDEX4(k,2,m,6, numEq,3,numComp)];
                                        const double B_0_7 = B_p[INDEX4(k,0,m,7, numEq,3,numComp)];
                                        const double B_1_7 = B_p[INDEX4(k,1,m,7, numEq,3,numComp)];
                                        const double B_2_7 = B_p[INDEX4(k,2,m,7, numEq,3,numComp)];
                                        const double tmp0 = w38*(B_2_1 + B_2_2);
                                        const double tmp1 = w42*(B_1_3 + B_1_7);
                                        const double tmp2 = w41*(B_0_3 + B_0_7);
                                        const double tmp3 = w37*(B_1_1 + B_1_5);
                                        const double tmp4 = w39*(B_0_2 + B_0_6);
                                        const double tmp5 = w45*(B_2_5 + B_2_6);
                                        const double tmp6 = w36*(B_0_1 + B_0_5);
                                        const double tmp7 = w40*(B_1_2 + B_1_6);
                                        const double tmp8 = w33*(B_0_0 + B_0_4);
                                        const double tmp9 = w34*(B_1_0 + B_1_4);
                                        const double tmp10 = w38*(B_2_4 + B_2_5 + B_2_6 + B_2_7);
                                        const double tmp11 = w42*(-B_1_6 - B_1_7);
                                        const double tmp12 = w41*(-B_0_5 - B_0_7);
                                        const double tmp13 = w37*(-B_1_4 - B_1_5);
                                        const double tmp14 = w39*(-B_0_4 - B_0_6);
                                        const double tmp15 = w45*(B_2_0 + B_2_1 + B_2_2 + B_2_3);
                                        const double tmp16 = w36*(-B_0_1 - B_0_3);
                                        const double tmp17 = w40*(-B_1_2 - B_1_3);
                                        const double tmp18 = w33*(-B_0_0 - B_0_2);
                                        const double tmp19 = w34*(-B_1_0 - B_1_1);
                                        const double tmp20 = w38*(-B_2_5 - B_2_7);
                                        const double tmp21 = w35*(-B_2_4 - B_2_6);
                                        const double tmp22 = w41*(B_0_1 + B_0_3);
                                        const double tmp23 = w37*(-B_1_2 - B_1_7);
                                        const double tmp24 = w39*(B_0_0 + B_0_2);
                                        const double tmp25 = w45*(-B_2_0 - B_2_2);
                                        const double tmp26 = w36*(B_0_5 + B_0_7);
                                        const double tmp27 = w40*(-B_1_0 - B_1_5);
                                        const double tmp28 = w33*(B_0_4 + B_0_6);
                                        const double tmp29 = w46*(-B_2_1 - B_2_3);
                                        const double tmp30 = w38*(B_2_0 + B_2_2);
                                        const double tmp31 = w35*(B_2_1 + B_2_3);
                                        const double tmp32 = w41*(-B_0_4 - B_0_6);
                                        const double tmp33 = w37*(B_1_0 + B_1_5);
                                        const double tmp34 = w39*(-B_0_5 - B_0_7);
                                        const double tmp35 = w45*(B_2_5 + B_2_7);
                                        const double tmp36 = w36*(-B_0_0 - B_0_2);
                                        const double tmp37 = w40*(B_1_2 + B_1_7);
                                        const double tmp38 = w33*(-B_0_1 - B_0_3);
                                        const double tmp39 = w46*(B_2_4 + B_2_6);
                                        const double tmp40 = w38*(-B_2_0 - B_2_1 - B_2_2 - B_2_3);
                                        const double tmp41 = w42*(B_1_0 + B_1_1);
                                        const double tmp42 = w41*(B_0_0 + B_0_2);
                                        const double tmp43 = w37*(B_1_2 + B_1_3);
                                        const double tmp44 = w39*(B_0_1 + B_0_3);
                                        const double tmp45 = w45*(-B_2_4 - B_2_5 - B_2_6 - B_2_7);
                                        const double tmp46 = w36*(B_0_4 + B_0_6);
                                        const double tmp47 = w40*(B_1_4 + B_1_5);
                                        const double tmp48 = w33*(B_0_5 + B_0_7);
                                        const double tmp49 = w34*(B_1_6 + B_1_7);
                                        const double tmp50 = w38*(B_2_0 + B_2_1);
                                        const double tmp51 = w42*(-B_1_4 - B_1_5);
                                        const double tmp52 = w35*(B_2_2 + B_2_3);
                                        const double tmp53 = w37*(-B_1_6 - B_1_7);
                                        const double tmp54 = w39*(B_0_0 + B_0_6);
                                        const double tmp55 = w45*(B_2_6 + B_2_7);
                                        const double tmp56 = w36*(B_0_1 + B_0_7);
                                        const double tmp57 = w40*(-B_1_0 - B_1_1);
                                        const double tmp58 = w46*(B_2_4 + B_2_5);
                                        const double tmp59 = w34*(-B_1_2 - B_1_3);
                                        const double tmp60 = w38*(-B_2_4 - B_2_5 - B_2_6 - B_2_7);
                                        const double tmp61 = w37*(-B_1_2 - B_1_3 - B_1_6 - B_1_7);
                                        const double tmp62 = w39*(-B_0_1 - B_0_3 - B_0_5 - B_0_7);
                                        const double tmp63 = w45*(-B_2_0 - B_2_1 - B_2_2 - B_2_3);
                                        const double tmp64 = w36*(-B_0_0 - B_0_2 - B_0_4 - B_0_6);
                                        const double tmp65 = w40*(-B_1_0 - B_1_1 - B_1_4 - B_1_5);
                                        const double tmp66 = w41*(B_0_4 + B_0_6);
                                        const double tmp67 = w39*(B_0_5 + B_0_7);
                                        const double tmp68 = w36*(B_0_0 + B_0_2);
                                        const double tmp69 = w33*(B_0_1 + B_0_3);
                                        const double tmp70 = w38*(-B_2_4 - B_2_7);
                                        const double tmp71 = w42*(B_1_2 + B_1_6);
                                        const double tmp72 = w41*(-B_0_2 - B_0_6);
                                        const double tmp73 = w37*(B_1_0 + B_1_4);
                                        const double tmp74 = w39*(-B_0_3 - B_0_7);
                                        const double tmp75 = w45*(-B_2_0 - B_2_3);
                                        const double tmp76 = w36*(-B_0_0 - B_0_4);
                                        const double tmp77 = w40*(B_1_3 + B_1_7);
                                        const double tmp78 = w33*(-B_0_1 - B_0_5);
                                        const double tmp79 = w34*(B_1_1 + B_1_5);
                                        const double tmp80 = w39*(B_0_0 + B_0_2 + B_0_4 + B_0_6);
                                        const double tmp81 = w36*(B_0_1 + B_0_3 + B_0_5 + B_0_7);
                                        const double tmp82 = w38*(B_2_0 + B_2_3);
                                        const double tmp83 = w42*(-B_1_1 - B_1_5);
                                        const double tmp84 = w41*(B_0_1 + B_0_5);
                                        const double tmp85 = w37*(-B_1_3 - B_1_7);
                                        const double tmp86 = w39*(B_0_0 + B_0_4);
                                        const double tmp87 = w45*(B_2_4 + B_2_7);
                                        const double tmp88 = w36*(B_0_3 + B_0_7);
                                        const double tmp89 = w40*(-B_1_0 - B_1_4);
                                        const double tmp90 = w33*(B_0_2 + B_0_6);
                                        const double tmp91 = w34*(-B_1_2 - B_1_6);
                                        const double tmp92 = w38*(-B_2_5 - B_2_6);
                                        const double tmp93 = w45*(-B_2_1 - B_2_2);
                                        const double tmp94 = w37*(B_1_0 + B_1_1 + B_1_4 + B_1_5);
                                        const double tmp95 = w40*(B_1_2 + B_1_3 + B_1_6 + B_1_7);
                                        const double tmp96 = w42*(-B_1_2 - B_1_3);
                                        const double tmp97 = w41*(-B_0_1 - B_0_3);
                                        const double tmp98 = w37*(-B_1_0 - B_1_1);
                                        const double tmp99 = w39*(-B_0_0 - B_0_2);
                                        const double tmp100 = w36*(-B_0_5 - B_0_7);
                                        const double tmp101 = w40*(-B_1_6 - B_1_7);
                                        const double tmp102 = w33*(-B_0_4 - B_0_6);
                                        const double tmp103 = w34*(-B_1_4 - B_1_5);
                                        const double tmp104 = w38*(B_2_6 + B_2_7);
                                        const double tmp105 = w35*(B_2_4 + B_2_5);
                                        const double tmp106 = w41*(B_0_2 + B_0_6);
                                        const double tmp107 = w37*(B_1_2 + B_1_3 + B_1_6 + B_1_7);
                                        const double tmp108 = w39*(B_0_3 + B_0_7);
                                        const double tmp109 = w45*(B_2_0 + B_2_1);
                                        const double tmp110 = w36*(B_0_0 + B_0_4);
                                        const double tmp111 = w40*(B_1_0 + B_1_1 + B_1_4 + B_1_5);
                                        const double tmp112 = w33*(B_0_1 + B_0_5);
                                        const double tmp113 = w46*(B_2_2 + B_2_3);
                                        const double tmp114 = w42*(-B_1_0 - B_1_4);
                                        const double tmp115 = w41*(-B_0_0 - B_0_4);
                                        const double tmp116 = w37*(-B_1_2 - B_1_6);
                                        const double tmp117 = w39*(-B_0_1 - B_0_5);
                                        const double tmp118 = w36*(-B_0_2 - B_0_6);
                                        const double tmp119 = w40*(-B_1_1 - B_1_5);
                                        const double tmp120 = w33*(-B_0_3 - B_0_7);
                                        const double tmp121 = w34*(-B_1_3 - B_1_7);
                                        const double tmp122 = w38*(B_2_2 + B_2_3);
                                        const double tmp123 = w42*(B_1_6 + B_1_7);
                                        const double tmp124 = w35*(B_2_0 + B_2_1);
                                        const double tmp125 = w37*(B_1_4 + B_1_5);
                                        const double tmp126 = w39*(-B_0_3 - B_0_5);
                                        const double tmp127 = w45*(B_2_4 + B_2_5);
                                        const double tmp128 = w36*(-B_0_2 - B_0_4);
                                        const double tmp129 = w40*(B_1_2 + B_1_3);
                                        const double tmp130 = w46*(B_2_6 + B_2_7);
                                        const double tmp131 = w34*(B_1_0 + B_1_1);
                                        const double tmp132 = w38*(-B_2_1 - B_2_2);
                                        const double tmp133 = w37*(B_1_2 + B_1_7);
                                        const double tmp134 = w39*(B_0_1 + B_0_7);
                                        const double tmp135 = w36*(B_0_0 + B_0_6);
                                        const double tmp136 = w40*(B_1_0 + B_1_5);
                                        const double tmp137 = w45*(-B_2_5 - B_2_6);
                                        const double tmp138 = w38*(-B_2_4 - B_2_6);
                                        const double tmp139 = w35*(-B_2_5 - B_2_7);
                                        const double tmp140 = w41*(-B_0_0 - B_0_2);
                                        const double tmp141 = w37*(B_1_1 + B_1_4);
                                        const double tmp142 = w39*(-B_0_1 - B_0_3);
                                        const double tmp143 = w45*(-B_2_1 - B_2_3);
                                        const double tmp144 = w36*(-B_0_4 - B_0_6);
                                        const double tmp145 = w40*(B_1_3 + B_1_6);
                                        const double tmp146 = w33*(-B_0_5 - B_0_7);
                                        const double tmp147 = w46*(-B_2_0 - B_2_2);
                                        const double tmp148 = w39*(B_0_2 + B_0_4);
                                        const double tmp149 = w36*(B_0_3 + B_0_5);
                                        const double tmp150 = w38*(B_2_5 + B_2_6);
                                        const double tmp151 = w37*(-B_1_0 - B_1_5);
                                        const double tmp152 = w39*(-B_0_0 - B_0_6);
                                        const double tmp153 = w45*(B_2_1 + B_2_2);
                                        const double tmp154 = w36*(-B_0_1 - B_0_7);
                                        const double tmp155 = w40*(-B_1_2 - B_1_7);
                                        const double tmp156 = w41*(-B_0_3 - B_0_7);
                                        const double tmp157 = w39*(-B_0_2 - B_0_6);
                                        const double tmp158 = w36*(-B_0_1 - B_0_5);
                                        const double tmp159 = w33*(-B_0_0 - B_0_4);
                                        const double tmp160 = w38*(-B_2_2 - B_2_3);
                                        const double tmp161 = w35*(-B_2_0 - B_2_1);
                                        const double tmp162 = w45*(-B_2_4 - B_2_5);
                                        const double tmp163 = w46*(-B_2_6 - B_2_7);
                                        const double tmp164 = w38*(-B_2_0 - B_2_3);
                                        const double tmp165 = w37*(B_1_3 + B_1_6);
                                        const double tmp166 = w40*(B_1_1 + B_1_4);
                                        const double tmp167 = w45*(-B_2_4 - B_2_7);
                                        const double tmp168 = w39*(B_0_3 + B_0_5);
                                        const double tmp169 = w36*(B_0_2 + B_0_4);
                                        const double tmp170 = w38*(B_2_1 + B_2_3);
                                        const double tmp171 = w35*(B_2_0 + B_2_2);
                                        const double tmp172 = w41*(B_0_5 + B_0_7);
                                        const double tmp173 = w37*(-B_1_3 - B_1_6);
                                        const double tmp174 = w39*(B_0_4 + B_0_6);
                                        const double tmp175 = w45*(B_2_4 + B_2_6);
                                        const double tmp176 = w36*(B_0_1 + B_0_3);
                                        const double tmp177 = w40*(-B_1_1 - B_1_4);
                                        const double tmp178 = w33*(B_0_0 + B_0_2);
                                        const double tmp179 = w46*(B_2_5 + B_2_7);
                                        const double tmp180 = w38*(B_2_5 + B_2_7);
                                        const double tmp181 = w42*(-B_1_3 - B_1_7);
                                        const double tmp182 = w35*(B_2_4 + B_2_6);
                                        const double tmp183 = w37*(-B_1_1 - B_1_5);
                                        const double tmp184 = w39*(B_0_1 + B_0_3 + B_0_5 + B_0_7);
                                        const double tmp185 = w45*(B_2_0 + B_2_2);
                                        const double tmp186 = w36*(B_0_0 + B_0_2 + B_0_4 + B_0_6);
                                        const double tmp187 = w40*(-B_1_2 - B_1_6);
                                        const double tmp188 = w46*(B_2_1 + B_2_3);
                                        const double tmp189 = w34*(-B_1_0 - B_1_4);
                                        const double tmp190 = w38*(B_2_4 + B_2_5);
                                        const double tmp191 = w35*(B_2_6 + B_2_7);
                                        const double tmp192 = w41*(-B_0_1 - B_0_5);
                                        const double tmp193 = w37*(-B_1_0 - B_1_1 - B_1_4 - B_1_5);
                                        const double tmp194 = w39*(-B_0_0 - B_0_4);
                                        const double tmp195 = w45*(B_2_2 + B_2_3);
                                        const double tmp196 = w36*(-B_0_3 - B_0_7);
                                        const double tmp197 = w40*(-B_1_2 - B_1_3 - B_1_6 - B_1_7);
                                        const double tmp198 = w33*(-B_0_2 - B_0_6);
                                        const double tmp199 = w46*(B_2_0 + B_2_1);
                                        const double tmp200 = w38*(-B_2_6 - B_2_7);
                                        const double tmp201 = w42*(B_1_2 + B_1_3);
                                        const double tmp202 = w35*(-B_2_4 - B_2_5);
                                        const double tmp203 = w37*(B_1_0 + B_1_1);
                                        const double tmp204 = w45*(-B_2_0 - B_2_1);
                                        const double tmp205 = w40*(B_1_6 + B_1_7);
                                        const double tmp206 = w46*(-B_2_2 - B_2_3);
                                        const double tmp207 = w34*(B_1_4 + B_1_5);
                                        const double tmp208 = w37*(-B_1_1 - B_1_4);
                                        const double tmp209 = w39*(-B_0_2 - B_0_4);
                                        const double tmp210 = w36*(-B_0_3 - B_0_5);
                                        const double tmp211 = w40*(-B_1_3 - B_1_6);
                                        const double tmp212 = w38*(B_2_4 + B_2_7);
                                        const double tmp213 = w45*(B_2_0 + B_2_3);
                                        const double tmp214 = w41*(B_0_0 + B_0_4);
                                        const double tmp215 = w39*(B_0_1 + B_0_5);
                                        const double tmp216 = w36*(B_0_2 + B_0_6);
                                        const double tmp217 = w33*(B_0_3 + B_0_7);
                                        const double tmp218 = w42*(B_1_1 + B_1_5);
                                        const double tmp219 = w37*(B_1_3 + B_1_7);
                                        const double tmp220 = w40*(B_1_0 + B_1_4);
                                        const double tmp221 = w34*(B_1_2 + B_1_6);
                                        const double tmp222 = w39*(-B_0_1 - B_0_7);
                                        const double tmp223 = w36*(-B_0_0 - B_0_6);
                                        const double tmp224 = w38*(-B_2_0 - B_2_1);
                                        const double tmp225 = w35*(-B_2_2 - B_2_3);
                                        const double tmp226 = w45*(-B_2_6 - B_2_7);
                                        const double tmp227 = w46*(-B_2_4 - B_2_5);
                                        const double tmp228 = w38*(B_2_4 + B_2_6);
                                        const double tmp229 = w42*(B_1_0 + B_1_4);
                                        const double tmp230 = w35*(B_2_5 + B_2_7);
                                        const double tmp231 = w37*(B_1_2 + B_1_6);
                                        const double tmp232 = w39*(-B_0_0 - B_0_2 - B_0_4 - B_0_6);
                                        const double tmp233 = w45*(B_2_1 + B_2_3);
                                        const double tmp234 = w36*(-B_0_1 - B_0_3 - B_0_5 - B_0_7);
                                        const double tmp235 = w40*(B_1_1 + B_1_5);
                                        const double tmp236 = w46*(B_2_0 + B_2_2);
                                        const double tmp237 = w34*(B_1_3 + B_1_7);
                                        const double tmp238 = w42*(-B_1_2 - B_1_6);
                                        const double tmp239 = w37*(-B_1_0 - B_1_4);
                                        const double tmp240 = w40*(-B_1_3 - B_1_7);
                                        const double tmp241 = w34*(-B_1_1 - B_1_5);
                                        const double tmp242 = w38*(-B_2_4 - B_2_5);
                                        const double tmp243 = w42*(-B_1_0 - B_1_1);
                                        const double tmp244 = w35*(-B_2_6 - B_2_7);
                                        const double tmp245 = w37*(-B_1_2 - B_1_3);
                                        const double tmp246 = w45*(-B_2_2 - B_2_3);
                                        const double tmp247 = w40*(-B_1_4 - B_1_5);
                                        const double tmp248 = w46*(-B_2_0 - B_2_1);
                                        const double tmp249 = w34*(-B_1_6 - B_1_7);
                                        const double tmp250 = w42*(B_1_4 + B_1_5);
                                        const double tmp251 = w37*(B_1_6 + B_1_7);
                                        const double tmp252 = w40*(B_1_0 + B_1_1);
                                        const double tmp253 = w34*(B_1_2 + B_1_3);
                                        const double tmp254 = w38*(-B_2_1 - B_2_3);
                                        const double tmp255 = w35*(-B_2_0 - B_2_2);
                                        const double tmp256 = w45*(-B_2_4 - B_2_6);
                                        const double tmp257 = w46*(-B_2_5 - B_2_7);
                                        const double tmp258 = w38*(B_2_0 + B_2_1 + B_2_2 + B_2_3);
                                        const double tmp259 = w45*(B_2_4 + B_2_5 + B_2_6 + B_2_7);
                                        const double tmp260 = w38*(-B_2_0 - B_2_2);
                                        const double tmp261 = w35*(-B_2_1 - B_2_3);
                                        const double tmp262 = w45*(-B_2_5 - B_2_7);
                                        const double tmp263 = w46*(-B_2_4 - B_2_6);
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=-B_0_0*w50 - B_0_1*w41 - B_0_6*w33 - B_0_7*w49 + B_1_0*w47 - B_1_2*w42 - B_1_5*w34 + B_1_7*w48 - B_2_0*w43 - B_2_3*w35 - B_2_4*w46 - B_2_7*w44 + tmp132 + tmp137 + tmp208 + tmp209 + tmp210 + tmp211;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=-B_0_0*w41 - B_0_1*w50 - B_0_6*w49 - B_0_7*w33 + tmp126 + tmp128 + tmp242 + tmp243 + tmp244 + tmp245 + tmp246 + tmp247 + tmp248 + tmp249;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=-B_1_0*w42 + B_1_2*w47 + B_1_5*w48 - B_1_7*w34 + tmp138 + tmp139 + tmp140 + tmp142 + tmp143 + tmp144 + tmp146 + tmp147 + tmp173 + tmp177;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=tmp100 + tmp101 + tmp102 + tmp103 + tmp40 + tmp45 + tmp96 + tmp97 + tmp98 + tmp99;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=-B_2_0*w46 - B_2_3*w44 - B_2_4*w43 - B_2_7*w35 + tmp114 + tmp115 + tmp116 + tmp117 + tmp118 + tmp119 + tmp120 + tmp121 + tmp92 + tmp93;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=tmp192 + tmp193 + tmp194 + tmp196 + tmp197 + tmp198 + tmp224 + tmp225 + tmp226 + tmp227;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=tmp232 + tmp234 + tmp238 + tmp239 + tmp240 + tmp241 + tmp260 + tmp261 + tmp262 + tmp263;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=tmp60 + tmp61 + tmp62 + tmp63 + tmp64 + tmp65;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=B_0_0*w50 + B_0_1*w41 + B_0_6*w33 + B_0_7*w49 + tmp148 + tmp149 + tmp242 + tmp243 + tmp244 + tmp245 + tmp246 + tmp247 + tmp248 + tmp249;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=B_0_0*w41 + B_0_1*w50 + B_0_6*w49 + B_0_7*w33 + B_1_1*w47 - B_1_3*w42 - B_1_4*w34 + B_1_6*w48 - B_2_1*w43 - B_2_2*w35 - B_2_5*w46 - B_2_6*w44 + tmp151 + tmp155 + tmp164 + tmp167 + tmp168 + tmp169;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=tmp101 + tmp103 + tmp40 + tmp42 + tmp44 + tmp45 + tmp46 + tmp48 + tmp96 + tmp98;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=-B_1_1*w42 + B_1_3*w47 + B_1_4*w48 - B_1_6*w34 + tmp20 + tmp21 + tmp22 + tmp23 + tmp24 + tmp25 + tmp26 + tmp27 + tmp28 + tmp29;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=tmp193 + tmp197 + tmp214 + tmp215 + tmp216 + tmp217 + tmp224 + tmp225 + tmp226 + tmp227;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=-B_2_1*w46 - B_2_2*w44 - B_2_5*w43 - B_2_6*w35 + tmp70 + tmp75 + tmp83 + tmp84 + tmp85 + tmp86 + tmp88 + tmp89 + tmp90 + tmp91;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=tmp60 + tmp61 + tmp63 + tmp65 + tmp80 + tmp81;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=tmp181 + tmp183 + tmp184 + tmp186 + tmp187 + tmp189 + tmp254 + tmp255 + tmp256 + tmp257;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=-B_1_0*w47 + B_1_2*w42 + B_1_5*w34 - B_1_7*w48 + tmp138 + tmp139 + tmp140 + tmp141 + tmp142 + tmp143 + tmp144 + tmp145 + tmp146 + tmp147;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=tmp100 + tmp102 + tmp40 + tmp41 + tmp43 + tmp45 + tmp47 + tmp49 + tmp97 + tmp99;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=-B_0_2*w50 - B_0_3*w41 - B_0_4*w33 - B_0_5*w49 + B_1_0*w42 - B_1_2*w47 - B_1_5*w48 + B_1_7*w34 - B_2_1*w35 - B_2_2*w43 - B_2_5*w44 - B_2_6*w46 + tmp152 + tmp154 + tmp164 + tmp165 + tmp166 + tmp167;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=-B_0_2*w41 - B_0_3*w50 - B_0_4*w49 - B_0_5*w33 + tmp200 + tmp201 + tmp202 + tmp203 + tmp204 + tmp205 + tmp206 + tmp207 + tmp222 + tmp223;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=tmp229 + tmp231 + tmp232 + tmp234 + tmp235 + tmp237 + tmp260 + tmp261 + tmp262 + tmp263;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=tmp60 + tmp62 + tmp63 + tmp64 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=-B_2_1*w44 - B_2_2*w46 - B_2_5*w35 - B_2_6*w43 + tmp70 + tmp71 + tmp72 + tmp73 + tmp74 + tmp75 + tmp76 + tmp77 + tmp78 + tmp79;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=tmp107 + tmp111 + tmp156 + tmp157 + tmp158 + tmp159 + tmp160 + tmp161 + tmp162 + tmp163;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=tmp40 + tmp41 + tmp42 + tmp43 + tmp44 + tmp45 + tmp46 + tmp47 + tmp48 + tmp49;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=-B_1_1*w47 + B_1_3*w42 + B_1_4*w34 - B_1_6*w48 + tmp20 + tmp21 + tmp22 + tmp24 + tmp25 + tmp26 + tmp28 + tmp29 + tmp33 + tmp37;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=B_0_2*w50 + B_0_3*w41 + B_0_4*w33 + B_0_5*w49 + tmp200 + tmp201 + tmp202 + tmp203 + tmp204 + tmp205 + tmp206 + tmp207 + tmp54 + tmp56;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=B_0_2*w41 + B_0_3*w50 + B_0_4*w49 + B_0_5*w33 + B_1_1*w42 - B_1_3*w47 - B_1_4*w48 + B_1_6*w34 - B_2_0*w35 - B_2_3*w43 - B_2_4*w44 - B_2_7*w46 + tmp132 + tmp133 + tmp134 + tmp135 + tmp136 + tmp137;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=tmp60 + tmp63 + tmp80 + tmp81 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=tmp184 + tmp186 + tmp218 + tmp219 + tmp220 + tmp221 + tmp254 + tmp255 + tmp256 + tmp257;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=tmp106 + tmp107 + tmp108 + tmp110 + tmp111 + tmp112 + tmp160 + tmp161 + tmp162 + tmp163;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=-B_2_0*w44 - B_2_3*w46 - B_2_4*w35 - B_2_7*w43 + tmp1 + tmp2 + tmp3 + tmp4 + tmp6 + tmp7 + tmp8 + tmp9 + tmp92 + tmp93;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=B_2_0*w43 + B_2_3*w35 + B_2_4*w46 + B_2_7*w44 + tmp0 + tmp114 + tmp115 + tmp116 + tmp117 + tmp118 + tmp119 + tmp120 + tmp121 + tmp5;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=tmp190 + tmp191 + tmp192 + tmp193 + tmp194 + tmp195 + tmp196 + tmp197 + tmp198 + tmp199;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=tmp228 + tmp230 + tmp232 + tmp233 + tmp234 + tmp236 + tmp238 + tmp239 + tmp240 + tmp241;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=tmp258 + tmp259 + tmp61 + tmp62 + tmp64 + tmp65;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=-B_0_2*w33 - B_0_3*w49 - B_0_4*w50 - B_0_5*w41 - B_1_1*w34 + B_1_3*w48 + B_1_4*w47 - B_1_6*w42 + B_2_0*w46 + B_2_3*w44 + B_2_4*w43 + B_2_7*w35 + tmp150 + tmp151 + tmp152 + tmp153 + tmp154 + tmp155;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=-B_0_2*w49 - B_0_3*w33 - B_0_4*w41 - B_0_5*w50 + tmp222 + tmp223 + tmp50 + tmp51 + tmp52 + tmp53 + tmp55 + tmp57 + tmp58 + tmp59;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=B_1_1*w48 - B_1_3*w34 - B_1_4*w42 + B_1_6*w47 + tmp23 + tmp27 + tmp30 + tmp31 + tmp32 + tmp34 + tmp35 + tmp36 + tmp38 + tmp39;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=tmp10 + tmp11 + tmp12 + tmp13 + tmp14 + tmp15 + tmp16 + tmp17 + tmp18 + tmp19;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=tmp190 + tmp191 + tmp193 + tmp195 + tmp197 + tmp199 + tmp214 + tmp215 + tmp216 + tmp217;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=B_2_1*w43 + B_2_2*w35 + B_2_5*w46 + B_2_6*w44 + tmp82 + tmp83 + tmp84 + tmp85 + tmp86 + tmp87 + tmp88 + tmp89 + tmp90 + tmp91;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=tmp258 + tmp259 + tmp61 + tmp65 + tmp80 + tmp81;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=tmp180 + tmp181 + tmp182 + tmp183 + tmp184 + tmp185 + tmp186 + tmp187 + tmp188 + tmp189;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=B_0_2*w33 + B_0_3*w49 + B_0_4*w50 + B_0_5*w41 + tmp50 + tmp51 + tmp52 + tmp53 + tmp54 + tmp55 + tmp56 + tmp57 + tmp58 + tmp59;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=B_0_2*w49 + B_0_3*w33 + B_0_4*w41 + B_0_5*w50 - B_1_0*w34 + B_1_2*w48 + B_1_5*w47 - B_1_7*w42 + B_2_1*w46 + B_2_2*w44 + B_2_5*w43 + B_2_6*w35 + tmp134 + tmp135 + tmp208 + tmp211 + tmp212 + tmp213;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=tmp10 + tmp11 + tmp13 + tmp15 + tmp17 + tmp19 + tmp66 + tmp67 + tmp68 + tmp69;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=B_1_0*w48 - B_1_2*w34 - B_1_5*w42 + B_1_7*w47 + tmp170 + tmp171 + tmp172 + tmp173 + tmp174 + tmp175 + tmp176 + tmp177 + tmp178 + tmp179;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=tmp228 + tmp229 + tmp230 + tmp231 + tmp232 + tmp233 + tmp234 + tmp235 + tmp236 + tmp237;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=tmp258 + tmp259 + tmp62 + tmp64 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=B_2_1*w35 + B_2_2*w43 + B_2_5*w44 + B_2_6*w46 + tmp71 + tmp72 + tmp73 + tmp74 + tmp76 + tmp77 + tmp78 + tmp79 + tmp82 + tmp87;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=tmp104 + tmp105 + tmp107 + tmp109 + tmp111 + tmp113 + tmp156 + tmp157 + tmp158 + tmp159;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=B_1_1*w34 - B_1_3*w48 - B_1_4*w47 + B_1_6*w42 + tmp30 + tmp31 + tmp32 + tmp33 + tmp34 + tmp35 + tmp36 + tmp37 + tmp38 + tmp39;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=tmp10 + tmp12 + tmp14 + tmp15 + tmp16 + tmp18 + tmp250 + tmp251 + tmp252 + tmp253;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=-B_0_0*w33 - B_0_1*w49 - B_0_6*w50 - B_0_7*w41 - B_1_1*w48 + B_1_3*w34 + B_1_4*w42 - B_1_6*w47 + B_2_1*w44 + B_2_2*w46 + B_2_5*w35 + B_2_6*w43 + tmp133 + tmp136 + tmp209 + tmp210 + tmp212 + tmp213;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=-B_0_0*w49 - B_0_1*w33 - B_0_6*w41 - B_0_7*w50 + tmp122 + tmp123 + tmp124 + tmp125 + tmp126 + tmp127 + tmp128 + tmp129 + tmp130 + tmp131;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=tmp258 + tmp259 + tmp80 + tmp81 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=tmp180 + tmp182 + tmp184 + tmp185 + tmp186 + tmp188 + tmp218 + tmp219 + tmp220 + tmp221;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=tmp104 + tmp105 + tmp106 + tmp107 + tmp108 + tmp109 + tmp110 + tmp111 + tmp112 + tmp113;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=B_2_0*w35 + B_2_3*w43 + B_2_4*w44 + B_2_7*w46 + tmp0 + tmp1 + tmp2 + tmp3 + tmp4 + tmp5 + tmp6 + tmp7 + tmp8 + tmp9;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=tmp10 + tmp15 + tmp250 + tmp251 + tmp252 + tmp253 + tmp66 + tmp67 + tmp68 + tmp69;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=B_1_0*w34 - B_1_2*w48 - B_1_5*w47 + B_1_7*w42 + tmp141 + tmp145 + tmp170 + tmp171 + tmp172 + tmp174 + tmp175 + tmp176 + tmp178 + tmp179;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=B_0_0*w33 + B_0_1*w49 + B_0_6*w50 + B_0_7*w41 + tmp122 + tmp123 + tmp124 + tmp125 + tmp127 + tmp129 + tmp130 + tmp131 + tmp148 + tmp149;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=B_0_0*w49 + B_0_1*w33 + B_0_6*w41 + B_0_7*w50 - B_1_0*w48 + B_1_2*w34 + B_1_5*w42 - B_1_7*w47 + B_2_0*w44 + B_2_3*w46 + B_2_4*w35 + B_2_7*w43 + tmp150 + tmp153 + tmp165 + tmp166 + tmp168 + tmp169;
                                    }
                                }
                            } else { // constant data
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double wB0 = B_p[INDEX3(k,0,m,numEq,3)]*w55;
                                        const double wB1 = B_p[INDEX3(k,1,m,numEq,3)]*w56;
                                        const double wB2 = B_p[INDEX3(k,2,m,numEq,3)]*w54;
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+= 4*wB0 + 4*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+= 4*wB0 + 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+= 2*wB0 + 4*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+= 2*wB0 + 2*wB1 +   wB2;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+= 2*wB0 + 2*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+= 2*wB0 +   wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=   wB0 + 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=   wB0 +   wB1 +   wB2;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=-4*wB0 + 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=-4*wB0 + 4*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=-2*wB0 + 2*wB1 +   wB2;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=-2*wB0 + 4*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=-2*wB0 +   wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=-2*wB0 + 2*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=  -wB0 +   wB1 +   wB2;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=  -wB0 + 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+= 2*wB0 - 4*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+= 2*wB0 - 2*wB1 +   wB2;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+= 4*wB0 - 4*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+= 4*wB0 - 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=   wB0 - 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=   wB0 -   wB1 +   wB2;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+= 2*wB0 - 2*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+= 2*wB0 -   wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=-2*wB0 - 2*wB1 +   wB2;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=-2*wB0 - 4*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=-4*wB0 - 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=-4*wB0 - 4*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=  -wB0 -   wB1 +   wB2;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=  -wB0 - 2*wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=-2*wB0 -   wB1 + 2*wB2;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=-2*wB0 - 2*wB1 + 4*wB2;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+= 2*wB0 + 2*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+= 2*wB0 +   wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=   wB0 + 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=   wB0 +   wB1 -   wB2;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+= 4*wB0 + 4*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+= 4*wB0 + 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+= 2*wB0 + 4*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+= 2*wB0 + 2*wB1 -   wB2;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=-2*wB0 +   wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=-2*wB0 + 2*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=  -wB0 +   wB1 -   wB2;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=  -wB0 + 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=-4*wB0 + 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=-4*wB0 + 4*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=-2*wB0 + 2*wB1 -   wB2;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=-2*wB0 + 4*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=   wB0 - 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=   wB0 -   wB1 -   wB2;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+= 2*wB0 - 2*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+= 2*wB0 -   wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+= 2*wB0 - 4*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+= 2*wB0 - 2*wB1 -   wB2;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+= 4*wB0 - 4*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+= 4*wB0 - 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=  -wB0 -   wB1 -   wB2;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=  -wB0 - 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=-2*wB0 -   wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=-2*wB0 - 2*wB1 - 4*wB2;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=-2*wB0 - 2*wB1 -   wB2;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=-2*wB0 - 4*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=-4*wB0 - 2*wB1 - 2*wB2;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=-4*wB0 - 4*wB1 - 4*wB2;
                                    }
                                }
                            }
                        }
                        ///////////////
                        // process C //
                        ///////////////
                        if (!C.isEmpty()) {
                            const double* C_p=C.getSampleDataRO(e);
                            if (C.actsExpanded()) {
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double C_0_0 = C_p[INDEX4(k,m,0, 0, numEq,numComp,3)];
                                        const double C_1_0 = C_p[INDEX4(k,m,1, 0, numEq,numComp,3)];
                                        const double C_2_0 = C_p[INDEX4(k,m,2, 0, numEq,numComp,3)];
                                        const double C_0_1 = C_p[INDEX4(k,m,0, 1, numEq,numComp,3)];
                                        const double C_1_1 = C_p[INDEX4(k,m,1, 1, numEq,numComp,3)];
                                        const double C_2_1 = C_p[INDEX4(k,m,2, 1, numEq,numComp,3)];
                                        const double C_0_2 = C_p[INDEX4(k,m,0, 2, numEq,numComp,3)];
                                        const double C_1_2 = C_p[INDEX4(k,m,1, 2, numEq,numComp,3)];
                                        const double C_2_2 = C_p[INDEX4(k,m,2, 2, numEq,numComp,3)];
                                        const double C_0_3 = C_p[INDEX4(k,m,0, 3, numEq,numComp,3)];
                                        const double C_1_3 = C_p[INDEX4(k,m,1, 3, numEq,numComp,3)];
                                        const double C_2_3 = C_p[INDEX4(k,m,2, 3, numEq,numComp,3)];
                                        const double C_0_4 = C_p[INDEX4(k,m,0, 4, numEq,numComp,3)];
                                        const double C_1_4 = C_p[INDEX4(k,m,1, 4, numEq,numComp,3)];
                                        const double C_2_4 = C_p[INDEX4(k,m,2, 4, numEq,numComp,3)];
                                        const double C_0_5 = C_p[INDEX4(k,m,0, 5, numEq,numComp,3)];
                                        const double C_1_5 = C_p[INDEX4(k,m,1, 5, numEq,numComp,3)];
                                        const double C_2_5 = C_p[INDEX4(k,m,2, 5, numEq,numComp,3)];
                                        const double C_0_6 = C_p[INDEX4(k,m,0, 6, numEq,numComp,3)];
                                        const double C_1_6 = C_p[INDEX4(k,m,1, 6, numEq,numComp,3)];
                                        const double C_2_6 = C_p[INDEX4(k,m,2, 6, numEq,numComp,3)];
                                        const double C_0_7 = C_p[INDEX4(k,m,0, 7, numEq,numComp,3)];
                                        const double C_1_7 = C_p[INDEX4(k,m,1, 7, numEq,numComp,3)];
                                        const double C_2_7 = C_p[INDEX4(k,m,2, 7, numEq,numComp,3)];
                                        const double tmp0 = w38*(-C_2_5 - C_2_6);
                                        const double tmp1 = w42*(C_1_3 + C_1_7);
                                        const double tmp2 = w41*(C_0_3 + C_0_7);
                                        const double tmp3 = w37*(C_1_1 + C_1_5);
                                        const double tmp4 = w39*(C_0_2 + C_0_6);
                                        const double tmp5 = w45*(-C_2_1 - C_2_2);
                                        const double tmp6 = w36*(C_0_1 + C_0_5);
                                        const double tmp7 = w40*(C_1_2 + C_1_6);
                                        const double tmp8 = w33*(C_0_0 + C_0_4);
                                        const double tmp9 = w34*(C_1_0 + C_1_4);
                                        const double tmp10 = w38*(C_2_4 + C_2_5 + C_2_6 + C_2_7);
                                        const double tmp11 = w42*(C_1_4 + C_1_5);
                                        const double tmp12 = w41*(C_0_4 + C_0_6);
                                        const double tmp13 = w37*(C_1_6 + C_1_7);
                                        const double tmp14 = w39*(C_0_5 + C_0_7);
                                        const double tmp15 = w45*(C_2_0 + C_2_1 + C_2_2 + C_2_3);
                                        const double tmp16 = w36*(C_0_0 + C_0_2);
                                        const double tmp17 = w40*(C_1_0 + C_1_1);
                                        const double tmp18 = w33*(C_0_1 + C_0_3);
                                        const double tmp19 = w34*(C_1_2 + C_1_3);
                                        const double tmp20 = w38*(-C_2_5 - C_2_7);
                                        const double tmp21 = w35*(-C_2_4 - C_2_6);
                                        const double tmp22 = w41*(C_0_1 + C_0_3);
                                        const double tmp23 = w37*(C_1_0 + C_1_5);
                                        const double tmp24 = w39*(C_0_0 + C_0_2);
                                        const double tmp25 = w45*(-C_2_0 - C_2_2);
                                        const double tmp26 = w36*(C_0_5 + C_0_7);
                                        const double tmp27 = w40*(C_1_2 + C_1_7);
                                        const double tmp28 = w33*(C_0_4 + C_0_6);
                                        const double tmp29 = w46*(-C_2_1 - C_2_3);
                                        const double tmp30 = w38*(C_2_0 + C_2_2);
                                        const double tmp31 = w35*(C_2_1 + C_2_3);
                                        const double tmp32 = w41*(-C_0_4 - C_0_6);
                                        const double tmp33 = w37*(-C_1_2 - C_1_7);
                                        const double tmp34 = w39*(-C_0_5 - C_0_7);
                                        const double tmp35 = w45*(C_2_5 + C_2_7);
                                        const double tmp36 = w36*(-C_0_0 - C_0_2);
                                        const double tmp37 = w40*(-C_1_0 - C_1_5);
                                        const double tmp38 = w33*(-C_0_1 - C_0_3);
                                        const double tmp39 = w46*(C_2_4 + C_2_6);
                                        const double tmp40 = w38*(-C_2_0 - C_2_1 - C_2_2 - C_2_3);
                                        const double tmp41 = w42*(-C_1_2 - C_1_3);
                                        const double tmp42 = w41*(-C_0_1 - C_0_3);
                                        const double tmp43 = w37*(-C_1_0 - C_1_1);
                                        const double tmp44 = w39*(-C_0_0 - C_0_2);
                                        const double tmp45 = w45*(-C_2_4 - C_2_5 - C_2_6 - C_2_7);
                                        const double tmp46 = w36*(-C_0_5 - C_0_7);
                                        const double tmp47 = w40*(-C_1_6 - C_1_7);
                                        const double tmp48 = w33*(-C_0_4 - C_0_6);
                                        const double tmp49 = w34*(-C_1_4 - C_1_5);
                                        const double tmp50 = w38*(C_2_0 + C_2_1);
                                        const double tmp51 = w42*(-C_1_4 - C_1_5);
                                        const double tmp52 = w35*(C_2_2 + C_2_3);
                                        const double tmp53 = w37*(-C_1_6 - C_1_7);
                                        const double tmp54 = w39*(-C_0_1 - C_0_7);
                                        const double tmp55 = w45*(C_2_6 + C_2_7);
                                        const double tmp56 = w36*(-C_0_0 - C_0_6);
                                        const double tmp57 = w40*(-C_1_0 - C_1_1);
                                        const double tmp58 = w46*(C_2_4 + C_2_5);
                                        const double tmp59 = w34*(-C_1_2 - C_1_3);
                                        const double tmp60 = w38*(C_2_0 + C_2_1 + C_2_2 + C_2_3);
                                        const double tmp61 = w37*(C_1_0 + C_1_1 + C_1_4 + C_1_5);
                                        const double tmp62 = w39*(C_0_0 + C_0_2 + C_0_4 + C_0_6);
                                        const double tmp63 = w45*(C_2_4 + C_2_5 + C_2_6 + C_2_7);
                                        const double tmp64 = w36*(C_0_1 + C_0_3 + C_0_5 + C_0_7);
                                        const double tmp65 = w40*(C_1_2 + C_1_3 + C_1_6 + C_1_7);
                                        const double tmp66 = w41*(-C_0_5 - C_0_7);
                                        const double tmp67 = w39*(-C_0_4 - C_0_6);
                                        const double tmp68 = w36*(-C_0_1 - C_0_3);
                                        const double tmp69 = w33*(-C_0_0 - C_0_2);
                                        const double tmp70 = w38*(C_2_0 + C_2_3);
                                        const double tmp71 = w42*(C_1_2 + C_1_6);
                                        const double tmp72 = w41*(-C_0_2 - C_0_6);
                                        const double tmp73 = w37*(C_1_0 + C_1_4);
                                        const double tmp74 = w39*(-C_0_3 - C_0_7);
                                        const double tmp75 = w45*(C_2_4 + C_2_7);
                                        const double tmp76 = w36*(-C_0_0 - C_0_4);
                                        const double tmp77 = w40*(C_1_3 + C_1_7);
                                        const double tmp78 = w33*(-C_0_1 - C_0_5);
                                        const double tmp79 = w34*(C_1_1 + C_1_5);
                                        const double tmp80 = w39*(-C_0_1 - C_0_3 - C_0_5 - C_0_7);
                                        const double tmp81 = w36*(-C_0_0 - C_0_2 - C_0_4 - C_0_6);
                                        const double tmp82 = w38*(-C_2_4 - C_2_7);
                                        const double tmp83 = w42*(-C_1_1 - C_1_5);
                                        const double tmp84 = w41*(C_0_1 + C_0_5);
                                        const double tmp85 = w37*(-C_1_3 - C_1_7);
                                        const double tmp86 = w39*(C_0_0 + C_0_4);
                                        const double tmp87 = w45*(-C_2_0 - C_2_3);
                                        const double tmp88 = w36*(C_0_3 + C_0_7);
                                        const double tmp89 = w40*(-C_1_0 - C_1_4);
                                        const double tmp90 = w33*(C_0_2 + C_0_6);
                                        const double tmp91 = w34*(-C_1_2 - C_1_6);
                                        const double tmp92 = w38*(C_2_1 + C_2_2);
                                        const double tmp93 = w45*(C_2_5 + C_2_6);
                                        const double tmp94 = w37*(-C_1_2 - C_1_3 - C_1_6 - C_1_7);
                                        const double tmp95 = w40*(-C_1_0 - C_1_1 - C_1_4 - C_1_5);
                                        const double tmp96 = w42*(C_1_0 + C_1_1);
                                        const double tmp97 = w41*(C_0_0 + C_0_2);
                                        const double tmp98 = w37*(C_1_2 + C_1_3);
                                        const double tmp99 = w39*(C_0_1 + C_0_3);
                                        const double tmp100 = w36*(C_0_4 + C_0_6);
                                        const double tmp101 = w40*(C_1_4 + C_1_5);
                                        const double tmp102 = w33*(C_0_5 + C_0_7);
                                        const double tmp103 = w34*(C_1_6 + C_1_7);
                                        const double tmp104 = w38*(-C_2_2 - C_2_3);
                                        const double tmp105 = w35*(-C_2_0 - C_2_1);
                                        const double tmp106 = w41*(-C_0_3 - C_0_7);
                                        const double tmp107 = w37*(C_1_2 + C_1_3 + C_1_6 + C_1_7);
                                        const double tmp108 = w39*(-C_0_2 - C_0_6);
                                        const double tmp109 = w45*(-C_2_4 - C_2_5);
                                        const double tmp110 = w36*(-C_0_1 - C_0_5);
                                        const double tmp111 = w40*(C_1_0 + C_1_1 + C_1_4 + C_1_5);
                                        const double tmp112 = w33*(-C_0_0 - C_0_4);
                                        const double tmp113 = w46*(-C_2_6 - C_2_7);
                                        const double tmp114 = w42*(-C_1_0 - C_1_4);
                                        const double tmp115 = w41*(-C_0_0 - C_0_4);
                                        const double tmp116 = w37*(-C_1_2 - C_1_6);
                                        const double tmp117 = w39*(-C_0_1 - C_0_5);
                                        const double tmp118 = w36*(-C_0_2 - C_0_6);
                                        const double tmp119 = w40*(-C_1_1 - C_1_5);
                                        const double tmp120 = w33*(-C_0_3 - C_0_7);
                                        const double tmp121 = w34*(-C_1_3 - C_1_7);
                                        const double tmp122 = w38*(C_2_2 + C_2_3);
                                        const double tmp123 = w42*(C_1_6 + C_1_7);
                                        const double tmp124 = w35*(C_2_0 + C_2_1);
                                        const double tmp125 = w37*(C_1_4 + C_1_5);
                                        const double tmp126 = w39*(C_0_2 + C_0_4);
                                        const double tmp127 = w45*(C_2_4 + C_2_5);
                                        const double tmp128 = w36*(C_0_3 + C_0_5);
                                        const double tmp129 = w40*(C_1_2 + C_1_3);
                                        const double tmp130 = w46*(C_2_6 + C_2_7);
                                        const double tmp131 = w34*(C_1_0 + C_1_1);
                                        const double tmp132 = w38*(-C_2_1 - C_2_2);
                                        const double tmp133 = w37*(C_1_2 + C_1_7);
                                        const double tmp134 = w39*(C_0_1 + C_0_7);
                                        const double tmp135 = w36*(C_0_0 + C_0_6);
                                        const double tmp136 = w40*(C_1_0 + C_1_5);
                                        const double tmp137 = w45*(-C_2_5 - C_2_6);
                                        const double tmp138 = w38*(-C_2_4 - C_2_6);
                                        const double tmp139 = w35*(-C_2_5 - C_2_7);
                                        const double tmp140 = w41*(-C_0_0 - C_0_2);
                                        const double tmp141 = w37*(-C_1_3 - C_1_6);
                                        const double tmp142 = w39*(-C_0_1 - C_0_3);
                                        const double tmp143 = w45*(-C_2_1 - C_2_3);
                                        const double tmp144 = w36*(-C_0_4 - C_0_6);
                                        const double tmp145 = w40*(-C_1_1 - C_1_4);
                                        const double tmp146 = w33*(-C_0_5 - C_0_7);
                                        const double tmp147 = w46*(-C_2_0 - C_2_2);
                                        const double tmp148 = w39*(-C_0_3 - C_0_5);
                                        const double tmp149 = w36*(-C_0_2 - C_0_4);
                                        const double tmp150 = w38*(C_2_5 + C_2_6);
                                        const double tmp151 = w37*(-C_1_0 - C_1_5);
                                        const double tmp152 = w39*(-C_0_0 - C_0_6);
                                        const double tmp153 = w45*(C_2_1 + C_2_2);
                                        const double tmp154 = w36*(-C_0_1 - C_0_7);
                                        const double tmp155 = w40*(-C_1_2 - C_1_7);
                                        const double tmp156 = w41*(C_0_2 + C_0_6);
                                        const double tmp157 = w39*(C_0_3 + C_0_7);
                                        const double tmp158 = w36*(C_0_0 + C_0_4);
                                        const double tmp159 = w33*(C_0_1 + C_0_5);
                                        const double tmp160 = w38*(C_2_6 + C_2_7);
                                        const double tmp161 = w35*(C_2_4 + C_2_5);
                                        const double tmp162 = w45*(C_2_0 + C_2_1);
                                        const double tmp163 = w46*(C_2_2 + C_2_3);
                                        const double tmp164 = w38*(-C_2_0 - C_2_3);
                                        const double tmp165 = w37*(C_1_3 + C_1_6);
                                        const double tmp166 = w40*(C_1_1 + C_1_4);
                                        const double tmp167 = w45*(-C_2_4 - C_2_7);
                                        const double tmp168 = w39*(C_0_3 + C_0_5);
                                        const double tmp169 = w36*(C_0_2 + C_0_4);
                                        const double tmp170 = w38*(C_2_1 + C_2_3);
                                        const double tmp171 = w35*(C_2_0 + C_2_2);
                                        const double tmp172 = w41*(C_0_5 + C_0_7);
                                        const double tmp173 = w37*(C_1_1 + C_1_4);
                                        const double tmp174 = w39*(C_0_4 + C_0_6);
                                        const double tmp175 = w45*(C_2_4 + C_2_6);
                                        const double tmp176 = w36*(C_0_1 + C_0_3);
                                        const double tmp177 = w40*(C_1_3 + C_1_6);
                                        const double tmp178 = w33*(C_0_0 + C_0_2);
                                        const double tmp179 = w46*(C_2_5 + C_2_7);
                                        const double tmp180 = w38*(-C_2_1 - C_2_3);
                                        const double tmp181 = w42*(C_1_1 + C_1_5);
                                        const double tmp182 = w35*(-C_2_0 - C_2_2);
                                        const double tmp183 = w37*(C_1_3 + C_1_7);
                                        const double tmp184 = w39*(C_0_1 + C_0_3 + C_0_5 + C_0_7);
                                        const double tmp185 = w45*(-C_2_4 - C_2_6);
                                        const double tmp186 = w36*(C_0_0 + C_0_2 + C_0_4 + C_0_6);
                                        const double tmp187 = w40*(C_1_0 + C_1_4);
                                        const double tmp188 = w46*(-C_2_5 - C_2_7);
                                        const double tmp189 = w34*(C_1_2 + C_1_6);
                                        const double tmp190 = w38*(-C_2_0 - C_2_1);
                                        const double tmp191 = w35*(-C_2_2 - C_2_3);
                                        const double tmp192 = w41*(C_0_0 + C_0_4);
                                        const double tmp193 = w37*(-C_1_0 - C_1_1 - C_1_4 - C_1_5);
                                        const double tmp194 = w39*(C_0_1 + C_0_5);
                                        const double tmp195 = w45*(-C_2_6 - C_2_7);
                                        const double tmp196 = w36*(C_0_2 + C_0_6);
                                        const double tmp197 = w40*(-C_1_2 - C_1_3 - C_1_6 - C_1_7);
                                        const double tmp198 = w33*(C_0_3 + C_0_7);
                                        const double tmp199 = w46*(-C_2_4 - C_2_5);
                                        const double tmp200 = w38*(-C_2_6 - C_2_7);
                                        const double tmp201 = w42*(C_1_2 + C_1_3);
                                        const double tmp202 = w35*(-C_2_4 - C_2_5);
                                        const double tmp203 = w37*(C_1_0 + C_1_1);
                                        const double tmp204 = w45*(-C_2_0 - C_2_1);
                                        const double tmp205 = w40*(C_1_6 + C_1_7);
                                        const double tmp206 = w46*(-C_2_2 - C_2_3);
                                        const double tmp207 = w34*(C_1_4 + C_1_5);
                                        const double tmp208 = w37*(-C_1_1 - C_1_4);
                                        const double tmp209 = w39*(-C_0_2 - C_0_4);
                                        const double tmp210 = w36*(-C_0_3 - C_0_5);
                                        const double tmp211 = w40*(-C_1_3 - C_1_6);
                                        const double tmp212 = w38*(C_2_4 + C_2_7);
                                        const double tmp213 = w45*(C_2_0 + C_2_3);
                                        const double tmp214 = w41*(-C_0_1 - C_0_5);
                                        const double tmp215 = w39*(-C_0_0 - C_0_4);
                                        const double tmp216 = w36*(-C_0_3 - C_0_7);
                                        const double tmp217 = w33*(-C_0_2 - C_0_6);
                                        const double tmp218 = w42*(-C_1_3 - C_1_7);
                                        const double tmp219 = w37*(-C_1_1 - C_1_5);
                                        const double tmp220 = w40*(-C_1_2 - C_1_6);
                                        const double tmp221 = w34*(-C_1_0 - C_1_4);
                                        const double tmp222 = w39*(C_0_0 + C_0_6);
                                        const double tmp223 = w36*(C_0_1 + C_0_7);
                                        const double tmp224 = w38*(C_2_4 + C_2_5);
                                        const double tmp225 = w35*(C_2_6 + C_2_7);
                                        const double tmp226 = w45*(C_2_2 + C_2_3);
                                        const double tmp227 = w46*(C_2_0 + C_2_1);
                                        const double tmp228 = w38*(-C_2_0 - C_2_2);
                                        const double tmp229 = w42*(-C_1_2 - C_1_6);
                                        const double tmp230 = w35*(-C_2_1 - C_2_3);
                                        const double tmp231 = w37*(-C_1_0 - C_1_4);
                                        const double tmp232 = w39*(-C_0_0 - C_0_2 - C_0_4 - C_0_6);
                                        const double tmp233 = w45*(-C_2_5 - C_2_7);
                                        const double tmp234 = w36*(-C_0_1 - C_0_3 - C_0_5 - C_0_7);
                                        const double tmp235 = w40*(-C_1_3 - C_1_7);
                                        const double tmp236 = w46*(-C_2_4 - C_2_6);
                                        const double tmp237 = w34*(-C_1_1 - C_1_5);
                                        const double tmp238 = w42*(C_1_0 + C_1_4);
                                        const double tmp239 = w37*(C_1_2 + C_1_6);
                                        const double tmp240 = w40*(C_1_1 + C_1_5);
                                        const double tmp241 = w34*(C_1_3 + C_1_7);
                                        const double tmp242 = w38*(-C_2_4 - C_2_5);
                                        const double tmp243 = w42*(-C_1_0 - C_1_1);
                                        const double tmp244 = w35*(-C_2_6 - C_2_7);
                                        const double tmp245 = w37*(-C_1_2 - C_1_3);
                                        const double tmp246 = w45*(-C_2_2 - C_2_3);
                                        const double tmp247 = w40*(-C_1_4 - C_1_5);
                                        const double tmp248 = w46*(-C_2_0 - C_2_1);
                                        const double tmp249 = w34*(-C_1_6 - C_1_7);
                                        const double tmp250 = w42*(-C_1_6 - C_1_7);
                                        const double tmp251 = w37*(-C_1_4 - C_1_5);
                                        const double tmp252 = w40*(-C_1_2 - C_1_3);
                                        const double tmp253 = w34*(-C_1_0 - C_1_1);
                                        const double tmp254 = w38*(C_2_5 + C_2_7);
                                        const double tmp255 = w35*(C_2_4 + C_2_6);
                                        const double tmp256 = w45*(C_2_0 + C_2_2);
                                        const double tmp257 = w46*(C_2_1 + C_2_3);
                                        const double tmp258 = w38*(-C_2_4 - C_2_5 - C_2_6 - C_2_7);
                                        const double tmp259 = w45*(-C_2_0 - C_2_1 - C_2_2 - C_2_3);
                                        const double tmp260 = w38*(C_2_4 + C_2_6);
                                        const double tmp261 = w35*(C_2_5 + C_2_7);
                                        const double tmp262 = w45*(C_2_1 + C_2_3);
                                        const double tmp263 = w46*(C_2_0 + C_2_2);
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=-C_0_0*w50 - C_0_1*w41 - C_0_6*w33 - C_0_7*w49 + C_1_0*w47 - C_1_2*w42 - C_1_5*w34 + C_1_7*w48 - C_2_0*w43 - C_2_3*w35 - C_2_4*w46 - C_2_7*w44 + tmp132 + tmp137 + tmp208 + tmp209 + tmp210 + tmp211;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=C_0_0*w50 + C_0_1*w41 + C_0_6*w33 + C_0_7*w49 + tmp126 + tmp128 + tmp242 + tmp243 + tmp244 + tmp245 + tmp246 + tmp247 + tmp248 + tmp249;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=-C_1_0*w47 + C_1_2*w42 + C_1_5*w34 - C_1_7*w48 + tmp138 + tmp139 + tmp140 + tmp142 + tmp143 + tmp144 + tmp146 + tmp147 + tmp173 + tmp177;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=tmp100 + tmp101 + tmp102 + tmp103 + tmp40 + tmp45 + tmp96 + tmp97 + tmp98 + tmp99;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=C_2_0*w43 + C_2_3*w35 + C_2_4*w46 + C_2_7*w44 + tmp114 + tmp115 + tmp116 + tmp117 + tmp118 + tmp119 + tmp120 + tmp121 + tmp92 + tmp93;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=tmp192 + tmp193 + tmp194 + tmp196 + tmp197 + tmp198 + tmp224 + tmp225 + tmp226 + tmp227;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=tmp232 + tmp234 + tmp238 + tmp239 + tmp240 + tmp241 + tmp260 + tmp261 + tmp262 + tmp263;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=tmp60 + tmp61 + tmp62 + tmp63 + tmp64 + tmp65;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=-C_0_0*w41 - C_0_1*w50 - C_0_6*w49 - C_0_7*w33 + tmp148 + tmp149 + tmp242 + tmp243 + tmp244 + tmp245 + tmp246 + tmp247 + tmp248 + tmp249;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=C_0_0*w41 + C_0_1*w50 + C_0_6*w49 + C_0_7*w33 + C_1_1*w47 - C_1_3*w42 - C_1_4*w34 + C_1_6*w48 - C_2_1*w43 - C_2_2*w35 - C_2_5*w46 - C_2_6*w44 + tmp151 + tmp155 + tmp164 + tmp167 + tmp168 + tmp169;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=tmp101 + tmp103 + tmp40 + tmp42 + tmp44 + tmp45 + tmp46 + tmp48 + tmp96 + tmp98;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=-C_1_1*w47 + C_1_3*w42 + C_1_4*w34 - C_1_6*w48 + tmp20 + tmp21 + tmp22 + tmp23 + tmp24 + tmp25 + tmp26 + tmp27 + tmp28 + tmp29;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=tmp193 + tmp197 + tmp214 + tmp215 + tmp216 + tmp217 + tmp224 + tmp225 + tmp226 + tmp227;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=C_2_1*w43 + C_2_2*w35 + C_2_5*w46 + C_2_6*w44 + tmp70 + tmp75 + tmp83 + tmp84 + tmp85 + tmp86 + tmp88 + tmp89 + tmp90 + tmp91;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=tmp60 + tmp61 + tmp63 + tmp65 + tmp80 + tmp81;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=tmp181 + tmp183 + tmp184 + tmp186 + tmp187 + tmp189 + tmp254 + tmp255 + tmp256 + tmp257;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=-C_1_0*w42 + C_1_2*w47 + C_1_5*w48 - C_1_7*w34 + tmp138 + tmp139 + tmp140 + tmp141 + tmp142 + tmp143 + tmp144 + tmp145 + tmp146 + tmp147;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=tmp100 + tmp102 + tmp40 + tmp41 + tmp43 + tmp45 + tmp47 + tmp49 + tmp97 + tmp99;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=-C_0_2*w50 - C_0_3*w41 - C_0_4*w33 - C_0_5*w49 + C_1_0*w42 - C_1_2*w47 - C_1_5*w48 + C_1_7*w34 - C_2_1*w35 - C_2_2*w43 - C_2_5*w44 - C_2_6*w46 + tmp152 + tmp154 + tmp164 + tmp165 + tmp166 + tmp167;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=C_0_2*w50 + C_0_3*w41 + C_0_4*w33 + C_0_5*w49 + tmp200 + tmp201 + tmp202 + tmp203 + tmp204 + tmp205 + tmp206 + tmp207 + tmp222 + tmp223;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=tmp229 + tmp231 + tmp232 + tmp234 + tmp235 + tmp237 + tmp260 + tmp261 + tmp262 + tmp263;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=tmp60 + tmp62 + tmp63 + tmp64 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=C_2_1*w35 + C_2_2*w43 + C_2_5*w44 + C_2_6*w46 + tmp70 + tmp71 + tmp72 + tmp73 + tmp74 + tmp75 + tmp76 + tmp77 + tmp78 + tmp79;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=tmp107 + tmp111 + tmp156 + tmp157 + tmp158 + tmp159 + tmp160 + tmp161 + tmp162 + tmp163;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=tmp40 + tmp41 + tmp42 + tmp43 + tmp44 + tmp45 + tmp46 + tmp47 + tmp48 + tmp49;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=-C_1_1*w42 + C_1_3*w47 + C_1_4*w48 - C_1_6*w34 + tmp20 + tmp21 + tmp22 + tmp24 + tmp25 + tmp26 + tmp28 + tmp29 + tmp33 + tmp37;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=-C_0_2*w41 - C_0_3*w50 - C_0_4*w49 - C_0_5*w33 + tmp200 + tmp201 + tmp202 + tmp203 + tmp204 + tmp205 + tmp206 + tmp207 + tmp54 + tmp56;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=C_0_2*w41 + C_0_3*w50 + C_0_4*w49 + C_0_5*w33 + C_1_1*w42 - C_1_3*w47 - C_1_4*w48 + C_1_6*w34 - C_2_0*w35 - C_2_3*w43 - C_2_4*w44 - C_2_7*w46 + tmp132 + tmp133 + tmp134 + tmp135 + tmp136 + tmp137;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=tmp60 + tmp63 + tmp80 + tmp81 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=tmp184 + tmp186 + tmp218 + tmp219 + tmp220 + tmp221 + tmp254 + tmp255 + tmp256 + tmp257;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=tmp106 + tmp107 + tmp108 + tmp110 + tmp111 + tmp112 + tmp160 + tmp161 + tmp162 + tmp163;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=C_2_0*w35 + C_2_3*w43 + C_2_4*w44 + C_2_7*w46 + tmp1 + tmp2 + tmp3 + tmp4 + tmp6 + tmp7 + tmp8 + tmp9 + tmp92 + tmp93;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=-C_2_0*w46 - C_2_3*w44 - C_2_4*w43 - C_2_7*w35 + tmp0 + tmp114 + tmp115 + tmp116 + tmp117 + tmp118 + tmp119 + tmp120 + tmp121 + tmp5;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=tmp190 + tmp191 + tmp192 + tmp193 + tmp194 + tmp195 + tmp196 + tmp197 + tmp198 + tmp199;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=tmp228 + tmp230 + tmp232 + tmp233 + tmp234 + tmp236 + tmp238 + tmp239 + tmp240 + tmp241;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=tmp258 + tmp259 + tmp61 + tmp62 + tmp64 + tmp65;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=-C_0_2*w33 - C_0_3*w49 - C_0_4*w50 - C_0_5*w41 - C_1_1*w34 + C_1_3*w48 + C_1_4*w47 - C_1_6*w42 + C_2_0*w46 + C_2_3*w44 + C_2_4*w43 + C_2_7*w35 + tmp150 + tmp151 + tmp152 + tmp153 + tmp154 + tmp155;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=C_0_2*w33 + C_0_3*w49 + C_0_4*w50 + C_0_5*w41 + tmp222 + tmp223 + tmp50 + tmp51 + tmp52 + tmp53 + tmp55 + tmp57 + tmp58 + tmp59;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=C_1_1*w34 - C_1_3*w48 - C_1_4*w47 + C_1_6*w42 + tmp23 + tmp27 + tmp30 + tmp31 + tmp32 + tmp34 + tmp35 + tmp36 + tmp38 + tmp39;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=tmp10 + tmp11 + tmp12 + tmp13 + tmp14 + tmp15 + tmp16 + tmp17 + tmp18 + tmp19;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=tmp190 + tmp191 + tmp193 + tmp195 + tmp197 + tmp199 + tmp214 + tmp215 + tmp216 + tmp217;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=-C_2_1*w46 - C_2_2*w44 - C_2_5*w43 - C_2_6*w35 + tmp82 + tmp83 + tmp84 + tmp85 + tmp86 + tmp87 + tmp88 + tmp89 + tmp90 + tmp91;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=tmp258 + tmp259 + tmp61 + tmp65 + tmp80 + tmp81;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=tmp180 + tmp181 + tmp182 + tmp183 + tmp184 + tmp185 + tmp186 + tmp187 + tmp188 + tmp189;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=-C_0_2*w49 - C_0_3*w33 - C_0_4*w41 - C_0_5*w50 + tmp50 + tmp51 + tmp52 + tmp53 + tmp54 + tmp55 + tmp56 + tmp57 + tmp58 + tmp59;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=C_0_2*w49 + C_0_3*w33 + C_0_4*w41 + C_0_5*w50 - C_1_0*w34 + C_1_2*w48 + C_1_5*w47 - C_1_7*w42 + C_2_1*w46 + C_2_2*w44 + C_2_5*w43 + C_2_6*w35 + tmp134 + tmp135 + tmp208 + tmp211 + tmp212 + tmp213;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=tmp10 + tmp11 + tmp13 + tmp15 + tmp17 + tmp19 + tmp66 + tmp67 + tmp68 + tmp69;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=C_1_0*w34 - C_1_2*w48 - C_1_5*w47 + C_1_7*w42 + tmp170 + tmp171 + tmp172 + tmp173 + tmp174 + tmp175 + tmp176 + tmp177 + tmp178 + tmp179;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=tmp228 + tmp229 + tmp230 + tmp231 + tmp232 + tmp233 + tmp234 + tmp235 + tmp236 + tmp237;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=tmp258 + tmp259 + tmp62 + tmp64 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=-C_2_1*w44 - C_2_2*w46 - C_2_5*w35 - C_2_6*w43 + tmp71 + tmp72 + tmp73 + tmp74 + tmp76 + tmp77 + tmp78 + tmp79 + tmp82 + tmp87;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=tmp104 + tmp105 + tmp107 + tmp109 + tmp111 + tmp113 + tmp156 + tmp157 + tmp158 + tmp159;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=C_1_1*w48 - C_1_3*w34 - C_1_4*w42 + C_1_6*w47 + tmp30 + tmp31 + tmp32 + tmp33 + tmp34 + tmp35 + tmp36 + tmp37 + tmp38 + tmp39;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=tmp10 + tmp12 + tmp14 + tmp15 + tmp16 + tmp18 + tmp250 + tmp251 + tmp252 + tmp253;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=-C_0_0*w33 - C_0_1*w49 - C_0_6*w50 - C_0_7*w41 - C_1_1*w48 + C_1_3*w34 + C_1_4*w42 - C_1_6*w47 + C_2_1*w44 + C_2_2*w46 + C_2_5*w35 + C_2_6*w43 + tmp133 + tmp136 + tmp209 + tmp210 + tmp212 + tmp213;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=C_0_0*w33 + C_0_1*w49 + C_0_6*w50 + C_0_7*w41 + tmp122 + tmp123 + tmp124 + tmp125 + tmp126 + tmp127 + tmp128 + tmp129 + tmp130 + tmp131;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=tmp258 + tmp259 + tmp80 + tmp81 + tmp94 + tmp95;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=tmp180 + tmp182 + tmp184 + tmp185 + tmp186 + tmp188 + tmp218 + tmp219 + tmp220 + tmp221;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=tmp104 + tmp105 + tmp106 + tmp107 + tmp108 + tmp109 + tmp110 + tmp111 + tmp112 + tmp113;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=-C_2_0*w44 - C_2_3*w46 - C_2_4*w35 - C_2_7*w43 + tmp0 + tmp1 + tmp2 + tmp3 + tmp4 + tmp5 + tmp6 + tmp7 + tmp8 + tmp9;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=tmp10 + tmp15 + tmp250 + tmp251 + tmp252 + tmp253 + tmp66 + tmp67 + tmp68 + tmp69;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=C_1_0*w48 - C_1_2*w34 - C_1_5*w42 + C_1_7*w47 + tmp141 + tmp145 + tmp170 + tmp171 + tmp172 + tmp174 + tmp175 + tmp176 + tmp178 + tmp179;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=-C_0_0*w49 - C_0_1*w33 - C_0_6*w41 - C_0_7*w50 + tmp122 + tmp123 + tmp124 + tmp125 + tmp127 + tmp129 + tmp130 + tmp131 + tmp148 + tmp149;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=C_0_0*w49 + C_0_1*w33 + C_0_6*w41 + C_0_7*w50 - C_1_0*w48 + C_1_2*w34 + C_1_5*w42 - C_1_7*w47 + C_2_0*w44 + C_2_3*w46 + C_2_4*w35 + C_2_7*w43 + tmp150 + tmp153 + tmp165 + tmp166 + tmp168 + tmp169;
                                    }
                                }
                            } else { // constant data
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double wC0 = C_p[INDEX3(k,m,0,numEq,numComp)]*w55;
                                        const double wC1 = C_p[INDEX3(k,m,1,numEq,numComp)]*w56;
                                        const double wC2 = C_p[INDEX3(k,m,2,numEq,numComp)]*w54;
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+= 4*wC0 + 4*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=-4*wC0 + 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+= 2*wC0 - 4*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=-2*wC0 - 2*wC1 +   wC2;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+= 2*wC0 + 2*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=-2*wC0 +   wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=   wC0 - 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=  -wC0 -   wC1 -   wC2;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+= 4*wC0 + 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=-4*wC0 + 4*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+= 2*wC0 - 2*wC1 +   wC2;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=-2*wC0 - 4*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+= 2*wC0 +   wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=-2*wC0 + 2*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=   wC0 -   wC1 -   wC2;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=  -wC0 - 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+= 2*wC0 + 4*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=-2*wC0 + 2*wC1 +   wC2;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+= 4*wC0 - 4*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=-4*wC0 - 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=   wC0 + 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=  -wC0 +   wC1 -   wC2;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+= 2*wC0 - 2*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=-2*wC0 -   wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+= 2*wC0 + 2*wC1 +   wC2;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=-2*wC0 + 4*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+= 4*wC0 - 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=-4*wC0 - 4*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=   wC0 +   wC1 -   wC2;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=  -wC0 + 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+= 2*wC0 -   wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=-2*wC0 - 2*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+= 2*wC0 + 2*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=-2*wC0 +   wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=   wC0 - 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=  -wC0 -   wC1 +   wC2;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+= 4*wC0 + 4*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=-4*wC0 + 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+= 2*wC0 - 4*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=-2*wC0 - 2*wC1 -   wC2;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+= 2*wC0 +   wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=-2*wC0 + 2*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=   wC0 -   wC1 +   wC2;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=  -wC0 - 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+= 4*wC0 + 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=-4*wC0 + 4*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+= 2*wC0 - 2*wC1 -   wC2;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=-2*wC0 - 4*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=   wC0 + 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=  -wC0 +   wC1 +   wC2;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+= 2*wC0 - 2*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=-2*wC0 -   wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+= 2*wC0 + 4*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=-2*wC0 + 2*wC1 -   wC2;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+= 4*wC0 - 4*wC1 - 4*wC2;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=-4*wC0 - 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=   wC0 +   wC1 +   wC2;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=  -wC0 + 2*wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+= 2*wC0 -   wC1 + 2*wC2;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=-2*wC0 - 2*wC1 + 4*wC2;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+= 2*wC0 + 2*wC1 -   wC2;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=-2*wC0 + 4*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+= 4*wC0 - 2*wC1 - 2*wC2;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=-4*wC0 - 4*wC1 - 4*wC2;
                                    }
                                }
                            }
                        }
                        ///////////////
                        // process D //
                        ///////////////
                        if (!D.isEmpty()) {
                            const double* D_p=D.getSampleDataRO(e);
                            if (D.actsExpanded()) {
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double D_0 = D_p[INDEX3(k,m,0,numEq,numComp)];
                                        const double D_1 = D_p[INDEX3(k,m,1,numEq,numComp)];
                                        const double D_2 = D_p[INDEX3(k,m,2,numEq,numComp)];
                                        const double D_3 = D_p[INDEX3(k,m,3,numEq,numComp)];
                                        const double D_4 = D_p[INDEX3(k,m,4,numEq,numComp)];
                                        const double D_5 = D_p[INDEX3(k,m,5,numEq,numComp)];
                                        const double D_6 = D_p[INDEX3(k,m,6,numEq,numComp)];
                                        const double D_7 = D_p[INDEX3(k,m,7,numEq,numComp)];
                                        const double tmp0 = w59*(D_3 + D_7);
                                        const double tmp1 = w57*(D_0 + D_4);
                                        const double tmp2 = w58*(D_1 + D_2 + D_5 + D_6);
                                        const double tmp3 = w60*(D_0 + D_1 + D_2 + D_3);
                                        const double tmp4 = w61*(D_4 + D_5 + D_6 + D_7);
                                        const double tmp5 = w59*(D_1 + D_3);
                                        const double tmp6 = w57*(D_4 + D_6);
                                        const double tmp7 = w58*(D_0 + D_2 + D_5 + D_7);
                                        const double tmp8 = w59*(D_4 + D_6);
                                        const double tmp9 = w57*(D_1 + D_3);
                                        const double tmp10 = w60*(D_4 + D_5 + D_6 + D_7);
                                        const double tmp11 = w61*(D_0 + D_1 + D_2 + D_3);
                                        const double tmp12 = w59*(D_4 + D_5);
                                        const double tmp13 = w57*(D_2 + D_3);
                                        const double tmp14 = w58*(D_0 + D_1 + D_6 + D_7);
                                        const double tmp15 = w58*(D_0 + D_1 + D_2 + D_3 + D_4 + D_5 + D_6 + D_7);
                                        const double tmp16 = w59*(D_2 + D_6);
                                        const double tmp17 = w57*(D_1 + D_5);
                                        const double tmp18 = w58*(D_0 + D_3 + D_4 + D_7);
                                        const double tmp19 = w59*(D_1 + D_5);
                                        const double tmp20 = w57*(D_2 + D_6);
                                        const double tmp21 = w60*(D_0 + D_1 + D_4 + D_5);
                                        const double tmp22 = w61*(D_2 + D_3 + D_6 + D_7);
                                        const double tmp23 = w59*(D_0 + D_4);
                                        const double tmp24 = w57*(D_3 + D_7);
                                        const double tmp25 = w59*(D_6 + D_7);
                                        const double tmp26 = w57*(D_0 + D_1);
                                        const double tmp27 = w58*(D_2 + D_3 + D_4 + D_5);
                                        const double tmp28 = w60*(D_0 + D_5 + D_6);
                                        const double tmp29 = w61*(D_1 + D_2 + D_7);
                                        const double tmp30 = w59*(D_0 + D_2);
                                        const double tmp31 = w57*(D_5 + D_7);
                                        const double tmp32 = w58*(D_1 + D_3 + D_4 + D_6);
                                        const double tmp33 = w60*(D_1 + D_2 + D_7);
                                        const double tmp34 = w61*(D_0 + D_5 + D_6);
                                        const double tmp35 = w60*(D_1 + D_4 + D_7);
                                        const double tmp36 = w61*(D_0 + D_3 + D_6);
                                        const double tmp37 = w60*(D_1 + D_2 + D_4);
                                        const double tmp38 = w61*(D_3 + D_5 + D_6);
                                        const double tmp39 = w59*(D_5 + D_7);
                                        const double tmp40 = w57*(D_0 + D_2);
                                        const double tmp41 = w60*(D_0 + D_2 + D_4 + D_6);
                                        const double tmp42 = w61*(D_1 + D_3 + D_5 + D_7);
                                        const double tmp43 = w60*(D_2 + D_3 + D_6 + D_7);
                                        const double tmp44 = w61*(D_0 + D_1 + D_4 + D_5);
                                        const double tmp45 = w60*(D_2 + D_4 + D_7);
                                        const double tmp46 = w61*(D_0 + D_3 + D_5);
                                        const double tmp47 = w59*(D_2 + D_3);
                                        const double tmp48 = w57*(D_4 + D_5);
                                        const double tmp49 = w60*(D_3 + D_5 + D_6);
                                        const double tmp50 = w61*(D_1 + D_2 + D_4);
                                        const double tmp51 = w60*(D_0 + D_3 + D_5);
                                        const double tmp52 = w61*(D_2 + D_4 + D_7);
                                        const double tmp53 = w60*(D_0 + D_3 + D_6);
                                        const double tmp54 = w61*(D_1 + D_4 + D_7);
                                        const double tmp55 = w60*(D_1 + D_3 + D_5 + D_7);
                                        const double tmp56 = w61*(D_0 + D_2 + D_4 + D_6);
                                        const double tmp57 = w59*(D_0 + D_1);
                                        const double tmp58 = w57*(D_6 + D_7);
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=D_0*w62 + D_7*w63 + tmp49 + tmp50;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=tmp27 + tmp57 + tmp58;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=tmp30 + tmp31 + tmp32;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=tmp10 + tmp11;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=tmp2 + tmp23 + tmp24;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=tmp43 + tmp44;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=tmp55 + tmp56;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=tmp27 + tmp57 + tmp58;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=D_1*w62 + D_6*w63 + tmp45 + tmp46;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=tmp10 + tmp11;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=tmp5 + tmp6 + tmp7;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=tmp43 + tmp44;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=tmp18 + tmp19 + tmp20;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=tmp41 + tmp42;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=tmp30 + tmp31 + tmp32;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=tmp10 + tmp11;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=D_2*w62 + D_5*w63 + tmp35 + tmp36;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=tmp14 + tmp47 + tmp48;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=tmp55 + tmp56;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=tmp16 + tmp17 + tmp18;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=tmp21 + tmp22;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=tmp10 + tmp11;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=tmp5 + tmp6 + tmp7;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=tmp14 + tmp47 + tmp48;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=D_3*w62 + D_4*w63 + tmp28 + tmp29;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=tmp41 + tmp42;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=tmp21 + tmp22;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=tmp0 + tmp1 + tmp2;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=tmp2 + tmp23 + tmp24;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=tmp43 + tmp44;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=tmp55 + tmp56;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=D_3*w63 + D_4*w62 + tmp33 + tmp34;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=tmp12 + tmp13 + tmp14;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=tmp7 + tmp8 + tmp9;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=tmp3 + tmp4;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=tmp43 + tmp44;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=tmp18 + tmp19 + tmp20;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=tmp41 + tmp42;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=tmp12 + tmp13 + tmp14;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=D_2*w63 + D_5*w62 + tmp53 + tmp54;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=tmp3 + tmp4;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=tmp32 + tmp39 + tmp40;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=tmp55 + tmp56;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=tmp16 + tmp17 + tmp18;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=tmp21 + tmp22;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=tmp7 + tmp8 + tmp9;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=tmp3 + tmp4;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=D_1*w63 + D_6*w62 + tmp51 + tmp52;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=tmp25 + tmp26 + tmp27;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=tmp15;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=tmp41 + tmp42;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=tmp21 + tmp22;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=tmp0 + tmp1 + tmp2;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=tmp3 + tmp4;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=tmp32 + tmp39 + tmp40;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=tmp25 + tmp26 + tmp27;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=D_0*w63 + D_7*w62 + tmp37 + tmp38;
                                    }
                                 }
                            } else { // constant data
                                for (index_t k=0; k<numEq; k++) {
                                    for (index_t m=0; m<numComp; m++) {
                                        const double wD0 = 8*D_p[INDEX2(k, m, numEq)]*w58;
                                        EM_S[INDEX4(k,m,0,0,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,0,1,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,0,2,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,0,3,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,0,4,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,0,5,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,0,6,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,0,7,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,1,0,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,1,1,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,1,2,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,1,3,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,1,4,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,1,5,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,1,6,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,1,7,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,2,0,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,2,1,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,2,2,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,2,3,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,2,4,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,2,5,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,2,6,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,2,7,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,3,0,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,3,1,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,3,2,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,3,3,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,3,4,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,3,5,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,3,6,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,3,7,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,4,0,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,4,1,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,4,2,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,4,3,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,4,4,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,4,5,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,4,6,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,4,7,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,5,0,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,5,1,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,5,2,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,5,3,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,5,4,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,5,5,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,5,6,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,5,7,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,6,0,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,6,1,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,6,2,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,6,3,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,6,4,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,6,5,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,6,6,numEq,numComp,8)]+=8*wD0;
                                        EM_S[INDEX4(k,m,6,7,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,7,0,numEq,numComp,8)]+=  wD0;
                                        EM_S[INDEX4(k,m,7,1,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,7,2,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,7,3,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,7,4,numEq,numComp,8)]+=2*wD0;
                                        EM_S[INDEX4(k,m,7,5,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,7,6,numEq,numComp,8)]+=4*wD0;
                                        EM_S[INDEX4(k,m,7,7,numEq,numComp,8)]+=8*wD0;
                                    }
                                }
                            }
                        }
                        ///////////////
                        // process X //
                        ///////////////
                        if (!du.isEmpty()) {
                            const double *du_p = du.getSampleDataRO(e);
                            const double *c11_p = c11.getSampleDataRO(e),
                                         *c13_p = c13.getSampleDataRO(e),
                                         *c33_p = c33.getSampleDataRO(e),
                                         *c44_p = c44.getSampleDataRO(e),
                                         *c66_p = c66.getSampleDataRO(e);
                            if (du.actsExpanded()) {
                                const double X_01_0 = -(c66_p[0]*(du_p[INDEX3(0,1,0,numEq,3)]+du_p[INDEX3(1,0,0,numEq,3)]));
                                const double X_01_1 = -(c66_p[1]*(du_p[INDEX3(0,1,1,numEq,3)]+du_p[INDEX3(1,0,1,numEq,3)]));
                                const double X_01_2 = -(c66_p[2]*(du_p[INDEX3(0,1,2,numEq,3)]+du_p[INDEX3(1,0,2,numEq,3)]));
                                const double X_01_3 = -(c66_p[3]*(du_p[INDEX3(0,1,3,numEq,3)]+du_p[INDEX3(1,0,3,numEq,3)]));
                                const double X_01_4 = -(c66_p[4]*(du_p[INDEX3(0,1,4,numEq,3)]+du_p[INDEX3(1,0,4,numEq,3)]));
                                const double X_01_5 = -(c66_p[5]*(du_p[INDEX3(0,1,5,numEq,3)]+du_p[INDEX3(1,0,5,numEq,3)]));
                                const double X_01_6 = -(c66_p[6]*(du_p[INDEX3(0,1,6,numEq,3)]+du_p[INDEX3(1,0,6,numEq,3)]));
                                const double X_01_7 = -(c66_p[7]*(du_p[INDEX3(0,1,7,numEq,3)]+du_p[INDEX3(1,0,7,numEq,3)]));
                                const double X_12_0 = -(c44_p[0]*(du_p[INDEX3(2,1,0,numEq,3)]+du_p[INDEX3(1,2,0,numEq,3)]));
                                const double X_12_1 = -(c44_p[1]*(du_p[INDEX3(2,1,1,numEq,3)]+du_p[INDEX3(1,2,1,numEq,3)]));
                                const double X_12_2 = -(c44_p[2]*(du_p[INDEX3(2,1,2,numEq,3)]+du_p[INDEX3(1,2,2,numEq,3)]));
                                const double X_12_3 = -(c44_p[3]*(du_p[INDEX3(2,1,3,numEq,3)]+du_p[INDEX3(1,2,3,numEq,3)]));
                                const double X_12_4 = -(c44_p[4]*(du_p[INDEX3(2,1,4,numEq,3)]+du_p[INDEX3(1,2,4,numEq,3)]));
                                const double X_12_5 = -(c44_p[5]*(du_p[INDEX3(2,1,5,numEq,3)]+du_p[INDEX3(1,2,5,numEq,3)]));
                                const double X_12_6 = -(c44_p[6]*(du_p[INDEX3(2,1,6,numEq,3)]+du_p[INDEX3(1,2,6,numEq,3)]));
                                const double X_12_7 = -(c44_p[7]*(du_p[INDEX3(2,1,7,numEq,3)]+du_p[INDEX3(1,2,7,numEq,3)]));
                                double X_00_0, X_00_1, X_00_2, X_00_3,
                                       X_00_4, X_00_5, X_00_6, X_00_7,
                                       X_02_0, X_02_1, X_02_2, X_02_3,
                                       X_02_4, X_02_5, X_02_6, X_02_7,
                                       X_11_0, X_11_1, X_11_2, X_11_3,
                                       X_11_4, X_11_5, X_11_6, X_11_7,
                                       X_22_0, X_22_1, X_22_2, X_22_3,
                                       X_22_4, X_22_5, X_22_6, X_22_7;
                                if (isVTI) {
                                    const double *c12_p = c12.getSampleDataRO(e);
                                    X_00_0 = -(c11_p[0]*du_p[INDEX3(0,0,0,numEq,3)]+c12_p[0]*du_p[INDEX3(1,1,0,numEq,3)]+c13_p[0]*du_p[INDEX3(2,2,0,numEq,3)]);
                                    X_00_1 = -(c11_p[1]*du_p[INDEX3(0,0,1,numEq,3)]+c12_p[1]*du_p[INDEX3(1,1,1,numEq,3)]+c13_p[1]*du_p[INDEX3(2,2,1,numEq,3)]);
                                    X_00_2 = -(c11_p[2]*du_p[INDEX3(0,0,2,numEq,3)]+c12_p[2]*du_p[INDEX3(1,1,2,numEq,3)]+c13_p[2]*du_p[INDEX3(2,2,2,numEq,3)]);
                                    X_00_3 = -(c11_p[3]*du_p[INDEX3(0,0,3,numEq,3)]+c12_p[3]*du_p[INDEX3(1,1,3,numEq,3)]+c13_p[3]*du_p[INDEX3(2,2,3,numEq,3)]);
                                    X_00_4 = -(c11_p[4]*du_p[INDEX3(0,0,4,numEq,3)]+c12_p[4]*du_p[INDEX3(1,1,4,numEq,3)]+c13_p[4]*du_p[INDEX3(2,2,4,numEq,3)]);
                                    X_00_5 = -(c11_p[5]*du_p[INDEX3(0,0,5,numEq,3)]+c12_p[5]*du_p[INDEX3(1,1,5,numEq,3)]+c13_p[5]*du_p[INDEX3(2,2,5,numEq,3)]);
                                    X_00_6 = -(c11_p[6]*du_p[INDEX3(0,0,6,numEq,3)]+c12_p[6]*du_p[INDEX3(1,1,6,numEq,3)]+c13_p[6]*du_p[INDEX3(2,2,6,numEq,3)]);
                                    X_00_7 = -(c11_p[7]*du_p[INDEX3(0,0,7,numEq,3)]+c12_p[7]*du_p[INDEX3(1,1,7,numEq,3)]+c13_p[7]*du_p[INDEX3(2,2,7,numEq,3)]);

                                    X_02_0 = -(c44_p[0]*(du_p[INDEX3(2,0,0,numEq,3)]+du_p[INDEX3(0,2,0,numEq,3)]));
                                    X_02_1 = -(c44_p[1]*(du_p[INDEX3(2,0,1,numEq,3)]+du_p[INDEX3(0,2,1,numEq,3)]));
                                    X_02_2 = -(c44_p[2]*(du_p[INDEX3(2,0,2,numEq,3)]+du_p[INDEX3(0,2,2,numEq,3)]));
                                    X_02_3 = -(c44_p[3]*(du_p[INDEX3(2,0,3,numEq,3)]+du_p[INDEX3(0,2,3,numEq,3)]));
                                    X_02_4 = -(c44_p[4]*(du_p[INDEX3(2,0,4,numEq,3)]+du_p[INDEX3(0,2,4,numEq,3)]));
                                    X_02_5 = -(c44_p[5]*(du_p[INDEX3(2,0,5,numEq,3)]+du_p[INDEX3(0,2,5,numEq,3)]));
                                    X_02_6 = -(c44_p[6]*(du_p[INDEX3(2,0,6,numEq,3)]+du_p[INDEX3(0,2,6,numEq,3)]));
                                    X_02_7 = -(c44_p[7]*(du_p[INDEX3(2,0,7,numEq,3)]+du_p[INDEX3(0,2,7,numEq,3)]));

                                    X_11_0 = -(c12_p[0]*du_p[INDEX3(0,0,0,numEq,3)]+c11_p[0]*du_p[INDEX3(1,1,0,numEq,3)]+c13_p[0]*du_p[INDEX3(2,2,0,numEq,3)]);
                                    X_11_1 = -(c12_p[1]*du_p[INDEX3(0,0,1,numEq,3)]+c11_p[1]*du_p[INDEX3(1,1,1,numEq,3)]+c13_p[1]*du_p[INDEX3(2,2,1,numEq,3)]);
                                    X_11_2 = -(c12_p[2]*du_p[INDEX3(0,0,2,numEq,3)]+c11_p[2]*du_p[INDEX3(1,1,2,numEq,3)]+c13_p[2]*du_p[INDEX3(2,2,2,numEq,3)]);
                                    X_11_3 = -(c12_p[3]*du_p[INDEX3(0,0,3,numEq,3)]+c11_p[3]*du_p[INDEX3(1,1,3,numEq,3)]+c13_p[3]*du_p[INDEX3(2,2,3,numEq,3)]);
                                    X_11_4 = -(c12_p[4]*du_p[INDEX3(0,0,4,numEq,3)]+c11_p[4]*du_p[INDEX3(1,1,4,numEq,3)]+c13_p[4]*du_p[INDEX3(2,2,4,numEq,3)]);
                                    X_11_5 = -(c12_p[5]*du_p[INDEX3(0,0,5,numEq,3)]+c11_p[5]*du_p[INDEX3(1,1,5,numEq,3)]+c13_p[5]*du_p[INDEX3(2,2,5,numEq,3)]);
                                    X_11_6 = -(c12_p[6]*du_p[INDEX3(0,0,6,numEq,3)]+c11_p[6]*du_p[INDEX3(1,1,6,numEq,3)]+c13_p[6]*du_p[INDEX3(2,2,6,numEq,3)]);
                                    X_11_7 = -(c12_p[7]*du_p[INDEX3(0,0,7,numEq,3)]+c11_p[7]*du_p[INDEX3(1,1,7,numEq,3)]+c13_p[7]*du_p[INDEX3(2,2,7,numEq,3)]);

                                    X_22_0 = -(c13_p[0]*(du_p[INDEX3(0,0,0,numEq,3)] + du_p[INDEX3(1,1,0,numEq,3)]) + c33_p[0]*du_p[INDEX3(2,2,0,numEq,3)]);
                                    X_22_1 = -(c13_p[1]*(du_p[INDEX3(0,0,1,numEq,3)] + du_p[INDEX3(1,1,1,numEq,3)]) + c33_p[1]*du_p[INDEX3(2,2,1,numEq,3)]);
                                    X_22_2 = -(c13_p[2]*(du_p[INDEX3(0,0,2,numEq,3)] + du_p[INDEX3(1,1,2,numEq,3)]) + c33_p[2]*du_p[INDEX3(2,2,2,numEq,3)]);
                                    X_22_3 = -(c13_p[3]*(du_p[INDEX3(0,0,3,numEq,3)] + du_p[INDEX3(1,1,3,numEq,3)]) + c33_p[3]*du_p[INDEX3(2,2,3,numEq,3)]);
                                    X_22_4 = -(c13_p[4]*(du_p[INDEX3(0,0,4,numEq,3)] + du_p[INDEX3(1,1,4,numEq,3)]) + c33_p[4]*du_p[INDEX3(2,2,4,numEq,3)]);
                                    X_22_5 = -(c13_p[5]*(du_p[INDEX3(0,0,5,numEq,3)] + du_p[INDEX3(1,1,5,numEq,3)]) + c33_p[5]*du_p[INDEX3(2,2,5,numEq,3)]);
                                    X_22_6 = -(c13_p[6]*(du_p[INDEX3(0,0,6,numEq,3)] + du_p[INDEX3(1,1,6,numEq,3)]) + c33_p[6]*du_p[INDEX3(2,2,6,numEq,3)]);
                                    X_22_7 = -(c13_p[7]*(du_p[INDEX3(0,0,7,numEq,3)] + du_p[INDEX3(1,1,7,numEq,3)]) + c33_p[7]*du_p[INDEX3(2,2,7,numEq,3)]);
                                } else { // isHTI
                                    const double *c23_p = c23.getSampleDataRO(e);
                                    X_00_0 = -(c11_p[0]*du_p[INDEX3(0,0,0,numEq,3)]+c13_p[0]*(du_p[INDEX3(1,1,0,numEq,3)]+du_p[INDEX3(2,2,0,numEq,3)]));
                                    X_00_1 = -(c11_p[1]*du_p[INDEX3(0,0,1,numEq,3)]+c13_p[1]*(du_p[INDEX3(1,1,1,numEq,3)]+du_p[INDEX3(2,2,1,numEq,3)]));
                                    X_00_2 = -(c11_p[2]*du_p[INDEX3(0,0,2,numEq,3)]+c13_p[2]*(du_p[INDEX3(1,1,2,numEq,3)]+du_p[INDEX3(2,2,2,numEq,3)]));
                                    X_00_3 = -(c11_p[3]*du_p[INDEX3(0,0,3,numEq,3)]+c13_p[3]*(du_p[INDEX3(1,1,3,numEq,3)]+du_p[INDEX3(2,2,3,numEq,3)]));
                                    X_00_4 = -(c11_p[4]*du_p[INDEX3(0,0,4,numEq,3)]+c13_p[4]*(du_p[INDEX3(1,1,4,numEq,3)]+du_p[INDEX3(2,2,4,numEq,3)]));
                                    X_00_5 = -(c11_p[5]*du_p[INDEX3(0,0,5,numEq,3)]+c13_p[5]*(du_p[INDEX3(1,1,5,numEq,3)]+du_p[INDEX3(2,2,5,numEq,3)]));
                                    X_00_6 = -(c11_p[6]*du_p[INDEX3(0,0,6,numEq,3)]+c13_p[6]*(du_p[INDEX3(1,1,6,numEq,3)]+du_p[INDEX3(2,2,6,numEq,3)]));
                                    X_00_7 = -(c11_p[7]*du_p[INDEX3(0,0,7,numEq,3)]+c13_p[7]*(du_p[INDEX3(1,1,7,numEq,3)]+du_p[INDEX3(2,2,7,numEq,3)]));

                                    X_02_0 = -(c66_p[0]*(du_p[INDEX3(2,0,0,numEq,3)]+du_p[INDEX3(0,2,0,numEq,3)]));
                                    X_02_1 = -(c66_p[1]*(du_p[INDEX3(2,0,1,numEq,3)]+du_p[INDEX3(0,2,1,numEq,3)]));
                                    X_02_2 = -(c66_p[2]*(du_p[INDEX3(2,0,2,numEq,3)]+du_p[INDEX3(0,2,2,numEq,3)]));
                                    X_02_3 = -(c66_p[3]*(du_p[INDEX3(2,0,3,numEq,3)]+du_p[INDEX3(0,2,3,numEq,3)]));
                                    X_02_4 = -(c66_p[4]*(du_p[INDEX3(2,0,4,numEq,3)]+du_p[INDEX3(0,2,4,numEq,3)]));
                                    X_02_5 = -(c66_p[5]*(du_p[INDEX3(2,0,5,numEq,3)]+du_p[INDEX3(0,2,5,numEq,3)]));
                                    X_02_6 = -(c66_p[6]*(du_p[INDEX3(2,0,6,numEq,3)]+du_p[INDEX3(0,2,6,numEq,3)]));
                                    X_02_7 = -(c66_p[7]*(du_p[INDEX3(2,0,7,numEq,3)]+du_p[INDEX3(0,2,7,numEq,3)]));

                                    X_11_0 = -(c13_p[0]*du_p[INDEX3(0,0,0,numEq,3)]+c33_p[0]*du_p[INDEX3(1,1,0,numEq,3)]+c23_p[0]*du_p[INDEX3(2,2,0,numEq,3)]);
                                    X_11_1 = -(c13_p[1]*du_p[INDEX3(0,0,1,numEq,3)]+c33_p[1]*du_p[INDEX3(1,1,1,numEq,3)]+c23_p[1]*du_p[INDEX3(2,2,1,numEq,3)]);
                                    X_11_2 = -(c13_p[2]*du_p[INDEX3(0,0,2,numEq,3)]+c33_p[2]*du_p[INDEX3(1,1,2,numEq,3)]+c23_p[2]*du_p[INDEX3(2,2,2,numEq,3)]);
                                    X_11_3 = -(c13_p[3]*du_p[INDEX3(0,0,3,numEq,3)]+c33_p[3]*du_p[INDEX3(1,1,3,numEq,3)]+c23_p[3]*du_p[INDEX3(2,2,3,numEq,3)]);
                                    X_11_4 = -(c13_p[4]*du_p[INDEX3(0,0,4,numEq,3)]+c33_p[4]*du_p[INDEX3(1,1,4,numEq,3)]+c23_p[4]*du_p[INDEX3(2,2,4,numEq,3)]);
                                    X_11_5 = -(c13_p[5]*du_p[INDEX3(0,0,5,numEq,3)]+c33_p[5]*du_p[INDEX3(1,1,5,numEq,3)]+c23_p[5]*du_p[INDEX3(2,2,5,numEq,3)]);
                                    X_11_6 = -(c13_p[6]*du_p[INDEX3(0,0,6,numEq,3)]+c33_p[6]*du_p[INDEX3(1,1,6,numEq,3)]+c23_p[6]*du_p[INDEX3(2,2,6,numEq,3)]);
                                    X_11_7 = -(c13_p[7]*du_p[INDEX3(0,0,7,numEq,3)]+c33_p[7]*du_p[INDEX3(1,1,7,numEq,3)]+c23_p[7]*du_p[INDEX3(2,2,7,numEq,3)]);

                                    X_22_0 = -(c13_p[0]*du_p[INDEX3(0,0,0,numEq,3)] + c23_p[0]*du_p[INDEX3(1,1,0,numEq,3)] + c33_p[0]*du_p[INDEX3(2,2,0,numEq,3)]);
                                    X_22_1 = -(c13_p[1]*du_p[INDEX3(0,0,1,numEq,3)] + c23_p[1]*du_p[INDEX3(1,1,1,numEq,3)] + c33_p[1]*du_p[INDEX3(2,2,1,numEq,3)]);
                                    X_22_2 = -(c13_p[2]*du_p[INDEX3(0,0,2,numEq,3)] + c23_p[2]*du_p[INDEX3(1,1,2,numEq,3)] + c33_p[2]*du_p[INDEX3(2,2,2,numEq,3)]);
                                    X_22_3 = -(c13_p[3]*du_p[INDEX3(0,0,3,numEq,3)] + c23_p[3]*du_p[INDEX3(1,1,3,numEq,3)] + c33_p[3]*du_p[INDEX3(2,2,3,numEq,3)]);
                                    X_22_4 = -(c13_p[4]*du_p[INDEX3(0,0,4,numEq,3)] + c23_p[4]*du_p[INDEX3(1,1,4,numEq,3)] + c33_p[4]*du_p[INDEX3(2,2,4,numEq,3)]);
                                    X_22_5 = -(c13_p[5]*du_p[INDEX3(0,0,5,numEq,3)] + c23_p[5]*du_p[INDEX3(1,1,5,numEq,3)] + c33_p[5]*du_p[INDEX3(2,2,5,numEq,3)]);
                                    X_22_6 = -(c13_p[6]*du_p[INDEX3(0,0,6,numEq,3)] + c23_p[6]*du_p[INDEX3(1,1,6,numEq,3)] + c33_p[6]*du_p[INDEX3(2,2,6,numEq,3)]);
                                    X_22_7 = -(c13_p[7]*du_p[INDEX3(0,0,7,numEq,3)] + c23_p[7]*du_p[INDEX3(1,1,7,numEq,3)] + c33_p[7]*du_p[INDEX3(2,2,7,numEq,3)]);
                                }

                                const double Atmp0 = w72*(X_00_6 + X_00_7);
                                const double Atmp1 = w66*(X_02_0 + X_02_4);
                                const double Atmp2 = w64*(X_00_0 + X_00_1);
                                const double Atmp3 = w68*(X_02_1 + X_02_2 + X_02_5 + X_02_6);
                                const double Atmp4 = w65*(X_01_0 + X_01_2);
                                const double Atmp5 = w70*(X_02_3 + X_02_7);
                                const double Atmp6 = w67*(X_01_1 + X_01_3 + X_01_4 + X_01_6);
                                const double Atmp7 = w71*(X_01_5 + X_01_7);
                                const double Atmp8 = w69*(X_00_2 + X_00_3 + X_00_4 + X_00_5);
                                const double Atmp9 = w72*(-X_00_6 - X_00_7);
                                const double Atmp10 = w66*(X_02_1 + X_02_5);
                                const double Atmp11 = w64*(-X_00_0 - X_00_1);
                                const double Atmp12 = w68*(X_02_0 + X_02_3 + X_02_4 + X_02_7);
                                const double Atmp13 = w65*(X_01_1 + X_01_3);
                                const double Atmp14 = w70*(X_02_2 + X_02_6);
                                const double Atmp15 = w67*(X_01_0 + X_01_2 + X_01_5 + X_01_7);
                                const double Atmp16 = w71*(X_01_4 + X_01_6);
                                const double Atmp17 = w69*(-X_00_2 - X_00_3 - X_00_4 - X_00_5);
                                const double Atmp18 = w72*(X_00_4 + X_00_5);
                                const double Atmp19 = w66*(X_02_2 + X_02_6);
                                const double Atmp20 = w64*(X_00_2 + X_00_3);
                                const double Atmp21 = w65*(-X_01_0 - X_01_2);
                                const double Atmp22 = w70*(X_02_1 + X_02_5);
                                const double Atmp23 = w67*(-X_01_1 - X_01_3 - X_01_4 - X_01_6);
                                const double Atmp24 = w71*(-X_01_5 - X_01_7);
                                const double Atmp25 = w69*(X_00_0 + X_00_1 + X_00_6 + X_00_7);
                                const double Atmp26 = w72*(-X_00_4 - X_00_5);
                                const double Atmp27 = w66*(X_02_3 + X_02_7);
                                const double Atmp28 = w64*(-X_00_2 - X_00_3);
                                const double Atmp29 = w65*(-X_01_1 - X_01_3);
                                const double Atmp30 = w70*(X_02_0 + X_02_4);
                                const double Atmp31 = w67*(-X_01_0 - X_01_2 - X_01_5 - X_01_7);
                                const double Atmp32 = w71*(-X_01_4 - X_01_6);
                                const double Atmp33 = w69*(-X_00_0 - X_00_1 - X_00_6 - X_00_7);
                                const double Atmp34 = w72*(X_00_2 + X_00_3);
                                const double Atmp35 = w66*(-X_02_0 - X_02_4);
                                const double Atmp36 = w64*(X_00_4 + X_00_5);
                                const double Atmp37 = w68*(-X_02_1 - X_02_2 - X_02_5 - X_02_6);
                                const double Atmp38 = w65*(X_01_4 + X_01_6);
                                const double Atmp39 = w70*(-X_02_3 - X_02_7);
                                const double Atmp40 = w71*(X_01_1 + X_01_3);
                                const double Atmp41 = w72*(-X_00_2 - X_00_3);
                                const double Atmp42 = w66*(-X_02_1 - X_02_5);
                                const double Atmp43 = w64*(-X_00_4 - X_00_5);
                                const double Atmp44 = w68*(-X_02_0 - X_02_3 - X_02_4 - X_02_7);
                                const double Atmp45 = w65*(X_01_5 + X_01_7);
                                const double Atmp46 = w70*(-X_02_2 - X_02_6);
                                const double Atmp47 = w71*(X_01_0 + X_01_2);
                                const double Atmp48 = w72*(X_00_0 + X_00_1);
                                const double Atmp49 = w66*(-X_02_2 - X_02_6);
                                const double Atmp50 = w64*(X_00_6 + X_00_7);
                                const double Atmp51 = w65*(-X_01_4 - X_01_6);
                                const double Atmp52 = w70*(-X_02_1 - X_02_5);
                                const double Atmp53 = w71*(-X_01_1 - X_01_3);
                                const double Atmp54 = w72*(-X_00_0 - X_00_1);
                                const double Atmp55 = w66*(-X_02_3 - X_02_7);
                                const double Atmp56 = w64*(-X_00_6 - X_00_7);
                                const double Atmp57 = w65*(-X_01_5 - X_01_7);
                                const double Atmp58 = w70*(-X_02_0 - X_02_4);
                                const double Atmp59 = w71*(-X_01_0 - X_01_2);
                                const double Btmp0 = w72*(X_01_6 + X_01_7);
                                const double Btmp1 = w66*(X_12_0 + X_12_4);
                                const double Btmp2 = w64*(X_01_0 + X_01_1);
                                const double Btmp3 = w68*(X_12_1 + X_12_2 + X_12_5 + X_12_6);
                                const double Btmp4 = w65*(X_11_0 + X_11_2);
                                const double Btmp5 = w70*(X_12_3 + X_12_7);
                                const double Btmp6 = w67*(X_11_1 + X_11_3 + X_11_4 + X_11_6);
                                const double Btmp7 = w71*(X_11_5 + X_11_7);
                                const double Btmp8 = w69*(X_01_2 + X_01_3 + X_01_4 + X_01_5);
                                const double Btmp9 = w72*(-X_01_6 - X_01_7);
                                const double Btmp10 = w66*(X_12_1 + X_12_5);
                                const double Btmp11 = w64*(-X_01_0 - X_01_1);
                                const double Btmp12 = w68*(X_12_0 + X_12_3 + X_12_4 + X_12_7);
                                const double Btmp13 = w65*(X_11_1 + X_11_3);
                                const double Btmp14 = w70*(X_12_2 + X_12_6);
                                const double Btmp15 = w67*(X_11_0 + X_11_2 + X_11_5 + X_11_7);
                                const double Btmp16 = w71*(X_11_4 + X_11_6);
                                const double Btmp17 = w69*(-X_01_2 - X_01_3 - X_01_4 - X_01_5);
                                const double Btmp18 = w72*(X_01_4 + X_01_5);
                                const double Btmp19 = w66*(X_12_2 + X_12_6);
                                const double Btmp20 = w64*(X_01_2 + X_01_3);
                                const double Btmp21 = w65*(-X_11_0 - X_11_2);
                                const double Btmp22 = w70*(X_12_1 + X_12_5);
                                const double Btmp23 = w67*(-X_11_1 - X_11_3 - X_11_4 - X_11_6);
                                const double Btmp24 = w71*(-X_11_5 - X_11_7);
                                const double Btmp25 = w69*(X_01_0 + X_01_1 + X_01_6 + X_01_7);
                                const double Btmp26 = w72*(-X_01_4 - X_01_5);
                                const double Btmp27 = w66*(X_12_3 + X_12_7);
                                const double Btmp28 = w64*(-X_01_2 - X_01_3);
                                const double Btmp29 = w65*(-X_11_1 - X_11_3);
                                const double Btmp30 = w70*(X_12_0 + X_12_4);
                                const double Btmp31 = w67*(-X_11_0 - X_11_2 - X_11_5 - X_11_7);
                                const double Btmp32 = w71*(-X_11_4 - X_11_6);
                                const double Btmp33 = w69*(-X_01_0 - X_01_1 - X_01_6 - X_01_7);
                                const double Btmp34 = w72*(X_01_2 + X_01_3);
                                const double Btmp35 = w66*(-X_12_0 - X_12_4);
                                const double Btmp36 = w64*(X_01_4 + X_01_5);
                                const double Btmp37 = w68*(-X_12_1 - X_12_2 - X_12_5 - X_12_6);
                                const double Btmp38 = w65*(X_11_4 + X_11_6);
                                const double Btmp39 = w70*(-X_12_3 - X_12_7);
                                const double Btmp40 = w71*(X_11_1 + X_11_3);
                                const double Btmp41 = w72*(-X_01_2 - X_01_3);
                                const double Btmp42 = w66*(-X_12_1 - X_12_5);
                                const double Btmp43 = w64*(-X_01_4 - X_01_5);
                                const double Btmp44 = w68*(-X_12_0 - X_12_3 - X_12_4 - X_12_7);
                                const double Btmp45 = w65*(X_11_5 + X_11_7);
                                const double Btmp46 = w70*(-X_12_2 - X_12_6);
                                const double Btmp47 = w71*(X_11_0 + X_11_2);
                                const double Btmp48 = w72*(X_01_0 + X_01_1);
                                const double Btmp49 = w66*(-X_12_2 - X_12_6);
                                const double Btmp50 = w64*(X_01_6 + X_01_7);
                                const double Btmp51 = w65*(-X_11_4 - X_11_6);
                                const double Btmp52 = w70*(-X_12_1 - X_12_5);
                                const double Btmp53 = w71*(-X_11_1 - X_11_3);
                                const double Btmp54 = w72*(-X_01_0 - X_01_1);
                                const double Btmp55 = w66*(-X_12_3 - X_12_7);
                                const double Btmp56 = w64*(-X_01_6 - X_01_7);
                                const double Btmp57 = w65*(-X_11_5 - X_11_7);
                                const double Btmp58 = w70*(-X_12_0 - X_12_4);
                                const double Btmp59 = w71*(-X_11_0 - X_11_2);
                                const double Ctmp0 = w72*(X_02_6 + X_02_7);
                                const double Ctmp1 = w66*(X_22_0 + X_22_4);
                                const double Ctmp2 = w64*(X_02_0 + X_02_1);
                                const double Ctmp3 = w68*(X_22_1 + X_22_2 + X_22_5 + X_22_6);
                                const double Ctmp4 = w65*(X_12_0 + X_12_2);
                                const double Ctmp5 = w70*(X_22_3 + X_22_7);
                                const double Ctmp6 = w67*(X_12_1 + X_12_3 + X_12_4 + X_12_6);
                                const double Ctmp7 = w71*(X_12_5 + X_12_7);
                                const double Ctmp8 = w69*(X_02_2 + X_02_3 + X_02_4 + X_02_5);
                                const double Ctmp9 = w72*(-X_02_6 - X_02_7);
                                const double Ctmp10 = w66*(X_22_1 + X_22_5);
                                const double Ctmp11 = w64*(-X_02_0 - X_02_1);
                                const double Ctmp12 = w68*(X_22_0 + X_22_3 + X_22_4 + X_22_7);
                                const double Ctmp13 = w65*(X_12_1 + X_12_3);
                                const double Ctmp14 = w70*(X_22_2 + X_22_6);
                                const double Ctmp15 = w67*(X_12_0 + X_12_2 + X_12_5 + X_12_7);
                                const double Ctmp16 = w71*(X_12_4 + X_12_6);
                                const double Ctmp17 = w69*(-X_02_2 - X_02_3 - X_02_4 - X_02_5);
                                const double Ctmp18 = w72*(X_02_4 + X_02_5);
                                const double Ctmp19 = w66*(X_22_2 + X_22_6);
                                const double Ctmp20 = w64*(X_02_2 + X_02_3);
                                const double Ctmp21 = w65*(-X_12_0 - X_12_2);
                                const double Ctmp22 = w70*(X_22_1 + X_22_5);
                                const double Ctmp23 = w67*(-X_12_1 - X_12_3 - X_12_4 - X_12_6);
                                const double Ctmp24 = w71*(-X_12_5 - X_12_7);
                                const double Ctmp25 = w69*(X_02_0 + X_02_1 + X_02_6 + X_02_7);
                                const double Ctmp26 = w72*(-X_02_4 - X_02_5);
                                const double Ctmp27 = w66*(X_22_3 + X_22_7);
                                const double Ctmp28 = w64*(-X_02_2 - X_02_3);
                                const double Ctmp29 = w65*(-X_12_1 - X_12_3);
                                const double Ctmp30 = w70*(X_22_0 + X_22_4);
                                const double Ctmp31 = w67*(-X_12_0 - X_12_2 - X_12_5 - X_12_7);
                                const double Ctmp32 = w71*(-X_12_4 - X_12_6);
                                const double Ctmp33 = w69*(-X_02_0 - X_02_1 - X_02_6 - X_02_7);
                                const double Ctmp34 = w72*(X_02_2 + X_02_3);
                                const double Ctmp35 = w66*(-X_22_0 - X_22_4);
                                const double Ctmp36 = w64*(X_02_4 + X_02_5);
                                const double Ctmp37 = w68*(-X_22_1 - X_22_2 - X_22_5 - X_22_6);
                                const double Ctmp38 = w65*(X_12_4 + X_12_6);
                                const double Ctmp39 = w70*(-X_22_3 - X_22_7);
                                const double Ctmp40 = w71*(X_12_1 + X_12_3);
                                const double Ctmp41 = w72*(-X_02_2 - X_02_3);
                                const double Ctmp42 = w66*(-X_22_1 - X_22_5);
                                const double Ctmp43 = w64*(-X_02_4 - X_02_5);
                                const double Ctmp44 = w68*(-X_22_0 - X_22_3 - X_22_4 - X_22_7);
                                const double Ctmp45 = w65*(X_12_5 + X_12_7);
                                const double Ctmp46 = w70*(-X_22_2 - X_22_6);
                                const double Ctmp47 = w71*(X_12_0 + X_12_2);
                                const double Ctmp48 = w72*(X_02_0 + X_02_1);
                                const double Ctmp49 = w66*(-X_22_2 - X_22_6);
                                const double Ctmp50 = w64*(X_02_6 + X_02_7);
                                const double Ctmp51 = w65*(-X_12_4 - X_12_6);
                                const double Ctmp52 = w70*(-X_22_1 - X_22_5);
                                const double Ctmp53 = w71*(-X_12_1 - X_12_3);
                                const double Ctmp54 = w72*(-X_02_0 - X_02_1);
                                const double Ctmp55 = w66*(-X_22_3 - X_22_7);
                                const double Ctmp56 = w64*(-X_02_6 - X_02_7);
                                const double Ctmp57 = w65*(-X_12_5 - X_12_7);
                                const double Ctmp58 = w70*(-X_22_0 - X_22_4);
                                const double Ctmp59 = w71*(-X_12_0 - X_12_2);
                                EM_F[INDEX2(0,0,numEq)]+=Atmp0 + Atmp1 + Atmp2 + Atmp3 + Atmp4 + Atmp5 + Atmp6 + Atmp7 + Atmp8;
                                EM_F[INDEX2(1,0,numEq)]+=Btmp0 + Btmp1 + Btmp2 + Btmp3 + Btmp4 + Btmp5 + Btmp6 + Btmp7 + Btmp8;
                                EM_F[INDEX2(2,0,numEq)]+=Ctmp0 + Ctmp1 + Ctmp2 + Ctmp3 + Ctmp4 + Ctmp5 + Ctmp6 + Ctmp7 + Ctmp8;
                                EM_F[INDEX2(0,1,numEq)]+=Atmp10 + Atmp11 + Atmp12 + Atmp13 + Atmp14 + Atmp15 + Atmp16 + Atmp17 + Atmp9;
                                EM_F[INDEX2(1,1,numEq)]+=Btmp10 + Btmp11 + Btmp12 + Btmp13 + Btmp14 + Btmp15 + Btmp16 + Btmp17 + Btmp9;
                                EM_F[INDEX2(2,1,numEq)]+=Ctmp10 + Ctmp11 + Ctmp12 + Ctmp13 + Ctmp14 + Ctmp15 + Ctmp16 + Ctmp17 + Ctmp9;
                                EM_F[INDEX2(0,2,numEq)]+=Atmp12 + Atmp18 + Atmp19 + Atmp20 + Atmp21 + Atmp22 + Atmp23 + Atmp24 + Atmp25;
                                EM_F[INDEX2(1,2,numEq)]+=Btmp12 + Btmp18 + Btmp19 + Btmp20 + Btmp21 + Btmp22 + Btmp23 + Btmp24 + Btmp25;
                                EM_F[INDEX2(2,2,numEq)]+=Ctmp12 + Ctmp18 + Ctmp19 + Ctmp20 + Ctmp21 + Ctmp22 + Ctmp23 + Ctmp24 + Ctmp25;
                                EM_F[INDEX2(0,3,numEq)]+=Atmp26 + Atmp27 + Atmp28 + Atmp29 + Atmp3 + Atmp30 + Atmp31 + Atmp32 + Atmp33;
                                EM_F[INDEX2(1,3,numEq)]+=Btmp26 + Btmp27 + Btmp28 + Btmp29 + Btmp3 + Btmp30 + Btmp31 + Btmp32 + Btmp33;
                                EM_F[INDEX2(2,3,numEq)]+=Ctmp26 + Ctmp27 + Ctmp28 + Ctmp29 + Ctmp3 + Ctmp30 + Ctmp31 + Ctmp32 + Ctmp33;
                                EM_F[INDEX2(0,4,numEq)]+=Atmp15 + Atmp25 + Atmp34 + Atmp35 + Atmp36 + Atmp37 + Atmp38 + Atmp39 + Atmp40;
                                EM_F[INDEX2(1,4,numEq)]+=Btmp15 + Btmp25 + Btmp34 + Btmp35 + Btmp36 + Btmp37 + Btmp38 + Btmp39 + Btmp40;
                                EM_F[INDEX2(2,4,numEq)]+=Ctmp15 + Ctmp25 + Ctmp34 + Ctmp35 + Ctmp36 + Ctmp37 + Ctmp38 + Ctmp39 + Ctmp40;
                                EM_F[INDEX2(0,5,numEq)]+=Atmp33 + Atmp41 + Atmp42 + Atmp43 + Atmp44 + Atmp45 + Atmp46 + Atmp47 + Atmp6;
                                EM_F[INDEX2(1,5,numEq)]+=Btmp33 + Btmp41 + Btmp42 + Btmp43 + Btmp44 + Btmp45 + Btmp46 + Btmp47 + Btmp6;
                                EM_F[INDEX2(2,5,numEq)]+=Ctmp33 + Ctmp41 + Ctmp42 + Ctmp43 + Ctmp44 + Ctmp45 + Ctmp46 + Ctmp47 + Ctmp6;
                                EM_F[INDEX2(0,6,numEq)]+=Atmp31 + Atmp44 + Atmp48 + Atmp49 + Atmp50 + Atmp51 + Atmp52 + Atmp53 + Atmp8;
                                EM_F[INDEX2(1,6,numEq)]+=Btmp31 + Btmp44 + Btmp48 + Btmp49 + Btmp50 + Btmp51 + Btmp52 + Btmp53 + Btmp8;
                                EM_F[INDEX2(2,6,numEq)]+=Ctmp31 + Ctmp44 + Ctmp48 + Ctmp49 + Ctmp50 + Ctmp51 + Ctmp52 + Ctmp53 + Ctmp8;
                                EM_F[INDEX2(0,7,numEq)]+=Atmp17 + Atmp23 + Atmp37 + Atmp54 + Atmp55 + Atmp56 + Atmp57 + Atmp58 + Atmp59;
                                EM_F[INDEX2(1,7,numEq)]+=Btmp17 + Btmp23 + Btmp37 + Btmp54 + Btmp55 + Btmp56 + Btmp57 + Btmp58 + Btmp59;
                                EM_F[INDEX2(2,7,numEq)]+=Ctmp17 + Ctmp23 + Ctmp37 + Ctmp54 + Ctmp55 + Ctmp56 + Ctmp57 + Ctmp58 + Ctmp59;
                            } else { // constant data
                                const double wX01 = 18*w56*-(c66_p[0]*(du_p[INDEX2(0,1,numEq)] + du_p[INDEX2(1,0,numEq)]));
                                const double wX10 = 18*w55*-(c66_p[0]*(du_p[INDEX2(0,1,numEq)] + du_p[INDEX2(1,0,numEq)]));
                                const double wX12 = 18*w54*-(c44_p[0]*(du_p[INDEX2(2,1,numEq)] + du_p[INDEX2(1,2,numEq)]));
                                const double wX21 = 18*w56*-(c44_p[0]*(du_p[INDEX2(2,1,numEq)] + du_p[INDEX2(1,2,numEq)]));
                                double wX00, wX11, wX22, wX02, wX20;
                                if (isVTI) {
                                    const double *c12_p = c12.getSampleDataRO(e);
                                    wX00 = 18*w55*-(c11_p[0]* du_p[INDEX2(0,0,numEq)] + c12_p[0] * du_p[INDEX2(1,1,numEq)] + c13_p[0]*du_p[INDEX2(2,2,numEq)]);
                                    wX11 = 18*w56*-(c12_p[0]* du_p[INDEX2(0,0,numEq)] + c11_p[0] * du_p[INDEX2(1,1,numEq)] + c13_p[0]*du_p[INDEX2(2,2,numEq)]);
                                    wX22 = 18*w54*-(c13_p[0]*(du_p[INDEX2(0,0,numEq)] + du_p[INDEX2(1,1,numEq)]) + c33_p[0]* du_p[INDEX2(2,2,numEq)]);
                                    wX02 = 18*w54*-(c44_p[0]*(du_p[INDEX2(2,0,numEq)] + du_p[INDEX2(0,2,numEq)]));
                                    wX20 = 18*w55*-(c44_p[0]*(du_p[INDEX2(2,0,numEq)] + du_p[INDEX2(0,2,numEq)]));                    
                                } else { // isHTI
                                    const double *c23_p = c23.getSampleDataRO(e);
                                    wX00 = 18*w55*-(c11_p[0]* du_p[INDEX2(0,0,numEq)] + c13_p[0] * (du_p[INDEX2(1,1,numEq)] + du_p[INDEX2(2,2,numEq)]));
                                    wX11 = 18*w56*-(c13_p[0]* du_p[INDEX2(0,0,numEq)] + c33_p[0] * du_p[INDEX2(1,1,numEq)] + c23_p[0]*du_p[INDEX2(2,2,numEq)]);
                                    wX22 = 18*w54*-(c13_p[0]* du_p[INDEX2(0,0,numEq)] + c23_p[0] * du_p[INDEX2(1,1,numEq)] + c33_p[0]* du_p[INDEX2(2,2,numEq)]);
                                    wX02 = 18*w54*-(c66_p[0]*(du_p[INDEX2(2,0,numEq)] + du_p[INDEX2(0,2,numEq)]));
                                    wX20 = 18*w55*-(c66_p[0]*(du_p[INDEX2(2,0,numEq)] + du_p[INDEX2(0,2,numEq)]));
                                }

                                EM_F[INDEX2(0,0,numEq)]+= wX00 + wX01 + wX02;
                                EM_F[INDEX2(0,1,numEq)]+=-wX00 + wX01 + wX02;
                                EM_F[INDEX2(0,2,numEq)]+= wX00 - wX01 + wX02;
                                EM_F[INDEX2(0,3,numEq)]+=-wX00 - wX01 + wX02;
                                EM_F[INDEX2(0,4,numEq)]+= wX00 + wX01 - wX02;
                                EM_F[INDEX2(0,5,numEq)]+=-wX00 + wX01 - wX02;
                                EM_F[INDEX2(0,6,numEq)]+= wX00 - wX01 - wX02;
                                EM_F[INDEX2(0,7,numEq)]+=-wX00 - wX01 - wX02;
                    
                                EM_F[INDEX2(1,0,numEq)]+= wX10 + wX11 + wX12;
                                EM_F[INDEX2(1,1,numEq)]+=-wX10 + wX11 + wX12;
                                EM_F[INDEX2(1,2,numEq)]+= wX10 - wX11 + wX12;
                                EM_F[INDEX2(1,3,numEq)]+=-wX10 - wX11 + wX12;
                                EM_F[INDEX2(1,4,numEq)]+= wX10 + wX11 - wX12;
                                EM_F[INDEX2(1,5,numEq)]+=-wX10 + wX11 - wX12;
                                EM_F[INDEX2(1,6,numEq)]+= wX10 - wX11 - wX12;
                                EM_F[INDEX2(1,7,numEq)]+=-wX10 - wX11 - wX12;
                    
                                EM_F[INDEX2(2,0,numEq)]+= wX20 + wX21 + wX22;
                                EM_F[INDEX2(2,1,numEq)]+=-wX20 + wX21 + wX22;
                                EM_F[INDEX2(2,2,numEq)]+= wX20 - wX21 + wX22;
                                EM_F[INDEX2(2,3,numEq)]+=-wX20 - wX21 + wX22;
                                EM_F[INDEX2(2,4,numEq)]+= wX20 + wX21 - wX22;
                                EM_F[INDEX2(2,5,numEq)]+=-wX20 + wX21 - wX22;
                                EM_F[INDEX2(2,6,numEq)]+= wX20 - wX21 - wX22;
                                EM_F[INDEX2(2,7,numEq)]+=-wX20 - wX21 - wX22;
                            }
                        }
                        ///////////////
                        // process Y //
                        ///////////////
                        if (!Y.isEmpty()) {
                            const double* Y_p=Y.getSampleDataRO(e);
                            if (Y.actsExpanded()) {
                                for (index_t k=0; k<numEq; k++) {
                                    const double Y_0 = Y_p[INDEX2(k, 0, numEq)];
                                    const double Y_1 = Y_p[INDEX2(k, 1, numEq)];
                                    const double Y_2 = Y_p[INDEX2(k, 2, numEq)];
                                    const double Y_3 = Y_p[INDEX2(k, 3, numEq)];
                                    const double Y_4 = Y_p[INDEX2(k, 4, numEq)];
                                    const double Y_5 = Y_p[INDEX2(k, 5, numEq)];
                                    const double Y_6 = Y_p[INDEX2(k, 6, numEq)];
                                    const double Y_7 = Y_p[INDEX2(k, 7, numEq)];
                                    const double tmp0 = w76*(Y_3 + Y_5 + Y_6);
                                    const double tmp1 = w75*(Y_1 + Y_2 + Y_4);
                                    const double tmp2 = w76*(Y_2 + Y_4 + Y_7);
                                    const double tmp3 = w75*(Y_0 + Y_3 + Y_5);
                                    const double tmp4 = w76*(Y_1 + Y_4 + Y_7);
                                    const double tmp5 = w75*(Y_0 + Y_3 + Y_6);
                                    const double tmp6 = w76*(Y_0 + Y_5 + Y_6);
                                    const double tmp7 = w75*(Y_1 + Y_2 + Y_7);
                                    const double tmp8 = w76*(Y_1 + Y_2 + Y_7);
                                    const double tmp9 = w75*(Y_0 + Y_5 + Y_6);
                                    const double tmp10 = w76*(Y_0 + Y_3 + Y_6);
                                    const double tmp11 = w75*(Y_1 + Y_4 + Y_7);
                                    const double tmp12 = w76*(Y_0 + Y_3 + Y_5);
                                    const double tmp13 = w75*(Y_2 + Y_4 + Y_7);
                                    const double tmp14 = w76*(Y_1 + Y_2 + Y_4);
                                    const double tmp15 = w75*(Y_3 + Y_5 + Y_6);
                                    EM_F[INDEX2(k,0,numEq)]+=Y_0*w74 + Y_7*w77 + tmp0 + tmp1;
                                    EM_F[INDEX2(k,1,numEq)]+=Y_1*w74 + Y_6*w77 + tmp2 + tmp3;
                                    EM_F[INDEX2(k,2,numEq)]+=Y_2*w74 + Y_5*w77 + tmp4 + tmp5;
                                    EM_F[INDEX2(k,3,numEq)]+=Y_3*w74 + Y_4*w77 + tmp6 + tmp7;
                                    EM_F[INDEX2(k,4,numEq)]+=Y_3*w77 + Y_4*w74 + tmp8 + tmp9;
                                    EM_F[INDEX2(k,5,numEq)]+=Y_2*w77 + Y_5*w74 + tmp10 + tmp11;
                                    EM_F[INDEX2(k,6,numEq)]+=Y_1*w77 + Y_6*w74 + tmp12 + tmp13;
                                    EM_F[INDEX2(k,7,numEq)]+=Y_0*w77 + Y_7*w74 + tmp14 + tmp15;
                                }
                            } else { // constant data
                                for (index_t k=0; k<numEq; k++) {
                                    EM_F[INDEX2(k,0,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,1,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,2,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,3,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,4,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,5,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,6,numEq)]+=216*Y_p[k]*w58;
                                    EM_F[INDEX2(k,7,numEq)]+=216*Y_p[k]*w58;
                                }
                            }
                        }

                        // add to matrix (if add_EM_S) and RHS (if add_EM_F)
                        const index_t firstNode=m_NN[0]*m_NN[1]*k2+m_NN[0]*k1+k0;
                        domain->addToMatrixAndRHS(mat, rhs, EM_S, EM_F, add_EM_S,
                                add_EM_F, firstNode, numEq, numComp);
                    } // end k0 loop
                } // end k1 loop
            } // end k2 loop
        } // end of colouring
    } // end of parallel region
}

} // namespace ripley

