/************************************************************************
 *
 * Copyright (C) 2009-2025 IRCAD France
 * Copyright (C) 2012-2020 IHU Strasbourg
 *
 * This file is part of Sight.
 *
 * Sight is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sight. If not, see <https://www.gnu.org/licenses/>.
 *
 ***********************************************************************/

/**
 * @brief This file defines the implementation of the class for managing a cardinal layout geometry.
 */

#include "ui/__/layout/cardinal.hpp"

#include "ui/__/detail/parser.hpp"

#include <core/base.hpp>

#include <boost/lexical_cast.hpp>
#include <boost/range/iterator_range.hpp>

namespace sight::ui::layout
{

//-----------------------------------------------------------------------------

const cardinal::registry_key_t cardinal::REGISTRY_KEY = "sight::ui::layout::cardinal";
//-----------------------------------------------------------------------------

const std::map<std::string, cardinal::align> cardinal::STRING_TO_ALIGN = {
    {"center", center},
    {"right", right},
    {"left", left},
    {"bottom", bottom},
    {"top", top},
};

//-----------------------------------------------------------------------------

void cardinal::initialize(const ui::config_t& _configuration)
{
    m_views.clear();

    const auto views_cfg = _configuration.equal_range("view");
    for(const auto& view : boost::make_iterator_range(views_cfg))
    {
        view_info vi;
        if(const auto view_cfg = view.second.get_child_optional("<xmlattr>"); view_cfg.has_value())
        {
            if(const auto align = view_cfg->get_optional<std::string>("align"); align.has_value())
            {
                const auto found = STRING_TO_ALIGN.find(*align);
                SIGHT_ASSERT("Align " << *align << " unknown", found != STRING_TO_ALIGN.end());
                vi.m_align = found->second;
            }

            vi.m_min_size = detail::parse_min_size(*view_cfg);
            vi.m_max_size = detail::parse_max_size(*view_cfg);

            vi.m_is_resizable   = view_cfg->get<bool>("resizable", vi.m_is_resizable);
            vi.m_position       = view_cfg->get<int>("position", vi.m_position);
            vi.m_layer          = view_cfg->get<int>("layer", vi.m_layer);
            vi.m_row            = view_cfg->get<int>("row", vi.m_row);
            vi.m_visible        = view_cfg->get<bool>("visible", vi.m_visible);
            vi.m_use_scroll_bar = view_cfg->get<bool>("useScrollBar", vi.m_use_scroll_bar);
            vi.m_tool_tip       = view_cfg->get<std::string>("toolTip", vi.m_tool_tip);
            vi.m_qss_key        = view_cfg->get<std::string>("QSSClass", "");

            if(auto caption = view_cfg->get_optional<std::string>("caption"); caption.has_value())
            {
                vi.m_caption.first  = true;
                vi.m_caption.second = caption.value();
            }

            vi.m_background_color = detail::parse_background_color(*view_cfg);
        }

        m_views.push_back(vi);
    }
}

//-----------------------------------------------------------------------------

} // namespace sight::ui::layout
