/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     macro.c                                                        */
/*                                                                          */
/* description: reading of macro triangulations; file includes              */
/*              1d/macro_1d.c, 2d/macro_2d.c, 3d/macro_3d.c which contain   */
/*              the dimension dependent parts.                              */ 
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*             Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Albert-Ludwigs-Universitaet Freiburg                         */
/*             Hermann-Herder-Str. 10                                       */
/*             D-79104 Freiburg                                             */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"
#include "alberta_intern.h"

#define VERT_IND(dim,i,j) ((i)*N_VERTICES(dim)+(j))
#define NEIGH_IND(dim,i,j) ((i)*N_NEIGH(dim)+(j))


/*--------------------------------------------------------------------------*/
/* opp_vertex_fast() checks whether the vertex/edge/face with vertices      */
/* test[0],..,test[dim-1] is part of mel's boundary. It returns the         */
/* opposite vertex if true else -1.                                         */
/*--------------------------------------------------------------------------*/

static S_CHAR opp_vertex(int dim, int *mel_vert, int *test)
{
  int      i, j, nv = 0, ov = 0;

  for (i = 0; i < N_VERTICES(dim); i++)
  {
    if (nv < i-1)  return(-1);

    for (j = 0; j < dim; j++)
    {
      if (mel_vert[i] == test[j])
      {
/*--------------------------------------------------------------------------*/
/* i is a common vertex                                                     */
/*--------------------------------------------------------------------------*/
	ov += i;
	nv++;
	break;
      }
    }

  }
  if (nv != dim) return(-1);
/*--------------------------------------------------------------------------*/
/*  the opposite vertex is 1/3/6 - (sum of indices of common vertices) in   */
/*  1d/2d/3d                                                                */
/*--------------------------------------------------------------------------*/
  if (dim == 1)
    return(1-ov);
  else if(dim == 2)
    return(3-ov);
  else
    return(6-ov);
}

/*--------------------------------------------------------------------------*/
/*  compute_neigh_fast() is an algorithm meant to speed up the task of      */
/*  computing neighbours. It does not use an N^2-algorithm.                 */
/*  The idea is to link vertices to elements sharing them to make the       */
/*  search for neighbours more efficient -  at the cost of some additional  */
/*  temporary memory usage.                                 Daniel Koester  */
/*--------------------------------------------------------------------------*/

void compute_neigh_fast(MACRO_DATA *data)
{
  FUNCNAME("compute_neigh_fast");
  int      dim = data->dim;
  int      i, j, k, index, vertices[DIM_OF_WORLD], info=0;
  int      neigh_found = false;
  S_CHAR   l;

  struct vert2elem {
    struct vert2elem *next;
    int mel;
  };

  typedef struct vert2elem VERT2ELEM;

  VERT2ELEM *buffer, *buffer2;
  VERT2ELEM **list = MEM_CALLOC(data->n_total_vertices, VERT2ELEM *);

  if(!data->neigh)
    data->neigh = MEM_ALLOC(data->n_macro_elements*N_NEIGH(dim), int);

/*--------------------------------------------------------------------------*/
/* first initialize elements  (-2 as "undefined")                           */
/*--------------------------------------------------------------------------*/

  for (i = 0; i < data->n_macro_elements; i++)
    for (j = 0; j < N_NEIGH(dim); j++)
      data->neigh[NEIGH_IND(dim,i,j)] = -2;  

/*--------------------------------------------------------------------------*/
/* fill the array "list" of linked lists                                    */
/*--------------------------------------------------------------------------*/

  for(i = 0; i < data->n_macro_elements; i++) {
    for(j = 0; j < N_VERTICES(dim); j++) {
       buffer = list[(index=data->mel_vertices[VERT_IND(dim,i,j)])];

       list[index] = MEM_ALLOC(1, VERT2ELEM);

       list[index]->next = buffer;

       list[index]->mel = i;
    }
  }

/*--------------------------------------------------------------------------*/
/* here comes the actual checking...                                        */
/*--------------------------------------------------------------------------*/

  for (i = 0; i < data->n_macro_elements; i++)
  {
    INFO(info,4,"Current element %d\n",i);
    INFO(info,6,"with vertices: ");
    
    for(j = 0; j < N_VERTICES(dim); j++)
      PRINT_INFO(info,6,"%d ", data->mel_vertices[VERT_IND(dim,i,j)]);
    PRINT_INFO(info,6,"\n");

    for (j = 0; j < N_NEIGH(dim); j++)
    {
      if (data->neigh[NEIGH_IND(dim,i,j)] == -2)
      {
	INFO(info,8,"looking for neighbour no %d\n", j);

	for (k = 0; k < dim; k++)
	  vertices[k] = data->mel_vertices[VERT_IND(dim, i, (j+k+1)%(dim+1))];
       
        buffer = list[vertices[0]];

	neigh_found = false;

        while(buffer) {
          if(buffer->mel != i) {
            if ((l = opp_vertex(dim, data->mel_vertices+buffer->mel*N_VERTICES(dim), vertices)) != -1) {
	      TEST_EXIT(!neigh_found,
	      "Found two neighbours on vertex/edge/face %d of macro el %d!\n",
	       j, i);
              data->neigh[NEIGH_IND(dim,i,j)] = buffer->mel;
              data->neigh[NEIGH_IND(dim,buffer->mel,l)] = i;
              INFO(info,8,"found element %d as neighbour...\n", buffer->mel);
	      neigh_found = true;
	    }
	  }

          buffer = buffer->next;
	}

	if(!neigh_found) {
          INFO(info,8,
 "no neighbour %d of element %d found: Assuming a boundary...\n", j, i);

          data->neigh[NEIGH_IND(dim,i,j)] = -1;
	}
      }
    }
  }

/*--------------------------------------------------------------------------*/
/* now is the time to clean up                                              */
/*--------------------------------------------------------------------------*/


  for(i = 0; i < data->n_total_vertices; i++) {
    buffer = list[i];

    while(buffer) {
       buffer2 = buffer->next;
       MEM_FREE(buffer, 1, VERT2ELEM);

       buffer = buffer2;
    }
  }

  MEM_FREE(list, data->n_total_vertices, VERT2ELEM *);

  return;
}


/*--------------------------------------------------------------------------*/
/*  sets the boundary of all edges without neigbour to DIRICHLET boundary   */
/*  type                                                                    */
/*--------------------------------------------------------------------------*/

void dirichlet_boundary(MACRO_DATA *data)
{
  int      i, dim = data->dim;

  if(!data->boundary)
    data->boundary = MEM_ALLOC(data->n_macro_elements*N_NEIGH(dim), S_CHAR);

  for (i = 0; i < data->n_macro_elements * N_NEIGH(dim); i++)
    data->boundary[i] = (data->neigh[i]>=0) ? INTERIOR : DIRICHLET;

  return;
}


#include "macro_1d.c"
#if DIM_OF_WORLD > 1
#include "macro_2d.c"
#if DIM_OF_WORLD > 2
#include "macro_3d.c"
#endif
#endif


/*--------------------------------------------------------------------------*/
/* read data->neigh into mel[].neigh[]                                      */
/* fill opp_vertex values and do a check on neighbour relations             */
/*--------------------------------------------------------------------------*/

static void fill_neigh_info(MACRO_EL *mel, const MACRO_DATA *data)
{
  FUNCNAME("fill_neigh_info");

  MACRO_EL      *neigh;
  int            i, j, k, index, dim = data->dim;

  for (i = 0; i < data->n_macro_elements; i++) {
    for (j = 0; j < N_NEIGH(dim); j++) {
      mel[i].neigh[j] =
	((index=data->neigh[NEIGH_IND(dim,i,j)]) >= 0) ? (mel+index) : nil;
    }
  }  

  for (i = 0; i < data->n_macro_elements; i++) {
    for (j = 0; j < N_NEIGH(dim); j++) {
      if ((neigh = mel[i].neigh[j])) {
        for (k = 0; k < N_NEIGH(dim); k++)
	  if (neigh->neigh[k] == mel+i)  break;
	
	TEST_EXIT(k<N_NEIGH(dim),"el %d is no neighbour of neighbour %d!\n", 
				  mel[i].index, neigh->index);
        mel[i].opp_vertex[j] = k;
      }
      else {
        mel[i].opp_vertex[j] = -1;
      }
    }
  }

  return;
}


/*--------------------------------------------------------------------------*/
/* domain size                                                              */
/*--------------------------------------------------------------------------*/

static void calculate_size(MESH *mesh, const MACRO_DATA *data)
{
  int         i,j;
  REAL_D      x_min, x_max;

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    x_min[j] = data->coords[0][j];
    x_max[j] = data->coords[0][j];
  }

  for (i = 0; i < mesh->n_vertices; i++)
  {
    for (j = 0; j < DIM_OF_WORLD; j++)
    {
      x_min[j] = MIN(x_min[j], data->coords[i][j]);
      x_max[j] = MAX(x_max[j], data->coords[i][j]);
    }
  }

  for (j = 0; j < DIM_OF_WORLD; j++)
    mesh->diam[j] = x_max[j] - x_min[j];

  return;
}


/*--------------------------------------------------------------------------*/
/*  read_indices()  reads dim+1 indices from  file  into  id[0-dim],        */
/*    returns true if dim+1 inputs arguments could be read successfully by  */
/*    fscanf(), else false                                                  */
/*--------------------------------------------------------------------------*/

static int read_indices(int dim, FILE *file, int id[])
{
  int      i;

  for (i = 0; i <= dim; i++)
    if (fscanf(file, "%d", id+i) != 1)
      return(false);
  return(true);
}

#define N_KEYS      9
#define N_MIN_KEYS  6              
static const char *keys[N_KEYS] = {"DIM",                 /*  0  */
				   "DIM_OF_WORLD",        /*  1  */
				   "number of vertices",  /*  2  */ 
				   "number of elements",  /*  3  */
				   "vertex coordinates",  /*  4  */
				   "element vertices",    /*  5  */
				   "element boundaries",  /*  6  */
				   "element neighbours",  /*  7  */
				   "element type"};       /*  8  */

static int get_key_no(const char *key)
{
  int     i;

  for (i = 0; i < N_KEYS; i++)
    if (!strcmp(keys[i], key))  return(i);

  return(-1);
}

static const char *read_key(const char *line)
{
  static char  key[100];
  char         *k = key;

  while(isspace(*line)) line++;
  while((*k++ = *line++) != ':');
  *--k = '\0';
  
  return((const char *) key);
}


/*--------------------------------------------------------------------------*/
/*  read_macro_data():                                                      */
/*    read macro triangulation from ascii file in ALBERTA format            */
/*    fills macro_data structure                                            */
/*    called by read_macro()                                                */
/*--------------------------------------------------------------------------*/

static MACRO_DATA *read_macro_data(const char *filename)
{
  FUNCNAME("read_macro_data");
  FILE       *file;
  MACRO_DATA *macro_data = nil;
  int        dim, dow, nv, ne, i, j, ind[DIM_OF_WORLD+1];
  REAL       dbl;
  char       name[128], line[256];
  int        line_no, n_keys, i_key, sort_key[N_KEYS], nv_key, ne_key;
  int        key_def[N_KEYS] = {0,0,0,0,0,0,0,0,0};
  const char *key;

  TEST_EXIT(filename,"no file specified; filename nil pointer\n");
  TEST_EXIT(strlen(filename) < (unsigned int) 127,
    "can only handle filenames up to 127 characters\n");

  TEST_EXIT((file=fopen(filename,"r")),"cannot open file %s\n",filename);
  strncpy(name, filename, 127);

/*--------------------------------------------------------------------------*/
/*  looking for all keys in the macro file ...                              */
/*--------------------------------------------------------------------------*/

  line_no = n_keys = 0;
  while (fgets(line, 255, file))
  {
    line_no++;
    if (!strchr(line, ':'))  continue;
    key = read_key(line);
    i_key = get_key_no(key);
    TEST_EXIT(i_key >= 0,
      "file %s: must not contain key %s on line %d\n",
       name, key, line_no);
    TEST_EXIT(!key_def[i_key],
      "file %s: key %s defined second time on line %d\n", name, key, line_no);

    sort_key[n_keys++] = i_key;
    key_def[i_key] = true;
  }
  fclose(file);

  for (i_key = 0; i_key < N_MIN_KEYS; i_key++)
  {
    for (j = 0; j < n_keys; j++)
      if (sort_key[j] == i_key)  break;
    TEST_EXIT(j < n_keys,
    "file %s: You do not have specified data for %s in %s\n",
     name, keys[i_key]);

    for (j = 0; j < n_keys; j++)
      if (sort_key[j] == 2)  break;
    nv_key = j;
    for (j = 0; j < n_keys; j++)
      if (sort_key[j] == 3)  break;
    ne_key = j;
    
    switch(i_key)
    {
    case 0:
    case 1:
      TEST_EXIT(sort_key[i_key] < 2,
	"file %s: You have to specify DIM or DIM_OF_WORLD before all other data\n", name);
      break;
    case 4: 
      TEST_EXIT(nv_key < i_key,
	"file %s: Before reading data for %s, you have to specify the %s\n",
	 name, keys[4], keys[2]);
      break;
    case 5: 
      TEST_EXIT(nv_key < i_key  &&  ne_key < i_key,
	"file %s: Before reading data for %s, you have to specify the %s and %s\n",
	 name, keys[5], keys[3], keys[2]);
    case 6:
    case 7:
    case 8:
      TEST_EXIT(ne_key < i_key,
	"file %s: Before reading data for %s, you have to specify the %s\n",
	 name, keys[i_key], keys[3]);
    }
  }

  for (i_key = 0; i_key < N_KEYS; i_key++)
    key_def[i_key] = false;

/*--------------------------------------------------------------------------*/
/*  and now, reading data ...                                               */
/*--------------------------------------------------------------------------*/
	
  TEST_EXIT((file=fopen(name,"r")),"cannot open file %s\n",name);

  for (i_key = 0; i_key < n_keys; i_key++)
  {
    switch(sort_key[i_key])
    {
    case 0:
      TEST_EXIT(fscanf(file, "%*s %d", &dim) == 1,
	"file %s: can not read DIM correctly\n", name);
      TEST_EXIT(dim <= DIM_OF_WORLD,"file %s: dimension = %d > DIM_OF_WORLD = %d\n", name, dim, DIM_OF_WORLD);
      key_def[0] = true;
      break;
    case 1:
      TEST_EXIT(fscanf(file, "%*s %d", &dow) == 1,
	"file %s: can not read DIM_OF_WORLD correctly\n", name);
      TEST_EXIT(dow == DIM_OF_WORLD,
	"file %s: dimension of world = %d != DIM_OF_WORLD = %d\n", name, dow, DIM_OF_WORLD);

      key_def[1] = true;
      break;
    case 2:
      TEST_EXIT(fscanf(file, "%*s %*s %*s %d", &nv) == 1,
	"file %s: can not read number of vertices correctly\n", name);
      TEST_EXIT(nv > 0,
	"file %s: number of vertices = %d must be bigger than 0\n", name, nv);

      key_def[2] = true;

      if(key_def[3]) 
        macro_data = alloc_macro_data(dim, nv, ne, 0);
      
      break;
    case 3:
      TEST_EXIT(fscanf(file, "%*s %*s %*s %d", &ne) == 1,
	"file %s: can not read number of elements correctly\n", name);
      TEST_EXIT(ne > 0,
	"file %s: number of elements = %d must be bigger than 0\n", name, ne);

      key_def[3] = true;

      if(key_def[2])
        macro_data = alloc_macro_data(dim, nv, ne, 0);

      break;
    case 4:
      fscanf(file, "%*s %*s");
      for (i = 0; i < nv; i++) 
      {
	for (j = 0; j < DIM_OF_WORLD; j++)
	{
	  TEST_EXIT(fscanf(file, "%lf", &dbl) == 1,
	    "file %s: error while reading coordinates, check file\n", name);

	  macro_data->coords[i][j] = dbl;
	}
      }

      key_def[4] = true;
      break;
    case 5:
      fscanf(file, "%*s %*s");
/*--------------------------------------------------------------------------*/
/* global index of vertices for each single element                         */
/*--------------------------------------------------------------------------*/

      for (i = 0; i < ne; i++)
      {
	TEST_EXIT(read_indices(dim, file, ind),
	  "file %s: can not read vertex indices of element %d\n",
	   name, i);

	for (j = 0; j < N_VERTICES(dim); j++)
	  macro_data->mel_vertices[VERT_IND(dim,i,j)] = ind[j];
      }

      key_def[5] = true;
      break;
    case 6:
      fscanf(file, "%*s %*s");

      if(dim == 0)
	ERROR_EXIT("Boundary types do not make sense in 0d!\n");
      else {
/*--------------------------------------------------------------------------*/
/* read boundary type of each vertex/edge/face (in 1d/2d/3d)                */
/*--------------------------------------------------------------------------*/
	macro_data->boundary = MEM_ALLOC(ne*N_NEIGH(dim), S_CHAR);

	for (i = 0; i < ne; i++) {
	  TEST_EXIT(read_indices(dim, file, ind),
	    "file %s: can not read boundary types of element %d\n", name, i);

	  for(j = 0; j < N_NEIGH(dim); j++)
	    macro_data->boundary[NEIGH_IND(dim,i,j)] = (S_CHAR) ind[j];
	}
      }

      key_def[6] = true;
      break;
    case 7:
      fscanf(file, "%*s %*s");

      if(dim == 0)
	ERROR_EXIT("Neighbour indices do not make sense in 0d!\n");
      else {
/*--------------------------------------------------------------------------*/
/* read neighbour indices:                                                  */
/*--------------------------------------------------------------------------*/
	macro_data->neigh = MEM_ALLOC(ne*N_NEIGH(dim), int);

	for (i = 0; i < ne; i++) {
	  TEST_EXIT(read_indices(dim, file, ind),
	    "file %s: can not read neighbour info of element %d\n", name, i);

	  for(j = 0; j < N_NEIGH(dim); j++)
	    macro_data->neigh[NEIGH_IND(dim,i,j)] = ind[j];
	}
      }

      key_def[7] = true;
      break;
    case 8:
      fscanf(file, "%*s %*s");
/*--------------------------------------------------------------------------*/
/* el_type is handled just like bound and neigh above                       */
/*--------------------------------------------------------------------------*/

      if(dim < 3) {
	WARNING("File %s: element type only used in 3d; will ignore data for el_type\n", name);
      }
      else {
	macro_data->el_type = MEM_ALLOC(ne, U_CHAR);

	for (i = 0; i < ne; i++) {
	  TEST_EXIT(fscanf(file, "%d", &j) == 1,
	    "file %s: can not read el_type of element %d\n", name, i);
	  
	  macro_data->el_type[i] = (U_CHAR) j;
	}
      }
      key_def[8] = true;

      break;
    }
  }
  fclose(file);

  return(macro_data);
}

/*--------------------------------------------------------------------------*/
/* read macro triangulation from file "filename" into macro_data data in    */
/* native binary format                                                     */
/*--------------------------------------------------------------------------*/

static MACRO_DATA *read_macro_data_bin(const char *name)
{
  FUNCNAME("read_macro_data_bin");
  FILE       *file;
  MACRO_DATA *macro_data;
  int        dim, i,length, nv, ne;
  char       *s;
  char       record_written;

  TEST_EXIT(file = fopen(name, "rb"),"cannot open file %s\n", name);

  length = MAX(strlen(ALBERTA_VERSION)+1, 21);
  s = MEM_ALLOC(length, char);

  fread(s, sizeof(char), length, file);
  TEST_EXIT(!strncmp(s, "ALBERTA", 6),"file %s: unknown file id:\"%s\"\n",name,s);

  MEM_FREE(s, length, char);

  fread(&i, sizeof(int), 1, file);
  TEST_EXIT(i == sizeof(REAL),"file %s: wrong sizeof(REAL) %d\n", name, i);

  fread(&dim, sizeof(int), 1, file);
  TEST_EXIT(dim <= DIM_OF_WORLD,"file %s: dimension = %d > DIM_OF_WORLD = %d\n", name, dim, DIM_OF_WORLD);

  fread(&i, sizeof(int), 1, file);
  TEST_EXIT(i == DIM_OF_WORLD,
    "file %s: dimension of world = %d != DIM_OF_WORLD = %d\n", name, i, DIM_OF_WORLD);
  
  fread(&nv, sizeof(int), 1, file);
  TEST_EXIT(nv > 0,
    "file %s: number of vertices = %d must be bigger than 0\n", name, nv);

  fread(&ne, sizeof(int), 1, file);
  TEST_EXIT(ne > 0,
    "file %s: number of elements = %d must be bigger than 0\n", name, ne);

  macro_data = alloc_macro_data(dim, nv, ne, 0);

  fread(macro_data->coords, sizeof(REAL_D), nv, file);
  fread(macro_data->mel_vertices, sizeof(int), N_VERTICES(dim) * ne, file);

  fread(&record_written, sizeof(char), 1, file);
  if(record_written) 
  {
    macro_data->boundary = MEM_ALLOC(ne*N_NEIGH(dim), S_CHAR);
    fread(macro_data->boundary, sizeof(S_CHAR), N_NEIGH(dim) * ne, file);
  }

  fread(&record_written, sizeof(char), 1, file);
  if(record_written) 
  {
    macro_data->neigh = MEM_ALLOC(ne*N_NEIGH(dim), int);
    fread(macro_data->neigh, sizeof(int), N_NEIGH(dim) * ne, file);
  }

  if(dim == 3) {
    fread(&record_written, sizeof(char), 1, file);
    if(record_written) {
	macro_data->el_type = MEM_ALLOC(ne, U_CHAR);
	fread(macro_data->el_type, sizeof(U_CHAR), ne, file);
    }
  }

  s = MEM_ALLOC(5, char); 

  TEST_EXIT(fread(s, sizeof(char), 4, file) == 4,
    "file %s: problem while reading FILE END MARK\n", name);

  TEST_EXIT(!strncmp(s, "EOF.", 4),"file %s: no FILE END MARK\n", name);
  MEM_FREE(s, 5, char);

  fclose(file);

  return(macro_data);
}


/*--------------------------------------------------------------------------*/
/* Some routines needed for interaction with xdr-files                      */
/* WARNING: These will need to be adapted if ALBERTA data types REAL, REAL_D*/
/* ,etc. change!                                                            */
/*--------------------------------------------------------------------------*/

static int xdr_dim = 0;

static bool_t xdr_REAL(XDR *xdr, REAL *rp)
{ 
  return (xdr_double(xdr,rp));
}

ALBERTA_DEFUNUSED(static bool_t xdr_U_CHAR(XDR *xdr, U_CHAR *ucp))
{ 
  return (xdr_u_char(xdr,ucp));
}

static bool_t xdr_S_CHAR(XDR *xdr, S_CHAR *cp)
{ 
  return (xdr_char(xdr,(char *)cp));
}

static bool_t xdr_REAL_D(XDR *xdr, REAL_D *dp)
{ 
  return (xdr_vector(xdr, (char *)dp, DIM_OF_WORLD, sizeof(REAL), (xdrproc_t) xdr_REAL));
}

static int read_xdr_file(char *xdr_file, char *buffer, int size)
{
  return ((int)fread(buffer, (size_t)size, 1, (FILE *)xdr_file));
} 

static int write_xdr_file(char *xdr_file, char *buffer, int size)
{
  return (fwrite(buffer, (size_t)size, 1, (FILE *)xdr_file) == 1 ? size : 0);
}


static XDR *xdr_open_file(const char *filename, enum xdr_op mode)
{
  XDR *xdr;
  FILE *xdr_file;

  if (!(xdr = MEM_ALLOC(1,XDR)))
  { 
    ERROR("can't allocate memory for xdr pointer.\n");

    return NULL;
  }
  
  if ((xdr_file = fopen(filename, mode == XDR_DECODE ? "r": "w")))
  {
    xdrrec_create(xdr, 65536, 65536, (caddr_t) xdr_file, 
                  read_xdr_file, write_xdr_file);
    
    xdr->x_op = mode;
    xdr->x_public = (caddr_t)xdr_file;
    
    if (mode == XDR_DECODE)
      xdrrec_skiprecord(xdr);
  
    return xdr;
  }
  else
  {
    ERROR("error opening xdr file.\n"); 
 
    MEM_FREE(xdr,1,XDR);

    return NULL;
  }
}


static int xdr_close_file(XDR *xdr)
{
  if (!xdr)
  {
    ERROR("NULL xdr pointer.\n");
    return 0;
  }

  if (xdr->x_op == XDR_ENCODE)
    xdrrec_endofrecord(xdr, 1);
   
  if (fclose((FILE *) xdr->x_public))
    ERROR("error closing file.\n");

  xdr_destroy(xdr);

  MEM_FREE(xdr,1,XDR);

  return 1;
}

/*--------------------------------------------------------------------------*/
/*  read_macro_data_xdr():                                                  */
/*    read macro triangulation from file in xdr-format                      */
/*    fills macro_data structure                                            */
/*    called by ....?                                                       */
/*--------------------------------------------------------------------------*/

static MACRO_DATA *read_macro_data_xdr(const char *name)
{
  FUNCNAME("read_macro_data_xdr");
  XDR        *xdrp;
  MACRO_DATA *macro_data;
  int        length, dow, nv, ne, size;
  char       *s;
  bool_t     record_written;
  caddr_t    array_loc;


  TEST_EXIT(name,"no file specified; filename nil pointer\n");

  if (!(xdrp = xdr_open_file(name, XDR_DECODE)))
    ERROR_EXIT("cannot open file %s\n",name);

  length = MAX(strlen(ALBERTA_VERSION)+1,21);     /* length with terminating \0 */
  s = MEM_ALLOC(length, char);

  TEST_EXIT(xdr_string(xdrp, &s, length),"file %s: could not read file id\n", name);
  TEST_EXIT(!strncmp(s, "ALBERTA", 6),"file %s: unknown file id: \"%s\"\n",name,s);

  MEM_FREE(s, length, char);

  TEST_EXIT(xdr_int(xdrp, &xdr_dim),"file %s: could not read dimension correctly\n",name);
  TEST_EXIT(xdr_dim <= DIM_OF_WORLD,"file %s: dimension = %d > DIM_OF_WORLD = %d\n", name, xdr_dim, DIM_OF_WORLD);
  

  TEST_EXIT(xdr_int(xdrp, &dow),"file %s: could not read dimension of world correctly\n",name);
  TEST_EXIT(dow == DIM_OF_WORLD,
    "file %s: dimension of world = %d != DIM_OF_WORLD = %d\n", name, dow, DIM_OF_WORLD);

  TEST_EXIT(xdr_int(xdrp, &nv),
    "file %s: can not read number of vertices correctly\n", name);
  TEST_EXIT(nv > 0,
    "file %s: number of vertices = %d must be bigger than 0\n", name, nv);

  TEST_EXIT(xdr_int(xdrp, &ne),
    "file %s: can not read number of elements correctly\n", name);
  TEST_EXIT(ne > 0,
    "file %s: number of elements = %d must be bigger than 0\n", name, ne);
  
  macro_data = alloc_macro_data(xdr_dim, nv, ne, 0);

  array_loc=(caddr_t) macro_data->coords;
  TEST_EXIT(xdr_array(xdrp, &array_loc, (u_int *) &nv, (u_int) nv, sizeof(REAL_D), (xdrproc_t) xdr_REAL_D),
    "file %s: error while reading coordinates, check file\n", name);

  array_loc=(caddr_t) macro_data->mel_vertices;
  size = ne * N_VERTICES(xdr_dim);
  TEST_EXIT(xdr_array(xdrp, &array_loc, (u_int *) &size, (u_int) size,
		      sizeof(int), (xdrproc_t) xdr_int),
    "file %s: can not read vertex indices\n", name);

  TEST_EXIT(xdr_bool(xdrp, &record_written),
    "file %s: could not determine whether to allocate memory for boundaries\n", name);
  if(record_written) 
  {
    macro_data->boundary = MEM_ALLOC(ne*N_NEIGH(xdr_dim), S_CHAR);

    array_loc=(caddr_t) macro_data->boundary;
    size = ne * N_NEIGH(xdr_dim);
    TEST_EXIT(xdr_array(xdrp, &array_loc, (u_int *) &size, (u_int) size,
			sizeof(S_CHAR), (xdrproc_t) xdr_S_CHAR),
      "file %s: could not read boundary types\n",name);
  }

  TEST_EXIT(xdr_bool(xdrp, &record_written),
    "file %s: could not determine whether to allocate memory for neighbours\n", name);
  if(record_written) 
  {
    macro_data->neigh = MEM_ALLOC(ne*N_NEIGH(xdr_dim), int);

    array_loc=(caddr_t) macro_data->neigh;
    size = ne * N_NEIGH(xdr_dim);
    TEST_EXIT(xdr_array(xdrp, &array_loc, (u_int *) &size, (u_int) size,
			sizeof(int), (xdrproc_t) xdr_int),
      "file %s: could not read neighbor info\n",name);
  }

  if(xdr_dim == 3) {
    TEST_EXIT(xdr_bool(xdrp, &record_written),
      "file %s: could not determine whether to allocate memory for element types\n", name);
    if(record_written) 
      {
	macro_data->el_type = MEM_ALLOC(ne, U_CHAR);

	array_loc=(caddr_t) macro_data->el_type;
	TEST_EXIT(xdr_array(xdrp, &array_loc, (u_int *) &ne, (u_int) ne, sizeof(U_CHAR), (xdrproc_t) xdr_U_CHAR),
	  "file %s: can not read element types\n", name);
      }
  }

  xdr_close_file(xdrp);

  return(macro_data);
}

/*--------------------------------------------------------------------------*/
/* supported file types for macro data files:                               */
/*--------------------------------------------------------------------------*/

typedef enum {ascii_format, binary_format, xdr_format} macro_format;

/*--------------------------------------------------------------------------*/
/* read macro triangulation from file "filename"                            */
/*--------------------------------------------------------------------------*/

static MACRO_DATA *read_macro_master(const char *filename, 
				     macro_format format)
{
  FUNCNAME("read_macro_master");
  MACRO_DATA *macro_data = nil;
  char       filenew[1024];

  TEST_EXIT(filename,"no file specified; filename nil pointer\n");

  switch(format) {
  case ascii_format:
    macro_data = read_macro_data(filename);
    break;
  case binary_format:
    macro_data = read_macro_data_bin(filename);
    break;
  case xdr_format:
    macro_data = read_macro_data_xdr(filename);
  }

  if(!macro_data->neigh && macro_data->dim > 0) compute_neigh_fast(macro_data);
  if(!macro_data->boundary && macro_data->dim > 0)
    dirichlet_boundary(macro_data);

  strncpy(filenew, filename, 1024); filenew[1023] = 0;
  strncat(filenew, ".new", 1024);   filenew[1023] = 0;
  macro_test(macro_data, filenew);

  return macro_data;
}


static void cleanup_write_macro(MACRO_DATA *data, DOF_INT_VEC *dof_vert_ind,
                                TRAVERSE_STACK *stack) 
{

  if (data) free_macro_data(data);
  free_dof_int_vec(dof_vert_ind);
  free_traverse_stack(stack);

  return;
}


/*--------------------------------------------------------------------------*/
/* mesh2macro_data(): counterpart to macro_data2mesh below. This routine    */
/* converts the information stored in the leaf elements of mesh to the raw  */
/* data type MACRO_DATA.                                                    */
/*--------------------------------------------------------------------------*/

extern MACRO_DATA *mesh2macro_data(MESH *mesh)
{
  FUNCNAME("mesh2macro_data");
  MACRO_DATA      *data;
  TRAVERSE_STACK  *stack;
  FLAGS           fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_BOUND|FILL_NEIGH;
  PARAMETRIC      *parametric;
  const DOF_ADMIN *admin;
  FE_SPACE        fe_space = {"write fe_space", nil, nil};
  const EL_INFO   *el_info;
  DOF_INT_VEC     *dof_vert_ind;
  int             dim = mesh->dim, n0, ne, nv, i, j, *vert_ind = nil;
  U_CHAR          write_el_type;
  static const REAL vertex_bary[N_VERTICES_MAX][N_LAMBDA] = 
    {{1.0, 0.0, 0.0, 0.0},
     {0.0, 1.0, 0.0, 0.0},
     {0.0, 0.0, 1.0, 0.0},
     {0.0, 0.0, 0.0, 1.0}};


  admin = get_vertex_admin(mesh);

  n0 = admin->n0_dof[VERTEX];
  fe_space.admin = admin;

  parametric = mesh->parametric;

  dof_vert_ind = get_dof_int_vec("vertex indices", &fe_space);
  GET_DOF_VEC(vert_ind, dof_vert_ind);
  FOR_ALL_DOFS(admin, vert_ind[dof] = -1);

  data = alloc_macro_data(dim, mesh->n_vertices, mesh->n_elements, 0);

  nv = ne = 0;
  write_el_type = false;

  stack = get_traverse_stack();

/*--------------------------------------------------------------------------*/
/* The first pass counts elements and vertices, checks these against the    */
/* entries of mesh->n_elements, mesh->n_vertices, and fills data->coords.   */
/* A check on whether an element has nonzero el_type is also done.          */
/*--------------------------------------------------------------------------*/
  for(el_info = traverse_first(stack, mesh, -1, CALL_LEAF_EL | FILL_COORDS);
      el_info;
      el_info = traverse_next(stack, el_info)) {

    if (parametric) {
      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, N_VERTICES(dim),
				 vertex_bary, (REAL_D *)el_info->coord);
    }

    for (i = 0; i < N_VERTICES(dim); i++) {
      if (vert_ind[el_info->el->dof[i][n0]] == -1) {
/*--------------------------------------------------------------------------*/
/* assign a global index to each vertex                                     */
/*--------------------------------------------------------------------------*/
        vert_ind[el_info->el->dof[i][n0]] = nv;    

	for (j = 0; j < DIM_OF_WORLD; j++) 
	  data->coords[nv][j] = el_info->coord[i][j];

        nv++;

        if(nv > mesh->n_vertices) 
	{
          cleanup_write_macro(data, dof_vert_ind, stack);
          ERROR("mesh %s: n_vertices (==%d) is too small! Writing aborted\n",
                mesh->name, mesh->n_vertices);
          return(nil);
        }
      }        
    }

    ne++;

    if(ne > mesh->n_elements) 
    {
      cleanup_write_macro(data, dof_vert_ind, stack);
      ERROR("mesh %s: n_elements (==%d) is too small! Writing aborted\n", 
            mesh->name, mesh->n_elements);
      return(nil);
    }
    if(dim == 3 && el_info->el_type) write_el_type = true;
  }

  if(ne < mesh->n_elements) 
  {
    cleanup_write_macro(data, dof_vert_ind, stack);
    ERROR("mesh %s: n_elements (==%d) is too large: only %d leaf elements counted -- writing aborted\n", mesh->name, mesh->n_elements, ne);
    return(nil);
  }
  if(nv < mesh->n_vertices)
  {
    cleanup_write_macro(data, dof_vert_ind, stack);
    ERROR("mesh %s: n_vertices (==%d) is too large: only %d vertices counted --  allocation of macro data aborted\n", mesh->name, mesh->n_vertices, nv);
    return(nil);
  }

  if(dim > 0)
    data->boundary = MEM_ALLOC(ne*N_NEIGH(dim), S_CHAR);

  if(write_el_type) data->el_type = MEM_ALLOC(ne, U_CHAR);  

  ne = 0;
  
/*--------------------------------------------------------------------------*/
/* The second pass assigns mel_vertices, boundary, and if necessary el_type */
/*--------------------------------------------------------------------------*/
  for(el_info = traverse_first(stack, mesh, -1, fill_flag);
      el_info;
      el_info = traverse_next(stack, el_info)) {

    for (i = 0; i < N_VERTICES(dim); i++)
      data->mel_vertices[VERT_IND(dim,ne,i)] = 
	vert_ind[el_info->el->dof[i][n0]];    

    if(dim > 0)
      for (i = 0; i < N_NEIGH(dim); i++)
	switch(dim) {
	case 1:
	  data->boundary[NEIGH_IND(dim,ne,i)] = el_info->vertex_bound[1-i];
	  break;
	case 2:
	  data->boundary[NEIGH_IND(dim,ne,i)] = el_info->edge_bound[i];
	  break;
	case 3:
	  data->boundary[NEIGH_IND(dim,ne,i)] = el_info->face_bound[i];
	}

    if(write_el_type) data->el_type[ne] = el_info->el_type;

    ++ne;
  }

/*--------------------------------------------------------------------------*/
/* Finally, we compute neighbour information. This seems to be the easiest  */
/* solution, since neighbor information in ALBERTA is only available as     */
/* pointers.                                                                */
/*--------------------------------------------------------------------------*/

  if(dim > 0)
    compute_neigh_fast(data);

  cleanup_write_macro(nil, dof_vert_ind, stack);

  return(data);
}


/*--------------------------------------------------------------------------*/
/* write_macro() writes the current mesh (at the level of leaf elements) as */
/* a macro triangulation to the specified file                              */
/*--------------------------------------------------------------------------*/

static int write_macro_master(MESH *mesh, const char *filename,
			      macro_format format)
{
  FUNCNAME("write_macro_master");
  int         result = 0; /* make gcc happy */
  MACRO_DATA  *data;  

  if (!filename)
  {
    ERROR("no filename specified, filename is nil pointer\n");
    return(0);
  }

  if (!mesh)
  {
    ERROR("no mesh specified, mesh is nil pointer\n");
    return(0);
  }

  if(!(data = mesh2macro_data(mesh))) {
    ERROR("Could not convert mesh to a macro data structure!\n");
    return(0);
  }

  switch(format) 
  {
  case ascii_format:
    result=write_macro_data(data, filename);
    break;
  case binary_format:
    result=write_macro_data_bin(data, filename);
    break;
  case xdr_format:
    result=write_macro_data_xdr(data, filename);
  }
       
  free_macro_data(data);

  return(result);
}


/*--------------------------------------------------------------------------*/
/* These routines are available to the user:                                */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  initialize and clear macro data structures                              */
/*--------------------------------------------------------------------------*/

extern MACRO_DATA *alloc_macro_data(int dim, int nv, int ne, FLAGS flag)
{
  FUNCNAME("alloc_macro_data");
  MACRO_DATA *data = MEM_CALLOC(1, MACRO_DATA);

  data->dim              = dim;
  data->n_total_vertices = nv;
  data->n_macro_elements = ne;

  data->coords       = MEM_ALLOC(nv, REAL_D);

  data->mel_vertices = MEM_ALLOC(ne*N_VERTICES(dim), int);

  if (flag & FILL_NEIGH)
    data->neigh = MEM_ALLOC(ne*N_NEIGH(dim), int);

  if (flag & FILL_BOUND)
    data->boundary = MEM_ALLOC(ne*N_NEIGH(dim), S_CHAR);

  if ((flag & FILL_EL_TYPE) && dim == 3)
    data->el_type = MEM_ALLOC(ne, U_CHAR);

  return(data);
}

extern void free_macro_data(MACRO_DATA *data)
{
  int dim = data->dim,
    ne = data->n_macro_elements,
    nv = data->n_total_vertices;

  MEM_FREE(data->coords, nv, REAL_D);

  MEM_FREE(data->mel_vertices, ne*N_VERTICES(dim), int);

  if(data->neigh) MEM_FREE(data->neigh, ne*N_NEIGH(dim), int);
  if(data->boundary) MEM_FREE(data->boundary, ne*N_NEIGH(dim), S_CHAR); 
  if(dim == 3 && data->el_type) MEM_FREE(data->el_type, ne, U_CHAR);  

  MEM_FREE(data, 1, MACRO_DATA);

  return;
}

extern MACRO_DATA *read_macro(const char *filename)
{
  return read_macro_master(filename, ascii_format);
}

extern MACRO_DATA *read_macro_bin(const char *filename)
{
  return read_macro_master(filename, binary_format);
}

extern MACRO_DATA *read_macro_xdr(const char *filename)
{
  return read_macro_master(filename, xdr_format);
}


extern int write_macro(MESH *mesh, const char *filename)
{
   return(write_macro_master(mesh, filename, ascii_format));
}

extern int write_macro_bin(MESH *mesh, const char *filename)
{
   return(write_macro_master(mesh, filename, binary_format));
}

extern int write_macro_xdr(MESH *mesh, const char *filename)
{
   return(write_macro_master(mesh, filename, xdr_format));
}

/*--------------------------------------------------------------------------*/
/* write raw macro triangulation in "data" to "filename" in standard ALBERTA*/
/* key format                                                               */
/*--------------------------------------------------------------------------*/

extern int write_macro_data(MACRO_DATA *data, const char *filename)
{
  FUNCNAME("write_macro_data");
  FILE    *macro_file;  
  int     i, j, dim = data->dim;

  if (!(macro_file = fopen(filename, "w")))
  {
    ERROR("could not open file %s for writing\n", filename);
    return(0);
  }

  fprintf(macro_file, "DIM: %d\n", dim);
  fprintf(macro_file, "DIM_OF_WORLD: %d\n\n", DIM_OF_WORLD);

  fprintf(macro_file, "number of vertices: %d\n", data->n_total_vertices);
  fprintf(macro_file, "number of elements: %d\n\n", data->n_macro_elements);

  fprintf(macro_file, "vertex coordinates:\n");
  for(i = 0; i < data->n_total_vertices; i++)
    for (j = 0; j < DIM_OF_WORLD; j++)
      fprintf(macro_file, "%17.10e%s", data->coords[i][j], 
	    j < DIM_OF_WORLD-1 ? " " : "\n");

  fprintf(macro_file, "\nelement vertices:\n");
  for(i = 0; i < data->n_macro_elements; i++)
    for (j = 0; j < N_VERTICES(dim); j++)
      fprintf(macro_file, "%5d%s", data->mel_vertices[VERT_IND(dim,i,j)], 
	    j < N_VERTICES(dim)-1 ? " " : "\n");

  if(data->boundary) {
    fprintf(macro_file, "\nelement boundaries:\n");
    for(i = 0; i < data->n_macro_elements; i++)
      for (j = 0; j < N_NEIGH(dim); j++)
	fprintf(macro_file, "%4d%s", data->boundary[NEIGH_IND(dim,i,j)], 
		j < N_NEIGH(dim)-1 ? " " : "\n");
  }

  if(data->neigh) {
    fprintf(macro_file, "\nelement neighbours:\n");
    for(i = 0; i < data->n_macro_elements; i++)
      for (j = 0; j < N_NEIGH(dim); j++)
	fprintf(macro_file, "%4d%s", data->neigh[NEIGH_IND(dim,i,j)], 
		j < N_NEIGH(dim)-1 ? " " : "\n");
  }

  if (dim == 3 && data->el_type) 
  {
    fprintf(macro_file, "\nelement type:\n");
    for(i = 0; i < data->n_macro_elements; i++) 
      fprintf(macro_file, "%d%s", data->el_type[i],  ((i+1)%20) ? " ": "\n");
  }

  if (!(i%20))
    fprintf(macro_file, "\n");

  fclose(macro_file);

  INFO(2,2,"wrote macro file %s\n", filename);

  return(1);
}

/*--------------------------------------------------------------------------*/
/* write raw macro triangulation in "data" to "filename" in native binary   */
/* format                                                                   */
/*--------------------------------------------------------------------------*/

extern int write_macro_data_bin(MACRO_DATA *data, const char *filename)
{
  FUNCNAME("write_macro_data_bin");
  FILE *file;
  int  i, dim = data->dim;
  char record_written=1;
  char record_not_written=0;


  if(!data) 
  {
    ERROR("no data - no file created\n");
    return(0);
  }

  if (!(file = fopen(filename, "wb")))
  {
    ERROR("cannot open file %s\n",filename);
    return(0);
  }

  fwrite(ALBERTA_VERSION, sizeof(char), strlen(ALBERTA_VERSION)+1, file);

  i = sizeof(REAL);
  fwrite(&i, sizeof(int), 1, file);

  fwrite(&dim, sizeof(int), 1, file);
  
  i = DIM_OF_WORLD;
  fwrite(&i, sizeof(int), 1, file);
  
  fwrite(&(data->n_total_vertices), sizeof(int), 1, file);
  fwrite(&(data->n_macro_elements), sizeof(int), 1, file);

  fwrite(data->coords, sizeof(REAL_D), data->n_total_vertices, file);
  fwrite(data->mel_vertices, sizeof(int),
	 N_VERTICES(dim) * data->n_macro_elements, file);

  if(data->boundary) {
    fwrite(&record_written, sizeof(char), 1, file);
    fwrite(data->boundary, sizeof(S_CHAR), 
	   N_NEIGH(dim) * data->n_macro_elements, file);
  }
  else fwrite(&record_not_written, sizeof(char), 1, file);

  if(data->neigh) {
    fwrite(&record_written, sizeof(char), 1, file);
    fwrite(data->neigh, sizeof(int),
	   N_NEIGH(dim) * data->n_macro_elements, file);
  }
  else fwrite(&record_not_written, sizeof(char), 1, file);

  if (dim == 3 && data->el_type) {
    fwrite(&record_written, sizeof(char), 1, file);
    fwrite(data->el_type, sizeof(U_CHAR), data->n_macro_elements, file);
  }
  else fwrite(&record_not_written, sizeof(char), 1, file);

  fwrite("EOF.", sizeof(char), 4, file);
  fclose(file);

  INFO(2,2,"wrote macro binary-file %s\n", filename);

  return(1);
}

/*--------------------------------------------------------------------------*/
/* write raw macro triangulation in "data" to "filename" in xdr format      */
/*--------------------------------------------------------------------------*/

extern int write_macro_data_xdr(MACRO_DATA *data, const char *filename)
{
  FUNCNAME("write_macro_data_xdr");
  XDR    *xdrp;
  int    i, length;
  char   *s;
  bool_t record_written=1;
  bool_t record_not_written=0;

  caddr_t array_loc;
  
  if(!data) 
  {
    ERROR("no data - no file created\n");
    return(0);
  }

  if (!(xdrp = xdr_open_file(filename, XDR_ENCODE)))
  {
    ERROR("cannot open file %s\n",filename);
    return(0);
  }

  length = MAX(strlen(ALBERTA_VERSION) + 1, 5);  /* length with terminating \0 */
  s=MEM_ALLOC(length, char);
  strcpy(s, ALBERTA_VERSION);
  xdr_string(xdrp, &s, length);
  MEM_FREE(s, length, char);

  xdr_dim = data->dim;

  xdr_int(xdrp, &xdr_dim);
  
  i = DIM_OF_WORLD;
  xdr_int(xdrp, &i); 
  
  xdr_int(xdrp, &(data->n_total_vertices));
  xdr_int(xdrp, &(data->n_macro_elements));

  array_loc=(caddr_t) data->coords;
  xdr_array(xdrp, &array_loc, (u_int *) &(data->n_total_vertices),
	    (u_int) data->n_total_vertices, sizeof(REAL_D),
	    (xdrproc_t) xdr_REAL_D);
 
  array_loc=(caddr_t) data->mel_vertices;
  xdr_array(xdrp, &array_loc, (u_int *) &(data->n_macro_elements),
	    (u_int) data->n_macro_elements * N_VERTICES(xdr_dim), sizeof(int),
	    (xdrproc_t) xdr_int);

  if(data->boundary) {
    xdr_bool(xdrp, &record_written);
    array_loc=(caddr_t) data->boundary;
    xdr_array(xdrp, &array_loc, (u_int *) &(data->n_macro_elements),
      (u_int) data->n_macro_elements * N_NEIGH(xdr_dim), sizeof(S_CHAR),
	      (xdrproc_t) xdr_S_CHAR);
  }
  else xdr_bool(xdrp, &record_not_written);

  if(data->neigh) {
    xdr_bool(xdrp, &record_written);
    array_loc=(caddr_t) data->neigh;
    xdr_array(xdrp, &array_loc, (u_int *) &(data->n_macro_elements),
	      (u_int) data->n_macro_elements * N_NEIGH(xdr_dim), sizeof(int),
	      (xdrproc_t) xdr_int);
  }
  else xdr_bool(xdrp, &record_not_written);

  if (xdr_dim == 3 && data->el_type) {
    xdr_bool(xdrp, &record_written);
    array_loc=(caddr_t) data->el_type;
    xdr_array(xdrp, &array_loc, (u_int *) &(data->n_macro_elements),
	      (u_int) data->n_macro_elements, sizeof(U_CHAR),
	      (xdrproc_t) xdr_U_CHAR);
  }
  else xdr_bool(xdrp, &record_not_written);

  xdr_close_file(xdrp);

  INFO(2,2,"wrote macro xdr-file %s\n", filename);

  return(1);
}


/***************************************************************************/
/*  macro_data2mesh():                                                     */
/*  copy macro data to the MESH structure "mesh" provided:                 */
/*  1) set most entries in "mesh"                                          */
/*  2) allocate macro elements and link them to "mesh"                     */
/*  3) calculate macro element orientation for 3D                          */ 
/*  4) calculate the mesh size for "mesh->diam"                            */ 
/*  5) Initialize slave meshes                                             */ 
/*                                                                         */
/*  the entire MACRO_DATA structure can be freed after use!                */
/***************************************************************************/

extern void macro_data2mesh(MESH *mesh, const MACRO_DATA *data,
	    NODE_PROJECTION *(*n_proj)(MESH *, MACRO_EL *, int))
{
  FUNCNAME("macro_data2mesh");
  int            i, j, dim = data->dim;
  MACRO_EL      *mel;
  REAL_D        *newcoords;

  TEST_EXIT(mesh,"no mesh, mesh is nil pointer!\n");

  mesh->dim = dim;

  mesh->n_elements = mesh->n_hier_elements = mesh->n_macro_el = data->n_macro_elements;
  mesh->n_vertices = data->n_total_vertices;

  mel = mesh->macro_els = MEM_CALLOC(data->n_macro_elements, MACRO_EL);

  newcoords = MEM_ALLOC(data->n_total_vertices, REAL_D);

  for(i = 0; i < data->n_total_vertices; i++)
    for(j = 0; j < DIM_OF_WORLD; j++)
      newcoords[i][j] = data->coords[i][j];

  ((MESH_MEM_INFO *)mesh->mem_info)->count = data->n_total_vertices;
  ((MESH_MEM_INFO *)mesh->mem_info)->coords = newcoords;

  for(i = 0; i < data->n_macro_elements; i++) {
    mel[i].el = get_element(mesh);

    mel[i].index = i;
#if ALBERTA_DEBUG
    mel[i].el->index = i;
#endif

    for(j = 0; j < N_VERTICES(dim); j++)
      mel[i].coord[j] = newcoords[data->mel_vertices[VERT_IND(dim,i,j)]];  

#if DIM_OF_WORLD == 3
    if(dim == 3) {
      mel[i].el_type = data->el_type ? data->el_type[i] : 0;
      mel[i].orientation = AI_get_orientation(mel + i);
    }
#endif
  }

  if (mesh->parametric)
    WARNING("mesh->diam not set, problems with graphical output may occur\n");
  else
    calculate_size(mesh, data);

  if(dim > 0) {
    if (!data->neigh) ERROR_EXIT("Neighbor information must be present!\n");
    fill_neigh_info(mel, data);

    if (!data->boundary) ERROR_EXIT("Boundary information must be present!\n");

    switch(dim) {
    case 1:
      fill_bound_info_1d(mesh, data);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      fill_bound_info_2d(mesh, data);
      count_edges_2d(mesh);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      fill_bound_info_3d(mesh, data);
      fill_more_bound_info_3d(mesh, data);
      break;
#endif
#endif
    default:
      ERROR_EXIT("Illegal dimension %d!\n", dim);
    }
  }

/****************************************************************************/
/* Call the user-defined new vertex projection assignment routine "n_proj". */
/****************************************************************************/
  if(n_proj) {
    for (i = 0; i < mesh->n_macro_el; i++) {
      mel[i].projection[0] = n_proj(mesh, mel + i, 0);
      
#if DIM_OF_WORLD > 1
      if(dim == 2)
	for(j = 1; j < N_NEIGH_2D + 1; j++)
	  mel[i].projection[j] = n_proj(mesh, mel + i, j);
#if DIM_OF_WORLD > 2
      else if(dim == 3)
	for(j = 1; j < N_NEIGH_3D + 1; j++)
	  mel[i].projection[j] = n_proj(mesh, mel + i, j);
#endif
#endif

#if 0 /* No point in doing this at the moment, DK. */
/****************************************************************************/
/* If necessary, copy projections to neighbour elements.                    */
/****************************************************************************/
      for(j = 0; j < N_NEIGH(dim); j++) {
	NODE_PROJECTION *tmp_proj;
	MACRO_EL *neigh;

	if(neigh = mel[i].neigh[j]) {
	  tmp_proj = mel[i].projection[0];
	  
	  if(mel[i].projection[j+1])
	    tmp_proj = mel[i].projection[j+1];
	  
	  /* Search for the correct subsimplex on the neighbour element. */
	  for(k = 0; k < N_NEIGH(dim); k++)
	    if(neigh->neigh[k] == mel + i) break;
	  
	  neigh->projection[k+1] = tmp_proj;
	}
#endif
    }
  }

  return;
}


extern void macro_test(MACRO_DATA *data, const char *new_name)
{
  FUNCNAME("macro_test");

  switch(data->dim) {
  case 0:
    break;
  case 1:
    macro_test_1d(data, new_name);
    break;
#if DIM_OF_WORLD > 1
  case 2:
    macro_test_2d(data, new_name);
    break;
#if DIM_OF_WORLD > 2
  case 3:
    macro_test_3d(data, new_name);
    break;
#endif
#endif
  default:
    ERROR_EXIT("Illegal dim == %d!\n", data->dim);
  }
}
