
##############################################################################
#
# Copyright (c) 2003-2017 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2017 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
Tests for integration on domains in the unit square/cube.

This file was auto-generated.
"""

import esys.escriptcore.utestselect as unittest
from esys.escript import *
import numpy

class Test_Util_Integration_noContact(unittest.TestCase):
    RES_TOL=1.e-8

    def test_integrateC_onFunction_fromData_ContinuousFunction_rank0(self):
        """
        tests integral of rank 0 Data on Function

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(-0.136080245421-0.684416740072j)*x[0]**o + ((-0.839305931283+0.0192526883543j))*x[0] + ((-0.0730680909085+0.708193534423j))*x[1]**o + ((-0.452780669865+0.646351329695j))*x[1]
            ref=(-0.750617468739+0.3446904062j)-((-0.034858056055+0.00396279905842j))*(o-1.)
        else:
            arg=(0.393054652003-0.920268849978j)*x[0]**o + ((0.388912501271-0.0510880123987j))*x[0] + ((-0.55357984142-0.387916408574j))*x[1]**o + ((0.380778014426-0.641941674697j))*x[1] + ((-0.637595521285-0.439454123991j))*x[2]**o + ((0.902408333704+0.969752205981j))*x[2]
            ref=(0.43698906935-0.735458431829j)-((-0.13302011845-0.291273230424j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ContinuousFunction_rank1(self):
        """
        tests integral of rank 1 Data on Function

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(0.0449968676495-0.97084060307j)*x[0]**o + ((-0.0666567131816+0.808098732751j))*x[0] + ((-0.0381202261616-0.445765477194j))*x[1]**o + ((-0.217831679616-0.446850647901j))*x[1]
            ref[(0,)]=(-0.138805875655-0.527678997707j)-((0.00114610691466-0.236101013377j))*(o-1.)
            arg[(1,)]=(-0.195458218672-0.585859112329j)*x[0]**o + ((-0.461705303456+0.938982442584j))*x[0] + ((0.896246254964+0.590365918932j))*x[1]**o + ((0.196948286415+0.205322763442j))*x[1]
            ref[(1,)]=(0.218015509625+0.574406006314j)-((0.116798006049+0.000751134433764j))*(o-1.)
            arg[(2,)]=(0.168960716466-0.222451211798j)*x[0]**o + ((-0.836872963892+0.679320375218j))*x[0] + ((0.988171737458+0.620373949447j))*x[1]**o + ((0.142921842054+0.597257030145j))*x[1]
            ref[(2,)]=(0.231590666043+0.837250071506j)-((0.192855408987+0.0663204562749j))*(o-1.)
            arg[(3,)]=(0.483237652324+0.877941620163j)*x[0]**o + ((-0.135039177508-0.440791040918j))*x[0] + ((-0.816960583666+0.842269658478j))*x[1]**o + ((0.776494905322-0.983208238941j))*x[1]
            ref[(3,)]=(0.153866398236+0.148105999391j)-((-0.0556204885569+0.286701879773j))*(o-1.)
        else:
            arg[(0,)]=(-0.510449138625+0.732037504546j)*x[0]**o + ((-0.817353168634+0.569297470189j))*x[0] + ((0.570057123236-0.808436382221j))*x[1]**o + ((0.937785307226-0.982778708703j))*x[1] + ((0.170866798731+0.855410030151j))*x[2]**o + ((0.758631736897+0.576139066242j))*x[2]
            ref[(0,)]=(0.554769329416+0.470834490102j)-((0.0384124638903+0.129835192079j))*(o-1.)
            arg[(1,)]=(-0.584173654322+0.807644747057j)*x[0]**o + ((0.55545850112-0.38178467032j))*x[0] + ((0.653549606011-0.538646643758j))*x[1]**o + ((0.203301069198+0.692917537966j))*x[1] + ((0.363257909371+0.0877479758101j))*x[2]**o + ((0.442058697873-0.5359361379j))*x[2]
            ref[(1,)]=(0.816726064625+0.0659714044273j)-((0.07210564351+0.0594576798514j))*(o-1.)
            arg[(2,)]=(0.826344235191-0.257890102118j)*x[0]**o + ((0.828372008554+0.889256355329j))*x[0] + ((0.383049637013-0.919317684865j))*x[1]**o + ((0.539172755484+0.419686368724j))*x[1] + ((-0.416624047095-0.952058028868j))*x[2]**o + ((-0.962570869637+0.725162930121j))*x[2]
            ref[(2,)]=(0.598871859755-0.0475800808384j)-((0.132128304185-0.354877635975j))*(o-1.)
            arg[(3,)]=(0.63058314827+0.00694802365864j)*x[0]**o + ((-0.390920963899+0.381391501418j))*x[0] + ((-0.563081166658+0.620529245646j))*x[1]**o + ((-0.48565390028+0.0843952946953j))*x[1] + ((-0.717012211394+0.307598180395j))*x[2]**o + ((0.892042105778+0.0562737461129j))*x[2]
            ref[(3,)]=(-0.317021494091+0.728567995963j)-((-0.108251704964+0.155845908283j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ContinuousFunction_rank2(self):
        """
        tests integral of rank 2 Data on Function

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4),w)
        ref=numpy.zeros((3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.724412797748+0.00989028456802j)*x[0]**o + ((-0.390537950213+0.550774780249j))*x[0] + ((-0.0141235872467-0.353161436056j))*x[1]**o + ((0.27305264677-0.52591124992j))*x[1]
            ref[(0, 0)]=(0.296401953529-0.159203810579j)-((0.118381535084-0.0572118585813j))*(o-1.)
            arg[(0, 1)]=(0.867680448203-0.290801500572j)*x[0]**o + ((-0.035326596811+0.608968094352j))*x[0] + ((0.440937989778+0.477929052197j))*x[1]**o + ((0.182759451116-0.963377050713j))*x[1]
            ref[(0, 1)]=(0.728025646143-0.083640702368j)-((0.218103072997+0.0311879252709j))*(o-1.)
            arg[(0, 2)]=(-0.346496717917+0.473818076769j)*x[0]**o + ((-0.464137693793+0.57346412444j))*x[0] + ((-0.2652875083-0.0636810900557j))*x[1]**o + ((0.345165153475-0.842654543956j))*x[1]
            ref[(0, 2)]=(-0.365378383268+0.0704732835989j)-((-0.101964037703+0.0683561644523j))*(o-1.)
            arg[(0, 3)]=(0.590820049788+0.577066048492j)*x[0]**o + ((0.395207040203-0.821811422553j))*x[0] + ((-0.720486974045-0.193415057729j))*x[1]**o + ((0.54391347489+0.894047847071j))*x[1]
            ref[(0, 3)]=(0.404726795418+0.22794370764j)-((-0.0216111540429+0.0639418317938j))*(o-1.)
            arg[(1, 0)]=(-0.90350618973-0.102406349621j)*x[0]**o + ((-0.429203111882-0.660633055096j))*x[0] + ((-0.0738063392198-0.241265012728j))*x[1]**o + ((0.191539456445+0.218471242276j))*x[1]
            ref[(1, 0)]=(-0.607488092193-0.392916587584j)-((-0.162885421492-0.0572785603914j))*(o-1.)
            arg[(1, 1)]=(-0.838946477772-0.512493054277j)*x[0]**o + ((-0.384867717261-0.0998381436971j))*x[0] + ((-0.836580354303-0.74263126768j))*x[1]**o + ((-0.155756033547+0.601405780715j))*x[1]
            ref[(1, 1)]=(-1.10807529144-0.376778342469j)-((-0.279254472013-0.209187386993j))*(o-1.)
            arg[(1, 2)]=(-0.123833166362+0.0890793906869j)*x[0]**o + ((0.360074178329+0.269454221425j))*x[0] + ((0.173134343301-0.391463565096j))*x[1]**o + ((0.709692920905-0.263072453353j))*x[1]
            ref[(1, 2)]=(0.559534138086-0.148001203168j)-((0.00821686282307-0.0503973624015j))*(o-1.)
            arg[(1, 3)]=(0.402659951336+0.229421133883j)*x[0]**o + ((-0.545770344692-0.0750128579142j))*x[0] + ((-0.913062099782+0.589353996729j))*x[1]**o + ((-0.294582632638-0.817760256535j))*x[1]
            ref[(1, 3)]=(-0.675377562888-0.0369989919184j)-((-0.085067024741+0.136462521769j))*(o-1.)
            arg[(2, 0)]=(-0.862307374651+0.301426117175j)*x[0]**o + ((0.00992729990314-0.748680615667j))*x[0] + ((-0.453561723106+0.258648914978j))*x[1]**o + ((0.594081595981-0.137741268289j))*x[1]
            ref[(2, 0)]=(-0.355930100936-0.163173425901j)-((-0.219311516293+0.0933458386922j))*(o-1.)
            arg[(2, 1)]=(0.957174318706-0.850311163927j)*x[0]**o + ((-0.72310421954-0.514525289228j))*x[0] + ((-0.611827553577-0.939151862012j))*x[1]**o + ((0.703867984888+0.957044336967j))*x[1]
            ref[(2, 1)]=(0.163055265239-0.6734719891j)-((0.0575577941881-0.298243837656j))*(o-1.)
            arg[(2, 2)]=(0.294290475686+0.176554845118j)*x[0]**o + ((0.5820559586+0.538487061789j))*x[0] + ((-0.0312147099558-0.0501244096285j))*x[1]**o + ((-0.614233081666+0.0407983588927j))*x[1]
            ref[(2, 2)]=(0.115449321332+0.352857928086j)-((0.0438459609551+0.0210717392483j))*(o-1.)
            arg[(2, 3)]=(0.900876753496-0.421470898534j)*x[0]**o + ((0.481237273344-0.208379636466j))*x[0] + ((-0.8063383547+0.4950733157j))*x[1]**o + ((0.615354393252-0.623412311827j))*x[1]
            ref[(2, 3)]=(0.595565032696-0.379094765564j)-((0.0157563997994+0.0122670695276j))*(o-1.)
        else:
            arg[(0, 0)]=(0.666738292306+0.780867785726j)*x[0]**o + ((-0.959327469072-0.532900103397j))*x[0] + ((-0.675788314127+0.930040543459j))*x[1]**o + ((0.531801438025-0.328879193529j))*x[1] + ((0.0640781080899+0.658643702562j))*x[2]**o + ((0.853591470352-0.215582624952j))*x[2]
            ref[(0, 0)]=(0.240546762787+0.646095054935j)-((0.00917134771151+0.394925338625j))*(o-1.)
            arg[(0, 1)]=(-0.802665664536+0.0158877041101j)*x[0]**o + ((-0.369493869164-0.301656749414j))*x[0] + ((0.902203303888+0.569301981091j))*x[1]**o + ((0.0676412373254+0.227686386771j))*x[1] + ((0.133383268457+0.0574906205162j))*x[2]**o + ((-0.077413442243+0.607822759217j))*x[2]
            ref[(0, 1)]=(-0.0731725831364+0.588266351145j)-((0.0388201513015+0.107113384286j))*(o-1.)
            arg[(0, 2)]=(-0.319097468513-0.355830900856j)*x[0]**o + ((-0.715256287018-0.662824967574j))*x[0] + ((0.926003059333-0.112075197316j))*x[1]**o + ((0.0665293682207+0.353265282571j))*x[1] + ((-0.185545780964+0.744232328299j))*x[2]**o + ((0.452192730319+0.452551971618j))*x[2]
            ref[(0, 2)]=(0.112412810689+0.209659258371j)-((0.070226634976+0.0460543716878j))*(o-1.)
            arg[(0, 3)]=(-0.468322269997-0.608367945589j)*x[0]**o + ((0.37354145518+0.86170988734j))*x[0] + ((-0.874297057552+0.516939053395j))*x[1]**o + ((0.762902421577-0.114442772194j))*x[1] + ((0.151290945882+0.776229091456j))*x[2]**o + ((0.068113967494+0.805667678388j))*x[2]
            ref[(0, 3)]=(0.00661473129222+1.1188674964j)-((-0.198554730278+0.114133366544j))*(o-1.)
            arg[(1, 0)]=(0.20690287013+0.512003463683j)*x[0]**o + ((-0.924047779827-0.557835067902j))*x[0] + ((0.788878751119+0.337745067703j))*x[1]**o + ((-0.558374967658+0.442575582707j))*x[1] + ((-0.696188566227-0.256338073885j))*x[2]**o + ((-0.451602736416-0.681575872603j))*x[2]
            ref[(1, 0)]=(-0.81721621444-0.101712450148j)-((0.0499321758369+0.098901742917j))*(o-1.)
            arg[(1, 1)]=(0.0245154323764-0.579478424097j)*x[0]**o + ((-0.3257047812-0.966844718585j))*x[0] + ((0.74618931135+0.110461375284j))*x[1]**o + ((-0.833576373521-0.744883675482j))*x[1] + ((-0.451191861474-0.446152024516j))*x[2]**o + ((-0.0862069273123-0.384309965821j))*x[2]
            ref[(1, 1)]=(-0.46298759989-1.50560371661j)-((0.053252147042-0.152528178888j))*(o-1.)
            arg[(1, 2)]=(0.366519237981+0.515156890266j)*x[0]**o + ((0.457187243878-0.934174678116j))*x[0] + ((0.542610216972+0.290972537626j))*x[1]**o + ((-0.682955242987-0.23220294653j))*x[1] + ((-0.854286923012+0.514976515746j))*x[2]**o + ((0.835656948978+0.313862061884j))*x[2]
            ref[(1, 2)]=(0.332365740905+0.234295190438j)-((0.00914042199018+0.22018432394j))*(o-1.)
            arg[(1, 3)]=(-0.629602839366+0.398159073219j)*x[0]**o + ((0.815219299314-0.616204794277j))*x[0] + ((-0.36269835603-0.381851586976j))*x[1]**o + ((-0.893212795662+0.531667866042j))*x[1] + ((0.13872763051-0.965378336105j))*x[2]**o + ((0.718157398451+0.730476990636j))*x[2]
            ref[(1, 3)]=(-0.106704831392-0.15156539373j)-((-0.142262260814-0.158178474977j))*(o-1.)
            arg[(2, 0)]=(-0.922661658816+0.117514484547j)*x[0]**o + ((-0.874029907272-0.337957083136j))*x[0] + ((0.770539887788-0.89627824385j))*x[1]**o + ((0.415217829572-0.157304533074j))*x[1] + ((0.840387202417+0.402531116867j))*x[2]**o + ((0.813662378866-0.081524013704j))*x[2]
            ref[(2, 0)]=(0.521557866278-0.476509136175j)-((0.114710905232-0.0627054404061j))*(o-1.)
            arg[(2, 1)]=(0.925754397243-0.655975434744j)*x[0]**o + ((-0.683415551024+0.0338515460168j))*x[0] + ((0.376495380996-0.193732520831j))*x[1]**o + ((-0.367268886349+0.975131924299j))*x[1] + ((0.902056157039+0.503051361455j))*x[2]**o + ((0.181596085437-0.445408466931j))*x[2]
            ref[(2, 1)]=(0.667608791671+0.108459204632j)-((0.367384322546-0.05777609902j))*(o-1.)
            arg[(2, 2)]=(0.457981238672+0.360449072435j)*x[0]**o + ((-0.316516851442-0.784686006828j))*x[0] + ((0.395254907361-0.784136114282j))*x[1]**o + ((-0.461398651004-0.676809430698j))*x[1] + ((0.507534833692+0.715288918893j))*x[2]**o + ((-0.695108885028+0.146758274195j))*x[2]
            ref[(2, 2)]=(-0.0561267038739-0.511567643143j)-((0.226795163288+0.0486003128411j))*(o-1.)
            arg[(2, 3)]=(0.769669897375+0.815779095921j)*x[0]**o + ((-0.820060213632+0.223214262488j))*x[0] + ((0.801587974821-0.387859610582j))*x[1]**o + ((0.245205777211-0.373443050053j))*x[1] + ((0.932155933872+0.123320592678j))*x[2]**o + ((-0.0306645041569-0.999273034694j))*x[2]
            ref[(2, 3)]=(0.948947432745-0.299130872121j)-((0.417235634345+0.0918733463362j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ContinuousFunction_rank3(self):
        """
        tests integral of rank 3 Data on Function

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(2, 2, 4),w)
        ref=numpy.zeros((2, 2, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.166909434506+0.925785135722j)*x[0]**o + ((-0.597623590934-0.470056350135j))*x[0] + ((0.871652888429+0.251221496867j))*x[1]**o + ((-0.807338840357+0.401627922693j))*x[1]
            ref[(0, 0, 0)]=(-0.183200054178+0.554289102573j)-((0.173093720489+0.196167772098j))*(o-1.)
            arg[(0, 0, 1)]=(-0.0970674135793+0.636113552919j)*x[0]**o + ((-0.0687266562729-0.164041211553j))*x[0] + ((-0.117253793664-0.746113574177j))*x[1]**o + ((0.392221415905+0.912719582537j))*x[1]
            ref[(0, 0, 1)]=(0.0545867761948+0.319339174863j)-((-0.0357202012072-0.0183333368764j))*(o-1.)
            arg[(0, 0, 2)]=(0.11437286969+0.217649856064j)*x[0]**o + ((0.51564240068-0.585046698305j))*x[0] + ((-0.26484818662+0.560170034187j))*x[1]**o + ((-0.178868396336-0.936835171923j))*x[1]
            ref[(0, 0, 2)]=(0.0931493437072-0.372030989988j)-((-0.0250792194882+0.129636648375j))*(o-1.)
            arg[(0, 0, 3)]=(-0.639934479541-0.402701545689j)*x[0]**o + ((0.167692963894-0.980068853552j))*x[0] + ((0.294482402357+0.542039180315j))*x[1]**o + ((0.690524659711+0.0750788430612j))*x[1]
            ref[(0, 0, 3)]=(0.256382773211-0.382826187932j)-((-0.0575753461973+0.0232229391044j))*(o-1.)
            arg[(0, 1, 0)]=(-0.889696569903+0.227447871813j)*x[0]**o + ((-0.757945815892-0.903748892001j))*x[0] + ((-0.672288349523-0.462744133296j))*x[1]**o + ((0.70685776928+0.406329165028j))*x[1]
            ref[(0, 1, 0)]=(-0.806536483019-0.366357994228j)-((-0.260330819904-0.0392160435805j))*(o-1.)
            arg[(0, 1, 1)]=(-0.939044753801+0.769335973907j)*x[0]**o + ((-0.430254722736-0.213721728626j))*x[0] + ((0.558044303118-0.703708568646j))*x[1]**o + ((0.175835911042+0.0634565112314j))*x[1]
            ref[(0, 1, 1)]=(-0.317709631188-0.0423189060667j)-((-0.0635000751138+0.0109379008769j))*(o-1.)
            arg[(0, 1, 2)]=(-0.337129895172+0.183521941881j)*x[0]**o + ((-0.307536488365-0.220329173859j))*x[0] + ((-0.190775179223-0.998258033625j))*x[1]**o + ((0.696130424176+0.483176091557j))*x[1]
            ref[(0, 1, 2)]=(-0.0696555692919-0.275944587022j)-((-0.0879841790659-0.135789348624j))*(o-1.)
            arg[(0, 1, 3)]=(0.0840659302951-0.550221158177j)*x[0]**o + ((0.183836696844-0.851887420961j))*x[0] + ((0.506318947109-0.659897797944j))*x[1]**o + ((-0.557665703166-0.146202585733j))*x[1]
            ref[(0, 1, 3)]=(0.108277935541-1.10410448141j)-((0.0983974795674-0.201686492687j))*(o-1.)
            arg[(1, 0, 0)]=(0.429862504409-0.192792273337j)*x[0]**o + ((-0.633450749461+0.700576179104j))*x[0] + ((0.983482954493-0.970787830124j))*x[1]**o + ((0.110610137407-0.28810330652j))*x[1]
            ref[(1, 0, 0)]=(0.445252423423-0.375553615439j)-((0.235557576484-0.193930017244j))*(o-1.)
            arg[(1, 0, 1)]=(0.708415300347+0.788263235648j)*x[0]**o + ((0.952217892269+0.889556342596j))*x[0] + ((-0.633627392719+0.105565528008j))*x[1]**o + ((-0.385010067147-0.153272097404j))*x[1]
            ref[(1, 0, 1)]=(0.320997866375+0.815056504424j)-((0.0124646512713+0.148971460609j))*(o-1.)
            arg[(1, 0, 2)]=(-0.168625542615+0.442752633078j)*x[0]**o + ((0.413181922969-0.71962177116j))*x[0] + ((-0.128176817232+0.908071949291j))*x[1]**o + ((-0.199788895525+0.495161779398j))*x[1]
            ref[(1, 0, 2)]=(-0.0417046662012+0.563182295303j)-((-0.0494670599744+0.225137430395j))*(o-1.)
            arg[(1, 0, 3)]=(0.884170006002+0.584180669235j)*x[0]**o + ((-0.118264362109-0.322276238218j))*x[0] + ((0.715750349096-0.323638784493j))*x[1]**o + ((-0.891205777769-0.0901386649997j))*x[1]
            ref[(1, 0, 3)]=(0.29522510761-0.075936509238j)-((0.266653392516+0.043423647457j))*(o-1.)
            arg[(1, 1, 0)]=(0.590929830404-0.0501681757695j)*x[0]**o + ((0.867257846902+0.636507570149j))*x[0] + ((-0.603881952791+0.33342264414j))*x[1]**o + ((0.364836889363+0.843375388814j))*x[1]
            ref[(1, 1, 0)]=(0.609571306939+0.881568713667j)-((-0.00215868706458+0.0472090780617j))*(o-1.)
            arg[(1, 1, 1)]=(-0.829243091862+0.052119182975j)*x[0]**o + ((0.471431226068-0.198639785548j))*x[0] + ((0.915649699973+0.292448015535j))*x[1]**o + ((-0.559804526046+0.220613241936j))*x[1]
            ref[(1, 1, 1)]=(-0.00098334593359+0.183270327449j)-((0.0144011013518+0.0574278664183j))*(o-1.)
            arg[(1, 1, 2)]=(-0.586469687012-0.00261628776236j)*x[0]**o + ((-0.947636758973-0.168258939884j))*x[0] + ((-0.374038190172-0.182690509544j))*x[1]**o + ((-0.971491959719-0.103192610458j))*x[1]
            ref[(1, 1, 2)]=(-1.43981829794-0.228379173824j)-((-0.160084646197-0.0308844662177j))*(o-1.)
            arg[(1, 1, 3)]=(-0.375705579993-0.396677418003j)*x[0]**o + ((-0.488079034426-0.951919883371j))*x[0] + ((-0.754907939732+0.266658466649j))*x[1]**o + ((0.498401482531-0.20606421891j))*x[1]
            ref[(1, 1, 3)]=(-0.56014553581-0.644001526817j)-((-0.188435586621-0.0216698252257j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(-0.150905631734-0.392476763344j)*x[0]**o + ((0.727306685512-0.438702269101j))*x[0] + ((0.186343145046+0.890701829618j))*x[1]**o + ((-0.216659948549+0.662476890716j))*x[1] + ((0.591199870248-0.276281581418j))*x[2]**o + ((-0.52648366077+0.27815379809j))*x[2]
            ref[(0, 0, 0)]=(0.305400229877+0.36193595228j)-((0.104439563927+0.0369905808094j))*(o-1.)
            arg[(0, 0, 1)]=(0.0559490796644+0.637853710585j)*x[0]**o + ((0.518081356359-0.950228234123j))*x[0] + ((0.345378292416+0.0436905574396j))*x[1]**o + ((-0.821564165157-0.331053981547j))*x[1] + ((0.466674998147-0.514037208959j))*x[2]**o + ((-0.572838381381+0.197929890273j))*x[2]
            ref[(0, 0, 1)]=(-0.00415940997604-0.457922633166j)-((0.144667061704+0.0279178431776j))*(o-1.)
            arg[(0, 0, 2)]=(-0.92756624738+0.366864343136j)*x[0]**o + ((-0.575963912405+0.159868883307j))*x[0] + ((-0.613953057421-0.211675197119j))*x[1]**o + ((-0.163528487871-0.181912729985j))*x[1] + ((0.665628109284-0.110068323685j))*x[2]**o + ((0.0418503602361+0.323393858453j))*x[2]
            ref[(0, 0, 2)]=(-0.786766617778+0.173235417053j)-((-0.145981865919+0.00752013705537j))*(o-1.)
            arg[(0, 0, 3)]=(-0.321469762157-0.0468665727306j)*x[0]**o + ((-0.268470798128+0.0203819100483j))*x[0] + ((0.746852445705-0.323348156206j))*x[1]**o + ((0.366322529284-0.177686501169j))*x[1] + ((-0.565635391771-0.977326053333j))*x[2]**o + ((-0.00802408076291-0.693094937936j))*x[2]
            ref[(0, 0, 3)]=(-0.0252125289153-1.09897015566j)-((-0.0233754513706-0.224590130378j))*(o-1.)
            arg[(0, 1, 0)]=(0.0633292808741+0.328789067948j)*x[0]**o + ((-0.505800840391-0.0192027473436j))*x[0] + ((-0.288033768256-0.404923662375j))*x[1]**o + ((0.221492902673+0.723737612885j))*x[1] + ((-0.223682684701-0.413632041188j))*x[2]**o + ((-0.326609855204-0.783691868466j))*x[2]
            ref[(0, 1, 0)]=(-0.529652482502-0.28446181927j)-((-0.074731195347-0.0816277726025j))*(o-1.)
            arg[(0, 1, 1)]=(-0.805965758927-0.519244323549j)*x[0]**o + ((-0.249587332657+0.31222452396j))*x[0] + ((0.782146877532-0.450207855767j))*x[1]**o + ((-0.596136691798-0.336967213593j))*x[1] + ((0.410091304982-0.540879101068j))*x[2]**o + ((0.451106396544-0.220624600011j))*x[2]
            ref[(0, 1, 1)]=(-0.00417260216209-0.877849285014j)-((0.0643787372645-0.251721880064j))*(o-1.)
            arg[(0, 1, 2)]=(0.610702939748+0.617086702859j)*x[0]**o + ((0.109762759174+0.0243865086797j))*x[0] + ((-0.739452481865+0.46032991231j))*x[1]**o + ((-0.376385425902-0.626554172387j))*x[1] + ((0.610003828452+0.316814182807j))*x[2]**o + ((-0.776459642121+0.84512433996j))*x[2]
            ref[(0, 1, 2)]=(-0.280914011256+0.818593737115j)-((0.0802090477226+0.232371799663j))*(o-1.)
            arg[(0, 1, 3)]=(0.879845926879+0.959979589326j)*x[0]**o + ((-0.698764534459-0.492172172875j))*x[0] + ((0.323018046942-0.712420877286j))*x[1]**o + ((-0.548738714021-0.933191055809j))*x[1] + ((0.829989118256+0.970262098919j))*x[2]**o + ((0.961593409826+0.955689786374j))*x[2]
            ref[(0, 1, 3)]=(0.873471626712+0.374073684324j)-((0.33880884868+0.20297013516j))*(o-1.)
            arg[(1, 0, 0)]=(-0.267232791601-0.0175389988417j)*x[0]**o + ((0.507003663716-0.229042285293j))*x[0] + ((0.995757368304-0.168400098388j))*x[1]**o + ((0.803655324664-0.933554571055j))*x[1] + ((-0.59072945251+0.360951660596j))*x[2]**o + ((0.191141483167-0.0880340438747j))*x[2]
            ref[(1, 0, 0)]=(0.81979779787-0.537809168428j)-((0.0229658540321+0.0291687605611j))*(o-1.)
            arg[(1, 0, 1)]=(-0.624919049376-0.660870391554j)*x[0]**o + ((-0.094339546563+0.972791579811j))*x[0] + ((-0.428379775075+0.984435301908j))*x[1]**o + ((0.205078602185-0.620458725308j))*x[1] + ((0.900284239146-0.469067680924j))*x[2]**o + ((-0.158890255431-0.14722246742j))*x[2]
            ref[(1, 0, 1)]=(-0.100582892557+0.0298038082561j)-((-0.0255024308841-0.0242504617618j))*(o-1.)
            arg[(1, 0, 2)]=(0.215717726018+0.988054492118j)*x[0]**o + ((-0.50389493-0.509251693892j))*x[0] + ((0.627093344268-0.846030800522j))*x[1]**o + ((-0.0196327535627-0.824101397158j))*x[1] + ((0.429297775158+0.114185740034j))*x[2]**o + ((0.217372215548+0.107529806804j))*x[2]
            ref[(1, 0, 2)]=(0.482976688715-0.484806926308j)-((0.212018140907+0.0427015719383j))*(o-1.)
            arg[(1, 0, 3)]=(-0.715335825742+0.291449574743j)*x[0]**o + ((-0.525266623955+0.56082118289j))*x[0] + ((0.137760775727+0.642212152613j))*x[1]**o + ((-0.644683610008-0.265619431857j))*x[1] + ((0.808969728519-0.0426791165404j))*x[2]**o + ((0.018420904323+0.5696679332j))*x[2]
            ref[(1, 0, 3)]=(-0.460067325568+0.877926147524j)-((0.0385657797508+0.148497101803j))*(o-1.)
            arg[(1, 1, 0)]=(0.256923516665-0.237922126207j)*x[0]**o + ((0.51867318284+0.544746844328j))*x[0] + ((-0.0471023474139+0.727100551733j))*x[1]**o + ((-0.246390853121+0.850988446957j))*x[1] + ((-0.188783575874+0.950215986884j))*x[2]**o + ((0.332961173821+0.550657591313j))*x[2]
            ref[(1, 1, 0)]=(0.313140548458+1.6928936475j)-((0.00350626556287+0.239899068735j))*(o-1.)
            arg[(1, 1, 1)]=(0.498020068273+0.234584079565j)*x[0]**o + ((-0.986405270611-0.566892651507j))*x[0] + ((0.721426067967+0.362437841367j))*x[1]**o + ((0.816579586763-0.522163279761j))*x[1] + ((-0.751266904584-0.961880897338j))*x[2]**o + ((-0.609712819282-0.53740199565j))*x[2]
            ref[(1, 1, 1)]=(-0.155679635737-0.995658451663j)-((0.0780298719427-0.0608098294011j))*(o-1.)
            arg[(1, 1, 2)]=(0.358236675365-0.956354238911j)*x[0]**o + ((0.0490519874555+0.936508205097j))*x[0] + ((0.932255697936-0.365148423307j))*x[1]**o + ((-0.843658483442-0.0532667205846j))*x[1] + ((-0.0673661092049-0.467428734837j))*x[2]**o + ((0.136960822658+0.216434506831j))*x[2]
            ref[(1, 1, 2)]=(0.282740295384-0.344627702856j)-((0.203854377349-0.298155232843j))*(o-1.)
            arg[(1, 1, 3)]=(0.288931277723+0.0419618105017j)*x[0]**o + ((0.437180182333-0.275705454206j))*x[0] + ((-0.474986083376-0.0421348406395j))*x[1]**o + ((0.708011472643+0.197420438076j))*x[1] + ((0.385640562306-0.437385051919j))*x[2]**o + ((-0.911422979333+0.789434221386j))*x[2]
            ref[(1, 1, 3)]=(0.216677216147+0.1367955616j)-((0.0332642927753-0.0729263470095j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 2, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ContinuousFunction_rank4(self):
        """
        tests integral of rank 4 Data on Function

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(2, 2, 2, 2),w)
        ref=numpy.zeros((2, 2, 2, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(0.521207810014+0.548432192496j)*x[0]**o + ((-0.812679484454+0.965745447801j))*x[0] + ((-0.274864804152+0.453886327798j))*x[1]**o + ((-0.805642037741-0.0180958369196j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.685989258167+0.974984065588j)-((0.0410571676435+0.167053086716j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.41904150986-0.874662039812j)*x[0]**o + ((-0.808256355102+0.548340692775j))*x[0] + ((-0.0921378010231-0.956845072276j))*x[1]**o + ((-0.585560172877+0.753576698179j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.952497919431-0.264794860567j)-((-0.0851965518138-0.305251185348j))*(o-1.)
            arg[(0, 0, 1, 0)]=(-0.36397668414+0.283755889462j)*x[0]**o + ((-0.961003627264-0.77753514884j))*x[0] + ((-0.465596072127-0.0521530154715j))*x[1]**o + ((0.513061990501-0.52456411494j))*x[1]
            ref[(0, 0, 1, 0)]=(-0.638757196515-0.535248194895j)-((-0.138262126044+0.0386004789984j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.791355847131+0.0653424534083j)*x[0]**o + ((0.496347320892-0.220461092031j))*x[0] + ((0.363305642897-0.173928500683j))*x[1]**o + ((0.617118727268+0.81963794111j))*x[1]
            ref[(0, 0, 1, 1)]=(0.342707921963+0.245295400902j)-((-0.0713417007057-0.0180976745458j))*(o-1.)
            arg[(0, 1, 0, 0)]=(0.212454641979-0.82565058009j)*x[0]**o + ((0.336535798505+0.726600069726j))*x[0] + ((-0.409416153595+0.871559878814j))*x[1]**o + ((0.455903067329-0.0516298932837j))*x[1]
            ref[(0, 1, 0, 0)]=(0.297738677109+0.360439737584j)-((-0.0328269186027+0.00765154978747j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.916759786205+0.232936185562j)*x[0]**o + ((0.95049049086+0.876121334971j))*x[0] + ((-0.0218936447148+0.185041387975j))*x[1]**o + ((-0.326080011637+0.600490603595j))*x[1]
            ref[(0, 1, 0, 1)]=(0.759638310357+0.947294756052j)-((0.149144356915+0.069662928923j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.646138133628-0.381419051587j)*x[0]**o + ((-0.940751487514+0.816883058829j))*x[0] + ((-0.435482587294+0.29909968889j))*x[1]**o + ((0.463940192205+0.914108770804j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.779216008115+0.824336233468j)-((-0.180270120154-0.0137198937829j))*(o-1.)
            arg[(0, 1, 1, 1)]=(0.0580241583018-0.44016235511j)*x[0]**o + ((0.340123294235+0.0383723463114j))*x[0] + ((-0.346867368033-0.867140024324j))*x[1]**o + ((-0.878803562038-0.440175412223j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.413761738767-0.854552722673j)-((-0.0481405349552-0.217883729906j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.035101310696-0.702515939592j)*x[0]**o + ((-0.74853924762+0.424515880929j))*x[0] + ((-0.0650514819428+0.0559520958724j))*x[1]**o + ((-0.405827386306+0.739046344589j))*x[1]
            ref[(1, 0, 0, 0)]=(-0.627259713282+0.258499190899j)-((-0.0166921321065-0.10776064062j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.0210455572117-0.0431512531237j)*x[0]**o + ((0.185276266059-0.631612152975j))*x[0] + ((0.690239949678+0.537617775233j))*x[1]**o + ((-0.922709679503-0.481103852855j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.0341195104887-0.30912474186j)-((0.111532398744+0.0824110870182j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.884846657369+0.543460782841j)*x[0]**o + ((0.447906153714-0.374236813643j))*x[0] + ((-0.503250622362+0.07071371395j))*x[1]**o + ((-0.174616887237-0.3750955877j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.557404006627-0.0675789522762j)-((-0.231349546622+0.102362416132j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.339326552759-0.628389210167j)*x[0]**o + ((0.401922227656-0.333203455965j))*x[0] + ((-0.28264432359+0.205221615311j))*x[1]**o + ((-0.537854463598+0.226638794006j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.378951556145-0.264866128407j)-((-0.103661812725-0.070527932476j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.0762907547927-0.90861871949j)*x[0]**o + ((-0.484850468108+0.986203620813j))*x[0] + ((0.678440505096-0.53276378722j))*x[1]**o + ((-0.429580407279+0.817726031502j))*x[1]
            ref[(1, 1, 0, 0)]=(-0.0798498077493+0.181273572802j)-((0.125788543315-0.240230417785j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.182501823217-0.562059652509j)*x[0]**o + ((0.320860568377-0.359158832657j))*x[0] + ((0.452502607588+0.0577043195432j))*x[1]**o + ((0.322349414296+0.764791539139j))*x[1]
            ref[(1, 1, 0, 1)]=(0.456605383522-0.0493613132419j)-((0.0450001307285-0.0840592221609j))*(o-1.)
            arg[(1, 1, 1, 0)]=(-0.546931424148-0.93070153638j)*x[0]**o + ((0.0199973110756+0.534588523168j))*x[0] + ((0.725806761955+0.517616177422j))*x[1]**o + ((0.669354783595+0.651834901211j))*x[1]
            ref[(1, 1, 1, 0)]=(0.434113716238+0.386669032711j)-((0.029812556301-0.0688475598262j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.0952950255595-0.273827322243j)*x[0]**o + ((-0.206240469997-0.953355170257j))*x[0] + ((-0.252248262576-0.131671966818j))*x[1]**o + ((-0.871725076762-0.0857972756774j))*x[1]
            ref[(1, 1, 1, 1)]=(-0.617459391888-0.722325867498j)-((-0.0261588728361-0.0675832148435j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(0.113131223729+0.789404465973j)*x[0]**o + ((0.100054030846+0.0535631597137j))*x[0] + ((0.273384704208+0.233529165682j))*x[1]**o + ((0.77619167462+0.414665247979j))*x[1] + ((0.674551695511-0.941079619122j))*x[2]**o + ((0.740349032558+0.462496389683j))*x[2]
            ref[(0, 0, 0, 0)]=(1.33883118074+0.506289404955j)-((0.176844603908+0.0136423354223j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.0383664243783-0.604840015319j)*x[0]**o + ((0.196094089068+0.494523962516j))*x[0] + ((0.420969099506+0.483409626287j))*x[1]**o + ((0.387091544027-0.0728600156126j))*x[1] + ((-0.865649812955-0.239234062019j))*x[2]**o + ((-0.770028390334+0.888508155099j))*x[2]
            ref[(0, 0, 0, 1)]=(-0.334944947533+0.474753825476j)-((-0.0805078563045-0.0601107418418j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.751432717975+0.815067373745j)*x[0]**o + ((-0.535257960279-0.54849752806j))*x[0] + ((0.38878774727+0.860479342945j))*x[1]**o + ((-0.184927369917-0.247365123407j))*x[1] + ((0.45974954307-0.11402976649j))*x[2]**o + ((-0.341464712731+0.424565168672j))*x[2]
            ref[(0, 0, 1, 0)]=(0.269159982694+0.595109733703j)-((0.266661668052+0.260252825033j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.637128768028-0.775918029285j)*x[0]**o + ((0.142745115972-0.171879622367j))*x[0] + ((0.335811399915+0.985048156938j))*x[1]**o + ((-0.585945896837+0.448012389951j))*x[1] + ((-0.416648855826-0.0087104105678j))*x[2]**o + ((-0.973487684963+0.390826880314j))*x[2]
            ref[(0, 0, 1, 1)]=(-1.06732734488+0.433689682491j)-((-0.119661037323+0.0334032861808j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.0498293855023-0.160482082452j)*x[0]**o + ((-0.480403052111-0.498155957595j))*x[0] + ((-0.0321197410949-0.316169801703j))*x[1]**o + ((0.429297157549+0.0767436721137j))*x[1] + ((0.0071020463985-0.690937681164j))*x[2]**o + ((0.673971847833+0.562211189873j))*x[2]
            ref[(0, 1, 0, 0)]=(0.274009436537-0.513395330464j)-((-0.0124745133664-0.194598260887j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.736361435105-0.814450650782j)*x[0]**o + ((0.685788623247-0.0473819916166j))*x[0] + ((-0.00873095076582+0.503553636043j))*x[1]**o + ((0.625064392689+0.807387083982j))*x[1] + ((0.40071307681+0.104022610711j))*x[2]**o + ((0.120717086667+0.179256590313j))*x[2]
            ref[(0, 1, 0, 1)]=(1.27995683188+0.366193639325j)-((0.188057260192-0.034479067338j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.0426982720606-0.0258561622552j)*x[0]**o + ((0.519390630353-0.716586075763j))*x[0] + ((0.0094837699013+0.809098333331j))*x[1]**o + ((-0.100285011038-0.266186673033j))*x[1] + ((-0.721937265263-0.556996796371j))*x[2]**o + ((-0.476295916811-0.305808441963j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.406171032459-0.531167908027j)-((-0.125858627904+0.0377075624508j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.892153947595-0.398917091509j)*x[0]**o + ((0.301881796322+0.693431365505j))*x[0] + ((-0.226104147589+0.216153723318j))*x[1]**o + ((-0.399393822828-0.377993250466j))*x[1] + ((-0.0535174726265-0.279559320029j))*x[2]**o + ((-0.341865265099+0.792925303945j))*x[2]
            ref[(0, 1, 1, 1)]=(-0.805576429708+0.323020365382j)-((-0.195295927968-0.0770537813701j))*(o-1.)
            arg[(1, 0, 0, 0)]=(0.63396848459+0.254896024923j)*x[0]**o + ((-0.00371769262638+0.635263815076j))*x[0] + ((0.680543278843+0.328912223005j))*x[1]**o + ((0.18338493529+0.826332133847j))*x[1] + ((0.882103232464+0.966139272246j))*x[2]**o + ((-0.85207731476+0.826348519409j))*x[2]
            ref[(1, 0, 0, 0)]=(0.7621024619+1.91894599425j)-((0.366102499316+0.258324586696j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.0476211284453+0.821886946427j)*x[0]**o + ((0.440372159956+0.0393492750241j))*x[0] + ((0.501288956288-0.635922857656j))*x[1]**o + ((-0.755444347419-0.0897333446675j))*x[1] + ((0.920644728531-0.511671688959j))*x[2]**o + ((0.469518606022+0.731291608817j))*x[2]
            ref[(1, 0, 0, 1)]=(0.764379487466+0.177599969492j)-((0.229052092729-0.0542846000315j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.604483345865-0.47804934175j)*x[0]**o + ((-0.318351763826-0.0797977410942j))*x[0] + ((-0.983824800102+0.713216995168j))*x[1]**o + ((0.895284444619-0.734889648324j))*x[1] + ((-0.584224719111+0.553227205537j))*x[2]**o + ((0.787387951852+0.984675783385j))*x[2]
            ref[(1, 0, 1, 0)]=(-0.404106116217+0.479191626461j)-((-0.362088810847+0.131399143159j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.908261422625-0.580479321562j)*x[0]**o + ((0.756837602705-0.739813858422j))*x[0] + ((0.284419655492-0.0850424485592j))*x[1]**o + ((0.305590501255+0.363988156133j))*x[1] + ((0.516135899173-0.229539904972j))*x[2]**o + ((0.717548296988-0.404686325962j))*x[2]
            ref[(1, 0, 1, 1)]=(0.836135266494-0.837786851672j)-((-0.0179509779934-0.149176945849j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.395268209597+0.0514602130362j)*x[0]**o + ((0.881924323666+0.758755386156j))*x[0] + ((0.822232149855+0.211182216625j))*x[1]**o + ((0.705458296387-0.736188517632j))*x[1] + ((0.129280904562+0.734459220673j))*x[2]**o + ((0.842175167531+0.209785230879j))*x[2]
            ref[(1, 1, 0, 0)]=(1.8881695258+0.614726874869j)-((0.224463544002+0.166183608389j))*(o-1.)
            arg[(1, 1, 0, 1)]=(0.688640582844-0.597388155115j)*x[0]**o + ((0.306928442735+0.287061950469j))*x[0] + ((-0.642338343823-0.0535606331208j))*x[1]**o + ((-0.880288058088+0.29188247144j))*x[1] + ((0.0936429254516+0.345508482621j))*x[2]**o + ((0.171183701617-0.286192129734j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.131115374632-0.00634400672024j)-((0.0233241940788-0.0509067176025j))*(o-1.)
            arg[(1, 1, 1, 0)]=(-0.377369223472-0.584720542605j)*x[0]**o + ((0.747106343052-0.986512845325j))*x[0] + ((-0.233867802797+0.467450682177j))*x[1]**o + ((-0.408865703615-0.394143192345j))*x[1] + ((0.764035254517+0.651017962859j))*x[2]**o + ((-0.617278037785+0.58997100791j))*x[2]
            ref[(1, 1, 1, 0)]=(-0.0631195850501-0.128468463665j)-((0.0254663713746+0.0889580170718j))*(o-1.)
            arg[(1, 1, 1, 1)]=(-0.356690908944-0.261660609718j)*x[0]**o + ((-0.48704864682-0.128606605157j))*x[0] + ((-0.899809846312-0.81929853462j))*x[1]**o + ((0.540682631377+0.783956702042j))*x[1] + ((-0.539375054722+0.727884217312j))*x[2]**o + ((-0.70089656385+0.182019045275j))*x[2]
            ref[(1, 1, 1, 1)]=(-1.22156919464+0.242147107567j)-((-0.299312634996-0.058845821171j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 2, 2, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Solution_rank0(self):
        """
        tests integral of rank 0 Data on Function

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(-0.0261557332468+0.280505197109j)*x[0]**o + ((-0.789992797974-0.207018136209j))*x[0] + ((-0.637179673524+0.801741401991j))*x[1]**o + ((-0.623594762326-0.277619471159j))*x[1]
            ref=(-1.03846148354+0.298804495866j)-((-0.110555901128+0.180374433183j))*(o-1.)
        else:
            arg=(-0.70244433864-0.98573035506j)*x[0]**o + ((0.872480666713+0.374286777181j))*x[0] + ((0.558465891402+0.989177407849j))*x[1]**o + ((-0.327906166898+0.286280869036j))*x[1] + ((0.977378453376-0.882519297935j))*x[2]**o + ((-0.65393265252-0.0420261820405j))*x[2]
            ref=(0.362020926716-0.130265390485j)-((0.138900001023-0.146512040858j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Solution_rank1(self):
        """
        tests integral of rank 1 Data on Function

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(2,),w)
        ref=numpy.zeros((2,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.137873057894-0.107806061318j)*x[0]**o + ((0.956514400708+0.499466275781j))*x[0] + ((0.985002473987+0.169106820889j))*x[1]**o + ((-0.828072298953-0.551145884941j))*x[1]
            ref[(0,)]=(0.487785758924+0.00481057520611j)-((0.141188236015+0.0102167932619j))*(o-1.)
            arg[(1,)]=(0.0468112592042-0.743043812152j)*x[0]**o + ((0.215282374535-0.986744748888j))*x[0] + ((-0.987932969474-0.176578181101j))*x[1]**o + ((0.0679529409691+0.723714461205j))*x[1]
            ref[(1,)]=(-0.328943197383-0.591326140468j)-((-0.156853618378-0.153270332209j))*(o-1.)
        else:
            arg[(0,)]=(0.403996141026+0.67065907708j)*x[0]**o + ((-0.492286700277+0.366145602098j))*x[0] + ((-0.364089086463+0.361123900658j))*x[1]**o + ((0.400422292772-0.936483995148j))*x[1] + ((-0.578043679817+0.878030628429j))*x[2]**o + ((-0.114259559566+0.805196605844j))*x[2]
            ref[(0,)]=(-0.372130296163+1.07233590948j)-((-0.0896894375422+0.318302267695j))*(o-1.)
            arg[(1,)]=(-0.392930937771-0.0582218251334j)*x[0]**o + ((0.491129388094-0.177777484656j))*x[0] + ((0.501781713951+0.0861365705241j))*x[1]**o + ((-0.500308680385+0.715475250327j))*x[1] + ((0.456426930839-0.476472002458j))*x[2]**o + ((0.537001586258+0.902134944074j))*x[2]
            ref[(1,)]=(0.546550000493+0.495637726339j)-((0.0942129511698-0.0747595428445j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Solution_rank2(self):
        """
        tests integral of rank 2 Data on Function

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(4, 3),w)
        ref=numpy.zeros((4, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(-0.693572936367+0.847224117585j)*x[0]**o + ((0.834188696189-0.243741175021j))*x[0] + ((0.512415191229+0.79023056362j))*x[1]**o + ((0.372542038427-0.671748686782j))*x[1]
            ref[(0, 0)]=(0.512786494739+0.360982409701j)-((-0.0301929575229+0.272909113534j))*(o-1.)
            arg[(0, 1)]=(0.77158396755+0.931351690936j)*x[0]**o + ((0.327837010528+0.128432396176j))*x[0] + ((-0.55486837528-0.290848130536j))*x[1]**o + ((-0.105974346826+0.18825432029j))*x[1]
            ref[(0, 1)]=(0.219289127986+0.478595138433j)-((0.0361192653784+0.1067505934j))*(o-1.)
            arg[(0, 2)]=(0.769920779702+0.959687349212j)*x[0]**o + ((-0.628776933643+0.900243610789j))*x[0] + ((0.0391034106076+0.301071832234j))*x[1]**o + ((-0.746213220055+0.66805223376j))*x[1]
            ref[(0, 2)]=(-0.282982981694+1.414527513j)-((0.134837365052+0.210126530241j))*(o-1.)
            arg[(1, 0)]=(0.368108459235+0.114505926631j)*x[0]**o + ((-0.701671813115+0.971249891431j))*x[0] + ((-0.423350021186+0.189526042297j))*x[1]**o + ((-0.465395942014-0.72918091074j))*x[1]
            ref[(1, 0)]=(-0.61115465854+0.27305047481j)-((-0.00920692699179+0.0506719948214j))*(o-1.)
            arg[(1, 1)]=(0.0164921155518+0.0573140971946j)*x[0]**o + ((0.72256360356+0.695943914395j))*x[0] + ((0.0939833389334-0.774027606268j))*x[1]**o + ((-0.989955935487+0.409718930262j))*x[1]
            ref[(1, 1)]=(-0.078458438721+0.194474667792j)-((0.0184125757475-0.119452251512j))*(o-1.)
            arg[(1, 2)]=(0.100359918534-0.40036537656j)*x[0]**o + ((-0.300485183294-0.445124378225j))*x[0] + ((-0.728214534711-0.767792297462j))*x[1]**o + ((-0.338352910545+0.58669093673j))*x[1]
            ref[(1, 2)]=(-0.633346355008-0.513295557759j)-((-0.104642436029-0.19469294567j))*(o-1.)
            arg[(2, 0)]=(-0.746233479348+0.0248305834145j)*x[0]**o + ((-0.293934979361+0.352849082605j))*x[0] + ((0.633648456628-0.514695026916j))*x[1]**o + ((-0.266266199045+0.588120814471j))*x[1]
            ref[(2, 0)]=(-0.336393100563+0.225552726787j)-((-0.0187641704534-0.081644073917j))*(o-1.)
            arg[(2, 1)]=(-0.194726054228-0.262870573625j)*x[0]**o + ((-0.996591193237-0.801548407508j))*x[0] + ((0.628612570344+0.787828778042j))*x[1]**o + ((-0.4994932221-0.440149453909j))*x[1]
            ref[(2, 1)]=(-0.531098949611-0.3583698285j)-((0.0723144193526+0.0874930340694j))*(o-1.)
            arg[(2, 2)]=(0.585659942291-0.939646842813j)*x[0]**o + ((0.249905756637-0.804578276508j))*x[0] + ((0.615085128408+0.726721860956j))*x[1]**o + ((0.750333853277+0.337271982704j))*x[1]
            ref[(2, 2)]=(1.10049234031-0.34011563783j)-((0.20012417845-0.0354874969762j))*(o-1.)
            arg[(3, 0)]=(-0.22514010717-0.43137850649j)*x[0]**o + ((-0.545094326839-0.836815000164j))*x[0] + ((0.201959858222-0.702002435076j))*x[1]**o + ((0.677795548549-0.466445761181j))*x[1]
            ref[(3, 0)]=(0.0547604863808-1.21832085146j)-((-0.00386337482469-0.188896823594j))*(o-1.)
            arg[(3, 1)]=(-0.881742139756+0.898323732285j)*x[0]**o + ((-0.279553409724+0.354289621602j))*x[0] + ((0.403543335322-0.384864837811j))*x[1]**o + ((-0.456986889961-0.878894789413j))*x[1]
            ref[(3, 1)]=(-0.60736955206-0.00557313666809j)-((-0.0796998007391+0.0855764824125j))*(o-1.)
            arg[(3, 2)]=(-0.149663147782+0.902321482347j)*x[0]**o + ((0.640218576348+0.0859609165352j))*x[0] + ((0.710561071168-0.984528199954j))*x[1]**o + ((0.395231497262+0.281087951202j))*x[1]
            ref[(3, 2)]=(0.798173998498+0.142421075065j)-((0.093482987231-0.0137011196012j))*(o-1.)
        else:
            arg[(0, 0)]=(0.583200455009-0.474891533225j)*x[0]**o + ((0.588907311843+0.518830908961j))*x[0] + ((-0.652808369915-0.168883951831j))*x[1]**o + ((-0.303116800568+0.418501546097j))*x[1] + ((0.44907114758-0.604214549364j))*x[2]**o + ((-0.122342303267+0.138497570185j))*x[2]
            ref[(0, 0)]=(0.271455720341-0.0860800045886j)-((0.0632438721124-0.20799833907j))*(o-1.)
            arg[(0, 1)]=(-0.596735161309+0.137538269943j)*x[0]**o + ((-0.64660296199+0.365910812786j))*x[0] + ((-0.387819583445-0.856219778817j))*x[1]**o + ((-0.258442760627+0.218869518662j))*x[1] + ((0.582080188388-0.442445814383j))*x[2]**o + ((-0.846614752681+0.0182392256408j))*x[2]
            ref[(0, 1)]=(-1.07706751583-0.279053883084j)-((-0.0670790927277-0.193521220543j))*(o-1.)
            arg[(0, 2)]=(0.684236068141-0.127236098096j)*x[0]**o + ((-0.461135925236-0.725996228036j))*x[0] + ((-0.968381599598+0.733162594223j))*x[1]**o + ((-0.998430639087-0.0601795450454j))*x[1] + ((0.0816325214635-0.816688870222j))*x[2]**o + ((0.821351241394+0.643547894581j))*x[2]
            ref[(0, 2)]=(-0.420364166462-0.176695126298j)-((-0.0337521683324-0.0351270623491j))*(o-1.)
            arg[(1, 0)]=(0.504351695489+0.367051953533j)*x[0]**o + ((0.0948145852456+0.682227038206j))*x[0] + ((0.12555686833+0.920417466829j))*x[1]**o + ((0.244388352336+0.585372553436j))*x[1] + ((-0.70111806824+0.280839209651j))*x[2]**o + ((0.251738204818-0.119762728342j))*x[2]
            ref[(1, 0)]=(0.259865818989+1.35807274666j)-((-0.0118682507368+0.261384771669j))*(o-1.)
            arg[(1, 1)]=(-0.262787566726-0.336031893535j)*x[0]**o + ((0.990973094298-0.322266471086j))*x[0] + ((-0.0317643859213-0.913169601349j))*x[1]**o + ((-0.591993775975+0.947865686393j))*x[1] + ((0.74699027617+0.364827406636j))*x[2]**o + ((0.890212497995+0.748860123197j))*x[2]
            ref[(1, 1)]=(0.87081506992+0.245042625128j)-((0.0754063872537-0.147395681375j))*(o-1.)
            arg[(1, 2)]=(-0.566150653709+0.233877322765j)*x[0]**o + ((0.7800844228-0.700999171646j))*x[0] + ((0.539659511406+0.545859908707j))*x[1]**o + ((0.552419825395+0.0977501186572j))*x[1] + ((-0.294455445318-0.437750608604j))*x[2]**o + ((0.623929216071+0.0771675209882j))*x[2]
            ref[(1, 2)]=(0.817743438322-0.092047454566j)-((-0.053491097937+0.056997770478j))*(o-1.)
            arg[(2, 0)]=(-0.0215209726083+0.103947487248j)*x[0]**o + ((0.896304275264-0.214384512844j))*x[0] + ((0.0381129522186+0.102853605152j))*x[1]**o + ((-0.831446812666-0.711390346273j))*x[1] + ((-0.721677894194+0.876924158379j))*x[2]**o + ((0.611491709198+0.0836391312246j))*x[2]
            ref[(2, 0)]=(-0.0143683713933+0.120794761443j)-((-0.117514319097+0.18062087513j))*(o-1.)
            arg[(2, 1)]=(-0.191228976914+0.377723078036j)*x[0]**o + ((0.71155137908+0.729106489896j))*x[0] + ((-0.859255371231-0.370403015902j))*x[1]**o + ((-0.770397341606-0.899101071533j))*x[1] + ((0.0732240145726-0.660832634757j))*x[2]**o + ((-0.911222747707-0.0568086807326j))*x[2]
            ref[(2, 1)]=(-0.973664521903-0.440157917496j)-((-0.162876722262-0.108918762104j))*(o-1.)
            arg[(2, 2)]=(0.758373210321+0.577431204737j)*x[0]**o + ((-0.333185636406-0.563342933066j))*x[0] + ((0.517824607574-0.342030678048j))*x[1]**o + ((-0.667024507471+0.394944449291j))*x[1] + ((0.899711526783+0.244212896607j))*x[2]**o + ((-0.0833345220477+0.526737164036j))*x[2]
            ref[(2, 2)]=(0.546182339377+0.418976051779j)-((0.362651557446+0.0799355705494j))*(o-1.)
            arg[(3, 0)]=(0.898908190848-0.27532795242j)*x[0]**o + ((0.410933213073+0.583485687257j))*x[0] + ((-0.905011401946+0.512180172095j))*x[1]**o + ((-0.898105684992-0.764064304245j))*x[1] + ((-0.691272455422-0.236064126335j))*x[2]**o + ((0.163903576127-0.589049701016j))*x[2]
            ref[(3, 0)]=(-0.510322281156-0.384420112333j)-((-0.116229277753+0.00013134888984j))*(o-1.)
            arg[(3, 1)]=(-0.709416797102-0.798793076845j)*x[0]**o + ((0.577480971777+0.129729099429j))*x[0] + ((0.477740641005-0.809150314708j))*x[1]**o + ((0.15768429815+0.730802908188j))*x[1] + ((-0.390070413512+0.00155138275333j))*x[2]**o + ((-0.755379952765+0.599917512142j))*x[2]
            ref[(3, 1)]=(-0.320980626223-0.0729712445196j)-((-0.103624428268-0.267732001467j))*(o-1.)
            arg[(3, 2)]=(0.715695974747-0.503416417564j)*x[0]**o + ((0.7019572546+0.196673918168j))*x[0] + ((-0.367293782921+0.173477218484j))*x[1]**o + ((0.676410038633+0.837067349985j))*x[1] + ((-0.705221726479+0.150566498876j))*x[2]**o + ((-0.643316938341+0.546169524318j))*x[2]
            ref[(3, 2)]=(0.18911541012+0.700269046134j)-((-0.0594699224421-0.0298954500339j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Solution_rank3(self):
        """
        tests integral of rank 3 Data on Function

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(3, 3, 2),w)
        ref=numpy.zeros((3, 3, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.585224848306-0.667330210971j)*x[0]**o + ((0.51115390976+0.862366866813j))*x[0] + ((0.371150386078+0.0731109554518j))*x[1]**o + ((0.662038345193-0.302976384835j))*x[1]
            ref[(0, 0, 0)]=(1.06478374467-0.0174143867703j)-((0.159395872397-0.0990365425865j))*(o-1.)
            arg[(0, 0, 1)]=(0.373549281101-0.38208125613j)*x[0]**o + ((0.466381993022-0.863575447173j))*x[0] + ((0.020660564457-0.257542933994j))*x[1]**o + ((-0.509560682372+0.926808036836j))*x[1]
            ref[(0, 0, 1)]=(0.175515578104-0.288195800231j)-((0.0657016409263-0.106604031687j))*(o-1.)
            arg[(0, 1, 0)]=(0.664605045758-0.0320664297715j)*x[0]**o + ((-0.988521708585-0.997931459117j))*x[0] + ((-0.219251094358-0.580228988351j))*x[1]**o + ((-0.814970138415-0.698126308841j))*x[1]
            ref[(0, 1, 0)]=(-0.6790689478-1.15417659304j)-((0.0742256585668-0.102049236354j))*(o-1.)
            arg[(0, 1, 1)]=(-0.866340538656-0.396822935142j)*x[0]**o + ((-0.526824564486+0.922344717242j))*x[0] + ((0.243219657016-0.436942089417j))*x[1]**o + ((-0.877638612686-0.849061658959j))*x[1]
            ref[(0, 1, 1)]=(-1.01379202941-0.380240983138j)-((-0.103853480273-0.138960837426j))*(o-1.)
            arg[(0, 2, 0)]=(-0.403305471706+0.334838264734j)*x[0]**o + ((0.681027179716+0.281597173882j))*x[0] + ((-0.147787353652+0.0593962812552j))*x[1]**o + ((-0.958120515476-0.331587692989j))*x[1]
            ref[(0, 2, 0)]=(-0.414093080559+0.172122013441j)-((-0.0918488042262+0.0657057576649j))*(o-1.)
            arg[(0, 2, 1)]=(-0.387981457652+0.385538802047j)*x[0]**o + ((-0.919412909583-0.870360987012j))*x[0] + ((-0.975725437169-0.14763819377j))*x[1]**o + ((-0.125587649357-0.0364786508171j))*x[1]
            ref[(0, 2, 1)]=(-1.20435372688-0.334469514776j)-((-0.22728448247+0.0396501013796j))*(o-1.)
            arg[(1, 0, 0)]=(-0.391427407422-0.103745654179j)*x[0]**o + ((0.149933137365-0.155592365377j))*x[0] + ((-0.182180687305+0.838009711324j))*x[1]**o + ((0.507044761632-0.112020938137j))*x[1]
            ref[(1, 0, 0)]=(0.0416849021351+0.233325376816j)-((-0.0956013491211+0.122377342858j))*(o-1.)
            arg[(1, 0, 1)]=(-0.274854455211+0.63954322369j)*x[0]**o + ((-0.411578453723-0.0904417936381j))*x[0] + ((-0.616952542922-0.0496166106896j))*x[1]**o + ((0.506359254178+0.238635058878j))*x[1]
            ref[(1, 0, 1)]=(-0.398513098839+0.36905993912j)-((-0.148634499689+0.0983211021668j))*(o-1.)
            arg[(1, 1, 0)]=(0.361252653904-0.115945880074j)*x[0]**o + ((-0.211433352208-0.620665839387j))*x[0] + ((0.919146600097-0.555458106971j))*x[1]**o + ((0.608446344378-0.270410744121j))*x[1]
            ref[(1, 1, 0)]=(0.838706123085-0.781240285276j)-((0.213399875667-0.111900664507j))*(o-1.)
            arg[(1, 1, 1)]=(-0.140324313314+0.761259521707j)*x[0]**o + ((0.0717327049188+0.794806470647j))*x[0] + ((-0.421347823575+0.345104042062j))*x[1]**o + ((0.885709942213-0.445880072867j))*x[1]
            ref[(1, 1, 1)]=(0.197885255121+0.727644980774j)-((-0.0936120228149+0.184393927295j))*(o-1.)
            arg[(1, 2, 0)]=(-0.273393216274+0.964376569678j)*x[0]**o + ((0.36333545434+0.307467071705j))*x[0] + ((-0.0683079416419-0.466958655384j))*x[1]**o + ((-0.478812909062+0.546314591924j))*x[1]
            ref[(1, 2, 0)]=(-0.22858930632+0.675599788961j)-((-0.056950192986+0.0829029857156j))*(o-1.)
            arg[(1, 2, 1)]=(-0.628098683318-0.389746625548j)*x[0]**o + ((0.459422159531-0.468312530408j))*x[0] + ((-0.631375427567+0.161099150555j))*x[1]**o + ((0.92498450217-0.164458263195j))*x[1]
            ref[(1, 2, 1)]=(0.0624662754081-0.430709134298j)-((-0.209912351814-0.0381079124988j))*(o-1.)
            arg[(2, 0, 0)]=(-0.725036289635-0.139725714309j)*x[0]**o + ((-0.629355429532+0.71023156594j))*x[0] + ((0.165056129953-0.292371629359j))*x[1]**o + ((0.764275189766-0.232226578362j))*x[1]
            ref[(2, 0, 0)]=(-0.212530199723+0.0229538219551j)-((-0.0933300266136-0.0720162239447j))*(o-1.)
            arg[(2, 0, 1)]=(0.990715639228+0.758519700048j)*x[0]**o + ((-0.0717360384462-0.802035616141j))*x[0] + ((0.915612946158+0.0134917013953j))*x[1]**o + ((-0.888608552252-0.0535026324393j))*x[1]
            ref[(2, 0, 1)]=(0.472991997344-0.0417634235683j)-((0.317721430898+0.128668566907j))*(o-1.)
            arg[(2, 1, 0)]=(-0.790142397391-0.404085977099j)*x[0]**o + ((-0.601771011298+0.125711437905j))*x[0] + ((-0.796084924965-0.799261276415j))*x[1]**o + ((-0.904860028741+0.0325019622849j))*x[1]
            ref[(2, 1, 0)]=(-1.5464291812-0.522566926662j)-((-0.264371220393-0.200557875586j))*(o-1.)
            arg[(2, 1, 1)]=(-0.880421106164-0.400367401196j)*x[0]**o + ((-0.0974126183086-0.972839183657j))*x[0] + ((0.764636930283+0.547611100355j))*x[1]**o + ((0.758285520425-0.0858310562807j))*x[1]
            ref[(2, 1, 1)]=(0.272544363118-0.455713270389j)-((-0.0192973626469+0.0245406165265j))*(o-1.)
            arg[(2, 2, 0)]=(-0.801119730831+0.742982186057j)*x[0]**o + ((-0.0305389534877+0.113795549548j))*x[0] + ((-0.581837873507+0.00703441411912j))*x[1]**o + ((0.114555584987-0.979264830457j))*x[1]
            ref[(2, 2, 0)]=(-0.64947048642-0.0577263403671j)-((-0.230492934056+0.125002766696j))*(o-1.)
            arg[(2, 2, 1)]=(-0.930487070129+0.500848964668j)*x[0]**o + ((0.822001163163+0.120982840355j))*x[0] + ((0.0468214027036-0.0799465523395j))*x[1]**o + ((-0.9770585525+0.657817760786j))*x[1]
            ref[(2, 2, 1)]=(-0.519361528381+0.599851506734j)-((-0.147277611238+0.0701504020547j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(0.553038990551+0.577291758411j)*x[0]**o + ((0.348229438592-0.0330777438953j))*x[0] + ((-0.095545306389-0.307687368028j))*x[1]**o + ((-0.354910579278+0.196283913192j))*x[1] + ((0.842240249012-0.926249699304j))*x[2]**o + ((-0.104158939704+0.0495438589367j))*x[2]
            ref[(0, 0, 0)]=(0.594446926392-0.221947640343j)-((0.216622322196-0.10944088482j))*(o-1.)
            arg[(0, 0, 1)]=(0.718166918751-0.278421520658j)*x[0]**o + ((0.566010934192+0.229564914479j))*x[0] + ((-0.622206551522+0.363919516125j))*x[1]**o + ((-0.465085898617-0.43942157782j))*x[1] + ((0.720920510631+0.802104925702j))*x[2]**o + ((-0.174448587628+0.0977612147083j))*x[2]
            ref[(0, 0, 1)]=(0.371678662903+0.387753736268j)-((0.136146812977+0.147933820195j))*(o-1.)
            arg[(0, 1, 0)]=(-0.105590826953-0.244523253327j)*x[0]**o + ((0.384918445007-0.108650908147j))*x[0] + ((-0.681382133732-0.884214626431j))*x[1]**o + ((0.98474788829+0.710733195324j))*x[1] + ((-0.24324135878+0.930669335835j))*x[2]**o + ((0.321323780923-0.237337859741j))*x[2]
            ref[(0, 1, 0)]=(0.330387897378+0.0833379417566j)-((-0.171702386578-0.0330114239872j))*(o-1.)
            arg[(0, 1, 1)]=(0.197919541918-0.775310090082j)*x[0]**o + ((-0.232499599017+0.749852361239j))*x[0] + ((-0.710871718308+0.402841906085j))*x[1]**o + ((-0.232459248152-0.534544236746j))*x[1] + ((0.735395211436-0.70178332878j))*x[2]**o + ((0.579510022788-0.963917184413j))*x[2]
            ref[(0, 1, 1)]=(0.168497105332-0.911430286348j)-((0.0370738391743-0.179041918796j))*(o-1.)
            arg[(0, 2, 0)]=(0.956565330308+0.171424442861j)*x[0]**o + ((-0.0705419737494-0.218533654773j))*x[0] + ((0.466422428477-0.284934967714j))*x[1]**o + ((-0.59070472067+0.517461354473j))*x[1] + ((0.0129355266365+0.394225020565j))*x[2]**o + ((0.702133614045+0.374797596079j))*x[2]
            ref[(0, 2, 0)]=(0.738405102523+0.477219895745j)-((0.23932054757+0.0467857492854j))*(o-1.)
            arg[(0, 2, 1)]=(-0.32068311571-0.0807430630762j)*x[0]**o + ((-0.314908146591+0.631678639301j))*x[0] + ((0.414079671551-0.0107410000432j))*x[1]**o + ((-0.364741478936-0.0663703453433j))*x[1] + ((-0.0680682888289-0.444746279279j))*x[2]**o + ((-0.533272856719-0.373312331843j))*x[2]
            ref[(0, 2, 1)]=(-0.593797107617-0.172117190142j)-((0.00422137783537-0.0893717237331j))*(o-1.)
            arg[(1, 0, 0)]=(0.915849059478-0.283570476678j)*x[0]**o + ((0.437287264744-0.0565933902319j))*x[0] + ((0.913315045838-0.547063810276j))*x[1]**o + ((0.193953112157+0.365545901758j))*x[1] + ((0.4188147111+0.60350951988j))*x[2]**o + ((-0.316478355124+0.0250588301359j))*x[2]
            ref[(1, 0, 0)]=(1.2813704191+0.0534432872939j)-((0.374663136069-0.0378541278457j))*(o-1.)
            arg[(1, 0, 1)]=(-0.156093327824-0.666703434327j)*x[0]**o + ((0.587418185936-0.635438772971j))*x[0] + ((0.309566107702-0.648727980964j))*x[1]**o + ((-0.495341526413+0.465328966473j))*x[1] + ((-0.728061122414+0.776640381041j))*x[2]**o + ((0.58390728346-0.447987385019j))*x[2]
            ref[(1, 0, 1)]=(0.0506978002237-0.578444112883j)-((-0.095764723756-0.0897985057083j))*(o-1.)
            arg[(1, 1, 0)]=(-0.322107412099+0.71765119577j)*x[0]**o + ((-0.424722300844+0.626084148738j))*x[0] + ((-0.0725311843047+0.580617298944j))*x[1]**o + ((0.37263875606+0.49380245607j))*x[1] + ((0.709737141034-0.148481937318j))*x[2]**o + ((-0.196092438824-0.269817099882j))*x[2]
            ref[(1, 1, 0)]=(0.0334612805113+0.99992803116j)-((0.0525164241051+0.191631092899j))*(o-1.)
            arg[(1, 1, 1)]=(0.914553966391+0.504560337566j)*x[0]**o + ((0.988951421298+0.0256444340608j))*x[0] + ((0.160844055118+0.237509138398j))*x[1]**o + ((0.25309590537-0.837791445877j))*x[1] + ((-0.778962879428-0.236354640496j))*x[2]**o + ((-0.859845667554-0.272273933882j))*x[2]
            ref[(1, 1, 1)]=(0.339318400598-0.289353055115j)-((0.0494058570135+0.0842858059112j))*(o-1.)
            arg[(1, 2, 0)]=(0.0402683840618-0.59989530818j)*x[0]**o + ((-0.409833486607+0.888719125981j))*x[0] + ((-0.460366179045-0.440499809818j))*x[1]**o + ((0.191457031757-0.300175616762j))*x[1] + ((-0.237517135285+0.0966766753456j))*x[2]**o + ((0.976328893898+0.9186362455j))*x[2]
            ref[(1, 2, 0)]=(0.0501687543897+0.281730656033j)-((-0.109602488378-0.157286407109j))*(o-1.)
            arg[(1, 2, 1)]=(-0.052196876906-0.921676809567j)*x[0]**o + ((-0.936027564878-0.14769499162j))*x[0] + ((0.712323895969-0.41816158553j))*x[1]**o + ((0.774331776255+0.494496929137j))*x[1] + ((0.215402021088+0.344189678301j))*x[2]**o + ((-0.105473260538-0.774877714963j))*x[2]
            ref[(1, 2, 1)]=(0.304179995495-0.711862247122j)-((0.145921506692-0.165941452799j))*(o-1.)
            arg[(2, 0, 0)]=(0.931079216542+0.351690287819j)*x[0]**o + ((0.4501799001-0.832565905798j))*x[0] + ((-0.744120743031-0.209822360553j))*x[1]**o + ((0.917826574518-0.870035743125j))*x[1] + ((-0.790830319465+0.0217506050201j))*x[2]**o + ((-0.565217957215+0.0192712825855j))*x[2]
            ref[(2, 0, 0)]=(0.0994583357243-0.759855917025j)-((-0.100645307659+0.0272697553812j))*(o-1.)
            arg[(2, 0, 1)]=(-0.173677463953+0.0481767818411j)*x[0]**o + ((-0.724115250217+0.112380494749j))*x[0] + ((-0.0331355554736-0.367456494855j))*x[1]**o + ((-0.667973556229-0.785029012921j))*x[1] + ((0.0104379563626-0.510628664352j))*x[2]**o + ((0.726022857144+0.971382671424j))*x[2]
            ref[(2, 0, 1)]=(-0.431220506183-0.265587112057j)-((-0.0327291771773-0.138318062894j))*(o-1.)
            arg[(2, 1, 0)]=(0.439315048069-0.139880678359j)*x[0]**o + ((-0.430952569727-0.479905265541j))*x[0] + ((0.985051920092-0.35796242364j))*x[1]**o + ((-0.094920835423+0.827202562995j))*x[1] + ((-0.822911350784+0.889162490658j))*x[2]**o + ((-0.804457776495+0.604161842036j))*x[2]
            ref[(2, 1, 0)]=(-0.364437782134+0.671389264075j)-((0.100242602896+0.0652198981099j))*(o-1.)
            arg[(2, 1, 1)]=(0.310607498357-0.501691496396j)*x[0]**o + ((0.619627045227-0.201371039786j))*x[0] + ((-0.08046802148+0.266142162956j))*x[1]**o + ((-0.612381165752+0.386514333757j))*x[1] + ((0.78653145943+0.0598534289821j))*x[2]**o + ((-0.371810360841+0.305019582449j))*x[2]
            ref[(2, 1, 1)]=(0.32605322747+0.157233485981j)-((0.169445156051-0.0292826507431j))*(o-1.)
            arg[(2, 2, 0)]=(0.797261241693+0.461269323891j)*x[0]**o + ((-0.946935029564+0.692540195494j))*x[0] + ((0.682002861893+0.240726968007j))*x[1]**o + ((0.701864430267+0.571090820702j))*x[1] + ((0.0838947401592+0.452811794317j))*x[2]**o + ((-0.884214844238+0.807892043895j))*x[2]
            ref[(2, 2, 0)]=(0.216936700105+1.61316557315j)-((0.260526473958+0.192468014369j))*(o-1.)
            arg[(2, 2, 1)]=(0.308295915089-0.876208475146j)*x[0]**o + ((0.671398430858-0.132183786922j))*x[0] + ((0.62109820329-0.619876631619j))*x[1]**o + ((-0.641124110161-0.560007807587j))*x[1] + ((0.560713158612+0.470011981677j))*x[2]**o + ((-0.642651325716+0.655928665413j))*x[2]
            ref[(2, 2, 1)]=(0.438865135986-0.531168027092j)-((0.248351212832-0.171012187515j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 3, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Solution_rank4(self):
        """
        tests integral of rank 4 Data on Function

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 4, 4, 2),w)
        ref=numpy.zeros((2, 4, 4, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.353246768041-0.570331764661j)*x[0]**o + ((0.510050925141-0.422063177939j))*x[0] + ((0.505787513047+0.886626047226j))*x[1]**o + ((-0.019441706576+0.00239121692685j))*x[1]
            ref[(0, 0, 0, 0)]=(0.321574981785-0.0516888392232j)-((0.0254234575009+0.052715713761j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.975189836838-0.0371146326851j)*x[0]**o + ((0.766859667907-0.909352669648j))*x[0] + ((0.597223357099-0.347373756075j))*x[1]**o + ((-0.902604760528+0.0145470125352j))*x[1]
            ref[(0, 0, 0, 1)]=(0.718334050658-0.639647022936j)-((0.262068865656-0.0640813981266j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.424509421293+0.325353055129j)*x[0]**o + ((0.25807523836+0.432799991617j))*x[0] + ((-0.618934611778-0.484659360062j))*x[1]**o + ((-0.868824558946+0.614109652875j))*x[1]
            ref[(0, 0, 1, 0)]=(-0.402587255535+0.44380166978j)-((-0.0324041984142-0.0265510508221j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.0763088644029+0.573793667975j)*x[0]**o + ((0.946315388239+0.27818463767j))*x[0] + ((-0.968079898208-0.0135662548245j))*x[1]**o + ((-0.690374175851+0.704144536295j))*x[1]
            ref[(0, 0, 1, 1)]=(-0.394223775111+0.771278293558j)-((-0.174064793768+0.0933712355251j))*(o-1.)
            arg[(0, 0, 2, 0)]=(-0.184644297938-0.380808997858j)*x[0]**o + ((-0.941009650624-0.100275979094j))*x[0] + ((0.412306139774+0.430667117653j))*x[1]**o + ((-0.492516737316+0.000524378611533j))*x[1]
            ref[(0, 0, 2, 0)]=(-0.602932273052-0.024946740344j)-((0.037943640306+0.0083096866324j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.501593260297+0.902189249124j)*x[0]**o + ((-0.503286757831+0.531976612212j))*x[0] + ((0.129775378308-0.0880594185365j))*x[1]**o + ((0.706582735752-0.214317880859j))*x[1]
            ref[(0, 0, 2, 1)]=(0.417332308263+0.56589428097j)-((0.105228106434+0.135688305098j))*(o-1.)
            arg[(0, 0, 3, 0)]=(0.747178906558-0.531821199156j)*x[0]**o + ((0.382233990019+0.104577682631j))*x[0] + ((-0.370985746665-0.644281267827j))*x[1]**o + ((0.163975541544-0.432581389463j))*x[1]
            ref[(0, 0, 3, 0)]=(0.461201345727-0.752053086907j)-((0.0626988599821-0.19601707783j))*(o-1.)
            arg[(0, 0, 3, 1)]=(-0.429626546164-0.781955968704j)*x[0]**o + ((0.624644088957+0.599453608828j))*x[0] + ((-0.843516198219+0.829043306967j))*x[1]**o + ((0.494416877291+0.574713387026j))*x[1]
            ref[(0, 0, 3, 1)]=(-0.0770408890677+0.610627167059j)-((-0.212190457397+0.00784788971055j))*(o-1.)
            arg[(0, 1, 0, 0)]=(0.317944584322-0.760093132217j)*x[0]**o + ((0.335894603119-0.467513356413j))*x[0] + ((-0.909210491095+0.321006147122j))*x[1]**o + ((0.662770795192+0.494064037942j))*x[1]
            ref[(0, 1, 0, 0)]=(0.203699745769-0.206268151783j)-((-0.0985443177956-0.0731811641826j))*(o-1.)
            arg[(0, 1, 0, 1)]=(-0.858060006705+0.155778426036j)*x[0]**o + ((-0.781409850842+0.497558822139j))*x[0] + ((0.277028615662+0.140302246903j))*x[1]**o + ((-0.719126474093+0.498207294707j))*x[1]
            ref[(0, 1, 0, 1)]=(-1.04078385799+0.645923394892j)-((-0.0968385651737+0.0493467788232j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.350204325551-0.509640489603j)*x[0]**o + ((-0.189813951742+0.0304378795136j))*x[0] + ((0.251347116113+0.146780949959j))*x[1]**o + ((-0.579042651187-0.769550081811j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.433856906184-0.550985870971j)-((-0.016476201573-0.0604765899406j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.915953196619+0.238301176234j)*x[0]**o + ((0.817198964014+0.100878676743j))*x[0] + ((0.699324652253+0.976072878679j))*x[1]**o + ((-0.669754128216+0.0889950623609j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.0345918542842+0.702123897009j)-((-0.0361047573943+0.202395675819j))*(o-1.)
            arg[(0, 1, 2, 0)]=(-0.763841425182-0.390931427254j)*x[0]**o + ((0.169769485179-0.739321117052j))*x[0] + ((-0.738386279942-0.958573608527j))*x[1]**o + ((0.852333023832-0.804962314799j))*x[1]
            ref[(0, 1, 2, 0)]=(-0.240062598056-1.44689423382j)-((-0.250371284187-0.224917505963j))*(o-1.)
            arg[(0, 1, 2, 1)]=(0.313561019317-0.553019858718j)*x[0]**o + ((0.692252100918+0.601913567933j))*x[0] + ((0.788270453203+0.674082738074j))*x[1]**o + ((-0.469880107985-0.0495939984067j))*x[1]
            ref[(0, 1, 2, 1)]=(0.662101732726+0.336691224441j)-((0.183638578753+0.0201771465595j))*(o-1.)
            arg[(0, 1, 3, 0)]=(0.952038018591-0.0590329947906j)*x[0]**o + ((-0.698089645663-0.622233426572j))*x[0] + ((0.716447734313-0.423725484988j))*x[1]**o + ((0.918014721476+0.372963971394j))*x[1]
            ref[(0, 1, 3, 0)]=(0.944205414359-0.366013967478j)-((0.278080958817-0.0804597466297j))*(o-1.)
            arg[(0, 1, 3, 1)]=(-0.95007074422-0.553502795231j)*x[0]**o + ((-0.266426875637+0.567385086308j))*x[0] + ((-0.209521570029+0.76057493746j))*x[1]**o + ((0.0818944831309-0.822320522973j))*x[1]
            ref[(0, 1, 3, 1)]=(-0.672062353378-0.0239316472183j)-((-0.193265385708+0.0345120237048j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.256406582596+0.105657423941j)*x[0]**o + ((-0.329972264874+0.883771977357j))*x[0] + ((0.876852006616-0.394075435146j))*x[1]**o + ((0.179152246184+0.390261056207j))*x[1]
            ref[(0, 2, 0, 0)]=(0.491219285261+0.49280751118j)-((0.188876431535-0.0480696685342j))*(o-1.)
            arg[(0, 2, 0, 1)]=(0.423176451145-0.718930887505j)*x[0]**o + ((0.0437639368075-0.834345481713j))*x[0] + ((-0.957361487617+0.349420665183j))*x[1]**o + ((-0.945036296386+0.299015171743j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.717728698025-0.452420266145j)-((-0.089030839412-0.0615850370536j))*(o-1.)
            arg[(0, 2, 1, 0)]=(-0.874698762368-0.341900976641j)*x[0]**o + ((-0.734099269661+0.676839134931j))*x[0] + ((0.878687551525-0.189271193291j))*x[1]**o + ((0.761026973396+0.740161634651j))*x[1]
            ref[(0, 2, 1, 0)]=(0.0154582464461+0.442914299825j)-((0.000664798192751-0.0885286949888j))*(o-1.)
            arg[(0, 2, 1, 1)]=(0.997382716863-0.545757360099j)*x[0]**o + ((0.482172136945+0.616816282295j))*x[0] + ((0.944733422877-0.0346264396639j))*x[1]**o + ((0.993316346726-0.327828198021j))*x[1]
            ref[(0, 2, 1, 1)]=(1.70880231171-0.145697857745j)-((0.32368602329-0.0967306332938j))*(o-1.)
            arg[(0, 2, 2, 0)]=(-0.101130515603+0.227348233656j)*x[0]**o + ((-0.682911304513+0.868170633501j))*x[0] + ((-0.771369555433+0.872476834563j))*x[1]**o + ((0.746058907757+0.196835399986j))*x[1]
            ref[(0, 2, 2, 0)]=(-0.404676233896+1.08241555085j)-((-0.145416678506+0.183304178036j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.213274286948-0.454494287649j)*x[0]**o + ((-0.509249966175+0.0154715075255j))*x[0] + ((0.502016436362+0.850805140969j))*x[1]**o + ((-0.693448322109-0.939730356065j))*x[1]
            ref[(0, 2, 2, 1)]=(-0.456978069435-0.26397399761j)-((0.0481236915691+0.0660518088866j))*(o-1.)
            arg[(0, 2, 3, 0)]=(-0.0188733958688+0.415202557852j)*x[0]**o + ((0.0182247984857-0.923160730933j))*x[0] + ((0.142078698777-0.681449467348j))*x[1]**o + ((0.372137018049-0.0509044222242j))*x[1]
            ref[(0, 2, 3, 0)]=(0.256783559721-0.620156031327j)-((0.0205342171514-0.044374484916j))*(o-1.)
            arg[(0, 2, 3, 1)]=(0.250073908186+0.168691138144j)*x[0]**o + ((-0.23654136447+0.250540342615j))*x[0] + ((-0.414591932254-0.609761280821j))*x[1]**o + ((0.327342353719+0.549823351705j))*x[1]
            ref[(0, 2, 3, 1)]=(-0.0368585174093+0.179646775822j)-((-0.027419670678-0.0735116904461j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.550047200378+0.997927717233j)*x[0]**o + ((-0.351946103283-0.343711068987j))*x[0] + ((0.643074007713-0.794904276763j))*x[1]**o + ((0.307005360141-0.732307222089j))*x[1]
            ref[(0, 3, 0, 0)]=(0.574090232475-0.436497425303j)-((0.198853534682+0.0338372400783j))*(o-1.)
            arg[(0, 3, 0, 1)]=(0.739437918474+0.37375991548j)*x[0]**o + ((0.541611378416+0.966516792427j))*x[0] + ((0.9213676565-0.267313592533j))*x[1]**o + ((0.834697295614+0.445757348972j))*x[1]
            ref[(0, 3, 0, 1)]=(1.5185571245+0.759360232173j)-((0.276800929162+0.0177410538244j))*(o-1.)
            arg[(0, 3, 1, 0)]=(-0.464940476332-0.538397330203j)*x[0]**o + ((-0.320168765856-0.200697146739j))*x[0] + ((0.474398367477-0.212941738451j))*x[1]**o + ((-0.270271612501-0.407381444848j))*x[1]
            ref[(0, 3, 1, 0)]=(-0.290491243606-0.679708830121j)-((0.00157631519087-0.125223178109j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.206106337901+0.504732781686j)*x[0]**o + ((-0.588030360212-0.610227347891j))*x[0] + ((-0.108362041157-0.157461527423j))*x[1]**o + ((0.738280345641-0.309339966573j))*x[1]
            ref[(0, 3, 1, 1)]=(-0.0821091968142-0.2861480301j)-((-0.0524113965096+0.0578785423772j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.198597697622+0.380567949713j)*x[0]**o + ((-0.950953537564+0.409185603316j))*x[0] + ((-0.924106830214+0.475888570373j))*x[1]**o + ((0.249964849753-0.482258179841j))*x[1]
            ref[(0, 3, 2, 0)]=(-0.713248910201+0.391691971781j)-((-0.120918188765+0.142742753348j))*(o-1.)
            arg[(0, 3, 2, 1)]=(0.898616120533-0.0200448395715j)*x[0]**o + ((0.416212566493-0.0956027659202j))*x[0] + ((0.802302043274-0.928851454891j))*x[1]**o + ((0.422097707855-0.0049651392121j))*x[1]
            ref[(0, 3, 2, 1)]=(1.26961421908-0.524732099797j)-((0.283486360635-0.15814938241j))*(o-1.)
            arg[(0, 3, 3, 0)]=(-0.25307946623+0.473517211937j)*x[0]**o + ((0.46608877829-0.717842164196j))*x[0] + ((0.548593541269+0.66192649629j))*x[1]**o + ((-0.343554418502-0.774575972724j))*x[1]
            ref[(0, 3, 3, 0)]=(0.209024217414-0.178487214346j)-((0.0492523458399+0.189240618038j))*(o-1.)
            arg[(0, 3, 3, 1)]=(0.958167996402-0.714715768108j)*x[0]**o + ((-0.855020689334+0.978467578415j))*x[0] + ((-0.593879805329+0.196296238193j))*x[1]**o + ((-0.7715662697+0.14534607047j))*x[1]
            ref[(0, 3, 3, 1)]=(-0.631149383981+0.302697059485j)-((0.0607146985122-0.0864032549858j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.897173993853-0.243614204769j)*x[0]**o + ((-0.786703235496-0.810771198802j))*x[0] + ((-0.840786144681+0.265908223073j))*x[1]**o + ((-0.492403953143+0.647028455891j))*x[1]
            ref[(1, 0, 0, 0)]=(-1.50853366359-0.0707243623039j)-((-0.289660023089+0.00371566971721j))*(o-1.)
            arg[(1, 0, 0, 1)]=(0.739803876606-0.437590557425j)*x[0]**o + ((-0.427703079702+0.568892906955j))*x[0] + ((-0.88726318744+0.106765603664j))*x[1]**o + ((-0.626750860589+0.692753024845j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.600956625563+0.46541048902j)-((-0.0245765518058-0.0551374922935j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.998534686339+0.362508782981j)*x[0]**o + ((-0.617803254618-0.79841590524j))*x[0] + ((-0.965436429636-0.400736833348j))*x[1]**o + ((-0.61531711132-0.962196476892j))*x[1]
            ref[(1, 0, 1, 0)]=(-1.59854574096-0.899420216249j)-((-0.327328519329-0.00637134172788j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.768975937451+0.843024160091j)*x[0]**o + ((0.942266705808+0.641523759932j))*x[0] + ((-0.651874461529+0.0957805078741j))*x[1]**o + ((-0.923481621015-0.00203428313184j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.701032657094+0.789147072383j)-((-0.23680839983+0.156467444661j))*(o-1.)
            arg[(1, 0, 2, 0)]=(-0.989213382305+0.452536299586j)*x[0]**o + ((0.855996786704+0.309857567928j))*x[0] + ((0.0446931251477-0.578107826474j))*x[1]**o + ((0.810193349578-0.717935623717j))*x[1]
            ref[(1, 0, 2, 0)]=(0.360834939562-0.266824791338j)-((-0.15742004286-0.0209285878147j))*(o-1.)
            arg[(1, 0, 2, 1)]=(-0.515646522729+0.806242262987j)*x[0]**o + ((-0.896511049732-0.512123351287j))*x[0] + ((0.643022125442-0.567882072984j))*x[1]**o + ((-0.999773441979+0.118451276257j))*x[1]
            ref[(1, 0, 2, 1)]=(-0.8844544445-0.0776559425133j)-((0.0212292671188+0.0397266983338j))*(o-1.)
            arg[(1, 0, 3, 0)]=(-0.599517815956-0.0963969527675j)*x[0]**o + ((0.75506471074+0.83358123105j))*x[0] + ((-0.199093627154+0.903292284501j))*x[1]**o + ((-0.648402747341+0.429472577464j))*x[1]
            ref[(1, 0, 3, 0)]=(-0.345974739856+1.03497457012j)-((-0.133101907185+0.134482555289j))*(o-1.)
            arg[(1, 0, 3, 1)]=(0.417672537592+0.867072022437j)*x[0]**o + ((-0.359714013929-0.0155653690353j))*x[0] + ((0.134614985681-0.736241640604j))*x[1]**o + ((-0.275840856668+0.39881154909j))*x[1]
            ref[(1, 0, 3, 1)]=(-0.0416336736623+0.257038280944j)-((0.0920479205455+0.0218050636389j))*(o-1.)
            arg[(1, 1, 0, 0)]=(-0.122937354334+0.0854129668752j)*x[0]**o + ((-0.81493594017+0.593860716164j))*x[0] + ((0.863507245498-0.835812953839j))*x[1]**o + ((-0.995048978891-0.236315088334j))*x[1]
            ref[(1, 1, 0, 0)]=(-0.534707513949-0.196427179567j)-((0.123428315194-0.125066664494j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.999326857882-0.995720739594j)*x[0]**o + ((-0.315372093965-0.0433574069513j))*x[0] + ((0.5494358715+0.567011463418j))*x[1]**o + ((0.62910564782-0.475457113797j))*x[1]
            ref[(1, 1, 0, 1)]=(-0.0680787162634-0.473761898462j)-((-0.0749818310636-0.0714515460294j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.743266650796-0.469404106265j)*x[0]**o + ((-0.234667983773-0.387953089711j))*x[0] + ((0.753578938002-0.46890816875j))*x[1]**o + ((0.524560561541-0.155066973489j))*x[1]
            ref[(1, 1, 1, 0)]=(0.893369083283-0.740666169107j)-((0.2494742648-0.156385379169j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.937418973952+0.494937696415j)*x[0]**o + ((-0.381231476678+0.0495893082881j))*x[0] + ((0.540947108141-0.471397934418j))*x[1]**o + ((-0.158882309878+0.516465467028j))*x[1]
            ref[(1, 1, 1, 1)]=(0.469126147768+0.294797268656j)-((0.246394347015+0.00392329366618j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.578350787897+0.826336812115j)*x[0]**o + ((0.998713406907+0.0877301472536j))*x[0] + ((0.669414581444+0.346178841273j))*x[1]**o + ((0.0191523299557+0.245149006377j))*x[1]
            ref[(1, 1, 2, 0)]=(1.1328155531+0.752697403509j)-((0.20796089489+0.195419275565j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.510337115349+0.901004443078j)*x[0]**o + ((0.0962019249879-0.85657545538j))*x[0] + ((-0.692992688163+0.785447554529j))*x[1]**o + ((0.838977839952-0.505942787648j))*x[1]
            ref[(1, 1, 2, 1)]=(-0.134075019286+0.16196687729j)-((-0.200554967252+0.281075332935j))*(o-1.)
            arg[(1, 1, 3, 0)]=(0.102279901336-0.248900203864j)*x[0]**o + ((-0.46907086896-0.262977732943j))*x[0] + ((0.740327866502+0.872184484979j))*x[1]**o + ((0.185933960197+0.621366065592j))*x[1]
            ref[(1, 1, 3, 0)]=(0.279735429537+0.490836306882j)-((0.140434627973+0.103880713519j))*(o-1.)
            arg[(1, 1, 3, 1)]=(-0.0166446181054+0.251447089153j)*x[0]**o + ((0.870982344648+0.177352636902j))*x[0] + ((0.97682989772-0.0853773450138j))*x[1]**o + ((0.798006273262+0.967305832767j))*x[1]
            ref[(1, 1, 3, 1)]=(1.31458694876+0.655364106904j)-((0.160030879936+0.0276782906898j))*(o-1.)
            arg[(1, 2, 0, 0)]=(-0.561573198816-0.542980194659j)*x[0]**o + ((-0.178287047913-0.51704086916j))*x[0] + ((-0.916053547684+0.641323319367j))*x[1]**o + ((-0.822350608939+0.408607424944j))*x[1]
            ref[(1, 2, 0, 0)]=(-1.23913220168-0.00504515975398j)-((-0.246271124417+0.0163905207847j))*(o-1.)
            arg[(1, 2, 0, 1)]=(0.876760895721-0.982917647582j)*x[0]**o + ((-0.592352593147+0.271100939796j))*x[0] + ((0.461026130231+0.757530732652j))*x[1]**o + ((-0.852002324217-0.665102303311j))*x[1]
            ref[(1, 2, 0, 1)]=(-0.0532839457056-0.309694139222j)-((0.222964504325-0.0375644858217j))*(o-1.)
            arg[(1, 2, 1, 0)]=(-0.744062965839+0.830103380154j)*x[0]**o + ((-0.911662489959-0.0659969847848j))*x[0] + ((0.369949074433+0.986009197818j))*x[1]**o + ((-0.73875136952-0.952475138521j))*x[1]
            ref[(1, 2, 1, 0)]=(-1.01226387544+0.398820227333j)-((-0.0623523152342+0.302685429662j))*(o-1.)
            arg[(1, 2, 1, 1)]=(-0.826532344062+0.510372591918j)*x[0]**o + ((-0.50301186101+0.17164687634j))*x[0] + ((-0.145297518105+0.218691483383j))*x[1]**o + ((-0.649206469163+0.421584069465j))*x[1]
            ref[(1, 2, 1, 1)]=(-1.06202409617+0.661147510553j)-((-0.161971643695+0.121510679217j))*(o-1.)
            arg[(1, 2, 2, 0)]=(0.47780150852-0.820749581877j)*x[0]**o + ((0.051179242751+0.503534632483j))*x[0] + ((0.370605233667-0.874244948121j))*x[1]**o + ((-0.901716568164-0.00283671671869j))*x[1]
            ref[(1, 2, 2, 0)]=(-0.00106529161295-0.597148307117j)-((0.141401123698-0.282499088333j))*(o-1.)
            arg[(1, 2, 2, 1)]=(0.325163211134-0.172663047641j)*x[0]**o + ((-0.17510613969+0.893905795469j))*x[0] + ((-0.539896477756+0.970446974397j))*x[1]**o + ((-0.560507647283-0.48156520181j))*x[1]
            ref[(1, 2, 2, 1)]=(-0.475173526798+0.605062260207j)-((-0.0357888777704+0.132963987793j))*(o-1.)
            arg[(1, 2, 3, 0)]=(-0.0211827227613+0.435398064646j)*x[0]**o + ((0.943393681533-0.463515760156j))*x[0] + ((0.464949176199+0.102885310352j))*x[1]**o + ((-0.539866239502-0.960794018836j))*x[1]
            ref[(1, 2, 3, 0)]=(0.423646947734-0.443013201997j)-((0.0739610755729+0.089713895833j))*(o-1.)
            arg[(1, 2, 3, 1)]=(-0.197437124314+0.589740171252j)*x[0]**o + ((-0.769594020536+0.959524939373j))*x[0] + ((0.781495856511+0.894903801421j))*x[1]**o + ((-0.93961979304-0.232141775282j))*x[1]
            ref[(1, 2, 3, 1)]=(-0.56257754069+1.10601356838j)-((0.0973431220327+0.247440662112j))*(o-1.)
            arg[(1, 3, 0, 0)]=(-0.274935291056-0.544334390369j)*x[0]**o + ((-0.813703534073-0.817133863724j))*x[0] + ((-0.264203500161+0.685100327908j))*x[1]**o + ((0.370563603006-0.0397586078223j))*x[1]
            ref[(1, 3, 0, 0)]=(-0.491139361142-0.358063267003j)-((-0.0898564652027+0.0234609895899j))*(o-1.)
            arg[(1, 3, 0, 1)]=(0.14629241383+0.812337619279j)*x[0]**o + ((-0.972553709123-0.310620183738j))*x[0] + ((0.0673655976513+0.630257024429j))*x[1]**o + ((-0.991408397516-0.885397482419j))*x[1]
            ref[(1, 3, 0, 1)]=(-0.875152047579+0.123288488776j)-((0.0356096685802+0.240432440618j))*(o-1.)
            arg[(1, 3, 1, 0)]=(-0.12037076987+0.302033312269j)*x[0]**o + ((0.542300329251+0.102694870199j))*x[0] + ((-0.683110202936-0.847915670549j))*x[1]**o + ((0.955996003073-0.0256247243126j))*x[1]
            ref[(1, 3, 1, 0)]=(0.347407679759-0.234406106197j)-((-0.133913495468-0.0909803930467j))*(o-1.)
            arg[(1, 3, 1, 1)]=(-0.183348739397+0.0331406318197j)*x[0]**o + ((-0.00507398837499-0.851555197842j))*x[0] + ((-0.951085395724-0.654931965325j))*x[1]**o + ((0.624069920446+0.920079955162j))*x[1]
            ref[(1, 3, 1, 1)]=(-0.257719101525-0.276633288093j)-((-0.189072355853-0.103631888917j))*(o-1.)
            arg[(1, 3, 2, 0)]=(0.0567476318915+0.424297442922j)*x[0]**o + ((-0.84648875347+0.0737892150973j))*x[0] + ((0.299450747994-0.724739789481j))*x[1]**o + ((-0.494819935499+0.711181039975j))*x[1]
            ref[(1, 3, 2, 0)]=(-0.492555154542+0.242263954257j)-((0.0593663966475-0.0500737244264j))*(o-1.)
            arg[(1, 3, 2, 1)]=(0.887635956725-0.468113283255j)*x[0]**o + ((0.388731412347+0.532169023486j))*x[0] + ((-0.180830490766-0.0426944330768j))*x[1]**o + ((0.116233939945-0.159346518612j))*x[1]
            ref[(1, 3, 2, 1)]=(0.605885409126-0.0689926057287j)-((0.117800910993-0.0851346193886j))*(o-1.)
            arg[(1, 3, 3, 0)]=(0.547333793267-0.352845215712j)*x[0]**o + ((0.377500001627+0.543896412907j))*x[0] + ((-0.0773033148827+0.0778049149335j))*x[1]**o + ((-0.307396025366+0.604560691268j))*x[1]
            ref[(1, 3, 3, 0)]=(0.270067227323+0.436708401698j)-((0.078338413064-0.0458400501298j))*(o-1.)
            arg[(1, 3, 3, 1)]=(0.959685445719-0.513931557872j)*x[0]**o + ((0.159399036922+0.491887433326j))*x[0] + ((0.989062113393-0.829535054342j))*x[1]**o + ((-0.663979330761-0.338710642966j))*x[1]
            ref[(1, 3, 3, 1)]=(0.722083632636-0.595144910926j)-((0.324791259852-0.223911102036j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(0.307329935841-0.385064445599j)*x[0]**o + ((0.116887786971-0.421331720884j))*x[0] + ((0.835411726878-0.185398065115j))*x[1]**o + ((0.154440161838+0.322692050963j))*x[1] + ((-0.254567803079-0.583453655659j))*x[2]**o + ((-0.839859856473-0.761981059406j))*x[2]
            ref[(0, 0, 0, 0)]=(0.159820975988-1.00726844785j)-((0.148028976607-0.192319361062j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.761851382202-0.876126164397j)*x[0]**o + ((0.363393452565-0.65352598529j))*x[0] + ((0.790218972292-0.739065004251j))*x[1]**o + ((0.253441737202-0.206175658215j))*x[1] + ((0.681877295579+0.611600052494j))*x[2]**o + ((-0.909804405541+0.41667214997j))*x[2]
            ref[(0, 0, 0, 1)]=(0.97048921715-0.723310304844j)-((0.372324608346-0.167265186026j))*(o-1.)
            arg[(0, 0, 1, 0)]=(-0.118449198475+0.632684768099j)*x[0]**o + ((-0.599539928679-0.427515224268j))*x[0] + ((-0.582889409948+0.514664677673j))*x[1]**o + ((-0.975134958678+0.218866669434j))*x[1] + ((-0.287899960177-0.0707439378761j))*x[2]**o + ((-0.978435645677-0.684106386451j))*x[2]
            ref[(0, 0, 1, 0)]=(-1.77117455082+0.0919252833053j)-((-0.164873094767+0.179434251316j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.104014374123+0.56083155713j)*x[0]**o + ((0.691471206295+0.0471590404722j))*x[0] + ((0.92731830385-0.893092103262j))*x[1]**o + ((-0.165479401529-0.141196405015j))*x[1] + ((0.860166627907+0.301642509579j))*x[2]**o + ((-0.936567165063-0.827013155453j))*x[2]
            ref[(0, 0, 1, 1)]=(0.740461972792-0.475834278274j)-((0.315249884313-0.00510300609223j))*(o-1.)
            arg[(0, 0, 2, 0)]=(0.264997220508-0.374392108818j)*x[0]**o + ((-0.897883420622-0.776767206512j))*x[0] + ((0.341695039424-0.218664687642j))*x[1]**o + ((0.753302292127-0.604508157578j))*x[1] + ((0.133913459996+0.607912769938j))*x[2]**o + ((-0.349463625211-0.524667644185j))*x[2]
            ref[(0, 0, 2, 0)]=(0.123280483111-0.945543517399j)-((0.123434286655+0.00247599557966j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.945374325076+0.859919937841j)*x[0]**o + ((-0.782162437316-0.747757288031j))*x[0] + ((-0.979060362123-0.632954707587j))*x[1]**o + ((0.587485384731-0.22312238505j))*x[1] + ((-0.650800393177-0.0157377354762j))*x[2]**o + ((0.62474539179-0.786745063589j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.127209045509-0.773198620946j)-((-0.114081071704+0.0352045824631j))*(o-1.)
            arg[(0, 0, 3, 0)]=(0.510922198127+0.621108466944j)*x[0]**o + ((0.850320582395-0.156690846368j))*x[0] + ((-0.738048663183-0.655892260511j))*x[1]**o + ((-0.741159065828+0.238708400468j))*x[1] + ((0.54038548946-0.640161678539j))*x[2]**o + ((-0.0310237214602-0.0542675978489j))*x[2]
            ref[(0, 0, 3, 0)]=(0.195698409755-0.323597757927j)-((0.0522098374007-0.112490912018j))*(o-1.)
            arg[(0, 0, 3, 1)]=(0.635545556666+0.266294609526j)*x[0]**o + ((0.0617982957212+0.560278424379j))*x[0] + ((-0.420659517348+0.324869772028j))*x[1]**o + ((0.24807221385+0.288542606575j))*x[1] + ((0.923654368744-0.373645752756j))*x[2]**o + ((-0.949101847938-0.00817882367925j))*x[2]
            ref[(0, 0, 3, 1)]=(0.249654534847+0.529080418036j)-((0.189756734677+0.0362531047996j))*(o-1.)
            arg[(0, 1, 0, 0)]=(0.55237232501+0.949074255673j)*x[0]**o + ((-0.134515674953-0.820082656873j))*x[0] + ((0.127726823184-0.16717333693j))*x[1]**o + ((0.0706870674071-0.819112821852j))*x[1] + ((-0.897749483675-0.065405392987j))*x[2]**o + ((-0.0729668211572+0.914713808914j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.177222882092-0.00399307202792j)-((-0.0362750559134+0.119415920959j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.701142280487-0.685506690634j)*x[0]**o + ((-0.560180874295-0.789699315459j))*x[0] + ((0.666054905465+0.467055994276j))*x[1]**o + ((0.0206287560575-0.0610068657744j))*x[1] + ((-0.932995684125+0.38818229442j))*x[2]**o + ((0.103295036332+0.871486713524j))*x[2]
            ref[(0, 1, 0, 1)]=(-0.00102779003865+0.0952560651758j)-((0.0723669169713+0.0282885996769j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.767302430596-0.984928748051j)*x[0]**o + ((0.141384505673-0.982542056663j))*x[0] + ((-0.874119719093+0.691258388339j))*x[1]**o + ((0.139554205232+0.52384315409j))*x[1] + ((0.112491435883-0.634256667827j))*x[2]**o + ((0.39855440725+0.809056452506j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.424718797825-0.288784738803j)-((-0.254821785634-0.15465450459j))*(o-1.)
            arg[(0, 1, 1, 1)]=(0.241696998668-0.578781357263j)*x[0]**o + ((0.43311687664-0.878326622182j))*x[0] + ((-0.846077494639-0.62649542901j))*x[1]**o + ((-0.744913144096-0.248766828467j))*x[1] + ((0.76877090815+0.255638554131j))*x[2]**o + ((0.434899337545+0.863158523702j))*x[2]
            ref[(0, 1, 1, 1)]=(0.143746741134-0.606786579544j)-((0.0273984020299-0.15827303869j))*(o-1.)
            arg[(0, 1, 2, 0)]=(-0.692975026841-0.183254141502j)*x[0]**o + ((-0.602390925219+0.421100300366j))*x[0] + ((0.739953278195-0.391457956348j))*x[1]**o + ((-0.869870089614+0.52985081043j))*x[1] + ((-0.494718489056-0.436029011892j))*x[2]**o + ((0.977126798156+0.971848561251j))*x[2]
            ref[(0, 1, 2, 0)]=(-0.47143722719+0.456029281153j)-((-0.0746233729504-0.168456851624j))*(o-1.)
            arg[(0, 1, 2, 1)]=(0.863388980652+0.333704479721j)*x[0]**o + ((0.161659038387+0.309690591721j))*x[0] + ((-0.901168361919+0.811619016347j))*x[1]**o + ((-0.734167044984+0.622375302537j))*x[1] + ((-0.0457524353194-0.914573202619j))*x[2]**o + ((-0.890093059891-0.607852891138j))*x[2]
            ref[(0, 1, 2, 1)]=(-0.773066441537+0.277481648285j)-((-0.013921969431+0.0384583822415j))*(o-1.)
            arg[(0, 1, 3, 0)]=(-0.682256217979-0.459401552803j)*x[0]**o + ((-0.829413102016-0.0525301417472j))*x[0] + ((0.52578539763-0.935392420321j))*x[1]**o + ((-0.855330588813-0.879812989484j))*x[1] + ((-0.117709990211+0.0778895753601j))*x[2]**o + ((-0.768074270912-0.507521322267j))*x[2]
            ref[(0, 1, 3, 0)]=(-1.36349938615-1.37838442563j)-((-0.0456968017599-0.219484066294j))*(o-1.)
            arg[(0, 1, 3, 1)]=(-0.352039564354+0.842756999743j)*x[0]**o + ((0.00709023910355-0.815670302777j))*x[0] + ((0.627857938382+0.888869094093j))*x[1]**o + ((0.455138257906+0.153460937537j))*x[1] + ((-0.67253518816+0.39430811553j))*x[2]**o + ((-0.495454534314-0.784311704903j))*x[2]
            ref[(0, 1, 3, 1)]=(-0.214971425719+0.339706569611j)-((-0.0661194690221+0.354322368228j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.0384541042453-0.818902814833j)*x[0]**o + ((-0.335683935189+0.171387992933j))*x[0] + ((-0.363244724167-0.771996920347j))*x[1]**o + ((0.585508346992+0.357517751704j))*x[1] + ((0.435278014377-0.790784885657j))*x[2]**o + ((0.693552753633-0.595049246626j))*x[2]
            ref[(0, 2, 0, 0)]=(0.526932279945-1.22391406141j)-((0.0184145657424-0.396947436806j))*(o-1.)
            arg[(0, 2, 0, 1)]=(-0.341525509364-0.0851274239135j)*x[0]**o + ((-0.0781257061015+0.0748728731991j))*x[0] + ((0.41875876273-0.632768978996j))*x[1]**o + ((-0.0900586427325+0.804624277837j))*x[1] + ((0.844026062303-0.996067474623j))*x[2]**o + ((0.00470939569418+0.891952037644j))*x[2]
            ref[(0, 2, 0, 1)]=(0.378892181265+0.0287426555737j)-((0.153543219278-0.285660646255j))*(o-1.)
            arg[(0, 2, 1, 0)]=(0.966968627246-0.415861161003j)*x[0]**o + ((0.0648821189143+0.57451497991j))*x[0] + ((0.76606093647-0.242824481931j))*x[1]**o + ((-0.387466697758+0.719081455903j))*x[1] + ((-0.706988460005-0.720661419717j))*x[2]**o + ((0.251578601265+0.382719309346j))*x[2]
            ref[(0, 2, 1, 0)]=(0.477517563066+0.148484341254j)-((0.171006850618-0.229891177108j))*(o-1.)
            arg[(0, 2, 1, 1)]=(0.38914051661-0.765556807835j)*x[0]**o + ((0.198706151858-0.899216765545j))*x[0] + ((-0.902808669483+0.402351874632j))*x[1]**o + ((-0.460026014975-0.588630325322j))*x[1] + ((-0.802892213454+0.181228526552j))*x[2]**o + ((0.45504922328-0.300397468488j))*x[2]
            ref[(0, 2, 1, 1)]=(-0.561415503082-0.985110483003j)-((-0.219426727721-0.0303294011085j))*(o-1.)
            arg[(0, 2, 2, 0)]=(0.243864159754-0.9038982012j)*x[0]**o + ((0.789975684404+0.851007113234j))*x[0] + ((-0.20768919481+0.730874566116j))*x[1]**o + ((0.202848967099+0.475897319644j))*x[1] + ((-0.198813412597-0.965955925173j))*x[2]**o + ((-0.90703915877-0.845978287022j))*x[2]
            ref[(0, 2, 2, 0)]=(-0.0384264774593-0.329026707201j)-((-0.027106407942-0.18982992671j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.985874873843-0.560774356421j)*x[0]**o + ((0.46271113273+0.924939250767j))*x[0] + ((-0.0383904729994+0.831456133392j))*x[1]**o + ((-0.364470595894-0.109253128421j))*x[1] + ((-0.717728121051+0.463994094796j))*x[2]**o + ((-0.0240156481764-0.849695883736j))*x[2]
            ref[(0, 2, 2, 1)]=(-0.833884289617+0.350333055188j)-((-0.290332244649+0.122445978628j))*(o-1.)
            arg[(0, 2, 3, 0)]=(0.960255655533+0.639954187716j)*x[0]**o + ((0.694311138866+0.88985639126j))*x[0] + ((-0.0718792989759-0.0938214795951j))*x[1]**o + ((-0.606661052987+0.223814807216j))*x[1] + ((-0.100617968497-0.945539204251j))*x[2]**o + ((0.738435785159+0.246927436131j))*x[2]
            ref[(0, 2, 3, 0)]=(0.806922129549+0.480596069238j)-((0.131293064677-0.0665677493551j))*(o-1.)
            arg[(0, 2, 3, 1)]=(-0.973491087226-0.784353360917j)*x[0]**o + ((-0.93997604312-0.0703554504545j))*x[0] + ((0.479130457738+0.706079383885j))*x[1]**o + ((0.721052101719-0.689949712635j))*x[1] + ((0.148102682525-0.726973338341j))*x[2]**o + ((-0.56082098626-0.379951359017j))*x[2]
            ref[(0, 2, 3, 1)]=(-0.563001437313-0.972751918739j)-((-0.0577096578272-0.134207885895j))*(o-1.)
            arg[(0, 3, 0, 0)]=(-0.527290709721-0.885556445285j)*x[0]**o + ((0.461608254797-0.42804400637j))*x[0] + ((-0.744574687349+0.614584613063j))*x[1]**o + ((-0.906310888546+0.983587007138j))*x[1] + ((0.65272112414-0.140523676167j))*x[2]**o + ((0.426145884002-0.00390668129261j))*x[2]
            ref[(0, 3, 0, 0)]=(-0.318850511338+0.0700704055433j)-((-0.103190712155-0.0685825847314j))*(o-1.)
            arg[(0, 3, 0, 1)]=(-0.235015126277-0.437386562499j)*x[0]**o + ((-0.07098469482+0.835619173346j))*x[0] + ((-0.451732015684-0.524638500399j))*x[1]**o + ((-0.6796851947-0.584544142759j))*x[1] + ((0.466002557283-0.82765856588j))*x[2]**o + ((-0.221603975079+0.68710248221j))*x[2]
            ref[(0, 3, 0, 1)]=(-0.596509224638-0.425753057991j)-((-0.0367907641128-0.298280604796j))*(o-1.)
            arg[(0, 3, 1, 0)]=(-0.987158560718-0.293324531506j)*x[0]**o + ((-0.857377021485-0.258826573909j))*x[0] + ((0.600248214076+0.357141156861j))*x[1]**o + ((0.724810109044+0.988667781167j))*x[1] + ((-0.224852208642+0.298914741016j))*x[2]**o + ((0.77140795576+0.816289990736j))*x[2]
            ref[(0, 3, 1, 0)]=(0.0135392440172+0.954431282182j)-((-0.101960425881+0.0604552277284j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.645449186663+0.254644681701j)*x[0]**o + ((0.138980353649-0.55707034584j))*x[0] + ((-0.950948883108-0.210908561258j))*x[1]**o + ((-0.499478970363-0.644049618354j))*x[1] + ((-0.598859865336-0.23329341135j))*x[2]**o + ((-0.539771797434+0.528800112454j))*x[2]
            ref[(0, 3, 1, 1)]=(-1.54776417463-0.430938571323j)-((-0.365876322518-0.0315928818179j))*(o-1.)
            arg[(0, 3, 2, 0)]=(-0.297956722542+0.336238835616j)*x[0]**o + ((-0.928185863293-0.912235558298j))*x[0] + ((-0.537376913916-0.517801968511j))*x[1]**o + ((-0.767802540455+0.106789012602j))*x[1] + ((-0.337338784347+0.473725249508j))*x[2]**o + ((0.257893948965+0.662157596779j))*x[2]
            ref[(0, 3, 2, 0)]=(-1.30538343779+0.0744365838474j)-((-0.195445403468+0.0486936861021j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.918011816397-0.523449031078j)*x[0]**o + ((-0.504941324369-0.887533029165j))*x[0] + ((-0.162747231164+0.672328372534j))*x[1]**o + ((-0.974777098409+0.00382609486369j))*x[1] + ((0.287432805804+0.986315787291j))*x[2]**o + ((0.270662976709-0.466473162152j))*x[2]
            ref[(0, 3, 2, 1)]=(-1.00119084391-0.107492483853j)-((-0.132221040293+0.189199188124j))*(o-1.)
            arg[(0, 3, 3, 0)]=(-0.204577660548-0.570615931973j)*x[0]**o + ((0.364288668769-0.175047755759j))*x[0] + ((0.104920582725+0.532473777527j))*x[1]**o + ((-0.493834382662+0.00106997063389j))*x[1] + ((0.217469064109+0.280306292556j))*x[2]**o + ((0.954080431681-0.403768479857j))*x[2]
            ref[(0, 3, 3, 0)]=(0.471173352037-0.167791063436j)-((0.0196353310476+0.0403606896851j))*(o-1.)
            arg[(0, 3, 3, 1)]=(-0.620072614585+0.402704561295j)*x[0]**o + ((0.701765403518+0.704070809243j))*x[0] + ((0.236634024753+0.279554752095j))*x[1]**o + ((-0.423344137427-0.545928132119j))*x[1] + ((-0.725105508183+0.66162306437j))*x[2]**o + ((-0.0555046510507-0.0802035686934j))*x[2]
            ref[(0, 3, 3, 1)]=(-0.442813741487+0.710910743095j)-((-0.184757349669+0.223980396293j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.966244294963-0.975411006614j)*x[0]**o + ((0.921283274973-0.74736231003j))*x[0] + ((-0.505211595413-0.378706324363j))*x[1]**o + ((-0.555323066679+0.789640397808j))*x[1] + ((-0.57971695869-0.476905919364j))*x[2]**o + ((0.807481088214+0.160323099139j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.438865776279-0.814211031712j)-((-0.341862141511-0.305170541723j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.207355422251+0.766475964764j)*x[0]**o + ((-0.0510413904543+0.19685177678j))*x[0] + ((0.644603592465-0.501874907522j))*x[1]**o + ((0.604046779105+0.403644574468j))*x[1] + ((-0.315847556692+0.370811988952j))*x[2]**o + ((-0.751076166729-0.976378969956j))*x[2]
            ref[(1, 0, 0, 1)]=(-0.0383350822783+0.129765213743j)-((0.0202334355869+0.105902174366j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.449692271525+0.866975804539j)*x[0]**o + ((-0.526369245617-0.190492197858j))*x[0] + ((0.745178223983+0.40315198831j))*x[1]**o + ((-0.940956676707+0.761792889847j))*x[1] + ((-0.41602323192+0.954634987685j))*x[2]**o + ((-0.830260474208-0.146981741318j))*x[2]
            ref[(1, 0, 1, 0)]=(-1.209061838+1.3245408656j)-((-0.0200895465771+0.370793796756j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.0268943007488-0.0938091735358j)*x[0]**o + ((0.987899729098+0.935198964273j))*x[0] + ((0.415169684396+0.389901300152j))*x[1]**o + ((-0.541209732306+0.918821003977j))*x[1] + ((0.911203250166+0.307398632653j))*x[2]**o + ((-0.160533856153+0.718848850683j))*x[2]
            ref[(1, 0, 1, 1)]=(0.792817387226+1.5881797891j)-((0.216579772302+0.100581793211j))*(o-1.)
            arg[(1, 0, 2, 0)]=(-0.120234598514-0.55455907723j)*x[0]**o + ((0.897569284661+0.00231708278326j))*x[0] + ((-0.353890513772-0.282763624251j))*x[1]**o + ((-0.849171704069-0.775386690122j))*x[1] + ((0.290210931143+0.807331253376j))*x[2]**o + ((-0.567694757314-0.89615134035j))*x[2]
            ref[(1, 0, 2, 0)]=(-0.351605678933-0.849606197897j)-((-0.0306523635238-0.00499857468416j))*(o-1.)
            arg[(1, 0, 2, 1)]=(0.820321262987-0.41787365701j)*x[0]**o + ((-0.6974244683-0.463573409838j))*x[0] + ((-0.525632922464+0.847406751608j))*x[1]**o + ((-0.837095009735-0.144106998136j))*x[1] + ((-0.751214039094-0.784671292133j))*x[2]**o + ((-0.427866064927-0.360368171968j))*x[2]
            ref[(1, 0, 2, 1)]=(-1.20945562077-0.661593388738j)-((-0.0760876164286-0.059189699589j))*(o-1.)
            arg[(1, 0, 3, 0)]=(0.311595692071+0.186044604373j)*x[0]**o + ((0.000677053302462-0.8831742348j))*x[0] + ((0.230089331842+0.000261428232845j))*x[1]**o + ((0.0120647364999+0.727360091268j))*x[1] + ((-0.425833931813-0.718953861196j))*x[2]**o + ((0.483659338506-0.802091330054j))*x[2]
            ref[(1, 0, 3, 0)]=(0.306126110204-0.745276651088j)-((0.0193085153499-0.0887746380983j))*(o-1.)
            arg[(1, 0, 3, 1)]=(-0.442338453481-0.773249331963j)*x[0]**o + ((0.901309071561-0.965097487017j))*x[0] + ((0.069271036931-0.533575329663j))*x[1]**o + ((-0.158075705963-0.450347837744j))*x[1] + ((0.0893874951516+0.215475534761j))*x[2]**o + ((0.690378936108+0.232409432775j))*x[2]
            ref[(1, 0, 3, 1)]=(0.574966190154-1.13719250943j)-((-0.0472799868998-0.181891521144j))*(o-1.)
            arg[(1, 1, 0, 0)]=(-0.732634904669-0.519976877992j)*x[0]**o + ((0.15757233544+0.932784185353j))*x[0] + ((-0.329652762688-0.0890744731849j))*x[1]**o + ((-0.297337226692+0.645398766448j))*x[1] + ((-0.438407127049+0.75498802507j))*x[2]**o + ((-0.992340971722+0.0806205016565j))*x[2]
            ref[(1, 1, 0, 0)]=(-1.31640032869+0.902370063675j)-((-0.250115799068+0.024322778982j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.147286816383-0.212400422107j)*x[0]**o + ((-0.280071505658-0.0986634680021j))*x[0] + ((-0.0751782087721+0.261053003457j))*x[1]**o + ((-0.462534928138+0.477737021906j))*x[1] + ((-0.111377712285-0.265535991281j))*x[2]**o + ((0.605920650281+0.021261234507j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.235264260477+0.0917256892403j)-((-0.05564045624-0.0361472349884j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.314448852714+0.668620068263j)*x[0]**o + ((0.199055468003+0.00953837138313j))*x[0] + ((-0.357944521351-0.00256432337885j))*x[1]**o + ((0.490460261731+0.778563733618j))*x[1] + ((0.842356560852+0.26997594248j))*x[2]**o + ((0.112707781286-0.722572822059j))*x[2]
            ref[(1, 1, 1, 0)]=(0.800542201617+0.500780485154j)-((0.133143482036+0.156005281227j))*(o-1.)
            arg[(1, 1, 1, 1)]=(-0.428782612899-0.428360481743j)*x[0]**o + ((0.0512690367602-0.796546073182j))*x[0] + ((0.0552250658039+0.447553112651j))*x[1]**o + ((0.548806527638-0.510366746896j))*x[1] + ((0.410781860162-0.162929371946j))*x[2]**o + ((-0.425692998507-0.849092739255j))*x[2]
            ref[(1, 1, 1, 1)]=(0.105803439479-1.14987115019j)-((0.00620405217784-0.0239561235065j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.87323623447-0.427149720624j)*x[0]**o + ((-0.14179101495+0.172665931804j))*x[0] + ((0.0711947517836+0.187050828545j))*x[1]**o + ((0.792672873568+0.493051383533j))*x[1] + ((-0.53672501421+0.00570926910459j))*x[2]**o + ((-0.112032534837-0.663706331539j))*x[2]
            ref[(1, 1, 2, 0)]=(0.473277647913-0.116189319588j)-((0.0679509953406-0.0390649371623j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.0775232988128-0.795802337654j)*x[0]**o + ((-0.85707824205-0.546487917059j))*x[0] + ((0.145235866083-0.439372231458j))*x[1]**o + ((0.471001744842+0.37272586152j))*x[1] + ((0.702097721424+0.921257986171j))*x[2]**o + ((0.123536265043-0.619128363749j))*x[2]
            ref[(1, 1, 2, 1)]=(0.253635028265-0.553403501114j)-((0.128301714782-0.0523194304901j))*(o-1.)
            arg[(1, 1, 3, 0)]=(0.914568727891+0.893987136303j)*x[0]**o + ((0.0948071613547-0.12313741106j))*x[0] + ((0.62651148299+0.275254403873j))*x[1]**o + ((-0.095503853271+0.243493684221j))*x[1] + ((-0.879771828496+0.177340611624j))*x[2]**o + ((0.566352485141+0.232316417753j))*x[2]
            ref[(1, 1, 3, 0)]=(0.613482087805+0.849627421357j)-((0.110218063731+0.224430358633j))*(o-1.)
            arg[(1, 1, 3, 1)]=(-0.8341930322+0.833365356254j)*x[0]**o + ((-0.352705118162-0.892729334619j))*x[0] + ((-0.121450238896-0.996088050984j))*x[1]**o + ((0.693601404465-0.974010796513j))*x[1] + ((0.0988872804515+0.115207179137j))*x[2]**o + ((0.30221568909-0.951212548539j))*x[2]
            ref[(1, 1, 3, 1)]=(-0.106822007626-1.43273409763j)-((-0.142792665108-0.00791925259878j))*(o-1.)
            arg[(1, 2, 0, 0)]=(0.89919788791-0.28160031226j)*x[0]**o + ((-0.821470494283+0.512119259493j))*x[0] + ((0.818117563989-0.700154092031j))*x[1]**o + ((0.776512374083-0.703454790572j))*x[1] + ((0.48726395614+0.48466891699j))*x[2]**o + ((0.771970969045-0.1124318793j))*x[2]
            ref[(1, 2, 0, 0)]=(1.46579612844-0.40042644884j)-((0.36742990134-0.0828475812169j))*(o-1.)
            arg[(1, 2, 0, 1)]=(-0.182745443847+0.578231073083j)*x[0]**o + ((-0.518070812422-0.913038429551j))*x[0] + ((-0.138384043654-0.0108485964293j))*x[1]**o + ((-0.830886402288+0.895477727502j))*x[1] + ((-0.88559403143+0.845610144061j))*x[2]**o + ((-0.0788732203315+0.0397724354838j))*x[2]
            ref[(1, 2, 0, 1)]=(-1.31727697699+0.717602177075j)-((-0.201120586489+0.235498770119j))*(o-1.)
            arg[(1, 2, 1, 0)]=(-0.110990157779+0.149043452115j)*x[0]**o + ((0.570043005761+0.595221927763j))*x[0] + ((-0.988449749448+0.570543937627j))*x[1]**o + ((-0.0535843389185+0.725212365063j))*x[1] + ((0.245288999931-0.346321101081j))*x[2]**o + ((0.538638045711-0.966311247104j))*x[2]
            ref[(1, 2, 1, 0)]=(0.100472902629+0.363694667192j)-((-0.142358484549+0.0622110481102j))*(o-1.)
            arg[(1, 2, 1, 1)]=(-0.296903530535+0.695908580936j)*x[0]**o + ((0.94349162789+0.0657857983499j))*x[0] + ((-0.440099652044-0.496118040877j))*x[1]**o + ((-0.251050202413+0.160113290991j))*x[1] + ((0.210943974437+0.356076944623j))*x[2]**o + ((0.414526443536-0.863827118499j))*x[2]
            ref[(1, 2, 1, 1)]=(0.290454330436-0.0410302722376j)-((-0.0876765346903+0.0926445807804j))*(o-1.)
            arg[(1, 2, 2, 0)]=(-0.165700662403+0.0450219430936j)*x[0]**o + ((-0.252208418089+0.560300203717j))*x[0] + ((0.297054236546+0.169252054819j))*x[1]**o + ((-0.609506081302+0.376797456985j))*x[1] + ((-0.25846301189+0.403405151104j))*x[2]**o + ((0.867812444068-0.800926738112j))*x[2]
            ref[(1, 2, 2, 0)]=(-0.0605057465351+0.376925035803j)-((-0.0211849062912+0.102946524836j))*(o-1.)
            arg[(1, 2, 2, 1)]=(-0.542443106155-0.742224843878j)*x[0]**o + ((0.287308387984+0.650534419937j))*x[0] + ((-0.189018073922-0.998328866371j))*x[1]**o + ((0.864613795341+0.417306837939j))*x[1] + ((-0.805085358028-0.666757330351j))*x[2]**o + ((-0.215664909188-0.616751364065j))*x[2]
            ref[(1, 2, 2, 1)]=(-0.300144631984-0.978110573394j)-((-0.256091089684-0.401218506766j))*(o-1.)
            arg[(1, 2, 3, 0)]=(0.789903144243+0.998320420651j)*x[0]**o + ((-0.857425524188+0.00835271850266j))*x[0] + ((-0.573743744186+0.825063933525j))*x[1]**o + ((0.742579679464-0.866997291349j))*x[1] + ((0.293220242094-0.474562062797j))*x[2]**o + ((0.524885113902-0.588961465805j))*x[2]
            ref[(1, 2, 3, 0)]=(0.459709455665-0.0493918736364j)-((0.0848966070252+0.22480371523j))*(o-1.)
            arg[(1, 2, 3, 1)]=(0.324671252293+0.304948244875j)*x[0]**o + ((-0.908306895392+0.191201011719j))*x[0] + ((0.0288329672048-0.664630797543j))*x[1]**o + ((0.550278186078-0.0165281930712j))*x[1] + ((0.40040296146-0.176753969196j))*x[2]**o + ((-0.79942459585-0.501102237106j))*x[2]
            ref[(1, 2, 3, 1)]=(-0.201773062103-0.431432970161j)-((0.125651196826-0.0894060869774j))*(o-1.)
            arg[(1, 3, 0, 0)]=(0.562135521861-0.581091423045j)*x[0]**o + ((0.889693010036-0.707354097661j))*x[0] + ((0.326619890654+0.724209229983j))*x[1]**o + ((-0.285577886925+0.018939771334j))*x[1] + ((0.208230725449+0.788481057967j))*x[2]**o + ((-0.14399531906-0.0901823212926j))*x[2]
            ref[(1, 3, 0, 0)]=(0.778552971007+0.0765011086431j)-((0.182831022994+0.155266477484j))*(o-1.)
            arg[(1, 3, 0, 1)]=(-0.827741752699+0.860616090876j)*x[0]**o + ((0.265457807355+0.217355407902j))*x[0] + ((0.225014111589+0.758016200745j))*x[1]**o + ((-0.670838828408+0.548373212501j))*x[1] + ((-0.0759343826957+0.974707971087j))*x[2]**o + ((-0.828995635765+0.668020278638j))*x[2]
            ref[(1, 3, 0, 1)]=(-0.956519340312+2.01354458087j)-((-0.113110337301+0.432223377118j))*(o-1.)
            arg[(1, 3, 1, 0)]=(-0.817737566412+0.689364857671j)*x[0]**o + ((0.285450305239-0.659049415542j))*x[0] + ((-0.300418586354+0.518759081067j))*x[1]**o + ((-0.352793947306-0.111824076924j))*x[1] + ((-0.7094174662-0.322725002349j))*x[2]**o + ((-0.349722389733-0.959267563106j))*x[2]
            ref[(1, 3, 1, 0)]=(-1.12231982538-0.422371059592j)-((-0.304595603161+0.147566489398j))*(o-1.)
            arg[(1, 3, 1, 1)]=(0.250563905115+0.859161709069j)*x[0]**o + ((-0.674313799118-0.215504222381j))*x[0] + ((-0.297928535368+0.972369584532j))*x[1]**o + ((0.1350516257+0.50555082988j))*x[1] + ((0.139126779576+0.662841295422j))*x[2]**o + ((-0.10852039419+0.240549715181j))*x[2]
            ref[(1, 3, 1, 1)]=(-0.278010209142+1.51248445585j)-((0.0152936915538+0.415728764837j))*(o-1.)
            arg[(1, 3, 2, 0)]=(0.428755551957+0.165105839635j)*x[0]**o + ((-0.492412947699-0.800908935187j))*x[0] + ((0.750823815562-0.941202633175j))*x[1]**o + ((0.22680344876-0.504462385942j))*x[1] + ((0.945018859573-0.756127851886j))*x[2]**o + ((0.410351400714+0.624733816714j))*x[2]
            ref[(1, 3, 2, 0)]=(1.13467006443-1.10643107492j)-((0.354099704515-0.255370774238j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.800151092199-0.0910502127959j)*x[0]**o + ((-0.824237541547-0.553968646695j))*x[0] + ((-0.443917465477+0.291618796307j))*x[1]**o + ((0.261439452413+0.719365474248j))*x[1] + ((0.536946520903+0.150124984078j))*x[2]**o + ((0.741378657738+0.582614576906j))*x[2]
            ref[(1, 3, 2, 1)]=(-0.264270734085+0.549352486024j)-((-0.117853672796+0.0584489279315j))*(o-1.)
            arg[(1, 3, 3, 0)]=(-0.902973409471-0.854477613295j)*x[0]**o + ((0.104961647303-0.113737337608j))*x[0] + ((-0.942456699682-0.140346761352j))*x[1]**o + ((0.359396926407+0.842155417953j))*x[1] + ((0.951827289603-0.919926071279j))*x[2]**o + ((0.32203028996-0.42741617193j))*x[2]
            ref[(1, 3, 3, 0)]=(-0.0536069779401-0.806874268756j)-((-0.148933803258-0.319125074321j))*(o-1.)
            arg[(1, 3, 3, 1)]=(0.105530579311+0.610409660155j)*x[0]**o + ((-0.609796305197+0.25724352445j))*x[0] + ((0.266928411223+0.333691047207j))*x[1]**o + ((0.38362181708+0.755873169635j))*x[1] + ((-0.576926872685+0.453580476411j))*x[2]**o + ((-0.176922268185-0.0460497368864j))*x[2]
            ref[(1, 3, 3, 1)]=(-0.303782319227+1.18237407049j)-((-0.0340779803587+0.232946863962j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 4, 4, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedSolution_rank0(self):
        """
        tests integral of rank 0 Data on Function

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.47809639955+0.183199572127j)*x[0] + ((-0.250136507414+0.965318831781j))*x[1]
            ref=(0.113979946068+0.574259201954j)
        else:
            arg=(0.269890288506+0.988587096063j)*x[0] + ((-0.312853967641-0.131721030493j))*x[1] + ((0.806938717502-0.469127011716j))*x[2]
            ref=(0.381987519184+0.193869526927j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedSolution_rank1(self):
        """
        tests integral of rank 1 Data on Function

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(3,),w)
        ref=numpy.zeros((3,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.374912817226-0.609784130463j)*x[0] + ((0.537066854566-0.0330431755369j))*x[1]
            ref[(0,)]=(0.0810770186699-0.321413653j)
            arg[(1,)]=(0.52806933078-0.144452326987j)*x[0] + ((-0.830047324655-0.923078331848j))*x[1]
            ref[(1,)]=(-0.150988996937-0.533765329417j)
            arg[(2,)]=(-0.453536014958+0.854205756859j)*x[0] + ((0.480595099909+0.888257312302j))*x[1]
            ref[(2,)]=(0.0135295424758+0.87123153458j)
        else:
            arg[(0,)]=(0.612833877359+0.373454744576j)*x[0] + ((0.396976940112-0.104748757369j))*x[1] + ((0.259980189464+0.835588471401j))*x[2]
            ref[(0,)]=(0.634895503468+0.552147229304j)
            arg[(1,)]=(-0.73920500524+0.664345223637j)*x[0] + ((0.638209513716+0.575185112599j))*x[1] + ((-0.207548299675+0.667487256288j))*x[2]
            ref[(1,)]=(-0.154271895599+0.953508796262j)
            arg[(2,)]=(-0.219165695879+0.570321107464j)*x[0] + ((-0.0337327269615+0.154130837419j))*x[1] + ((-0.990994404248-0.223028982596j))*x[2]
            ref[(2,)]=(-0.621946413544+0.250711481144j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedSolution_rank2(self):
        """
        tests integral of rank 2 Data on Function

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4),w)
        ref=numpy.zeros((3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.917365948809+0.291406422221j)*x[0] + ((0.0730571909916+0.743325257238j))*x[1]
            ref[(0, 0)]=(0.4952115699+0.51736583973j)
            arg[(0, 1)]=(-0.818494447148-0.729160296561j)*x[0] + ((0.993000516835+0.456448541586j))*x[1]
            ref[(0, 1)]=(0.0872530348437-0.136355877487j)
            arg[(0, 2)]=(0.876798094703-0.576285032622j)*x[0] + ((-0.07448685234-0.286697245646j))*x[1]
            ref[(0, 2)]=(0.401155621181-0.431491139134j)
            arg[(0, 3)]=(0.816410652361-0.787439367398j)*x[0] + ((-0.187384077982+0.63069038916j))*x[1]
            ref[(0, 3)]=(0.31451328719-0.078374489119j)
            arg[(1, 0)]=(0.526288750984-0.257717240817j)*x[0] + ((-0.684913594571-0.482157949397j))*x[1]
            ref[(1, 0)]=(-0.0793124217935-0.369937595107j)
            arg[(1, 1)]=(0.978062736258+0.133010012421j)*x[0] + ((-0.238657826788+0.283995993199j))*x[1]
            ref[(1, 1)]=(0.369702454735+0.20850300281j)
            arg[(1, 2)]=(0.758472866566+0.194607751187j)*x[0] + ((0.468586879599-0.385773592004j))*x[1]
            ref[(1, 2)]=(0.613529873082-0.0955829204084j)
            arg[(1, 3)]=(-0.978761721288+0.35292325251j)*x[0] + ((-0.543495246403-0.210431532589j))*x[1]
            ref[(1, 3)]=(-0.761128483846+0.0712458599606j)
            arg[(2, 0)]=(-0.541175739527-0.739084944249j)*x[0] + ((0.189160024599+0.227485215068j))*x[1]
            ref[(2, 0)]=(-0.176007857464-0.255799864591j)
            arg[(2, 1)]=(-0.359155085915-0.980335627583j)*x[0] + ((-0.633315116712-0.725856743711j))*x[1]
            ref[(2, 1)]=(-0.496235101313-0.853096185647j)
            arg[(2, 2)]=(0.372683734689-0.278291000358j)*x[0] + ((-0.0494417279588-0.0653847892904j))*x[1]
            ref[(2, 2)]=(0.161621003365-0.171837894824j)
            arg[(2, 3)]=(0.695288250571-0.609556986341j)*x[0] + ((0.614620392656-0.658205209271j))*x[1]
            ref[(2, 3)]=(0.654954321613-0.633881097806j)
        else:
            arg[(0, 0)]=(-0.891426872514-0.587926810139j)*x[0] + ((-0.192456008221+0.896092861395j))*x[1] + ((-0.679538480861-0.534836617151j))*x[2]
            ref[(0, 0)]=(-0.881710680798-0.113335282948j)
            arg[(0, 1)]=(-0.297041747227-0.495039627823j)*x[0] + ((-0.544433747154-0.278348348237j))*x[1] + ((-0.618446041422+0.5982378315j))*x[2]
            ref[(0, 1)]=(-0.729960767901-0.0875750722802j)
            arg[(0, 2)]=(-0.339211843573+0.185531520808j)*x[0] + ((-0.785503455436-0.945323432345j))*x[1] + ((-0.267352466648+0.898018980939j))*x[2]
            ref[(0, 2)]=(-0.696033882828+0.0691135347009j)
            arg[(0, 3)]=(-0.0453800747284+0.652417810091j)*x[0] + ((0.212422057052-0.359312339196j))*x[1] + ((-0.630338680373+0.216588069912j))*x[2]
            ref[(0, 3)]=(-0.231648349024+0.254846770404j)
            arg[(1, 0)]=(-0.482614821514+0.044211650196j)*x[0] + ((-0.109014311641-0.855186608453j))*x[1] + ((0.183307790806-0.966066194056j))*x[2]
            ref[(1, 0)]=(-0.204160671174-0.888520576157j)
            arg[(1, 1)]=(-0.694451976764+0.304378197389j)*x[0] + ((0.702539539217-0.980533695687j))*x[1] + ((-0.59363317286-0.644779053401j))*x[2]
            ref[(1, 1)]=(-0.292772805203-0.66046727585j)
            arg[(1, 2)]=(0.801078242944-0.202826814382j)*x[0] + ((0.82401586053-0.30770796068j))*x[1] + ((-0.91539853893+0.690789573725j))*x[2]
            ref[(1, 2)]=(0.354847782272+0.0901273993312j)
            arg[(1, 3)]=(-0.570563811112-0.301772136029j)*x[0] + ((0.990722542331-0.320943100139j))*x[1] + ((0.728078967911+0.923418155448j))*x[2]
            ref[(1, 3)]=(0.574118849565+0.15035145964j)
            arg[(2, 0)]=(-0.077746418305+0.712881433618j)*x[0] + ((0.175381147312+0.514550167302j))*x[1] + ((-0.300530942768-0.024028802557j))*x[2]
            ref[(2, 0)]=(-0.101448106881+0.601701399181j)
            arg[(2, 1)]=(0.0175530852057+0.332650575516j)*x[0] + ((0.014669045161-0.0808454570925j))*x[1] + ((-0.557354507495-0.358127728345j))*x[2]
            ref[(2, 1)]=(-0.262566188564-0.0531613049604j)
            arg[(2, 2)]=(-0.511360591927+0.850596617487j)*x[0] + ((0.362626820691-0.498308663882j))*x[1] + ((0.612575631206-0.787870287578j))*x[2]
            ref[(2, 2)]=(0.231920929985-0.217791166986j)
            arg[(2, 3)]=(-0.900017536529+0.228434876429j)*x[0] + ((-0.551465566079-0.232882508251j))*x[1] + ((-0.756089060389-0.36195250345j))*x[2]
            ref[(2, 3)]=(-1.1037860815-0.183200067636j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedSolution_rank3(self):
        """
        tests integral of rank 3 Data on Function

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(4, 2, 4),w)
        ref=numpy.zeros((4, 2, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.874376503307+0.802470384299j)*x[0] + ((0.724647989123-0.850715589127j))*x[1]
            ref[(0, 0, 0)]=(-0.0748642570919-0.0241226024139j)
            arg[(0, 0, 1)]=(0.75779166808+0.580240241215j)*x[0] + ((0.645782019991-0.297563015771j))*x[1]
            ref[(0, 0, 1)]=(0.701786844036+0.141338612722j)
            arg[(0, 0, 2)]=(-0.731488289502+0.87592178286j)*x[0] + ((-0.962739632451+0.119526994773j))*x[1]
            ref[(0, 0, 2)]=(-0.847113960977+0.497724388817j)
            arg[(0, 0, 3)]=(0.715286282042+0.52428813702j)*x[0] + ((0.432076722818-0.409400341135j))*x[1]
            ref[(0, 0, 3)]=(0.57368150243+0.0574438979427j)
            arg[(0, 1, 0)]=(-0.669488958885+0.28721937095j)*x[0] + ((0.418199464475+0.0111980205183j))*x[1]
            ref[(0, 1, 0)]=(-0.125644747205+0.149208695734j)
            arg[(0, 1, 1)]=(0.620627339028+0.857967127712j)*x[0] + ((-0.221541425752+0.855781992867j))*x[1]
            ref[(0, 1, 1)]=(0.199542956638+0.85687456029j)
            arg[(0, 1, 2)]=(0.840677687426+0.247762842194j)*x[0] + ((-0.908025600875+0.585349586925j))*x[1]
            ref[(0, 1, 2)]=(-0.0336739567244+0.416556214559j)
            arg[(0, 1, 3)]=(0.768980952294-0.304671101527j)*x[0] + ((-0.855944015774-0.506536935574j))*x[1]
            ref[(0, 1, 3)]=(-0.04348153174-0.40560401855j)
            arg[(1, 0, 0)]=(0.855896744936-0.952732324132j)*x[0] + ((0.674186322606-0.832380122335j))*x[1]
            ref[(1, 0, 0)]=(0.765041533771-0.892556223233j)
            arg[(1, 0, 1)]=(0.251657425067-0.972739058923j)*x[0] + ((0.00601679552795+0.113780722326j))*x[1]
            ref[(1, 0, 1)]=(0.128837110298-0.429479168299j)
            arg[(1, 0, 2)]=(-0.364747308335-0.950906129394j)*x[0] + ((0.838697726954+0.217020357444j))*x[1]
            ref[(1, 0, 2)]=(0.236975209309-0.366942885975j)
            arg[(1, 0, 3)]=(-0.103535701163+0.823620776051j)*x[0] + ((-0.23021023525-0.161169737178j))*x[1]
            ref[(1, 0, 3)]=(-0.166872968206+0.331225519437j)
            arg[(1, 1, 0)]=(-0.0484016896677+0.392333515031j)*x[0] + ((0.350724974643-0.719247587321j))*x[1]
            ref[(1, 1, 0)]=(0.151161642488-0.163457036145j)
            arg[(1, 1, 1)]=(0.352919106345+0.228142748324j)*x[0] + ((0.983328868866+0.173318276201j))*x[1]
            ref[(1, 1, 1)]=(0.668123987606+0.200730512263j)
            arg[(1, 1, 2)]=(0.220598943546-0.990119818476j)*x[0] + ((0.131743140307+0.128823117676j))*x[1]
            ref[(1, 1, 2)]=(0.176171041927-0.4306483504j)
            arg[(1, 1, 3)]=(0.858932744496+0.660780366909j)*x[0] + ((0.660885471783+0.92296805634j))*x[1]
            ref[(1, 1, 3)]=(0.759909108139+0.791874211624j)
            arg[(2, 0, 0)]=(0.238459293126+0.41849397556j)*x[0] + ((0.280441188781-0.869850989357j))*x[1]
            ref[(2, 0, 0)]=(0.259450240954-0.225678506898j)
            arg[(2, 0, 1)]=(-0.95417979973-0.181563797561j)*x[0] + ((-0.405803708134-0.0309706979719j))*x[1]
            ref[(2, 0, 1)]=(-0.679991753932-0.106267247767j)
            arg[(2, 0, 2)]=(-0.401130491346-0.309305353062j)*x[0] + ((-0.015879838469-0.770915287736j))*x[1]
            ref[(2, 0, 2)]=(-0.208505164907-0.540110320399j)
            arg[(2, 0, 3)]=(-0.976561643901-0.812847443363j)*x[0] + ((0.0845322669126+0.461198627018j))*x[1]
            ref[(2, 0, 3)]=(-0.446014688494-0.175824408172j)
            arg[(2, 1, 0)]=(0.134482251388+0.632695357024j)*x[0] + ((-0.719114797702+0.560538521588j))*x[1]
            ref[(2, 1, 0)]=(-0.292316273157+0.596616939306j)
            arg[(2, 1, 1)]=(-0.292550120238-0.708342488826j)*x[0] + ((-0.475654395449-0.675444072489j))*x[1]
            ref[(2, 1, 1)]=(-0.384102257844-0.691893280658j)
            arg[(2, 1, 2)]=(0.598879392558-0.483583872873j)*x[0] + ((0.970894361586+0.0178982889405j))*x[1]
            ref[(2, 1, 2)]=(0.784886877072-0.232842791966j)
            arg[(2, 1, 3)]=(-0.464214223639-0.177082222316j)*x[0] + ((-0.89584455568+0.702528260865j))*x[1]
            ref[(2, 1, 3)]=(-0.680029389659+0.262723019274j)
            arg[(3, 0, 0)]=(-0.572919046743-0.468384488651j)*x[0] + ((0.738057007468+0.364598622025j))*x[1]
            ref[(3, 0, 0)]=(0.0825689803622-0.0518929333132j)
            arg[(3, 0, 1)]=(0.502263792212+0.563500546332j)*x[0] + ((-0.25340274461-0.194862240382j))*x[1]
            ref[(3, 0, 1)]=(0.124430523801+0.184319152975j)
            arg[(3, 0, 2)]=(-0.95360040984+0.0278846644295j)*x[0] + ((-0.0512134037523+0.702023908904j))*x[1]
            ref[(3, 0, 2)]=(-0.502406906796+0.364954286667j)
            arg[(3, 0, 3)]=(-0.444424350534-0.307473732509j)*x[0] + ((-0.0662006480832-0.599656842543j))*x[1]
            ref[(3, 0, 3)]=(-0.255312499308-0.453565287526j)
            arg[(3, 1, 0)]=(0.282763107093+0.957510794033j)*x[0] + ((0.774818923345+0.0729543475303j))*x[1]
            ref[(3, 1, 0)]=(0.528791015219+0.515232570782j)
            arg[(3, 1, 1)]=(0.434852665618+0.495466462612j)*x[0] + ((0.104795496821-0.583879980065j))*x[1]
            ref[(3, 1, 1)]=(0.269824081219-0.0442067587264j)
            arg[(3, 1, 2)]=(0.258807310417+0.425867972729j)*x[0] + ((0.290703253128-0.980393613045j))*x[1]
            ref[(3, 1, 2)]=(0.274755281773-0.277262820158j)
            arg[(3, 1, 3)]=(0.781081290415-0.748885669447j)*x[0] + ((-0.655826768304-0.1287875948j))*x[1]
            ref[(3, 1, 3)]=(0.0626272610553-0.438836632123j)
        else:
            arg[(0, 0, 0)]=(-0.573433246861+0.0294236552004j)*x[0] + ((-0.252038265157+0.935689339955j))*x[1] + ((-0.307414598128+0.186431994002j))*x[2]
            ref[(0, 0, 0)]=(-0.566443055073+0.575772494579j)
            arg[(0, 0, 1)]=(-0.796061051705+0.359363891502j)*x[0] + ((-0.0823385174398-0.403632385289j))*x[1] + ((-0.823348760996+0.897490244203j))*x[2]
            ref[(0, 0, 1)]=(-0.85087416507+0.426610875208j)
            arg[(0, 0, 2)]=(0.937007493001-0.867435886338j)*x[0] + ((-0.0745964634919-0.0821840217539j))*x[1] + ((0.248877762196-0.521452795111j))*x[2]
            ref[(0, 0, 2)]=(0.555644395852-0.735536351601j)
            arg[(0, 0, 3)]=(-0.499167320472+0.958105441674j)*x[0] + ((0.0739111518137+0.160720252762j))*x[1] + ((-0.542164405964-0.0489107153471j))*x[2]
            ref[(0, 0, 3)]=(-0.483710287311+0.534957489544j)
            arg[(0, 1, 0)]=(0.0239094597021-0.740226490306j)*x[0] + ((0.938177325774+0.24784517691j))*x[1] + ((-0.114198529872-0.129803607443j))*x[2]
            ref[(0, 1, 0)]=(0.423944127802-0.311092460419j)
            arg[(0, 1, 1)]=(-0.949967778364-0.60954109481j)*x[0] + ((0.203132457939+0.723532220326j))*x[1] + ((-0.828227391632-0.555428027828j))*x[2]
            ref[(0, 1, 1)]=(-0.787531356028-0.220718451157j)
            arg[(0, 1, 2)]=(0.456628556362+0.961980734652j)*x[0] + ((-0.768758159449+0.715365661203j))*x[1] + ((-0.850103499742+0.362642189642j))*x[2]
            ref[(0, 1, 2)]=(-0.581116551414+1.01999429275j)
            arg[(0, 1, 3)]=(-0.698298427729+0.53417594215j)*x[0] + ((0.644862226388-0.355191530375j))*x[1] + ((-0.299211879476-0.46471013495j))*x[2]
            ref[(0, 1, 3)]=(-0.176324040409-0.142862861587j)
            arg[(1, 0, 0)]=(-0.249777834568-0.543987119547j)*x[0] + ((-0.887806485402+0.752555717508j))*x[1] + ((-0.630858220212+0.208485683913j))*x[2]
            ref[(1, 0, 0)]=(-0.884221270091+0.208527140937j)
            arg[(1, 0, 1)]=(-0.880802792761-0.291796173425j)*x[0] + ((0.828209127446+0.073266668044j))*x[1] + ((-0.959944855265+0.152142612925j))*x[2]
            ref[(1, 0, 1)]=(-0.506269260289-0.0331934462282j)
            arg[(1, 0, 2)]=(0.334007026282-0.370141535629j)*x[0] + ((0.269887120652+0.415007979357j))*x[1] + ((-0.22478331861+0.722144018356j))*x[2]
            ref[(1, 0, 2)]=(0.189555414162+0.383505231042j)
            arg[(1, 0, 3)]=(-0.0648378994295+0.00393738176447j)*x[0] + ((-0.185047073269-0.887564514435j))*x[1] + ((0.825747870229-0.0856441921408j))*x[2]
            ref[(1, 0, 3)]=(0.287931448765-0.484635662406j)
            arg[(1, 1, 0)]=(-0.876020891241+0.445130634699j)*x[0] + ((-0.204059463125-0.129732309983j))*x[1] + ((0.431581872151+0.585626331583j))*x[2]
            ref[(1, 1, 0)]=(-0.324249241107+0.45051232815j)
            arg[(1, 1, 1)]=(0.044178760902-0.133776788943j)*x[0] + ((0.181995344209-0.361122202254j))*x[1] + ((0.0339906178328-0.28536654311j))*x[2]
            ref[(1, 1, 1)]=(0.130082361472-0.390132767153j)
            arg[(1, 1, 2)]=(0.267899654628-0.57701426588j)*x[0] + ((0.761053538404-0.111408486404j))*x[1] + ((0.35764380678-0.061397868986j))*x[2]
            ref[(1, 1, 2)]=(0.693298499906-0.374910310635j)
            arg[(1, 1, 3)]=(0.360568924278-0.105656727798j)*x[0] + ((-0.181361457823-0.00740440481005j))*x[1] + ((-0.841512210811+0.487000690018j))*x[2]
            ref[(1, 1, 3)]=(-0.331152372178+0.186969778705j)
            arg[(2, 0, 0)]=(0.0553984655993+0.0659213327096j)*x[0] + ((-0.863691086318-0.227189589679j))*x[1] + ((0.165832030909+0.987311999025j))*x[2]
            ref[(2, 0, 0)]=(-0.321230294905+0.413021871028j)
            arg[(2, 0, 1)]=(0.559828821449+0.184044421483j)*x[0] + ((0.81151970521-0.803053427418j))*x[1] + ((0.0965743322869-0.0487570553671j))*x[2]
            ref[(2, 0, 1)]=(0.733961429473-0.333883030651j)
            arg[(2, 0, 2)]=(-0.334829461292+0.552023094241j)*x[0] + ((-0.135829409462-0.67896744119j))*x[1] + ((-0.350506531037-0.34270151793j))*x[2]
            ref[(2, 0, 2)]=(-0.410582700896-0.23482293244j)
            arg[(2, 0, 3)]=(-0.0175511646734+0.583166087333j)*x[0] + ((-0.294952692253+0.740391269553j))*x[1] + ((-0.0110082487503+0.12140212298j))*x[2]
            ref[(2, 0, 3)]=(-0.161756052839+0.722479739933j)
            arg[(2, 1, 0)]=(0.692558715464-0.484693600141j)*x[0] + ((-0.970337322397+0.166420397527j))*x[1] + ((-0.817064143419-0.788802132448j))*x[2]
            ref[(2, 1, 0)]=(-0.547421375176-0.553537667531j)
            arg[(2, 1, 1)]=(-0.0591482876425+0.406334831083j)*x[0] + ((-0.618872129571-0.367470133662j))*x[1] + ((0.370555152467-0.710288198647j))*x[2]
            ref[(2, 1, 1)]=(-0.153732632373-0.335711750614j)
            arg[(2, 1, 2)]=(0.510121521294-0.244160094524j)*x[0] + ((0.622263953089-0.867713094557j))*x[1] + ((0.289048678992-0.266717454395j))*x[2]
            ref[(2, 1, 2)]=(0.710717076687-0.689295321738j)
            arg[(2, 1, 3)]=(0.177103901509-0.195630571364j)*x[0] + ((0.261581249653-0.471113316015j))*x[1] + ((0.51855049736+0.238408983109j))*x[2]
            ref[(2, 1, 3)]=(0.478617824261-0.214167452135j)
            arg[(3, 0, 0)]=(0.940088565011-0.838346919153j)*x[0] + ((-0.653009021288-0.215893890639j))*x[1] + ((0.685601434536+0.890090406926j))*x[2]
            ref[(3, 0, 0)]=(0.48634048913-0.0820752014332j)
            arg[(3, 0, 1)]=(-0.880687979867+0.320513882876j)*x[0] + ((0.232810382734+0.112586539379j))*x[1] + ((0.0845558231884+0.640597248625j))*x[2]
            ref[(3, 0, 1)]=(-0.281660886972+0.53684883544j)
            arg[(3, 0, 2)]=(0.99818462046+0.982450174237j)*x[0] + ((0.36432433361-0.606471720965j))*x[1] + ((-0.255169913236-0.111403681717j))*x[2]
            ref[(3, 0, 2)]=(0.553669520417+0.132287385777j)
            arg[(3, 0, 3)]=(0.964503258659-0.592792675383j)*x[0] + ((0.651266378803+0.634093219123j))*x[1] + ((-0.540874038621-0.664220579334j))*x[2]
            ref[(3, 0, 3)]=(0.53744779942-0.311460017796j)
            arg[(3, 1, 0)]=(-0.653387488533+0.864141665506j)*x[0] + ((-0.531529558374-0.140612916729j))*x[1] + ((-0.524744724798+0.935430623828j))*x[2]
            ref[(3, 1, 0)]=(-0.854830885853+0.829479686302j)
            arg[(3, 1, 1)]=(0.659772909202-0.853532310511j)*x[0] + ((-0.0565860068351+0.904239785981j))*x[1] + ((-0.816124869447+0.890934521234j))*x[2]
            ref[(3, 1, 1)]=(-0.10646898354+0.470820998353j)
            arg[(3, 1, 2)]=(-0.818902626288-0.316182856524j)*x[0] + ((-0.823343364583-0.098554487781j))*x[1] + ((-0.672581102766+0.0110395176183j))*x[2]
            ref[(3, 1, 2)]=(-1.15741354682-0.201848913343j)
            arg[(3, 1, 3)]=(-0.555875989923+0.56811374179j)*x[0] + ((-0.788577998445-0.705129633255j))*x[1] + ((0.608980994226-0.560257470402j))*x[2]
            ref[(3, 1, 3)]=(-0.367736497071-0.348636680934j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 2, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedSolution_rank4(self):
        """
        tests integral of rank 4 Data on Function

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 4, 4, 2),w)
        ref=numpy.zeros((2, 4, 4, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(0.190398670262-0.878046464743j)*x[0] + ((0.603310171679+0.177380232184j))*x[1]
            ref[(0, 0, 0, 0)]=(0.396854420971-0.35033311628j)
            arg[(0, 0, 0, 1)]=(-0.75423896089-0.431849142902j)*x[0] + ((-0.68202226916-0.975530518013j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.718130615025-0.703689830458j)
            arg[(0, 0, 1, 0)]=(-0.726007693614-0.401156624409j)*x[0] + ((0.861199797863-0.624107817094j))*x[1]
            ref[(0, 0, 1, 0)]=(0.0675960521241-0.512632220751j)
            arg[(0, 0, 1, 1)]=(0.447233555836-0.589138695969j)*x[0] + ((-0.434864110963-0.773609454408j))*x[1]
            ref[(0, 0, 1, 1)]=(0.00618472243671-0.681374075189j)
            arg[(0, 0, 2, 0)]=(-0.997877968023-0.544835318782j)*x[0] + ((-0.519543486512+0.686671672307j))*x[1]
            ref[(0, 0, 2, 0)]=(-0.758710727267+0.0709181767621j)
            arg[(0, 0, 2, 1)]=(-0.26694528296-0.349863850903j)*x[0] + ((-0.204963506147-0.618133318451j))*x[1]
            ref[(0, 0, 2, 1)]=(-0.235954394553-0.483998584677j)
            arg[(0, 0, 3, 0)]=(0.701936732415-0.270357009383j)*x[0] + ((0.832370957115-0.710147692903j))*x[1]
            ref[(0, 0, 3, 0)]=(0.767153844765-0.490252351143j)
            arg[(0, 0, 3, 1)]=(0.323782990155-0.583842941249j)*x[0] + ((-0.697777381392-0.542066129549j))*x[1]
            ref[(0, 0, 3, 1)]=(-0.186997195618-0.562954535399j)
            arg[(0, 1, 0, 0)]=(-0.915251728781+0.904094245094j)*x[0] + ((-0.24386377229+0.271582064589j))*x[1]
            ref[(0, 1, 0, 0)]=(-0.579557750535+0.587838154841j)
            arg[(0, 1, 0, 1)]=(-0.850274006717+0.750601043425j)*x[0] + ((-0.549316133635+0.425007780958j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.699795070176+0.587804412192j)
            arg[(0, 1, 1, 0)]=(0.382983812028-0.958961499511j)*x[0] + ((0.873937772223+0.344998884631j))*x[1]
            ref[(0, 1, 1, 0)]=(0.628460792126-0.30698130744j)
            arg[(0, 1, 1, 1)]=(0.350613846639+0.611300093992j)*x[0] + ((-0.264816201704-0.926846399667j))*x[1]
            ref[(0, 1, 1, 1)]=(0.0428988224675-0.157773152838j)
            arg[(0, 1, 2, 0)]=(0.179437385038+0.499586089205j)*x[0] + ((0.72276757284-0.0703423687718j))*x[1]
            ref[(0, 1, 2, 0)]=(0.451102478939+0.214621860217j)
            arg[(0, 1, 2, 1)]=(-0.287265181928-0.567445829335j)*x[0] + ((0.226629153611+0.0277027304622j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.0303180141584-0.269871549436j)
            arg[(0, 1, 3, 0)]=(-0.109313952664+0.324596805509j)*x[0] + ((0.59151656883+0.0783121319822j))*x[1]
            ref[(0, 1, 3, 0)]=(0.241101308083+0.201454468745j)
            arg[(0, 1, 3, 1)]=(-0.203642495423+0.544236419068j)*x[0] + ((0.0194180291539+0.621163746346j))*x[1]
            ref[(0, 1, 3, 1)]=(-0.0921122331345+0.582700082707j)
            arg[(0, 2, 0, 0)]=(0.96702799193-0.517578344687j)*x[0] + ((-0.0213324990425+0.433165775475j))*x[1]
            ref[(0, 2, 0, 0)]=(0.472847746444-0.0422062846057j)
            arg[(0, 2, 0, 1)]=(-0.519106379254-0.108033845064j)*x[0] + ((0.343758712691+0.449062005901j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.0876738332816+0.170514080419j)
            arg[(0, 2, 1, 0)]=(0.276977590904-0.870287966833j)*x[0] + ((0.886804678856-0.805474607326j))*x[1]
            ref[(0, 2, 1, 0)]=(0.58189113488-0.83788128708j)
            arg[(0, 2, 1, 1)]=(0.446813745055-0.97880614869j)*x[0] + ((0.81007366393+0.153505135169j))*x[1]
            ref[(0, 2, 1, 1)]=(0.628443704493-0.412650506761j)
            arg[(0, 2, 2, 0)]=(-0.844069791125+0.221472247278j)*x[0] + ((0.926854411591+0.907752129614j))*x[1]
            ref[(0, 2, 2, 0)]=(0.0413923102334+0.564612188446j)
            arg[(0, 2, 2, 1)]=(0.711735349242-0.888678325902j)*x[0] + ((0.241989613621-0.651537340411j))*x[1]
            ref[(0, 2, 2, 1)]=(0.476862481432-0.770107833157j)
            arg[(0, 2, 3, 0)]=(-0.282194865768-0.915829549935j)*x[0] + ((-0.311246135083+0.721520592651j))*x[1]
            ref[(0, 2, 3, 0)]=(-0.296720500425-0.0971544786424j)
            arg[(0, 2, 3, 1)]=(0.393687191763+0.761964824938j)*x[0] + ((-0.515379302785+0.117807637063j))*x[1]
            ref[(0, 2, 3, 1)]=(-0.0608460555111+0.439886231001j)
            arg[(0, 3, 0, 0)]=(-0.491449388038+0.352585068094j)*x[0] + ((0.636719560032+0.581390836191j))*x[1]
            ref[(0, 3, 0, 0)]=(0.0726350859968+0.466987952142j)
            arg[(0, 3, 0, 1)]=(-0.0717000517334-0.0643158766579j)*x[0] + ((0.932468252064-0.750890248706j))*x[1]
            ref[(0, 3, 0, 1)]=(0.430384100165-0.407603062682j)
            arg[(0, 3, 1, 0)]=(-0.683894717095-0.592507325754j)*x[0] + ((-0.907049537243-0.695748851066j))*x[1]
            ref[(0, 3, 1, 0)]=(-0.795472127169-0.64412808841j)
            arg[(0, 3, 1, 1)]=(-0.675084193485+0.245663949267j)*x[0] + ((0.979222838339-0.0643346442372j))*x[1]
            ref[(0, 3, 1, 1)]=(0.152069322427+0.0906646525149j)
            arg[(0, 3, 2, 0)]=(0.875757799316-0.252748038209j)*x[0] + ((-0.126745308608-0.196635986227j))*x[1]
            ref[(0, 3, 2, 0)]=(0.374506245354-0.224692012218j)
            arg[(0, 3, 2, 1)]=(0.741840167883+0.0524632325329j)*x[0] + ((-0.76182046199+0.222013903909j))*x[1]
            ref[(0, 3, 2, 1)]=(-0.00999014705345+0.137238568221j)
            arg[(0, 3, 3, 0)]=(-0.500888616128-0.49963978121j)*x[0] + ((0.728984726245-0.109396380517j))*x[1]
            ref[(0, 3, 3, 0)]=(0.114048055059-0.304518080864j)
            arg[(0, 3, 3, 1)]=(0.175917777192-0.577983247351j)*x[0] + ((0.603624400349+0.550509104837j))*x[1]
            ref[(0, 3, 3, 1)]=(0.389771088771-0.0137370712571j)
            arg[(1, 0, 0, 0)]=(-0.437967508641-0.926819163398j)*x[0] + ((-0.590065680602+0.838870318446j))*x[1]
            ref[(1, 0, 0, 0)]=(-0.514016594622-0.0439744224761j)
            arg[(1, 0, 0, 1)]=(-0.736617347616-0.414800563479j)*x[0] + ((0.177201815282+0.464757925685j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.279707766167+0.0249786811032j)
            arg[(1, 0, 1, 0)]=(0.739684138033+0.888816501844j)*x[0] + ((-0.372278015313-0.0293074111159j))*x[1]
            ref[(1, 0, 1, 0)]=(0.18370306136+0.429754545364j)
            arg[(1, 0, 1, 1)]=(-0.274415907826+0.804549233541j)*x[0] + ((0.165634298653-0.731504463783j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.0543908045861+0.0365223848792j)
            arg[(1, 0, 2, 0)]=(0.893063856296+0.268526291784j)*x[0] + ((-0.675154522295-0.47993741799j))*x[1]
            ref[(1, 0, 2, 0)]=(0.108954667001-0.105705563103j)
            arg[(1, 0, 2, 1)]=(-0.321673957833-0.129596131771j)*x[0] + ((-0.286898174709-0.428034920434j))*x[1]
            ref[(1, 0, 2, 1)]=(-0.304286066271-0.278815526102j)
            arg[(1, 0, 3, 0)]=(0.889175443474-0.575510160222j)*x[0] + ((0.531190160119+0.62245900086j))*x[1]
            ref[(1, 0, 3, 0)]=(0.710182801796+0.0234744203188j)
            arg[(1, 0, 3, 1)]=(0.713202006857-0.756609957215j)*x[0] + ((0.00355776044819+0.540487326553j))*x[1]
            ref[(1, 0, 3, 1)]=(0.358379883653-0.108061315331j)
            arg[(1, 1, 0, 0)]=(-0.00419869341741+0.952237240326j)*x[0] + ((0.280317342175-0.92997008031j))*x[1]
            ref[(1, 1, 0, 0)]=(0.138059324379+0.0111335800079j)
            arg[(1, 1, 0, 1)]=(0.370336327478-0.710680053824j)*x[0] + ((-0.260825512172+0.975740665432j))*x[1]
            ref[(1, 1, 0, 1)]=(0.054755407653+0.132530305804j)
            arg[(1, 1, 1, 0)]=(0.565748579075-0.0136551514077j)*x[0] + ((0.136238492685+0.699222835876j))*x[1]
            ref[(1, 1, 1, 0)]=(0.35099353588+0.342783842234j)
            arg[(1, 1, 1, 1)]=(0.292662584308-0.903381038494j)*x[0] + ((0.129301207683+0.522107997613j))*x[1]
            ref[(1, 1, 1, 1)]=(0.210981895996-0.190636520441j)
            arg[(1, 1, 2, 0)]=(0.540644953167+0.781660559899j)*x[0] + ((0.0598984634647-0.465744433233j))*x[1]
            ref[(1, 1, 2, 0)]=(0.300271708316+0.157958063333j)
            arg[(1, 1, 2, 1)]=(0.602494377557-0.0297750492727j)*x[0] + ((0.603165189074+0.579830120022j))*x[1]
            ref[(1, 1, 2, 1)]=(0.602829783316+0.275027535375j)
            arg[(1, 1, 3, 0)]=(-0.972238594117+0.985262291978j)*x[0] + ((0.680542470614-0.426886992492j))*x[1]
            ref[(1, 1, 3, 0)]=(-0.145848061751+0.279187649743j)
            arg[(1, 1, 3, 1)]=(-0.45777851701+0.143648852891j)*x[0] + ((-0.348152472101-0.946006590289j))*x[1]
            ref[(1, 1, 3, 1)]=(-0.402965494556-0.401178868699j)
            arg[(1, 2, 0, 0)]=(-0.998365618959+0.30618221986j)*x[0] + ((-0.374796892752-0.956036523902j))*x[1]
            ref[(1, 2, 0, 0)]=(-0.686581255856-0.324927152021j)
            arg[(1, 2, 0, 1)]=(0.560629214643-0.274667853594j)*x[0] + ((0.525437789431-0.105523419309j))*x[1]
            ref[(1, 2, 0, 1)]=(0.543033502037-0.190095636451j)
            arg[(1, 2, 1, 0)]=(-0.514976611071-0.77509230783j)*x[0] + ((0.810767238801+0.671597013309j))*x[1]
            ref[(1, 2, 1, 0)]=(0.147895313865-0.0517476472608j)
            arg[(1, 2, 1, 1)]=(0.578179059718+0.733376640105j)*x[0] + ((0.581500178191-0.496262898598j))*x[1]
            ref[(1, 2, 1, 1)]=(0.579839618954+0.118556870754j)
            arg[(1, 2, 2, 0)]=(-0.630543048101-0.000956751930941j)*x[0] + ((0.792215035997-0.133421700946j))*x[1]
            ref[(1, 2, 2, 0)]=(0.0808359939479-0.0671892264384j)
            arg[(1, 2, 2, 1)]=(0.383880846631+0.31688207058j)*x[0] + ((0.713626253514-0.26469434894j))*x[1]
            ref[(1, 2, 2, 1)]=(0.548753550073+0.02609386082j)
            arg[(1, 2, 3, 0)]=(-0.585645653371+0.986431030466j)*x[0] + ((0.910775098194+0.908339577236j))*x[1]
            ref[(1, 2, 3, 0)]=(0.162564722411+0.947385303851j)
            arg[(1, 2, 3, 1)]=(0.599906893905-0.713862517313j)*x[0] + ((0.435672174864+0.419281596832j))*x[1]
            ref[(1, 2, 3, 1)]=(0.517789534384-0.147290460241j)
            arg[(1, 3, 0, 0)]=(-0.133700918509+0.046862811179j)*x[0] + ((-0.766462589837+0.273876511332j))*x[1]
            ref[(1, 3, 0, 0)]=(-0.450081754173+0.160369661255j)
            arg[(1, 3, 0, 1)]=(-0.500455730026+0.84836624145j)*x[0] + ((-0.933513305656+0.24241607866j))*x[1]
            ref[(1, 3, 0, 1)]=(-0.716984517841+0.545391160055j)
            arg[(1, 3, 1, 0)]=(0.137348387332-0.0962528922707j)*x[0] + ((-0.824733754566-0.827193215926j))*x[1]
            ref[(1, 3, 1, 0)]=(-0.343692683617-0.461723054098j)
            arg[(1, 3, 1, 1)]=(0.647146564757-0.651653765831j)*x[0] + ((-0.0215966663366-0.437003047566j))*x[1]
            ref[(1, 3, 1, 1)]=(0.31277494921-0.544328406698j)
            arg[(1, 3, 2, 0)]=(-0.828202623657+0.567076686797j)*x[0] + ((-0.646816802963+0.701390645808j))*x[1]
            ref[(1, 3, 2, 0)]=(-0.73750971331+0.634233666302j)
            arg[(1, 3, 2, 1)]=(0.794073596129+0.421655922139j)*x[0] + ((0.531860543047-0.63894092314j))*x[1]
            ref[(1, 3, 2, 1)]=(0.662967069588-0.1086425005j)
            arg[(1, 3, 3, 0)]=(-0.85773111625-0.0478954933975j)*x[0] + ((0.358801689559+0.0875536184467j))*x[1]
            ref[(1, 3, 3, 0)]=(-0.249464713345+0.0198290625246j)
            arg[(1, 3, 3, 1)]=(0.726475695917+0.0648836463247j)*x[0] + ((-0.529195676693-0.0273843006825j))*x[1]
            ref[(1, 3, 3, 1)]=(0.0986400096122+0.0187496728211j)
        else:
            arg[(0, 0, 0, 0)]=(-0.88911714083+0.530334945349j)*x[0] + ((-0.646062028025+0.288960792547j))*x[1] + ((-0.578408257505+0.963357175942j))*x[2]
            ref[(0, 0, 0, 0)]=(-1.05679371318+0.891326456919j)
            arg[(0, 0, 0, 1)]=(-0.896110981902-0.17935439055j)*x[0] + ((-0.64436117975-0.955276197111j))*x[1] + ((-0.612146333882-0.746692154954j))*x[2]
            ref[(0, 0, 0, 1)]=(-1.07630924777-0.940661371308j)
            arg[(0, 0, 1, 0)]=(0.757133518974-0.384954630295j)*x[0] + ((-0.316790909716+0.22906783726j))*x[1] + ((0.955060569261+0.0446450554261j))*x[2]
            ref[(0, 0, 1, 0)]=(0.69770158926-0.0556208688043j)
            arg[(0, 0, 1, 1)]=(0.340653753819+0.64593921917j)*x[0] + ((0.729684463135-0.545779996376j))*x[1] + ((0.541706738392+0.01730515619j))*x[2]
            ref[(0, 0, 1, 1)]=(0.806022477673+0.0587321894923j)
            arg[(0, 0, 2, 0)]=(-0.616722729631-0.430815959353j)*x[0] + ((0.833427599652+0.29297631202j))*x[1] + ((0.821200455191-0.409096801132j))*x[2]
            ref[(0, 0, 2, 0)]=(0.518952662606-0.273468224232j)
            arg[(0, 0, 2, 1)]=(-0.285659707574+0.961161364922j)*x[0] + ((0.385228633991-0.201603610091j))*x[1] + ((-0.754636597706+0.98417011505j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.327533835645+0.87186393494j)
            arg[(0, 0, 3, 0)]=(-0.75208420978-0.18706707654j)*x[0] + ((-0.883726226205+0.545603664203j))*x[1] + ((0.291925059349+0.75271829547j))*x[2]
            ref[(0, 0, 3, 0)]=(-0.671942688318+0.555627441566j)
            arg[(0, 0, 3, 1)]=(0.959900031889+0.328122678138j)*x[0] + ((0.830314797539-0.347666332403j))*x[1] + ((0.842649354943+0.291113273668j))*x[2]
            ref[(0, 0, 3, 1)]=(1.31643209219+0.135784809702j)
            arg[(0, 1, 0, 0)]=(0.808671415358+0.537542360152j)*x[0] + ((-0.629172489145-0.972125714168j))*x[1] + ((-0.465954320296+0.498558756585j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.143227697041+0.0319877012844j)
            arg[(0, 1, 0, 1)]=(0.519114926662-0.875312260813j)*x[0] + ((0.0126590646679+0.156448259947j))*x[1] + ((-0.730747718455-0.961917719368j))*x[2]
            ref[(0, 1, 0, 1)]=(-0.099486863563-0.840390860117j)
            arg[(0, 1, 1, 0)]=(0.016142045454+0.237015876696j)*x[0] + ((-0.143003444868-0.454379324873j))*x[1] + ((-0.474615979143+0.895103551219j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.300738689278+0.338870051521j)
            arg[(0, 1, 1, 1)]=(-0.112818513659-0.988799582537j)*x[0] + ((-0.263853215773-0.976686188276j))*x[1] + ((-0.0543845463952+0.977253681052j))*x[2]
            ref[(0, 1, 1, 1)]=(-0.215528137914-0.49411604488j)
            arg[(0, 1, 2, 0)]=(-0.289035940591+0.191849908895j)*x[0] + ((-0.833486212739-0.33514646194j))*x[1] + ((0.123795893057+0.458678712173j))*x[2]
            ref[(0, 1, 2, 0)]=(-0.499363130136+0.157691079564j)
            arg[(0, 1, 2, 1)]=(0.103715545016-0.590189646618j)*x[0] + ((0.248662479643+0.900806390971j))*x[1] + ((0.000159238852312-0.070614213964j))*x[2]
            ref[(0, 1, 2, 1)]=(0.176268631756+0.120001265194j)
            arg[(0, 1, 3, 0)]=(0.941507710555-0.720572618313j)*x[0] + ((0.343082681402+0.208487363167j))*x[1] + ((-0.305936026285+0.995437305308j))*x[2]
            ref[(0, 1, 3, 0)]=(0.489327182836+0.241676025081j)
            arg[(0, 1, 3, 1)]=(-0.236657751437-0.705616933561j)*x[0] + ((-0.00820963491272-0.0444202229477j))*x[1] + ((-0.517532848098+0.244943081023j))*x[2]
            ref[(0, 1, 3, 1)]=(-0.381200117224-0.252547037743j)
            arg[(0, 2, 0, 0)]=(0.765658850372-0.1947629928j)*x[0] + ((-0.690859339185-0.991847021954j))*x[1] + ((0.194493578328+0.130462186031j))*x[2]
            ref[(0, 2, 0, 0)]=(0.134646544757-0.528073914362j)
            arg[(0, 2, 0, 1)]=(0.590091642575+0.38839738746j)*x[0] + ((-0.963946839207+0.321675926154j))*x[1] + ((-0.569691558255+0.134274707375j))*x[2]
            ref[(0, 2, 0, 1)]=(-0.471773377444+0.422174010494j)
            arg[(0, 2, 1, 0)]=(-0.592026315845-0.0307225948257j)*x[0] + ((-0.639476850172+0.747113619762j))*x[1] + ((-0.933192246924-0.0654011752544j))*x[2]
            ref[(0, 2, 1, 0)]=(-1.08234770647+0.325494924841j)
            arg[(0, 2, 1, 1)]=(-0.132164445414-0.676087040381j)*x[0] + ((-0.517288755354+0.254384365819j))*x[1] + ((0.660896980276-0.831040402318j))*x[2]
            ref[(0, 2, 1, 1)]=(0.00572188975385-0.62637153844j)
            arg[(0, 2, 2, 0)]=(0.83473139732+0.693181158032j)*x[0] + ((-0.573845555645+0.274425350603j))*x[1] + ((0.759359965524+0.741512097994j))*x[2]
            ref[(0, 2, 2, 0)]=(0.510122903599+0.854559303314j)
            arg[(0, 2, 2, 1)]=(-0.835114971896-0.496261651975j)*x[0] + ((-0.459519013835-0.137231406018j))*x[1] + ((-0.877570661862+0.026217819946j))*x[2]
            ref[(0, 2, 2, 1)]=(-1.0861023238-0.303637619024j)
            arg[(0, 2, 3, 0)]=(0.789251773621+0.950488217138j)*x[0] + ((-0.981479146734-0.314393999904j))*x[1] + ((0.272232073398-0.73809398178j))*x[2]
            ref[(0, 2, 3, 0)]=(0.0400023501422-0.050999882273j)
            arg[(0, 2, 3, 1)]=(0.364437916686-0.927291736931j)*x[0] + ((-0.635989722997-0.0187609908085j))*x[1] + ((0.132967787297+0.894659059523j))*x[2]
            ref[(0, 2, 3, 1)]=(-0.0692920095069-0.0256968341082j)
            arg[(0, 3, 0, 0)]=(0.256465712208-0.0820275145412j)*x[0] + ((0.871326386759-0.470910577575j))*x[1] + ((-0.628520530034+0.352798378766j))*x[2]
            ref[(0, 3, 0, 0)]=(0.249635784467-0.100069856675j)
            arg[(0, 3, 0, 1)]=(0.920816095814-0.983695239166j)*x[0] + ((0.894810900281+0.332305071583j))*x[1] + ((0.638225219933+0.86787050038j))*x[2]
            ref[(0, 3, 0, 1)]=(1.22692610801+0.108240166398j)
            arg[(0, 3, 1, 0)]=(-0.796682762201-0.192516595784j)*x[0] + ((0.466047755293-0.760287520161j))*x[1] + ((0.606713219787-0.202751231405j))*x[2]
            ref[(0, 3, 1, 0)]=(0.13803910644-0.577777673675j)
            arg[(0, 3, 1, 1)]=(-0.98285808391-0.253473064881j)*x[0] + ((-0.338156654938+0.105590312949j))*x[1] + ((-0.12584446328+0.784277661916j))*x[2]
            ref[(0, 3, 1, 1)]=(-0.723429601064+0.318197454992j)
            arg[(0, 3, 2, 0)]=(0.556724574791-0.394014088577j)*x[0] + ((-0.56104501792-0.549299571283j))*x[1] + ((-0.000712339979012+0.0592730236885j))*x[2]
            ref[(0, 3, 2, 0)]=(-0.00251639155403-0.442020318086j)
            arg[(0, 3, 2, 1)]=(0.859000240298-0.151016309515j)*x[0] + ((-0.19945275825+0.882947248039j))*x[1] + ((0.548829626265+0.587342204093j))*x[2]
            ref[(0, 3, 2, 1)]=(0.604188554156+0.659636571309j)
            arg[(0, 3, 3, 0)]=(0.633561027873-0.497457411653j)*x[0] + ((0.622360777032+0.94809790404j))*x[1] + ((0.387789040707+0.847279724963j))*x[2]
            ref[(0, 3, 3, 0)]=(0.821855422806+0.648960108676j)
            arg[(0, 3, 3, 1)]=(-0.477174778638+0.878650003311j)*x[0] + ((0.412863398901-0.110326784373j))*x[1] + ((-0.355785427689-0.869121758274j))*x[2]
            ref[(0, 3, 3, 1)]=(-0.210048403713-0.0503992696685j)
            arg[(1, 0, 0, 0)]=(-0.619978647279-0.787251917669j)*x[0] + ((-0.932171567044-0.860175642379j))*x[1] + ((0.397756934146+0.607006775132j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.577196640088-0.520210392458j)
            arg[(1, 0, 0, 1)]=(0.583325644471-0.314688112043j)*x[0] + ((-0.90430866144-0.983077205383j))*x[1] + ((0.985608813225-0.857867397976j))*x[2]
            ref[(1, 0, 0, 1)]=(0.332312898128-1.0778163577j)
            arg[(1, 0, 1, 0)]=(0.558395070959-0.57245270738j)*x[0] + ((0.443130618798+0.0532804525069j))*x[1] + ((-0.0230531571472-0.878970910363j))*x[2]
            ref[(1, 0, 1, 0)]=(0.489236266305-0.699071582618j)
            arg[(1, 0, 1, 1)]=(0.28996490291-0.958311673923j)*x[0] + ((0.444292523835+0.234023661894j))*x[1] + ((0.493495346446+0.528687690104j))*x[2]
            ref[(1, 0, 1, 1)]=(0.613876386596-0.0978001609623j)
            arg[(1, 0, 2, 0)]=(0.860195175538-0.951982219814j)*x[0] + ((0.280993107047-0.967399982583j))*x[1] + ((0.119299944505-0.619527681988j))*x[2]
            ref[(1, 0, 2, 0)]=(0.630244113545-1.26945494219j)
            arg[(1, 0, 2, 1)]=(-0.895291993993-0.322387059159j)*x[0] + ((-0.462557745056+0.456868576881j))*x[1] + ((0.120707579625-0.867002463823j))*x[2]
            ref[(1, 0, 2, 1)]=(-0.618571079712-0.366260473051j)
            arg[(1, 0, 3, 0)]=(-0.775537607264-0.438209161184j)*x[0] + ((-0.824972983771+0.33534246545j))*x[1] + ((-0.199427001756-0.430452365953j))*x[2]
            ref[(1, 0, 3, 0)]=(-0.899968796396-0.266659530844j)
            arg[(1, 0, 3, 1)]=(-0.447880464364+0.153180278138j)*x[0] + ((0.0308416779261-0.405255115126j))*x[1] + ((0.871534781374-0.45857137422j))*x[2]
            ref[(1, 0, 3, 1)]=(0.227247997468-0.355323105604j)
            arg[(1, 1, 0, 0)]=(0.915268090523-0.83148108662j)*x[0] + ((0.0536724364262-0.37734893184j))*x[1] + ((0.707085019998-0.803738070337j))*x[2]
            ref[(1, 1, 0, 0)]=(0.838012773473-1.0062840444j)
            arg[(1, 1, 0, 1)]=(-0.499458490653-0.183124015848j)*x[0] + ((0.552380821134-0.786190322769j))*x[1] + ((-0.163390940736+0.808931617553j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.0552343051277-0.0801913605323j)
            arg[(1, 1, 1, 0)]=(0.662829577829-0.0737661486924j)*x[0] + ((-0.92459876214+0.546862720789j))*x[1] + ((-0.425912889428-0.0530122355493j))*x[2]
            ref[(1, 1, 1, 0)]=(-0.34384103687+0.210042168274j)
            arg[(1, 1, 1, 1)]=(-0.791201446789+0.459252705948j)*x[0] + ((-0.165733352505-0.927592092587j))*x[1] + ((0.0613991627561+0.429043865617j))*x[2]
            ref[(1, 1, 1, 1)]=(-0.447767818269-0.019647760511j)
            arg[(1, 1, 2, 0)]=(0.89160477019-0.560276817837j)*x[0] + ((-0.316221282507+0.811024849559j))*x[1] + ((0.920770696388-0.22121548319j))*x[2]
            ref[(1, 1, 2, 0)]=(0.748077092035+0.0147662742661j)
            arg[(1, 1, 2, 1)]=(-0.192425124239-0.954020297751j)*x[0] + ((-0.997856675927-0.362588612755j))*x[1] + ((-0.867610337299+0.219194694042j))*x[2]
            ref[(1, 1, 2, 1)]=(-1.02894606873-0.548707108232j)
            arg[(1, 1, 3, 0)]=(0.202983582691+0.55338679386j)*x[0] + ((0.185561275115+0.22459756343j))*x[1] + ((0.443855570264-0.414227203516j))*x[2]
            ref[(1, 1, 3, 0)]=(0.416200214035+0.181878576887j)
            arg[(1, 1, 3, 1)]=(0.464059927909+0.565155264971j)*x[0] + ((0.799801885918+0.438289318485j))*x[1] + ((-0.593874122109+0.416922030705j))*x[2]
            ref[(1, 1, 3, 1)]=(0.334993845858+0.71018330708j)
            arg[(1, 2, 0, 0)]=(-0.779636948182+0.330981624702j)*x[0] + ((0.0344544895891-0.946497513667j))*x[1] + ((-0.063329886868-0.586216581778j))*x[2]
            ref[(1, 2, 0, 0)]=(-0.40425617273-0.600866235371j)
            arg[(1, 2, 0, 1)]=(-0.527885361143-0.711516883341j)*x[0] + ((-0.821021738728+0.840929582028j))*x[1] + ((-0.823957258899-0.92698005979j))*x[2]
            ref[(1, 2, 0, 1)]=(-1.08643217938-0.398783680552j)
            arg[(1, 2, 1, 0)]=(-0.278488023048+0.570792876872j)*x[0] + ((-0.584988836133-0.312081666868j))*x[1] + ((-0.412952146096-0.170723045484j))*x[2]
            ref[(1, 2, 1, 0)]=(-0.638214502638+0.0439940822598j)
            arg[(1, 2, 1, 1)]=(0.866097377219-0.80186035439j)*x[0] + ((-0.12831659339+0.353130078492j))*x[1] + ((0.543863427125+0.243591563113j))*x[2]
            ref[(1, 2, 1, 1)]=(0.640822105477-0.102569356393j)
            arg[(1, 2, 2, 0)]=(-0.415000389835-0.936149321741j)*x[0] + ((0.225952492124+0.806738709673j))*x[1] + ((0.110993516664+0.910391447217j))*x[2]
            ref[(1, 2, 2, 0)]=(-0.0390271905235+0.390490417575j)
            arg[(1, 2, 2, 1)]=(-0.0488448809711+0.720162350694j)*x[0] + ((-0.332208404364+0.318680841349j))*x[1] + ((-0.409052532053-0.831124665481j))*x[2]
            ref[(1, 2, 2, 1)]=(-0.395052908694+0.103859263281j)
            arg[(1, 2, 3, 0)]=(0.96315926163-0.912563285435j)*x[0] + ((-0.544302022932-0.567969795455j))*x[1] + ((-0.681919815874+0.610944705294j))*x[2]
            ref[(1, 2, 3, 0)]=(-0.131531288588-0.434794187798j)
            arg[(1, 2, 3, 1)]=(0.723589559982+0.0768784772325j)*x[0] + ((0.917737511713-0.70638939519j))*x[1] + ((0.285855629378-0.702403715289j))*x[2]
            ref[(1, 2, 3, 1)]=(0.963591350536-0.665957316623j)
            arg[(1, 3, 0, 0)]=(-0.803348806745-0.271825109013j)*x[0] + ((-0.767095293217+0.11499091337j))*x[1] + ((-0.497627924631+0.430056931528j))*x[2]
            ref[(1, 3, 0, 0)]=(-1.0340360123+0.136611367943j)
            arg[(1, 3, 0, 1)]=(0.647459294864+0.659190830495j)*x[0] + ((0.195060638643+0.215589198911j))*x[1] + ((-0.0718781389825+0.682207545716j))*x[2]
            ref[(1, 3, 0, 1)]=(0.385320897262+0.778493787561j)
            arg[(1, 3, 1, 0)]=(0.122771185757-0.883116505985j)*x[0] + ((-0.44847849077+0.517516472095j))*x[1] + ((-0.708188105914-0.97269125697j))*x[2]
            ref[(1, 3, 1, 0)]=(-0.516947705463-0.66914564543j)
            arg[(1, 3, 1, 1)]=(-0.692353425977+0.256158075311j)*x[0] + ((0.519898142703+0.208395195451j))*x[1] + ((-0.776135144644-0.291221136879j))*x[2]
            ref[(1, 3, 1, 1)]=(-0.474295213959+0.0866660669418j)
            arg[(1, 3, 2, 0)]=(-0.614320346686-0.93263226432j)*x[0] + ((-0.99314592507+0.211085382239j))*x[1] + ((0.0517549764833+0.688188879734j))*x[2]
            ref[(1, 3, 2, 0)]=(-0.777855647636-0.0166790011734j)
            arg[(1, 3, 2, 1)]=(0.612047283975+0.475160150598j)*x[0] + ((-0.712343243746+0.504594032031j))*x[1] + ((-0.155354753934-0.799349167407j))*x[2]
            ref[(1, 3, 2, 1)]=(-0.127825356852+0.090202507611j)
            arg[(1, 3, 3, 0)]=(0.330410874771+0.0345281516893j)*x[0] + ((-0.985638604065-0.0384333130532j))*x[1] + ((0.573297960671-0.262548378103j))*x[2]
            ref[(1, 3, 3, 0)]=(-0.0409648843119-0.133226769734j)
            arg[(1, 3, 3, 1)]=(-0.820713198986+0.56720032534j)*x[0] + ((-0.745102137441+0.792363147115j))*x[1] + ((0.71593925308-0.164244443785j))*x[2]
            ref[(1, 3, 3, 1)]=(-0.424938041673+0.597659514335j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 4, 4, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedContinuousFunction_rank0(self):
        """
        tests integral of rank 0 Data on Function

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.033605177261-0.644264153247j)*x[0] + ((-0.205517924257+0.385085803184j))*x[1]
            ref=(-0.0859563734978-0.129589175032j)
        else:
            arg=(0.288365512375+0.424326943319j)*x[0] + ((0.712680249776+0.965758706668j))*x[1] + ((-0.356566162871+0.174564103127j))*x[2]
            ref=(0.32223979964+0.782324876557j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedContinuousFunction_rank1(self):
        """
        tests integral of rank 1 Data on Function

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3,),w)
        ref=numpy.zeros((3,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.12863871833+0.733541658207j)*x[0] + ((0.35293554815-0.588178433236j))*x[1]
            ref[(0,)]=(0.11214841491+0.0726816124856j)
            arg[(1,)]=(0.271189229245+0.94480079294j)*x[0] + ((-0.925590540451-0.303380388763j))*x[1]
            ref[(1,)]=(-0.327200655603+0.320710202088j)
            arg[(2,)]=(0.800535012311-0.451180840699j)*x[0] + ((-0.0491978481748-0.416356867935j))*x[1]
            ref[(2,)]=(0.375668582068-0.433768854317j)
        else:
            arg[(0,)]=(0.695888135213+0.445354182429j)*x[0] + ((-0.869889943447-0.358992788639j))*x[1] + ((0.737232902001+0.470914289468j))*x[2]
            ref[(0,)]=(0.281615546884+0.278637841629j)
            arg[(1,)]=(-0.580289450428-0.264272689632j)*x[0] + ((0.188377837249+0.685957523016j))*x[1] + ((-0.0397129810337+0.916434047893j))*x[2]
            ref[(1,)]=(-0.215812297107+0.669059440639j)
            arg[(2,)]=(0.67235081569+0.0557308747658j)*x[0] + ((-0.368301332232+0.507796684651j))*x[1] + ((-0.136157718199+0.470506069914j))*x[2]
            ref[(2,)]=(0.0839458826292+0.517016814666j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedContinuousFunction_rank2(self):
        """
        tests integral of rank 2 Data on Function

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4, 2),w)
        ref=numpy.zeros((4, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.332860873662+0.412294419015j)*x[0] + ((0.883172061966-0.0585509565643j))*x[1]
            ref[(0, 0)]=(0.608016467814+0.176871731225j)
            arg[(0, 1)]=(0.239037328279+0.923901352574j)*x[0] + ((0.128776904478+0.928284452025j))*x[1]
            ref[(0, 1)]=(0.183907116379+0.926092902299j)
            arg[(1, 0)]=(-0.270686511189-0.2050431467j)*x[0] + ((-0.870617261276-0.391302123276j))*x[1]
            ref[(1, 0)]=(-0.570651886232-0.298172634988j)
            arg[(1, 1)]=(-0.00559142470953+0.102481129567j)*x[0] + ((0.11908359083-0.514469979908j))*x[1]
            ref[(1, 1)]=(0.0567460830604-0.20599442517j)
            arg[(2, 0)]=(-0.660250531189-0.0216733610427j)*x[0] + ((0.933199165567+0.0622497994538j))*x[1]
            ref[(2, 0)]=(0.136474317189+0.0202882192056j)
            arg[(2, 1)]=(-0.728641157087+0.449640616641j)*x[0] + ((-0.662479258168-0.432708856286j))*x[1]
            ref[(2, 1)]=(-0.695560207628+0.00846588017753j)
            arg[(3, 0)]=(-0.692920425064-0.100237963881j)*x[0] + ((0.135247392063-0.472923359517j))*x[1]
            ref[(3, 0)]=(-0.278836516501-0.286580661699j)
            arg[(3, 1)]=(0.846191912472+0.651293978184j)*x[0] + ((-0.117053452795+0.94648163342j))*x[1]
            ref[(3, 1)]=(0.364569229839+0.798887805802j)
        else:
            arg[(0, 0)]=(0.620308381816-0.352493222607j)*x[0] + ((0.767285728264+0.269937340104j))*x[1] + ((0.893256252133-0.0137004602371j))*x[2]
            ref[(0, 0)]=(1.14042518111-0.0481281713702j)
            arg[(0, 1)]=(-0.506283686823+0.928210155193j)*x[0] + ((-0.0756298324372-0.605023229323j))*x[1] + ((0.263812582944-0.309394729318j))*x[2]
            ref[(0, 1)]=(-0.159050468158+0.00689609827632j)
            arg[(1, 0)]=(0.845893942686+0.291912073046j)*x[0] + ((0.917787723806+0.366697731718j))*x[1] + ((0.693456116139+0.616874049766j))*x[2]
            ref[(1, 0)]=(1.22856889132+0.637741927265j)
            arg[(1, 1)]=(0.144543861775+0.11827519199j)*x[0] + ((0.308021319605+0.210161954982j))*x[1] + ((-0.292432520124-0.0521922065098j))*x[2]
            ref[(1, 1)]=(0.0800663306277+0.138122470231j)
            arg[(2, 0)]=(-0.377074478326+0.72652334785j)*x[0] + ((-0.271152350554-0.398325048499j))*x[1] + ((-0.0887022880817+0.41477058951j))*x[2]
            ref[(2, 0)]=(-0.368464558481+0.37148444443j)
            arg[(2, 1)]=(0.191085033514-0.69362576459j)*x[0] + ((0.545436588623+0.302587325156j))*x[1] + ((0.737603767532+0.537364424192j))*x[2]
            ref[(2, 1)]=(0.737062694834+0.0731629923792j)
            arg[(3, 0)]=(0.945187977441+0.358845974456j)*x[0] + ((-0.449155018337+0.162796594295j))*x[1] + ((0.345690776881+0.0886869768335j))*x[2]
            ref[(3, 0)]=(0.420861867992+0.305164772793j)
            arg[(3, 1)]=(0.445153524739-0.438564078905j)*x[0] + ((0.459604814788+0.961172872961j))*x[1] + ((0.935216108478-0.815297398015j))*x[2]
            ref[(3, 1)]=(0.919987224003-0.146344301979j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedContinuousFunction_rank3(self):
        """
        tests integral of rank 3 Data on Function

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(2, 3, 3),w)
        ref=numpy.zeros((2, 3, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.428258213144+0.536202157252j)*x[0] + ((-0.12789078597-0.930436019652j))*x[1]
            ref[(0, 0, 0)]=(-0.278074499557-0.1971169312j)
            arg[(0, 0, 1)]=(-0.839861498634+0.157387099848j)*x[0] + ((-0.548775454788+0.910018538785j))*x[1]
            ref[(0, 0, 1)]=(-0.694318476711+0.533702819316j)
            arg[(0, 0, 2)]=(-0.390269753916-0.116449324344j)*x[0] + ((0.143784240524+0.275197225064j))*x[1]
            ref[(0, 0, 2)]=(-0.123242756696+0.0793739503599j)
            arg[(0, 1, 0)]=(0.736233488293+0.13907219812j)*x[0] + ((0.846891989599-0.672876893175j))*x[1]
            ref[(0, 1, 0)]=(0.791562738946-0.266902347527j)
            arg[(0, 1, 1)]=(-0.403046149888-0.819295165991j)*x[0] + ((0.610199920278+0.470049005023j))*x[1]
            ref[(0, 1, 1)]=(0.103576885195-0.174623080484j)
            arg[(0, 1, 2)]=(-0.729108408424-0.641472987809j)*x[0] + ((0.944845096349-0.90014093477j))*x[1]
            ref[(0, 1, 2)]=(0.107868343963-0.77080696129j)
            arg[(0, 2, 0)]=(0.33078545382+0.889197096121j)*x[0] + ((0.0903513555457-0.134569571365j))*x[1]
            ref[(0, 2, 0)]=(0.210568404683+0.377313762378j)
            arg[(0, 2, 1)]=(0.216829460532-0.880065332306j)*x[0] + ((-0.982525426568-0.854237333366j))*x[1]
            ref[(0, 2, 1)]=(-0.382847983018-0.867151332836j)
            arg[(0, 2, 2)]=(0.28058354688-0.534265764573j)*x[0] + ((0.746407367296-0.691666804225j))*x[1]
            ref[(0, 2, 2)]=(0.513495457088-0.612966284399j)
            arg[(1, 0, 0)]=(-0.697111177137-0.393348025658j)*x[0] + ((0.885588334141+0.998357534894j))*x[1]
            ref[(1, 0, 0)]=(0.0942385785021+0.302504754618j)
            arg[(1, 0, 1)]=(-0.247988371496-0.375373673675j)*x[0] + ((0.792124942663-0.729404598192j))*x[1]
            ref[(1, 0, 1)]=(0.272068285584-0.552389135933j)
            arg[(1, 0, 2)]=(-0.302160505677+0.706083855916j)*x[0] + ((-0.258135862906-0.311249106555j))*x[1]
            ref[(1, 0, 2)]=(-0.280148184292+0.197417374681j)
            arg[(1, 1, 0)]=(-0.959391062426-0.564044793841j)*x[0] + ((-0.754668808725-0.324482391379j))*x[1]
            ref[(1, 1, 0)]=(-0.857029935576-0.44426359261j)
            arg[(1, 1, 1)]=(-0.15877515457+0.810448822891j)*x[0] + ((-0.911469095287-0.571428895925j))*x[1]
            ref[(1, 1, 1)]=(-0.535122124929+0.119509963483j)
            arg[(1, 1, 2)]=(-0.633480878076-0.857266204602j)*x[0] + ((0.178120847079+0.621185256314j))*x[1]
            ref[(1, 1, 2)]=(-0.227680015498-0.118040474144j)
            arg[(1, 2, 0)]=(-0.284174922613-0.561339856417j)*x[0] + ((0.745629545353-0.454540765762j))*x[1]
            ref[(1, 2, 0)]=(0.23072731137-0.507940311089j)
            arg[(1, 2, 1)]=(0.508128390124-0.506420492019j)*x[0] + ((-0.00338145190045+0.902450146863j))*x[1]
            ref[(1, 2, 1)]=(0.252373469112+0.198014827422j)
            arg[(1, 2, 2)]=(0.681960526041+0.531241033598j)*x[0] + ((0.511562699093-0.603264531826j))*x[1]
            ref[(1, 2, 2)]=(0.596761612567-0.0360117491141j)
        else:
            arg[(0, 0, 0)]=(-0.0780172290962-0.877449866287j)*x[0] + ((-0.411209004498-0.684571819208j))*x[1] + ((-0.187219461042+0.338069820474j))*x[2]
            ref[(0, 0, 0)]=(-0.338222847318-0.61197593251j)
            arg[(0, 0, 1)]=(-0.92380112902-0.949337352407j)*x[0] + ((0.702294938718+0.380324159348j))*x[1] + ((-0.250531209179+0.26915659556j))*x[2]
            ref[(0, 0, 1)]=(-0.236018699741-0.14992829875j)
            arg[(0, 0, 2)]=(-0.810098633987-0.918367189694j)*x[0] + ((0.947185204968+0.792745599589j))*x[1] + ((0.158961371401-0.702296424632j))*x[2]
            ref[(0, 0, 2)]=(0.148023971191-0.413959007369j)
            arg[(0, 1, 0)]=(-0.493362927544-0.336689227217j)*x[0] + ((-0.0227007201289-0.787914172346j))*x[1] + ((-0.117096732369-0.625958735279j))*x[2]
            ref[(0, 1, 0)]=(-0.316580190021-0.875281067421j)
            arg[(0, 1, 1)]=(0.147014512656+0.670037604719j)*x[0] + ((-0.28488464355+0.746363561939j))*x[1] + ((0.991739822318-0.188799874228j))*x[2]
            ref[(0, 1, 1)]=(0.426934845712+0.613800646215j)
            arg[(0, 1, 2)]=(0.798283566234+0.837361629468j)*x[0] + ((0.00590371626389-0.730412658655j))*x[1] + ((-0.147790704927-0.0119556466725j))*x[2]
            ref[(0, 1, 2)]=(0.328198288785+0.0474966620705j)
            arg[(0, 2, 0)]=(0.0155962199426+0.46033353207j)*x[0] + ((-0.506843322414-0.20932465041j))*x[1] + ((-0.881982360425-0.827762922228j))*x[2]
            ref[(0, 2, 0)]=(-0.686614731448-0.288377020284j)
            arg[(0, 2, 1)]=(0.529704560163+0.791937328064j)*x[0] + ((-0.986030067248+0.0527002224765j))*x[1] + ((0.652320163119+0.37323606129j))*x[2]
            ref[(0, 2, 1)]=(0.0979973280171+0.608936805915j)
            arg[(0, 2, 2)]=(-0.320532737697-0.449534559083j)*x[0] + ((0.995317099461+0.252511325031j))*x[1] + ((-0.393733476094-0.128088707826j))*x[2]
            ref[(0, 2, 2)]=(0.140525442835-0.162555970939j)
            arg[(1, 0, 0)]=(0.943018281348-0.192096947993j)*x[0] + ((-0.0637085112678+0.258457214082j))*x[1] + ((0.37545011291-0.372327985084j))*x[2]
            ref[(1, 0, 0)]=(0.627379941495-0.152983859498j)
            arg[(1, 0, 1)]=(-0.416809968004+0.866014689206j)*x[0] + ((-0.0445080649384-0.421285688528j))*x[1] + ((-0.624510233516-0.539346692031j))*x[2]
            ref[(1, 0, 1)]=(-0.542914133229-0.0473088456763j)
            arg[(1, 0, 2)]=(-0.533022349362+0.837516729361j)*x[0] + ((-0.949738631906+0.157829483948j))*x[1] + ((-0.760989148454-0.228622940793j))*x[2]
            ref[(1, 0, 2)]=(-1.12187506486+0.383361636258j)
            arg[(1, 1, 0)]=(0.447709711938-0.0798449626079j)*x[0] + ((-0.554424101482-0.975545647913j))*x[1] + ((0.820216381881-0.506723761828j))*x[2]
            ref[(1, 1, 0)]=(0.356750996169-0.781057186174j)
            arg[(1, 1, 1)]=(0.714830054602-0.580640884403j)*x[0] + ((0.852725547563+0.678960487005j))*x[1] + ((0.709021420039+0.376932797817j))*x[2]
            ref[(1, 1, 1)]=(1.1382885111+0.237626200209j)
            arg[(1, 1, 2)]=(-0.39931730579-0.107305082404j)*x[0] + ((-0.451083815056+0.185062480735j))*x[1] + ((-0.227344927468+0.835936625697j))*x[2]
            ref[(1, 1, 2)]=(-0.538873024157+0.456847012014j)
            arg[(1, 2, 0)]=(-0.864385752481-0.665709702021j)*x[0] + ((-0.522198207072+0.167614871203j))*x[1] + ((0.697679019915+0.812466698047j))*x[2]
            ref[(1, 2, 0)]=(-0.344452469819+0.157185933614j)
            arg[(1, 2, 1)]=(0.771643654153+0.99380499424j)*x[0] + ((0.0942574580975+0.469004979962j))*x[1] + ((0.0981039034916-0.738799037303j))*x[2]
            ref[(1, 2, 1)]=(0.482002507871+0.362005468449j)
            arg[(1, 2, 2)]=(0.0991269826729-0.73996341206j)*x[0] + ((-0.160325776527-0.0660210844552j))*x[1] + ((0.759583071357-0.723067464246j))*x[2]
            ref[(1, 2, 2)]=(0.349192138752-0.764525980381j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 3, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_ReducedContinuousFunction_rank4(self):
        """
        tests integral of rank 4 Data on Function

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 4, 4),w)
        ref=numpy.zeros((3, 4, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.798312850754-0.268604300624j)*x[0] + ((0.873926684238-0.419608926371j))*x[1]
            ref[(0, 0, 0, 0)]=(0.037806916742-0.344106613498j)
            arg[(0, 0, 0, 1)]=(0.523485818504+0.313397807231j)*x[0] + ((0.562791839642-0.961593191776j))*x[1]
            ref[(0, 0, 0, 1)]=(0.543138829073-0.324097692272j)
            arg[(0, 0, 0, 2)]=(0.24554977892+0.549599455356j)*x[0] + ((0.957641159774+0.313542981498j))*x[1]
            ref[(0, 0, 0, 2)]=(0.601595469347+0.431571218427j)
            arg[(0, 0, 0, 3)]=(0.958248346951+0.604973724654j)*x[0] + ((-0.914124896338+0.610665796654j))*x[1]
            ref[(0, 0, 0, 3)]=(0.0220617253067+0.607819760654j)
            arg[(0, 0, 1, 0)]=(-0.340172796537+0.90774753482j)*x[0] + ((0.916867680257-0.967829882872j))*x[1]
            ref[(0, 0, 1, 0)]=(0.28834744186-0.0300411740256j)
            arg[(0, 0, 1, 1)]=(0.600823017836+0.574576042899j)*x[0] + ((0.0439024115364+0.861189791049j))*x[1]
            ref[(0, 0, 1, 1)]=(0.322362714686+0.717882916974j)
            arg[(0, 0, 1, 2)]=(0.625891908636+0.857907345693j)*x[0] + ((0.269837249435+0.169383641919j))*x[1]
            ref[(0, 0, 1, 2)]=(0.447864579035+0.513645493806j)
            arg[(0, 0, 1, 3)]=(0.610136967497-0.00981675030623j)*x[0] + ((-0.0475479685709+0.0278272903383j))*x[1]
            ref[(0, 0, 1, 3)]=(0.281294499463+0.00900527001602j)
            arg[(0, 0, 2, 0)]=(0.0535728638221-0.349695010338j)*x[0] + ((0.531576496071+0.888604600185j))*x[1]
            ref[(0, 0, 2, 0)]=(0.292574679946+0.269454794923j)
            arg[(0, 0, 2, 1)]=(-0.45945783666+0.280323309293j)*x[0] + ((0.987229162627+0.0791879776308j))*x[1]
            ref[(0, 0, 2, 1)]=(0.263885662984+0.179755643462j)
            arg[(0, 0, 2, 2)]=(-0.162596829473+0.33534625146j)*x[0] + ((-0.101953723487+0.424030179911j))*x[1]
            ref[(0, 0, 2, 2)]=(-0.13227527648+0.379688215686j)
            arg[(0, 0, 2, 3)]=(-0.410753942508-0.831351745691j)*x[0] + ((-0.153521491681+0.689446325534j))*x[1]
            ref[(0, 0, 2, 3)]=(-0.282137717095-0.0709527100789j)
            arg[(0, 0, 3, 0)]=(0.527464219658-0.51076891333j)*x[0] + ((0.0258061853074+0.243973686299j))*x[1]
            ref[(0, 0, 3, 0)]=(0.276635202482-0.133397613516j)
            arg[(0, 0, 3, 1)]=(0.261366130518-0.0623042630064j)*x[0] + ((-0.923718482046+0.709733260252j))*x[1]
            ref[(0, 0, 3, 1)]=(-0.331176175764+0.323714498623j)
            arg[(0, 0, 3, 2)]=(0.13602232939+0.644508298794j)*x[0] + ((0.325684462092-0.0204724007759j))*x[1]
            ref[(0, 0, 3, 2)]=(0.230853395741+0.312017949009j)
            arg[(0, 0, 3, 3)]=(0.449890875254-0.181988514961j)*x[0] + ((0.663763526435-0.513960291883j))*x[1]
            ref[(0, 0, 3, 3)]=(0.556827200844-0.347974403422j)
            arg[(0, 1, 0, 0)]=(-0.556226699793-0.83117531816j)*x[0] + ((-0.0474767091739+0.606580811548j))*x[1]
            ref[(0, 1, 0, 0)]=(-0.301851704484-0.112297253306j)
            arg[(0, 1, 0, 1)]=(0.61834353477-0.90517895803j)*x[0] + ((0.156786460247+0.125425884484j))*x[1]
            ref[(0, 1, 0, 1)]=(0.387564997508-0.389876536773j)
            arg[(0, 1, 0, 2)]=(0.928933656112+0.33032219503j)*x[0] + ((-0.666547473335-0.00578030158346j))*x[1]
            ref[(0, 1, 0, 2)]=(0.131193091388+0.162270946723j)
            arg[(0, 1, 0, 3)]=(0.639179106175-0.99368768286j)*x[0] + ((0.206281133555+0.825274811896j))*x[1]
            ref[(0, 1, 0, 3)]=(0.422730119865-0.0842064354823j)
            arg[(0, 1, 1, 0)]=(-0.262230185685-0.986752242691j)*x[0] + ((-0.0705850246747+0.476534651909j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.16640760518-0.255108795391j)
            arg[(0, 1, 1, 1)]=(0.535795230719+0.998098604068j)*x[0] + ((-0.678933155957+0.407873101001j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.0715689626188+0.702985852535j)
            arg[(0, 1, 1, 2)]=(0.238732466188+0.496092308351j)*x[0] + ((0.925241360447-0.643497870199j))*x[1]
            ref[(0, 1, 1, 2)]=(0.581986913317-0.0737027809238j)
            arg[(0, 1, 1, 3)]=(0.488522190061-0.163507348783j)*x[0] + ((-0.474235246249-0.104788796937j))*x[1]
            ref[(0, 1, 1, 3)]=(0.00714347190568-0.13414807286j)
            arg[(0, 1, 2, 0)]=(-0.547400532094+0.248553426621j)*x[0] + ((-0.222721697457+0.987104124164j))*x[1]
            ref[(0, 1, 2, 0)]=(-0.385061114775+0.617828775393j)
            arg[(0, 1, 2, 1)]=(0.223695017834-0.483285948699j)*x[0] + ((0.996490152005-0.269907546149j))*x[1]
            ref[(0, 1, 2, 1)]=(0.610092584919-0.376596747424j)
            arg[(0, 1, 2, 2)]=(0.441908189004+0.200834727043j)*x[0] + ((-0.919965781693+0.996770809287j))*x[1]
            ref[(0, 1, 2, 2)]=(-0.239028796345+0.598802768165j)
            arg[(0, 1, 2, 3)]=(-0.508665778075+0.369206627755j)*x[0] + ((-0.785063819012-0.159573959021j))*x[1]
            ref[(0, 1, 2, 3)]=(-0.646864798544+0.104816334367j)
            arg[(0, 1, 3, 0)]=(0.0972243330376+0.949019777147j)*x[0] + ((-0.938326204735+0.479906779002j))*x[1]
            ref[(0, 1, 3, 0)]=(-0.420550935848+0.714463278074j)
            arg[(0, 1, 3, 1)]=(0.67040600826-0.288243549113j)*x[0] + ((-0.300939785819+0.679129007921j))*x[1]
            ref[(0, 1, 3, 1)]=(0.18473311122+0.195442729404j)
            arg[(0, 1, 3, 2)]=(0.607834443672-0.902018842798j)*x[0] + ((-0.311003148299+0.652062062088j))*x[1]
            ref[(0, 1, 3, 2)]=(0.148415647687-0.124978390355j)
            arg[(0, 1, 3, 3)]=(0.31068111316-0.320469802861j)*x[0] + ((-0.116847526543+0.0438674859138j))*x[1]
            ref[(0, 1, 3, 3)]=(0.0969167933089-0.138301158474j)
            arg[(0, 2, 0, 0)]=(0.652832867335+0.404333139912j)*x[0] + ((0.978746105121+0.659494094956j))*x[1]
            ref[(0, 2, 0, 0)]=(0.815789486228+0.531913617434j)
            arg[(0, 2, 0, 1)]=(0.656000998205-0.622249145997j)*x[0] + ((0.414907552672-0.549477439833j))*x[1]
            ref[(0, 2, 0, 1)]=(0.535454275439-0.585863292915j)
            arg[(0, 2, 0, 2)]=(-0.743444619388+0.880514907947j)*x[0] + ((-0.0738897908885+0.591086585387j))*x[1]
            ref[(0, 2, 0, 2)]=(-0.408667205138+0.735800746667j)
            arg[(0, 2, 0, 3)]=(0.203510829405+0.574311367689j)*x[0] + ((-0.0617350804382-0.0955223331299j))*x[1]
            ref[(0, 2, 0, 3)]=(0.0708878744832+0.23939451728j)
            arg[(0, 2, 1, 0)]=(-0.339274162024-0.314875630259j)*x[0] + ((0.875618121681+0.553922478831j))*x[1]
            ref[(0, 2, 1, 0)]=(0.268171979828+0.119523424286j)
            arg[(0, 2, 1, 1)]=(-0.650806604368+0.254989560036j)*x[0] + ((0.147336577097-0.687846872393j))*x[1]
            ref[(0, 2, 1, 1)]=(-0.251735013635-0.216428656179j)
            arg[(0, 2, 1, 2)]=(0.902963313565+0.110255633039j)*x[0] + ((0.837844439203+0.976413312614j))*x[1]
            ref[(0, 2, 1, 2)]=(0.870403876384+0.543334472827j)
            arg[(0, 2, 1, 3)]=(-0.004875448519+0.417500592456j)*x[0] + ((-0.322349495749-0.800909194067j))*x[1]
            ref[(0, 2, 1, 3)]=(-0.163612472134-0.191704300806j)
            arg[(0, 2, 2, 0)]=(-0.695831959186+0.251049760159j)*x[0] + ((0.676687926781-0.00528379795918j))*x[1]
            ref[(0, 2, 2, 0)]=(-0.00957201620215+0.1228829811j)
            arg[(0, 2, 2, 1)]=(0.464264636748-0.351470715115j)*x[0] + ((-0.0526788556508+0.899689335199j))*x[1]
            ref[(0, 2, 2, 1)]=(0.205792890549+0.274109310042j)
            arg[(0, 2, 2, 2)]=(0.0339388661625+0.761481166043j)*x[0] + ((-0.0212959566466-0.126396681246j))*x[1]
            ref[(0, 2, 2, 2)]=(0.00632145475794+0.317542242399j)
            arg[(0, 2, 2, 3)]=(0.128127702011+0.36928186157j)*x[0] + ((-0.190005082889-0.47618700334j))*x[1]
            ref[(0, 2, 2, 3)]=(-0.0309386904389-0.053452570885j)
            arg[(0, 2, 3, 0)]=(-0.672289697272+0.691553419705j)*x[0] + ((0.896964500343+0.330021508409j))*x[1]
            ref[(0, 2, 3, 0)]=(0.112337401536+0.510787464057j)
            arg[(0, 2, 3, 1)]=(-0.631703335712+0.537261822175j)*x[0] + ((0.704881175855+0.578979880413j))*x[1]
            ref[(0, 2, 3, 1)]=(0.0365889200714+0.558120851294j)
            arg[(0, 2, 3, 2)]=(-0.653845193052-0.71757020942j)*x[0] + ((-0.176634970328-0.418107475309j))*x[1]
            ref[(0, 2, 3, 2)]=(-0.41524008169-0.567838842365j)
            arg[(0, 2, 3, 3)]=(-0.603908148273-0.333788219201j)*x[0] + ((-0.343795513774-0.275404204032j))*x[1]
            ref[(0, 2, 3, 3)]=(-0.473851831023-0.304596211617j)
            arg[(0, 3, 0, 0)]=(0.531890810065+0.336836410295j)*x[0] + ((0.647653900299+0.56482133521j))*x[1]
            ref[(0, 3, 0, 0)]=(0.589772355182+0.450828872752j)
            arg[(0, 3, 0, 1)]=(-0.13239322973+0.800605580614j)*x[0] + ((0.786120066425-0.145884732499j))*x[1]
            ref[(0, 3, 0, 1)]=(0.326863418348+0.327360424058j)
            arg[(0, 3, 0, 2)]=(-0.822943798013-0.35227139739j)*x[0] + ((0.417430493283+0.464877491645j))*x[1]
            ref[(0, 3, 0, 2)]=(-0.202756652365+0.0563030471275j)
            arg[(0, 3, 0, 3)]=(0.576698552544+0.632021605746j)*x[0] + ((0.958103953156-0.354242052244j))*x[1]
            ref[(0, 3, 0, 3)]=(0.76740125285+0.138889776751j)
            arg[(0, 3, 1, 0)]=(-0.918760124708-0.415003413456j)*x[0] + ((0.273208938939+0.365349696795j))*x[1]
            ref[(0, 3, 1, 0)]=(-0.322775592884-0.0248268583306j)
            arg[(0, 3, 1, 1)]=(-0.380471836594+0.90686715675j)*x[0] + ((0.32881646022+0.498731894363j))*x[1]
            ref[(0, 3, 1, 1)]=(-0.025827688187+0.702799525557j)
            arg[(0, 3, 1, 2)]=(0.621572326734-0.0846218361532j)*x[0] + ((-0.478681568793-0.521892682302j))*x[1]
            ref[(0, 3, 1, 2)]=(0.0714453789705-0.303257259228j)
            arg[(0, 3, 1, 3)]=(-0.548217377258-0.139755085916j)*x[0] + ((0.97982283086+0.334754091717j))*x[1]
            ref[(0, 3, 1, 3)]=(0.215802726801+0.0974995029004j)
            arg[(0, 3, 2, 0)]=(-0.604944962454+0.93108804949j)*x[0] + ((-0.944070197104-0.6081941493j))*x[1]
            ref[(0, 3, 2, 0)]=(-0.774507579779+0.161446950095j)
            arg[(0, 3, 2, 1)]=(-0.548412950257+0.306153311249j)*x[0] + ((-0.977782202285-0.151435554247j))*x[1]
            ref[(0, 3, 2, 1)]=(-0.763097576271+0.0773588785011j)
            arg[(0, 3, 2, 2)]=(0.907802538764+0.11583209003j)*x[0] + ((-0.466331481456+0.662975698347j))*x[1]
            ref[(0, 3, 2, 2)]=(0.220735528654+0.389403894189j)
            arg[(0, 3, 2, 3)]=(-0.936777307336+0.983932878966j)*x[0] + ((-0.590438126839+0.530844363222j))*x[1]
            ref[(0, 3, 2, 3)]=(-0.763607717088+0.757388621094j)
            arg[(0, 3, 3, 0)]=(-0.0304136219627-0.667244717315j)*x[0] + ((-0.887202091113-0.556375237998j))*x[1]
            ref[(0, 3, 3, 0)]=(-0.458807856538-0.611809977657j)
            arg[(0, 3, 3, 1)]=(0.661669800196-0.0469018500956j)*x[0] + ((0.813795402211+0.699739651819j))*x[1]
            ref[(0, 3, 3, 1)]=(0.737732601204+0.326418900862j)
            arg[(0, 3, 3, 2)]=(-0.802059012671-0.697076355126j)*x[0] + ((-0.865433731367+0.481696835066j))*x[1]
            ref[(0, 3, 3, 2)]=(-0.833746372019-0.10768976003j)
            arg[(0, 3, 3, 3)]=(-0.460636795978+0.382223016512j)*x[0] + ((0.424063572939+0.601873178237j))*x[1]
            ref[(0, 3, 3, 3)]=(-0.0182866115196+0.492048097375j)
            arg[(1, 0, 0, 0)]=(0.966080389613-0.0595875709658j)*x[0] + ((0.0226706528568-0.995520315365j))*x[1]
            ref[(1, 0, 0, 0)]=(0.494375521235-0.527553943166j)
            arg[(1, 0, 0, 1)]=(0.650737068843-0.520271710602j)*x[0] + ((-0.126490271144-0.655042259606j))*x[1]
            ref[(1, 0, 0, 1)]=(0.262123398849-0.587656985104j)
            arg[(1, 0, 0, 2)]=(-0.0268882423893+0.355777615775j)*x[0] + ((0.0485219662856-0.727348933326j))*x[1]
            ref[(1, 0, 0, 2)]=(0.0108168619482-0.185785658775j)
            arg[(1, 0, 0, 3)]=(-0.261334202448+0.979134072708j)*x[0] + ((-0.378618662529-0.922123067609j))*x[1]
            ref[(1, 0, 0, 3)]=(-0.319976432488+0.0285055025499j)
            arg[(1, 0, 1, 0)]=(0.271254838601-0.862894039883j)*x[0] + ((-0.270257133238-0.365696522984j))*x[1]
            ref[(1, 0, 1, 0)]=(0.000498852681174-0.614295281434j)
            arg[(1, 0, 1, 1)]=(0.86015883232+0.443544218094j)*x[0] + ((-0.800423432187+0.475013956043j))*x[1]
            ref[(1, 0, 1, 1)]=(0.0298677000664+0.459279087068j)
            arg[(1, 0, 1, 2)]=(-0.514439706268+0.21912006967j)*x[0] + ((0.965926186598+0.311798912304j))*x[1]
            ref[(1, 0, 1, 2)]=(0.225743240165+0.265459490987j)
            arg[(1, 0, 1, 3)]=(0.87669665292-0.385833611241j)*x[0] + ((-0.509844218599-0.217547256861j))*x[1]
            ref[(1, 0, 1, 3)]=(0.18342621716-0.301690434051j)
            arg[(1, 0, 2, 0)]=(-0.000158784700266+0.0537097981786j)*x[0] + ((-0.852512817169+0.450574230631j))*x[1]
            ref[(1, 0, 2, 0)]=(-0.426335800935+0.252142014405j)
            arg[(1, 0, 2, 1)]=(0.374486084646+0.14446841493j)*x[0] + ((0.0809477303637+0.738897768486j))*x[1]
            ref[(1, 0, 2, 1)]=(0.227716907505+0.441683091708j)
            arg[(1, 0, 2, 2)]=(-0.356551357698+0.969512568319j)*x[0] + ((-0.434455967786-0.925577985099j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.395503662742+0.0219672916098j)
            arg[(1, 0, 2, 3)]=(0.694765974088+0.529916346559j)*x[0] + ((0.838289171733-0.139921984417j))*x[1]
            ref[(1, 0, 2, 3)]=(0.76652757291+0.194997181071j)
            arg[(1, 0, 3, 0)]=(0.730662110673-0.0837093222756j)*x[0] + ((0.966990939925-0.794173396067j))*x[1]
            ref[(1, 0, 3, 0)]=(0.848826525299-0.438941359172j)
            arg[(1, 0, 3, 1)]=(-0.354269478593-0.3527404392j)*x[0] + ((-0.937459908757+0.0400908224006j))*x[1]
            ref[(1, 0, 3, 1)]=(-0.645864693675-0.1563248084j)
            arg[(1, 0, 3, 2)]=(0.741845813445+0.393757078606j)*x[0] + ((0.618007721767-0.404730011398j))*x[1]
            ref[(1, 0, 3, 2)]=(0.679926767606-0.00548646639596j)
            arg[(1, 0, 3, 3)]=(-0.0296644891887-0.221913790015j)*x[0] + ((-0.92135379378+0.246574665287j))*x[1]
            ref[(1, 0, 3, 3)]=(-0.475509141485+0.0123304376358j)
            arg[(1, 1, 0, 0)]=(0.614864334547+0.551292826878j)*x[0] + ((0.584651770661-0.165665932707j))*x[1]
            ref[(1, 1, 0, 0)]=(0.599758052604+0.192813447085j)
            arg[(1, 1, 0, 1)]=(-0.70445998507-0.6015715459j)*x[0] + ((0.733677008792+0.616887234447j))*x[1]
            ref[(1, 1, 0, 1)]=(0.0146085118611+0.0076578442733j)
            arg[(1, 1, 0, 2)]=(-0.418616260302+0.931824428917j)*x[0] + ((-0.0786938709492+0.4669656852j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.248655065626+0.699395057059j)
            arg[(1, 1, 0, 3)]=(0.509829087744-0.337361509383j)*x[0] + ((0.175428326259+0.753559931985j))*x[1]
            ref[(1, 1, 0, 3)]=(0.342628707001+0.208099211301j)
            arg[(1, 1, 1, 0)]=(0.357569242462-0.907750513985j)*x[0] + ((-0.00125858719675-0.452932503868j))*x[1]
            ref[(1, 1, 1, 0)]=(0.178155327633-0.680341508927j)
            arg[(1, 1, 1, 1)]=(0.0986407685664+0.0449001747908j)*x[0] + ((-0.658193964337-0.142224735143j))*x[1]
            ref[(1, 1, 1, 1)]=(-0.279776597886-0.0486622801762j)
            arg[(1, 1, 1, 2)]=(-0.886190094346+0.848367531963j)*x[0] + ((0.932372192276-0.346618015624j))*x[1]
            ref[(1, 1, 1, 2)]=(0.0230910489648+0.250874758169j)
            arg[(1, 1, 1, 3)]=(0.953027748089-0.543507421309j)*x[0] + ((-0.0817012579545-0.0550062938929j))*x[1]
            ref[(1, 1, 1, 3)]=(0.435663245067-0.299256857601j)
            arg[(1, 1, 2, 0)]=(0.183053904242+0.966175225215j)*x[0] + ((0.842017860971+0.680365112237j))*x[1]
            ref[(1, 1, 2, 0)]=(0.512535882607+0.823270168726j)
            arg[(1, 1, 2, 1)]=(-0.466097580773-0.275713445377j)*x[0] + ((-0.613252536911-0.246607489165j))*x[1]
            ref[(1, 1, 2, 1)]=(-0.539675058842-0.261160467271j)
            arg[(1, 1, 2, 2)]=(-0.209461051706-0.0104521914792j)*x[0] + ((-0.428754324035-0.766317956193j))*x[1]
            ref[(1, 1, 2, 2)]=(-0.31910768787-0.388385073836j)
            arg[(1, 1, 2, 3)]=(-0.0087944298782-0.177841424414j)*x[0] + ((0.716394942016-0.855754453356j))*x[1]
            ref[(1, 1, 2, 3)]=(0.353800256069-0.516797938885j)
            arg[(1, 1, 3, 0)]=(-0.111425598338-0.0243683954542j)*x[0] + ((-0.340286276102-0.400379748879j))*x[1]
            ref[(1, 1, 3, 0)]=(-0.22585593722-0.212374072166j)
            arg[(1, 1, 3, 1)]=(-0.796250167631-0.0294270488621j)*x[0] + ((-0.159465825058-0.418116537791j))*x[1]
            ref[(1, 1, 3, 1)]=(-0.477857996344-0.223771793327j)
            arg[(1, 1, 3, 2)]=(0.0929786905869-0.00682252091759j)*x[0] + ((-0.929491010505-0.295885433446j))*x[1]
            ref[(1, 1, 3, 2)]=(-0.418256159959-0.151353977182j)
            arg[(1, 1, 3, 3)]=(0.304994816573+0.839174126762j)*x[0] + ((0.0453015572739+0.341545972472j))*x[1]
            ref[(1, 1, 3, 3)]=(0.175148186923+0.590360049617j)
            arg[(1, 2, 0, 0)]=(0.979614626454+0.0514075357558j)*x[0] + ((0.992528391427+0.379717522619j))*x[1]
            ref[(1, 2, 0, 0)]=(0.98607150894+0.215562529187j)
            arg[(1, 2, 0, 1)]=(0.504411750751+0.713611044419j)*x[0] + ((-0.413543615047+0.996268978865j))*x[1]
            ref[(1, 2, 0, 1)]=(0.0454340678522+0.854940011642j)
            arg[(1, 2, 0, 2)]=(0.862413747608+0.512924775511j)*x[0] + ((0.523642103181-0.668834758795j))*x[1]
            ref[(1, 2, 0, 2)]=(0.693027925394-0.0779549916418j)
            arg[(1, 2, 0, 3)]=(-0.371189985801-0.835012296985j)*x[0] + ((-0.150067951116-0.224673910488j))*x[1]
            ref[(1, 2, 0, 3)]=(-0.260628968459-0.529843103737j)
            arg[(1, 2, 1, 0)]=(-0.122242888316+0.456217889637j)*x[0] + ((0.127330401059+0.376987978419j))*x[1]
            ref[(1, 2, 1, 0)]=(0.00254375637179+0.416602934028j)
            arg[(1, 2, 1, 1)]=(-0.716518810767+0.364919960935j)*x[0] + ((0.173901989625+0.462706056351j))*x[1]
            ref[(1, 2, 1, 1)]=(-0.271308410571+0.413813008643j)
            arg[(1, 2, 1, 2)]=(-0.967483893412+0.0745994450871j)*x[0] + ((-0.182287277916+0.152882549099j))*x[1]
            ref[(1, 2, 1, 2)]=(-0.574885585664+0.113740997093j)
            arg[(1, 2, 1, 3)]=(-0.969762659415+0.242618356348j)*x[0] + ((-0.831242379882+0.160240134072j))*x[1]
            ref[(1, 2, 1, 3)]=(-0.900502519649+0.20142924521j)
            arg[(1, 2, 2, 0)]=(-0.596261780827-0.788098764064j)*x[0] + ((-0.307363969158-0.950976571727j))*x[1]
            ref[(1, 2, 2, 0)]=(-0.451812874992-0.869537667895j)
            arg[(1, 2, 2, 1)]=(0.526916129793-0.895071324217j)*x[0] + ((0.955027448845-0.163305869069j))*x[1]
            ref[(1, 2, 2, 1)]=(0.740971789319-0.529188596643j)
            arg[(1, 2, 2, 2)]=(0.720824153719-0.576236938891j)*x[0] + ((0.313745742977+0.4103301027j))*x[1]
            ref[(1, 2, 2, 2)]=(0.517284948348-0.0829534180957j)
            arg[(1, 2, 2, 3)]=(-0.223312970742+0.768108247645j)*x[0] + ((-0.398020688375-0.0905739528678j))*x[1]
            ref[(1, 2, 2, 3)]=(-0.310666829558+0.338767147389j)
            arg[(1, 2, 3, 0)]=(0.86762918186+0.636977384463j)*x[0] + ((0.660148105971+0.142756648236j))*x[1]
            ref[(1, 2, 3, 0)]=(0.763888643915+0.389867016349j)
            arg[(1, 2, 3, 1)]=(0.607580065335-0.264847636717j)*x[0] + ((0.284706850453+0.851661881061j))*x[1]
            ref[(1, 2, 3, 1)]=(0.446143457894+0.293407122172j)
            arg[(1, 2, 3, 2)]=(0.816739466001-0.654476456527j)*x[0] + ((0.386162492237-0.868028206226j))*x[1]
            ref[(1, 2, 3, 2)]=(0.601450979119-0.761252331376j)
            arg[(1, 2, 3, 3)]=(-0.736236979466-0.59963624002j)*x[0] + ((0.416993820802+0.445048001113j))*x[1]
            ref[(1, 2, 3, 3)]=(-0.159621579332-0.0772941194533j)
            arg[(1, 3, 0, 0)]=(0.763108559005+0.105906449048j)*x[0] + ((0.432237306058+0.243319159632j))*x[1]
            ref[(1, 3, 0, 0)]=(0.597672932532+0.17461280434j)
            arg[(1, 3, 0, 1)]=(0.0641377166225-0.440691622811j)*x[0] + ((0.374692208146+0.0208428949208j))*x[1]
            ref[(1, 3, 0, 1)]=(0.219414962384-0.209924363945j)
            arg[(1, 3, 0, 2)]=(0.125523809883+0.0993861040577j)*x[0] + ((-0.464467947089+0.525066447808j))*x[1]
            ref[(1, 3, 0, 2)]=(-0.169472068603+0.312226275933j)
            arg[(1, 3, 0, 3)]=(0.239325532687+0.165329389638j)*x[0] + ((0.463420784364+0.207099034352j))*x[1]
            ref[(1, 3, 0, 3)]=(0.351373158525+0.186214211995j)
            arg[(1, 3, 1, 0)]=(0.71650867594-0.306404418649j)*x[0] + ((-0.69776829332-0.278392830331j))*x[1]
            ref[(1, 3, 1, 0)]=(0.00937019131002-0.29239862449j)
            arg[(1, 3, 1, 1)]=(-0.0518678309952+0.166168778842j)*x[0] + ((-0.847523159545+0.344484450593j))*x[1]
            ref[(1, 3, 1, 1)]=(-0.44969549527+0.255326614718j)
            arg[(1, 3, 1, 2)]=(0.0674976002168+0.696532929303j)*x[0] + ((0.023721602878-0.100592045887j))*x[1]
            ref[(1, 3, 1, 2)]=(0.0456096015474+0.297970441708j)
            arg[(1, 3, 1, 3)]=(-0.950286003463-0.978345873647j)*x[0] + ((-0.607476951835+0.647665698018j))*x[1]
            ref[(1, 3, 1, 3)]=(-0.778881477649-0.165340087815j)
            arg[(1, 3, 2, 0)]=(-0.339815122701-0.226651525626j)*x[0] + ((0.194841977539+0.80221611532j))*x[1]
            ref[(1, 3, 2, 0)]=(-0.0724865725811+0.287782294847j)
            arg[(1, 3, 2, 1)]=(0.0838331436584+0.932300357188j)*x[0] + ((-0.517726794506+0.98891586576j))*x[1]
            ref[(1, 3, 2, 1)]=(-0.216946825424+0.960608111474j)
            arg[(1, 3, 2, 2)]=(0.881141532865-0.321413758086j)*x[0] + ((-0.876138772188+0.928590367714j))*x[1]
            ref[(1, 3, 2, 2)]=(0.00250138033848+0.303588304814j)
            arg[(1, 3, 2, 3)]=(0.242506784938+0.0237807495762j)*x[0] + ((0.0485996097917-0.438758027082j))*x[1]
            ref[(1, 3, 2, 3)]=(0.145553197365-0.207488638753j)
            arg[(1, 3, 3, 0)]=(0.0850294371956+0.102480101547j)*x[0] + ((0.404230394309-0.69653921934j))*x[1]
            ref[(1, 3, 3, 0)]=(0.244629915752-0.297029558896j)
            arg[(1, 3, 3, 1)]=(0.356516587785+0.779764575301j)*x[0] + ((-0.0462326167509-0.52779775758j))*x[1]
            ref[(1, 3, 3, 1)]=(0.155141985517+0.12598340886j)
            arg[(1, 3, 3, 2)]=(-0.84821874261-0.487370074705j)*x[0] + ((0.818864058333+0.313801202822j))*x[1]
            ref[(1, 3, 3, 2)]=(-0.0146773421384-0.0867844359419j)
            arg[(1, 3, 3, 3)]=(0.256156708403-0.38622403095j)*x[0] + ((-0.706888036032-0.499762248768j))*x[1]
            ref[(1, 3, 3, 3)]=(-0.225365663815-0.442993139859j)
            arg[(2, 0, 0, 0)]=(-0.0309334395902+0.103737383883j)*x[0] + ((-0.0510637427317-0.817858965299j))*x[1]
            ref[(2, 0, 0, 0)]=(-0.040998591161-0.357060790708j)
            arg[(2, 0, 0, 1)]=(0.069134129995+0.393726701978j)*x[0] + ((0.773679076571-0.883352764767j))*x[1]
            ref[(2, 0, 0, 1)]=(0.421406603283-0.244813031395j)
            arg[(2, 0, 0, 2)]=(0.326918754643+0.0279330366489j)*x[0] + ((0.777537716613+0.818134971267j))*x[1]
            ref[(2, 0, 0, 2)]=(0.552228235628+0.423034003958j)
            arg[(2, 0, 0, 3)]=(0.57601685368-0.856628112624j)*x[0] + ((-0.570140837065-0.0581483971247j))*x[1]
            ref[(2, 0, 0, 3)]=(0.00293800830725-0.457388254875j)
            arg[(2, 0, 1, 0)]=(0.140408454917+0.84483167466j)*x[0] + ((0.208254014206-0.457933680419j))*x[1]
            ref[(2, 0, 1, 0)]=(0.174331234562+0.193448997121j)
            arg[(2, 0, 1, 1)]=(0.969212679761+0.148420296195j)*x[0] + ((-0.0511067952465+0.870208350632j))*x[1]
            ref[(2, 0, 1, 1)]=(0.459052942257+0.509314323414j)
            arg[(2, 0, 1, 2)]=(0.574295462236-0.369968904137j)*x[0] + ((0.748757496555+0.702147481208j))*x[1]
            ref[(2, 0, 1, 2)]=(0.661526479395+0.166089288535j)
            arg[(2, 0, 1, 3)]=(0.230804078069-0.656494126388j)*x[0] + ((0.87835626108-0.724096034988j))*x[1]
            ref[(2, 0, 1, 3)]=(0.554580169575-0.690295080688j)
            arg[(2, 0, 2, 0)]=(0.480081140181-0.497864337507j)*x[0] + ((-0.958006105674-0.0748698585331j))*x[1]
            ref[(2, 0, 2, 0)]=(-0.238962482746-0.28636709802j)
            arg[(2, 0, 2, 1)]=(-0.937773322599+0.168114508417j)*x[0] + ((0.445174454589+0.347555998525j))*x[1]
            ref[(2, 0, 2, 1)]=(-0.246299434005+0.257835253471j)
            arg[(2, 0, 2, 2)]=(-0.0825869499598+0.876031807613j)*x[0] + ((-0.413041995278-0.4525065186j))*x[1]
            ref[(2, 0, 2, 2)]=(-0.247814472619+0.211762644506j)
            arg[(2, 0, 2, 3)]=(0.301386310057-0.304983959555j)*x[0] + ((0.412062647593+0.089116714716j))*x[1]
            ref[(2, 0, 2, 3)]=(0.356724478825-0.10793362242j)
            arg[(2, 0, 3, 0)]=(-0.147278890942+0.311658175428j)*x[0] + ((-0.25883506231-0.825274343473j))*x[1]
            ref[(2, 0, 3, 0)]=(-0.203056976626-0.256808084022j)
            arg[(2, 0, 3, 1)]=(0.386839245632-0.359386390126j)*x[0] + ((0.508776514215-0.0674902681703j))*x[1]
            ref[(2, 0, 3, 1)]=(0.447807879924-0.213438329148j)
            arg[(2, 0, 3, 2)]=(-0.231558992291+0.145044547036j)*x[0] + ((-0.860849777762-0.282761600328j))*x[1]
            ref[(2, 0, 3, 2)]=(-0.546204385027-0.068858526646j)
            arg[(2, 0, 3, 3)]=(-0.56692801068+0.665889377326j)*x[0] + ((0.920735617295-0.984280210538j))*x[1]
            ref[(2, 0, 3, 3)]=(0.176903803308-0.159195416606j)
            arg[(2, 1, 0, 0)]=(-0.688084721536+0.527152979052j)*x[0] + ((-0.897514403152-0.921400115528j))*x[1]
            ref[(2, 1, 0, 0)]=(-0.792799562344-0.197123568238j)
            arg[(2, 1, 0, 1)]=(0.967246536983-0.543188427423j)*x[0] + ((-0.0761331633941-0.816757758716j))*x[1]
            ref[(2, 1, 0, 1)]=(0.445556686794-0.67997309307j)
            arg[(2, 1, 0, 2)]=(0.856283981414+0.659442672989j)*x[0] + ((-0.163928656945+0.0992693369625j))*x[1]
            ref[(2, 1, 0, 2)]=(0.346177662234+0.379356004976j)
            arg[(2, 1, 0, 3)]=(-0.64683300585-0.288549282004j)*x[0] + ((0.799453025114-0.182204010781j))*x[1]
            ref[(2, 1, 0, 3)]=(0.0763100096318-0.235376646392j)
            arg[(2, 1, 1, 0)]=(0.703640117293-0.758382852548j)*x[0] + ((0.230298583783+0.9803195585j))*x[1]
            ref[(2, 1, 1, 0)]=(0.466969350538+0.110968352976j)
            arg[(2, 1, 1, 1)]=(-0.0926223996904-0.581653094412j)*x[0] + ((-0.382317545228-0.263415694099j))*x[1]
            ref[(2, 1, 1, 1)]=(-0.237469972459-0.422534394256j)
            arg[(2, 1, 1, 2)]=(0.0644418983757+0.36190914642j)*x[0] + ((-0.507219790877+0.00335100608252j))*x[1]
            ref[(2, 1, 1, 2)]=(-0.22138894625+0.182630076251j)
            arg[(2, 1, 1, 3)]=(-0.682539077125+0.926084672259j)*x[0] + ((0.950155182458+0.983667741983j))*x[1]
            ref[(2, 1, 1, 3)]=(0.133808052667+0.954876207121j)
            arg[(2, 1, 2, 0)]=(0.905413767436-0.176724599902j)*x[0] + ((-0.790066074078-0.454355711456j))*x[1]
            ref[(2, 1, 2, 0)]=(0.0576738466793-0.315540155679j)
            arg[(2, 1, 2, 1)]=(0.902219973904+0.944488225807j)*x[0] + ((-0.679552686449+0.0545826187208j))*x[1]
            ref[(2, 1, 2, 1)]=(0.111333643728+0.499535422264j)
            arg[(2, 1, 2, 2)]=(0.930389228567-0.0466491164635j)*x[0] + ((-0.803536803084+0.740138546843j))*x[1]
            ref[(2, 1, 2, 2)]=(0.0634262127418+0.34674471519j)
            arg[(2, 1, 2, 3)]=(-0.950468452689+0.309963076665j)*x[0] + ((0.873786279708+0.578523695027j))*x[1]
            ref[(2, 1, 2, 3)]=(-0.0383410864904+0.444243385846j)
            arg[(2, 1, 3, 0)]=(-0.788629357874-0.984952718934j)*x[0] + ((0.610425723269+0.666517983405j))*x[1]
            ref[(2, 1, 3, 0)]=(-0.0891018173027-0.159217367765j)
            arg[(2, 1, 3, 1)]=(-0.61380790428+0.988486635287j)*x[0] + ((0.0718934236994+0.657369207824j))*x[1]
            ref[(2, 1, 3, 1)]=(-0.27095724029+0.822927921555j)
            arg[(2, 1, 3, 2)]=(-0.882503120127-0.240463198919j)*x[0] + ((-0.269992447419-0.426400997923j))*x[1]
            ref[(2, 1, 3, 2)]=(-0.576247783773-0.333432098421j)
            arg[(2, 1, 3, 3)]=(-0.720180079388+0.826031123601j)*x[0] + ((0.426596520533-0.417372093234j))*x[1]
            ref[(2, 1, 3, 3)]=(-0.146791779427+0.204329515183j)
            arg[(2, 2, 0, 0)]=(-0.143886842561-0.055747065674j)*x[0] + ((0.308679724646-0.947758791349j))*x[1]
            ref[(2, 2, 0, 0)]=(0.0823964410424-0.501752928511j)
            arg[(2, 2, 0, 1)]=(0.935052319752+0.684884415956j)*x[0] + ((0.975381189681-0.999063347903j))*x[1]
            ref[(2, 2, 0, 1)]=(0.955216754716-0.157089465973j)
            arg[(2, 2, 0, 2)]=(-0.00639706036925+0.820897530641j)*x[0] + ((0.41548689582+0.523469466355j))*x[1]
            ref[(2, 2, 0, 2)]=(0.204544917725+0.672183498498j)
            arg[(2, 2, 0, 3)]=(0.767948410611+0.119679724749j)*x[0] + ((0.0968879355151+0.287986649519j))*x[1]
            ref[(2, 2, 0, 3)]=(0.432418173063+0.203833187134j)
            arg[(2, 2, 1, 0)]=(0.104474475603+0.819571565792j)*x[0] + ((0.977495568975-0.147582276336j))*x[1]
            ref[(2, 2, 1, 0)]=(0.540985022289+0.335994644728j)
            arg[(2, 2, 1, 1)]=(-0.678582800894-0.277487866637j)*x[0] + ((-0.722068517614+0.153061734789j))*x[1]
            ref[(2, 2, 1, 1)]=(-0.700325659254-0.0622130659242j)
            arg[(2, 2, 1, 2)]=(-0.590248769495+0.598190828327j)*x[0] + ((-0.284350649278-0.310558193835j))*x[1]
            ref[(2, 2, 1, 2)]=(-0.437299709386+0.143816317246j)
            arg[(2, 2, 1, 3)]=(-0.127585205684-0.399331457967j)*x[0] + ((0.173415494159+0.445127483763j))*x[1]
            ref[(2, 2, 1, 3)]=(0.0229151442376+0.0228980128981j)
            arg[(2, 2, 2, 0)]=(0.511055297611-0.155118018383j)*x[0] + ((-0.807821351371+0.510768988447j))*x[1]
            ref[(2, 2, 2, 0)]=(-0.14838302688+0.177825485032j)
            arg[(2, 2, 2, 1)]=(0.332744382965+0.419647942949j)*x[0] + ((-0.82148417497-0.0688524521819j))*x[1]
            ref[(2, 2, 2, 1)]=(-0.244369896003+0.175397745384j)
            arg[(2, 2, 2, 2)]=(0.874018173338+0.551626501142j)*x[0] + ((0.307327533891+0.378615909478j))*x[1]
            ref[(2, 2, 2, 2)]=(0.590672853614+0.46512120531j)
            arg[(2, 2, 2, 3)]=(0.797300503691+0.912184419598j)*x[0] + ((0.247193971887-0.377312126123j))*x[1]
            ref[(2, 2, 2, 3)]=(0.522247237789+0.267436146737j)
            arg[(2, 2, 3, 0)]=(-0.676291102846+0.327240086728j)*x[0] + ((-0.92559577065+0.476458042849j))*x[1]
            ref[(2, 2, 3, 0)]=(-0.800943436748+0.401849064788j)
            arg[(2, 2, 3, 1)]=(-0.773693231029+0.21911240799j)*x[0] + ((-0.375841021765+0.506822512114j))*x[1]
            ref[(2, 2, 3, 1)]=(-0.574767126397+0.362967460052j)
            arg[(2, 2, 3, 2)]=(0.377578092106-0.34301207467j)*x[0] + ((-0.664892337648-0.669383643041j))*x[1]
            ref[(2, 2, 3, 2)]=(-0.143657122771-0.506197858855j)
            arg[(2, 2, 3, 3)]=(-0.0596747110262-0.719161124288j)*x[0] + ((-0.51123088535+0.0946174330004j))*x[1]
            ref[(2, 2, 3, 3)]=(-0.285452798188-0.312271845644j)
            arg[(2, 3, 0, 0)]=(-0.166849108665+0.217162652534j)*x[0] + ((-0.334725130406-0.798607900159j))*x[1]
            ref[(2, 3, 0, 0)]=(-0.250787119535-0.290722623813j)
            arg[(2, 3, 0, 1)]=(0.217767011229-0.777679256284j)*x[0] + ((-0.499205536488-0.0720314690809j))*x[1]
            ref[(2, 3, 0, 1)]=(-0.140719262629-0.424855362683j)
            arg[(2, 3, 0, 2)]=(-0.714610301181-0.69079402159j)*x[0] + ((-0.133316658655-0.992917015766j))*x[1]
            ref[(2, 3, 0, 2)]=(-0.423963479918-0.841855518678j)
            arg[(2, 3, 0, 3)]=(-0.278973081309+0.115658462745j)*x[0] + ((-0.966173315849-0.992900874377j))*x[1]
            ref[(2, 3, 0, 3)]=(-0.622573198579-0.438621205816j)
            arg[(2, 3, 1, 0)]=(0.576439558915-0.589836033524j)*x[0] + ((0.709406938635+0.713140879898j))*x[1]
            ref[(2, 3, 1, 0)]=(0.642923248775+0.0616524231869j)
            arg[(2, 3, 1, 1)]=(-0.787659668002-0.0530407303765j)*x[0] + ((-0.365404913253-0.865467353191j))*x[1]
            ref[(2, 3, 1, 1)]=(-0.576532290627-0.459254041784j)
            arg[(2, 3, 1, 2)]=(-0.235752793854+0.754183731212j)*x[0] + ((0.617280332483+0.698840392992j))*x[1]
            ref[(2, 3, 1, 2)]=(0.190763769315+0.726512062102j)
            arg[(2, 3, 1, 3)]=(-0.439074944988+0.98873359369j)*x[0] + ((0.565688920427+0.802748854419j))*x[1]
            ref[(2, 3, 1, 3)]=(0.0633069877197+0.895741224054j)
            arg[(2, 3, 2, 0)]=(0.595382686103+0.681237349894j)*x[0] + ((0.482821857458-0.302601749706j))*x[1]
            ref[(2, 3, 2, 0)]=(0.539102271781+0.189317800094j)
            arg[(2, 3, 2, 1)]=(0.690304087426-0.173940294352j)*x[0] + ((0.574813065106-0.06464636053j))*x[1]
            ref[(2, 3, 2, 1)]=(0.632558576266-0.119293327441j)
            arg[(2, 3, 2, 2)]=(-0.641636457922+0.77221333228j)*x[0] + ((-0.651338777057+0.777290761657j))*x[1]
            ref[(2, 3, 2, 2)]=(-0.64648761749+0.774752046968j)
            arg[(2, 3, 2, 3)]=(-0.607860408138-0.821315053635j)*x[0] + ((-0.000602278617829+0.696698552798j))*x[1]
            ref[(2, 3, 2, 3)]=(-0.304231343378-0.0623082504183j)
            arg[(2, 3, 3, 0)]=(0.879621541734-0.306629974229j)*x[0] + ((0.0170107336919-0.865154160067j))*x[1]
            ref[(2, 3, 3, 0)]=(0.448316137713-0.585892067148j)
            arg[(2, 3, 3, 1)]=(-0.856498373252-0.183510826022j)*x[0] + ((0.906143784207+0.419516037972j))*x[1]
            ref[(2, 3, 3, 1)]=(0.0248227054778+0.118002605975j)
            arg[(2, 3, 3, 2)]=(-0.740419363345-0.131548573436j)*x[0] + ((-0.268790525383-0.0962973313097j))*x[1]
            ref[(2, 3, 3, 2)]=(-0.504604944364-0.113922952373j)
            arg[(2, 3, 3, 3)]=(0.110108984886-0.476973806551j)*x[0] + ((-0.636727371565+0.363923234697j))*x[1]
            ref[(2, 3, 3, 3)]=(-0.263309193339-0.0565252859269j)
        else:
            arg[(0, 0, 0, 0)]=(0.25079438227+0.106563566718j)*x[0] + ((-0.539756677859+0.654918919864j))*x[1] + ((-0.229271766422-0.19970114257j))*x[2]
            ref[(0, 0, 0, 0)]=(-0.259117031006+0.280890672006j)
            arg[(0, 0, 0, 1)]=(-0.360144094012-0.760668200218j)*x[0] + ((-0.0566009000981+0.184629877505j))*x[1] + ((0.608200057522+0.689936222845j))*x[2]
            ref[(0, 0, 0, 1)]=(0.0957275317057+0.0569489500657j)
            arg[(0, 0, 0, 2)]=(-0.143031466659+0.735283624084j)*x[0] + ((0.00503675627267-0.780844385963j))*x[1] + ((-0.192370178253+0.147931909488j))*x[2]
            ref[(0, 0, 0, 2)]=(-0.16518244432+0.0511855738049j)
            arg[(0, 0, 0, 3)]=(-0.280541119008-0.527180711645j)*x[0] + ((0.0577199459463-0.89522822522j))*x[1] + ((-0.310279036671-0.174145588692j))*x[2]
            ref[(0, 0, 0, 3)]=(-0.266550104867-0.798277262778j)
            arg[(0, 0, 1, 0)]=(-0.314481599412-0.891701832441j)*x[0] + ((0.966452044271+0.71731740116j))*x[1] + ((0.120018425359+0.910830735383j))*x[2]
            ref[(0, 0, 1, 0)]=(0.385994435109+0.368223152051j)
            arg[(0, 0, 1, 1)]=(-0.681392034517-0.338227291952j)*x[0] + ((-0.587752645153-0.712169712912j))*x[1] + ((0.293978808607+0.505947599193j))*x[2]
            ref[(0, 0, 1, 1)]=(-0.487582935531-0.272224702835j)
            arg[(0, 0, 1, 2)]=(0.761859153994-0.21879382482j)*x[0] + ((-0.352817019049-0.389944067014j))*x[1] + ((0.533198366097+0.401909767217j))*x[2]
            ref[(0, 0, 1, 2)]=(0.471120250521-0.103414062308j)
            arg[(0, 0, 1, 3)]=(-0.678929680649+0.601379724545j)*x[0] + ((-0.568634331114-0.393486269867j))*x[1] + ((0.375486699561+0.535900353601j))*x[2]
            ref[(0, 0, 1, 3)]=(-0.436038656101+0.371896904139j)
            arg[(0, 0, 2, 0)]=(0.83161257539-0.483388825577j)*x[0] + ((0.861970644656+0.876096050215j))*x[1] + ((0.507577972441+0.179051900546j))*x[2]
            ref[(0, 0, 2, 0)]=(1.10058059624+0.285879562592j)
            arg[(0, 0, 2, 1)]=(-0.276713601784+0.53617588699j)*x[0] + ((-0.849121810109+0.743042971485j))*x[1] + ((-0.0979445946383-0.462267758982j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.611890003266+0.408475549747j)
            arg[(0, 0, 2, 2)]=(0.639436323547-0.0232010239288j)*x[0] + ((0.37341919766-0.329038224942j))*x[1] + ((-0.589718287894-0.420565570254j))*x[2]
            ref[(0, 0, 2, 2)]=(0.211568616657-0.386402409562j)
            arg[(0, 0, 2, 3)]=(0.132433840783+0.300346008394j)*x[0] + ((0.620170047826+0.371299883304j))*x[1] + ((0.0123337212661+0.425007268237j))*x[2]
            ref[(0, 0, 2, 3)]=(0.382468804937+0.548326579968j)
            arg[(0, 0, 3, 0)]=(0.780293631528+0.363677936521j)*x[0] + ((-0.421281186627-0.154299396467j))*x[1] + ((-0.934687163749-0.304435206358j))*x[2]
            ref[(0, 0, 3, 0)]=(-0.287837359424-0.0475283331519j)
            arg[(0, 0, 3, 1)]=(-0.679692517958-0.260939924987j)*x[0] + ((0.915321454968+0.837643494008j))*x[1] + ((-0.641409478781-0.64030569604j))*x[2]
            ref[(0, 0, 3, 1)]=(-0.202890270886-0.0318010635097j)
            arg[(0, 0, 3, 2)]=(0.995071292872-0.0334878002244j)*x[0] + ((-0.177622772649+0.00511944858329j))*x[1] + ((-0.907742605588-0.469452618361j))*x[2]
            ref[(0, 0, 3, 2)]=(-0.0451470426822-0.248910485001j)
            arg[(0, 0, 3, 3)]=(-0.958695004783+0.252770186004j)*x[0] + ((-0.713108074919+0.623894528172j))*x[1] + ((0.940030818303+0.651839876764j))*x[2]
            ref[(0, 0, 3, 3)]=(-0.365886130699+0.76425229547j)
            arg[(0, 1, 0, 0)]=(-0.985332370105+0.806165140015j)*x[0] + ((-0.349087806414-0.665840370558j))*x[1] + ((-0.891610970657+0.236636808666j))*x[2]
            ref[(0, 1, 0, 0)]=(-1.11301557359+0.188480789061j)
            arg[(0, 1, 0, 1)]=(0.00539720757708-0.906423351203j)*x[0] + ((0.530522545776+0.251951312358j))*x[1] + ((0.153315656734-0.622812831783j))*x[2]
            ref[(0, 1, 0, 1)]=(0.344617705044-0.638642435314j)
            arg[(0, 1, 0, 2)]=(-0.894753821876-0.035381633613j)*x[0] + ((0.792279768864+0.29853664536j))*x[1] + ((0.721249666026+0.661170362101j))*x[2]
            ref[(0, 1, 0, 2)]=(0.309387806507+0.462162686924j)
            arg[(0, 1, 0, 3)]=(-0.896235197866-0.727054591894j)*x[0] + ((0.101754756173+0.839166986181j))*x[1] + ((0.46418044196-0.0053240677835j))*x[2]
            ref[(0, 1, 0, 3)]=(-0.165149999866+0.0533941632518j)
            arg[(0, 1, 1, 0)]=(-0.707114746615-0.133341687438j)*x[0] + ((0.367960388809+0.138553328319j))*x[1] + ((0.0934875928193-0.875091557189j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.122833382493-0.434939958154j)
            arg[(0, 1, 1, 1)]=(0.298302776537+0.980712251983j)*x[0] + ((-0.996188489445+0.290594759615j))*x[1] + ((-0.708720165837+0.87788817774j))*x[2]
            ref[(0, 1, 1, 1)]=(-0.703302939373+1.07459759467j)
            arg[(0, 1, 1, 2)]=(0.205490381472-0.718906355569j)*x[0] + ((-0.5865004781-0.807566575096j))*x[1] + ((0.93392082258-0.303254996547j))*x[2]
            ref[(0, 1, 1, 2)]=(0.276455362976-0.914863963607j)
            arg[(0, 1, 1, 3)]=(-0.847970116993+0.156817944047j)*x[0] + ((0.596690831515+0.738827385437j))*x[1] + ((-0.691805780277-0.394123053425j))*x[2]
            ref[(0, 1, 1, 3)]=(-0.471542532878+0.25076113803j)
            arg[(0, 1, 2, 0)]=(-0.13699360836+0.335065641641j)*x[0] + ((-0.976944780514+0.969272381572j))*x[1] + ((0.865670667906-0.990262565501j))*x[2]
            ref[(0, 1, 2, 0)]=(-0.124133860484+0.157037728856j)
            arg[(0, 1, 2, 1)]=(-0.639031811895+0.90927984208j)*x[0] + ((0.13142351986-0.501629007955j))*x[1] + ((-0.188574509416+0.258579853255j))*x[2]
            ref[(0, 1, 2, 1)]=(-0.348091400726+0.33311534369j)
            arg[(0, 1, 2, 2)]=(-0.829351911232+0.237048845637j)*x[0] + ((0.217185839292+0.427752581604j))*x[1] + ((0.964498783776+0.721704668378j))*x[2]
            ref[(0, 1, 2, 2)]=(0.176166355918+0.693253047809j)
            arg[(0, 1, 2, 3)]=(0.616845544009+0.177568009549j)*x[0] + ((0.574279814185-0.270939131974j))*x[1] + ((-0.856307029023-0.371782286432j))*x[2]
            ref[(0, 1, 2, 3)]=(0.167409164586-0.232576704429j)
            arg[(0, 1, 3, 0)]=(0.00211299305989-0.756445866244j)*x[0] + ((-0.879128146018+0.885936842799j))*x[1] + ((0.660285352089+0.0603754647357j))*x[2]
            ref[(0, 1, 3, 0)]=(-0.108364900434+0.0949332206453j)
            arg[(0, 1, 3, 1)]=(0.213071802576-0.866788849286j)*x[0] + ((0.254685371996+0.508776225167j))*x[1] + ((-0.421850544929+0.847720098719j))*x[2]
            ref[(0, 1, 3, 1)]=(0.0229533148213+0.2448537373j)
            arg[(0, 1, 3, 2)]=(0.513038385559+0.484287413648j)*x[0] + ((0.026342897834+0.470890678481j))*x[1] + ((-0.421690697783-0.420723744913j))*x[2]
            ref[(0, 1, 3, 2)]=(0.0588452928049+0.267227173608j)
            arg[(0, 1, 3, 3)]=(-0.0622737569375+0.212203832184j)*x[0] + ((-0.701661958314-0.667031660842j))*x[1] + ((0.785844066244+0.585447718724j))*x[2]
            ref[(0, 1, 3, 3)]=(0.0109541754964+0.0653099450329j)
            arg[(0, 2, 0, 0)]=(0.957257276978-0.708015907083j)*x[0] + ((-0.403462488178+0.216618324824j))*x[1] + ((0.162717359954+0.349200448107j))*x[2]
            ref[(0, 2, 0, 0)]=(0.358256074377-0.0710985670758j)
            arg[(0, 2, 0, 1)]=(-0.952453086542+0.887892330814j)*x[0] + ((0.240241994737+0.784934700617j))*x[1] + ((-0.585106894973-0.475395088199j))*x[2]
            ref[(0, 2, 0, 1)]=(-0.648658993389+0.598715971616j)
            arg[(0, 2, 0, 2)]=(-0.983682009164+0.496605105853j)*x[0] + ((0.569013235207+0.659405814985j))*x[1] + ((-0.39576325067+0.611844450548j))*x[2]
            ref[(0, 2, 0, 2)]=(-0.405216012314+0.883927685693j)
            arg[(0, 2, 0, 3)]=(0.898155316951-0.180476207353j)*x[0] + ((-0.896221252608+0.282100085681j))*x[1] + ((-0.277754564188-0.10984917585j))*x[2]
            ref[(0, 2, 0, 3)]=(-0.137910249922-0.00411264876093j)
            arg[(0, 2, 1, 0)]=(0.30638019373-0.141117586321j)*x[0] + ((0.991394762666+0.287874813847j))*x[1] + ((0.985589996505-0.765972882634j))*x[2]
            ref[(0, 2, 1, 0)]=(1.14168247645-0.309607827554j)
            arg[(0, 2, 1, 1)]=(0.325266461531-0.2855438237j)*x[0] + ((-0.598038210923+0.0954342322422j))*x[1] + ((0.900372049704+0.943526671672j))*x[2]
            ref[(0, 2, 1, 1)]=(0.313800150156+0.376708540107j)
            arg[(0, 2, 1, 2)]=(-0.625704694784+0.471471298501j)*x[0] + ((0.187111397526-0.994848052042j))*x[1] + ((-0.251652049262-0.100880418169j))*x[2]
            ref[(0, 2, 1, 2)]=(-0.34512267326-0.312128585855j)
            arg[(0, 2, 1, 3)]=(-0.978130483466+0.620790704636j)*x[0] + ((-0.145408135959-0.343648523275j))*x[1] + ((-0.208937218652-0.713165397486j))*x[2]
            ref[(0, 2, 1, 3)]=(-0.666237919038-0.218011608062j)
            arg[(0, 2, 2, 0)]=(0.710342123303-0.863205010593j)*x[0] + ((-0.0983735286536-0.624183784029j))*x[1] + ((-0.332797079408-0.664204172135j))*x[2]
            ref[(0, 2, 2, 0)]=(0.139585757621-1.07579648338j)
            arg[(0, 2, 2, 1)]=(-0.262513610002+0.852937831047j)*x[0] + ((-0.711994457144-0.484874432059j))*x[1] + ((-0.678476659796-0.950133396119j))*x[2]
            ref[(0, 2, 2, 1)]=(-0.826492363471-0.291034998565j)
            arg[(0, 2, 2, 2)]=(-0.828415324497-0.204194774631j)*x[0] + ((-0.57457910809-0.990664841514j))*x[1] + ((-0.569000597296-0.242796429118j))*x[2]
            ref[(0, 2, 2, 2)]=(-0.985997514941-0.718828022632j)
            arg[(0, 2, 2, 3)]=(0.318381855444-0.502936215079j)*x[0] + ((-0.600303085345+0.037375703564j))*x[1] + ((0.119204793087-0.00488278928839j))*x[2]
            ref[(0, 2, 2, 3)]=(-0.0813582184073-0.235221650402j)
            arg[(0, 2, 3, 0)]=(0.872109529094+0.142953013112j)*x[0] + ((-0.219127913681-0.045881366978j))*x[1] + ((-0.926225044763-0.104121036518j))*x[2]
            ref[(0, 2, 3, 0)]=(-0.136621714675-0.00352469519212j)
            arg[(0, 2, 3, 1)]=(0.0459993216195+0.303168407318j)*x[0] + ((-0.188660047063-0.920839136507j))*x[1] + ((-0.331089509785-0.343872559064j))*x[2]
            ref[(0, 2, 3, 1)]=(-0.236875117614-0.480771644126j)
            arg[(0, 2, 3, 2)]=(0.688011100211-0.798223777944j)*x[0] + ((0.62980547653-0.585821352242j))*x[1] + ((-0.291379967441+0.788149678821j))*x[2]
            ref[(0, 2, 3, 2)]=(0.51321830465-0.297947725683j)
            arg[(0, 2, 3, 3)]=(0.853668086548+0.368412055807j)*x[0] + ((0.069753644773-0.384646604539j))*x[1] + ((-0.117753591846-0.0742016740578j))*x[2]
            ref[(0, 2, 3, 3)]=(0.402834069737-0.0452181113946j)
            arg[(0, 3, 0, 0)]=(0.445452911805+0.842560889686j)*x[0] + ((-0.517650371624-0.663705548319j))*x[1] + ((-0.825564973766+0.943173379176j))*x[2]
            ref[(0, 3, 0, 0)]=(-0.448881216793+0.561014360272j)
            arg[(0, 3, 0, 1)]=(0.713408393368+0.592421769423j)*x[0] + ((-0.80232124459+0.694855605401j))*x[1] + ((-0.0104336210865+0.881219358176j))*x[2]
            ref[(0, 3, 0, 1)]=(-0.0496732361544+1.0842483665j)
            arg[(0, 3, 0, 2)]=(-0.808930466867+0.439108847932j)*x[0] + ((0.325544801308+0.893431150308j))*x[1] + ((0.208692125889+0.31071675515j))*x[2]
            ref[(0, 3, 0, 2)]=(-0.137346769835+0.821628376695j)
            arg[(0, 3, 0, 3)]=(0.764753580921+0.745475752123j)*x[0] + ((-0.365206697582-0.273519869905j))*x[1] + ((-0.302575396187-0.122401734941j))*x[2]
            ref[(0, 3, 0, 3)]=(0.048485743576+0.174777073638j)
            arg[(0, 3, 1, 0)]=(0.467499116478-0.990210875277j)*x[0] + ((0.469060353775-0.889986597683j))*x[1] + ((0.0493840246343+0.207274876832j))*x[2]
            ref[(0, 3, 1, 0)]=(0.492971747443-0.836461298064j)
            arg[(0, 3, 1, 1)]=(0.692039926039+0.543123901106j)*x[0] + ((-0.631020671108-0.232829689187j))*x[1] + ((-0.845515008315-0.847554479962j))*x[2]
            ref[(0, 3, 1, 1)]=(-0.392247876692-0.268630134021j)
            arg[(0, 3, 1, 2)]=(-0.411686062991+0.259269847622j)*x[0] + ((0.466090122159-0.255242059134j))*x[1] + ((0.173341205094-0.668625221557j))*x[2]
            ref[(0, 3, 1, 2)]=(0.113872632131-0.332298716534j)
            arg[(0, 3, 1, 3)]=(0.490902040488-0.749746701039j)*x[0] + ((-0.41187318505-0.407665686114j))*x[1] + ((-0.349655176819+0.497580612363j))*x[2]
            ref[(0, 3, 1, 3)]=(-0.135313160691-0.329915887395j)
            arg[(0, 3, 2, 0)]=(0.376176731622-0.792239165066j)*x[0] + ((-0.315052162038-0.777150567928j))*x[1] + ((0.466457689784-0.123135720603j))*x[2]
            ref[(0, 3, 2, 0)]=(0.263791129684-0.846262726799j)
            arg[(0, 3, 2, 1)]=(0.234178338525+0.543990702719j)*x[0] + ((-0.725619767593+0.464826393853j))*x[1] + ((-0.871150720881-0.0797215734799j))*x[2]
            ref[(0, 3, 2, 1)]=(-0.681296074975+0.464547761546j)
            arg[(0, 3, 2, 2)]=(-0.1569796309-0.493154279485j)*x[0] + ((-0.774878258573+0.762343890514j))*x[1] + ((0.0866534707778-0.66201120007j))*x[2]
            ref[(0, 3, 2, 2)]=(-0.422602209347-0.196410794521j)
            arg[(0, 3, 2, 3)]=(-0.245103071017+0.879497819806j)*x[0] + ((0.0427642315021+0.49677749922j))*x[1] + ((0.293080845936+0.604270531099j))*x[2]
            ref[(0, 3, 2, 3)]=(0.0453710032103+0.990272925063j)
            arg[(0, 3, 3, 0)]=(0.266192947115-0.319663208199j)*x[0] + ((0.857482543085-0.555250368648j))*x[1] + ((-0.229984508976+0.31239627784j))*x[2]
            ref[(0, 3, 3, 0)]=(0.446845490612-0.281258649504j)
            arg[(0, 3, 3, 1)]=(0.0154961260811-0.270834947531j)*x[0] + ((-0.108623205104-0.331543973975j))*x[1] + ((0.825087839729+0.145980058414j))*x[2]
            ref[(0, 3, 3, 1)]=(0.365980380353-0.228199431546j)
            arg[(0, 3, 3, 2)]=(-0.202400703155+0.527850120825j)*x[0] + ((0.655661833348-0.0216828758455j))*x[1] + ((0.610718430707-0.964331433658j))*x[2]
            ref[(0, 3, 3, 2)]=(0.53198978045-0.229082094339j)
            arg[(0, 3, 3, 3)]=(0.367407765676-0.1843668476j)*x[0] + ((0.297167512293-0.996243863055j))*x[1] + ((-0.14043162975+0.687062010968j))*x[2]
            ref[(0, 3, 3, 3)]=(0.26207182411-0.246774349844j)
            arg[(1, 0, 0, 0)]=(-0.924594018883+0.014096844313j)*x[0] + ((-0.287449664102-0.28698065028j))*x[1] + ((-0.216838651506-0.956915377458j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.714441167246-0.614899591713j)
            arg[(1, 0, 0, 1)]=(-0.488902475986-0.694842444094j)*x[0] + ((-0.70048675159+0.671989981693j))*x[1] + ((-0.286883570619+0.757972467698j))*x[2]
            ref[(1, 0, 0, 1)]=(-0.738136399097+0.367560002649j)
            arg[(1, 0, 0, 2)]=(0.200846307244-0.180035753398j)*x[0] + ((0.21225455635-0.599224257229j))*x[1] + ((0.779740587216+0.808598103046j))*x[2]
            ref[(1, 0, 0, 2)]=(0.596420725405+0.0146690462097j)
            arg[(1, 0, 0, 3)]=(0.447361844153-0.529689857705j)*x[0] + ((-0.309317743752-0.153684663693j))*x[1] + ((0.0304026627658+0.927221550366j))*x[2]
            ref[(1, 0, 0, 3)]=(0.0842233815838+0.121923514484j)
            arg[(1, 0, 1, 0)]=(0.517109226141+0.00812147642989j)*x[0] + ((-0.0707014926248-0.239869911334j))*x[1] + ((-0.393554772373+0.916163630947j))*x[2]
            ref[(1, 0, 1, 0)]=(0.0264264805716+0.342207598021j)
            arg[(1, 0, 1, 1)]=(-0.580924703564+0.312008445864j)*x[0] + ((-0.513016333193-0.331795936851j))*x[1] + ((0.72692717093-0.566419552064j))*x[2]
            ref[(1, 0, 1, 1)]=(-0.183506932914-0.293103521525j)
            arg[(1, 0, 1, 2)]=(0.557869184326+0.528669748845j)*x[0] + ((0.811049012275+0.263859848239j))*x[1] + ((0.102931909898+0.890980884941j))*x[2]
            ref[(1, 0, 1, 2)]=(0.73592505325+0.841755241013j)
            arg[(1, 0, 1, 3)]=(0.855713367663-0.926916196273j)*x[0] + ((-0.096142722774-0.99751630312j))*x[1] + ((0.52961845612+0.0588961264611j))*x[2]
            ref[(1, 0, 1, 3)]=(0.644594550505-0.932768186466j)
            arg[(1, 0, 2, 0)]=(-0.730511502731+0.716774133249j)*x[0] + ((0.847811474492+0.436138229212j))*x[1] + ((0.710643047367+0.212938705938j))*x[2]
            ref[(1, 0, 2, 0)]=(0.413971509564+0.6829255342j)
            arg[(1, 0, 2, 1)]=(-0.516499289587+0.873955474164j)*x[0] + ((0.530221603854+0.977011565421j))*x[1] + ((0.860000557801+0.910393618255j))*x[2]
            ref[(1, 0, 2, 1)]=(0.436861436034+1.38068032892j)
            arg[(1, 0, 2, 2)]=(0.0912435217619+0.127080453911j)*x[0] + ((0.78625582807-0.236986770051j))*x[1] + ((-0.16524554192+0.995485229004j))*x[2]
            ref[(1, 0, 2, 2)]=(0.356126903956+0.442789456432j)
            arg[(1, 0, 2, 3)]=(-0.0178032471474-0.450343805793j)*x[0] + ((0.486522550437+0.353499381846j))*x[1] + ((0.223678568227+0.26896614777j))*x[2]
            ref[(1, 0, 2, 3)]=(0.346198935758+0.0860608619117j)
            arg[(1, 0, 3, 0)]=(0.959014047765+0.913645317293j)*x[0] + ((-0.647122631371+0.797240377712j))*x[1] + ((-0.0855332179067-0.620686067577j))*x[2]
            ref[(1, 0, 3, 0)]=(0.113179099243+0.545099813714j)
            arg[(1, 0, 3, 1)]=(0.785857875884-0.993220085389j)*x[0] + ((0.609180208933+0.658984986531j))*x[1] + ((0.627030400538+0.525644679149j))*x[2]
            ref[(1, 0, 3, 1)]=(1.01103424268+0.0957047901454j)
            arg[(1, 0, 3, 2)]=(0.60103618375-0.790450721644j)*x[0] + ((-0.384713228102+0.880134521241j))*x[1] + ((0.363925546884+0.204942155118j))*x[2]
            ref[(1, 0, 3, 2)]=(0.290124251266+0.147312977357j)
            arg[(1, 0, 3, 3)]=(0.565660937913-0.756399676032j)*x[0] + ((-0.278948915438-0.951285358053j))*x[1] + ((-0.537674046865-0.319748402726j))*x[2]
            ref[(1, 0, 3, 3)]=(-0.125481012195-1.01371671841j)
            arg[(1, 1, 0, 0)]=(0.150836636033+0.216164294877j)*x[0] + ((-0.105647098611-0.433336460861j))*x[1] + ((0.949918227845+0.97817506878j))*x[2]
            ref[(1, 1, 0, 0)]=(0.497553882633+0.380501451398j)
            arg[(1, 1, 0, 1)]=(-0.300545487018-0.768866897956j)*x[0] + ((-0.240096502918-0.598717439662j))*x[1] + ((-0.767035314355-0.82616815529j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.653838652145-1.09687624645j)
            arg[(1, 1, 0, 2)]=(-0.0510734268924+0.158139984545j)*x[0] + ((0.810026019054+0.156424162002j))*x[1] + ((0.602639395952-0.471918086867j))*x[2]
            ref[(1, 1, 0, 2)]=(0.680795994057-0.0786769701602j)
            arg[(1, 1, 0, 3)]=(-0.768583145793-0.720538244794j)*x[0] + ((-0.901420341555-0.212993495122j))*x[1] + ((0.951143998915+0.604992485025j))*x[2]
            ref[(1, 1, 0, 3)]=(-0.359429744217-0.164269627446j)
            arg[(1, 1, 1, 0)]=(-0.215148512796-0.611891549781j)*x[0] + ((-0.613099699462+0.955513504513j))*x[1] + ((0.203415183962-0.468494969531j))*x[2]
            ref[(1, 1, 1, 0)]=(-0.312416514148-0.0624365073994j)
            arg[(1, 1, 1, 1)]=(-0.816724066309-0.633426573211j)*x[0] + ((0.725947925989+0.560508206686j))*x[1] + ((0.929143671502+0.168660313671j))*x[2]
            ref[(1, 1, 1, 1)]=(0.419183765591+0.0478709735732j)
            arg[(1, 1, 1, 2)]=(-0.408121410691-0.877019032946j)*x[0] + ((0.178159857939+0.859320919712j))*x[1] + ((-0.171899017652-0.68138991998j))*x[2]
            ref[(1, 1, 1, 2)]=(-0.200930285202-0.349544016607j)
            arg[(1, 1, 1, 3)]=(-0.739733100259+0.402738642117j)*x[0] + ((-0.95536760545+0.435259384364j))*x[1] + ((-0.397415951818-0.317360789554j))*x[2]
            ref[(1, 1, 1, 3)]=(-1.04625832876+0.260318618464j)
            arg[(1, 1, 2, 0)]=(0.775057181104+0.108120278971j)*x[0] + ((-0.208260530361+0.687539620165j))*x[1] + ((-0.42258753548+0.35880171267j))*x[2]
            ref[(1, 1, 2, 0)]=(0.0721045576319+0.577230805903j)
            arg[(1, 1, 2, 1)]=(0.505048230242+0.115976054997j)*x[0] + ((0.909183698499-0.876891127677j))*x[1] + ((0.988448100589+0.05036300159j))*x[2]
            ref[(1, 1, 2, 1)]=(1.20134001467-0.355276035545j)
            arg[(1, 1, 2, 2)]=(0.99227861867+0.587819203535j)*x[0] + ((-0.702169979237-0.42377085312j))*x[1] + ((0.35273592192-0.112369465269j))*x[2]
            ref[(1, 1, 2, 2)]=(0.321422280676+0.0258394425729j)
            arg[(1, 1, 2, 3)]=(0.696849750639+0.864171078682j)*x[0] + ((-0.980831463293+0.8900172502j))*x[1] + ((-0.659246874949+0.529425191941j))*x[2]
            ref[(1, 1, 2, 3)]=(-0.471614293802+1.14180676041j)
            arg[(1, 1, 3, 0)]=(0.694114748655+0.202361025151j)*x[0] + ((-0.0926448950917-0.276458373611j))*x[1] + ((-0.418459813342-0.0387684903769j))*x[2]
            ref[(1, 1, 3, 0)]=(0.0915050201103-0.0564329194184j)
            arg[(1, 1, 3, 1)]=(-0.170773250788+0.153154455273j)*x[0] + ((0.0511303068069-0.456723624438j))*x[1] + ((0.998179092307-0.761305882762j))*x[2]
            ref[(1, 1, 3, 1)]=(0.439268074163-0.532437525964j)
            arg[(1, 1, 3, 2)]=(0.0430423483978+0.866248355903j)*x[0] + ((0.894740381852+0.783702098704j))*x[1] + ((-0.21610982029+0.631403198981j))*x[2]
            ref[(1, 1, 3, 2)]=(0.36083645498+1.14067682679j)
            arg[(1, 1, 3, 3)]=(0.583770410932-0.517141638847j)*x[0] + ((-0.999681467317-0.960812586441j))*x[1] + ((-0.464071309118+0.427100097576j))*x[2]
            ref[(1, 1, 3, 3)]=(-0.439991182751-0.525427063856j)
            arg[(1, 2, 0, 0)]=(-0.677422347027+0.269026598452j)*x[0] + ((-0.976376780855-0.747469041211j))*x[1] + ((-0.402178303379+0.659951379324j))*x[2]
            ref[(1, 2, 0, 0)]=(-1.02798871563+0.0907544682825j)
            arg[(1, 2, 0, 1)]=(0.01441063794-0.694288941028j)*x[0] + ((-0.987124364539+0.245823788107j))*x[1] + ((-0.66509903977+0.762846899182j))*x[2]
            ref[(1, 2, 0, 1)]=(-0.818906383185+0.157190873131j)
            arg[(1, 2, 0, 2)]=(0.342226663689-0.18184438361j)*x[0] + ((-0.885010489962+0.813238655312j))*x[1] + ((0.90848692819+0.364528308415j))*x[2]
            ref[(1, 2, 0, 2)]=(0.182851550959+0.497961290058j)
            arg[(1, 2, 0, 3)]=(-0.432077957557+0.735156078815j)*x[0] + ((0.632671651235+0.486309490085j))*x[1] + ((0.850050900658-0.439956861694j))*x[2]
            ref[(1, 2, 0, 3)]=(0.525322297168+0.390754353603j)
            arg[(1, 2, 1, 0)]=(0.995274184334-0.0957289809887j)*x[0] + ((-0.743077893602-0.564976195437j))*x[1] + ((-0.432482280094-0.483772205169j))*x[2]
            ref[(1, 2, 1, 0)]=(-0.090142994681-0.572238690797j)
            arg[(1, 2, 1, 1)]=(-0.84590015859+0.927937170494j)*x[0] + ((0.172102641613+0.393501042661j))*x[1] + ((-0.668042583312+0.616734256687j))*x[2]
            ref[(1, 2, 1, 1)]=(-0.670920050144+0.969086234921j)
            arg[(1, 2, 1, 2)]=(0.225760578606-0.590701598166j)*x[0] + ((0.154616101887+0.843857236204j))*x[1] + ((0.512277862583+0.881142864752j))*x[2]
            ref[(1, 2, 1, 2)]=(0.446327271538+0.567149251395j)
            arg[(1, 2, 1, 3)]=(0.785447565926+0.467257993803j)*x[0] + ((-0.926669597787-0.821601530549j))*x[1] + ((0.0908581796752-0.766700663036j))*x[2]
            ref[(1, 2, 1, 3)]=(-0.0251819260932-0.560522099891j)
            arg[(1, 2, 2, 0)]=(0.770077068378+0.264736723003j)*x[0] + ((0.290841992706+0.359936714648j))*x[1] + ((0.526003152456+0.843294749638j))*x[2]
            ref[(1, 2, 2, 0)]=(0.79346110677+0.733984093645j)
            arg[(1, 2, 2, 1)]=(0.527716488452+0.975927230891j)*x[0] + ((0.262514468115+0.864767507034j))*x[1] + ((-0.0638587328897-0.445627253012j))*x[2]
            ref[(1, 2, 2, 1)]=(0.363186111838+0.697533742457j)
            arg[(1, 2, 2, 2)]=(-0.257715123685-0.764312357986j)*x[0] + ((0.647263442238+0.840084142301j))*x[1] + ((0.573542834866-0.762467755101j))*x[2]
            ref[(1, 2, 2, 2)]=(0.48154557671-0.343347985393j)
            arg[(1, 2, 2, 3)]=(-0.604956362233-0.36132690431j)*x[0] + ((0.643485073007+0.637126443322j))*x[1] + ((0.906589091078+0.412765398908j))*x[2]
            ref[(1, 2, 2, 3)]=(0.472558900926+0.34428246896j)
            arg[(1, 2, 3, 0)]=(0.303997608267-0.459568405646j)*x[0] + ((-0.258742682743-0.265902348186j))*x[1] + ((-0.910556185109-0.848921254273j))*x[2]
            ref[(1, 2, 3, 0)]=(-0.432650629793-0.787196004053j)
            arg[(1, 2, 3, 1)]=(0.844387794727+0.163435006486j)*x[0] + ((0.245813576127+0.439686415461j))*x[1] + ((-0.194072889911-0.601050537177j))*x[2]
            ref[(1, 2, 3, 1)]=(0.448064240471+0.001035442385j)
            arg[(1, 2, 3, 2)]=(0.306669288597-0.906879132739j)*x[0] + ((-0.248331283111-0.763255740634j))*x[1] + ((0.13062919846+0.702602540122j))*x[2]
            ref[(1, 2, 3, 2)]=(0.0944836019728-0.483766166625j)
            arg[(1, 2, 3, 3)]=(-0.847378294386-0.400338468092j)*x[0] + ((-0.218146704283-0.164011445225j))*x[1] + ((0.170759761347-0.319168757178j))*x[2]
            ref[(1, 2, 3, 3)]=(-0.447382618661-0.441759335248j)
            arg[(1, 3, 0, 0)]=(0.134433641287+0.714319239155j)*x[0] + ((-0.973918424656-0.540440954469j))*x[1] + ((0.232339213284+0.683548767264j))*x[2]
            ref[(1, 3, 0, 0)]=(-0.303572785043+0.428713525975j)
            arg[(1, 3, 0, 1)]=(-0.462209576833+0.0721486565275j)*x[0] + ((-0.558286063529-0.300342041063j))*x[1] + ((-0.0278952482044+0.781163651199j))*x[2]
            ref[(1, 3, 0, 1)]=(-0.524195444283+0.276485133332j)
            arg[(1, 3, 0, 2)]=(0.756561786403+0.653046211178j)*x[0] + ((-0.958242709125-0.945688628057j))*x[1] + ((-0.451013800449-0.732217855404j))*x[2]
            ref[(1, 3, 0, 2)]=(-0.326347361586-0.512430136142j)
            arg[(1, 3, 0, 3)]=(-0.173356683066+0.213870343609j)*x[0] + ((-0.441743633019+0.707751569873j))*x[1] + ((-0.881405528701+0.968635189023j))*x[2]
            ref[(1, 3, 0, 3)]=(-0.748252922393+0.945128551253j)
            arg[(1, 3, 1, 0)]=(-0.445084465901+0.507803513218j)*x[0] + ((0.837455576108-0.313905039474j))*x[1] + ((-0.616198300289-0.461679336272j))*x[2]
            ref[(1, 3, 1, 0)]=(-0.111913595041-0.133890431264j)
            arg[(1, 3, 1, 1)]=(0.847090233437-0.316911208179j)*x[0] + ((-0.02075430359-0.164809791029j))*x[1] + ((-0.629005527168-0.674965424734j))*x[2]
            ref[(1, 3, 1, 1)]=(0.0986652013395-0.578343211971j)
            arg[(1, 3, 1, 2)]=(0.199345152379-0.572975275518j)*x[0] + ((0.883403095313+0.214319193015j))*x[1] + ((0.0327259927572-0.826782559349j))*x[2]
            ref[(1, 3, 1, 2)]=(0.557737120225-0.592719320926j)
            arg[(1, 3, 1, 3)]=(0.0453806865587-0.226097441819j)*x[0] + ((0.38170495826+0.282825983257j))*x[1] + ((-0.392903701024-0.813554771052j))*x[2]
            ref[(1, 3, 1, 3)]=(0.0170909718974-0.378413114807j)
            arg[(1, 3, 2, 0)]=(0.883061126956+0.457636950132j)*x[0] + ((0.942222625126+0.958372179596j))*x[1] + ((0.192531469445+0.819380050469j))*x[2]
            ref[(1, 3, 2, 0)]=(1.00890761076+1.1176945901j)
            arg[(1, 3, 2, 1)]=(0.0409741415363-0.729956152673j)*x[0] + ((0.586942654765+0.487293608526j))*x[1] + ((0.989264019632+0.53260267139j))*x[2]
            ref[(1, 3, 2, 1)]=(0.808590407966+0.144970063622j)
            arg[(1, 3, 2, 2)]=(0.94089630585-0.908597116116j)*x[0] + ((-0.664236314165+0.653694923767j))*x[1] + ((0.430568103572+0.432014523857j))*x[2]
            ref[(1, 3, 2, 2)]=(0.353614047628+0.0885561657539j)
            arg[(1, 3, 2, 3)]=(-0.686989937214+0.601587350734j)*x[0] + ((0.853283216374-0.586150759704j))*x[1] + ((0.239058878848-0.701776680052j))*x[2]
            ref[(1, 3, 2, 3)]=(0.202676079004-0.343170044511j)
            arg[(1, 3, 3, 0)]=(0.214957920505+0.521262648653j)*x[0] + ((0.39626997539+0.852536488043j))*x[1] + ((0.486746003618+0.42893875927j))*x[2]
            ref[(1, 3, 3, 0)]=(0.548986949757+0.901368947983j)
            arg[(1, 3, 3, 1)]=(-0.636057464865+0.454310462425j)*x[0] + ((0.745005702373+0.199633555417j))*x[1] + ((-0.946226183883+0.706990855543j))*x[2]
            ref[(1, 3, 3, 1)]=(-0.418638973187+0.680467436693j)
            arg[(1, 3, 3, 2)]=(-0.226468825975+0.00185688022555j)*x[0] + ((0.092880930325-0.963735961703j))*x[1] + ((-0.635884453178-0.0878318679024j))*x[2]
            ref[(1, 3, 3, 2)]=(-0.384736174414-0.52485547469j)
            arg[(1, 3, 3, 3)]=(-0.801221797236+0.0366796083324j)*x[0] + ((-0.443803013+0.663309058867j))*x[1] + ((0.699760248605-0.20499829718j))*x[2]
            ref[(1, 3, 3, 3)]=(-0.272632280815+0.24749518501j)
            arg[(2, 0, 0, 0)]=(0.769446000299+0.0433553341193j)*x[0] + ((0.692972046926-0.583247850675j))*x[1] + ((-0.391647301624-0.910180184746j))*x[2]
            ref[(2, 0, 0, 0)]=(0.535385372801-0.725036350651j)
            arg[(2, 0, 0, 1)]=(0.138300851446-0.00774814475351j)*x[0] + ((-0.892164482752+0.623980620767j))*x[1] + ((-0.94654545432+0.443194295904j))*x[2]
            ref[(2, 0, 0, 1)]=(-0.850204542813+0.529713385959j)
            arg[(2, 0, 0, 2)]=(0.00168318398192+0.673282819028j)*x[0] + ((0.140680411374+0.366428477923j))*x[1] + ((0.706655663348-0.260259781791j))*x[2]
            ref[(2, 0, 0, 2)]=(0.424509629352+0.38972575758j)
            arg[(2, 0, 0, 3)]=(0.52199660108+0.972964689688j)*x[0] + ((0.260153995091+0.331257136615j))*x[1] + ((-0.489680963277+0.265841809659j))*x[2]
            ref[(2, 0, 0, 3)]=(0.146234816447+0.785031817981j)
            arg[(2, 0, 1, 0)]=(-0.631909067056-0.670270782301j)*x[0] + ((-0.246326175534+0.519722448405j))*x[1] + ((0.0885183194882+0.680916860925j))*x[2]
            ref[(2, 0, 1, 0)]=(-0.394858461551+0.265184263514j)
            arg[(2, 0, 1, 1)]=(-0.332038572394+0.255573711058j)*x[0] + ((-0.399675019787+0.0581394917015j))*x[1] + ((-0.560954816238-0.30367965319j))*x[2]
            ref[(2, 0, 1, 1)]=(-0.64633420421+0.00501677478488j)
            arg[(2, 0, 1, 2)]=(-0.974789528691-0.987004145228j)*x[0] + ((-0.397550752472+0.118467268139j))*x[1] + ((0.412411235786-0.391975167926j))*x[2]
            ref[(2, 0, 1, 2)]=(-0.479964522689-0.630256022508j)
            arg[(2, 0, 1, 3)]=(-0.143599625819-0.840535887275j)*x[0] + ((-0.326757590053-0.61084716411j))*x[1] + ((8.66404122821e-05+0.771096720974j))*x[2]
            ref[(2, 0, 1, 3)]=(-0.23513528773-0.340143165206j)
            arg[(2, 0, 2, 0)]=(0.974766848614-0.538926146073j)*x[0] + ((-0.623861858012-0.139996407323j))*x[1] + ((-0.0267948770644-0.667412000707j))*x[2]
            ref[(2, 0, 2, 0)]=(0.162055056768-0.673167277051j)
            arg[(2, 0, 2, 1)]=(-0.253843923606+0.69982513281j)*x[0] + ((0.931197229103-0.23262162645j))*x[1] + ((-0.83868355341-0.36592152814j))*x[2]
            ref[(2, 0, 2, 1)]=(-0.0806651239565+0.0506409891104j)
            arg[(2, 0, 2, 2)]=(0.834485325756-0.381237049455j)*x[0] + ((-0.916142362621+0.52746466293j))*x[1] + ((-0.622759547826-0.23879750617j))*x[2]
            ref[(2, 0, 2, 2)]=(-0.352208292345-0.0462849463476j)
            arg[(2, 0, 2, 3)]=(-0.603636166863-0.0518729569234j)*x[0] + ((0.815966639727+0.709466445436j))*x[1] + ((-0.383445946022+0.637832298324j))*x[2]
            ref[(2, 0, 2, 3)]=(-0.085557736579+0.647712893418j)
            arg[(2, 0, 3, 0)]=(-0.931231577516+0.183514658449j)*x[0] + ((-0.0982560303549+0.589224224578j))*x[1] + ((-0.125933830936+0.867148795327j))*x[2]
            ref[(2, 0, 3, 0)]=(-0.577710719404+0.819943839177j)
            arg[(2, 0, 3, 1)]=(0.686763842948+0.572956794711j)*x[0] + ((-0.862038703415-0.925679476696j))*x[1] + ((-0.0837276587367-0.751776505718j))*x[2]
            ref[(2, 0, 3, 1)]=(-0.129501259602-0.552249593852j)
            arg[(2, 0, 3, 2)]=(-0.0968450328614-0.714369290444j)*x[0] + ((0.387315921124-0.983081685059j))*x[1] + ((0.388358041137-0.198755245199j))*x[2]
            ref[(2, 0, 3, 2)]=(0.3394144647-0.948103110351j)
            arg[(2, 0, 3, 3)]=(0.814124810874+0.756216872571j)*x[0] + ((-0.0901606480921+0.193719919571j))*x[1] + ((-0.100216317344-0.118135260731j))*x[2]
            ref[(2, 0, 3, 3)]=(0.311873922719+0.415900765706j)
            arg[(2, 1, 0, 0)]=(0.997273706803+0.916986179408j)*x[0] + ((0.799372674507+0.676867354548j))*x[1] + ((0.201570616548-0.554028873439j))*x[2]
            ref[(2, 1, 0, 0)]=(0.999108498929+0.519912330259j)
            arg[(2, 1, 0, 1)]=(-0.552572642923+0.686276472403j)*x[0] + ((-0.27966777748-0.90635082783j))*x[1] + ((0.29199851125-0.54715118167j))*x[2]
            ref[(2, 1, 0, 1)]=(-0.270120954576-0.383612768549j)
            arg[(2, 1, 0, 2)]=(-0.0277783855061-0.90769305248j)*x[0] + ((0.639644057821+0.220550139803j))*x[1] + ((-0.26586168045-0.233312359378j))*x[2]
            ref[(2, 1, 0, 2)]=(0.173001995933-0.460227636028j)
            arg[(2, 1, 0, 3)]=(0.966290445323-0.450357144098j)*x[0] + ((-0.7988071624+0.292782392648j))*x[1] + ((0.244233883246+0.0329889281287j))*x[2]
            ref[(2, 1, 0, 3)]=(0.205858583085-0.062292911661j)
            arg[(2, 1, 1, 0)]=(0.753341298745+0.615964904201j)*x[0] + ((0.312839753489-0.182341419846j))*x[1] + ((0.304056695773+0.907381023451j))*x[2]
            ref[(2, 1, 1, 0)]=(0.685118874004+0.670502253903j)
            arg[(2, 1, 1, 1)]=(0.0980329857105+0.261632166465j)*x[0] + ((-0.59007909442-0.230973313255j))*x[1] + ((-0.964480599054-0.379114705882j))*x[2]
            ref[(2, 1, 1, 1)]=(-0.728263353882-0.174227926336j)
            arg[(2, 1, 1, 2)]=(0.299264278391-0.985997642809j)*x[0] + ((0.320257670902-0.666993267434j))*x[1] + ((-0.996382154725-0.649789898902j))*x[2]
            ref[(2, 1, 1, 2)]=(-0.188430102716-1.15139040457j)
            arg[(2, 1, 1, 3)]=(0.569146869332+0.490033515126j)*x[0] + ((-0.63446223161-0.786903816441j))*x[1] + ((0.082542922585+0.264994948803j))*x[2]
            ref[(2, 1, 1, 3)]=(0.00861378015331-0.0159376762561j)
            arg[(2, 1, 2, 0)]=(-0.845222164203-0.0375680615719j)*x[0] + ((-0.433980978473-0.223195154311j))*x[1] + ((-0.536898148662-0.613514160423j))*x[2]
            ref[(2, 1, 2, 0)]=(-0.908050645669-0.437138688153j)
            arg[(2, 1, 2, 1)]=(-0.0249517640441-0.972292251834j)*x[0] + ((0.0196073267536-0.174290036777j))*x[1] + ((-0.786345514645-0.522665201725j))*x[2]
            ref[(2, 1, 2, 1)]=(-0.395844975968-0.834623745168j)
            arg[(2, 1, 2, 2)]=(-0.683108525621+0.558870969638j)*x[0] + ((0.425591980196-0.771892563392j))*x[1] + ((0.0481044783545+0.059106161743j))*x[2]
            ref[(2, 1, 2, 2)]=(-0.104706033535-0.0769577160057j)
            arg[(2, 1, 2, 3)]=(-0.785316468973-0.349502561883j)*x[0] + ((0.662387562232-0.358346885065j))*x[1] + ((0.00407260460025-0.971250941745j))*x[2]
            ref[(2, 1, 2, 3)]=(-0.0594281510704-0.839550194347j)
            arg[(2, 1, 3, 0)]=(-0.0804359062479-0.327773856699j)*x[0] + ((-0.395676372772+0.0632089721565j))*x[1] + ((0.397052617628+0.161134500281j))*x[2]
            ref[(2, 1, 3, 0)]=(-0.0395298306958-0.0517151921307j)
            arg[(2, 1, 3, 1)]=(-0.831526039448-0.248446814005j)*x[0] + ((-0.544480828265+0.656461587357j))*x[1] + ((0.286594385575-0.219312784187j))*x[2]
            ref[(2, 1, 3, 1)]=(-0.544706241069+0.0943509945825j)
            arg[(2, 1, 3, 2)]=(-0.100098306861-0.879224664451j)*x[0] + ((-0.988859121153-0.422430479337j))*x[1] + ((0.862973930557-0.570669153477j))*x[2]
            ref[(2, 1, 3, 2)]=(-0.112991748728-0.936162148633j)
            arg[(2, 1, 3, 3)]=(-0.754804038246+0.73632371637j)*x[0] + ((0.143539041107+0.639006154811j))*x[1] + ((-0.306278976245-0.701568035258j))*x[2]
            ref[(2, 1, 3, 3)]=(-0.458771986692+0.336880917961j)
            arg[(2, 2, 0, 0)]=(-0.379131156073+0.394217342587j)*x[0] + ((-0.661959166618-0.528374632173j))*x[1] + ((-0.356455000019-0.846437915633j))*x[2]
            ref[(2, 2, 0, 0)]=(-0.698772661355-0.490297602609j)
            arg[(2, 2, 0, 1)]=(0.428945020593+0.773394590846j)*x[0] + ((0.983753998985-0.433904541308j))*x[1] + ((-0.305264727696+0.113290517033j))*x[2]
            ref[(2, 2, 0, 1)]=(0.553717145941+0.226390283286j)
            arg[(2, 2, 0, 2)]=(0.568929858488-0.22791011807j)*x[0] + ((0.679441493216-0.834883075529j))*x[1] + ((0.503326067147+0.406616707257j))*x[2]
            ref[(2, 2, 0, 2)]=(0.875848709425-0.328088243171j)
            arg[(2, 2, 0, 3)]=(-0.2770124605+0.0800394555048j)*x[0] + ((-0.879851995536+0.243941705199j))*x[1] + ((0.874853753904+0.428284130763j))*x[2]
            ref[(2, 2, 0, 3)]=(-0.141005351066+0.376132645733j)
            arg[(2, 2, 1, 0)]=(0.618357930796-0.931549135706j)*x[0] + ((0.915587256352+0.739529309791j))*x[1] + ((-0.228148455513+0.237883854882j))*x[2]
            ref[(2, 2, 1, 0)]=(0.652898365817+0.0229320144837j)
            arg[(2, 2, 1, 1)]=(0.767862033602-0.0120324199481j)*x[0] + ((-0.758038481179+0.531145659285j))*x[1] + ((0.613583744156+0.0404378498796j))*x[2]
            ref[(2, 2, 1, 1)]=(0.311703648289+0.279775544608j)
            arg[(2, 2, 1, 2)]=(-0.455081311798-0.991347049306j)*x[0] + ((-0.255433745599+0.413671821102j))*x[1] + ((0.145671246174+0.21074731052j))*x[2]
            ref[(2, 2, 1, 2)]=(-0.282421905612-0.183463958842j)
            arg[(2, 2, 1, 3)]=(-0.143742938651-0.027012421726j)*x[0] + ((0.164772839889-0.586212490859j))*x[1] + ((0.124477204932-0.802301879248j))*x[2]
            ref[(2, 2, 1, 3)]=(0.072753553085-0.707763395917j)
            arg[(2, 2, 2, 0)]=(-0.350068568673+0.957491558929j)*x[0] + ((-0.442984275275+0.22673763517j))*x[1] + ((0.844250467056+0.158415013329j))*x[2]
            ref[(2, 2, 2, 0)]=(0.025598811554+0.671322103713j)
            arg[(2, 2, 2, 1)]=(-0.542172466268-0.316908142093j)*x[0] + ((0.258473582522+0.155078204377j))*x[1] + ((-0.0260144696198+0.577011037364j))*x[2]
            ref[(2, 2, 2, 1)]=(-0.154856676683+0.207590549824j)
            arg[(2, 2, 2, 2)]=(-0.211105396595-0.225673845832j)*x[0] + ((0.0830750666312-0.608871927148j))*x[1] + ((0.220487857078+0.903562137427j))*x[2]
            ref[(2, 2, 2, 2)]=(0.0462287635568+0.0345081822236j)
            arg[(2, 2, 2, 3)]=(-0.0465570023609+0.295113924316j)*x[0] + ((0.844110266802+0.880540670885j))*x[1] + ((-0.63843206119-0.623231658352j))*x[2]
            ref[(2, 2, 2, 3)]=(0.0795606016256+0.276211468424j)
            arg[(2, 2, 3, 0)]=(0.119184942297+0.260489456726j)*x[0] + ((-0.426400794517-0.0825456115104j))*x[1] + ((0.508452394741+0.0980848534242j))*x[2]
            ref[(2, 2, 3, 0)]=(0.100618271261+0.13801434932j)
            arg[(2, 2, 3, 1)]=(-0.0192782696365+0.157730155582j)*x[0] + ((-0.286690469782-0.335603086511j))*x[1] + ((0.594972200621-0.739244968625j))*x[2]
            ref[(2, 2, 3, 1)]=(0.144501730601-0.458558949777j)
            arg[(2, 2, 3, 2)]=(-0.779416043524+0.698889267376j)*x[0] + ((-0.545122365145-0.0227245472235j))*x[1] + ((-0.506820302654-0.560129376525j))*x[2]
            ref[(2, 2, 3, 2)]=(-0.915679355661+0.058017671814j)
            arg[(2, 2, 3, 3)]=(0.793100046776-0.594161356518j)*x[0] + ((0.716047873292-0.84428199872j))*x[1] + ((-0.93611966507+0.746987104787j))*x[2]
            ref[(2, 2, 3, 3)]=(0.286514127499-0.345728125225j)
            arg[(2, 3, 0, 0)]=(-0.0424880292545+0.587509831414j)*x[0] + ((0.567696493787+0.613088311364j))*x[1] + ((-0.458087958419+0.507850030452j))*x[2]
            ref[(2, 3, 0, 0)]=(0.0335602530568+0.854224086615j)
            arg[(2, 3, 0, 1)]=(0.780093650575+0.0983562904352j)*x[0] + ((0.600854732607+0.861156768752j))*x[1] + ((-0.560405604396+0.770293712694j))*x[2]
            ref[(2, 3, 0, 1)]=(0.410271389393+0.864903385941j)
            arg[(2, 3, 0, 2)]=(-0.251326101144+0.360606434829j)*x[0] + ((-0.688466394302+0.146579694169j))*x[1] + ((-0.835582711568-0.834917689222j))*x[2]
            ref[(2, 3, 0, 2)]=(-0.887687603507-0.163865780112j)
            arg[(2, 3, 0, 3)]=(-0.90217870417-0.649120598128j)*x[0] + ((-0.971461577968-0.918572417343j))*x[1] + ((0.590976815537-0.601665717389j))*x[2]
            ref[(2, 3, 0, 3)]=(-0.641331733301-1.08467936643j)
            arg[(2, 3, 1, 0)]=(0.213180098796-0.123848064884j)*x[0] + ((0.213995714288-0.49422653588j))*x[1] + ((-0.955133846943-0.459952954029j))*x[2]
            ref[(2, 3, 1, 0)]=(-0.26397901693-0.539013777396j)
            arg[(2, 3, 1, 1)]=(-0.730398049772+0.261427322946j)*x[0] + ((-0.0491792166473+0.590298520819j))*x[1] + ((-0.57230300132-0.937220099026j))*x[2]
            ref[(2, 3, 1, 1)]=(-0.675940133869-0.0427471276302j)
            arg[(2, 3, 1, 2)]=(-0.258208075172+0.897000163174j)*x[0] + ((-0.266431454854-0.447818400097j))*x[1] + ((0.836072828822-0.0405852708711j))*x[2]
            ref[(2, 3, 1, 2)]=(0.155716649398+0.204298246103j)
            arg[(2, 3, 1, 3)]=(0.297211542535-0.335892050502j)*x[0] + ((0.215365491297-0.736311033045j))*x[1] + ((-0.681234470609+0.0708355299125j))*x[2]
            ref[(2, 3, 1, 3)]=(-0.0843287183882-0.500683776817j)
            arg[(2, 3, 2, 0)]=(0.488683519888+0.325052684168j)*x[0] + ((-0.573358537849+0.622599857786j))*x[1] + ((-0.991762618861-0.177130190198j))*x[2]
            ref[(2, 3, 2, 0)]=(-0.538218818411+0.385261175878j)
            arg[(2, 3, 2, 1)]=(0.183077403021-0.455311617615j)*x[0] + ((0.849117894005-0.178194277482j))*x[1] + ((0.928209283049+0.993492385643j))*x[2]
            ref[(2, 3, 2, 1)]=(0.980202290038+0.179993245273j)
            arg[(2, 3, 2, 2)]=(-0.601885458364-0.397607276668j)*x[0] + ((-0.589283790757-0.962685940667j))*x[1] + ((-0.156119598521-0.528206748469j))*x[2]
            ref[(2, 3, 2, 2)]=(-0.673644423821-0.944249982902j)
            arg[(2, 3, 2, 3)]=(-0.257684493605-0.42706467637j)*x[0] + ((0.866832357635+0.487955333262j))*x[1] + ((0.3184880098-0.802082469923j))*x[2]
            ref[(2, 3, 2, 3)]=(0.463817936915-0.370595906515j)
            arg[(2, 3, 3, 0)]=(0.103893781514-0.778095923224j)*x[0] + ((0.847485623406+0.422737352091j))*x[1] + ((0.687551232661+0.0413254731508j))*x[2]
            ref[(2, 3, 3, 0)]=(0.81946531879-0.157016548991j)
            arg[(2, 3, 3, 1)]=(0.26229874564-0.35350793535j)*x[0] + ((-0.208720189287-0.0985136201411j))*x[1] + ((0.490745532242+0.409724416411j))*x[2]
            ref[(2, 3, 3, 1)]=(0.272162044297-0.0211485695399j)
            arg[(2, 3, 3, 2)]=(0.00607225177825-0.231360870178j)*x[0] + ((-0.686308714813-0.138912559844j))*x[1] + ((0.912433067257-0.506544770204j))*x[2]
            ref[(2, 3, 3, 2)]=(0.116098302111-0.438409100113j)
            arg[(2, 3, 3, 3)]=(0.133939156583+0.0781258498431j)*x[0] + ((-0.295271464383+0.590944818968j))*x[1] + ((0.691430582845-0.439681312413j))*x[2]
            ref[(2, 3, 3, 3)]=(0.265049137522+0.114694678199j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Function_rank0(self):
        """
        tests integral of rank 0 Data on Function

        assumptions: Function(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Function(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.157214840144+0.158923440908j)*x[0]**o + ((-0.474429075582+0.614954658463j))*x[0] + ((0.693015308128-0.841363910479j))*x[1]**o + ((-0.851104180375+0.905487764921j))*x[1]
            ref=(-0.237651553843+0.419000976906j)-((0.141705024712-0.113740078262j))*(o-1.)
        else:
            arg=(0.47762182879-0.272504355554j)*x[0]**o + ((-0.937632277566+0.623590175305j))*x[0] + ((0.817111796065-0.249114209887j))*x[1]**o + ((0.839279279714-0.123687251064j))*x[1] + ((0.32397177414+0.619117091131j))*x[2]**o + ((-0.556828004823-0.131216996514j))*x[2]
            ref=(0.481762198161+0.233092226708j)-((0.269784233166+0.0162497542817j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Function_rank1(self):
        """
        tests integral of rank 1 Data on Function

        assumptions: Function(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Function(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.249589467478-0.0154175342228j)*x[0]**o + ((0.764423701454-0.885759918021j))*x[0] + ((0.944790795666-0.608726571563j))*x[1]**o + ((0.531604700602-0.863041310435j))*x[1]
            ref[(0,)]=(0.995614865122-1.18647266712j)-((0.115866888031-0.104024017631j))*(o-1.)
            arg[(1,)]=(0.930004580414+0.627732871704j)*x[0]**o + ((-0.323102643208+0.382332197074j))*x[0] + ((-0.0820319779796-0.0016250373115j))*x[1]**o + ((0.0324538298765+0.177610679113j))*x[1]
            ref[(1,)]=(0.278661894551+0.59302535529j)-((0.141328767072+0.104351305732j))*(o-1.)
            arg[(2,)]=(-0.342539676348+0.22364813516j)*x[0]**o + ((0.554691976581-0.258113703546j))*x[0] + ((-0.480229089911-0.898037423073j))*x[1]**o + ((-0.229038981045+0.679543842052j))*x[1]
            ref[(2,)]=(-0.248557885362-0.126479574704j)-((-0.13712812771-0.112398214652j))*(o-1.)
            arg[(3,)]=(-0.798927235915+0.524760071163j)*x[0]**o + ((0.215353938911+0.773371456478j))*x[0] + ((0.366376007934+0.461547434644j))*x[1]**o + ((-0.567366560103-0.0806692614793j))*x[1]
            ref[(3,)]=(-0.392281924586+0.839504850403j)-((-0.0720918713301+0.164384584301j))*(o-1.)
        else:
            arg[(0,)]=(-0.96877554581+0.804978186654j)*x[0]**o + ((-0.136452668102+0.431123357817j))*x[0] + ((0.689851265209-0.591375674729j))*x[1]**o + ((0.165006761943+0.185369338931j))*x[1] + ((-0.96654842846+0.695952812778j))*x[2]**o + ((0.53346476302+0.357053175582j))*x[2]
            ref[(0,)]=(-0.341726926101+0.941550598517j)-((-0.207578784844+0.151592554117j))*(o-1.)
            arg[(1,)]=(-0.350962272552-0.506067516916j)*x[0]**o + ((0.123331887715+0.0044148153308j))*x[0] + ((-0.106485493102+0.871675722653j))*x[1]**o + ((-0.549863513496+0.208193052383j))*x[1] + ((0.983723690242-0.0356895140884j))*x[2]**o + ((0.396839665752-0.7710610087j))*x[2]
            ref[(1,)]=(0.24829198228-0.114267224669j)-((0.087712654098+0.054986448608j))*(o-1.)
            arg[(2,)]=(0.792440232867+0.644254412396j)*x[0]**o + ((-0.712078467059-0.926843883899j))*x[0] + ((0.971520992441+0.858411574928j))*x[1]**o + ((0.247856056448-0.481139954858j))*x[1] + ((0.488446076775-0.339542981813j))*x[2]**o + ((-0.679461344573-0.841620479194j))*x[2]
            ref[(2,)]=(0.554361773449-0.54324065622j)-((0.375401217014+0.193853834252j))*(o-1.)
            arg[(3,)]=(-0.68171920556-0.673417546248j)*x[0]**o + ((0.931452801761-0.183802212223j))*x[0] + ((0.739397195151-0.354296991722j))*x[1]**o + ((0.486473055113+0.205409072239j))*x[1] + ((0.33228622327+0.847275686785j))*x[2]**o + ((0.816881048869-0.416257356995j))*x[2]
            ref[(3,)]=(1.3123855593-0.287544674082j)-((0.0649940354768-0.0300731418641j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Function_rank2(self):
        """
        tests integral of rank 2 Data on Function

        assumptions: Function(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Function(self.domain)
        x = w.getX()
        arg = Data(0,(4, 4),w)
        ref=numpy.zeros((4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(-0.289418106043-0.309855372446j)*x[0]**o + ((-0.292894762382+0.915718863586j))*x[0] + ((0.450734921055+0.93190586253j))*x[1]**o + ((0.382021508279-0.520742933724j))*x[1]
            ref[(0, 0)]=(0.125221780455+0.508513209973j)-((0.0268861358354+0.103675081681j))*(o-1.)
            arg[(0, 1)]=(0.852504902125+0.414704591089j)*x[0]**o + ((-0.219791342102-0.140279782054j))*x[0] + ((-0.958118094799-0.917600296204j))*x[1]**o + ((-0.757897399554-0.770456781004j))*x[1]
            ref[(0, 1)]=(-0.541650967165-0.706816134087j)-((-0.0176021987789-0.0838159508525j))*(o-1.)
            arg[(0, 2)]=(-0.599754843839-0.554929721512j)*x[0]**o + ((0.759536449091-0.542807431155j))*x[0] + ((-0.913745419547+0.815882910059j))*x[1]**o + ((-0.855731681721+0.360717794169j))*x[1]
            ref[(0, 2)]=(-0.804847748008+0.0394317757805j)-((-0.252250043898+0.0434921980912j))*(o-1.)
            arg[(0, 3)]=(0.500854897269-0.142427388471j)*x[0]**o + ((-0.826872516367-0.540629117409j))*x[0] + ((0.989258009404-0.441544690574j))*x[1]**o + ((-0.392417373132+0.826523445887j))*x[1]
            ref[(0, 3)]=(0.135411508587-0.149038875284j)-((0.248352151112-0.0973286798409j))*(o-1.)
            arg[(1, 0)]=(-0.30879008529+0.29820162285j)*x[0]**o + ((0.95400508135+0.301051605033j))*x[0] + ((-0.398581883866+0.918784004221j))*x[1]**o + ((0.753080494537-0.579681129981j))*x[1]
            ref[(1, 0)]=(0.499856803365+0.469178051062j)-((-0.117895328193+0.202830937845j))*(o-1.)
            arg[(1, 1)]=(-0.214390962378+0.762010702225j)*x[0]**o + ((0.235710787186-0.816716072719j))*x[0] + ((-0.600003860788-0.987360957135j))*x[1]**o + ((-0.304859342569-0.660918270409j))*x[1]
            ref[(1, 1)]=(-0.441771689274-0.851492299019j)-((-0.135732470528-0.0375583758182j))*(o-1.)
            arg[(1, 2)]=(-0.876906574324+0.529635088573j)*x[0]**o + ((-0.779744146995+0.606370316363j))*x[0] + ((-0.336571803556-0.380886699754j))*x[1]**o + ((-0.0682166546336+0.539238205028j))*x[1]
            ref[(1, 2)]=(-1.03071958975+0.647178455105j)-((-0.202246396313+0.0247913981366j))*(o-1.)
            arg[(1, 3)]=(-0.822309112237+0.649388131891j)*x[0]**o + ((0.415360418897+0.953094863872j))*x[0] + ((-0.87680683442-0.17720359661j))*x[1]**o + ((-0.722213516219+0.218356497558j))*x[1]
            ref[(1, 3)]=(-1.00298452199+0.821817948355j)-((-0.28318599111+0.0786974225468j))*(o-1.)
            arg[(2, 0)]=(0.916408586949+0.993110906274j)*x[0]**o + ((-0.808168574385-0.717103768497j))*x[0] + ((-0.511615973852-0.523589628322j))*x[1]**o + ((0.794989065226-0.799169834052j))*x[1]
            ref[(2, 0)]=(0.195806551969-0.523376162299j)-((0.0674654355161+0.0782535463253j))*(o-1.)
            arg[(2, 1)]=(-0.596798093279+0.300782382716j)*x[0]**o + ((-0.635285951012-0.71793847274j))*x[0] + ((-0.244226087403+0.271358166264j))*x[1]**o + ((0.0455364232004+0.762393774842j))*x[1]
            ref[(2, 1)]=(-0.715386854246+0.308297925541j)-((-0.14017069678+0.0953567581634j))*(o-1.)
            arg[(2, 2)]=(-0.905471625142+0.785069600675j)*x[0]**o + ((-0.481153470388+0.732388689975j))*x[0] + ((0.696433457734-0.58984140668j))*x[1]**o + ((0.0605334498495+0.934560565916j))*x[1]
            ref[(2, 2)]=(-0.314829093973+0.931088724943j)-((-0.034839694568+0.0325380323325j))*(o-1.)
            arg[(2, 3)]=(-0.891916872321+0.89095573289j)*x[0]**o + ((-0.383262846888+0.372866116153j))*x[0] + ((0.0702524273977-0.202652075149j))*x[1]**o + ((-0.1775785281+0.565782195752j))*x[1]
            ref[(2, 3)]=(-0.691252909956+0.813475984823j)-((-0.136944074154+0.11471727629j))*(o-1.)
            arg[(3, 0)]=(0.220726269585+0.640305882865j)*x[0]**o + ((0.440642385491-0.351903565914j))*x[0] + ((-0.897630995446-0.697543308241j))*x[1]**o + ((-0.170301346897-0.470166369148j))*x[1]
            ref[(3, 0)]=(-0.203281843634-0.439653680219j)-((-0.11281745431-0.00953957089612j))*(o-1.)
            arg[(3, 1)]=(-0.256101435328+0.566950431405j)*x[0]**o + ((0.993462679271-0.779105851013j))*x[0] + ((0.457514176105+0.39187885773j))*x[1]**o + ((0.322976841383-0.0132482863067j))*x[1]
            ref[(3, 1)]=(0.758926130716+0.0832375759075j)-((0.0335687901295+0.159804881522j))*(o-1.)
            arg[(3, 2)]=(0.472819804113+0.521474144165j)*x[0]**o + ((0.13388983119-0.66803250039j))*x[0] + ((-0.326924609838+0.592925037726j))*x[1]**o + ((0.494847418139+0.581800265759j))*x[1]
            ref[(3, 2)]=(0.387316221802+0.51408347363j)-((0.0243158657125+0.185733196982j))*(o-1.)
            arg[(3, 3)]=(-0.70972063899+0.392849264892j)*x[0]**o + ((0.324571758044-0.369051112022j))*x[0] + ((-0.514333630829-0.46877277721j))*x[1]**o + ((-0.90502236513-0.939433106476j))*x[1]
            ref[(3, 3)]=(-0.902252438452-0.692203865408j)-((-0.20400904497-0.0126539187198j))*(o-1.)
        else:
            arg[(0, 0)]=(-0.181103877354+0.181890553639j)*x[0]**o + ((0.962240110945+0.181140002851j))*x[0] + ((-0.467240779583-0.223865969961j))*x[1]**o + ((0.749825014004-0.19937356558j))*x[1] + ((0.927814829229+0.901757221122j))*x[2]**o + ((0.619743074226-0.0953161335962j))*x[2]
            ref[(0, 0)]=(1.30563918573+0.373116054237j)-((0.0465783620487+0.143296967467j))*(o-1.)
            arg[(0, 1)]=(0.102823000844-0.485503719758j)*x[0]**o + ((0.206313612573+0.331917773693j))*x[0] + ((-0.0249489744819-0.851188367737j))*x[1]**o + ((-0.570003323972+0.769927584619j))*x[1] + ((-0.335303265308-0.192323251318j))*x[2]**o + ((0.765088624769-0.720324487749j))*x[2]
            ref[(0, 1)]=(0.0719848372119-0.573747234125j)-((-0.0429048731577-0.254835889802j))*(o-1.)
            arg[(0, 2)]=(0.207100106752-0.563457209454j)*x[0]**o + ((-0.358970453261-0.047943823731j))*x[0] + ((0.719790189837+0.78115417991j))*x[1]**o + ((-0.329759725095-0.350514010313j))*x[1] + ((-0.564453959565-0.00757373251682j))*x[2]**o + ((-0.230183034525-0.61552778862j))*x[2]
            ref[(0, 2)]=(-0.278238437928-0.401931192362j)-((0.0604060561706+0.0350205396565j))*(o-1.)
            arg[(0, 3)]=(0.969359696683+0.924303824148j)*x[0]**o + ((0.746497613889-0.40996732583j))*x[0] + ((-0.834105512688+0.00176099399862j))*x[1]**o + ((-0.644586478568-0.723821130281j))*x[1] + ((0.634313933516+0.421402754357j))*x[2]**o + ((-0.344974910317-0.961610739606j))*x[2]
            ref[(0, 3)]=(0.263252171258-0.373965811606j)-((0.128261352919+0.224577928751j))*(o-1.)
            arg[(1, 0)]=(0.633937617438-0.907905628613j)*x[0]**o + ((0.311720294192-0.947589616989j))*x[0] + ((0.709230179382+0.362888187182j))*x[1]**o + ((-0.90491737483-0.945523698246j))*x[1] + ((0.4779326531-0.5032779404j))*x[2]**o + ((-0.627355350346-0.78388489558j))*x[2]
            ref[(1, 0)]=(0.300274009468-1.86264679632j)-((0.303516741653-0.174715896972j))*(o-1.)
            arg[(1, 1)]=(0.745171365547+0.436146881688j)*x[0]**o + ((-0.353739787396+0.132876401186j))*x[0] + ((-0.074215040279-0.152891935693j))*x[1]**o + ((-0.304540885513+0.68007216356j))*x[1] + ((-0.283279436493+0.319238198485j))*x[2]**o + ((0.334202906881+0.307212341507j))*x[2]
            ref[(1, 1)]=(0.0317995613731+0.861327025366j)-((0.0646128147958+0.10041552408j))*(o-1.)
            arg[(1, 2)]=(-0.593925891178-0.295320232273j)*x[0]**o + ((-0.308040705343+0.908916177263j))*x[0] + ((0.463659865525+0.939133151202j))*x[1]**o + ((-0.409197677399-0.775140335245j))*x[1] + ((-0.331897529092+0.107492809653j))*x[2]**o + ((0.595210717281-0.156426384765j))*x[2]
            ref[(1, 2)]=(-0.292095610103+0.364327592917j)-((-0.0770272591242+0.12521762143j))*(o-1.)
            arg[(1, 3)]=(0.807772978922-0.968789089568j)*x[0]**o + ((-0.81049035318-0.330298027794j))*x[0] + ((-0.774431225204+0.32154490973j))*x[1]**o + ((0.8380451145+0.398816665502j))*x[1] + ((0.466640799693-0.420051018012j))*x[2]**o + ((-0.143749846321-0.162244892354j))*x[2]
            ref[(1, 3)]=(0.191893734204-0.580510726248j)-((0.0833304255684-0.177882532975j))*(o-1.)
            arg[(2, 0)]=(-0.0897263602041+0.341036216184j)*x[0]**o + ((0.567955967541-0.859917505846j))*x[0] + ((-0.367718422509-0.659899212229j))*x[1]**o + ((-0.348406934719-0.589802817005j))*x[1] + ((0.200951882397+0.273163090024j))*x[2]**o + ((0.185209343423+0.32480152756j))*x[2]
            ref[(2, 0)]=(0.0741327379644-0.585309350656j)-((-0.0427488167194-0.00761665100356j))*(o-1.)
            arg[(2, 1)]=(0.722835442286-0.415542366776j)*x[0]**o + ((0.389532818275-0.36350037902j))*x[0] + ((-0.202933744104-0.266758009229j))*x[1]**o + ((-0.243239079604-0.741000760061j))*x[1] + ((-0.121543377147+0.435302933879j))*x[2]**o + ((0.839133766611-0.432502354571j))*x[2]
            ref[(2, 1)]=(0.691892913159-0.892000467889j)-((0.0663930535059-0.0411662403543j))*(o-1.)
            arg[(2, 2)]=(0.832851625409-0.880828371618j)*x[0]**o + ((0.826044739905+0.578984938617j))*x[0] + ((-0.338598470688-0.619709384546j))*x[1]**o + ((0.35358787778-0.0667952256593j))*x[1] + ((-0.142669748953+0.675940751402j))*x[2]**o + ((0.372905856856-0.649270591086j))*x[2]
            ref[(2, 2)]=(0.952060940155-0.480838941445j)-((0.0585972342947-0.137432834127j))*(o-1.)
            arg[(2, 3)]=(-0.230359027065-0.221977883535j)*x[0]**o + ((-0.516853620373+0.700246003828j))*x[0] + ((0.840211342034-0.197534450604j))*x[1]**o + ((0.331002505839+0.799915537425j))*x[1] + ((0.0805451183736+0.583963660601j))*x[2]**o + ((-0.382415272407+0.867752785414j))*x[2]
            ref[(2, 3)]=(0.0610655232006+1.26618282656j)-((0.11506623889+0.0274085544105j))*(o-1.)
            arg[(3, 0)]=(0.0101678165992-0.866388553236j)*x[0]**o + ((-0.287927701017-0.0327503750812j))*x[0] + ((-0.136313576833-0.00842299444519j))*x[1]**o + ((0.28795804532+0.27541970319j))*x[1] + ((-0.0411665421228-0.306182610149j))*x[2]**o + ((0.502847450095+0.995241597112j))*x[2]
            ref[(3, 0)]=(0.167782746021+0.0284583836954j)-((-0.0278853837261-0.196832359638j))*(o-1.)
            arg[(3, 1)]=(-0.0795136531806-0.036126417474j)*x[0]**o + ((0.391837790504+0.577121024478j))*x[0] + ((-0.572784272422+0.363842612557j))*x[1]**o + ((-0.739969366537+0.493534069548j))*x[1] + ((-0.00652399150928-0.243298721236j))*x[2]**o + ((0.255522010271+0.119861600045j))*x[2]
            ref[(3, 1)]=(-0.375715741437+0.63746708396j)-((-0.109803652852+0.0140695789745j))*(o-1.)
            arg[(3, 2)]=(0.467362655505-0.256044914055j)*x[0]**o + ((-0.241080028157-0.515480637204j))*x[0] + ((-0.0650741477265-0.0680009560319j))*x[1]**o + ((-0.347289509681+0.313238063298j))*x[1] + ((-0.17858447468+0.689779060941j))*x[2]**o + ((-0.251902622617-0.652323547762j))*x[2]
            ref[(3, 2)]=(-0.308284063678-0.244416465406j)-((0.0372840055164+0.0609555318091j))*(o-1.)
            arg[(3, 3)]=(-0.046317578979+0.471890596324j)*x[0]**o + ((0.898776752933+0.883901082599j))*x[0] + ((0.699344463308+0.157645397077j))*x[1]**o + ((0.15958331895-0.78675446512j))*x[1] + ((-0.216144386818-0.167790182951j))*x[2]**o + ((0.384389210563-0.426842900549j))*x[2]
            ref[(3, 3)]=(0.939815889979+0.0660247636901j)-((0.0728137495852+0.076957635075j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Function_rank3(self):
        """
        tests integral of rank 3 Data on Function

        assumptions: Function(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Function(self.domain)
        x = w.getX()
        arg = Data(0,(3, 3, 4),w)
        ref=numpy.zeros((3, 3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.761889099371-0.727794358059j)*x[0]**o + ((0.705464809685-0.14803181579j))*x[0] + ((0.733102491321+0.475500037952j))*x[1]**o + ((-0.993731241254-0.750048910336j))*x[1]
            ref[(0, 0, 0)]=(0.603362579562-0.575187523117j)-((0.249165265115-0.0420490533512j))*(o-1.)
            arg[(0, 0, 1)]=(0.934496341774+0.251346472054j)*x[0]**o + ((-0.570792576711+0.585614727741j))*x[0] + ((0.822182639317+0.738084924374j))*x[1]**o + ((-0.406452027964+0.256013612472j))*x[1]
            ref[(0, 0, 1)]=(0.389717188208+0.915529868321j)-((0.292779830182+0.164905232738j))*(o-1.)
            arg[(0, 0, 2)]=(0.473938722031+0.587881920212j)*x[0]**o + ((6.53313630465e-05-0.87729898965j))*x[0] + ((-0.139689877038+0.106723244276j))*x[1]**o + ((0.0968521724786+0.792097827858j))*x[1]
            ref[(0, 0, 2)]=(0.215583174417+0.304702001348j)-((0.0557081408321+0.115767527415j))*(o-1.)
            arg[(0, 0, 3)]=(0.428100150683+0.342936580783j)*x[0]**o + ((-0.456784292428+0.951894990003j))*x[0] + ((0.362202074783-0.754580140081j))*x[1]**o + ((0.0130167665056+0.255487901113j))*x[1]
            ref[(0, 0, 3)]=(0.173267349772+0.397869665909j)-((0.131717037578-0.0686072598831j))*(o-1.)
            arg[(0, 1, 0)]=(-0.535707387892-0.61843062774j)*x[0]**o + ((0.398236324523+0.593912547688j))*x[0] + ((-0.711861391473+0.549457089042j))*x[1]**o + ((-0.448136129573-0.600453569447j))*x[1]
            ref[(0, 1, 0)]=(-0.648734292208-0.0377572802284j)-((-0.207928129894-0.0114955897829j))*(o-1.)
            arg[(0, 1, 1)]=(0.634707515845+0.950844451004j)*x[0]**o + ((-0.486629321089+0.766299581547j))*x[0] + ((-0.122793280311-0.535107520201j))*x[1]**o + ((-0.978820694439+0.437881411999j))*x[1]
            ref[(0, 1, 1)]=(-0.476767889997+0.809958962174j)-((0.0853190392557+0.0692894884671j))*(o-1.)
            arg[(0, 1, 2)]=(0.510534916993+0.335873490815j)*x[0]**o + ((-0.443830919716+0.602096718535j))*x[0] + ((-0.13688920978+0.757866583592j))*x[1]**o + ((0.271400793267+0.494957275769j))*x[1]
            ref[(0, 1, 2)]=(0.100607790382+1.09539703436j)-((0.0622742845355+0.182290012401j))*(o-1.)
            arg[(0, 1, 3)]=(0.943320817206+0.167032093594j)*x[0]**o + ((-0.246596946581+0.1938006777j))*x[0] + ((0.3402077386-0.645403870028j))*x[1]**o + ((0.0539582651305+0.610199808043j))*x[1]
            ref[(0, 1, 3)]=(0.545444937177+0.162814354654j)-((0.213921425968-0.0797286294057j))*(o-1.)
            arg[(0, 2, 0)]=(-0.388453360506+0.212765373888j)*x[0]**o + ((0.0863722444754+0.0099389718673j))*x[0] + ((0.0144626561082-0.924761917752j))*x[1]**o + ((-0.684373719743+0.310942940783j))*x[1]
            ref[(0, 2, 0)]=(-0.485996089833-0.195557315607j)-((-0.0623317840664-0.118666090644j))*(o-1.)
            arg[(0, 2, 1)]=(-0.0743904679421+0.854395077204j)*x[0]**o + ((-0.10798084239-0.725881587938j))*x[0] + ((0.849197560503+0.971796444971j))*x[1]**o + ((0.493973404489+0.17548452343j))*x[1]
            ref[(0, 2, 1)]=(0.58039982733+0.637897228834j)-((0.129134515427+0.304365253696j))*(o-1.)
            arg[(0, 2, 2)]=(0.845748953548-0.595195836993j)*x[0]**o + ((0.198968845979-0.606732871601j))*x[0] + ((-0.30471491431-0.57236859684j))*x[1]**o + ((0.443002147513+0.764806907284j))*x[1]
            ref[(0, 2, 2)]=(0.591502516365-0.504745199075j)-((0.090172339873-0.194594072306j))*(o-1.)
            arg[(0, 2, 3)]=(0.437295496864-0.755453241335j)*x[0]**o + ((0.514869672017-0.349202498234j))*x[0] + ((-0.726572672548-0.560529324281j))*x[1]**o + ((-0.939253692528+0.345555548043j))*x[1]
            ref[(0, 2, 3)]=(-0.356830598098-0.659814757903j)-((-0.048212862614-0.219330427603j))*(o-1.)
            arg[(1, 0, 0)]=(-0.675885974397+0.367970795786j)*x[0]**o + ((0.104946314266-0.826003407386j))*x[0] + ((0.285598894114-0.623177025138j))*x[1]**o + ((-0.877377026724-0.410450521591j))*x[1]
            ref[(1, 0, 0)]=(-0.58135889637-0.745830079165j)-((-0.0650478467137-0.0425343715587j))*(o-1.)
            arg[(1, 0, 1)]=(-0.568083268726-0.457270888009j)*x[0]**o + ((0.23594317888+0.864778199264j))*x[0] + ((-0.243650970997-0.577762703562j))*x[1]**o + ((0.137310943191-0.244755906696j))*x[1]
            ref[(1, 0, 1)]=(-0.219240058826-0.207505649501j)-((-0.135289039954-0.172505598595j))*(o-1.)
            arg[(1, 0, 2)]=(0.248340348334+0.861490704407j)*x[0]**o + ((-0.661652735561+0.732379276704j))*x[0] + ((-0.209738095819-0.805951992709j))*x[1]**o + ((0.331833619307-0.660109643676j))*x[1]
            ref[(1, 0, 2)]=(-0.145608431869+0.0639041723633j)-((0.00643370875243+0.00925645194972j))*(o-1.)
            arg[(1, 0, 3)]=(-0.585431519374-0.219629956577j)*x[0]**o + ((0.232068918251-0.512669776766j))*x[0] + ((-0.727664037945-0.726720407302j))*x[1]**o + ((0.677114544261-0.463907022477j))*x[1]
            ref[(1, 0, 3)]=(-0.201956047403-0.961463581562j)-((-0.218849259553-0.157725060647j))*(o-1.)
            arg[(1, 1, 0)]=(0.0881554658098-0.581017206572j)*x[0]**o + ((-0.966663962915-0.49166309756j))*x[0] + ((-0.73362052831-0.304908925498j))*x[1]**o + ((0.939925214988+0.108603627527j))*x[1]
            ref[(1, 1, 0)]=(-0.336101905213-0.634492801051j)-((-0.107577510417-0.147654355345j))*(o-1.)
            arg[(1, 1, 1)]=(0.701101746427-0.611589636841j)*x[0]**o + ((0.979676190821-0.961198896254j))*x[0] + ((0.156542573955+0.492680242584j))*x[1]**o + ((-0.570240170055-0.740882735794j))*x[1]
            ref[(1, 1, 1)]=(0.633540170574-0.910495513153j)-((0.142940720064-0.0198182323762j))*(o-1.)
            arg[(1, 1, 2)]=(-0.668588734171-0.466215876566j)*x[0]**o + ((-0.751837206784+0.227681059376j))*x[0] + ((0.0890897785969+0.656060074647j))*x[1]**o + ((-0.170757359778-0.18958789261j))*x[1]
            ref[(1, 1, 2)]=(-0.751046761068+0.113968682424j)-((-0.0965831592624+0.0316406996802j))*(o-1.)
            arg[(1, 1, 3)]=(0.877173631378-0.983281652627j)*x[0]**o + ((0.728990084138-0.195468636915j))*x[0] + ((-0.414417944825-0.373604837747j))*x[1]**o + ((0.217018761532-0.225479636426j))*x[1]
            ref[(1, 1, 3)]=(0.704382266112-0.888917381858j)-((0.0771259477587-0.226147748396j))*(o-1.)
            arg[(1, 2, 0)]=(0.970705204117+0.797407533794j)*x[0]**o + ((0.18810076589+0.0373283714252j))*x[0] + ((-0.639221291222-0.431944796395j))*x[1]**o + ((-0.321600103689-0.345737873659j))*x[1]
            ref[(1, 2, 0)]=(0.0989922875479+0.0285266175825j)-((0.0552473188159+0.0609104562332j))*(o-1.)
            arg[(1, 2, 1)]=(0.745025617868+0.692166810745j)*x[0]**o + ((-0.370374309168-0.661723027972j))*x[0] + ((0.838436759227+0.708896683151j))*x[1]**o + ((0.278749680808-0.117609512422j))*x[1]
            ref[(1, 2, 1)]=(0.745918874367+0.310865476751j)-((0.263910396182+0.233510582316j))*(o-1.)
            arg[(1, 2, 2)]=(0.294450534139+0.42011483584j)*x[0]**o + ((0.686762610656-0.895054631176j))*x[0] + ((-0.958870099266+0.583660320713j))*x[1]**o + ((-0.700888063822-0.708084189574j))*x[1]
            ref[(1, 2, 2)]=(-0.339272509147-0.299681832099j)-((-0.110736594188+0.167295859425j))*(o-1.)
            arg[(1, 2, 3)]=(0.456178351426+0.70159274066j)*x[0]**o + ((-0.20434932591-0.0385267408426j))*x[0] + ((0.286926245972-0.312101684264j))*x[1]**o + ((-0.441196538263-0.348364611785j))*x[1]
            ref[(1, 2, 3)]=(0.0487793666122+0.00129985188387j)-((0.123850766233+0.0649151760659j))*(o-1.)
            arg[(2, 0, 0)]=(0.182117594426-0.0606022983241j)*x[0]**o + ((-0.681751712618+0.685174847073j))*x[0] + ((0.219692507694+0.757486110135j))*x[1]**o + ((0.37964126053+0.656325537432j))*x[1]
            ref[(2, 0, 0)]=(0.0498498250161+1.01919209816j)-((0.0669683503534+0.116147301969j))*(o-1.)
            arg[(2, 0, 1)]=(0.911337497181+0.433926888585j)*x[0]**o + ((0.459815361908-0.960297185471j))*x[0] + ((-0.730593859502-0.0613752467087j))*x[1]**o + ((0.749789922731-0.992061224134j))*x[1]
            ref[(2, 0, 1)]=(0.695174461159-0.789903383865j)-((0.0301239396132+0.0620919403127j))*(o-1.)
            arg[(2, 0, 2)]=(0.411679773478-0.806012383148j)*x[0]**o + ((-0.39487771448-0.62206023172j))*x[0] + ((0.977590835774-0.0721818952802j))*x[1]**o + ((0.175182129472-0.728561815409j))*x[1]
            ref[(2, 0, 2)]=(0.584787512122-1.11440816278j)-((0.231545101542-0.146365713071j))*(o-1.)
            arg[(2, 0, 3)]=(0.476145043616+0.922279609335j)*x[0]**o + ((-0.580420447876+0.10049174189j))*x[0] + ((-0.456680169744+0.350002888532j))*x[1]**o + ((0.999818637859-0.208999945392j))*x[1]
            ref[(2, 0, 3)]=(0.219431531928+0.581887147183j)-((0.00324414564537+0.212047082978j))*(o-1.)
            arg[(2, 1, 0)]=(0.533331672446+0.380764692765j)*x[0]**o + ((-0.633530640412-0.969665230138j))*x[0] + ((0.273930057511-0.0400900144834j))*x[1]**o + ((-0.944132500624-0.178349490568j))*x[1]
            ref[(2, 1, 0)]=(-0.38520070554-0.403670021213j)-((0.134543621659+0.0567791130469j))*(o-1.)
            arg[(2, 1, 1)]=(0.82691751967-0.99977680426j)*x[0]**o + ((0.438146552031-0.406041469448j))*x[0] + ((0.371958303354+0.429510556642j))*x[1]**o + ((-0.685571414365-0.485206919318j))*x[1]
            ref[(2, 1, 1)]=(0.475725480345-0.730757318192j)-((0.199812637171-0.0950443746029j))*(o-1.)
            arg[(2, 1, 2)]=(0.43137265574-0.3602482247j)*x[0]**o + ((0.421995814791+0.435091482461j))*x[0] + ((0.00631902273433+0.194772388585j))*x[1]**o + ((0.185060840379-0.600605321919j))*x[1]
            ref[(2, 1, 2)]=(0.522374166822-0.165494837787j)-((0.0729486130791-0.0275793060192j))*(o-1.)
            arg[(2, 1, 3)]=(0.538465572951+0.633965404207j)*x[0]**o + ((0.670391562031-0.938804756716j))*x[0] + ((-0.859273475786+0.550588920323j))*x[1]**o + ((-0.75013363777+0.460352557017j))*x[1]
            ref[(2, 1, 3)]=(-0.200274989287+0.353051062416j)-((-0.0534679838058+0.197425720755j))*(o-1.)
            arg[(2, 2, 0)]=(0.655814467839+0.719598833082j)*x[0]**o + ((-0.271479988328-0.160130633129j))*x[0] + ((0.539567787632-0.289318036277j))*x[1]**o + ((-0.400000568489-0.398980901215j))*x[1]
            ref[(2, 2, 0)]=(0.261950849327-0.0644153687696j)-((0.199230375912+0.0717134661342j))*(o-1.)
            arg[(2, 2, 1)]=(-0.40291967521-0.486058139672j)*x[0]**o + ((-0.0889529122119-0.196116020041j))*x[0] + ((-0.403306267513-0.780953843711j))*x[1]**o + ((-0.990050330707+0.149202651721j))*x[1]
            ref[(2, 2, 1)]=(-0.942614592821-0.656962675851j)-((-0.134370990454-0.211168663897j))*(o-1.)
            arg[(2, 2, 2)]=(-0.105173634217+0.22577032157j)*x[0]**o + ((-0.395652903367-0.0415134481029j))*x[0] + ((-0.58538087477+0.852744274266j))*x[1]**o + ((0.881718875196-0.521455477935j))*x[1]
            ref[(2, 2, 2)]=(-0.102244268579+0.257772834899j)-((-0.115092418164+0.179752432639j))*(o-1.)
            arg[(2, 2, 3)]=(0.357182916294-0.587334814885j)*x[0]**o + ((-0.871224136758+0.0975438399963j))*x[0] + ((0.580096491304+0.368737706428j))*x[1]**o + ((0.60207296592-0.558072560997j))*x[1]
            ref[(2, 2, 3)]=(0.33406411838-0.339562914729j)-((0.1562132346-0.0364328514095j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(0.489239168192+0.473824987981j)*x[0]**o + ((-0.887845764962-0.345980408184j))*x[0] + ((-0.651094852464+0.445558183013j))*x[1]**o + ((-0.736646857729+0.317331920503j))*x[1] + ((-0.294941408962+0.00671703950304j))*x[2]**o + ((0.551662917672-0.37682175783j))*x[2]
            ref[(0, 0, 0)]=(-0.764813399127+0.260314982493j)-((-0.0761328488724+0.154350035083j))*(o-1.)
            arg[(0, 0, 1)]=(-0.500415556356+0.640928422821j)*x[0]**o + ((0.255598501889+0.386688760422j))*x[0] + ((0.983961111349-0.610616216029j))*x[1]**o + ((-0.127930937318-0.248781977684j))*x[1] + ((0.242007643874+0.27492654168j))*x[2]**o + ((0.290270183339-0.949485370741j))*x[2]
            ref[(0, 0, 1)]=(0.571745473389-0.253169919765j)-((0.120925533144+0.0508731247454j))*(o-1.)
            arg[(0, 0, 2)]=(0.994571685523-0.111010272994j)*x[0]**o + ((0.732107339272+0.29772742987j))*x[0] + ((-0.288550450982+0.360387469236j))*x[1]**o + ((-0.859784386261-0.0312762591841j))*x[1] + ((0.662864423887-0.0612663100568j))*x[2]**o + ((-0.627318320528-0.252735489453j))*x[2]
            ref[(0, 0, 2)]=(0.306945145456+0.100913283709j)-((0.228147609738+0.0313518143642j))*(o-1.)
            arg[(0, 0, 3)]=(0.816168523208+0.884850876187j)*x[0]**o + ((0.549984946576+0.6306267172j))*x[0] + ((-0.70499005931+0.0401808360449j))*x[1]**o + ((-0.869122332565-0.11350411474j))*x[1] + ((-0.557560883269-0.769474511039j))*x[2]**o + ((0.441488075341+0.847239276559j))*x[2]
            ref[(0, 0, 3)]=(-0.16201586501+0.759959540105j)-((-0.0743970698953+0.0259262001987j))*(o-1.)
            arg[(0, 1, 0)]=(0.280652847973+0.680841229835j)*x[0]**o + ((0.920807330628+0.714730758076j))*x[0] + ((-0.299255212824+0.95106470677j))*x[1]**o + ((-0.232115049625-0.320222160159j))*x[1] + ((-0.931316431047-0.531380073369j))*x[2]**o + ((-0.0534337735039-0.0179204296442j))*x[2]
            ref[(0, 1, 0)]=(-0.1573301442+0.738557015755j)-((-0.158319799316+0.183420977206j))*(o-1.)
            arg[(0, 1, 1)]=(0.157567877301-0.927054942847j)*x[0]**o + ((0.615254300372-0.159069591384j))*x[0] + ((-0.889363776472-0.820156618842j))*x[1]**o + ((-0.100162795123-0.308275954425j))*x[1] + ((-0.044165132095+0.713400303668j))*x[2]**o + ((0.877202262874+0.0730002973161j))*x[2]
            ref[(0, 1, 1)]=(0.308166368428-0.714078253257j)-((-0.129326838544-0.172301876337j))*(o-1.)
            arg[(0, 1, 2)]=(-0.0450782839792+0.104274894046j)*x[0]**o + ((0.0807062759123+0.265890634029j))*x[0] + ((-0.286481262617-0.473412287547j))*x[1]**o + ((-0.117435894941+0.474813084303j))*x[1] + ((0.973097809556-0.666564694387j))*x[2]**o + ((0.563823716096-0.927685433338j))*x[2]
            ref[(0, 1, 2)]=(0.584316180013-0.611341901447j)-((0.106923043827-0.172617014648j))*(o-1.)
            arg[(0, 1, 3)]=(0.0440267794628+0.976140246137j)*x[0]**o + ((-0.03529426236-0.451024728256j))*x[0] + ((-0.845484679775-0.693248006781j))*x[1]**o + ((-0.556434387236-0.101629074189j))*x[1] + ((0.104057868058+0.832927358064j))*x[2]**o + ((-0.225700214364+0.599205490447j))*x[2]
            ref[(0, 1, 3)]=(-0.757414448107+0.581185642711j)-((-0.116233338709+0.185969932903j))*(o-1.)
            arg[(0, 2, 0)]=(0.722632718639+0.592721608066j)*x[0]**o + ((-0.857101343268+0.546125359527j))*x[0] + ((0.902011409834-0.804635240895j))*x[1]**o + ((0.873838428007+0.85894923716j))*x[1] + ((0.413532760151+0.484114741432j))*x[2]**o + ((0.117049929879-0.988269002377j))*x[2]
            ref[(0, 2, 0)]=(1.08598195162+0.344503351457j)-((0.339696148104+0.0453668514339j))*(o-1.)
            arg[(0, 2, 1)]=(-0.569215832818-0.257807979125j)*x[0]**o + ((-0.0180644790958+0.633830476009j))*x[0] + ((-0.237843448971-0.483806632718j))*x[1]**o + ((0.951916700878+0.0350477214447j))*x[1] + ((-0.608375631201-0.67028573348j))*x[2]**o + ((0.932711814289-0.53247898232j))*x[2]
            ref[(0, 2, 1)]=(0.22556456154-0.637750565095j)-((-0.235905818832-0.235316724221j))*(o-1.)
            arg[(0, 2, 2)]=(0.767390428224-0.72183694087j)*x[0]**o + ((0.993973337618+0.343589214283j))*x[0] + ((0.409454414357+0.289600767228j))*x[1]**o + ((0.332867056249-0.575968014983j))*x[1] + ((-0.322485847256+0.0739598502547j))*x[2]**o + ((0.291645369168-0.0304074845459j))*x[2]
            ref[(0, 2, 2)]=(1.23642237918-0.310531304316j)-((0.142393165887-0.0597127205644j))*(o-1.)
            arg[(0, 2, 3)]=(0.815647754453+0.833459194316j)*x[0]**o + ((-0.936971047013-0.317200630065j))*x[0] + ((0.796015284273+0.707057243742j))*x[1]**o + ((-0.273673684407+0.973544798427j))*x[1] + ((-0.111474328971+0.0216268748122j))*x[2]**o + ((-0.930166937608-0.168759235237j))*x[2]
            ref[(0, 2, 3)]=(-0.320311479636+1.024864123j)-((0.250031451626+0.260357218812j))*(o-1.)
            arg[(1, 0, 0)]=(0.790723953861+0.956331692203j)*x[0]**o + ((0.0807428569983+0.0452520649354j))*x[0] + ((0.80934469108-0.724630777031j))*x[1]**o + ((0.970592357439-0.739748835931j))*x[1] + ((0.220294525927+0.468608757238j))*x[2]**o + ((-0.53603433038+0.339045651658j))*x[2]
            ref[(1, 0, 0)]=(1.16783202746+0.172429276536j)-((0.303393861811+0.116718278735j))*(o-1.)
            arg[(1, 0, 1)]=(-0.674767868097+0.198294021117j)*x[0]**o + ((-0.781958147382+0.053399836155j))*x[0] + ((0.760423370753-0.284110085793j))*x[1]**o + ((-0.676872809997-0.702618310535j))*x[1] + ((-0.355651283154-0.326123010432j))*x[2]**o + ((-0.550168375206-0.07485417849j))*x[2]
            ref[(1, 0, 1)]=(-1.13949755654-0.568005863989j)-((-0.0449992967498-0.068656512518j))*(o-1.)
            arg[(1, 0, 2)]=(-0.751386508593+0.729644532721j)*x[0]**o + ((0.940883934836+0.964460126164j))*x[0] + ((-0.540764758821-0.305430000852j))*x[1]**o + ((-0.205907111458-0.653283358679j))*x[1] + ((-0.273592209559+0.610678459511j))*x[2]**o + ((0.233201017176+0.914016213757j))*x[2]
            ref[(1, 0, 2)]=(-0.298782818209+1.13004298631j)-((-0.260957246162+0.17248216523j))*(o-1.)
            arg[(1, 0, 3)]=(0.746897942241-0.714510433968j)*x[0]**o + ((0.882998922788-0.0406533985426j))*x[0] + ((-0.977640848696+0.101547095641j))*x[1]**o + ((0.154206616729+0.728319437554j))*x[1] + ((-0.339720933901+0.879075592416j))*x[2]**o + ((0.716495082703+0.0479651898286j))*x[2]
            ref[(1, 0, 3)]=(0.591618390931+0.500871741465j)-((-0.0950773067262+0.0443520423483j))*(o-1.)
            arg[(1, 1, 0)]=(-0.522969208386+0.590893454292j)*x[0]**o + ((0.881766082485-0.158019264151j))*x[0] + ((0.134704228167+0.965789478632j))*x[1]**o + ((0.732264829491+0.477690648914j))*x[1] + ((-0.313086258898+0.228464227021j))*x[2]**o + ((0.11727790911-0.0657978524322j))*x[2]
            ref[(1, 1, 0)]=(0.514978790985+1.01951034614j)-((-0.116891873186+0.297524526658j))*(o-1.)
            arg[(1, 1, 1)]=(-0.240712827014+0.0616017958218j)*x[0]**o + ((-0.694205806346-0.202277662127j))*x[0] + ((-0.607698327401+0.752719625731j))*x[1]**o + ((-0.939140944228-0.206411852892j))*x[1] + ((0.342515025387+0.0102069282576j))*x[2]**o + ((0.339920295885+0.878924441592j))*x[2]
            ref[(1, 1, 1)]=(-0.899661291858+0.647381638192j)-((-0.0843160215047+0.137421391635j))*(o-1.)
            arg[(1, 1, 2)]=(-0.935909703103-0.2174081123j)*x[0]**o + ((0.49431616178-0.746786870513j))*x[0] + ((0.529045462361+0.828181567404j))*x[1]**o + ((0.386067438386-0.557735014947j))*x[1] + ((-0.660739076143-0.0856410842264j))*x[2]**o + ((-0.698108134922-0.827958999634j))*x[2]
            ref[(1, 1, 2)]=(-0.442663925821-0.803674257108j)-((-0.177933886148+0.0875220618131j))*(o-1.)
            arg[(1, 1, 3)]=(0.088937823563-0.544918705281j)*x[0]**o + ((0.361192430846+0.0312899890741j))*x[0] + ((-0.578998936395+0.033845990507j))*x[1]**o + ((-0.107755095997-0.8401260809j))*x[1] + ((-0.261220470649+0.760769661539j))*x[2]**o + ((-0.692774352742+0.608887384122j))*x[2]
            ref[(1, 1, 3)]=(-0.595309300687+0.0248741195304j)-((-0.125213597247+0.0416161577942j))*(o-1.)
            arg[(1, 2, 0)]=(0.387820897314+0.857314109401j)*x[0]**o + ((-0.184449632679-0.032721436627j))*x[0] + ((0.518283213199-0.0832069352306j))*x[1]**o + ((0.890789330409-0.367479317901j))*x[1] + ((-0.579252286757+0.870711854747j))*x[2]**o + ((-0.16909663213+0.435300103237j))*x[2]
            ref[(1, 2, 0)]=(0.432047444678+0.839959188813j)-((0.0544753039594+0.27413650482j))*(o-1.)
            arg[(1, 2, 1)]=(-0.634288020291-0.850227558383j)*x[0]**o + ((-0.945245253479-0.814538990352j))*x[0] + ((0.192244484495-0.139968749742j))*x[1]**o + ((-0.16279381909-0.819728410964j))*x[1] + ((-0.340620040239+0.548283498959j))*x[2]**o + ((0.984887782289-0.42696077957j))*x[2]
            ref[(1, 2, 1)]=(-0.452907433157-1.25157049503j)-((-0.130443929339-0.0736521348612j))*(o-1.)
            arg[(1, 2, 2)]=(-0.888982818586-0.0987917674821j)*x[0]**o + ((0.272597604588-0.788707519027j))*x[0] + ((-0.475960304012+0.623035560476j))*x[1]**o + ((-0.75634240932+0.0732058778518j))*x[1] + ((0.31915541639+0.346654696752j))*x[2]**o + ((0.204221876647-0.502017517848j))*x[2]
            ref[(1, 2, 2)]=(-0.662655317147-0.173310334638j)-((-0.174297951035+0.145149748291j))*(o-1.)
            arg[(1, 2, 3)]=(0.145465437999+0.9141014133j)*x[0]**o + ((-0.818392977982-0.727036823712j))*x[0] + ((0.366200669843+0.157665828085j))*x[1]**o + ((-0.0198173112412-0.510064697404j))*x[1] + ((-0.0816839250288-0.455168984518j))*x[2]**o + ((0.682036109888+0.327324492707j))*x[2]
            ref[(1, 2, 3)]=(0.136904001739-0.146589385772j)-((0.0716636971356+0.102766376144j))*(o-1.)
            arg[(2, 0, 0)]=(-0.317524759223-0.32505999367j)*x[0]**o + ((-0.89089277914-0.331904704433j))*x[0] + ((-0.68729314797+0.974547430742j))*x[1]**o + ((0.776115714224-0.149963299224j))*x[1] + ((0.561092673195+0.924773509691j))*x[2]**o + ((0.966979785634-0.679898727629j))*x[2]
            ref[(2, 0, 0)]=(0.204238743361+0.206247107738j)-((-0.0739542056661+0.26237682446j))*(o-1.)
            arg[(2, 0, 1)]=(-0.474804042562-0.960299294734j)*x[0]**o + ((-0.023356676168-0.328521662107j))*x[0] + ((0.68237333863+0.551026764084j))*x[1]**o + ((0.842485126012+0.495500998757j))*x[1] + ((0.69678024347-0.650142175769j))*x[2]**o + ((0.0796014383112+0.679331271733j))*x[2]
            ref[(2, 0, 1)]=(0.901539713847-0.106552049018j)-((0.150724923256-0.176569117737j))*(o-1.)
            arg[(2, 0, 2)]=(-0.0298694464624-0.910742086016j)*x[0]**o + ((0.637974493789+0.629801568387j))*x[0] + ((-0.590226344011-0.607658575405j))*x[1]**o + ((-0.905223845351+0.95045927447j))*x[1] + ((0.968826572025-0.415391926214j))*x[2]**o + ((-0.718127216917+0.431261522172j))*x[2]
            ref[(2, 0, 2)]=(-0.318322893464+0.0388648886973j)-((0.0581217969252-0.322298764606j))*(o-1.)
            arg[(2, 0, 3)]=(0.512088333973-0.689279279181j)*x[0]**o + ((-0.987797312192-0.412763766513j))*x[0] + ((0.419191839352-0.208229573934j))*x[1]**o + ((-0.0715778879149-0.103339099159j))*x[1] + ((-0.886214620141-0.263932301534j))*x[2]**o + ((0.442664201478+0.970923649893j))*x[2]
            ref[(2, 0, 3)]=(-0.285822722722-0.353310185214j)-((0.0075109255307-0.193573525775j))*(o-1.)
            arg[(2, 1, 0)]=(-0.621954154719-0.476372629122j)*x[0]**o + ((-0.833407380664-0.175925346966j))*x[0] + ((-0.855139632783+0.715589327319j))*x[1]**o + ((0.0565459026225-0.458013310485j))*x[1] + ((-0.853010024474+0.340821609436j))*x[2]**o + ((0.209793849346+0.0509042713633j))*x[2]
            ref[(2, 1, 0)]=(-1.44858572034-0.00149803922785j)-((-0.388350635329+0.096673051272j))*(o-1.)
            arg[(2, 1, 1)]=(0.722755686002+0.363154274177j)*x[0]**o + ((0.225629814966-0.290735773865j))*x[0] + ((-0.99553859558+0.669424303206j))*x[1]**o + ((0.22942303921+0.44886984928j))*x[1] + ((0.286287807616-0.0522724864579j))*x[2]**o + ((-0.973937532837-0.332873494379j))*x[2]
            ref[(2, 1, 1)]=(-0.252689890312+0.402783335981j)-((0.00225081633951+0.163384348488j))*(o-1.)
            arg[(2, 1, 2)]=(-0.801950218599-0.261967722977j)*x[0]**o + ((-0.259327535678+0.688819090214j))*x[0] + ((-0.846175427682+0.331858837843j))*x[1]**o + ((-0.138370609269-0.202642688984j))*x[1] + ((0.67382726332-0.641364450662j))*x[2]**o + ((-0.62215179213+0.850327475208j))*x[2]
            ref[(2, 1, 2)]=(-0.997074160019+0.382515270321j)-((-0.162383063827-0.095245555966j))*(o-1.)
            arg[(2, 1, 3)]=(-0.601699192645-0.338724832292j)*x[0]**o + ((-0.741495331294-0.78022904868j))*x[0] + ((0.205734110571+0.328845829693j))*x[1]**o + ((0.241472754611-0.198573663205j))*x[1] + ((0.378470701264+0.90380925809j))*x[2]**o + ((-0.0219610819109-0.730821100743j))*x[2]
            ref[(2, 1, 3)]=(-0.269739019702-0.407846778568j)-((-0.00291573013502+0.148988375915j))*(o-1.)
            arg[(2, 2, 0)]=(-0.237999003536-0.744606635727j)*x[0]**o + ((0.944699650559+0.400643815896j))*x[0] + ((-0.337533356476+0.0673911624916j))*x[1]**o + ((-0.799256012166-0.699006033645j))*x[1] + ((-0.807700184852-0.667755974786j))*x[2]**o + ((-0.0897329768748+0.0986870640534j))*x[2]
            ref[(2, 2, 0)]=(-0.663760941673-0.772323300859j)-((-0.230538757477-0.224161908004j))*(o-1.)
            arg[(2, 2, 1)]=(-0.603207103312+0.908842751267j)*x[0]**o + ((0.545223308964-0.173729547618j))*x[0] + ((0.914906861668-0.225148747936j))*x[1]**o + ((0.390523410689+0.610190607322j))*x[1] + ((0.910933651828-0.167869664968j))*x[2]**o + ((-0.641643309675-0.650964898258j))*x[2]
            ref[(2, 2, 1)]=(0.758368410081+0.150660249905j)-((0.203772235031+0.0859707230605j))*(o-1.)
            arg[(2, 2, 2)]=(0.737644985557+0.485677034958j)*x[0]**o + ((-0.241864870472-0.758132751666j))*x[0] + ((0.871112475699-0.776815511925j))*x[1]**o + ((-0.807000362082-0.522255125174j))*x[1] + ((0.662848018236-0.459500686894j))*x[2]**o + ((0.0378835540869+0.914899022796j))*x[2]
            ref[(2, 2, 2)]=(0.630311900512-0.558064008953j)-((0.378600913249-0.12510652731j))*(o-1.)
            arg[(2, 2, 3)]=(0.560821937332+0.0527465984192j)*x[0]**o + ((-0.00143900185686-0.603346292858j))*x[0] + ((0.711752114352-0.684580900705j))*x[1]**o + ((0.722007355009-0.410844586761j))*x[1] + ((0.985510410341-0.356578542538j))*x[2]**o + ((-0.407619137656-0.344491409522j))*x[2]
            ref[(2, 2, 3)]=(1.28551683876-1.17354756698j)-((0.376347410337-0.164735474137j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunction_fromData_Function_rank4(self):
        """
        tests integral of rank 4 Data on Function

        assumptions: Function(self.domain) exists
                     self.domain supports integral on Function
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = Function(self.domain)
        w = Function(self.domain)
        x = w.getX()
        arg = Data(0,(4, 4, 3, 2),w)
        ref=numpy.zeros((4, 4, 3, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.787420617704-0.665070593639j)*x[0]**o + ((0.257482049418-0.927613401845j))*x[0] + ((-0.358090667147-0.507424776221j))*x[1]**o + ((-0.381646219026-0.398291784985j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.63483772723-1.24920027835j)-((-0.190918547475-0.195415894977j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.610554520359+0.90818794069j)*x[0]**o + ((-0.657723616236+0.353841315019j))*x[0] + ((-0.520005659869+0.26919754497j))*x[1]**o + ((0.865804093084+0.844393545381j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.46123985169+1.18781017303j)-((-0.188426696705+0.196230914277j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.16130546855-0.210028507911j)*x[0]**o + ((-0.800450999918+0.285232069054j))*x[0] + ((0.327099393977-0.291497177742j))*x[1]**o + ((-0.368529853264+0.938853721433j))*x[1]
            ref[(0, 0, 1, 0)]=(-0.340287995328+0.361280052417j)-((0.0814008104211-0.0835876142754j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.295917694786+0.000819943897388j)*x[0]**o + ((-0.614470155906-0.378970842139j))*x[0] + ((0.891120341279+0.569353567369j))*x[1]**o + ((-0.838625617828+0.372168539584j))*x[1]
            ref[(0, 0, 1, 1)]=(-0.133028868834+0.281685604356j)-((0.197839672677+0.0950289185443j))*(o-1.)
            arg[(0, 0, 2, 0)]=(0.78036736619-0.955690379769j)*x[0]**o + ((0.415920083381-0.667813179408j))*x[0] + ((0.083778864916+0.648030544904j))*x[1]**o + ((0.406962891093+0.61343502493j))*x[1]
            ref[(0, 0, 2, 0)]=(0.84351460279-0.181018994672j)-((0.144024371851-0.0512766391442j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.590039754833+0.437497267697j)*x[0]**o + ((-0.460763640128+0.21772636348j))*x[0] + ((0.506143299801-0.137068539618j))*x[1]**o + ((0.014173737834+0.915320544172j))*x[1]
            ref[(0, 0, 2, 1)]=(0.32479657617+0.716737817865j)-((0.182697175772+0.0500714546798j))*(o-1.)
            arg[(0, 1, 0, 0)]=(0.418252152013+0.0469779002096j)*x[0]**o + ((0.984960504269-0.733405897185j))*x[0] + ((-0.27936931004+0.0324142035766j))*x[1]**o + ((-0.483470380521-0.417242440249j))*x[1]
            ref[(0, 1, 0, 0)]=(0.32018648286-0.535628116824j)-((0.0231471403288+0.0132320172977j))*(o-1.)
            arg[(0, 1, 0, 1)]=(-0.937541044528-0.508292068707j)*x[0]**o + ((0.8485663803-0.083671625087j))*x[0] + ((0.555188150654+0.152479836573j))*x[1]**o + ((0.418182412526+0.208454989081j))*x[1]
            ref[(0, 1, 0, 1)]=(0.442197949476-0.11551443407j)-((-0.0637254823123-0.0593020386889j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.604305974074+0.342942045616j)*x[0]**o + ((0.103306362297-0.0142194636466j))*x[0] + ((-0.513951741348-0.132345497021j))*x[1]**o + ((-0.915455443869-0.568515330782j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.965203398497-0.186069122917j)-((-0.186376285904+0.0350994247659j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.688374048696+0.666667576899j)*x[0]**o + ((-0.997385775217+0.580082743351j))*x[0] + ((0.40465260985-0.675072042952j))*x[1]**o + ((-0.749317563469-0.429537852666j))*x[1]
            ref[(0, 1, 1, 1)]=(-1.01521238877+0.0710702123162j)-((-0.0472869064743-0.00140074434218j))*(o-1.)
            arg[(0, 1, 2, 0)]=(-0.978288788351+0.772060342033j)*x[0]**o + ((-0.151149481731+0.856420652877j))*x[0] + ((0.696189962573+0.188473066315j))*x[1]**o + ((-0.432824435958-0.492059066939j))*x[1]
            ref[(0, 1, 2, 0)]=(-0.433036371734+0.662447497143j)-((-0.047016470963+0.160088901391j))*(o-1.)
            arg[(0, 1, 2, 1)]=(0.0820912096672-0.0242012244207j)*x[0]**o + ((-0.0906775411649-0.0665919806942j))*x[0] + ((-0.673211330394+0.236924068118j))*x[1]**o + ((-0.0844535473453-0.176649690084j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.383125604618-0.0152594135404j)-((-0.0985200201211+0.0354538072829j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.0398566481256+0.179415196966j)*x[0]**o + ((0.470026498456-0.754699180102j))*x[0] + ((-0.580022495684-0.283925767257j))*x[1]**o + ((-0.5105360391+0.874785458458j))*x[1]
            ref[(0, 2, 0, 0)]=(-0.290337694101+0.00778785403231j)-((-0.0900276412597-0.0174184283818j))*(o-1.)
            arg[(0, 2, 0, 1)]=(0.11221455396+0.471631748316j)*x[0]**o + ((-0.956094719608+0.0809682923284j))*x[0] + ((0.48563772843+0.105513217967j))*x[1]**o + ((-0.586875783445+0.530722385049j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.472559110331+0.59441782183j)-((0.0996420470651+0.0961908277139j))*(o-1.)
            arg[(0, 2, 1, 0)]=(0.547810339792+0.343017725044j)*x[0]**o + ((-0.825746317172+0.46423568678j))*x[0] + ((-0.974754419106+0.283968804196j))*x[1]**o + ((-0.373398858455-0.504157315297j))*x[1]
            ref[(0, 2, 1, 0)]=(-0.81304462747+0.293532450361j)-((-0.0711573465523+0.104497754873j))*(o-1.)
            arg[(0, 2, 1, 1)]=(-0.808654406994-0.271903732598j)*x[0]**o + ((0.845091509358-0.720418032179j))*x[0] + ((-0.656219442762-0.0798032934958j))*x[1]**o + ((-0.928710094269-0.368392827252j))*x[1]
            ref[(0, 2, 1, 1)]=(-0.774246217334-0.720258942762j)-((-0.244145641626-0.0586178376823j))*(o-1.)
            arg[(0, 2, 2, 0)]=(-0.480654582667-0.41587535494j)*x[0]**o + ((0.938769870748+0.784948949509j))*x[0] + ((0.788433532949-0.649452056074j))*x[1]**o + ((-0.482834759125+0.903737808627j))*x[1]
            ref[(0, 2, 2, 0)]=(0.381857030952+0.311679673561j)-((0.0512964917137-0.177554568502j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.810001187299-0.567627686901j)*x[0]**o + ((-0.639899305904+0.668314037431j))*x[0] + ((0.406524714489-0.56927093514j))*x[1]**o + ((0.247607468557+0.460090870581j))*x[1]
            ref[(0, 2, 2, 1)]=(-0.397884155078-0.00424685701439j)-((-0.0672460788017-0.189483103674j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.215633691519+0.792378267158j)*x[0]**o + ((0.879611077424+0.169190415062j))*x[0] + ((0.929243471895+0.493262613977j))*x[1]**o + ((0.293167484302-0.150409393776j))*x[1]
            ref[(0, 3, 0, 0)]=(1.15882786257+0.65221095121j)-((0.190812860569+0.214273480189j))*(o-1.)
            arg[(0, 3, 0, 1)]=(0.289316363039-0.437660707623j)*x[0]**o + ((0.697099754833+0.996179826038j))*x[0] + ((-0.390631105636-0.184530228343j))*x[1]**o + ((-0.977648446855-0.850069534284j))*x[1]
            ref[(0, 3, 0, 1)]=(-0.19093171731-0.238040322106j)-((-0.0168857904329-0.103698489328j))*(o-1.)
            arg[(0, 3, 1, 0)]=(-0.770121988137+0.954392905203j)*x[0]**o + ((-0.335512447482-0.620237000663j))*x[0] + ((0.455641606072+0.105006271532j))*x[1]**o + ((-0.813162058356+0.897848585167j))*x[1]
            ref[(0, 3, 1, 0)]=(-0.731577443952+0.66850538062j)-((-0.0524133970108+0.176566529456j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.147597812963+0.131759927975j)*x[0]**o + ((-0.506433633202-0.471194632629j))*x[0] + ((-0.171634172053+0.299066667456j))*x[1]**o + ((0.895725446298+0.838144547149j))*x[1]
            ref[(0, 3, 1, 1)]=(0.0350299140405+0.398888254975j)-((-0.053205330836+0.0718044325718j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.636899420534+0.285360084074j)*x[0]**o + ((0.834173004008-0.181301095456j))*x[0] + ((0.0826744486251+0.701681295222j))*x[1]**o + ((-0.338402776215-0.0375310245075j))*x[1]
            ref[(0, 3, 2, 0)]=(0.607672048476+0.384104629666j)-((0.119928978193+0.164506896549j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.861881891361-0.51465232248j)*x[0]**o + ((-0.411323693268-0.590255607506j))*x[0] + ((0.0439679195117-0.0973273356247j))*x[1]**o + ((-0.413656567613-0.98121752991j))*x[1]
            ref[(0, 3, 2, 1)]=(-0.821447116365-1.09172639776j)-((-0.136318995308-0.101996609684j))*(o-1.)
            arg[(1, 0, 0, 0)]=(0.457256686014-0.0365562205415j)*x[0]**o + ((-0.518277609547+0.0762537018088j))*x[0] + ((-0.695450904537-0.755080915826j))*x[1]**o + ((-0.277147009431-0.0993101056635j))*x[1]
            ref[(1, 0, 0, 0)]=(-0.51680941875-0.407346770111j)-((-0.0396990364204-0.131939522728j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.800766304053+0.77898850145j)*x[0]**o + ((-0.53099034102+0.739840899299j))*x[0] + ((0.595844313226+0.0125107106377j))*x[1]**o + ((-0.60922069732-0.705489339766j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.672566514583+0.41292538581j)-((-0.0341536651377+0.131916535348j))*(o-1.)
            arg[(1, 0, 1, 0)]=(0.506645523348+0.964733573422j)*x[0]**o + ((0.433347044201+0.915835882461j))*x[0] + ((-0.19656814325-0.134117482604j))*x[1]**o + ((-0.968918159165-0.809593738391j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.112746867432+0.468429117444j)-((0.0516795633498+0.138436015136j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.159997999661+0.585953577603j)*x[0]**o + ((0.755305762504-0.396088326805j))*x[0] + ((-0.213488147986-0.556536249155j))*x[1]**o + ((-0.6893105029+0.810803624015j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.153745444022+0.222066312829j)-((-0.0622476912746+0.00490288807473j))*(o-1.)
            arg[(1, 0, 2, 0)]=(0.185208849037+0.444640087496j)*x[0]**o + ((-0.308827478363+0.299453891866j))*x[0] + ((-0.290495820236+0.727594475571j))*x[1]**o + ((0.842754959435+0.200594143111j))*x[1]
            ref[(1, 0, 2, 0)]=(0.214320254937+0.836141299022j)-((-0.0175478285331+0.195372427178j))*(o-1.)
            arg[(1, 0, 2, 1)]=(-0.871106952342+0.70799700807j)*x[0]**o + ((-0.109218675807+0.53653604452j))*x[0] + ((0.398177332227-0.987002289651j))*x[1]**o + ((0.202404789119-0.0768827184826j))*x[1]
            ref[(1, 0, 2, 1)]=(-0.189871753402+0.0903240222276j)-((-0.0788216033525-0.0465008802637j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.79507816319-0.159749297072j)*x[0]**o + ((-0.39095689946+0.267924604028j))*x[0] + ((-0.959773376849-0.601315132788j))*x[1]**o + ((0.990716104946-0.0650930449181j))*x[1]
            ref[(1, 1, 0, 0)]=(0.217531995913-0.279116435375j)-((-0.0274492022765-0.126844071643j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.972603277201+0.522013768858j)*x[0]**o + ((0.149596632303-0.980373697956j))*x[0] + ((0.717048988103+0.596237977526j))*x[1]**o + ((-0.370079849986+0.25410644905j))*x[1]
            ref[(1, 1, 0, 1)]=(-0.238018753391+0.195992248739j)-((-0.0425923815164+0.186375291064j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.693709315318+0.972956311455j)*x[0]**o + ((-0.610364703205-0.0418435119655j))*x[0] + ((-0.804195730847-0.140309938569j))*x[1]**o + ((0.968370522446+0.479084827994j))*x[1]
            ref[(1, 1, 1, 0)]=(0.123759701856+0.634943844457j)-((-0.0184144025882+0.138774395481j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.211996092504+0.480228641149j)*x[0]**o + ((0.626020963314-0.410581198054j))*x[0] + ((-0.264784921109+0.425204299771j))*x[1]**o + ((-0.673912099764+0.259824710347j))*x[1]
            ref[(1, 1, 1, 1)]=(-0.0503399825278+0.377338226607j)-((-0.00879813810096+0.150905490153j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.701338395302+0.48412023127j)*x[0]**o + ((0.012880228763+0.262844034989j))*x[0] + ((-0.639174921401+0.979846752903j))*x[1]**o + ((-0.288509013288+0.55208305217j))*x[1]
            ref[(1, 1, 2, 0)]=(-0.106732655312+1.13944703567j)-((0.0103605789834+0.243994497362j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.118537560003-0.463761848254j)*x[0]**o + ((-0.172974839741-0.97303309127j))*x[0] + ((0.610113522181-0.864828238296j))*x[1]**o + ((-0.389749225328+0.80383132156j))*x[1]
            ref[(1, 1, 2, 1)]=(-0.0355740514454-0.74889592813j)-((0.0819293270297-0.221431681092j))*(o-1.)
            arg[(1, 2, 0, 0)]=(0.716617554921-0.452384481085j)*x[0]**o + ((-0.390044548697-0.722969397352j))*x[0] + ((0.943185184572-0.167395660075j))*x[1]**o + ((-0.0604092844123-0.941406931513j))*x[1]
            ref[(1, 2, 0, 0)]=(0.604674453192-1.14207823501j)-((0.276633789915-0.103296690193j))*(o-1.)
            arg[(1, 2, 0, 1)]=(0.73282428661-0.437168217472j)*x[0]**o + ((-0.0726877237075+0.473113463454j))*x[0] + ((0.554396862103+0.994981321788j))*x[1]**o + ((-0.641469511905+0.6450889422j))*x[1]
            ref[(1, 2, 0, 1)]=(0.28653195655+0.838007754985j)-((0.214536858119+0.0929688507194j))*(o-1.)
            arg[(1, 2, 1, 0)]=(-0.862474681119+0.663098655609j)*x[0]**o + ((0.606236848839+0.584379415795j))*x[0] + ((-0.317243490603-0.872983707953j))*x[1]**o + ((-0.409173920864+0.508470520761j))*x[1]
            ref[(1, 2, 1, 0)]=(-0.491327621873+0.441482442106j)-((-0.196619695287-0.0349808420573j))*(o-1.)
            arg[(1, 2, 1, 1)]=(0.975077594251-0.395419539837j)*x[0]**o + ((0.352848738408-0.819635074843j))*x[0] + ((-0.735373780589-0.791700591072j))*x[1]**o + ((-0.645659425709-0.328342045838j))*x[1]
            ref[(1, 2, 1, 1)]=(-0.0265534368194-1.16754862579j)-((0.0399506356102-0.197853355152j))*(o-1.)
            arg[(1, 2, 2, 0)]=(-0.603016184399-0.787888249516j)*x[0]**o + ((-0.0572521972689-0.967411189872j))*x[0] + ((-0.0980036143125-0.347780351604j))*x[1]**o + ((0.237828809523+0.277015259539j))*x[1]
            ref[(1, 2, 2, 0)]=(-0.260221593228-0.913032265727j)-((-0.116836633119-0.189278100187j))*(o-1.)
            arg[(1, 2, 2, 1)]=(-0.361575241468-0.0494598084078j)*x[0]**o + ((-0.93954948213-0.126634006447j))*x[0] + ((0.997121528587-0.389272915943j))*x[1]**o + ((-0.375163195326-0.643862135652j))*x[1]
            ref[(1, 2, 2, 1)]=(-0.339583195169-0.604614433225j)-((0.105924381186-0.0731221207252j))*(o-1.)
            arg[(1, 3, 0, 0)]=(-0.342765316257-0.833150841651j)*x[0]**o + ((0.203265587456-0.521057660381j))*x[0] + ((-0.548837030062-0.471031209648j))*x[1]**o + ((-0.631076857978-0.913436951021j))*x[1]
            ref[(1, 3, 0, 0)]=(-0.65970680842-1.36933833135j)-((-0.148600391053-0.217363675216j))*(o-1.)
            arg[(1, 3, 0, 1)]=(0.677686502593+0.411786248282j)*x[0]**o + ((-0.395664020936-0.802333645078j))*x[0] + ((0.375103431642+0.269714353032j))*x[1]**o + ((0.711098073283-0.596824455523j))*x[1]
            ref[(1, 3, 0, 1)]=(0.684111993291-0.358828749644j)-((0.175464989039+0.113583433552j))*(o-1.)
            arg[(1, 3, 1, 0)]=(0.226176406392+0.458724303778j)*x[0]**o + ((-0.998890524321-0.700253647788j))*x[0] + ((0.576234209791+0.933676045921j))*x[1]**o + ((0.210797552418+0.289769213317j))*x[1]
            ref[(1, 3, 1, 0)]=(0.00715882213975+0.490957957614j)-((0.133735102697+0.23206672495j))*(o-1.)
            arg[(1, 3, 1, 1)]=(-0.191210854592+0.826712247132j)*x[0]**o + ((0.437917988652-0.860912716055j))*x[0] + ((-0.405819108894-0.138942191901j))*x[1]**o + ((-0.251496244094+0.642992708943j))*x[1]
            ref[(1, 3, 1, 1)]=(-0.205304109464+0.23492502406j)-((-0.0995049939143+0.114628342539j))*(o-1.)
            arg[(1, 3, 2, 0)]=(-0.338330538369-0.456216462347j)*x[0]**o + ((-0.681714497603+0.275811631234j))*x[0] + ((-0.710807330369-0.451426594183j))*x[1]**o + ((0.514906677073+0.232502062485j))*x[1]
            ref[(1, 3, 2, 0)]=(-0.607972844634-0.199664681405j)-((-0.174856311456-0.151273842755j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.631150877312-0.851168440766j)*x[0]**o + ((0.416097983518+0.736483881087j))*x[0] + ((0.911789173502+0.863583232133j))*x[1]**o + ((0.140001241021-0.603411771121j))*x[1]
            ref[(1, 3, 2, 1)]=(0.418368760364+0.0727434506663j)-((0.0467730493649+0.00206913189441j))*(o-1.)
            arg[(2, 0, 0, 0)]=(-0.433787164244-0.615400149872j)*x[0]**o + ((0.714667513152+0.559277831525j))*x[0] + ((-0.219057963905+0.497905775911j))*x[1]**o + ((-0.0875959228384+0.465962709883j))*x[1]
            ref[(2, 0, 0, 0)]=(-0.0128867689176+0.453873083723j)-((-0.108807521358-0.0195823956602j))*(o-1.)
            arg[(2, 0, 0, 1)]=(0.355076612566-0.9668945955j)*x[0]**o + ((0.519265037465+0.848978414871j))*x[0] + ((0.245867037208-0.593868109668j))*x[1]**o + ((0.959054285852+0.190045029925j))*x[1]
            ref[(2, 0, 0, 1)]=(1.03963148655-0.260869630186j)-((0.100157274962-0.260127117528j))*(o-1.)
            arg[(2, 0, 1, 0)]=(0.102295065529+0.874321260728j)*x[0]**o + ((-0.795150122997+0.828627291512j))*x[0] + ((0.0820398907107+0.597550378464j))*x[1]**o + ((-0.884224222713-0.722188709106j))*x[1]
            ref[(2, 0, 1, 0)]=(-0.747519694735+0.789155110799j)-((0.0307224927067+0.245311939865j))*(o-1.)
            arg[(2, 0, 1, 1)]=(-0.313913094194-0.484700951422j)*x[0]**o + ((0.750932311945+0.471108122409j))*x[0] + ((-0.12218662499+0.70031148685j))*x[1]**o + ((-0.497234880786-0.621876463104j))*x[1]
            ref[(2, 0, 1, 1)]=(-0.0912011440122+0.0324210973665j)-((-0.0726832865307+0.035935089238j))*(o-1.)
            arg[(2, 0, 2, 0)]=(-0.658833430404-0.851244873528j)*x[0]**o + ((-0.0294116250365-0.426917190124j))*x[0] + ((0.147609079887+0.0206341141459j))*x[1]**o + ((0.217057045836-0.779872076971j))*x[1]
            ref[(2, 0, 2, 0)]=(-0.161789464858-1.01870001324j)-((-0.0852040584195-0.138435126564j))*(o-1.)
            arg[(2, 0, 2, 1)]=(-0.96789106786+0.495910906076j)*x[0]**o + ((-0.675167564827-0.337726536191j))*x[0] + ((0.266921846494-0.00696722791664j))*x[1]**o + ((0.598988942592+0.402967332759j))*x[1]
            ref[(2, 0, 2, 1)]=(-0.3885739218+0.277092237364j)-((-0.116828203561+0.0814906130266j))*(o-1.)
            arg[(2, 1, 0, 0)]=(0.493847723727-0.6366187055j)*x[0]**o + ((-0.830341478702+0.00534409619283j))*x[0] + ((-0.10781652013+0.472423804511j))*x[1]**o + ((-0.754995647075-0.219264302151j))*x[1]
            ref[(2, 1, 0, 0)]=(-0.59965296109-0.189057553474j)-((0.0643385339329-0.0273658168315j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.176017279478+0.460007508696j)*x[0]**o + ((0.622844451665+0.801430138308j))*x[0] + ((0.377256647367-0.0484227110213j))*x[1]**o + ((-0.828530820658-0.0570270865524j))*x[1]
            ref[(2, 1, 0, 1)]=(-0.0022235005517+0.577993924715j)-((0.0335398946482+0.0685974662791j))*(o-1.)
            arg[(2, 1, 1, 0)]=(0.497345369226-0.624488675265j)*x[0]**o + ((0.621870174933+0.979638384791j))*x[0] + ((-0.790513690407+0.486356645492j))*x[1]**o + ((-0.292846210484+0.927239060593j))*x[1]
            ref[(2, 1, 1, 0)]=(0.017927821634+0.884372707805j)-((-0.0488613868635-0.0230220049622j))*(o-1.)
            arg[(2, 1, 1, 1)]=(0.45634374882+0.653477080681j)*x[0]**o + ((-0.136680425627-0.983015263754j))*x[0] + ((0.883529136674-0.453103689783j))*x[1]**o + ((-0.594157558024+0.611568073764j))*x[1]
            ref[(2, 1, 1, 1)]=(0.304517450922-0.0855368995462j)-((0.223312147582+0.0333955651496j))*(o-1.)
            arg[(2, 1, 2, 0)]=(0.895218736786-0.599939367013j)*x[0]**o + ((0.524171067675-0.494329673674j))*x[0] + ((0.655050313493-0.983491095142j))*x[1]**o + ((-0.673348165788-0.285950224348j))*x[1]
            ref[(2, 1, 2, 0)]=(0.700545976083-1.18185518009j)-((0.258378175046-0.263905077026j))*(o-1.)
            arg[(2, 1, 2, 1)]=(0.799984787002-0.145063359959j)*x[0]**o + ((0.798341674042+0.340447875248j))*x[0] + ((0.424533050117-0.983175116676j))*x[1]**o + ((0.156478219936+0.373920571051j))*x[1]
            ref[(2, 1, 2, 1)]=(1.08966886555-0.206935015168j)-((0.204086306187-0.188039746106j))*(o-1.)
            arg[(2, 2, 0, 0)]=(-0.456388741014+0.033058048729j)*x[0]**o + ((0.0048322529307+0.485453705228j))*x[0] + ((0.191068126761-0.631080679609j))*x[1]**o + ((0.879073111235+0.300959024341j))*x[1]
            ref[(2, 2, 0, 0)]=(0.309292374956+0.0941950493446j)-((-0.0442201023755-0.09967043848j))*(o-1.)
            arg[(2, 2, 0, 1)]=(-0.415841261608+0.946342194941j)*x[0]**o + ((-0.637137479898+0.473890045483j))*x[0] + ((0.95012621543-0.0621362618826j))*x[1]**o + ((-0.476022621875-0.660261840749j))*x[1]
            ref[(2, 2, 0, 1)]=(-0.289437573975+0.348917068896j)-((0.0890474923036+0.14736765551j))*(o-1.)
            arg[(2, 2, 1, 0)]=(0.165728950937-0.841441851765j)*x[0]**o + ((0.122634846698+0.794196668574j))*x[0] + ((0.850306284765+0.020935551357j))*x[1]**o + ((0.303020126712-0.421255533298j))*x[1]
            ref[(2, 2, 1, 0)]=(0.720845104556-0.223782582566j)-((0.16933920595-0.136751050068j))*(o-1.)
            arg[(2, 2, 1, 1)]=(0.206871997837-0.696247833608j)*x[0]**o + ((0.556297375454+0.336578815512j))*x[0] + ((-0.695129919311-0.150379005607j))*x[1]**o + ((-0.946812120873+0.163451664747j))*x[1]
            ref[(2, 2, 1, 1)]=(-0.439386333446-0.173298179478j)-((-0.0813763202457-0.141104473202j))*(o-1.)
            arg[(2, 2, 2, 0)]=(0.682132617076-0.819089870107j)*x[0]**o + ((0.00795993309438+0.827569413954j))*x[0] + ((-0.400765987251-0.863717213772j))*x[1]**o + ((-0.147595524345+0.147557116288j))*x[1]
            ref[(2, 2, 2, 0)]=(0.0708655192873-0.353840276819j)-((0.0468944383042-0.280467847313j))*(o-1.)
            arg[(2, 2, 2, 1)]=(-0.42413295448+0.976170782182j)*x[0]**o + ((-0.22925017923+0.432634465257j))*x[0] + ((0.309165075661-0.940571735769j))*x[1]**o + ((-0.190985126841+0.266230100292j))*x[1]
            ref[(2, 2, 2, 1)]=(-0.267601592445+0.367231805981j)-((-0.0191613131365+0.00593317440219j))*(o-1.)
            arg[(2, 3, 0, 0)]=(-0.174242554302+0.824611324505j)*x[0]**o + ((0.115354413143+0.77902756229j))*x[0] + ((0.386236699972+0.900726472177j))*x[1]**o + ((0.750793152462-0.960926603025j))*x[1]
            ref[(2, 3, 0, 0)]=(0.539070855637+0.771719377973j)-((0.0353323576116+0.287556299447j))*(o-1.)
            arg[(2, 3, 0, 1)]=(0.863495739956+0.457483012208j)*x[0]**o + ((0.49253939739+0.720154617833j))*x[0] + ((0.147979595701+0.452487627895j))*x[1]**o + ((0.398648947197-0.327808913169j))*x[1]
            ref[(2, 3, 0, 1)]=(0.951331840122+0.651158172383j)-((0.168579222609+0.15166177335j))*(o-1.)
            arg[(2, 3, 1, 0)]=(-0.507061099264+0.826078135744j)*x[0]**o + ((-0.109522167959-0.678345666499j))*x[0] + ((0.00766385377161+0.220758467479j))*x[1]**o + ((0.222249915372-0.529016691819j))*x[1]
            ref[(2, 3, 1, 0)]=(-0.19333474904-0.0802628775476j)-((-0.0832328742488+0.174472767204j))*(o-1.)
            arg[(2, 3, 1, 1)]=(-0.303017133982+0.408762455738j)*x[0]**o + ((0.579698078051-0.277271791964j))*x[0] + ((0.26309571818-0.7913196785j))*x[1]**o + ((0.148732162736-0.70685666889j))*x[1]
            ref[(2, 3, 1, 1)]=(0.344254412492-0.683342841808j)-((-0.00665356930032-0.0637595371269j))*(o-1.)
            arg[(2, 3, 2, 0)]=(-0.117455350437-0.911116254291j)*x[0]**o + ((-0.284486808499+0.939190139178j))*x[0] + ((-0.346604254719-0.398322654883j))*x[1]**o + ((0.749560159453+0.419898807974j))*x[1]
            ref[(2, 3, 2, 0)]=(0.000506872899105+0.024825018989j)-((-0.0773432675259-0.218239818196j))*(o-1.)
            arg[(2, 3, 2, 1)]=(0.694192605691+0.153082586564j)*x[0]**o + ((0.986428552243+0.221668277118j))*x[0] + ((0.896784351202+0.362638236507j))*x[1]**o + ((-0.682764071862-0.659335883382j))*x[1]
            ref[(2, 3, 2, 1)]=(0.947320718637+0.0390266084028j)-((0.265162826149+0.0859534705117j))*(o-1.)
            arg[(3, 0, 0, 0)]=(0.587834815034+0.931173220166j)*x[0]**o + ((0.227195759112-0.371002860436j))*x[0] + ((0.908831293912-0.733522084401j))*x[1]**o + ((-0.535800291954-0.843661970325j))*x[1]
            ref[(3, 0, 0, 0)]=(0.594030788052-0.508506847498j)-((0.249444351491+0.0329418559608j))*(o-1.)
            arg[(3, 0, 0, 1)]=(0.443430373626-0.293978053659j)*x[0]**o + ((-0.134031111672-0.0986469793894j))*x[0] + ((0.0774077277518+0.772876479786j))*x[1]**o + ((-0.170259783297+0.440136526435j))*x[1]
            ref[(3, 0, 0, 1)]=(0.108273603204+0.410193986586j)-((0.0868063502297+0.0798164043546j))*(o-1.)
            arg[(3, 0, 1, 0)]=(0.729754820726+0.423093016284j)*x[0]**o + ((-0.661974033167-0.499294823999j))*x[0] + ((-0.233250347432-0.742410072309j))*x[1]**o + ((0.656551665361+0.144135311787j))*x[1]
            ref[(3, 0, 1, 0)]=(0.245541052744-0.337238284119j)-((0.0827507455489-0.0532195093375j))*(o-1.)
            arg[(3, 0, 1, 1)]=(-0.845108890608-0.91610633226j)*x[0]**o + ((0.0770118689866-0.115020922219j))*x[0] + ((0.759143331839+0.464467586459j))*x[1]**o + ((-0.216438911439+0.918703769978j))*x[1]
            ref[(3, 0, 1, 1)]=(-0.112696300611+0.176022050979j)-((-0.0143275931282-0.0752731243002j))*(o-1.)
            arg[(3, 0, 2, 0)]=(-0.970982633176-0.664710297655j)*x[0]**o + ((0.000617978380867+0.37502185483j))*x[0] + ((0.157610111526+0.356952210908j))*x[1]**o + ((0.17962363619-0.569277636553j))*x[1]
            ref[(3, 0, 2, 0)]=(-0.31656545354-0.251006934235j)-((-0.135562086942-0.0512930144579j))*(o-1.)
            arg[(3, 0, 2, 1)]=(-0.559938645551+0.474267814564j)*x[0]**o + ((0.476983846407+0.783768637702j))*x[0] + ((-0.407731656564-0.723039796562j))*x[1]**o + ((-0.710664664216-0.571836294236j))*x[1]
            ref[(3, 0, 2, 1)]=(-0.600675559962-0.0184198192662j)-((-0.161278383686-0.0414619969996j))*(o-1.)
            arg[(3, 1, 0, 0)]=(-0.0898130924622-0.869638430117j)*x[0]**o + ((0.921152625387+0.901650865597j))*x[0] + ((-0.818251990674-0.0355222990619j))*x[1]**o + ((0.701723680725-0.0689328731996j))*x[1]
            ref[(3, 1, 0, 0)]=(0.357405611487-0.0362213683903j)-((-0.151344180523-0.15086012153j))*(o-1.)
            arg[(3, 1, 0, 1)]=(-0.258558375275+0.532154108086j)*x[0]**o + ((-0.000680604197579-0.477350995902j))*x[0] + ((0.689375669777-0.476339737252j))*x[1]**o + ((0.887172260735-0.713210463399j))*x[1]
            ref[(3, 1, 0, 1)]=(0.65865447552-0.567373544233j)-((0.071802882417+0.00930239513914j))*(o-1.)
            arg[(3, 1, 1, 0)]=(-0.421995034306+0.519166718301j)*x[0]**o + ((-0.769054797014+0.894950781422j))*x[0] + ((0.742989549503-0.409357106932j))*x[1]**o + ((-0.13918811014-0.512564503002j))*x[1]
            ref[(3, 1, 1, 0)]=(-0.293624195978+0.246097944894j)-((0.0534990858662+0.0183016018948j))*(o-1.)
            arg[(3, 1, 1, 1)]=(0.578628832831+0.832766068551j)*x[0]**o + ((0.817402549781+0.80970359516j))*x[0] + ((0.938994589156-0.730257118451j))*x[1]**o + ((0.130732635707+0.684262634093j))*x[1]
            ref[(3, 1, 1, 1)]=(1.23287930374+0.798237589676j)-((0.252937236998+0.0170848250165j))*(o-1.)
            arg[(3, 1, 2, 0)]=(0.386991956562-0.38715579409j)*x[0]**o + ((0.621794898058-0.943243779733j))*x[0] + ((-0.945449836853+0.118695159261j))*x[1]**o + ((0.0521871670067+0.229809740637j))*x[1]
            ref[(3, 1, 2, 0)]=(0.0577620923868-0.490947336962j)-((-0.0930763133818-0.0447434391381j))*(o-1.)
            arg[(3, 1, 2, 1)]=(0.648057290478+0.867839305415j)*x[0]**o + ((0.0197548667313+0.520889183635j))*x[0] + ((0.263836893156-0.574120484516j))*x[1]**o + ((0.985924647552+0.178610543205j))*x[1]
            ref[(3, 1, 2, 1)]=(0.958786848959+0.496609273869j)-((0.151982363939+0.0489531368165j))*(o-1.)
            arg[(3, 2, 0, 0)]=(0.836318534998+0.601496839621j)*x[0]**o + ((-0.789955396845+0.140264902798j))*x[0] + ((0.746535301143-0.21283960168j))*x[1]**o + ((-0.849089289645+0.368824256987j))*x[1]
            ref[(3, 2, 0, 0)]=(-0.0280954251745+0.448873198863j)-((0.26380897269+0.0647762063234j))*(o-1.)
            arg[(3, 2, 0, 1)]=(0.280600772541+0.498744141048j)*x[0]**o + ((-0.421677557774+0.325658050795j))*x[0] + ((-0.934056292054-0.695106425383j))*x[1]**o + ((0.979012847722-0.486242771729j))*x[1]
            ref[(3, 2, 0, 1)]=(-0.0480601147821-0.178473502634j)-((-0.108909253252-0.0327270473892j))*(o-1.)
            arg[(3, 2, 1, 0)]=(0.898024725493-0.415767266554j)*x[0]**o + ((0.322059666876-0.895911746291j))*x[0] + ((0.813157746176+0.524242265042j))*x[1]**o + ((-0.463687449605-0.947890107267j))*x[1]
            ref[(3, 2, 1, 0)]=(0.78477734447-0.867663427535j)-((0.285197078612+0.0180791664146j))*(o-1.)
            arg[(3, 2, 1, 1)]=(0.710954536721-0.869214808048j)*x[0]**o + ((0.284015125069+0.371568484189j))*x[0] + ((0.746024678907+0.248780575273j))*x[1]**o + ((-0.69992033076+0.913692460813j))*x[1]
            ref[(3, 2, 1, 1)]=(0.520537004969+0.332413356113j)-((0.242829869271-0.103405705463j))*(o-1.)
            arg[(3, 2, 2, 0)]=(-0.988443925894+0.613471079971j)*x[0]**o + ((-0.437736215246-0.672418285065j))*x[0] + ((-0.983856707447+0.136462368705j))*x[1]**o + ((-0.00463175742688+0.00321361530333j))*x[1]
            ref[(3, 2, 2, 0)]=(-1.20733430301+0.0403643894567j)-((-0.328716772224+0.124988908113j))*(o-1.)
            arg[(3, 2, 2, 1)]=(-0.441472561601+0.779016831993j)*x[0]**o + ((-0.745480869519-0.763055280771j))*x[0] + ((-0.186312553841-0.365686651529j))*x[1]**o + ((-0.0690954013512-0.351436975386j))*x[1]
            ref[(3, 2, 2, 1)]=(-0.721180693156-0.350581037847j)-((-0.104630852574+0.0688883634107j))*(o-1.)
            arg[(3, 3, 0, 0)]=(-0.806315355976+0.353238867405j)*x[0]**o + ((0.271537412852-0.852233705449j))*x[0] + ((-0.751185176803+0.357651659435j))*x[1]**o + ((-0.848968700095+0.00787719488661j))*x[1]
            ref[(3, 3, 0, 0)]=(-1.06746591001-0.0667329918612j)-((-0.25958342213+0.118481754473j))*(o-1.)
            arg[(3, 3, 0, 1)]=(-0.112315445453-0.886800058113j)*x[0]**o + ((0.244060718352+0.464954149418j))*x[0] + ((0.334629153942+0.64873662681j))*x[1]**o + ((0.961843343236+0.733890884995j))*x[1]
            ref[(3, 3, 0, 1)]=(0.714108885039+0.480390801555j)-((0.0370522847482-0.0396772385506j))*(o-1.)
            arg[(3, 3, 1, 0)]=(0.806817450201-0.0442536350796j)*x[0]**o + ((-0.618445290338+0.184435456201j))*x[0] + ((-0.0683648946804-0.279375429839j))*x[1]**o + ((0.530364743531-0.365793236485j))*x[1]
            ref[(3, 3, 1, 0)]=(0.325186004357-0.252493422602j)-((0.12307542592-0.0539381774865j))*(o-1.)
            arg[(3, 3, 1, 1)]=(0.626511813738-0.549586440663j)*x[0]**o + ((-0.825414635053+0.824574411019j))*x[0] + ((0.297606132797-0.407495110955j))*x[1]**o + ((0.784636840034+0.877718105443j))*x[1]
            ref[(3, 3, 1, 1)]=(0.441670075758+0.372605482422j)-((0.154019657756-0.159513591936j))*(o-1.)
            arg[(3, 3, 2, 0)]=(0.68850887425+0.593950756629j)*x[0]**o + ((-0.330881588681-0.236351363572j))*x[0] + ((-0.570027521168+0.840018606496j))*x[1]**o + ((0.685893723365+0.828379261049j))*x[1]
            ref[(3, 3, 2, 0)]=(0.236746743883+1.0129986303j)-((0.0197468921803+0.238994893854j))*(o-1.)
            arg[(3, 3, 2, 1)]=(-0.782102572148-0.702867346106j)*x[0]**o + ((-0.364988688348-0.932902817626j))*x[0] + ((-0.535522560354-0.788596834581j))*x[1]**o + ((-0.533435000725-0.183372757991j))*x[1]
            ref[(3, 3, 2, 1)]=(-1.10802441079-1.30386987815j)-((-0.21960418875-0.248577363448j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(0.985069051759-0.625898470147j)*x[0]**o + ((0.821935975195-0.139584964453j))*x[0] + ((0.575255349579-0.769698355329j))*x[1]**o + ((0.440216328436-0.753278084349j))*x[1] + ((0.943562993166-0.3802187866j))*x[2]**o + ((-0.0423499851525-0.556845257832j))*x[2]
            ref[(0, 0, 0, 0)]=(1.86184485649-1.61276195935j)-((0.417314565751-0.295969268679j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.767670972641-0.770930325389j)*x[0]**o + ((0.253899954161+0.478248542159j))*x[0] + ((-0.0330444621335+0.234901837246j))*x[1]**o + ((-0.896279162069-0.656686961231j))*x[1] + ((0.447564945008+0.253366244186j))*x[2]**o + ((0.776315029605+0.605695633489j))*x[2]
            ref[(0, 0, 0, 1)]=(0.658063638606+0.0722974852301j)-((0.197031909253-0.0471103739928j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.0282545587519+0.506446972409j)*x[0]**o + ((0.795755355362+0.0436529843944j))*x[0] + ((0.132236272147+0.226547470993j))*x[1]**o + ((-0.549679201779+0.46162929728j))*x[1] + ((0.362107884318+0.307276103746j))*x[2]**o + ((-0.922630497827-0.15760615968j))*x[2]
            ref[(0, 0, 1, 0)]=(-0.0769778145135+0.693973334571j)-((0.0870997858694+0.173378424525j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.543626647786+0.134865969272j)*x[0]**o + ((-0.156174509222+0.804562919395j))*x[0] + ((0.818241377068-0.656116850873j))*x[1]**o + ((-0.469899463-0.874083954608j))*x[1] + ((0.486637548308+0.208914744245j))*x[2]**o + ((0.0408931194368-0.869658858483j))*x[2]
            ref[(0, 0, 1, 1)]=(0.631662360189-0.625758015526j)-((0.308084262194-0.0520560228926j))*(o-1.)
            arg[(0, 0, 2, 0)]=(-0.283699144871+0.20928966873j)*x[0]**o + ((0.0634010209085+0.752261992517j))*x[0] + ((-0.398700608197+0.699303835624j))*x[1]**o + ((0.567308925375-0.703717438015j))*x[1] + ((0.475279077239+0.0360521455695j))*x[2]**o + ((-0.498377047763+0.937040738292j))*x[2]
            ref[(0, 0, 2, 0)]=(-0.0373938886537+0.965115471358j)-((-0.0345201126381+0.157440941654j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.979188586031-0.323547395874j)*x[0]**o + ((0.4080011402-0.924063674738j))*x[0] + ((-0.495259823153-0.631423810132j))*x[1]**o + ((0.0317093589042-0.00625052104389j))*x[1] + ((0.274572149793-0.700430845264j))*x[2]**o + ((-0.315214228372-0.581119287761j))*x[2]
            ref[(0, 0, 2, 1)]=(0.441498591702-1.58341776741j)-((0.126416818778-0.275900341878j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.332996919958-0.395776605699j)*x[0]**o + ((-0.574195094368+0.187683274302j))*x[0] + ((0.314507466874+0.247821899095j))*x[1]**o + ((0.967494193059-0.389349414229j))*x[1] + ((0.592589345034+0.852494608101j))*x[2]**o + ((-0.0753482526558+0.109891455088j))*x[2]
            ref[(0, 1, 0, 0)]=(0.446025368993+0.306382608329j)-((0.0956833153251+0.117423316916j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.616996332404-0.0851905058482j)*x[0]**o + ((0.309810622056+0.267772041517j))*x[0] + ((0.171893815635-0.722972033044j))*x[1]**o + ((-0.92113737059+0.0157691447357j))*x[1] + ((-0.896090238541+0.0143848688914j))*x[2]**o + ((0.233730463145+0.732884205092j))*x[2]
            ref[(0, 1, 0, 1)]=(-0.242398187945+0.111323860672j)-((-0.0178666817502-0.132296278333j))*(o-1.)
            arg[(0, 1, 1, 0)]=(0.943394951634-0.386027721948j)*x[0]**o + ((-0.0662200207067+0.595229202538j))*x[0] + ((0.402781211778-0.0247826111779j))*x[1]**o + ((0.716107775599+0.153798802444j))*x[1] + ((-0.460465665771-0.659922506584j))*x[2]**o + ((0.565521140245-0.788150119023j))*x[2]
            ref[(0, 1, 1, 0)]=(1.05055969639-0.554927476875j)-((0.147618416274-0.178455473285j))*(o-1.)
            arg[(0, 1, 1, 1)]=(0.250297143993-0.191512640762j)*x[0]**o + ((-0.992268526805-0.836559501582j))*x[0] + ((-0.699627596519-0.582875195115j))*x[1]**o + ((0.670077664818-0.961498931047j))*x[1] + ((0.488672946382+0.289722056234j))*x[2]**o + ((0.408837631437-0.688961431335j))*x[2]
            ref[(0, 1, 1, 1)]=(0.0629946316532-1.4858428218j)-((0.00655708230946-0.0807776299406j))*(o-1.)
            arg[(0, 1, 2, 0)]=(0.305197417695+0.681123617975j)*x[0]**o + ((-0.0282739244027-0.544507182168j))*x[0] + ((-0.423395972477-0.301330659123j))*x[1]**o + ((-0.909884528959-0.802085451832j))*x[1] + ((-0.584043534936+0.73996038903j))*x[2]**o + ((0.948021500759-0.927127341135j))*x[2]
            ref[(0, 1, 2, 0)]=(-0.34618952116-0.576983313626j)-((-0.117040348286+0.18662555798j))*(o-1.)
            arg[(0, 1, 2, 1)]=(0.537083278395-0.342156509825j)*x[0]**o + ((0.517130558972+0.759043584056j))*x[0] + ((-0.397819521063+0.500085153957j))*x[1]**o + ((0.913315597427-0.718358328941j))*x[1] + ((0.496064500338-0.538815909854j))*x[2]**o + ((0.767891501024-0.80780787419j))*x[2]
            ref[(0, 1, 2, 1)]=(1.41683295755-0.574004942398j)-((0.105888042945-0.0634812109536j))*(o-1.)
            arg[(0, 2, 0, 0)]=(-0.877195171362+0.704683543171j)*x[0]**o + ((-0.41600330493-0.814388376125j))*x[0] + ((-0.156371290659-0.889303614754j))*x[1]**o + ((-0.136027605172-0.133096583723j))*x[1] + ((-0.11235502166-0.691423480747j))*x[2]**o + ((-0.394441762532-0.863306355405j))*x[2]
            ref[(0, 2, 0, 0)]=(-1.04619707816-1.34341743379j)-((-0.190986913947-0.146007258722j))*(o-1.)
            arg[(0, 2, 0, 1)]=(0.430163519075+0.796372496474j)*x[0]**o + ((-0.244609594275+0.911435966771j))*x[0] + ((0.912157530543+0.725317565549j))*x[1]**o + ((0.965124859193-0.701231078974j))*x[1] + ((0.209196302195-0.643210332683j))*x[2]**o + ((-0.576382443905+0.822818168383j))*x[2]
            ref[(0, 2, 0, 1)]=(0.847825086412+0.955751392761j)-((0.258586225302+0.146413288224j))*(o-1.)
            arg[(0, 2, 1, 0)]=(0.372343284637-0.773776964331j)*x[0]**o + ((0.926819612981-0.596364825808j))*x[0] + ((-0.674727977637-0.170012237399j))*x[1]**o + ((-0.466407213261+0.634875981387j))*x[1] + ((-0.253664576386-0.418491852439j))*x[2]**o + ((-0.237154823402-0.900626938177j))*x[2]
            ref[(0, 2, 1, 0)]=(-0.166395846534-1.11219841838j)-((-0.0926748782309-0.227046842362j))*(o-1.)
            arg[(0, 2, 1, 1)]=(-0.68915721358+0.0410274941304j)*x[0]**o + ((0.939689404416-0.522917910828j))*x[0] + ((-0.604127100057+0.120505763871j))*x[1]**o + ((0.684320787935-0.137027804221j))*x[1] + ((0.291028177096-0.137750415146j))*x[2]**o + ((0.639722233836+0.915106558116j))*x[2]
            ref[(0, 2, 1, 1)]=(0.630738144823+0.139471842961j)-((-0.167042689423+0.00396380714261j))*(o-1.)
            arg[(0, 2, 2, 0)]=(0.150729945344-0.123638100247j)*x[0]**o + ((0.702181022378+0.955115464971j))*x[0] + ((0.892959699284+0.642664321182j))*x[1]**o + ((0.38153225689+0.64149124323j))*x[1] + ((-0.225091476991+0.192088374267j))*x[2]**o + ((0.951690753276-0.24772880174j))*x[2]
            ref[(0, 2, 2, 0)]=(1.42700110009+1.02999625083j)-((0.13643302794+0.1185190992j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.209705595484+0.459470223409j)*x[0]**o + ((0.0482307695376-0.250700471213j))*x[0] + ((-0.760325219373+0.824988100085j))*x[1]**o + ((0.690019887518-0.710009569367j))*x[1] + ((-0.652501439336-0.785439372829j))*x[2]**o + ((-0.282458302753+0.933181067106j))*x[2]
            ref[(0, 2, 2, 1)]=(-0.583369949945+0.235744988595j)-((-0.270422042366+0.0831698251108j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.100230071878-0.650348693374j)*x[0]**o + ((-0.115345512698-0.632590164185j))*x[0] + ((0.273959993358-0.496005173518j))*x[1]**o + ((0.412357989187-0.703716805163j))*x[1] + ((0.911889272175+0.822547129899j))*x[2]**o + ((0.925191769046-0.78587197613j))*x[2]
            ref[(0, 3, 0, 0)]=(1.25414179147-1.22299284124j)-((0.214346556235-0.0539677894988j))*(o-1.)
            arg[(0, 3, 0, 1)]=(0.70421057295-0.484983216731j)*x[0]**o + ((0.0581316432002+0.659605919721j))*x[0] + ((0.766999734828-0.636488096162j))*x[1]**o + ((-0.614926383668+0.581048558883j))*x[1] + ((-0.943585574393-0.828677910189j))*x[2]**o + ((-0.162107095331+0.429960644525j))*x[2]
            ref[(0, 3, 0, 1)]=(-0.0956385512072-0.139767049977j)-((0.0879374555641-0.325024870514j))*(o-1.)
            arg[(0, 3, 1, 0)]=(0.893550963709+0.180452642222j)*x[0]**o + ((0.438183392497-0.143637376763j))*x[0] + ((0.762337084145+0.923248114169j))*x[1]**o + ((0.48850984144+0.96758393531j))*x[1] + ((-0.270795047541-0.694384304949j))*x[2]**o + ((0.516097058548-0.939694512808j))*x[2]
            ref[(0, 3, 1, 0)]=(1.4139416464+0.146784248591j)-((0.230848833385+0.0682194085738j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.838410870357+0.583583148311j)*x[0]**o + ((0.809233660972+0.0896689230325j))*x[0] + ((0.651991556566+0.836200300744j))*x[1]**o + ((0.576466798107+0.629282220888j))*x[1] + ((-0.232441965881+0.211979281843j))*x[2]**o + ((-0.51417258558-0.0585702743124j))*x[2]
            ref[(0, 3, 1, 1)]=(0.226333296913+1.14607180025j)-((-0.0698102132787+0.27196045515j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.303889182271-0.125227164911j)*x[0]**o + ((0.214794577731+0.958440510252j))*x[0] + ((-0.936061103038+0.374499334429j))*x[1]**o + ((-0.329932737218+0.455791888491j))*x[1] + ((0.065136659649-0.188122367119j))*x[2]**o + ((-0.564965305017-0.349926543019j))*x[2]
            ref[(0, 3, 2, 0)]=(-0.623569362811+0.562727829062j)-((-0.094505876853+0.0101916337331j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.605859296223+0.118665295023j)*x[0]**o + ((0.506119942096-0.876155735862j))*x[0] + ((0.00519821024674+0.676651434072j))*x[1]**o + ((-0.272580378234-0.122441360007j))*x[1] + ((-0.769773539377+0.344016657583j))*x[2]**o + ((0.146749246877-0.963245248414j))*x[2]
            ref[(0, 3, 2, 1)]=(-0.495072907307-0.411254478803j)-((-0.228405770892+0.18988889778j))*(o-1.)
            arg[(1, 0, 0, 0)]=(0.00129735345442+0.542713945786j)*x[0]**o + ((-0.734898962388+0.439053017074j))*x[0] + ((0.295155466064+0.991198425469j))*x[1]**o + ((0.661964997732-0.902065649962j))*x[1] + ((0.789148564682-0.0781090888894j))*x[2]**o + ((0.856326809501-0.744478776244j))*x[2]
            ref[(1, 0, 0, 0)]=(0.934497114523+0.124155936617j)-((0.180933564033+0.242633880394j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.200350241542+0.41636894697j)*x[0]**o + ((-0.906296595271-0.746079371448j))*x[0] + ((-0.0854291896539-0.0747542750887j))*x[1]**o + ((-0.521836757434-0.430050245946j))*x[1] + ((-0.568608354256+0.53284597463j))*x[2]**o + ((0.783931690221+0.345998920769j))*x[2]
            ref[(1, 0, 0, 1)]=(-0.749294723968+0.0221649749433j)-((-0.142397964242+0.145743441085j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.928691629786-0.790903727897j)*x[0]**o + ((0.246672200852+0.363490037116j))*x[0] + ((-0.077633428211+0.667790995927j))*x[1]**o + ((0.693193389336-0.628980041849j))*x[1] + ((0.249627422496+0.5330646851j))*x[2]**o + ((0.168703994718+0.83037068658j))*x[2]
            ref[(1, 0, 1, 0)]=(0.175935974702+0.487416317489j)-((-0.126116272584+0.0683253255217j))*(o-1.)
            arg[(1, 0, 1, 1)]=(0.48238364865+0.365351450669j)*x[0]**o + ((0.0780873431249-0.309140558894j))*x[0] + ((0.871389828841+0.892620212115j))*x[1]**o + ((-0.334231174405-0.139360315219j))*x[1] + ((-0.849703385526-0.948247200964j))*x[2]**o + ((0.675436649341+0.384984452221j))*x[2]
            ref[(1, 0, 1, 1)]=(0.461681455013+0.123104019963j)-((0.0840116819942+0.0516207436365j))*(o-1.)
            arg[(1, 0, 2, 0)]=(0.542715960506-0.390203605366j)*x[0]**o + ((-0.161818774901-0.221878601786j))*x[0] + ((0.436282436075+0.486468834807j))*x[1]**o + ((0.497516551819+0.465826412998j))*x[1] + ((0.106342723023-0.451937232017j))*x[2]**o + ((0.342311943174+0.742151419316j))*x[2]
            ref[(1, 0, 2, 0)]=(0.881675419848+0.315213613977j)-((0.180890186601-0.059278667096j))*(o-1.)
            arg[(1, 0, 2, 1)]=(0.0715071963642+0.206063794316j)*x[0]**o + ((-0.553912988896+0.492468858518j))*x[0] + ((-0.331480127354-0.757532298101j))*x[1]**o + ((0.851791116183-0.368714646265j))*x[1] + ((-0.438356383766-0.240565625303j))*x[2]**o + ((0.874665225614+0.366548700289j))*x[2]
            ref[(1, 0, 2, 1)]=(0.237107019072-0.150865608273j)-((-0.116388219126-0.132005688181j))*(o-1.)
            arg[(1, 1, 0, 0)]=(-0.0351717973149+0.612423459401j)*x[0]**o + ((-0.441837106875+0.365286589706j))*x[0] + ((-0.728677809911-0.9599287921j))*x[1]**o + ((-0.0401417081663+0.608361160655j))*x[1] + ((-0.524146506331+0.203656178766j))*x[2]**o + ((-0.0146059169849+0.739145765255j))*x[2]
            ref[(1, 1, 0, 0)]=(-0.892290422792+0.784472180841j)-((-0.214666018926-0.0239748589889j))*(o-1.)
            arg[(1, 1, 0, 1)]=(0.607135964611+0.774488337441j)*x[0]**o + ((0.414379529872-0.446852231794j))*x[0] + ((0.454749961067+0.242367537443j))*x[1]**o + ((0.580122541268-0.670022637973j))*x[1] + ((0.327171491105-0.296208906293j))*x[2]**o + ((-0.243690438145-0.689621731864j))*x[2]
            ref[(1, 1, 0, 1)]=(1.06993452489-0.54292481652j)-((0.231509569464+0.120107828098j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.381368118392+0.252737898704j)*x[0]**o + ((0.496319058505+0.595994780312j))*x[0] + ((0.657909397212-0.75601352245j))*x[1]**o + ((-0.550412830336+0.446112397274j))*x[1] + ((-0.145807527009+0.0669820398193j))*x[2]**o + ((0.0489222112277+0.53209301564j))*x[2]
            ref[(1, 1, 1, 0)]=(0.444149213996+0.56895330465j)-((0.148911664766-0.0727155973212j))*(o-1.)
            arg[(1, 1, 1, 1)]=(-0.499223843302-0.28991627822j)*x[0]**o + ((-0.442028313037-0.0217761566892j))*x[0] + ((0.166650343004-0.903851588304j))*x[1]**o + ((-0.659092123988-0.658900917101j))*x[1] + ((-0.961365861933-0.0472012513876j))*x[2]**o + ((-0.462209182257-0.450009065896j))*x[2]
            ref[(1, 1, 1, 1)]=(-1.42863449076-1.1858276288j)-((-0.215656560372-0.206828186319j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.644383431086+0.155223464719j)*x[0]**o + ((0.816359111202-0.226778138262j))*x[0] + ((0.477649893521+0.152958159753j))*x[1]**o + ((-0.142405686454-0.604953875451j))*x[1] + ((0.622854728495-0.861148309858j))*x[2]**o + ((0.920531418792+0.642694446983j))*x[2]
            ref[(1, 1, 2, 0)]=(1.66968644832-0.371002126058j)-((0.290814675517-0.0921611142309j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.143691269816-0.970021129114j)*x[0]**o + ((0.657667691974+0.768977420477j))*x[0] + ((0.125298363489-0.738869655082j))*x[1]**o + ((-0.998177481488-0.763783742895j))*x[1] + ((0.0636283234956-0.552478661194j))*x[2]**o + ((0.312047090689-0.53553764725j))*x[2]
            ref[(1, 1, 2, 1)]=(0.00838635917199-1.39585670753j)-((0.00753923619488-0.376894907565j))*(o-1.)
            arg[(1, 2, 0, 0)]=(-0.0500260804335+0.476275081799j)*x[0]**o + ((0.407875514631-0.200117740942j))*x[0] + ((-0.900352874566+0.600882756153j))*x[1]**o + ((-0.490530626892-0.771269601403j))*x[1] + ((0.888090256905+0.00155737288782j))*x[2]**o + ((-0.18694658893+0.0186418151358j))*x[2]
            ref[(1, 2, 0, 0)]=(-0.165945199642+0.0629848418151j)-((-0.0103814496824+0.179785868473j))*(o-1.)
            arg[(1, 2, 0, 1)]=(-0.572780001356+0.515062752352j)*x[0]**o + ((-0.227514010301+0.653973499942j))*x[0] + ((0.915831975591-0.866528263867j))*x[1]**o + ((-0.722587449649+0.835530750275j))*x[1] + ((-0.273500881303-0.833551451817j))*x[2]**o + ((0.389249569401+0.296648099463j))*x[2]
            ref[(1, 2, 0, 1)]=(-0.245650398808+0.300567693174j)-((0.0115918488221-0.197502827222j))*(o-1.)
            arg[(1, 2, 1, 0)]=(0.104412506719+0.660870654763j)*x[0]**o + ((-0.974848236821-0.396125729886j))*x[0] + ((-0.249826391625+0.569491603359j))*x[1]**o + ((0.817803980549+0.303619945491j))*x[1] + ((-0.550610832136-0.870489026327j))*x[2]**o + ((-0.00756854796803-0.811135189736j))*x[2]
            ref[(1, 2, 1, 0)]=(-0.430318760641-0.271883871168j)-((-0.116004119507+0.0599788719659j))*(o-1.)
            arg[(1, 2, 1, 1)]=(0.111202265857-0.373535785474j)*x[0]**o + ((0.688777711831-0.656361094219j))*x[0] + ((0.952295353444-0.197983229459j))*x[1]**o + ((0.410956147829-0.391979431401j))*x[1] + ((-0.237094602496-0.0252103058072j))*x[2]**o + ((0.544370416484+0.689883831853j))*x[2]
            ref[(1, 2, 1, 1)]=(1.23525364647-0.477593007253j)-((0.137733836134-0.09945488679j))*(o-1.)
            arg[(1, 2, 2, 0)]=(0.590810393813+0.846381432786j)*x[0]**o + ((-0.83641323602-0.040744035795j))*x[0] + ((-0.442140418583+0.434423839416j))*x[1]**o + ((-0.954301061833-0.63054862857j))*x[1] + ((-0.687463485242+0.699023950531j))*x[2]**o + ((0.71315623601-0.435976017868j))*x[2]
            ref[(1, 2, 2, 0)]=(-0.808175785928+0.43628027025j)-((-0.0897989183354+0.329971537122j))*(o-1.)
            arg[(1, 2, 2, 1)]=(0.258254844153+0.730739526436j)*x[0]**o + ((-0.914294626815+0.659982884721j))*x[0] + ((-0.47073438345+0.343394393068j))*x[1]**o + ((-0.0405833380568+0.245618166994j))*x[1] + ((-0.683995975869+0.603745534075j))*x[2]**o + ((0.479352417559+0.9931926133j))*x[2]
            ref[(1, 2, 2, 1)]=(-0.686000531239+1.7883365593j)-((-0.149412585861+0.279646575596j))*(o-1.)
            arg[(1, 3, 0, 0)]=(0.66857599448-0.733761839483j)*x[0]**o + ((-0.206399323897+0.41181763365j))*x[0] + ((-0.727108040113-0.899592156303j))*x[1]**o + ((0.914797396891-0.422403773037j))*x[1] + ((-0.991489297684+0.697837854493j))*x[2]**o + ((-0.0300952376328-0.531396446589j))*x[2]
            ref[(1, 3, 0, 0)]=(-0.185859253978-0.738749363634j)-((-0.17500355722-0.155919356882j))*(o-1.)
            arg[(1, 3, 0, 1)]=(-0.607596537962+0.374823301874j)*x[0]**o + ((0.000272992598763+0.258083038598j))*x[0] + ((0.309127997264-0.592087534391j))*x[1]**o + ((-0.872084603498+0.612579896679j))*x[1] + ((0.927449465331-0.426275119012j))*x[2]**o + ((0.869239414868+0.409425923515j))*x[2]
            ref[(1, 3, 0, 1)]=(0.313204364301+0.318274753631j)-((0.104830154105-0.107256558588j))*(o-1.)
            arg[(1, 3, 1, 0)]=(-0.107915263152-0.340537784469j)*x[0]**o + ((0.889939154641-0.686127419128j))*x[0] + ((-0.82075927711+0.649439424981j))*x[1]**o + ((0.928623015453-0.0470746771646j))*x[1] + ((0.405391907428-0.986459263825j))*x[2]**o + ((-0.0864921938137+0.597767710608j))*x[2]
            ref[(1, 3, 1, 0)]=(0.604393671723-0.406496004499j)-((-0.087213772139-0.112926270552j))*(o-1.)
            arg[(1, 3, 1, 1)]=(0.602275932353-0.364713945069j)*x[0]**o + ((0.167736263909+0.640738324134j))*x[0] + ((-0.777839001496+0.338023963389j))*x[1]**o + ((0.24315108242-0.613430777503j))*x[1] + ((0.347034727113-0.105518543464j))*x[2]**o + ((-0.520353632971+0.994117953474j))*x[2]
            ref[(1, 3, 1, 1)]=(0.0310026856642+0.44460848748j)-((0.0285786096617-0.0220347541908j))*(o-1.)
            arg[(1, 3, 2, 0)]=(0.359802245086-0.683332746075j)*x[0]**o + ((0.617983892453-0.607688921448j))*x[0] + ((0.209615993708+0.466736615167j))*x[1]**o + ((-0.876026550016-0.529059362016j))*x[1] + ((-0.973701218725+0.285011070427j))*x[2]**o + ((0.862783484721-0.197434107452j))*x[2]
            ref[(1, 3, 2, 0)]=(0.100228923614-0.632883725698j)-((-0.0673804966552+0.0114024899198j))*(o-1.)
            arg[(1, 3, 2, 1)]=(0.546405029025-0.000377692727477j)*x[0]**o + ((0.599075843629-0.266515326501j))*x[0] + ((-0.748550356551+0.380920429595j))*x[1]**o + ((0.639600993756+0.428498590483j))*x[1] + ((0.829097073087-0.778048727841j))*x[2]**o + ((-0.36989907883-0.837303342355j))*x[2]
            ref[(1, 3, 2, 1)]=(0.747864752059-0.536413034674j)-((0.104491957594-0.0662509984957j))*(o-1.)
            arg[(2, 0, 0, 0)]=(-0.609097822689-0.733678141974j)*x[0]**o + ((0.152998254244+0.875975235998j))*x[0] + ((0.613212152109+0.23852130422j))*x[1]**o + ((-0.0471673727763+0.495442980791j))*x[1] + ((0.404970420033-0.370973979725j))*x[2]**o + ((-0.816326428166-0.633685058252j))*x[2]
            ref[(2, 0, 0, 0)]=(-0.150705398622-0.0641988294708j)-((0.0681807915757-0.144355136246j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.936355713126-0.421157131145j)*x[0]**o + ((-0.488746162097+0.831786418801j))*x[0] + ((0.862735617523+0.342206001768j))*x[1]**o + ((-0.0432731025199+0.109118415676j))*x[1] + ((0.336827864377+0.322682940148j))*x[2]**o + ((-0.730984089325+0.0786930111954j))*x[2]
            ref[(2, 0, 0, 1)]=(-0.499897792584+0.631664828222j)-((0.0438679614624+0.040621968462j))*(o-1.)
            arg[(2, 0, 1, 0)]=(-0.158087303852+0.403635708551j)*x[0]**o + ((-0.505737741963+0.324372873011j))*x[0] + ((0.88857673102+0.896883082877j))*x[1]**o + ((0.150116199581+0.870197263166j))*x[1] + ((0.606990658543-0.259065861778j))*x[2]**o + ((-0.981110672591+0.357082694996j))*x[2]
            ref[(2, 0, 1, 0)]=(0.000373935368895+1.29655288041j)-((0.222913347619+0.173575488275j))*(o-1.)
            arg[(2, 0, 1, 1)]=(0.997699142278-0.817739466425j)*x[0]**o + ((0.818909687077+0.300365676665j))*x[0] + ((0.997126449571+0.358868639701j))*x[1]**o + ((-0.718338600715-0.500967908298j))*x[1] + ((0.438218916586+0.0397187702058j))*x[2]**o + ((-0.0973013270218-0.833633053319j))*x[2]
            ref[(2, 0, 1, 1)]=(1.21815713389-0.726693670736j)-((0.405507418073-0.0698586760865j))*(o-1.)
            arg[(2, 0, 2, 0)]=(-0.630952341822+0.855899824647j)*x[0]**o + ((-0.289981416552-0.244343876596j))*x[0] + ((0.964137881522-0.157615004431j))*x[1]**o + ((0.0240686899564+0.935880283479j))*x[1] + ((-0.367828378284-0.392654046259j))*x[2]**o + ((0.58917851303-0.970479634203j))*x[2]
            ref[(2, 0, 2, 0)]=(0.144311473925+0.0133437733183j)-((-0.0057738064306+0.0509384623262j))*(o-1.)
            arg[(2, 0, 2, 1)]=(-0.356631563824-0.37337610954j)*x[0]**o + ((0.530741992017+0.494711266247j))*x[0] + ((0.453190889267-0.601897476636j))*x[1]**o + ((-0.681824186135-0.841968511408j))*x[1] + ((-0.61949369017-0.61096932449j))*x[2]**o + ((0.920167928633+0.767878711447j))*x[2]
            ref[(2, 0, 2, 1)]=(0.123075684894-0.58281072219j)-((-0.0871557274544-0.264373818444j))*(o-1.)
            arg[(2, 1, 0, 0)]=(-0.190549836985+0.289059005235j)*x[0]**o + ((-0.483171572513-0.640854135128j))*x[0] + ((0.783682957944-0.106179040579j))*x[1]**o + ((0.962298317971-0.386850391684j))*x[1] + ((0.246961324475-0.529156259844j))*x[2]**o + ((-0.497293898068+0.675101729933j))*x[2]
            ref[(2, 1, 0, 0)]=(0.410963646413-0.349439546034j)-((0.140015740906-0.0577127158647j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.921893964174+0.669024284603j)*x[0]**o + ((-0.739504033215-0.0149408937163j))*x[0] + ((0.538219654389-0.456410521365j))*x[1]**o + ((-0.240414233485-0.607864028109j))*x[1] + ((-0.657766250198-0.360211398792j))*x[2]**o + ((-0.704097228597-0.800025024499j))*x[2]
            ref[(2, 1, 0, 1)]=(-1.36272802764-0.785213790939j)-((-0.173573426664-0.0245996059256j))*(o-1.)
            arg[(2, 1, 1, 0)]=(-0.966446278707-0.844421775732j)*x[0]**o + ((0.918332452861+0.217153008226j))*x[0] + ((-0.532105494083+0.596443657778j))*x[1]**o + ((-0.212139108423+0.886625494874j))*x[1] + ((0.220653400768+0.199665809311j))*x[2]**o + ((-0.468656640006+0.93751363957j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.520180833795+0.996489917013j)-((-0.212983062004-0.00805205144064j))*(o-1.)
            arg[(2, 1, 1, 1)]=(0.939554096388+0.236323079012j)*x[0]**o + ((-0.44347322516+0.640734415187j))*x[0] + ((0.21477393026-0.607528923753j))*x[1]**o + ((0.625114806269+0.452024078525j))*x[1] + ((-0.331054135173-0.645137444667j))*x[2]**o + ((-0.48393730237+0.601301711273j))*x[2]
            ref[(2, 1, 1, 1)]=(0.260489085107+0.338858457788j)-((0.137212315246-0.169390548235j))*(o-1.)
            arg[(2, 1, 2, 0)]=(0.345644807657+0.391028508223j)*x[0]**o + ((0.939361082034-0.60697138008j))*x[0] + ((-0.180860454026+0.370652030244j))*x[1]**o + ((0.653536392027-0.315191011546j))*x[1] + ((-0.886099044839-0.415949204183j))*x[2]**o + ((-0.612432805321-0.621684355714j))*x[2]
            ref[(2, 1, 2, 0)]=(0.129574988765-0.599057706528j)-((-0.120219115201+0.0576218890473j))*(o-1.)
            arg[(2, 1, 2, 1)]=(0.981863374186-0.232382947695j)*x[0]**o + ((-0.531656215515-0.346249550188j))*x[0] + ((0.201723339561+0.411730205373j))*x[1]**o + ((-0.77131547428+0.266409076749j))*x[1] + ((0.59815412764+0.417491458405j))*x[2]**o + ((0.597483981565-0.0283122883619j))*x[2]
            ref[(2, 1, 2, 1)]=(0.538126566578+0.244342977141j)-((0.296956806898+0.0994731193472j))*(o-1.)
            arg[(2, 2, 0, 0)]=(0.687803729385-0.457416207619j)*x[0]**o + ((0.110759055892-0.620005115798j))*x[0] + ((-0.585355173749-0.749047959714j))*x[1]**o + ((-0.666069238065+0.344480860179j))*x[1] + ((-0.449727045635-0.983064047241j))*x[2]**o + ((0.741267816802-0.238939658479j))*x[2]
            ref[(2, 2, 0, 0)]=(-0.0806604276847-1.35199606434j)-((-0.0578797483331-0.364921369096j))*(o-1.)
            arg[(2, 2, 0, 1)]=(-0.0413769860276+0.494600926838j)*x[0]**o + ((0.395195214729+0.608683626856j))*x[0] + ((-0.774109667705-0.21501979109j))*x[1]**o + ((-0.884634134315+0.420074091532j))*x[1] + ((0.490370465662+0.0640294169441j))*x[2]**o + ((0.0202202993276-0.215806072121j))*x[2]
            ref[(2, 2, 0, 1)]=(-0.397167404165+0.578281099479j)-((-0.0541860313451+0.0572684254487j))*(o-1.)
            arg[(2, 2, 1, 0)]=(-0.425061485762-0.224794831645j)*x[0]**o + ((-0.90615479216-0.96889737025j))*x[0] + ((0.376715512201+0.659692377415j))*x[1]**o + ((0.789040812553-0.335873286917j))*x[1] + ((-0.327397470037+0.191941332748j))*x[2]**o + ((-0.902744305236+0.666271233209j))*x[2]
            ref[(2, 2, 1, 0)]=(-0.697800864221-0.00583027272017j)-((-0.0626239072663+0.10447314642j))*(o-1.)
            arg[(2, 2, 1, 1)]=(0.0480036859294+0.604350716914j)*x[0]**o + ((-0.744804970634-0.113392395439j))*x[0] + ((0.82668022248-0.785391474216j))*x[1]**o + ((-0.853781071593-0.558554778899j))*x[1] + ((0.116136105292-0.763241439444j))*x[2]**o + ((0.899179435864-0.952086968804j))*x[2]
            ref[(2, 2, 1, 1)]=(0.14570670367-1.28415816994j)-((0.16513666895-0.157380366124j))*(o-1.)
            arg[(2, 2, 2, 0)]=(-0.345106898932+0.73275412584j)*x[0]**o + ((0.6928907466+0.267790830833j))*x[0] + ((-0.271356114259-0.971809342889j))*x[1]**o + ((0.589442569344+0.739988177856j))*x[1] + ((-0.548971559662+0.429552299262j))*x[2]**o + ((0.263977644941+0.245102696513j))*x[2]
            ref[(2, 2, 2, 0)]=(0.190438194016+0.721689393707j)-((-0.194239095476+0.0317495137021j))*(o-1.)
            arg[(2, 2, 2, 1)]=(0.98513343793-0.329138723945j)*x[0]**o + ((0.867156913968+0.766182525996j))*x[0] + ((-0.340692386819+0.350728582944j))*x[1]**o + ((-0.0136943229046+0.722961688559j))*x[1] + ((-0.776471188487-0.703746998705j))*x[2]**o + ((0.480454763382-0.633969665017j))*x[2]
            ref[(2, 2, 2, 1)]=(0.600943608535+0.0865087049162j)-((-0.0220050228961-0.113692856618j))*(o-1.)
            arg[(2, 3, 0, 0)]=(0.983040970695+0.300217440676j)*x[0]**o + ((-0.298934447611+0.00760262092767j))*x[0] + ((-0.0504467794547-0.0320515979328j))*x[1]**o + ((0.946769427177-0.114783914066j))*x[1] + ((-0.561284785248+0.946409296233j))*x[2]**o + ((-0.0938831088373+0.610650055765j))*x[2]
            ref[(2, 3, 0, 0)]=(0.46263063836+0.859021950801j)-((0.0618849009987+0.202429189829j))*(o-1.)
            arg[(2, 3, 0, 1)]=(0.549583201191-0.0878009433448j)*x[0]**o + ((0.00990807793096-0.274163959085j))*x[0] + ((-0.0106643203015-0.956039043847j))*x[1]**o + ((0.390872896689+0.650447015873j))*x[1] + ((0.790575332439-0.370000357311j))*x[2]**o + ((-0.799105306019+0.64624921225j))*x[2]
            ref[(2, 3, 0, 1)]=(0.465584940965-0.195654037733j)-((0.221582368888-0.235640057417j))*(o-1.)
            arg[(2, 3, 1, 0)]=(0.231813637816+0.751660604091j)*x[0]**o + ((0.737614101918+0.662313790364j))*x[0] + ((0.242363575223-0.965902175474j))*x[1]**o + ((0.741136083544+0.756658461711j))*x[1] + ((-0.0214348586372+0.628485951051j))*x[2]**o + ((-0.248459940483-0.207684559668j))*x[2]
            ref[(2, 3, 1, 0)]=(0.841516299691+0.812766036037j)-((0.0754570590669+0.0690407299447j))*(o-1.)
            arg[(2, 3, 1, 1)]=(-0.215894325425+0.0677355544697j)*x[0]**o + ((0.204186299671+0.683410858971j))*x[0] + ((0.301403013538+0.251821797795j))*x[1]**o + ((-0.124942759379-0.949529823368j))*x[1] + ((-0.721511360567+0.475572976774j))*x[2]**o + ((-0.266135965064+0.240039612468j))*x[2]
            ref[(2, 3, 1, 1)]=(-0.411447548613+0.384525488555j)-((-0.106000445409+0.132521721506j))*(o-1.)
            arg[(2, 3, 2, 0)]=(0.607106374499-0.339295018706j)*x[0]**o + ((-0.287460006247-0.916239650882j))*x[0] + ((-0.676438611989-0.808252693932j))*x[1]**o + ((-0.0748702793031-0.887531734155j))*x[1] + ((-0.980151745768+0.0123040921761j))*x[2]**o + ((-0.950709540806+0.467435393773j))*x[2]
            ref[(2, 3, 2, 0)]=(-1.18126190481-1.23578980586j)-((-0.17491399721-0.189207270077j))*(o-1.)
            arg[(2, 3, 2, 1)]=(-0.701405047466+0.53594593476j)*x[0]**o + ((-0.0655910724536+0.875016001818j))*x[0] + ((0.635439117224+0.0414776812461j))*x[1]**o + ((0.709892603394+0.0662502739184j))*x[1] + ((0.648485565147-0.996685762097j))*x[2]**o + ((0.899173182081-0.117118773312j))*x[2]
            ref[(2, 3, 2, 1)]=(1.06299717396+0.202442678167j)-((0.0970866058175-0.0698770243485j))*(o-1.)
            arg[(3, 0, 0, 0)]=(-0.135316091434+0.804542948623j)*x[0]**o + ((-0.475722978094+0.978108151962j))*x[0] + ((-0.406518901856-0.325373077221j))*x[1]**o + ((0.684699490745+0.818763040095j))*x[1] + ((0.297627114726+0.590648055713j))*x[2]**o + ((-0.0195776422936-0.364549794232j))*x[2]
            ref[(3, 0, 0, 0)]=(-0.0274045041032+1.25106966247j)-((-0.040701313094+0.178302987853j))*(o-1.)
            arg[(3, 0, 0, 1)]=(-0.0624866800473+0.939581059628j)*x[0]**o + ((-0.194244955313-0.62717895108j))*x[0] + ((0.0281491056385-0.800528499402j))*x[1]**o + ((-0.0631795524683-0.441581011799j))*x[1] + ((0.94055158647+0.796278889421j))*x[2]**o + ((-0.217087715955-0.704527063366j))*x[2]
            ref[(3, 0, 0, 1)]=(0.215850894162-0.418977788299j)-((0.151035668677+0.155888574941j))*(o-1.)
            arg[(3, 0, 1, 0)]=(-0.48361641197+0.0483781650693j)*x[0]**o + ((0.459145918137-0.770552443952j))*x[0] + ((-0.0313654685334+0.364259944744j))*x[1]**o + ((0.165479567313+0.0398604936636j))*x[1] + ((-0.357783820692+0.550423418308j))*x[2]**o + ((0.450729467468+0.349938954812j))*x[2]
            ref[(3, 0, 1, 0)]=(0.101294625861+0.291154266322j)-((-0.145460950199+0.160510254687j))*(o-1.)
            arg[(3, 0, 1, 1)]=(0.484124454608-0.119667273698j)*x[0]**o + ((0.603565268022+0.149971235359j))*x[0] + ((0.505286800158+0.788612654325j))*x[1]**o + ((0.232072481857-0.230219154897j))*x[1] + ((-0.293054329333+0.528702150495j))*x[2]**o + ((0.886580989092-0.95615817022j))*x[2]
            ref[(3, 0, 1, 1)]=(1.2092878322+0.0806207206815j)-((0.116059487572+0.199607921854j))*(o-1.)
            arg[(3, 0, 2, 0)]=(0.89750625773+0.377945152767j)*x[0]**o + ((-0.434239811423-0.599792742567j))*x[0] + ((0.618817606272-0.593279621069j))*x[1]**o + ((0.183669502398+0.91481928554j))*x[1] + ((0.825063471294+0.46879364958j))*x[2]**o + ((-0.393120455803-0.932409642245j))*x[2]
            ref[(3, 0, 2, 0)]=(0.848848285234-0.181961958997j)-((0.390231222549+0.0422431968797j))*(o-1.)
            arg[(3, 0, 2, 1)]=(-0.664750767842-0.479404475815j)*x[0]**o + ((-0.699253785875+0.820436617319j))*x[0] + ((0.794604530444-0.118416915546j))*x[1]**o + ((-0.659611805311-0.192545035916j))*x[1] + ((-0.81862826534+0.512950739042j))*x[2]**o + ((0.808164504415+0.643213601072j))*x[2]
            ref[(3, 0, 2, 1)]=(-0.619737794755+0.593117265078j)-((-0.114795750456-0.0141451087198j))*(o-1.)
            arg[(3, 1, 0, 0)]=(-0.814311187743-0.94413643353j)*x[0]**o + ((0.859576744401+0.530190238607j))*x[0] + ((0.355919733374-0.883684540113j))*x[1]**o + ((0.516603990002+0.378096450514j))*x[1] + ((-0.246248617692-0.824928857445j))*x[2]**o + ((0.322153517909-0.0917693367368j))*x[2]
            ref[(3, 1, 0, 0)]=(0.496847090125-0.918116239353j)-((-0.11744001201-0.442124971848j))*(o-1.)
            arg[(3, 1, 0, 1)]=(0.967420609926+0.451468628165j)*x[0]**o + ((-0.283602015225-0.876267146398j))*x[0] + ((0.434595044988+0.508055995866j))*x[1]**o + ((-0.218803143427+0.927048007782j))*x[1] + ((0.212408436013-0.297550464441j))*x[2]**o + ((0.98169077575-0.16160294822j))*x[2]
            ref[(3, 1, 0, 1)]=(1.04685485401+0.275576036377j)-((0.269070681821+0.110329026598j))*(o-1.)
            arg[(3, 1, 1, 0)]=(0.602012361037-0.0459976127785j)*x[0]**o + ((-0.491029952389-0.177933020193j))*x[0] + ((-0.871680706998+0.102397227628j))*x[1]**o + ((-0.177555922328-0.503809446537j))*x[1] + ((-0.855985305114-0.727331193085j))*x[2]**o + ((-0.408354683861-0.177092377314j))*x[2]
            ref[(3, 1, 1, 0)]=(-1.10129710483-0.76488321114j)-((-0.187608941846-0.111821929706j))*(o-1.)
            arg[(3, 1, 1, 1)]=(-0.466029425105+0.657368970361j)*x[0]**o + ((-0.117083374178-0.67541464399j))*x[0] + ((0.483811445016-0.67792839467j))*x[1]**o + ((0.300271068471-0.607537676994j))*x[1] + ((0.0747020124449-0.645344538614j))*x[2]**o + ((-0.920821939271+0.980589577819j))*x[2]
            ref[(3, 1, 1, 1)]=(-0.322575106312-0.484133353044j)-((0.0154140053926-0.11098399382j))*(o-1.)
            arg[(3, 1, 2, 0)]=(-0.207624895307+0.119824983008j)*x[0]**o + ((-0.840494962731+0.788789306211j))*x[0] + ((0.145582599665+0.222322035017j))*x[1]**o + ((0.738717772135-0.551649612371j))*x[1] + ((-0.588133128864-0.768572745772j))*x[2]**o + ((-0.245855655346+0.330415213237j))*x[2]
            ref[(3, 1, 2, 0)]=(-0.498904135224+0.0705645896647j)-((-0.108362570751-0.0710709546246j))*(o-1.)
            arg[(3, 1, 2, 1)]=(0.944195426016-0.33728603884j)*x[0]**o + ((0.260102664501-0.825906486491j))*x[0] + ((-0.937425270184+0.115907208172j))*x[1]**o + ((-0.526902615144-0.619441741308j))*x[1] + ((-0.312482038628+0.33903945359j))*x[2]**o + ((-0.00767744932908-0.503620517177j))*x[2]
            ref[(3, 1, 2, 1)]=(-0.290094641384-0.915654061027j)-((-0.050951980466+0.0196101038204j))*(o-1.)
            arg[(3, 2, 0, 0)]=(-0.683700746794-0.818531044003j)*x[0]**o + ((0.584174034494+0.149774118566j))*x[0] + ((-0.643198814952+0.3585866366j))*x[1]**o + ((0.332681397946-0.756444176229j))*x[1] + ((0.953850002959+0.0590094800493j))*x[2]**o + ((0.748834441025+0.559863145158j))*x[2]
            ref[(3, 2, 0, 0)]=(0.646320157339-0.223870919929j)-((-0.0621749264644-0.0668224878922j))*(o-1.)
            arg[(3, 2, 0, 1)]=(-0.522537396377+0.378978292548j)*x[0]**o + ((-0.31925226023-0.43447603556j))*x[0] + ((0.502614108814-0.645006047835j))*x[1]**o + ((0.482900617497-0.0822845883746j))*x[1] + ((0.422881211505-0.937211489235j))*x[2]**o + ((0.0834827128639-0.381697175437j))*x[2]
            ref[(3, 2, 0, 1)]=(0.325044497036-1.05084852195j)-((0.0671596539903-0.200539874087j))*(o-1.)
            arg[(3, 2, 1, 0)]=(-0.832622478046+0.0554405110553j)*x[0]**o + ((0.104216107371-0.932641850561j))*x[0] + ((0.505210565256-0.0564240650361j))*x[1]**o + ((0.966061882829+0.903308950917j))*x[1] + ((0.0527367780744+0.662017458344j))*x[2]**o + ((0.410907806698-0.841367045488j))*x[2]
            ref[(3, 2, 1, 0)]=(0.603255331091-0.104833020385j)-((-0.0457791891192+0.110172317394j))*(o-1.)
            arg[(3, 2, 1, 1)]=(-0.218169153298-0.0104750236721j)*x[0]**o + ((0.0288859597959+0.19454708215j))*x[0] + ((-0.407303598375+0.173987936177j))*x[1]**o + ((-0.743277463072+0.466995342765j))*x[1] + ((-0.779667012001-0.308968775833j))*x[2]**o + ((-0.416643098315-0.623850079768j))*x[2]
            ref[(3, 2, 1, 1)]=(-1.26808718263-0.0538817590905j)-((-0.234189960612-0.0242426438879j))*(o-1.)
            arg[(3, 2, 2, 0)]=(-0.612134982705+0.949046661482j)*x[0]**o + ((0.219517068097+0.263284886798j))*x[0] + ((0.148949360965+0.413461106198j))*x[1]**o + ((0.790030223506-0.637368314679j))*x[1] + ((0.206712940635-0.0432795126816j))*x[2]**o + ((-0.688009507552-0.0269518047904j))*x[2]
            ref[(3, 2, 2, 0)]=(0.0325325514731+0.459096511164j)-((-0.0427454468507+0.219871375833j))*(o-1.)
            arg[(3, 2, 2, 1)]=(-0.0921293701793-0.0172140762487j)*x[0]**o + ((0.409356365271-0.229154815929j))*x[0] + ((0.664564935169-0.430238841888j))*x[1]**o + ((0.163140614632+0.872814981517j))*x[1] + ((0.456379754881+0.955303725946j))*x[2]**o + ((0.716288279104+0.188825149541j))*x[2]
            ref[(3, 2, 2, 1)]=(1.15880028944+0.670168061469j)-((0.171469219979+0.0846418013015j))*(o-1.)
            arg[(3, 3, 0, 0)]=(0.0152571365348-0.297823151255j)*x[0]**o + ((0.801123196975-0.808828855579j))*x[0] + ((-0.823896301327+0.67043007681j))*x[1]**o + ((-0.378529400195+0.762577264519j))*x[1] + ((-0.921380100796-0.745732320872j))*x[2]**o + ((-0.589358198498+0.025199647146j))*x[2]
            ref[(3, 3, 0, 0)]=(-0.948391833653-0.197088669615j)-((-0.288336544265-0.062187565886j))*(o-1.)
            arg[(3, 3, 0, 1)]=(-0.0781379264266+0.496133321909j)*x[0]**o + ((0.468244119848-0.311276682126j))*x[0] + ((-0.975472441881-0.723806510799j))*x[1]**o + ((-0.0419640918778-0.786409492872j))*x[1] + ((0.995376177305+0.300116351605j))*x[2]**o + ((0.465991950861+0.400373970603j))*x[2]
            ref[(3, 3, 0, 1)]=(0.417018893914-0.31243452084j)-((-0.00970569850043+0.0120738604524j))*(o-1.)
            arg[(3, 3, 1, 0)]=(-0.689367986746+0.274761048379j)*x[0]**o + ((-0.609795463642-0.97177491702j))*x[0] + ((-0.419216705209+0.811944722825j))*x[1]**o + ((0.121914979777+0.357358368641j))*x[1] + ((0.386995488404-0.724859913972j))*x[2]**o + ((0.462939873322-0.984035622408j))*x[2]
            ref[(3, 3, 1, 0)]=(-0.373264907047-0.618303156777j)-((-0.120264867258+0.0603076428721j))*(o-1.)
            arg[(3, 3, 1, 1)]=(-0.136372783386+0.0575950163549j)*x[0]**o + ((-0.415168251798-0.297916730163j))*x[0] + ((-0.706711675042+0.293700503418j))*x[1]**o + ((0.51746198816-0.00513120423364j))*x[1] + ((0.888279483945-0.912873762646j))*x[2]**o + ((-0.469563042538+0.226192745578j))*x[2]
            ref[(3, 3, 1, 1)]=(-0.161037140329-0.319216715846j)-((0.00753250425282-0.0935963738121j))*(o-1.)
            arg[(3, 3, 2, 0)]=(-0.138664435454+0.908311966718j)*x[0]**o + ((-0.137138657624-0.967040034942j))*x[0] + ((-0.102763875402+0.00554819445598j))*x[1]**o + ((-0.189131193434+0.709751919229j))*x[1] + ((-0.886976390765+0.469522570212j))*x[2]**o + ((0.283383263772-0.225768392976j))*x[2]
            ref[(3, 3, 2, 0)]=(-0.585645644454+0.450163111349j)-((-0.18806745027+0.230563788564j))*(o-1.)
            arg[(3, 3, 2, 1)]=(-0.492213984959-0.561203370508j)*x[0]**o + ((0.20377073787+0.95936590823j))*x[0] + ((0.65027865881+0.905710834388j))*x[1]**o + ((-0.29281873462-0.414574161211j))*x[1] + ((0.779999997531+0.913192800594j))*x[2]**o + ((-0.00610224078864+0.714828299621j))*x[2]
            ref[(3, 3, 2, 1)]=(0.421457216922+1.25866015556j)-((0.156344111897+0.209616710746j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 4, 3, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ContinuousFunction_rank0(self):
        """
        tests integral of rank 0 Data on ReducedFunction

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.667980480619+0.788355050374j)*x[0]**o + ((0.819098724292-0.548707975564j))*x[0] + ((-0.0824865927456-0.135586286295j))*x[1]**o + ((-0.38614692608+0.802802725652j))*x[1]
            ref=(0.509222843042+0.453431757084j)-((0.0975823146456+0.108794794013j))*(o-1.)
        else:
            arg=(-0.26210602865-0.114152078732j)*x[0]**o + ((0.698454530497-0.495996007042j))*x[0] + ((0.70784642022+0.14223837232j))*x[1]**o + ((0.700445814524-0.783947918289j))*x[1] + ((0.60641812845+0.241185719504j))*x[2]**o + ((0.381721404137-0.28059217345j))*x[2]
            ref=(1.41639013459-0.645632042845j)-((0.175359753337+0.0448786688487j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ContinuousFunction_rank1(self):
        """
        tests integral of rank 1 Data on ReducedFunction

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(0.119364819154+0.312739074173j)*x[0]**o + ((-0.787745091554-0.969247847104j))*x[0] + ((-0.515098771724-0.418650592607j))*x[1]**o + ((0.105153259393-0.937581380647j))*x[1]
            ref[(0,)]=(-0.539162892365-1.00637037309j)-((-0.0659556587617-0.017651919739j))*(o-1.)
            arg[(1,)]=(0.132491641793+0.83598235478j)*x[0]**o + ((-0.672318596905+0.363239856526j))*x[0] + ((0.438200620135+0.378660075861j))*x[1]**o + ((0.511513106793+0.14033023043j))*x[1]
            ref[(1,)]=(0.204943385907+0.859106258799j)-((0.0951153769879+0.202440405107j))*(o-1.)
            arg[(2,)]=(-0.308751829628-0.673315531023j)*x[0]**o + ((0.650286832255-0.526085105642j))*x[0] + ((0.727002358908-0.278758436305j))*x[1]**o + ((0.672786091003+0.858116633248j))*x[1]
            ref[(2,)]=(0.870661726269-0.310021219861j)-((0.0697084215466-0.158678994555j))*(o-1.)
            arg[(3,)]=(0.933035260212+0.193280997265j)*x[0]**o + ((0.420898925908+0.532075485182j))*x[0] + ((-0.62827547508+0.63322493947j))*x[1]**o + ((-0.185065271342-0.66220554471j))*x[1]
            ref[(3,)]=(0.27029671985+0.348187938604j)-((0.0507932975221+0.137750989456j))*(o-1.)
        else:
            arg[(0,)]=(0.10182386038+0.421033414853j)*x[0]**o + ((-0.896547429848-0.659998577847j))*x[0] + ((0.621310100054-0.144664879017j))*x[1]**o + ((0.588408541572-0.213394583282j))*x[1] + ((0.464796754165-0.56526873526j))*x[2]**o + ((-0.86862640055+0.0193436536864j))*x[2]
            ref[(0,)]=(0.00558271288604-0.571474853433j)-((0.197988452433-0.0481500332373j))*(o-1.)
            arg[(1,)]=(0.637081059302+0.982698134504j)*x[0]**o + ((-0.470313227635+0.723707103348j))*x[0] + ((0.198621456201-0.711348265965j))*x[1]**o + ((0.235451766817+0.702672040455j))*x[1] + ((-0.0850651392494+0.753696437481j))*x[2]**o + ((-0.139751903352-0.160762122253j))*x[2]
            ref[(1,)]=(0.188012006042+1.14533166378j)-((0.125106229376+0.170841051003j))*(o-1.)
            arg[(2,)]=(0.83532911754-0.319922082013j)*x[0]**o + ((0.128535963055-0.834445529281j))*x[0] + ((0.037070470959+0.66940636324j))*x[1]**o + ((0.773289960066+0.500938220161j))*x[1] + ((-0.413284185341+0.380511121952j))*x[2]**o + ((0.797062133786-0.613828194847j))*x[2]
            ref[(2,)]=(1.07900173003-0.108670050394j)-((0.0765192338598+0.12166590053j))*(o-1.)
            arg[(3,)]=(0.975839917986+0.939071595593j)*x[0]**o + ((0.13662402118-0.831052551014j))*x[0] + ((-0.0447898745373+0.505895170194j))*x[1]**o + ((-0.780927429436+0.828233611944j))*x[1] + ((-0.800206762303+0.683651190752j))*x[2]**o + ((0.300828126837+0.268068593037j))*x[2]
            ref[(3,)]=(-0.106316000136+1.19693380525j)-((0.0218072135243+0.354769659423j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ContinuousFunction_rank2(self):
        """
        tests integral of rank 2 Data on ReducedFunction

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 2),w)
        ref=numpy.zeros((3, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.936769615504-0.772200025548j)*x[0]**o + ((0.545741965893-0.577442619558j))*x[0] + ((0.610334408287+0.587078498993j))*x[1]**o + ((0.735211971619+0.901272445244j))*x[1]
            ref[(0, 0)]=(1.41402898065+0.069354149566j)-((0.257850670632-0.0308535877591j))*(o-1.)
            arg[(0, 1)]=(-0.874083833261+0.519560533893j)*x[0]**o + ((-0.825913634413+0.282187084474j))*x[0] + ((0.551424169472+0.10899379677j))*x[1]**o + ((0.20498453764+0.209185862103j))*x[1]
            ref[(0, 1)]=(-0.471794380281+0.55996363862j)-((-0.0537766106316+0.10475905511j))*(o-1.)
            arg[(1, 0)]=(-0.845364582619+0.959671573642j)*x[0]**o + ((0.164623162565-0.461666790723j))*x[0] + ((-0.164663119914+0.2258228069j))*x[1]**o + ((-0.397736769071-0.181210570252j))*x[1]
            ref[(1, 0)]=(-0.62157065452+0.271308509784j)-((-0.168337950422+0.197582396757j))*(o-1.)
            arg[(1, 1)]=(-0.896604393271+0.516423995328j)*x[0]**o + ((-0.707167664771-0.0236810056882j))*x[0] + ((0.494795800219+0.481239240869j))*x[1]**o + ((0.700236941921+0.892823736758j))*x[1]
            ref[(1, 1)]=(-0.204369657951+0.933402983633j)-((-0.0669680988421+0.166277206033j))*(o-1.)
            arg[(2, 0)]=(0.927331163265-0.0709931965525j)*x[0]**o + ((0.836889190538+0.114988459921j))*x[0] + ((-0.686311358894+0.249795399221j))*x[1]**o + ((0.698652072114-0.749048037502j))*x[1]
            ref[(2, 0)]=(0.888280533512-0.227628687456j)-((0.0401699673951+0.0298003671114j))*(o-1.)
            arg[(2, 1)]=(-0.727629756668-0.671804378868j)*x[0]**o + ((0.668843112954-0.125316552806j))*x[0] + ((0.943278058264-0.746370062681j))*x[1]**o + ((-0.922543530574-0.611786400297j))*x[1]
            ref[(2, 1)]=(-0.019026058012-1.07763869733j)-((0.0359413835993-0.236362406925j))*(o-1.)
        else:
            arg[(0, 0)]=(-0.730758020938+0.697710923566j)*x[0]**o + ((-0.120577249163-0.982398952505j))*x[0] + ((-0.307690675389-0.15130528168j))*x[1]**o + ((0.707561575221-0.845552577282j))*x[1] + ((-0.969575574358-0.819051397866j))*x[2]**o + ((0.764529611317+0.433664425877j))*x[2]
            ref[(0, 0)]=(-0.328255166654-0.833466429946j)-((-0.334670711781-0.0454409593301j))*(o-1.)
            arg[(0, 1)]=(-0.155693529826-0.247550596631j)*x[0]**o + ((-0.907357036068-0.256576435415j))*x[0] + ((0.509794576549+0.607048892556j))*x[1]**o + ((-0.916109738131+0.657695517516j))*x[1] + ((0.214962530011+0.553940422025j))*x[2]**o + ((0.787012268191+0.324055315167j))*x[2]
            ref[(0, 1)]=(-0.233695464637+0.819306557609j)-((0.0948439294556+0.152239786325j))*(o-1.)
            arg[(1, 0)]=(0.850791853199+0.368901818279j)*x[0]**o + ((0.9891863569+0.263087052011j))*x[0] + ((0.514588721227-0.430534805631j))*x[1]**o + ((-0.938039016413-0.0879620806141j))*x[1] + ((0.146697126364-0.0704857997597j))*x[2]**o + ((0.902427956784-0.125564964783j))*x[2]
            ref[(1, 0)]=(1.23282649903-0.0412793902486j)-((0.252012950132-0.0220197978519j))*(o-1.)
            arg[(1, 1)]=(-0.0673152799673+0.634199316978j)*x[0]**o + ((0.50669351162-0.388426226307j))*x[0] + ((0.888652255789+0.757525891357j))*x[1]**o + ((-0.365874640531-0.586068411811j))*x[1] + ((-0.10856124162+0.337485121341j))*x[2]**o + ((-0.396520420961+0.716075166851j))*x[2]
            ref[(1, 1)]=(0.228537092165+0.735395429204j)-((0.1187959557+0.288201721613j))*(o-1.)
            arg[(2, 0)]=(0.63752424159-0.970022590811j)*x[0]**o + ((-0.916278102902+0.241628128234j))*x[0] + ((0.807871180004-0.466853528056j))*x[1]**o + ((0.913764187877+0.0115888406469j))*x[1] + ((0.809316243508+0.851311718922j))*x[2]**o + ((-0.166997878474+0.940897578729j))*x[2]
            ref[(2, 0)]=(1.0425999358+0.304275073832j)-((0.375785277517-0.0975940666576j))*(o-1.)
            arg[(2, 1)]=(0.687627655176+0.705872916791j)*x[0]**o + ((0.122810710278+0.394788214747j))*x[0] + ((-0.67350479909+0.0423469693381j))*x[1]**o + ((0.490759109994+0.166726409845j))*x[1] + ((-0.0810476839007-0.716541344204j))*x[2]**o + ((-0.902576417768-0.945474516328j))*x[2]
            ref[(2, 1)]=(-0.177965712655-0.176140674905j)-((-0.0111541379691+0.00527975698766j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ContinuousFunction_rank3(self):
        """
        tests integral of rank 3 Data on ReducedFunction

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 4),w)
        ref=numpy.zeros((3, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.814129451274+0.0528885520992j)*x[0]**o + ((0.976229568838-0.462337754907j))*x[0] + ((0.105960671224+0.739152593638j))*x[1]**o + ((-0.550557671315+0.0353323102142j))*x[1]
            ref[(0, 0, 0)]=(-0.141248441263+0.182517850523j)-((-0.118028130008+0.132006857623j))*(o-1.)
            arg[(0, 0, 1)]=(-0.582275228588+0.37543587014j)*x[0]**o + ((0.625986750293+0.817213490774j))*x[0] + ((-0.207835691877-0.0796851644502j))*x[1]**o + ((-0.604870839541-0.497079287223j))*x[1]
            ref[(0, 0, 1)]=(-0.384497504857+0.30794245462j)-((-0.131685153411+0.0492917842816j))*(o-1.)
            arg[(0, 0, 2)]=(0.201144433578-0.487045495184j)*x[0]**o + ((0.384821598389-0.69336136553j))*x[0] + ((0.725128606972-0.548174476527j))*x[1]**o + ((0.655546077206-0.833895333788j))*x[1]
            ref[(0, 0, 2)]=(0.983320358073-1.28123833551j)-((0.154378840092-0.172536661952j))*(o-1.)
            arg[(0, 0, 3)]=(0.785092573045+0.674549576188j)*x[0]**o + ((0.128089109942-0.0994922914579j))*x[0] + ((-0.433150205044-0.260725711234j))*x[1]**o + ((-0.838434834864+0.141675970117j))*x[1]
            ref[(0, 0, 3)]=(-0.17920167846+0.228003771807j)-((0.0586570613335+0.0689706441591j))*(o-1.)
            arg[(0, 1, 0)]=(-0.00492990656417-0.982065938034j)*x[0]**o + ((-0.822336655088-0.0178234712945j))*x[0] + ((0.462564102276+0.0916600167886j))*x[1]**o + ((0.829216232828+0.72242177189j))*x[1]
            ref[(0, 1, 0)]=(0.232256886726-0.0929038103248j)-((0.076272365952-0.148400986874j))*(o-1.)
            arg[(0, 1, 1)]=(-0.298570015554+0.513998561119j)*x[0]**o + ((0.0326540114776-0.593016350405j))*x[0] + ((-0.281476379912-0.828369093821j))*x[1]**o + ((-0.588022881411-0.104920056964j))*x[1]
            ref[(0, 1, 1)]=(-0.5677076327-0.506153470036j)-((-0.0966743992443-0.0523950887837j))*(o-1.)
            arg[(0, 1, 2)]=(-0.14948724446-0.936232192024j)*x[0]**o + ((-0.915554845991-0.501096756751j))*x[0] + ((-0.172477431912+0.69670433329j))*x[1]**o + ((0.107655777026+0.673598285671j))*x[1]
            ref[(0, 1, 2)]=(-0.564931872669-0.0335131649065j)-((-0.0536607793954-0.039921309789j))*(o-1.)
            arg[(0, 1, 3)]=(0.298986435146-0.94233633462j)*x[0]**o + ((0.7452614405-0.217537208909j))*x[0] + ((-0.579846281636-0.790155842847j))*x[1]**o + ((0.0142277768417-0.0955700276615j))*x[1]
            ref[(0, 1, 3)]=(0.239314685426-1.02279970702j)-((-0.0468099744149-0.288748696245j))*(o-1.)
            arg[(0, 2, 0)]=(0.882617423113+0.311370837286j)*x[0]**o + ((-0.785233167361-0.475608400768j))*x[0] + ((-0.118200588618+0.964458242884j))*x[1]**o + ((0.445531320884+0.600559952024j))*x[1]
            ref[(0, 2, 0)]=(0.212357494009+0.700390315713j)-((0.127402805749+0.212638180028j))*(o-1.)
            arg[(0, 2, 1)]=(-0.434606432131+0.506709193268j)*x[0]**o + ((0.771197377577+0.576473477061j))*x[0] + ((0.114166172065+0.160630219049j))*x[1]**o + ((0.365835595745+0.499277007634j))*x[1]
            ref[(0, 2, 1)]=(0.408296356628+0.871544948506j)-((-0.053406710011+0.111223235386j))*(o-1.)
            arg[(0, 2, 2)]=(-0.50774036759-0.11112209413j)*x[0]**o + ((0.508835654444+0.186900701354j))*x[0] + ((-0.247110027182-0.0918524007457j))*x[1]**o + ((-0.701898863491+0.834294522017j))*x[1]
            ref[(0, 2, 2)]=(-0.47395680191+0.409110364247j)-((-0.125808399129-0.0338290824794j))*(o-1.)
            arg[(0, 2, 3)]=(-0.798302279286-0.831212954096j)*x[0]**o + ((0.549752993841+0.604729190806j))*x[0] + ((0.272114721078-0.773627236008j))*x[1]**o + ((-0.34980521684-0.639700854349j))*x[1]
            ref[(0, 2, 3)]=(-0.163119890603-0.819905926823j)-((-0.087697926368-0.267473365017j))*(o-1.)
            arg[(0, 3, 0)]=(0.613098207465+0.696130558406j)*x[0]**o + ((-0.0316622312391-0.86828661813j))*x[0] + ((0.131439013594-0.994432148221j))*x[1]**o + ((0.425677069347+0.645790955755j))*x[1]
            ref[(0, 3, 0)]=(0.569276029583-0.260398626095j)-((0.124089536843-0.0497169316359j))*(o-1.)
            arg[(0, 3, 1)]=(0.322925992786-0.505702805021j)*x[0]**o + ((-0.532102239357-0.335827838735j))*x[0] + ((-0.466849305371-0.928025927823j))*x[1]**o + ((-0.915708426453+0.276057381598j))*x[1]
            ref[(0, 3, 1)]=(-0.795866989198-0.746749594991j)-((-0.0239872187642-0.238954788807j))*(o-1.)
            arg[(0, 3, 2)]=(-0.372583156579-0.196942683116j)*x[0]**o + ((-0.301142360963-0.514411131907j))*x[0] + ((0.786315954514+0.241444213723j))*x[1]**o + ((-0.725514195104-0.821071341796j))*x[1]
            ref[(0, 3, 2)]=(-0.306461879066-0.645490471548j)-((0.0689554663224+0.00741692176796j))*(o-1.)
            arg[(0, 3, 3)]=(0.968307372624+0.944537208766j)*x[0]**o + ((-0.597490603064+0.707383607183j))*x[0] + ((-0.969674151182+0.784103440453j))*x[1]**o + ((-0.508822960786-0.189417764113j))*x[1]
            ref[(0, 3, 3)]=(-0.553840171204+1.12330324614j)-((-0.000227796426417+0.28810677487j))*(o-1.)
            arg[(1, 0, 0)]=(-0.173253690359+0.105005207479j)*x[0]**o + ((-0.454893228403-0.426365260183j))*x[0] + ((-0.632521190488+0.478106718577j))*x[1]**o + ((-0.674355540928+0.809543590638j))*x[1]
            ref[(1, 0, 0)]=(-0.967511825089+0.483145128256j)-((-0.134295813474+0.0971853210095j))*(o-1.)
            arg[(1, 0, 1)]=(-0.269453367135-0.0928584847736j)*x[0]**o + ((-0.663883280419+0.879019225764j))*x[0] + ((-0.996323668939+0.446252798808j))*x[1]**o + ((0.745160208529+0.235050417565j))*x[1]
            ref[(1, 0, 1)]=(-0.592250053981+0.733731978681j)-((-0.210962839346+0.0588990523391j))*(o-1.)
            arg[(1, 0, 2)]=(-0.441247485594+0.792344871883j)*x[0]**o + ((-0.106363229734+0.163253156484j))*x[0] + ((0.621849342776-0.324265369367j))*x[1]**o + ((-0.508259149454+0.865614243775j))*x[1]
            ref[(1, 0, 2)]=(-0.217010261003+0.748473451388j)-((0.0301003095302+0.0780132504194j))*(o-1.)
            arg[(1, 0, 3)]=(-0.991162330567-0.857513932825j)*x[0]**o + ((0.811109897594-0.8904305823j))*x[0] + ((-0.430018976242+0.999000208804j))*x[1]**o + ((0.422455795511+0.191395687214j))*x[1]
            ref[(1, 0, 3)]=(-0.093807806852-0.278774309554j)-((-0.236863551135+0.0235810459965j))*(o-1.)
            arg[(1, 1, 0)]=(0.597775436098-0.0150597007218j)*x[0]**o + ((0.496055936504+0.0894558828446j))*x[0] + ((0.0693948526017-0.439347186744j))*x[1]**o + ((-0.837099310758-0.238562687757j))*x[1]
            ref[(1, 1, 0)]=(0.163063457223-0.301756846189j)-((0.111195048117-0.0757344812443j))*(o-1.)
            arg[(1, 1, 1)]=(-0.3937388261+0.60487524737j)*x[0]**o + ((-0.35989845404-0.044631766103j))*x[0] + ((-0.887938803079-0.164023049696j))*x[1]**o + ((-0.704082729-0.979822303688j))*x[1]
            ref[(1, 1, 1)]=(-1.17282940611-0.291800936058j)-((-0.213612938196+0.0734753662791j))*(o-1.)
            arg[(1, 1, 2)]=(0.297206343986+0.0913465632946j)*x[0]**o + ((0.681067747704-0.911054259308j))*x[0] + ((-0.382703657212+0.6717742573j))*x[1]**o + ((0.686699315123-0.702062819549j))*x[1]
            ref[(1, 1, 2)]=(0.641134874801-0.424998129131j)-((-0.0142495522042+0.127186803432j))*(o-1.)
            arg[(1, 1, 3)]=(0.633177753819-0.27390382258j)*x[0]**o + ((-0.765796517958-0.406213420742j))*x[0] + ((-0.463890596726-0.707240447805j))*x[1]**o + ((0.62849107426+0.455710181407j))*x[1]
            ref[(1, 1, 3)]=(0.0159908566974-0.46582375486j)-((0.0282145261821-0.163524045064j))*(o-1.)
            arg[(1, 2, 0)]=(0.630254476991-0.352427441707j)*x[0]**o + ((0.695772969753-0.17076768343j))*x[0] + ((-0.897227368747-0.384675081243j))*x[1]**o + ((0.031504082784-0.419044797003j))*x[1]
            ref[(1, 2, 0)]=(0.230152080391-0.663457501692j)-((-0.0444954819592-0.122850420492j))*(o-1.)
            arg[(1, 2, 1)]=(0.157634067316-0.510276111639j)*x[0]**o + ((-0.675274107654+0.624938894279j))*x[0] + ((-0.869297786682+0.591061297926j))*x[1]**o + ((-0.139145273862-0.0133161516394j))*x[1]
            ref[(1, 2, 1)]=(-0.763041550441+0.346203964463j)-((-0.118610619894+0.0134641977145j))*(o-1.)
            arg[(1, 2, 2)]=(-0.445240456832+0.611263933774j)*x[0]**o + ((-0.454031885272-0.602117760589j))*x[0] + ((0.507284930023-0.930970224224j))*x[1]**o + ((0.0153319631387+0.393721194314j))*x[1]
            ref[(1, 2, 2)]=(-0.188327724471-0.264051428363j)-((0.0103407455318-0.0532843817417j))*(o-1.)
            arg[(1, 2, 3)]=(-0.405745647344-0.849310565197j)*x[0]**o + ((0.98036427648+0.721360142821j))*x[0] + ((0.460267237733-0.208155197358j))*x[1]**o + ((-0.168483785625+0.745049382822j))*x[1]
            ref[(1, 2, 3)]=(0.433201040622+0.204471881544j)-((0.00908693173147-0.176244293759j))*(o-1.)
            arg[(1, 3, 0)]=(0.0363198004738-0.26987613514j)*x[0]**o + ((-0.292679886672-0.875140111238j))*x[0] + ((0.116400058595+0.823476787252j))*x[1]**o + ((-0.982825048798-0.664212966442j))*x[1]
            ref[(1, 3, 0)]=(-0.5613925382-0.492876212784j)-((0.0254533098448+0.0922667753519j))*(o-1.)
            arg[(1, 3, 1)]=(0.599948819208-0.631780805357j)*x[0]**o + ((0.945886654962-0.156216028261j))*x[0] + ((-0.0446668024243-0.728967307966j))*x[1]**o + ((0.288214842703-0.844145817714j))*x[1]
            ref[(1, 3, 1)]=(0.894691757225-1.18055497965j)-((0.0925470027973-0.226791352221j))*(o-1.)
            arg[(1, 3, 2)]=(-0.9418738559+0.770781210194j)*x[0]**o + ((-0.108824175468+0.175430964595j))*x[0] + ((-0.216587689563+0.292450076567j))*x[1]**o + ((0.478959631233-0.454164590007j))*x[1]
            ref[(1, 3, 2)]=(-0.394163044849+0.392248830675j)-((-0.193076924244+0.17720521446j))*(o-1.)
            arg[(1, 3, 3)]=(0.0192963364751+0.765977614265j)*x[0]**o + ((-0.00441605182377-0.629116175291j))*x[0] + ((-0.96943401943-0.869513838619j))*x[1]**o + ((-0.909300690873-0.769094190656j))*x[1]
            ref[(1, 3, 3)]=(-0.931927212826-0.750873295151j)-((-0.158356280492-0.0172560373924j))*(o-1.)
            arg[(2, 0, 0)]=(-0.0252554702868+0.459973783693j)*x[0]**o + ((0.688900091182+0.175364071561j))*x[0] + ((0.564382079534+0.0846686431482j))*x[1]**o + ((0.660595997621+0.818128082802j))*x[1]
            ref[(2, 0, 0)]=(0.944311349025+0.769067290602j)-((0.0898544348745+0.0907737378068j))*(o-1.)
            arg[(2, 0, 1)]=(-0.696240182972-0.703934307642j)*x[0]**o + ((0.74416679673+0.770412990867j))*x[0] + ((0.685010999665+0.725140187175j))*x[1]**o + ((0.410544381162+0.58537350382j))*x[1]
            ref[(2, 0, 1)]=(0.571740997293+0.68849618711j)-((-0.00187153055112+0.00353431325555j))*(o-1.)
            arg[(2, 0, 2)]=(0.406732478414-0.0561428354683j)*x[0]**o + ((-0.381401122504-0.54235122479j))*x[0] + ((-0.577388719612-0.219027233918j))*x[1]**o + ((0.729356732007+0.755830299715j))*x[1]
            ref[(2, 0, 2)]=(0.088649684152-0.0308454972309j)-((-0.0284427068664-0.0458616782311j))*(o-1.)
            arg[(2, 0, 3)]=(0.266422248634+0.354785154767j)*x[0]**o + ((-0.943913904956+0.3320563399j))*x[0] + ((0.694437878832-0.716277457474j))*x[1]**o + ((0.361076267387-0.780090655713j))*x[1]
            ref[(2, 0, 3)]=(0.189011244948-0.40476330926j)-((0.160143354578-0.0602487171178j))*(o-1.)
            arg[(2, 1, 0)]=(0.535378648491+0.836484297513j)*x[0]**o + ((-0.713969440173+0.312689281286j))*x[0] + ((0.729974133154+0.19522278173j))*x[1]**o + ((-0.293926791392+0.85987505667j))*x[1]
            ref[(2, 1, 0)]=(0.12872827504+1.1021357086j)-((0.210892130274+0.171951179874j))*(o-1.)
            arg[(2, 1, 1)]=(-0.270240188875+0.423547049216j)*x[0]**o + ((-0.746636857905-0.266773860254j))*x[0] + ((-0.950101296456-0.395015977564j))*x[1]**o + ((0.72544072713+0.823972157278j))*x[1]
            ref[(2, 1, 1)]=(-0.620768808053+0.292864684338j)-((-0.203390247555+0.00475517860863j))*(o-1.)
            arg[(2, 1, 2)]=(0.0657744366258-0.921985396583j)*x[0]**o + ((-0.11023966589-0.556831496767j))*x[0] + ((-0.370340065204+0.0780783959464j))*x[1]**o + ((-0.839141703052+0.61675355964j))*x[1]
            ref[(2, 1, 2)]=(-0.62697349876-0.391992468881j)-((-0.0507609380963-0.140651166773j))*(o-1.)
            arg[(2, 1, 3)]=(0.481261636589+0.296777450047j)*x[0]**o + ((0.874819965476-0.196326078954j))*x[0] + ((0.128240513726+0.562935689788j))*x[1]**o + ((-0.190908332417+0.393516843733j))*x[1]
            ref[(2, 1, 3)]=(0.646706891687+0.528451952308j)-((0.101583691719+0.143285523306j))*(o-1.)
            arg[(2, 2, 0)]=(-0.283962775686-0.00691405086811j)*x[0]**o + ((-0.663953652369-0.799882933159j))*x[0] + ((-0.0314191149978-0.16248033553j))*x[1]**o + ((-0.378523590019-0.532276704247j))*x[1]
            ref[(2, 2, 0)]=(-0.678929566536-0.750777011902j)-((-0.0525636484473-0.028232397733j))*(o-1.)
            arg[(2, 2, 1)]=(0.132380972876-0.266170753128j)*x[0]**o + ((0.640875450555+0.804953629247j))*x[0] + ((-0.845513170979+0.580326301498j))*x[1]**o + ((-0.504610196751-0.5672157146j))*x[1]
            ref[(2, 2, 1)]=(-0.288433472149+0.275946731508j)-((-0.11885536635+0.0523592580616j))*(o-1.)
            arg[(2, 2, 2)]=(0.334820510195-0.860018196626j)*x[0]**o + ((0.697948184317-0.342843407178j))*x[0] + ((0.566039974421-0.796058242207j))*x[1]**o + ((-0.268034225721-0.92077556897j))*x[1]
            ref[(2, 2, 2)]=(0.665387221605-1.45984770749j)-((0.150143414103-0.276012739805j))*(o-1.)
            arg[(2, 2, 3)]=(0.999034410019-0.758589293616j)*x[0]**o + ((-0.132416257416+0.459017473946j))*x[0] + ((0.71829561889-0.126517120374j))*x[1]**o + ((0.601502254565+0.97380429604j))*x[1]
            ref[(2, 2, 3)]=(1.09320801303+0.273857677998j)-((0.286221671485-0.147517735665j))*(o-1.)
            arg[(2, 3, 0)]=(-0.793483795938+0.241143344044j)*x[0]**o + ((-0.597691131049+0.726450665875j))*x[0] + ((0.0399139311255+0.175610047291j))*x[1]**o + ((0.210541167185+0.25406726459j))*x[1]
            ref[(2, 3, 0)]=(-0.570359914338+0.6986356609j)-((-0.125594977469+0.0694588985558j))*(o-1.)
            arg[(2, 3, 1)]=(0.478043405438+0.50605915376j)*x[0]**o + ((-0.880472866355-0.344002737908j))*x[0] + ((0.939742863731-0.582548718832j))*x[1]**o + ((0.159821635578+0.938843443559j))*x[1]
            ref[(2, 3, 1)]=(0.348567519196+0.259175570289j)-((0.236297711528-0.0127482608454j))*(o-1.)
            arg[(2, 3, 2)]=(0.9962073387-0.380126074118j)*x[0]**o + ((0.701036269773+0.253351060985j))*x[0] + ((0.805723300478-0.221101756583j))*x[1]**o + ((0.329256121241+0.545589223906j))*x[1]
            ref[(2, 3, 2)]=(1.4161115151+0.0988562270947j)-((0.300321773196-0.10020463845j))*(o-1.)
            arg[(2, 3, 3)]=(-0.735522460251+0.208231823929j)*x[0]**o + ((0.831416800325+0.712366470407j))*x[0] + ((-0.495173596367+0.127825748656j))*x[1]**o + ((0.45696923823-0.0191080898211j))*x[1]
            ref[(2, 3, 3)]=(0.0288449909682+0.514657976586j)-((-0.205116009436+0.0560095954309j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(-0.692687750135+0.460953796755j)*x[0]**o + ((-0.514241884401-0.146077174452j))*x[0] + ((-0.563577125223+0.426188369484j))*x[1]**o + ((0.317560040604+0.151057000601j))*x[1] + ((0.229221047699-0.762017035448j))*x[2]**o + ((-0.247744537801-0.0436654507781j))*x[2]
            ref[(0, 0, 0)]=(-0.735735104628+0.0432197530809j)-((-0.171173971276+0.020854188465j))*(o-1.)
            arg[(0, 0, 1)]=(-0.478753663869-0.712645322245j)*x[0]**o + ((0.92118230303-0.866323827716j))*x[0] + ((0.333872521533-0.0309365665551j))*x[1]**o + ((0.175110398343+0.981298778357j))*x[1] + ((0.805462662846-0.779615163297j))*x[2]**o + ((0.893042107608-0.612452674887j))*x[2]
            ref[(0, 0, 1)]=(1.32495816475-1.01033738817j)-((0.110096920085-0.253866175349j))*(o-1.)
            arg[(0, 0, 2)]=(0.178596095849+0.380339159898j)*x[0]**o + ((-0.263001069991+0.946836555158j))*x[0] + ((-0.57017151402-0.63009152884j))*x[1]**o + ((0.567545316579-0.278398973928j))*x[1] + ((-0.919684752575+0.0373446161755j))*x[2]**o + ((-0.253918044971-0.485466768588j))*x[2]
            ref[(0, 0, 2)]=(-0.630316984564-0.0147184700621j)-((-0.218543361791-0.0354012921277j))*(o-1.)
            arg[(0, 0, 3)]=(-0.576769081776-0.539929891971j)*x[0]**o + ((-0.742736770721-0.128705805441j))*x[0] + ((-0.857795577403-0.0953603106022j))*x[1]**o + ((0.832575563823-0.771339420683j))*x[1] + ((-0.0498575056791+0.931590567412j))*x[2]**o + ((-0.305499581835-0.0796616020009j))*x[2]
            ref[(0, 0, 3)]=(-0.850041476795-0.341703231643j)-((-0.247403694143+0.0493833941398j))*(o-1.)
            arg[(0, 1, 0)]=(0.808303667412+0.260843508585j)*x[0]**o + ((-0.418836011834-0.498470013903j))*x[0] + ((-0.123703761283-0.336347823079j))*x[1]**o + ((0.15182169645+0.419948341293j))*x[1] + ((-0.0413730120515+0.512087089029j))*x[2]**o + ((-0.964243331268-0.167387952073j))*x[2]
            ref[(0, 1, 0)]=(-0.294015376287+0.0953365749261j)-((0.107204482346+0.0727637957559j))*(o-1.)
            arg[(0, 1, 1)]=(-0.0762176086648+0.391946700127j)*x[0]**o + ((0.767439800331+0.947463377065j))*x[0] + ((0.21064684478+0.623794572153j))*x[1]**o + ((-0.438030889711+0.0114803985875j))*x[1] + ((0.142606535914+0.712656125083j))*x[2]**o + ((-0.194864565253+0.841440648713j))*x[2]
            ref[(0, 1, 1)]=(0.205790058698+1.76439091086j)-((0.0461726286714+0.288066232894j))*(o-1.)
            arg[(0, 1, 2)]=(-0.709108421739-0.532588986579j)*x[0]**o + ((-0.993527321296-0.89263191257j))*x[0] + ((-0.448625825862+0.333678268938j))*x[1]**o + ((0.937426536908-0.526633542605j))*x[1] + ((0.73658675746+0.726440305973j))*x[2]**o + ((0.712247354784-0.385145822623j))*x[2]
            ref[(0, 1, 2)]=(0.117499540128-0.638440844733j)-((-0.0701912483567+0.0879215980554j))*(o-1.)
            arg[(0, 1, 3)]=(-0.889121542486+0.75464568479j)*x[0]**o + ((0.41638835339-0.6714476728j))*x[0] + ((-0.626935454967+0.705604587358j))*x[1]**o + ((-0.267983609693+0.624417505329j))*x[1] + ((-0.215574742686+0.292534640008j))*x[2]**o + ((0.00515103844391+0.383071842344j))*x[2]
            ref[(0, 1, 3)]=(-0.789037978999+1.04441329351j)-((-0.288605290023+0.292130818693j))*(o-1.)
            arg[(0, 2, 0)]=(0.760915256992-0.820972025109j)*x[0]**o + ((-0.0486689735807-0.790336233947j))*x[0] + ((0.195310860535+0.486257925091j))*x[1]**o + ((0.358218313785+0.710103710725j))*x[1] + ((-0.75632687611+0.350529685396j))*x[2]**o + ((0.442745211661-0.157391834065j))*x[2]
            ref[(0, 2, 0)]=(0.476096896641-0.110904385954j)-((0.0333165402361+0.00263593089646j))*(o-1.)
            arg[(0, 2, 1)]=(-0.705832390943+0.815967769051j)*x[0]**o + ((-0.0961858871257+0.483385302936j))*x[0] + ((0.97310732026+0.457366456621j))*x[1]**o + ((0.432333885292-0.811585560291j))*x[1] + ((-0.915890837889+0.502670083828j))*x[2]**o + ((-0.758804186222+0.706553500859j))*x[2]
            ref[(0, 2, 1)]=(-0.535636048314+1.0771787765j)-((-0.108102651429+0.29600071825j))*(o-1.)
            arg[(0, 2, 2)]=(-0.338176470168-0.143794583087j)*x[0]**o + ((0.730936632373-0.27148946471j))*x[0] + ((0.261240874643+0.0549504508745j))*x[1]**o + ((-0.76335997282-0.957869083476j))*x[1] + ((0.90985008178-0.819118424457j))*x[2]**o + ((0.0745756395855+0.0577378829323j))*x[2]
            ref[(0, 2, 2)]=(0.437533392697-1.03979161096j)-((0.138819081042-0.151327092778j))*(o-1.)
            arg[(0, 2, 3)]=(0.44388453454+0.169239996428j)*x[0]**o + ((-0.123067260588+0.537714806031j))*x[0] + ((0.659839007667+0.369539587364j))*x[1]**o + ((0.0119720626453-0.222501651873j))*x[1] + ((0.107606504351+0.217629265031j))*x[2]**o + ((-0.444812137533-0.952558497611j))*x[2]
            ref[(0, 2, 3)]=(0.327711355541+0.0595317526843j)-((0.201888341093+0.12606814147j))*(o-1.)
            arg[(0, 3, 0)]=(0.371208085155+0.247433271971j)*x[0]**o + ((0.213811520799+0.342521743469j))*x[0] + ((0.0725377271184-0.504516091907j))*x[1]**o + ((-0.471017598857+0.700217758559j))*x[1] + ((-0.223839179952-0.355567450326j))*x[2]**o + ((-0.839793288257-0.381635527935j))*x[2]
            ref[(0, 3, 0)]=(-0.438546366997+0.0242268519154j)-((0.0366511053869-0.102108378377j))*(o-1.)
            arg[(0, 3, 1)]=(-0.666459154938-0.643219120601j)*x[0]**o + ((0.925621876117-0.621163918199j))*x[0] + ((-0.278354528882-0.58874114934j))*x[1]**o + ((0.417850253276-0.902757561707j))*x[1] + ((0.0870661352132+0.331483770484j))*x[2]**o + ((0.137389688162-0.0856920123134j))*x[2]
            ref[(0, 3, 1)]=(0.311557134474-1.25504499584j)-((-0.142957924768-0.150079416576j))*(o-1.)
            arg[(0, 3, 2)]=(-0.1912924936-0.489796682829j)*x[0]**o + ((0.531262317018+0.200286629959j))*x[0] + ((-0.383722891351+0.984272867922j))*x[1]**o + ((0.878713438649-0.745216424259j))*x[1] + ((0.811802194768-0.722269721974j))*x[2]**o + ((-0.131988479516+0.770714593078j))*x[2]
            ref[(0, 3, 2)]=(0.757387042983-0.00100436905102j)-((0.0394644683028-0.03796558948j))*(o-1.)
            arg[(0, 3, 3)]=(0.580685220124-0.890752875401j)*x[0]**o + ((-0.919591406599+0.628226350501j))*x[0] + ((-0.626391319686+0.106057893469j))*x[1]**o + ((0.406618241412-0.251981960831j))*x[1] + ((-0.645602335765+0.567948296243j))*x[2]**o + ((-0.397697438088+0.353589419658j))*x[2]
            ref[(0, 3, 3)]=(-0.800989519301+0.256543561819j)-((-0.115218072554-0.0361244476149j))*(o-1.)
            arg[(1, 0, 0)]=(-0.384510797773+0.809901728401j)*x[0]**o + ((-0.806521341126-0.699053492258j))*x[0] + ((0.932667685438+0.569326345469j))*x[1]**o + ((0.107810636319+0.315413531176j))*x[1] + ((-0.0722175623345-0.502182797623j))*x[2]**o + ((0.282609430969+0.850517424902j))*x[2]
            ref[(1, 0, 0)]=(0.0299190257462+0.671961370034j)-((0.0793232208884+0.146174212708j))*(o-1.)
            arg[(1, 0, 1)]=(0.788073370575+0.755009119343j)*x[0]**o + ((0.0959104129966-0.855296000262j))*x[0] + ((0.56489842194-0.739994805747j))*x[1]**o + ((0.0824707931251-0.346744835596j))*x[1] + ((-0.0383658612559+0.973473011219j))*x[2]**o + ((-0.478320767853-0.424991175594j))*x[2]
            ref[(1, 0, 1)]=(0.507333184764-0.319272343319j)-((0.219100988543+0.164747887469j))*(o-1.)
            arg[(1, 0, 2)]=(-0.195954877921+0.541886100402j)*x[0]**o + ((-0.658435042648-0.468243823414j))*x[0] + ((-0.585675054266-0.384230221149j))*x[1]**o + ((-0.664173150496-0.253975791039j))*x[1] + ((0.915734398343+0.95396155542j))*x[2]**o + ((-0.310430931502-0.13293152313j))*x[2]
            ref[(1, 0, 2)]=(-0.749467329245+0.128233148545j)-((0.0223507443593+0.185269572446j))*(o-1.)
            arg[(1, 0, 3)]=(-0.973378389781+0.354965514097j)*x[0]**o + ((-0.223575907568-0.187256662722j))*x[0] + ((0.801323996414-0.995974204252j))*x[1]**o + ((-0.169741327275-0.517605989288j))*x[1] + ((-0.158598977443-0.684381424015j))*x[2]**o + ((0.929504929524-0.196346444981j))*x[2]
            ref[(1, 0, 3)]=(0.102767161936-1.11329960558j)-((-0.0551088951349-0.220898352362j))*(o-1.)
            arg[(1, 1, 0)]=(-0.111822232569+0.623337527394j)*x[0]**o + ((0.648284908676+0.949054178299j))*x[0] + ((-0.748435979616-0.390230022161j))*x[1]**o + ((0.795915547756+0.993718634677j))*x[1] + ((-0.170776032443+0.855969079123j))*x[2]**o + ((0.794070273605+0.743144106167j))*x[2]
            ref[(1, 1, 0)]=(0.603618242704+1.88749675175j)-((-0.171839040771+0.181512764059j))*(o-1.)
            arg[(1, 1, 1)]=(0.663599057533-0.49732176328j)*x[0]**o + ((-0.460506096958+0.499013379337j))*x[0] + ((-0.63772856719+0.292395242068j))*x[1]**o + ((0.168599406149-0.115238744311j))*x[1] + ((-0.00938044422187+0.991160441369j))*x[2]**o + ((-0.0350490419611+0.201444696646j))*x[2]
            ref[(1, 1, 1)]=(-0.155232843324+0.685726625914j)-((0.00274834102027+0.131038986693j))*(o-1.)
            arg[(1, 1, 2)]=(-0.91294199044-0.448789223764j)*x[0]**o + ((-0.22068116528-0.073725225445j))*x[0] + ((-0.225575513192-0.0205405516981j))*x[1]**o + ((0.339089860453+0.0546174871955j))*x[1] + ((0.253553969839-0.476674321257j))*x[2]**o + ((-0.0025232238377-0.866743235754j))*x[2]
            ref[(1, 1, 2)]=(-0.384539031229-0.915927535361j)-((-0.147493922299-0.157667349453j))*(o-1.)
            arg[(1, 1, 3)]=(-0.627202087464+0.658556617835j)*x[0]**o + ((-0.773309073251-0.659311603256j))*x[0] + ((0.0756378602581+0.453956511684j))*x[1]**o + ((-0.661929763411+0.0186249418307j))*x[1] + ((0.364824613171-0.819007804417j))*x[2]**o + ((0.157014632879-0.608225474194j))*x[2]
            ref[(1, 1, 3)]=(-0.732481908909-0.477703405259j)-((-0.0311232690058+0.0489175541835j))*(o-1.)
            arg[(1, 2, 0)]=(0.831914784021+0.209205034642j)*x[0]**o + ((-0.0727675848744+0.0161662954388j))*x[0] + ((-0.582907060509-0.802979746984j))*x[1]**o + ((-0.0727350717606-0.665999868031j))*x[1] + ((0.981145628391-0.862294986324j))*x[2]**o + ((0.362638883693-0.0538293115396j))*x[2]
            ref[(1, 2, 0)]=(0.723644789481-1.0798662914j)-((0.205025558651-0.242678283111j))*(o-1.)
            arg[(1, 2, 1)]=(-0.129940782926-0.0843640746983j)*x[0]**o + ((0.850656320177+0.93086994168j))*x[0] + ((0.355836541255-0.538757852201j))*x[1]**o + ((-0.282316724571+0.627412604889j))*x[1] + ((0.79463795289-0.578014574588j))*x[2]**o + ((-0.0691574294468-0.100213107469j))*x[2]
            ref[(1, 2, 1)]=(0.759857938689+0.128466468807j)-((0.17008895187-0.200189416914j))*(o-1.)
            arg[(1, 2, 2)]=(-0.156037483845-0.769432891542j)*x[0]**o + ((0.619525385564+0.556163216653j))*x[0] + ((-0.433639064839+0.115689367294j))*x[1]**o + ((0.0914743025432+0.197431166034j))*x[1] + ((0.918354551539-0.890972248306j))*x[2]**o + ((0.777721783446-0.867479981363j))*x[2]
            ref[(1, 2, 2)]=(0.908699737204-0.829300685615j)-((0.0547796671424-0.257452628759j))*(o-1.)
            arg[(1, 2, 3)]=(0.730518050262+0.913399405612j)*x[0]**o + ((-0.325546179792-0.335768098865j))*x[0] + ((-0.0631063642904+0.16449826523j))*x[1]**o + ((0.296785301566+0.427618221946j))*x[1] + ((-0.614250453849-0.0555299490188j))*x[2]**o + ((0.284034155478+0.0643070350527j))*x[2]
            ref[(1, 2, 3)]=(0.154217254687+0.589262439978j)-((0.00886020535381+0.170394620304j))*(o-1.)
            arg[(1, 3, 0)]=(0.597414848543-0.302041817456j)*x[0]**o + ((0.871899048917+0.678325892565j))*x[0] + ((0.281821261648+0.723582578368j))*x[1]**o + ((-0.501413789805-0.942018845407j))*x[1] + ((-0.329983088342+0.856894727573j))*x[2]**o + ((-0.401373609882+0.568627010036j))*x[2]
            ref[(1, 3, 0)]=(0.25918233554+0.79168477284j)-((0.0915421703082+0.213072581414j))*(o-1.)
            arg[(1, 3, 1)]=(0.423908012158+0.300706470356j)*x[0]**o + ((0.186119348348+0.460344256394j))*x[0] + ((0.252891005317-0.387550980483j))*x[1]**o + ((0.0854065572732-0.171978029404j))*x[1] + ((-0.65265388942+0.833890203673j))*x[2]**o + ((0.0167584453035+0.800514139053j))*x[2]
            ref[(1, 3, 1)]=(0.15621473949+0.917963029794j)-((0.00402418800913+0.124507615591j))*(o-1.)
            arg[(1, 3, 2)]=(0.0272544892789-0.765921101941j)*x[0]**o + ((0.561146241557+0.925566098772j))*x[0] + ((0.0293770467907-0.76232858857j))*x[1]**o + ((0.155036259694+0.637060169342j))*x[1] + ((-0.869131678689-0.585199146443j))*x[2]**o + ((-0.534668498075+0.614152406697j))*x[2]
            ref[(1, 3, 2)]=(-0.315493069722+0.0316649189287j)-((-0.135416690437-0.352241472826j))*(o-1.)
            arg[(1, 3, 3)]=(-0.663253925778-0.568454154166j)*x[0]**o + ((-0.790964110416+0.748482146907j))*x[0] + ((-0.938359288881+0.743864556219j))*x[1]**o + ((0.731014090855-0.694481174798j))*x[1] + ((-0.402528017639+0.0109282048518j))*x[2]**o + ((0.828334954221-0.620873456891j))*x[2]
            ref[(1, 3, 3)]=(-0.617878148819-0.190266938939j)-((-0.334023538716+0.0310564344841j))*(o-1.)
            arg[(2, 0, 0)]=(0.583885027564-0.820188691055j)*x[0]**o + ((-0.385285675156+0.0806272558063j))*x[0] + ((0.256264727565+0.4346351214j))*x[1]**o + ((-0.234720487722+0.550500776051j))*x[1] + ((0.151234816713-0.0200087603542j))*x[2]**o + ((0.797153502525+0.704086729124j))*x[2]
            ref[(2, 0, 0)]=(0.584265955745+0.464826215487j)-((0.165230761974-0.0675937216681j))*(o-1.)
            arg[(2, 0, 1)]=(-0.599837659538-0.0735396540122j)*x[0]**o + ((0.37854279202-0.332302817567j))*x[0] + ((-0.40129212963-0.337608468464j))*x[1]**o + ((0.514611135485-0.359338454093j))*x[1] + ((-0.758883520663-0.414271523458j))*x[2]**o + ((0.0870253653565+0.746661400224j))*x[2]
            ref[(2, 0, 1)]=(-0.389917008484-0.385199758685j)-((-0.293335551638-0.137569940989j))*(o-1.)
            arg[(2, 0, 2)]=(-0.198855905389-0.716843486679j)*x[0]**o + ((0.811486421876-0.612201632599j))*x[0] + ((0.672411828713-0.799320415758j))*x[1]**o + ((0.0982184047273+0.86652839042j))*x[1] + ((0.0500802124368+0.254111358966j))*x[2]**o + ((-0.647288836042+0.714173329266j))*x[2]
            ref[(2, 0, 2)]=(0.393026063161-0.146776228192j)-((0.0872726892935-0.210342090578j))*(o-1.)
            arg[(2, 0, 3)]=(0.292440219838-0.0684734060042j)*x[0]**o + ((-0.722912440095-0.839710098565j))*x[0] + ((0.28447028359+0.873381082881j))*x[1]**o + ((0.482936867606+0.483395965158j))*x[1] + ((0.227798485316+0.669168137221j))*x[2]**o + ((-0.883534991078+0.690636522457j))*x[2]
            ref[(2, 0, 3)]=(-0.159400787412+0.904199101574j)-((0.134118164791+0.24567930235j))*(o-1.)
            arg[(2, 1, 0)]=(0.869911305533+0.177330664517j)*x[0]**o + ((-0.434129373062+0.714217504321j))*x[0] + ((0.284494942011-0.640165139685j))*x[1]**o + ((0.43888402361+0.339672557557j))*x[1] + ((-0.809298589484+0.582160517235j))*x[2]**o + ((0.691159860927+0.0169878339575j))*x[2]
            ref[(2, 1, 0)]=(0.520511084767+0.595101968951j)-((0.0575179430099+0.0198876736779j))*(o-1.)
            arg[(2, 1, 1)]=(-0.971752678371-0.940697759121j)*x[0]**o + ((0.718514180046+0.225516473286j))*x[0] + ((-0.129976342426+0.390761070292j))*x[1]**o + ((-0.675511331614-0.323727727282j))*x[1] + ((0.868959860585-0.506748211372j))*x[2]**o + ((0.882626130034+0.255906906507j))*x[2]
            ref[(2, 1, 1)]=(0.346429909127-0.449494623845j)-((-0.0387948600353-0.176114150034j))*(o-1.)
            arg[(2, 1, 2)]=(-0.109615339982-0.409770175334j)*x[0]**o + ((-0.794593047773-0.93592411535j))*x[0] + ((0.706499550993+0.104181427893j))*x[1]**o + ((0.195999295269-0.232993550268j))*x[1] + ((0.976720411102-0.1873555769j))*x[2]**o + ((0.389594692108+0.173117011354j))*x[2]
            ref[(2, 1, 2)]=(0.682302780859-0.744372489303j)-((0.262267437019-0.0821573873902j))*(o-1.)
            arg[(2, 1, 3)]=(0.83959662297+0.365303966529j)*x[0]**o + ((0.555260587028+0.615343755403j))*x[0] + ((-0.823776200301+0.685407274752j))*x[1]**o + ((0.299757731942+0.18204446893j))*x[1] + ((-0.656841814738+0.0911111843532j))*x[2]**o + ((-0.160251589324+0.557232664918j))*x[2]
            ref[(2, 1, 3)]=(0.0268726687891+1.24822165744j)-((-0.106836898678+0.190303737606j))*(o-1.)
            arg[(2, 2, 0)]=(-0.787546932129+0.994370623366j)*x[0]**o + ((0.743119945757-0.904071929451j))*x[0] + ((0.0379322043168+0.495622039974j))*x[1]**o + ((0.156847005926-0.0686778717603j))*x[1] + ((-0.974767499849+0.0403570978278j))*x[2]**o + ((-0.372061754925-0.889486253489j))*x[2]
            ref[(2, 2, 0)]=(-0.598238515451-0.165943146766j)-((-0.287397037943+0.255058293528j))*(o-1.)
            arg[(2, 2, 1)]=(-0.0903951172017-0.428818018911j)*x[0]**o + ((-0.62973352383+0.406019381051j))*x[0] + ((0.543541707202-0.496456215177j))*x[1]**o + ((-0.921818051567-0.793855616431j))*x[1] + ((0.927366693607-0.944694258123j))*x[2]**o + ((0.532064175729+0.649102970844j))*x[2]
            ref[(2, 2, 1)]=(0.18051294197-0.804350878373j)-((0.230085547268-0.311661415368j))*(o-1.)
            arg[(2, 2, 2)]=(0.711314967525-0.825681977257j)*x[0]**o + ((-0.888370902073+0.275018446196j))*x[0] + ((0.335599532437+0.171279546819j))*x[1]**o + ((0.200916771834-0.643088414722j))*x[1] + ((-0.662632476693-0.170688596004j))*x[2]**o + ((-0.220087038131+0.772214895345j))*x[2]
            ref[(2, 2, 2)]=(-0.26162957255-0.210473049811j)-((0.0640470038783-0.137515171074j))*(o-1.)
            arg[(2, 2, 3)]=(-0.539208059612+0.846353848924j)*x[0]**o + ((-0.845229804447+0.587124781505j))*x[0] + ((-0.197315435145+0.504655320572j))*x[1]**o + ((0.594455053475-0.223516373765j))*x[1] + ((-0.853903354989+0.487992368506j))*x[2]**o + ((0.58752621876+0.226370802803j))*x[2]
            ref[(2, 2, 3)]=(-0.626837690979+1.21449037427j)-((-0.265071141624+0.306500256334j))*(o-1.)
            arg[(2, 3, 0)]=(-0.952852096061+0.174297570862j)*x[0]**o + ((0.955228378845-0.796573770254j))*x[0] + ((0.926661536609-0.145981249932j))*x[1]**o + ((0.303483263947+0.568423465752j))*x[1] + ((-0.138488887338-0.815892389054j))*x[2]**o + ((-0.484132610776+0.366410846605j))*x[2]
            ref[(2, 3, 0)]=(0.304949792613-0.324657763011j)-((-0.0274465744651-0.131262678021j))*(o-1.)
            arg[(2, 3, 1)]=(0.935243547875-0.36077380347j)*x[0]**o + ((0.0702109288902-0.544071581951j))*x[0] + ((0.366611717641-0.931528461524j))*x[1]**o + ((-0.269915526212-0.734264016631j))*x[1] + ((0.213680386382-0.669956591391j))*x[2]**o + ((-0.649136153299+0.375989332883j))*x[2]
            ref[(2, 3, 1)]=(0.333347450639-1.43230256104j)-((0.252589275316-0.327043142731j))*(o-1.)
            arg[(2, 3, 2)]=(0.791578008548-0.350172953874j)*x[0]**o + ((0.0785396290458+0.40919849284j))*x[0] + ((0.115749305475+0.213517371067j))*x[1]**o + ((0.00203302907959+0.971408227734j))*x[1] + ((0.623630485925-0.796989780294j))*x[2]**o + ((-0.204038937851+0.366546726019j))*x[2]
            ref[(2, 3, 2)]=(0.703745760111+0.406754041745j)-((0.255159633325-0.155607560517j))*(o-1.)
            arg[(2, 3, 3)]=(0.222859816609-0.969363015766j)*x[0]**o + ((-0.764708835839+0.578365031404j))*x[0] + ((0.487155250171-0.421159370262j))*x[1]**o + ((0.936315924039+0.3335231745j))*x[1] + ((-0.549521851895+0.662974693943j))*x[2]**o + ((0.854279583405+0.137438118585j))*x[2]
            ref[(2, 3, 3)]=(0.593189943245+0.160889316202j)-((0.0267488691476-0.121257948681j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ContinuousFunction_rank4(self):
        """
        tests integral of rank 4 Data on ReducedFunction

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 3, 3),w)
        ref=numpy.zeros((3, 4, 3, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.268539356423-0.278903038502j)*x[0]**o + ((-0.419256495498+0.143073550691j))*x[0] + ((-0.504984826243+0.5686747403j))*x[1]**o + ((-0.487950041037-0.197383551806j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.8403653596+0.117730850341j)-((-0.128920697111+0.048295283633j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.677634740824+0.851866160908j)*x[0]**o + ((-0.0827672014775-0.368048565535j))*x[0] + ((-0.443490848936+0.449211734604j))*x[1]**o + ((-0.919156692002+0.354156604382j))*x[1]
            ref[(0, 0, 0, 1)]=(-1.06152474162+0.64359296718j)-((-0.18685426496+0.216846315919j))*(o-1.)
            arg[(0, 0, 0, 2)]=(-0.962395634348+0.237765492524j)*x[0]**o + ((0.455361885866+0.590902272908j))*x[0] + ((0.399313328916+0.945800228198j))*x[1]**o + ((0.420491453264+0.23184879471j))*x[1]
            ref[(0, 0, 0, 2)]=(0.156385516849+1.00315839417j)-((-0.0938470509053+0.197260953454j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.714170379135-0.326760500433j)*x[0]**o + ((0.533380544087-0.689702432637j))*x[0] + ((0.610031378307-0.235771819123j))*x[1]**o + ((0.858672513248+0.182480132147j))*x[1]
            ref[(0, 0, 1, 0)]=(1.35812740739-0.534877310023j)-((0.220700292907-0.0937553865927j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.0122459242717+0.512924449376j)*x[0]**o + ((0.236117230351-0.672411935268j))*x[0] + ((-0.230075605407-0.423895432924j))*x[1]**o + ((0.781065059156-0.0258983471199j))*x[1]
            ref[(0, 0, 1, 1)]=(0.387430379914-0.304640632968j)-((-0.0403869216132+0.0148381694086j))*(o-1.)
            arg[(0, 0, 1, 2)]=(-0.768703225608+0.189891421322j)*x[0]**o + ((0.903310318152-0.130024123807j))*x[0] + ((0.0651882423549+0.797174362767j))*x[1]**o + ((-0.6592304326-0.897582284016j))*x[1]
            ref[(0, 0, 1, 2)]=(-0.229717548851-0.0202703118672j)-((-0.117252497209+0.164510964015j))*(o-1.)
            arg[(0, 0, 2, 0)]=(-0.515467089452-0.719364252755j)*x[0]**o + ((-0.272474501032+0.796036205543j))*x[0] + ((0.727211163495-0.162178955763j))*x[1]**o + ((0.973140645248-0.196200419825j))*x[1]
            ref[(0, 0, 2, 0)]=(0.456205109129-0.1408537114j)-((0.0352906790072-0.146923868086j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.0518149063357+0.797967671558j)*x[0]**o + ((0.940942632007-0.48714033471j))*x[0] + ((0.983115959236+0.231307903771j))*x[1]**o + ((0.70484000011-0.346720665664j))*x[1]
            ref[(0, 0, 2, 1)]=(1.34035674884+0.097707287478j)-((0.172488477595+0.171545929222j))*(o-1.)
            arg[(0, 0, 2, 2)]=(0.712839086141-0.58275611612j)*x[0]**o + ((0.932903040459+0.114818925379j))*x[0] + ((-0.997612144393-0.188539175219j))*x[1]**o + ((0.41851438258-0.89534521524j))*x[1]
            ref[(0, 0, 2, 2)]=(0.533322182393-0.7759107906j)-((-0.0474621763754-0.128549215223j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.310912785452-0.635151088221j)*x[0]**o + ((0.906498782553-0.765364185406j))*x[0] + ((-0.55380129961-0.117328550722j))*x[1]**o + ((0.380401875651+0.986549931218j))*x[1]
            ref[(0, 1, 0, 0)]=(0.211093286571-0.265646946566j)-((-0.144119014177-0.125413273157j))*(o-1.)
            arg[(0, 1, 0, 1)]=(-0.252124912323-0.975328734671j)*x[0]**o + ((-0.849533394448+0.579643162117j))*x[0] + ((0.28723040706+0.0728063273662j))*x[1]**o + ((0.941363477018-0.767588329078j))*x[1]
            ref[(0, 1, 0, 1)]=(0.0634677886538-0.545233787133j)-((0.00585091578959-0.150420401218j))*(o-1.)
            arg[(0, 1, 0, 2)]=(0.403455615169+0.911437029262j)*x[0]**o + ((0.0311238972852-0.756207087732j))*x[0] + ((-0.00914891595551-0.389089751578j))*x[1]**o + ((0.624012365331+0.61644331373j))*x[1]
            ref[(0, 1, 0, 2)]=(0.524721480915+0.191291751841j)-((0.0657177832022+0.0870578796139j))*(o-1.)
            arg[(0, 1, 1, 0)]=(0.534478775658-0.625666965919j)*x[0]**o + ((-0.730982066909-0.838143366509j))*x[0] + ((0.298620697807-0.160029270807j))*x[1]**o + ((0.995279586532-0.483987279656j))*x[1]
            ref[(0, 1, 1, 0)]=(0.548698496544-1.05391344145j)-((0.138849912244-0.130949372788j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.543381470603-0.869031056522j)*x[0]**o + ((0.183253230911-0.353564483562j))*x[0] + ((0.320883729719-0.163344353422j))*x[1]**o + ((0.513860990781+0.327173572226j))*x[1]
            ref[(0, 1, 1, 1)]=(0.237308240404-0.52938316064j)-((-0.0370829568139-0.172062568324j))*(o-1.)
            arg[(0, 1, 1, 2)]=(-0.521302660745+0.297907869793j)*x[0]**o + ((-0.388190496221-0.832115753277j))*x[0] + ((-0.16144743278+0.138544775951j))*x[1]**o + ((0.55822762969-0.883833165068j))*x[1]
            ref[(0, 1, 1, 2)]=(-0.256356480028-0.6397481363j)-((-0.113791682254+0.0727421076241j))*(o-1.)
            arg[(0, 1, 2, 0)]=(0.845596413921+0.862223836863j)*x[0]**o + ((-0.195799150846+0.919684935604j))*x[0] + ((0.381366822204+0.969320487582j))*x[1]**o + ((-0.772418763193-0.205584331163j))*x[1]
            ref[(0, 1, 2, 0)]=(0.129372661043+1.27282246444j)-((0.204493872688+0.305257387407j))*(o-1.)
            arg[(0, 1, 2, 1)]=(-0.480265592829+0.731746295433j)*x[0]**o + ((-0.0590574921824+0.994913016847j))*x[0] + ((0.119739021883+0.969089549942j))*x[1]**o + ((-0.823610907769-0.957870139895j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.621597485449+0.868939361164j)-((-0.0600877618244+0.283472640896j))*(o-1.)
            arg[(0, 1, 2, 2)]=(0.197688172035+0.832825488431j)*x[0]**o + ((0.209598604073+0.395186318037j))*x[0] + ((0.168274988733+0.519752764707j))*x[1]**o + ((0.153958449347-0.150356868651j))*x[1]
            ref[(0, 1, 2, 2)]=(0.364760107094+0.798703851262j)-((0.060993860128+0.225429708856j))*(o-1.)
            arg[(0, 2, 0, 0)]=(-0.409227491302+0.53646730613j)*x[0]**o + ((-0.157958353155+0.763597549535j))*x[0] + ((0.0307303940008+0.658380290918j))*x[1]**o + ((-0.935620566981+0.254275832164j))*x[1]
            ref[(0, 2, 0, 0)]=(-0.736038008719+1.10636048937j)-((-0.0630828495503+0.199141266175j))*(o-1.)
            arg[(0, 2, 0, 1)]=(-0.829020839063+0.682882449013j)*x[0]**o + ((0.629362359154+0.0271325399891j))*x[0] + ((-0.830101375482+0.187555249973j))*x[1]**o + ((0.127680666775+0.412635233276j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.451039594308+0.655102736125j)-((-0.276520369091+0.145072949831j))*(o-1.)
            arg[(0, 2, 0, 2)]=(-0.383607090087+0.695416601454j)*x[0]**o + ((-0.315757052843-0.732006117868j))*x[0] + ((-0.851701205107-0.8169617456j))*x[1]**o + ((0.582041542503-0.841549361879j))*x[1]
            ref[(0, 2, 0, 2)]=(-0.484511902767-0.847550311947j)-((-0.205884715866-0.0202575240244j))*(o-1.)
            arg[(0, 2, 1, 0)]=(-0.38063583964+0.544273805275j)*x[0]**o + ((0.596674451084+0.941993887224j))*x[0] + ((0.368068397037+0.0839601821543j))*x[1]**o + ((0.810999013883+0.474932391896j))*x[1]
            ref[(0, 2, 1, 0)]=(0.697553011182+1.02258013327j)-((-0.00209457376711+0.104705664572j))*(o-1.)
            arg[(0, 2, 1, 1)]=(-0.987832807733+0.070003973745j)*x[0]**o + ((0.4406525623+0.255698913842j))*x[0] + ((0.708199381646-0.0318178752379j))*x[1]**o + ((0.0315028114623-0.593973260835j))*x[1]
            ref[(0, 2, 1, 1)]=(0.0962609738378-0.150044124242j)-((-0.0466055710145+0.0063643497512j))*(o-1.)
            arg[(0, 2, 1, 2)]=(-0.583066657563-0.947946205119j)*x[0]**o + ((0.254673977108-0.374676390536j))*x[0] + ((-0.310116913305-0.797207863204j))*x[1]**o + ((0.238634037364+0.198299055261j))*x[1]
            ref[(0, 2, 1, 2)]=(-0.199937778198-0.960765701799j)-((-0.148863928478-0.290859011387j))*(o-1.)
            arg[(0, 2, 2, 0)]=(-0.424852017342-0.948171135063j)*x[0]**o + ((0.610929380642-0.425544173854j))*x[0] + ((0.65256799734-0.14417334601j))*x[1]**o + ((0.954974433648+0.964443222297j))*x[1]
            ref[(0, 2, 2, 0)]=(0.896809897144-0.276722716316j)-((0.0379526633331-0.182057413512j))*(o-1.)
            arg[(0, 2, 2, 1)]=(0.0115894720233-0.243101859088j)*x[0]**o + ((-0.102787561221+0.955952356773j))*x[0] + ((-0.723847456703+0.61347994434j))*x[1]**o + ((-0.859269829157-0.903396260329j))*x[1]
            ref[(0, 2, 2, 1)]=(-0.83715768753+0.211467090848j)-((-0.118709664113+0.0617296808754j))*(o-1.)
            arg[(0, 2, 2, 2)]=(-0.695735773471-0.546921705563j)*x[0]**o + ((0.0640444385429+0.267830289261j))*x[0] + ((0.731326525917-0.585737436937j))*x[1]**o + ((0.0656351010734-0.0235473189306j))*x[1]
            ref[(0, 2, 2, 2)]=(0.0826351460313-0.444188086085j)-((0.00593179207438-0.18877652375j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.704336544479+0.702792698582j)*x[0]**o + ((0.732453456414-0.706679544303j))*x[0] + ((0.118153956866-0.397056974627j))*x[1]**o + ((-0.849189747128+0.0195096739884j))*x[1]
            ref[(0, 3, 0, 0)]=(0.352877105316-0.19071707318j)-((0.137081750224+0.0509559539925j))*(o-1.)
            arg[(0, 3, 0, 1)]=(-0.252590709523-0.914714222301j)*x[0]**o + ((0.719190246493+0.31064324548j))*x[0] + ((0.0604285516198-0.0602693052502j))*x[1]**o + ((0.836165797549+0.878446279998j))*x[1]
            ref[(0, 3, 0, 1)]=(0.681596943069+0.107052998964j)-((-0.0320270263172-0.162497254592j))*(o-1.)
            arg[(0, 3, 0, 2)]=(0.998379194913-0.24756264549j)*x[0]**o + ((-0.959543819797-0.9700240795j))*x[0] + ((-0.674098859478-0.318274418485j))*x[1]**o + ((-0.617200522613-0.190967835794j))*x[1]
            ref[(0, 3, 0, 2)]=(-0.626232003488-0.863414489635j)-((0.0540467225725-0.0943061773292j))*(o-1.)
            arg[(0, 3, 1, 0)]=(-0.18803867135-0.991907320513j)*x[0]**o + ((0.822696278014-0.776523729403j))*x[0] + ((-0.584628776414+0.627950888323j))*x[1]**o + ((0.245705512918+0.0470688247722j))*x[1]
            ref[(0, 3, 1, 0)]=(0.147867171584-0.54670566841j)-((-0.128777907961-0.060659405365j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.605964555741+0.936419594485j)*x[0]**o + ((-0.295595444072+0.209266785051j))*x[0] + ((-0.753599817457+0.873551594711j))*x[1]**o + ((0.654139436172-0.402233566579j))*x[1]
            ref[(0, 3, 1, 1)]=(-0.500510190549+0.808502203834j)-((-0.2265940622+0.301661864866j))*(o-1.)
            arg[(0, 3, 1, 2)]=(0.951697387573+0.756802549978j)*x[0]**o + ((0.208343323102+0.95194894549j))*x[0] + ((0.00697161084532-0.199297538492j))*x[1]**o + ((-0.457044169238-0.179326622491j))*x[1]
            ref[(0, 3, 1, 2)]=(0.354984076141+0.665063667242j)-((0.159778166403+0.0929175019143j))*(o-1.)
            arg[(0, 3, 2, 0)]=(-0.480250934225+0.555128694925j)*x[0]**o + ((-0.794803494462-0.0811146236731j))*x[0] + ((-0.841455999237+0.618377788803j))*x[1]**o + ((-0.610762830633-0.327892066326j))*x[1]
            ref[(0, 3, 2, 0)]=(-1.36363662928+0.382249896865j)-((-0.22028448891+0.195584413955j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.155974627378+0.319688008944j)*x[0]**o + ((0.400178019512-0.626431340283j))*x[0] + ((0.373927975288+0.961349877067j))*x[1]**o + ((-0.425516511802+0.0981075351463j))*x[1]
            ref[(0, 3, 2, 1)]=(0.0963074278097+0.376357040437j)-((0.036325557985+0.213506314335j))*(o-1.)
            arg[(0, 3, 2, 2)]=(-0.257531372835-0.984110284571j)*x[0]**o + ((0.0890819159046-0.838050467902j))*x[0] + ((0.50197339399-0.313176420148j))*x[1]**o + ((0.580230521931+0.593098630219j))*x[1]
            ref[(0, 3, 2, 2)]=(0.456877229495-0.771119271201j)-((0.0407403368591-0.216214450787j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.327996589607+0.473062655817j)*x[0]**o + ((0.184795203483+0.54676988264j))*x[0] + ((0.93754143437-0.416946903j))*x[1]**o + ((-0.41027731618-0.124754192391j))*x[1]
            ref[(1, 0, 0, 0)]=(0.192031366033+0.239065721533j)-((0.101590807461+0.00935262546949j))*(o-1.)
            arg[(1, 0, 0, 1)]=(0.467109312638+0.0523363709201j)*x[0]**o + ((0.436593311229-0.25656334897j))*x[0] + ((-0.876493232574-0.445109153323j))*x[1]**o + ((0.0355870286832+0.803666100573j))*x[1]
            ref[(1, 0, 0, 1)]=(0.0313982099883+0.0771649845997j)-((-0.0682306533227-0.0654621304005j))*(o-1.)
            arg[(1, 0, 0, 2)]=(-0.854325421069+0.89295077122j)*x[0]**o + ((-0.475660106361+0.0286264589126j))*x[0] + ((-0.590570894519-0.580183912555j))*x[1]**o + ((0.991445588504+0.707495647461j))*x[1]
            ref[(1, 0, 0, 2)]=(-0.464555416722+0.524444482519j)-((-0.240816052598+0.0521278097775j))*(o-1.)
            arg[(1, 0, 1, 0)]=(0.142395053776-0.161796644758j)*x[0]**o + ((-0.703040844379+0.980657850052j))*x[0] + ((-0.871053323515+0.678179298888j))*x[1]**o + ((0.0521656182992+0.577495955561j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.68976674791+1.03726822987j)-((-0.121443044956+0.0860637756884j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.692355496286-0.604175881019j)*x[0]**o + ((0.990973657965+0.27250350221j))*x[0] + ((-0.963463204372-0.133696395442j))*x[1]**o + ((0.45718601998-0.74866732207j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.103829511357-0.607018048161j)-((-0.275969783443-0.122978712744j))*(o-1.)
            arg[(1, 0, 1, 2)]=(0.670531437823-0.656098640281j)*x[0]**o + ((0.289616906368-0.108560606476j))*x[0] + ((-0.855040521436+0.0190114701338j))*x[1]**o + ((0.28583497224-0.813933520876j))*x[1]
            ref[(1, 0, 1, 2)]=(0.195471397497-0.77979064875j)-((-0.0307515139356-0.106181195025j))*(o-1.)
            arg[(1, 0, 2, 0)]=(0.797860628338-0.240290159787j)*x[0]**o + ((-0.547055033524-0.538317710043j))*x[0] + ((-0.23293351514+0.658312503405j))*x[1]**o + ((0.945117338079-0.45358415236j))*x[1]
            ref[(1, 0, 2, 0)]=(0.481494708876-0.286939759392j)-((0.0941545188663+0.069670390603j))*(o-1.)
            arg[(1, 0, 2, 1)]=(0.442015505325-0.157471655542j)*x[0]**o + ((-0.363465678847-0.393228806933j))*x[0] + ((0.514280018521+0.455100753605j))*x[1]**o + ((0.656134478292-0.261208100153j))*x[1]
            ref[(1, 0, 2, 1)]=(0.624482161646-0.178403904511j)-((0.159382587308+0.0496048496772j))*(o-1.)
            arg[(1, 0, 2, 2)]=(0.451294437842+0.279507782711j)*x[0]**o + ((0.154870463822-0.590116748713j))*x[0] + ((-0.336352648125-0.686783151523j))*x[1]**o + ((-0.71053690122+0.997054478996j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.22036232384-0.000168819264776j)-((0.0191569649529-0.0678792281354j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.0210369692061+0.591874781363j)*x[0]**o + ((0.802183419593-0.584547261462j))*x[0] + ((-0.519843917973-0.0922012025853j))*x[1]**o + ((0.601750574067-0.591587131505j))*x[1]
            ref[(1, 1, 0, 0)]=(0.452563522446-0.338230407095j)-((-0.0831344914612+0.0832789297963j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.339929727231+0.933398581061j)*x[0]**o + ((-0.926677058946+0.701653948707j))*x[0] + ((0.753951986807-0.338825828784j))*x[1]**o + ((0.532564551754+0.708988310997j))*x[1]
            ref[(1, 1, 0, 1)]=(0.00995487619207+1.00260750599j)-((0.0690037099293+0.0990954587127j))*(o-1.)
            arg[(1, 1, 0, 2)]=(-0.552444708886+0.593215140818j)*x[0]**o + ((0.818865120819-0.593524312745j))*x[0] + ((-0.561601002744+0.906958432223j))*x[1]**o + ((-0.339059324572+0.987202084764j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.317119957692+0.94692567253j)-((-0.185674285272+0.25002892884j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.250597956679-0.0771463950101j)*x[0]**o + ((-0.488373118099+0.942628828494j))*x[0] + ((0.379001527783-0.172808352837j))*x[1]**o + ((-0.0944070159045+0.750227263856j))*x[1]
            ref[(1, 1, 1, 0)]=(0.0234096752294+0.721450672251j)-((0.10493324741-0.0416591246413j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.57957546409+0.308257264367j)*x[0]**o + ((0.304237516935+0.533204891892j))*x[0] + ((-0.505903635392-0.611829377607j))*x[1]**o + ((-0.372152897921-0.374381814462j))*x[1]
            ref[(1, 1, 1, 1)]=(0.00287822385548-0.0723745179051j)-((0.0122786381163-0.0505953522066j))*(o-1.)
            arg[(1, 1, 1, 2)]=(-0.745117336207+0.865746647419j)*x[0]**o + ((0.798004226033+0.350471640931j))*x[0] + ((0.289387057123-0.946361857047j))*x[1]**o + ((-0.387169862655+0.358450593754j))*x[1]
            ref[(1, 1, 1, 2)]=(-0.0224479578526+0.314153512528j)-((-0.0759550465139-0.0134358682713j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.846873348641+0.0690767261633j)*x[0]**o + ((0.899951821364-0.0166002017717j))*x[0] + ((0.786934772508+0.70405793503j))*x[1]**o + ((-0.239090707792+0.838727995359j))*x[1]
            ref[(1, 1, 2, 0)]=(1.14733461736+0.79763122739j)-((0.272301353525+0.128855776866j))*(o-1.)
            arg[(1, 1, 2, 1)]=(0.655238670764+0.950283268509j)*x[0]**o + ((0.0417347540592+0.00784199490051j))*x[0] + ((-0.495959991433-0.304149035184j))*x[1]**o + ((0.527662508647-0.887791860355j))*x[1]
            ref[(1, 1, 2, 1)]=(0.364337971018-0.116907816065j)-((0.0265464465551+0.107689038887j))*(o-1.)
            arg[(1, 1, 2, 2)]=(0.0467404734812+0.431432850389j)*x[0]**o + ((0.251738500504+0.0814949153023j))*x[0] + ((-0.663435278096-0.514559791267j))*x[1]**o + ((0.864779439398+0.56522414811j))*x[1]
            ref[(1, 1, 2, 2)]=(0.249911567644+0.281796061267j)-((-0.102782467436-0.0138544901465j))*(o-1.)
            arg[(1, 2, 0, 0)]=(-0.398758705629+0.96919688619j)*x[0]**o + ((-0.361941392433-0.393124123221j))*x[0] + ((0.562319912625+0.404267240072j))*x[1]**o + ((0.173003645632+0.464926798308j))*x[1]
            ref[(1, 2, 0, 0)]=(-0.0126882699026+0.722633400674j)-((0.027260201166+0.22891068771j))*(o-1.)
            arg[(1, 2, 0, 1)]=(0.379827035197-0.244887905841j)*x[0]**o + ((0.0854725309781-0.624463767319j))*x[0] + ((-0.894730405429+0.147971752005j))*x[1]**o + ((-0.592360301058-0.526788428214j))*x[1]
            ref[(1, 2, 0, 1)]=(-0.510895570156-0.624084174684j)-((-0.085817228372-0.016152692306j))*(o-1.)
            arg[(1, 2, 0, 2)]=(-0.0376995379454+0.888774089795j)*x[0]**o + ((0.0744235532529-0.193269628228j))*x[0] + ((-0.309724914394+0.80882334715j))*x[1]**o + ((0.504729849095-0.652004699748j))*x[1]
            ref[(1, 2, 0, 2)]=(0.115864475004+0.426161554484j)-((-0.0579040753899+0.282932906158j))*(o-1.)
            arg[(1, 2, 1, 0)]=(0.00538979710782+0.646214277709j)*x[0]**o + ((-0.275466046509-0.541958534314j))*x[0] + ((0.45314801423-0.852942730867j))*x[1]**o + ((-0.16109942908-0.598097662736j))*x[1]
            ref[(1, 2, 1, 0)]=(0.010986167874-0.673392325104j)-((0.0764229685563-0.034454742193j))*(o-1.)
            arg[(1, 2, 1, 1)]=(-0.241346176933-0.297008189928j)*x[0]**o + ((0.892668852219+0.262545292818j))*x[0] + ((-0.317777261671+0.307775898909j))*x[1]**o + ((-0.0494557044219+0.0965711481952j))*x[1]
            ref[(1, 2, 1, 1)]=(0.142044854597+0.184942074997j)-((-0.0931872397673+0.00179461816353j))*(o-1.)
            arg[(1, 2, 1, 2)]=(0.913250656834-0.851183966245j)*x[0]**o + ((-0.438476248448+0.649101100905j))*x[0] + ((-0.772397000195+0.099703898481j))*x[1]**o + ((0.205142941346+0.802106335177j))*x[1]
            ref[(1, 2, 1, 2)]=(-0.0462398252312+0.349863684159j)-((0.0234756094399-0.125246677961j))*(o-1.)
            arg[(1, 2, 2, 0)]=(-0.713219096082+0.729317782038j)*x[0]**o + ((0.942966710353-0.825977204835j))*x[0] + ((-0.666584995956+0.808268167143j))*x[1]**o + ((-0.003284402072+0.607110161662j))*x[1]
            ref[(1, 2, 2, 0)]=(-0.220060891879+0.659359453004j)-((-0.229967348673+0.256264324864j))*(o-1.)
            arg[(1, 2, 2, 1)]=(0.937846640459-0.73375414492j)*x[0]**o + ((0.212197287005-0.328137485814j))*x[0] + ((0.675465131662+0.632031229339j))*x[1]**o + ((0.476888365197+0.317507471974j))*x[1]
            ref[(1, 2, 2, 1)]=(1.15119871216-0.0561764647104j)-((0.268885295353-0.0169538192635j))*(o-1.)
            arg[(1, 2, 2, 2)]=(-0.83004454681-0.582809720883j)*x[0]**o + ((-0.543852497448+0.853123994392j))*x[0] + ((-0.960255471717+0.716299868151j))*x[1]**o + ((0.132066857386-0.339719498889j))*x[1]
            ref[(1, 2, 2, 2)]=(-1.10104282929+0.323447321386j)-((-0.298383336421+0.0222483578781j))*(o-1.)
            arg[(1, 3, 0, 0)]=(-0.797445296796+0.935417038752j)*x[0]**o + ((0.44129415684+0.0207301852981j))*x[0] + ((0.828466808462+0.177234297658j))*x[1]**o + ((-0.583116707796+0.792582428513j))*x[1]
            ref[(1, 3, 0, 0)]=(-0.055400519645+0.96298197511j)-((0.0051702519444+0.185441889402j))*(o-1.)
            arg[(1, 3, 0, 1)]=(-0.497236824605+0.0348904971152j)*x[0]**o + ((-0.0860022848296-0.366927985837j))*x[0] + ((0.418671791657-0.346100946599j))*x[1]**o + ((0.0774231414038-0.0559526063515j))*x[1]
            ref[(1, 3, 0, 1)]=(-0.0435720881868-0.367045520836j)-((-0.013094172158-0.0518684082472j))*(o-1.)
            arg[(1, 3, 0, 2)]=(0.294911588941-0.305469001315j)*x[0]**o + ((0.857366918268-0.463298048154j))*x[0] + ((0.820277415538+0.575856725316j))*x[1]**o + ((0.358856947227-0.121387123745j))*x[1]
            ref[(1, 3, 0, 2)]=(1.16570643499-0.157148723949j)-((0.18586483408+0.0450646206669j))*(o-1.)
            arg[(1, 3, 1, 0)]=(0.748432727233+0.302074872953j)*x[0]**o + ((-0.635029780992+0.84144430465j))*x[0] + ((-0.251013143867-0.305455618776j))*x[1]**o + ((0.273791705788-0.397405752968j))*x[1]
            ref[(1, 3, 1, 0)]=(0.0680907540813+0.220328902929j)-((0.0829032638944-0.000563457637275j))*(o-1.)
            arg[(1, 3, 1, 1)]=(0.982570817989+0.081652771912j)*x[0]**o + ((-0.158469896566-0.0840220078921j))*x[0] + ((-0.00774525374682+0.032303602255j))*x[1]**o + ((0.609741336403-0.589294561932j))*x[1]
            ref[(1, 3, 1, 1)]=(0.71304850204-0.279680097829j)-((0.162470927374+0.0189927290278j))*(o-1.)
            arg[(1, 3, 1, 2)]=(0.317093115611+0.899900905067j)*x[0]**o + ((0.766581223915+0.473221084077j))*x[0] + ((0.0469968214711+0.836826258575j))*x[1]**o + ((-0.926583687024-0.0341229172352j))*x[1]
            ref[(1, 3, 1, 2)]=(0.102043736987+1.08791266524j)-((0.0606816561804+0.289454527274j))*(o-1.)
            arg[(1, 3, 2, 0)]=(0.157186404779-0.539014622349j)*x[0]**o + ((0.703342102477+0.789125035504j))*x[0] + ((-0.586532507719+0.364096821425j))*x[1]**o + ((0.150564517578-0.0232304763523j))*x[1]
            ref[(1, 3, 2, 0)]=(0.212280258558+0.295488379114j)-((-0.0715576838233-0.0291529668206j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.590321738736+0.904114651384j)*x[0]**o + ((-0.503663413633+0.369140985063j))*x[0] + ((-0.114470065565-0.44527404925j))*x[1]**o + ((0.73774523925+0.651460311168j))*x[1]
            ref[(1, 3, 2, 1)]=(-0.235354989342+0.739720949182j)-((-0.117465300717+0.076473433689j))*(o-1.)
            arg[(1, 3, 2, 2)]=(0.165937090552+0.911279488234j)*x[0]**o + ((0.7168036023-0.988907164338j))*x[0] + ((0.0229437825293+0.239079525137j))*x[1]**o + ((0.0321425972058-0.950778261364j))*x[1]
            ref[(1, 3, 2, 2)]=(0.468913536294-0.394663206165j)-((0.0314801455136+0.191726502228j))*(o-1.)
            arg[(2, 0, 0, 0)]=(0.43376916877-0.566187674951j)*x[0]**o + ((-0.852052698541-0.564391578248j))*x[0] + ((0.866997030989-0.582112318699j))*x[1]**o + ((0.227455475065+0.764047852491j))*x[1]
            ref[(2, 0, 0, 0)]=(0.338084488141-0.474321859703j)-((0.216794366626-0.191383332275j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.237877949134-0.365866788985j)*x[0]**o + ((0.165825056865+0.834533799079j))*x[0] + ((-0.598997427744+0.414427513663j))*x[1]**o + ((0.818605654075-0.0312599287624j))*x[1]
            ref[(2, 0, 0, 1)]=(0.0737776670309+0.425917297497j)-((-0.13947922948+0.00809345411309j))*(o-1.)
            arg[(2, 0, 0, 2)]=(0.15378123243-0.500989652434j)*x[0]**o + ((-0.844198219582-0.92122748816j))*x[0] + ((-0.539815265376-0.401922310594j))*x[1]**o + ((-0.794964353803+0.904389500005j))*x[1]
            ref[(2, 0, 0, 2)]=(-1.01259830317-0.459874975591j)-((-0.0643390054911-0.150485327171j))*(o-1.)
            arg[(2, 0, 1, 0)]=(-0.525735379112-0.200855992873j)*x[0]**o + ((0.57011836088+0.638247375163j))*x[0] + ((0.306615550182-0.84414705936j))*x[1]**o + ((-0.653826779465-0.848575984041j))*x[1]
            ref[(2, 0, 1, 0)]=(-0.151414123758-0.627665830555j)-((-0.0365199714882-0.174167175372j))*(o-1.)
            arg[(2, 0, 1, 1)]=(-0.401218099873+0.19757936859j)*x[0]**o + ((0.824082334778+0.944069058801j))*x[0] + ((0.236273106042+0.340442621841j))*x[1]**o + ((-0.931012030537+0.801600459172j))*x[1]
            ref[(2, 0, 1, 1)]=(-0.135937344795+1.1418457542j)-((-0.0274908323052+0.0896703317384j))*(o-1.)
            arg[(2, 0, 1, 2)]=(-0.534060576092-0.957290819071j)*x[0]**o + ((0.373933685718-0.333420055666j))*x[0] + ((0.633203120986+0.837918289308j))*x[1]**o + ((-0.527591562223+0.0896628154245j))*x[1]
            ref[(2, 0, 1, 2)]=(-0.0272576658058-0.181564885002j)-((0.0165237574823-0.0198954216272j))*(o-1.)
            arg[(2, 0, 2, 0)]=(0.186592653359+0.573486386929j)*x[0]**o + ((-0.625736763299+0.397686588601j))*x[0] + ((0.656589398158+0.858395532039j))*x[1]**o + ((0.327438598295-0.986424747155j))*x[1]
            ref[(2, 0, 2, 0)]=(0.272441943257+0.421571880207j)-((0.14053034192+0.238646986495j))*(o-1.)
            arg[(2, 0, 2, 1)]=(0.572017382363-0.467235945965j)*x[0]**o + ((-0.304286073958-0.225667683488j))*x[0] + ((0.320585280264-0.794856739723j))*x[1]**o + ((-0.765088345957+0.62557347842j))*x[1]
            ref[(2, 0, 2, 1)]=(-0.0883858786443-0.431093445378j)-((0.148767110438-0.210348780948j))*(o-1.)
            arg[(2, 0, 2, 2)]=(0.582834979998-0.382486824665j)*x[0]**o + ((0.652734435355-0.814220466634j))*x[0] + ((-0.269708994753+0.864096096075j))*x[1]**o + ((-0.385944882534+0.894561022805j))*x[1]
            ref[(2, 0, 2, 2)]=(0.289957769033+0.28097491379j)-((0.0521876642076+0.0802682119015j))*(o-1.)
            arg[(2, 1, 0, 0)]=(0.0431099836697-0.218598697116j)*x[0]**o + ((-0.881555030926-0.551427017895j))*x[0] + ((0.51676695896-0.108202826368j))*x[1]**o + ((0.646983710118-0.315508203942j))*x[1]
            ref[(2, 1, 0, 0)]=(0.162652810911-0.596868372661j)-((0.0933128237716-0.0544669205806j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.890468736459-0.791369678605j)*x[0]**o + ((0.304339878619-0.542538775376j))*x[0] + ((-0.482882322321+0.0623666194517j))*x[1]**o + ((0.294128242104+0.507637653765j))*x[1]
            ref[(2, 1, 0, 1)]=(-0.387441469029-0.381952090382j)-((-0.22889184313-0.121500509859j))*(o-1.)
            arg[(2, 1, 0, 2)]=(0.871567180979-0.258568562538j)*x[0]**o + ((0.282436812851-0.591938876257j))*x[0] + ((-0.958331148673-0.0407894024853j))*x[1]**o + ((-0.713094083787-0.355853166691j))*x[1]
            ref[(2, 1, 0, 2)]=(-0.258710619315-0.623575003985j)-((-0.0144606612824-0.0498929941705j))*(o-1.)
            arg[(2, 1, 1, 0)]=(0.171372370011-0.99290061862j)*x[0]**o + ((-0.490432437812+0.130577408813j))*x[0] + ((0.337362655756+0.399623208455j))*x[1]**o + ((0.324453757364+0.393684047082j))*x[1]
            ref[(2, 1, 1, 0)]=(0.17137817266-0.0345079771354j)-((0.0847891709612-0.0988795683609j))*(o-1.)
            arg[(2, 1, 1, 1)]=(-0.373496691469-0.676105463326j)*x[0]**o + ((0.234595450039-0.218555522123j))*x[0] + ((0.480908177376+0.0354281200706j))*x[1]**o + ((0.217118547387+0.238466212908j))*x[1]
            ref[(2, 1, 1, 1)]=(0.279562741666-0.310383326235j)-((0.0179019143178-0.106779557209j))*(o-1.)
            arg[(2, 1, 1, 2)]=(0.320906025709-0.466358334561j)*x[0]**o + ((0.936270178152-0.74558861518j))*x[0] + ((-0.749105173572+0.307481193338j))*x[1]**o + ((0.201622035608+0.622243035308j))*x[1]
            ref[(2, 1, 1, 2)]=(0.354846532949-0.141111360547j)-((-0.0713665246439-0.0264795235371j))*(o-1.)
            arg[(2, 1, 2, 0)]=(0.158904970604+0.439003127468j)*x[0]**o + ((-0.543208370736-0.992535470272j))*x[0] + ((-0.728668844943-0.479575172597j))*x[1]**o + ((-0.171078270065-0.56564151517j))*x[1]
            ref[(2, 1, 2, 0)]=(-0.64202525757-0.799374515285j)-((-0.0949606457232-0.00676200752151j))*(o-1.)
            arg[(2, 1, 2, 1)]=(0.973789050724-0.667932024813j)*x[0]**o + ((-0.988044771883+0.53829015566j))*x[0] + ((0.727361388779-0.681182875758j))*x[1]**o + ((-0.230375584306+0.919285728664j))*x[1]
            ref[(2, 1, 2, 1)]=(0.241365041657+0.0542304918764j)-((0.283525073251-0.224852483429j))*(o-1.)
            arg[(2, 1, 2, 2)]=(0.148830489542-0.232366708982j)*x[0]**o + ((0.0787261057989+0.386131527188j))*x[0] + ((0.195333139992+0.118336436765j))*x[1]**o + ((-0.477885983008-0.558359345763j))*x[1]
            ref[(2, 1, 2, 2)]=(-0.0274981238378-0.143129045396j)-((0.0573606049223-0.0190050453695j))*(o-1.)
            arg[(2, 2, 0, 0)]=(-0.093152664682+0.398817330424j)*x[0]**o + ((0.772459086927+0.248455918478j))*x[0] + ((0.195635666099-0.612822177375j))*x[1]**o + ((0.485954328848+0.411572784604j))*x[1]
            ref[(2, 2, 0, 0)]=(0.680448208596+0.223011928065j)-((0.0170805002363-0.0356674744918j))*(o-1.)
            arg[(2, 2, 0, 1)]=(0.976747363939+0.33287739426j)*x[0]**o + ((-0.962819303675+0.304026504326j))*x[0] + ((0.846314774861+0.123910147891j))*x[1]**o + ((0.856718944599-0.903588532946j))*x[1]
            ref[(2, 2, 0, 1)]=(0.858480889862-0.0713872432339j)-((0.3038436898+0.0761312570253j))*(o-1.)
            arg[(2, 2, 0, 2)]=(-0.933218923275-0.970953272191j)*x[0]**o + ((0.083133209818-0.0362584601657j))*x[0] + ((0.53876236735+0.751900473117j))*x[1]**o + ((-0.75981166561+0.959509988643j))*x[1]
            ref[(2, 2, 0, 2)]=(-0.535567505858+0.352099364702j)-((-0.0657427593208-0.0365087998455j))*(o-1.)
            arg[(2, 2, 1, 0)]=(0.189188123122-0.618866163932j)*x[0]**o + ((-0.978796183549+0.166491002461j))*x[0] + ((-0.529644266064-0.229047600008j))*x[1]**o + ((-0.485843318898-0.907429683615j))*x[1]
            ref[(2, 2, 1, 0)]=(-0.902547822694-0.794426222546j)-((-0.0567426904902-0.141318960657j))*(o-1.)
            arg[(2, 2, 1, 1)]=(-0.267670125455+0.127151349167j)*x[0]**o + ((-0.152415881737+0.726032169647j))*x[0] + ((0.673795596084+0.469238908157j))*x[1]**o + ((-0.666023841998+0.848526675793j))*x[1]
            ref[(2, 2, 1, 1)]=(-0.206157126553+1.08547455138j)-((0.0676875784382+0.0993983762206j))*(o-1.)
            arg[(2, 2, 1, 2)]=(0.579710306985+0.512033433517j)*x[0]**o + ((0.670163449341+0.0820736392308j))*x[0] + ((-0.301112413001-0.660165165921j))*x[1]**o + ((-0.0659830492435+0.00580627810947j))*x[1]
            ref[(2, 2, 1, 2)]=(0.441389147041-0.0301259075318j)-((0.0464329823308-0.0246886220673j))*(o-1.)
            arg[(2, 2, 2, 0)]=(-0.970344071833+0.829834381669j)*x[0]**o + ((0.231251808249+0.942234338667j))*x[0] + ((0.414408108323-0.115391269255j))*x[1]**o + ((-0.538692440264+0.0836284072553j))*x[1]
            ref[(2, 2, 2, 0)]=(-0.431688297763+0.870152929168j)-((-0.0926559939184+0.119073852069j))*(o-1.)
            arg[(2, 2, 2, 1)]=(-0.087530377126-0.198317347028j)*x[0]**o + ((-0.437986068421+0.539946615766j))*x[0] + ((-0.468941515084-0.376790531002j))*x[1]**o + ((0.899973797351+0.0702009533532j))*x[1]
            ref[(2, 2, 2, 1)]=(-0.0472420816402+0.0175198455446j)-((-0.0927453153684-0.095851313005j))*(o-1.)
            arg[(2, 2, 2, 2)]=(-0.930853850542-0.137701432612j)*x[0]**o + ((0.780442168754-0.440588322556j))*x[0] + ((0.675490890338-0.337922358652j))*x[1]**o + ((-0.178432154184+0.20221604734j))*x[1]
            ref[(2, 2, 2, 2)]=(0.173323527183-0.35699803324j)-((-0.0425604933674-0.0792706318773j))*(o-1.)
            arg[(2, 3, 0, 0)]=(0.00402872488358-0.520142209733j)*x[0]**o + ((0.918679974913-0.203973941006j))*x[0] + ((-0.470376833841+0.042361506533j))*x[1]**o + ((0.406995176541-0.254911001526j))*x[1]
            ref[(2, 3, 0, 0)]=(0.429663521248-0.468332822866j)-((-0.0777246848263-0.0796301171999j))*(o-1.)
            arg[(2, 3, 0, 1)]=(-0.627274718994+0.257053730826j)*x[0]**o + ((-0.139195557914+0.118781317168j))*x[0] + ((-0.309342484038-0.389858050456j))*x[1]**o + ((0.423485196355+0.942859129807j))*x[1]
            ref[(2, 3, 0, 1)]=(-0.326163782296+0.464418063672j)-((-0.156102867172-0.0221340532717j))*(o-1.)
            arg[(2, 3, 0, 2)]=(-0.543611004559-0.952235458405j)*x[0]**o + ((-0.598181441106+0.0644459713506j))*x[0] + ((0.833660482418+0.294500491465j))*x[1]**o + ((0.260221462906+0.716138644649j))*x[1]
            ref[(2, 3, 0, 2)]=(-0.0239552501704+0.06142482453j)-((0.0483415796432-0.10962249449j))*(o-1.)
            arg[(2, 3, 1, 0)]=(0.799091481788-0.484799819074j)*x[0]**o + ((-0.99285817589-0.589469961342j))*x[0] + ((-0.11437379699+0.924014537864j))*x[1]**o + ((-0.0552195223499+0.521477692688j))*x[1]
            ref[(2, 3, 1, 0)]=(-0.181680006721+0.185611225068j)-((0.114119614133+0.0732024531315j))*(o-1.)
            arg[(2, 3, 1, 1)]=(0.885083115828-0.919525734082j)*x[0]**o + ((0.0223437233979-0.922536861575j))*x[0] + ((0.245987813775-0.0404654477079j))*x[1]**o + ((-0.842952868853-0.777532377045j))*x[1]
            ref[(2, 3, 1, 1)]=(0.155230892074-1.33003021021j)-((0.1885118216-0.159998530298j))*(o-1.)
            arg[(2, 3, 1, 2)]=(-0.30997023109-0.248448132774j)*x[0]**o + ((0.340283431651+0.986727046615j))*x[0] + ((-0.86033556966-0.223400749776j))*x[1]**o + ((-0.792802592309+0.465746620285j))*x[1]
            ref[(2, 3, 1, 2)]=(-0.811412480703+0.490312392175j)-((-0.195050966792-0.0786414804249j))*(o-1.)
            arg[(2, 3, 2, 0)]=(0.523874049-0.961806825659j)*x[0]**o + ((-0.283589752339+0.725599325422j))*x[0] + ((-0.865611287061+0.461735110969j))*x[1]**o + ((-0.197549800488+0.180733292188j))*x[1]
            ref[(2, 3, 2, 0)]=(-0.411438395444+0.20313045146j)-((-0.0569562063435-0.0833452857818j))*(o-1.)
            arg[(2, 3, 2, 1)]=(0.0104832594301+0.623857934857j)*x[0]**o + ((0.0243248165213+0.346398096362j))*x[0] + ((0.486161211632-0.321789380304j))*x[1]**o + ((-0.497362477755-0.979590605601j))*x[1]
            ref[(2, 3, 2, 1)]=(0.0118034049144-0.165561977343j)-((0.0827740785104+0.0503447590922j))*(o-1.)
            arg[(2, 3, 2, 2)]=(0.426294251501-0.768375579946j)*x[0]**o + ((0.683163780355-0.277902545065j))*x[0] + ((-0.0604277231535-0.645590580863j))*x[1]**o + ((0.919378365266+0.503941017609j))*x[1]
            ref[(2, 3, 2, 2)]=(0.984204336984-0.593963844133j)-((0.0609777547246-0.235661026802j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(-0.487771782752+0.990400440382j)*x[0]**o + ((0.182608756173-0.803274573071j))*x[0] + ((-0.709464513668-0.142946514949j))*x[1]**o + ((0.184951221092-0.652912924499j))*x[1] + ((0.903153429329+0.525461248122j))*x[2]**o + ((0.3770877519+0.481109014115j))*x[2]
            ref[(0, 0, 0, 0)]=(0.225282431037+0.198918345051j)-((-0.0490138111818+0.228819195593j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.165585792326+0.734633263921j)*x[0]**o + ((-0.710154978491-0.722860264077j))*x[0] + ((-0.478591165822-0.977704760909j))*x[1]**o + ((-0.988090690146+0.394062694942j))*x[1] + ((-0.543663150956-0.215880284335j))*x[2]**o + ((0.999926838513+0.211198916142j))*x[2]
            ref[(0, 0, 0, 1)]=(-0.943079469614-0.288275217158j)-((-0.197973351517-0.0764919635539j))*(o-1.)
            arg[(0, 0, 0, 2)]=(0.957083643704-0.775619695005j)*x[0]**o + ((-0.590745454894-0.80821314883j))*x[0] + ((-0.806152564222+0.230494504591j))*x[1]**o + ((0.157024245454-0.155388859761j))*x[1] + ((0.902631347613-0.917027616758j))*x[2]**o + ((-0.379913750397-0.0262498724196j))*x[2]
            ref[(0, 0, 0, 2)]=(0.119963733629-1.22600234409j)-((0.175593737849-0.243692134529j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.354765372268-0.985438316916j)*x[0]**o + ((-0.761631786459+0.964985012659j))*x[0] + ((-0.768574779241-0.489468591864j))*x[1]**o + ((0.409556267166+0.0563592587766j))*x[1] + ((-0.843497278197+0.941683286314j))*x[2]**o + ((-0.166423743179-0.287785518646j))*x[2]
            ref[(0, 0, 1, 0)]=(-0.88790297382+0.100167565162j)-((-0.209551114195-0.0888706037443j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.257711609663-0.784982886707j)*x[0]**o + ((0.913578162493-0.386350456044j))*x[0] + ((0.934933922491+0.776898615795j))*x[1]**o + ((0.958631024512+0.673238087356j))*x[1] + ((0.320568966687-0.161286928962j))*x[2]**o + ((0.688538877607-0.859746008828j))*x[2]
            ref[(0, 0, 1, 1)]=(2.03698128173-0.371114788695j)-((0.252202416474-0.0282285333122j))*(o-1.)
            arg[(0, 0, 1, 2)]=(-0.7999383582-0.474595485126j)*x[0]**o + ((0.395843606598+0.145037041706j))*x[0] + ((0.307003875521+0.525528692427j))*x[1]**o + ((-0.831390238749-0.0702856461207j))*x[1] + ((0.525756170923+0.8321149811j))*x[2]**o + ((0.148663359755-0.17314553512j))*x[2]
            ref[(0, 0, 1, 2)]=(-0.127030792076+0.392327024433j)-((0.00547028137406+0.147174698067j))*(o-1.)
            arg[(0, 0, 2, 0)]=(0.470284454805-0.717258272148j)*x[0]**o + ((-0.331145835725-0.595542239271j))*x[0] + ((0.698222777214+0.867102401574j))*x[1]**o + ((-0.917236852491-0.247965643748j))*x[1] + ((0.617880996022+0.695878392075j))*x[2]**o + ((0.126246353064+0.298622069962j))*x[2]
            ref[(0, 0, 2, 0)]=(0.332125946444+0.150418354223j)-((0.29773137134+0.140953753584j))*(o-1.)
            arg[(0, 0, 2, 1)]=(-0.550595495249+0.622028746686j)*x[0]**o + ((0.180139146974+0.0622854949903j))*x[0] + ((0.728587297137-0.563530958809j))*x[1]**o + ((-0.413986689083+0.716264733615j))*x[1] + ((-0.802781268155-0.144126342424j))*x[2]**o + ((0.182925012016-0.406199765683j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.33785599818+0.143360954188j)-((-0.104131577711-0.0142714257578j))*(o-1.)
            arg[(0, 0, 2, 2)]=(0.581431790782+0.993709977465j)*x[0]**o + ((0.589861158616+0.94319108425j))*x[0] + ((0.311433628584-0.349125089473j))*x[1]**o + ((-0.411445281603-0.754181171679j))*x[1] + ((0.97538376942+0.940284409464j))*x[2]**o + ((0.23782645851+0.155494508765j))*x[2]
            ref[(0, 0, 2, 2)]=(1.14224576215+0.964686859396j)-((0.311374864798+0.264144882909j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.323391446346+0.855910181885j)*x[0]**o + ((0.992795581792+0.159123548944j))*x[0] + ((-0.995461807559+0.282484051906j))*x[1]**o + ((0.465893086651-0.814439699105j))*x[1] + ((-0.102819570222-0.137390340406j))*x[2]**o + ((-0.808609313644+0.994314962695j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.385796734664+0.670001352959j)-((-0.236945470688+0.166833982231j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.155379609031-0.880515989527j)*x[0]**o + ((-0.998473577259-0.465166444408j))*x[0] + ((-0.239612809363+0.888792456809j))*x[1]**o + ((0.50392388391+0.457779996867j))*x[1] + ((-0.476697564207-0.700005913777j))*x[2]**o + ((0.447376842261-0.321606082565j))*x[2]
            ref[(0, 1, 0, 1)]=(-0.304051807813-0.510360988301j)-((-0.0934884607564-0.115288241083j))*(o-1.)
            arg[(0, 1, 0, 2)]=(0.647903959192-0.31915215743j)*x[0]**o + ((-0.360890092166+0.889604051619j))*x[0] + ((0.483403959173-0.526656491448j))*x[1]**o + ((0.798738269026-0.402877106804j))*x[1] + ((-0.735068868209-0.790012351205j))*x[2]**o + ((-0.0680574941844-0.771664380391j))*x[2]
            ref[(0, 1, 0, 2)]=(0.383014866416-0.96037921783j)-((0.0660398416927-0.272636833347j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.25730622615+0.81270324261j)*x[0]**o + ((0.74945876014-0.757362325066j))*x[0] + ((0.422590593266-0.968979055103j))*x[1]**o + ((-0.792272161375-0.401253739242j))*x[1] + ((0.3780878191-0.34433886883j))*x[2]**o + ((0.225620318979+0.651358935971j))*x[2]
            ref[(0, 1, 1, 0)]=(0.363089551981-0.50393590483j)-((0.090562031036-0.0834357802206j))*(o-1.)
            arg[(0, 1, 1, 1)]=(0.205054672383+0.546481188433j)*x[0]**o + ((-0.873766882773-0.250876638774j))*x[0] + ((-0.275258922651-0.309940755194j))*x[1]**o + ((0.657078538499-0.338978032289j))*x[1] + ((0.777490363412+0.347853759535j))*x[2]**o + ((-0.285827972004+0.57595910514j))*x[2]
            ref[(0, 1, 1, 1)]=(0.102384898433+0.285249313425j)-((0.117881018857+0.0973990321289j))*(o-1.)
            arg[(0, 1, 1, 2)]=(0.512931577965-0.512961741872j)*x[0]**o + ((-0.0451005776664+0.593333631167j))*x[0] + ((0.611808312917-0.705192217805j))*x[1]**o + ((-0.627656097558-0.444290633589j))*x[1] + ((-0.0207294959112+0.916252247417j))*x[2]**o + ((0.537058331873-0.444119461346j))*x[2]
            ref[(0, 1, 1, 2)]=(0.48415602581-0.298489088014j)-((0.184001732495-0.0503169520433j))*(o-1.)
            arg[(0, 1, 2, 0)]=(-0.565767158597-0.0888278799865j)*x[0]**o + ((0.912433641889+0.358860854877j))*x[0] + ((0.330139761028+0.203678050674j))*x[1]**o + ((0.319520949744-0.196428670639j))*x[1] + ((0.13884446142+0.639739483817j))*x[2]**o + ((0.680340842309-0.513718899247j))*x[2]
            ref[(0, 1, 2, 0)]=(0.907756248896+0.201651469747j)-((-0.0161304893581+0.125764942417j))*(o-1.)
            arg[(0, 1, 2, 1)]=(-0.312591690363-0.687569415507j)*x[0]**o + ((-0.654508578058+0.498994710562j))*x[0] + ((-0.765869139564-0.490580219985j))*x[1]**o + ((0.613164740545+0.33249413273j))*x[1] + ((-0.625771557335-0.976841994112j))*x[2]**o + ((0.796765080682+0.160401581196j))*x[2]
            ref[(0, 1, 2, 1)]=(-0.474405572046-0.581550602558j)-((-0.28403873121-0.359165271601j))*(o-1.)
            arg[(0, 1, 2, 2)]=(0.433626632975-0.884311568479j)*x[0]**o + ((0.411190886317+0.193916293315j))*x[0] + ((-0.328191120372-0.466221261742j))*x[1]**o + ((-0.726950363219+0.219005729444j))*x[1] + ((-0.417495039057-0.476889698778j))*x[2]**o + ((0.062242021925-0.114044076713j))*x[2]
            ref[(0, 1, 2, 2)]=(-0.282788490715-0.764272291476j)-((-0.0520099210756-0.3045704215j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.517949936465-0.886595026095j)*x[0]**o + ((-0.84274684775-0.914350631285j))*x[0] + ((-0.688530887731+0.647877623524j))*x[1]**o + ((-0.15366916575-0.0799005232125j))*x[1] + ((-0.0275903881136+0.393257914443j))*x[2]**o + ((0.439628421018+0.431215244715j))*x[2]
            ref[(0, 2, 0, 0)]=(-0.377479465931-0.204247698955j)-((-0.0330285565633+0.0257567519788j))*(o-1.)
            arg[(0, 2, 0, 1)]=(-0.858358546181-0.43283746961j)*x[0]**o + ((-0.803311017691+0.144766239515j))*x[0] + ((0.913201558027-0.883206251347j))*x[1]**o + ((-0.406922108217-0.287942578423j))*x[1] + ((0.508286228925+0.874005701786j))*x[2]**o + ((-0.907727640024+0.315191572853j))*x[2]
            ref[(0, 2, 0, 1)]=(-0.77741576258-0.135011392613j)-((0.0938548734619-0.0736730031951j))*(o-1.)
            arg[(0, 2, 0, 2)]=(0.792654415906-0.43486092153j)*x[0]**o + ((0.800330388818-0.435927111425j))*x[0] + ((-0.756095452529+0.61239231598j))*x[1]**o + ((0.302539574104-0.554743998788j))*x[1] + ((-0.196006109224-0.183802847288j))*x[2]**o + ((-0.235388044729+0.845170690141j))*x[2]
            ref[(0, 2, 0, 2)]=(0.354017386173-0.0758859364553j)-((-0.0265745243078-0.00104524213965j))*(o-1.)
            arg[(0, 2, 1, 0)]=(0.0788758040246+0.809486311306j)*x[0]**o + ((-0.75230630262+0.570359748288j))*x[0] + ((0.386941568916+0.178154768999j))*x[1]**o + ((0.185341111676+0.664091113195j))*x[1] + ((0.306977737697+0.563224705853j))*x[2]**o + ((0.575216836037-0.314488211022j))*x[2]
            ref[(0, 2, 1, 0)]=(0.390523377865+1.23541421831j)-((0.128799185106+0.258477631026j))*(o-1.)
            arg[(0, 2, 1, 1)]=(-0.113133181261+0.800208327878j)*x[0]**o + ((-0.67938188354+0.0318800408453j))*x[0] + ((-0.652878467814+0.418483163138j))*x[1]**o + ((0.873972098205-0.611018319215j))*x[1] + ((-0.41473407458-0.0132506669833j))*x[2]**o + ((0.682480792344-0.324161661252j))*x[2]
            ref[(0, 2, 1, 1)]=(-0.151837358323+0.151070442205j)-((-0.196790953942+0.200906804005j))*(o-1.)
            arg[(0, 2, 1, 2)]=(-0.823399185489+0.634398403342j)*x[0]**o + ((0.560656114537-0.329384637579j))*x[0] + ((-0.15506585441+0.62755366168j))*x[1]**o + ((0.388703176335+0.791123929358j))*x[1] + ((-0.486898096119+0.213564061318j))*x[2]**o + ((0.322821072289-0.143015641149j))*x[2]
            ref[(0, 2, 1, 2)]=(-0.0965913864282+0.897119888484j)-((-0.244227189336+0.24591935439j))*(o-1.)
            arg[(0, 2, 2, 0)]=(-0.485720504408-0.0949815130234j)*x[0]**o + ((-0.820122425141-0.575005897145j))*x[0] + ((-0.0787780038229-0.882095444505j))*x[1]**o + ((-0.59544941117+0.834435829543j))*x[1] + ((-0.335291741722+0.435237930559j))*x[2]**o + ((0.34671208096-0.252809985552j))*x[2]
            ref[(0, 2, 2, 0)]=(-0.984325002652-0.267609540062j)-((-0.149965041659-0.0903065044948j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.010335865882+0.590314326864j)*x[0]**o + ((0.509293423963+0.323400747789j))*x[0] + ((-0.905161716158+0.344957423971j))*x[1]**o + ((0.548573024265-0.509058489157j))*x[1] + ((0.946604073781-0.692274497511j))*x[2]**o + ((-0.61503780017-0.416673475935j))*x[2]
            ref[(0, 2, 2, 1)]=(0.2369675699-0.17966698199j)-((0.00518441529027+0.0404995422206j))*(o-1.)
            arg[(0, 2, 2, 2)]=(-0.791778631604-0.450012621961j)*x[0]**o + ((0.73224410577-0.488951835903j))*x[0] + ((-0.0443008446305-0.934368874645j))*x[1]**o + ((0.678746593356+0.12115559156j))*x[1] + ((0.0434385779849-0.366668252066j))*x[2]**o + ((-0.0297262707898-0.320222089499j))*x[2]
            ref[(0, 2, 2, 2)]=(0.294311765044-1.21953404126j)-((-0.132106816375-0.291841624779j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.497715661082-0.0800875044233j)*x[0]**o + ((0.4469490519-0.513482283661j))*x[0] + ((-0.501595194737+0.149639378009j))*x[1]**o + ((0.805720873174-0.561861330757j))*x[1] + ((0.662421171677-0.168344757639j))*x[2]**o + ((-0.629206371752-0.998828255272j))*x[2]
            ref[(0, 3, 0, 0)]=(0.641002595672-1.08648237687j)-((0.10975693967-0.0164654806756j))*(o-1.)
            arg[(0, 3, 0, 1)]=(-0.116955418718-0.519937046644j)*x[0]**o + ((0.809071057169-0.824211083288j))*x[0] + ((-0.881506625006-0.0504984597031j))*x[1]**o + ((-0.586359312182-0.870054986637j))*x[1] + ((-0.922516141611+0.113089821003j))*x[2]**o + ((0.503276877023-0.141544632698j))*x[2]
            ref[(0, 3, 0, 1)]=(-0.597494781662-1.14657819398j)-((-0.320163030889-0.0762242808907j))*(o-1.)
            arg[(0, 3, 0, 2)]=(-0.795152191229+0.542279405338j)*x[0]**o + ((-0.365001893592-0.200286007089j))*x[0] + ((-0.0723838460367-0.0640897607675j))*x[1]**o + ((0.13119058481+0.172176989031j))*x[1] + ((-0.414381082749-0.202492382234j))*x[2]**o + ((-0.605147178082+0.150059199676j))*x[2]
            ref[(0, 3, 0, 2)]=(-1.06043780344+0.198823721977j)-((-0.213652853336+0.0459495437227j))*(o-1.)
            arg[(0, 3, 1, 0)]=(-0.469885421267+0.698947575677j)*x[0]**o + ((-0.698281681726-0.929861235485j))*x[0] + ((-0.633880734582+0.281342466034j))*x[1]**o + ((0.338222193755+0.9115102473j))*x[1] + ((-0.645105448089-0.618292130447j))*x[2]**o + ((-0.353520310107+0.232275649855j))*x[2]
            ref[(0, 3, 1, 0)]=(-1.23122570101+0.287961286468j)-((-0.291478600656+0.0603329852107j))*(o-1.)
            arg[(0, 3, 1, 1)]=(-0.868747430947+0.305531021514j)*x[0]**o + ((0.0539803434658-0.504280358663j))*x[0] + ((0.168468356919+0.781219103186j))*x[1]**o + ((-0.499401082743-0.0216329115796j))*x[1] + ((-0.437566663131+0.603358190343j))*x[2]**o + ((0.851872938779-0.067005935189j))*x[2]
            ref[(0, 3, 1, 1)]=(-0.365696768829+0.548594554806j)-((-0.189640956193+0.281684719174j))*(o-1.)
            arg[(0, 3, 1, 2)]=(0.242190675307-0.851825872178j)*x[0]**o + ((0.906713641611-0.845168562205j))*x[0] + ((0.740321811338+0.0720849697962j))*x[1]**o + ((0.0223807031661-0.510917265851j))*x[1] + ((-0.564593715338+0.310344105139j))*x[2]**o + ((-0.253816475647-0.703425632508j))*x[2]
            ref[(0, 3, 1, 2)]=(0.546598320218-1.2644541289j)-((0.0696531285511-0.0782327995405j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.7031076094+0.914305552856j)*x[0]**o + ((-0.707691405736+0.274242944894j))*x[0] + ((0.950379917362+0.192869653669j))*x[1]**o + ((0.117039795151+0.0618471259586j))*x[1] + ((0.787628113033+0.251681664531j))*x[2]**o + ((-0.632556388638+0.116444451343j))*x[2]
            ref[(0, 3, 2, 0)]=(0.608953820285+0.905695696626j)-((0.406852606632+0.226476145176j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.980515434228-0.417373484097j)*x[0]**o + ((-0.421921214643-0.555641792198j))*x[0] + ((0.478831859353+0.297724528408j))*x[1]**o + ((-0.400928187562+0.554158891015j))*x[1] + ((-0.113783450225-0.665391088038j))*x[2]**o + ((0.0387731623764+0.153164499989j))*x[2]
            ref[(0, 3, 2, 1)]=(-0.699771632464-0.316679222461j)-((-0.102577837517-0.130840007288j))*(o-1.)
            arg[(0, 3, 2, 2)]=(-0.251562950391+0.312486135101j)*x[0]**o + ((-0.993911779994-0.168550042163j))*x[0] + ((-0.580317237646+0.489433668502j))*x[1]**o + ((-0.0247258890439-0.105500473585j))*x[1] + ((0.616888653509+0.503950848213j))*x[2]**o + ((-0.904388583162+0.0963631619676j))*x[2]
            ref[(0, 3, 2, 2)]=(-1.06900889336+0.564091649018j)-((-0.0358319224214+0.217645108636j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.255496988339-0.255908542314j)*x[0]**o + ((-0.853165015915-0.316384952306j))*x[0] + ((0.661609289497-0.441356283848j))*x[1]**o + ((-0.455709314666-0.619276181317j))*x[1] + ((0.225204238419-0.927492386128j))*x[2]**o + ((-0.649016492948+0.855940629541j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.663287141977-0.852238858186j)-((0.105219423263-0.270792868715j))*(o-1.)
            arg[(1, 0, 0, 1)]=(0.177467631323+0.647822913099j)*x[0]**o + ((0.0179501391405-0.192973308054j))*x[0] + ((0.933639282498+0.4599719534j))*x[1]**o + ((0.612503607923+0.683274822887j))*x[1] + ((0.855223667034+0.602609257569j))*x[2]**o + ((0.658131761887+0.253222312945j))*x[2]
            ref[(1, 0, 0, 1)]=(1.6274580449+1.22696397592j)-((0.327721763476+0.285067354011j))*(o-1.)
            arg[(1, 0, 0, 2)]=(-0.66856290354+0.314183064802j)*x[0]**o + ((0.595423942355-0.289717884945j))*x[0] + ((0.551122001098+0.881839032921j))*x[1]**o + ((0.653336335219+0.09775760853j))*x[1] + ((-0.197295155872+0.181422877947j))*x[2]**o + ((-0.86513838614+0.724888233883j))*x[2]
            ref[(1, 0, 0, 2)]=(0.0344429165593+0.955186466569j)-((-0.0524560097191+0.229574162612j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.399031285968-0.295995386437j)*x[0]**o + ((0.575231268423-0.171052308853j))*x[0] + ((-0.869078475034-0.705563589618j))*x[1]**o + ((-0.422652743611-0.786523907408j))*x[1] + ((0.0429968021208+0.367525362043j))*x[2]**o + ((0.23015397952+0.700216997445j))*x[2]
            ref[(1, 0, 1, 0)]=(-0.421190227275-0.445696416414j)-((-0.204185493147-0.105672269002j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.298200339401-0.757543214807j)*x[0]**o + ((0.965151162983-0.382834767363j))*x[0] + ((0.199411700249-0.101124658766j))*x[1]**o + ((-0.375593902928+0.538018181743j))*x[1] + ((-0.374430980094+0.711217324275j))*x[2]**o + ((-0.839843317463+0.195639444331j))*x[2]
            ref[(1, 0, 1, 1)]=(-0.361752838327+0.101686154707j)-((-0.0788699365411-0.0245750915496j))*(o-1.)
            arg[(1, 0, 1, 2)]=(-0.0342020912013+0.653527885149j)*x[0]**o + ((0.810187831522+0.250736186881j))*x[0] + ((0.527067896885-0.925242463764j))*x[1]**o + ((-0.283131741345+0.417870314657j))*x[1] + ((-0.456722364618+0.808306510707j))*x[2]**o + ((0.865613206648-0.0239312836451j))*x[2]
            ref[(1, 0, 1, 2)]=(0.714406368945+0.590633574992j)-((0.00602390684432+0.089431988682j))*(o-1.)
            arg[(1, 0, 2, 0)]=(-0.332726959426-0.258168350849j)*x[0]**o + ((-0.281989932035-0.446871047191j))*x[0] + ((0.421006853153-0.996649202757j))*x[1]**o + ((-0.0886567002129+0.249627297683j))*x[1] + ((-0.75032063753-0.13170967825j))*x[2]**o + ((0.0644061859553+0.91666088778j))*x[2]
            ref[(1, 0, 2, 0)]=(-0.484140595048-0.333555046792j)-((-0.110340123967-0.231087871976j))*(o-1.)
            arg[(1, 0, 2, 1)]=(-0.965652642457+0.479802369003j)*x[0]**o + ((-0.312062431522+0.139024995227j))*x[0] + ((0.39374022383+0.218286457595j))*x[1]**o + ((-0.0771806328384-0.152941529139j))*x[1] + ((0.953888783624+0.926777643942j))*x[2]**o + ((-0.490244405126-0.445481500688j))*x[2]
            ref[(1, 0, 2, 1)]=(-0.248755552244+0.582734217971j)-((0.0636627274995+0.270811078424j))*(o-1.)
            arg[(1, 0, 2, 2)]=(-0.135091862714-0.772368221202j)*x[0]**o + ((0.89671866292-0.538484681198j))*x[0] + ((-0.744008162498-0.254024664673j))*x[1]**o + ((-0.216470144207-0.788244049308j))*x[1] + ((-0.778140156035-0.869912397879j))*x[2]**o + ((-0.419751578973-0.767254835037j))*x[2]
            ref[(1, 0, 2, 2)]=(-0.698371620754-1.99514442465j)-((-0.276206696875-0.316050880626j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.346406564369+0.708333636277j)*x[0]**o + ((0.803150963475-0.406580971291j))*x[0] + ((0.224458103034-0.284785685288j))*x[1]**o + ((-0.660031501871+0.265850941826j))*x[1] + ((-0.365701965914-0.913620468724j))*x[2]**o + ((-0.198923927845+0.0420978826554j))*x[2]
            ref[(1, 1, 0, 0)]=(0.0746791176237-0.294352332272j)-((0.0341937835815-0.0816787529559j))*(o-1.)
            arg[(1, 1, 0, 1)]=(0.541265321433+0.770116247963j)*x[0]**o + ((0.225533062643+0.714155436023j))*x[0] + ((0.417226173937+0.662849508127j))*x[1]**o + ((-0.238922602187+0.165846892264j))*x[1] + ((-0.351596775665-0.567815565277j))*x[2]**o + ((0.253205406115-0.249990383891j))*x[2]
            ref[(1, 1, 0, 1)]=(0.423355293138+0.747581067604j)-((0.101149119951+0.144191698469j))*(o-1.)
            arg[(1, 1, 0, 2)]=(0.682649075452+0.36259552713j)*x[0]**o + ((0.586389651977+0.832557927913j))*x[0] + ((0.977627195603+0.930164728061j))*x[1]**o + ((0.271768632958+0.210734367562j))*x[1] + ((-0.167752927674-0.424342010559j))*x[2]**o + ((-0.0250367339423-0.812878777036j))*x[2]
            ref[(1, 1, 0, 2)]=(1.16282244719+0.549415881536j)-((0.248753890563+0.144736374105j))*(o-1.)
            arg[(1, 1, 1, 0)]=(-0.0932671561351-0.917453892033j)*x[0]**o + ((0.18099117041-0.384134077951j))*x[0] + ((-0.0175413612889+0.923567874209j))*x[1]**o + ((0.375176646708-0.156798020016j))*x[1] + ((-0.00483446639649-0.326712733537j))*x[2]**o + ((-0.871206003587-0.432299006415j))*x[2]
            ref[(1, 1, 1, 0)]=(-0.215340585145-0.646914927871j)-((-0.0192738306367-0.0534331252267j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.759569691455+0.474763753894j)*x[0]**o + ((0.0235381813278-0.180713877571j))*x[0] + ((0.883665577699+0.556030713041j))*x[1]**o + ((-0.235489290674-0.210163367044j))*x[1] + ((0.171242629885+0.157918583683j))*x[2]**o + ((-0.252734227087-0.391627114879j))*x[2]
            ref[(1, 1, 1, 1)]=(0.674896281303+0.203104345562j)-((0.302412983173+0.19811884177j))*(o-1.)
            arg[(1, 1, 1, 2)]=(-0.805461862462-0.190072042685j)*x[0]**o + ((0.958581026058-0.722091225031j))*x[0] + ((-0.122624805582-0.447828280998j))*x[1]**o + ((-0.17383107323-0.713359475854j))*x[1] + ((-0.879963787255-0.344755259941j))*x[2]**o + ((-0.132598977213-0.302892740176j))*x[2]
            ref[(1, 1, 1, 2)]=(-0.577949739842-1.36049951234j)-((-0.30134174255-0.163775930604j))*(o-1.)
            arg[(1, 1, 2, 0)]=(0.279138623801-0.794073544161j)*x[0]**o + ((-0.0536442946737+0.766746113317j))*x[0] + ((-0.49686084048-0.255422223013j))*x[1]**o + ((0.455638506524-0.905302187322j))*x[1] + ((0.13018940999-0.887024060374j))*x[2]**o + ((-0.561164928301-0.895007343498j))*x[2]
            ref[(1, 1, 2, 0)]=(-0.12335176157-1.48504162253j)-((-0.014588801115-0.322753304591j))*(o-1.)
            arg[(1, 1, 2, 1)]=(0.784871780604+0.679848514168j)*x[0]**o + ((-0.287090725823-0.190257581515j))*x[0] + ((-0.318888568834-0.56708419793j))*x[1]**o + ((0.781748543122+0.577867325153j))*x[1] + ((0.375758915594-0.0391040450404j))*x[2]**o + ((-0.455979702795+0.0806833786552j))*x[2]
            ref[(1, 1, 2, 1)]=(0.440210120934+0.270976696746j)-((0.140290354561+0.0122767118663j))*(o-1.)
            arg[(1, 1, 2, 2)]=(-0.242787273455-0.198325491031j)*x[0]**o + ((0.507037910961-0.56152074898j))*x[0] + ((-0.993613894821+0.864502930177j))*x[1]**o + ((-0.0696499583889-0.0694571086145j))*x[1] + ((-0.207210495639+0.682669257878j))*x[2]**o + ((0.755830695357+0.0798197159764j))*x[2]
            ref[(1, 1, 2, 2)]=(-0.125196507993+0.398844277703j)-((-0.240601943986+0.224807782837j))*(o-1.)
            arg[(1, 2, 0, 0)]=(0.797580412816+0.184335904972j)*x[0]**o + ((0.0793106550454-0.241346654533j))*x[0] + ((-0.595991641003+0.231414176107j))*x[1]**o + ((0.449160610415-0.730816585394j))*x[1] + ((-0.706745074159+0.946204333627j))*x[2]**o + ((-0.94072208238+0.536303504153j))*x[2]
            ref[(1, 2, 0, 0)]=(-0.458703559633+0.463047339465j)-((-0.0841927170576+0.226992402451j))*(o-1.)
            arg[(1, 2, 0, 1)]=(0.45368567148+0.831989629005j)*x[0]**o + ((-0.426671661233-0.545176977848j))*x[0] + ((0.999908610585-0.0302627949076j))*x[1]**o + ((-0.937813328491-0.581672091895j))*x[1] + ((0.509231661346-0.940202954294j))*x[2]**o + ((0.191805604572+0.604397785839j))*x[2]
            ref[(1, 2, 0, 1)]=(0.39507327913-0.33046370205j)-((0.327137657235-0.0230793533662j))*(o-1.)
            arg[(1, 2, 0, 2)]=(0.441205608806-0.0358927634482j)*x[0]**o + ((0.859307138297-0.858591972711j))*x[0] + ((-0.695946416574-0.146707717545j))*x[1]**o + ((0.328436498757-0.181522693373j))*x[1] + ((-0.840443361902+0.878508947998j))*x[2]**o + ((-0.573223210722-0.702332126498j))*x[2]
            ref[(1, 2, 0, 2)]=(-0.240331871668-0.523269162788j)-((-0.182530694945+0.115984744501j))*(o-1.)
            arg[(1, 2, 1, 0)]=(-0.114858240512-0.335062297696j)*x[0]**o + ((-0.227929957669-0.255549634137j))*x[0] + ((-0.913398126598-0.20878028713j))*x[1]**o + ((0.175076051483+0.444755764196j))*x[1] + ((-0.25427973566+0.157328592558j))*x[2]**o + ((-0.0111448667504+0.323202298972j))*x[2]
            ref[(1, 2, 1, 0)]=(-0.673267437854+0.0629472183818j)-((-0.213756017128-0.0644189987114j))*(o-1.)
            arg[(1, 2, 1, 1)]=(-0.0646229441612+0.412052522601j)*x[0]**o + ((-0.409966851245-0.732976471204j))*x[0] + ((-0.798169030192-0.471995065366j))*x[1]**o + ((-0.113722088462+0.93395258762j))*x[1] + ((-0.749143474009-0.528382179997j))*x[2]**o + ((-0.283270569281-0.0260606892871j))*x[2]
            ref[(1, 2, 1, 1)]=(-1.20944747868-0.206704647817j)-((-0.26865590806-0.0980541204602j))*(o-1.)
            arg[(1, 2, 1, 2)]=(0.980583411036-0.253849576208j)*x[0]**o + ((-0.410449235107+0.804597742133j))*x[0] + ((0.486752704809-0.595188876361j))*x[1]**o + ((0.288077658614-0.231838044219j))*x[1] + ((0.570136228936-0.23919460606j))*x[2]**o + ((-0.458407931487-0.803920835475j))*x[2]
            ref[(1, 2, 1, 2)]=(0.7283464184-0.659697098095j)-((0.33957872413-0.181372176438j))*(o-1.)
            arg[(1, 2, 2, 0)]=(-0.179336190546-0.691752253176j)*x[0]**o + ((-0.181847068891+0.73529601771j))*x[0] + ((0.595433036505-0.187084005383j))*x[1]**o + ((-0.594030472261-0.908040426268j))*x[1] + ((0.476107504777+0.653096230659j))*x[2]**o + ((-0.145162437185-0.0953328451908j))*x[2]
            ref[(1, 2, 2, 0)]=(-0.0144178138002-0.246908640825j)-((0.148700725123-0.0376233379833j))*(o-1.)
            arg[(1, 2, 2, 1)]=(0.569607023751+0.940118876343j)*x[0]**o + ((0.655274169618-0.0499002078729j))*x[0] + ((-0.487668318563+0.225952151841j))*x[1]**o + ((-0.152527294557+0.148434538192j))*x[1] + ((0.331588545231+0.884847342865j))*x[2]**o + ((-0.635881365752-0.949913405549j))*x[2]
            ref[(1, 2, 2, 1)]=(0.140196379864+0.599769647909j)-((0.0689212084032+0.341819728508j))*(o-1.)
            arg[(1, 2, 2, 2)]=(-0.0621408514275+0.653974495729j)*x[0]**o + ((-0.578698824344-0.238595591964j))*x[0] + ((0.820448595789-0.544485779164j))*x[1]**o + ((0.966626467315-0.781278566672j))*x[1] + ((0.732848003249-0.955823302725j))*x[2]**o + ((-0.348003758968-0.218598654308j))*x[2]
            ref[(1, 2, 2, 2)]=(0.765539815806-1.04240369955j)-((0.248525957935-0.14105576436j))*(o-1.)
            arg[(1, 3, 0, 0)]=(-0.573485054996-0.240086943537j)*x[0]**o + ((-0.831236808231+0.978560137672j))*x[0] + ((0.927293624533-0.480745993128j))*x[1]**o + ((0.229342415277-0.977241677466j))*x[1] + ((0.572904159527-0.654739639266j))*x[2]**o + ((-0.327505230954-0.203107750775j))*x[2]
            ref[(1, 3, 0, 0)]=(-0.00134344742136-0.78868093325j)-((0.154452121511-0.229262095988j))*(o-1.)
            arg[(1, 3, 0, 1)]=(0.22719605109+0.358015264937j)*x[0]**o + ((0.4880493442+0.98079779585j))*x[0] + ((0.79258211847+0.0684314660043j))*x[1]**o + ((-0.779503578748+0.188220668198j))*x[1] + ((0.750736288924-0.934992589288j))*x[2]**o + ((-0.879129469767+0.767882277071j))*x[2]
            ref[(1, 3, 0, 1)]=(0.299965377084+0.714177441386j)-((0.295085743081-0.0847576430578j))*(o-1.)
            arg[(1, 3, 0, 2)]=(-0.806632488038-0.224526406787j)*x[0]**o + ((-0.156057996403-0.200909831829j))*x[0] + ((-0.11431102804+0.550127856598j))*x[1]**o + ((-0.0221931771139-0.218525286876j))*x[1] + ((0.13017131846+0.683912767116j))*x[2]**o + ((0.380903375244+0.255149303244j))*x[2]
            ref[(1, 3, 0, 2)]=(-0.294059997945+0.422614200733j)-((-0.13179536627+0.168252369488j))*(o-1.)
            arg[(1, 3, 1, 0)]=(0.949823918676+0.316178331093j)*x[0]**o + ((-0.956994436961+0.369233939539j))*x[0] + ((-0.669706027472+0.235678524062j))*x[1]**o + ((-0.429116004368+0.513972976591j))*x[1] + ((-0.378071739997-0.101913244466j))*x[2]**o + ((0.439786042205-0.651125294781j))*x[2]
            ref[(1, 3, 1, 0)]=(-0.522139123958+0.341012616019j)-((-0.0163256414654+0.0749906017814j))*(o-1.)
            arg[(1, 3, 1, 1)]=(-0.167829281971-0.929581558927j)*x[0]**o + ((-0.729721114213+0.944261603293j))*x[0] + ((0.383481999836-0.0379970386665j))*x[1]**o + ((0.0189400133381-0.919909214391j))*x[1] + ((0.573177027986+0.0939231755004j))*x[2]**o + ((0.0139113906024-0.140370638444j))*x[2]
            ref[(1, 3, 1, 1)]=(0.0459800177893-0.494836835818j)-((0.131471624308-0.145609237015j))*(o-1.)
            arg[(1, 3, 1, 2)]=(0.531636628715+0.254443146922j)*x[0]**o + ((0.372464588236-0.552321700555j))*x[0] + ((-0.597256104039-0.675569157563j))*x[1]**o + ((0.284275437591+0.154005419169j))*x[1] + ((0.443721574037-0.0796975855473j))*x[2]**o + ((0.156663442864-0.541185335948j))*x[2]
            ref[(1, 3, 1, 2)]=(0.595752783702-0.720162606761j)-((0.0630170164522-0.0834705993648j))*(o-1.)
            arg[(1, 3, 2, 0)]=(-0.94002348168+0.844914456699j)*x[0]**o + ((0.525349318761+0.37991644269j))*x[0] + ((-0.616160579747-0.681368686598j))*x[1]**o + ((0.701614576204+0.921975820021j))*x[1] + ((0.00161508683958-0.205564881589j))*x[2]**o + ((0.290601532648+0.244262821061j))*x[2]
            ref[(1, 3, 2, 0)]=(-0.0185017734876+0.752067986142j)-((-0.259094829098-0.00700318524788j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.925886853428-0.687467080771j)*x[0]**o + ((-0.617961085109+0.834445970515j))*x[0] + ((0.240512406538+0.846882375062j))*x[1]**o + ((0.578288331157+0.878500481643j))*x[1] + ((-0.99525473255-0.362820810753j))*x[2]**o + ((-0.308320247898-0.711369635658j))*x[2]
            ref[(1, 3, 2, 1)]=(-1.01431109065+0.399085650019j)-((-0.28010486324-0.0339009194103j))*(o-1.)
            arg[(1, 3, 2, 2)]=(0.439117137893+0.627636228743j)*x[0]**o + ((0.512981819318+0.912091293639j))*x[0] + ((0.760840078081+0.078580796848j))*x[1]**o + ((0.0858598585981+0.433334021561j))*x[1] + ((-0.42965052838-0.120900697873j))*x[2]**o + ((-0.195321936359-0.303406615367j))*x[2]
            ref[(1, 3, 2, 2)]=(0.586913214576+0.813667513776j)-((0.128384447932+0.0975527212864j))*(o-1.)
            arg[(2, 0, 0, 0)]=(0.791935990146+0.931283269697j)*x[0]**o + ((-0.470024929355-0.486691108638j))*x[0] + ((-0.826611117283-0.441998202416j))*x[1]**o + ((0.720480088578-0.735863025015j))*x[1] + ((-0.958129634812-0.118147576315j))*x[2]**o + ((0.430302503525+0.961157683301j))*x[2]
            ref[(2, 0, 0, 0)]=(-0.1560235496+0.0548705203067j)-((-0.165467460325+0.0618562484942j))*(o-1.)
            arg[(2, 0, 0, 1)]=(0.11967471322+0.344043447836j)*x[0]**o + ((-0.969185398217-0.78556043242j))*x[0] + ((0.456966351244-0.342660982297j))*x[1]**o + ((-0.425677679664+0.00369278587531j))*x[1] + ((-0.819825467027-0.553534973049j))*x[2]**o + ((-0.0769858915446-0.557552905324j))*x[2]
            ref[(2, 0, 0, 1)]=(-0.857516685995-0.945786529689j)-((-0.0405307337606-0.0920254179183j))*(o-1.)
            arg[(2, 0, 0, 2)]=(0.0464311313695+0.990981837132j)*x[0]**o + ((0.651814825996+0.822307053331j))*x[0] + ((0.0700558892882+0.0915175398031j))*x[1]**o + ((-0.674742538619+0.321966708205j))*x[1] + ((-0.895118693862-0.153876294184j))*x[2]**o + ((-0.899728385604+0.15242214523j))*x[2]
            ref[(2, 0, 0, 2)]=(-0.850643885715+1.11265949476j)-((-0.129771945534+0.154770513792j))*(o-1.)
            arg[(2, 0, 1, 0)]=(0.21607209183-0.852398328314j)*x[0]**o + ((0.575445709556-0.155033292968j))*x[0] + ((0.210890632137+0.356206087153j))*x[1]**o + ((0.885995687658+0.452056304069j))*x[1] + ((-0.752708120655+0.222000033888j))*x[2]**o + ((0.932914018784-0.0147946604368j))*x[2]
            ref[(2, 0, 1, 0)]=(1.03430500965+0.00401807169544j)-((-0.0542908994481-0.0456987012122j))*(o-1.)
            arg[(2, 0, 1, 1)]=(-0.997769329663+0.763451882455j)*x[0]**o + ((0.360949196025-0.13154910639j))*x[0] + ((-0.904488991686-0.828258974145j))*x[1]**o + ((0.565190805882-0.17992866459j))*x[1] + ((-0.809370872341-0.167845044706j))*x[2]**o + ((0.772494964751-0.511116562733j))*x[2]
            ref[(2, 0, 1, 1)]=(-0.506497113516-0.527623235054j)-((-0.451938198948-0.038775356066j))*(o-1.)
            arg[(2, 0, 1, 2)]=(-0.0932596903304-0.453307741685j)*x[0]**o + ((-0.037603806255+0.371028531038j))*x[0] + ((-0.545033635845+0.333446299303j))*x[1]**o + ((-0.477931110033-0.603323037558j))*x[1] + ((0.7780527859-0.404891073095j))*x[2]**o + ((0.615590036394-0.439551128684j))*x[2]
            ref[(2, 0, 1, 2)]=(0.119907289916-0.598299075341j)-((0.0232932432876-0.0874587525795j))*(o-1.)
            arg[(2, 0, 2, 0)]=(0.472970682482+0.394654040582j)*x[0]**o + ((-0.674684621325-0.531414600605j))*x[0] + ((0.633647851335-0.757796325522j))*x[1]**o + ((-0.455529102303+0.21943374735j))*x[1] + ((0.588764743391+0.00215811692344j))*x[2]**o + ((0.987200101264+0.334576245213j))*x[2]
            ref[(2, 0, 2, 0)]=(0.776184827422-0.16919438803j)-((0.282563879535-0.0601640280029j))*(o-1.)
            arg[(2, 0, 2, 1)]=(-0.703062128151+0.626912281407j)*x[0]**o + ((-0.991605838261+0.998078317264j))*x[0] + ((0.741639369484-0.133934740166j))*x[1]**o + ((-0.621766364515-0.768737819462j))*x[1] + ((-0.983011956354+0.874647910972j))*x[2]**o + ((0.777764156586+0.651635634403j))*x[2]
            ref[(2, 0, 2, 1)]=(-0.890021380605+1.12430079221j)-((-0.157405785837+0.227937575369j))*(o-1.)
            arg[(2, 0, 2, 2)]=(-0.99615163243-0.328450399863j)*x[0]**o + ((0.420204149576+0.134316170323j))*x[0] + ((-0.608996645187+0.824234053048j))*x[1]**o + ((-0.961973629953+0.304834040888j))*x[1] + ((0.412287985992-0.569141244399j))*x[2]**o + ((-0.693425960439+0.147329167449j))*x[2]
            ref[(2, 0, 2, 2)]=(-1.21402786622+0.256560893723j)-((-0.198810048604-0.0122262652024j))*(o-1.)
            arg[(2, 1, 0, 0)]=(-0.862667156578-0.304754200453j)*x[0]**o + ((-0.998752559898+0.0326515484128j))*x[0] + ((-0.597284682129-0.827142149806j))*x[1]**o + ((-0.194832154159+0.139552358774j))*x[1] + ((-0.991266807679-0.383088016315j))*x[2]**o + ((0.324762508837-0.879829691671j))*x[2]
            ref[(2, 1, 0, 0)]=(-1.6600204258-1.11130507553j)-((-0.408536441064-0.252497394429j))*(o-1.)
            arg[(2, 1, 0, 1)]=(0.936949648843-0.663317374973j)*x[0]**o + ((-0.320744062886+0.0265024487785j))*x[0] + ((-0.192323511118+0.0715019879241j))*x[1]**o + ((-0.0388373392598-0.0843701315247j))*x[1] + ((-0.401219234023+0.500956692714j))*x[2]**o + ((0.359296116336-0.911382296604j))*x[2]
            ref[(2, 1, 0, 1)]=(0.171560808947-0.530054336843j)-((0.0572344839504-0.0151431157225j))*(o-1.)
            arg[(2, 1, 0, 2)]=(-0.0754780541675+0.151528538097j)*x[0]**o + ((-0.569859303259+0.677348876386j))*x[0] + ((0.793295897975+0.204077675112j))*x[1]**o + ((0.3158832577+0.366676533156j))*x[1] + ((-0.1859718056-0.535677885076j))*x[2]**o + ((0.739462224154+0.62772257459j))*x[2]
            ref[(2, 1, 0, 2)]=(0.508666108401+0.745838156132j)-((0.0886410063679-0.0300119453112j))*(o-1.)
            arg[(2, 1, 1, 0)]=(0.197933073385-0.581633662809j)*x[0]**o + ((0.463434938879-0.222885541547j))*x[0] + ((-0.609994673339+0.811169482582j))*x[1]**o + ((-0.592979318772+0.463085736889j))*x[1] + ((-0.463941891088-0.157626322956j))*x[2]**o + ((0.548382998868-0.366197534259j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.228582436033-0.02704392105j)-((-0.14600058184+0.0119849161361j))*(o-1.)
            arg[(2, 1, 1, 1)]=(-0.723859109984-0.118548667149j)*x[0]**o + ((-0.205057228119-0.150785622395j))*x[0] + ((-0.246363896784+0.486146959924j))*x[1]**o + ((-0.281501463331-0.812426932633j))*x[1] + ((0.836883079121-0.0799010176375j))*x[2]**o + ((0.641325230965-0.929662099644j))*x[2]
            ref[(2, 1, 1, 1)]=(0.0107133059339-0.802588689767j)-((-0.0222233212745+0.0479495458564j))*(o-1.)
            arg[(2, 1, 1, 2)]=(0.747192090857-0.624292833471j)*x[0]**o + ((-0.56645039779+0.342489325845j))*x[0] + ((0.522634084543-0.699737159016j))*x[1]**o + ((0.422598957831+0.0249070637425j))*x[1] + ((-0.811803997775-0.924045826988j))*x[2]**o + ((-0.646440538204-0.14279520676j))*x[2]
            ref[(2, 1, 1, 2)]=(-0.166134900269-1.01173731832j)-((0.0763370296043-0.374679303246j))*(o-1.)
            arg[(2, 1, 2, 0)]=(0.19225766236-0.637541047452j)*x[0]**o + ((-0.0817297510521-0.726749812373j))*x[0] + ((0.13779170349+0.256563777664j))*x[1]**o + ((-0.716769381806+0.194574950319j))*x[1] + ((0.388865546056-0.194172402992j))*x[2]**o + ((-0.759101575768-0.140892290075j))*x[2]
            ref[(2, 1, 2, 0)]=(-0.41934289836-0.624108412455j)-((0.119819151984-0.0958582787968j))*(o-1.)
            arg[(2, 1, 2, 1)]=(-0.780980587631-0.0495881972674j)*x[0]**o + ((-0.865081489531+0.525502065353j))*x[0] + ((0.84012224693-0.199586537154j))*x[1]**o + ((0.625828349917-0.358731179282j))*x[1] + ((0.478570661805+0.352253051487j))*x[2]**o + ((0.497057532068+0.524611909475j))*x[2]
            ref[(2, 1, 2, 1)]=(0.39775835678+0.397230556306j)-((0.089618720184+0.0171797195109j))*(o-1.)
            arg[(2, 1, 2, 2)]=(-0.480912192903-0.0195403238356j)*x[0]**o + ((0.292153394535+0.395938579778j))*x[0] + ((-0.104361710849-0.390446607612j))*x[1]**o + ((0.139177270736-0.416188740414j))*x[1] + ((0.0394728404843-0.791671278275j))*x[2]**o + ((-0.701730094649+0.992303376409j))*x[2]
            ref[(2, 1, 2, 2)]=(-0.408100246323-0.114802496974j)-((-0.090966843878-0.200276368287j))*(o-1.)
            arg[(2, 2, 0, 0)]=(-0.832998014022+0.639034155139j)*x[0]**o + ((-0.30819866388+0.768859809292j))*x[0] + ((0.642767059837-0.0984576316187j))*x[1]**o + ((-0.124280854375-0.0130720844199j))*x[1] + ((-0.16036386173+0.959137730648j))*x[2]**o + ((0.149164752763-0.446681125328j))*x[2]
            ref[(2, 2, 0, 0)]=(-0.316954790704+0.904410426857j)-((-0.0584324693193+0.249952375695j))*(o-1.)
            arg[(2, 2, 0, 1)]=(-0.103779764535-0.806344522939j)*x[0]**o + ((0.33351382074-0.213224178745j))*x[0] + ((-0.161994846849-0.536263537472j))*x[1]**o + ((-0.637752239358-0.738594566094j))*x[1] + ((0.061100151368+0.767351182402j))*x[2]**o + ((-0.885676837551+0.819273717398j))*x[2]
            ref[(2, 2, 0, 1)]=(-0.697294858092-0.353900952725j)-((-0.0341124100026-0.095876146335j))*(o-1.)
            arg[(2, 2, 0, 2)]=(-0.0910014807028+0.101540763779j)*x[0]**o + ((0.0315246084019-0.247788903457j))*x[0] + ((0.577296639498+0.606878762551j))*x[1]**o + ((-0.318518492517-0.782384111842j))*x[1] + ((-0.180534200761-0.634815348973j))*x[2]**o + ((0.556198568141+0.540380720609j))*x[2]
            ref[(2, 2, 0, 2)]=(0.28748282103-0.208094058667j)-((0.0509601596723+0.0122673628927j))*(o-1.)
            arg[(2, 2, 1, 0)]=(-0.4300115902-0.543563213117j)*x[0]**o + ((-0.752413394678+0.0974934866401j))*x[0] + ((0.42787895548+0.472074939461j))*x[1]**o + ((0.873271583149+0.180212414867j))*x[1] + ((-0.576752861458-0.829180103947j))*x[2]**o + ((0.356833268399+0.128899557365j))*x[2]
            ref[(2, 2, 1, 0)]=(-0.0505970196541-0.247031459365j)-((-0.0964809160297-0.150111396267j))*(o-1.)
            arg[(2, 2, 1, 1)]=(-0.371786823883-0.841881587032j)*x[0]**o + ((-0.166643227205-0.7365526682j))*x[0] + ((0.202112075454+0.786009413796j))*x[1]**o + ((0.878144630403+0.0456215369349j))*x[1] + ((0.640028756859+0.421886983941j))*x[2]**o + ((-0.331520932199-0.611981950284j))*x[2]
            ref[(2, 2, 1, 1)]=(0.425167239714-0.468449135422j)-((0.0783923347384+0.0610024684509j))*(o-1.)
            arg[(2, 2, 1, 2)]=(0.836771405424+0.119644477921j)*x[0]**o + ((0.812260697004+0.71488190042j))*x[0] + ((0.604623838343-0.691735719617j))*x[1]**o + ((0.787961109136+0.807806001191j))*x[1] + ((-0.875216146106-0.696187162386j))*x[2]**o + ((0.82310004439+0.337737948651j))*x[2]
            ref[(2, 2, 1, 2)]=(1.4947504741+0.296073723091j)-((0.0943631829435-0.211379734014j))*(o-1.)
            arg[(2, 2, 2, 0)]=(0.873463651916+0.152551912407j)*x[0]**o + ((0.616551280296-0.763593505716j))*x[0] + ((-0.971688351077+0.213657696467j))*x[1]**o + ((-0.105065998143-0.730914254714j))*x[1] + ((-0.121241953778+0.715203988297j))*x[2]**o + ((-0.748255140455-0.737880339627j))*x[2]
            ref[(2, 2, 2, 0)]=(-0.228118255621-0.575487251443j)-((-0.0365777754899+0.180235599529j))*(o-1.)
            arg[(2, 2, 2, 1)]=(-0.300228931052+0.573956860303j)*x[0]**o + ((0.275732086702-0.716974155443j))*x[0] + ((0.0163158051195-0.598282376261j))*x[1]**o + ((-0.822369173039+0.139720615818j))*x[1] + ((0.633886687068-0.128903048719j))*x[2]**o + ((-0.455641828134-0.557057336509j))*x[2]
            ref[(2, 2, 2, 1)]=(-0.326152676668-0.643769720406j)-((0.0583289268559-0.025538094113j))*(o-1.)
            arg[(2, 2, 2, 2)]=(-0.944167773822+0.183120455302j)*x[0]**o + ((0.679635511381+0.57500842313j))*x[0] + ((-0.020564861751+0.397063342013j))*x[1]**o + ((-0.483352859219-0.138969333606j))*x[1] + ((-0.777031338385+0.889275501561j))*x[2]**o + ((0.781092274291-0.418509252579j))*x[2]
            ref[(2, 2, 2, 2)]=(-0.382194523752+0.743494567911j)-((-0.29029399566+0.244909883146j))*(o-1.)
            arg[(2, 3, 0, 0)]=(-0.478234101233+0.35049706621j)*x[0]**o + ((-0.889840738959+0.780781993087j))*x[0] + ((0.648457387953+0.587438363617j))*x[1]**o + ((0.142581528164+0.0675845035419j))*x[1] + ((0.70379672395-0.894862913056j))*x[2]**o + ((-0.648201416496-0.650871987773j))*x[2]
            ref[(2, 3, 0, 0)]=(-0.260720308311+0.120283512813j)-((0.145670001778+0.00717875279518j))*(o-1.)
            arg[(2, 3, 0, 1)]=(0.266741325007-0.626090688149j)*x[0]**o + ((0.439386419501+0.537636058252j))*x[0] + ((0.948512940168-0.0484057851643j))*x[1]**o + ((-0.847567738317+0.809751793986j))*x[1] + ((0.868694034314-0.776840232657j))*x[2]**o + ((-0.446312115648-0.0289462080405j))*x[2]
            ref[(2, 3, 0, 1)]=(0.614727432513-0.0664475308863j)-((0.347324716582-0.241889450995j))*(o-1.)
            arg[(2, 3, 0, 2)]=(0.526957422213+0.574499184441j)*x[0]**o + ((0.0525708433774-0.187293694246j))*x[0] + ((-0.557575848704-0.409595416857j))*x[1]**o + ((-0.728085279861-0.129658221453j))*x[1] + ((0.82340963343+0.342588992953j))*x[2]**o + ((0.874519152267-0.479711040559j))*x[2]
            ref[(2, 3, 0, 2)]=(0.495897961362-0.144585097861j)-((0.132131867823+0.0845821267561j))*(o-1.)
            arg[(2, 3, 1, 0)]=(0.147083486335+0.201248412116j)*x[0]**o + ((0.851240545675+0.895027170077j))*x[0] + ((-0.357360951023+0.987148575242j))*x[1]**o + ((-0.698055383012+0.373740410021j))*x[1] + ((0.430399981922+0.0607396865534j))*x[2]**o + ((-0.352808377645-0.490378902348j))*x[2]
            ref[(2, 3, 1, 0)]=(0.0102496511258+1.01376267583j)-((0.0366870862056+0.208189445652j))*(o-1.)
            arg[(2, 3, 1, 1)]=(-0.961861262622-0.888306543703j)*x[0]**o + ((-0.874753977083+0.401685558499j))*x[0] + ((-0.388640194909+0.886762527165j))*x[1]**o + ((0.0308593746207+0.684401057669j))*x[1] + ((-0.0577535304389+0.325694884117j))*x[2]**o + ((0.817124174217+0.935549137669j))*x[2]
            ref[(2, 3, 1, 1)]=(-0.717512708107+1.17289331071j)-((-0.234709164662+0.0540251445965j))*(o-1.)
            arg[(2, 3, 1, 2)]=(0.372123975098+0.773557242603j)*x[0]**o + ((-0.578606665314-0.581439990879j))*x[0] + ((0.783812663883-0.412969133904j))*x[1]**o + ((-0.754737176095+0.410875847455j))*x[1] + ((0.517509527777-0.85181859088j))*x[2]**o + ((0.81285330623+0.926235904319j))*x[2]
            ref[(2, 3, 1, 2)]=(0.576477815789+0.132220639357j)-((0.27890769446-0.0818717470302j))*(o-1.)
            arg[(2, 3, 2, 0)]=(-0.64979809732-0.340326131733j)*x[0]**o + ((0.672386009481-0.128300654417j))*x[0] + ((-0.584028924885+0.945122172048j))*x[1]**o + ((0.0540375771124-0.488425945096j))*x[1] + ((0.369998408848+0.641068277632j))*x[2]**o + ((0.626189818918+0.140624071091j))*x[2]
            ref[(2, 3, 2, 0)]=(0.244392396078+0.384880894762j)-((-0.143971435559+0.207644052991j))*(o-1.)
            arg[(2, 3, 2, 1)]=(-0.303456542109-0.174977741065j)*x[0]**o + ((-0.368851103754+0.617447755763j))*x[0] + ((-0.947333627849-0.743525130966j))*x[1]**o + ((-0.802059209519-0.907006015679j))*x[1] + ((-0.578381065612+0.968596075927j))*x[2]**o + ((-0.389669398406+0.652655095073j))*x[2]
            ref[(2, 3, 2, 1)]=(-1.69487547362+0.206595019527j)-((-0.304861872595+0.00834886731598j))*(o-1.)
            arg[(2, 3, 2, 2)]=(0.689371641127-0.640741120718j)*x[0]**o + ((0.399239622126+0.0530459793882j))*x[0] + ((-0.971536701535-0.765595113251j))*x[1]**o + ((0.469615785604-0.670897011025j))*x[1] + ((-0.772786502576+0.434641758472j))*x[2]**o + ((-0.243715183939-0.340579922295j))*x[2]
            ref[(2, 3, 2, 2)]=(-0.214905669596-0.965062714715j)-((-0.175825260497-0.16194907925j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 3, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_Solution_rank0(self):
        """
        tests integral of rank 0 Data on ReducedFunction

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.367483050586-0.351981356665j)*x[0]**o + ((0.786262023815-0.538017056352j))*x[0] + ((-0.857997878028-0.221543552453j))*x[1]**o + ((0.185321199683+0.518050402134j))*x[1]
            ref=(0.240534198029-0.296745781668j)-((-0.0817524712402-0.0955874848529j))*(o-1.)
        else:
            arg=(0.645570471198-0.297561200278j)*x[0]**o + ((0.931724075916+0.362438283714j))*x[0] + ((-0.283339450017+0.288285689817j))*x[1]**o + ((-0.0413481845301+0.00718213279571j))*x[1] + ((0.694540228824-0.875742497323j))*x[2]**o + ((-0.502728041888-0.40433173033j))*x[2]
            ref=(0.722209549752-0.459864660802j)-((0.176128541668-0.147503001297j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_Solution_rank1(self):
        """
        tests integral of rank 1 Data on ReducedFunction

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(3,),w)
        ref=numpy.zeros((3,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.672126547225-0.80787325375j)*x[0]**o + ((-0.156591594338-0.721267875898j))*x[0] + ((-0.400688598107-0.644715381311j))*x[1]**o + ((0.613488823783-0.284322066745j))*x[1]
            ref[(0,)]=(-0.307958957944-1.22908928885j)-((-0.178802524222-0.242098105844j))*(o-1.)
            arg[(1,)]=(0.124240983968-0.036200808654j)*x[0]**o + ((0.480076907698+0.571028167156j))*x[0] + ((0.846096772997-0.559377611028j))*x[1]**o + ((-0.806793890195+0.826826011017j))*x[1]
            ref[(1,)]=(0.321810387234+0.401137879246j)-((0.161722959494-0.0992630699469j))*(o-1.)
            arg[(2,)]=(-0.631140835177+0.591302218579j)*x[0]**o + ((0.500243741621-0.995021170821j))*x[0] + ((0.602402848151+0.438580873247j))*x[1]**o + ((0.0791757679333+0.457332693999j))*x[1]
            ref[(2,)]=(0.275340761264+0.246097307502j)-((-0.00478966450435+0.171647181971j))*(o-1.)
        else:
            arg[(0,)]=(-0.24247460833+0.0836621371211j)*x[0]**o + ((-0.59210720425+0.925065181804j))*x[0] + ((0.64819552836-0.513807437686j))*x[1]**o + ((0.853054516143+0.833771923204j))*x[1] + ((-0.910519522308+0.400127822087j))*x[2]**o + ((-0.560426194487-0.136686557366j))*x[2]
            ref[(0,)]=(-0.402138742437+0.796066534582j)-((-0.0841331003797-0.00500291307963j))*(o-1.)
            arg[(1,)]=(0.236966258965+0.188432135725j)*x[0]**o + ((-0.128030435893+0.0218565561564j))*x[0] + ((-0.917938840823-0.118107330134j))*x[1]**o + ((0.0499366072318+0.34029185628j))*x[1] + ((0.712255057403-0.312968216475j))*x[2]**o + ((-0.812352765541-0.977860794426j))*x[2]
            ref[(1,)]=(-0.429582059329-0.429177896436j)-((0.00521374592412-0.0404405684805j))*(o-1.)
            arg[(2,)]=(-0.329635982767+0.0730411938229j)*x[0]**o + ((-0.682461967636-0.528773974538j))*x[0] + ((0.0199048535149+0.0738571457722j))*x[1]**o + ((-0.837550811414-0.40346848042j))*x[1] + ((-0.905460578625-0.350333421672j))*x[2]**o + ((0.0633767003886+0.708681979204j))*x[2]
            ref[(2,)]=(-1.33591389327-0.213497778916j)-((-0.202531951313-0.0339058470129j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_Solution_rank2(self):
        """
        tests integral of rank 2 Data on ReducedFunction

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 2),w)
        ref=numpy.zeros((2, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.752316713648-0.498119906274j)*x[0]**o + ((0.932530234299-0.427091603874j))*x[0] + ((-0.147465240416+0.0532165782623j))*x[1]**o + ((0.925888594767+0.610647601007j))*x[1]
            ref[(0, 0)]=(1.23163515115-0.130673665439j)-((0.100808578872-0.0741505546686j))*(o-1.)
            arg[(0, 1)]=(0.0556603681674+0.261879133642j)*x[0]**o + ((-0.228906459933+0.951467234991j))*x[0] + ((-0.128192618174-0.83107562346j))*x[1]**o + ((0.572378903279-0.415365649887j))*x[1]
            ref[(0, 1)]=(0.13547009667-0.0165474523569j)-((-0.0120887083345-0.0948660816363j))*(o-1.)
            arg[(1, 0)]=(-0.324078098178-0.242945419187j)*x[0]**o + ((0.273291953382-0.619132992953j))*x[0] + ((-0.366253699493+0.991514955124j))*x[1]**o + ((0.610069649605-0.0145971285914j))*x[1]
            ref[(1, 0)]=(0.096514902658+0.0574197071964j)-((-0.115055299612+0.124761589323j))*(o-1.)
            arg[(1, 1)]=(0.798077326427-0.806763399706j)*x[0]**o + ((-0.380220822744+0.0919758699168j))*x[0] + ((0.393337031758+0.661861362069j))*x[1]**o + ((-0.304609419601-0.678655864752j))*x[1]
            ref[(1, 1)]=(0.25329205792-0.365791016236j)-((0.198569059697-0.0241503396062j))*(o-1.)
        else:
            arg[(0, 0)]=(0.0971627596719-0.176478130399j)*x[0]**o + ((0.77841501423-0.426921551334j))*x[0] + ((0.825405889255+0.448246167772j))*x[1]**o + ((-0.691393234803+0.919056640884j))*x[1] + ((0.415946048469-0.675065891628j))*x[2]**o + ((-0.433026638078-0.929320831213j))*x[2]
            ref[(0, 0)]=(0.496254919372-0.420241797959j)-((0.223085782899-0.0672163090425j))*(o-1.)
            arg[(0, 1)]=(-0.272880854038-0.914763761019j)*x[0]**o + ((-0.318854019914-0.79957263578j))*x[0] + ((-0.32366107069+0.782415143799j))*x[1]**o + ((-0.346017198461-0.661544196785j))*x[1] + ((0.559997495265-0.0537292194714j))*x[2]**o + ((0.476337797868-0.513976066003j))*x[2]
            ref[(0, 1)]=(-0.112538924985-1.08058536763j)-((-0.00609073824383-0.031012972782j))*(o-1.)
            arg[(1, 0)]=(0.897655739152+0.176762274973j)*x[0]**o + ((-0.503806432158+0.301173514996j))*x[0] + ((-0.721167237922+0.288210221332j))*x[1]**o + ((0.989154504204+0.021907323004j))*x[1] + ((0.041089353374-0.838178280741j))*x[2]**o + ((0.305446961414-0.793723611079j))*x[2]
            ref[(1, 0)]=(0.504186444032-0.421924278758j)-((0.0362629757673-0.0622009640727j))*(o-1.)
            arg[(1, 1)]=(-0.921329067885+0.533126206473j)*x[0]**o + ((-0.388148723726-0.901845391026j))*x[0] + ((-0.227651944367+0.927849393105j))*x[1]**o + ((0.148560986616+0.363902390588j))*x[1] + ((-0.126914974677+0.0745625975642j))*x[2]**o + ((-0.264881096547-0.896178844497j))*x[2]
            ref[(1, 1)]=(-0.890182410293+0.0507081761035j)-((-0.212649331155+0.255923032857j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_Solution_rank3(self):
        """
        tests integral of rank 3 Data on ReducedFunction

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 4, 4),w)
        ref=numpy.zeros((2, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.275205810086+0.0297230059982j)*x[0]**o + ((0.51384235455-0.468958981929j))*x[0] + ((0.226798473108+0.344787189437j))*x[1]**o + ((-0.479515321385+0.475715160914j))*x[1]
            ref[(0, 0, 0)]=(0.26816565818+0.19063318721j)-((0.0836673805324+0.062418365906j))*(o-1.)
            arg[(0, 0, 1)]=(0.118730137439+0.086483962365j)*x[0]**o + ((-0.825721823681-0.785636714284j))*x[0] + ((0.654151259847+0.171293972769j))*x[1]**o + ((-0.72243921222-0.70629409466j))*x[1]
            ref[(0, 0, 1)]=(-0.387639819308-0.617076436905j)-((0.128813566214+0.0429629891889j))*(o-1.)
            arg[(0, 0, 2)]=(-0.122952532758+0.25493473149j)*x[0]**o + ((0.423463998908-0.942487860994j))*x[0] + ((-0.706728866271+0.0852078355698j))*x[1]**o + ((-0.880345631521-0.365086178272j))*x[1]
            ref[(0, 0, 2)]=(-0.643281515822-0.483715736103j)-((-0.138280233172+0.0566904278433j))*(o-1.)
            arg[(0, 0, 3)]=(-0.0237430624218-0.940719951187j)*x[0]**o + ((0.0672471365654+0.457305374691j))*x[0] + ((-0.517666860935+0.370424469235j))*x[1]**o + ((0.426948145223+0.803179541242j))*x[1]
            ref[(0, 0, 3)]=(-0.0236073207842+0.34509471699j)-((-0.0902349872261-0.0950492469921j))*(o-1.)
            arg[(0, 1, 0)]=(0.0679674045704-0.421083484099j)*x[0]**o + ((0.519352892713-0.676595054492j))*x[0] + ((0.0691741222181-0.722725962434j))*x[1]**o + ((-0.999086470855+0.374746173178j))*x[1]
            ref[(0, 1, 0)]=(-0.171296025677-0.722829163923j)-((0.0228569211314-0.190634907756j))*(o-1.)
            arg[(0, 1, 1)]=(0.906981291646-0.0569015542056j)*x[0]**o + ((-0.00928145156789-0.317175816527j))*x[0] + ((-0.389634571982-0.786904083867j))*x[1]**o + ((-0.792902899859-0.14155969936j))*x[1]
            ref[(0, 1, 1)]=(-0.142418815881-0.65127057698j)-((0.0862244532774-0.140634273012j))*(o-1.)
            arg[(0, 1, 2)]=(0.299605922695+0.543154149418j)*x[0]**o + ((-0.632372057989-0.945793148839j))*x[0] + ((-0.817402046409-0.0335762667272j))*x[1]**o + ((0.788039111449+0.0754124538895j))*x[1]
            ref[(0, 1, 2)]=(-0.181064535127-0.180401406129j)-((-0.0862993539523+0.0849296471152j))*(o-1.)
            arg[(0, 1, 3)]=(0.380909738643+0.677180277722j)*x[0]**o + ((0.2567138337+0.142989040242j))*x[0] + ((-0.762731093837+0.66087197102j))*x[1]**o + ((0.0106967436233+0.696148199651j))*x[1]
            ref[(0, 1, 3)]=(-0.0572053889356+1.08859474432j)-((-0.0636368925324+0.223008708124j))*(o-1.)
            arg[(0, 2, 0)]=(-0.0327093405407-0.416069513088j)*x[0]**o + ((-0.737481625722+0.3867953184j))*x[0] + ((0.34976985411+0.830058744468j))*x[1]**o + ((0.850050626039-0.910668347259j))*x[1]
            ref[(0, 2, 0)]=(0.214814756943-0.0549418987394j)-((0.0528434189282+0.06899820523j))*(o-1.)
            arg[(0, 2, 1)]=(0.258984240676-0.570256668413j)*x[0]**o + ((0.901218634465+0.138481583915j))*x[0] + ((0.0513110398756+0.529130808083j))*x[1]**o + ((0.138456522075+0.190517739865j))*x[1]
            ref[(0, 2, 1)]=(0.674985218546+0.143936731725j)-((0.0517158800919-0.00685431005505j))*(o-1.)
            arg[(0, 2, 2)]=(-0.744206330641-0.730933963961j)*x[0]**o + ((0.0774985944818+0.545791517896j))*x[0] + ((-0.470101869952+0.73287509539j))*x[1]**o + ((-0.940042035801-0.385475063472j))*x[1]
            ref[(0, 2, 2)]=(-1.03842582096+0.0811287929266j)-((-0.202384700099+0.000323521904836j))*(o-1.)
            arg[(0, 2, 3)]=(-0.582541958678-0.496550208194j)*x[0]**o + ((-0.768872210617+0.899864093044j))*x[0] + ((-0.442641005216+0.576499710988j))*x[1]**o + ((-0.470933908048-0.555400469919j))*x[1]
            ref[(0, 2, 3)]=(-1.13249454128+0.21220656296j)-((-0.170863827316+0.0133249171323j))*(o-1.)
            arg[(0, 3, 0)]=(-0.872738824826+0.0377770799646j)*x[0]**o + ((0.704587436938+0.630170684265j))*x[0] + ((0.202204282333+0.0549856086213j))*x[1]**o + ((0.215682850188-0.528120052864j))*x[1]
            ref[(0, 3, 0)]=(0.124867872317+0.0974066599932j)-((-0.111755757082+0.0154604480976j))*(o-1.)
            arg[(0, 3, 1)]=(0.811179853044+0.102008664439j)*x[0]**o + ((-0.791629621799-0.243394098922j))*x[0] + ((0.784223836334+0.355494000142j))*x[1]**o + ((-0.0457434657656+0.905736088792j))*x[1]
            ref[(0, 3, 1)]=(0.379015300907+0.559922327226j)-((0.265900614896+0.0762504440968j))*(o-1.)
            arg[(0, 3, 2)]=(0.158161987603-0.292206166601j)*x[0]**o + ((0.628469885882-0.762116858364j))*x[0] + ((-0.121520555562+0.679285334057j))*x[1]**o + ((0.554518664729-0.212949508931j))*x[1]
            ref[(0, 3, 2)]=(0.609814991325-0.293993599919j)-((0.0061069053401+0.0645131945759j))*(o-1.)
            arg[(0, 3, 3)]=(-0.347340241137+0.42590877805j)*x[0]**o + ((-0.879753326162-0.42735935191j))*x[0] + ((-0.558871590535+0.104459837519j))*x[1]**o + ((0.00293156863329-0.0583351845982j))*x[1]
            ref[(0, 3, 3)]=(-0.891516794601+0.0223370395307j)-((-0.151035305279+0.0883947692617j))*(o-1.)
            arg[(1, 0, 0)]=(-0.814110076168+0.821266009074j)*x[0]**o + ((-0.829347042431+0.461916003976j))*x[0] + ((-0.798608027946+0.794032631575j))*x[1]**o + ((0.795379662216+0.0208448211799j))*x[1]
            ref[(1, 0, 0)]=(-0.823342742165+1.0490297329j)-((-0.268786350686+0.269216440108j))*(o-1.)
            arg[(1, 0, 1)]=(0.2411495143+0.111429337221j)*x[0]**o + ((0.692252709879+0.649003535752j))*x[0] + ((0.695880662261+0.326965740798j))*x[1]**o + ((0.389834393155+0.0776515765801j))*x[1]
            ref[(1, 0, 1)]=(1.0095586398+0.582525095176j)-((0.156171696094+0.0730658463364j))*(o-1.)
            arg[(1, 0, 2)]=(0.988848912674+0.773645103516j)*x[0]**o + ((-0.000380824959843+0.0663300406896j))*x[0] + ((-0.0424313727303+0.581127898414j))*x[1]**o + ((0.252041691282+0.419232582585j))*x[1]
            ref[(1, 0, 2)]=(0.599039203133+0.920167812602j)-((0.157736256657+0.225795500322j))*(o-1.)
            arg[(1, 0, 3)]=(0.56695981599+0.175018216823j)*x[0]**o + ((0.53283735368-0.48293990156j))*x[0] + ((0.385892415349-0.112985669874j))*x[1]**o + ((-0.134404793416+0.104744568058j))*x[1]
            ref[(1, 0, 3)]=(0.675642395802-0.158081393276j)-((0.158808705223+0.0103387578247j))*(o-1.)
            arg[(1, 1, 0)]=(0.432737002573-0.404026542852j)*x[0]**o + ((-0.0980213686301-0.662875935715j))*x[0] + ((0.701689680682+0.604352902602j))*x[1]**o + ((-0.336918326236-0.299414771198j))*x[1]
            ref[(1, 1, 0)]=(0.349743494194-0.380982173582j)-((0.189071113876+0.033387726625j))*(o-1.)
            arg[(1, 1, 1)]=(-0.13699210573-0.427855170576j)*x[0]**o + ((-0.923464675862+0.152587577602j))*x[0] + ((0.758234127878-0.642751063671j))*x[1]**o + ((-0.586782471992-0.488127446702j))*x[1]
            ref[(1, 1, 1)]=(-0.444502562853-0.703073051673j)-((0.103540337025-0.178434372374j))*(o-1.)
            arg[(1, 1, 2)]=(0.366731451946+0.462211082446j)*x[0]**o + ((0.180258789663+0.72030226205j))*x[0] + ((-0.476928246604-0.282592941022j))*x[1]**o + ((0.691683699029+0.000551728167857j))*x[1]
            ref[(1, 1, 2)]=(0.380872847017+0.450236065821j)-((-0.018366132443+0.029936356904j))*(o-1.)
            arg[(1, 1, 3)]=(0.496838678977+0.695757732027j)*x[0]**o + ((0.197607883846-0.161052351507j))*x[0] + ((0.424881564776-0.0923768161457j))*x[1]**o + ((-0.625743447379-0.365525663469j))*x[1]
            ref[(1, 1, 3)]=(0.24679234011+0.0384014504525j)-((0.153620040626+0.10056348598j))*(o-1.)
            arg[(1, 2, 0)]=(-0.164018378624-0.328036175278j)*x[0]**o + ((-0.556140239847+0.55830582197j))*x[0] + ((-0.777131029329+0.674938785785j))*x[1]**o + ((-0.630491118938-0.135500013852j))*x[1]
            ref[(1, 2, 0)]=(-1.06389038337+0.384854209313j)-((-0.156858234659+0.0578171017512j))*(o-1.)
            arg[(1, 2, 1)]=(0.261006945176-0.528772185687j)*x[0]**o + ((-0.770394494609+0.90285819483j))*x[0] + ((0.0608985549401-0.251264117365j))*x[1]**o + ((-0.542839722392+0.245276867952j))*x[1]
            ref[(1, 2, 1)]=(-0.495664358442+0.184049379865j)-((0.0536509166861-0.130006050509j))*(o-1.)
            arg[(1, 2, 2)]=(-0.281028953665-0.576338272793j)*x[0]**o + ((-0.946782719636-0.929011276189j))*x[0] + ((0.856994165666+0.447760896869j))*x[1]**o + ((-0.144449068796+0.0199137863127j))*x[1]
            ref[(1, 2, 2)]=(-0.257633288216-0.5188374329j)-((0.0959942020001-0.0214295626541j))*(o-1.)
            arg[(1, 2, 3)]=(0.387748467281-0.356190864917j)*x[0]**o + ((-0.946694339131-0.269937443305j))*x[0] + ((-0.728631265111+0.367875507122j))*x[1]**o + ((-0.934369353582-0.923719960119j))*x[1]
            ref[(1, 2, 3)]=(-1.11097324527-0.590986380609j)-((-0.0568137996384+0.00194744036756j))*(o-1.)
            arg[(1, 3, 0)]=(-0.895139288094-0.156605769819j)*x[0]**o + ((0.0668918458362+0.0395051872998j))*x[0] + ((-0.470510894928-0.318033173398j))*x[1]**o + ((-0.831854079875+0.0739972332655j))*x[1]
            ref[(1, 3, 0)]=(-1.06530620853-0.180568261326j)-((-0.227608363837-0.0791064905363j))*(o-1.)
            arg[(1, 3, 1)]=(-0.989094673313+0.907640784277j)*x[0]**o + ((-0.941707603467+0.269550756433j))*x[0] + ((-0.545080326738-0.947907754485j))*x[1]**o + ((-0.678102810448-0.220436771496j))*x[1]
            ref[(1, 3, 1)]=(-1.57699270698+0.00442350736417j)-((-0.255695833342-0.0067111617014j))*(o-1.)
            arg[(1, 3, 2)]=(0.0260037498309-0.00693456734958j)*x[0]**o + ((-0.641193454166+0.521525630206j))*x[0] + ((-0.364466922193-0.629617090633j))*x[1]**o + ((-0.598374905565+0.0493308972202j))*x[1]
            ref[(1, 3, 2)]=(-0.789015766047-0.0328475652778j)-((-0.0564105287271-0.106091942997j))*(o-1.)
            arg[(1, 3, 3)]=(0.520834195507-0.924704262987j)*x[0]**o + ((-0.492244864286-0.625265269275j))*x[0] + ((0.747117061532-0.413797591499j))*x[1]**o + ((0.67657476543+0.580515352266j))*x[1]
            ref[(1, 3, 3)]=(0.726140579092-0.691625885748j)-((0.211325209507-0.223083642414j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(-0.401734869412+0.133712990472j)*x[0]**o + ((0.881672950366-0.383481654487j))*x[0] + ((-0.0417628427736+0.406905343798j))*x[1]**o + ((-0.381836151408-0.30902088641j))*x[1] + ((0.674061148317-0.55875537382j))*x[2]**o + ((0.0403110757959-0.607841582142j))*x[2]
            ref[(0, 0, 0)]=(0.385355655443-0.659240581294j)-((0.0384272393552-0.0030228399249j))*(o-1.)
            arg[(0, 0, 1)]=(0.349632848192+0.888233156092j)*x[0]**o + ((0.499600033765+0.503666398606j))*x[0] + ((0.261412377308+0.57704262656j))*x[1]**o + ((0.286437174085+0.296739680953j))*x[1] + ((-0.764063654504+0.844835848192j))*x[2]**o + ((-0.577751787755-0.802629348997j))*x[2]
            ref[(0, 0, 1)]=(0.0276334955453+1.1539441807j)-((-0.0255030715007+0.385018605141j))*(o-1.)
            arg[(0, 0, 2)]=(-0.498866320109-0.367915558274j)*x[0]**o + ((-0.676809772591-0.947515365931j))*x[0] + ((-0.774875171336+0.213653825847j))*x[1]**o + ((-0.82485156187+0.246490666402j))*x[1] + ((0.349670565913+0.0497773933995j))*x[2]**o + ((-0.989827516232+0.563502974067j))*x[2]
            ref[(0, 0, 2)]=(-1.70777988811-0.121003032245j)-((-0.154011820922-0.0174140565047j))*(o-1.)
            arg[(0, 0, 3)]=(0.958234943902+0.961157274076j)*x[0]**o + ((0.141323219664+0.849442254852j))*x[0] + ((-0.253171168606-0.494413516942j))*x[1]**o + ((-0.894908275135+0.415202789456j))*x[1] + ((-0.336077949851-0.103306552948j))*x[2]**o + ((-0.55447340895+0.839776836291j))*x[2]
            ref[(0, 0, 3)]=(-0.469536319488+1.23392954239j)-((0.0614976375741+0.0605728673643j))*(o-1.)
            arg[(0, 1, 0)]=(0.425730744576-0.795529874983j)*x[0]**o + ((0.519558014622-0.338516449875j))*x[0] + ((0.92195653222-0.631386168288j))*x[1]**o + ((0.0362846299126+0.16249220114j))*x[1] + ((-0.722621956918-0.817710241844j))*x[2]**o + ((0.23055633943+0.18183000996j))*x[2]
            ref[(0, 1, 0)]=(0.705732151922-1.11941026195j)-((0.104177553313-0.374104380853j))*(o-1.)
            arg[(0, 1, 1)]=(-0.606493826631-0.551866397272j)*x[0]**o + ((0.0439660767408-0.420041067094j))*x[0] + ((0.578895411363+0.470523258128j))*x[1]**o + ((-0.748433304136+0.0665659531745j))*x[1] + ((0.845609768185+0.916699935344j))*x[2]**o + ((0.0606654622918-0.381334983966j))*x[2]
            ref[(0, 1, 1)]=(0.0871047939071+0.0502733491566j)-((0.136335225486+0.1392261327j))*(o-1.)
            arg[(0, 1, 2)]=(0.130540625964-0.400781456742j)*x[0]**o + ((0.154562758743-0.345047720575j))*x[0] + ((-0.755395863761-0.520136884006j))*x[1]**o + ((-0.959567667246+0.123687941658j))*x[1] + ((-0.832972357987-0.488711268017j))*x[2]**o + ((0.529403392169-0.794514418426j))*x[2]
            ref[(0, 1, 2)]=(-0.866714556059-1.21275190305j)-((-0.242971265964-0.234938268128j))*(o-1.)
            arg[(0, 1, 3)]=(-0.590412292499+0.00598752483169j)*x[0]**o + ((-0.0614929195338-0.432508474917j))*x[0] + ((0.738177948769-0.0957870169193j))*x[1]**o + ((-0.370805084311-0.251505102698j))*x[1] + ((0.197774319836+0.344728730813j))*x[2]**o + ((-0.896642004158-0.561612623948j))*x[2]
            ref[(0, 1, 3)]=(-0.491700015949-0.495348481419j)-((0.0575899960176+0.0424882064543j))*(o-1.)
            arg[(0, 2, 0)]=(0.405090624486+0.236136619296j)*x[0]**o + ((0.765201026323-0.206810663045j))*x[0] + ((0.900683209919+0.536837495306j))*x[1]**o + ((0.487709096289+0.393918234844j))*x[1] + ((-0.922602506713-0.616581650102j))*x[2]**o + ((0.280160159292-0.941744995796j))*x[2]
            ref[(0, 2, 0)]=(0.958120804799-0.299122479749j)-((0.0638618879488+0.0260654107501j))*(o-1.)
            arg[(0, 2, 1)]=(0.883355612331+0.138534652714j)*x[0]**o + ((0.154334451546-0.0846161121242j))*x[0] + ((-0.0726982309774+0.151247096306j))*x[1]**o + ((-0.0866389249933-0.203759446124j))*x[1] + ((-0.27511190749+0.516758646328j))*x[2]**o + ((0.842529549528-0.21663675025j))*x[2]
            ref[(0, 2, 1)]=(0.722885274972+0.150764043425j)-((0.0892575789774+0.134423399225j))*(o-1.)
            arg[(0, 2, 2)]=(-0.931949412801-0.351933528582j)*x[0]**o + ((-0.362035201047+0.106816805746j))*x[0] + ((0.443397363915+0.011094660583j))*x[1]**o + ((0.474930259669+0.100651046009j))*x[1] + ((0.118763111543+0.871539759411j))*x[2]**o + ((0.309422438663-0.398382411278j))*x[2]
            ref[(0, 2, 2)]=(0.0262642799709+0.169893165944j)-((-0.0616314895571+0.0884501485686j))*(o-1.)
            arg[(0, 2, 3)]=(-0.878072769374-0.440501330577j)*x[0]**o + ((-0.390815623927+0.399098289476j))*x[0] + ((0.848946019019-0.663486649862j))*x[1]**o + ((-0.651058264247-0.255049921307j))*x[1] + ((-0.950864025055-0.524389822987j))*x[2]**o + ((0.975199852166-0.309066960433j))*x[2]
            ref[(0, 2, 3)]=(-0.523332405709-0.896698197845j)-((-0.163331795902-0.271396300571j))*(o-1.)
            arg[(0, 3, 0)]=(-0.0668784350958+0.340780202661j)*x[0]**o + ((0.273069426651-0.647662109979j))*x[0] + ((-0.216976518924+0.366060344663j))*x[1]**o + ((0.358628144713+0.501132153774j))*x[1] + ((-0.589310185068-0.87193185157j))*x[2]**o + ((-0.0120301263212+0.654898033297j))*x[2]
            ref[(0, 3, 0)]=(-0.126748847022+0.171638386423j)-((-0.145527523181-0.0275152173744j))*(o-1.)
            arg[(0, 3, 1)]=(-0.584878098507+0.290439415211j)*x[0]**o + ((0.728734006924+0.149601972099j))*x[0] + ((-0.11041959156-0.565618893205j))*x[1]**o + ((0.918607574923+0.277458084333j))*x[1] + ((0.359801768671-0.975975094624j))*x[2]**o + ((-0.0161223151869-0.132389112685j))*x[2]
            ref[(0, 3, 1)]=(0.647861672632-0.478241814435j)-((-0.0559159868993-0.208525762103j))*(o-1.)
            arg[(0, 3, 2)]=(0.855737513399+0.205010646382j)*x[0]**o + ((-0.862843318819-0.881226856444j))*x[0] + ((-0.805782265148+0.274879748593j))*x[1]**o + ((0.253774230374-0.863819472062j))*x[1] + ((-0.9192000754+0.154245326863j))*x[2]**o + ((0.931669569498-0.549342168235j))*x[2]
            ref[(0, 3, 2)]=(-0.273322173048-0.830126387451j)-((-0.144874137858+0.105689286973j))*(o-1.)
            arg[(0, 3, 3)]=(0.729663244776+0.457629640377j)*x[0]**o + ((-0.376748942652-0.766677102049j))*x[0] + ((-0.119856675733+0.771910549786j))*x[1]**o + ((0.183219552512-0.268690477592j))*x[1] + ((0.797275832051+0.074578830531j))*x[2]**o + ((0.707185116105+0.328103690377j))*x[2]
            ref[(0, 3, 3)]=(0.96036906353+0.298427565715j)-((0.234513733516+0.217353170116j))*(o-1.)
            arg[(1, 0, 0)]=(-0.0643191758074-0.171587687991j)*x[0]**o + ((0.579170797583+0.970570890973j))*x[0] + ((-0.861333667596-0.832168340538j))*x[1]**o + ((-0.415981141398+0.512329384455j))*x[1] + ((-0.978253190151-0.201554791766j))*x[2]**o + ((0.337866928621-0.536964312578j))*x[2]
            ref[(1, 0, 0)]=(-0.701424724374-0.129687428723j)-((-0.317317672259-0.200885136716j))*(o-1.)
            arg[(1, 0, 1)]=(0.871529300977+0.668618435807j)*x[0]**o + ((0.794491718963-0.145662759951j))*x[0] + ((-0.634890054833+0.230203534486j))*x[1]**o + ((-0.00298051359801+0.961889737165j))*x[1] + ((0.676519585046+0.340197308722j))*x[2]**o + ((-0.593231007483-0.831948525937j))*x[2]
            ref[(1, 0, 1)]=(0.555719514536+0.611648865147j)-((0.152193138532+0.206503213169j))*(o-1.)
            arg[(1, 0, 2)]=(-0.345994338156+0.869001599432j)*x[0]**o + ((-0.749161592427+0.343782105234j))*x[0] + ((-0.00742584879243+0.065744843744j))*x[1]**o + ((-0.845155357849+0.982595866439j))*x[1] + ((0.673110876905-0.836711647615j))*x[2]**o + ((-0.283816706065+0.188826867271j))*x[2]
            ref[(1, 0, 2)]=(-0.779221483192+0.806619817252j)-((0.0532817816595+0.0163391325936j))*(o-1.)
            arg[(1, 0, 3)]=(0.840931272089-0.29361642351j)*x[0]**o + ((-0.286948752899-0.838600641197j))*x[0] + ((-0.39977959273+0.36422980269j))*x[1]**o + ((0.158716545705-0.363348103549j))*x[1] + ((-0.798133795688-0.86569208576j))*x[2]**o + ((-0.507276763126-0.48572267692j))*x[2]
            ref[(1, 0, 3)]=(-0.496245543325-1.24137506412j)-((-0.0594970193882-0.132513117763j))*(o-1.)
            arg[(1, 1, 0)]=(0.19437187537+0.740874275467j)*x[0]**o + ((0.709361631473-0.260917472825j))*x[0] + ((0.020517745441+0.669769479967j))*x[1]**o + ((-0.590446449158-0.864763229875j))*x[1] + ((0.583828928594-0.453238609657j))*x[2]**o + ((0.210421833755-0.932462031542j))*x[2]
            ref[(1, 1, 0)]=(0.564027782737-0.550368794233j)-((0.133119758234+0.159567524296j))*(o-1.)
            arg[(1, 1, 1)]=(-0.621374899963-0.671720659358j)*x[0]**o + ((0.885716193263+0.123209933312j))*x[0] + ((-0.576225074511+0.757366256329j))*x[1]**o + ((0.362362828049-0.825201976764j))*x[1] + ((0.835745042685+0.339414549343j))*x[2]**o + ((0.839124874954+0.326167581965j))*x[2]
            ref[(1, 1, 1)]=(0.862674482238+0.0246178424129j)-((-0.0603091552982+0.070843357719j))*(o-1.)
            arg[(1, 1, 2)]=(-0.688514278622+0.975517558099j)*x[0]**o + ((-0.575867883681-0.446932551243j))*x[0] + ((-0.688877072167-0.366760923871j))*x[1]**o + ((0.786218913134-0.745416189581j))*x[1] + ((-0.0773455410559+0.359251896776j))*x[2]**o + ((-0.68680780045+0.345200982328j))*x[2]
            ref[(1, 1, 2)]=(-0.965596831421+0.0604303862537j)-((-0.242456148641+0.161334755167j))*(o-1.)
            arg[(1, 1, 3)]=(-0.456189046437+0.735105611277j)*x[0]**o + ((0.126075193919+0.412482683985j))*x[0] + ((0.783902113219-0.189246227462j))*x[1]**o + ((0.60914115181-0.742993387906j))*x[1] + ((-0.568317397656-0.889175163045j))*x[2]**o + ((-0.113599638931+0.59413976209j))*x[2]
            ref[(1, 1, 3)]=(0.190506187963-0.0398433605309j)-((-0.0401007218122-0.0572192965384j))*(o-1.)
            arg[(1, 2, 0)]=(-0.0222375366763-0.501283022134j)*x[0]**o + ((0.089191297346-0.262994154683j))*x[0] + ((-0.806722927438+0.574018186891j))*x[1]**o + ((0.260145690096+0.479867929672j))*x[1] + ((-0.994115030615-0.49782196276j))*x[2]**o + ((0.243902558378-0.634613664289j))*x[2]
            ref[(1, 2, 0)]=(-0.614917974455-0.421413343652j)-((-0.303845915788-0.0708477996672j))*(o-1.)
            arg[(1, 2, 1)]=(0.305782991559-0.899007495725j)*x[0]**o + ((-0.0179758587062-0.617981013917j))*x[0] + ((0.919738091274-0.132490964572j))*x[1]**o + ((0.0185593420901+0.233450087509j))*x[1] + ((0.270676962225-0.198030452688j))*x[2]**o + ((-0.95141194232-0.663134856172j))*x[2]
            ref[(1, 2, 1)]=(0.272684793061-1.13859734778j)-((0.249366340843-0.204921485498j))*(o-1.)
            arg[(1, 2, 2)]=(-0.255478415064-0.229609624122j)*x[0]**o + ((0.220147327467-0.561200477118j))*x[0] + ((0.0240954308025+0.443406837689j))*x[1]**o + ((0.787227541884-0.622797098279j))*x[1] + ((0.141691597923-0.0341774126645j))*x[2]**o + ((-0.176631976354+0.481382550432j))*x[2]
            ref[(1, 2, 2)]=(0.370525753329-0.261497612032j)-((-0.0149485643898+0.0299366334836j))*(o-1.)
            arg[(1, 2, 3)]=(0.40705270306-0.3688676048j)*x[0]**o + ((0.457579867858+0.401506709427j))*x[0] + ((-0.332482264433-0.0261258025079j))*x[1]**o + ((0.693094578647-0.701567770706j))*x[1] + ((-0.914631630177-0.925883178393j))*x[2]**o + ((0.143516091368-0.223171953494j))*x[2]
            ref[(1, 2, 3)]=(0.227064673161-0.922054800237j)-((-0.140010198592-0.220146097617j))*(o-1.)
            arg[(1, 3, 0)]=(-0.904988745096-0.528256214398j)*x[0]**o + ((0.916886993883+0.699925936658j))*x[0] + ((-0.704630949638+0.149763165413j))*x[1]**o + ((0.435281849084+0.705769681993j))*x[1] + ((0.606683379196-0.52302763452j))*x[2]**o + ((0.364576736119+0.828710980998j))*x[2]
            ref[(1, 3, 0)]=(0.356904631774+0.666442958072j)-((-0.16715605259-0.150253447251j))*(o-1.)
            arg[(1, 3, 1)]=(0.360828488943+0.38694081032j)*x[0]**o + ((0.940356206764-0.4809483609j))*x[0] + ((0.241795211649-0.820590920649j))*x[1]**o + ((0.493353774993-0.0283959265751j))*x[1] + ((-0.110222895347+0.279342894869j))*x[2]**o + ((0.549626546355-0.0515744850732j))*x[2]
            ref[(1, 3, 1)]=(1.23786866668-0.357612994004j)-((0.0820668008742-0.0257178692432j))*(o-1.)
            arg[(1, 3, 2)]=(0.798891570703-0.765111047699j)*x[0]**o + ((0.531812136129+0.508578910144j))*x[0] + ((0.181024056808-0.64203204643j))*x[1]**o + ((-0.971577289891+0.154825594792j))*x[1] + ((-0.839212878432+0.787375326195j))*x[2]**o + ((-0.392925155405+0.468074510929j))*x[2]
            ref[(1, 3, 2)]=(-0.345993780044+0.255855623966j)-((0.0234504581798-0.103294627989j))*(o-1.)
            arg[(1, 3, 3)]=(0.605911477289+0.417713783439j)*x[0]**o + ((0.91851210423-0.762614379973j))*x[0] + ((-0.901808973626+0.798757408792j))*x[1]**o + ((0.976352474909+0.791835111905j))*x[1] + ((0.671267149908-0.227590556846j))*x[2]**o + ((-0.99171171157+0.908074190785j))*x[2]
            ref[(1, 3, 3)]=(0.639261260569+0.963087779051j)-((0.0625616089283+0.164813439231j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_Solution_rank4(self):
        """
        tests integral of rank 4 Data on ReducedFunction

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 3, 4),w)
        ref=numpy.zeros((3, 4, 3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.590972508684-0.822450732153j)*x[0]**o + ((0.208953921563-0.694248039721j))*x[0] + ((0.221871190203+0.955174614773j))*x[1]**o + ((-0.638719063152-0.777407160252j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.399433230035-0.669465658676j)-((-0.0615168864135+0.0221206471033j))*(o-1.)
            arg[(0, 0, 0, 1)]=(-0.280236491134+0.476660355271j)*x[0]**o + ((0.196599521786-0.907450387207j))*x[0] + ((-0.892860061536-0.130474843427j))*x[1]**o + ((0.450674734773+0.90927613887j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.262911148055+0.174005631753j)-((-0.195516092112+0.0576975853074j))*(o-1.)
            arg[(0, 0, 0, 2)]=(0.690373586249-0.630182807555j)*x[0]**o + ((-0.351256082561-0.612896236662j))*x[0] + ((-0.799698084037-0.950031329505j))*x[1]**o + ((-0.645526056655-0.987205461621j))*x[1]
            ref[(0, 0, 0, 2)]=(-0.553053318502-1.59015791767j)-((-0.0182207496312-0.263369022843j))*(o-1.)
            arg[(0, 0, 0, 3)]=(-0.329941894199-0.574247303746j)*x[0]**o + ((0.716229379859+0.388282195564j))*x[0] + ((0.64722816344+0.214556676181j))*x[1]**o + ((-0.112910526458+0.322898959272j))*x[1]
            ref[(0, 0, 0, 3)]=(0.460302561321+0.175745263636j)-((0.0528810448734-0.0599484379275j))*(o-1.)
            arg[(0, 0, 1, 0)]=(-0.273475033147-0.647691894633j)*x[0]**o + ((-0.645781599768-0.877437498527j))*x[0] + ((-0.755578213043+0.15857646112j))*x[1]**o + ((-0.921815620747-0.459848386085j))*x[1]
            ref[(0, 0, 1, 0)]=(-1.29832523335-0.913200659062j)-((-0.171508874365-0.0815192389189j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.0693925474167+0.439051549336j)*x[0]**o + ((0.0920831201352-0.278658516137j))*x[0] + ((-0.51126311732+0.52267107054j))*x[1]**o + ((-0.986789980459-0.650346168231j))*x[1]
            ref[(0, 0, 1, 1)]=(-0.668288715114+0.0163589677537j)-((-0.073645094984+0.160287103313j))*(o-1.)
            arg[(0, 0, 1, 2)]=(-0.555290519702-0.561360499752j)*x[0]**o + ((0.726179900006-0.914174331628j))*x[0] + ((-0.134848339371-0.566077682573j))*x[1]**o + ((0.373931334982+0.0210745970819j))*x[1]
            ref[(0, 0, 1, 2)]=(0.204986187958-1.01026895844j)-((-0.115023143179-0.187906363721j))*(o-1.)
            arg[(0, 0, 1, 3)]=(-0.692203405097-0.318793018997j)*x[0]**o + ((0.215101541276-0.519795637036j))*x[0] + ((0.342328439182+0.312396599647j))*x[1]**o + ((-0.566709321203+0.772662056524j))*x[1]
            ref[(0, 0, 1, 3)]=(-0.350741372921+0.123235000069j)-((-0.0583124943193-0.00106606989169j))*(o-1.)
            arg[(0, 0, 2, 0)]=(0.443867585929+0.995208086004j)*x[0]**o + ((-0.748261570886+0.766614425625j))*x[0] + ((-0.600592078715-0.611735176714j))*x[1]**o + ((-0.00182072183282+0.250200039241j))*x[1]
            ref[(0, 0, 2, 0)]=(-0.453403392753+0.700143687078j)-((-0.0261207487977+0.0639121515482j))*(o-1.)
            arg[(0, 0, 2, 1)]=(-0.16901258052-0.965953538828j)*x[0]**o + ((-0.0556830763216+0.328818893199j))*x[0] + ((-0.73646030283+0.179133954161j))*x[1]**o + ((0.574222861551-0.393067972697j))*x[1]
            ref[(0, 0, 2, 1)]=(-0.193466549061-0.425534332082j)-((-0.150912147225-0.131136597444j))*(o-1.)
            arg[(0, 0, 2, 2)]=(0.0151316896525-0.359492437725j)*x[0]**o + ((0.287087075669+0.616300577036j))*x[0] + ((-0.455169034418-0.630211334758j))*x[1]**o + ((-0.825425882645+0.837004150652j))*x[1]
            ref[(0, 0, 2, 2)]=(-0.489188075871+0.231800477603j)-((-0.073339557461-0.164950628747j))*(o-1.)
            arg[(0, 0, 2, 3)]=(-0.192293801854+0.86807391635j)*x[0]**o + ((-0.132259454265+0.263565261862j))*x[0] + ((0.798618265683+0.0271989508749j))*x[1]**o + ((0.863553485702-0.231166967307j))*x[1]
            ref[(0, 0, 2, 3)]=(0.668809247634+0.46383558089j)-((0.101054077305+0.149212144538j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.301524708509+0.544783262704j)*x[0]**o + ((-0.183668298134+0.134039326062j))*x[0] + ((0.392439017436+0.722669963959j))*x[1]**o + ((-0.116983618525+0.701791204352j))*x[1]
            ref[(0, 1, 0, 0)]=(-0.104868803866+1.05164187854j)-((0.0151523848212+0.211242204444j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.0577999514479+0.554123040913j)*x[0]**o + ((-0.4388916663+0.456522106016j))*x[0] + ((-0.496185637341-0.569144952454j))*x[1]**o + ((-0.798198044658-0.901936403071j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.837737698425-0.230218104298j)-((-0.0730642809821-0.00250365192343j))*(o-1.)
            arg[(0, 1, 0, 2)]=(0.244348075208+0.450661384618j)*x[0]**o + ((-0.538598561994-0.232743790518j))*x[0] + ((-0.702999478417+0.572030725825j))*x[1]**o + ((-0.98879345458-0.535082212752j))*x[1]
            ref[(0, 1, 0, 2)]=(-0.993021709892+0.127433053586j)-((-0.0764419005349+0.170448685074j))*(o-1.)
            arg[(0, 1, 0, 3)]=(0.0527322613762+0.0134977719832j)*x[0]**o + ((-0.342543860356-0.0852022513124j))*x[0] + ((0.899760575312-0.422234108773j))*x[1]**o + ((0.803304082881-0.340495607122j))*x[1]
            ref[(0, 1, 0, 3)]=(0.706626529607-0.417217097612j)-((0.158748806115-0.0681227227983j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.106163387477+0.567045586519j)*x[0]**o + ((0.66675790806+0.491132470636j))*x[0] + ((0.726238064331-0.289295367974j))*x[1]**o + ((-0.998268140248-0.111766681491j))*x[1]
            ref[(0, 1, 1, 0)]=(0.144282222333+0.328558003845j)-((0.103345779476+0.0462917030907j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.993622474745-0.727562485074j)*x[0]**o + ((0.706293189663-0.510515903588j))*x[0] + ((-0.662913268961+0.894456245225j))*x[1]**o + ((0.953185054575+0.12339319555j))*x[1]
            ref[(0, 1, 1, 1)]=(0.0014712502657-0.110114473944j)-((-0.276089290618+0.0278156266918j))*(o-1.)
            arg[(0, 1, 1, 2)]=(-0.258344250641+0.82023109646j)*x[0]**o + ((0.781189328073-0.197639877958j))*x[0] + ((-0.00802802325832-0.289031934972j))*x[1]**o + ((-0.903691695816+0.264806210773j))*x[1]
            ref[(0, 1, 1, 2)]=(-0.194437320821+0.299182747151j)-((-0.0443953789832+0.0885331935812j))*(o-1.)
            arg[(0, 1, 1, 3)]=(-0.792428409448+0.746623600479j)*x[0]**o + ((0.778982421572-0.15844747307j))*x[0] + ((-0.390973015533-0.370693724548j))*x[1]**o + ((0.667402919258+0.934859335987j))*x[1]
            ref[(0, 1, 1, 3)]=(0.131491957925+0.576170869424j)-((-0.19723357083+0.0626549793218j))*(o-1.)
            arg[(0, 1, 2, 0)]=(0.727572070514-0.961311078024j)*x[0]**o + ((0.0463866001966+0.976267541557j))*x[0] + ((-0.960560745488+0.709570169958j))*x[1]**o + ((0.83341173377-0.581648717635j))*x[1]
            ref[(0, 1, 2, 0)]=(0.323404829496+0.0714389579283j)-((-0.0388314458291-0.0419568180109j))*(o-1.)
            arg[(0, 1, 2, 1)]=(0.365750718659+0.256619474196j)*x[0]**o + ((0.340246560974+0.734365102286j))*x[0] + ((0.500442190599-0.458709063136j))*x[1]**o + ((-0.474565650014-0.258522453209j))*x[1]
            ref[(0, 1, 2, 1)]=(0.365936910109+0.136876530069j)-((0.144365484876-0.0336815981566j))*(o-1.)
            arg[(0, 1, 2, 2)]=(-0.628628087192-0.25217504208j)*x[0]**o + ((0.420444137519+0.557546136168j))*x[0] + ((-0.0775672570496+0.12419568553j))*x[1]**o + ((0.0513939344826+0.475058764095j))*x[1]
            ref[(0, 1, 2, 2)]=(-0.11717863612+0.452312771857j)-((-0.11769922404-0.0213298927583j))*(o-1.)
            arg[(0, 1, 2, 3)]=(0.256788456674+0.195035346638j)*x[0]**o + ((-0.570188611288-0.643110854273j))*x[0] + ((0.608587116733-0.991782426934j))*x[1]**o + ((0.636243880675+0.0765082224796j))*x[1]
            ref[(0, 1, 2, 3)]=(0.465715421396-0.681674856045j)-((0.144229262234-0.132791180049j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.21257844534-0.216456051664j)*x[0]**o + ((0.525453430182+0.49915087931j))*x[0] + ((-0.0650482788096+0.83901216145j))*x[1]**o + ((-0.823428323883-0.642572920971j))*x[1]
            ref[(0, 2, 0, 0)]=(-0.0752223635856+0.239567034062j)-((0.0245883610884+0.103759351631j))*(o-1.)
            arg[(0, 2, 0, 1)]=(-0.911044742499-0.80038553065j)*x[0]**o + ((0.490279485401-0.770913277338j))*x[0] + ((-0.95155490394-0.619888411165j))*x[1]**o + ((0.158011232301-0.44062406578j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.607154464369-1.31590564247j)-((-0.310433274407-0.236712323636j))*(o-1.)
            arg[(0, 2, 0, 2)]=(-0.131281704919+0.977523466499j)*x[0]**o + ((0.543531608778-0.905177508631j))*x[0] + ((-0.253204946968-0.76717555276j))*x[1]**o + ((0.0247841287708+0.337137268594j))*x[1]
            ref[(0, 2, 0, 2)]=(0.0919145428305-0.178846163149j)-((-0.0640811086479+0.0350579856232j))*(o-1.)
            arg[(0, 2, 0, 3)]=(-0.609385058045-0.592030910538j)*x[0]**o + ((0.975225370259+0.806750023211j))*x[0] + ((0.744019693767+0.0696235807047j))*x[1]**o + ((-0.328545741189-0.675681763363j))*x[1]
            ref[(0, 2, 0, 3)]=(0.390657132396-0.195669534992j)-((0.0224391059537-0.0870678883055j))*(o-1.)
            arg[(0, 2, 1, 0)]=(0.497126035092+0.236613573802j)*x[0]**o + ((-0.156633781395+0.890473560045j))*x[0] + ((-0.252655859266+0.805864691371j))*x[1]**o + ((-0.473910075816-0.605748084688j))*x[1]
            ref[(0, 2, 1, 0)]=(-0.193036840693+0.663601870265j)-((0.0407450293043+0.173746377529j))*(o-1.)
            arg[(0, 2, 1, 1)]=(-0.164842614898-0.420044999335j)*x[0]**o + ((-0.570205969403+0.500715400592j))*x[0] + ((-0.151087820541+0.473315150228j))*x[1]**o + ((0.87998342944+0.731792101069j))*x[1]
            ref[(0, 2, 1, 1)]=(-0.0030764877011+0.642888826278j)-((-0.0526550725731+0.00887835848224j))*(o-1.)
            arg[(0, 2, 1, 2)]=(-0.511869831858-0.533852454717j)*x[0]**o + ((0.49510787568+0.511826872278j))*x[0] + ((0.236669033721+0.136966664225j))*x[1]**o + ((-0.473135258526-0.192204331162j))*x[1]
            ref[(0, 2, 1, 2)]=(-0.126614090491-0.0386316246877j)-((-0.0458667996895-0.0661476317487j))*(o-1.)
            arg[(0, 2, 1, 3)]=(0.00255094241351-0.275998884839j)*x[0]**o + ((-0.481001064986+0.0465841303206j))*x[0] + ((-0.110642401302-0.42401488921j))*x[1]**o + ((-0.411402601182+0.685450152691j))*x[1]
            ref[(0, 2, 1, 3)]=(-0.500247562528+0.0160102544809j)-((-0.0180152431481-0.116668962342j))*(o-1.)
            arg[(0, 2, 2, 0)]=(-0.653822880255-0.93130700421j)*x[0]**o + ((-0.567073409746+0.638853187277j))*x[0] + ((-0.0641717239666-0.321437924969j))*x[1]**o + ((0.286384467651+0.992665816919j))*x[1]
            ref[(0, 2, 2, 0)]=(-0.499341773158+0.189387037509j)-((-0.11966576737-0.20879082153j))*(o-1.)
            arg[(0, 2, 2, 1)]=(0.670256768485+0.477549929525j)*x[0]**o + ((-0.603058158058+0.21563967805j))*x[0] + ((0.999574899552+0.872258792209j))*x[1]**o + ((-0.983974531432+0.325793651952j))*x[1]
            ref[(0, 2, 2, 1)]=(0.0413994892736+0.945621025868j)-((0.278305278006+0.224968120289j))*(o-1.)
            arg[(0, 2, 2, 2)]=(-0.435230886678-0.83897412114j)*x[0]**o + ((0.205684839097+0.500681906335j))*x[0] + ((0.392370697027-0.790429007851j))*x[1]**o + ((-0.305257441202+0.673177647234j))*x[1]
            ref[(0, 2, 2, 2)]=(-0.0712163958784-0.227771787711j)-((-0.00714336494192-0.271567188165j))*(o-1.)
            arg[(0, 2, 2, 3)]=(0.656130382364+0.442646747193j)*x[0]**o + ((0.438353218387-0.268242717472j))*x[0] + ((-0.992301797807-0.63452325957j))*x[1]**o + ((0.350499347238-0.521065068446j))*x[1]
            ref[(0, 2, 2, 3)]=(0.226340575091-0.490592149148j)-((-0.0560285692404-0.0319794187295j))*(o-1.)
            arg[(0, 3, 0, 0)]=(-0.644621825903-0.796785734733j)*x[0]**o + ((0.159223517429-0.865706144619j))*x[0] + ((-0.944633012465+0.953587075128j))*x[1]**o + ((-0.457564014144-0.605465535433j))*x[1]
            ref[(0, 3, 0, 0)]=(-0.943797667542-0.657185169829j)-((-0.264875806395+0.0261335567325j))*(o-1.)
            arg[(0, 3, 0, 1)]=(0.256006360031-0.381518883019j)*x[0]**o + ((-0.695903705435+0.205000589189j))*x[0] + ((0.357933535566-0.925341970533j))*x[1]**o + ((0.494649933419+0.639877093198j))*x[1]
            ref[(0, 3, 0, 1)]=(0.20634306179-0.230991585583j)-((0.102323315933-0.217810142259j))*(o-1.)
            arg[(0, 3, 0, 2)]=(0.365617835361-0.861765914459j)*x[0]**o + ((-0.755853852811+0.469444106737j))*x[0] + ((-0.413040193782-0.0084044599126j))*x[1]**o + ((0.339949934282+0.649077293883j))*x[1]
            ref[(0, 3, 0, 2)]=(-0.231663138475+0.124175513124j)-((-0.00790372640338-0.145028395729j))*(o-1.)
            arg[(0, 3, 0, 3)]=(0.78514446585-0.488310753779j)*x[0]**o + ((0.850866164669+0.413295010975j))*x[0] + ((0.347607500227+0.4258920733j))*x[1]**o + ((-0.752449876286-0.491922481814j))*x[1]
            ref[(0, 3, 0, 3)]=(0.61558412723-0.0705230756593j)-((0.188791994346-0.0104031134132j))*(o-1.)
            arg[(0, 3, 1, 0)]=(0.674501733225+0.40657453438j)*x[0]**o + ((-0.103091121493+0.726698184175j))*x[0] + ((0.69823177329+0.889128151922j))*x[1]**o + ((-0.900621627376+0.738694248009j))*x[1]
            ref[(0, 3, 1, 0)]=(0.184510378823+1.38054755924j)-((0.228788917752+0.215950447717j))*(o-1.)
            arg[(0, 3, 1, 1)]=(0.765934944851-0.824888278308j)*x[0]**o + ((0.596781206417+0.0564415650618j))*x[0] + ((-0.154761975317+0.909236316388j))*x[1]**o + ((-0.188118992233+0.674282956999j))*x[1]
            ref[(0, 3, 1, 1)]=(0.509917591859+0.40753628007j)-((0.101862161589+0.0140580063467j))*(o-1.)
            arg[(0, 3, 1, 2)]=(-0.882871543664+0.982828250646j)*x[0]**o + ((0.857362149886-0.335275120149j))*x[0] + ((-0.473561495501-0.726281052304j))*x[1]**o + ((-0.765533184119-0.888214172066j))*x[1]
            ref[(0, 3, 1, 2)]=(-0.632302036699-0.483471046936j)-((-0.226072173194+0.0427578663904j))*(o-1.)
            arg[(0, 3, 1, 3)]=(-0.30960663386+0.239886390657j)*x[0]**o + ((0.248911174242-0.908228407632j))*x[0] + ((0.132403090894+0.708051519436j))*x[1]**o + ((0.302886608997-0.642990736522j))*x[1]
            ref[(0, 3, 1, 3)]=(0.187297120137-0.30164061703j)-((-0.0295339238276+0.157989651682j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.892635305404+0.624829802617j)*x[0]**o + ((-0.00345732809393-0.337407018317j))*x[0] + ((-0.193949443806-0.17562612086j))*x[1]**o + ((0.0226267208966+0.479700737267j))*x[1]
            ref[(0, 3, 2, 0)]=(0.3589276272+0.295748700354j)-((0.1164476436+0.0748672802929j))*(o-1.)
            arg[(0, 3, 2, 1)]=(-0.131221514814-0.414234576337j)*x[0]**o + ((0.958305803016-0.726070473768j))*x[0] + ((0.861951672033+0.206896343077j))*x[1]**o + ((-0.152265298862+0.501879315175j))*x[1]
            ref[(0, 3, 2, 1)]=(0.768385330686-0.215764695926j)-((0.121788359536-0.0345563722099j))*(o-1.)
            arg[(0, 3, 2, 2)]=(0.573346210812-0.45997221066j)*x[0]**o + ((0.109096948052-0.916089063574j))*x[0] + ((0.671536978887+0.0499825448767j))*x[1]**o + ((-0.929204072315-0.669406554841j))*x[1]
            ref[(0, 3, 2, 2)]=(0.212388032718-0.997742642099j)-((0.207480531617-0.0683316109638j))*(o-1.)
            arg[(0, 3, 2, 3)]=(-0.203073070828-0.319349322454j)*x[0]**o + ((-0.691150967589+0.593060838864j))*x[0] + ((0.812055016515+0.644745706248j))*x[1]**o + ((0.076273147704-0.852826161933j))*x[1]
            ref[(0, 3, 2, 3)]=(-0.00294793709934+0.0328155303621j)-((0.101496990948+0.0542327306323j))*(o-1.)
            arg[(1, 0, 0, 0)]=(0.838202250955+0.875839608537j)*x[0]**o + ((0.850326017154-0.670129215947j))*x[0] + ((0.151003336407+0.75831043979j))*x[1]**o + ((0.268764929254-0.243401630181j))*x[1]
            ref[(1, 0, 0, 0)]=(1.05414826688+0.360309601099j)-((0.164867597894+0.272358341388j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.550328806877+0.950337102202j)*x[0]**o + ((0.268810472445-0.928658502116j))*x[0] + ((0.459686784682-0.929087112669j))*x[1]**o + ((-0.813814284049-0.743606689952j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.317822916899-0.825507601267j)-((-0.0151070036991+0.00354166492216j))*(o-1.)
            arg[(1, 0, 0, 2)]=(-0.39571024873-0.533131987694j)*x[0]**o + ((0.0827693005885-0.34743768613j))*x[0] + ((0.531499015933-0.190806069924j))*x[1]**o + ((-0.465182782158-0.37233601623j))*x[1]
            ref[(1, 0, 0, 2)]=(-0.123312357183-0.721855879989j)-((0.0226314612006-0.120656342936j))*(o-1.)
            arg[(1, 0, 0, 3)]=(-0.342490292531-0.806074512225j)*x[0]**o + ((0.236379125824+0.221964797227j))*x[0] + ((-0.749022061067-0.270486681099j))*x[1]**o + ((-0.331587426846+0.735755974323j))*x[1]
            ref[(1, 0, 0, 3)]=(-0.59336032731-0.0594202108873j)-((-0.1819187256-0.179426865554j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.949264746381-0.953262910339j)*x[0]**o + ((-0.379661086476+0.566725970598j))*x[0] + ((0.155263260614+0.882015731548j))*x[1]**o + ((0.930491780361+0.893272546452j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.121585395941+0.69437566913j)-((-0.132333580961-0.0118745297985j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.759375519066+0.381538538781j)*x[0]**o + ((0.174135468356-0.130859541493j))*x[0] + ((-0.581010122779-0.108726429384j))*x[1]**o + ((0.356663356911-0.442611497698j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.404793408289-0.150329464897j)-((-0.223397606974+0.0454686848995j))*(o-1.)
            arg[(1, 0, 1, 2)]=(-0.25202117632-0.30948576478j)*x[0]**o + ((0.764423088578-0.235318114396j))*x[0] + ((0.970746433367-0.125409687847j))*x[1]**o + ((0.460555376828+0.63256162973j))*x[1]
            ref[(1, 0, 1, 2)]=(0.971851861226-0.0188259686463j)-((0.119787542841-0.0724825754379j))*(o-1.)
            arg[(1, 0, 1, 3)]=(-0.145439564368-0.332057904523j)*x[0]**o + ((-0.750302024562-0.0365878863223j))*x[0] + ((-0.938550060085-0.0900614430287j))*x[1]**o + ((-0.111412168858+0.730089522918j))*x[1]
            ref[(1, 0, 1, 3)]=(-0.972851908937+0.135691144522j)-((-0.180664937409-0.0703532245919j))*(o-1.)
            arg[(1, 0, 2, 0)]=(0.106196057034+0.677239981692j)*x[0]**o + ((-0.713937282016-0.467828987084j))*x[0] + ((0.726381167383+0.0879140424233j))*x[1]**o + ((0.206733945078-0.55607353566j))*x[1]
            ref[(1, 0, 2, 0)]=(0.16268694374-0.129374249315j)-((0.138762870736+0.127525670686j))*(o-1.)
            arg[(1, 0, 2, 1)]=(-0.124544934667+0.633534855314j)*x[0]**o + ((0.322998624588+0.348636092646j))*x[0] + ((-0.340733758431+0.00790186418867j))*x[1]**o + ((0.166694261585-0.0673211013548j))*x[1]
            ref[(1, 0, 2, 1)]=(0.0122070965375+0.461375855397j)-((-0.0775464488497+0.106906119917j))*(o-1.)
            arg[(1, 0, 2, 2)]=(-0.405485741097-0.0760992638797j)*x[0]**o + ((0.328185320611+0.753324632414j))*x[0] + ((0.415161526834-0.810939089376j))*x[1]**o + ((0.749935439629+0.898307823988j))*x[1]
            ref[(1, 0, 2, 2)]=(0.543898272988+0.382297051573j)-((0.00161263095618-0.147839725543j))*(o-1.)
            arg[(1, 0, 2, 3)]=(-0.931932607959+0.974573878891j)*x[0]**o + ((-0.829026554916-0.414805452387j))*x[0] + ((-0.0417892159307-0.323767967338j))*x[1]**o + ((0.235417242262-0.973726702453j))*x[1]
            ref[(1, 0, 2, 3)]=(-0.783665568272-0.368863121644j)-((-0.162286970648+0.108467651926j))*(o-1.)
            arg[(1, 1, 0, 0)]=(-0.219546692957+0.285775775036j)*x[0]**o + ((0.0406057491157-0.382510083655j))*x[0] + ((0.25092387898+0.71361111473j))*x[1]**o + ((0.237911921698-0.480542688693j))*x[1]
            ref[(1, 1, 0, 0)]=(0.154947428419+0.068167058709j)-((0.00522953100388+0.166564481628j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.310894262027-0.412788183619j)*x[0]**o + ((0.00470527451204+0.00185996774552j))*x[0] + ((0.850169163525-0.989689388723j))*x[1]**o + ((0.733275112243+0.205113443261j))*x[1]
            ref[(1, 1, 0, 1)]=(0.638627644126-0.597752080668j)-((0.0898791502496-0.233746262057j))*(o-1.)
            arg[(1, 1, 0, 2)]=(-0.328928984538+0.463315638169j)*x[0]**o + ((-0.226620418462-0.0864733035394j))*x[0] + ((-0.159223087713-0.844035623708j))*x[1]**o + ((-0.991444627281+0.325883787921j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.853108558997-0.0706547505787j)-((-0.0813586787085-0.0634533309232j))*(o-1.)
            arg[(1, 1, 0, 3)]=(-0.537596673376-0.361857135074j)*x[0]**o + ((-0.0911234866562-0.64652098446j))*x[0] + ((0.527641567157-0.0755391598981j))*x[1]**o + ((-0.914440518242+0.658045885708j))*x[1]
            ref[(1, 1, 0, 3)]=(-0.507759555559-0.212935696862j)-((-0.00165918436989-0.0728993824953j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.708165052552+0.692037982393j)*x[0]**o + ((-0.993784015274-0.477829382095j))*x[0] + ((0.87634164926+0.820149770006j))*x[1]**o + ((-0.927888559521-0.4635231665j))*x[1]
            ref[(1, 1, 1, 0)]=(-0.168582936491+0.285417601902j)-((0.264084450302+0.252031292067j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.837266523887-0.92642416467j)*x[0]**o + ((0.972253570021+0.16064748227j))*x[0] + ((0.949958689677-0.52743420058j))*x[1]**o + ((0.409527048604+0.206697030919j))*x[1]
            ref[(1, 1, 1, 1)]=(1.5845029161-0.54325692603j)-((0.297870868927-0.242309727542j))*(o-1.)
            arg[(1, 1, 1, 2)]=(0.401261432975+0.493377527529j)*x[0]**o + ((-0.0465369207552-0.359556613657j))*x[0] + ((0.743173020791+0.0794390831782j))*x[1]**o + ((0.125865778567+0.651647606532j))*x[1]
            ref[(1, 1, 1, 2)]=(0.611881655789+0.432453801791j)-((0.190739075628+0.0954694351178j))*(o-1.)
            arg[(1, 1, 1, 3)]=(-0.279151248724-0.506106526591j)*x[0]**o + ((0.122706915186+0.343449566692j))*x[0] + ((-0.719239392138+0.775059122147j))*x[1]**o + ((-0.54222056485-0.237612820879j))*x[1]
            ref[(1, 1, 1, 3)]=(-0.708952145263+0.187394670685j)-((-0.166398440144+0.0448254325926j))*(o-1.)
            arg[(1, 1, 2, 0)]=(-0.642306291434-0.541611923564j)*x[0]**o + ((-0.522173549459+0.428650828797j))*x[0] + ((0.959362942468+0.346519259605j))*x[1]**o + ((-0.827965042328-0.675005531747j))*x[1]
            ref[(1, 1, 2, 0)]=(-0.516540970377-0.220723683454j)-((0.0528427751723-0.0325154439931j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.817760102472+0.220753847064j)*x[0]**o + ((0.370003490556+0.126215458039j))*x[0] + ((0.216124758076+0.270601455048j))*x[1]**o + ((0.482370900181-0.372349384266j))*x[1]
            ref[(1, 1, 2, 1)]=(0.125369523171+0.122610687942j)-((-0.100272557399+0.081892550352j))*(o-1.)
            arg[(1, 1, 2, 2)]=(-0.930767463839+0.0582879413736j)*x[0]**o + ((-0.368507731457+0.653372878876j))*x[0] + ((0.751136666178-0.954965563822j))*x[1]**o + ((-0.612847215151-0.899709523139j))*x[1]
            ref[(1, 1, 2, 2)]=(-0.580492872135-0.571507133356j)-((-0.0299384662769-0.149446270408j))*(o-1.)
            arg[(1, 1, 2, 3)]=(0.603489445837+0.845092819865j)*x[0]**o + ((0.873688394884-0.873763202143j))*x[0] + ((0.383600045588+0.121266871687j))*x[1]**o + ((-0.583833876281+0.330616932641j))*x[1]
            ref[(1, 1, 2, 3)]=(0.638472005014+0.211606711025j)-((0.164514915237+0.161059948592j))*(o-1.)
            arg[(1, 2, 0, 0)]=(-0.905242487865+0.812703371343j)*x[0]**o + ((-0.939292576321-0.509139490201j))*x[0] + ((0.412954688951+0.676216739278j))*x[1]**o + ((-0.0278766949754-0.094212094853j))*x[1]
            ref[(1, 2, 0, 0)]=(-0.729728535106+0.442784262783j)-((-0.0820479664857+0.24815335177j))*(o-1.)
            arg[(1, 2, 0, 1)]=(-0.773527925464+0.25774671861j)*x[0]**o + ((0.530879594327+0.467614697345j))*x[0] + ((-0.69330751056+0.938327665863j))*x[1]**o + ((-0.112183319591-0.290778128662j))*x[1]
            ref[(1, 2, 0, 1)]=(-0.524069580644+0.686455476578j)-((-0.244472572671+0.199345730745j))*(o-1.)
            arg[(1, 2, 0, 2)]=(-0.0969582296396+0.934111103328j)*x[0]**o + ((0.165520888709+0.998930959046j))*x[0] + ((-0.2674768795-0.645619916814j))*x[1]**o + ((0.599553713591-0.213046629529j))*x[1]
            ref[(1, 2, 0, 2)]=(0.20031974658+0.537187758016j)-((-0.0607391848566+0.0480818644191j))*(o-1.)
            arg[(1, 2, 0, 3)]=(0.782791039052-0.694713692693j)*x[0]**o + ((-0.608118483932-0.833823538281j))*x[0] + ((0.960613233389+0.617044716078j))*x[1]**o + ((0.561526485724+0.0552836452347j))*x[1]
            ref[(1, 2, 0, 3)]=(0.848406137116-0.428104434831j)-((0.29056737874-0.0129448294359j))*(o-1.)
            arg[(1, 2, 1, 0)]=(0.534977228469+0.0459405294363j)*x[0]**o + ((-0.483867116081+0.494204701179j))*x[0] + ((0.616101667041-0.199683407992j))*x[1]**o + ((-0.473823271091+0.44942550222j))*x[1]
            ref[(1, 2, 1, 0)]=(0.0966942541691+0.394943662422j)-((0.191846482585-0.0256238130926j))*(o-1.)
            arg[(1, 2, 1, 1)]=(0.862292529094+0.0246443917417j)*x[0]**o + ((-0.0098317225533+0.923703784211j))*x[0] + ((0.120728023676+0.59774790383j))*x[1]**o + ((0.897314627417+0.233407785941j))*x[1]
            ref[(1, 2, 1, 1)]=(0.935251728817+0.889751932862j)-((0.163836758795+0.103732049262j))*(o-1.)
            arg[(1, 2, 1, 2)]=(-0.947255470068+0.939858776321j)*x[0]**o + ((-0.158215211083-0.399611314021j))*x[0] + ((-0.36394283745-0.295653474822j))*x[1]**o + ((-0.794044391598-0.704671310141j))*x[1]
            ref[(1, 2, 1, 2)]=(-1.1317289551-0.230038661331j)-((-0.218533051253+0.10736755025j))*(o-1.)
            arg[(1, 2, 1, 3)]=(0.839760060673+0.306331107898j)*x[0]**o + ((-0.590157384418-0.607402624352j))*x[0] + ((-0.293902492871+0.68493748062j))*x[1]**o + ((-0.314431527198-0.145969742426j))*x[1]
            ref[(1, 2, 1, 3)]=(-0.179365671907+0.11894811087j)-((0.0909762613005+0.16521143142j))*(o-1.)
            arg[(1, 2, 2, 0)]=(0.11392975587+0.149534297147j)*x[0]**o + ((-0.920148713043+0.333985290559j))*x[0] + ((0.901047634114-0.109159685558j))*x[1]**o + ((0.720152847223+0.195347754386j))*x[1]
            ref[(1, 2, 2, 0)]=(0.407490762082+0.284853828267j)-((0.169162898331+0.00672910193151j))*(o-1.)
            arg[(1, 2, 2, 1)]=(-0.163959941595-0.488765475575j)*x[0]**o + ((-0.519578315022+0.100980632015j))*x[0] + ((0.556576601666+0.558550341319j))*x[1]**o + ((0.565535969538-0.430767278839j))*x[1]
            ref[(1, 2, 2, 1)]=(0.219287157293-0.13000089054j)-((0.0654361100119+0.0116308109573j))*(o-1.)
            arg[(1, 2, 2, 2)]=(-0.801098901255-0.601498728355j)*x[0]**o + ((0.720763561763-0.955616382589j))*x[0] + ((-0.199390958424-0.960431453254j))*x[1]**o + ((-0.165237476295-0.00681515068495j))*x[1]
            ref[(1, 2, 2, 2)]=(-0.222481887106-1.26218085744j)-((-0.166748309947-0.260321696935j))*(o-1.)
            arg[(1, 2, 2, 3)]=(0.310341712108-0.94222897104j)*x[0]**o + ((-0.0444940036862+0.0108233157549j))*x[0] + ((0.682613492228-0.794055582765j))*x[1]**o + ((0.113570292154+0.149587891784j))*x[1]
            ref[(1, 2, 2, 3)]=(0.531015746402-0.787936673134j)-((0.165492534056-0.289380758968j))*(o-1.)
            arg[(1, 3, 0, 0)]=(0.350140639127-0.234445416244j)*x[0]**o + ((-0.167155584285+0.448373612074j))*x[0] + ((0.555302074399+0.427769924856j))*x[1]**o + ((0.46990701609-0.0614509885765j))*x[1]
            ref[(1, 3, 0, 0)]=(0.604097072665+0.290123566055j)-((0.150907118921+0.0322207514353j))*(o-1.)
            arg[(1, 3, 0, 1)]=(0.412098614964-0.816142772452j)*x[0]**o + ((-0.565907861681+0.952593957981j))*x[0] + ((-0.554742692897+0.140259087006j))*x[1]**o + ((-0.744779312905+0.507410100949j))*x[1]
            ref[(1, 3, 0, 1)]=(-0.726665626259+0.392060186742j)-((-0.0237740129888-0.112647280908j))*(o-1.)
            arg[(1, 3, 0, 2)]=(-0.570474489801-0.753205137568j)*x[0]**o + ((-0.511459224659+0.794800660287j))*x[0] + ((-0.646213938724-0.202734469926j))*x[1]**o + ((-0.0290023861108-0.98844585158j))*x[1]
            ref[(1, 3, 0, 2)]=(-0.878575019647-0.574792399394j)-((-0.202781404754-0.159323267916j))*(o-1.)
            arg[(1, 3, 0, 3)]=(0.423016158501+0.646163545939j)*x[0]**o + ((0.83193631428-0.184217116998j))*x[0] + ((-0.255918849531+0.609193174754j))*x[1]**o + ((-0.737716059553-0.416837917507j))*x[1]
            ref[(1, 3, 0, 3)]=(0.130658781849+0.327150843094j)-((0.0278495514951+0.209226120116j))*(o-1.)
            arg[(1, 3, 1, 0)]=(-0.617872193754+0.552052166356j)*x[0]**o + ((0.055388092518+0.260224778476j))*x[0] + ((-0.518656927385+0.626504488674j))*x[1]**o + ((-0.871778675627-0.0818226037849j))*x[1]
            ref[(1, 3, 1, 0)]=(-0.976459852124+0.67847941486j)-((-0.18942152019+0.196426109172j))*(o-1.)
            arg[(1, 3, 1, 1)]=(-0.0230883480076-0.134734722421j)*x[0]**o + ((0.585588318099-0.728856908633j))*x[0] + ((0.302610985565-0.247002982519j))*x[1]**o + ((0.614820137198-0.681830629506j))*x[1]
            ref[(1, 3, 1, 1)]=(0.739965546427-0.89621262154j)-((0.0465871062596-0.0636229508233j))*(o-1.)
            arg[(1, 3, 1, 2)]=(0.278225064509+0.341980022638j)*x[0]**o + ((-0.417968577649+0.951388271534j))*x[0] + ((0.768475476014-0.914125830932j))*x[1]**o + ((-0.177186532597+0.967334053992j))*x[1]
            ref[(1, 3, 1, 2)]=(0.225772715138+0.673288258616j)-((0.174450090087-0.0953576347157j))*(o-1.)
            arg[(1, 3, 1, 3)]=(-0.0360859001564-0.583449232441j)*x[0]**o + ((0.830505627318+0.994814663001j))*x[0] + ((0.285744428049-0.180331259129j))*x[1]**o + ((-0.71474210227+0.0130666607672j))*x[1]
            ref[(1, 3, 1, 3)]=(0.18271102647+0.122050416099j)-((0.0416097546488-0.127296748595j))*(o-1.)
            arg[(1, 3, 2, 0)]=(0.863931157674-0.386595001686j)*x[0]**o + ((-0.735879790437-0.47262513633j))*x[0] + ((0.833860927921-0.270836950185j))*x[1]**o + ((0.79232292769+0.364061448293j))*x[1]
            ref[(1, 3, 2, 0)]=(0.877117611424-0.382997819954j)-((0.282965347599-0.109571991979j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.444195641739+0.703833348732j)*x[0]**o + ((-0.318610834382-0.0728939097405j))*x[0] + ((0.553515672494-0.805469986204j))*x[1]**o + ((0.876692597236-0.312765851395j))*x[1]
            ref[(1, 3, 2, 1)]=(0.333700896804-0.243648199304j)-((0.0182200051258-0.0169394395787j))*(o-1.)
            arg[(1, 3, 2, 2)]=(0.446986466601+0.567228367102j)*x[0]**o + ((-0.335101646153+0.909731750191j))*x[0] + ((-0.185339186908+0.860795340059j))*x[1]**o + ((-0.00098700504835+0.0194527476765j))*x[1]
            ref[(1, 3, 2, 2)]=(-0.0372206857541+1.17860410251j)-((0.0436078799488+0.238003951193j))*(o-1.)
            arg[(1, 3, 2, 3)]=(-0.395639243611-0.708426850486j)*x[0]**o + ((-0.913786844341-0.408388741902j))*x[0] + ((0.451237999968+0.610394536132j))*x[1]**o + ((0.726154989488+0.614066303116j))*x[1]
            ref[(1, 3, 2, 3)]=(-0.066016549248+0.0538226234301j)-((0.00926645939282-0.016338719059j))*(o-1.)
            arg[(2, 0, 0, 0)]=(-0.0729895473333+0.544591903348j)*x[0]**o + ((-0.130926161891-0.259925389076j))*x[0] + ((0.739350071813-0.482488077659j))*x[1]**o + ((-0.901561236421+0.491424907902j))*x[1]
            ref[(2, 0, 0, 0)]=(-0.183063436916+0.146801672257j)-((0.111060087413+0.0103506376149j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.222378416237-0.00768823092623j)*x[0]**o + ((0.278968355248-0.436237865563j))*x[0] + ((-0.0347025473103+0.346293840042j))*x[1]**o + ((-0.102556738817+0.085239544495j))*x[1]
            ref[(2, 0, 0, 1)]=(-0.0403346735581-0.00619635597574j)-((-0.0428468272579+0.056434268186j))*(o-1.)
            arg[(2, 0, 0, 2)]=(-0.939045382005+0.589413767732j)*x[0]**o + ((-0.802793234849+0.530416062552j))*x[0] + ((-0.720106663433-0.822123049175j))*x[1]**o + ((-0.689929162196-0.313866782779j))*x[1]
            ref[(2, 0, 0, 2)]=(-1.57593722124-0.00808000083484j)-((-0.276525340906-0.0387848802405j))*(o-1.)
            arg[(2, 0, 0, 3)]=(-0.941879068119+0.574433805138j)*x[0]**o + ((0.565356773116+0.815055400532j))*x[0] + ((0.875625652679+0.328037046123j))*x[1]**o + ((-0.571009826776-0.97821341898j))*x[1]
            ref[(2, 0, 0, 3)]=(-0.0359532345498+0.369656416406j)-((-0.0110422359066+0.150411808543j))*(o-1.)
            arg[(2, 0, 1, 0)]=(-0.687562035289+0.549449743779j)*x[0]**o + ((-0.745067359947+0.52266658641j))*x[0] + ((0.826325183505+0.381870582325j))*x[1]**o + ((-0.627694872906+0.00814812395567j))*x[1]
            ref[(2, 0, 1, 0)]=(-0.616999542318+0.731067518235j)-((0.0231271913693+0.155220054351j))*(o-1.)
            arg[(2, 0, 1, 1)]=(0.169087552775-0.479276631367j)*x[0]**o + ((0.271533445839-0.915044736218j))*x[0] + ((0.518184339368+0.137435990533j))*x[1]**o + ((-0.885116755147-0.0799394679679j))*x[1]
            ref[(2, 0, 1, 1)]=(0.0368442914177-0.66841242251j)-((0.114545315357-0.0569734401389j))*(o-1.)
            arg[(2, 0, 1, 2)]=(0.897281462997+0.563423180946j)*x[0]**o + ((-0.214359552487+0.815160340195j))*x[0] + ((-0.452736929039-0.797980738437j))*x[1]**o + ((0.994560082962-0.0623790838246j))*x[1]
            ref[(2, 0, 1, 2)]=(0.612372532217+0.25911184944j)-((0.0740907556597-0.0390929262484j))*(o-1.)
            arg[(2, 0, 1, 3)]=(-0.427795968357-0.805633079884j)*x[0]**o + ((0.763373333552-0.0707772592485j))*x[0] + ((0.666036953982+0.385061467413j))*x[1]**o + ((-0.0857879624356-0.335206363362j))*x[1]
            ref[(2, 0, 1, 3)]=(0.457913178371-0.413277617541j)-((0.0397068309374-0.0700952687452j))*(o-1.)
            arg[(2, 0, 2, 0)]=(-0.654916212555-0.349380973054j)*x[0]**o + ((-0.799354980238-0.965037079702j))*x[0] + ((0.745780122418+0.627451626925j))*x[1]**o + ((-0.0547880155115-0.873101706266j))*x[1]
            ref[(2, 0, 2, 0)]=(-0.381639542943-0.780034066049j)-((0.0151439849771+0.0463451089785j))*(o-1.)
            arg[(2, 0, 2, 1)]=(0.877440435812-0.731418820935j)*x[0]**o + ((0.299176116535-0.991199146492j))*x[0] + ((-0.335908423543+0.846210850126j))*x[1]**o + ((0.755042370273-0.82761169771j))*x[1]
            ref[(2, 0, 2, 1)]=(0.797875249538-0.852009407506j)-((0.0902553353781+0.0191320048651j))*(o-1.)
            arg[(2, 0, 2, 2)]=(-0.850756588314+0.424063698216j)*x[0]**o + ((0.0216835464781-0.4471301913j))*x[0] + ((0.51119034306-0.425103628838j))*x[1]**o + ((-0.788072315993-0.196553064385j))*x[1]
            ref[(2, 0, 2, 2)]=(-0.552977507384-0.322361593154j)-((-0.0565943742089-0.000173321770407j))*(o-1.)
            arg[(2, 0, 2, 3)]=(-0.190722104508+0.437018352882j)*x[0]**o + ((0.0491160426023+0.0146273067688j))*x[0] + ((-0.119923944209+0.694843202207j))*x[1]**o + ((0.580622136869+0.630281828492j))*x[1]
            ref[(2, 0, 2, 3)]=(0.159546065377+0.888385345175j)-((-0.0517743414528+0.188643592515j))*(o-1.)
            arg[(2, 1, 0, 0)]=(-0.240393586604+0.276860902457j)*x[0]**o + ((0.861923830538+0.786601365626j))*x[0] + ((0.711615283795+0.0147093951873j))*x[1]**o + ((-0.701116630946+0.264607273026j))*x[1]
            ref[(2, 1, 0, 0)]=(0.316014448392+0.671389468148j)-((0.078536949532+0.0485950496074j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.474241438002+0.867563513927j)*x[0]**o + ((0.0656729000183-0.568157205744j))*x[0] + ((-0.744465291856-0.485084857799j))*x[1]**o + ((0.664027206647-0.213571557519j))*x[1]
            ref[(2, 1, 0, 1)]=(-0.244503311596-0.199625053568j)-((-0.20311778831+0.063746442688j))*(o-1.)
            arg[(2, 1, 0, 2)]=(-0.535010968955-0.245727303705j)*x[0]**o + ((-0.302058168701-0.816876991416j))*x[0] + ((-0.171246593519-0.790222129573j))*x[1]**o + ((-0.353609458692-0.258644515655j))*x[1]
            ref[(2, 1, 0, 2)]=(-0.680962594933-1.05573547017j)-((-0.117709593746-0.17265823888j))*(o-1.)
            arg[(2, 1, 0, 3)]=(0.885143667941-0.25681813788j)*x[0]**o + ((0.625537675755-0.983124044827j))*x[0] + ((-0.00970397016007-0.732435697468j))*x[1]**o + ((0.201332561688-0.955104889571j))*x[1]
            ref[(2, 1, 0, 3)]=(0.851154967612-1.46374138487j)-((0.145906616297-0.164875639225j))*(o-1.)
            arg[(2, 1, 1, 0)]=(-0.136942074664-0.866577989291j)*x[0]**o + ((0.463552425884-0.146040756833j))*x[0] + ((-0.971009847822-0.596636093222j))*x[1]**o + ((-0.583249917547+0.596022039966j))*x[1]
            ref[(2, 1, 1, 0)]=(-0.613824707074-0.50661639969j)-((-0.184658653748-0.243869013752j))*(o-1.)
            arg[(2, 1, 1, 1)]=(-0.199343513641-0.368980787277j)*x[0]**o + ((0.80148043566+0.247275950632j))*x[0] + ((0.475629651573-0.834508626456j))*x[1]**o + ((-0.830657790763-0.0726332378154j))*x[1]
            ref[(2, 1, 1, 1)]=(0.123554391415-0.514423350458j)-((0.0460476896553-0.200581568955j))*(o-1.)
            arg[(2, 1, 1, 2)]=(-0.308219422219+0.485166573374j)*x[0]**o + ((-0.754630181531-0.328232717012j))*x[0] + ((0.436274694247-0.156961133792j))*x[1]**o + ((-0.559670475932-0.544360063184j))*x[1]
            ref[(2, 1, 1, 2)]=(-0.593122692718-0.272193670307j)-((0.0213425453379+0.054700906597j))*(o-1.)
            arg[(2, 1, 1, 3)]=(0.627987776883+0.0966399923444j)*x[0]**o + ((0.669539793415-0.478824453511j))*x[0] + ((0.599682177444+0.555436857888j))*x[1]**o + ((-0.958274750639-0.424426502196j))*x[1]
            ref[(2, 1, 1, 3)]=(0.469467498551-0.125587052738j)-((0.204611659054+0.108679475039j))*(o-1.)
            arg[(2, 1, 2, 0)]=(0.173949187711+0.649083150807j)*x[0]**o + ((0.669212351931+0.605312593021j))*x[0] + ((0.67446326051-0.269040313859j))*x[1]**o + ((0.725111256848-0.290861191996j))*x[1]
            ref[(2, 1, 2, 0)]=(1.1213680285+0.347247118986j)-((0.141402074703+0.0633404728246j))*(o-1.)
            arg[(2, 1, 2, 1)]=(-0.452463266569+0.367466543729j)*x[0]**o + ((0.518334632531-0.275865160219j))*x[0] + ((-0.557826654381+0.198987497702j))*x[1]**o + ((-0.608566777369+0.148246596386j))*x[1]
            ref[(2, 1, 2, 1)]=(-0.550261032894+0.219417738799j)-((-0.168381653492+0.0944090069052j))*(o-1.)
            arg[(2, 1, 2, 2)]=(0.924801455038-0.538773703046j)*x[0]**o + ((0.667998840156+0.0280179223329j))*x[0] + ((0.0178827024404-0.330518686182j))*x[1]**o + ((0.42575590195+0.897872732184j))*x[1]
            ref[(2, 1, 2, 2)]=(1.01821944979+0.0282991326441j)-((0.157114026246-0.144882064871j))*(o-1.)
            arg[(2, 1, 2, 3)]=(0.779771406078-0.454094893918j)*x[0]**o + ((0.50918122304-0.305926994092j))*x[0] + ((0.329941171194-0.20563413514j))*x[1]**o + ((-0.243779996659-0.15009498344j))*x[1]
            ref[(2, 1, 2, 3)]=(0.687556901827-0.557875503295j)-((0.184952096212-0.109954838176j))*(o-1.)
            arg[(2, 2, 0, 0)]=(-0.145510649455-0.907207641271j)*x[0]**o + ((0.102479811212+0.0110414092714j))*x[0] + ((-0.68806894883+0.216400855507j))*x[1]**o + ((-0.368690512649+0.399438685151j))*x[1]
            ref[(2, 2, 0, 0)]=(-0.549895149861-0.140163345671j)-((-0.138929933048-0.115134464294j))*(o-1.)
            arg[(2, 2, 0, 1)]=(-0.109000251061+0.956008948859j)*x[0]**o + ((-0.328904724578+0.0790491523873j))*x[0] + ((-0.449140150685-0.759089726021j))*x[1]**o + ((-0.825425243405-0.316959535659j))*x[1]
            ref[(2, 2, 0, 1)]=(-0.856235184864-0.0204955802168j)-((-0.0930234002909+0.032819870473j))*(o-1.)
            arg[(2, 2, 0, 2)]=(0.436467712214+0.621935352564j)*x[0]**o + ((-0.064638922529-0.813803313257j))*x[0] + ((-0.574529980614+0.0756567443233j))*x[1]**o + ((-0.696852690469-0.400610509836j))*x[1]
            ref[(2, 2, 0, 2)]=(-0.449776940699-0.258410863103j)-((-0.0230103780667+0.116265349481j))*(o-1.)
            arg[(2, 2, 0, 3)]=(0.69451440245+0.13164402535j)*x[0]**o + ((0.387181080629+0.181381918624j))*x[0] + ((0.119310188256+0.989181757699j))*x[1]**o + ((-0.711913587583-0.626777590719j))*x[1]
            ref[(2, 2, 0, 3)]=(0.244546041877+0.337715055477j)-((0.135637431784+0.186804297175j))*(o-1.)
            arg[(2, 2, 1, 0)]=(-0.687477121019+0.805735564646j)*x[0]**o + ((-0.0353331886997-0.27258644984j))*x[0] + ((-0.267457209739+0.891008648473j))*x[1]**o + ((0.840226929781+0.0702958520286j))*x[1]
            ref[(2, 2, 1, 0)]=(-0.0750202948385+0.747226807654j)-((-0.159155721793+0.282790702187j))*(o-1.)
            arg[(2, 2, 1, 1)]=(-0.465921333081+0.196451697344j)*x[0]**o + ((-0.40979299197+0.0664965922612j))*x[0] + ((-0.0107110379058-0.46806023961j))*x[1]**o + ((-0.431390429372+0.789107475915j))*x[1]
            ref[(2, 2, 1, 1)]=(-0.658907896164+0.291997762955j)-((-0.0794387284978-0.0452680903776j))*(o-1.)
            arg[(2, 2, 1, 2)]=(-0.574575908835-0.0023535412651j)*x[0]**o + ((0.483274963692-0.304004683247j))*x[0] + ((0.461127658716+0.970192365407j))*x[1]**o + ((-0.915791224847+0.306101039605j))*x[1]
            ref[(2, 2, 1, 2)]=(-0.272982255637+0.48496759025j)-((-0.0189080416865+0.16130647069j))*(o-1.)
            arg[(2, 2, 1, 3)]=(-0.219159103881+0.426096404772j)*x[0]**o + ((-0.0281128974985+0.96865701771j))*x[0] + ((-0.940609730425+0.825909069738j))*x[1]**o + ((0.347981156507-0.290703781298j))*x[1]
            ref[(2, 2, 1, 3)]=(-0.419950287648+0.964979355461j)-((-0.193294805718+0.208667579085j))*(o-1.)
            arg[(2, 2, 2, 0)]=(-0.99724715944+0.488099004059j)*x[0]**o + ((-0.966380552111-0.869363579064j))*x[0] + ((0.608357599+0.729116282073j))*x[1]**o + ((0.644733825007+0.301309206047j))*x[1]
            ref[(2, 2, 2, 0)]=(-0.355268143772+0.324580456557j)-((-0.06481492674+0.202869214355j))*(o-1.)
            arg[(2, 2, 2, 1)]=(0.332582806794+0.383330281331j)*x[0]**o + ((-0.447827850238-0.51559205388j))*x[0] + ((-0.645928063008-0.313377230764j))*x[1]**o + ((-0.993054632177+0.331422790714j))*x[1]
            ref[(2, 2, 2, 1)]=(-0.877113869315-0.0571081062992j)-((-0.052224209369+0.0116588417612j))*(o-1.)
            arg[(2, 2, 2, 2)]=(0.668482947338-0.13997211815j)*x[0]**o + ((0.422917722832-0.774977743986j))*x[0] + ((-0.197278850532+0.888517906493j))*x[1]**o + ((-0.59710705709-0.11682790525j))*x[1]
            ref[(2, 2, 2, 2)]=(0.148507381274-0.0716299304465j)-((0.0785340161343+0.124757631391j))*(o-1.)
            arg[(2, 2, 2, 3)]=(0.247166562097-0.890316980179j)*x[0]**o + ((-0.492034868239-0.0564592180886j))*x[0] + ((0.551970912312-0.494054576171j))*x[1]**o + ((-0.733688859762+0.600135813682j))*x[1]
            ref[(2, 2, 2, 3)]=(-0.213293126796-0.420347480378j)-((0.133189579068-0.230728592725j))*(o-1.)
            arg[(2, 3, 0, 0)]=(0.361933859668-0.0408382542399j)*x[0]**o + ((0.0258584976143+0.00390979867507j))*x[0] + ((-0.112774037524-0.10930678646j))*x[1]**o + ((0.187710471077+0.873433538165j))*x[1]
            ref[(2, 3, 0, 0)]=(0.231364395418+0.36359914807j)-((0.0415266370241-0.02502417345j))*(o-1.)
            arg[(2, 3, 0, 1)]=(0.95728509932+0.221323180272j)*x[0]**o + ((0.955828884199+0.433800169044j))*x[0] + ((0.357897423028+0.737617045191j))*x[1]**o + ((0.884115880336-0.382274778483j))*x[1]
            ref[(2, 3, 0, 1)]=(1.57756364344+0.505232808012j)-((0.219197087058+0.159823370911j))*(o-1.)
            arg[(2, 3, 0, 2)]=(0.107185964274-0.456126205703j)*x[0]**o + ((-0.342816670244-0.562134658772j))*x[0] + ((-0.542016149215+0.280534435605j))*x[1]**o + ((-0.604811221009-0.302247503425j))*x[1]
            ref[(2, 3, 0, 2)]=(-0.691229038097-0.519986966147j)-((-0.0724716974902-0.0292652950164j))*(o-1.)
            arg[(2, 3, 0, 3)]=(-0.573155444565-0.345384220496j)*x[0]**o + ((-0.114928201398+0.787848312507j))*x[0] + ((0.791756294897-0.131908570845j))*x[1]**o + ((-0.115831457941+0.75394310525j))*x[1]
            ref[(2, 3, 0, 3)]=(-0.006079404503+0.532249313208j)-((0.0364334750554-0.0795487985568j))*(o-1.)
            arg[(2, 3, 1, 0)]=(-0.50084915956-0.440680098171j)*x[0]**o + ((0.239311146909+0.145802792018j))*x[0] + ((0.0408468880455-0.138022118898j))*x[1]**o + ((-0.712002630221+0.701938944787j))*x[1]
            ref[(2, 3, 1, 0)]=(-0.466346877413+0.134519759868j)-((-0.0766670452524-0.0964503695115j))*(o-1.)
            arg[(2, 3, 1, 1)]=(-0.573306391768+0.796042317551j)*x[0]**o + ((0.337319544599+0.187676939951j))*x[0] + ((0.148375661032+0.765600147286j))*x[1]**o + ((0.793413204687-0.360161274953j))*x[1]
            ref[(2, 3, 1, 1)]=(0.352901009275+0.694579064918j)-((-0.070821788456+0.260273744139j))*(o-1.)
            arg[(2, 3, 1, 2)]=(-0.727969524445+0.91656483483j)*x[0]**o + ((0.105422873674-0.867911327339j))*x[0] + ((-0.308761919036+0.459867204065j))*x[1]**o + ((-0.352564186502-0.902802433555j))*x[1]
            ref[(2, 3, 1, 2)]=(-0.641936378154-0.197140860999j)-((-0.172788573913+0.229405339816j))*(o-1.)
            arg[(2, 3, 1, 3)]=(-0.890757129047+0.849823108369j)*x[0]**o + ((0.286902250741+0.979773680646j))*x[0] + ((0.778887977058-0.893485864784j))*x[1]**o + ((-0.956543248701+0.0822475597027j))*x[1]
            ref[(2, 3, 1, 3)]=(-0.390755074975+0.509179241967j)-((-0.0186448586648-0.00727712606911j))*(o-1.)
            arg[(2, 3, 2, 0)]=(0.104364154044+0.563174327671j)*x[0]**o + ((0.0400465152973-0.268490986001j))*x[0] + ((0.82043103093-0.0430950744152j))*x[1]**o + ((0.419328557573+0.299247757276j))*x[1]
            ref[(2, 3, 2, 0)]=(0.692085128922+0.275418012266j)-((0.154132530829+0.0866798755427j))*(o-1.)
            arg[(2, 3, 2, 1)]=(0.0179573877364-0.263059466006j)*x[0]**o + ((-0.813220843669+0.489315490465j))*x[0] + ((0.145253059176+0.332171995487j))*x[1]**o + ((-0.28481660403-0.731687861671j))*x[1]
            ref[(2, 3, 2, 1)]=(-0.467413500394-0.0866299208629j)-((0.027201741152+0.0115187549134j))*(o-1.)
            arg[(2, 3, 2, 2)]=(-0.913468226933-0.574854283975j)*x[0]**o + ((-0.0614636712668-0.984232361547j))*x[0] + ((0.69633049661+0.802704612839j))*x[1]**o + ((0.227710286353+0.802016247385j))*x[1]
            ref[(2, 3, 2, 2)]=(-0.0254455576179+0.0228171073508j)-((-0.0361896217203+0.0379750548107j))*(o-1.)
            arg[(2, 3, 2, 3)]=(-0.982848560529-0.317859901788j)*x[0]**o + ((0.861923806852-0.641415184663j))*x[0] + ((-0.565893074976+0.267620264547j))*x[1]**o + ((0.695346281935+0.735873099j))*x[1]
            ref[(2, 3, 2, 3)]=(0.00426422664095+0.0221091385482j)-((-0.258123605917-0.00837327287341j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(-0.600194929991-0.144781445515j)*x[0]**o + ((0.91295950263+0.315098312901j))*x[0] + ((-0.892247576643-0.741785166354j))*x[1]**o + ((0.722511017595-0.590031622939j))*x[1] + ((0.876544785626-0.74765195373j))*x[2]**o + ((0.761923519329+0.0207968344102j))*x[2]
            ref[(0, 0, 0, 0)]=(0.890748159273-0.944177520614j)-((-0.102649620168-0.272369760933j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.391614430907+0.7301779016j)*x[0]**o + ((0.0629097025532+0.0908793369411j))*x[0] + ((0.0727456142855+0.238925803281j))*x[1]**o + ((0.703786904506+0.761007037514j))*x[1] + ((-0.512090356536+0.419958188772j))*x[2]**o + ((-0.118988473474+0.617982457866j))*x[2]
            ref[(0, 0, 0, 1)]=(0.29998891112+1.42946536299j)-((-0.00795505189069+0.231510315609j))*(o-1.)
            arg[(0, 0, 0, 2)]=(-0.907218978165-0.662282992452j)*x[0]**o + ((-0.561724949732-0.389766734044j))*x[0] + ((0.186366115697+0.681177677552j))*x[1]**o + ((-0.212416350593-0.361181610372j))*x[1] + ((0.582771552814-0.620850182226j))*x[2]**o + ((0.733868324185-0.495204733476j))*x[2]
            ref[(0, 0, 0, 2)]=(-0.0891771428969-0.924054287509j)-((-0.0230135516091-0.100325916188j))*(o-1.)
            arg[(0, 0, 0, 3)]=(0.853869935272-0.480601504305j)*x[0]**o + ((0.983086421649+0.769850500551j))*x[0] + ((0.36818799254+0.228889837908j))*x[1]**o + ((-0.599539002904-0.177425142038j))*x[1] + ((-0.00397972568743+0.333243558631j))*x[2]**o + ((-0.161840394545+0.684574520789j))*x[2]
            ref[(0, 0, 0, 3)]=(0.719892613162+0.679265885768j)-((0.203013033687+0.0135886487057j))*(o-1.)
            arg[(0, 0, 1, 0)]=(-0.667866634606+0.24741431949j)*x[0]**o + ((-0.359819940573+0.912272913028j))*x[0] + ((-0.448739367217+0.869801099687j))*x[1]**o + ((-0.615923582468-0.384627060348j))*x[1] + ((0.367895553301+0.948854548293j))*x[2]**o + ((-0.96290671418+0.251241969958j))*x[2]
            ref[(0, 0, 1, 0)]=(-1.34368034287+1.42247889505j)-((-0.124785074754+0.344344994578j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.717763635736+0.674333750029j)*x[0]**o + ((0.244285207726-0.352271288389j))*x[0] + ((0.0337129566403-0.226257753745j))*x[1]**o + ((0.196236902724-0.473786758336j))*x[1] + ((0.377467344095+0.705691849037j))*x[2]**o + ((0.983578084564-0.195871253584j))*x[2]
            ref[(0, 0, 1, 1)]=(0.558758430007+0.0659192725058j)-((-0.0510972225+0.192294640887j))*(o-1.)
            arg[(0, 0, 1, 2)]=(0.984793937951-0.362933600052j)*x[0]**o + ((-0.503330471831-0.416320917977j))*x[0] + ((-0.577615419083-0.0535593240278j))*x[1]**o + ((-0.637644080966+0.843243777365j))*x[1] + ((-0.780941246173+0.310417037437j))*x[2]**o + ((-0.522154282017+0.605849185313j))*x[2]
            ref[(0, 0, 1, 2)]=(-1.01844578106+0.463348079029j)-((-0.0622937878841-0.0176793144405j))*(o-1.)
            arg[(0, 0, 1, 3)]=(-0.242613580896+0.745094532601j)*x[0]**o + ((0.317781321805+0.938196994022j))*x[0] + ((-0.958058713878-0.850342812096j))*x[1]**o + ((-0.223721216255-0.326186492969j))*x[1] + ((0.643770345365+0.21652818701j))*x[2]**o + ((0.339627877926+0.176557542646j))*x[2]
            ref[(0, 0, 1, 3)]=(-0.0616069829667+0.449923975607j)-((-0.0928169915682+0.0185466512525j))*(o-1.)
            arg[(0, 0, 2, 0)]=(-0.391590735596+0.0915149184539j)*x[0]**o + ((0.479938998273+0.700600816989j))*x[0] + ((0.681689254483-0.977528360267j))*x[1]**o + ((0.302200905623+0.0940749428692j))*x[1] + ((-0.79782994344-0.753435054612j))*x[2]**o + ((0.160580585608+0.391625052377j))*x[2]
            ref[(0, 0, 2, 0)]=(0.217494532475-0.226573842095j)-((-0.0846219040921-0.273241416071j))*(o-1.)
            arg[(0, 0, 2, 1)]=(0.812841339159-0.142016886219j)*x[0]**o + ((-0.534404091837+0.512935086685j))*x[0] + ((0.220635560237-0.816134973767j))*x[1]**o + ((-0.903550415776-0.171078404743j))*x[1] + ((0.74935319435-0.765902424635j))*x[2]**o + ((0.693416887284+0.887832592589j))*x[2]
            ref[(0, 0, 2, 1)]=(0.519146236709-0.247182505045j)-((0.297138348958-0.28734238077j))*(o-1.)
            arg[(0, 0, 2, 2)]=(0.633422422202-0.101190167637j)*x[0]**o + ((0.97239945992+0.316575018624j))*x[0] + ((0.238435393475+0.949826408108j))*x[1]**o + ((0.0341402577101+0.163555044873j))*x[1] + ((0.0746124080792-0.170728082202j))*x[2]**o + ((0.955873192765-0.377876745998j))*x[2]
            ref[(0, 0, 2, 2)]=(1.45444156708+0.390080737884j)-((0.157745037293+0.112984693045j))*(o-1.)
            arg[(0, 0, 2, 3)]=(0.604808788547-0.398423827079j)*x[0]**o + ((-0.643156110454+0.54609673965j))*x[0] + ((0.417404608813-0.350103076655j))*x[1]**o + ((0.296740360874+0.774267154946j))*x[1] + ((-0.57985302924-0.477494316198j))*x[2]**o + ((0.921146841242-0.467392118996j))*x[2]
            ref[(0, 0, 2, 3)]=(0.508545729891-0.186524722166j)-((0.0737267280202-0.204336869989j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.779472454709-0.8369378028j)*x[0]**o + ((-0.8275596467+0.844461967166j))*x[0] + ((0.179964404584-0.450890121031j))*x[1]**o + ((-0.992959549883-0.232893413057j))*x[1] + ((0.781508543213-0.0180331788241j))*x[2]**o + ((0.243979157891-0.0715234719828j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.697269772802-0.382908010264j)-((0.0303334155147-0.217643517109j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.677835982664-0.431449466708j)*x[0]**o + ((-0.796694526823+0.633794834983j))*x[0] + ((0.697162124845-0.177486701632j))*x[1]**o + ((0.920530591914+0.0933909222267j))*x[1] + ((-0.0780435172236-0.347459799067j))*x[2]**o + ((-0.447252522483-0.980147527407j))*x[2]
            ref[(0, 1, 0, 1)]=(0.486769066447-0.604678868802j)-((0.216159098381-0.159399327901j))*(o-1.)
            arg[(0, 1, 0, 2)]=(0.144628974761-0.626079196343j)*x[0]**o + ((-0.402147959184+0.131717333357j))*x[0] + ((0.367931914326+0.658399347026j))*x[1]**o + ((-0.314822756755-0.597284579886j))*x[1] + ((-0.302574219797+0.515852594976j))*x[2]**o + ((0.927720464658+0.0412277977902j))*x[2]
            ref[(0, 1, 0, 2)]=(0.210368209005+0.0619166484602j)-((0.034997778215+0.0913621242766j))*(o-1.)
            arg[(0, 1, 0, 3)]=(0.334165103355-0.417160622132j)*x[0]**o + ((-0.23255335885+0.260416685792j))*x[0] + ((-0.417892344864-0.408789700883j))*x[1]**o + ((-0.74128581414-0.924565865741j))*x[1] + ((-0.381006530857-0.316457709684j))*x[2]**o + ((0.139008710567-0.590557519204j))*x[2]
            ref[(0, 1, 0, 3)]=(-0.649782117395-1.19855736593j)-((-0.0774556287277-0.190401338783j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.646909896706-0.531732782548j)*x[0]**o + ((-0.451091363748-0.367328304145j))*x[0] + ((0.747700844855-0.693680760054j))*x[1]**o + ((-0.738819077641-0.759906588468j))*x[1] + ((0.823333792422-0.377807677561j))*x[2]**o + ((-0.13436465709-0.873297557623j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.200075178954-1.8018768352j)-((0.154020790095-0.267203536694j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.883515518012+0.433350170922j)*x[0]**o + ((-0.736402423075+0.994693945177j))*x[0] + ((-0.869386719768-0.654715827206j))*x[1]**o + ((-0.736797489193+0.241415507223j))*x[1] + ((0.142786797051+0.85439822528j))*x[2]**o + ((-0.495388350475-0.900289939524j))*x[2]
            ref[(0, 1, 1, 1)]=(-1.78935185174+0.484426040936j)-((-0.268352573455+0.105505428166j))*(o-1.)
            arg[(0, 1, 1, 2)]=(-0.765435160688-0.416683628192j)*x[0]**o + ((-0.659698315897+0.525102101587j))*x[0] + ((-0.925483214347-0.949425696884j))*x[1]**o + ((0.438567791325+0.341341389691j))*x[1] + ((0.859087628228-0.767365305255j))*x[2]**o + ((-0.10655296382+0.36962314393j))*x[2]
            ref[(0, 1, 1, 2)]=(-0.579757117599-0.448703997561j)-((-0.138638457801-0.355579105055j))*(o-1.)
            arg[(0, 1, 1, 3)]=(-0.0765000115031-0.190390704219j)*x[0]**o + ((0.463697517789+0.764407721015j))*x[0] + ((-0.0338237314949-0.180846400081j))*x[1]**o + ((-0.21299147038-0.830698927828j))*x[1] + ((0.583596144987-0.870035489212j))*x[2]**o + ((-0.609064434942-0.0674194015487j))*x[2]
            ref[(0, 1, 1, 3)]=(0.0574570072282-0.687491600937j)-((0.0788787336648-0.206878765585j))*(o-1.)
            arg[(0, 1, 2, 0)]=(0.606016784218+0.999597599904j)*x[0]**o + ((-0.494258348007-0.842205387168j))*x[0] + ((0.300901112455-0.620042287428j))*x[1]**o + ((0.303610691645-0.153853159325j))*x[1] + ((0.18740754021+0.443378215782j))*x[2]**o + ((0.0381725804428+0.247833139381j))*x[2]
            ref[(0, 1, 2, 0)]=(0.470925180482+0.0373540605733j)-((0.182387572814+0.137155588043j))*(o-1.)
            arg[(0, 1, 2, 1)]=(-0.632497712155+0.622132249668j)*x[0]**o + ((-0.066143198172+0.649071446615j))*x[0] + ((0.525090011672+0.464773999809j))*x[1]**o + ((0.46056456206+0.158826420098j))*x[1] + ((0.743756182992-0.856359753045j))*x[2]**o + ((0.988429561473+0.15318700049j))*x[2]
            ref[(0, 1, 2, 1)]=(1.00959970393+0.595815681817j)-((0.106058080418+0.0384244160719j))*(o-1.)
            arg[(0, 1, 2, 2)]=(0.291787597204-0.122045825198j)*x[0]**o + ((-0.23244588372+0.75924540952j))*x[0] + ((0.358177569336+0.542745384244j))*x[1]**o + ((-0.275015204212+0.0301466680069j))*x[1] + ((0.751759912608+0.00684637981466j))*x[2]**o + ((-0.821840930577-0.107396485856j))*x[2]
            ref[(0, 1, 2, 2)]=(0.0362115303197+0.554770765266j)-((0.233620846525+0.0712576564769j))*(o-1.)
            arg[(0, 1, 2, 3)]=(0.0244522191622+0.457653909006j)*x[0]**o + ((0.878407964265-0.197956503938j))*x[0] + ((0.696281041802-0.536855501157j))*x[1]**o + ((0.875603323397-0.453006052841j))*x[1] + ((-0.599561385118-0.545417837287j))*x[2]**o + ((-0.113533702199-0.910711820052j))*x[2]
            ref[(0, 1, 2, 3)]=(0.880824730654-1.09314690313j)-((0.0201953126409-0.10410323824j))*(o-1.)
            arg[(0, 2, 0, 0)]=(0.690826891768-0.670917440113j)*x[0]**o + ((-0.587118361265+0.356851446521j))*x[0] + ((-0.132909120779+0.613647448868j))*x[1]**o + ((-0.0689134741522+0.962348785548j))*x[1] + ((-0.797136452617-0.109868247209j))*x[2]**o + ((-0.440807837711-0.377740048998j))*x[2]
            ref[(0, 2, 0, 0)]=(-0.668029177378+0.387160972308j)-((-0.0398697802713-0.0278563730758j))*(o-1.)
            arg[(0, 2, 0, 1)]=(0.726027906772+0.776300306423j)*x[0]**o + ((0.896919161425+0.619123290035j))*x[0] + ((-0.118500979977+0.933287943421j))*x[1]**o + ((0.463974091784-0.527078809847j))*x[1] + ((-0.460967339376-0.461856571687j))*x[2]**o + ((0.609442794109+0.22729094645j))*x[2]
            ref[(0, 2, 0, 1)]=(1.05844781737+0.783533552397j)-((0.0244265979032+0.207955279693j))*(o-1.)
            arg[(0, 2, 0, 2)]=(0.47410239884-0.0138884382686j)*x[0]**o + ((0.0859106113647-0.856621137259j))*x[0] + ((-0.574460008882-0.156700124834j))*x[1]**o + ((-0.710359048977+0.519798710822j))*x[1] + ((-0.320681911053-0.699928636101j))*x[2]**o + ((0.730764379728-0.330257084048j))*x[2]
            ref[(0, 2, 0, 2)]=(-0.15736178949-0.768798354844j)-((-0.0701732535158-0.145086199867j))*(o-1.)
            arg[(0, 2, 0, 3)]=(0.834955347442-0.32171343058j)*x[0]**o + ((0.649851929602+0.382334067041j))*x[0] + ((0.920742461013+0.279490380507j))*x[1]**o + ((0.986355169315-0.0313668313873j))*x[1] + ((0.102842666687-0.93283915585j))*x[2]**o + ((0.273736309632-0.942889864756j))*x[2]
            ref[(0, 2, 0, 3)]=(1.88424194185-0.783492417513j)-((0.309756745857-0.162510367654j))*(o-1.)
            arg[(0, 2, 1, 0)]=(-0.0152518447157-0.347090391465j)*x[0]**o + ((0.130578759976+0.804087977995j))*x[0] + ((0.877936697719-0.338965232001j))*x[1]**o + ((-0.496209701204+0.416713542227j))*x[1] + ((-0.795695002806-0.516312123088j))*x[2]**o + ((-0.539955527408-0.408269038154j))*x[2]
            ref[(0, 2, 1, 0)]=(-0.419298309219-0.194917632243j)-((0.0111649750328-0.200394624426j))*(o-1.)
            arg[(0, 2, 1, 1)]=(0.0443850969922-0.864921286833j)*x[0]**o + ((-0.797206917863+0.113844719869j))*x[0] + ((0.925373364618+0.0301413649934j))*x[1]**o + ((0.304106363668-0.198869692458j))*x[1] + ((0.502644947134+0.317039340935j))*x[2]**o + ((-0.736669737185+0.404449818846j))*x[2]
            ref[(0, 2, 1, 1)]=(0.121316558683-0.0991578673237j)-((0.245400568124-0.0862900968174j))*(o-1.)
            arg[(0, 2, 1, 2)]=(0.559294049483-0.9070828602j)*x[0]**o + ((0.462557104813-0.958987276249j))*x[0] + ((-0.0348456012231+0.97869260533j))*x[1]**o + ((0.376705765814+0.235640042541j))*x[1] + ((-0.0157818212306-0.834427411032j))*x[2]**o + ((-0.311117736705+0.20299304432j))*x[2]
            ref[(0, 2, 1, 2)]=(0.518405880475-0.641585927645j)-((0.0847777711715-0.12713627765j))*(o-1.)
            arg[(0, 2, 1, 3)]=(0.725304872257-0.370115885162j)*x[0]**o + ((-0.316401047394+0.72397566844j))*x[0] + ((-0.270885342191-0.822438140337j))*x[1]**o + ((0.642510453758+0.418691676087j))*x[1] + ((0.47909741039-0.83478542566j))*x[2]**o + ((-0.362041115461+0.622682674091j))*x[2]
            ref[(0, 2, 1, 3)]=(0.44879261568-0.13099471627j)-((0.155586156743-0.337889908526j))*(o-1.)
            arg[(0, 2, 2, 0)]=(0.600169002458+0.271288921345j)*x[0]**o + ((0.6605652353+0.321492871487j))*x[0] + ((0.992806786778-0.796877969062j))*x[1]**o + ((-0.914947364681-0.906399393136j))*x[1] + ((0.0884092996926-0.555855476879j))*x[2]**o + ((0.423126663509+0.215688909162j))*x[2]
            ref[(0, 2, 2, 0)]=(0.925064811528-0.725331068542j)-((0.280230848155-0.180240754099j))*(o-1.)
            arg[(0, 2, 2, 1)]=(-0.612836642619+0.261377081972j)*x[0]**o + ((-0.814715951571-0.678103576903j))*x[0] + ((0.132146394659-0.353969653271j))*x[1]**o + ((0.921043215614-0.8538271457j))*x[1] + ((0.143897369931+0.623201098527j))*x[2]**o + ((-0.153777924891+0.295611458361j))*x[2]
            ref[(0, 2, 2, 1)]=(-0.192121769438-0.352855368507j)-((-0.0561321463381+0.088434754538j))*(o-1.)
            arg[(0, 2, 2, 2)]=(0.332163771588-0.710994936518j)*x[0]**o + ((0.455132387248-0.706938949064j))*x[0] + ((-0.0211156814879+0.503073840012j))*x[1]**o + ((0.696596805187-0.270420521424j))*x[1] + ((0.940940792987+0.415448192146j))*x[2]**o + ((-0.0432048884004+0.84044993821j))*x[2]
            ref[(0, 2, 2, 2)]=(1.18025659356+0.0353087816806j)-((0.208664813848+0.0345878492733j))*(o-1.)
            arg[(0, 2, 2, 3)]=(-0.222211935732+0.568835586137j)*x[0]**o + ((-0.193086622539+0.640573827941j))*x[0] + ((0.696327410182-0.471624610209j))*x[1]**o + ((-0.598458674776+0.567897843667j))*x[1] + ((0.29412508929-0.129117647873j))*x[2]**o + ((-0.148568819868-0.00723851307951j))*x[2]
            ref[(0, 2, 2, 3)]=(-0.0859367767212+0.584663243292j)-((0.128040093957-0.00531777865755j))*(o-1.)
            arg[(0, 3, 0, 0)]=(0.476749445821-0.440862168789j)*x[0]**o + ((-0.200938281256+0.0944299543187j))*x[0] + ((-0.564243560235-0.504649333391j))*x[1]**o + ((-0.555735406643+0.761330973988j))*x[1] + ((-0.454317274768-0.650260432979j))*x[2]**o + ((0.537853862234-0.727705471361j))*x[2]
            ref[(0, 3, 0, 0)]=(-0.380315607423-0.733858239107j)-((-0.090301898197-0.265961989193j))*(o-1.)
            arg[(0, 3, 0, 1)]=(-0.844947393938+0.396086580606j)*x[0]**o + ((0.998319340185-0.161541130686j))*x[0] + ((-0.678503208669-0.664164669216j))*x[1]**o + ((0.0274128957868+0.15314411135j))*x[1] + ((0.188517763555+0.10817537525j))*x[2]**o + ((0.543935498653+0.192578289393j))*x[2]
            ref[(0, 3, 0, 1)]=(0.117367447787+0.0121392783488j)-((-0.222488806509-0.0266504522267j))*(o-1.)
            arg[(0, 3, 0, 2)]=(-0.819315577065-0.233885963068j)*x[0]**o + ((-0.60190564434+0.30522847351j))*x[0] + ((0.39997240583+0.694186868628j))*x[1]**o + ((0.882573992164-0.575309701496j))*x[1] + ((0.347094991095-0.0713140228145j))*x[2]**o + ((-0.155208278759+0.19388837339j))*x[2]
            ref[(0, 3, 0, 2)]=(0.0266059444625+0.156397014074j)-((-0.0120413633567+0.0648311471241j))*(o-1.)
            arg[(0, 3, 0, 3)]=(0.962172861635+0.858968336528j)*x[0]**o + ((-0.358540948478+0.618867178096j))*x[0] + ((-0.181909172398-0.59713345194j))*x[1]**o + ((0.85849203164+0.131866219692j))*x[1] + ((0.360762878145-0.521736370474j))*x[2]**o + ((0.726613153027+0.904575401275j))*x[2]
            ref[(0, 3, 0, 3)]=(1.18379540179+0.697703656588j)-((0.190171094564-0.0433169143144j))*(o-1.)
            arg[(0, 3, 1, 0)]=(0.0141292726803+0.802278062879j)*x[0]**o + ((0.234500779253-0.62716415979j))*x[0] + ((-0.967917667134+0.64971514805j))*x[1]**o + ((0.0244669782054-0.00486330663787j))*x[1] + ((0.168637236849+0.38908324302j))*x[2]**o + ((0.0906464982893+0.0725489171211j))*x[2]
            ref[(0, 3, 1, 0)]=(-0.217768450929+0.640798952321j)-((-0.130858526267+0.306846075658j))*(o-1.)
            arg[(0, 3, 1, 1)]=(0.0988689626079+0.995864857825j)*x[0]**o + ((0.243702278653+0.36396178979j))*x[0] + ((-0.423858001115-0.132438952986j))*x[1]**o + ((0.779224426826+0.049721047998j))*x[1] + ((0.0895507862411-0.487256520451j))*x[2]**o + ((0.937781087761+0.763263117943j))*x[2]
            ref[(0, 3, 1, 1)]=(0.862634770487+0.776557670059j)-((-0.0392397087111+0.062694897398j))*(o-1.)
            arg[(0, 3, 1, 2)]=(-0.114304317846+0.781710059681j)*x[0]**o + ((-0.170820239565+0.708279301043j))*x[0] + ((0.460357332121-0.544689251857j))*x[1]**o + ((0.837068644205-0.213569961471j))*x[1] + ((-0.428888294838-0.643694076829j))*x[2]**o + ((0.790781276957+0.842451682327j))*x[2]
            ref[(0, 3, 1, 2)]=(0.687097200517+0.465243876447j)-((-0.0138058800938-0.0677788781676j))*(o-1.)
            arg[(0, 3, 1, 3)]=(-0.626537636327-0.781711137043j)*x[0]**o + ((-0.503134349978+0.917242475643j))*x[0] + ((-0.0697859758273-0.694027157581j))*x[1]**o + ((-0.0798978998867-0.320960623682j))*x[1] + ((-0.411555752279-0.695249241553j))*x[2]**o + ((-0.848938218654+0.00820354174758j))*x[2]
            ref[(0, 3, 1, 3)]=(-1.26992491648-0.783251071235j)-((-0.184646560739-0.36183125603j))*(o-1.)
            arg[(0, 3, 2, 0)]=(0.715605878929+0.740056088093j)*x[0]**o + ((-0.438369422524+0.558742900845j))*x[0] + ((0.946796949895+0.613761212323j))*x[1]**o + ((0.079632514559+0.891623009875j))*x[1] + ((-0.893053335521-0.289163058547j))*x[2]**o + ((-0.910108668183+0.958919426811j))*x[2]
            ref[(0, 3, 2, 0)]=(-0.249748041422+1.7369697897j)-((0.128224915551+0.177442373645j))*(o-1.)
            arg[(0, 3, 2, 1)]=(0.560665089114+0.463369786135j)*x[0]**o + ((-0.218937424859-0.767475351834j))*x[0] + ((-0.685717777437+0.719944918232j))*x[1]**o + ((-0.617362668768-0.1661943103j))*x[1] + ((-0.496289176418+0.696788226453j))*x[2]**o + ((0.991612849803-0.049687608719j))*x[2]
            ref[(0, 3, 2, 1)]=(-0.233014554283+0.448372829984j)-((-0.103556977457+0.31335048847j))*(o-1.)
            arg[(0, 3, 2, 2)]=(-0.799633438126-0.976893032071j)*x[0]**o + ((0.140069639565-0.529349991904j))*x[0] + ((-0.24340583269+0.569305322918j))*x[1]**o + ((0.526253658662+0.87275320333j))*x[1] + ((-0.923184112992-0.428906777678j))*x[2]**o + ((0.160170024924-0.487518900086j))*x[2]
            ref[(0, 3, 2, 2)]=(-0.569865030329-0.490305087745j)-((-0.327703897301-0.139415747805j))*(o-1.)
            arg[(0, 3, 2, 3)]=(-0.565105742832-0.76684892998j)*x[0]**o + ((0.271746543784+0.894081525018j))*x[0] + ((-0.939973470815-0.184014705539j))*x[1]**o + ((-0.732543477627-0.8526978878j))*x[1] + ((-0.727153539026+0.899899212334j))*x[2]**o + ((0.530331474042-0.0434658227197j))*x[2]
            ref[(0, 3, 2, 3)]=(-1.08134910624-0.0265233043435j)-((-0.372038792112-0.00849407053093j))*(o-1.)
            arg[(1, 0, 0, 0)]=(0.641299722573+0.614072917694j)*x[0]**o + ((-0.155492590658+0.217864167018j))*x[0] + ((0.573062188329+0.421253836226j))*x[1]**o + ((-0.15965791542-0.957178842282j))*x[1] + ((-0.346722711624-0.696187774948j))*x[2]**o + ((0.497309449902+0.9233047021j))*x[2]
            ref[(1, 0, 0, 0)]=(0.524899071551+0.261564502904j)-((0.144606533213+0.056523163162j))*(o-1.)
            arg[(1, 0, 0, 1)]=(-0.276278265836+0.174961842546j)*x[0]**o + ((0.101039709878-0.333287795228j))*x[0] + ((-0.456110827991-0.594990804854j))*x[1]**o + ((-0.400013225776+0.847811734217j))*x[1] + ((0.530317673127+0.81604165439j))*x[2]**o + ((0.589327543853+0.39652419161j))*x[2]
            ref[(1, 0, 0, 1)]=(0.0441413036272+0.653530411341j)-((-0.0336785701168+0.066002115347j))*(o-1.)
            arg[(1, 0, 0, 2)]=(0.294950347928-0.779778933664j)*x[0]**o + ((-0.432803555426+0.0372965022512j))*x[0] + ((-0.733163121597+0.152878249377j))*x[1]**o + ((0.121731690242+0.257851678177j))*x[1] + ((-0.284688467927-0.868231984655j))*x[2]**o + ((0.00711815215358-0.0598080811795j))*x[2]
            ref[(1, 0, 0, 2)]=(-0.513427477313-0.629896284846j)-((-0.120483540266-0.249188778157j))*(o-1.)
            arg[(1, 0, 0, 3)]=(-0.124576422888+0.570811515421j)*x[0]**o + ((0.216476099942+0.27470593531j))*x[0] + ((-0.779720287075-0.00232854802574j))*x[1]**o + ((-0.5544641785+0.612749616624j))*x[1] + ((-0.00850044747073-0.169160574327j))*x[2]**o + ((0.744009381846+0.752702819646j))*x[2]
            ref[(1, 0, 0, 3)]=(-0.253387927073+1.01974038232j)-((-0.152132859572+0.0665537321781j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.28269747054-0.00118071275814j)*x[0]**o + ((0.324364218197-0.202052222903j))*x[0] + ((-0.635046772638+0.539129704407j))*x[1]**o + ((-0.632133920135-0.37052222253j))*x[1] + ((-0.143827502962+0.893289801537j))*x[2]**o + ((-0.440768801425+0.0991776776497j))*x[2]
            ref[(1, 0, 1, 0)]=(-0.905055124752+0.478921012701j)-((-0.176928624357+0.238539798864j))*(o-1.)
            arg[(1, 0, 1, 1)]=(0.186224633778-0.657760806517j)*x[0]**o + ((-0.04344549558+0.772142395955j))*x[0] + ((0.495769979306-0.590979062876j))*x[1]**o + ((-0.00233494715361-0.939830361245j))*x[1] + ((0.473685015826-0.132759526792j))*x[2]**o + ((-0.631221561319+0.652487889126j))*x[2]
            ref[(1, 0, 1, 1)]=(0.239338812429-0.448349736174j)-((0.192613271485-0.230249899364j))*(o-1.)
            arg[(1, 0, 1, 2)]=(-0.864158680993-0.815484075167j)*x[0]**o + ((0.503855460007-0.623136293271j))*x[0] + ((0.88524925451-0.453513740407j))*x[1]**o + ((-0.95598894457+0.599387241534j))*x[1] + ((0.397014024076-0.384019194974j))*x[2]**o + ((0.991842928249-0.379990654682j))*x[2]
            ref[(1, 0, 1, 2)]=(0.47890702064-1.02837835848j)-((0.069684099599-0.275502835091j))*(o-1.)
            arg[(1, 0, 1, 3)]=(0.282883079817+0.668837743209j)*x[0]**o + ((-0.648310788911+0.973803390958j))*x[0] + ((0.670444472723+0.139230775141j))*x[1]**o + ((0.595545770154+0.957860854187j))*x[1] + ((-0.604464720019-0.335696512819j))*x[2]**o + ((-0.60968493225-0.428947249156j))*x[2]
            ref[(1, 0, 1, 3)]=(-0.156793559243+0.98754450076j)-((0.0581438054202+0.0787286675885j))*(o-1.)
            arg[(1, 0, 2, 0)]=(-0.487394865191+0.655020330522j)*x[0]**o + ((0.445728980475+0.135630045537j))*x[0] + ((0.187055516229-0.0240263885157j))*x[1]**o + ((0.0782715820298+0.863000478527j))*x[1] + ((0.869627788739-0.150282193952j))*x[2]**o + ((0.510209774548-0.160068510773j))*x[2]
            ref[(1, 0, 2, 0)]=(0.801749388414+0.659636880673j)-((0.0948814066294+0.0801186246757j))*(o-1.)
            arg[(1, 0, 2, 1)]=(0.00358739313214+0.766304704751j)*x[0]**o + ((-0.422794189732+0.120244771678j))*x[0] + ((-0.766217376181-0.0864079635924j))*x[1]**o + ((-0.691422990734+0.952949203085j))*x[1] + ((-0.228913085227-0.478046726034j))*x[2]**o + ((0.519348043567-0.10310503708j))*x[2]
            ref[(1, 0, 2, 1)]=(-0.793206102588+0.585969476403j)-((-0.165257178046+0.0336416691875j))*(o-1.)
            arg[(1, 0, 2, 2)]=(-0.322867374786-0.743836979052j)*x[0]**o + ((0.519880312624-0.229938846475j))*x[0] + ((0.240147458124-0.754352599344j))*x[1]**o + ((-0.630759351386-0.949330131744j))*x[1] + ((-0.807140445625+0.801770186814j))*x[2]**o + ((0.418166534216-0.263380470273j))*x[2]
            ref[(1, 0, 2, 2)]=(-0.291286433417-1.06953442004j)-((-0.148310060381-0.116069898597j))*(o-1.)
            arg[(1, 0, 2, 3)]=(-0.214996490264-0.344538215705j)*x[0]**o + ((0.411670566398+0.919476015804j))*x[0] + ((0.251393673848+0.476851197848j))*x[1]**o + ((-0.879575107739+0.411129656124j))*x[1] + ((0.320187010548+0.469914399861j))*x[2]**o + ((-0.451204309144+0.515576565978j))*x[2]
            ref[(1, 0, 2, 3)]=(-0.281262328177+1.22420480996j)-((0.0594306990219+0.100371230334j))*(o-1.)
            arg[(1, 1, 0, 0)]=(-0.295274205539+0.0526364892561j)*x[0]**o + ((0.0838106214683+0.739200265635j))*x[0] + ((0.97785994904+0.786380736998j))*x[1]**o + ((-0.297666933322+0.11310166266j))*x[1] + ((0.69431242716-0.386535702944j))*x[2]**o + ((0.500417118393-0.00613701419104j))*x[2]
            ref[(1, 1, 0, 0)]=(0.8317294886+0.649323218707j)-((0.229483028443+0.0754135872183j))*(o-1.)
            arg[(1, 1, 0, 1)]=(0.910699332221-0.17989517704j)*x[0]**o + ((0.00648085134407-0.170672363566j))*x[0] + ((-0.704355654173+0.919818637937j))*x[1]**o + ((-0.829208869576-0.636777945038j))*x[1] + ((0.953101591575-0.535680704843j))*x[2]**o + ((-0.209425560594-0.824227528485j))*x[2]
            ref[(1, 1, 0, 1)]=(0.0636458453985-0.713717540518j)-((0.193240878271+0.0340404593423j))*(o-1.)
            arg[(1, 1, 0, 2)]=(-0.256503975352+0.937189299198j)*x[0]**o + ((-0.98245251498-0.645362110453j))*x[0] + ((-0.375819085434+0.591559252156j))*x[1]**o + ((0.338744416715+0.678715373792j))*x[1] + ((-0.591997233676+0.0731917562872j))*x[2]**o + ((0.843685520866-0.144872026549j))*x[2]
            ref[(1, 1, 0, 2)]=(-0.512171435931+0.745210772216j)-((-0.20405338241+0.266990051274j))*(o-1.)
            arg[(1, 1, 0, 3)]=(-0.168408915495-0.378835874935j)*x[0]**o + ((0.669936622701-0.485362670808j))*x[0] + ((0.636599781462+0.588159347801j))*x[1]**o + ((-0.468931671319+0.89415807855j))*x[1] + ((0.658580110862+0.728268902084j))*x[2]**o + ((0.905673904596+0.98401076942j))*x[2]
            ref[(1, 1, 0, 3)]=(1.1167249164+1.16519927606j)-((0.187795162805+0.156265395825j))*(o-1.)
            arg[(1, 1, 1, 0)]=(-0.0281508099849+0.744293508452j)*x[0]**o + ((-0.3879770914+0.804155169593j))*x[0] + ((-0.716818574142+0.536730220345j))*x[1]**o + ((0.355756177727+0.249884079817j))*x[1] + ((-0.935510937234-0.91107337714j))*x[2]**o + ((-0.999937167445+0.395278388486j))*x[2]
            ref[(1, 1, 1, 0)]=(-1.35631920124+0.909633994776j)-((-0.28008005356+0.0616583919428j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.880386793782+0.648910500589j)*x[0]**o + ((-0.594433810874-0.969774262954j))*x[0] + ((-0.312344738179-0.996021711912j))*x[1]**o + ((0.561872227136+0.294693712577j))*x[1] + ((0.94119230519+0.0577845766757j))*x[2]**o + ((0.795575529462+0.933835224784j))*x[2]
            ref[(1, 1, 1, 1)]=(1.13612415326-0.0152859801201j)-((0.251539060132-0.0482211057746j))*(o-1.)
            arg[(1, 1, 1, 2)]=(0.0824098198944-0.3611157461j)*x[0]**o + ((-0.447617855621+0.457162705197j))*x[0] + ((0.995927131936-0.723756464032j))*x[1]**o + ((-0.289114807381+0.595453616711j))*x[1] + ((0.00600141327198+0.16380686509j))*x[2]**o + ((0.917807270177+0.726491944804j))*x[2]
            ref[(1, 1, 1, 2)]=(0.632706486139+0.429021460835j)-((0.18072306085-0.15351089084j))*(o-1.)
            arg[(1, 1, 1, 3)]=(-0.06325935207+0.95823759893j)*x[0]**o + ((0.31522184045+0.453970972254j))*x[0] + ((0.99185965941-0.318780045276j))*x[1]**o + ((0.857885694613+0.945426645475j))*x[1] + ((-0.533461212461-0.707230633107j))*x[2]**o + ((0.343757335577-0.453827960954j))*x[2]
            ref[(1, 1, 1, 3)]=(0.95600198276+0.438898288661j)-((0.0658565158131-0.0112955132423j))*(o-1.)
            arg[(1, 1, 2, 0)]=(-0.86449898323-0.396404802792j)*x[0]**o + ((0.932392889729-0.0299529119538j))*x[0] + ((0.987868061985-0.354231136692j))*x[1]**o + ((-0.279931289617-0.121473821653j))*x[1] + ((0.643767733406+0.173045361471j))*x[2]**o + ((0.656635981617+0.716654649417j))*x[2]
            ref[(1, 1, 2, 0)]=(1.03811719695-0.00618133110152j)-((0.12785613536-0.0962650963356j))*(o-1.)
            arg[(1, 1, 2, 1)]=(-0.630609074404+0.420575559169j)*x[0]**o + ((-0.117599263148-0.943059910774j))*x[0] + ((-0.0662027498024+0.844024759079j))*x[1]**o + ((0.89563386163-0.40968540559j))*x[1] + ((0.768722558542+0.425901593542j))*x[2]**o + ((-0.015076052056+0.0887537956957j))*x[2]
            ref[(1, 1, 2, 1)]=(0.417434640381+0.213255195561j)-((0.0119851223893+0.281750318632j))*(o-1.)
            arg[(1, 1, 2, 2)]=(-0.0168083846612+0.807188593702j)*x[0]**o + ((-0.414396835412-0.741994683054j))*x[0] + ((0.150589975394-0.0640584239518j))*x[1]**o + ((-0.910630974623+0.0925183031142j))*x[1] + ((-0.514882173518-0.674725586285j))*x[2]**o + ((0.0750501324651-0.323338635817j))*x[2]
            ref[(1, 1, 2, 2)]=(-0.815539130178-0.452205216146j)-((-0.0635167637975+0.0114007639109j))*(o-1.)
            arg[(1, 1, 2, 3)]=(-0.759597597026-0.434634010139j)*x[0]**o + ((0.142558896455+0.98832762756j))*x[0] + ((0.742492650769-0.467099464462j))*x[1]**o + ((0.0376586910642+0.867008892399j))*x[1] + ((-0.733617834997+0.436686281124j))*x[2]**o + ((0.55752250078+0.439608849344j))*x[2]
            ref[(1, 1, 2, 3)]=(-0.00649134647717+0.914949087913j)-((-0.125120463542-0.0775078655794j))*(o-1.)
            arg[(1, 2, 0, 0)]=(-0.46958095905-0.767610675264j)*x[0]**o + ((0.909055647804-0.0115360351498j))*x[0] + ((-0.575771400084-0.909163341882j))*x[1]**o + ((0.138980914351+0.626512504687j))*x[1] + ((-0.0984035464091+0.0335230138469j))*x[2]**o + ((-0.291307886632-0.66796403063j))*x[2]
            ref[(1, 2, 0, 0)]=(-0.19351361501-0.848119282196j)-((-0.190625984257-0.273875167217j))*(o-1.)
            arg[(1, 2, 0, 1)]=(-0.376834589678-0.405416851373j)*x[0]**o + ((-0.78078604957-0.0837429582654j))*x[0] + ((-0.365921750075+0.330356672168j))*x[1]**o + ((0.286506571523-0.0795838398262j))*x[1] + ((0.230099370928+0.218850880462j))*x[2]**o + ((-0.484135180507+0.57962366379j))*x[2]
            ref[(1, 2, 0, 1)]=(-0.745535813689+0.280043783477j)-((-0.0854428281375+0.0239651168761j))*(o-1.)
            arg[(1, 2, 0, 2)]=(-0.544030114804-0.143770110211j)*x[0]**o + ((-0.138174306208+0.615735839306j))*x[0] + ((-0.612684231672-0.0196858309879j))*x[1]**o + ((0.160577265157-0.850974516991j))*x[1] + ((0.623795935038-0.606469223635j))*x[2]**o + ((0.656161374765-0.107748844822j))*x[2]
            ref[(1, 2, 0, 2)]=(0.0728229611378-0.55645634367j)-((-0.0888197352396-0.128320860806j))*(o-1.)
            arg[(1, 2, 0, 3)]=(0.92775499777+0.201936892721j)*x[0]**o + ((-0.00427450343474-0.00362493443907j))*x[0] + ((0.127978025008-0.670125186933j))*x[1]**o + ((0.116055916939+0.252985680761j))*x[1] + ((0.971201142502-0.980801743857j))*x[2]**o + ((0.0124943048481-0.0240123012858j))*x[2]
            ref[(1, 2, 0, 3)]=(1.07560494182-0.611820796517j)-((0.33782236088-0.241498339678j))*(o-1.)
            arg[(1, 2, 1, 0)]=(0.332773747588+0.339617236204j)*x[0]**o + ((-0.0698357437255-0.486024699765j))*x[0] + ((0.732788157581-0.157545075449j))*x[1]**o + ((0.0203925755681-0.925759482537j))*x[1] + ((0.17834030785-0.640162456618j))*x[2]**o + ((0.259225869106+0.619725236153j))*x[2]
            ref[(1, 2, 1, 0)]=(0.726842456984-0.625074621006j)-((0.207317035503-0.0763483826439j))*(o-1.)
            arg[(1, 2, 1, 1)]=(0.699569505568+0.00481284957173j)*x[0]**o + ((0.781011917931+0.908954483041j))*x[0] + ((-0.896425200886+0.197985858166j))*x[1]**o + ((0.77293772573-0.784949893868j))*x[1] + ((-0.36209160645-0.893650855148j))*x[2]**o + ((-0.857788702443-0.484887878704j))*x[2]
            ref[(1, 2, 1, 1)]=(0.068606819725-0.525867718471j)-((-0.0931578836279-0.115142024568j))*(o-1.)
            arg[(1, 2, 1, 2)]=(-0.642257118774+0.490566016731j)*x[0]**o + ((-0.297751177421-0.499442956255j))*x[0] + ((-0.343640082888-0.0326423424484j))*x[1]**o + ((-0.342102924019+0.792208708385j))*x[1] + ((-0.986872913114-0.0184852904887j))*x[2]**o + ((-0.746115527909-0.426719084313j))*x[2]
            ref[(1, 2, 1, 2)]=(-1.67936987206+0.152742525806j)-((-0.328795019129+0.0732397306323j))*(o-1.)
            arg[(1, 2, 1, 3)]=(0.882639780176-0.804817551964j)*x[0]**o + ((0.499368284558-0.0176740980905j))*x[0] + ((0.0706940592708+0.841752748905j))*x[1]**o + ((-0.984645215378-0.57296195315j))*x[1] + ((0.675695364669+0.306354167945j))*x[2]**o + ((0.944185548977+0.186618237025j))*x[2]
            ref[(1, 2, 1, 3)]=(1.04396891114-0.0303642246646j)-((0.271504867353+0.0572148941478j))*(o-1.)
            arg[(1, 2, 2, 0)]=(0.78527451249-0.0279067022615j)*x[0]**o + ((0.53246453454+0.515831423019j))*x[0] + ((0.611431736741-0.0419336618593j))*x[1]**o + ((0.962266369001+0.661202974122j))*x[1] + ((-0.954800504105+0.0619820163894j))*x[2]**o + ((-0.254842817407-0.837892902288j))*x[2]
            ref[(1, 2, 2, 0)]=(0.84089691563+0.165641573561j)-((0.073650957521-0.00130972462191j))*(o-1.)
            arg[(1, 2, 2, 1)]=(0.99554482036-0.771105620434j)*x[0]**o + ((-0.845991779256-0.377037016218j))*x[0] + ((0.39950941159-0.720101727573j))*x[1]**o + ((-0.975897521722-0.112531705657j))*x[1] + ((-0.577175420993-0.747516228101j))*x[2]**o + ((-0.272283099296+0.348534732518j))*x[2]
            ref[(1, 2, 2, 1)]=(-0.638146794658-1.18987878273j)-((0.136313135159-0.373120596018j))*(o-1.)
            arg[(1, 2, 2, 2)]=(-0.812135177321-0.389502986177j)*x[0]**o + ((0.540868429635-0.373966005051j))*x[0] + ((0.638082352646-0.933679639631j))*x[1]**o + ((-0.584221648881-0.628738257736j))*x[1] + ((-0.844616345656-0.376101171813j))*x[2]**o + ((-0.296247852703-0.413827547557j))*x[2]
            ref[(1, 2, 2, 2)]=(-0.67913512114-1.55790780398j)-((-0.169778195055-0.28321396627j))*(o-1.)
            arg[(1, 2, 2, 3)]=(0.187235706463-0.863065539632j)*x[0]**o + ((-0.0816907507638+0.108652944657j))*x[0] + ((0.306178687616+0.474022765934j))*x[1]**o + ((0.241526739349+0.944878139737j))*x[1] + ((-0.0902547454433+0.205766690516j))*x[2]**o + ((-0.942775861367-0.900192496995j))*x[2]
            ref[(1, 2, 2, 3)]=(-0.189890112073-0.0149687478918j)-((0.0671932747726-0.0305460138637j))*(o-1.)
            arg[(1, 3, 0, 0)]=(0.373446547705+0.888989194247j)*x[0]**o + ((-0.244024621257+0.618751004942j))*x[0] + ((-0.951768785347+0.0402722692004j))*x[1]**o + ((0.482509470035-0.261750393703j))*x[1] + ((0.233665822675+0.608373658179j))*x[2]**o + ((0.308093105476-0.0316649262013j))*x[2]
            ref[(1, 3, 0, 0)]=(0.100960769644+0.931485403332j)-((-0.0574427358277+0.256272520271j))*(o-1.)
            arg[(1, 3, 0, 1)]=(0.963993951043-0.556505965657j)*x[0]**o + ((-0.829839486626+0.0500962467778j))*x[0] + ((0.801530687142-0.2348266485j))*x[1]**o + ((-0.919058796487-0.177078039326j))*x[1] + ((-0.46822599021+0.758040060623j))*x[2]**o + ((0.800447697902-0.472783087683j))*x[2]
            ref[(1, 3, 0, 1)]=(0.174424031382-0.316528716882j)-((0.216216441329-0.00554875892236j))*(o-1.)
            arg[(1, 3, 0, 2)]=(0.231041252204-0.719835901012j)*x[0]**o + ((-0.0637949864345+0.619265817086j))*x[0] + ((-0.623153312109-0.468938645682j))*x[1]**o + ((-0.522379861759+0.920518487573j))*x[1] + ((-0.449044910087-0.737028632853j))*x[2]**o + ((0.0327421820622+0.162227424411j))*x[2]
            ref[(1, 3, 0, 2)]=(-0.697294818062-0.111895725238j)-((-0.140192828332-0.320967196591j))*(o-1.)
            arg[(1, 3, 0, 3)]=(-0.708419817755-0.341401043502j)*x[0]**o + ((0.748129193074+0.370679164293j))*x[0] + ((-0.335877889943+0.0661685465217j))*x[1]**o + ((0.540749753143-0.631685391975j))*x[1] + ((0.141128563575-0.361734402615j))*x[2]**o + ((-0.77334532005+0.656443509715j))*x[2]
            ref[(1, 3, 0, 3)]=(-0.193817758979-0.120764808781j)-((-0.150528190687-0.106161149933j))*(o-1.)
            arg[(1, 3, 1, 0)]=(0.54294021606+0.33942664165j)*x[0]**o + ((0.583774879585+0.643117845388j))*x[0] + ((0.389883091338+0.625940971255j))*x[1]**o + ((0.106010534478+0.849243081285j))*x[1] + ((0.590978687667+0.831599541454j))*x[2]**o + ((0.715886304254-0.802146252396j))*x[2]
            ref[(1, 3, 1, 0)]=(1.46473685669+1.24359091432j)-((0.253966999178+0.299494525727j))*(o-1.)
            arg[(1, 3, 1, 1)]=(-0.833801291343-0.481829201969j)*x[0]**o + ((0.493205464781-0.264182377465j))*x[0] + ((-0.398421430591+0.859141893182j))*x[1]**o + ((-0.174930901695+0.252309695778j))*x[1] + ((0.729974428031-0.428149254567j))*x[2]**o + ((-0.798787178037-0.532191970107j))*x[2]
            ref[(1, 3, 1, 1)]=(-0.491380454427-0.297450607574j)-((-0.0837080489838-0.00847276055896j))*(o-1.)
            arg[(1, 3, 1, 2)]=(0.989444074641-0.173910012737j)*x[0]**o + ((0.227156783136+0.541098624804j))*x[0] + ((-0.88838652431+0.43066944643j))*x[1]**o + ((-0.876530171306-0.800859751217j))*x[1] + ((0.214936093263+0.0414391295777j))*x[2]**o + ((0.276292788705-0.975760247423j))*x[2]
            ref[(1, 3, 1, 2)]=(-0.0285434779361-0.468661405283j)-((0.0526656072656+0.0496997605452j))*(o-1.)
            arg[(1, 3, 1, 3)]=(0.664776315279+0.519438253789j)*x[0]**o + ((0.15406242698+0.122344065075j))*x[0] + ((-0.54009706579+0.645430818255j))*x[1]**o + ((0.0251581745834+0.487585241538j))*x[1] + ((0.154854719724+0.0149990785892j))*x[2]**o + ((-0.474137197146+0.130598645971j))*x[2]
            ref[(1, 3, 1, 3)]=(-0.00769131318431+0.960198051608j)-((0.0465889948689+0.196644691772j))*(o-1.)
            arg[(1, 3, 2, 0)]=(-0.539921030947-0.684311113492j)*x[0]**o + ((0.855859119999-0.802140518413j))*x[0] + ((-0.247207982242-0.533686499158j))*x[1]**o + ((0.322193856163-0.171997127212j))*x[1] + ((0.791329354867+0.777776046766j))*x[2]**o + ((-0.742755489576-0.487228304835j))*x[2]
            ref[(1, 3, 2, 0)]=(0.219748914132-0.950793758172j)-((0.000700056946209-0.0733702609806j))*(o-1.)
            arg[(1, 3, 2, 1)]=(-0.0260777163431-0.133278925141j)*x[0]**o + ((-0.0224277787235+0.00661115371379j))*x[0] + ((0.280122124427+0.196529556258j))*x[1]**o + ((-0.250671385888+0.167404353713j))*x[1] + ((0.22855901071+0.380931071868j))*x[2]**o + ((-0.805100167334-0.946425904518j))*x[2]
            ref[(1, 3, 2, 1)]=(-0.297797956576-0.164114347053j)-((0.0804339031323+0.0740302838309j))*(o-1.)
            arg[(1, 3, 2, 2)]=(-0.290591417879-0.532919049581j)*x[0]**o + ((-0.872448573918-0.828488503696j))*x[0] + ((0.741404943481-0.619451563618j))*x[1]**o + ((0.0213817713306-0.0972388243162j))*x[1] + ((0.777655427794-0.360204531718j))*x[2]**o + ((-0.557123931725+0.265447035954j))*x[2]
            ref[(1, 3, 2, 2)]=(-0.0898608904579-1.08642771849j)-((0.204744825566-0.252095857486j))*(o-1.)
            arg[(1, 3, 2, 3)]=(-0.575711344493-0.978720321048j)*x[0]**o + ((-0.440600747247+0.323822657807j))*x[0] + ((0.294752250936-0.709491411588j))*x[1]**o + ((-0.548733960059-0.280169435344j))*x[1] + ((-0.935571652288+0.609343529013j))*x[2]**o + ((-0.988878958653-0.91068403687j))*x[2]
            ref[(1, 3, 2, 3)]=(-1.5973722059-0.972949509015j)-((-0.202755124307-0.17981136727j))*(o-1.)
            arg[(2, 0, 0, 0)]=(-0.199317355741+0.865648797777j)*x[0]**o + ((0.0970524889375+0.465558793083j))*x[0] + ((-0.892897504946-0.349365953127j))*x[1]**o + ((0.549287351158+0.327812805038j))*x[1] + ((-0.551905986111+0.675449868886j))*x[2]**o + ((-0.247367436458+0.859652982642j))*x[2]
            ref[(2, 0, 0, 0)]=(-0.62257422158+1.42237864715j)-((-0.274020141133+0.198622118923j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.658651049623-0.498780846217j)*x[0]**o + ((0.118397385707-0.266377268814j))*x[0] + ((0.744713260323+0.636752726891j))*x[1]**o + ((0.878877863798+0.472233542534j))*x[1] + ((0.937522795175+0.166433053478j))*x[2]**o + ((-0.850574458402-0.289593311135j))*x[2]
            ref[(2, 0, 0, 1)]=(0.585142898489+0.110333948368j)-((0.170597500979+0.050734155692j))*(o-1.)
            arg[(2, 0, 0, 2)]=(-0.512148900952-0.446600571309j)*x[0]**o + ((0.456419582072+0.607189136569j))*x[0] + ((-0.192348880309+0.0843383624285j))*x[1]**o + ((-0.623376866405-0.16974267261j))*x[1] + ((0.183309645774-0.462487799815j))*x[2]**o + ((-0.663154316534-0.294703935058j))*x[2]
            ref[(2, 0, 0, 2)]=(-0.675649868177-0.341003739897j)-((-0.0868646892478-0.137458334782j))*(o-1.)
            arg[(2, 0, 0, 3)]=(0.6841748666+0.692585557542j)*x[0]**o + ((-0.145494787957-0.220811514068j))*x[0] + ((0.447174880727-0.0411560182431j))*x[1]**o + ((-0.8358840135-0.844879708135j))*x[1] + ((-0.994239068995-0.76735984264j))*x[2]**o + ((-0.717429250018+0.452156144497j))*x[2]
            ref[(2, 0, 0, 3)]=(-0.780848686571-0.364732690524j)-((0.022851779722-0.0193217172236j))*(o-1.)
            arg[(2, 0, 1, 0)]=(0.940532892504-0.9240368621j)*x[0]**o + ((-0.520745888035-0.131206179976j))*x[0] + ((-0.200827769307+0.230986074724j))*x[1]**o + ((-0.855694256008-0.278213603336j))*x[1] + ((0.862291540088-0.580880153622j))*x[2]**o + ((-0.815603769675+0.552612229812j))*x[2]
            ref[(2, 0, 1, 0)]=(-0.295023625217-0.565369247249j)-((0.266999443881-0.2123218235j))*(o-1.)
            arg[(2, 0, 1, 1)]=(0.00246355029728+0.89514246934j)*x[0]**o + ((0.286437279732-0.93090789727j))*x[0] + ((-0.72305567644-0.146349627673j))*x[1]**o + ((0.638799398833+0.0248395867002j))*x[1] + ((-0.945351715419-0.632177471883j))*x[2]**o + ((-0.108184955993+0.41937182628j))*x[2]
            ref[(2, 0, 1, 1)]=(-0.424446059495-0.185040557253j)-((-0.277657306927+0.019435894964j))*(o-1.)
            arg[(2, 0, 1, 2)]=(-0.71561989519-0.930731705547j)*x[0]**o + ((-0.0604197185163-0.330746247102j))*x[0] + ((0.120444510293-0.923451348799j))*x[1]**o + ((0.188930116505+0.467930941813j))*x[1] + ((0.383282708614+0.622555991358j))*x[2]**o + ((0.680971792662+0.0759625157163j))*x[2]
            ref[(2, 0, 1, 2)]=(0.298794757183-0.50923992628j)-((-0.0353154460473-0.205271177165j))*(o-1.)
            arg[(2, 0, 1, 3)]=(-0.79594203006-0.146488841203j)*x[0]**o + ((0.706391581637+0.858195155116j))*x[0] + ((-0.899942712961-0.941978541273j))*x[1]**o + ((-0.935964567407+0.577639844132j))*x[1] + ((-0.293804138543-0.629933320359j))*x[2]**o + ((-0.416768153155-0.857559218181j))*x[2]
            ref[(2, 0, 1, 3)]=(-1.31801501024-0.570062460884j)-((-0.331614813594-0.286400117139j))*(o-1.)
            arg[(2, 0, 2, 0)]=(0.862186229723+0.74354861666j)*x[0]**o + ((-0.677330382453+0.1634128014j))*x[0] + ((0.194390964339-0.491306836725j))*x[1]**o + ((0.237084866128+0.10322633343j))*x[1] + ((-0.0283195386207+0.273321608169j))*x[2]**o + ((-0.458116328869-0.911499953542j))*x[2]
            ref[(2, 0, 2, 0)]=(0.0649479051235-0.0596487153033j)-((0.171376275907+0.0875938980175j))*(o-1.)
            arg[(2, 0, 2, 1)]=(0.440049505369-0.299993321648j)*x[0]**o + ((0.848560923098+0.8453185466j))*x[0] + ((-0.495018841024-0.298257535652j))*x[1]**o + ((-0.354711895161+0.501768059104j))*x[1] + ((-0.423440109515-0.849394960354j))*x[2]**o + ((0.234142975422-0.618439741263j))*x[2]
            ref[(2, 0, 2, 1)]=(0.124791279094-0.359499476607j)-((-0.0797349075283-0.241274302942j))*(o-1.)
            arg[(2, 0, 2, 2)]=(0.809066994716+0.300602293452j)*x[0]**o + ((0.298590819543+0.524128584794j))*x[0] + ((-0.0227411387521-0.364459139431j))*x[1]**o + ((-0.196532424389+0.944954810033j))*x[1] + ((0.0241113584954-0.644938364919j))*x[2]**o + ((0.284607973217+0.259259767113j))*x[2]
            ref[(2, 0, 2, 2)]=(0.598551791415+0.509773975521j)-((0.135072869077-0.11813253515j))*(o-1.)
            arg[(2, 0, 2, 3)]=(0.546013745861-0.977545075402j)*x[0]**o + ((0.167047997117+0.35115551765j))*x[0] + ((0.871310887025-0.552800643536j))*x[1]**o + ((0.477957337896-0.112063379499j))*x[1] + ((0.778556372653+0.0470587393883j))*x[2]**o + ((-0.625426087966+0.886849953532j))*x[2]
            ref[(2, 0, 2, 3)]=(1.10773012629-0.178672443933j)-((0.36598016759-0.247214496592j))*(o-1.)
            arg[(2, 1, 0, 0)]=(0.776891394054-0.442273620549j)*x[0]**o + ((-0.237737112371+0.458288242741j))*x[0] + ((-0.585165143531+0.336542090803j))*x[1]**o + ((-0.342241669585+0.843298166771j))*x[1] + ((-0.572292687421+0.435540193401j))*x[2]**o + ((0.818220450215+0.371487968411j))*x[2]
            ref[(2, 1, 0, 0)]=(-0.0711623843196+1.00144152079j)-((-0.0634277394831+0.0549681106093j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.473034253506-0.650558595633j)*x[0]**o + ((-0.678739713821-0.219804289996j))*x[0] + ((0.7857697981-0.434904417596j))*x[1]**o + ((-0.187037226643-0.169988978796j))*x[1] + ((-0.832677027893-0.834074809147j))*x[2]**o + ((0.083752919783+0.155980431259j))*x[2]
            ref[(2, 1, 0, 1)]=(-0.65098275199-1.07667532995j)-((-0.0866569138833-0.319922970396j))*(o-1.)
            arg[(2, 1, 0, 2)]=(-0.346604873604-0.714890126062j)*x[0]**o + ((0.721633894625+0.244664376941j))*x[0] + ((0.289809743396+0.172191752366j))*x[1]**o + ((-0.368247139468-0.290562131756j))*x[1] + ((-0.897199371341+0.259659250739j))*x[2]**o + ((-0.727542357997-0.795929848604j))*x[2]
            ref[(2, 1, 0, 2)]=(-0.664075052195-0.562433363188j)-((-0.158999083591-0.0471731871596j))*(o-1.)
            arg[(2, 1, 0, 3)]=(0.337387813571+0.97843701585j)*x[0]**o + ((0.704163912393+0.617865792186j))*x[0] + ((-0.023348197308-0.365767351776j))*x[1]**o + ((0.63473377861-0.417783629385j))*x[1] + ((-0.500730397017+0.55480919674j))*x[2]**o + ((0.0757122826865-0.999194831756j))*x[2]
            ref[(2, 1, 0, 3)]=(0.613959596468+0.18418309593j)-((-0.0311151301256+0.194579810136j))*(o-1.)
            arg[(2, 1, 1, 0)]=(0.522619483687-0.996161393513j)*x[0]**o + ((-0.321378252562-0.213560397516j))*x[0] + ((-0.300091289411-0.73159614794j))*x[1]**o + ((0.971490450817+0.223966869622j))*x[1] + ((-0.202070683536+0.574858096888j))*x[2]**o + ((-0.747932879811-0.898877467934j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.038681585408-1.0206852202j)-((0.00340958512333-0.192149907427j))*(o-1.)
            arg[(2, 1, 1, 1)]=(0.442945804181+0.285069134871j)*x[0]**o + ((0.946998189771+0.290411497164j))*x[0] + ((0.202644396369-0.860386836798j))*x[1]**o + ((0.285737541157-0.263480857305j))*x[1] + ((-0.937585539771-0.399939716842j))*x[2]**o + ((0.3065648598+0.500314442343j))*x[2]
            ref[(2, 1, 1, 1)]=(0.623652625754-0.224006168283j)-((-0.0486658898701-0.162542903128j))*(o-1.)
            arg[(2, 1, 1, 2)]=(0.372743600281+0.879622029197j)*x[0]**o + ((0.365263808167-0.299458532707j))*x[0] + ((-0.239008606459-0.51307872815j))*x[1]**o + ((-0.776148826452-0.231813408376j))*x[1] + ((-0.435553914435-0.344845216756j))*x[2]**o + ((-0.841657961699+0.171997342281j))*x[2]
            ref[(2, 1, 1, 2)]=(-0.777180950298-0.168788257256j)-((-0.0503031534356+0.00361634738177j))*(o-1.)
            arg[(2, 1, 1, 3)]=(0.593247268844+0.827419931033j)*x[0]**o + ((0.856950615841-0.314969324232j))*x[0] + ((-0.841580848585-0.493745417393j))*x[1]**o + ((0.257543362233-0.386910520728j))*x[1] + ((0.273217845705-0.450926014499j))*x[2]**o + ((-0.145579733461+0.785091051071j))*x[2]
            ref[(2, 1, 1, 3)]=(0.496899255289-0.0170201473739j)-((0.00414737766083-0.0195419168099j))*(o-1.)
            arg[(2, 1, 2, 0)]=(-0.0471071758564-0.00187891070657j)*x[0]**o + ((-0.364984918707+0.269320549579j))*x[0] + ((0.691230231726-0.622947875644j))*x[1]**o + ((0.761006916317-0.19379318298j))*x[1] + ((-0.110046758295-0.481203537357j))*x[2]**o + ((-0.982045487212-0.189711402049j))*x[2]
            ref[(2, 1, 2, 0)]=(-0.025973596014-0.610107179579j)-((0.0890127162624-0.184338387285j))*(o-1.)
            arg[(2, 1, 2, 1)]=(-0.191496420463+0.659653321274j)*x[0]**o + ((0.31057813182+0.162395870702j))*x[0] + ((0.74904930821+0.728618069194j))*x[1]**o + ((0.9153073701+0.553941556076j))*x[1] + ((-0.130862819392-0.898372968881j))*x[2]**o + ((0.028579541742+0.158898526426j))*x[2]
            ref[(2, 1, 2, 1)]=(0.840577556009+0.682567187396j)-((0.0711150113926+0.0816497369312j))*(o-1.)
            arg[(2, 1, 2, 2)]=(0.450287643442+0.450410717789j)*x[0]**o + ((-0.196884535996-0.782788594014j))*x[0] + ((-0.643078826282+0.49902076872j))*x[1]**o + ((-0.38951722403+0.408754609222j))*x[1] + ((-0.464315943873-0.267525703925j))*x[2]**o + ((-0.666716563719+0.753165913717j))*x[2]
            ref[(2, 1, 2, 2)]=(-0.955112725229+0.530518855755j)-((-0.109517854452+0.113650963764j))*(o-1.)
            arg[(2, 1, 2, 3)]=(-0.549361454541-0.197564389664j)*x[0]**o + ((0.683515730367+0.97201359553j))*x[0] + ((0.472183087147-0.824926311855j))*x[1]**o + ((-0.91145110597+0.513440769312j))*x[1] + ((0.820542330212-0.383107428674j))*x[2]**o + ((0.765594798417+0.732871879494j))*x[2]
            ref[(2, 1, 2, 3)]=(0.640511692816+0.406364057072j)-((0.123893993803-0.234266355032j))*(o-1.)
            arg[(2, 2, 0, 0)]=(0.112333882477-0.713014331413j)*x[0]**o + ((0.718172278049+0.844315121172j))*x[0] + ((-0.42632699945+0.149770687275j))*x[1]**o + ((-0.195412365612+0.646031507987j))*x[1] + ((0.916189428813+0.514797622669j))*x[2]**o + ((-0.168365430478-0.476828978252j))*x[2]
            ref[(2, 2, 0, 0)]=(0.478295396899+0.482535814719j)-((0.100366051973-0.00807433691153j))*(o-1.)
            arg[(2, 2, 0, 1)]=(0.824469232152-0.347640509027j)*x[0]**o + ((0.331802925866-0.49090083721j))*x[0] + ((0.66135175217-0.394922546611j))*x[1]**o + ((0.896023994453+0.275888258199j))*x[1] + ((-0.278542944584-0.122214199919j))*x[2]**o + ((5.73278560743e-05+0.260265527051j))*x[2]
            ref[(2, 2, 0, 1)]=(1.21758114396-0.409762153759j)-((0.201213006623-0.144129542593j))*(o-1.)
            arg[(2, 2, 0, 2)]=(0.745468885438+0.13083116725j)*x[0]**o + ((-0.619408892111-0.470722791794j))*x[0] + ((-0.195538063276-0.62625653652j))*x[1]**o + ((-0.619116899706-0.0459550930047j))*x[1] + ((-0.701303125406+0.789648654328j))*x[2]**o + ((0.37216044432-0.456792223005j))*x[2]
            ref[(2, 2, 0, 2)]=(-0.508868825371-0.339623411373j)-((-0.0252287172074+0.0490372141763j))*(o-1.)
            arg[(2, 2, 0, 3)]=(-0.878790390027+0.331062500521j)*x[0]**o + ((0.199610292915-0.984807316752j))*x[0] + ((0.74443042353+0.769025207085j))*x[1]**o + ((-0.852731579963-0.605980277548j))*x[1] + ((-0.836874233107+0.842833148403j))*x[2]**o + ((-0.402995250513-0.280715023043j))*x[2]
            ref[(2, 2, 0, 3)]=(-1.01367536858+0.0357091193327j)-((-0.161872366601+0.323820142668j))*(o-1.)
            arg[(2, 2, 1, 0)]=(0.793373937561-0.83422143529j)*x[0]**o + ((-0.47376347773-0.0724564403269j))*x[0] + ((0.447997675069-0.662054838957j))*x[1]**o + ((0.545640780114+0.878368920557j))*x[1] + ((0.9911244336-0.671256933477j))*x[2]**o + ((-0.533252237009-0.761921928868j))*x[2]
            ref[(2, 2, 1, 0)]=(0.885560555802-1.06177132818j)-((0.372082674372-0.361255534621j))*(o-1.)
            arg[(2, 2, 1, 1)]=(0.158751274556+0.156019886507j)*x[0]**o + ((0.265719604976+0.372587534523j))*x[0] + ((0.0748326834904+0.748755046932j))*x[1]**o + ((-0.137301192489+0.292624935153j))*x[1] + ((-0.513124496093-0.119076889988j))*x[2]**o + ((0.281478576596-0.734568222404j))*x[2]
            ref[(2, 2, 1, 1)]=(0.0651782255179+0.358171145362j)-((-0.0465900896744+0.130949673909j))*(o-1.)
            arg[(2, 2, 1, 2)]=(0.27652071404-0.032061047653j)*x[0]**o + ((0.333699280572-0.711598950165j))*x[0] + ((-0.471894251319-0.115547405206j))*x[1]**o + ((0.707887879793-0.4546684168j))*x[1] + ((0.0221670767035+0.253071230774j))*x[2]**o + ((-0.851039721843-0.772629717161j))*x[2]
            ref[(2, 2, 1, 2)]=(0.00867048897318-0.916717153105j)-((-0.0288677434293+0.0175771296526j))*(o-1.)
            arg[(2, 2, 1, 3)]=(0.871786739041-0.16054084211j)*x[0]**o + ((-0.864004059805+0.562326750302j))*x[0] + ((-0.612725772868+0.559415631397j))*x[1]**o + ((0.553531460103+0.578708959099j))*x[1] + ((-0.476069066975+0.0606757626114j))*x[2]**o + ((-0.0617434137339+0.218526519279j))*x[2]
            ref[(2, 2, 1, 3)]=(-0.294612057119+0.90955639029j)-((-0.0361680168004+0.0765917586498j))*(o-1.)
            arg[(2, 2, 2, 0)]=(0.155232457133+0.317521858425j)*x[0]**o + ((0.119526908268-0.42476821298j))*x[0] + ((-0.664627686074-0.719849046089j))*x[1]**o + ((-0.499842335718+0.869081178283j))*x[1] + ((-0.553112275158+0.730909109641j))*x[2]**o + ((-0.597072755869+0.339494088292j))*x[2]
            ref[(2, 2, 2, 0)]=(-1.01994784371+0.556194487786j)-((-0.177084584016+0.054763653663j))*(o-1.)
            arg[(2, 2, 2, 1)]=(0.325453086144+0.262319293562j)*x[0]**o + ((-0.290863449719+0.582324300543j))*x[0] + ((-0.0946219665714+0.0807673764889j))*x[1]**o + ((0.263390709938-0.0951162696842j))*x[1] + ((-0.53704765363-0.167609513095j))*x[2]**o + ((-0.616030169129+0.0925029992747j))*x[2]
            ref[(2, 2, 2, 1)]=(-0.474859721484+0.377594093544j)-((-0.0510360890095+0.0292461928259j))*(o-1.)
            arg[(2, 2, 2, 2)]=(0.224648128517+0.3594146436j)*x[0]**o + ((-0.616933536029-0.852767897926j))*x[0] + ((-0.950307362306+0.381831558055j))*x[1]**o + ((0.135161784768+0.852683110994j))*x[1] + ((0.776015029322+0.457449888674j))*x[2]**o + ((-0.395425496579+0.613598549246j))*x[2]
            ref[(2, 2, 2, 2)]=(-0.413420726154+0.906104926321j)-((0.00839263258879+0.199782681721j))*(o-1.)
            arg[(2, 2, 2, 3)]=(0.276194675101-0.0809246423578j)*x[0]**o + ((0.216280858116+0.972544272397j))*x[0] + ((-0.328721875906-0.0964997958201j))*x[1]**o + ((0.913253112471+0.469291595585j))*x[1] + ((-0.967227832688-0.760499428321j))*x[2]**o + ((0.474547119057-0.386606331043j))*x[2]
            ref[(2, 2, 2, 3)]=(0.292163028075+0.0586528352197j)-((-0.169959172249-0.156320644417j))*(o-1.)
            arg[(2, 3, 0, 0)]=(-0.656571072736-0.902404946376j)*x[0]**o + ((-0.886766828753+0.161357177736j))*x[0] + ((-0.120676755008-0.814085163792j))*x[1]**o + ((0.728695950487-0.68969250278j))*x[1] + ((0.257460541666+0.770289344424j))*x[2]**o + ((-0.0385741146017+0.918817117943j))*x[2]
            ref[(2, 3, 0, 0)]=(-0.358216139472-0.277859486423j)-((-0.0866312143462-0.157700127624j))*(o-1.)
            arg[(2, 3, 0, 1)]=(0.717014928392-0.57049869327j)*x[0]**o + ((-0.0406610339249-0.693797780215j))*x[0] + ((-0.0604991673225-0.0515894925907j))*x[1]**o + ((0.29590604794-0.223967173411j))*x[1] + ((0.545261847283+0.536380677906j))*x[2]**o + ((0.00472854559211+0.801632804221j))*x[2]
            ref[(2, 3, 0, 1)]=(0.730875583979-0.10091982868j)-((0.200296268059-0.0142845846592j))*(o-1.)
            arg[(2, 3, 0, 2)]=(0.391274254458-0.514694690511j)*x[0]**o + ((-0.373042111663-0.931687749992j))*x[0] + ((0.666187051946+0.719686852312j))*x[1]**o + ((-0.67475746713+0.408179583806j))*x[1] + ((0.628955134382-0.68658253206j))*x[2]**o + ((0.950393427129-0.900858713571j))*x[2]
            ref[(2, 3, 0, 2)]=(0.794505144561-0.952978625008j)-((0.281069406798-0.0802650617098j))*(o-1.)
            arg[(2, 3, 0, 3)]=(-0.656038080786-0.350992347423j)*x[0]**o + ((-0.960404635803-0.813752441046j))*x[0] + ((0.994184296278-0.179714588702j))*x[1]**o + ((0.153497022717+0.970960574765j))*x[1] + ((0.844349684726-0.980353713651j))*x[2]**o + ((0.351829329812-0.0414260026397j))*x[2]
            ref[(2, 3, 0, 3)]=(0.363708808472-0.697639259349j)-((0.197082650036-0.251843441629j))*(o-1.)
            arg[(2, 3, 1, 0)]=(-0.677414899279+0.270375464956j)*x[0]**o + ((-0.0346119110731-0.089020390789j))*x[0] + ((0.947286734833-0.129438130581j))*x[1]**o + ((0.240096359342+0.733113780419j))*x[1] + ((0.408849025158+0.126564305852j))*x[2]**o + ((-0.178071657982-0.353325721104j))*x[2]
            ref[(2, 3, 1, 0)]=(0.3530668255+0.279134654376j)-((0.113120143452+0.0445836067045j))*(o-1.)
            arg[(2, 3, 1, 1)]=(0.359078212071+0.524786983789j)*x[0]**o + ((-0.949494218781+0.555839852746j))*x[0] + ((-0.352765137761-0.128774936873j))*x[1]**o + ((0.0769644491628+0.762539142903j))*x[1] + ((0.853226868612+0.448109502402j))*x[2]**o + ((-0.969293982065+0.435387911444j))*x[2]
            ref[(2, 3, 1, 1)]=(-0.491141904381+1.29894422821j)-((0.143256657154+0.140686924886j))*(o-1.)
            arg[(2, 3, 1, 2)]=(0.657588019929-0.838760867249j)*x[0]**o + ((0.0104962925381+0.679363169781j))*x[0] + ((-0.947810021242+0.461093394639j))*x[1]**o + ((-0.865990213048-0.737342028334j))*x[1] + ((0.33313630639+0.644111429033j))*x[2]**o + ((0.541888811748+0.0423025523353j))*x[2]
            ref[(2, 3, 1, 2)]=(-0.135345401843+0.125383825103j)-((0.00715238417943+0.0444073260705j))*(o-1.)
            arg[(2, 3, 1, 3)]=(-0.968824707558+0.167896440164j)*x[0]**o + ((-0.138504625542+0.275941934922j))*x[0] + ((-0.942960805285+0.77439312113j))*x[1]**o + ((0.722481753416+0.836864290935j))*x[1] + ((-0.416078889968+0.0722391542129j))*x[2]**o + ((-0.641461894368-0.661802411731j))*x[2]
            ref[(2, 3, 1, 3)]=(-1.19267458465+0.732766264816j)-((-0.387977400469+0.169088119251j))*(o-1.)
            arg[(2, 3, 2, 0)]=(-0.907994613668+0.832528276706j)*x[0]**o + ((-0.0059813180479-0.623139897534j))*x[0] + ((0.493742062179-0.115547341066j))*x[1]**o + ((0.791342818941-0.486094732629j))*x[1] + ((0.587464492192-0.425983924324j))*x[2]**o + ((0.415662449756-0.13233675344j))*x[2]
            ref[(2, 3, 2, 0)]=(0.687117945676-0.475287186144j)-((0.0288686567839+0.048499501886j))*(o-1.)
            arg[(2, 3, 2, 1)]=(-0.244334587197-0.905351410685j)*x[0]**o + ((0.632681320886-0.79971401418j))*x[0] + ((0.656133454631-0.301713982343j))*x[1]**o + ((-0.70914113275+0.677696085377j))*x[1] + ((0.650937111795-0.250913383415j))*x[2]**o + ((-0.215267910557+0.984165782615j))*x[2]
            ref[(2, 3, 2, 1)]=(0.385504128404-0.297915461316j)-((0.177122663205-0.242996462741j))*(o-1.)
            arg[(2, 3, 2, 2)]=(-0.68680862724-0.0485876880289j)*x[0]**o + ((-0.700571584196+0.390150086045j))*x[0] + ((0.0612158476396-0.301352007173j))*x[1]**o + ((0.831781414328-0.443243844829j))*x[1] + ((-0.617411838546-0.0826847328565j))*x[2]**o + ((-0.0836390205517+0.710818854121j))*x[2]
            ref[(2, 3, 2, 2)]=(-0.597716904283+0.112550333639j)-((-0.207167436358-0.072104071343j))*(o-1.)
            arg[(2, 3, 2, 3)]=(0.162045056877+0.600076117248j)*x[0]**o + ((0.649129540063-0.828535568233j))*x[0] + ((-0.9863498619+0.123292631461j))*x[1]**o + ((0.488516171931+0.340787964138j))*x[1] + ((0.482189818715-0.0479207308317j))*x[2]**o + ((-0.736111796024+0.740778052903j))*x[2]
            ref[(2, 3, 2, 3)]=(0.0297094648314+0.464239233342j)-((-0.0570191643846+0.112574669646j))*(o-1.)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedSolution_rank0(self):
        """
        tests integral of rank 0 Data on ReducedFunction

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.413158804865-0.0102734045722j)*x[0] + ((-0.542838213649+0.37616470416j))*x[1]
            ref=(-0.0648397043921+0.182945649794j)
        else:
            arg=(-0.87238838479+0.465055331366j)*x[0] + ((-0.476423818121-0.247359942012j))*x[1] + ((0.184430924532-0.39199247784j))*x[2]
            ref=(-0.58219063919-0.0871485442427j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedSolution_rank1(self):
        """
        tests integral of rank 1 Data on ReducedFunction

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.12144649708-0.918224620021j)*x[0] + ((-0.802750502716+0.282114790615j))*x[1]
            ref[(0,)]=(-0.462098499898-0.318054914703j)
            arg[(1,)]=(-0.780034806704+0.312327345719j)*x[0] + ((0.473935431337-0.557478968593j))*x[1]
            ref[(1,)]=(-0.153049687684-0.122575811437j)
            arg[(2,)]=(-0.46716991632-0.196269079237j)*x[0] + ((0.135462713887-0.653687031785j))*x[1]
            ref[(2,)]=(-0.165853601216-0.424978055511j)
            arg[(3,)]=(-0.0103821000842+0.876058903099j)*x[0] + ((-0.869279756499+0.294124485577j))*x[1]
            ref[(3,)]=(-0.439830928291+0.585091694338j)
        else:
            arg[(0,)]=(0.00287909419752+0.969380890611j)*x[0] + ((-0.286026723735+0.374472281292j))*x[1] + ((-0.0789198243362+0.874286794358j))*x[2]
            ref[(0,)]=(-0.181033726937+1.10906998313j)
            arg[(1,)]=(-0.0122508545645+0.553189007752j)*x[0] + ((-0.783922851515+0.604938246803j))*x[1] + ((0.936987681831+0.377124620624j))*x[2]
            ref[(1,)]=(0.0704069878758+0.767625937589j)
            arg[(2,)]=(-0.556189055082+0.226961290804j)*x[0] + ((0.656130157967-0.687624133115j))*x[1] + ((0.802302895804-0.988535601552j))*x[2]
            ref[(2,)]=(0.451121999345-0.724599221932j)
            arg[(3,)]=(-0.417589987064-0.667531262114j)*x[0] + ((0.65308040807+0.0926153063121j))*x[1] + ((-0.525624509674-0.262121849838j))*x[2]
            ref[(3,)]=(-0.145067044334-0.41851890282j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedSolution_rank2(self):
        """
        tests integral of rank 2 Data on ReducedFunction

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 3),w)
        ref=numpy.zeros((2, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.956102707769-0.028258491544j)*x[0] + ((0.834388100824+0.406200174737j))*x[1]
            ref[(0, 0)]=(0.895245404296+0.188970841596j)
            arg[(0, 1)]=(0.239500831199+0.793790394302j)*x[0] + ((-0.903722869791+0.298776181344j))*x[1]
            ref[(0, 1)]=(-0.332111019296+0.546283287823j)
            arg[(0, 2)]=(0.224422612204+0.344914921173j)*x[0] + ((0.462336455793-0.215575459196j))*x[1]
            ref[(0, 2)]=(0.343379533999+0.0646697309885j)
            arg[(1, 0)]=(0.544845448904-0.186723243379j)*x[0] + ((0.379454435167+0.324191089077j))*x[1]
            ref[(1, 0)]=(0.462149942035+0.0687339228489j)
            arg[(1, 1)]=(-0.0637240552364+0.666615011091j)*x[0] + ((0.676715760826-0.685013337656j))*x[1]
            ref[(1, 1)]=(0.306495852795-0.00919916328288j)
            arg[(1, 2)]=(-0.997394042316-0.0312905835973j)*x[0] + ((0.507554273144+0.127726826824j))*x[1]
            ref[(1, 2)]=(-0.244919884586+0.0482181216131j)
        else:
            arg[(0, 0)]=(-0.116592568109-0.764333346504j)*x[0] + ((-0.351788033298+0.0470118319873j))*x[1] + ((0.838394501877+0.1907248553j))*x[2]
            ref[(0, 0)]=(0.185006950235-0.263298329608j)
            arg[(0, 1)]=(-0.692416076002+0.263242432846j)*x[0] + ((-0.000106875846724+0.982390171531j))*x[1] + ((0.523211179158-0.646316629523j))*x[2]
            ref[(0, 1)]=(-0.0846558863454+0.299657987427j)
            arg[(0, 2)]=(0.143008702195+0.24850033635j)*x[0] + ((0.60848506922-0.881329864685j))*x[1] + ((-0.731687208557+0.422048511607j))*x[2]
            ref[(0, 2)]=(0.00990328142874-0.105390508364j)
            arg[(1, 0)]=(-0.500882451852-0.531468370875j)*x[0] + ((-0.836909074531+0.453810013558j))*x[1] + ((-0.988210461818+0.668065559649j))*x[2]
            ref[(1, 0)]=(-1.1630009941+0.295203601166j)
            arg[(1, 1)]=(0.443990604749+0.408471164058j)*x[0] + ((0.00305586232618-0.475128462825j))*x[1] + ((0.83776745953+0.421583394528j))*x[2]
            ref[(1, 1)]=(0.642406963303+0.17746304788j)
            arg[(1, 2)]=(0.140507159733+0.154333082609j)*x[0] + ((0.989530856411+0.141830809358j))*x[1] + ((0.748578250097+0.93860017888j))*x[2]
            ref[(1, 2)]=(0.93930813312+0.617382035424j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedSolution_rank3(self):
        """
        tests integral of rank 3 Data on ReducedFunction

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(2, 3, 3),w)
        ref=numpy.zeros((2, 3, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.17111179621+0.0504413903971j)*x[0] + ((0.241313082841+0.865862943866j))*x[1]
            ref[(0, 0, 0)]=(0.0351006433154+0.458152167131j)
            arg[(0, 0, 1)]=(0.258439987214+0.0981489949938j)*x[0] + ((-0.512284148341+0.787526764395j))*x[1]
            ref[(0, 0, 1)]=(-0.126922080563+0.442837879694j)
            arg[(0, 0, 2)]=(-0.383327043306-0.535751562701j)*x[0] + ((0.606925630679+0.121964071784j))*x[1]
            ref[(0, 0, 2)]=(0.111799293686-0.206893745458j)
            arg[(0, 1, 0)]=(0.200765124773+0.401400608133j)*x[0] + ((-0.167366736554-0.0956575425354j))*x[1]
            ref[(0, 1, 0)]=(0.0166991941097+0.152871532799j)
            arg[(0, 1, 1)]=(0.925075194705-0.327682821226j)*x[0] + ((0.675048042806+0.230265163741j))*x[1]
            ref[(0, 1, 1)]=(0.800061618756-0.0487088287423j)
            arg[(0, 1, 2)]=(-0.134859133886+0.0591737962602j)*x[0] + ((-0.434360447095-0.061500411257j))*x[1]
            ref[(0, 1, 2)]=(-0.284609790491-0.0011633074984j)
            arg[(0, 2, 0)]=(-0.23221910989-0.369801629017j)*x[0] + ((-0.125455811876-0.387989886413j))*x[1]
            ref[(0, 2, 0)]=(-0.178837460883-0.378895757715j)
            arg[(0, 2, 1)]=(0.629320305757+0.685679335153j)*x[0] + ((-0.811168980828-0.78258926655j))*x[1]
            ref[(0, 2, 1)]=(-0.0909243375353-0.0484549656987j)
            arg[(0, 2, 2)]=(0.897806104071+0.42377845375j)*x[0] + ((-0.281081832453-0.0643606929803j))*x[1]
            ref[(0, 2, 2)]=(0.308362135809+0.179708880385j)
            arg[(1, 0, 0)]=(-0.247293728435-0.893259224227j)*x[0] + ((0.144470987312-0.573868649205j))*x[1]
            ref[(1, 0, 0)]=(-0.0514113705612-0.733563936716j)
            arg[(1, 0, 1)]=(0.275268753136+0.566237777481j)*x[0] + ((0.23307687699-0.0103072853056j))*x[1]
            ref[(1, 0, 1)]=(0.254172815063+0.277965246088j)
            arg[(1, 0, 2)]=(-0.528846295653+0.833939124292j)*x[0] + ((0.415277209631+0.395328259532j))*x[1]
            ref[(1, 0, 2)]=(-0.0567845430109+0.614633691912j)
            arg[(1, 1, 0)]=(0.0937659018103-0.746023763148j)*x[0] + ((-0.491365736381-0.654671925342j))*x[1]
            ref[(1, 1, 0)]=(-0.198799917285-0.700347844245j)
            arg[(1, 1, 1)]=(-0.685236820328+0.633134505859j)*x[0] + ((0.183228851895+0.804373044527j))*x[1]
            ref[(1, 1, 1)]=(-0.251003984216+0.718753775193j)
            arg[(1, 1, 2)]=(0.0904803516073-0.939877601141j)*x[0] + ((-0.431959505684-0.367003575819j))*x[1]
            ref[(1, 1, 2)]=(-0.170739577038-0.65344058848j)
            arg[(1, 2, 0)]=(0.0690839212138+0.85731229508j)*x[0] + ((-0.986572849832-0.950177599906j))*x[1]
            ref[(1, 2, 0)]=(-0.458744464309-0.0464326524131j)
            arg[(1, 2, 1)]=(-0.663100575384-0.609960986662j)*x[0] + ((-0.977332082743-0.00979442206463j))*x[1]
            ref[(1, 2, 1)]=(-0.820216329063-0.309877704363j)
            arg[(1, 2, 2)]=(0.931313206215-0.0247379127821j)*x[0] + ((-0.666444029843-0.0195288181568j))*x[1]
            ref[(1, 2, 2)]=(0.132434588186-0.0221333654695j)
        else:
            arg[(0, 0, 0)]=(0.908087205786+0.765783579445j)*x[0] + ((-0.000876343065817-0.224877646762j))*x[1] + ((0.378402921857+0.0997450462016j))*x[2]
            ref[(0, 0, 0)]=(0.642806892289+0.320325489442j)
            arg[(0, 0, 1)]=(0.285791856676-0.104743506549j)*x[0] + ((0.776046637885-0.989062644522j))*x[1] + ((-0.000986838073126+0.272640781153j))*x[2]
            ref[(0, 0, 1)]=(0.530425828244-0.410582684959j)
            arg[(0, 0, 2)]=(0.823023872735-0.399516681262j)*x[0] + ((-0.24172896245-0.227777792656j))*x[1] + ((-0.849736861101-0.118680404687j))*x[2]
            ref[(0, 0, 2)]=(-0.134220975408-0.372987439302j)
            arg[(0, 1, 0)]=(-0.204750060827+0.286884584106j)*x[0] + ((-0.925846718566-0.905482779609j))*x[1] + ((-0.586394609224+0.0454074324306j))*x[2]
            ref[(0, 1, 0)]=(-0.858495694308-0.286595381536j)
            arg[(0, 1, 1)]=(-0.0423588889373+0.140397949618j)*x[0] + ((0.498882974284+0.847095848872j))*x[1] + ((0.155745356466-0.218437425055j))*x[2]
            ref[(0, 1, 1)]=(0.306134720906+0.384528186717j)
            arg[(0, 1, 2)]=(-0.162332875269-0.706778623601j)*x[0] + ((0.862156995651+0.55383899126j))*x[1] + ((-0.0345274266073+0.678576486881j))*x[2]
            ref[(0, 1, 2)]=(0.332648346887+0.26281842727j)
            arg[(0, 2, 0)]=(-0.0858851496071-0.819560741214j)*x[0] + ((0.00892795765807+0.743135224601j))*x[1] + ((0.0181907550572+0.672995513455j))*x[2]
            ref[(0, 2, 0)]=(-0.0293832184459+0.298284998421j)
            arg[(0, 2, 1)]=(0.557313699461-0.493152677797j)*x[0] + ((-0.357052414606+0.852051246862j))*x[1] + ((-0.65317597611+0.892900473473j))*x[2]
            ref[(0, 2, 1)]=(-0.226457345628+0.625899521269j)
            arg[(0, 2, 2)]=(-0.944053623206-0.870096150229j)*x[0] + ((-0.42894957029+0.722705049438j))*x[1] + ((0.432356384396-0.0370593079447j))*x[2]
            ref[(0, 2, 2)]=(-0.47032340455-0.0922252043678j)
            arg[(1, 0, 0)]=(0.845394339803+0.158434103099j)*x[0] + ((-0.324045042685+0.342335427085j))*x[1] + ((-0.394236343456+0.947077788717j))*x[2]
            ref[(1, 0, 0)]=(0.0635564768311+0.72392365945j)
            arg[(1, 0, 1)]=(-0.465116060528-0.233159368234j)*x[0] + ((0.953800207029+0.433250364103j))*x[1] + ((0.394697038041+0.0801278933461j))*x[2]
            ref[(1, 0, 1)]=(0.441690592271+0.140109444608j)
            arg[(1, 0, 2)]=(-0.199464912825+0.634421138571j)*x[0] + ((0.822994606332+0.829393473486j))*x[1] + ((0.491684571662+0.116561071499j))*x[2]
            ref[(1, 0, 2)]=(0.557607132584+0.790187841778j)
            arg[(1, 1, 0)]=(-0.504074338504-0.664288658852j)*x[0] + ((-0.732596278631+0.338002289561j))*x[1] + ((-0.423610828602+0.620662317108j))*x[2]
            ref[(1, 1, 0)]=(-0.830140722869+0.147187973908j)
            arg[(1, 1, 1)]=(0.96762028545-0.507599640263j)*x[0] + ((-0.632795079631-0.411331477305j))*x[1] + ((0.486879120553-0.227326075031j))*x[2]
            ref[(1, 1, 1)]=(0.410852163186-0.573128596299j)
            arg[(1, 1, 2)]=(0.124510309498-0.622524224844j)*x[0] + ((-0.461600728348+0.532310659221j))*x[1] + ((-0.341603863238+0.291160049583j))*x[2]
            ref[(1, 1, 2)]=(-0.339347141044+0.10047324198j)
            arg[(1, 2, 0)]=(-0.237748166958-0.198602462178j)*x[0] + ((0.590509350539+0.733366499294j))*x[1] + ((0.648246455183+0.409633748591j))*x[2]
            ref[(1, 2, 0)]=(0.500503819382+0.472198892853j)
            arg[(1, 2, 1)]=(0.52997209739-0.256199589604j)*x[0] + ((0.538342830968-0.200220401526j))*x[1] + ((0.70822747439+0.164545577988j))*x[2]
            ref[(1, 2, 1)]=(0.888271201374-0.145937206571j)
            arg[(1, 2, 2)]=(0.611770009259+0.115845362937j)*x[0] + ((-0.537760302077+0.120532975752j))*x[1] + ((0.637741182573+0.560858269754j))*x[2]
            ref[(1, 2, 2)]=(0.355875444877+0.398618304221j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 3, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedSolution_rank4(self):
        """
        tests integral of rank 4 Data on ReducedFunction

        assumptions: ReducedSolution(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedSolution(self.domain)
        x = w.getX()
        arg = Data(0,(3, 2, 4, 4),w)
        ref=numpy.zeros((3, 2, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.899839703802+0.609703601329j)*x[0] + ((-0.046624247601-0.419757959959j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.473231975701+0.0949728206849j)
            arg[(0, 0, 0, 1)]=(-0.287937783661+0.251140619055j)*x[0] + ((0.0819272107282-0.840215074841j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.103005286466-0.294537227893j)
            arg[(0, 0, 0, 2)]=(0.687325420289-0.46741036808j)*x[0] + ((-0.966819681538+0.865994447605j))*x[1]
            ref[(0, 0, 0, 2)]=(-0.139747130625+0.199292039762j)
            arg[(0, 0, 0, 3)]=(0.946279400394-0.720100957248j)*x[0] + ((0.33217823937-0.540614395452j))*x[1]
            ref[(0, 0, 0, 3)]=(0.639228819882-0.63035767635j)
            arg[(0, 0, 1, 0)]=(0.737805383249+0.930542539802j)*x[0] + ((-0.360305545377+0.119670743651j))*x[1]
            ref[(0, 0, 1, 0)]=(0.188749918936+0.525106641727j)
            arg[(0, 0, 1, 1)]=(0.738206311686+0.492317555629j)*x[0] + ((0.927571437753-0.761534445452j))*x[1]
            ref[(0, 0, 1, 1)]=(0.832888874719-0.134608444911j)
            arg[(0, 0, 1, 2)]=(-0.404676719902-0.591681581022j)*x[0] + ((-0.191507241843+0.51393146488j))*x[1]
            ref[(0, 0, 1, 2)]=(-0.298091980873-0.0388750580711j)
            arg[(0, 0, 1, 3)]=(-0.0838543843744-0.743610132137j)*x[0] + ((0.190642515601+0.505891759817j))*x[1]
            ref[(0, 0, 1, 3)]=(0.0533940656132-0.11885918616j)
            arg[(0, 0, 2, 0)]=(-0.0131681015128+0.647120316239j)*x[0] + ((0.157864162911+0.829654522825j))*x[1]
            ref[(0, 0, 2, 0)]=(0.0723480306991+0.738387419532j)
            arg[(0, 0, 2, 1)]=(-0.679720732998+0.696302165784j)*x[0] + ((-0.1456378091+0.828249255302j))*x[1]
            ref[(0, 0, 2, 1)]=(-0.412679271049+0.762275710543j)
            arg[(0, 0, 2, 2)]=(0.535689718536-0.724171534266j)*x[0] + ((0.520769424315+0.764852494831j))*x[1]
            ref[(0, 0, 2, 2)]=(0.528229571426+0.0203404802828j)
            arg[(0, 0, 2, 3)]=(-0.853044568619-0.965655455975j)*x[0] + ((0.701481406451-0.370936886676j))*x[1]
            ref[(0, 0, 2, 3)]=(-0.0757815810841-0.668296171326j)
            arg[(0, 0, 3, 0)]=(0.518380747471+0.629683191236j)*x[0] + ((-0.948666543801-0.0583907249054j))*x[1]
            ref[(0, 0, 3, 0)]=(-0.215142898165+0.285646233165j)
            arg[(0, 0, 3, 1)]=(-0.0762353252214-0.109748230695j)*x[0] + ((-0.594812663839-0.0627231678818j))*x[1]
            ref[(0, 0, 3, 1)]=(-0.33552399453-0.0862356992883j)
            arg[(0, 0, 3, 2)]=(0.357507903128+0.315701526643j)*x[0] + ((-0.0169251217098-0.362798377689j))*x[1]
            ref[(0, 0, 3, 2)]=(0.170291390709-0.0235484255232j)
            arg[(0, 0, 3, 3)]=(0.777265798046-0.841173193002j)*x[0] + ((-0.348884439585-0.82506097637j))*x[1]
            ref[(0, 0, 3, 3)]=(0.21419067923-0.833117084686j)
            arg[(0, 1, 0, 0)]=(0.624348147018+0.382292878577j)*x[0] + ((-0.543633481107+0.678828214074j))*x[1]
            ref[(0, 1, 0, 0)]=(0.0403573329555+0.530560546326j)
            arg[(0, 1, 0, 1)]=(-0.428852026269-0.277001265539j)*x[0] + ((-0.403728501371+0.657782681556j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.41629026382+0.190390708008j)
            arg[(0, 1, 0, 2)]=(-0.975973607113-0.983373385546j)*x[0] + ((-0.0940488123435-0.896859598912j))*x[1]
            ref[(0, 1, 0, 2)]=(-0.535011209728-0.940116492229j)
            arg[(0, 1, 0, 3)]=(0.585773149531-0.703019830038j)*x[0] + ((0.355173058372-0.708078917292j))*x[1]
            ref[(0, 1, 0, 3)]=(0.470473103951-0.705549373665j)
            arg[(0, 1, 1, 0)]=(0.91048443197+0.945120388122j)*x[0] + ((0.442892553895-0.996395844j))*x[1]
            ref[(0, 1, 1, 0)]=(0.676688492933-0.0256377279394j)
            arg[(0, 1, 1, 1)]=(-0.483861208902+0.64124061887j)*x[0] + ((-0.523449265694-0.314957895923j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.503655237298+0.163141361473j)
            arg[(0, 1, 1, 2)]=(-0.0407348765964-0.596935352219j)*x[0] + ((-0.234046138607+0.820980328978j))*x[1]
            ref[(0, 1, 1, 2)]=(-0.137390507602+0.112022488379j)
            arg[(0, 1, 1, 3)]=(-0.904842482241+0.911603624608j)*x[0] + ((0.390591268702-0.731055953887j))*x[1]
            ref[(0, 1, 1, 3)]=(-0.257125606769+0.0902738353604j)
            arg[(0, 1, 2, 0)]=(0.480491997796+0.58659057206j)*x[0] + ((-0.928581795757-0.924338559329j))*x[1]
            ref[(0, 1, 2, 0)]=(-0.22404489898-0.168873993635j)
            arg[(0, 1, 2, 1)]=(0.398181986219+0.663004904937j)*x[0] + ((-0.795460764756-0.610333670302j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.198639389268+0.0263356173177j)
            arg[(0, 1, 2, 2)]=(0.609285071978-0.548243523856j)*x[0] + ((0.313338103386-0.664561166199j))*x[1]
            ref[(0, 1, 2, 2)]=(0.461311587682-0.606402345027j)
            arg[(0, 1, 2, 3)]=(-0.0769671871469-0.463151645372j)*x[0] + ((-0.267254931735-0.232817909195j))*x[1]
            ref[(0, 1, 2, 3)]=(-0.172111059441-0.347984777283j)
            arg[(0, 1, 3, 0)]=(0.321709440108+0.731124951626j)*x[0] + ((-0.54847783987-0.89827161888j))*x[1]
            ref[(0, 1, 3, 0)]=(-0.113384199881-0.0835733336269j)
            arg[(0, 1, 3, 1)]=(-0.790117934643+0.280756957792j)*x[0] + ((-0.648688954568+0.276177043532j))*x[1]
            ref[(0, 1, 3, 1)]=(-0.719403444606+0.278467000662j)
            arg[(0, 1, 3, 2)]=(0.301949388931-0.727208221817j)*x[0] + ((0.37389426923+0.677000936325j))*x[1]
            ref[(0, 1, 3, 2)]=(0.33792182908-0.0251036427457j)
            arg[(0, 1, 3, 3)]=(0.0704056524245+0.0011157976077j)*x[0] + ((0.0523029898763+0.420963781765j))*x[1]
            ref[(0, 1, 3, 3)]=(0.0613543211504+0.211039789686j)
            arg[(1, 0, 0, 0)]=(0.426939712918-0.904052533943j)*x[0] + ((0.995222939841+0.291444294239j))*x[1]
            ref[(1, 0, 0, 0)]=(0.711081326379-0.306304119852j)
            arg[(1, 0, 0, 1)]=(0.37883631668+0.0341274711087j)*x[0] + ((-0.890610821674+0.381524131439j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.255887252497+0.207825801274j)
            arg[(1, 0, 0, 2)]=(-0.130349841148+0.789068045091j)*x[0] + ((0.751934358068-0.998409802155j))*x[1]
            ref[(1, 0, 0, 2)]=(0.31079225846-0.104670878532j)
            arg[(1, 0, 0, 3)]=(0.15860714834+0.695378009847j)*x[0] + ((-0.682664164919-0.511010059977j))*x[1]
            ref[(1, 0, 0, 3)]=(-0.262028508289+0.092183974935j)
            arg[(1, 0, 1, 0)]=(0.82520384048+0.402068392317j)*x[0] + ((-0.381402715932-0.334563530551j))*x[1]
            ref[(1, 0, 1, 0)]=(0.221900562274+0.0337524308831j)
            arg[(1, 0, 1, 1)]=(-0.150475906712-0.394457689233j)*x[0] + ((-0.380825680223-0.621071636135j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.265650793468-0.507764662684j)
            arg[(1, 0, 1, 2)]=(0.369666504574-0.891514088082j)*x[0] + ((0.933829801431+0.729490338154j))*x[1]
            ref[(1, 0, 1, 2)]=(0.651748153003-0.0810118749639j)
            arg[(1, 0, 1, 3)]=(0.826774121071-0.982630750108j)*x[0] + ((0.712509764449-0.12071267334j))*x[1]
            ref[(1, 0, 1, 3)]=(0.76964194276-0.551671711724j)
            arg[(1, 0, 2, 0)]=(-0.412378732177-0.44281811947j)*x[0] + ((0.925261916021+0.1207378347j))*x[1]
            ref[(1, 0, 2, 0)]=(0.256441591922-0.161040142385j)
            arg[(1, 0, 2, 1)]=(0.832666668226+0.604771972408j)*x[0] + ((0.835146122123+0.568912514446j))*x[1]
            ref[(1, 0, 2, 1)]=(0.833906395175+0.586842243427j)
            arg[(1, 0, 2, 2)]=(-0.786600641329+0.468492449784j)*x[0] + ((0.725910959169-0.953277159405j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.03034484108-0.24239235481j)
            arg[(1, 0, 2, 3)]=(-0.0515288180919-0.268698385957j)*x[0] + ((-0.0462923711829-0.358680815525j))*x[1]
            ref[(1, 0, 2, 3)]=(-0.0489105946374-0.313689600741j)
            arg[(1, 0, 3, 0)]=(0.701423956075-0.752351384485j)*x[0] + ((0.971862305357+0.517813591379j))*x[1]
            ref[(1, 0, 3, 0)]=(0.836643130716-0.117268896553j)
            arg[(1, 0, 3, 1)]=(-0.309415180436+0.808618791282j)*x[0] + ((0.0990183957204-0.31753168557j))*x[1]
            ref[(1, 0, 3, 1)]=(-0.105198392358+0.245543552856j)
            arg[(1, 0, 3, 2)]=(-0.707842462612+0.334225711962j)*x[0] + ((-0.517997557694+0.0987298332425j))*x[1]
            ref[(1, 0, 3, 2)]=(-0.612920010153+0.216477772602j)
            arg[(1, 0, 3, 3)]=(0.122666003223+0.0541940130412j)*x[0] + ((0.593873279287+0.916358879699j))*x[1]
            ref[(1, 0, 3, 3)]=(0.358269641255+0.48527644637j)
            arg[(1, 1, 0, 0)]=(-0.853955344476-0.583294974434j)*x[0] + ((-0.561037464491-0.378660155615j))*x[1]
            ref[(1, 1, 0, 0)]=(-0.707496404484-0.480977565024j)
            arg[(1, 1, 0, 1)]=(0.91439458918-0.016355504598j)*x[0] + ((-0.573752271404+0.554747631315j))*x[1]
            ref[(1, 1, 0, 1)]=(0.170321158888+0.269196063358j)
            arg[(1, 1, 0, 2)]=(0.935486912971+0.0318245070917j)*x[0] + ((-0.516096684527-0.808453545983j))*x[1]
            ref[(1, 1, 0, 2)]=(0.209695114222-0.388314519446j)
            arg[(1, 1, 0, 3)]=(0.970832258646-0.611234483741j)*x[0] + ((0.793119528971+0.627506482662j))*x[1]
            ref[(1, 1, 0, 3)]=(0.881975893808+0.00813599946056j)
            arg[(1, 1, 1, 0)]=(-0.347743143119+0.861431134244j)*x[0] + ((0.56439705791-0.557426947412j))*x[1]
            ref[(1, 1, 1, 0)]=(0.108326957395+0.152002093416j)
            arg[(1, 1, 1, 1)]=(0.893044032375-0.860809630714j)*x[0] + ((0.316740137141+0.88190026161j))*x[1]
            ref[(1, 1, 1, 1)]=(0.604892084758+0.0105453154479j)
            arg[(1, 1, 1, 2)]=(-0.888464057942+0.805654085344j)*x[0] + ((0.584222725217+0.830080293392j))*x[1]
            ref[(1, 1, 1, 2)]=(-0.152120666363+0.817867189368j)
            arg[(1, 1, 1, 3)]=(-0.162298301324+0.242814300687j)*x[0] + ((-0.100680964061+0.165281675307j))*x[1]
            ref[(1, 1, 1, 3)]=(-0.131489632693+0.204047987997j)
            arg[(1, 1, 2, 0)]=(-0.587259654503-0.199024539966j)*x[0] + ((0.194097201992+0.97074169615j))*x[1]
            ref[(1, 1, 2, 0)]=(-0.196581226256+0.385858578092j)
            arg[(1, 1, 2, 1)]=(0.629232700827-0.00651903594531j)*x[0] + ((0.872538697671+0.718397749161j))*x[1]
            ref[(1, 1, 2, 1)]=(0.750885699249+0.355939356608j)
            arg[(1, 1, 2, 2)]=(-0.00323550025838+0.608115567407j)*x[0] + ((-0.351468787331-0.741752075904j))*x[1]
            ref[(1, 1, 2, 2)]=(-0.177352143794-0.0668182542481j)
            arg[(1, 1, 2, 3)]=(0.618104706648-0.526130395571j)*x[0] + ((0.156431732141+0.255427515836j))*x[1]
            ref[(1, 1, 2, 3)]=(0.387268219395-0.135351439868j)
            arg[(1, 1, 3, 0)]=(0.449124745011+0.763389137685j)*x[0] + ((0.820330795643+0.145770792487j))*x[1]
            ref[(1, 1, 3, 0)]=(0.634727770327+0.454579965086j)
            arg[(1, 1, 3, 1)]=(0.129351413284-0.00872969650669j)*x[0] + ((-0.0935828936332+0.718953963527j))*x[1]
            ref[(1, 1, 3, 1)]=(0.0178842598254+0.35511213351j)
            arg[(1, 1, 3, 2)]=(0.553869543442+0.147257126439j)*x[0] + ((0.98998476646-0.159732473601j))*x[1]
            ref[(1, 1, 3, 2)]=(0.771927154951-0.00623767358084j)
            arg[(1, 1, 3, 3)]=(-0.318507443769+0.205196580421j)*x[0] + ((0.379250398423+0.300400839917j))*x[1]
            ref[(1, 1, 3, 3)]=(0.0303714773269+0.252798710169j)
            arg[(2, 0, 0, 0)]=(-0.911209061583+0.811484427149j)*x[0] + ((0.703050478233-0.341323010586j))*x[1]
            ref[(2, 0, 0, 0)]=(-0.104079291675+0.235080708281j)
            arg[(2, 0, 0, 1)]=(-0.221176495823+0.88656349782j)*x[0] + ((0.701713368684-0.426312176205j))*x[1]
            ref[(2, 0, 0, 1)]=(0.240268436431+0.230125660808j)
            arg[(2, 0, 0, 2)]=(0.841824885601-0.703510781324j)*x[0] + ((0.885446497484-0.363225068684j))*x[1]
            ref[(2, 0, 0, 2)]=(0.863635691542-0.533367925004j)
            arg[(2, 0, 0, 3)]=(-0.24580659722-0.909294672115j)*x[0] + ((0.0943965894164-0.060286808129j))*x[1]
            ref[(2, 0, 0, 3)]=(-0.0757050039019-0.484790740122j)
            arg[(2, 0, 1, 0)]=(-0.0237913501359-0.519585073279j)*x[0] + ((0.983038939334-0.127169083225j))*x[1]
            ref[(2, 0, 1, 0)]=(0.479623794599-0.323377078252j)
            arg[(2, 0, 1, 1)]=(-0.300353612594+0.784610101515j)*x[0] + ((-0.420474496031+0.28493614246j))*x[1]
            ref[(2, 0, 1, 1)]=(-0.360414054312+0.534773121987j)
            arg[(2, 0, 1, 2)]=(-0.179324906301+0.477499837168j)*x[0] + ((0.492883280053-0.47348620478j))*x[1]
            ref[(2, 0, 1, 2)]=(0.156779186876+0.00200681619418j)
            arg[(2, 0, 1, 3)]=(0.659433915504-0.222237566706j)*x[0] + ((-0.954579103849-0.174612390786j))*x[1]
            ref[(2, 0, 1, 3)]=(-0.147572594173-0.198424978746j)
            arg[(2, 0, 2, 0)]=(0.82253952957+0.635081075533j)*x[0] + ((0.810285033802-0.161131384743j))*x[1]
            ref[(2, 0, 2, 0)]=(0.816412281686+0.236974845395j)
            arg[(2, 0, 2, 1)]=(0.143071129741+0.318458863054j)*x[0] + ((0.421065564478+0.363563250216j))*x[1]
            ref[(2, 0, 2, 1)]=(0.282068347109+0.341011056635j)
            arg[(2, 0, 2, 2)]=(0.514122878043-0.0365245445421j)*x[0] + ((0.820717618077-0.583547693212j))*x[1]
            ref[(2, 0, 2, 2)]=(0.66742024806-0.310036118877j)
            arg[(2, 0, 2, 3)]=(0.20472700305-0.0525675449221j)*x[0] + ((0.72530186746-0.853014714383j))*x[1]
            ref[(2, 0, 2, 3)]=(0.465014435255-0.452791129653j)
            arg[(2, 0, 3, 0)]=(0.510227325061+0.11059649921j)*x[0] + ((-0.879784870333-0.00620742751614j))*x[1]
            ref[(2, 0, 3, 0)]=(-0.184778772636+0.0521945358467j)
            arg[(2, 0, 3, 1)]=(0.210181782286-0.678445770514j)*x[0] + ((0.471147924996-0.964543225074j))*x[1]
            ref[(2, 0, 3, 1)]=(0.340664853641-0.821494497794j)
            arg[(2, 0, 3, 2)]=(-0.860131965201+0.24708514177j)*x[0] + ((-0.5486229437+0.0412830720947j))*x[1]
            ref[(2, 0, 3, 2)]=(-0.704377454451+0.144184106932j)
            arg[(2, 0, 3, 3)]=(0.636145988054+0.274030408264j)*x[0] + ((-0.63302201758-0.915781013345j))*x[1]
            ref[(2, 0, 3, 3)]=(0.00156198523717-0.320875302541j)
            arg[(2, 1, 0, 0)]=(-0.727731846125+0.43305794252j)*x[0] + ((-0.283241933168+0.103787761643j))*x[1]
            ref[(2, 1, 0, 0)]=(-0.505486889647+0.268422852081j)
            arg[(2, 1, 0, 1)]=(0.200903169178+0.443663981513j)*x[0] + ((-0.922989223959+0.702305421228j))*x[1]
            ref[(2, 1, 0, 1)]=(-0.361043027391+0.57298470137j)
            arg[(2, 1, 0, 2)]=(-0.251464941735+0.660246030922j)*x[0] + ((-0.73770654245+0.63571079928j))*x[1]
            ref[(2, 1, 0, 2)]=(-0.494585742093+0.647978415101j)
            arg[(2, 1, 0, 3)]=(0.619790618432+0.0842458453964j)*x[0] + ((0.656733674244-0.394571414492j))*x[1]
            ref[(2, 1, 0, 3)]=(0.638262146338-0.155162784548j)
            arg[(2, 1, 1, 0)]=(-0.485611969003-0.819868328088j)*x[0] + ((0.226158414122+0.0887952506704j))*x[1]
            ref[(2, 1, 1, 0)]=(-0.129726777441-0.365536538709j)
            arg[(2, 1, 1, 1)]=(-0.168269478904+0.430402340749j)*x[0] + ((0.667939491061+0.228860257401j))*x[1]
            ref[(2, 1, 1, 1)]=(0.249835006079+0.329631299075j)
            arg[(2, 1, 1, 2)]=(-0.246842319695-0.197429111861j)*x[0] + ((0.134719488289+0.215139189826j))*x[1]
            ref[(2, 1, 1, 2)]=(-0.056061415703+0.00885503898225j)
            arg[(2, 1, 1, 3)]=(0.340657237211+0.602027882616j)*x[0] + ((-0.6128457019+0.703965081136j))*x[1]
            ref[(2, 1, 1, 3)]=(-0.136094232344+0.652996481876j)
            arg[(2, 1, 2, 0)]=(0.795281283049-0.89322816426j)*x[0] + ((0.92232736201+0.170390782589j))*x[1]
            ref[(2, 1, 2, 0)]=(0.858804322529-0.361418690836j)
            arg[(2, 1, 2, 1)]=(-0.189235275576-0.237536653611j)*x[0] + ((-0.809344170837+0.63126320162j))*x[1]
            ref[(2, 1, 2, 1)]=(-0.499289723207+0.196863274005j)
            arg[(2, 1, 2, 2)]=(-0.0902479898163-0.879161738085j)*x[0] + ((0.721319021763-0.976256654143j))*x[1]
            ref[(2, 1, 2, 2)]=(0.315535515973-0.927709196114j)
            arg[(2, 1, 2, 3)]=(-0.853507340123-0.229479292523j)*x[0] + ((-0.151796890109+0.930087915217j))*x[1]
            ref[(2, 1, 2, 3)]=(-0.502652115116+0.350304311347j)
            arg[(2, 1, 3, 0)]=(-0.334624459333-0.675326222741j)*x[0] + ((0.504427764892-0.90451535023j))*x[1]
            ref[(2, 1, 3, 0)]=(0.0849016527796-0.789920786485j)
            arg[(2, 1, 3, 1)]=(-0.100485132722+0.0575111311878j)*x[0] + ((0.0714785509696+0.613912134552j))*x[1]
            ref[(2, 1, 3, 1)]=(-0.0145032908765+0.33571163287j)
            arg[(2, 1, 3, 2)]=(-0.551828984922-0.189138583548j)*x[0] + ((0.48273845032-0.172096513671j))*x[1]
            ref[(2, 1, 3, 2)]=(-0.0345452673011-0.180617548609j)
            arg[(2, 1, 3, 3)]=(0.863264356068+0.726018070429j)*x[0] + ((-0.48330990062+0.773804147761j))*x[1]
            ref[(2, 1, 3, 3)]=(0.189977227724+0.749911109095j)
        else:
            arg[(0, 0, 0, 0)]=(0.985583796634+0.677389176572j)*x[0] + ((0.97828337203+0.110011183328j))*x[1] + ((0.493393498027+0.908934869978j))*x[2]
            ref[(0, 0, 0, 0)]=(1.22863033335+0.848167614939j)
            arg[(0, 0, 0, 1)]=(0.496098261198-0.882934109969j)*x[0] + ((0.113912249211+0.280156629534j))*x[1] + ((-0.915183121341-0.126725047417j))*x[2]
            ref[(0, 0, 0, 1)]=(-0.152586305466-0.364751263926j)
            arg[(0, 0, 0, 2)]=(0.520538065462+0.62356364416j)*x[0] + ((0.447108401966+0.716852005442j))*x[1] + ((-0.752359793428-0.0481830905553j))*x[2]
            ref[(0, 0, 0, 2)]=(0.107643337+0.646116279524j)
            arg[(0, 0, 0, 3)]=(-0.271404884337+0.535242551775j)*x[0] + ((0.445159710814-0.700499412008j))*x[1] + ((-0.864956331768-0.364126133439j))*x[2]
            ref[(0, 0, 0, 3)]=(-0.345600752646-0.264691496836j)
            arg[(0, 0, 1, 0)]=(-0.423465756695-0.125814599562j)*x[0] + ((-0.325801138724+0.677555890751j))*x[1] + ((0.618124156899-0.0947552190386j))*x[2]
            ref[(0, 0, 1, 0)]=(-0.0655713692595+0.228493036075j)
            arg[(0, 0, 1, 1)]=(0.952818016459-0.597543108586j)*x[0] + ((-0.751099209466-0.923176611216j))*x[1] + ((-0.10607095247+0.473233080127j))*x[2]
            ref[(0, 0, 1, 1)]=(0.0478239272617-0.523743319837j)
            arg[(0, 0, 1, 2)]=(0.397544787511-0.489376193969j)*x[0] + ((0.193338057962-0.376953376991j))*x[1] + ((-0.925157588258+0.860895378676j))*x[2]
            ref[(0, 0, 1, 2)]=(-0.167137371392-0.00271709614212j)
            arg[(0, 0, 1, 3)]=(0.364453146378-0.495758566147j)*x[0] + ((0.655212202647+0.459044329478j))*x[1] + ((-0.419322670903-0.603758438872j))*x[2]
            ref[(0, 0, 1, 3)]=(0.300171339061-0.32023633777j)
            arg[(0, 0, 2, 0)]=(-0.39071361889+0.659698106971j)*x[0] + ((-0.668269210032-0.112157903713j))*x[1] + ((-0.962970697069+0.876633383476j))*x[2]
            ref[(0, 0, 2, 0)]=(-1.010976763+0.712086793367j)
            arg[(0, 0, 2, 1)]=(-0.446020914338+0.20606529356j)*x[0] + ((-0.484570952056-0.617298111452j))*x[1] + ((0.52243546651-0.801906955268j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.204078199942-0.60656988658j)
            arg[(0, 0, 2, 2)]=(-0.149261311586-0.284742684576j)*x[0] + ((-0.0631913286776-0.340646392781j))*x[1] + ((-0.679597632903+0.914007340456j))*x[2]
            ref[(0, 0, 2, 2)]=(-0.446025136583+0.14430913155j)
            arg[(0, 0, 2, 3)]=(-0.436690439764+0.803331529283j)*x[0] + ((-0.967483341061-0.439829037734j))*x[1] + ((0.00315019718063-0.791620741611j))*x[2]
            ref[(0, 0, 2, 3)]=(-0.700511791822-0.214059125031j)
            arg[(0, 0, 3, 0)]=(0.797161406993+0.0858384332429j)*x[0] + ((-0.748268160392+0.817896261094j))*x[1] + ((0.266599907643-0.657126217603j))*x[2]
            ref[(0, 0, 3, 0)]=(0.157746577122+0.123304238367j)
            arg[(0, 0, 3, 1)]=(-0.298127346285-0.377903266656j)*x[0] + ((0.483274664343-0.694165970049j))*x[1] + ((-0.409923922605+0.0615539650637j))*x[2]
            ref[(0, 0, 3, 1)]=(-0.112388302273-0.505257635821j)
            arg[(0, 0, 3, 2)]=(-0.484141908329-0.691160616575j)*x[0] + ((-0.20483697669-0.0759190199568j))*x[1] + ((-0.766923091444-0.0512834082611j))*x[2]
            ref[(0, 0, 3, 2)]=(-0.727950988231-0.409181522397j)
            arg[(0, 0, 3, 3)]=(-0.264963990713-0.559886690821j)*x[0] + ((-0.0157357225845+0.251828324893j))*x[1] + ((0.613215861172+0.658984033041j))*x[2]
            ref[(0, 0, 3, 3)]=(0.166258073937+0.175462833556j)
            arg[(0, 1, 0, 0)]=(0.0522738603945-0.0174516051874j)*x[0] + ((0.00660225643061+0.578832972662j))*x[1] + ((0.62915660311+0.438630078286j))*x[2]
            ref[(0, 1, 0, 0)]=(0.344016359968+0.50000572288j)
            arg[(0, 1, 0, 1)]=(-0.231594629511-0.906439540696j)*x[0] + ((0.191347457056-0.0326247741203j))*x[1] + ((-0.67676060189-0.829564306171j))*x[2]
            ref[(0, 1, 0, 1)]=(-0.358503887172-0.884314310493j)
            arg[(0, 1, 0, 2)]=(0.437100400799-0.481691344484j)*x[0] + ((-0.0832113871787-0.0919069358666j))*x[1] + ((0.603537745449+0.729693079476j))*x[2]
            ref[(0, 1, 0, 2)]=(0.478713379535+0.0780473995628j)
            arg[(0, 1, 0, 3)]=(-0.895202153096-0.282022083874j)*x[0] + ((0.0423872847602+0.0154160217929j))*x[1] + ((0.468650378408-0.775762901735j))*x[2]
            ref[(0, 1, 0, 3)]=(-0.192082244964-0.521184481908j)
            arg[(0, 1, 1, 0)]=(-0.528139712477+0.0493995181117j)*x[0] + ((-0.978827454357+0.949117578515j))*x[1] + ((0.301173751445+0.997329263645j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.602896707694+0.997923180136j)
            arg[(0, 1, 1, 1)]=(0.679858282897-0.10685042231j)*x[0] + ((-0.569370439493+0.918757497064j))*x[1] + ((-0.536375375885+0.115782464998j))*x[2]
            ref[(0, 1, 1, 1)]=(-0.21294376624+0.463844769876j)
            arg[(0, 1, 1, 2)]=(0.153306838168-0.30175563661j)*x[0] + ((0.763358901354+0.158048479146j))*x[1] + ((-0.98635814685+0.315502051318j))*x[2]
            ref[(0, 1, 1, 2)]=(-0.0348462036641+0.0858974469268j)
            arg[(0, 1, 1, 3)]=(0.850417248068+0.783069868358j)*x[0] + ((-0.643191784358-0.117824850195j))*x[1] + ((-0.130963133462+0.144135782748j))*x[2]
            ref[(0, 1, 1, 3)]=(0.0381311651238+0.404690400456j)
            arg[(0, 1, 2, 0)]=(0.762249919837+0.157603775391j)*x[0] + ((-0.900039035828+0.14782325343j))*x[1] + ((-0.405686934695+0.695603870108j))*x[2]
            ref[(0, 1, 2, 0)]=(-0.271738025343+0.500515449464j)
            arg[(0, 1, 2, 1)]=(0.138621840072-0.84129156218j)*x[0] + ((0.640496002511-0.0678138042205j))*x[1] + ((0.86840857291+0.552822548149j))*x[2]
            ref[(0, 1, 2, 1)]=(0.823763207746-0.178141409126j)
            arg[(0, 1, 2, 2)]=(0.526118989736+0.331030040015j)*x[0] + ((-0.159125262298+0.687568615173j))*x[1] + ((0.887338355349+0.961647758722j))*x[2]
            ref[(0, 1, 2, 2)]=(0.627166041393+0.990123206955j)
            arg[(0, 1, 2, 3)]=(-0.378081367542-0.668310211069j)*x[0] + ((-0.114260281417-0.471182354076j))*x[1] + ((0.336667698169+0.665875075178j))*x[2]
            ref[(0, 1, 2, 3)]=(-0.077836975395-0.236808744983j)
            arg[(0, 1, 3, 0)]=(0.996977783847+0.598427185867j)*x[0] + ((-0.726790586412-0.652862224236j))*x[1] + ((-0.919704113112-0.775291600907j))*x[2]
            ref[(0, 1, 3, 0)]=(-0.324758457839-0.414863319638j)
            arg[(0, 1, 3, 1)]=(-0.289672935818+0.587653229148j)*x[0] + ((-0.193329649845-0.918634545608j))*x[1] + ((-0.0904394300584+0.00681744286915j))*x[2]
            ref[(0, 1, 3, 1)]=(-0.286721007861-0.162081936795j)
            arg[(0, 1, 3, 2)]=(0.186040370662+0.994427103443j)*x[0] + ((-0.794193732429-0.381086458681j))*x[1] + ((-0.277729958694+0.181134420302j))*x[2]
            ref[(0, 1, 3, 2)]=(-0.44294166023+0.397237532532j)
            arg[(0, 1, 3, 3)]=(0.451168663212-0.846327772743j)*x[0] + ((0.641217440894+0.217286630241j))*x[1] + ((-0.252444912771-0.211009201021j))*x[2]
            ref[(0, 1, 3, 3)]=(0.419970595667-0.420025171761j)
            arg[(1, 0, 0, 0)]=(0.832975123648-0.914025536317j)*x[0] + ((0.516632502717-0.757778098628j))*x[1] + ((-0.920190922244+0.837091482579j))*x[2]
            ref[(1, 0, 0, 0)]=(0.214708352061-0.417356076183j)
            arg[(1, 0, 0, 1)]=(-0.190552735424+0.120886609862j)*x[0] + ((-0.0564564704884-0.68287302482j))*x[1] + ((0.0196920882809-0.499085123796j))*x[2]
            ref[(1, 0, 0, 1)]=(-0.113658558816-0.530535769377j)
            arg[(1, 0, 0, 2)]=(0.588260140911+0.587039941024j)*x[0] + ((-0.454753295248+0.57756556547j))*x[1] + ((-0.592718745439-0.45299992292j))*x[2]
            ref[(1, 0, 0, 2)]=(-0.229605949887+0.355802791787j)
            arg[(1, 0, 0, 3)]=(-0.0436394652462-0.501928360476j)*x[0] + ((0.264178081101-0.666866719154j))*x[1] + ((0.72655849711+0.210243901956j))*x[2]
            ref[(1, 0, 0, 3)]=(0.473548556482-0.479275588837j)
            arg[(1, 0, 1, 0)]=(-0.17224520541+0.408711553934j)*x[0] + ((0.725741997684-0.750127515575j))*x[1] + ((-0.420402402391+0.784274393145j))*x[2]
            ref[(1, 0, 1, 0)]=(0.066547194941+0.221429215752j)
            arg[(1, 0, 1, 1)]=(0.306704728895+0.355496764914j)*x[0] + ((-0.839457433497+0.667699926855j))*x[1] + ((0.220568398899-0.0582019907171j))*x[2]
            ref[(1, 0, 1, 1)]=(-0.156092152852+0.482497350526j)
            arg[(1, 0, 1, 2)]=(-0.447246330173+0.649224104848j)*x[0] + ((-0.0869886329306-0.684635704405j))*x[1] + ((0.36283721334-0.772049457754j))*x[2]
            ref[(1, 0, 1, 2)]=(-0.085698874882-0.403730528656j)
            arg[(1, 0, 1, 3)]=(0.849581251463+0.40005452752j)*x[0] + ((0.0213885370144-0.600299785201j))*x[1] + ((-0.774185296785-0.390820513789j))*x[2]
            ref[(1, 0, 1, 3)]=(0.0483922458461-0.295532885735j)
            arg[(1, 0, 2, 0)]=(0.768336773438-0.365641624327j)*x[0] + ((-0.21807030393+0.575452685209j))*x[1] + ((0.61659913711+0.74923388907j))*x[2]
            ref[(1, 0, 2, 0)]=(0.583432803309+0.479522474976j)
            arg[(1, 0, 2, 1)]=(-0.0126204305519-0.18975812375j)*x[0] + ((0.451749206935+0.519003881535j))*x[1] + ((-0.586425186089+0.783756641875j))*x[2]
            ref[(1, 0, 2, 1)]=(-0.073648204853+0.55650119983j)
            arg[(1, 0, 2, 2)]=(0.604363835744+0.888783702346j)*x[0] + ((-0.653403932482-0.598166065679j))*x[1] + ((0.0281599740055+0.443852979203j))*x[2]
            ref[(1, 0, 2, 2)]=(-0.010440061366+0.367235307935j)
            arg[(1, 0, 2, 3)]=(-0.350254471769-0.923099142778j)*x[0] + ((0.295254315145-0.956898706603j))*x[1] + ((0.836119886983-0.86606694608j))*x[2]
            ref[(1, 0, 2, 3)]=(0.390559865179-1.37303239773j)
            arg[(1, 0, 3, 0)]=(0.46225878192+0.000616427477363j)*x[0] + ((-0.86956612909+0.989547317323j))*x[1] + ((0.44846986037+0.213497747485j))*x[2]
            ref[(1, 0, 3, 0)]=(0.0205812566002+0.601830746143j)
            arg[(1, 0, 3, 1)]=(-0.309667000442+0.0028699042918j)*x[0] + ((0.394753488226-0.464987407691j))*x[1] + ((0.638144410187-0.125724217351j))*x[2]
            ref[(1, 0, 3, 1)]=(0.361615448986-0.293920860375j)
            arg[(1, 0, 3, 2)]=(-0.629196117639-0.116214109118j)*x[0] + ((0.0445408132384-0.901315364035j))*x[1] + ((0.851717080397+0.86103525109j))*x[2]
            ref[(1, 0, 3, 2)]=(0.133530887998-0.0782471110317j)
            arg[(1, 0, 3, 3)]=(-0.388163217336-0.602801331817j)*x[0] + ((-0.519711631312-0.186951666834j))*x[1] + ((-0.324677159954+0.112116521624j))*x[2]
            ref[(1, 0, 3, 3)]=(-0.616276004301-0.338818238514j)
            arg[(1, 1, 0, 0)]=(-0.200058670062+0.601429005978j)*x[0] + ((0.490372011836+0.983518350526j))*x[1] + ((0.0459742020928-0.746682517877j))*x[2]
            ref[(1, 1, 0, 0)]=(0.168143771934+0.419132419313j)
            arg[(1, 1, 0, 1)]=(0.152540544393+0.26406565709j)*x[0] + ((0.0791493381663-0.31654480041j))*x[1] + ((-0.442331951896-0.515079844797j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.105321034668-0.283779494059j)
            arg[(1, 1, 0, 2)]=(0.38299695655+0.912108563535j)*x[0] + ((0.184873877491+0.154858406634j))*x[1] + ((-0.866944340505-0.29319115655j))*x[2]
            ref[(1, 1, 0, 2)]=(-0.149536753232+0.386887906809j)
            arg[(1, 1, 0, 3)]=(-0.691984508004-0.835366314046j)*x[0] + ((-0.574237751246+0.895501632855j))*x[1] + ((-0.530312035376+0.705309442861j))*x[2]
            ref[(1, 1, 0, 3)]=(-0.898267147313+0.382722380835j)
            arg[(1, 1, 1, 0)]=(-0.291131245667-0.98446731718j)*x[0] + ((-0.819054517561+0.653478724766j))*x[1] + ((0.654340565259-0.427818975121j))*x[2]
            ref[(1, 1, 1, 0)]=(-0.227922598985-0.379403783767j)
            arg[(1, 1, 1, 1)]=(-0.437288958908-0.116887067259j)*x[0] + ((-0.65138958128+0.0969600513961j))*x[1] + ((-0.176694564974-0.195289286161j))*x[2]
            ref[(1, 1, 1, 1)]=(-0.632686552581-0.107608151012j)
            arg[(1, 1, 1, 2)]=(0.817512717437+0.314319223636j)*x[0] + ((0.97104060012-0.356796235263j))*x[1] + ((-0.903430513248-0.483277452382j))*x[2]
            ref[(1, 1, 1, 2)]=(0.442561402154-0.262877232004j)
            arg[(1, 1, 1, 3)]=(-0.152873328581+0.42331750873j)*x[0] + ((-0.121382532725-0.378750066439j))*x[1] + ((0.132197311958+0.512612552283j))*x[2]
            ref[(1, 1, 1, 3)]=(-0.0710292746742+0.278589997287j)
            arg[(1, 1, 2, 0)]=(0.794150730729-0.36391752134j)*x[0] + ((0.278188979099-0.895730233331j))*x[1] + ((-0.661442091301+0.158911181611j))*x[2]
            ref[(1, 1, 2, 0)]=(0.205448809264-0.550368286529j)
            arg[(1, 1, 2, 1)]=(-0.726156691147-0.368617422504j)*x[0] + ((0.0422102105633+0.278691665747j))*x[1] + ((-0.644995706377-0.497201670056j))*x[2]
            ref[(1, 1, 2, 1)]=(-0.66447109348-0.293563713406j)
            arg[(1, 1, 2, 2)]=(0.787961989589+0.14855902369j)*x[0] + ((0.303809965385-0.591207437533j))*x[1] + ((-0.961001425366-0.257206779874j))*x[2]
            ref[(1, 1, 2, 2)]=(0.0653852648039-0.349927596859j)
            arg[(1, 1, 2, 3)]=(0.79874882934+0.6155534165j)*x[0] + ((0.412179140678-0.791067398016j))*x[1] + ((0.269713097954-0.330104587924j))*x[2]
            ref[(1, 1, 2, 3)]=(0.740320533986-0.252809284719j)
            arg[(1, 1, 3, 0)]=(0.61551060035-0.736152757948j)*x[0] + ((0.163964765976-0.65607546484j))*x[1] + ((0.431114475713-0.344222557912j))*x[2]
            ref[(1, 1, 3, 0)]=(0.605294921019-0.868225390349j)
            arg[(1, 1, 3, 1)]=(-0.243926217822+0.785984018493j)*x[0] + ((-0.713110042368+0.838915714544j))*x[1] + ((-0.548040188147-0.66585926461j))*x[2]
            ref[(1, 1, 3, 1)]=(-0.752538224168+0.479520234213j)
            arg[(1, 1, 3, 2)]=(-0.0569440352823-0.856411142698j)*x[0] + ((0.866730864164-0.305071471087j))*x[1] + ((0.849727488432-0.259739407304j))*x[2]
            ref[(1, 1, 3, 2)]=(0.829757158657-0.710611010545j)
            arg[(1, 1, 3, 3)]=(0.958903969246-0.0649166546127j)*x[0] + ((0.973138918374-0.288514927599j))*x[1] + ((-0.731435851099-0.416516303827j))*x[2]
            ref[(1, 1, 3, 3)]=(0.600303518261-0.384973943019j)
            arg[(2, 0, 0, 0)]=(-0.857841027053+0.285758886742j)*x[0] + ((-0.564273480727-0.978108637743j))*x[1] + ((0.141682245588-0.854804399995j))*x[2]
            ref[(2, 0, 0, 0)]=(-0.640216131096-0.773577075498j)
            arg[(2, 0, 0, 1)]=(0.529934130796-0.742832877162j)*x[0] + ((-0.111242911929-0.0241189103803j))*x[1] + ((0.88197796669-0.433492156184j))*x[2]
            ref[(2, 0, 0, 1)]=(0.650334592778-0.600221971863j)
            arg[(2, 0, 0, 2)]=(0.487229296086+0.0140451906627j)*x[0] + ((-0.169892646599+0.41056784364j))*x[1] + ((-0.1694681966+0.761476075113j))*x[2]
            ref[(2, 0, 0, 2)]=(0.0739342264434+0.593044554708j)
            arg[(2, 0, 0, 3)]=(-0.939828100547+0.605820708868j)*x[0] + ((-0.594545585121+0.509444781352j))*x[1] + ((-0.844434730452+0.621642835281j))*x[2]
            ref[(2, 0, 0, 3)]=(-1.18940420806+0.86845416275j)
            arg[(2, 0, 1, 0)]=(-0.443616456506+0.547828972547j)*x[0] + ((0.276040291725+0.789040918601j))*x[1] + ((-0.320594569126-0.2237198839j))*x[2]
            ref[(2, 0, 1, 0)]=(-0.244085366954+0.556575003624j)
            arg[(2, 0, 1, 1)]=(-0.337158693648+0.545141390093j)*x[0] + ((0.350098334742+0.604644218168j))*x[1] + ((0.505905016997+0.845363110883j))*x[2]
            ref[(2, 0, 1, 1)]=(0.259422329045+0.997574359572j)
            arg[(2, 0, 1, 2)]=(0.432502635846-0.665645567068j)*x[0] + ((0.117866723461+0.13847394003j))*x[1] + ((-0.413022389786+0.336201827073j))*x[2]
            ref[(2, 0, 1, 2)]=(0.0686734847605-0.0954848999826j)
            arg[(2, 0, 1, 3)]=(-0.280050066534+0.337136248179j)*x[0] + ((-0.744695707214-0.275509564362j))*x[1] + ((0.318342400832-0.235103448421j))*x[2]
            ref[(2, 0, 1, 3)]=(-0.353201686457-0.0867383823019j)
            arg[(2, 0, 2, 0)]=(-0.442164250354+0.853536560921j)*x[0] + ((-0.202064904799-0.371164499721j))*x[1] + ((0.370458470882+0.625991253499j))*x[2]
            ref[(2, 0, 2, 0)]=(-0.136885342135+0.55418165735j)
            arg[(2, 0, 2, 1)]=(0.532172671646-0.714731530446j)*x[0] + ((-0.277459034348-0.250420512074j))*x[1] + ((-0.860974052818-0.329229408419j))*x[2]
            ref[(2, 0, 2, 1)]=(-0.30313020776-0.64719072547j)
            arg[(2, 0, 2, 2)]=(-0.643769069532-0.725291409157j)*x[0] + ((0.708217361341-0.293559095325j))*x[1] + ((-0.674133907642+0.45873316789j))*x[2]
            ref[(2, 0, 2, 2)]=(-0.304842807916-0.280058668296j)
            arg[(2, 0, 2, 3)]=(-0.583348405826-0.220690929701j)*x[0] + ((-0.882721512367+0.891112168776j))*x[1] + ((-0.749234010205-0.624146775194j))*x[2]
            ref[(2, 0, 2, 3)]=(-1.1076519642+0.0231372319409j)
            arg[(2, 0, 3, 0)]=(-0.751597780965+0.802428904454j)*x[0] + ((0.569163367805-0.374828588438j))*x[1] + ((-0.228589403736-0.260975839557j))*x[2]
            ref[(2, 0, 3, 0)]=(-0.205511908448+0.0833122382292j)
            arg[(2, 0, 3, 1)]=(-0.843539988285-0.522624151106j)*x[0] + ((0.227411367545-0.799345562207j))*x[1] + ((0.0576811491108-0.824459177251j))*x[2]
            ref[(2, 0, 3, 1)]=(-0.279223735815-1.07321444528j)
            arg[(2, 0, 3, 2)]=(-0.432769025304-0.320791836906j)*x[0] + ((-0.0788097613882-0.755841142133j))*x[1] + ((0.293357986611+0.113327247978j))*x[2]
            ref[(2, 0, 3, 2)]=(-0.10911040004-0.48165286553j)
            arg[(2, 0, 3, 3)]=(0.508672664705+0.737263564741j)*x[0] + ((0.491655780507+0.466947055394j))*x[1] + ((0.806256983323-0.243910048628j))*x[2]
            ref[(2, 0, 3, 3)]=(0.903292714267+0.480150285754j)
            arg[(2, 1, 0, 0)]=(0.00780499335632+0.482231837731j)*x[0] + ((-0.134929486406-0.906974675157j))*x[1] + ((-0.095822817455-0.344799996662j))*x[2]
            ref[(2, 1, 0, 0)]=(-0.111473655252-0.384771417044j)
            arg[(2, 1, 0, 1)]=(0.277593095245-0.494647988314j)*x[0] + ((-0.683810181653-0.749742561516j))*x[1] + ((-0.57718590384-0.2545505089j))*x[2]
            ref[(2, 1, 0, 1)]=(-0.491701495124-0.749470529365j)
            arg[(2, 1, 0, 2)]=(0.925924029227-0.884381081759j)*x[0] + ((-0.106998781417+0.655967578179j))*x[1] + ((0.0853540804883-0.161077748735j))*x[2]
            ref[(2, 1, 0, 2)]=(0.452139664149-0.194745626157j)
            arg[(2, 1, 0, 3)]=(-0.10480538237-0.462202124131j)*x[0] + ((0.321006055852-0.842792670894j))*x[1] + ((-0.80907695523-0.382153346313j))*x[2]
            ref[(2, 1, 0, 3)]=(-0.296438140874-0.843574070669j)
            arg[(2, 1, 1, 0)]=(-0.907489896309-0.134298126529j)*x[0] + ((0.860703057979-0.637930736581j))*x[1] + ((-0.0868925137597-0.835049459687j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.0668396760451-0.803639161399j)
            arg[(2, 1, 1, 1)]=(0.878020688699-0.742104057343j)*x[0] + ((0.51221846669-0.745327160164j))*x[1] + ((0.157909896234-0.490041844626j))*x[2]
            ref[(2, 1, 1, 1)]=(0.774074525811-0.988736531066j)
            arg[(2, 1, 1, 2)]=(0.587947308699+0.522391545566j)*x[0] + ((-0.579194697328-0.0837395619325j))*x[1] + ((0.904546761734+0.427156221245j))*x[2]
            ref[(2, 1, 1, 2)]=(0.456649686552+0.43290410244j)
            arg[(2, 1, 1, 3)]=(0.2582687969-0.0654102139969j)*x[0] + ((-0.990024526672-0.599081562558j))*x[1] + ((-0.154876978476+0.290093781295j))*x[2]
            ref[(2, 1, 1, 3)]=(-0.443316354124-0.18719899763j)
            arg[(2, 1, 2, 0)]=(0.68056773871+0.816395971723j)*x[0] + ((0.473710486047-0.753454014569j))*x[1] + ((-0.347131693638+0.526332723957j))*x[2]
            ref[(2, 1, 2, 0)]=(0.40357326556+0.294637340555j)
            arg[(2, 1, 2, 1)]=(0.985219471845-0.589074732605j)*x[0] + ((-0.351602299161-0.0920792381434j))*x[1] + ((0.411609596931+0.476556487749j))*x[2]
            ref[(2, 1, 2, 1)]=(0.522613384807-0.1022987415j)
            arg[(2, 1, 2, 2)]=(-0.654884308059+0.750185036364j)*x[0] + ((0.547508717185-0.142170407782j))*x[1] + ((0.161157444951+0.151527972603j))*x[2]
            ref[(2, 1, 2, 2)]=(0.0268909270382+0.379771300592j)
            arg[(2, 1, 2, 3)]=(-0.144623867324+0.202081973755j)*x[0] + ((-0.657318463539+0.0438622986493j))*x[1] + ((0.789262599858+0.791503464244j))*x[2]
            ref[(2, 1, 2, 3)]=(-0.00633986550292+0.518723868324j)
            arg[(2, 1, 3, 0)]=(-0.252155623965-0.447707878608j)*x[0] + ((0.626977886538+0.687402726198j))*x[1] + ((0.275951061009-0.261324537633j))*x[2]
            ref[(2, 1, 3, 0)]=(0.325386661791-0.0108148450214j)
            arg[(2, 1, 3, 1)]=(-0.746376922139+0.177842607957j)*x[0] + ((0.212073158498+0.0211715238233j))*x[1] + ((-0.181505512767-0.841156816076j))*x[2]
            ref[(2, 1, 3, 1)]=(-0.357904638204-0.321071342148j)
            arg[(2, 1, 3, 2)]=(-0.854549682445-0.138125892909j)*x[0] + ((-0.668524559943-0.411391078182j))*x[1] + ((0.0985949149493+0.559347645922j))*x[2]
            ref[(2, 1, 3, 2)]=(-0.712239663719+0.00491533741558j)
            arg[(2, 1, 3, 3)]=(-0.252611112406-0.63181562464j)*x[0] + ((0.10057516137+0.8601868809j))*x[1] + ((0.600572839414+0.576673697263j))*x[2]
            ref[(2, 1, 3, 3)]=(0.224268444189+0.402522476762j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 2, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedContinuousFunction_rank0(self):
        """
        tests integral of rank 0 Data on ReducedFunction

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.64164760851+0.488645792629j)*x[0] + ((0.048077286981-0.21853002464j))*x[1]
            ref=(0.344862447746+0.135057883994j)
        else:
            arg=(0.115433898234+0.466205301775j)*x[0] + ((0.90819265641+0.459547462277j))*x[1] + ((-0.725712845213+0.544657657681j))*x[2]
            ref=(0.148956854716+0.735205210866j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedContinuousFunction_rank1(self):
        """
        tests integral of rank 1 Data on ReducedFunction

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.0212639004617-0.206099730194j)*x[0] + ((-0.337116905894-0.295636508931j))*x[1]
            ref[(0,)]=(-0.179190403178-0.250868119562j)
            arg[(1,)]=(0.92367482074+0.774166233117j)*x[0] + ((0.975188044986-0.476498634845j))*x[1]
            ref[(1,)]=(0.949431432863+0.148833799136j)
            arg[(2,)]=(-0.730484013572-0.874732427235j)*x[0] + ((-0.867055391383+0.566383312839j))*x[1]
            ref[(2,)]=(-0.798769702478-0.154174557198j)
            arg[(3,)]=(0.540466902263+0.737487926563j)*x[0] + ((0.790592567041+0.456430664224j))*x[1]
            ref[(3,)]=(0.665529734652+0.596959295393j)
        else:
            arg[(0,)]=(-0.570014857948-0.898077134773j)*x[0] + ((-0.00959647287716-0.905596811923j))*x[1] + ((-0.517422915622-0.94023100602j))*x[2]
            ref[(0,)]=(-0.548517123224-1.37195247636j)
            arg[(1,)]=(-0.52694024417-0.412457913988j)*x[0] + ((-0.921965273949+0.564482015401j))*x[1] + ((0.385425620535-0.901714797133j))*x[2]
            ref[(1,)]=(-0.531739948792-0.37484534786j)
            arg[(2,)]=(-0.812912305208+0.633871315633j)*x[0] + ((-0.362173452161+0.467930921645j))*x[1] + ((-0.417314325357-0.629060534994j))*x[2]
            ref[(2,)]=(-0.796200041362+0.236370851142j)
            arg[(3,)]=(0.839747242172-0.0562148137949j)*x[0] + ((0.918785238122-0.99107645451j))*x[1] + ((0.458787576982+0.484556804939j))*x[2]
            ref[(3,)]=(1.10866002864-0.281367231683j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedContinuousFunction_rank2(self):
        """
        tests integral of rank 2 Data on ReducedFunction

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 2),w)
        ref=numpy.zeros((3, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.051307155875-0.583602678929j)*x[0] + ((-0.673382171681+0.413200513072j))*x[1]
            ref[(0, 0)]=(-0.311037507903-0.0852010829284j)
            arg[(0, 1)]=(0.780708611057-0.576916879673j)*x[0] + ((-0.926610429243+0.639517363118j))*x[1]
            ref[(0, 1)]=(-0.0729509090933+0.0313002417226j)
            arg[(1, 0)]=(-0.573500194661+0.44835494916j)*x[0] + ((0.91367696062+0.585109628246j))*x[1]
            ref[(1, 0)]=(0.170088382979+0.516732288703j)
            arg[(1, 1)]=(-0.601840318938-0.737805876613j)*x[0] + ((0.69697954513+0.613402076714j))*x[1]
            ref[(1, 1)]=(0.0475696130961-0.0622018999495j)
            arg[(2, 0)]=(0.483214493115+0.968246045068j)*x[0] + ((0.339132739756-0.218757872926j))*x[1]
            ref[(2, 0)]=(0.411173616436+0.374744086071j)
            arg[(2, 1)]=(-0.188721919331+0.28381379727j)*x[0] + ((-0.885915872531+0.772380918539j))*x[1]
            ref[(2, 1)]=(-0.537318895931+0.528097357905j)
        else:
            arg[(0, 0)]=(-0.135512150599+0.236056603411j)*x[0] + ((0.599401556282+0.976290669886j))*x[1] + ((0.155961126535+0.849285891323j))*x[2]
            ref[(0, 0)]=(0.309925266109+1.03081658231j)
            arg[(0, 1)]=(0.573270299004+0.559828502483j)*x[0] + ((0.389469771689-0.417075383118j))*x[1] + ((-0.613329886342-0.742075130485j))*x[2]
            ref[(0, 1)]=(0.174705092175-0.29966100556j)
            arg[(1, 0)]=(-0.480022450905-0.318056356636j)*x[0] + ((0.421591187023-0.875531565467j))*x[1] + ((-0.740709930563-0.402109784225j))*x[2]
            ref[(1, 0)]=(-0.399570597222-0.797848853164j)
            arg[(1, 1)]=(-0.394904508557-0.60576113334j)*x[0] + ((-0.222697406217-0.0475297539387j))*x[1] + ((0.490609304052+0.932060565022j))*x[2]
            ref[(1, 1)]=(-0.0634963053611+0.139384838872j)
            arg[(2, 0)]=(0.571110626928+0.543899415384j)*x[0] + ((0.794278196832-0.42380839615j))*x[1] + ((-0.989016164579-0.884677182072j))*x[2]
            ref[(2, 0)]=(0.18818632959-0.382293081419j)
            arg[(2, 1)]=(-0.640363455008+0.291486763774j)*x[0] + ((0.741144919068-0.268211626987j))*x[1] + ((0.995337526223-0.814327064772j))*x[2]
            ref[(2, 1)]=(0.548059495142-0.395525963993j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedContinuousFunction_rank3(self):
        """
        tests integral of rank 3 Data on ReducedFunction

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 3),w)
        ref=numpy.zeros((3, 4, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.910447122788+0.633340478666j)*x[0] + ((0.480920406119+0.755809437671j))*x[1]
            ref[(0, 0, 0)]=(0.695683764453+0.694574958168j)
            arg[(0, 0, 1)]=(0.280391005429+0.424150424142j)*x[0] + ((0.771003687101-0.307155684571j))*x[1]
            ref[(0, 0, 1)]=(0.525697346265+0.0584973697854j)
            arg[(0, 0, 2)]=(-0.251997236884-0.322031972532j)*x[0] + ((0.853175161068+0.895962539024j))*x[1]
            ref[(0, 0, 2)]=(0.300588962092+0.286965283246j)
            arg[(0, 1, 0)]=(-0.895138311421-0.170834447639j)*x[0] + ((-0.555569413545-0.552802045065j))*x[1]
            ref[(0, 1, 0)]=(-0.725353862483-0.361818246352j)
            arg[(0, 1, 1)]=(-0.667324056518+0.0846045025038j)*x[0] + ((0.506524607159-0.87266314375j))*x[1]
            ref[(0, 1, 1)]=(-0.0803997246797-0.394029320623j)
            arg[(0, 1, 2)]=(-0.662320090576+0.00366895767907j)*x[0] + ((0.577886214604+0.800025268235j))*x[1]
            ref[(0, 1, 2)]=(-0.0422169379857+0.401847112957j)
            arg[(0, 2, 0)]=(-0.813131257065-0.39206372952j)*x[0] + ((-0.825486193014-0.826890367087j))*x[1]
            ref[(0, 2, 0)]=(-0.819308725039-0.609477048303j)
            arg[(0, 2, 1)]=(-0.656464408337+0.74308883209j)*x[0] + ((-0.548808209411-0.34371852957j))*x[1]
            ref[(0, 2, 1)]=(-0.602636308874+0.19968515126j)
            arg[(0, 2, 2)]=(-0.0400533425961+0.534377097838j)*x[0] + ((0.600140143191-0.489780535614j))*x[1]
            ref[(0, 2, 2)]=(0.280043400297+0.0222982811119j)
            arg[(0, 3, 0)]=(-0.0368391419482-0.645908783998j)*x[0] + ((-0.458129700981+0.24097122273j))*x[1]
            ref[(0, 3, 0)]=(-0.247484421464-0.202468780634j)
            arg[(0, 3, 1)]=(-0.648548360765-0.330866581038j)*x[0] + ((0.960561479678+0.901741647031j))*x[1]
            ref[(0, 3, 1)]=(0.156006559456+0.285437532996j)
            arg[(0, 3, 2)]=(0.424668053582-0.699398038963j)*x[0] + ((0.918573440685-0.0602901862676j))*x[1]
            ref[(0, 3, 2)]=(0.671620747133-0.379844112615j)
            arg[(1, 0, 0)]=(-0.0707674547359+0.147771538194j)*x[0] + ((0.442987277835-0.873973562212j))*x[1]
            ref[(1, 0, 0)]=(0.186109911549-0.363101012009j)
            arg[(1, 0, 1)]=(-0.0527321943681-0.630482185917j)*x[0] + ((0.736753306621+0.0412296420951j))*x[1]
            ref[(1, 0, 1)]=(0.342010556127-0.294626271911j)
            arg[(1, 0, 2)]=(-0.874087391725+0.878433243174j)*x[0] + ((0.699628599682-0.303855727676j))*x[1]
            ref[(1, 0, 2)]=(-0.0872293960215+0.287288757749j)
            arg[(1, 1, 0)]=(-0.603097228583-0.969097262977j)*x[0] + ((0.780204942661+0.380516717239j))*x[1]
            ref[(1, 1, 0)]=(0.0885538570386-0.294290272869j)
            arg[(1, 1, 1)]=(0.284717027471-0.681993767099j)*x[0] + ((0.460331699255-0.620728368368j))*x[1]
            ref[(1, 1, 1)]=(0.372524363363-0.651361067734j)
            arg[(1, 1, 2)]=(-0.713255022669-0.0486986393298j)*x[0] + ((-0.480519692097+0.438033291927j))*x[1]
            ref[(1, 1, 2)]=(-0.596887357383+0.194667326298j)
            arg[(1, 2, 0)]=(0.168815041874-0.659635629806j)*x[0] + ((0.361042636787-0.549699678553j))*x[1]
            ref[(1, 2, 0)]=(0.264928839331-0.604667654179j)
            arg[(1, 2, 1)]=(-0.0777729383257+0.144082218983j)*x[0] + ((0.736676502496+0.701671999508j))*x[1]
            ref[(1, 2, 1)]=(0.329451782085+0.422877109246j)
            arg[(1, 2, 2)]=(0.90008195221-0.0411123125505j)*x[0] + ((-0.564495609936-0.489472495134j))*x[1]
            ref[(1, 2, 2)]=(0.167793171137-0.265292403842j)
            arg[(1, 3, 0)]=(-0.580234814936+0.902773066757j)*x[0] + ((0.356660867241+0.821944536399j))*x[1]
            ref[(1, 3, 0)]=(-0.111786973847+0.862358801578j)
            arg[(1, 3, 1)]=(0.0407164928424+0.414278707512j)*x[0] + ((0.976248395562-0.888311938891j))*x[1]
            ref[(1, 3, 1)]=(0.508482444202-0.23701661569j)
            arg[(1, 3, 2)]=(-0.0100798593525+0.611232773593j)*x[0] + ((-0.681153005408-0.510161327056j))*x[1]
            ref[(1, 3, 2)]=(-0.34561643238+0.0505357232684j)
            arg[(2, 0, 0)]=(-0.0569424094987-0.647053983269j)*x[0] + ((0.655363153429+0.165541434994j))*x[1]
            ref[(2, 0, 0)]=(0.299210371965-0.240756274138j)
            arg[(2, 0, 1)]=(0.423907872754+0.704008788455j)*x[0] + ((-0.936668352232+0.653975910764j))*x[1]
            ref[(2, 0, 1)]=(-0.256380239739+0.67899234961j)
            arg[(2, 0, 2)]=(-0.673272231679-0.789160093346j)*x[0] + ((-0.499494716766-0.468235400888j))*x[1]
            ref[(2, 0, 2)]=(-0.586383474222-0.628697747117j)
            arg[(2, 1, 0)]=(0.11512680987-0.906410478684j)*x[0] + ((-0.917206475783+0.219037578484j))*x[1]
            ref[(2, 1, 0)]=(-0.401039832956-0.3436864501j)
            arg[(2, 1, 1)]=(-0.0206775794235-0.151005277695j)*x[0] + ((-0.863921489955-0.0484445974983j))*x[1]
            ref[(2, 1, 1)]=(-0.442299534689-0.0997249375966j)
            arg[(2, 1, 2)]=(-0.527576787557+0.454468604916j)*x[0] + ((-0.672172306873+0.291771522024j))*x[1]
            ref[(2, 1, 2)]=(-0.599874547215+0.37312006347j)
            arg[(2, 2, 0)]=(0.0856738289975+0.667172488852j)*x[0] + ((0.933332106558-0.888760135335j))*x[1]
            ref[(2, 2, 0)]=(0.509502967778-0.110793823241j)
            arg[(2, 2, 1)]=(-0.858936872268-0.538739803162j)*x[0] + ((-0.0571153929+0.905465045931j))*x[1]
            ref[(2, 2, 1)]=(-0.458026132584+0.183362621384j)
            arg[(2, 2, 2)]=(0.480328438681+0.600028039223j)*x[0] + ((0.857734990498-0.569442984466j))*x[1]
            ref[(2, 2, 2)]=(0.669031714589+0.0152925273785j)
            arg[(2, 3, 0)]=(-0.987997700833+0.146878003769j)*x[0] + ((0.711995136128-0.622104838719j))*x[1]
            ref[(2, 3, 0)]=(-0.138001282353-0.237613417475j)
            arg[(2, 3, 1)]=(-0.0661141105292-0.263437857596j)*x[0] + ((0.0353712756853-0.663038899971j))*x[1]
            ref[(2, 3, 1)]=(-0.015371417422-0.463238378784j)
            arg[(2, 3, 2)]=(-0.667365008639-0.643303175855j)*x[0] + ((0.524762684709+0.262202388581j))*x[1]
            ref[(2, 3, 2)]=(-0.0713011619651-0.190550393637j)
        else:
            arg[(0, 0, 0)]=(0.998662656818+0.0130471602265j)*x[0] + ((-0.916939895022+0.747738457836j))*x[1] + ((0.981004125298+0.816063418522j))*x[2]
            ref[(0, 0, 0)]=(0.531363443547+0.788424518292j)
            arg[(0, 0, 1)]=(-0.251424045602-0.458429267635j)*x[0] + ((-0.300668301588-0.760177641777j))*x[1] + ((-0.415150962489-0.511268748434j))*x[2]
            ref[(0, 0, 1)]=(-0.48362165484-0.864937828923j)
            arg[(0, 0, 2)]=(-0.268325907281-0.269576654992j)*x[0] + ((0.738441342905-0.442314618823j))*x[1] + ((-0.71152956075+0.60495246102j))*x[2]
            ref[(0, 0, 2)]=(-0.120707062563-0.0534694063975j)
            arg[(0, 1, 0)]=(-0.398889048071+0.362118187573j)*x[0] + ((0.407939194097+0.729671860242j))*x[1] + ((0.203363440057+0.194177481255j))*x[2]
            ref[(0, 1, 0)]=(0.106206793042+0.642983764535j)
            arg[(0, 1, 1)]=(-0.00554989635291+0.96361180934j)*x[0] + ((0.093190893732-0.679335855719j))*x[1] + ((0.417483239128-0.353159796295j))*x[2]
            ref[(0, 1, 1)]=(0.252562118253-0.0344419213372j)
            arg[(0, 1, 2)]=(0.260552651339-0.875044693487j)*x[0] + ((0.674446045063-0.775241994332j))*x[1] + ((-0.969575594552+0.589733242924j))*x[2]
            ref[(0, 1, 2)]=(-0.0172884490751-0.530276722448j)
            arg[(0, 2, 0)]=(-0.997326479444-0.70138701904j)*x[0] + ((0.828447194192+0.265895037441j))*x[1] + ((0.653406726949+0.649747277877j))*x[2]
            ref[(0, 2, 0)]=(0.242263720848+0.107127648139j)
            arg[(0, 2, 1)]=(0.460919835926-0.68588552543j)*x[0] + ((-0.0847361963926-0.669103170974j))*x[1] + ((0.71894489399+0.704056830207j))*x[2]
            ref[(0, 2, 1)]=(0.547564266762-0.325465933098j)
            arg[(0, 2, 2)]=(0.868216760526-0.0282590651181j)*x[0] + ((-0.787857838959+0.519018159381j))*x[1] + ((-0.271766611359+0.264046998291j))*x[2]
            ref[(0, 2, 2)]=(-0.0957038448958+0.377403046277j)
            arg[(0, 3, 0)]=(-0.79942491158+0.486617522447j)*x[0] + ((-0.0730517237191+0.418924082777j))*x[1] + ((0.202792775747-0.721407269645j))*x[2]
            ref[(0, 3, 0)]=(-0.334841929776+0.0920671677894j)
            arg[(0, 3, 1)]=(-0.986540345738+0.146882296327j)*x[0] + ((0.869900438556+0.675923656253j))*x[1] + ((0.884613459896+0.0964316872171j))*x[2]
            ref[(0, 3, 1)]=(0.383986776357+0.459618819899j)
            arg[(0, 3, 2)]=(0.273316874256-0.807122706276j)*x[0] + ((0.619406779422+0.754521878024j))*x[1] + ((-0.883164174684+0.914732111843j))*x[2]
            ref[(0, 3, 2)]=(0.00477973949732+0.431065641795j)
            arg[(1, 0, 0)]=(-0.228267081731-0.862983619652j)*x[0] + ((-0.644584626168+0.814770361641j))*x[1] + ((-0.0928511191599-0.340030461032j))*x[2]
            ref[(1, 0, 0)]=(-0.48285141353-0.194121859522j)
            arg[(1, 0, 1)]=(-0.11960335412+0.872968854838j)*x[0] + ((0.975535034377+0.565365642524j))*x[1] + ((-0.0258407120697+0.555153877279j))*x[2]
            ref[(1, 0, 1)]=(0.415045484094+0.99674418732j)
            arg[(1, 0, 2)]=(-0.393589192239+0.337356378389j)*x[0] + ((0.132605332795+0.714131411462j))*x[1] + ((-0.349928033234-0.413985962203j))*x[2]
            ref[(1, 0, 2)]=(-0.305455946339+0.318750913824j)
            arg[(1, 1, 0)]=(0.138026241386-0.583578068806j)*x[0] + ((-0.493207887061+0.369375745188j))*x[1] + ((-0.452646265872-0.834876445513j))*x[2]
            ref[(1, 1, 0)]=(-0.403913955773-0.524539384566j)
            arg[(1, 1, 1)]=(0.000557184147109+0.879673922221j)*x[0] + ((0.054096896832-0.502996430393j))*x[1] + ((-0.305343023422-0.782569177111j))*x[2]
            ref[(1, 1, 1)]=(-0.125344471221-0.202945842641j)
            arg[(1, 1, 2)]=(0.410420208758+0.494301107941j)*x[0] + ((-0.221793912025-0.513932240885j))*x[1] + ((-0.350040143594+0.69634261769j))*x[2]
            ref[(1, 1, 2)]=(-0.0807069234302+0.338355742373j)
            arg[(1, 2, 0)]=(-0.503303604955-0.806608210773j)*x[0] + ((-0.827075696471-0.65750982804j))*x[1] + ((0.0431730689295-0.287992368505j))*x[2]
            ref[(1, 2, 0)]=(-0.643603116248-0.876055203659j)
            arg[(1, 2, 1)]=(-0.329479504324-0.100284644529j)*x[0] + ((0.881574503596+0.667172979284j))*x[1] + ((0.141516693288+0.135046984846j))*x[2]
            ref[(1, 2, 1)]=(0.34680584628+0.3509676598j)
            arg[(1, 2, 2)]=(0.281774438225-0.682627380598j)*x[0] + ((0.0516565111159+0.250081238939j))*x[1] + ((0.730964028591+0.715149719674j))*x[2]
            ref[(1, 2, 2)]=(0.532197488966+0.141301789008j)
            arg[(1, 3, 0)]=(0.371303489977-0.800203499401j)*x[0] + ((0.23900143674+0.0277445006431j))*x[1] + ((0.570838699409-0.748287529269j))*x[2]
            ref[(1, 3, 0)]=(0.590571813063-0.760373264013j)
            arg[(1, 3, 1)]=(-0.392675519699+0.496953285291j)*x[0] + ((-0.686211114835-0.711810128216j))*x[1] + ((-0.378064325006-0.285086427831j))*x[2]
            ref[(1, 3, 1)]=(-0.72847547977-0.249971635378j)
            arg[(1, 3, 2)]=(-0.193411662692-0.920082736856j)*x[0] + ((-0.374608402842+0.585860309237j))*x[1] + ((0.929851588626-0.769843876948j))*x[2]
            ref[(1, 3, 2)]=(0.180915761546-0.552033152283j)
            arg[(2, 0, 0)]=(-0.405830445225-0.394956762861j)*x[0] + ((-0.750423754701-0.590108999449j))*x[1] + ((-0.2080029487+0.346775465236j))*x[2]
            ref[(2, 0, 0)]=(-0.682128574313-0.319145148537j)
            arg[(2, 0, 1)]=(0.787375191976-0.157544498748j)*x[0] + ((0.274912547658-0.238208046173j))*x[1] + ((0.427847393835-0.713363509933j))*x[2]
            ref[(2, 0, 1)]=(0.745067566735-0.554558027427j)
            arg[(2, 0, 2)]=(-0.78298723595-0.110485058209j)*x[0] + ((-0.775599457592-0.991260080787j))*x[1] + ((0.538679366179+0.504136244758j))*x[2]
            ref[(2, 0, 2)]=(-0.509953663681-0.298804447119j)
            arg[(2, 1, 0)]=(-0.98191244649+0.775575886433j)*x[0] + ((-0.816610571669+0.516002365706j))*x[1] + ((0.454795241157-0.456165641072j))*x[2]
            ref[(2, 1, 0)]=(-0.671863888501+0.417706305534j)
            arg[(2, 1, 1)]=(0.699378872371-0.253935818951j)*x[0] + ((0.363083905433+0.00456930111686j))*x[1] + ((-0.143279872752+0.133277775072j))*x[2]
            ref[(2, 1, 1)]=(0.459591452526-0.0580443713813j)
            arg[(2, 1, 2)]=(-0.235979154263-0.244954994306j)*x[0] + ((-0.822168464142+0.3921131336j))*x[1] + ((0.410812242231+0.126363175308j))*x[2]
            ref[(2, 1, 2)]=(-0.323667688087+0.136760657301j)
            arg[(2, 2, 0)]=(-0.481544979213-0.0433462498824j)*x[0] + ((-0.0540579031158+0.122388295672j))*x[1] + ((0.961406961777+0.638870916122j))*x[2]
            ref[(2, 2, 0)]=(0.212902039724+0.358956480956j)
            arg[(2, 2, 1)]=(-0.212631899374-0.321004717281j)*x[0] + ((0.152920642198-0.973296467732j))*x[1] + ((-0.0489375241819+0.981589644329j))*x[2]
            ref[(2, 2, 1)]=(-0.0543243906788-0.156355770342j)
            arg[(2, 2, 2)]=(-0.646873745833-0.901041408546j)*x[0] + ((0.479392895572+0.0341411629645j))*x[1] + ((-0.0974263525241+0.274667120746j))*x[2]
            ref[(2, 2, 2)]=(-0.132453601392-0.296116562418j)
            arg[(2, 3, 0)]=(0.574698105024+0.857582859586j)*x[0] + ((-0.624212440058-0.397366073849j))*x[1] + ((0.153847570609+0.653406441329j))*x[2]
            ref[(2, 3, 0)]=(0.0521666177875+0.556811613533j)
            arg[(2, 3, 1)]=(-0.120478153136+0.761674613752j)*x[0] + ((0.7965450347+0.534782864284j))*x[1] + ((-0.298107648917+0.229901023918j))*x[2]
            ref[(2, 3, 1)]=(0.188979616324+0.763179250977j)
            arg[(2, 3, 2)]=(-0.0265347806174-0.749338133753j)*x[0] + ((0.765180450763-0.15525881474j))*x[1] + ((-0.22959020021+0.377239500612j))*x[2]
            ref[(2, 3, 2)]=(0.254527734968-0.26367872394j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedContinuousFunction_rank4(self):
        """
        tests integral of rank 4 Data on ReducedFunction

        assumptions: ReducedContinuousFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 2, 3, 4),w)
        ref=numpy.zeros((3, 2, 3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(0.11899304241+0.149184132438j)*x[0] + ((-0.777613471501+0.510417888024j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.329310214546+0.329801010231j)
            arg[(0, 0, 0, 1)]=(0.633580917888+0.179238321773j)*x[0] + ((-0.34613665128+0.871005321537j))*x[1]
            ref[(0, 0, 0, 1)]=(0.143722133304+0.525121821655j)
            arg[(0, 0, 0, 2)]=(0.508397845107-0.0699599126622j)*x[0] + ((0.768862030336+0.526853721637j))*x[1]
            ref[(0, 0, 0, 2)]=(0.638629937721+0.228446904487j)
            arg[(0, 0, 0, 3)]=(-0.188640293955+0.0204735191394j)*x[0] + ((-0.0920787467818-0.22873750325j))*x[1]
            ref[(0, 0, 0, 3)]=(-0.140359520368-0.104131992055j)
            arg[(0, 0, 1, 0)]=(0.380666878432+0.506376630053j)*x[0] + ((-0.233794199351+0.993271221533j))*x[1]
            ref[(0, 0, 1, 0)]=(0.0734363395406+0.749823925793j)
            arg[(0, 0, 1, 1)]=(-0.65954024408-0.310679705383j)*x[0] + ((0.972002594746+0.581373221402j))*x[1]
            ref[(0, 0, 1, 1)]=(0.156231175333+0.135346758009j)
            arg[(0, 0, 1, 2)]=(0.633412801822+0.814284787473j)*x[0] + ((-0.562515077159+0.562391071498j))*x[1]
            ref[(0, 0, 1, 2)]=(0.0354488623314+0.688337929486j)
            arg[(0, 0, 1, 3)]=(-0.80115960663+0.940728942753j)*x[0] + ((0.273154313217+0.535052486469j))*x[1]
            ref[(0, 0, 1, 3)]=(-0.264002646706+0.737890714611j)
            arg[(0, 0, 2, 0)]=(0.958009524024-0.884068583986j)*x[0] + ((0.545846647353-0.802405632086j))*x[1]
            ref[(0, 0, 2, 0)]=(0.751928085688-0.843237108036j)
            arg[(0, 0, 2, 1)]=(-0.532026304797+0.950040742329j)*x[0] + ((0.553556070059-0.675088312109j))*x[1]
            ref[(0, 0, 2, 1)]=(0.0107648826308+0.13747621511j)
            arg[(0, 0, 2, 2)]=(0.25300733542-0.866594699872j)*x[0] + ((0.898209102146+0.177496311046j))*x[1]
            ref[(0, 0, 2, 2)]=(0.575608218783-0.344549194413j)
            arg[(0, 0, 2, 3)]=(-0.466724902833+0.846977492687j)*x[0] + ((0.144246151636+0.179910237714j))*x[1]
            ref[(0, 0, 2, 3)]=(-0.161239375599+0.513443865201j)
            arg[(0, 1, 0, 0)]=(0.397522695968+0.561511179234j)*x[0] + ((0.380448840743+0.732541452083j))*x[1]
            ref[(0, 1, 0, 0)]=(0.388985768356+0.647026315658j)
            arg[(0, 1, 0, 1)]=(-0.811355592827+0.473291571403j)*x[0] + ((-0.842123202719+0.116222313111j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.826739397773+0.294756942257j)
            arg[(0, 1, 0, 2)]=(-0.705922223073+0.270250377854j)*x[0] + ((-0.533651089991+0.32519741648j))*x[1]
            ref[(0, 1, 0, 2)]=(-0.619786656532+0.297723897167j)
            arg[(0, 1, 0, 3)]=(-0.389961938883-0.50933653894j)*x[0] + ((0.149688978729-0.515622774894j))*x[1]
            ref[(0, 1, 0, 3)]=(-0.120136480077-0.512479656917j)
            arg[(0, 1, 1, 0)]=(-0.471047597749-0.884456850505j)*x[0] + ((-0.873928836572-0.107110272998j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.672488217161-0.495783561751j)
            arg[(0, 1, 1, 1)]=(-0.998284786128-0.717420624444j)*x[0] + ((0.549952643087-0.372461680099j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.22416607152-0.544941152272j)
            arg[(0, 1, 1, 2)]=(0.927453935191+0.455880771516j)*x[0] + ((0.580154697656+0.640344898162j))*x[1]
            ref[(0, 1, 1, 2)]=(0.753804316423+0.548112834839j)
            arg[(0, 1, 1, 3)]=(-0.802645471595-0.733196232008j)*x[0] + ((0.796776880431+0.667006153264j))*x[1]
            ref[(0, 1, 1, 3)]=(-0.00293429558212-0.0330950393718j)
            arg[(0, 1, 2, 0)]=(0.938874155641-0.817291901946j)*x[0] + ((0.340630131211+0.678225798263j))*x[1]
            ref[(0, 1, 2, 0)]=(0.639752143426-0.0695330518412j)
            arg[(0, 1, 2, 1)]=(-0.331184788668-0.65641751006j)*x[0] + ((-0.265541396431+0.987719151984j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.29836309255+0.165650820962j)
            arg[(0, 1, 2, 2)]=(0.905568337248+0.394341241557j)*x[0] + ((0.346335027219+0.25301346276j))*x[1]
            ref[(0, 1, 2, 2)]=(0.625951682233+0.323677352159j)
            arg[(0, 1, 2, 3)]=(0.742543411593-0.529277095076j)*x[0] + ((-0.864363636872-0.376463313285j))*x[1]
            ref[(0, 1, 2, 3)]=(-0.0609101126397-0.45287020418j)
            arg[(1, 0, 0, 0)]=(-0.323478971821-0.398127677548j)*x[0] + ((-0.147358898451+0.141511532361j))*x[1]
            ref[(1, 0, 0, 0)]=(-0.235418935136-0.128308072593j)
            arg[(1, 0, 0, 1)]=(-0.553268891611+0.9645325358j)*x[0] + ((0.0804244571743-0.0637896023754j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.236422217218+0.450371466712j)
            arg[(1, 0, 0, 2)]=(0.801042426989+0.948402649316j)*x[0] + ((0.576352887483+0.740753438092j))*x[1]
            ref[(1, 0, 0, 2)]=(0.688697657236+0.844578043704j)
            arg[(1, 0, 0, 3)]=(0.152926724689-0.275662723128j)*x[0] + ((0.132055830522-0.104370662318j))*x[1]
            ref[(1, 0, 0, 3)]=(0.142491277606-0.190016692723j)
            arg[(1, 0, 1, 0)]=(0.192805971596+0.0231564926202j)*x[0] + ((-0.937593412596-0.519017332694j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.3723937205-0.247930420037j)
            arg[(1, 0, 1, 1)]=(0.462346943976+0.218377408902j)*x[0] + ((-0.272779142241-0.711841842689j))*x[1]
            ref[(1, 0, 1, 1)]=(0.0947839008677-0.246732216893j)
            arg[(1, 0, 1, 2)]=(-0.435522585716+0.37783689619j)*x[0] + ((0.915071334515+0.04782630728j))*x[1]
            ref[(1, 0, 1, 2)]=(0.239774374399+0.212831601735j)
            arg[(1, 0, 1, 3)]=(-0.449755388598+0.95939279579j)*x[0] + ((-0.834116552648+0.772934839711j))*x[1]
            ref[(1, 0, 1, 3)]=(-0.641935970623+0.86616381775j)
            arg[(1, 0, 2, 0)]=(-0.468088825607-0.638153176055j)*x[0] + ((-0.0631417357268-0.530257517471j))*x[1]
            ref[(1, 0, 2, 0)]=(-0.265615280667-0.584205346763j)
            arg[(1, 0, 2, 1)]=(-0.487937288995+0.0333378345772j)*x[0] + ((-0.296279004818+0.0219862583145j))*x[1]
            ref[(1, 0, 2, 1)]=(-0.392108146906+0.0276620464459j)
            arg[(1, 0, 2, 2)]=(-0.7203538398-0.698981141594j)*x[0] + ((-0.856806944102+0.487017491479j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.788580391951-0.105981825058j)
            arg[(1, 0, 2, 3)]=(0.9746630039+0.318072637575j)*x[0] + ((0.523753979496-0.180077616761j))*x[1]
            ref[(1, 0, 2, 3)]=(0.749208491698+0.068997510407j)
            arg[(1, 1, 0, 0)]=(-0.178307764517+0.972469715102j)*x[0] + ((0.855567073637+0.750523012278j))*x[1]
            ref[(1, 1, 0, 0)]=(0.33862965456+0.86149636369j)
            arg[(1, 1, 0, 1)]=(0.211522437115+0.840064223229j)*x[0] + ((-0.625744690315+0.234837023769j))*x[1]
            ref[(1, 1, 0, 1)]=(-0.2071111266+0.537450623499j)
            arg[(1, 1, 0, 2)]=(-0.198561542102+0.655349238481j)*x[0] + ((-0.488558348266+0.498130127067j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.343559945184+0.576739682774j)
            arg[(1, 1, 0, 3)]=(0.141420353561-0.108267805443j)*x[0] + ((-0.444404165584+0.157866149334j))*x[1]
            ref[(1, 1, 0, 3)]=(-0.151491906011+0.0247991719454j)
            arg[(1, 1, 1, 0)]=(-0.7690221224+0.136287089226j)*x[0] + ((0.0728711272683+0.523624370031j))*x[1]
            ref[(1, 1, 1, 0)]=(-0.348075497566+0.329955729628j)
            arg[(1, 1, 1, 1)]=(-0.599602539334+0.694082885749j)*x[0] + ((-0.886090194073-0.824899745111j))*x[1]
            ref[(1, 1, 1, 1)]=(-0.742846366704-0.0654084296808j)
            arg[(1, 1, 1, 2)]=(-0.345469267879+0.289745685376j)*x[0] + ((-0.758428085822-0.420812228918j))*x[1]
            ref[(1, 1, 1, 2)]=(-0.551948676851-0.0655332717715j)
            arg[(1, 1, 1, 3)]=(0.759119417373+0.99580206963j)*x[0] + ((0.988764240811-0.799084942288j))*x[1]
            ref[(1, 1, 1, 3)]=(0.873941829092+0.0983585636707j)
            arg[(1, 1, 2, 0)]=(-0.644466823252-0.0511473833247j)*x[0] + ((-0.84956812243-0.509824055309j))*x[1]
            ref[(1, 1, 2, 0)]=(-0.747017472841-0.280485719317j)
            arg[(1, 1, 2, 1)]=(-0.841780317171-0.862898353531j)*x[0] + ((-0.912410338125+0.155488209594j))*x[1]
            ref[(1, 1, 2, 1)]=(-0.877095327648-0.353705071968j)
            arg[(1, 1, 2, 2)]=(0.757104555813+0.841358089467j)*x[0] + ((-0.367353952375+0.806429723424j))*x[1]
            ref[(1, 1, 2, 2)]=(0.194875301719+0.823893906445j)
            arg[(1, 1, 2, 3)]=(-0.0488193020879+0.328322434328j)*x[0] + ((0.879839350235-0.310256796637j))*x[1]
            ref[(1, 1, 2, 3)]=(0.415510024073+0.00903281884566j)
            arg[(2, 0, 0, 0)]=(-0.955194807279+0.821269887459j)*x[0] + ((-0.628959014682-0.695233372897j))*x[1]
            ref[(2, 0, 0, 0)]=(-0.79207691098+0.0630182572807j)
            arg[(2, 0, 0, 1)]=(-0.902803842204+0.736111683008j)*x[0] + ((0.561451549597+0.383055487919j))*x[1]
            ref[(2, 0, 0, 1)]=(-0.170676146303+0.559583585464j)
            arg[(2, 0, 0, 2)]=(0.403882645094+0.738655503846j)*x[0] + ((-0.0811931336062-0.983409579111j))*x[1]
            ref[(2, 0, 0, 2)]=(0.161344755744-0.122377037633j)
            arg[(2, 0, 0, 3)]=(-0.830536016462-0.201867322568j)*x[0] + ((0.365428103894-0.441792062191j))*x[1]
            ref[(2, 0, 0, 3)]=(-0.232553956284-0.32182969238j)
            arg[(2, 0, 1, 0)]=(0.795868378191+0.973646771021j)*x[0] + ((0.0241839247819-0.744403349994j))*x[1]
            ref[(2, 0, 1, 0)]=(0.410026151486+0.114621710513j)
            arg[(2, 0, 1, 1)]=(0.785800327278-0.584626924793j)*x[0] + ((-0.930448653502+0.0145721796764j))*x[1]
            ref[(2, 0, 1, 1)]=(-0.0723241631117-0.285027372559j)
            arg[(2, 0, 1, 2)]=(-0.0121804231696+0.420422740865j)*x[0] + ((-0.24124177115+0.629315381556j))*x[1]
            ref[(2, 0, 1, 2)]=(-0.12671109716+0.524869061211j)
            arg[(2, 0, 1, 3)]=(0.347456210009+0.179079620883j)*x[0] + ((-0.302770667791-0.167882486639j))*x[1]
            ref[(2, 0, 1, 3)]=(0.0223427711088+0.00559856712238j)
            arg[(2, 0, 2, 0)]=(-0.066629187773-0.462683352848j)*x[0] + ((-0.0502784624897-0.681920325475j))*x[1]
            ref[(2, 0, 2, 0)]=(-0.0584538251314-0.572301839162j)
            arg[(2, 0, 2, 1)]=(-0.759473794609-0.904259697852j)*x[0] + ((0.17606547713-0.461827462016j))*x[1]
            ref[(2, 0, 2, 1)]=(-0.291704158739-0.683043579934j)
            arg[(2, 0, 2, 2)]=(-0.638759009468-0.159112781577j)*x[0] + ((0.914022433675+0.302606925266j))*x[1]
            ref[(2, 0, 2, 2)]=(0.137631712104+0.0717470718443j)
            arg[(2, 0, 2, 3)]=(0.957263045095+0.406076950235j)*x[0] + ((0.708135518548+0.275742773991j))*x[1]
            ref[(2, 0, 2, 3)]=(0.832699281821+0.340909862113j)
            arg[(2, 1, 0, 0)]=(-0.777146834891+0.648803689763j)*x[0] + ((0.437135460036-0.00195920199822j))*x[1]
            ref[(2, 1, 0, 0)]=(-0.170005687428+0.323422243882j)
            arg[(2, 1, 0, 1)]=(0.578082262637-0.393529965231j)*x[0] + ((0.104971645364+0.747417465728j))*x[1]
            ref[(2, 1, 0, 1)]=(0.341526954+0.176943750248j)
            arg[(2, 1, 0, 2)]=(-0.346452689413-0.386109956048j)*x[0] + ((-0.368317170306-0.468675307189j))*x[1]
            ref[(2, 1, 0, 2)]=(-0.35738492986-0.427392631618j)
            arg[(2, 1, 0, 3)]=(-0.463601863667+0.417377010779j)*x[0] + ((-0.148739328629+0.254734212313j))*x[1]
            ref[(2, 1, 0, 3)]=(-0.306170596148+0.336055611546j)
            arg[(2, 1, 1, 0)]=(0.86754078473-0.406045739351j)*x[0] + ((0.974224290692+0.998790668401j))*x[1]
            ref[(2, 1, 1, 0)]=(0.920882537711+0.296372464525j)
            arg[(2, 1, 1, 1)]=(-0.695373200743+0.238823570091j)*x[0] + ((-0.849049484938+0.820509406263j))*x[1]
            ref[(2, 1, 1, 1)]=(-0.77221134284+0.529666488177j)
            arg[(2, 1, 1, 2)]=(0.770410472565-0.72423146996j)*x[0] + ((0.19597281499-0.809158649778j))*x[1]
            ref[(2, 1, 1, 2)]=(0.483191643777-0.766695059869j)
            arg[(2, 1, 1, 3)]=(-0.761852883475+0.336536778947j)*x[0] + ((-0.90706088455+0.714928850054j))*x[1]
            ref[(2, 1, 1, 3)]=(-0.834456884013+0.525732814501j)
            arg[(2, 1, 2, 0)]=(-0.590327378264+0.286983896827j)*x[0] + ((0.854124420759+0.349988818901j))*x[1]
            ref[(2, 1, 2, 0)]=(0.131898521248+0.318486357864j)
            arg[(2, 1, 2, 1)]=(0.3032180499-0.038492432642j)*x[0] + ((-0.482926919412-0.415675969436j))*x[1]
            ref[(2, 1, 2, 1)]=(-0.0898544347562-0.227084201039j)
            arg[(2, 1, 2, 2)]=(0.817186163132-0.315837263975j)*x[0] + ((0.918784103855-0.423441716749j))*x[1]
            ref[(2, 1, 2, 2)]=(0.867985133493-0.369639490362j)
            arg[(2, 1, 2, 3)]=(0.319970341547-0.0734674848494j)*x[0] + ((0.747994148104-0.593726794011j))*x[1]
            ref[(2, 1, 2, 3)]=(0.533982244826-0.33359713943j)
        else:
            arg[(0, 0, 0, 0)]=(-0.995815165904+0.733943193013j)*x[0] + ((-0.169583984684+0.222028347371j))*x[1] + ((-0.350852409826-0.103347543583j))*x[2]
            ref[(0, 0, 0, 0)]=(-0.758125780207+0.4263119984j)
            arg[(0, 0, 0, 1)]=(0.880238269096-0.995991215115j)*x[0] + ((0.779691367712-0.0833357772227j))*x[1] + ((0.634319803847+0.440094551576j))*x[2]
            ref[(0, 0, 0, 1)]=(1.14712472033-0.319616220381j)
            arg[(0, 0, 0, 2)]=(0.745336051435+0.257400671674j)*x[0] + ((-0.679509935506-0.805438026392j))*x[1] + ((0.448039142054+0.154216779508j))*x[2]
            ref[(0, 0, 0, 2)]=(0.256932628992-0.196910287605j)
            arg[(0, 0, 0, 3)]=(0.589219798464+0.778068313011j)*x[0] + ((0.54869819775+0.558433845114j))*x[1] + ((-0.258804004673+0.60894855191j))*x[2]
            ref[(0, 0, 0, 3)]=(0.43955699577+0.972725355017j)
            arg[(0, 0, 1, 0)]=(0.915486187114+0.0979581144306j)*x[0] + ((0.697734151313+0.464282964612j))*x[1] + ((0.734535253039+0.760479836601j))*x[2]
            ref[(0, 0, 1, 0)]=(1.17387779573+0.661360457822j)
            arg[(0, 0, 1, 1)]=(0.907876674761-0.772411056679j)*x[0] + ((-0.336693885635+0.61638322338j))*x[1] + ((-0.651888648157-0.152537388605j))*x[2]
            ref[(0, 0, 1, 1)]=(-0.0403529295159-0.154282610952j)
            arg[(0, 0, 1, 2)]=(-0.587249234681-0.308377422828j)*x[0] + ((-0.626514450078-0.587729215249j))*x[1] + ((-0.886686154043-0.721784558839j))*x[2]
            ref[(0, 0, 1, 2)]=(-1.0502249194-0.808945598458j)
            arg[(0, 0, 1, 3)]=(0.841493095606-0.661493997123j)*x[0] + ((0.344546672361+0.84414258358j))*x[1] + ((-0.247730198378+0.0431666652808j))*x[2]
            ref[(0, 0, 1, 3)]=(0.469154784794+0.112907625869j)
            arg[(0, 0, 2, 0)]=(-0.383108738696+0.0573468702187j)*x[0] + ((-0.928944307436+0.495674198181j))*x[1] + ((0.628474622195-0.0361164186668j))*x[2]
            ref[(0, 0, 2, 0)]=(-0.341789211969+0.258452324866j)
            arg[(0, 0, 2, 1)]=(0.861363323869+0.819452672084j)*x[0] + ((-0.782642150472-0.730885918769j))*x[1] + ((-0.644323317223-0.505728683482j))*x[2]
            ref[(0, 0, 2, 1)]=(-0.282801071913-0.208580965083j)
            arg[(0, 0, 2, 2)]=(-0.384223297777-0.51288236055j)*x[0] + ((-0.392080277741+0.413514745536j))*x[1] + ((0.681040937155-0.536281272952j))*x[2]
            ref[(0, 0, 2, 2)]=(-0.0476313191812-0.317824443983j)
            arg[(0, 0, 2, 3)]=(0.416149210509-0.178312398954j)*x[0] + ((0.846964682356+0.497863125929j))*x[1] + ((-0.635441717521-0.803699508937j))*x[2]
            ref[(0, 0, 2, 3)]=(0.313836087672-0.242074390981j)
            arg[(0, 1, 0, 0)]=(0.145205252602-0.404601679096j)*x[0] + ((-0.419095564169-0.192444350443j))*x[1] + ((0.346769944586+0.629487335474j))*x[2]
            ref[(0, 1, 0, 0)]=(0.0364398165093+0.016220652967j)
            arg[(0, 1, 0, 1)]=(0.733011164787+0.973274839699j)*x[0] + ((0.219477195836-0.978354478742j))*x[1] + ((-0.634127840135-0.977535731526j))*x[2]
            ref[(0, 1, 0, 1)]=(0.159180260244-0.491307685285j)
            arg[(0, 1, 0, 2)]=(0.757754518064+0.0861211328841j)*x[0] + ((0.28794365721+0.568830812425j))*x[1] + ((0.386548525204+0.208164944376j))*x[2]
            ref[(0, 1, 0, 2)]=(0.716123350239+0.431558444842j)
            arg[(0, 1, 0, 3)]=(-0.77693460312+0.000273636118082j)*x[0] + ((0.348539698297+0.843996468198j))*x[1] + ((0.0567907766283-0.799572036953j))*x[2]
            ref[(0, 1, 0, 3)]=(-0.185802064097+0.0223490336812j)
            arg[(0, 1, 1, 0)]=(-0.550812031156+0.204525119097j)*x[0] + ((0.491226853886-0.403175882059j))*x[1] + ((-0.517769381402-0.807103788227j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.288677279336-0.502877275594j)
            arg[(0, 1, 1, 1)]=(0.0510635754496+0.0392246658594j)*x[0] + ((-0.694569070198-0.569359346135j))*x[1] + ((0.717789090807+0.833091400977j))*x[2]
            ref[(0, 1, 1, 1)]=(0.0371417980297+0.151478360351j)
            arg[(0, 1, 1, 2)]=(-0.075139668299+0.568895495056j)*x[0] + ((0.250259807163+0.105576697935j))*x[1] + ((0.372493362356-0.522578207237j))*x[2]
            ref[(0, 1, 1, 2)]=(0.27380675061+0.0759469928768j)
            arg[(0, 1, 1, 3)]=(-0.753860346151+0.36327383463j)*x[0] + ((-0.72149936366+0.787525817656j))*x[1] + ((0.367948683081+0.852630185347j))*x[2]
            ref[(0, 1, 1, 3)]=(-0.553705513365+1.00171491882j)
            arg[(0, 1, 2, 0)]=(0.831007149589-0.492167432278j)*x[0] + ((-0.150039305855+0.410487584113j))*x[1] + ((0.254829224525-0.957498861995j))*x[2]
            ref[(0, 1, 2, 0)]=(0.46789853413-0.51958935508j)
            arg[(0, 1, 2, 1)]=(-0.615875210814-0.940609177653j)*x[0] + ((0.370803836445+0.0317429112697j))*x[1] + ((-0.0565181979476+0.76511970236j))*x[2]
            ref[(0, 1, 2, 1)]=(-0.150794786158-0.0718732820116j)
            arg[(0, 1, 2, 2)]=(-0.231716160231-0.347195239469j)*x[0] + ((0.769910835506+0.897912311872j))*x[1] + ((-0.279469512531-0.610660864053j))*x[2]
            ref[(0, 1, 2, 2)]=(0.129362581372-0.0299718958252j)
            arg[(0, 1, 2, 3)]=(0.953659194486+0.793440476864j)*x[0] + ((0.0136258849508-0.0588574176099j))*x[1] + ((-0.561535168428-0.139839649877j))*x[2]
            ref[(0, 1, 2, 3)]=(0.202874955504+0.297371704689j)
            arg[(1, 0, 0, 0)]=(0.182897502367-0.817178646914j)*x[0] + ((-0.617352147199-0.973802863677j))*x[1] + ((-0.469798969317-0.565543027065j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.452126807074-1.17826226883j)
            arg[(1, 0, 0, 1)]=(0.66532422016+0.730712313782j)*x[0] + ((0.152338428369-0.930799499292j))*x[1] + ((-0.341791381492-0.930480435993j))*x[2]
            ref[(1, 0, 0, 1)]=(0.237935633519-0.565283810752j)
            arg[(1, 0, 0, 2)]=(0.652583609101+0.55242635586j)*x[0] + ((-0.748186133576+0.191178154433j))*x[1] + ((-0.782161093339-0.879722277183j))*x[2]
            ref[(1, 0, 0, 2)]=(-0.438881808907-0.0680588834449j)
            arg[(1, 0, 0, 3)]=(-0.470241449-0.145281467766j)*x[0] + ((0.501632304138+0.616764721149j))*x[1] + ((0.33566050532+0.0533718877519j))*x[2]
            ref[(1, 0, 0, 3)]=(0.183525680229+0.262427570567j)
            arg[(1, 0, 1, 0)]=(0.425923101666+0.680440708625j)*x[0] + ((0.344540621543+0.144357836067j))*x[1] + ((-0.676001070891+0.670995715826j))*x[2]
            ref[(1, 0, 1, 0)]=(0.0472313261591+0.747897130259j)
            arg[(1, 0, 1, 1)]=(-0.273841220857+0.681557020261j)*x[0] + ((-0.678001919021-0.57010356436j))*x[1] + ((-0.71019704162-0.795503504224j))*x[2]
            ref[(1, 0, 1, 1)]=(-0.831020090749-0.342025024161j)
            arg[(1, 0, 1, 2)]=(-0.704847942188-0.975116588764j)*x[0] + ((0.318017393998+0.634352569621j))*x[1] + ((-0.412660943+0.649422821627j))*x[2]
            ref[(1, 0, 1, 2)]=(-0.399745745595+0.154329401242j)
            arg[(1, 0, 1, 3)]=(-0.825025208724-0.787178113051j)*x[0] + ((-0.87677457835+0.465442005361j))*x[1] + ((-0.255605865815-0.519412169265j))*x[2]
            ref[(1, 0, 1, 3)]=(-0.978702826445-0.420574138478j)
            arg[(1, 0, 2, 0)]=(0.414826517866+0.393373615595j)*x[0] + ((-0.796991066944+0.486469732311j))*x[1] + ((-0.790845309544+0.552482620461j))*x[2]
            ref[(1, 0, 2, 0)]=(-0.586504929311+0.716162984183j)
            arg[(1, 0, 2, 1)]=(0.0716386388442-0.067239518565j)*x[0] + ((0.878376465999+0.213187085301j))*x[1] + ((-0.0605593206273+0.468143256876j))*x[2]
            ref[(1, 0, 2, 1)]=(0.444727892108+0.307045411806j)
            arg[(1, 0, 2, 2)]=(-0.228226497187-0.205538965398j)*x[0] + ((0.0397825614856-0.135849052707j))*x[1] + ((-0.279373593176+0.110486581635j))*x[2]
            ref[(1, 0, 2, 2)]=(-0.233908764438-0.115450718235j)
            arg[(1, 0, 2, 3)]=(-0.624714526293-0.744614354475j)*x[0] + ((-0.129191773705-0.435605828216j))*x[1] + ((-0.222207753976-0.600849742037j))*x[2]
            ref[(1, 0, 2, 3)]=(-0.488057026987-0.890534962364j)
            arg[(1, 1, 0, 0)]=(-0.403855170105-0.782624025583j)*x[0] + ((0.860288058746+0.47987267916j))*x[1] + ((-0.824297163706+0.969195426123j))*x[2]
            ref[(1, 1, 0, 0)]=(-0.183932137532+0.33322203985j)
            arg[(1, 1, 0, 1)]=(-0.8663163965+0.428325430879j)*x[0] + ((-0.26624395059+0.472019189087j))*x[1] + ((0.0878425088249+0.182080142256j))*x[2]
            ref[(1, 1, 0, 1)]=(-0.522358919132+0.541212381111j)
            arg[(1, 1, 0, 2)]=(0.118500164684+0.842477187135j)*x[0] + ((-0.467406068129-0.0840796967166j))*x[1] + ((0.469587181459-0.480390964714j))*x[2]
            ref[(1, 1, 0, 2)]=(0.0603406390072+0.139003262852j)
            arg[(1, 1, 0, 3)]=(-0.257485752522+0.30829297801j)*x[0] + ((0.215079755714-0.00520013634059j))*x[1] + ((-0.86235654987+0.0607274746683j))*x[2]
            ref[(1, 1, 0, 3)]=(-0.452381273339+0.181910158169j)
            arg[(1, 1, 1, 0)]=(0.477286704686-0.467871529233j)*x[0] + ((-0.792678469263-0.569258572991j))*x[1] + ((0.680737354866+0.603706733294j))*x[2]
            ref[(1, 1, 1, 0)]=(0.182672795145-0.216711684465j)
            arg[(1, 1, 1, 1)]=(-0.0621221490371-0.495281680599j)*x[0] + ((-0.345320077149-0.0942186647887j))*x[1] + ((-0.510742300745+0.343748864751j))*x[2]
            ref[(1, 1, 1, 1)]=(-0.459092263465-0.122875740318j)
            arg[(1, 1, 1, 2)]=(0.811644857437+0.938479938786j)*x[0] + ((0.336458746923+0.0492532029855j))*x[1] + ((-0.0269635170317+0.336029887846j))*x[2]
            ref[(1, 1, 1, 2)]=(0.560570043664+0.661881514809j)
            arg[(1, 1, 1, 3)]=(-0.85195128168+0.532838995031j)*x[0] + ((0.269945673471+0.684569373519j))*x[1] + ((0.415407833584+0.200100511199j))*x[2]
            ref[(1, 1, 1, 3)]=(-0.0832988873128+0.708754439875j)
            arg[(1, 1, 2, 0)]=(-0.566703929391-0.398542748051j)*x[0] + ((0.799708214671+0.362848905005j))*x[1] + ((0.760440387323+0.983426538632j))*x[2]
            ref[(1, 1, 2, 0)]=(0.496722336301+0.473866347793j)
            arg[(1, 1, 2, 1)]=(0.95155944311-0.555181610891j)*x[0] + ((0.188216908122-0.552479826102j))*x[1] + ((0.696689659111-0.424424855074j))*x[2]
            ref[(1, 1, 2, 1)]=(0.918233005172-0.766043146034j)
            arg[(1, 1, 2, 2)]=(-0.572043722951+0.903549173041j)*x[0] + ((0.954174845621-0.592173584444j))*x[1] + ((0.95357551822-0.0975797296073j))*x[2]
            ref[(1, 1, 2, 2)]=(0.667853320445+0.106897929495j)
            arg[(1, 1, 2, 3)]=(-0.76479756937-0.229332158149j)*x[0] + ((0.496245790724+0.0348729710081j))*x[1] + ((0.56464986033+0.417052585941j))*x[2]
            ref[(1, 1, 2, 3)]=(0.148049040842+0.1112966994j)
            arg[(2, 0, 0, 0)]=(-0.209712753798+0.54553970742j)*x[0] + ((0.0660536754545-0.0357730189272j))*x[1] + ((0.252995654514-0.377077605732j))*x[2]
            ref[(2, 0, 0, 0)]=(0.0546682880854+0.0663445413803j)
            arg[(2, 0, 0, 1)]=(0.533240055693+0.758204505592j)*x[0] + ((0.784518798698-0.736583076881j))*x[1] + ((-0.956224566683+0.277512667963j))*x[2]
            ref[(2, 0, 0, 1)]=(0.180767143854+0.149567048337j)
            arg[(2, 0, 0, 2)]=(-0.207272321486+0.242420673438j)*x[0] + ((-0.355273879828+0.694727272438j))*x[1] + ((0.0494382145817-0.765952091359j))*x[2]
            ref[(2, 0, 0, 2)]=(-0.256553993366+0.0855979272585j)
            arg[(2, 0, 0, 3)]=(-0.694746391891+0.0968018892707j)*x[0] + ((0.689064072688+0.909617693729j))*x[1] + ((-0.982322466209+0.924925435246j))*x[2]
            ref[(2, 0, 0, 3)]=(-0.494002392706+0.965672509123j)
            arg[(2, 0, 1, 0)]=(-0.269041333338+0.376403684969j)*x[0] + ((-0.812643818012+0.938544319205j))*x[1] + ((0.488163433196+0.842969225758j))*x[2]
            ref[(2, 0, 1, 0)]=(-0.296760859077+1.07895861497j)
            arg[(2, 0, 1, 1)]=(-0.883362108382-0.497804362624j)*x[0] + ((0.507578918316+0.72043531606j))*x[1] + ((0.729443147145-0.911448194714j))*x[2]
            ref[(2, 0, 1, 1)]=(0.176829978539-0.344408620639j)
            arg[(2, 0, 1, 2)]=(-0.934804082235+0.105353282055j)*x[0] + ((-0.314030669961+0.40189617245j))*x[1] + ((0.897428057781-0.0566036232935j))*x[2]
            ref[(2, 0, 1, 2)]=(-0.175703347208+0.225322915606j)
            arg[(2, 0, 1, 3)]=(-0.39703892154-0.19943975231j)*x[0] + ((-0.728997576958+0.793809741922j))*x[1] + ((-0.683497927397+0.544223543027j))*x[2]
            ref[(2, 0, 1, 3)]=(-0.904767212947+0.56929676632j)
            arg[(2, 0, 2, 0)]=(0.0376959628551+0.358996634752j)*x[0] + ((0.745656049014-0.153766275317j))*x[1] + ((0.381530263806+0.629785591967j))*x[2]
            ref[(2, 0, 2, 0)]=(0.582441137837+0.417507975701j)
            arg[(2, 0, 2, 1)]=(-0.848470303562-0.889389466021j)*x[0] + ((0.442417500223-0.250929707411j))*x[1] + ((0.8674488273+0.864113345816j))*x[2]
            ref[(2, 0, 2, 1)]=(0.230698011981-0.138102913808j)
            arg[(2, 0, 2, 2)]=(-0.897961648153-0.795834469981j)*x[0] + ((0.0348016574508-0.626565916764j))*x[1] + ((-0.000582745241646-0.749604126118j))*x[2]
            ref[(2, 0, 2, 2)]=(-0.431871367972-1.08600225643j)
            arg[(2, 0, 2, 3)]=(-0.353361017423+0.577966322317j)*x[0] + ((0.552248993773-0.624321073157j))*x[1] + ((-0.92839067564+0.879298171934j))*x[2]
            ref[(2, 0, 2, 3)]=(-0.364751349645+0.416471710547j)
            arg[(2, 1, 0, 0)]=(-0.845823124293-0.922793212411j)*x[0] + ((0.190050360926+0.159447086857j))*x[1] + ((0.700587249307+0.767196902776j))*x[2]
            ref[(2, 1, 0, 0)]=(0.0224072429701+0.00192538861118j)
            arg[(2, 1, 0, 1)]=(0.713498686604-0.230278800709j)*x[0] + ((-0.508541885447-0.189007287849j))*x[1] + ((-0.879038728372-0.683592958752j))*x[2]
            ref[(2, 1, 0, 1)]=(-0.337040963608-0.551439523655j)
            arg[(2, 1, 0, 2)]=(0.788350966969+0.496450620342j)*x[0] + ((0.193007437993+0.188979464803j))*x[1] + ((0.451499865286+0.627909094813j))*x[2]
            ref[(2, 1, 0, 2)]=(0.716429135124+0.656669589979j)
            arg[(2, 1, 0, 3)]=(0.947347124468-0.710595914254j)*x[0] + ((-0.937305609917+0.838847944254j))*x[1] + ((0.971158373732+0.358043070891j))*x[2]
            ref[(2, 1, 0, 3)]=(0.490599944142+0.243147550445j)
            arg[(2, 1, 1, 0)]=(-0.648038060208+0.376889301296j)*x[0] + ((-0.71762872583+0.201180122317j))*x[1] + ((-0.171617531202-0.621811249956j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.76864215862-0.0218709131713j)
            arg[(2, 1, 1, 1)]=(-0.997313395232-0.542606988719j)*x[0] + ((-0.266533888216+0.24648214011j))*x[1] + ((0.493703577824-0.115066884033j))*x[2]
            ref[(2, 1, 1, 1)]=(-0.385071852812-0.205595866321j)
            arg[(2, 1, 1, 2)]=(0.608071199961+0.390649483471j)*x[0] + ((-0.721860332298-0.308000265526j))*x[1] + ((-0.585637313751+0.820101117908j))*x[2]
            ref[(2, 1, 1, 2)]=(-0.349713223044+0.451375167927j)
            arg[(2, 1, 1, 3)]=(-0.250588208295-0.622894253171j)*x[0] + ((0.154707518563+0.126398638082j))*x[1] + ((-0.616140983119+0.129658456176j))*x[2]
            ref[(2, 1, 1, 3)]=(-0.356010836426-0.183418579457j)
            arg[(2, 1, 2, 0)]=(0.374615172789-0.854604365062j)*x[0] + ((0.696728491601-0.971297193893j))*x[1] + ((0.862066785992+0.93389447655j))*x[2]
            ref[(2, 1, 2, 0)]=(0.966705225191-0.446003541203j)
            arg[(2, 1, 2, 1)]=(0.140425376944-0.6298055906j)*x[0] + ((0.825809426953+0.175581695093j))*x[1] + ((-0.763208049098+0.210526546392j))*x[2]
            ref[(2, 1, 2, 1)]=(0.1015133774-0.121848674558j)
            arg[(2, 1, 2, 2)]=(-0.996389005902+0.76389236622j)*x[0] + ((0.235090455919-0.663594729888j))*x[1] + ((0.130406618531-0.505374398216j))*x[2]
            ref[(2, 1, 2, 2)]=(-0.315445965726-0.202538380942j)
            arg[(2, 1, 2, 3)]=(-0.955446140498-0.655027133984j)*x[0] + ((0.318964183609+0.95959159127j))*x[1] + ((-0.881044855279+0.453477454995j))*x[2]
            ref[(2, 1, 2, 3)]=(-0.758763406084+0.379020956141j)
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 2, 3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedFunction_rank0(self):
        """
        tests integral of rank 0 Data on ReducedFunction

        assumptions: ReducedFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(-0.0108939910479+0.079538921822j)*x[0]**o + ((0.333644172892+0.395923617916j))*x[0] + ((-0.371284939448+0.620908078904j))*x[1]**o + ((0.1556269485+0.230476103212j))*x[1]
            ref=(0.0535460954482+0.663423360927j)-((-0.063696488416+0.116741166788j))*(o-1.)
        else:
            arg=(-0.0914804050248-0.00126584886498j)*x[0]**o + ((-0.574624698268+0.485768797458j))*x[0] + ((0.029469601788+0.256083885293j))*x[1]**o + ((-0.788380237765+0.0154075391639j))*x[1] + ((-0.855478111375-0.248614750228j))*x[2]**o + ((-0.973537250364-0.192227239058j))*x[2]
            ref=(-1.6270155505+0.157576191882j)-((-0.152914819102+0.00103388103331j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedFunction_rank1(self):
        """
        tests integral of rank 1 Data on ReducedFunction

        assumptions: ReducedFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3,),w)
        ref=numpy.zeros((3,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.940157899807+0.166110706557j)*x[0]**o + ((0.656348835986+0.455428122594j))*x[0] + ((0.275806505283-0.227853935107j))*x[1]**o + ((0.521602421831+0.779479799961j))*x[1]
            ref[(0,)]=(0.256799931647+0.586582347002j)-((-0.110725232421-0.0102905380917j))*(o-1.)
            arg[(1,)]=(-0.661371102385-0.499284374162j)*x[0]**o + ((0.136376079054-0.825641823942j))*x[0] + ((-0.422249909539+0.957772012196j))*x[1]**o + ((0.370080463797-0.00152753224685j))*x[1]
            ref[(1,)]=(-0.288582234536-0.184340859077j)-((-0.180603501987+0.076414606339j))*(o-1.)
            arg[(2,)]=(-0.763108428366+0.353454916314j)*x[0]**o + ((0.675793987162-0.473054904811j))*x[0] + ((-0.925002959787+0.165643857522j))*x[1]**o + ((-0.870835415051+0.52178793586j))*x[1]
            ref[(2,)]=(-0.941576408021+0.283915902442j)-((-0.281351898025+0.0865164623059j))*(o-1.)
        else:
            arg[(0,)]=(0.98798966142+0.539987050376j)*x[0]**o + ((-0.109213720813+0.196364445945j))*x[0] + ((0.246470768634-0.487895261146j))*x[1]**o + ((0.401508259179+0.0113063138355j))*x[1] + ((-0.71715659942+0.266165011636j))*x[2]**o + ((0.115698125654+0.702555886987j))*x[2]
            ref[(0,)]=(0.462648247327+0.614241723817j)-((0.0862173051056+0.0530428001443j))*(o-1.)
            arg[(1,)]=(0.555071206104-0.0697854750845j)*x[0]**o + ((-0.575253682888-0.816498199517j))*x[0] + ((0.67724715971-0.589119336446j))*x[1]**o + ((0.695572538747+0.734808663731j))*x[1] + ((-0.719815585642-0.459222013473j))*x[2]**o + ((0.401878433589+0.960634085522j))*x[2]
            ref[(1,)]=(0.51735003481-0.119591137634j)-((0.0854171300286-0.186354470834j))*(o-1.)
            arg[(2,)]=(-0.340410011498+0.404545541818j)*x[0]**o + ((0.364199257276-0.0518721480508j))*x[0] + ((-0.769504824459+0.878702329158j))*x[1]**o + ((0.613200056843+0.7062492926j))*x[1] + ((-0.0930031807351-0.373634628109j))*x[2]**o + ((0.66765071553+0.87298879756j))*x[2]
            ref[(2,)]=(0.221066006479+1.21848959249j)-((-0.200486336115+0.151602207144j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedFunction_rank2(self):
        """
        tests integral of rank 2 Data on ReducedFunction

        assumptions: ReducedFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedFunction(self.domain)
        x = w.getX()
        arg = Data(0,(2, 3),w)
        ref=numpy.zeros((2, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.263432106368+0.0560252311984j)*x[0]**o + ((0.0837258853529-0.761935525208j))*x[0] + ((-0.483056514676+0.163226757824j))*x[1]**o + ((0.620743574166+0.0417769044047j))*x[1]
            ref[(0, 0)]=(0.242422525605-0.25045331589j)-((-0.0366040680514+0.0365419981704j))*(o-1.)
            arg[(0, 1)]=(-0.615078415576+0.151244601193j)*x[0]**o + ((0.661112947621-0.474646466763j))*x[0] + ((0.849893404122-0.0499647175885j))*x[1]**o + ((0.804159507937+0.170278802875j))*x[1]
            ref[(0, 1)]=(0.850043722052-0.101543890142j)-((0.0391358314242+0.0168799806007j))*(o-1.)
            arg[(0, 2)]=(0.317268806771-0.835868361484j)*x[0]**o + ((-0.505931755131+0.337136206135j))*x[0] + ((-0.251942886383-0.612713774187j))*x[1]**o + ((-0.561358439025-0.882631669443j))*x[1]
            ref[(0, 2)]=(-0.500982136884-0.99703879949j)-((0.0108876533979-0.241430355945j))*(o-1.)
            arg[(1, 0)]=(0.307831750476+0.192664793861j)*x[0]**o + ((-0.667889768678-0.779474352447j))*x[0] + ((-0.56378559497-0.381394538328j))*x[1]**o + ((-0.852110992312-0.247953186105j))*x[1]
            ref[(1, 0)]=(-0.887977302742-0.608078641509j)-((-0.0426589740823-0.0314549574112j))*(o-1.)
            arg[(1, 1)]=(-0.604016446055+0.474697321115j)*x[0]**o + ((-0.462231471696-0.115163565133j))*x[0] + ((0.214836788285+0.444834411496j))*x[1]**o + ((-0.879025020675-0.391104957394j))*x[1]
            ref[(1, 1)]=(-0.86521807507+0.206631605042j)-((-0.0648632762951+0.153255288769j))*(o-1.)
            arg[(1, 2)]=(0.886790700508+0.072123386009j)*x[0]**o + ((-0.682200650055-0.0940236219037j))*x[0] + ((0.807527645968-0.203713591775j))*x[1]**o + ((-0.711490934008+0.225122616975j))*x[1]
            ref[(1, 2)]=(0.150313381207-0.000245605347575j)-((0.282386391079-0.021931700961j))*(o-1.)
        else:
            arg[(0, 0)]=(0.99321095328+0.480869091978j)*x[0]**o + ((0.0935807844755-0.0794657770862j))*x[0] + ((-0.0894526477852+0.868865021344j))*x[1]**o + ((-0.333011396969+0.639101202894j))*x[1] + ((-0.241550026781+0.300536283782j))*x[2]**o + ((-0.887834905338-0.436084743879j))*x[2]
            ref[(0, 0)]=(-0.232528619559+0.886910539517j)-((0.110368046452+0.275045066184j))*(o-1.)
            arg[(0, 1)]=(0.863926017253+0.99790810939j)*x[0]**o + ((-0.67184097933-0.681116591062j))*x[0] + ((-0.235657985606+0.577066364011j))*x[1]**o + ((-0.424487693893-0.621747713486j))*x[1] + ((0.115929915408+0.209537803667j))*x[2]**o + ((-0.0607768205906+0.821036913814j))*x[2]
            ref[(0, 1)]=(-0.206453773379+0.651342443166j)-((0.124032991176+0.297418712845j))*(o-1.)
            arg[(0, 2)]=(-0.377947552273+0.169883136894j)*x[0]**o + ((0.646804786576-0.82937133988j))*x[0] + ((-0.514123079556-0.0570664451778j))*x[1]**o + ((-0.167273131432+0.418236255964j))*x[1] + ((-0.964923400887+0.925613472262j))*x[2]**o + ((0.00351776941109-0.337779916721j))*x[2]
            ref[(0, 2)]=(-0.68697230408+0.14475758167j)-((-0.309499005453+0.173071693996j))*(o-1.)
            arg[(1, 0)]=(0.720537046342-0.738518895229j)*x[0]**o + ((-0.113963049554+0.743689828266j))*x[0] + ((0.110864786424+0.444530040771j))*x[1]**o + ((-0.907702712398-0.487785273951j))*x[1] + ((-0.09258252693-0.774238303798j))*x[2]**o + ((0.42562414211+0.484356181521j))*x[2]
            ref[(1, 0)]=(0.0713888429972-0.16398321121j)-((0.123136550973-0.178037859709j))*(o-1.)
            arg[(1, 1)]=(0.386687920598-0.893972101717j)*x[0]**o + ((0.237607262031+0.11628634848j))*x[0] + ((0.323099968623-0.692013538878j))*x[1]**o + ((-0.229868594418+0.41081754757j))*x[1] + ((-0.743579771667-0.235361584023j))*x[2]**o + ((0.195070598859-0.408600413137j))*x[2]
            ref[(1, 1)]=(0.0845086920124-0.851421870852j)-((-0.00563198040764-0.30355787077j))*(o-1.)
            arg[(1, 2)]=(-0.377636615819-0.18238833577j)*x[0]**o + ((0.268340509175-0.652704416678j))*x[0] + ((-0.644093327352+0.47167794299j))*x[1]**o + ((-0.443587558882+0.957925757809j))*x[1] + ((0.18336567563-0.433369643127j))*x[2]**o + ((-0.187108123816-0.784633767469j))*x[2]
            ref[(1, 2)]=(-0.600359720532-0.311746231123j)-((-0.139727377924-0.0240133393179j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedFunction_rank3(self):
        """
        tests integral of rank 3 Data on ReducedFunction

        assumptions: ReducedFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 4, 4),w)
        ref=numpy.zeros((3, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.617566171644+0.857265193399j)*x[0]**o + ((0.0452554986511+0.561679778463j))*x[0] + ((-0.371622150394-0.805355704088j))*x[1]**o + ((0.947495896359-0.47146140136j))*x[1]
            ref[(0, 0, 0)]=(0.00178153648616+0.0710639332067j)-((-0.16486472034+0.0086515815518j))*(o-1.)
            arg[(0, 0, 1)]=(0.724164060143-0.383584636796j)*x[0]**o + ((0.653480343246+0.951527522742j))*x[0] + ((-0.618776066869-0.860057805281j))*x[1]**o + ((-0.956367869729-0.686777724044j))*x[1]
            ref[(0, 0, 1)]=(-0.0987497666048-0.489446321689j)-((0.0175646655456-0.207273740346j))*(o-1.)
            arg[(0, 0, 2)]=(0.643156418697+0.793770836209j)*x[0]**o + ((-0.616599097751-0.824910399213j))*x[0] + ((-0.327756028456-0.892610126463j))*x[1]**o + ((0.556757162984-0.325471317773j))*x[1]
            ref[(0, 0, 2)]=(0.127779227737-0.62461050362j)-((0.0525667317068-0.0164732150423j))*(o-1.)
            arg[(0, 0, 3)]=(-0.633926745362-0.520361120157j)*x[0]**o + ((0.0684460910085-0.254600713338j))*x[0] + ((-0.423274368508+0.161495433597j))*x[1]**o + ((-0.231989723841-0.705249574449j))*x[1]
            ref[(0, 0, 3)]=(-0.610372373351-0.659357987173j)-((-0.176200185645-0.0598109477601j))*(o-1.)
            arg[(0, 1, 0)]=(0.963860747897-0.245598704644j)*x[0]**o + ((-0.439168365279+0.171407730669j))*x[0] + ((0.771359429232-0.144132879079j))*x[1]**o + ((-0.321293002063-0.973361912982j))*x[1]
            ref[(0, 1, 0)]=(0.487379404894-0.595842883018j)-((0.289203362855-0.0649552639539j))*(o-1.)
            arg[(0, 1, 1)]=(-0.118411980392-0.174836437148j)*x[0]**o + ((0.550243243901+0.671511359521j))*x[0] + ((0.271610973703+0.979107143289j))*x[1]**o + ((-0.910637160328-0.146836924613j))*x[1]
            ref[(0, 1, 1)]=(-0.103597461558+0.664472570524j)-((0.0255331655518+0.13404511769j))*(o-1.)
            arg[(0, 1, 2)]=(-0.360809661849+0.0529177752635j)*x[0]**o + ((-0.0192466387762+0.684267281266j))*x[0] + ((0.388807105754-0.850220211669j))*x[1]**o + ((0.737661794853+0.0111044663485j))*x[1]
            ref[(0, 1, 2)]=(0.373206299991-0.0509653443952j)-((0.00466624065092-0.132883739401j))*(o-1.)
            arg[(0, 1, 3)]=(-0.520177093427-0.0352399477103j)*x[0]**o + ((-0.216835817699-0.25899348692j))*x[0] + ((0.221973385705-0.365144523133j))*x[1]**o + ((0.581721231586+0.386766500234j))*x[1]
            ref[(0, 1, 3)]=(0.0333408530827-0.136305728765j)-((-0.0497006179536-0.0667307451406j))*(o-1.)
            arg[(0, 2, 0)]=(-0.870831206583+0.460967317797j)*x[0]**o + ((0.904287305443-0.563329594358j))*x[0] + ((-0.858758973757-0.94351011666j))*x[1]**o + ((-0.650032233758+0.443459644378j))*x[1]
            ref[(0, 2, 0)]=(-0.737667554328-0.301206374422j)-((-0.288265030057-0.0804237998106j))*(o-1.)
            arg[(0, 2, 1)]=(0.617315447759+0.503204387916j)*x[0]**o + ((-0.384334845401+0.123192597357j))*x[0] + ((-0.567376296856-0.384117180888j))*x[1]**o + ((-0.330785317424+0.824382276256j))*x[1]
            ref[(0, 2, 1)]=(-0.332590505961+0.53333104032j)-((0.00832319181713+0.019847867838j))*(o-1.)
            arg[(0, 2, 2)]=(0.162732312577+0.792185554362j)*x[0]**o + ((-0.339276113929-0.35430974936j))*x[0] + ((0.141720209331-0.926761267784j))*x[1]**o + ((-0.783328088686+0.058401783881j))*x[1]
            ref[(0, 2, 2)]=(-0.409075840354-0.21524183945j)-((0.0507420869848-0.0224292855703j))*(o-1.)
            arg[(0, 2, 3)]=(-0.804578597463-0.180308889704j)*x[0]**o + ((0.13953790755-0.716761267762j))*x[0] + ((-0.109760551594-0.293916614299j))*x[1]**o + ((-0.13119693004+0.162819453281j))*x[1]
            ref[(0, 2, 3)]=(-0.452999085773-0.514083659242j)-((-0.152389858176-0.0790375840004j))*(o-1.)
            arg[(0, 3, 0)]=(-0.754270306783-0.940807918027j)*x[0]**o + ((0.483301450935+0.228322753941j))*x[0] + ((-0.273732958531+0.800812874785j))*x[1]**o + ((-0.0950938086825-0.54420864753j))*x[1]
            ref[(0, 3, 0)]=(-0.319897811531-0.227940468416j)-((-0.171333877552-0.0233325072071j))*(o-1.)
            arg[(0, 3, 1)]=(0.939523941462-0.239663989805j)*x[0]**o + ((0.573616981796-0.174873198003j))*x[0] + ((-0.811007836761+0.83737452228j))*x[1]**o + ((0.055417119866-0.802242138774j))*x[1]
            ref[(0, 3, 1)]=(0.378775103182-0.189702402151j)-((0.0214193507836+0.0996184220792j))*(o-1.)
            arg[(0, 3, 2)]=(0.105643427594+0.0362323747067j)*x[0]**o + ((-0.942966875847-0.640938488586j))*x[0] + ((-0.217526660418-0.177024688721j))*x[1]**o + ((0.951483038401+0.830467524044j))*x[1]
            ref[(0, 3, 2)]=(-0.0516835351344+0.0243683607218j)-((-0.0186472054706-0.0234653856691j))*(o-1.)
            arg[(0, 3, 3)]=(0.6858782692+0.896165731557j)*x[0]**o + ((0.817318952277+0.417673992258j))*x[0] + ((-0.375936247785+0.584143242654j))*x[1]**o + ((0.589834857137-0.915169966328j))*x[1]
            ref[(0, 3, 3)]=(0.858547915414+0.491406500071j)-((0.0516570035691+0.246718162369j))*(o-1.)
            arg[(1, 0, 0)]=(-0.370537437397-0.609273025364j)*x[0]**o + ((-0.311299893025+0.131830616028j))*x[0] + ((-0.423154895565-0.439868721636j))*x[1]**o + ((-0.719072211869+0.389859976545j))*x[1]
            ref[(1, 0, 0)]=(-0.912032218928-0.263725577213j)-((-0.132282055494-0.174856957833j))*(o-1.)
            arg[(1, 0, 1)]=(0.853289738839+0.177861066274j)*x[0]**o + ((0.169549438078+0.17395550346j))*x[0] + ((-0.440700284954+0.179780255938j))*x[1]**o + ((-0.268100467376+0.840373523397j))*x[1]
            ref[(1, 0, 1)]=(0.157019212293+0.685985174534j)-((0.0687649089808+0.0596068870354j))*(o-1.)
            arg[(1, 0, 2)]=(0.513718356055+0.74047932716j)*x[0]**o + ((-0.0741163646413+0.676285748712j))*x[0] + ((0.00748075020255-0.133073846235j))*x[1]**o + ((-0.394288314161-0.493129309966j))*x[1]
            ref[(1, 0, 2)]=(0.0263972137279+0.395280959835j)-((0.0868665177096+0.101234246821j))*(o-1.)
            arg[(1, 0, 3)]=(-0.476581203243+0.797868768599j)*x[0]**o + ((-0.292093680264-0.405915038667j))*x[0] + ((-0.746764728562-0.390051024825j))*x[1]**o + ((-0.0162039205245-0.41047380796j))*x[1]
            ref[(1, 0, 3)]=(-0.765821766296-0.204285551427j)-((-0.203890988634+0.0679696239623j))*(o-1.)
            arg[(1, 1, 0)]=(-0.156953192671-0.803299773739j)*x[0]**o + ((0.19319297687+0.633139536318j))*x[0] + ((0.753083375856-0.882714179894j))*x[1]**o + ((-0.00482021797909+0.811849824539j))*x[1]
            ref[(1, 1, 0)]=(0.392251471038-0.120512296388j)-((0.0993550305308-0.281002325606j))*(o-1.)
            arg[(1, 1, 1)]=(0.516422267773-0.202020799368j)*x[0]**o + ((-0.607103175966-0.818372041654j))*x[0] + ((-0.589119303559+0.983092852419j))*x[1]**o + ((0.651870114066+0.15016148036j))*x[1]
            ref[(1, 1, 1)]=(-0.0139650488424+0.0564307458788j)-((-0.0121161726309+0.130178675509j))*(o-1.)
            arg[(1, 1, 2)]=(-0.877909511016-0.311146963835j)*x[0]**o + ((0.0591971651885-0.645607779322j))*x[0] + ((0.117351264641-0.947852560928j))*x[1]**o + ((0.872082238786-0.499412957881j))*x[1]
            ref[(1, 1, 2)]=(0.0853605788003-1.20201013098j)-((-0.126759707729-0.209833254127j))*(o-1.)
            arg[(1, 1, 3)]=(-0.888875307498-0.850479620096j)*x[0]**o + ((-0.364092367065-0.058028525095j))*x[0] + ((0.695020068127-0.169578077355j))*x[1]**o + ((-0.273452559053-0.265623820453j))*x[1]
            ref[(1, 1, 3)]=(-0.415700082745-0.6718550215j)-((-0.0323092065619-0.170009616242j))*(o-1.)
            arg[(1, 2, 0)]=(0.588437894153-0.957430883731j)*x[0]**o + ((-0.209742245169+0.431791096699j))*x[0] + ((0.471202069944+0.97783541995j))*x[1]**o + ((0.981694870415-0.883985283551j))*x[1]
            ref[(1, 2, 0)]=(0.915796294672-0.215894825317j)-((0.176606660683+0.00340075603653j))*(o-1.)
            arg[(1, 2, 1)]=(0.190055284066+0.357062172902j)*x[0]**o + ((0.355878764455-0.547129595708j))*x[0] + ((-0.605617592437+0.486956544646j))*x[1]**o + ((-0.983379176802-0.065628064009j))*x[1]
            ref[(1, 2, 1)]=(-0.52153136036+0.115630528915j)-((-0.0692603847286+0.140669786258j))*(o-1.)
            arg[(1, 2, 2)]=(0.90178943421+0.0324383903254j)*x[0]**o + ((-0.757602226157+0.737000621504j))*x[0] + ((0.439587757981-0.783857024771j))*x[1]**o + ((0.440149518979+0.862072371853j))*x[1]
            ref[(1, 2, 2)]=(0.511962242507+0.423827179455j)-((0.223562865365-0.125236439074j))*(o-1.)
            arg[(1, 2, 3)]=(0.865921243012+0.0145920653013j)*x[0]**o + ((0.543000665099-0.720853681624j))*x[0] + ((-0.184162869601-0.630130929156j))*x[1]**o + ((-0.240386014299+0.889115645597j))*x[1]
            ref[(1, 2, 3)]=(0.492186512105-0.223638449941j)-((0.113626395568-0.102589810642j))*(o-1.)
            arg[(1, 3, 0)]=(-0.0973000408993-0.596885330028j)*x[0]**o + ((-0.833186356811-0.305747188142j))*x[0] + ((0.0878155240385+0.839648881041j))*x[1]**o + ((-0.884516402512-0.761979130959j))*x[1]
            ref[(1, 3, 0)]=(-0.863593638092-0.412481384044j)-((-0.00158075281014+0.0404605918355j))*(o-1.)
            arg[(1, 3, 1)]=(0.0239718909385+0.195051371759j)*x[0]**o + ((-0.877365046595-0.0234393309316j))*x[0] + ((-0.350095265929+0.301137461932j))*x[1]**o + ((0.549459297325-0.281642703022j))*x[1]
            ref[(1, 3, 1)]=(-0.32701456213+0.0955533998688j)-((-0.0543538958318+0.0826981389485j))*(o-1.)
            arg[(1, 3, 2)]=(0.706695115316-0.392749733301j)*x[0]**o + ((-0.496731523052+0.055008809175j))*x[0] + ((0.190963696999-0.976143437589j))*x[1]**o + ((0.811163617568-0.483968848319j))*x[1]
            ref[(1, 3, 2)]=(0.606045453415-0.898926605017j)-((0.149609802052-0.228148861815j))*(o-1.)
            arg[(1, 3, 3)]=(-0.622906619889-0.810645662544j)*x[0]**o + ((0.841950673531+0.0795133616277j))*x[0] + ((-0.358479442658+0.775154762962j))*x[1]**o + ((0.750370945916-0.216397763466j))*x[1]
            ref[(1, 3, 3)]=(0.30546777845-0.08618765071j)-((-0.163564343758-0.00591514993032j))*(o-1.)
            arg[(2, 0, 0)]=(-0.364492573277-0.911643754773j)*x[0]**o + ((0.910314840824+0.933683370287j))*x[0] + ((-0.488311639697+0.665359098309j))*x[1]**o + ((-0.504549755393+0.553172869087j))*x[1]
            ref[(2, 0, 0)]=(-0.223519563771+0.620285791455j)-((-0.142134035496-0.0410474427441j))*(o-1.)
            arg[(2, 0, 1)]=(0.975676569754+0.187237773268j)*x[0]**o + ((-0.693552948666-0.89179612014j))*x[0] + ((-0.314899674146-0.901537461588j))*x[1]**o + ((0.0934043768441+0.868684738661j))*x[1]
            ref[(2, 0, 1)]=(0.030314161893-0.368705534899j)-((0.110129482601-0.119049948053j))*(o-1.)
            arg[(2, 0, 2)]=(-0.9251557541-0.511116692822j)*x[0]**o + ((0.904676312481+0.0783834942898j))*x[0] + ((0.262074096217-0.0191270597963j))*x[1]**o + ((0.421394264219+0.797900201046j))*x[1]
            ref[(2, 0, 2)]=(0.331494459409+0.173019971359j)-((-0.110513609647-0.0883739587696j))*(o-1.)
            arg[(2, 0, 3)]=(0.470270402933+0.842034353192j)*x[0]**o + ((0.726919639687-0.38109209363j))*x[0] + ((-0.979637904637-0.857830185673j))*x[1]**o + ((0.860494680437+0.076755904422j))*x[1]
            ref[(2, 0, 3)]=(0.53902340921-0.160066010844j)-((-0.0848945836173-0.00263263874689j))*(o-1.)
            arg[(2, 1, 0)]=(-0.222721567543+0.128627612998j)*x[0]**o + ((-0.490591279862-0.849443638219j))*x[0] + ((-0.784114213046-0.216051454352j))*x[1]**o + ((0.29262347433+0.945044571649j))*x[1]
            ref[(2, 1, 0)]=(-0.60240179306+0.00408854603784j)-((-0.167805963431-0.0145706402258j))*(o-1.)
            arg[(2, 1, 1)]=(0.284931578955+0.522558936025j)*x[0]**o + ((-0.100150402809+0.834346922996j))*x[0] + ((-0.65462478427+0.0506709818816j))*x[1]**o + ((-0.475825854375+0.515852926117j))*x[1]
            ref[(2, 1, 1)]=(-0.472834731249+0.96171488351j)-((-0.0616155342191+0.0955383196511j))*(o-1.)
            arg[(2, 1, 2)]=(-0.517996585737+0.222086306219j)*x[0]**o + ((-0.952355053997-0.932205780222j))*x[0] + ((-0.125649768695+0.664171478504j))*x[1]**o + ((-0.789650248345+0.591823791938j))*x[1]
            ref[(2, 1, 2)]=(-1.19282582839+0.272937898219j)-((-0.107274392405+0.147709630787j))*(o-1.)
            arg[(2, 1, 3)]=(0.314891059088-0.832933159841j)*x[0]**o + ((-0.427077017659-0.753743111239j))*x[0] + ((0.399856760249+0.0796759524395j))*x[1]**o + ((-0.43048964344+0.802120446036j))*x[1]
            ref[(2, 1, 3)]=(-0.0714094208813-0.352439936302j)-((0.119124636556-0.1255428679j))*(o-1.)
            arg[(2, 2, 0)]=(-0.183916883787-0.526141771042j)*x[0]**o + ((-0.772541829042+0.431564380113j))*x[0] + ((0.20285070269+0.687054928121j))*x[1]**o + ((0.900442676949+0.178930314721j))*x[1]
            ref[(2, 2, 0)]=(0.0734173334049+0.385703925956j)-((0.00315563648389+0.0268188595132j))*(o-1.)
            arg[(2, 2, 1)]=(-0.222559137703+0.984093734511j)*x[0]**o + ((0.862471524598-0.0578188821857j))*x[0] + ((0.725158060143+0.890276133485j))*x[1]**o + ((-0.224084022331-0.0932271951332j))*x[1]
            ref[(2, 2, 1)]=(0.570493212353+0.861661895338j)-((0.0837664870734+0.312394977999j))*(o-1.)
            arg[(2, 2, 2)]=(-0.828202015343-0.246350095619j)*x[0]**o + ((-0.974189306869+0.866966389318j))*x[0] + ((0.342103091786+0.507655349888j))*x[1]**o + ((0.858522671349+0.234474325352j))*x[1]
            ref[(2, 2, 2)]=(-0.300882779539+0.68137298447j)-((-0.0810164872595+0.0435508757116j))*(o-1.)
            arg[(2, 2, 3)]=(-0.271774417846-0.682575811604j)*x[0]**o + ((-0.613204522306+0.61165946373j))*x[0] + ((-0.204070432266-0.0736980187106j))*x[1]**o + ((0.723742376632+0.286018688641j))*x[1]
            ref[(2, 2, 3)]=(-0.182653497893+0.070702161028j)-((-0.0793074750187-0.126045638386j))*(o-1.)
            arg[(2, 3, 0)]=(0.269884840408-0.728951347868j)*x[0]**o + ((0.689487227994+0.183114460613j))*x[0] + ((-0.442948442887+0.275348504729j))*x[1]**o + ((0.120796314048+0.0980359102543j))*x[1]
            ref[(2, 3, 0)]=(0.318609969782-0.086226236136j)-((-0.0288439337464-0.0756004738566j))*(o-1.)
            arg[(2, 3, 1)]=(-0.711343770448-0.486697090599j)*x[0]**o + ((0.850205403154-0.427558276122j))*x[0] + ((-0.586283012943-0.365900701663j))*x[1]**o + ((-0.677242148916-0.0860798564949j))*x[1]
            ref[(2, 3, 1)]=(-0.562331764576-0.68311796244j)-((-0.216271130565-0.142099632044j))*(o-1.)
            arg[(2, 3, 2)]=(0.959012120536+0.427211740951j)*x[0]**o + ((0.193209440132+0.537135484684j))*x[0] + ((-0.69897140691-0.0404403779077j))*x[1]**o + ((0.356617407887+0.498858791229j))*x[1]
            ref[(2, 3, 2)]=(0.404933780822+0.711382819478j)-((0.0433401189375+0.0644618938406j))*(o-1.)
            arg[(2, 3, 3)]=(0.710300741142+0.208720052037j)*x[0]**o + ((0.148387941209+0.723573882271j))*x[0] + ((0.736655970977+0.37736827817j))*x[1]**o + ((-0.00418677176904+0.429836798059j))*x[1]
            ref[(2, 3, 3)]=(0.795578940779+0.869749505268j)-((0.24115945202+0.0976813883678j))*(o-1.)
        else:
            arg[(0, 0, 0)]=(0.812498101869+0.642088211909j)*x[0]**o + ((0.202872570675+0.376754438546j))*x[0] + ((0.222004499978+0.479424204869j))*x[1]**o + ((0.452858777906-0.521706412231j))*x[1] + ((0.472536712261-0.237583238783j))*x[2]**o + ((-0.218253347566-0.799947485339j))*x[2]
            ref[(0, 0, 0)]=(0.972258657562-0.0304851405146j)-((0.251173219018+0.147321529666j))*(o-1.)
            arg[(0, 0, 1)]=(-0.0775800063129-0.315928899118j)*x[0]**o + ((-0.150409801981-0.513677058572j))*x[0] + ((-0.383305735054+0.740446638401j))*x[1]**o + ((-0.555833314887-0.0776887245194j))*x[1] + ((-0.719848111951+0.268081236701j))*x[2]**o + ((0.429278480651+0.766642693835j))*x[2]
            ref[(0, 0, 1)]=(-0.728849244768+0.433937943364j)-((-0.196788975553+0.115433162664j))*(o-1.)
            arg[(0, 0, 2)]=(0.514505402833+0.567192717364j)*x[0]**o + ((-0.285481067633-0.372616662835j))*x[0] + ((0.418037615086+0.720564183868j))*x[1]**o + ((0.218821683061-0.771613021052j))*x[1] + ((-0.00343647318822-0.538862885743j))*x[2]**o + ((0.0299441046549-0.365299091546j))*x[2]
            ref[(0, 0, 2)]=(0.446195632407-0.380317379972j)-((0.154851090788+0.124815669248j))*(o-1.)
            arg[(0, 0, 3)]=(0.545981004973-0.161544441659j)*x[0]**o + ((0.700052930566-0.905364772576j))*x[0] + ((-0.268079030208-0.859905570521j))*x[1]**o + ((-0.405970543264-0.502731623093j))*x[1] + ((0.640323412255-0.256729559476j))*x[2]**o + ((-0.451304696999+0.382603311181j))*x[2]
            ref[(0, 0, 3)]=(0.380501538661-1.15183632807j)-((0.153037564503-0.213029928609j))*(o-1.)
            arg[(0, 1, 0)]=(-0.3572455587+0.753370926192j)*x[0]**o + ((-0.918365874573-0.65820594839j))*x[0] + ((-0.309543963841+0.0365113074996j))*x[1]**o + ((-0.960455594225-0.721777305213j))*x[1] + ((-0.0832943378041+0.885254204536j))*x[2]**o + ((-0.712204215525+0.806331813601j))*x[2]
            ref[(0, 1, 0)]=(-1.67055477233+0.550742499112j)-((-0.125013976724+0.279189406371j))*(o-1.)
            arg[(0, 1, 1)]=(0.609477189242-0.70128471766j)*x[0]**o + ((0.709514916761-0.831380748471j))*x[0] + ((-0.0382539886963-0.0980884257923j))*x[1]**o + ((0.949016666481+0.419839852806j))*x[1] + ((-0.693046814427-0.255009978942j))*x[2]**o + ((-0.293341134556+0.956345310194j))*x[2]
            ref[(0, 1, 1)]=(0.621683417403-0.254789353933j)-((-0.0203039356468-0.175730520399j))*(o-1.)
            arg[(0, 1, 2)]=(0.533426042413+0.120212955046j)*x[0]**o + ((-0.676046747338-0.842243902826j))*x[0] + ((0.557767518416-0.48707688686j))*x[1]**o + ((-0.0947746780368-0.149199921509j))*x[1] + ((-0.505729110803+0.754842966356j))*x[2]**o + ((0.552866630692-0.575159773945j))*x[2]
            ref[(0, 1, 2)]=(0.183754827671-0.589312281869j)-((0.0975774083375+0.0646631724236j))*(o-1.)
            arg[(0, 1, 3)]=(-0.735409901458+0.131262581262j)*x[0]**o + ((0.977808502141-0.108833823987j))*x[0] + ((0.462441264773+0.0902319979545j))*x[1]**o + ((0.928110263665+0.198271004087j))*x[1] + ((-0.719228425931-0.288952845746j))*x[2]**o + ((0.0532910517923+0.538317034635j))*x[2]
            ref[(0, 1, 3)]=(0.483506377491+0.280147974103j)-((-0.165366177103-0.0112430444216j))*(o-1.)
            arg[(0, 2, 0)]=(0.636489505686+0.595295277794j)*x[0]**o + ((0.163680651128+0.00555113993306j))*x[0] + ((-0.0598252791875-0.419491260753j))*x[1]**o + ((0.265777029462-0.00721737438653j))*x[1] + ((0.971732147132+0.97452853593j))*x[2]**o + ((-0.178822060195-0.997678675414j))*x[2]
            ref[(0, 2, 0)]=(0.899515997013+0.0754938215516j)-((0.258066062272+0.191722092162j))*(o-1.)
            arg[(0, 2, 1)]=(-0.782797706046-0.873422223962j)*x[0]**o + ((-0.080335744768+0.166635703912j))*x[0] + ((-0.568357918653+0.933455617166j))*x[1]**o + ((0.0169018884085-0.414589701864j))*x[1] + ((-0.177491368097-0.478030278796j))*x[2]**o + ((-0.494515446986-0.773546705071j))*x[2]
            ref[(0, 2, 1)]=(-1.04329814807-0.719748794307j)-((-0.254774498799-0.0696661475987j))*(o-1.)
            arg[(0, 2, 2)]=(-0.760651579794+0.199305571558j)*x[0]**o + ((0.539479364128-0.60890871985j))*x[0] + ((-0.364992372156+0.0626762127684j))*x[1]**o + ((-0.679137387062-0.984738646178j))*x[1] + ((0.099357457009-0.582781798354j))*x[2]**o + ((0.687274089143-0.791839686131j))*x[2]
            ref[(0, 2, 2)]=(-0.239335214366-1.35314353309j)-((-0.171047749157-0.0534666690046j))*(o-1.)
            arg[(0, 2, 3)]=(-0.882958643877+0.857579994025j)*x[0]**o + ((0.499205268456+0.435656758365j))*x[0] + ((-0.834107792956-0.909610723253j))*x[1]**o + ((0.0715163968482+0.947441508098j))*x[1] + ((-0.302267174225-0.168364199331j))*x[2]**o + ((0.462479147894-0.603110983996j))*x[2]
            ref[(0, 2, 3)]=(-0.49306639893+0.279796176954j)-((-0.336555601843-0.0367324880931j))*(o-1.)
            arg[(0, 3, 0)]=(-0.986832825336-0.250681479128j)*x[0]**o + ((-0.788479344342+0.382680258731j))*x[0] + ((-0.640126665328+0.815388810166j))*x[1]**o + ((0.601887321793-0.442368871642j))*x[1] + ((-0.0274779464049-0.930681905528j))*x[2]**o + ((0.0515124227896+0.403122720806j))*x[2]
            ref[(0, 3, 0)]=(-0.894758518414-0.0112702332979j)-((-0.275739572845-0.060995762415j))*(o-1.)
            arg[(0, 3, 1)]=(0.744570368829-0.00710782166778j)*x[0]**o + ((0.628192255746+0.332086239918j))*x[0] + ((-0.892551394746+0.955955370808j))*x[1]**o + ((-0.724742336472-0.708647772419j))*x[1] + ((-0.255926329048-0.932219851172j))*x[2]**o + ((-0.909683832314-0.233569334398j))*x[2]
            ref[(0, 3, 1)]=(-0.705070634003-0.296751584466j)-((-0.0673178924942+0.00277128299469j))*(o-1.)
            arg[(0, 3, 2)]=(-0.851350437437-0.482493258604j)*x[0]**o + ((-0.627961219517+0.956637797809j))*x[0] + ((-0.1717662199+0.869677381367j))*x[1]**o + ((0.771952582543-0.360342632942j))*x[1] + ((-0.825191030498-0.15671585059j))*x[2]**o + ((0.29897521397+0.220414305563j))*x[2]
            ref[(0, 3, 2)]=(-0.70267055542+0.523588871302j)-((-0.308051281306+0.0384113786956j))*(o-1.)
            arg[(0, 3, 3)]=(0.834839044839+0.106946276149j)*x[0]**o + ((0.051828762273-0.50044776021j))*x[0] + ((-0.450559871504-0.807374405261j))*x[1]**o + ((-0.358385996479+0.0478286640062j))*x[1] + ((-0.132469077912-0.903712285174j))*x[2]**o + ((0.685221203923+0.0169810485264j))*x[2]
            ref[(0, 3, 3)]=(0.31523703257-1.01988923098j)-((0.0419683492373-0.267356735714j))*(o-1.)
            arg[(1, 0, 0)]=(0.896542289349-0.594543473383j)*x[0]**o + ((-0.0382738984662-0.532881487176j))*x[0] + ((-0.989519867681+0.158506009454j))*x[1]**o + ((-0.404167288851+0.804029851245j))*x[1] + ((-0.749114482148-0.43506938955j))*x[2]**o + ((-0.962817412093-0.184227266125j))*x[2]
            ref[(1, 0, 0)]=(-1.12367532995-0.392092877767j)-((-0.140348676747-0.14518447558j))*(o-1.)
            arg[(1, 0, 1)]=(-0.219762510285+0.745531627203j)*x[0]**o + ((0.458105054921-0.292583206558j))*x[0] + ((-0.645366633412+0.853255176451j))*x[1]**o + ((0.26079759416+0.782323248287j))*x[1] + ((-0.332068260621+0.777437368887j))*x[2]**o + ((-0.271613641833+0.151471723728j))*x[2]
            ref[(1, 0, 1)]=(-0.374954198535+1.508717969j)-((-0.19953290072+0.39603736209j))*(o-1.)
            arg[(1, 0, 2)]=(-0.574704571323-0.427222564775j)*x[0]**o + ((0.06969086706+0.658671685635j))*x[0] + ((-0.743381239071+0.72939638774j))*x[1]**o + ((0.672350250774+0.595911692634j))*x[1] + ((0.21652056228+0.0957614599517j))*x[2]**o + ((-0.0280950928859-0.546171295678j))*x[2]
            ref[(1, 0, 2)]=(-0.193809611583+0.553173682753j)-((-0.183594208019+0.0663225471527j))*(o-1.)
            arg[(1, 0, 3)]=(-0.958470927248+0.35759770049j)*x[0]**o + ((0.924623856415-0.49390222264j))*x[0] + ((0.0984245145467+0.516779389303j))*x[1]**o + ((-0.551490007138+0.34639928868j))*x[1] + ((0.900552090426+0.682005783666j))*x[2]**o + ((-0.515994143539+0.464434675484j))*x[2]
            ref[(1, 0, 3)]=(-0.0511773082685+0.936657307491j)-((0.00675094628744+0.259397145576j))*(o-1.)
            arg[(1, 1, 0)]=(-0.558111718839-0.0580155554054j)*x[0]**o + ((0.206694252886-0.572586120132j))*x[0] + ((0.429610946865-0.444815908591j))*x[1]**o + ((0.918194901516+0.572346637876j))*x[1] + ((0.57331348909+0.262300620814j))*x[2]**o + ((-0.494329555727+0.0120544067937j))*x[2]
            ref[(1, 1, 0)]=(0.537686157896-0.114357959322j)-((0.0741354528527-0.0400884738637j))*(o-1.)
            arg[(1, 1, 1)]=(0.753143133932-0.715768538109j)*x[0]**o + ((-0.216021991759-0.523703895975j))*x[0] + ((0.27815735682+0.138942670907j))*x[1]**o + ((0.676795314685+0.472954016611j))*x[1] + ((-0.401153706573+0.161087542565j))*x[2]**o + ((0.114470400465-0.971875419534j))*x[2]
            ref[(1, 1, 1)]=(0.602695253785-0.719181811767j)-((0.10502446403-0.0692897207727j))*(o-1.)
            arg[(1, 1, 2)]=(0.989387902214+0.266675563557j)*x[0]**o + ((0.911505961455+0.0465600413975j))*x[0] + ((-0.0549099431536+0.73819994782j))*x[1]**o + ((0.894561808869+0.260795566011j))*x[1] + ((0.553775399111-0.211381490324j))*x[2]**o + ((0.633501612505+0.196632531111j))*x[2]
            ref[(1, 1, 2)]=(1.9639113705+0.648741079787j)-((0.248042226362+0.132249003509j))*(o-1.)
            arg[(1, 1, 3)]=(-0.441845747427-0.285550270474j)*x[0]**o + ((-0.795009844674+0.830706510991j))*x[0] + ((0.489510326729-0.89168445579j))*x[1]**o + ((0.238453860187+0.912131944502j))*x[1] + ((0.65744844206-0.227953333088j))*x[2]**o + ((-0.946148473397-0.781987118714j))*x[2]
            ref[(1, 1, 3)]=(-0.398795718261-0.222168361286j)-((0.117518836894-0.234198009892j))*(o-1.)
            arg[(1, 2, 0)]=(-0.447528876278+0.585158603646j)*x[0]**o + ((0.60575500305+0.619609133263j))*x[0] + ((0.326196956986+0.649744241599j))*x[1]**o + ((-0.944723119305-0.744558897063j))*x[1] + ((0.73675940105-0.417543576025j))*x[2]**o + ((0.0317919707876+0.822211855201j))*x[2]
            ref[(1, 2, 0)]=(0.154125668146+0.75731068031j)-((0.10257124696+0.13622654487j))*(o-1.)
            arg[(1, 2, 1)]=(0.795847637584-0.697121406015j)*x[0]**o + ((-0.887855752167+0.0539779807136j))*x[0] + ((-0.145092413971-0.524764819052j))*x[1]**o + ((0.804139003087+0.606474275874j))*x[1] + ((-0.11337254828-0.86167742827j))*x[2]**o + ((0.589104079759+0.78603196548j))*x[2]
            ref[(1, 2, 1)]=(0.521385003006-0.318539715634j)-((0.0895637792223-0.347260608889j))*(o-1.)
            arg[(1, 2, 2)]=(-0.652821973218+0.141229328091j)*x[0]**o + ((-0.285076507466+0.910143851195j))*x[0] + ((-0.311270398794-0.621733860777j))*x[1]**o + ((-0.159223045361-0.371245678639j))*x[1] + ((0.490187225687+0.687525258744j))*x[2]**o + ((0.445029162271-0.809227407816j))*x[2]
            ref[(1, 2, 2)]=(-0.236587768441-0.0316542546015j)-((-0.0789841910543+0.0345034543429j))*(o-1.)
            arg[(1, 2, 3)]=(-0.0100513467536+0.209299363748j)*x[0]**o + ((-0.743576501092-0.846359160261j))*x[0] + ((-0.675990001815-0.583753058847j))*x[1]**o + ((0.480482014174-0.277410747649j))*x[1] + ((0.819952039007-0.772399992576j))*x[2]**o + ((-0.970388300587-0.906813311686j))*x[2]
            ref[(1, 2, 3)]=(-0.549786048533-1.58871845364j)-((0.0223184484065-0.191142281279j))*(o-1.)
            arg[(1, 3, 0)]=(0.533857164627-0.760928283536j)*x[0]**o + ((0.329275335173-0.424252809664j))*x[0] + ((0.893018348775+0.683569701465j))*x[1]**o + ((0.414291873405+0.491015852348j))*x[1] + ((0.1783042169-0.96024432877j))*x[2]**o + ((0.836417162721+0.998366662274j))*x[2]
            ref[(1, 3, 0)]=(1.5925820508+0.0137633970594j)-((0.26752995505-0.172933818473j))*(o-1.)
            arg[(1, 3, 1)]=(0.775530342929-0.15950164982j)*x[0]**o + ((0.165319572648+0.685796036896j))*x[0] + ((-0.2334763553-0.495028067521j))*x[1]**o + ((-0.166139757816+0.247267296541j))*x[1] + ((0.624453880667-0.375070518521j))*x[2]**o + ((0.939238772767+0.851241542937j))*x[2]
            ref[(1, 3, 1)]=(1.05246322795+0.377352320256j)-((0.194417978049-0.17160003931j))*(o-1.)
            arg[(1, 3, 2)]=(-0.379145377524-0.769615631397j)*x[0]**o + ((0.363321933478-0.529037977935j))*x[0] + ((-0.0313814753577-0.228813369706j))*x[1]**o + ((-0.287885273239-0.4059431154j))*x[1] + ((0.242370099205+0.806756919365j))*x[2]**o + ((0.0731596460812+0.729015223525j))*x[2]
            ref[(1, 3, 2)]=(-0.00978022367821-0.198818975773j)-((-0.0280261256128-0.0319453469562j))*(o-1.)
            arg[(1, 3, 3)]=(-0.5307636617-0.364683167685j)*x[0]**o + ((0.0226701606383+0.972062091974j))*x[0] + ((0.703886293505-0.228624778731j))*x[1]**o + ((-0.463676947387+0.227332536109j))*x[1] + ((0.0865732542489+0.706453490762j))*x[2]**o + ((0.904497017394-0.709997825203j))*x[2]
            ref[(1, 3, 3)]=(0.36159305835+0.301271173613j)-((0.0432826476758+0.0188575907243j))*(o-1.)
            arg[(2, 0, 0)]=(-0.507956644655+0.184106696555j)*x[0]**o + ((-0.754684110962+0.768089470887j))*x[0] + ((-0.0087326736971-0.804678859263j))*x[1]**o + ((0.0849206732403+0.150808924113j))*x[1] + ((-0.274658991779-0.505691439823j))*x[2]**o + ((0.436111089271+0.442445464767j))*x[2]
            ref[(2, 0, 0)]=(-0.512500329291+0.117540128619j)-((-0.131891385022-0.187710600422j))*(o-1.)
            arg[(2, 0, 1)]=(0.813582701504-0.284706803456j)*x[0]**o + ((-0.263158574151+0.991768350131j))*x[0] + ((-0.778336575467+0.728069963221j))*x[1]**o + ((0.454768929169+0.846065108964j))*x[1] + ((0.304102521531+0.199549984516j))*x[2]**o + ((0.647252900643+0.43069597321j))*x[2]
            ref[(2, 0, 1)]=(0.589105951615+1.45572128829j)-((0.0565581079281+0.107152190714j))*(o-1.)
            arg[(2, 0, 2)]=(0.291746088183+0.0438494542793j)*x[0]**o + ((-0.648806096453+0.851144991908j))*x[0] + ((-0.865802902513+0.682763790642j))*x[1]**o + ((0.382511644369-0.508578028524j))*x[1] + ((-0.747836034757-0.118587959248j))*x[2]**o + ((0.576556861649+0.834677091761j))*x[2]
            ref[(2, 0, 2)]=(-0.505815219761+0.89263467041j)-((-0.220315474848+0.101337547612j))*(o-1.)
            arg[(2, 0, 3)]=(-0.838567493741+0.428418107048j)*x[0]**o + ((-0.817687206628+0.0938958714527j))*x[0] + ((0.509215918356-0.599771425958j))*x[1]**o + ((0.886246620795-0.96029641662j))*x[1] + ((0.332772681828-0.336874447022j))*x[2]**o + ((0.497514722338+0.339152257164j))*x[2]
            ref[(2, 0, 3)]=(0.284747621474-0.517738026967j)-((0.000570184407177-0.0847046276552j))*(o-1.)
            arg[(2, 1, 0)]=(0.0941124187278+0.532269900179j)*x[0]**o + ((0.641638147507+0.815325428961j))*x[0] + ((0.313661608719-0.592186377165j))*x[1]**o + ((-0.345659277837-0.549432167823j))*x[1] + ((-0.983430947022+0.808555383258j))*x[2]**o + ((0.320389544365+0.902532427444j))*x[2]
            ref[(2, 1, 0)]=(0.0203557472305+0.958532297427j)-((-0.0959428199292+0.124773151045j))*(o-1.)
            arg[(2, 1, 1)]=(0.972962146553+0.681635068792j)*x[0]**o + ((0.739596778684-0.608999067173j))*x[0] + ((-0.656608224397+0.456832662563j))*x[1]**o + ((0.218965183058-0.395384012265j))*x[1] + ((-0.670512573264+0.887168764372j))*x[2]**o + ((0.0240848375476-0.369990235922j))*x[2]
            ref[(2, 1, 1)]=(0.31424407409+0.325631590183j)-((-0.0590264418514+0.337606082621j))*(o-1.)
            arg[(2, 1, 2)]=(0.439508581093+0.737412977587j)*x[0]**o + ((0.46906984772-0.266226825484j))*x[0] + ((-0.20132293965-0.711835707607j))*x[1]**o + ((-0.658186314268-0.016447986131j))*x[1] + ((-0.17442095913+0.276682245932j))*x[2]**o + ((0.514649353802+0.765581723364j))*x[2]
            ref[(2, 1, 2)]=(0.194648784784+0.39258321383j)-((0.0106274470521+0.0503765859853j))*(o-1.)
            arg[(2, 1, 3)]=(0.71230275989+0.421141228629j)*x[0]**o + ((0.485598884342+0.752853511029j))*x[0] + ((0.0932276702322+0.21366932775j))*x[1]**o + ((-0.239908516046-0.435204410948j))*x[1] + ((-0.364202691338+0.748178271692j))*x[2]**o + ((0.219610090346+0.954033619891j))*x[2]
            ref[(2, 1, 3)]=(0.453314098713+1.32733577402j)-((0.0735546231308+0.230498138012j))*(o-1.)
            arg[(2, 2, 0)]=(0.860140265041-0.490266214821j)*x[0]**o + ((0.747635048308+0.401130013293j))*x[0] + ((-0.695128375426-0.842237128166j))*x[1]**o + ((-0.13079969829+0.626128884152j))*x[1] + ((-0.36468138828+0.290510598032j))*x[2]**o + ((0.827338533396-0.959153500611j))*x[2]
            ref[(2, 2, 0)]=(0.622252192375-0.486943674061j)-((-0.0332782497775-0.173665457493j))*(o-1.)
            arg[(2, 2, 1)]=(-0.501784966513-0.121345017501j)*x[0]**o + ((-0.0508181340529+0.0903787082719j))*x[0] + ((0.245636229889-0.835962715238j))*x[1]**o + ((-0.438548477322-0.0662924030949j))*x[1] + ((-0.316425142415-0.0256104498681j))*x[2]**o + ((0.937235719828+0.0460390321693j))*x[2]
            ref[(2, 2, 1)]=(-0.0623523852931-0.456396422631j)-((-0.0954289798399-0.163819697101j))*(o-1.)
            arg[(2, 2, 2)]=(-0.0600342507909+0.943697586908j)*x[0]**o + ((0.887150279214+0.0408863874344j))*x[0] + ((0.110153064761+0.905971073955j))*x[1]**o + ((0.148010089704+0.613893509147j))*x[1] + ((-0.433819116127-0.0485602051945j))*x[2]**o + ((-0.376736678012+0.191089438671j))*x[2]
            ref[(2, 2, 2)]=(0.137361694375+1.32348889546j)-((-0.0639500503595+0.300184742611j))*(o-1.)
            arg[(2, 2, 3)]=(-0.312009125132-0.73787481907j)*x[0]**o + ((0.75222551705+0.0139662029558j))*x[0] + ((-0.602688437723-0.651889416145j))*x[1]**o + ((0.832035594259+0.773130988953j))*x[1] + ((0.437834335157-0.763717054395j))*x[2]**o + ((0.238118989614+0.832028033025j))*x[2]
            ref[(2, 2, 3)]=(0.672758436612-0.267178032338j)-((-0.0794772046164-0.358913548268j))*(o-1.)
            arg[(2, 3, 0)]=(0.159544364063-0.772859653456j)*x[0]**o + ((0.45208001732-0.978529245996j))*x[0] + ((-0.129461207644-0.554381332008j))*x[1]**o + ((0.89063097596-0.451822581462j))*x[1] + ((-0.0803274550815-0.0180295187435j))*x[2]**o + ((-0.912414880056-0.239613804983j))*x[2]
            ref[(2, 3, 0)]=(0.190025907281-1.50761806832j)-((-0.00837404977705-0.224211750701j))*(o-1.)
            arg[(2, 3, 1)]=(-0.619173212548-0.627446117108j)*x[0]**o + ((0.410898189459+0.0663515322575j))*x[0] + ((0.30508127752-0.979246563342j))*x[1]**o + ((0.563634921145+0.741295989133j))*x[1] + ((-0.172735391141+0.297977253628j))*x[2]**o + ((0.602018756856+0.493286271629j))*x[2]
            ref[(2, 3, 1)]=(0.544862270645-0.00389081690215j)-((-0.081137887695-0.218119237804j))*(o-1.)
            arg[(2, 3, 2)]=(-0.979974457774+0.0636350060915j)*x[0]**o + ((0.668255530622+0.000623095824008j))*x[0] + ((-0.0757999562731+0.930306954125j))*x[1]**o + ((-0.514344682638-0.597692802483j))*x[1] + ((0.468455213754+0.301032048084j))*x[2]**o + ((0.074786630879+0.564666491014j))*x[2]
            ref[(2, 3, 2)]=(-0.179310860714+0.631285396328j)-((-0.0978865333821+0.215829001384j))*(o-1.)
            arg[(2, 3, 3)]=(-0.178156281475+0.589769501813j)*x[0]**o + ((-0.653448885487+0.042646652558j))*x[0] + ((-0.500428843146-0.731859914463j))*x[1]**o + ((0.851792171403+0.39419046052j))*x[1] + ((-0.305683400103-0.222203782383j))*x[2]**o + ((-0.357969809476+0.089496443892j))*x[2]
            ref[(2, 3, 3)]=(-0.571947524142+0.0810196809684j)-((-0.164044754121-0.0607156991722j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onReducedFunction_fromData_ReducedFunction_rank4(self):
        """
        tests integral of rank 4 Data on ReducedFunction

        assumptions: ReducedFunction(self.domain) exists
                     self.domain supports integral on ReducedFunction
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = ReducedFunction(self.domain)
        w = ReducedFunction(self.domain)
        x = w.getX()
        arg = Data(0,(3, 2, 2, 3),w)
        ref=numpy.zeros((3, 2, 2, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(0.0608896940982-0.446430160804j)*x[0]**o + ((-0.907065581165+0.698274089421j))*x[0] + ((0.792620456828-0.183581475698j))*x[1]**o + ((-0.388826942009+0.872078675165j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.221191186124+0.470170564042j)-((0.142251691821-0.105001939417j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.88055535379+0.732898902864j)*x[0]**o + ((-0.454775564887+0.257789512507j))*x[0] + ((0.84559464065-0.967840605446j))*x[1]**o + ((-0.369472807076-0.123052964221j))*x[1]
            ref[(0, 0, 0, 1)]=(0.450950811238-0.050102577148j)-((0.28769166574-0.0391569504304j))*(o-1.)
            arg[(0, 0, 0, 2)]=(-0.441420733873-0.627115654427j)*x[0]**o + ((0.0688717449939+0.354055387833j))*x[0] + ((0.362780319185-0.212894937312j))*x[1]**o + ((-0.942532642995+0.377665918901j))*x[1]
            ref[(0, 0, 0, 2)]=(-0.476150656344-0.0541446425026j)-((-0.0131067357813-0.14000176529j))*(o-1.)
            arg[(0, 0, 1, 0)]=(0.437418544613-0.406860672973j)*x[0]**o + ((-0.937046932232-0.187604929521j))*x[0] + ((-0.675048201798+0.728508693213j))*x[1]**o + ((0.730677235378-0.904250963341j))*x[1]
            ref[(0, 0, 1, 0)]=(-0.22199967702-0.385103936311j)-((-0.0396049428642+0.0536080033734j))*(o-1.)
            arg[(0, 0, 1, 1)]=(0.0760941547383+0.231372185188j)*x[0]**o + ((0.349187869291+0.225735391029j))*x[0] + ((-0.344206420704+0.0285313244329j))*x[1]**o + ((-0.997922394334+0.878369125362j))*x[1]
            ref[(0, 0, 1, 1)]=(-0.458423395505+0.682004013006j)-((-0.044685377661+0.0433172516036j))*(o-1.)
            arg[(0, 0, 1, 2)]=(0.273806076814+0.558664118475j)*x[0]**o + ((-0.279705403661+0.800950257723j))*x[0] + ((-0.32740555872-0.819313114318j))*x[1]**o + ((0.296387132024-0.23692735877j))*x[1]
            ref[(0, 0, 1, 2)]=(-0.0184588767709+0.151686951554j)-((-0.00893324698425-0.0434414993073j))*(o-1.)
            arg[(0, 1, 0, 0)]=(0.455187147559-0.697566692119j)*x[0]**o + ((-0.667469988531-0.276040720638j))*x[0] + ((0.858371979983-0.807775450489j))*x[1]**o + ((-0.944834303913-0.872726463274j))*x[1]
            ref[(0, 1, 0, 0)]=(-0.149372582451-1.32705466326j)-((0.218926521257-0.250890357101j))*(o-1.)
            arg[(0, 1, 0, 1)]=(-0.15590428887-0.755631143685j)*x[0]**o + ((-0.21430592889+0.145571537547j))*x[0] + ((0.719555758187-0.989126672765j))*x[1]**o + ((-0.969697759867+0.739710333142j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.31017610972-0.42973797288j)-((0.0939419115529-0.290792969408j))*(o-1.)
            arg[(0, 1, 0, 2)]=(-0.505001919501+0.532371292708j)*x[0]**o + ((0.836064429858-0.00535462834846j))*x[0] + ((0.892258930565+0.814620655355j))*x[1]**o + ((-0.223856784527+0.123455461048j))*x[1]
            ref[(0, 1, 0, 2)]=(0.499732328197+0.732546390382j)-((0.0645428351772+0.224498658011j))*(o-1.)
            arg[(0, 1, 1, 0)]=(-0.424578646789+0.67270002427j)*x[0]**o + ((0.427645008064+0.873337825613j))*x[0] + ((-0.226059507291-0.231470626171j))*x[1]**o + ((-0.941830956726-0.816492788696j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.582412051371+0.249037217508j)-((-0.108439692347+0.0735382330165j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.80522499306+0.86815350766j)*x[0]**o + ((-0.23987026311+0.609799806255j))*x[0] + ((-0.818564857811+0.637161691963j))*x[1]**o + ((0.917101474394-0.904902662211j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.473279319794+0.605106171833j)-((-0.270631641812+0.250885866604j))*(o-1.)
            arg[(0, 1, 1, 2)]=(0.672741419999-0.416913541454j)*x[0]**o + ((0.664802263173+0.314755413389j))*x[0] + ((0.0756865746685+0.198551568178j))*x[1]**o + ((-0.575678696125+0.838831582318j))*x[1]
            ref[(0, 1, 1, 2)]=(0.418775780858+0.467612511215j)-((0.124737999111-0.0363936622128j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.581936708626+0.421392124664j)*x[0]**o + ((0.544305049155+0.727393326202j))*x[0] + ((-0.331182930109-0.291272967464j))*x[1]**o + ((-0.534682617493-0.230659712084j))*x[1]
            ref[(1, 0, 0, 0)]=(-0.451748603536+0.313426385659j)-((-0.152186606456+0.0216865261999j))*(o-1.)
            arg[(1, 0, 0, 1)]=(0.271913271497+0.631682043879j)*x[0]**o + ((-0.931546704904+0.576092895032j))*x[0] + ((-0.549677595898+0.597189448883j))*x[1]**o + ((0.871961542605-0.89672792988j))*x[1]
            ref[(1, 0, 0, 1)]=(-0.16867474335+0.454118228957j)-((-0.0462940540668+0.20481191546j))*(o-1.)
            arg[(1, 0, 0, 2)]=(0.560881784436-0.0412576811003j)*x[0]**o + ((-0.8169699307+0.369460664416j))*x[0] + ((-0.0670967127505-0.434947626565j))*x[1]**o + ((-0.93044618238+0.387491868433j))*x[1]
            ref[(1, 0, 0, 2)]=(-0.626815520697+0.140373612592j)-((0.0822975119476-0.0793675512776j))*(o-1.)
            arg[(1, 0, 1, 0)]=(0.824801002332-0.585192654785j)*x[0]**o + ((0.717346571262-0.272153708118j))*x[0] + ((-0.48329857376-0.553946689826j))*x[1]**o + ((-0.437577092806+0.360384021404j))*x[1]
            ref[(1, 0, 1, 0)]=(0.310635953514-0.525454515663j)-((0.0569170714286-0.189856557435j))*(o-1.)
            arg[(1, 0, 1, 1)]=(-0.0231429927281-0.251859908997j)*x[0]**o + ((-0.844704865182+0.278860522092j))*x[0] + ((-0.571655635652+0.45272974985j))*x[1]**o + ((0.681962477703-0.311016405957j))*x[1]
            ref[(1, 0, 1, 1)]=(-0.37877050793+0.084356978494j)-((-0.09913310473+0.0334783068088j))*(o-1.)
            arg[(1, 0, 1, 2)]=(-0.896715578021+0.565917049003j)*x[0]**o + ((0.454582205857+0.0628583990111j))*x[0] + ((-0.322421160615+0.704938625256j))*x[1]**o + ((-0.677863300618-0.582044354627j))*x[1]
            ref[(1, 0, 1, 2)]=(-0.721208916699+0.375834859322j)-((-0.203189456439+0.211809279043j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.522952486939+0.812010929048j)*x[0]**o + ((-0.812169636913+0.251212796014j))*x[0] + ((0.267095864318+0.178795433538j))*x[1]**o + ((-0.0442449761823-0.220333657755j))*x[1]
            ref[(1, 1, 0, 0)]=(-0.0331831309189+0.510842750422j)-((0.131674725209+0.165134393764j))*(o-1.)
            arg[(1, 1, 0, 1)]=(-0.711188978322-0.391182138457j)*x[0]**o + ((0.551554546682+0.318602640731j))*x[0] + ((-0.0662465915-0.542833135564j))*x[1]**o + ((0.992171559864-0.591165870942j))*x[1]
            ref[(1, 1, 0, 1)]=(0.383145268362-0.603289252116j)-((-0.12957259497-0.155669212337j))*(o-1.)
            arg[(1, 1, 0, 2)]=(-0.266214059578+0.757670362768j)*x[0]**o + ((-0.324730708542-0.980998227422j))*x[0] + ((0.0190207029855-0.019830226211j))*x[1]**o + ((-0.583073279189-0.492571270745j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.577498672161-0.367864680805j)-((-0.0411988927654+0.122973356093j))*(o-1.)
            arg[(1, 1, 1, 0)]=(0.713325154923-0.407553066632j)*x[0]**o + ((0.994437256706+0.96413700318j))*x[0] + ((0.103948474675+0.887139286826j))*x[1]**o + ((-0.619898099966-0.129202357407j))*x[1]
            ref[(1, 1, 1, 0)]=(0.595906393169+0.657260432983j)-((0.1362122716+0.079931036699j))*(o-1.)
            arg[(1, 1, 1, 1)]=(-0.12773256751-0.894163276886j)*x[0]**o + ((0.477470746915-0.787867587122j))*x[0] + ((-0.109665490073+0.994122826444j))*x[1]**o + ((0.338330136893+0.531879514977j))*x[1]
            ref[(1, 1, 1, 1)]=(0.289201413112-0.0780142612934j)-((-0.0395663429305+0.0166599249264j))*(o-1.)
            arg[(1, 1, 1, 2)]=(0.605716267558+0.0330165719365j)*x[0]**o + ((-0.226460939989+0.984826579246j))*x[0] + ((-0.520425574783-0.296832161647j))*x[1]**o + ((0.952319913865+0.154942297881j))*x[1]
            ref[(1, 1, 1, 2)]=(0.405574833326+0.437976643708j)-((0.0142151154626-0.0439692649517j))*(o-1.)
            arg[(2, 0, 0, 0)]=(0.357699042064+0.350723507006j)*x[0]**o + ((0.899196082333-0.555115267253j))*x[0] + ((-0.0137879524334+0.621513950316j))*x[1]**o + ((-0.55769137925+0.936112792509j))*x[1]
            ref[(2, 0, 0, 0)]=(0.342707896357+0.67661749129j)-((0.0573185149385+0.16203957622j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.527640710191+0.906403219634j)*x[0]**o + ((-0.104051638639-0.424041729293j))*x[0] + ((0.183943310795+0.703977992994j))*x[1]**o + ((0.237042154689+0.477254802237j))*x[1]
            ref[(2, 0, 0, 1)]=(-0.105353441673+0.831797142786j)-((-0.0572828998993+0.268396868771j))*(o-1.)
            arg[(2, 0, 0, 2)]=(0.79220270677-0.497852128206j)*x[0]**o + ((0.75880893324+0.951764669575j))*x[0] + ((0.630164590526+0.433365562976j))*x[1]**o + ((0.17391791609-0.873229965724j))*x[1]
            ref[(2, 0, 0, 2)]=(1.17754707331+0.00702406931008j)-((0.237061216216-0.0107477608717j))*(o-1.)
            arg[(2, 0, 1, 0)]=(-0.844434150748+0.0546964824094j)*x[0]**o + ((-0.186241892574+0.872633597334j))*x[0] + ((0.505600659818-0.804209258295j))*x[1]**o + ((0.238865913209-0.107176098371j))*x[1]
            ref[(2, 0, 1, 0)]=(-0.143104735148+0.00797236153884j)-((-0.0564722484883-0.124918795981j))*(o-1.)
            arg[(2, 0, 1, 1)]=(0.190262795277-0.183122454113j)*x[0]**o + ((-0.0425522942483+0.559704106559j))*x[0] + ((-0.684223231906-0.814830860067j))*x[1]**o + ((0.963992888347+0.416400214502j))*x[1]
            ref[(2, 0, 1, 1)]=(0.213740078735-0.0109244965591j)-((-0.0823267394381-0.166325552363j))*(o-1.)
            arg[(2, 0, 1, 2)]=(0.150070723485+0.663324117941j)*x[0]**o + ((0.593835280629+0.611144764731j))*x[0] + ((-0.574746863738-0.665784240548j))*x[1]**o + ((0.62541466156-0.987082536944j))*x[1]
            ref[(2, 0, 1, 2)]=(0.397286900968-0.18919894741j)-((-0.070779356709-0.000410020434421j))*(o-1.)
            arg[(2, 1, 0, 0)]=(0.756752595717+0.120143950903j)*x[0]**o + ((-0.0501446579493+0.433805088596j))*x[0] + ((0.363188549651+0.245951953369j))*x[1]**o + ((0.68755879027+0.887814923883j))*x[1]
            ref[(2, 1, 0, 0)]=(0.878677638845+0.843857958375j)-((0.186656857561+0.0610159840453j))*(o-1.)
            arg[(2, 1, 0, 1)]=(0.00257176198345+0.855413069692j)*x[0]**o + ((-0.375601333452+0.171254783568j))*x[0] + ((-0.145967412526+0.588412497501j))*x[1]**o + ((-0.993152724697+0.618768826737j))*x[1]
            ref[(2, 1, 0, 1)]=(-0.756074854346+1.11692458875j)-((-0.0238992750905+0.240637594532j))*(o-1.)
            arg[(2, 1, 0, 2)]=(-0.595986181133-0.866232865307j)*x[0]**o + ((-0.958464184366+0.762260327661j))*x[0] + ((-0.538130483791-0.324830215239j))*x[1]**o + ((0.548698372517+0.382204396553j))*x[1]
            ref[(2, 1, 0, 2)]=(-0.771941238387-0.0232991781662j)-((-0.189019444154-0.198510513424j))*(o-1.)
            arg[(2, 1, 1, 0)]=(0.758106142139-0.113231469757j)*x[0]**o + ((0.0663938146408-0.620076604936j))*x[0] + ((0.446051796841+0.375860187203j))*x[1]**o + ((0.896173977167-0.935060237622j))*x[1]
            ref[(2, 1, 1, 0)]=(1.08336286539-0.646254062556j)-((0.20069298983+0.0437714529076j))*(o-1.)
            arg[(2, 1, 1, 1)]=(-0.402937628251+0.71475473808j)*x[0]**o + ((-0.498673411785+0.343776154574j))*x[0] + ((0.187224140173+0.584275449348j))*x[1]**o + ((-0.291408970512-0.912816720306j))*x[1]
            ref[(2, 1, 1, 1)]=(-0.502897935188+0.364994810848j)-((-0.0359522480131+0.216505031238j))*(o-1.)
            arg[(2, 1, 1, 2)]=(-0.47817315375+0.0572723624776j)*x[0]**o + ((-0.431563824388+0.00995082286808j))*x[0] + ((-0.733991003104-0.248202053134j))*x[1]**o + ((-0.763384971745+0.634727092539j))*x[1]
            ref[(2, 1, 1, 2)]=(-1.20355647649+0.226874112375j)-((-0.202027359476-0.0318216151095j))*(o-1.)
        else:
            arg[(0, 0, 0, 0)]=(-0.817028094082-0.0781880228904j)*x[0]**o + ((-0.182494493146+0.208798172804j))*x[0] + ((-0.367245458256+0.180650432108j))*x[1]**o + ((0.233232500353+0.00597905015144j))*x[1] + ((-0.0634163690995-0.75623620054j))*x[2]**o + ((0.534200618716-0.761928604565j))*x[2]
            ref[(0, 0, 0, 0)]=(-0.331375647757-0.600462586466j)-((-0.20794832024-0.108962298554j))*(o-1.)
            arg[(0, 0, 0, 1)]=(0.94284644272-0.43959174266j)*x[0]**o + ((0.172970052017+0.595045341947j))*x[0] + ((0.831205176217+0.852806100535j))*x[1]**o + ((0.194394882504-0.967706120106j))*x[1] + ((-0.949696666035+0.21972766005j))*x[2]**o + ((0.322679678635-0.200106406064j))*x[2]
            ref[(0, 0, 0, 1)]=(0.757199783029+0.0300874168512j)-((0.13739249215+0.105490336321j))*(o-1.)
            arg[(0, 0, 0, 2)]=(-0.901399063143+0.582110835977j)*x[0]**o + ((-0.961700368596-0.886379786214j))*x[0] + ((-0.954681016135-0.46232144134j))*x[1]**o + ((-0.704451300877-0.416051494384j))*x[1] + ((0.994604071567-0.996468545913j))*x[2]**o + ((0.351612450725+0.348744595267j))*x[2]
            ref[(0, 0, 0, 2)]=(-1.08800761323-0.915182918303j)-((-0.143579334619-0.146113191879j))*(o-1.)
            arg[(0, 0, 1, 0)]=(-0.111562101253+0.946851833616j)*x[0]**o + ((0.604610368288-0.51332016343j))*x[0] + ((-0.771328019048+0.535776240632j))*x[1]**o + ((-0.105748082106+0.500015837213j))*x[1] + ((0.826634378533+0.831135417751j))*x[2]**o + ((-0.868811378042+0.960739025875j))*x[2]
            ref[(0, 0, 1, 0)]=(-0.213102416814+1.63059909583j)-((-0.00937595696135+0.385627248666j))*(o-1.)
            arg[(0, 0, 1, 1)]=(-0.302027321034-0.838382987169j)*x[0]**o + ((0.269317376707-0.298069917503j))*x[0] + ((-0.576166043021+0.873842226821j))*x[1]**o + ((-0.794141479221-0.278149606708j))*x[1] + ((0.648983559881-0.678440416509j))*x[2]**o + ((-0.26235635127-0.315831154636j))*x[2]
            ref[(0, 0, 1, 1)]=(-0.508195128979-0.767515927852j)-((-0.0382016340289-0.107163529476j))*(o-1.)
            arg[(0, 0, 1, 2)]=(-0.272879309522+0.364670919623j)*x[0]**o + ((-0.14210082744-0.863554333345j))*x[0] + ((0.0537434933795+0.360493779851j))*x[1]**o + ((-0.462804462089-0.106295731102j))*x[1] + ((-0.839114384549+0.285361582517j))*x[2]**o + ((-0.183702090436-0.178240904582j))*x[2]
            ref[(0, 0, 1, 2)]=(-0.923428790328-0.0687823435185j)-((-0.176375033449+0.168421046999j))*(o-1.)
            arg[(0, 1, 0, 0)]=(-0.447746434494+0.519086601755j)*x[0]**o + ((-0.264869511539-0.641275371978j))*x[0] + ((-0.434465825144-0.40921793971j))*x[1]**o + ((0.0211776799417-0.93832338954j))*x[1] + ((-0.594565746956-0.510291529282j))*x[2]**o + ((0.408937485163+0.336457579892j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.655766176514-0.821782024431j)-((-0.246129667766-0.0667371445395j))*(o-1.)
            arg[(0, 1, 0, 1)]=(0.302362381099-0.950725665067j)*x[0]**o + ((0.998734665388-0.308898563435j))*x[0] + ((0.463642698483-0.229294282717j))*x[1]**o + ((0.924929109925+0.868229666372j))*x[1] + ((-0.7730388506-0.32777399304j))*x[2]**o + ((0.693821595639-0.940629668743j))*x[2]
            ref[(0, 1, 0, 1)]=(1.30522579997-0.944546253315j)-((-0.00117229516975-0.251298990137j))*(o-1.)
            arg[(0, 1, 0, 2)]=(-0.268228478883+0.926667449575j)*x[0]**o + ((0.664160887212-0.875891768894j))*x[0] + ((0.0830148733352-0.105712131933j))*x[1]**o + ((-0.640519791787+0.331257480956j))*x[1] + ((0.185206572305+0.134390447101j))*x[2]**o + ((0.374359845547+0.142936164538j))*x[2]
            ref[(0, 1, 0, 2)]=(0.198996953865+0.276823820672j)-((-1.17220699018e-06+0.159224294124j))*(o-1.)
            arg[(0, 1, 1, 0)]=(0.516717346342-0.308833823397j)*x[0]**o + ((-0.910051341916-0.406204185361j))*x[0] + ((0.919973919651-0.502813304055j))*x[1]**o + ((-0.446401768886-0.0812251463845j))*x[1] + ((-0.41974583416+0.166923530439j))*x[2]**o + ((-0.41429952735-0.0617350830366j))*x[2]
            ref[(0, 1, 1, 0)]=(-0.376903603159-0.596944005898j)-((0.169490905306-0.107453932836j))*(o-1.)
            arg[(0, 1, 1, 1)]=(-0.779937606909+0.619394567967j)*x[0]**o + ((0.702238743057+0.738402306074j))*x[0] + ((0.749787070768+0.396797040186j))*x[1]**o + ((0.032893966452-0.582549528184j))*x[1] + ((-0.025293234902-0.577457804178j))*x[2]**o + ((-0.0538652468847-0.237401246779j))*x[2]
            ref[(0, 1, 1, 1)]=(0.312911845791+0.178592667543j)-((-0.00924062850711+0.0731223006624j))*(o-1.)
            arg[(0, 1, 1, 2)]=(-0.619127587494+0.491424993797j)*x[0]**o + ((0.122867201692-0.571074916312j))*x[0] + ((-0.348063511786+0.185409730511j))*x[1]**o + ((0.125274031695+0.222831312515j))*x[1] + ((-0.698034190618-0.199721778711j))*x[2]**o + ((0.926729524135-0.393122477442j))*x[2]
            ref[(0, 1, 1, 2)]=(-0.245177266188-0.132126567821j)-((-0.277537548316+0.0795188242662j))*(o-1.)
            arg[(1, 0, 0, 0)]=(-0.442381849138-0.246859028013j)*x[0]**o + ((0.74088623201+0.448023166525j))*x[0] + ((0.200408169703-0.233542104195j))*x[1]**o + ((0.24427757621-0.365462620031j))*x[1] + ((0.645328741892+0.0440304178429j))*x[2]**o + ((0.0788782652083+0.674002736192j))*x[2]
            ref[(1, 0, 0, 0)]=(0.733698567942+0.160096284161j)-((0.0672258437428-0.0727284523942j))*(o-1.)
            arg[(1, 0, 0, 1)]=(0.533592536924-0.518666902381j)*x[0]**o + ((-0.0820339424064+0.0906415871834j))*x[0] + ((-0.593814932803+0.136937951109j))*x[1]**o + ((-0.761344341511-0.197608347183j))*x[1] + ((-0.274297180744-0.935911242707j))*x[2]**o + ((-0.886557277929+0.703174824175j))*x[2]
            ref[(1, 0, 0, 1)]=(-1.03222756923-0.360716064902j)-((-0.0557532627704-0.219606698996j))*(o-1.)
            arg[(1, 0, 0, 2)]=(0.258598734019-0.179177166666j)*x[0]**o + ((0.38435897311-0.3375392934j))*x[0] + ((-0.143373497177+0.911800635165j))*x[1]**o + ((0.49445350091+0.554195748134j))*x[1] + ((-0.0266712615172+0.382212753144j))*x[2]**o + ((-0.0450018134572+0.393384927822j))*x[2]
            ref[(1, 0, 0, 2)]=(0.461182317944+0.8624388021j)-((0.0147589958875+0.185806036941j))*(o-1.)
            arg[(1, 0, 1, 0)]=(-0.8434157769+0.665003468692j)*x[0]**o + ((0.702932992852+0.309279398592j))*x[0] + ((-0.303766836899-0.180964642645j))*x[1]**o + ((-0.125501323761+0.272193776284j))*x[1] + ((-0.119857082489+0.539687736049j))*x[2]**o + ((0.205141336461+0.0281682724591j))*x[2]
            ref[(1, 0, 1, 0)]=(-0.242233345368+0.816684004716j)-((-0.211173282715+0.170621093683j))*(o-1.)
            arg[(1, 0, 1, 1)]=(0.446581224253-0.73402057343j)*x[0]**o + ((0.770462468647-0.226005310894j))*x[0] + ((0.626260396812-0.965814665803j))*x[1]**o + ((0.855185589978-0.755558552232j))*x[1] + ((-0.228307143913+0.598837386065j))*x[2]**o + ((-0.929598585617+0.839183202774j))*x[2]
            ref[(1, 0, 1, 1)]=(0.77029197508-0.62168925676j)-((0.140755746192-0.183499642195j))*(o-1.)
            arg[(1, 0, 1, 2)]=(-0.808572153276-0.868088993274j)*x[0]**o + ((0.685651182343+0.231715044523j))*x[0] + ((-0.282265289355-0.728709081164j))*x[1]**o + ((-0.491593772369+0.63698912822j))*x[1] + ((-0.682112757212+0.664597268009j))*x[2]**o + ((0.162073870023+0.270810210401j))*x[2]
            ref[(1, 0, 1, 2)]=(-0.708409459923+0.103656788358j)-((-0.295491699974-0.155366801072j))*(o-1.)
            arg[(1, 1, 0, 0)]=(0.589414462432+0.706406820867j)*x[0]**o + ((-0.560630480829-0.885502139205j))*x[0] + ((0.274871374858+0.702345638316j))*x[1]**o + ((-0.285123230173-0.294212716543j))*x[1] + ((-0.5547007441-0.213814812724j))*x[2]**o + ((0.283229092642+0.883112804407j))*x[2]
            ref[(1, 1, 0, 0)]=(-0.126469762585+0.449167797559j)-((0.0515975155316+0.19915627441j))*(o-1.)
            arg[(1, 1, 0, 1)]=(0.764164443204+0.613275222139j)*x[0]**o + ((0.898906913308-0.298402902854j))*x[0] + ((-0.534738706317+0.545070604388j))*x[1]**o + ((-0.248430500461-0.623140487876j))*x[1] + ((0.63325378151+0.767993547028j))*x[2]**o + ((0.640431858446+0.245876921645j))*x[2]
            ref[(1, 1, 0, 1)]=(1.07679389485+0.625336452235j)-((0.143779919733+0.321056562259j))*(o-1.)
            arg[(1, 1, 0, 2)]=(-0.0676858018384-0.00181796816562j)*x[0]**o + ((0.866241966035+0.727779762529j))*x[0] + ((-0.679748653402-0.683034172452j))*x[1]**o + ((0.972610705189-0.663149388829j))*x[1] + ((0.98979179439+0.754271158725j))*x[2]**o + ((-0.14582573517-0.798854774965j))*x[2]
            ref[(1, 1, 0, 2)]=(0.967692137602-0.332402691578j)-((0.0403928898582+0.0115698363513j))*(o-1.)
            arg[(1, 1, 1, 0)]=(-0.533115376135+0.592395679291j)*x[0]**o + ((0.215349993986-0.757133068283j))*x[0] + ((0.661268909077+0.578702368599j))*x[1]**o + ((0.803918512266+0.759934519535j))*x[1] + ((0.891612105775-0.442288886112j))*x[2]**o + ((-0.844726688625+0.762588143005j))*x[2]
            ref[(1, 1, 1, 0)]=(0.597153728172+0.747099378018j)-((0.169960939786+0.12146819363j))*(o-1.)
            arg[(1, 1, 1, 1)]=(0.526184964613-0.202595501212j)*x[0]**o + ((0.0233986926537+0.709404369742j))*x[0] + ((0.311019454322-0.22176169823j))*x[1]**o + ((0.19413425827-0.458066348701j))*x[1] + ((0.0598663032773+0.771196106891j))*x[2]**o + ((-0.484124987992+0.869209529439j))*x[2]
            ref[(1, 1, 1, 1)]=(0.315239342572+0.733693228964j)-((0.149511787035+0.0578064845747j))*(o-1.)
            arg[(1, 1, 1, 2)]=(-0.796427812103+0.4566708582j)*x[0]**o + ((-0.113520027404-0.785720526679j))*x[0] + ((-0.936553620587+0.432734782681j))*x[1]**o + ((0.239861107837+0.0783651512882j))*x[1] + ((-0.188713750622+0.45370492025j))*x[2]**o + ((0.718133575045+0.667597322546j))*x[2]
            ref[(1, 1, 1, 2)]=(-0.538610263917+0.651676254143j)-((-0.320282530552+0.223851760188j))*(o-1.)
            arg[(2, 0, 0, 0)]=(0.206552568224+0.0244930187306j)*x[0]**o + ((-0.468974958336+0.10082175004j))*x[0] + ((0.451010109288+0.564288118727j))*x[1]**o + ((-0.529945397321+0.899654890478j))*x[1] + ((-0.570406774068+0.258485524147j))*x[2]**o + ((-0.0676830643846+0.326781542708j))*x[2]
            ref[(2, 0, 0, 0)]=(-0.489723758299+1.08726242242j)-((0.0145259839071+0.141211110267j))*(o-1.)
            arg[(2, 0, 0, 1)]=(-0.0405803770644-0.677799444715j)*x[0]**o + ((-0.755752391773-0.185988674616j))*x[0] + ((-0.704325690987-0.752908662358j))*x[1]**o + ((-0.502850760147-0.430905778004j))*x[1] + ((-0.651313278593+0.311381101102j))*x[2]**o + ((-0.541047696409-0.0683538462666j))*x[2]
            ref[(2, 0, 0, 1)]=(-1.59793509749-0.902287652429j)-((-0.232703224441-0.186554500995j))*(o-1.)
            arg[(2, 0, 0, 2)]=(0.0361318800613-0.442648045252j)*x[0]**o + ((0.316699397149+0.3255753077j))*x[0] + ((0.505634011821-0.123732553849j))*x[1]**o + ((0.270136085678-0.322056744111j))*x[1] + ((0.808846343373-0.172878224028j))*x[2]**o + ((0.482430700388+0.82218858766j))*x[2]
            ref[(2, 0, 0, 2)]=(1.20993920923+0.0432241640596j)-((0.225102039209-0.123209803855j))*(o-1.)
            arg[(2, 0, 1, 0)]=(-0.555881041559-0.691423251957j)*x[0]**o + ((-0.763056378763+0.282095790087j))*x[0] + ((0.511736849256-0.663690692907j))*x[1]**o + ((-0.436175352424-0.0227355983197j))*x[1] + ((-0.142700600677+0.470954309649j))*x[2]**o + ((-0.926146847416-0.740734066083j))*x[2]
            ref[(2, 0, 1, 0)]=(-1.15611168579-0.682766754766j)-((-0.03114079883-0.147359939203j))*(o-1.)
            arg[(2, 0, 1, 1)]=(-0.412425705266-0.222131921009j)*x[0]**o + ((-0.159138173136-0.730707686687j))*x[0] + ((0.809767350874-0.887284125244j))*x[1]**o + ((0.559898752165-0.984235805825j))*x[1] + ((-0.804494749657+0.720418730573j))*x[2]**o + ((-0.000500727323897-0.904622366149j))*x[2]
            ref[(2, 0, 1, 1)]=(-0.00344662617189-1.50428158717j)-((-0.0678588506748-0.0648328859467j))*(o-1.)
            arg[(2, 0, 1, 2)]=(0.795689694033+0.612729165733j)*x[0]**o + ((0.777351814156+0.743070543827j))*x[0] + ((-0.129677576607-0.575657333842j))*x[1]**o + ((0.899277054677+0.382908053617j))*x[1] + ((0.209886579794-0.704828677539j))*x[2]**o + ((0.400832934075-0.645198797628j))*x[2]
            ref[(2, 0, 1, 2)]=(1.47668025006-0.0934885229162j)-((0.145983116203-0.111292807608j))*(o-1.)
            arg[(2, 1, 0, 0)]=(-0.0551813664621+0.90849855869j)*x[0]**o + ((-0.236780549788-0.216685525942j))*x[0] + ((-0.467838677867-0.790195006616j))*x[1]**o + ((0.473117309386-0.0659395743899j))*x[1] + ((-0.783548761569-0.957538106464j))*x[2]**o + ((0.3856946817-0.119770772995j))*x[2]
            ref[(2, 1, 0, 0)]=(-0.3422686823-0.620815213859j)-((-0.21776146765-0.139872425732j))*(o-1.)
            arg[(2, 1, 0, 1)]=(-0.141902966552+0.422043102896j)*x[0]**o + ((0.146163973047+0.739126434804j))*x[0] + ((0.48866654742-0.296457389341j))*x[1]**o + ((0.14481594999+0.96914865426j))*x[1] + ((-0.553183259447-0.0513352748738j))*x[2]**o + ((0.138586828071+0.648939440498j))*x[2]
            ref[(2, 1, 0, 1)]=(0.111573536264+1.21573248412j)-((-0.0344032797631+0.0123750731135j))*(o-1.)
            arg[(2, 1, 0, 2)]=(0.536862674307-0.310815060658j)*x[0]**o + ((0.0414292036386-0.452918354449j))*x[0] + ((0.630280621558-0.980524622588j))*x[1]**o + ((-0.570637927153+0.345510118917j))*x[1] + ((0.903531891366+0.191309505765j))*x[2]**o + ((-0.660695774196+0.330368272142j))*x[2]
            ref[(2, 1, 0, 2)]=(0.44038534476-0.438535070436j)-((0.345112531205-0.183338362913j))*(o-1.)
            arg[(2, 1, 1, 0)]=(-0.00293790126227+0.178286415441j)*x[0]**o + ((-0.255287995522+0.551873727332j))*x[0] + ((0.491267797661-0.756145413817j))*x[1]**o + ((-0.000963084719112-0.214212631722j))*x[1] + ((-0.398936951191+0.965316199745j))*x[2]**o + ((-0.260328841992-0.435578402227j))*x[2]
            ref[(2, 1, 1, 0)]=(-0.213593488513+0.144769947376j)-((0.0148988242013+0.0645762002281j))*(o-1.)
            arg[(2, 1, 1, 1)]=(0.852610894153-0.795804547501j)*x[0]**o + ((0.496675948856+0.762514166401j))*x[0] + ((0.841272404654-0.858664838043j))*x[1]**o + ((0.591986915273+0.784207938607j))*x[1] + ((-0.487038511265-0.26218808571j))*x[2]**o + ((0.511646274071+0.475863415908j))*x[2]
            ref[(2, 1, 1, 1)]=(1.40357696287+0.052964024831j)-((0.201140797924-0.319442911876j))*(o-1.)
            arg[(2, 1, 1, 2)]=(-0.657419018799+0.735119561599j)*x[0]**o + ((-0.62584505856-0.788530892412j))*x[0] + ((-0.276669904045+0.352085720975j))*x[1]**o + ((-0.37726013806-0.469572818968j))*x[1] + ((0.0703720230613+0.483119441429j))*x[2]**o + ((0.576980335296+0.458521743583j))*x[2]
            ref[(2, 1, 1, 2)]=(-0.644920880554+0.385371378103j)-((-0.14395281663+0.261720787334j))*(o-1.)
        res=integrate(arg)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (3, 2, 2, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_ContinuousFunction_rank0(self):
        """
        tests integral of rank 0 Data on FunctionOnBoundary

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(0.198581155815+0.384989999904j)*x[0]**o + ((-0.90120976501-0.561804126183j))*x[0] + ((-0.0830446135197+0.349463852647j))*x[1]**o + ((-0.0357018703926+0.352658919703j))*x[1]
            ref=(0.115536542295+0.734453852551j)*(1+2.*(dim-1)/(o+1.)) + ((-0.936911635403-0.20914520648j))*dim
        else:
            arg=(-0.705375053124-0.898219658423j)*x[0]**o + ((-0.432744905517-0.839177711995j))*x[0] + ((0.940717585392+0.921577096334j))*x[1]**o + ((0.819798759495+0.154769900066j))*x[1] + ((-0.934956262379-0.769304830264j))*x[2]**o + ((-0.103700006685+0.853743217891j))*x[2]
            ref=(-0.699613730111-0.745947392353j)*(1+2.*(dim-1)/(o+1.)) + ((0.283353847294+0.169335405962j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_ContinuousFunction_rank1(self):
        """
        tests integral of rank 1 Data on FunctionOnBoundary

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4,),w)
        ref=numpy.zeros((4,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.809831232504-0.00386975677892j)*x[0]**o + ((0.566313859918+0.0824606724976j))*x[0] + ((0.438978261523+0.584260267311j))*x[1]**o + ((-0.990380177392-0.893077099586j))*x[1]
            ref[(0,)]=(-0.370852970981+0.580390510532j)*(1+2.*(dim-1)/(o+1.)) + ((-0.424066317474-0.810616427089j))*dim
            arg[(1,)]=(-0.556284151134+0.548836592967j)*x[0]**o + ((0.781015118325-0.601158705975j))*x[0] + ((-0.384613992771+0.546426364676j))*x[1]**o + ((-0.532693554751-0.658678807716j))*x[1]
            ref[(1,)]=(-0.940898143905+1.09526295764j)*(1+2.*(dim-1)/(o+1.)) + ((0.248321563574-1.25983751369j))*dim
            arg[(2,)]=(-0.477734325963+0.0656756488343j)*x[0]**o + ((-0.417284702446-0.516832551928j))*x[0] + ((0.395055751679+0.471205066518j))*x[1]**o + ((-0.0111479105449+0.781452511147j))*x[1]
            ref[(2,)]=(-0.0826785742836+0.536880715353j)*(1+2.*(dim-1)/(o+1.)) + ((-0.428432612991+0.264619959219j))*dim
            arg[(3,)]=(-0.662646777195+0.486517736555j)*x[0]**o + ((0.569360348216-0.855590740172j))*x[0] + ((-0.192023084532+0.260661832752j))*x[1]**o + ((0.248435392284+0.685556922098j))*x[1]
            ref[(3,)]=(-0.854669861727+0.747179569308j)*(1+2.*(dim-1)/(o+1.)) + ((0.817795740501-0.170033818074j))*dim
        else:
            arg[(0,)]=(0.264346862803+0.447421930417j)*x[0]**o + ((-0.503546175869-0.354402386638j))*x[0] + ((0.756485370584+0.334895194647j))*x[1]**o + ((0.807696962034+0.0751723991385j))*x[1] + ((-0.751264894585+0.425277554134j))*x[2]**o + ((-0.207946853951+0.278470934564j))*x[2]
            ref[(0,)]=(0.269567338802+1.2075946792j)*(1+2.*(dim-1)/(o+1.)) + ((0.096203932214-0.000759052935719j))*dim
            arg[(1,)]=(-0.672355361783+0.421855053649j)*x[0]**o + ((-0.859288550663-0.519615590955j))*x[0] + ((0.678252073412+0.606469802555j))*x[1]**o + ((-0.26823539367-0.35997552876j))*x[1] + ((-0.916318417694-0.281283689687j))*x[2]**o + ((-0.0235344769401-0.107496495841j))*x[2]
            ref[(1,)]=(-0.910421706065+0.747041166516j)*(1+2.*(dim-1)/(o+1.)) + ((-1.15105842127-0.987087615556j))*dim
            arg[(2,)]=(0.286691615314+0.729349401554j)*x[0]**o + ((0.17299769659-0.0876403598609j))*x[0] + ((-0.00537042954698+0.277972230924j))*x[1]**o + ((-0.928361932019+0.914575577889j))*x[1] + ((-0.742701503996+0.0715279031025j))*x[2]**o + ((0.416090643573+0.953816631496j))*x[2]
            ref[(2,)]=(-0.461380318229+1.07884953558j)*(1+2.*(dim-1)/(o+1.)) + ((-0.339273591857+1.78075184952j))*dim
            arg[(3,)]=(0.668313569774-0.710146131267j)*x[0]**o + ((-0.39877006774-0.869233840214j))*x[0] + ((0.204091602894-0.0120748440555j))*x[1]**o + ((-0.400540011903+0.23460393729j))*x[1] + ((0.316164790133+0.626939690675j))*x[2]**o + ((-0.745751722003-0.441739531054j))*x[2]
            ref[(3,)]=(1.1885699628-0.0952812846475j)*(1+2.*(dim-1)/(o+1.)) + ((-1.54506180165-1.07636943398j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_ContinuousFunction_rank2(self):
        """
        tests integral of rank 2 Data on FunctionOnBoundary

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4, 2),w)
        ref=numpy.zeros((4, 2),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.585559949736+0.753261492689j)*x[0]**o + ((0.522103426861-0.574870897603j))*x[0] + ((0.35160825634-0.987381496292j))*x[1]**o + ((-0.536845832523+0.885742968881j))*x[1]
            ref[(0, 0)]=(0.937168206076-0.234120003603j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0147424056614+0.310872071278j))*dim
            arg[(0, 1)]=(0.32581148941+0.877081446949j)*x[0]**o + ((-0.694181720986+0.991462619598j))*x[0] + ((0.260134714965-0.74614457623j))*x[1]**o + ((0.678342904906+0.872748426702j))*x[1]
            ref[(0, 1)]=(0.585946204375+0.130936870719j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0158388160802+1.8642110463j))*dim
            arg[(1, 0)]=(0.382239958295-0.389429030952j)*x[0]**o + ((0.728517673088-0.961774990243j))*x[0] + ((-0.364965177489-0.515317132021j))*x[1]**o + ((-0.813899077628-0.200631635772j))*x[1]
            ref[(1, 0)]=(0.017274780806-0.904746162974j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0853814045403-1.16240662601j))*dim
            arg[(1, 1)]=(-0.418036261759-0.373953026914j)*x[0]**o + ((-0.0911288541937-0.940218510303j))*x[0] + ((0.342003311964+0.423406625934j))*x[1]**o + ((0.130299789028-0.227270516253j))*x[1]
            ref[(1, 1)]=(-0.0760329497953+0.0494535990196j)*(1+2.*(dim-1)/(o+1.)) + ((0.0391709348347-1.16748902656j))*dim
            arg[(2, 0)]=(-0.691338338981-0.481473147596j)*x[0]**o + ((-0.212675570454+0.0169867310481j))*x[0] + ((0.573174592889+0.262339733039j))*x[1]**o + ((0.239627429109-0.247860041214j))*x[1]
            ref[(2, 0)]=(-0.118163746092-0.219133414557j)*(1+2.*(dim-1)/(o+1.)) + ((0.0269518586545-0.230873310166j))*dim
            arg[(2, 1)]=(0.867819351033+0.244586131966j)*x[0]**o + ((0.286115344319-0.019176335632j))*x[0] + ((0.228149051138+0.663626209323j))*x[1]**o + ((0.705525433679-0.048759616575j))*x[1]
            ref[(2, 1)]=(1.09596840217+0.908212341289j)*(1+2.*(dim-1)/(o+1.)) + ((0.991640777998-0.067935952207j))*dim
            arg[(3, 0)]=(0.181671694988+0.0696947854967j)*x[0]**o + ((-0.976080564142+0.288967292029j))*x[0] + ((-0.0072785238973+0.983955047671j))*x[1]**o + ((0.649043464132-0.900641601889j))*x[1]
            ref[(3, 0)]=(0.174393171091+1.05364983317j)*(1+2.*(dim-1)/(o+1.)) + ((-0.32703710001-0.61167430986j))*dim
            arg[(3, 1)]=(0.546146163945-0.47227750069j)*x[0]**o + ((-0.285976200287+0.529154099337j))*x[0] + ((-0.866653129339-0.949927166653j))*x[1]**o + ((-0.818390809675-0.104499497595j))*x[1]
            ref[(3, 1)]=(-0.320506965394-1.42220466734j)*(1+2.*(dim-1)/(o+1.)) + ((-1.10436700996+0.424654601743j))*dim
        else:
            arg[(0, 0)]=(0.127496011772+0.35377168851j)*x[0]**o + ((-0.351019917831-0.0164641615684j))*x[0] + ((0.876156227922+0.577790268318j))*x[1]**o + ((0.536999139807+0.268390924848j))*x[1] + ((0.970636474905-0.47966563304j))*x[2]**o + ((0.537688725918+0.0673019255908j))*x[2]
            ref[(0, 0)]=(1.9742887146+0.451896323788j)*(1+2.*(dim-1)/(o+1.)) + ((0.723667947893+0.31922868887j))*dim
            arg[(0, 1)]=(0.872102338422+0.0965949054737j)*x[0]**o + ((-0.904313157442+0.688082682034j))*x[0] + ((-0.800188598658+0.885587205394j))*x[1]**o + ((-0.322413656329+0.0482377439355j))*x[1] + ((0.344736318893+0.392460161357j))*x[2]**o + ((-0.639248095276-0.204587553256j))*x[2]
            ref[(0, 1)]=(0.416650058657+1.37464227223j)*(1+2.*(dim-1)/(o+1.)) + ((-1.86597490905+0.531732872713j))*dim
            arg[(1, 0)]=(0.0491555009778-0.495998634493j)*x[0]**o + ((0.0151845953304-0.86802333204j))*x[0] + ((0.181412052223-0.891951083464j))*x[1]**o + ((-0.971798872809+0.408826835545j))*x[1] + ((-0.0739132201466-0.492219785321j))*x[2]**o + ((0.461412109609+0.0381194910668j))*x[2]
            ref[(1, 0)]=(0.156654333054-1.88016950328j)*(1+2.*(dim-1)/(o+1.)) + ((-0.495202167869-0.421077005429j))*dim
            arg[(1, 1)]=(0.0650661469927-0.407849859231j)*x[0]**o + ((0.0789020519426-0.730905315975j))*x[0] + ((0.459922740352+0.346567143773j))*x[1]**o + ((0.74034443292-0.193884266879j))*x[1] + ((-0.462769373948-0.801032477425j))*x[2]**o + ((0.277490466695+0.448834762645j))*x[2]
            ref[(1, 1)]=(0.0622195133964-0.862315192883j)*(1+2.*(dim-1)/(o+1.)) + ((1.09673695156-0.475954820209j))*dim
            arg[(2, 0)]=(-0.850430399649+0.808053901833j)*x[0]**o + ((0.123333604923-0.749957734095j))*x[0] + ((-0.499670002087-0.916428353183j))*x[1]**o + ((-0.69742983751+0.416466935012j))*x[1] + ((-0.236137051582+0.76284563047j))*x[2]**o + ((0.924515818615-0.439503112384j))*x[2]
            ref[(2, 0)]=(-1.58623745332+0.654471179121j)*(1+2.*(dim-1)/(o+1.)) + ((0.350419586028-0.772993911467j))*dim
            arg[(2, 1)]=(-0.969240346314-0.258690239145j)*x[0]**o + ((0.194480071192+0.563609799253j))*x[0] + ((-0.930434293114+0.346479146525j))*x[1]**o + ((0.726113061835-0.466797810368j))*x[1] + ((-0.290210768489+0.352222459179j))*x[2]**o + ((0.226442232031+0.67609396895j))*x[2]
            ref[(2, 1)]=(-2.18988540792+0.440011366558j)*(1+2.*(dim-1)/(o+1.)) + ((1.14703536506+0.772905957835j))*dim
            arg[(3, 0)]=(0.671467810745-0.226837788305j)*x[0]**o + ((-0.670357883349+0.373015947705j))*x[0] + ((-0.772107906411+0.469714142557j))*x[1]**o + ((0.221859734991-0.341852293964j))*x[1] + ((0.721475227345-0.928593996681j))*x[2]**o + ((-0.844078376795+0.273583602663j))*x[2]
            ref[(3, 0)]=(0.620835131679-0.685717642429j)*(1+2.*(dim-1)/(o+1.)) + ((-1.29257652515+0.304747256404j))*dim
            arg[(3, 1)]=(0.159868940318-0.487368116807j)*x[0]**o + ((-0.586502064322+0.249026790031j))*x[0] + ((0.299485814662+0.841534003267j))*x[1]**o + ((-0.307033685619+0.0569596024879j))*x[1] + ((0.964550952676+0.327679562877j))*x[2]**o + ((0.782311007687-0.345813276982j))*x[2]
            ref[(3, 1)]=(1.42390570766+0.681845449338j)*(1+2.*(dim-1)/(o+1.)) + ((-0.111224742254-0.0398268844626j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 2),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_ContinuousFunction_rank3(self):
        """
        tests integral of rank 3 Data on FunctionOnBoundary

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(4, 4, 3),w)
        ref=numpy.zeros((4, 4, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(0.728893280968-0.621737720881j)*x[0]**o + ((-0.512535294458-0.440071744002j))*x[0] + ((-0.145828656457-0.566495536065j))*x[1]**o + ((-0.226507478032+0.185935880351j))*x[1]
            ref[(0, 0, 0)]=(0.58306462451-1.18823325695j)*(1+2.*(dim-1)/(o+1.)) + ((-0.73904277249-0.254135863651j))*dim
            arg[(0, 0, 1)]=(0.98905515711-0.866763929416j)*x[0]**o + ((-0.148766430783+0.374890045702j))*x[0] + ((-0.349913088129+0.56245923725j))*x[1]**o + ((-0.498012005996+0.891602859005j))*x[1]
            ref[(0, 0, 1)]=(0.639142068981-0.304304692166j)*(1+2.*(dim-1)/(o+1.)) + ((-0.646778436779+1.26649290471j))*dim
            arg[(0, 0, 2)]=(0.801267762273+0.542200141801j)*x[0]**o + ((-0.768462905595+0.877130757609j))*x[0] + ((0.648784123957+0.840570043276j))*x[1]**o + ((0.42056010407-0.195242660599j))*x[1]
            ref[(0, 0, 2)]=(1.45005188623+1.38277018508j)*(1+2.*(dim-1)/(o+1.)) + ((-0.347902801526+0.68188809701j))*dim
            arg[(0, 1, 0)]=(0.438367422861+0.108107891766j)*x[0]**o + ((0.57649808299+0.586754163296j))*x[0] + ((0.0163381617605-0.92220283526j))*x[1]**o + ((-0.0635160067314+0.488024960279j))*x[1]
            ref[(0, 1, 0)]=(0.454705584621-0.814094943494j)*(1+2.*(dim-1)/(o+1.)) + ((0.512982076259+1.07477912358j))*dim
            arg[(0, 1, 1)]=(-0.221106115618-0.105272906167j)*x[0]**o + ((-0.43005060575-0.0140868610377j))*x[0] + ((0.156357060204-0.763781007643j))*x[1]**o + ((0.326810644814-0.439113455744j))*x[1]
            ref[(0, 1, 1)]=(-0.0647490554146-0.86905391381j)*(1+2.*(dim-1)/(o+1.)) + ((-0.103239960936-0.453200316782j))*dim
            arg[(0, 1, 2)]=(-0.173348995083+0.784422719668j)*x[0]**o + ((0.635180169255+0.2099617441j))*x[0] + ((-0.050248958941-0.655939776631j))*x[1]**o + ((0.297274107732+0.834064643763j))*x[1]
            ref[(0, 1, 2)]=(-0.223597954024+0.128482943037j)*(1+2.*(dim-1)/(o+1.)) + ((0.932454276987+1.04402638786j))*dim
            arg[(0, 2, 0)]=(0.634249066744+0.576858267126j)*x[0]**o + ((-0.491296199308+0.362775160307j))*x[0] + ((0.473667912795+0.881891048463j))*x[1]**o + ((0.118265378454-0.378894768159j))*x[1]
            ref[(0, 2, 0)]=(1.10791697954+1.45874931559j)*(1+2.*(dim-1)/(o+1.)) + ((-0.373030820854-0.0161196078516j))*dim
            arg[(0, 2, 1)]=(-0.31652195791-0.506228022055j)*x[0]**o + ((0.285776359517+0.921403151886j))*x[0] + ((0.816783870381-0.937884639248j))*x[1]**o + ((-0.177853748396+0.882765657922j))*x[1]
            ref[(0, 2, 1)]=(0.500261912471-1.4441126613j)*(1+2.*(dim-1)/(o+1.)) + ((0.107922611121+1.80416880981j))*dim
            arg[(0, 2, 2)]=(-0.0916544565566-0.941733652998j)*x[0]**o + ((0.245609412312+0.818447012865j))*x[0] + ((-0.196652015984-0.764635292905j))*x[1]**o + ((0.993819140157+0.00490116289546j))*x[1]
            ref[(0, 2, 2)]=(-0.288306472541-1.7063689459j)*(1+2.*(dim-1)/(o+1.)) + ((1.23942855247+0.82334817576j))*dim
            arg[(0, 3, 0)]=(0.829407742738+0.889071072366j)*x[0]**o + ((-0.0174084223404-0.466170640509j))*x[0] + ((-0.140652417359+0.403036097968j))*x[1]**o + ((-0.454318540975-0.708514638125j))*x[1]
            ref[(0, 3, 0)]=(0.688755325379+1.29210717033j)*(1+2.*(dim-1)/(o+1.)) + ((-0.471726963316-1.17468527863j))*dim
            arg[(0, 3, 1)]=(-0.0701558501916-0.765248010215j)*x[0]**o + ((0.0556830581768+0.700471090002j))*x[0] + ((0.966436059075+0.758012706046j))*x[1]**o + ((0.83904301541+0.761401423176j))*x[1]
            ref[(0, 3, 1)]=(0.896280208883-0.00723530416941j)*(1+2.*(dim-1)/(o+1.)) + ((0.894726073587+1.46187251318j))*dim
            arg[(0, 3, 2)]=(0.940507035561-0.737977089381j)*x[0]**o + ((-0.0454722129373-0.759912042531j))*x[0] + ((0.437870087877+0.469713396791j))*x[1]**o + ((0.580455234392+0.158845166204j))*x[1]
            ref[(0, 3, 2)]=(1.37837712344-0.26826369259j)*(1+2.*(dim-1)/(o+1.)) + ((0.534983021455-0.601066876327j))*dim
            arg[(1, 0, 0)]=(-0.497762591777+0.83941797863j)*x[0]**o + ((0.618371969774-0.170808753486j))*x[0] + ((0.296223306481+0.345409245588j))*x[1]**o + ((0.516977113941-0.703543420113j))*x[1]
            ref[(1, 0, 0)]=(-0.201539285296+1.18482722422j)*(1+2.*(dim-1)/(o+1.)) + ((1.13534908372-0.874352173599j))*dim
            arg[(1, 0, 1)]=(0.982794980511+0.27412148284j)*x[0]**o + ((0.278040495459-0.110722182827j))*x[0] + ((0.556225670877+0.119414683029j))*x[1]**o + ((0.14671870629+0.238876093324j))*x[1]
            ref[(1, 0, 1)]=(1.53902065139+0.39353616587j)*(1+2.*(dim-1)/(o+1.)) + ((0.424759201749+0.128153910497j))*dim
            arg[(1, 0, 2)]=(0.960047476636+0.570158519106j)*x[0]**o + ((0.396620903784+0.97271620087j))*x[0] + ((0.355227722678-0.630412146164j))*x[1]**o + ((0.855186151321-0.219232496386j))*x[1]
            ref[(1, 0, 2)]=(1.31527519931-0.0602536270583j)*(1+2.*(dim-1)/(o+1.)) + ((1.2518070551+0.753483704484j))*dim
            arg[(1, 1, 0)]=(0.324481863923+0.795084369631j)*x[0]**o + ((-0.313622989623+0.782324762728j))*x[0] + ((0.229828110928-0.854904245683j))*x[1]**o + ((0.579200098353-0.445674769373j))*x[1]
            ref[(1, 1, 0)]=(0.554309974852-0.0598198760518j)*(1+2.*(dim-1)/(o+1.)) + ((0.26557710873+0.336649993355j))*dim
            arg[(1, 1, 1)]=(-0.772814968088-0.708343682206j)*x[0]**o + ((-0.346844338462+0.998134676129j))*x[0] + ((-0.0924710941936+0.179869819646j))*x[1]**o + ((-0.1083222054-0.221831603419j))*x[1]
            ref[(1, 1, 1)]=(-0.865286062281-0.52847386256j)*(1+2.*(dim-1)/(o+1.)) + ((-0.455166543862+0.776303072711j))*dim
            arg[(1, 1, 2)]=(0.641927334914-0.204973469431j)*x[0]**o + ((-0.157893432259+0.535297499905j))*x[0] + ((0.97335707866+0.485284790255j))*x[1]**o + ((0.903629933283+0.199220475156j))*x[1]
            ref[(1, 1, 2)]=(1.61528441357+0.280311320824j)*(1+2.*(dim-1)/(o+1.)) + ((0.745736501025+0.734517975061j))*dim
            arg[(1, 2, 0)]=(-0.737286005205+0.990769776369j)*x[0]**o + ((-0.770103666118-0.466741651176j))*x[0] + ((0.585993509914+0.209772955223j))*x[1]**o + ((-0.107277728354+0.719718985894j))*x[1]
            ref[(1, 2, 0)]=(-0.151292495291+1.20054273159j)*(1+2.*(dim-1)/(o+1.)) + ((-0.877381394472+0.252977334718j))*dim
            arg[(1, 2, 1)]=(0.737642181309+0.465911648683j)*x[0]**o + ((-0.04464666003+0.202557879434j))*x[0] + ((-0.276945451492-0.274147173143j))*x[1]**o + ((0.321697027019+0.931838932303j))*x[1]
            ref[(1, 2, 1)]=(0.460696729817+0.19176447554j)*(1+2.*(dim-1)/(o+1.)) + ((0.277050366989+1.13439681174j))*dim
            arg[(1, 2, 2)]=(-0.293937946222-0.154939339268j)*x[0]**o + ((0.490200884764-0.799237433679j))*x[0] + ((-0.695141759073+0.989758566644j))*x[1]**o + ((-0.316412828088+0.944047292218j))*x[1]
            ref[(1, 2, 2)]=(-0.989079705295+0.834819227376j)*(1+2.*(dim-1)/(o+1.)) + ((0.173788056676+0.144809858539j))*dim
            arg[(1, 3, 0)]=(-0.0527730631265+0.721175114501j)*x[0]**o + ((0.201853075677-0.870378248069j))*x[0] + ((0.322402003496-0.69557771445j))*x[1]**o + ((-0.921675032526-0.055576109841j))*x[1]
            ref[(1, 3, 0)]=(0.269628940369+0.0255974000512j)*(1+2.*(dim-1)/(o+1.)) + ((-0.719821956849-0.92595435791j))*dim
            arg[(1, 3, 1)]=(-0.731189802061+0.573269808804j)*x[0]**o + ((0.618297367442-0.506275409112j))*x[0] + ((0.183704439248+0.51672377939j))*x[1]**o + ((-0.956536346795-0.187313630916j))*x[1]
            ref[(1, 3, 1)]=(-0.547485362813+1.08999358819j)*(1+2.*(dim-1)/(o+1.)) + ((-0.338238979353-0.693589040028j))*dim
            arg[(1, 3, 2)]=(-0.571488775525+0.418570545093j)*x[0]**o + ((-0.168848456347+0.60491949394j))*x[0] + ((-0.750649787878-0.300714032996j))*x[1]**o + ((0.445234246263-0.417713694454j))*x[1]
            ref[(1, 3, 2)]=(-1.3221385634+0.117856512097j)*(1+2.*(dim-1)/(o+1.)) + ((0.276385789916+0.187205799486j))*dim
            arg[(2, 0, 0)]=(0.15785415525-0.283796132699j)*x[0]**o + ((0.491235546587-0.776214633343j))*x[0] + ((-0.767973063579+0.76800866711j))*x[1]**o + ((-0.350958536502+0.538548327351j))*x[1]
            ref[(2, 0, 0)]=(-0.610118908329+0.484212534411j)*(1+2.*(dim-1)/(o+1.)) + ((0.140277010085-0.237666305992j))*dim
            arg[(2, 0, 1)]=(-0.259030612855+0.316539933952j)*x[0]**o + ((-0.416750694622-0.367346120874j))*x[0] + ((0.978610649211+0.165363259374j))*x[1]**o + ((-0.610185525121-0.758543616022j))*x[1]
            ref[(2, 0, 1)]=(0.719580036356+0.481903193326j)*(1+2.*(dim-1)/(o+1.)) + ((-1.02693621974-1.1258897369j))*dim
            arg[(2, 0, 2)]=(-0.840436379915-0.575964564508j)*x[0]**o + ((-0.992798254713-0.2049599718j))*x[0] + ((-0.328383373638-0.34593983886j))*x[1]**o + ((0.141885806166+0.511619363947j))*x[1]
            ref[(2, 0, 2)]=(-1.16881975355-0.921904403368j)*(1+2.*(dim-1)/(o+1.)) + ((-0.850912448547+0.306659392148j))*dim
            arg[(2, 1, 0)]=(0.773389485666-0.313584875784j)*x[0]**o + ((-0.24692136203-0.662888139021j))*x[0] + ((-0.500519619759+0.480813517891j))*x[1]**o + ((0.399028174112-0.172114932458j))*x[1]
            ref[(2, 1, 0)]=(0.272869865907+0.167228642107j)*(1+2.*(dim-1)/(o+1.)) + ((0.152106812082-0.835003071479j))*dim
            arg[(2, 1, 1)]=(0.446531395588-0.584539363996j)*x[0]**o + ((-0.943889548667+0.832468690999j))*x[0] + ((-0.78664311297-0.088405882863j))*x[1]**o + ((0.440298229601+0.763205333556j))*x[1]
            ref[(2, 1, 1)]=(-0.340111717382-0.672945246859j)*(1+2.*(dim-1)/(o+1.)) + ((-0.503591319067+1.59567402455j))*dim
            arg[(2, 1, 2)]=(0.0125248850677-0.526308940204j)*x[0]**o + ((-0.286076006105+0.178241953602j))*x[0] + ((-0.78026013472+0.784443188565j))*x[1]**o + ((-0.731190103472+0.347374446755j))*x[1]
            ref[(2, 1, 2)]=(-0.767735249652+0.258134248361j)*(1+2.*(dim-1)/(o+1.)) + ((-1.01726610958+0.525616400357j))*dim
            arg[(2, 2, 0)]=(0.00926742468683+0.840573082145j)*x[0]**o + ((-0.750494207888+0.804682978685j))*x[0] + ((0.996792084368+0.456806618865j))*x[1]**o + ((-0.118332177908-0.667773021582j))*x[1]
            ref[(2, 2, 0)]=(1.00605950905+1.29737970101j)*(1+2.*(dim-1)/(o+1.)) + ((-0.868826385796+0.136909957103j))*dim
            arg[(2, 2, 1)]=(-0.371414641942-0.600502045324j)*x[0]**o + ((-0.318563872324-0.44877450821j))*x[0] + ((0.594973580074+0.00835544268879j))*x[1]**o + ((0.0104627849503+0.147287648885j))*x[1]
            ref[(2, 2, 1)]=(0.223558938132-0.592146602635j)*(1+2.*(dim-1)/(o+1.)) + ((-0.308101087374-0.301486859326j))*dim
            arg[(2, 2, 2)]=(-0.895203135174-0.350409630142j)*x[0]**o + ((-0.482430640208-0.704122620549j))*x[0] + ((0.414588872301-0.260040389442j))*x[1]**o + ((0.981888972941+0.605699973846j))*x[1]
            ref[(2, 2, 2)]=(-0.480614262872-0.610450019584j)*(1+2.*(dim-1)/(o+1.)) + ((0.499458332733-0.0984226467027j))*dim
            arg[(2, 3, 0)]=(-0.854646318612+0.628791182227j)*x[0]**o + ((0.959735407304-0.666861280412j))*x[0] + ((0.497857589347-0.286609365398j))*x[1]**o + ((-0.166004334593-0.0910792343378j))*x[1]
            ref[(2, 3, 0)]=(-0.356788729265+0.342181816829j)*(1+2.*(dim-1)/(o+1.)) + ((0.793731072711-0.75794051475j))*dim
            arg[(2, 3, 1)]=(-0.984589845624-0.279795274236j)*x[0]**o + ((0.125741379901+0.847145125809j))*x[0] + ((-0.931788993854+0.0509926772232j))*x[1]**o + ((-0.287408929265+0.933642397928j))*x[1]
            ref[(2, 3, 1)]=(-1.91637883948-0.228802597013j)*(1+2.*(dim-1)/(o+1.)) + ((-0.161667549364+1.78078752374j))*dim
            arg[(2, 3, 2)]=(-0.788051538858+0.368269592569j)*x[0]**o + ((-0.552941083416-0.498994419937j))*x[0] + ((-0.313501808515+0.19302288711j))*x[1]**o + ((-0.534276818815-0.138025039797j))*x[1]
            ref[(2, 3, 2)]=(-1.10155334737+0.561292479679j)*(1+2.*(dim-1)/(o+1.)) + ((-1.08721790223-0.637019459734j))*dim
            arg[(3, 0, 0)]=(0.924237977124+0.575732534022j)*x[0]**o + ((0.425683448817+0.950280232859j))*x[0] + ((-0.998730061495-0.491892644434j))*x[1]**o + ((0.613497925964+0.86185106435j))*x[1]
            ref[(3, 0, 0)]=(-0.0744920843711+0.0838398895885j)*(1+2.*(dim-1)/(o+1.)) + ((1.03918137478+1.81213129721j))*dim
            arg[(3, 0, 1)]=(0.665489470988+0.172513518924j)*x[0]**o + ((-0.505834525449-0.717031005365j))*x[0] + ((-0.45470103717+0.864136892895j))*x[1]**o + ((0.0400329032813-0.622893009141j))*x[1]
            ref[(3, 0, 1)]=(0.210788433818+1.03665041182j)*(1+2.*(dim-1)/(o+1.)) + ((-0.465801622168-1.33992401451j))*dim
            arg[(3, 0, 2)]=(-0.280285018212-0.65449002977j)*x[0]**o + ((-0.920704362645-0.936737613204j))*x[0] + ((-0.851973854075-0.175054421861j))*x[1]**o + ((0.847033351846+0.971483579572j))*x[1]
            ref[(3, 0, 2)]=(-1.13225887229-0.829544451631j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0736710107989+0.0347459663674j))*dim
            arg[(3, 1, 0)]=(-0.108022424071-0.0107362103115j)*x[0]**o + ((0.687492930338-0.0670198061325j))*x[0] + ((0.0467343850689-0.529308450191j))*x[1]**o + ((0.0178814557885-0.933659790412j))*x[1]
            ref[(3, 1, 0)]=(-0.0612880390025-0.540044660503j)*(1+2.*(dim-1)/(o+1.)) + ((0.705374386126-1.00067959654j))*dim
            arg[(3, 1, 1)]=(-0.246528523067-0.262121480795j)*x[0]**o + ((0.956490089544+0.505188081152j))*x[0] + ((-0.977315838014-0.301045935722j))*x[1]**o + ((0.0794824784061-0.868567624847j))*x[1]
            ref[(3, 1, 1)]=(-1.22384436108-0.563167416517j)*(1+2.*(dim-1)/(o+1.)) + ((1.03597256795-0.363379543694j))*dim
            arg[(3, 1, 2)]=(0.304655392199+0.179022975443j)*x[0]**o + ((-0.903769565308-0.0484112911403j))*x[0] + ((0.261566667926-0.500290185734j))*x[1]**o + ((-0.0173940603492-0.237080593577j))*x[1]
            ref[(3, 1, 2)]=(0.566222060125-0.321267210291j)*(1+2.*(dim-1)/(o+1.)) + ((-0.921163625658-0.285491884717j))*dim
            arg[(3, 2, 0)]=(-0.788418730057+0.53409140864j)*x[0]**o + ((-0.748837009097-0.0613118928879j))*x[0] + ((0.329577332821-0.387788230632j))*x[1]**o + ((-0.487315755364-0.521489181651j))*x[1]
            ref[(3, 2, 0)]=(-0.458841397236+0.146303178008j)*(1+2.*(dim-1)/(o+1.)) + ((-1.23615276446-0.582801074539j))*dim
            arg[(3, 2, 1)]=(0.656344349666+0.800441741285j)*x[0]**o + ((0.393311694772+0.323150828789j))*x[0] + ((0.233995275953-0.36862481076j))*x[1]**o + ((-0.927427387827-0.979651140169j))*x[1]
            ref[(3, 2, 1)]=(0.890339625619+0.431816930524j)*(1+2.*(dim-1)/(o+1.)) + ((-0.534115693055-0.656500311381j))*dim
            arg[(3, 2, 2)]=(0.122310327866-0.319789162549j)*x[0]**o + ((-0.876295269235+0.153384545072j))*x[0] + ((0.614011318493+0.599095532033j))*x[1]**o + ((0.212727198659+0.205722253603j))*x[1]
            ref[(3, 2, 2)]=(0.736321646359+0.279306369484j)*(1+2.*(dim-1)/(o+1.)) + ((-0.663568070576+0.359106798674j))*dim
            arg[(3, 3, 0)]=(0.245423756457-0.881553427455j)*x[0]**o + ((0.00337098156858-0.559187056352j))*x[0] + ((-0.758369129334-0.65629762173j))*x[1]**o + ((-0.549026019135+0.785232601228j))*x[1]
            ref[(3, 3, 0)]=(-0.512945372877-1.53785104919j)*(1+2.*(dim-1)/(o+1.)) + ((-0.545655037567+0.226045544876j))*dim
            arg[(3, 3, 1)]=(-0.949749497829-0.583256686554j)*x[0]**o + ((-0.792237358502-0.529027772628j))*x[0] + ((-0.742057338661+0.388354757391j))*x[1]**o + ((0.14189842012-0.568991905872j))*x[1]
            ref[(3, 3, 1)]=(-1.69180683649-0.194901929163j)*(1+2.*(dim-1)/(o+1.)) + ((-0.650338938382-1.0980196785j))*dim
            arg[(3, 3, 2)]=(0.182397301817-0.597206318135j)*x[0]**o + ((-0.468292831461-0.324136722947j))*x[0] + ((0.51127712413+0.986385957106j))*x[1]**o + ((0.690352399998-0.283902138953j))*x[1]
            ref[(3, 3, 2)]=(0.693674425948+0.389179638971j)*(1+2.*(dim-1)/(o+1.)) + ((0.222059568537-0.6080388619j))*dim
        else:
            arg[(0, 0, 0)]=(-0.32524260558+0.963329498514j)*x[0]**o + ((-0.417741830228+0.118273286422j))*x[0] + ((-0.259822453079+0.374417961334j))*x[1]**o + ((0.553894076421-0.167370678201j))*x[1] + ((0.260300408012+0.205573275402j))*x[2]**o + ((-0.117797051969-0.73002173434j))*x[2]
            ref[(0, 0, 0)]=(-0.324764650647+1.54332073525j)*(1+2.*(dim-1)/(o+1.)) + ((0.0183551942239-0.779119126119j))*dim
            arg[(0, 0, 1)]=(0.123342765874+0.548517381167j)*x[0]**o + ((0.00979914104191+0.831716034387j))*x[0] + ((0.835917219141+0.00824032254262j))*x[1]**o + ((0.057015214431-0.630804158542j))*x[1] + ((0.994433131476-0.0429079620085j))*x[2]**o + ((-0.763202254207-0.927555638933j))*x[2]
            ref[(0, 0, 1)]=(1.95369311649+0.513849741701j)*(1+2.*(dim-1)/(o+1.)) + ((-0.696387898734-0.726643763088j))*dim
            arg[(0, 0, 2)]=(0.501273681531+0.660762147127j)*x[0]**o + ((0.209852541073+0.264725542996j))*x[0] + ((-0.87178924186-0.272207291884j))*x[1]**o + ((-0.407437423714-0.458646446844j))*x[1] + ((-0.585616616862+0.091550314899j))*x[2]**o + ((0.367640700423+0.986268235721j))*x[2]
            ref[(0, 0, 2)]=(-0.956132177191+0.480105170142j)*(1+2.*(dim-1)/(o+1.)) + ((0.170055817782+0.792347331874j))*dim
            arg[(0, 1, 0)]=(-0.764003579764+0.0186596526732j)*x[0]**o + ((0.18176555656+0.182531804395j))*x[0] + ((-0.365865177082-0.477246026369j))*x[1]**o + ((0.32909659568+0.923890910348j))*x[1] + ((0.497151040409-0.221971487295j))*x[2]**o + ((0.872213566663-0.0487440979557j))*x[2]
            ref[(0, 1, 0)]=(-0.632717716438-0.680557860991j)*(1+2.*(dim-1)/(o+1.)) + ((1.3830757189+1.05767861679j))*dim
            arg[(0, 1, 1)]=(-0.389455996337-0.93318779957j)*x[0]**o + ((0.10350601051+0.97925356363j))*x[0] + ((0.196100079485+0.711168846183j))*x[1]**o + ((-0.864554579443+0.0943075014634j))*x[1] + ((-0.150163509872+0.849837265226j))*x[2]**o + ((-0.831844858712+0.39622702499j))*x[2]
            ref[(0, 1, 1)]=(-0.343519426723+0.627818311839j)*(1+2.*(dim-1)/(o+1.)) + ((-1.59289342765+1.46978809008j))*dim
            arg[(0, 1, 2)]=(-0.425172251591-0.300383466915j)*x[0]**o + ((0.858424427025-0.33490450144j))*x[0] + ((0.513460551021+0.906934427795j))*x[1]**o + ((-0.610319232736-0.484000022862j))*x[1] + ((0.0943904301727-0.646644493507j))*x[2]**o + ((0.16626956263+0.642557677654j))*x[2]
            ref[(0, 1, 2)]=(0.182678729603-0.0400935326272j)*(1+2.*(dim-1)/(o+1.)) + ((0.41437475692-0.176346846648j))*dim
            arg[(0, 2, 0)]=(-0.445546653325+0.781408364825j)*x[0]**o + ((0.0410900704823+0.635202431422j))*x[0] + ((-0.705183809217+0.169024736592j))*x[1]**o + ((0.721159525939+0.293257133108j))*x[1] + ((-0.800381024696-0.892728435106j))*x[2]**o + ((-0.782856807006+0.529171911668j))*x[2]
            ref[(0, 2, 0)]=(-1.95111148724+0.0577046663109j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0206072105846+1.4576314762j))*dim
            arg[(0, 2, 1)]=(-0.791706217594-0.417014740546j)*x[0]**o + ((-0.662202641233+0.41087819655j))*x[0] + ((-0.783077582498-0.666351739641j))*x[1]**o + ((0.290603093721+0.0402003136507j))*x[1] + ((-0.672464142674+0.625533741995j))*x[2]**o + ((-0.941199870952+0.730687064355j))*x[2]
            ref[(0, 2, 1)]=(-2.24724794277-0.457832738191j)*(1+2.*(dim-1)/(o+1.)) + ((-1.31279941846+1.18176557456j))*dim
            arg[(0, 2, 2)]=(-0.0950218702392-0.758187089477j)*x[0]**o + ((-0.0771813034352-0.413238074805j))*x[0] + ((-0.88507398616-0.740648476211j))*x[1]**o + ((0.881340991869-0.780245016812j))*x[1] + ((-0.0938919347347+0.0855192902462j))*x[2]**o + ((-0.588324006929+0.953122576505j))*x[2]
            ref[(0, 2, 2)]=(-1.07398779113-1.41331627544j)*(1+2.*(dim-1)/(o+1.)) + ((0.215835681505-0.240360515113j))*dim
            arg[(0, 3, 0)]=(0.925984545623-0.00210813570644j)*x[0]**o + ((0.76117530311+0.819111028201j))*x[0] + ((0.507062596503-0.260815174794j))*x[1]**o + ((0.575673107088+0.497912201691j))*x[1] + ((0.316060925065+0.220189233161j))*x[2]**o + ((0.405312478567-0.904804858585j))*x[2]
            ref[(0, 3, 0)]=(1.74910806719-0.0427340773391j)*(1+2.*(dim-1)/(o+1.)) + ((1.74216088876+0.412218371307j))*dim
            arg[(0, 3, 1)]=(-0.0874193080521+0.408529821311j)*x[0]**o + ((0.629110967511-0.919014610893j))*x[0] + ((-0.348683250473+0.151730624511j))*x[1]**o + ((-0.828832697925-0.588629856358j))*x[1] + ((0.645313814048+0.262351103231j))*x[2]**o + ((-0.662358503584+0.578942726336j))*x[2]
            ref[(0, 3, 1)]=(0.209211255523+0.822611549053j)*(1+2.*(dim-1)/(o+1.)) + ((-0.862080233998-0.928701740916j))*dim
            arg[(0, 3, 2)]=(-0.126328550786+0.915926266026j)*x[0]**o + ((-0.352578435823-0.61987710688j))*x[0] + ((-0.78153618836+0.50281505847j))*x[1]**o + ((-0.538241856756+0.910024033506j))*x[1] + ((-0.354906750274+0.355987286153j))*x[2]**o + ((0.0763783205311+0.156978342145j))*x[2]
            ref[(0, 3, 2)]=(-1.26277148942+1.77472861065j)*(1+2.*(dim-1)/(o+1.)) + ((-0.814441972048+0.447125268771j))*dim
            arg[(1, 0, 0)]=(0.0948384335522-0.357552459781j)*x[0]**o + ((-0.399419839837-0.443485228346j))*x[0] + ((-0.701377325988-0.837079739539j))*x[1]**o + ((-0.75811582873-0.0775371087613j))*x[1] + ((-0.856762417072+0.525316182006j))*x[2]**o + ((0.731418523103+0.0109387496671j))*x[2]
            ref[(1, 0, 0)]=(-1.46330130951-0.669316017314j)*(1+2.*(dim-1)/(o+1.)) + ((-0.426117145464-0.51008358744j))*dim
            arg[(1, 0, 1)]=(-0.317867262102+0.142607627884j)*x[0]**o + ((0.435053634382+0.124444976588j))*x[0] + ((0.514517276665-0.962232278158j))*x[1]**o + ((0.758663838472-0.76785067789j))*x[1] + ((0.712778444694+0.221116394046j))*x[2]**o + ((0.653715571652-0.692978270332j))*x[2]
            ref[(1, 0, 1)]=(0.909428459258-0.598508256228j)*(1+2.*(dim-1)/(o+1.)) + ((1.84743304451-1.33638397163j))*dim
            arg[(1, 0, 2)]=(0.954842862473-0.40298469627j)*x[0]**o + ((-0.118237959571-0.336837444629j))*x[0] + ((0.169805044332+0.194994766867j))*x[1]**o + ((-0.544065173566+0.697694770064j))*x[1] + ((-0.351472439835+0.963274007033j))*x[2]**o + ((-0.3472028866+0.472911596047j))*x[2]
            ref[(1, 0, 2)]=(0.77317546697+0.75528407763j)*(1+2.*(dim-1)/(o+1.)) + ((-1.00950601974+0.833768921482j))*dim
            arg[(1, 1, 0)]=(0.640018657057+0.645101647171j)*x[0]**o + ((0.591061756726+0.31620553588j))*x[0] + ((0.0975782400543+0.637525352776j))*x[1]**o + ((0.874270331748+0.780788052923j))*x[1] + ((-0.0496768101814-0.158218904811j))*x[2]**o + ((-0.214478591244+0.201372753833j))*x[2]
            ref[(1, 1, 0)]=(0.68792008693+1.12440809514j)*(1+2.*(dim-1)/(o+1.)) + ((1.25085349723+1.29836634264j))*dim
            arg[(1, 1, 1)]=(-0.128767076207-0.67529750371j)*x[0]**o + ((0.0876541899961-0.0742182975664j))*x[0] + ((-0.753273676267+0.744601198721j))*x[1]**o + ((-0.238531015823+0.232183831179j))*x[1] + ((0.0349502763246+0.540302586977j))*x[2]**o + ((-0.748968623807+0.386230711107j))*x[2]
            ref[(1, 1, 1)]=(-0.847090476149+0.609606281988j)*(1+2.*(dim-1)/(o+1.)) + ((-0.899845449634+0.544196244719j))*dim
            arg[(1, 1, 2)]=(0.658892200417-0.317610981974j)*x[0]**o + ((-0.939833523903+0.351815807959j))*x[0] + ((0.643649901624+0.0851242048236j))*x[1]**o + ((-0.949961953558-0.137059014197j))*x[1] + ((-0.46794235311+0.322402097137j))*x[2]**o + ((0.899798721409-0.192544844089j))*x[2]
            ref[(1, 1, 2)]=(0.834599748931+0.0899153199868j)*(1+2.*(dim-1)/(o+1.)) + ((-0.989996756052+0.0222119496729j))*dim
            arg[(1, 2, 0)]=(-0.481727155733+0.548457482738j)*x[0]**o + ((0.278801908198+0.955140446594j))*x[0] + ((0.316934832125+0.545261440455j))*x[1]**o + ((0.879916045892+0.322082265397j))*x[1] + ((0.600022156714-0.327844253706j))*x[2]**o + ((0.167359143534-0.921661639902j))*x[2]
            ref[(1, 2, 0)]=(0.435229833106+0.765874669487j)*(1+2.*(dim-1)/(o+1.)) + ((1.32607709762+0.35556107209j))*dim
            arg[(1, 2, 1)]=(-0.747973598208+0.554736236611j)*x[0]**o + ((0.773294262398-0.347544135456j))*x[0] + ((0.914199009915-0.301445663703j))*x[1]**o + ((-0.817401859929+0.391597278935j))*x[1] + ((0.52912497113-0.219015960093j))*x[2]**o + ((0.224707915478+0.0454862925455j))*x[2]
            ref[(1, 2, 1)]=(0.695350382837+0.0342746128154j)*(1+2.*(dim-1)/(o+1.)) + ((0.180600317947+0.0895394360241j))*dim
            arg[(1, 2, 2)]=(-0.134831204309+0.282991474177j)*x[0]**o + ((-0.878925238301+0.980492173855j))*x[0] + ((-0.161822910345-0.735471854782j))*x[1]**o + ((0.375793811496-0.839600324486j))*x[1] + ((-0.293889490558-0.278596333243j))*x[2]**o + ((0.192259478538+0.239965193057j))*x[2]
            ref[(1, 2, 2)]=(-0.590543605212-0.731076713849j)*(1+2.*(dim-1)/(o+1.)) + ((-0.310871948267+0.380857042425j))*dim
            arg[(1, 3, 0)]=(-0.107809299738+0.384250901764j)*x[0]**o + ((0.468882549044+0.0154766474759j))*x[0] + ((-0.542717528289+0.640012256612j))*x[1]**o + ((-0.28392841917-0.165066525911j))*x[1] + ((0.676425425731+0.00710707703592j))*x[2]**o + ((-0.27827044158+0.400096028915j))*x[2]
            ref[(1, 3, 0)]=(0.0258985977045+1.03137023541j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0933163117055+0.25050615048j))*dim
            arg[(1, 3, 1)]=(-0.309772509719+0.894765772225j)*x[0]**o + ((0.321313473729-0.497616668227j))*x[0] + ((0.976285077104+0.363645906149j))*x[1]**o + ((0.00380525357468+0.702683621735j))*x[1] + ((0.110261570805+0.972850943431j))*x[2]**o + ((-0.921270392204-0.407844766078j))*x[2]
            ref[(1, 3, 1)]=(0.77677413819+2.2312626218j)*(1+2.*(dim-1)/(o+1.)) + ((-0.596151664901-0.202777812569j))*dim
            arg[(1, 3, 2)]=(-0.0182480871252-0.0977730286766j)*x[0]**o + ((0.733059913833+0.461043076181j))*x[0] + ((0.942084647842-0.596894847587j))*x[1]**o + ((0.401474040665+0.12644445495j))*x[1] + ((-0.858314124894-0.469930866696j))*x[2]**o + ((-0.688369043929+0.408442569905j))*x[2]
            ref[(1, 3, 2)]=(0.0655224358224-1.16459874296j)*(1+2.*(dim-1)/(o+1.)) + ((0.446164910569+0.995930101035j))*dim
            arg[(2, 0, 0)]=(-0.933294620946-0.906174163024j)*x[0]**o + ((-0.812214999014-0.15738752833j))*x[0] + ((-0.784187623612+0.649511430319j))*x[1]**o + ((-0.233977068293-0.783591228468j))*x[1] + ((-0.390199148171-0.187760872435j))*x[2]**o + ((-0.902554865968+0.133084020098j))*x[2]
            ref[(2, 0, 0)]=(-2.10768139273-0.444423605141j)*(1+2.*(dim-1)/(o+1.)) + ((-1.94874693327-0.8078947367j))*dim
            arg[(2, 0, 1)]=(-0.847356240998+0.0803345161349j)*x[0]**o + ((0.77818480305-0.727061179336j))*x[0] + ((-0.954028846708+0.29855789379j))*x[1]**o + ((-0.733454614979+0.565067781613j))*x[1] + ((-0.431958594855-0.112158920232j))*x[2]**o + ((-0.213975953554-0.425045432089j))*x[2]
            ref[(2, 0, 1)]=(-2.23334368256+0.266733489693j)*(1+2.*(dim-1)/(o+1.)) + ((-0.169245765483-0.587038829812j))*dim
            arg[(2, 0, 2)]=(-0.69563936736-0.364629643389j)*x[0]**o + ((-0.78578971034+0.382834607669j))*x[0] + ((-0.879424617879+0.683540410091j))*x[1]**o + ((0.38633309571+0.34637530906j))*x[1] + ((0.336247937139+0.489871275726j))*x[2]**o + ((-0.592070862923+0.438474969246j))*x[2]
            ref[(2, 0, 2)]=(-1.2388160481+0.808782042428j)*(1+2.*(dim-1)/(o+1.)) + ((-0.991527477554+1.16768488598j))*dim
            arg[(2, 1, 0)]=(0.560454595309+0.136724517346j)*x[0]**o + ((0.571618555187-0.794662317725j))*x[0] + ((-0.231329090502-0.202168490979j))*x[1]**o + ((-0.593841937131-0.589368715043j))*x[1] + ((-0.959433197337+0.565784980417j))*x[2]**o + ((-0.29300492444+0.72352581681j))*x[2]
            ref[(2, 1, 0)]=(-0.630307692529+0.500341006784j)*(1+2.*(dim-1)/(o+1.)) + ((-0.315228306384-0.660505215958j))*dim
            arg[(2, 1, 1)]=(-0.978905401166+0.215609233616j)*x[0]**o + ((-0.886584825278+0.795555952988j))*x[0] + ((0.671537445115+0.631726647603j))*x[1]**o + ((0.802597600433+0.971006135102j))*x[1] + ((0.551046350299-0.274395740608j))*x[2]**o + ((0.331828481341-0.365128387658j))*x[2]
            ref[(2, 1, 1)]=(0.243678394248+0.57294014061j)*(1+2.*(dim-1)/(o+1.)) + ((0.247841256496+1.40143370043j))*dim
            arg[(2, 1, 2)]=(0.0361739068396+0.100314239811j)*x[0]**o + ((-0.744669768578-0.578639702826j))*x[0] + ((-0.120637095001+0.729550384085j))*x[1]**o + ((0.568805439486-0.299205970674j))*x[1] + ((-0.846331857716-0.885581980174j))*x[2]**o + ((0.596702706921-0.384635444531j))*x[2]
            ref[(2, 1, 2)]=(-0.930795045878-0.0557173562781j)*(1+2.*(dim-1)/(o+1.)) + ((0.420838377829-1.26248111803j))*dim
            arg[(2, 2, 0)]=(-0.38197589789-0.560237319976j)*x[0]**o + ((-0.788052533787-0.0305235322669j))*x[0] + ((-0.080100719433-0.102176414543j))*x[1]**o + ((0.685370435943-0.554255917815j))*x[1] + ((-0.0572012454832-0.25963408599j))*x[2]**o + ((-0.14208982066-0.158257056278j))*x[2]
            ref[(2, 2, 0)]=(-0.519277862806-0.922047820509j)*(1+2.*(dim-1)/(o+1.)) + ((-0.244771918505-0.743036506361j))*dim
            arg[(2, 2, 1)]=(-0.337185820004-0.0970280167512j)*x[0]**o + ((-0.978084560828+0.769098604693j))*x[0] + ((0.306998091425+0.0664823013778j))*x[1]**o + ((-0.206063462696+0.18654063225j))*x[1] + ((-0.487233079353-0.0697508130196j))*x[2]**o + ((0.853208098322+0.673222468615j))*x[2]
            ref[(2, 2, 1)]=(-0.517420807932-0.100296528393j)*(1+2.*(dim-1)/(o+1.)) + ((-0.330939925201+1.62886170556j))*dim
            arg[(2, 2, 2)]=(-0.0862984173912-0.421885767458j)*x[0]**o + ((0.973654617608+0.594940973571j))*x[0] + ((-0.620285602843-0.192017526948j))*x[1]**o + ((0.645585522433+0.401734411207j))*x[1] + ((0.00813186851583+0.249043791731j))*x[2]**o + ((0.123567647007+0.690151160583j))*x[2]
            ref[(2, 2, 2)]=(-0.698452151719-0.364859502674j)*(1+2.*(dim-1)/(o+1.)) + ((1.74280778705+1.68682654536j))*dim
            arg[(2, 3, 0)]=(-0.627745045905-0.00721547676493j)*x[0]**o + ((0.762295678171+0.961348843262j))*x[0] + ((-0.616399294199+0.453608796687j))*x[1]**o + ((-0.281186891153+0.0998368950383j))*x[1] + ((0.194833837267+0.644587678225j))*x[2]**o + ((-0.248961571827-0.0957052799659j))*x[2]
            ref[(2, 3, 0)]=(-1.04931050284+1.09098099815j)*(1+2.*(dim-1)/(o+1.)) + ((0.23214721519+0.965480458335j))*dim
            arg[(2, 3, 1)]=(-0.821476168165-0.430233591393j)*x[0]**o + ((-0.491085314892-0.370189833777j))*x[0] + ((0.875423884224+0.792798941884j))*x[1]**o + ((0.144582859441-0.99274040882j))*x[1] + ((-0.766622155728+0.867960870029j))*x[2]**o + ((-0.0720302998521+0.28775220397j))*x[2]
            ref[(2, 3, 1)]=(-0.712674439669+1.23052622052j)*(1+2.*(dim-1)/(o+1.)) + ((-0.418532755302-1.07517803863j))*dim
            arg[(2, 3, 2)]=(-0.245312078056+0.593553975111j)*x[0]**o + ((0.267109390096+0.189860821917j))*x[0] + ((-0.753035017835-0.752952241262j))*x[1]**o + ((-0.0452691384354-0.445563327292j))*x[1] + ((-0.296376257726-0.643073789118j))*x[2]**o + ((-0.279277126421+0.492530123371j))*x[2]
            ref[(2, 3, 2)]=(-1.29472335362-0.802472055268j)*(1+2.*(dim-1)/(o+1.)) + ((-0.05743687476+0.236827617995j))*dim
            arg[(3, 0, 0)]=(-0.0241254432206+0.688106386918j)*x[0]**o + ((-0.722760754087-0.197778387052j))*x[0] + ((-0.140116654564+0.680890190588j))*x[1]**o + ((0.214598750266+0.649158357018j))*x[1] + ((-0.442007967501+0.409284494974j))*x[2]**o + ((0.0665357668607+0.626119328628j))*x[2]
            ref[(3, 0, 0)]=(-0.606250065285+1.77828107248j)*(1+2.*(dim-1)/(o+1.)) + ((-0.44162623696+1.07749929859j))*dim
            arg[(3, 0, 1)]=(-0.465270677232-0.211319038039j)*x[0]**o + ((-0.88163849338-0.603084638145j))*x[0] + ((0.776934614184-0.331776225957j))*x[1]**o + ((0.975604071337+0.638655061202j))*x[1] + ((0.440341345107-0.516893212998j))*x[2]**o + ((-0.0456905609986-0.465068065467j))*x[2]
            ref[(3, 0, 1)]=(0.75200528206-1.059988477j)*(1+2.*(dim-1)/(o+1.)) + ((0.0482750169582-0.42949764241j))*dim
            arg[(3, 0, 2)]=(-0.238023398521-0.346259151283j)*x[0]**o + ((0.990275139521+0.703211735404j))*x[0] + ((-0.594424870864+0.980309270087j))*x[1]**o + ((0.945046951854-0.888752975843j))*x[1] + ((0.919122256893+0.0546039428799j))*x[2]**o + ((0.46060756524-0.777745578419j))*x[2]
            ref[(3, 0, 2)]=(0.0866739875084+0.688654061683j)*(1+2.*(dim-1)/(o+1.)) + ((2.39592965661-0.963286818858j))*dim
            arg[(3, 1, 0)]=(0.513780741885+0.887373483258j)*x[0]**o + ((0.0472454990289-0.974055510038j))*x[0] + ((-0.314195462694+0.249060823164j))*x[1]**o + ((0.462522600891-0.33816351464j))*x[1] + ((0.0772903177842-0.477759892204j))*x[2]**o + ((0.26226206808+0.592109714656j))*x[2]
            ref[(3, 1, 0)]=(0.276875596975+0.658674414219j)*(1+2.*(dim-1)/(o+1.)) + ((0.772030168-0.720109310022j))*dim
            arg[(3, 1, 1)]=(0.213030220579-0.910727327026j)*x[0]**o + ((-0.330592990974-0.545201309695j))*x[0] + ((-0.867259720961-0.709854477595j))*x[1]**o + ((0.04079073342-0.817802511194j))*x[1] + ((-0.0725250598276+0.708338683459j))*x[2]**o + ((-0.31462807447+0.0404727742493j))*x[2]
            ref[(3, 1, 1)]=(-0.72675456021-0.912243121162j)*(1+2.*(dim-1)/(o+1.)) + ((-0.604430332024-1.32253104664j))*dim
            arg[(3, 1, 2)]=(0.851274780231+0.186215121762j)*x[0]**o + ((0.826801373241+0.861948188667j))*x[0] + ((-0.171800791175-0.634706143093j))*x[1]**o + ((-0.34119519902+0.352775723979j))*x[1] + ((0.0389335222379-0.530145911895j))*x[2]**o + ((0.409161670086-0.808177716353j))*x[2]
            ref[(3, 1, 2)]=(0.718407511294-0.978636933226j)*(1+2.*(dim-1)/(o+1.)) + ((0.894767844307+0.406546196293j))*dim
            arg[(3, 2, 0)]=(-0.510454684483-0.734565856603j)*x[0]**o + ((-0.4219177824-0.402230523306j))*x[0] + ((-0.482470320516+0.162018223899j))*x[1]**o + ((-0.694548766986-0.749926158102j))*x[1] + ((-0.0395589933284+0.40922930146j))*x[2]**o + ((0.849984076351-0.324316985447j))*x[2]
            ref[(3, 2, 0)]=(-1.03248399833-0.163318331244j)*(1+2.*(dim-1)/(o+1.)) + ((-0.266482473035-1.47647366686j))*dim
            arg[(3, 2, 1)]=(0.937321380411-0.333059165868j)*x[0]**o + ((0.833311280961+0.55291905518j))*x[0] + ((0.89044809914+0.413655280227j))*x[1]**o + ((-0.881090469399-0.69080568479j))*x[1] + ((0.0743849623449-0.724952884051j))*x[2]**o + ((0.452338942789+0.754735207683j))*x[2]
            ref[(3, 2, 1)]=(1.9021544419-0.644356769692j)*(1+2.*(dim-1)/(o+1.)) + ((0.404559754352+0.616848578073j))*dim
            arg[(3, 2, 2)]=(-0.855477909376+0.335927035187j)*x[0]**o + ((-0.275884270324-0.533291179569j))*x[0] + ((-0.856803945708+0.467233673305j))*x[1]**o + ((-0.845852714122+0.0236681413819j))*x[1] + ((0.004001344389+0.627005928353j))*x[2]**o + ((0.585950062653-0.339895882371j))*x[2]
            ref[(3, 2, 2)]=(-1.70828051069+1.43016663685j)*(1+2.*(dim-1)/(o+1.)) + ((-0.535786921793-0.849518920558j))*dim
            arg[(3, 3, 0)]=(0.298147775758+0.79942221579j)*x[0]**o + ((0.937925489499+0.683321703348j))*x[0] + ((-0.482359196281-0.709030743897j))*x[1]**o + ((-0.141713802569-0.805328909415j))*x[1] + ((-0.686370632229+0.332865200651j))*x[2]**o + ((0.76559725667+0.177116043174j))*x[2]
            ref[(3, 3, 0)]=(-0.870582052752+0.423256672544j)*(1+2.*(dim-1)/(o+1.)) + ((1.5618089436+0.0551088371073j))*dim
            arg[(3, 3, 1)]=(0.560255564426+0.77984608872j)*x[0]**o + ((0.827020432263+0.185010724606j))*x[0] + ((0.722292248261+0.122230439739j))*x[1]**o + ((0.0692847979569+0.82960093361j))*x[1] + ((0.234856834383-0.485639198229j))*x[2]**o + ((0.0819457273342-0.00374079504334j))*x[2]
            ref[(3, 3, 1)]=(1.51740464707+0.416437330231j)*(1+2.*(dim-1)/(o+1.)) + ((0.978250957554+1.01087086317j))*dim
            arg[(3, 3, 2)]=(-0.512040190606-0.886597452687j)*x[0]**o + ((0.552558594536+0.441346350112j))*x[0] + ((0.899090917494+0.919536212754j))*x[1]**o + ((0.31186822299-0.755856016964j))*x[1] + ((-0.0653072248025+0.305383702006j))*x[2]**o + ((0.763428672405-0.18191566562j))*x[2]
            ref[(3, 3, 2)]=(0.321743502086+0.338322462072j)*(1+2.*(dim-1)/(o+1.)) + ((1.62785548993-0.496425332473j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 4, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_ContinuousFunction_rank4(self):
        """
        tests integral of rank 4 Data on FunctionOnBoundary

        assumptions: ContinuousFunction(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = ContinuousFunction(self.domain)
        x = w.getX()
        arg = Data(0,(2, 4, 3, 4),w)
        ref=numpy.zeros((2, 4, 3, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(-0.736688000117-0.579482819206j)*x[0]**o + ((-0.905024768493+0.438196538173j))*x[0] + ((0.602239546664+0.013673457225j))*x[1]**o + ((0.859013740163-0.510143427932j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.134448453453-0.565809361981j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0460110283304-0.0719468897591j))*dim
            arg[(0, 0, 0, 1)]=(-0.838893921741+0.46026582278j)*x[0]**o + ((0.993709502339+0.462152056336j))*x[0] + ((0.230973933636+0.0181352875221j))*x[1]**o + ((0.095116340062-0.614348803638j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.607919988105+0.478401110302j)*(1+2.*(dim-1)/(o+1.)) + ((1.0888258424-0.152196747302j))*dim
            arg[(0, 0, 0, 2)]=(0.382458873479-0.509891102983j)*x[0]**o + ((0.468551858928+0.494588095364j))*x[0] + ((0.420469024456+0.652522441584j))*x[1]**o + ((0.519888663751-0.300097981198j))*x[1]
            ref[(0, 0, 0, 2)]=(0.802927897935+0.142631338601j)*(1+2.*(dim-1)/(o+1.)) + ((0.988440522679+0.194490114166j))*dim
            arg[(0, 0, 0, 3)]=(-0.0551340007259-0.629178644153j)*x[0]**o + ((-0.747303111834-0.981308630607j))*x[0] + ((0.885960692334+0.664119604634j))*x[1]**o + ((0.0317725832479-0.715760893849j))*x[1]
            ref[(0, 0, 0, 3)]=(0.830826691608+0.0349409604805j)*(1+2.*(dim-1)/(o+1.)) + ((-0.715530528586-1.69706952446j))*dim
            arg[(0, 0, 1, 0)]=(0.512466209069+0.708132305166j)*x[0]**o + ((-0.252182267306-0.798801453027j))*x[0] + ((-0.220212275372-0.0652894847659j))*x[1]**o + ((-0.709615239291+0.544281870559j))*x[1]
            ref[(0, 0, 1, 0)]=(0.292253933697+0.6428428204j)*(1+2.*(dim-1)/(o+1.)) + ((-0.961797506597-0.254519582468j))*dim
            arg[(0, 0, 1, 1)]=(-0.834620803036-0.657702222073j)*x[0]**o + ((-0.931781105192+0.807416074388j))*x[0] + ((-0.0220252925127+0.606724579655j))*x[1]**o + ((0.578309815945-0.941665156851j))*x[1]
            ref[(0, 0, 1, 1)]=(-0.856646095549-0.0509776424173j)*(1+2.*(dim-1)/(o+1.)) + ((-0.353471289248-0.134249082463j))*dim
            arg[(0, 0, 1, 2)]=(0.984214091458-0.174345775657j)*x[0]**o + ((0.336518728355+0.832440344152j))*x[0] + ((0.889877440465+0.0714151783676j))*x[1]**o + ((0.869007306651+0.740441155114j))*x[1]
            ref[(0, 0, 1, 2)]=(1.87409153192-0.102930597289j)*(1+2.*(dim-1)/(o+1.)) + ((1.20552603501+1.57288149927j))*dim
            arg[(0, 0, 1, 3)]=(-0.529809744367-0.559720266663j)*x[0]**o + ((-0.458610828992-0.00362915141365j))*x[0] + ((-0.183277226398-0.0512686078193j))*x[1]**o + ((-0.293545059139-0.034137327459j))*x[1]
            ref[(0, 0, 1, 3)]=(-0.713086970765-0.610988874482j)*(1+2.*(dim-1)/(o+1.)) + ((-0.752155888131-0.0377664788726j))*dim
            arg[(0, 0, 2, 0)]=(0.793860055506+0.364130936494j)*x[0]**o + ((0.913856475542-0.212145943335j))*x[0] + ((-0.783673766511+0.235500462321j))*x[1]**o + ((0.583548353713+0.917471673303j))*x[1]
            ref[(0, 0, 2, 0)]=(0.0101862889948+0.599631398815j)*(1+2.*(dim-1)/(o+1.)) + ((1.49740482925+0.705325729968j))*dim
            arg[(0, 0, 2, 1)]=(-0.539220793332-0.616703741018j)*x[0]**o + ((0.522254675697-0.890575000088j))*x[0] + ((0.0506388023966+0.976196118736j))*x[1]**o + ((0.699919740337-0.885400943551j))*x[1]
            ref[(0, 0, 2, 1)]=(-0.488581990935+0.359492377718j)*(1+2.*(dim-1)/(o+1.)) + ((1.22217441603-1.77597594364j))*dim
            arg[(0, 0, 2, 2)]=(-0.376171004424-0.319656123868j)*x[0]**o + ((0.677225769303+0.506935911935j))*x[0] + ((0.0493093679358+0.860359244361j))*x[1]**o + ((0.855058047249+0.788997164236j))*x[1]
            ref[(0, 0, 2, 2)]=(-0.326861636488+0.540703120493j)*(1+2.*(dim-1)/(o+1.)) + ((1.53228381655+1.29593307617j))*dim
            arg[(0, 0, 2, 3)]=(-0.674450169266+0.180804628132j)*x[0]**o + ((-0.329332807717-0.639287292238j))*x[0] + ((-0.0544895268866+0.0348801865729j))*x[1]**o + ((-0.927326981681+0.995588667991j))*x[1]
            ref[(0, 0, 2, 3)]=(-0.728939696152+0.215684814704j)*(1+2.*(dim-1)/(o+1.)) + ((-1.2566597894+0.356301375753j))*dim
            arg[(0, 1, 0, 0)]=(0.0993064359753-0.203438681918j)*x[0]**o + ((-0.326575478564+0.237642542683j))*x[0] + ((-0.0948348152365+0.691649388287j))*x[1]**o + ((0.682677173953-0.266722970232j))*x[1]
            ref[(0, 1, 0, 0)]=(0.00447162073886+0.488210706369j)*(1+2.*(dim-1)/(o+1.)) + ((0.356101695389-0.0290804275495j))*dim
            arg[(0, 1, 0, 1)]=(-0.26661368418-0.559795469483j)*x[0]**o + ((-0.25846784391+0.00324584601723j))*x[0] + ((-0.190912455011+0.693588803757j))*x[1]**o + ((-0.170410148794-0.625007312711j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.457526139191+0.133793334274j)*(1+2.*(dim-1)/(o+1.)) + ((-0.428877992704-0.621761466694j))*dim
            arg[(0, 1, 0, 2)]=(-0.0273740528159-0.193908443758j)*x[0]**o + ((-0.308706194487+0.630868110547j))*x[0] + ((0.557692177199-0.00608709016041j))*x[1]**o + ((-0.535081839276-0.700317192058j))*x[1]
            ref[(0, 1, 0, 2)]=(0.530318124383-0.199995533918j)*(1+2.*(dim-1)/(o+1.)) + ((-0.843788033763-0.0694490815117j))*dim
            arg[(0, 1, 0, 3)]=(0.586161803878+0.935844937629j)*x[0]**o + ((-0.403647829828-0.366538558335j))*x[0] + ((0.0648158089573-0.77498678382j))*x[1]**o + ((-0.814885162841-0.645242634989j))*x[1]
            ref[(0, 1, 0, 3)]=(0.650977612836+0.160858153809j)*(1+2.*(dim-1)/(o+1.)) + ((-1.21853299267-1.01178119332j))*dim
            arg[(0, 1, 1, 0)]=(0.338353834366+0.952153892516j)*x[0]**o + ((0.488586527959+0.185326364267j))*x[0] + ((-0.541975861525-0.121998197045j))*x[1]**o + ((-0.386674607201-0.686565627746j))*x[1]
            ref[(0, 1, 1, 0)]=(-0.203622027159+0.830155695471j)*(1+2.*(dim-1)/(o+1.)) + ((0.101911920758-0.501239263479j))*dim
            arg[(0, 1, 1, 1)]=(-0.724657437673-0.0131990191814j)*x[0]**o + ((0.0190186439348+0.62677042091j))*x[0] + ((-0.987435222773+0.984657405048j))*x[1]**o + ((-0.638133498806-0.529164573914j))*x[1]
            ref[(0, 1, 1, 1)]=(-1.71209266045+0.971458385866j)*(1+2.*(dim-1)/(o+1.)) + ((-0.619114854871+0.0976058469961j))*dim
            arg[(0, 1, 1, 2)]=(-0.00215331913248+0.763160426227j)*x[0]**o + ((0.33020721643-0.995845420992j))*x[0] + ((0.592998051721+0.76683986865j))*x[1]**o + ((-0.636862226978+0.373715820739j))*x[1]
            ref[(0, 1, 1, 2)]=(0.590844732588+1.53000029488j)*(1+2.*(dim-1)/(o+1.)) + ((-0.306655010548-0.622129600253j))*dim
            arg[(0, 1, 1, 3)]=(0.30766842751+0.929282559972j)*x[0]**o + ((0.646694277786+0.464015397369j))*x[0] + ((-0.998251995129+0.374292217842j))*x[1]**o + ((-0.384343957284+0.540221581141j))*x[1]
            ref[(0, 1, 1, 3)]=(-0.690583567619+1.30357477781j)*(1+2.*(dim-1)/(o+1.)) + ((0.262350320502+1.00423697851j))*dim
            arg[(0, 1, 2, 0)]=(-0.594341917499-0.601321509107j)*x[0]**o + ((-0.565627402885-0.211828446178j))*x[0] + ((0.306992102999+0.138499427271j))*x[1]**o + ((-0.792425514642+0.911733752886j))*x[1]
            ref[(0, 1, 2, 0)]=(-0.2873498145-0.462822081836j)*(1+2.*(dim-1)/(o+1.)) + ((-1.35805291753+0.699905306708j))*dim
            arg[(0, 1, 2, 1)]=(-0.571513995364+0.770591295464j)*x[0]**o + ((-0.564292805114-0.997462934593j))*x[0] + ((0.00304906989875-0.894935866549j))*x[1]**o + ((0.376533927393+0.264024803673j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.568464925465-0.124344571085j)*(1+2.*(dim-1)/(o+1.)) + ((-0.187758877722-0.73343813092j))*dim
            arg[(0, 1, 2, 2)]=(0.515364773038+0.099724289883j)*x[0]**o + ((-0.133769081453-0.191532343559j))*x[0] + ((0.0116965567333-0.948123244441j))*x[1]**o + ((-0.187969622202-0.84063107771j))*x[1]
            ref[(0, 1, 2, 2)]=(0.527061329771-0.848398954558j)*(1+2.*(dim-1)/(o+1.)) + ((-0.321738703655-1.03216342127j))*dim
            arg[(0, 1, 2, 3)]=(-0.123294176296-0.188692155887j)*x[0]**o + ((-0.886241776997+0.73714532343j))*x[0] + ((0.485576616857+0.0687286112086j))*x[1]**o + ((0.570692657205+0.229620180491j))*x[1]
            ref[(0, 1, 2, 3)]=(0.362282440561-0.119963544678j)*(1+2.*(dim-1)/(o+1.)) + ((-0.315549119792+0.966765503921j))*dim
            arg[(0, 2, 0, 0)]=(-0.0395650800556+0.475509282821j)*x[0]**o + ((0.451134688911+0.778299239654j))*x[0] + ((-0.0296687796076+0.460117333879j))*x[1]**o + ((0.852530630521-0.00553877655007j))*x[1]
            ref[(0, 2, 0, 0)]=(-0.0692338596632+0.9356266167j)*(1+2.*(dim-1)/(o+1.)) + ((1.30366531943+0.772760463104j))*dim
            arg[(0, 2, 0, 1)]=(0.550521939525-0.547843000186j)*x[0]**o + ((-0.461601433752+0.288364501297j))*x[0] + ((-0.53121302637-0.00683553572217j))*x[1]**o + ((0.43403902534+0.657038593173j))*x[1]
            ref[(0, 2, 0, 1)]=(0.0193089131552-0.554678535908j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0275624084115+0.945403094469j))*dim
            arg[(0, 2, 0, 2)]=(-0.201372943707+0.363832917102j)*x[0]**o + ((-0.36084132809-0.390291414744j))*x[0] + ((-0.501507448779+0.338593782474j))*x[1]**o + ((-0.946852047433+0.348968036338j))*x[1]
            ref[(0, 2, 0, 2)]=(-0.702880392487+0.702426699576j)*(1+2.*(dim-1)/(o+1.)) + ((-1.30769337552-0.0413233784063j))*dim
            arg[(0, 2, 0, 3)]=(-0.382703444115-0.278931354645j)*x[0]**o + ((-0.802283022462+0.909624814704j))*x[0] + ((-0.825767441329+0.184185645527j))*x[1]**o + ((0.748361299912-0.0161057535661j))*x[1]
            ref[(0, 2, 0, 3)]=(-1.20847088544-0.0947457091185j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0539217225494+0.893519061138j))*dim
            arg[(0, 2, 1, 0)]=(-0.229659472755-0.523127250484j)*x[0]**o + ((0.849219590075+0.540596292248j))*x[0] + ((-0.567472571835+0.700044602765j))*x[1]**o + ((-0.382528680462-0.61350828138j))*x[1]
            ref[(0, 2, 1, 0)]=(-0.79713204459+0.176917352281j)*(1+2.*(dim-1)/(o+1.)) + ((0.466690909612-0.0729119891312j))*dim
            arg[(0, 2, 1, 1)]=(0.524216669519+0.661975430823j)*x[0]**o + ((-0.998672892337+0.775523971771j))*x[0] + ((0.431934044388-0.209512208599j))*x[1]**o + ((-0.566385875075+0.491978519371j))*x[1]
            ref[(0, 2, 1, 1)]=(0.956150713908+0.452463222223j)*(1+2.*(dim-1)/(o+1.)) + ((-1.56505876741+1.26750249114j))*dim
            arg[(0, 2, 1, 2)]=(-0.29779827055+0.732228917237j)*x[0]**o + ((0.986679071778+0.914777283667j))*x[0] + ((0.529776685357+0.965862407822j))*x[1]**o + ((-0.727339294742+0.916000702678j))*x[1]
            ref[(0, 2, 1, 2)]=(0.231978414807+1.69809132506j)*(1+2.*(dim-1)/(o+1.)) + ((0.259339777036+1.83077798634j))*dim
            arg[(0, 2, 1, 3)]=(-0.188038197304-0.662331288997j)*x[0]**o + ((-0.871399240102-0.0316282906226j))*x[0] + ((0.247968379789-0.269349963001j))*x[1]**o + ((-0.973683984943+0.305783145495j))*x[1]
            ref[(0, 2, 1, 3)]=(0.0599301824843-0.931681251998j)*(1+2.*(dim-1)/(o+1.)) + ((-1.84508322505+0.274154854873j))*dim
            arg[(0, 2, 2, 0)]=(-0.532724253173+0.0680824336052j)*x[0]**o + ((0.175695228116+0.957340094386j))*x[0] + ((0.108786878361-0.804442898018j))*x[1]**o + ((-0.159566995683-0.283268393724j))*x[1]
            ref[(0, 2, 2, 0)]=(-0.423937374811-0.736360464413j)*(1+2.*(dim-1)/(o+1.)) + ((0.0161282324331+0.674071700662j))*dim
            arg[(0, 2, 2, 1)]=(0.389652548143-0.760452155968j)*x[0]**o + ((-0.457334078875-0.858091333098j))*x[0] + ((0.320238815724-0.366038828936j))*x[1]**o + ((0.248467049583+0.870954173889j))*x[1]
            ref[(0, 2, 2, 1)]=(0.709891363867-1.1264909849j)*(1+2.*(dim-1)/(o+1.)) + ((-0.208867029291+0.012862840791j))*dim
            arg[(0, 2, 2, 2)]=(0.944618295053+0.288416101685j)*x[0]**o + ((0.210275583282+0.46304178347j))*x[0] + ((0.565787977009+0.0305966667068j))*x[1]**o + ((-0.724482733715+0.304048140196j))*x[1]
            ref[(0, 2, 2, 2)]=(1.51040627206+0.319012768391j)*(1+2.*(dim-1)/(o+1.)) + ((-0.514207150432+0.767089923666j))*dim
            arg[(0, 2, 2, 3)]=(-0.0903666183752+0.328038885952j)*x[0]**o + ((-0.0311590474936+0.240142795354j))*x[0] + ((-0.0888867298648-0.188319885046j))*x[1]**o + ((0.666298298414+0.950018220157j))*x[1]
            ref[(0, 2, 2, 3)]=(-0.17925334824+0.139719000906j)*(1+2.*(dim-1)/(o+1.)) + ((0.63513925092+1.19016101551j))*dim
            arg[(0, 3, 0, 0)]=(-0.952229045423-0.700728810125j)*x[0]**o + ((-0.329559160983-0.797540887273j))*x[0] + ((0.754347249598+0.785814224545j))*x[1]**o + ((-0.363756088906-0.0957849104977j))*x[1]
            ref[(0, 3, 0, 0)]=(-0.197881795825+0.0850854144196j)*(1+2.*(dim-1)/(o+1.)) + ((-0.693315249889-0.893325797771j))*dim
            arg[(0, 3, 0, 1)]=(0.319920010181-0.848668732618j)*x[0]**o + ((0.815060586676+0.72472335654j))*x[0] + ((0.615219767261+0.127590419155j))*x[1]**o + ((0.263688082633+0.98935970812j))*x[1]
            ref[(0, 3, 0, 1)]=(0.935139777443-0.721078313463j)*(1+2.*(dim-1)/(o+1.)) + ((1.07874866931+1.71408306466j))*dim
            arg[(0, 3, 0, 2)]=(0.406627299752+0.976526491841j)*x[0]**o + ((-0.281625564247-0.866038138125j))*x[0] + ((-0.599444497907-0.707150946547j))*x[1]**o + ((0.137934076831-0.336203189647j))*x[1]
            ref[(0, 3, 0, 2)]=(-0.192817198156+0.269375545294j)*(1+2.*(dim-1)/(o+1.)) + ((-0.143691487415-1.20224132777j))*dim
            arg[(0, 3, 0, 3)]=(0.112197080809+0.708629616712j)*x[0]**o + ((0.353672666469-0.760678409063j))*x[0] + ((-0.974665874845-0.661633885943j))*x[1]**o + ((0.777969644207-0.203860062076j))*x[1]
            ref[(0, 3, 0, 3)]=(-0.862468794036+0.0469957307689j)*(1+2.*(dim-1)/(o+1.)) + ((1.13164231068-0.964538471139j))*dim
            arg[(0, 3, 1, 0)]=(-0.75589396701-0.989504018015j)*x[0]**o + ((-0.296279646596+0.396356802455j))*x[0] + ((0.900832562453+0.65820604857j))*x[1]**o + ((0.320437159031+0.0449041104478j))*x[1]
            ref[(0, 3, 1, 0)]=(0.144938595443-0.331297969445j)*(1+2.*(dim-1)/(o+1.)) + ((0.0241575124347+0.441260912903j))*dim
            arg[(0, 3, 1, 1)]=(-0.74530657087+0.852693318248j)*x[0]**o + ((-0.863756047337+0.65928227179j))*x[0] + ((0.166111715416+0.907940017499j))*x[1]**o + ((-0.00958168678724+0.704549531393j))*x[1]
            ref[(0, 3, 1, 1)]=(-0.579194855455+1.76063333575j)*(1+2.*(dim-1)/(o+1.)) + ((-0.873337734124+1.36383180318j))*dim
            arg[(0, 3, 1, 2)]=(0.601863292715+0.00779577591675j)*x[0]**o + ((-0.171887657625+0.701861967388j))*x[0] + ((0.251698531406-0.917195449311j))*x[1]**o + ((-0.700835206615-0.225489490388j))*x[1]
            ref[(0, 3, 1, 2)]=(0.853561824121-0.909399673394j)*(1+2.*(dim-1)/(o+1.)) + ((-0.87272286424+0.476372477j))*dim
            arg[(0, 3, 1, 3)]=(-0.840537876776+0.948980006841j)*x[0]**o + ((0.0226241782806-0.207275808494j))*x[0] + ((0.913658937881-0.941428596286j))*x[1]**o + ((-0.847767460307+0.591296865711j))*x[1]
            ref[(0, 3, 1, 3)]=(0.0731210611056+0.00755141055481j)*(1+2.*(dim-1)/(o+1.)) + ((-0.825143282026+0.384021057216j))*dim
            arg[(0, 3, 2, 0)]=(-0.0780378394494+0.413398177045j)*x[0]**o + ((-0.320598162182-0.993161241779j))*x[0] + ((0.208324423564-0.0311994990497j))*x[1]**o + ((0.761807768432-0.0864027052648j))*x[1]
            ref[(0, 3, 2, 0)]=(0.130286584114+0.382198677996j)*(1+2.*(dim-1)/(o+1.)) + ((0.44120960625-1.07956394704j))*dim
            arg[(0, 3, 2, 1)]=(-0.251349158423-0.284388935957j)*x[0]**o + ((-0.589003481947-0.743767930464j))*x[0] + ((0.0393476789036+0.992469163458j))*x[1]**o + ((-0.253364472066+0.39312742662j))*x[1]
            ref[(0, 3, 2, 1)]=(-0.21200147952+0.708080227501j)*(1+2.*(dim-1)/(o+1.)) + ((-0.842367954013-0.350640503844j))*dim
            arg[(0, 3, 2, 2)]=(-0.410723364486+0.125325685399j)*x[0]**o + ((0.216805234291+0.746221071583j))*x[0] + ((0.246168437202+0.92619662124j))*x[1]**o + ((-0.32822214875+0.315790026227j))*x[1]
            ref[(0, 3, 2, 2)]=(-0.164554927284+1.05152230664j)*(1+2.*(dim-1)/(o+1.)) + ((-0.111416914458+1.06201109781j))*dim
            arg[(0, 3, 2, 3)]=(0.327946011985-0.969529549959j)*x[0]**o + ((-0.212734283575+0.553114487428j))*x[0] + ((-0.714880110525+0.323234544521j))*x[1]**o + ((-0.822784041186+0.379427495275j))*x[1]
            ref[(0, 3, 2, 3)]=(-0.38693409854-0.646295005438j)*(1+2.*(dim-1)/(o+1.)) + ((-1.03551832476+0.932541982704j))*dim
            arg[(1, 0, 0, 0)]=(0.634682631894+0.713816492035j)*x[0]**o + ((0.334108990276-0.694171439988j))*x[0] + ((0.986968137933-0.0926124534854j))*x[1]**o + ((-0.924017400588+0.666146409005j))*x[1]
            ref[(1, 0, 0, 0)]=(1.62165076983+0.62120403855j)*(1+2.*(dim-1)/(o+1.)) + ((-0.589908410312-0.0280250309826j))*dim
            arg[(1, 0, 0, 1)]=(-0.835046111669-0.437739342718j)*x[0]**o + ((0.388791504238+0.797217300016j))*x[0] + ((-0.526453987621-0.890123105289j))*x[1]**o + ((-0.379688739552-0.217573051204j))*x[1]
            ref[(1, 0, 0, 1)]=(-1.36150009929-1.32786244801j)*(1+2.*(dim-1)/(o+1.)) + ((0.00910276468571+0.579644248812j))*dim
            arg[(1, 0, 0, 2)]=(0.247744122888-0.258307506248j)*x[0]**o + ((0.814105337398+0.860798095552j))*x[0] + ((0.530655074717+0.0367663393535j))*x[1]**o + ((-0.206737018678+0.00198438597797j))*x[1]
            ref[(1, 0, 0, 2)]=(0.778399197605-0.221541166895j)*(1+2.*(dim-1)/(o+1.)) + ((0.60736831872+0.86278248153j))*dim
            arg[(1, 0, 0, 3)]=(0.493490951069+0.0155409960418j)*x[0]**o + ((0.999259970373+0.500158645222j))*x[0] + ((-0.492189211036-0.23601812501j))*x[1]**o + ((0.276856771819+0.74629421525j))*x[1]
            ref[(1, 0, 0, 3)]=(0.00130174003243-0.220477128968j)*(1+2.*(dim-1)/(o+1.)) + ((1.27611674219+1.24645286047j))*dim
            arg[(1, 0, 1, 0)]=(-0.95691418402+0.970159840787j)*x[0]**o + ((-0.284180335126-0.00426690019388j))*x[0] + ((0.0711499720557+0.0958094312323j))*x[1]**o + ((0.175694337333-0.872845524617j))*x[1]
            ref[(1, 0, 1, 0)]=(-0.885764211964+1.06596927202j)*(1+2.*(dim-1)/(o+1.)) + ((-0.108485997793-0.87711242481j))*dim
            arg[(1, 0, 1, 1)]=(0.531502386241-0.129634433076j)*x[0]**o + ((-0.301383637983-0.138178666482j))*x[0] + ((0.69937350373-0.193517087241j))*x[1]**o + ((-0.634863254836+0.850526164509j))*x[1]
            ref[(1, 0, 1, 1)]=(1.23087588997-0.323151520317j)*(1+2.*(dim-1)/(o+1.)) + ((-0.936246892819+0.712347498027j))*dim
            arg[(1, 0, 1, 2)]=(-0.693253522981+0.147893343832j)*x[0]**o + ((0.721248276149-0.0998570907488j))*x[0] + ((0.879177629363-0.276530637047j))*x[1]**o + ((-0.509305834837-0.170257125342j))*x[1]
            ref[(1, 0, 1, 2)]=(0.185924106383-0.128637293215j)*(1+2.*(dim-1)/(o+1.)) + ((0.211942441312-0.270114216091j))*dim
            arg[(1, 0, 1, 3)]=(-0.765586228824-0.963107595333j)*x[0]**o + ((0.570972710566-0.324044700304j))*x[0] + ((-0.126020423473+0.339939607932j))*x[1]**o + ((0.393601688008+0.266102999988j))*x[1]
            ref[(1, 0, 1, 3)]=(-0.891606652297-0.623167987401j)*(1+2.*(dim-1)/(o+1.)) + ((0.964574398574-0.0579417003157j))*dim
            arg[(1, 0, 2, 0)]=(-0.262483817216-0.615092650659j)*x[0]**o + ((-0.143476967546+0.915033138685j))*x[0] + ((0.0194355002362+0.484992115268j))*x[1]**o + ((-0.038876703953+0.688143429658j))*x[1]
            ref[(1, 0, 2, 0)]=(-0.24304831698-0.130100535391j)*(1+2.*(dim-1)/(o+1.)) + ((-0.182353671499+1.60317656834j))*dim
            arg[(1, 0, 2, 1)]=(0.267291578133+0.334284143427j)*x[0]**o + ((-0.792818753602-0.968239724146j))*x[0] + ((-0.740893844691+0.818723490371j))*x[1]**o + ((-0.73948721742+0.192389114657j))*x[1]
            ref[(1, 0, 2, 1)]=(-0.473602266558+1.1530076338j)*(1+2.*(dim-1)/(o+1.)) + ((-1.53230597102-0.775850609489j))*dim
            arg[(1, 0, 2, 2)]=(-0.315288851995-0.470947221968j)*x[0]**o + ((-0.586631393813+0.495603812303j))*x[0] + ((-0.220542329656-0.184702013821j))*x[1]**o + ((-0.620148111281-0.940113937058j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.535831181651-0.655649235789j)*(1+2.*(dim-1)/(o+1.)) + ((-1.20677950509-0.444510124755j))*dim
            arg[(1, 0, 2, 3)]=(-0.504000096025+0.256000065612j)*x[0]**o + ((0.45085178735+0.460133173429j))*x[0] + ((0.221549494661-0.787147098227j))*x[1]**o + ((0.688955925866-0.128898814889j))*x[1]
            ref[(1, 0, 2, 3)]=(-0.282450601363-0.531147032615j)*(1+2.*(dim-1)/(o+1.)) + ((1.13980771322+0.33123435854j))*dim
            arg[(1, 1, 0, 0)]=(-0.975894196031+0.962191843375j)*x[0]**o + ((-0.0355768426257+0.790688355252j))*x[0] + ((0.929349471973-0.282873452746j))*x[1]**o + ((0.441609112656+0.610388601509j))*x[1]
            ref[(1, 1, 0, 0)]=(-0.0465447240579+0.679318390628j)*(1+2.*(dim-1)/(o+1.)) + ((0.40603227003+1.40107695676j))*dim
            arg[(1, 1, 0, 1)]=(0.995696089801+0.165395292988j)*x[0]**o + ((0.650890760579-0.199412395823j))*x[0] + ((-0.207054001417+0.473359196431j))*x[1]**o + ((0.226086023379+0.740480547921j))*x[1]
            ref[(1, 1, 0, 1)]=(0.788642088385+0.638754489419j)*(1+2.*(dim-1)/(o+1.)) + ((0.876976783958+0.541068152099j))*dim
            arg[(1, 1, 0, 2)]=(-0.214036924912-0.00131172636009j)*x[0]**o + ((0.387712204025+0.710755193464j))*x[0] + ((0.296279883523+0.0293234752987j))*x[1]**o + ((0.348947014116-0.59924431971j))*x[1]
            ref[(1, 1, 0, 2)]=(0.0822429586115+0.0280117489386j)*(1+2.*(dim-1)/(o+1.)) + ((0.736659218141+0.111510873753j))*dim
            arg[(1, 1, 0, 3)]=(-0.147327888012-0.0613055656448j)*x[0]**o + ((0.780550024254-0.562082219638j))*x[0] + ((0.985007417003+0.511987690565j))*x[1]**o + ((0.518649969889+0.777054651258j))*x[1]
            ref[(1, 1, 0, 3)]=(0.837679528991+0.45068212492j)*(1+2.*(dim-1)/(o+1.)) + ((1.29919999414+0.21497243162j))*dim
            arg[(1, 1, 1, 0)]=(-0.644066433791+0.61677656056j)*x[0]**o + ((-0.240774107865-0.220123009368j))*x[0] + ((0.0209700502768-0.140877514454j))*x[1]**o + ((-0.766146858077-0.828987902852j))*x[1]
            ref[(1, 1, 1, 0)]=(-0.623096383514+0.475899046105j)*(1+2.*(dim-1)/(o+1.)) + ((-1.00692096594-1.04911091222j))*dim
            arg[(1, 1, 1, 1)]=(-0.418600803383+0.0615736149364j)*x[0]**o + ((0.634521030208-0.360061856409j))*x[0] + ((-0.70717367939-0.657544261857j))*x[1]**o + ((0.911194000315+0.785186721091j))*x[1]
            ref[(1, 1, 1, 1)]=(-1.12577448277-0.595970646921j)*(1+2.*(dim-1)/(o+1.)) + ((1.54571503052+0.425124864681j))*dim
            arg[(1, 1, 1, 2)]=(-0.221280256382+0.424842216284j)*x[0]**o + ((-0.188600234857+0.0360679505796j))*x[0] + ((-0.211506486096-0.485225641918j))*x[1]**o + ((0.492998307916+0.568692078364j))*x[1]
            ref[(1, 1, 1, 2)]=(-0.432786742478-0.0603834256333j)*(1+2.*(dim-1)/(o+1.)) + ((0.304398073059+0.604760028943j))*dim
            arg[(1, 1, 1, 3)]=(0.126666520178+0.846952649379j)*x[0]**o + ((-0.822126419274+0.33110448776j))*x[0] + ((0.923816077564+0.797600612918j))*x[1]**o + ((0.236175897363-0.437652552112j))*x[1]
            ref[(1, 1, 1, 3)]=(1.05048259774+1.6445532623j)*(1+2.*(dim-1)/(o+1.)) + ((-0.585950521911-0.106548064352j))*dim
            arg[(1, 1, 2, 0)]=(0.608334364353+0.0236316198682j)*x[0]**o + ((-0.705730734464-0.738486257243j))*x[0] + ((-0.341682421171-0.640679888665j))*x[1]**o + ((0.147277809324-0.336790247027j))*x[1]
            ref[(1, 1, 2, 0)]=(0.266651943182-0.617048268797j)*(1+2.*(dim-1)/(o+1.)) + ((-0.55845292514-1.07527650427j))*dim
            arg[(1, 1, 2, 1)]=(0.220254484846+0.400454026771j)*x[0]**o + ((0.97815570268+0.588447935548j))*x[0] + ((-0.0718683229803+0.428867239585j))*x[1]**o + ((0.531536423835-0.683448442274j))*x[1]
            ref[(1, 1, 2, 1)]=(0.148386161866+0.829321266356j)*(1+2.*(dim-1)/(o+1.)) + ((1.50969212652-0.0950005067258j))*dim
            arg[(1, 1, 2, 2)]=(0.328269661423+0.666481629882j)*x[0]**o + ((0.284841998452-0.87067022179j))*x[0] + ((0.867906074666-0.508862589204j))*x[1]**o + ((0.395375991351+0.347708224894j))*x[1]
            ref[(1, 1, 2, 2)]=(1.19617573609+0.157619040678j)*(1+2.*(dim-1)/(o+1.)) + ((0.680217989803-0.522961996896j))*dim
            arg[(1, 1, 2, 3)]=(-0.189531483786-0.884611766705j)*x[0]**o + ((0.666923618672-0.925219983832j))*x[0] + ((-0.363696228509+0.130086764584j))*x[1]**o + ((0.290086801721+0.136356185053j))*x[1]
            ref[(1, 1, 2, 3)]=(-0.553227712295-0.754525002121j)*(1+2.*(dim-1)/(o+1.)) + ((0.957010420393-0.78886379878j))*dim
            arg[(1, 2, 0, 0)]=(-0.529539216624-0.939580343959j)*x[0]**o + ((0.0436078287896+0.704650789896j))*x[0] + ((-0.867754931126+0.779992978294j))*x[1]**o + ((0.225309520772+0.0760775888822j))*x[1]
            ref[(1, 2, 0, 0)]=(-1.39729414775-0.159587365664j)*(1+2.*(dim-1)/(o+1.)) + ((0.268917349561+0.780728378778j))*dim
            arg[(1, 2, 0, 1)]=(-0.623865068668+0.996242381966j)*x[0]**o + ((0.078306688199-0.930997077345j))*x[0] + ((-0.134311528116-0.239232148651j))*x[1]**o + ((-0.244656956794+0.654055565952j))*x[1]
            ref[(1, 2, 0, 1)]=(-0.758176596784+0.757010233315j)*(1+2.*(dim-1)/(o+1.)) + ((-0.166350268595-0.276941511394j))*dim
            arg[(1, 2, 0, 2)]=(-0.395530135516-0.176375427862j)*x[0]**o + ((0.738525508617-0.412722024345j))*x[0] + ((-0.295427511959+0.582959825133j))*x[1]**o + ((0.568418322456+0.466855188879j))*x[1]
            ref[(1, 2, 0, 2)]=(-0.690957647476+0.406584397271j)*(1+2.*(dim-1)/(o+1.)) + ((1.30694383107+0.0541331645338j))*dim
            arg[(1, 2, 0, 3)]=(0.176111710514-0.696315974286j)*x[0]**o + ((-0.859354417067-0.250328244637j))*x[0] + ((0.186565996446-0.824125855004j))*x[1]**o + ((-0.936362990104+0.465534603756j))*x[1]
            ref[(1, 2, 0, 3)]=(0.36267770696-1.52044182929j)*(1+2.*(dim-1)/(o+1.)) + ((-1.79571740717+0.215206359119j))*dim
            arg[(1, 2, 1, 0)]=(-0.330850375408+0.0599337606623j)*x[0]**o + ((0.351161295349+0.146495063175j))*x[0] + ((0.430201818291+0.804072877611j))*x[1]**o + ((-0.116944826667+0.209296892091j))*x[1]
            ref[(1, 2, 1, 0)]=(0.0993514428824+0.864006638273j)*(1+2.*(dim-1)/(o+1.)) + ((0.234216468682+0.355791955266j))*dim
            arg[(1, 2, 1, 1)]=(0.345851095648-0.239713013904j)*x[0]**o + ((-0.750592291624-0.553179173761j))*x[0] + ((-0.573419869902+0.593032724709j))*x[1]**o + ((0.209139416702+0.172813775466j))*x[1]
            ref[(1, 2, 1, 1)]=(-0.227568774254+0.353319710805j)*(1+2.*(dim-1)/(o+1.)) + ((-0.541452874922-0.380365398295j))*dim
            arg[(1, 2, 1, 2)]=(0.545727110501-0.736517282863j)*x[0]**o + ((0.617623639471+0.871729455093j))*x[0] + ((0.106817041935-0.253573586554j))*x[1]**o + ((0.854051589997-0.560008771683j))*x[1]
            ref[(1, 2, 1, 2)]=(0.652544152436-0.990090869417j)*(1+2.*(dim-1)/(o+1.)) + ((1.47167522947+0.31172068341j))*dim
            arg[(1, 2, 1, 3)]=(-0.426069926436-0.0388312294878j)*x[0]**o + ((0.599021174981-0.933261376981j))*x[0] + ((-0.954389159777+0.263440627139j))*x[1]**o + ((0.390281186302-0.861313456969j))*x[1]
            ref[(1, 2, 1, 3)]=(-1.38045908621+0.224609397651j)*(1+2.*(dim-1)/(o+1.)) + ((0.989302361283-1.79457483395j))*dim
            arg[(1, 2, 2, 0)]=(0.967190100869+0.896971619957j)*x[0]**o + ((0.314062608795+0.330216986471j))*x[0] + ((0.847016827621-0.526601130578j))*x[1]**o + ((0.521752159841-0.644469118739j))*x[1]
            ref[(1, 2, 2, 0)]=(1.81420692849+0.370370489379j)*(1+2.*(dim-1)/(o+1.)) + ((0.835814768636-0.314252132267j))*dim
            arg[(1, 2, 2, 1)]=(-0.52251234273-0.504277363431j)*x[0]**o + ((-0.0550379186751-0.949595827117j))*x[0] + ((0.329267698541-0.572485174755j))*x[1]**o + ((0.336239679159+0.259683008044j))*x[1]
            ref[(1, 2, 2, 1)]=(-0.193244644189-1.07676253819j)*(1+2.*(dim-1)/(o+1.)) + ((0.281201760483-0.689912819074j))*dim
            arg[(1, 2, 2, 2)]=(-0.682133945908+0.831043780262j)*x[0]**o + ((0.0317062939332-0.387416789235j))*x[0] + ((0.60882670367+0.221069382723j))*x[1]**o + ((0.600779191947-0.138920237659j))*x[1]
            ref[(1, 2, 2, 2)]=(-0.0733072422382+1.05211316298j)*(1+2.*(dim-1)/(o+1.)) + ((0.63248548588-0.526337026894j))*dim
            arg[(1, 2, 2, 3)]=(-0.0592445162328-0.0969922890219j)*x[0]**o + ((-0.314901723218-0.628163847961j))*x[0] + ((-0.29456495591-0.902509358531j))*x[1]**o + ((0.0819756743735+0.110643600591j))*x[1]
            ref[(1, 2, 2, 3)]=(-0.353809472143-0.999501647553j)*(1+2.*(dim-1)/(o+1.)) + ((-0.232926048845-0.51752024737j))*dim
            arg[(1, 3, 0, 0)]=(0.721348747686-0.62787921015j)*x[0]**o + ((0.719936388306+0.291226612649j))*x[0] + ((-0.662234957663+0.79634350993j))*x[1]**o + ((0.287740950157-0.718369369514j))*x[1]
            ref[(1, 3, 0, 0)]=(0.0591137900225+0.16846429978j)*(1+2.*(dim-1)/(o+1.)) + ((1.00767733846-0.427142756865j))*dim
            arg[(1, 3, 0, 1)]=(-0.214156562581+0.83736066517j)*x[0]**o + ((0.832412829957-0.98393528313j))*x[0] + ((-0.192164834134+0.411003363284j))*x[1]**o + ((0.424089869215+0.397308990763j))*x[1]
            ref[(1, 3, 0, 1)]=(-0.406321396714+1.24836402845j)*(1+2.*(dim-1)/(o+1.)) + ((1.25650269917-0.586626292367j))*dim
            arg[(1, 3, 0, 2)]=(-0.99002322822+0.910441757431j)*x[0]**o + ((-0.953000250763-0.729205322603j))*x[0] + ((0.951500606927-0.60054903934j))*x[1]**o + ((0.166132671987-0.593134733124j))*x[1]
            ref[(1, 3, 0, 2)]=(-0.0385226212932+0.309892718091j)*(1+2.*(dim-1)/(o+1.)) + ((-0.786867578777-1.32234005573j))*dim
            arg[(1, 3, 0, 3)]=(0.0873475244077+0.138374422117j)*x[0]**o + ((0.311930443356+0.0518362580206j))*x[0] + ((-0.567717888392-0.357227689878j))*x[1]**o + ((-0.481558819165+0.133264150427j))*x[1]
            ref[(1, 3, 0, 3)]=(-0.480370363984-0.218853267761j)*(1+2.*(dim-1)/(o+1.)) + ((-0.169628375809+0.185100408448j))*dim
            arg[(1, 3, 1, 0)]=(0.988878564047+0.48080224423j)*x[0]**o + ((-0.365665339277+0.36730854088j))*x[0] + ((0.768769992493-0.480932686295j))*x[1]**o + ((0.489850271253-0.708511557301j))*x[1]
            ref[(1, 3, 1, 0)]=(1.75764855654-0.000130442065467j)*(1+2.*(dim-1)/(o+1.)) + ((0.124184931975-0.341203016421j))*dim
            arg[(1, 3, 1, 1)]=(0.472818171103+0.0565849319734j)*x[0]**o + ((0.376068627912+0.851615657618j))*x[0] + ((0.259971908269+0.595968298314j))*x[1]**o + ((0.473556536261+0.317504766232j))*x[1]
            ref[(1, 3, 1, 1)]=(0.732790079372+0.652553230287j)*(1+2.*(dim-1)/(o+1.)) + ((0.849625164173+1.16912042385j))*dim
            arg[(1, 3, 1, 2)]=(0.318448339556-0.775536378874j)*x[0]**o + ((0.932518288131+0.0779302410839j))*x[0] + ((-0.870950195575+0.656567152793j))*x[1]**o + ((-0.96272614288-0.903756142436j))*x[1]
            ref[(1, 3, 1, 2)]=(-0.552501856019-0.118969226081j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0302078547493-0.825825901352j))*dim
            arg[(1, 3, 1, 3)]=(0.0163996080743-0.0484821749175j)*x[0]**o + ((0.250999701397-0.51277567966j))*x[0] + ((-0.134282776699-0.607768900478j))*x[1]**o + ((-0.578039190911-0.0574635883625j))*x[1]
            ref[(1, 3, 1, 3)]=(-0.117883168624-0.656251075395j)*(1+2.*(dim-1)/(o+1.)) + ((-0.327039489514-0.570239268022j))*dim
            arg[(1, 3, 2, 0)]=(0.204432047184+0.147563438761j)*x[0]**o + ((0.502902840359-0.687655685088j))*x[0] + ((0.949384224805-0.567860214176j))*x[1]**o + ((-0.00329397998044+0.00508867497639j))*x[1]
            ref[(1, 3, 2, 0)]=(1.15381627199-0.420296775415j)*(1+2.*(dim-1)/(o+1.)) + ((0.499608860379-0.682567010112j))*dim
            arg[(1, 3, 2, 1)]=(0.544547004481+0.450411873589j)*x[0]**o + ((0.498844135616+0.16279023234j))*x[0] + ((-0.342647788187+0.682390684186j))*x[1]**o + ((-0.925573273783+0.894545912542j))*x[1]
            ref[(1, 3, 2, 1)]=(0.201899216294+1.13280255777j)*(1+2.*(dim-1)/(o+1.)) + ((-0.426729138167+1.05733614488j))*dim
            arg[(1, 3, 2, 2)]=(0.408040908618-0.787622300055j)*x[0]**o + ((0.954593078993-0.327341555801j))*x[0] + ((-0.87238134249+0.449524830156j))*x[1]**o + ((-0.222401920826+0.160696438376j))*x[1]
            ref[(1, 3, 2, 2)]=(-0.464340433871-0.338097469899j)*(1+2.*(dim-1)/(o+1.)) + ((0.732191158167-0.166645117425j))*dim
            arg[(1, 3, 2, 3)]=(-0.0485009362499-0.0284134202052j)*x[0]**o + ((-0.132405273571-0.0397169176451j))*x[0] + ((0.770666063006+0.257752353279j))*x[1]**o + ((0.263006768827+0.114929785024j))*x[1]
            ref[(1, 3, 2, 3)]=(0.722165126756+0.229338933074j)*(1+2.*(dim-1)/(o+1.)) + ((0.130601495256+0.0752128673793j))*dim
        else:
            arg[(0, 0, 0, 0)]=(0.181094396594-0.0162050932758j)*x[0]**o + ((0.731917661792+0.132229428298j))*x[0] + ((-0.325352679113-0.805328106994j))*x[1]**o + ((-0.991778054043-0.497551086715j))*x[1] + ((-0.826400158737-0.0931703022937j))*x[2]**o + ((-0.218081033326+0.371827533224j))*x[2]
            ref[(0, 0, 0, 0)]=(-0.970658441256-0.914703502564j)*(1+2.*(dim-1)/(o+1.)) + ((-0.477941425576+0.0065058748072j))*dim
            arg[(0, 0, 0, 1)]=(0.265341496447-0.625306745038j)*x[0]**o + ((0.869778096001+0.431467425219j))*x[0] + ((-0.262516092568-0.692650640742j))*x[1]**o + ((0.565339586563-0.722367679903j))*x[1] + ((0.149729018529-0.739624628418j))*x[2]**o + ((0.338231573755-0.513188360693j))*x[2]
            ref[(0, 0, 0, 1)]=(0.152554422408-2.0575820142j)*(1+2.*(dim-1)/(o+1.)) + ((1.77334925632-0.804088615377j))*dim
            arg[(0, 0, 0, 2)]=(0.666011166757-0.0700276068433j)*x[0]**o + ((-0.26833900039-0.88061928269j))*x[0] + ((0.789849581645-0.0491975219207j))*x[1]**o + ((-0.755413669567-0.339958860948j))*x[1] + ((0.393962545186+0.652319928546j))*x[2]**o + ((0.163344660794+0.662128552628j))*x[2]
            ref[(0, 0, 0, 2)]=(1.84982329359+0.533094799782j)*(1+2.*(dim-1)/(o+1.)) + ((-0.860408009163-0.55844959101j))*dim
            arg[(0, 0, 0, 3)]=(-0.771312124719-0.018799680434j)*x[0]**o + ((-0.823005336309-0.388302678053j))*x[0] + ((-0.0510151027343+0.652474815198j))*x[1]**o + ((-0.180688174377-0.500832760518j))*x[1] + ((-0.587921970344-0.864669640798j))*x[2]**o + ((0.586468001257+0.162375812161j))*x[2]
            ref[(0, 0, 0, 3)]=(-1.4102491978-0.230994506033j)*(1+2.*(dim-1)/(o+1.)) + ((-0.417225509429-0.72675962641j))*dim
            arg[(0, 0, 1, 0)]=(-0.0666110936789-0.0764956391485j)*x[0]**o + ((0.991723992342+0.341224729735j))*x[0] + ((-0.612860476831-0.0708837041755j))*x[1]**o + ((0.74860781563+0.78977072364j))*x[1] + ((-0.454364906407-0.556553399213j))*x[2]**o + ((-0.313276003406-0.424683996521j))*x[2]
            ref[(0, 0, 1, 0)]=(-1.13383647692-0.703932742537j)*(1+2.*(dim-1)/(o+1.)) + ((1.42705580457+0.706311456854j))*dim
            arg[(0, 0, 1, 1)]=(0.365758450129-0.0703552699213j)*x[0]**o + ((-0.450719134776+0.795597854156j))*x[0] + ((0.592825321593+0.323312225239j))*x[1]**o + ((-0.490909502488-0.21902918174j))*x[1] + ((0.231862422848-0.142844171376j))*x[2]**o + ((-0.107815402061-0.542883273272j))*x[2]
            ref[(0, 0, 1, 1)]=(1.19044619457+0.110112783941j)*(1+2.*(dim-1)/(o+1.)) + ((-1.04944403932+0.0336853991438j))*dim
            arg[(0, 0, 1, 2)]=(0.8000240377-0.619071053921j)*x[0]**o + ((0.645534310246-0.864604187651j))*x[0] + ((-0.265501536257-0.116529430644j))*x[1]**o + ((-0.0503320850943+0.0768126629125j))*x[1] + ((0.561965330331+0.219493381821j))*x[2]**o + ((0.41977192866-0.137038162392j))*x[2]
            ref[(0, 0, 1, 2)]=(1.09648783178-0.516107102744j)*(1+2.*(dim-1)/(o+1.)) + ((1.01497415381-0.92482968713j))*dim
            arg[(0, 0, 1, 3)]=(-0.701307745731-0.183358595969j)*x[0]**o + ((0.760740183851-0.0613037850743j))*x[0] + ((-0.470168545287+0.204727728673j))*x[1]**o + ((0.148029254086+0.118327296417j))*x[1] + ((0.553055860251-0.632243462404j))*x[2]**o + ((0.328012364379-0.414041238708j))*x[2]
            ref[(0, 0, 1, 3)]=(-0.618420430768-0.610874329701j)*(1+2.*(dim-1)/(o+1.)) + ((1.23678180232-0.357017727365j))*dim
            arg[(0, 0, 2, 0)]=(0.374378117392-0.223304193724j)*x[0]**o + ((0.957499220379-0.519252262962j))*x[0] + ((-0.456372302269+0.694726093434j))*x[1]**o + ((-0.688124080724-0.0432369610404j))*x[1] + ((0.315491371118+0.655584787743j))*x[2]**o + ((-0.31077019189-0.355215338253j))*x[2]
            ref[(0, 0, 2, 0)]=(0.233497186241+1.12700668745j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0413950522345-0.917704562255j))*dim
            arg[(0, 0, 2, 1)]=(-0.872615245824-0.684637847202j)*x[0]**o + ((-0.885908274886-0.639024267094j))*x[0] + ((-0.96299779451-0.483824357722j))*x[1]**o + ((-0.342508171877-0.432251246064j))*x[1] + ((0.584926773222-0.381599018905j))*x[2]**o + ((0.401022826466+0.527759177381j))*x[2]
            ref[(0, 0, 2, 1)]=(-1.25068626711-1.55006122383j)*(1+2.*(dim-1)/(o+1.)) + ((-0.827393620297-0.543516335777j))*dim
            arg[(0, 0, 2, 2)]=(0.132780651574-0.252519451843j)*x[0]**o + ((0.28074370984+0.758191040161j))*x[0] + ((0.160014519188+0.394065973508j))*x[1]**o + ((0.316571633014+0.58875651354j))*x[1] + ((-0.240556922464-0.680011851917j))*x[2]**o + ((-0.888890973893+0.591487297696j))*x[2]
            ref[(0, 0, 2, 2)]=(0.0522382482984-0.538465330252j)*(1+2.*(dim-1)/(o+1.)) + ((-0.291575631039+1.9384348514j))*dim
            arg[(0, 0, 2, 3)]=(0.740445614472+0.611014661149j)*x[0]**o + ((-0.765151648243-0.0171573618279j))*x[0] + ((0.923770671679-0.147652907023j))*x[1]**o + ((-0.489026248252-0.454900621751j))*x[1] + ((0.678923870969-0.318742365123j))*x[2]**o + ((0.159907625774+0.791017733492j))*x[2]
            ref[(0, 0, 2, 3)]=(2.34314015712+0.144619389003j)*(1+2.*(dim-1)/(o+1.)) + ((-1.09427027072+0.318959749912j))*dim
            arg[(0, 1, 0, 0)]=(0.633053054999+0.307880316664j)*x[0]**o + ((-0.89127699023+0.721941631491j))*x[0] + ((-0.384747255216-0.335606384154j))*x[1]**o + ((-0.554384007074+0.329857399801j))*x[1] + ((-0.682842608979+0.108573488332j))*x[2]**o + ((0.0416004837441+0.687181065238j))*x[2]
            ref[(0, 1, 0, 0)]=(-0.434536809196+0.0808474208427j)*(1+2.*(dim-1)/(o+1.)) + ((-1.40406051356+1.73898009653j))*dim
            arg[(0, 1, 0, 1)]=(0.360358901954+0.988355687395j)*x[0]**o + ((0.79948497818-0.497960967777j))*x[0] + ((-0.870446047607+0.132570128893j))*x[1]**o + ((0.80380876942-0.278886251729j))*x[1] + ((0.679792720135+0.233510905918j))*x[2]**o + ((-0.294691822799+0.225213132179j))*x[2]
            ref[(0, 1, 0, 1)]=(0.169705574482+1.35443672221j)*(1+2.*(dim-1)/(o+1.)) + ((1.3086019248-0.551634087327j))*dim
            arg[(0, 1, 0, 2)]=(0.591037762277-0.949255601671j)*x[0]**o + ((-0.490899551869+0.595964391697j))*x[0] + ((0.631897621954-0.194744563568j))*x[1]**o + ((-0.407547087098+0.418241974655j))*x[1] + ((-0.250818722066-0.33182457509j))*x[2]**o + ((0.318093895132-0.21669563395j))*x[2]
            ref[(0, 1, 0, 2)]=(0.972116662165-1.47582474033j)*(1+2.*(dim-1)/(o+1.)) + ((-0.580352743835+0.797510732402j))*dim
            arg[(0, 1, 0, 3)]=(-0.0608689908695+0.748895062312j)*x[0]**o + ((-0.972597558048+0.886525608652j))*x[0] + ((0.0161714218098-0.997796642978j))*x[1]**o + ((-0.899423774864+0.0126405639912j))*x[1] + ((-0.212598510568-0.594780749194j))*x[2]**o + ((-0.935053055272+0.817765694397j))*x[2]
            ref[(0, 1, 0, 3)]=(-0.257296079627-0.843682329861j)*(1+2.*(dim-1)/(o+1.)) + ((-2.80707438818+1.71693186704j))*dim
            arg[(0, 1, 1, 0)]=(-0.633461031345+0.992432298545j)*x[0]**o + ((0.647456667305+0.763676841853j))*x[0] + ((-0.775750150629+0.0492811746991j))*x[1]**o + ((0.691185467697-0.360609424749j))*x[1] + ((-0.567234081893-0.246039495001j))*x[2]**o + ((-0.812694275345+0.661204521814j))*x[2]
            ref[(0, 1, 1, 0)]=(-1.97644526387+0.795673978243j)*(1+2.*(dim-1)/(o+1.)) + ((0.525947859656+1.06427193892j))*dim
            arg[(0, 1, 1, 1)]=(-0.731675267912-0.15452589358j)*x[0]**o + ((0.153121889653+0.634492858288j))*x[0] + ((-0.477304194988-0.674678529848j))*x[1]**o + ((-0.824034926849-0.93454609976j))*x[1] + ((-0.048049350696+0.890614003155j))*x[2]**o + ((0.813199908988-0.328547380416j))*x[2]
            ref[(0, 1, 1, 1)]=(-1.2570288136+0.061409579726j)*(1+2.*(dim-1)/(o+1.)) + ((0.142286871792-0.628600621888j))*dim
            arg[(0, 1, 1, 2)]=(0.476769705156-0.386567726017j)*x[0]**o + ((0.539511001452+0.108680727295j))*x[0] + ((0.172410273894-0.663729895821j))*x[1]**o + ((0.692972564687-0.306921980751j))*x[1] + ((0.664883005665+0.92783287984j))*x[2]**o + ((0.58667361175+0.0817144763932j))*x[2]
            ref[(0, 1, 1, 2)]=(1.31406298471-0.122464741997j)*(1+2.*(dim-1)/(o+1.)) + ((1.81915717789-0.116526777062j))*dim
            arg[(0, 1, 1, 3)]=(0.945967513081+0.523012808822j)*x[0]**o + ((-0.411623953875-0.115117591935j))*x[0] + ((0.504803659285-0.0370974777867j))*x[1]**o + ((-0.808472189248-0.234329263556j))*x[1] + ((0.482130409772+0.481529898684j))*x[2]**o + ((-0.523181798173+0.33736165933j))*x[2]
            ref[(0, 1, 1, 3)]=(1.93290158214+0.96744522972j)*(1+2.*(dim-1)/(o+1.)) + ((-1.7432779413-0.0120851961616j))*dim
            arg[(0, 1, 2, 0)]=(-0.823130644072+0.284049685606j)*x[0]**o + ((0.703661440793+0.126398509309j))*x[0] + ((-0.864023489336+0.434682714602j))*x[1]**o + ((0.218539998007+0.688470461171j))*x[1] + ((0.176600617378+0.858958377066j))*x[2]**o + ((-0.364836100456+0.992891989411j))*x[2]
            ref[(0, 1, 2, 0)]=(-1.51055351603+1.57769077727j)*(1+2.*(dim-1)/(o+1.)) + ((0.557365338345+1.80776095989j))*dim
            arg[(0, 1, 2, 1)]=(0.371799998809+0.0436444083827j)*x[0]**o + ((0.453644862999+0.148142602465j))*x[0] + ((0.266629163307+0.784393732118j))*x[1]**o + ((-0.714590135866+0.605391708552j))*x[1] + ((-0.734451379558-0.97895060442j))*x[2]**o + ((0.188206023867-0.711644088803j))*x[2]
            ref[(0, 1, 2, 1)]=(-0.096022217441-0.150912463919j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0727392490004+0.0418902222137j))*dim
            arg[(0, 1, 2, 2)]=(-0.602296969127-0.808395678764j)*x[0]**o + ((0.323668246869+0.0136877674507j))*x[0] + ((0.857865006047-0.109856866133j))*x[1]**o + ((0.323520861388+0.765224496107j))*x[1] + ((0.573978711617-0.00312182878263j))*x[2]**o + ((0.83615554506-0.94553350891j))*x[2]
            ref[(0, 1, 2, 2)]=(0.829546748537-0.92137437368j)*(1+2.*(dim-1)/(o+1.)) + ((1.48334465332-0.166621245353j))*dim
            arg[(0, 1, 2, 3)]=(0.613633572817-0.492831741607j)*x[0]**o + ((-0.0204965980259-0.735312054834j))*x[0] + ((0.954489120282+0.392653531069j))*x[1]**o + ((0.802960993714+0.809272693251j))*x[1] + ((-0.257272552668+0.824550052586j))*x[2]**o + ((0.584681183118+0.842823447101j))*x[2]
            ref[(0, 1, 2, 3)]=(1.31085014043+0.724371842049j)*(1+2.*(dim-1)/(o+1.)) + ((1.36714557881+0.916784085517j))*dim
            arg[(0, 2, 0, 0)]=(-0.40119920274+0.386139729852j)*x[0]**o + ((0.197099963316+0.405540800129j))*x[0] + ((0.788060770462+0.769045754477j))*x[1]**o + ((-0.157233387374+0.424543453482j))*x[1] + ((-0.970339633667-0.603882998248j))*x[2]**o + ((0.672844608379+0.863442806333j))*x[2]
            ref[(0, 2, 0, 0)]=(-0.583478065945+0.551302486081j)*(1+2.*(dim-1)/(o+1.)) + ((0.712711184321+1.69352705994j))*dim
            arg[(0, 2, 0, 1)]=(0.939792911151-0.588894468314j)*x[0]**o + ((0.0189872352185-0.846834771907j))*x[0] + ((-0.430086256496-0.870772826903j))*x[1]**o + ((0.569698139451-0.0303667822727j))*x[1] + ((-0.74919367579-0.631234797497j))*x[2]**o + ((-0.864187983515-0.369006471609j))*x[2]
            ref[(0, 2, 0, 1)]=(-0.239487021135-2.09090209271j)*(1+2.*(dim-1)/(o+1.)) + ((-0.275502608845-1.24620802579j))*dim
            arg[(0, 2, 0, 2)]=(-0.132178136007+0.379564276747j)*x[0]**o + ((0.605831720821-0.673879589261j))*x[0] + ((0.424406886802-0.896797338588j))*x[1]**o + ((-0.414905282831+0.533413622021j))*x[1] + ((-0.749779653609-0.692415051397j))*x[2]**o + ((-0.75327986672-0.631967348704j))*x[2]
            ref[(0, 2, 0, 2)]=(-0.457550902814-1.20964811324j)*(1+2.*(dim-1)/(o+1.)) + ((-0.56235342873-0.772433315945j))*dim
            arg[(0, 2, 0, 3)]=(0.813747078888-0.20070622702j)*x[0]**o + ((0.478586197962-0.245865015532j))*x[0] + ((-0.192313611003-0.693759106074j))*x[1]**o + ((-0.228545318764-0.202178031808j))*x[1] + ((-0.891651112685+0.329893458097j))*x[2]**o + ((-0.794479265699-0.611042654627j))*x[2]
            ref[(0, 2, 0, 3)]=(-0.2702176448-0.564571874998j)*(1+2.*(dim-1)/(o+1.)) + ((-0.5444383865-1.05908570197j))*dim
            arg[(0, 2, 1, 0)]=(-0.500510267057-0.161696649866j)*x[0]**o + ((0.434418554793-0.909482525545j))*x[0] + ((0.704111977526+0.695320017932j))*x[1]**o + ((-0.366356967104-0.431587392401j))*x[1] + ((-0.4108006892+0.880826985739j))*x[2]**o + ((-0.00756597844917+0.299169122837j))*x[2]
            ref[(0, 2, 1, 0)]=(-0.20719897873+1.4144503538j)*(1+2.*(dim-1)/(o+1.)) + ((0.0604956092398-1.04190079511j))*dim
            arg[(0, 2, 1, 1)]=(-0.79235340746-0.795019213933j)*x[0]**o + ((-0.446143248635+0.359941981785j))*x[0] + ((0.619845437382-0.559802278272j))*x[1]**o + ((0.732116972497+0.0492199007531j))*x[1] + ((-0.26593033357+0.780079390974j))*x[2]**o + ((0.930554707676+0.631881269584j))*x[2]
            ref[(0, 2, 1, 1)]=(-0.438438303647-0.574742101231j)*(1+2.*(dim-1)/(o+1.)) + ((1.21652843154+1.04104315212j))*dim
            arg[(0, 2, 1, 2)]=(0.145330272071-0.178779332476j)*x[0]**o + ((-0.171311248777+0.936759939321j))*x[0] + ((-0.0509785790325+0.493834446488j))*x[1]**o + ((0.223946366258+0.936209196938j))*x[1] + ((-0.747526564047-0.819229261059j))*x[2]**o + ((0.622223415896+0.0806799997268j))*x[2]
            ref[(0, 2, 1, 2)]=(-0.653174871008-0.504174147047j)*(1+2.*(dim-1)/(o+1.)) + ((0.674858533377+1.95364913599j))*dim
            arg[(0, 2, 1, 3)]=(0.0464754222971-0.82263611918j)*x[0]**o + ((-0.360412360955-0.985813609093j))*x[0] + ((0.6858548821+0.0122262699522j))*x[1]**o + ((0.520691614+0.917492891437j))*x[1] + ((-0.181857914431+0.101422117637j))*x[2]**o + ((-0.951955600864-0.0586587232108j))*x[2]
            ref[(0, 2, 1, 3)]=(0.550472389966-0.70898773159j)*(1+2.*(dim-1)/(o+1.)) + ((-0.79167634782-0.126979440867j))*dim
            arg[(0, 2, 2, 0)]=(-0.166044917616-0.743101185486j)*x[0]**o + ((0.636897829449+0.436769977405j))*x[0] + ((0.439721320105-0.00761739085564j))*x[1]**o + ((-0.622204875269+0.728313427465j))*x[1] + ((0.317218446306-0.18249682751j))*x[2]**o + ((0.245504532623+0.884163554457j))*x[2]
            ref[(0, 2, 2, 0)]=(0.590894848794-0.933215403851j)*(1+2.*(dim-1)/(o+1.)) + ((0.260197486802+2.04924695933j))*dim
            arg[(0, 2, 2, 1)]=(-0.23025851752-0.816375863051j)*x[0]**o + ((-0.999413684612+0.587737376347j))*x[0] + ((0.340032598635-0.369574458842j))*x[1]**o + ((0.844115637306-0.937712022091j))*x[1] + ((-0.816610772035+0.927785475763j))*x[2]**o + ((-0.658621678481-0.454895417311j))*x[2]
            ref[(0, 2, 2, 1)]=(-0.706836690921-0.25816484613j)*(1+2.*(dim-1)/(o+1.)) + ((-0.813919725787-0.804870063056j))*dim
            arg[(0, 2, 2, 2)]=(-0.122173230259-0.557260017827j)*x[0]**o + ((-0.0800915621267+0.315029853029j))*x[0] + ((-0.474841603344-0.745020566959j))*x[1]**o + ((-0.557037521354+0.582131015229j))*x[1] + ((0.613441918844+0.401321396362j))*x[2]**o + ((-0.461392183247+0.518896498164j))*x[2]
            ref[(0, 2, 2, 2)]=(0.0164270852417-0.900959188423j)*(1+2.*(dim-1)/(o+1.)) + ((-1.09852126673+1.41605736642j))*dim
            arg[(0, 2, 2, 3)]=(0.0689213498764+0.591581870725j)*x[0]**o + ((0.552608440581+0.302446552352j))*x[0] + ((0.0758774058482-0.915220145913j))*x[1]**o + ((-0.252525008995+0.716771779082j))*x[1] + ((-0.179957434027+0.746906981496j))*x[2]**o + ((-0.152590217241+0.871811462471j))*x[2]
            ref[(0, 2, 2, 3)]=(-0.0351586783025+0.423268706307j)*(1+2.*(dim-1)/(o+1.)) + ((0.147493214345+1.89102979391j))*dim
            arg[(0, 3, 0, 0)]=(0.242911135815-0.852882404741j)*x[0]**o + ((-0.176648771677+0.997589645572j))*x[0] + ((-0.2203266667+0.28727158879j))*x[1]**o + ((0.220703745361+0.871204850643j))*x[1] + ((-0.95970333877-0.435603194237j))*x[2]**o + ((0.400965781717+0.771682086985j))*x[2]
            ref[(0, 3, 0, 0)]=(-0.937118869655-1.00121401019j)*(1+2.*(dim-1)/(o+1.)) + ((0.445020755402+2.6404765832j))*dim
            arg[(0, 3, 0, 1)]=(0.235112876026-0.750616627353j)*x[0]**o + ((-0.204269617222+0.84813954537j))*x[0] + ((-0.593433706652-0.341691690409j))*x[1]**o + ((-0.9859474417+0.639364292029j))*x[1] + ((-0.412304753263+0.0933901489141j))*x[2]**o + ((0.160817264553-0.466464439814j))*x[2]
            ref[(0, 3, 0, 1)]=(-0.77062558389-0.998918168848j)*(1+2.*(dim-1)/(o+1.)) + ((-1.02939979437+1.02103939758j))*dim
            arg[(0, 3, 0, 2)]=(-0.094331706458+0.0468287605918j)*x[0]**o + ((0.737090908572-0.167784944022j))*x[0] + ((0.728969417805-0.773966799047j))*x[1]**o + ((0.0723554296795-0.546137017096j))*x[1] + ((-0.405790863382+0.403677188791j))*x[2]**o + ((0.308241831096-0.909040246906j))*x[2]
            ref[(0, 3, 0, 2)]=(0.228846847965-0.323460849664j)*(1+2.*(dim-1)/(o+1.)) + ((1.11768816935-1.62296220802j))*dim
            arg[(0, 3, 0, 3)]=(-0.179071479569+0.0859754461317j)*x[0]**o + ((-0.291415220974+0.719198206894j))*x[0] + ((0.134556421718-0.697888183398j))*x[1]**o + ((-0.0554507488662+0.24384396183j))*x[1] + ((-0.289531162926-0.161222485084j))*x[2]**o + ((0.753347775975-0.802779199652j))*x[2]
            ref[(0, 3, 0, 3)]=(-0.334046220777-0.77313522235j)*(1+2.*(dim-1)/(o+1.)) + ((0.406481806135+0.160262969072j))*dim
            arg[(0, 3, 1, 0)]=(-0.455857412854+0.22531187162j)*x[0]**o + ((0.252997297563-0.964741757835j))*x[0] + ((0.270978502904+0.342889302784j))*x[1]**o + ((-0.255829394681-0.817799514273j))*x[1] + ((-0.485028317358-0.0255529507891j))*x[2]**o + ((0.223536900849-0.733116051002j))*x[2]
            ref[(0, 3, 1, 0)]=(-0.669907227308+0.542648223615j)*(1+2.*(dim-1)/(o+1.)) + ((0.220704803731-2.51565732311j))*dim
            arg[(0, 3, 1, 1)]=(0.528643645989-0.844253985621j)*x[0]**o + ((-0.320122423445+0.0109478490385j))*x[0] + ((0.448134083215+0.0443103241459j))*x[1]**o + ((-0.856026760868-0.839078379778j))*x[1] + ((-0.628885585481-0.243615413979j))*x[2]**o + ((0.279754579278+0.756935268776j))*x[2]
            ref[(0, 3, 1, 1)]=(0.347892143723-1.04355907545j)*(1+2.*(dim-1)/(o+1.)) + ((-0.896394605034-0.0711952619637j))*dim
            arg[(0, 3, 1, 2)]=(0.717002836937-0.0776924958256j)*x[0]**o + ((0.376444116816+0.751148510455j))*x[0] + ((-0.703294220856-0.30142357176j))*x[1]**o + ((0.992932960094-0.84673009302j))*x[1] + ((0.902502863704+0.595235691082j))*x[2]**o + ((-0.852657596943+0.739300906437j))*x[2]
            ref[(0, 3, 1, 2)]=(0.916211479786+0.216119623496j)*(1+2.*(dim-1)/(o+1.)) + ((0.516719479968+0.643719323871j))*dim
            arg[(0, 3, 1, 3)]=(0.110000723091+0.932649100591j)*x[0]**o + ((0.658909732818+0.612924829262j))*x[0] + ((-0.824796469937-0.734190077218j))*x[1]**o + ((-0.656295558867+0.205609549613j))*x[1] + ((-0.434364025166+0.863451193087j))*x[2]**o + ((-0.482861177664+0.423534827788j))*x[2]
            ref[(0, 3, 1, 3)]=(-1.14915977201+1.06191021646j)*(1+2.*(dim-1)/(o+1.)) + ((-0.480247003712+1.24206920666j))*dim
            arg[(0, 3, 2, 0)]=(-0.391055719551+0.159218773809j)*x[0]**o + ((-0.881285806453+0.886455711628j))*x[0] + ((0.893168588483-0.748944530317j))*x[1]**o + ((-0.0625921406029-0.554050619835j))*x[1] + ((0.684132474907-0.832224493649j))*x[2]**o + ((-0.864862798276+0.0911985385335j))*x[2]
            ref[(0, 3, 2, 0)]=(1.18624534384-1.42195025016j)*(1+2.*(dim-1)/(o+1.)) + ((-1.80874074533+0.423603630326j))*dim
            arg[(0, 3, 2, 1)]=(-0.156737948216+0.230408272139j)*x[0]**o + ((0.472082248215+0.283768272069j))*x[0] + ((0.0396689029226-0.0771509632575j))*x[1]**o + ((0.859829148091+0.634657553235j))*x[1] + ((0.238878676328+0.133051802705j))*x[2]**o + ((-0.742518961404-0.30920010654j))*x[2]
            ref[(0, 3, 2, 1)]=(0.121809631035+0.286309111587j)*(1+2.*(dim-1)/(o+1.)) + ((0.589392434902+0.609225718763j))*dim
            arg[(0, 3, 2, 2)]=(-0.699764313098-0.833923450886j)*x[0]**o + ((-0.721288877193-0.153876630551j))*x[0] + ((0.461962086714-0.0626496352906j))*x[1]**o + ((0.289653182781-0.986677746254j))*x[1] + ((-0.936296326737+0.58039908999j))*x[2]**o + ((-0.231479568939+0.733151426157j))*x[2]
            ref[(0, 3, 2, 2)]=(-1.17409855312-0.316173996186j)*(1+2.*(dim-1)/(o+1.)) + ((-0.663115263351-0.407402950648j))*dim
            arg[(0, 3, 2, 3)]=(-0.682427157168-0.912562896556j)*x[0]**o + ((0.539868662289+0.114946927229j))*x[0] + ((0.915996570158-0.27743572771j))*x[1]**o + ((-0.66451197451+0.75713791497j))*x[1] + ((-0.868568545758-0.478403132079j))*x[2]**o + ((-0.978999461668-0.301710835331j))*x[2]
            ref[(0, 3, 2, 3)]=(-0.634999132768-1.66840175634j)*(1+2.*(dim-1)/(o+1.)) + ((-1.10364277389+0.570374006868j))*dim
            arg[(1, 0, 0, 0)]=(-0.169402232148+0.723869877651j)*x[0]**o + ((0.549068587872+0.620611722639j))*x[0] + ((0.801973643057+0.172073906755j))*x[1]**o + ((0.327862490131+0.121582346755j))*x[1] + ((-0.998906599848-0.861015686502j))*x[2]**o + ((-0.120254048547+0.980998988422j))*x[2]
            ref[(1, 0, 0, 0)]=(-0.366335188939+0.034928097904j)*(1+2.*(dim-1)/(o+1.)) + ((0.756677029456+1.72319305782j))*dim
            arg[(1, 0, 0, 1)]=(0.683543939174+0.437222379362j)*x[0]**o + ((-0.975958375135+0.0166714055745j))*x[0] + ((0.427213903421+0.918846351888j))*x[1]**o + ((0.65347476657+0.157553752063j))*x[1] + ((0.688502513218+0.79413919803j))*x[2]**o + ((-0.0578938257705+0.147097296809j))*x[2]
            ref[(1, 0, 0, 1)]=(1.79926035581+2.15020792928j)*(1+2.*(dim-1)/(o+1.)) + ((-0.380377434335+0.321322454446j))*dim
            arg[(1, 0, 0, 2)]=(0.291501015251-0.494772851082j)*x[0]**o + ((0.711041794548+0.395707655932j))*x[0] + ((-0.91189855624-0.155491363129j))*x[1]**o + ((0.302886376642-0.567831212888j))*x[1] + ((0.92784666769+0.700588286543j))*x[2]**o + ((0.724382400329-0.298218960932j))*x[2]
            ref[(1, 0, 0, 2)]=(0.307449126701+0.050324072332j)*(1+2.*(dim-1)/(o+1.)) + ((1.73831057152-0.470342517887j))*dim
            arg[(1, 0, 0, 3)]=(0.548659996077+0.764396699516j)*x[0]**o + ((0.623981495378+0.226378922359j))*x[0] + ((0.0539437075065-0.0769104101925j))*x[1]**o + ((-0.0241894637855+0.621278732731j))*x[1] + ((-0.359927838113+0.875159730724j))*x[2]**o + ((-0.168258766221-0.469299770149j))*x[2]
            ref[(1, 0, 0, 3)]=(0.24267586547+1.56264602005j)*(1+2.*(dim-1)/(o+1.)) + ((0.431533265372+0.378357884941j))*dim
            arg[(1, 0, 1, 0)]=(0.175489831343-0.068360216313j)*x[0]**o + ((0.116707085729-0.785937880813j))*x[0] + ((0.622495210449+0.756797130527j))*x[1]**o + ((-0.918714191266-0.931420993228j))*x[1] + ((-0.237391414011-0.894485479953j))*x[2]**o + ((0.887222704243-0.860907452816j))*x[2]
            ref[(1, 0, 1, 0)]=(0.560593627781-0.206048565739j)*(1+2.*(dim-1)/(o+1.)) + ((0.0852155987059-2.57826632686j))*dim
            arg[(1, 0, 1, 1)]=(0.927084860961+0.474557173313j)*x[0]**o + ((-0.625755021307+0.738674504352j))*x[0] + ((-0.909731218649-0.782564218159j))*x[1]**o + ((0.322962453993+0.55069037705j))*x[1] + ((-0.354989331011+0.839657819103j))*x[2]**o + ((-0.597848935435-0.181144761136j))*x[2]
            ref[(1, 0, 1, 1)]=(-0.337635688699+0.531650774257j)*(1+2.*(dim-1)/(o+1.)) + ((-0.90064150275+1.10822012027j))*dim
            arg[(1, 0, 1, 2)]=(0.073706791196+0.984432061903j)*x[0]**o + ((0.716894760624+0.0419894107423j))*x[0] + ((-0.190637764248+0.639588055754j))*x[1]**o + ((-0.163128166877-0.127458895886j))*x[1] + ((0.350399125537-0.0354487668194j))*x[2]**o + ((0.83755421694+0.981178894513j))*x[2]
            ref[(1, 0, 1, 2)]=(0.233468152485+1.58857135084j)*(1+2.*(dim-1)/(o+1.)) + ((1.39132081069+0.895709409369j))*dim
            arg[(1, 0, 1, 3)]=(0.849340409559-0.559518816814j)*x[0]**o + ((0.15255206121-0.653861602012j))*x[0] + ((0.440300993148+0.758806666053j))*x[1]**o + ((0.861979665229-0.859793857089j))*x[1] + ((-0.0286674936262-0.390200465773j))*x[2]**o + ((-0.130236780864-0.647107462107j))*x[2]
            ref[(1, 0, 1, 3)]=(1.26097390908-0.190912616534j)*(1+2.*(dim-1)/(o+1.)) + ((0.884294945576-2.16076292121j))*dim
            arg[(1, 0, 2, 0)]=(-0.689084334105-0.611051840179j)*x[0]**o + ((0.743133834055+0.0356627565323j))*x[0] + ((-0.143343767606-0.573976808407j))*x[1]**o + ((-0.16254818597-0.561734435409j))*x[1] + ((0.362781301005-0.618297209177j))*x[2]**o + ((-0.857230104551-0.246567625701j))*x[2]
            ref[(1, 0, 2, 0)]=(-0.469646800707-1.80332585776j)*(1+2.*(dim-1)/(o+1.)) + ((-0.276644456466-0.772639304577j))*dim
            arg[(1, 0, 2, 1)]=(-0.988839682673+0.172252477659j)*x[0]**o + ((-0.6332210646+0.960219044939j))*x[0] + ((0.797233607693-0.502763067793j))*x[1]**o + ((-0.781745888766-0.394343841108j))*x[1] + ((0.0811066381435-0.0537794932079j))*x[2]**o + ((-0.727425046891-0.735382326194j))*x[2]
            ref[(1, 0, 2, 1)]=(-0.110499436837-0.384290083341j)*(1+2.*(dim-1)/(o+1.)) + ((-2.14239200026-0.169507122364j))*dim
            arg[(1, 0, 2, 2)]=(0.553761152623+0.799840409617j)*x[0]**o + ((0.0487451712169+0.983434600564j))*x[0] + ((0.185534649185-0.505934960838j))*x[1]**o + ((0.311630142172-0.0618837550885j))*x[1] + ((-0.769698752265-0.129630106978j))*x[2]**o + ((-0.405977413989-0.550299874041j))*x[2]
            ref[(1, 0, 2, 2)]=(-0.0304029504572+0.164275341801j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0456021006001+0.371250971434j))*dim
            arg[(1, 0, 2, 3)]=(0.0928708704476-0.892573448312j)*x[0]**o + ((0.849767114515+0.663088540816j))*x[0] + ((0.253432251688+0.405188892078j))*x[1]**o + ((-0.700347181471-0.403514446907j))*x[1] + ((0.436469521387+0.811127974972j))*x[2]**o + ((-0.172257397331-0.150336454161j))*x[2]
            ref[(1, 0, 2, 3)]=(0.782772643522+0.323743418738j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0228374642867+0.109237639749j))*dim
            arg[(1, 1, 0, 0)]=(0.425746934952+0.10072570387j)*x[0]**o + ((0.872465937813-0.523491703164j))*x[0] + ((0.895867877779-0.231741728902j))*x[1]**o + ((-0.823774040994-0.443957090566j))*x[1] + ((0.199402985824-0.906566334077j))*x[2]**o + ((0.0923389643505-0.75546935583j))*x[2]
            ref[(1, 1, 0, 0)]=(1.52101779856-1.03758235911j)*(1+2.*(dim-1)/(o+1.)) + ((0.14103086117-1.72291814956j))*dim
            arg[(1, 1, 0, 1)]=(0.68144028164-0.736996991074j)*x[0]**o + ((-0.154949629-0.658197007765j))*x[0] + ((0.905208533402-0.787165816963j))*x[1]**o + ((0.561503683542+0.319122764159j))*x[1] + ((-0.484177298043-0.94425483562j))*x[2]**o + ((-0.390327491686-0.890988451421j))*x[2]
            ref[(1, 1, 0, 1)]=(1.102471517-2.46841764366j)*(1+2.*(dim-1)/(o+1.)) + ((0.0162265628563-1.23006269503j))*dim
            arg[(1, 1, 0, 2)]=(0.484337165472-0.21047686984j)*x[0]**o + ((0.727861236078-0.478346308817j))*x[0] + ((-0.841278955275-0.782125463806j))*x[1]**o + ((0.147817678008-0.107267208468j))*x[1] + ((0.373788762529+0.132502278169j))*x[2]**o + ((-0.248432065319-0.0849060503968j))*x[2]
            ref[(1, 1, 0, 2)]=(0.0168469727258-0.860100055477j)*(1+2.*(dim-1)/(o+1.)) + ((0.627246848767-0.670519567681j))*dim
            arg[(1, 1, 0, 3)]=(-0.605124981735-0.818224064488j)*x[0]**o + ((0.38786550788+0.618135712554j))*x[0] + ((-0.785014400115+0.848065631411j))*x[1]**o + ((0.0107988567369-0.148957409339j))*x[1] + ((-0.167604850338+0.0362314257977j))*x[2]**o + ((0.358999381828+0.577851683641j))*x[2]
            ref[(1, 1, 0, 3)]=(-1.55774423219+0.0660729927199j)*(1+2.*(dim-1)/(o+1.)) + ((0.757663746445+1.04702998686j))*dim
            arg[(1, 1, 1, 0)]=(-0.212941232908+0.362598189809j)*x[0]**o + ((0.635685613156-0.0279676361086j))*x[0] + ((0.535938502289+0.575875929859j))*x[1]**o + ((-0.497598608351-0.806525757947j))*x[1] + ((0.131057305155+0.2585999657j))*x[2]**o + ((0.402036401871-0.181333231437j))*x[2]
            ref[(1, 1, 1, 0)]=(0.454054574537+1.19707408537j)*(1+2.*(dim-1)/(o+1.)) + ((0.540123406676-1.01582662549j))*dim
            arg[(1, 1, 1, 1)]=(-0.474994223446+0.800316413544j)*x[0]**o + ((-0.852767992394-0.941780180319j))*x[0] + ((-0.259321410638+0.111008629166j))*x[1]**o + ((0.552601102136-0.102975630531j))*x[1] + ((0.54464688917+0.807172944199j))*x[2]**o + ((0.142731110645-0.69076828882j))*x[2]
            ref[(1, 1, 1, 1)]=(-0.189668744914+1.71849798691j)*(1+2.*(dim-1)/(o+1.)) + ((-0.157435779613-1.73552409967j))*dim
            arg[(1, 1, 1, 2)]=(0.718366022178+0.365293285989j)*x[0]**o + ((0.351545482374+0.856450487489j))*x[0] + ((0.370459696621-0.488256055274j))*x[1]**o + ((-0.783748416894-0.443644789579j))*x[1] + ((0.717541234646-0.144884060901j))*x[2]**o + ((-0.854991949603+0.388378685683j))*x[2]
            ref[(1, 1, 1, 2)]=(1.80636695345-0.267846830185j)*(1+2.*(dim-1)/(o+1.)) + ((-1.28719488412+0.801184383593j))*dim
            arg[(1, 1, 1, 3)]=(-0.602017467476-0.362054774502j)*x[0]**o + ((0.378747500277-0.753996444193j))*x[0] + ((-0.909615775476-0.482402669715j))*x[1]**o + ((-0.118678611082-0.754799298131j))*x[1] + ((-0.745402025865-0.193694642601j))*x[2]**o + ((0.708255973826-0.586599086139j))*x[2]
            ref[(1, 1, 1, 3)]=(-2.25703526882-1.03815208682j)*(1+2.*(dim-1)/(o+1.)) + ((0.96832486302-2.09539482846j))*dim
            arg[(1, 1, 2, 0)]=(0.319159336415-0.118433093905j)*x[0]**o + ((-0.536919592588-0.654439940299j))*x[0] + ((0.921554150988+0.305739306415j))*x[1]**o + ((-0.221332517808+0.65367837782j))*x[1] + ((-0.653296805226-0.870001343611j))*x[2]**o + ((0.762241353684-0.977582480891j))*x[2]
            ref[(1, 1, 2, 0)]=(0.587416682177-0.682695131102j)*(1+2.*(dim-1)/(o+1.)) + ((0.00398924328868-0.978344043371j))*dim
            arg[(1, 1, 2, 1)]=(-0.011305795927+0.440899822609j)*x[0]**o + ((-0.808657497197+0.0126254775436j))*x[0] + ((-0.326351250686-0.660415483077j))*x[1]**o + ((-0.724779425811-0.112411864222j))*x[1] + ((0.566973373179+0.989303557567j))*x[2]**o + ((-0.585587834019-0.428047313736j))*x[2]
            ref[(1, 1, 2, 1)]=(0.229316326567+0.769787897098j)*(1+2.*(dim-1)/(o+1.)) + ((-2.11902475703-0.527833700413j))*dim
            arg[(1, 1, 2, 2)]=(0.946371281793+0.761127002906j)*x[0]**o + ((-0.400837014104+0.0634178064556j))*x[0] + ((-0.748492822588+0.322401426922j))*x[1]**o + ((-0.479694995214+0.0864527320601j))*x[1] + ((-0.566639117012-0.00239713058075j))*x[2]**o + ((-0.736255779108+0.439289970085j))*x[2]
            ref[(1, 1, 2, 2)]=(-0.368760657807+1.08113129925j)*(1+2.*(dim-1)/(o+1.)) + ((-1.61678778843+0.589160508601j))*dim
            arg[(1, 1, 2, 3)]=(0.785044461582-0.176189742351j)*x[0]**o + ((-0.47240089555-0.937475053957j))*x[0] + ((-0.193142052753+0.869148580568j))*x[1]**o + ((-0.493767652526+0.0560400171827j))*x[1] + ((0.830662305697-0.806763819025j))*x[2]**o + ((-0.659118818109-0.0771181839444j))*x[2]
            ref[(1, 1, 2, 3)]=(1.42256471453-0.113804980808j)*(1+2.*(dim-1)/(o+1.)) + ((-1.62528736619-0.958553220719j))*dim
            arg[(1, 2, 0, 0)]=(-0.34127017452-0.74449924053j)*x[0]**o + ((0.923356765147-0.8774604867j))*x[0] + ((0.269176620541-0.983329874008j))*x[1]**o + ((-0.379424739561-0.522810847052j))*x[1] + ((0.362410592665-0.450280321566j))*x[2]**o + ((0.377467258609+0.856970837775j))*x[2]
            ref[(1, 2, 0, 0)]=(0.290317038686-2.1781094361j)*(1+2.*(dim-1)/(o+1.)) + ((0.921399284195-0.543300495977j))*dim
            arg[(1, 2, 0, 1)]=(-0.00547934635069-0.936728487188j)*x[0]**o + ((-0.405836966857-0.3813372489j))*x[0] + ((0.0652166976045+0.398353005591j))*x[1]**o + ((0.75158890748-0.390148470484j))*x[1] + ((0.504017966153-0.0226122369129j))*x[2]**o + ((0.993377603324+0.5734438995j))*x[2]
            ref[(1, 2, 0, 1)]=(0.563755317407-0.56098771851j)*(1+2.*(dim-1)/(o+1.)) + ((1.33912954395-0.198041819885j))*dim
            arg[(1, 2, 0, 2)]=(-0.815608957388-0.503638873305j)*x[0]**o + ((-0.916872134716+0.912741930355j))*x[0] + ((-0.958093953105+0.32943055569j))*x[1]**o + ((0.77669002462-0.542072367333j))*x[1] + ((-0.119663912777-0.351856024551j))*x[2]**o + ((0.256394758824+0.550279432731j))*x[2]
            ref[(1, 2, 0, 2)]=(-1.89336682327-0.526064342167j)*(1+2.*(dim-1)/(o+1.)) + ((0.116212648728+0.920948995753j))*dim
            arg[(1, 2, 0, 3)]=(0.906334306228-0.528825390206j)*x[0]**o + ((-0.0440908494606-0.535392011124j))*x[0] + ((0.565789534396-0.267554058548j))*x[1]**o + ((0.282994250636-0.0212833820457j))*x[1] + ((-0.0050645305224-0.855172941688j))*x[2]**o + ((0.220540650797-0.390889710713j))*x[2]
            ref[(1, 2, 0, 3)]=(1.4670593101-1.65155239044j)*(1+2.*(dim-1)/(o+1.)) + ((0.459444051973-0.947565103883j))*dim
            arg[(1, 2, 1, 0)]=(-0.542483339128+0.745881894331j)*x[0]**o + ((-0.502124162223-0.0259436437073j))*x[0] + ((0.640063899636-0.876391063789j))*x[1]**o + ((0.744959579694-0.203277832717j))*x[1] + ((0.458993513137+0.455945971632j))*x[2]**o + ((-0.198839047151+0.614347182091j))*x[2]
            ref[(1, 2, 1, 0)]=(0.556574073644+0.325436802174j)*(1+2.*(dim-1)/(o+1.)) + ((0.0439963703205+0.385125705666j))*dim
            arg[(1, 2, 1, 1)]=(-0.720846889514-0.228601207454j)*x[0]**o + ((-0.728542272817-0.97214467549j))*x[0] + ((0.254030868353+0.232133214109j))*x[1]**o + ((-0.819614787919+0.132422422691j))*x[1] + ((0.029412432404+0.0213197655626j))*x[2]**o + ((0.207096693545+0.628304411232j))*x[2]
            ref[(1, 2, 1, 1)]=(-0.437403588757+0.024851772218j)*(1+2.*(dim-1)/(o+1.)) + ((-1.34106036719-0.211417841567j))*dim
            arg[(1, 2, 1, 2)]=(0.0385126969068-0.302726850897j)*x[0]**o + ((-0.068614349704-0.0440620162816j))*x[0] + ((0.0437486719624+0.501522379115j))*x[1]**o + ((0.315091761447+0.711874523911j))*x[1] + ((-0.618077092633-0.0859369527136j))*x[2]**o + ((-0.367096132713-0.325128987027j))*x[2]
            ref[(1, 2, 1, 2)]=(-0.535815723763+0.112858575504j)*(1+2.*(dim-1)/(o+1.)) + ((-0.12061872097+0.342683520603j))*dim
            arg[(1, 2, 1, 3)]=(0.990105799275+0.0227508108402j)*x[0]**o + ((-0.556359288655-0.394298350364j))*x[0] + ((0.647805323717+0.227681741145j))*x[1]**o + ((-0.771179696499+0.090922664361j))*x[1] + ((-0.843886074979+0.387120873418j))*x[2]**o + ((0.952870265439-0.716549705239j))*x[2]
            ref[(1, 2, 1, 3)]=(0.794025048013+0.637553425403j)*(1+2.*(dim-1)/(o+1.)) + ((-0.374668719715-1.01992539124j))*dim
            arg[(1, 2, 2, 0)]=(-0.130847737913-0.727306500591j)*x[0]**o + ((-0.136401493158-0.477771158458j))*x[0] + ((0.974381422146+0.761246196293j))*x[1]**o + ((0.964390630826-0.532262985669j))*x[1] + ((-0.729766207174+0.374940095705j))*x[2]**o + ((0.628666061707-0.766663356466j))*x[2]
            ref[(1, 2, 2, 0)]=(0.11376747706+0.408879791406j)*(1+2.*(dim-1)/(o+1.)) + ((1.45665519937-1.77669750059j))*dim
            arg[(1, 2, 2, 1)]=(-0.897631559068-0.393447305467j)*x[0]**o + ((0.447450734588-0.335094065105j))*x[0] + ((0.737217146201-0.778699756879j))*x[1]**o + ((-0.857019631945+0.67114996615j))*x[1] + ((-0.666332573941-0.834491151544j))*x[2]**o + ((0.558003271393+0.49282621607j))*x[2]
            ref[(1, 2, 2, 1)]=(-0.826746986808-2.00663821389j)*(1+2.*(dim-1)/(o+1.)) + ((0.148434374035+0.828882117115j))*dim
            arg[(1, 2, 2, 2)]=(0.23554151207-0.859489955183j)*x[0]**o + ((-0.755576421316-0.0767411984708j))*x[0] + ((-0.0845953871579+0.0296279534215j))*x[1]**o + ((0.26470794641-0.968599461562j))*x[1] + ((-0.338391928789-0.393575247471j))*x[2]**o + ((0.907500437238-0.963945072196j))*x[2]
            ref[(1, 2, 2, 2)]=(-0.187445803876-1.22343724923j)*(1+2.*(dim-1)/(o+1.)) + ((0.416631962332-2.00928573223j))*dim
            arg[(1, 2, 2, 3)]=(0.620215970805+0.529865948557j)*x[0]**o + ((0.153864894561-0.898040650359j))*x[0] + ((-0.00684066473625-0.881867229899j))*x[1]**o + ((-0.241813175451+0.530265833348j))*x[1] + ((-0.394264559358+0.678252392491j))*x[2]**o + ((-0.287400122672-0.0502046832815j))*x[2]
            ref[(1, 2, 2, 3)]=(0.219110746711+0.326251111148j)*(1+2.*(dim-1)/(o+1.)) + ((-0.375348403562-0.417979500292j))*dim
            arg[(1, 3, 0, 0)]=(-0.768007649475+0.0213690979919j)*x[0]**o + ((-0.864746749568-0.910241413375j))*x[0] + ((-0.459344485584+0.703836374266j))*x[1]**o + ((-0.41659832578+0.811547799718j))*x[1] + ((-0.994630641385-0.514106341077j))*x[2]**o + ((-0.836848805154+0.817453328328j))*x[2]
            ref[(1, 3, 0, 0)]=(-2.22198277644+0.211099131181j)*(1+2.*(dim-1)/(o+1.)) + ((-2.1181938805+0.718759714671j))*dim
            arg[(1, 3, 0, 1)]=(0.995181730701-0.878178780993j)*x[0]**o + ((0.508355576605+0.769957858441j))*x[0] + ((0.507317986078-0.583805869725j))*x[1]**o + ((0.916826308875-0.344654679602j))*x[1] + ((0.6813797986-0.130438378294j))*x[2]**o + ((-0.372218283487+0.730224669647j))*x[2]
            ref[(1, 3, 0, 1)]=(2.18387951538-1.59242302901j)*(1+2.*(dim-1)/(o+1.)) + ((1.05296360199+1.15552784849j))*dim
            arg[(1, 3, 0, 2)]=(-0.705334714324-0.412183045424j)*x[0]**o + ((-0.0655551434498+0.815110473227j))*x[0] + ((0.81276981148+0.839315182456j))*x[1]**o + ((0.280098406316+0.0302161391381j))*x[1] + ((-0.61222517201-0.4496360026j))*x[2]**o + ((0.375264422704-0.365948445971j))*x[2]
            ref[(1, 3, 0, 2)]=(-0.504790074855-0.0225038655684j)*(1+2.*(dim-1)/(o+1.)) + ((0.589807685571+0.479378166395j))*dim
            arg[(1, 3, 0, 3)]=(0.401900303579+0.43614850781j)*x[0]**o + ((-0.440575219243-0.445515528202j))*x[0] + ((0.740865702272-0.162400161528j))*x[1]**o + ((0.161457305063+0.867110045035j))*x[1] + ((0.51109348263+0.421231844528j))*x[2]**o + ((0.132318490874-0.401986174592j))*x[2]
            ref[(1, 3, 0, 3)]=(1.65385948848+0.694980190811j)*(1+2.*(dim-1)/(o+1.)) + ((-0.146799423306+0.0196083422409j))*dim
            arg[(1, 3, 1, 0)]=(0.72994317077-0.0654959876761j)*x[0]**o + ((-0.611273041453+0.198171777226j))*x[0] + ((-0.226666741324-0.32730984915j))*x[1]**o + ((0.171353970925+0.388696494763j))*x[1] + ((0.563589543291+0.232542803923j))*x[2]**o + ((0.402278462908+0.973692235422j))*x[2]
            ref[(1, 3, 1, 0)]=(1.06686597274-0.160263032903j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0376406076199+1.56056050741j))*dim
            arg[(1, 3, 1, 1)]=(-0.980047440785-0.957320965594j)*x[0]**o + ((-0.192488377513+0.814423454638j))*x[0] + ((0.488501844627-0.631385170258j))*x[1]**o + ((0.170914356701+0.887260478294j))*x[1] + ((-0.000206995952564+0.510980953005j))*x[2]**o + ((0.605235896949+0.273488498025j))*x[2]
            ref[(1, 3, 1, 1)]=(-0.491752592111-1.07772518285j)*(1+2.*(dim-1)/(o+1.)) + ((0.583661876138+1.97517243096j))*dim
            arg[(1, 3, 1, 2)]=(-0.948186770039-0.493709549488j)*x[0]**o + ((0.673331485592-0.912364001321j))*x[0] + ((-0.429516692107-0.865132488672j))*x[1]**o + ((0.41175879221-0.626323737612j))*x[1] + ((0.215084849016+0.155405087486j))*x[2]**o + ((0.252165350107+0.912475575069j))*x[2]
            ref[(1, 3, 1, 2)]=(-1.16261861313-1.20343695067j)*(1+2.*(dim-1)/(o+1.)) + ((1.33725562791-0.626212163863j))*dim
            arg[(1, 3, 1, 3)]=(-0.384448759772+0.0957531510649j)*x[0]**o + ((-0.0330987783781+0.342145628779j))*x[0] + ((-0.127807580084+0.899414199724j))*x[1]**o + ((0.910618169043-0.185152041609j))*x[1] + ((0.702034263665+0.468306349313j))*x[2]**o + ((0.782341066683-0.299471258447j))*x[2]
            ref[(1, 3, 1, 3)]=(0.189777923808+1.4634737001j)*(1+2.*(dim-1)/(o+1.)) + ((1.65986045735-0.142477671277j))*dim
            arg[(1, 3, 2, 0)]=(-0.349538291054-0.984311927182j)*x[0]**o + ((-0.0730879605975+0.803370917806j))*x[0] + ((-0.742775070472-0.281058049312j))*x[1]**o + ((-0.804221248814+0.254251571195j))*x[1] + ((-0.630572764746+0.00947003559451j))*x[2]**o + ((-0.484933158343+0.630811284214j))*x[2]
            ref[(1, 3, 2, 0)]=(-1.72288612627-1.2558999409j)*(1+2.*(dim-1)/(o+1.)) + ((-1.36224236775+1.68843377321j))*dim
            arg[(1, 3, 2, 1)]=(0.374134080471+0.0486865106836j)*x[0]**o + ((-0.0565061733636+0.61422059601j))*x[0] + ((-0.980930951384-0.276738087912j))*x[1]**o + ((-0.0284862148296-0.325234362984j))*x[1] + ((-0.654642399881-0.908230476553j))*x[2]**o + ((0.923645725154-0.00304394057302j))*x[2]
            ref[(1, 3, 2, 1)]=(-1.26143927079-1.13628205378j)*(1+2.*(dim-1)/(o+1.)) + ((0.83865333696+0.285942292453j))*dim
            arg[(1, 3, 2, 2)]=(0.471596428809+0.100785790198j)*x[0]**o + ((-0.400217740035-0.638054897327j))*x[0] + ((0.1244587968-0.878079869096j))*x[1]**o + ((-0.819340359973+0.176397501481j))*x[1] + ((-0.11390355987+0.320897165193j))*x[2]**o + ((-0.27946986202+0.986759878804j))*x[2]
            ref[(1, 3, 2, 2)]=(0.482151665738-0.456396913705j)*(1+2.*(dim-1)/(o+1.)) + ((-1.49902796203+0.525102482958j))*dim
            arg[(1, 3, 2, 3)]=(0.119226985533-0.953927413315j)*x[0]**o + ((-0.719788307053-0.732293998854j))*x[0] + ((-0.246409581332+0.610402308752j))*x[1]**o + ((-0.97995836647-0.58045953195j))*x[1] + ((0.237544685243+0.287139387819j))*x[2]**o + ((0.468504681088-0.700616115814j))*x[2]
            ref[(1, 3, 2, 3)]=(0.110362089444-0.0563857167442j)*(1+2.*(dim-1)/(o+1.)) + ((-1.23124199244-2.01336964662j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2, 4, 3, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_Solution_rank0(self):
        """
        tests integral of rank 0 Data on FunctionOnBoundary

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(),w)
        if dim==2:
            arg=(-0.976878000244+0.323541437159j)*x[0]**o + ((0.785467134174-0.334266802263j))*x[0] + ((-0.694821583844-0.0953612586318j))*x[1]**o + ((-0.536456141283+0.774459168934j))*x[1]
            ref=(-1.67169958409+0.228180178527j)*(1+2.*(dim-1)/(o+1.)) + ((0.249010992891+0.44019236667j))*dim
        else:
            arg=(-0.78969565121-0.224858777839j)*x[0]**o + ((0.305115723608+0.206137638863j))*x[0] + ((0.155763876755+0.640717277988j))*x[1]**o + ((-0.860839319161-0.119430388063j))*x[1] + ((0.293031169331-0.697497889901j))*x[2]**o + ((0.473149201257-0.337594117138j))*x[2]
            ref=(-0.340900605123-0.281639389752j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0825743942966-0.250886866339j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,complex),'wrong type of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_Solution_rank1(self):
        """
        tests integral of rank 1 Data on FunctionOnBoundary

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(2,),w)
        ref=numpy.zeros((2,),numpy.complex_)
        if dim==2:
            arg[(0,)]=(-0.239419316459-0.860224399558j)*x[0]**o + ((0.101006402419-0.545007018286j))*x[0] + ((-0.799094825353-0.415067885865j))*x[1]**o + ((-0.0668867627279+0.384809702868j))*x[1]
            ref[(0,)]=(-1.03851414181-1.27529228542j)*(1+2.*(dim-1)/(o+1.)) + ((0.0341196396911-0.160197315418j))*dim
            arg[(1,)]=(0.676958645193-0.479880606465j)*x[0]**o + ((0.28761677357+0.60791142678j))*x[0] + ((0.501518375699+0.134257529234j))*x[1]**o + ((-0.667344754485-0.144902808787j))*x[1]
            ref[(1,)]=(1.17847702089-0.345623077231j)*(1+2.*(dim-1)/(o+1.)) + ((-0.379727980915+0.463008617993j))*dim
        else:
            arg[(0,)]=(0.217216895933+0.0940987040237j)*x[0]**o + ((-0.891587178817+0.0249231069339j))*x[0] + ((0.783065364699-0.503566511087j))*x[1]**o + ((0.0389635958782-0.451069019759j))*x[1] + ((0.0331163639087-0.850475786j))*x[2]**o + ((-0.0126803262323-0.803302321586j))*x[2]
            ref[(0,)]=(1.03339862454-1.25994359306j)*(1+2.*(dim-1)/(o+1.)) + ((-0.865303909171-1.22944823441j))*dim
            arg[(1,)]=(-0.552723142576+0.61939367774j)*x[0]**o + ((0.0575606557937+0.0236077171996j))*x[0] + ((0.0339076030471+0.887188725083j))*x[1]**o + ((-0.043147461755+0.207858848164j))*x[1] + ((0.511645436329-0.216833268084j))*x[2]**o + ((0.809086988616-0.515136387202j))*x[2]
            ref[(1,)]=(-0.00717010319991+1.28974913474j)*(1+2.*(dim-1)/(o+1.)) + ((0.823500182655-0.283669821839j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (2,),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_Solution_rank2(self):
        """
        tests integral of rank 2 Data on FunctionOnBoundary

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(4, 3),w)
        ref=numpy.zeros((4, 3),numpy.complex_)
        if dim==2:
            arg[(0, 0)]=(0.728893796233+0.609847260281j)*x[0]**o + ((0.375426733699+0.569095641377j))*x[0] + ((0.386841559073+0.134457426122j))*x[1]**o + ((-0.775080700678-0.361392367341j))*x[1]
            ref[(0, 0)]=(1.11573535531+0.744304686404j)*(1+2.*(dim-1)/(o+1.)) + ((-0.399653966979+0.207703274036j))*dim
            arg[(0, 1)]=(0.313156451543-0.633550274595j)*x[0]**o + ((0.964060873372-0.654559811523j))*x[0] + ((0.346399905262-0.657185260028j))*x[1]**o + ((0.0725880258497+0.881905176898j))*x[1]
            ref[(0, 1)]=(0.659556356805-1.29073553462j)*(1+2.*(dim-1)/(o+1.)) + ((1.03664889922+0.227345365375j))*dim
            arg[(0, 2)]=(0.80315096066-0.395373210355j)*x[0]**o + ((0.477530591633+0.498725104098j))*x[0] + ((0.368126948988+0.417328838097j))*x[1]**o + ((-0.862888736091+0.60728411491j))*x[1]
            ref[(0, 2)]=(1.17127790965+0.0219556277423j)*(1+2.*(dim-1)/(o+1.)) + ((-0.385358144458+1.10600921901j))*dim
            arg[(1, 0)]=(-0.812438354077-0.444355312773j)*x[0]**o + ((-0.777703225729-0.989131794361j))*x[0] + ((-0.482977730062-0.538062813468j))*x[1]**o + ((-0.0620840521635-0.797636538533j))*x[1]
            ref[(1, 0)]=(-1.29541608414-0.982418126241j)*(1+2.*(dim-1)/(o+1.)) + ((-0.839787277893-1.78676833289j))*dim
            arg[(1, 1)]=(-0.950295199367-0.132602532296j)*x[0]**o + ((0.798410361249+0.32997579905j))*x[0] + ((-0.129971542072-0.97466416798j))*x[1]**o + ((-0.59190133288+0.103046473432j))*x[1]
            ref[(1, 1)]=(-1.08026674144-1.10726670028j)*(1+2.*(dim-1)/(o+1.)) + ((0.20650902837+0.433022272481j))*dim
            arg[(1, 2)]=(0.027630046615-0.631407359705j)*x[0]**o + ((0.712713277541+0.309078424714j))*x[0] + ((-0.637736849044-0.974051640699j))*x[1]**o + ((-0.0286966688096-0.839520687947j))*x[1]
            ref[(1, 2)]=(-0.610106802429-1.6054590004j)*(1+2.*(dim-1)/(o+1.)) + ((0.684016608731-0.530442263233j))*dim
            arg[(2, 0)]=(-0.429213235675+0.326378998921j)*x[0]**o + ((0.0561356889089+0.0536765716093j))*x[0] + ((-0.616152443481+0.125412908396j))*x[1]**o + ((0.434394980215-0.933414909989j))*x[1]
            ref[(2, 0)]=(-1.04536567916+0.451791907318j)*(1+2.*(dim-1)/(o+1.)) + ((0.490530669123-0.87973833838j))*dim
            arg[(2, 1)]=(-0.144460479747-0.322914802549j)*x[0]**o + ((0.581102249244-0.29061482812j))*x[0] + ((0.0400062868011-0.870950590725j))*x[1]**o + ((-0.675706911509-0.325613690633j))*x[1]
            ref[(2, 1)]=(-0.104454192946-1.19386539327j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0946046622648-0.616228518752j))*dim
            arg[(2, 2)]=(-0.448378973166-0.396390186101j)*x[0]**o + ((-0.866683449354-0.487199336449j))*x[0] + ((-0.0577865039406-0.810761146098j))*x[1]**o + ((0.108371468688-0.738547944074j))*x[1]
            ref[(2, 2)]=(-0.506165477107-1.2071513322j)*(1+2.*(dim-1)/(o+1.)) + ((-0.758311980666-1.22574728052j))*dim
            arg[(3, 0)]=(0.82278746834-0.0469966357865j)*x[0]**o + ((0.205112726308-0.159585788096j))*x[0] + ((-0.998698716427-0.672540354171j))*x[1]**o + ((-0.826883015295+0.280482752401j))*x[1]
            ref[(3, 0)]=(-0.175911248087-0.719536989958j)*(1+2.*(dim-1)/(o+1.)) + ((-0.621770288987+0.120896964305j))*dim
            arg[(3, 1)]=(-0.826566928141-0.151371874451j)*x[0]**o + ((-0.412099935583+0.432891690834j))*x[0] + ((-0.763817599841-0.0199891874846j))*x[1]**o + ((-0.702572079903+0.695138715557j))*x[1]
            ref[(3, 1)]=(-1.59038452798-0.171361061936j)*(1+2.*(dim-1)/(o+1.)) + ((-1.11467201549+1.12803040639j))*dim
            arg[(3, 2)]=(-0.840087888758-0.845975244805j)*x[0]**o + ((0.543047698148-0.555601135507j))*x[0] + ((0.231218969056-0.380968896982j))*x[1]**o + ((-0.553636241874+0.662913202509j))*x[1]
            ref[(3, 2)]=(-0.608868919702-1.22694414179j)*(1+2.*(dim-1)/(o+1.)) + ((-0.010588543726+0.107312067003j))*dim
        else:
            arg[(0, 0)]=(0.727218709301+0.592155062742j)*x[0]**o + ((-0.601598475615-0.0481178839906j))*x[0] + ((-0.866405398775+0.903274524441j))*x[1]**o + ((-0.675591335357-0.159892160003j))*x[1] + ((0.991316066538-0.267781804494j))*x[2]**o + ((0.749224504081-0.0718676232158j))*x[2]
            ref[(0, 0)]=(0.852129377064+1.22764778269j)*(1+2.*(dim-1)/(o+1.)) + ((-0.52796530689-0.279877667209j))*dim
            arg[(0, 1)]=(-0.20817684799-0.911032761157j)*x[0]**o + ((-0.934293323345-0.505061905359j))*x[0] + ((-0.0189722130153+0.108510960482j))*x[1]**o + ((0.49016422516-0.0670216618515j))*x[1] + ((-0.831959627445-0.440151690164j))*x[2]**o + ((-0.410970245707+0.0221120451437j))*x[2]
            ref[(0, 1)]=(-1.05910868845-1.24267349084j)*(1+2.*(dim-1)/(o+1.)) + ((-0.855099343892-0.549971522067j))*dim
            arg[(0, 2)]=(0.216850769596+0.329617838656j)*x[0]**o + ((0.273778126476-0.3182890961j))*x[0] + ((0.949084220938-0.143731835015j))*x[1]**o + ((0.211131020855-0.578059982171j))*x[1] + ((-0.0336102281496-0.803708287168j))*x[2]**o + ((0.855353874769+0.244271134143j))*x[2]
            ref[(0, 2)]=(1.13232476238-0.617822283527j)*(1+2.*(dim-1)/(o+1.)) + ((1.3402630221-0.652077944129j))*dim
            arg[(1, 0)]=(-0.595376793601-0.892094847996j)*x[0]**o + ((0.488221600628+0.0593435988011j))*x[0] + ((0.885872773465+0.656556464075j))*x[1]**o + ((-0.553619282757-0.247636046409j))*x[1] + ((-0.553772095373-0.794048043569j))*x[2]**o + ((-0.355435394226-0.492500819955j))*x[2]
            ref[(1, 0)]=(-0.26327611551-1.02958642749j)*(1+2.*(dim-1)/(o+1.)) + ((-0.420833076355-0.680793267563j))*dim
            arg[(1, 1)]=(-0.17897733418+0.137793449864j)*x[0]**o + ((-0.753423977544-0.741917578972j))*x[0] + ((0.821911587893+0.45499651471j))*x[1]**o + ((-0.295154027445-0.230475674517j))*x[1] + ((-0.354377684801+0.824488202817j))*x[2]**o + ((-0.480603065579+0.426046678967j))*x[2]
            ref[(1, 1)]=(0.288556568912+1.41727816739j)*(1+2.*(dim-1)/(o+1.)) + ((-1.52918107057-0.546346574522j))*dim
            arg[(1, 2)]=(-0.226935898677-0.779157604862j)*x[0]**o + ((0.829193077449-0.942265126012j))*x[0] + ((-0.44491728444-0.653681622744j))*x[1]**o + ((-0.152034761969-0.463384106402j))*x[1] + ((-0.619942363753-0.653306413122j))*x[2]**o + ((0.341937575546-0.197702621452j))*x[2]
            ref[(1, 2)]=(-1.29179554687-2.08614564073j)*(1+2.*(dim-1)/(o+1.)) + ((1.01909589103-1.60335185387j))*dim
            arg[(2, 0)]=(0.910752312611+0.672638230278j)*x[0]**o + ((0.653357932665-0.427560749113j))*x[0] + ((0.25103229939+0.0762292447552j))*x[1]**o + ((0.450261063744-0.974109915571j))*x[1] + ((-0.72739786185+0.805319245636j))*x[2]**o + ((0.46419283619+0.775546203881j))*x[2]
            ref[(2, 0)]=(0.434386750151+1.55418672067j)*(1+2.*(dim-1)/(o+1.)) + ((1.5678118326-0.626124460803j))*dim
            arg[(2, 1)]=(-0.200016182834-0.923493357894j)*x[0]**o + ((0.621811580441+0.233253002037j))*x[0] + ((-0.567263302661-0.19486070756j))*x[1]**o + ((-0.601850377141+0.310868955895j))*x[1] + ((0.0245456837217+0.659790444804j))*x[2]**o + ((-0.541290232336-0.536982181205j))*x[2]
            ref[(2, 1)]=(-0.742733801773-0.458563620651j)*(1+2.*(dim-1)/(o+1.)) + ((-0.521329029037+0.00713977672786j))*dim
            arg[(2, 2)]=(0.182747462505-0.548192595942j)*x[0]**o + ((-0.108714716979-0.803102236818j))*x[0] + ((0.21158491298+0.757265652381j))*x[1]**o + ((0.454698289026-0.206313755471j))*x[1] + ((0.795902972659+0.217292370424j))*x[2]**o + ((0.807904830777-0.267013680073j))*x[2]
            ref[(2, 2)]=(1.19023534814+0.426365426863j)*(1+2.*(dim-1)/(o+1.)) + ((1.15388840282-1.27642967236j))*dim
            arg[(3, 0)]=(-0.428921335601-0.490481388065j)*x[0]**o + ((-0.766373578405-0.975834907351j))*x[0] + ((-0.768811810876+0.255873780776j))*x[1]**o + ((0.271761491656-0.683635251142j))*x[1] + ((-0.646548628898+0.797696999896j))*x[2]**o + ((0.0225733521078+0.9728076063j))*x[2]
            ref[(3, 0)]=(-1.84428177538+0.563089392607j)*(1+2.*(dim-1)/(o+1.)) + ((-0.472038734641-0.686662552194j))*dim
            arg[(3, 1)]=(0.101483764523-0.361700229906j)*x[0]**o + ((0.675604703078+0.632816537197j))*x[0] + ((0.0600862463387+0.205758144239j))*x[1]**o + ((-0.347019518773-0.69020585958j))*x[1] + ((0.775190413277-0.906623995964j))*x[2]**o + ((0.851926235796-0.311743042541j))*x[2]
            ref[(3, 1)]=(0.936760424139-1.06256608163j)*(1+2.*(dim-1)/(o+1.)) + ((1.1805114201-0.369132364924j))*dim
            arg[(3, 2)]=(0.910514537375-0.628696476305j)*x[0]**o + ((-0.849990006723+0.584929180872j))*x[0] + ((-0.516603400088-0.997557184621j))*x[1]**o + ((0.972176687115+0.654433566622j))*x[1] + ((-0.672311809878-0.511071516807j))*x[2]**o + ((-0.203841103882-0.661344128286j))*x[2]
            ref[(3, 2)]=(-0.278400672592-2.13732517773j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0816544234904+0.578018619208j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 3),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_Solution_rank3(self):
        """
        tests integral of rank 3 Data on FunctionOnBoundary

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(4, 4, 4),w)
        ref=numpy.zeros((4, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0)]=(-0.741643231947+0.392529643397j)*x[0]**o + ((-0.641494359405+0.851048442851j))*x[0] + ((0.47726184457+0.552716949789j))*x[1]**o + ((-0.165794723857-0.563170485072j))*x[1]
            ref[(0, 0, 0)]=(-0.264381387377+0.945246593186j)*(1+2.*(dim-1)/(o+1.)) + ((-0.807289083263+0.287877957779j))*dim
            arg[(0, 0, 1)]=(0.430661806003-0.0531594910151j)*x[0]**o + ((0.667734622807+0.0224935132527j))*x[0] + ((0.789447592698+0.403594099569j))*x[1]**o + ((0.808678830808+0.24088023916j))*x[1]
            ref[(0, 0, 1)]=(1.2201093987+0.350434608554j)*(1+2.*(dim-1)/(o+1.)) + ((1.47641345361+0.263373752413j))*dim
            arg[(0, 0, 2)]=(0.626280980842+0.925126137839j)*x[0]**o + ((0.814334916131-0.802244478441j))*x[0] + ((-0.355159449475-0.705712819348j))*x[1]**o + ((-0.926206806547+0.602500256929j))*x[1]
            ref[(0, 0, 2)]=(0.271121531367+0.21941331849j)*(1+2.*(dim-1)/(o+1.)) + ((-0.111871890416-0.199744221512j))*dim
            arg[(0, 0, 3)]=(0.27001057582+0.172130813988j)*x[0]**o + ((0.319255614272-0.815776115831j))*x[0] + ((-0.777859506941+0.146216817429j))*x[1]**o + ((0.296672805615+0.493371155672j))*x[1]
            ref[(0, 0, 3)]=(-0.507848931121+0.318347631417j)*(1+2.*(dim-1)/(o+1.)) + ((0.615928419886-0.322404960159j))*dim
            arg[(0, 1, 0)]=(0.487639465313-0.680140002471j)*x[0]**o + ((-0.708823387201+0.813769776372j))*x[0] + ((0.62776793915+0.38471503333j))*x[1]**o + ((-0.477043973778-0.0699160168364j))*x[1]
            ref[(0, 1, 0)]=(1.11540740446-0.29542496914j)*(1+2.*(dim-1)/(o+1.)) + ((-1.18586736098+0.743853759536j))*dim
            arg[(0, 1, 1)]=(-0.622829586222-0.543627235293j)*x[0]**o + ((-0.10190004007-0.0330857279156j))*x[0] + ((-0.72208357514-0.646791855013j))*x[1]**o + ((0.0177568593282+0.507117883275j))*x[1]
            ref[(0, 1, 1)]=(-1.34491316136-1.19041909031j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0841431807414+0.47403215536j))*dim
            arg[(0, 1, 2)]=(0.606773967082-0.594981747477j)*x[0]**o + ((0.619048241405+0.022199118627j))*x[0] + ((-0.358899642138-0.072114500233j))*x[1]**o + ((-0.101637637707+0.507715021391j))*x[1]
            ref[(0, 1, 2)]=(0.247874324944-0.66709624771j)*(1+2.*(dim-1)/(o+1.)) + ((0.517410603697+0.529914140018j))*dim
            arg[(0, 1, 3)]=(-0.911857249315-0.0659702787295j)*x[0]**o + ((0.383836862514-0.483854052132j))*x[0] + ((0.762262088654+0.123962266761j))*x[1]**o + ((-0.0709872733445+0.115667131837j))*x[1]
            ref[(0, 1, 3)]=(-0.14959516066+0.057991988031j)*(1+2.*(dim-1)/(o+1.)) + ((0.312849589169-0.368186920295j))*dim
            arg[(0, 2, 0)]=(0.200304803357+0.296823098174j)*x[0]**o + ((-0.327079388603-0.0360787084403j))*x[0] + ((-0.577217898557+0.0462172190496j))*x[1]**o + ((0.873255649943+0.287766081944j))*x[1]
            ref[(0, 2, 0)]=(-0.3769130952+0.343040317224j)*(1+2.*(dim-1)/(o+1.)) + ((0.54617626134+0.251687373503j))*dim
            arg[(0, 2, 1)]=(0.199816275081+0.193066591343j)*x[0]**o + ((-0.219672232602-0.401002595501j))*x[0] + ((0.138156617712-0.909772116415j))*x[1]**o + ((-0.585429544201+0.705596979406j))*x[1]
            ref[(0, 2, 1)]=(0.337972892793-0.716705525072j)*(1+2.*(dim-1)/(o+1.)) + ((-0.805101776804+0.304594383905j))*dim
            arg[(0, 2, 2)]=(0.244482525033+0.222817974975j)*x[0]**o + ((0.463110419932-0.0864592968799j))*x[0] + ((-0.92104262098-0.245686372894j))*x[1]**o + ((0.494042241491-0.667237528301j))*x[1]
            ref[(0, 2, 2)]=(-0.676560095947-0.0228683979185j)*(1+2.*(dim-1)/(o+1.)) + ((0.957152661423-0.753696825181j))*dim
            arg[(0, 2, 3)]=(0.924333839928+0.629407662732j)*x[0]**o + ((-0.560656559015-0.741643019804j))*x[0] + ((-0.0321473294715+0.719702061587j))*x[1]**o + ((-0.331231200979+0.30316878876j))*x[1]
            ref[(0, 2, 3)]=(0.892186510456+1.34910972432j)*(1+2.*(dim-1)/(o+1.)) + ((-0.891887759995-0.438474231044j))*dim
            arg[(0, 3, 0)]=(0.235997977614+0.0612643293819j)*x[0]**o + ((-0.463573099714+0.781256050312j))*x[0] + ((-0.479071650329-0.947093599034j))*x[1]**o + ((0.505588744845-0.504134611024j))*x[1]
            ref[(0, 3, 0)]=(-0.243073672715-0.885829269652j)*(1+2.*(dim-1)/(o+1.)) + ((0.0420156451314+0.277121439288j))*dim
            arg[(0, 3, 1)]=(-0.395929234943-0.441011539969j)*x[0]**o + ((-0.066264594006+0.571321448146j))*x[0] + ((0.863776867813-0.907828176023j))*x[1]**o + ((-0.962555607129-0.0944118028008j))*x[1]
            ref[(0, 3, 1)]=(0.46784763287-1.34883971599j)*(1+2.*(dim-1)/(o+1.)) + ((-1.02882020114+0.476909645345j))*dim
            arg[(0, 3, 2)]=(-0.00810251547872-0.645752376646j)*x[0]**o + ((0.5053628896+0.64770013442j))*x[0] + ((-0.849342037729-0.741271715678j))*x[1]**o + ((0.631094188974+0.126469822912j))*x[1]
            ref[(0, 3, 2)]=(-0.857444553207-1.38702409232j)*(1+2.*(dim-1)/(o+1.)) + ((1.13645707857+0.774169957332j))*dim
            arg[(0, 3, 3)]=(-0.231361555653+0.543075013592j)*x[0]**o + ((0.337529374031-0.754536442775j))*x[0] + ((-0.529198218267-0.142238460901j))*x[1]**o + ((-0.602610178671+0.0517901557943j))*x[1]
            ref[(0, 3, 3)]=(-0.76055977392+0.40083655269j)*(1+2.*(dim-1)/(o+1.)) + ((-0.26508080464-0.70274628698j))*dim
            arg[(1, 0, 0)]=(-0.20963005681+0.515459866041j)*x[0]**o + ((-0.244980033803+0.906805247311j))*x[0] + ((-0.79470853301-0.975870952522j))*x[1]**o + ((-0.0212191665904+0.073719206709j))*x[1]
            ref[(1, 0, 0)]=(-1.00433858982-0.460411086481j)*(1+2.*(dim-1)/(o+1.)) + ((-0.266199200393+0.98052445402j))*dim
            arg[(1, 0, 1)]=(0.183069163626-0.580101525363j)*x[0]**o + ((0.820429602407+0.279143312311j))*x[0] + ((-0.992578811929-0.536741177669j))*x[1]**o + ((0.864428577459+0.555800606504j))*x[1]
            ref[(1, 0, 1)]=(-0.809509648302-1.11684270303j)*(1+2.*(dim-1)/(o+1.)) + ((1.68485817987+0.834943918815j))*dim
            arg[(1, 0, 2)]=(0.0531799710127-0.197722176745j)*x[0]**o + ((-0.413998245333+0.765764048095j))*x[0] + ((-0.14430934083+0.189508512663j))*x[1]**o + ((-0.322375855284+0.898378972599j))*x[1]
            ref[(1, 0, 2)]=(-0.0911293698172-0.00821366408209j)*(1+2.*(dim-1)/(o+1.)) + ((-0.736374100617+1.66414302069j))*dim
            arg[(1, 0, 3)]=(0.836549139976-0.835590459598j)*x[0]**o + ((0.309680207068+0.730175460607j))*x[0] + ((-0.852514952207-0.0367066056843j))*x[1]**o + ((0.389203825664-0.578920374798j))*x[1]
            ref[(1, 0, 3)]=(-0.0159658122309-0.872297065283j)*(1+2.*(dim-1)/(o+1.)) + ((0.698884032732+0.15125508581j))*dim
            arg[(1, 1, 0)]=(-0.570885874212+0.406232127912j)*x[0]**o + ((-0.407015955751+0.320157888082j))*x[0] + ((-0.74482472444+0.0558211605104j))*x[1]**o + ((-0.953917603554+0.48446505164j))*x[1]
            ref[(1, 1, 0)]=(-1.31571059865+0.462053288422j)*(1+2.*(dim-1)/(o+1.)) + ((-1.3609335593+0.804622939722j))*dim
            arg[(1, 1, 1)]=(0.959920199368+0.0913250743508j)*x[0]**o + ((-0.542763788032+0.113945808522j))*x[0] + ((-0.601864810366-0.348783957052j))*x[1]**o + ((0.574024370528+0.560407076457j))*x[1]
            ref[(1, 1, 1)]=(0.358055389002-0.257458882702j)*(1+2.*(dim-1)/(o+1.)) + ((0.0312605824961+0.674352884979j))*dim
            arg[(1, 1, 2)]=(-0.117999242604-0.373777707355j)*x[0]**o + ((0.294464930496+0.0363233402196j))*x[0] + ((0.602670729688+0.828742864899j))*x[1]**o + ((-0.919706160447-0.443776546061j))*x[1]
            ref[(1, 1, 2)]=(0.484671487083+0.454965157544j)*(1+2.*(dim-1)/(o+1.)) + ((-0.625241229952-0.407453205841j))*dim
            arg[(1, 1, 3)]=(0.437025043327-0.500630896788j)*x[0]**o + ((0.0332436735949+0.305973816495j))*x[0] + ((-0.860382362251-0.71202811437j))*x[1]**o + ((-0.345274857235-0.312005610576j))*x[1]
            ref[(1, 1, 3)]=(-0.423357318924-1.21265901116j)*(1+2.*(dim-1)/(o+1.)) + ((-0.31203118364-0.00603179408145j))*dim
            arg[(1, 2, 0)]=(0.888045614689+0.828719165858j)*x[0]**o + ((-0.758503805835-0.955759692095j))*x[0] + ((0.823230028419+0.0323500117262j))*x[1]**o + ((0.609889654628+0.0480059959563j))*x[1]
            ref[(1, 2, 0)]=(1.71127564311+0.861069177584j)*(1+2.*(dim-1)/(o+1.)) + ((-0.148614151207-0.907753696139j))*dim
            arg[(1, 2, 1)]=(-0.894320735388+0.834902903333j)*x[0]**o + ((0.690392671953-0.174763598467j))*x[0] + ((0.234455321478+0.516189066626j))*x[1]**o + ((-0.923367453232+0.204104481377j))*x[1]
            ref[(1, 2, 1)]=(-0.65986541391+1.35109196996j)*(1+2.*(dim-1)/(o+1.)) + ((-0.232974781279+0.0293408829098j))*dim
            arg[(1, 2, 2)]=(0.290834423773-0.454216836468j)*x[0]**o + ((-0.509067442118+0.18097454022j))*x[0] + ((-0.110111734099+0.104515941784j))*x[1]**o + ((0.831653927912-0.251375075747j))*x[1]
            ref[(1, 2, 2)]=(0.180722689674-0.349700894685j)*(1+2.*(dim-1)/(o+1.)) + ((0.322586485794-0.0704005355272j))*dim
            arg[(1, 2, 3)]=(0.793458824083-0.71401326823j)*x[0]**o + ((-0.75736855999+0.7234587844j))*x[0] + ((0.857349588813+0.906729809747j))*x[1]**o + ((0.755594175189+0.578280653801j))*x[1]
            ref[(1, 2, 3)]=(1.6508084129+0.192716541517j)*(1+2.*(dim-1)/(o+1.)) + ((-0.00177438480133+1.3017394382j))*dim
            arg[(1, 3, 0)]=(0.478691498001+0.232780638234j)*x[0]**o + ((0.605849604062-0.177614392974j))*x[0] + ((0.958911307548-0.102968322509j))*x[1]**o + ((-0.927543689367+0.805606000657j))*x[1]
            ref[(1, 3, 0)]=(1.43760280555+0.129812315725j)*(1+2.*(dim-1)/(o+1.)) + ((-0.321694085305+0.627991607683j))*dim
            arg[(1, 3, 1)]=(0.891106232063+0.802633932846j)*x[0]**o + ((0.537285173092-0.180153118525j))*x[0] + ((-0.57825319751+0.0974364786052j))*x[1]**o + ((0.170494073116+0.289191915917j))*x[1]
            ref[(1, 3, 1)]=(0.312853034553+0.900070411451j)*(1+2.*(dim-1)/(o+1.)) + ((0.707779246208+0.109038797392j))*dim
            arg[(1, 3, 2)]=(0.574493501524+0.472671527627j)*x[0]**o + ((0.0452299246167+0.161358959672j))*x[0] + ((0.498295055725+0.942994568924j))*x[1]**o + ((0.615185143619-0.184017979853j))*x[1]
            ref[(1, 3, 2)]=(1.07278855725+1.41566609655j)*(1+2.*(dim-1)/(o+1.)) + ((0.660415068236-0.0226590201809j))*dim
            arg[(1, 3, 3)]=(-0.547301446629-0.44094107405j)*x[0]**o + ((-0.855359109959-0.84585102077j))*x[0] + ((0.901062639572-0.724531317521j))*x[1]**o + ((-0.729913786376-0.817208050038j))*x[1]
            ref[(1, 3, 3)]=(0.353761192943-1.16547239157j)*(1+2.*(dim-1)/(o+1.)) + ((-1.58527289633-1.66305907081j))*dim
            arg[(2, 0, 0)]=(0.389750439939-0.851739950601j)*x[0]**o + ((-0.643315380028+0.238898440511j))*x[0] + ((-0.6295194741+0.0734983700873j))*x[1]**o + ((-0.935876359743+0.459758863484j))*x[1]
            ref[(2, 0, 0)]=(-0.239769034161-0.778241580513j)*(1+2.*(dim-1)/(o+1.)) + ((-1.57919173977+0.698657303994j))*dim
            arg[(2, 0, 1)]=(0.564358364512-0.889390962981j)*x[0]**o + ((-0.3633977304-0.015620542285j))*x[0] + ((0.556327319358-0.562309317042j))*x[1]**o + ((0.727649861203+0.869978765058j))*x[1]
            ref[(2, 0, 1)]=(1.12068568387-1.45170028002j)*(1+2.*(dim-1)/(o+1.)) + ((0.364252130803+0.854358222773j))*dim
            arg[(2, 0, 2)]=(0.782811496871-0.393316771899j)*x[0]**o + ((0.734005038676+0.734505959858j))*x[0] + ((0.558601432834+0.640561031797j))*x[1]**o + ((-0.761573196125-0.901561516463j))*x[1]
            ref[(2, 0, 2)]=(1.3414129297+0.247244259898j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0275681574483-0.167055556605j))*dim
            arg[(2, 0, 3)]=(0.273677632595+0.353287376321j)*x[0]**o + ((0.749171243372-0.166350084895j))*x[0] + ((-0.306917822304+0.581824744959j))*x[1]**o + ((0.908099442478+0.123984587645j))*x[1]
            ref[(2, 0, 3)]=(-0.0332401897086+0.935112121279j)*(1+2.*(dim-1)/(o+1.)) + ((1.65727068585-0.042365497249j))*dim
            arg[(2, 1, 0)]=(-0.0482498583672-0.695476996538j)*x[0]**o + ((0.745320288823-0.697221840533j))*x[0] + ((0.905311611713-0.994453130233j))*x[1]**o + ((0.00734924911393+0.191731262343j))*x[1]
            ref[(2, 1, 0)]=(0.857061753346-1.68993012677j)*(1+2.*(dim-1)/(o+1.)) + ((0.752669537937-0.50549057819j))*dim
            arg[(2, 1, 1)]=(-0.99731379493+0.180634361053j)*x[0]**o + ((-0.934107760975-0.50513359926j))*x[0] + ((0.725475869422-0.235354046915j))*x[1]**o + ((-0.595230644035+0.441364259341j))*x[1]
            ref[(2, 1, 1)]=(-0.271837925509-0.0547196858626j)*(1+2.*(dim-1)/(o+1.)) + ((-1.52933840501-0.0637693399188j))*dim
            arg[(2, 1, 2)]=(0.224546170077+0.327527235878j)*x[0]**o + ((-0.666402552205+0.882279457052j))*x[0] + ((0.302806073965+0.831966200349j))*x[1]**o + ((-0.64268175783-0.124833169619j))*x[1]
            ref[(2, 1, 2)]=(0.527352244042+1.15949343623j)*(1+2.*(dim-1)/(o+1.)) + ((-1.30908431003+0.757446287433j))*dim
            arg[(2, 1, 3)]=(0.0723205611301+0.895695540363j)*x[0]**o + ((-0.202246588856+0.003050475108j))*x[0] + ((-0.634253891681+0.0935891385071j))*x[1]**o + ((0.566261563691-0.0138239211617j))*x[1]
            ref[(2, 1, 3)]=(-0.561933330551+0.98928467887j)*(1+2.*(dim-1)/(o+1.)) + ((0.364014974835-0.0107734460537j))*dim
            arg[(2, 2, 0)]=(-0.230203879485-0.0537962864805j)*x[0]**o + ((-0.9304111912-0.132787423341j))*x[0] + ((-0.93378742138+0.784234902375j))*x[1]**o + ((-0.212197749118+0.599654571714j))*x[1]
            ref[(2, 2, 0)]=(-1.16399130087+0.730438615894j)*(1+2.*(dim-1)/(o+1.)) + ((-1.14260894032+0.466867148374j))*dim
            arg[(2, 2, 1)]=(0.899533788082-0.423748793016j)*x[0]**o + ((0.0618514120339+0.85904092384j))*x[0] + ((-0.819723953886-0.226969535561j))*x[1]**o + ((0.651604809782-0.104880869561j))*x[1]
            ref[(2, 2, 1)]=(0.0798098341955-0.650718328578j)*(1+2.*(dim-1)/(o+1.)) + ((0.713456221816+0.754160054279j))*dim
            arg[(2, 2, 2)]=(-0.847679494336-0.869823420441j)*x[0]**o + ((-0.260627755929-0.0856970513345j))*x[0] + ((0.451765909938+0.0513964659988j))*x[1]**o + ((0.137095545612+0.433721149922j))*x[1]
            ref[(2, 2, 2)]=(-0.395913584398-0.818426954442j)*(1+2.*(dim-1)/(o+1.)) + ((-0.123532210318+0.348024098588j))*dim
            arg[(2, 2, 3)]=(-0.0150523269862+0.894153911589j)*x[0]**o + ((-0.0922093939653+0.162493525164j))*x[0] + ((-0.646195852545+0.78953502718j))*x[1]**o + ((0.603921002448-0.9847200778j))*x[1]
            ref[(2, 2, 3)]=(-0.661248179531+1.68368893877j)*(1+2.*(dim-1)/(o+1.)) + ((0.511711608483-0.822226552636j))*dim
            arg[(2, 3, 0)]=(-0.769862728349+0.900610178901j)*x[0]**o + ((-0.612988367515+0.970373137207j))*x[0] + ((-0.193637529451+0.488011661114j))*x[1]**o + ((-0.788387860154-0.0312533492432j))*x[1]
            ref[(2, 3, 0)]=(-0.9635002578+1.38862184002j)*(1+2.*(dim-1)/(o+1.)) + ((-1.40137622767+0.939119787964j))*dim
            arg[(2, 3, 1)]=(0.481709075821-0.613566075589j)*x[0]**o + ((-0.462391374256+0.665236154229j))*x[0] + ((0.0305954870935-0.607539877682j))*x[1]**o + ((0.00404179077581-0.335955543563j))*x[1]
            ref[(2, 3, 1)]=(0.512304562915-1.22110595327j)*(1+2.*(dim-1)/(o+1.)) + ((-0.458349583481+0.329280610666j))*dim
            arg[(2, 3, 2)]=(0.835118351583+0.669484763766j)*x[0]**o + ((0.26369272777-0.415703266278j))*x[0] + ((-0.154576983652+0.180485157408j))*x[1]**o + ((-0.969380999543+0.739136368737j))*x[1]
            ref[(2, 3, 2)]=(0.680541367931+0.849969921173j)*(1+2.*(dim-1)/(o+1.)) + ((-0.705688271773+0.323433102459j))*dim
            arg[(2, 3, 3)]=(0.141688013072+0.611923704381j)*x[0]**o + ((-0.83663585151+0.42450956792j))*x[0] + ((0.00590922796829-0.387698500386j))*x[1]**o + ((0.0243850270438+0.795966705686j))*x[1]
            ref[(2, 3, 3)]=(0.147597241041+0.224225203996j)*(1+2.*(dim-1)/(o+1.)) + ((-0.812250824466+1.22047627361j))*dim
            arg[(3, 0, 0)]=(0.448833086176-0.910898043981j)*x[0]**o + ((0.701821627139-0.12065554612j))*x[0] + ((0.982158449059-0.496618001125j))*x[1]**o + ((0.793530866935+0.695584483509j))*x[1]
            ref[(3, 0, 0)]=(1.43099153523-1.40751604511j)*(1+2.*(dim-1)/(o+1.)) + ((1.49535249407+0.574928937389j))*dim
            arg[(3, 0, 1)]=(-0.0990680915456-0.992392887251j)*x[0]**o + ((0.795892021014-0.17632962714j))*x[0] + ((0.5840907974-0.239563207517j))*x[1]**o + ((0.634581567143-0.627311447383j))*x[1]
            ref[(3, 0, 1)]=(0.485022705855-1.23195609477j)*(1+2.*(dim-1)/(o+1.)) + ((1.43047358816-0.803641074522j))*dim
            arg[(3, 0, 2)]=(-0.0393373178469-0.697623409367j)*x[0]**o + ((0.134399924113+0.494065534276j))*x[0] + ((0.731139386+0.250579925858j))*x[1]**o + ((-0.0530063925445+0.374353031612j))*x[1]
            ref[(3, 0, 2)]=(0.691802068153-0.447043483509j)*(1+2.*(dim-1)/(o+1.)) + ((0.081393531569+0.868418565888j))*dim
            arg[(3, 0, 3)]=(0.348405431126-0.208621569937j)*x[0]**o + ((-0.699815100151+0.881736678828j))*x[0] + ((0.0230902029222-0.971945356569j))*x[1]**o + ((0.802337618582+0.939099092028j))*x[1]
            ref[(3, 0, 3)]=(0.371495634048-1.18056692651j)*(1+2.*(dim-1)/(o+1.)) + ((0.10252251843+1.82083577086j))*dim
            arg[(3, 1, 0)]=(-0.123959603021-0.268851397607j)*x[0]**o + ((0.628729947117-0.583674380617j))*x[0] + ((0.619106557044-0.981322109679j))*x[1]**o + ((0.971612773967-0.760281530099j))*x[1]
            ref[(3, 1, 0)]=(0.495146954023-1.25017350729j)*(1+2.*(dim-1)/(o+1.)) + ((1.60034272108-1.34395591072j))*dim
            arg[(3, 1, 1)]=(0.0794588670867+0.294015609303j)*x[0]**o + ((0.716192779317+0.196525437966j))*x[0] + ((-0.827650459136-0.832289068617j))*x[1]**o + ((-0.343235570858-0.912798130772j))*x[1]
            ref[(3, 1, 1)]=(-0.74819159205-0.538273459314j)*(1+2.*(dim-1)/(o+1.)) + ((0.37295720846-0.716272692805j))*dim
            arg[(3, 1, 2)]=(-0.64572257236+0.49744614383j)*x[0]**o + ((0.426850315457-0.0504298337776j))*x[0] + ((0.552155911726+0.844357705868j))*x[1]**o + ((-0.308932010709-0.648823859432j))*x[1]
            ref[(3, 1, 2)]=(-0.0935666606338+1.3418038497j)*(1+2.*(dim-1)/(o+1.)) + ((0.117918304748-0.699253693209j))*dim
            arg[(3, 1, 3)]=(-0.045454776355+0.479135073111j)*x[0]**o + ((-0.311402871159+0.279985850252j))*x[0] + ((-0.952293312604+0.886258744575j))*x[1]**o + ((-0.389043775266+0.116864934129j))*x[1]
            ref[(3, 1, 3)]=(-0.997748088959+1.36539381769j)*(1+2.*(dim-1)/(o+1.)) + ((-0.700446646425+0.396850784381j))*dim
            arg[(3, 2, 0)]=(0.490353159675-0.270458690978j)*x[0]**o + ((-0.0465783818757-0.101501989144j))*x[0] + ((-0.841792103664+0.664481486704j))*x[1]**o + ((-0.375865107711-0.212185513546j))*x[1]
            ref[(3, 2, 0)]=(-0.351438943989+0.394022795726j)*(1+2.*(dim-1)/(o+1.)) + ((-0.422443489587-0.31368750269j))*dim
            arg[(3, 2, 1)]=(-0.956494618732+0.400444844557j)*x[0]**o + ((0.0175606561068+0.297460094986j))*x[0] + ((0.310247722614-0.502667315425j))*x[1]**o + ((-0.630984853293-0.121911912632j))*x[1]
            ref[(3, 2, 1)]=(-0.646246896117-0.102222470868j)*(1+2.*(dim-1)/(o+1.)) + ((-0.613424197187+0.175548182354j))*dim
            arg[(3, 2, 2)]=(0.0963898664042-0.642801817387j)*x[0]**o + ((-0.063885789622-0.786769045626j))*x[0] + ((0.312203348823-0.169258538716j))*x[1]**o + ((0.130954031078-0.830602315703j))*x[1]
            ref[(3, 2, 2)]=(0.408593215227-0.812060356104j)*(1+2.*(dim-1)/(o+1.)) + ((0.0670682414561-1.61737136133j))*dim
            arg[(3, 2, 3)]=(-0.481945194556-0.394002688615j)*x[0]**o + ((-0.417721975995-0.370657727533j))*x[0] + ((-0.0422710541177+0.429711704928j))*x[1]**o + ((-0.690046159088-0.870913756007j))*x[1]
            ref[(3, 2, 3)]=(-0.524216248674+0.0357090163129j)*(1+2.*(dim-1)/(o+1.)) + ((-1.10776813508-1.24157148354j))*dim
            arg[(3, 3, 0)]=(-0.803011730387+0.780288856852j)*x[0]**o + ((-0.04780811569+0.816585541092j))*x[0] + ((0.158234420206-0.10377432393j))*x[1]**o + ((-0.164279620184+0.792339099478j))*x[1]
            ref[(3, 3, 0)]=(-0.644777310181+0.676514532922j)*(1+2.*(dim-1)/(o+1.)) + ((-0.212087735874+1.60892464057j))*dim
            arg[(3, 3, 1)]=(-0.609920685685-0.45367809027j)*x[0]**o + ((-0.402457961603+0.354643677493j))*x[0] + ((0.582780630138-0.844150744934j))*x[1]**o + ((-0.0760036361536+0.0529357997816j))*x[1]
            ref[(3, 3, 1)]=(-0.0271400555474-1.2978288352j)*(1+2.*(dim-1)/(o+1.)) + ((-0.478461597757+0.407579477275j))*dim
            arg[(3, 3, 2)]=(0.636689602719-0.291731073457j)*x[0]**o + ((-0.255873901287-0.392383174582j))*x[0] + ((-0.635676308465+0.456516162368j))*x[1]**o + ((0.139796103985-0.289477119762j))*x[1]
            ref[(3, 3, 2)]=(0.00101329425457+0.164785088911j)*(1+2.*(dim-1)/(o+1.)) + ((-0.116077797302-0.681860294344j))*dim
            arg[(3, 3, 3)]=(0.459734964125-0.00111200032175j)*x[0]**o + ((-0.17680604433+0.802516687447j))*x[0] + ((0.158863731589-0.559229725592j))*x[1]**o + ((-0.590956288407-0.0305197590714j))*x[1]
            ref[(3, 3, 3)]=(0.618598695713-0.560341725913j)*(1+2.*(dim-1)/(o+1.)) + ((-0.767762332738+0.771996928375j))*dim
        else:
            arg[(0, 0, 0)]=(0.184268034071+0.913639292929j)*x[0]**o + ((0.632519775892-0.515754512055j))*x[0] + ((-0.31087280567-0.145048697334j))*x[1]**o + ((0.98472861541-0.560341112998j))*x[1] + ((-0.610323785205-0.584134134215j))*x[2]**o + ((0.904096031395-0.0181532600634j))*x[2]
            ref[(0, 0, 0)]=(-0.736928556804+0.184456461381j)*(1+2.*(dim-1)/(o+1.)) + ((2.5213444227-1.09424888512j))*dim
            arg[(0, 0, 1)]=(0.782379444514+0.8217486655j)*x[0]**o + ((0.525835368971-0.957153784899j))*x[0] + ((0.447755233647-0.233644198921j))*x[1]**o + ((-0.359078727192-0.588008194974j))*x[1] + ((-0.219002324372-0.828476245881j))*x[2]**o + ((0.0530070804494+0.0318947406656j))*x[2]
            ref[(0, 0, 1)]=(1.01113235379-0.240371779301j)*(1+2.*(dim-1)/(o+1.)) + ((0.219763722228-1.51326723921j))*dim
            arg[(0, 0, 2)]=(0.432354994282+0.952866296592j)*x[0]**o + ((0.33532530616+0.471724277128j))*x[0] + ((0.246608471658-0.552016787868j))*x[1]**o + ((-0.274290368776+0.0914035627638j))*x[1] + ((0.828249657651-0.182571987754j))*x[2]**o + ((-0.591202421585-0.257445951265j))*x[2]
            ref[(0, 0, 2)]=(1.50721312359+0.21827752097j)*(1+2.*(dim-1)/(o+1.)) + ((-0.530167484201+0.305681888627j))*dim
            arg[(0, 0, 3)]=(0.567409565863+0.0311097060645j)*x[0]**o + ((-0.217815646628+0.195248527488j))*x[0] + ((-0.538985071718+0.208396342135j))*x[1]**o + ((0.36881965252+0.823780687032j))*x[1] + ((0.745735472585+0.256509377095j))*x[2]**o + ((0.415029354399-0.0463303050981j))*x[2]
            ref[(0, 0, 3)]=(0.77415996673+0.496015425294j)*(1+2.*(dim-1)/(o+1.)) + ((0.566033360291+0.972698909422j))*dim
            arg[(0, 1, 0)]=(0.173950025989+0.691175316716j)*x[0]**o + ((-0.233614827355-0.0243772578365j))*x[0] + ((0.109548752056-0.141745246225j))*x[1]**o + ((-0.910705439484-0.810190236231j))*x[1] + ((0.377622266256+0.614166584093j))*x[2]**o + ((-0.692732424972-0.776327645436j))*x[2]
            ref[(0, 1, 0)]=(0.661121044301+1.16359665458j)*(1+2.*(dim-1)/(o+1.)) + ((-1.83705269181-1.6108951395j))*dim
            arg[(0, 1, 1)]=(-0.420599208145+0.208965560983j)*x[0]**o + ((0.487631254966+0.812452156045j))*x[0] + ((-0.252196641613-0.463596000319j))*x[1]**o + ((0.641692541688+0.467122921108j))*x[1] + ((-0.530005421232-0.902379620264j))*x[2]**o + ((-0.944701087013+0.15214633051j))*x[2]
            ref[(0, 1, 1)]=(-1.20280127099-1.1570100596j)*(1+2.*(dim-1)/(o+1.)) + ((0.184622709641+1.43172140766j))*dim
            arg[(0, 1, 2)]=(-0.765133722834+0.211710316434j)*x[0]**o + ((-0.835490631312-0.406078014511j))*x[0] + ((-0.622098187095-0.797046966115j))*x[1]**o + ((0.711264295911-0.670552317025j))*x[1] + ((-0.825132015328-0.695496775068j))*x[2]**o + ((0.704174675607-0.234361679994j))*x[2]
            ref[(0, 1, 2)]=(-2.21236392526-1.28083342475j)*(1+2.*(dim-1)/(o+1.)) + ((0.579948340206-1.31099201153j))*dim
            arg[(0, 1, 3)]=(-0.00683559813551-0.207926992751j)*x[0]**o + ((0.732577344422+0.461821152901j))*x[0] + ((0.878762534487-0.695256035722j))*x[1]**o + ((-0.164380230276+0.84015723533j))*x[1] + ((-0.837021668491-0.470354840341j))*x[2]**o + ((-0.221450748773-0.520885295055j))*x[2]
            ref[(0, 1, 3)]=(0.0349052678602-1.37353786881j)*(1+2.*(dim-1)/(o+1.)) + ((0.346746365372+0.781093093176j))*dim
            arg[(0, 2, 0)]=(-0.711974059885-0.595677035886j)*x[0]**o + ((0.870807509449-0.657447323509j))*x[0] + ((-0.393056681583+0.412250112052j))*x[1]**o + ((0.0368767496277+0.116779318966j))*x[1] + ((-0.0563521264176+0.842329153263j))*x[2]**o + ((0.835794153855+0.386093954124j))*x[2]
            ref[(0, 2, 0)]=(-1.16138286789+0.658902229428j)*(1+2.*(dim-1)/(o+1.)) + ((1.74347841293-0.154574050419j))*dim
            arg[(0, 2, 1)]=(0.284555364548+0.110823586877j)*x[0]**o + ((-0.859154301774+0.327653027062j))*x[0] + ((-0.148850172185-0.571907614641j))*x[1]**o + ((-0.241667239787+0.510275402772j))*x[1] + ((0.737380264294+0.840799692932j))*x[2]**o + ((0.658913497719+0.872345707055j))*x[2]
            ref[(0, 2, 1)]=(0.873085456657+0.379715665169j)*(1+2.*(dim-1)/(o+1.)) + ((-0.441908043842+1.71027413689j))*dim
            arg[(0, 2, 2)]=(-0.751353934752-0.368024745476j)*x[0]**o + ((0.65876102974-0.614483483587j))*x[0] + ((-0.894055660466+0.495352743654j))*x[1]**o + ((0.641376760399-0.505536026929j))*x[1] + ((-0.431138007713+0.262089633073j))*x[2]**o + ((-0.892989720054+0.671223682374j))*x[2]
            ref[(0, 2, 2)]=(-2.07654760293+0.389417631251j)*(1+2.*(dim-1)/(o+1.)) + ((0.407148070085-0.448795828143j))*dim
            arg[(0, 2, 3)]=(-0.820325028257-0.952522032858j)*x[0]**o + ((0.707824404867+0.72769412652j))*x[0] + ((0.0287161949052+0.437288271684j))*x[1]**o + ((-0.0525463013715+0.344776225777j))*x[1] + ((-0.452997124741-0.261905910783j))*x[2]**o + ((-0.915253341948-0.0595586452383j))*x[2]
            ref[(0, 2, 3)]=(-1.24460595809-0.777139671957j)*(1+2.*(dim-1)/(o+1.)) + ((-0.259975238453+1.01291170706j))*dim
            arg[(0, 3, 0)]=(0.218397388478+0.0110669270734j)*x[0]**o + ((-0.442891047718-0.820182702756j))*x[0] + ((0.816253832204+0.00419151631353j))*x[1]**o + ((0.631212313133+0.671692437607j))*x[1] + ((-0.778304816626+0.376637450816j))*x[2]**o + ((-0.758516933778-0.791329474653j))*x[2]
            ref[(0, 3, 0)]=(0.256346404056+0.391895894203j)*(1+2.*(dim-1)/(o+1.)) + ((-0.570195668364-0.939819739802j))*dim
            arg[(0, 3, 1)]=(-0.0223047364787+0.445760636023j)*x[0]**o + ((0.53406616384-0.672662929917j))*x[0] + ((0.0923567132532+0.810846487567j))*x[1]**o + ((-0.468694878917-0.581369375125j))*x[1] + ((-0.603419284974+0.982147053144j))*x[2]**o + ((0.683719832774-0.985337869241j))*x[2]
            ref[(0, 3, 1)]=(-0.533367308199+2.23875417673j)*(1+2.*(dim-1)/(o+1.)) + ((0.749091117697-2.23937017428j))*dim
            arg[(0, 3, 2)]=(-0.389164425495+0.752440145088j)*x[0]**o + ((-0.764411353734-0.768809878347j))*x[0] + ((-0.117871620281+0.569215755376j))*x[1]**o + ((0.534879354534+0.956477964058j))*x[1] + ((0.111052596039-0.154779029885j))*x[2]**o + ((-0.950210182464-0.0146462549024j))*x[2]
            ref[(0, 3, 2)]=(-0.395983449737+1.16687687058j)*(1+2.*(dim-1)/(o+1.)) + ((-1.17974218166+0.173021830808j))*dim
            arg[(0, 3, 3)]=(0.74782058519-0.391739791007j)*x[0]**o + ((0.46182753259-0.969830062225j))*x[0] + ((0.198120910402+0.111584654341j))*x[1]**o + ((0.84531693253+0.908262355532j))*x[1] + ((0.984300335906+0.660395829578j))*x[2]**o + ((0.748997854286+0.151494945782j))*x[2]
            ref[(0, 3, 3)]=(1.9302418315+0.380240692912j)*(1+2.*(dim-1)/(o+1.)) + ((2.05614231941+0.0899272390896j))*dim
            arg[(1, 0, 0)]=(0.19128730581-0.864295668485j)*x[0]**o + ((-0.692481576746+0.951660215829j))*x[0] + ((-0.737305903107-0.128994443042j))*x[1]**o + ((0.885040564583-0.0840682546054j))*x[1] + ((0.163587937352+0.515392015972j))*x[2]**o + ((-0.82955631558-0.0545356602267j))*x[2]
            ref[(1, 0, 0)]=(-0.382430659945-0.477898095555j)*(1+2.*(dim-1)/(o+1.)) + ((-0.636997327744+0.813056300997j))*dim
            arg[(1, 0, 1)]=(0.245036264342+0.113125113558j)*x[0]**o + ((0.50713072687+0.494753041184j))*x[0] + ((-0.255380179745+0.494682797878j))*x[1]**o + ((-0.226627311781-0.909786664682j))*x[1] + ((0.0338095285108-0.485862160096j))*x[2]**o + ((0.74116251061-0.872071186694j))*x[2]
            ref[(1, 0, 1)]=(0.0234656131079+0.12194575134j)*(1+2.*(dim-1)/(o+1.)) + ((1.0216659257-1.28710481019j))*dim
            arg[(1, 0, 2)]=(-0.229878820224-0.666534034216j)*x[0]**o + ((0.119701084675-0.593750635251j))*x[0] + ((-0.688460401822-0.16329152665j))*x[1]**o + ((0.170113254702-0.811912008932j))*x[1] + ((-0.723258201173+0.12439156222j))*x[2]**o + ((-0.0323330509403-0.890302955194j))*x[2]
            ref[(1, 0, 2)]=(-1.64159742322-0.705433998647j)*(1+2.*(dim-1)/(o+1.)) + ((0.257481288437-2.29596559938j))*dim
            arg[(1, 0, 3)]=(-0.261037816771+0.364718799932j)*x[0]**o + ((-0.229482782923-0.0556125106935j))*x[0] + ((0.174443565306-0.265803315044j))*x[1]**o + ((0.678506134382-0.494241012439j))*x[1] + ((-0.647703084249+0.180070841965j))*x[2]**o + ((-0.295503449835-0.207273296434j))*x[2]
            ref[(1, 0, 3)]=(-0.734297335715+0.278986326853j)*(1+2.*(dim-1)/(o+1.)) + ((0.153519901625-0.757126819566j))*dim
            arg[(1, 1, 0)]=(-0.502729933205+0.807177966968j)*x[0]**o + ((-0.425052341117-0.680767294127j))*x[0] + ((0.567732178053-0.351897014585j))*x[1]**o + ((0.353849143711-0.153997454943j))*x[1] + ((0.297127264971-0.176325289698j))*x[2]**o + ((0.00814837243853+0.720829344392j))*x[2]
            ref[(1, 1, 0)]=(0.362129509819+0.278955662685j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0630548249675-0.113935404678j))*dim
            arg[(1, 1, 1)]=(-0.986750472573+0.67232718115j)*x[0]**o + ((-0.454230841196-0.073919134792j))*x[0] + ((0.15816418726-0.678654066489j))*x[1]**o + ((0.121312247299+0.148873333339j))*x[1] + ((-0.291959301877+0.51448342502j))*x[2]**o + ((0.255315381733+0.619976296351j))*x[2]
            ref[(1, 1, 1)]=(-1.12054558719+0.508156539681j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0776032121633+0.694930494898j))*dim
            arg[(1, 1, 2)]=(-0.303297797775-0.60126669584j)*x[0]**o + ((0.567229943715+0.24638496431j))*x[0] + ((0.434018631714-0.701435117732j))*x[1]**o + ((-0.454370963138-0.656024232997j))*x[1] + ((-0.514885597582+0.168658018198j))*x[2]**o + ((0.583400328931-0.113511273057j))*x[2]
            ref[(1, 1, 2)]=(-0.384164763642-1.13404379537j)*(1+2.*(dim-1)/(o+1.)) + ((0.696259309508-0.523150541743j))*dim
            arg[(1, 1, 3)]=(-0.945226994808-0.23842645013j)*x[0]**o + ((0.878317705205-0.816980136835j))*x[0] + ((0.285431871022+0.309946060376j))*x[1]**o + ((-0.343870129456+0.32865026999j))*x[1] + ((-0.556702587218-0.502443165393j))*x[2]**o + ((-0.896344555287-0.274822988089j))*x[2]
            ref[(1, 1, 3)]=(-1.216497711-0.430923555147j)*(1+2.*(dim-1)/(o+1.)) + ((-0.361896979538-0.763152854935j))*dim
            arg[(1, 2, 0)]=(-0.514066017614+0.471663723824j)*x[0]**o + ((-0.191855142891-0.634167912547j))*x[0] + ((0.245166154378+0.817922762528j))*x[1]**o + ((0.802405865113+0.136146451255j))*x[1] + ((0.989562348423+0.211011956723j))*x[2]**o + ((0.547166859916+0.650406171839j))*x[2]
            ref[(1, 2, 0)]=(0.720662485187+1.50059844307j)*(1+2.*(dim-1)/(o+1.)) + ((1.15771758214+0.152384710547j))*dim
            arg[(1, 2, 1)]=(0.601663358394-0.947969292813j)*x[0]**o + ((0.672638053403-0.413465589319j))*x[0] + ((0.322635863252-0.62647054376j))*x[1]**o + ((0.651192172792-0.426349328354j))*x[1] + ((0.414637202118+0.187066746672j))*x[2]**o + ((-0.841698763897+0.614329773525j))*x[2]
            ref[(1, 2, 1)]=(1.33893642377-1.3873730899j)*(1+2.*(dim-1)/(o+1.)) + ((0.482131462298-0.225485144149j))*dim
            arg[(1, 2, 2)]=(-0.970714461208+0.512564805153j)*x[0]**o + ((0.945318758931-0.797438884914j))*x[0] + ((0.294034475733-0.163196711347j))*x[1]**o + ((-0.511507973879-0.107310011927j))*x[1] + ((-0.34437730897+0.344951160772j))*x[2]**o + ((-0.787376661707+0.18176592965j))*x[2]
            ref[(1, 2, 2)]=(-1.02105729445+0.694319254577j)*(1+2.*(dim-1)/(o+1.)) + ((-0.353565876655-0.72298296719j))*dim
            arg[(1, 2, 3)]=(0.685571272891-0.868245907241j)*x[0]**o + ((-0.79606151441-0.0234060952976j))*x[0] + ((0.32754521285+0.41207896904j))*x[1]**o + ((-0.791217075993+0.601498177856j))*x[1] + ((0.0151602413225-0.908111779226j))*x[2]**o + ((0.100129694115+0.846692317773j))*x[2]
            ref[(1, 2, 3)]=(1.02827672706-1.36427871743j)*(1+2.*(dim-1)/(o+1.)) + ((-1.48714889629+1.42478440033j))*dim
            arg[(1, 3, 0)]=(-0.434532459161-0.926163890228j)*x[0]**o + ((-0.366474318535-0.104736109657j))*x[0] + ((-0.473782878753+0.654134594779j))*x[1]**o + ((0.386884658229+0.649837296164j))*x[1] + ((-0.278026842051-0.833127685673j))*x[2]**o + ((-0.510830721384+0.793639122275j))*x[2]
            ref[(1, 3, 0)]=(-1.18634217997-1.10515698112j)*(1+2.*(dim-1)/(o+1.)) + ((-0.490420381689+1.33874030878j))*dim
            arg[(1, 3, 1)]=(0.232338186725+0.315838791259j)*x[0]**o + ((0.610273889331-0.0624824478141j))*x[0] + ((0.740045687078-0.761349158856j))*x[1]**o + ((0.883687795899+0.208253421203j))*x[1] + ((0.859525544895-0.511045599916j))*x[2]**o + ((-0.379607844405-0.4154658462j))*x[2]
            ref[(1, 3, 1)]=(1.8319094187-0.956555967512j)*(1+2.*(dim-1)/(o+1.)) + ((1.11435384083-0.269694872811j))*dim
            arg[(1, 3, 2)]=(0.26450128305+0.333572953707j)*x[0]**o + ((-0.107698794106-0.852927480031j))*x[0] + ((-0.776716934006-0.248858368769j))*x[1]**o + ((0.353846530657-0.112482428197j))*x[1] + ((0.520657095856+0.535918808661j))*x[2]**o + ((0.783905314157+0.82429010177j))*x[2]
            ref[(1, 3, 2)]=(0.00844144489936+0.620633393599j)*(1+2.*(dim-1)/(o+1.)) + ((1.03005305071-0.141119806458j))*dim
            arg[(1, 3, 3)]=(-0.504020252896-0.0509473316711j)*x[0]**o + ((0.0546241901979+0.442005344847j))*x[0] + ((0.806720039957+0.453050135589j))*x[1]**o + ((0.804146766428+0.242599794085j))*x[1] + ((0.720229765055-0.223647428176j))*x[2]**o + ((0.749498872855-0.489773509824j))*x[2]
            ref[(1, 3, 3)]=(1.02292955212+0.178455375742j)*(1+2.*(dim-1)/(o+1.)) + ((1.60826982948+0.194831629108j))*dim
            arg[(2, 0, 0)]=(-0.43756146698+0.0447124538828j)*x[0]**o + ((-0.417910459812-0.816181175002j))*x[0] + ((0.0955806209599+0.352274143912j))*x[1]**o + ((-0.97570009723+0.807280898241j))*x[1] + ((0.75773839312+0.596474496513j))*x[2]**o + ((-0.510370460632+0.396385780926j))*x[2]
            ref[(2, 0, 0)]=(0.4157575471+0.993461094307j)*(1+2.*(dim-1)/(o+1.)) + ((-1.90398101767+0.387485504165j))*dim
            arg[(2, 0, 1)]=(0.403653652415-0.465450247729j)*x[0]**o + ((-0.596542347452-0.436165711889j))*x[0] + ((-0.322453507244-0.137683803251j))*x[1]**o + ((0.553026672792+0.446677823758j))*x[1] + ((0.552189859193-0.665576782402j))*x[2]**o + ((0.18965290216+0.844531438118j))*x[2]
            ref[(2, 0, 1)]=(0.633390004365-1.26871083338j)*(1+2.*(dim-1)/(o+1.)) + ((0.1461372275+0.855043549987j))*dim
            arg[(2, 0, 2)]=(-0.954919114076-0.592674079502j)*x[0]**o + ((-0.61099411415+0.930695334955j))*x[0] + ((0.89764611609-0.502487315584j))*x[1]**o + ((-0.449511957606+0.332882793688j))*x[1] + ((-0.71322717113+0.583484745743j))*x[2]**o + ((-0.369438536278+0.0457231401623j))*x[2]
            ref[(2, 0, 2)]=(-0.770500169116-0.511676649343j)*(1+2.*(dim-1)/(o+1.)) + ((-1.42994460803+1.30930126881j))*dim
            arg[(2, 0, 3)]=(-0.55665740527+0.68659224052j)*x[0]**o + ((-0.333994693407-0.682285983204j))*x[0] + ((-0.273254005626+0.609607820092j))*x[1]**o + ((0.0211371897486-0.958816831908j))*x[1] + ((-0.0998045547131+0.92568716407j))*x[2]**o + ((-0.636996494083+0.574324322441j))*x[2]
            ref[(2, 0, 3)]=(-0.929715965609+2.22188722468j)*(1+2.*(dim-1)/(o+1.)) + ((-0.949853997742-1.06677849267j))*dim
            arg[(2, 1, 0)]=(-0.602008884675+0.119924926336j)*x[0]**o + ((0.674757707143+0.652027185989j))*x[0] + ((0.0447044729149+0.239856996749j))*x[1]**o + ((0.465017800286+0.23726283377j))*x[1] + ((0.579018581607+0.793359293229j))*x[2]**o + ((-0.732210182649-0.814657262425j))*x[2]
            ref[(2, 1, 0)]=(0.0217141698476+1.15314121631j)*(1+2.*(dim-1)/(o+1.)) + ((0.407565324779+0.0746327573335j))*dim
            arg[(2, 1, 1)]=(-0.317034949815-0.491835962868j)*x[0]**o + ((-0.541080286494-0.627338575424j))*x[0] + ((0.630265483068+0.311774379163j))*x[1]**o + ((0.385285479174+0.226613511917j))*x[1] + ((-0.857662049581+0.799971810346j))*x[2]**o + ((-0.797332080083-0.142448575799j))*x[2]
            ref[(2, 1, 1)]=(-0.544431516328+0.619910226641j)*(1+2.*(dim-1)/(o+1.)) + ((-0.953126887403-0.543173639306j))*dim
            arg[(2, 1, 2)]=(-0.675129141169+0.662685534661j)*x[0]**o + ((0.782085584972+0.215786374765j))*x[0] + ((0.598064063013+0.411715079245j))*x[1]**o + ((-0.296507841388+0.997939403618j))*x[1] + ((-0.672662870636+0.708540598978j))*x[2]**o + ((0.125233742225+0.707690390381j))*x[2]
            ref[(2, 1, 2)]=(-0.749727948793+1.78294121288j)*(1+2.*(dim-1)/(o+1.)) + ((0.610811485808+1.92141616876j))*dim
            arg[(2, 1, 3)]=(0.607204410117-0.0905865962986j)*x[0]**o + ((-0.596766344304+0.259812192131j))*x[0] + ((-0.595505768513+0.681624859947j))*x[1]**o + ((0.273372950545+0.240042759434j))*x[1] + ((0.580660581246-0.0470606458336j))*x[2]**o + ((-0.88636541409-0.0845979700049j))*x[2]
            ref[(2, 1, 3)]=(0.592359222851+0.543977617815j)*(1+2.*(dim-1)/(o+1.)) + ((-1.20975880785+0.415256981559j))*dim
            arg[(2, 2, 0)]=(0.236860671185+0.398805987774j)*x[0]**o + ((0.874936562497-0.432678328069j))*x[0] + ((-0.48167055723-0.105740429473j))*x[1]**o + ((-0.486160320016+0.318550858008j))*x[1] + ((-0.0781385941465+0.371945013712j))*x[2]**o + ((0.414999965584-0.669739454454j))*x[2]
            ref[(2, 2, 0)]=(-0.322948480191+0.665010572014j)*(1+2.*(dim-1)/(o+1.)) + ((0.803776208065-0.783866924515j))*dim
            arg[(2, 2, 1)]=(0.352215508799+0.0520977994513j)*x[0]**o + ((-0.557984576905+0.245322662056j))*x[0] + ((-0.793886967767+0.037643589325j))*x[1]**o + ((0.449808407416-0.0308382763374j))*x[1] + ((-0.312420989589+0.0131305656961j))*x[2]**o + ((-0.168835093841-0.986377137528j))*x[2]
            ref[(2, 2, 1)]=(-0.754092448557+0.102871954472j)*(1+2.*(dim-1)/(o+1.)) + ((-0.27701126333-0.77189275181j))*dim
            arg[(2, 2, 2)]=(0.420426956107-0.650362677471j)*x[0]**o + ((0.804626973913-0.167437423478j))*x[0] + ((0.294675936323-0.418879085233j))*x[1]**o + ((0.140370280132+0.82234077498j))*x[1] + ((0.371888203541-0.628644310545j))*x[2]**o + ((-0.729607559016+0.553958324327j))*x[2]
            ref[(2, 2, 2)]=(1.08699109597-1.69788607325j)*(1+2.*(dim-1)/(o+1.)) + ((0.215389695029+1.20886167583j))*dim
            arg[(2, 2, 3)]=(0.0319885865518+0.210843795276j)*x[0]**o + ((0.689737759679+0.913276276162j))*x[0] + ((-0.32423605804+0.0847361810263j))*x[1]**o + ((0.684382276949+0.0882451605711j))*x[1] + ((0.579563308751-0.193470383742j))*x[2]**o + ((-0.834261855702+0.992106354329j))*x[2]
            ref[(2, 2, 3)]=(0.287315837262+0.102109592561j)*(1+2.*(dim-1)/(o+1.)) + ((0.539858180926+1.99362779106j))*dim
            arg[(2, 3, 0)]=(0.730289376861+0.562234945778j)*x[0]**o + ((0.211950340597+0.657167686538j))*x[0] + ((-0.682410602868+0.128974203158j))*x[1]**o + ((-0.530015383164+0.707173450513j))*x[1] + ((0.479386498128+0.236871454606j))*x[2]**o + ((0.668662234354+0.273023105256j))*x[2]
            ref[(2, 3, 0)]=(0.527265272121+0.928080603543j)*(1+2.*(dim-1)/(o+1.)) + ((0.350597191787+1.63736424231j))*dim
            arg[(2, 3, 1)]=(0.256379955214-0.740539705282j)*x[0]**o + ((-0.142277875819-0.248520242843j))*x[0] + ((-0.736739122587-0.273713512285j))*x[1]**o + ((0.129880598266-0.255335867381j))*x[1] + ((-0.149815125198+0.853452087247j))*x[2]**o + ((0.0417336730253-0.172400480691j))*x[2]
            ref[(2, 3, 1)]=(-0.630174292572-0.16080113032j)*(1+2.*(dim-1)/(o+1.)) + ((0.0293363954731-0.676256590916j))*dim
            arg[(2, 3, 2)]=(-0.986936833577-0.890653801368j)*x[0]**o + ((-0.371683539786-0.0455965029895j))*x[0] + ((0.983152453019+0.300292485396j))*x[1]**o + ((0.0779722534291+0.207777057285j))*x[1] + ((0.382428590301+0.617299849562j))*x[2]**o + ((0.592409892456+0.0794765675429j))*x[2]
            ref[(2, 3, 2)]=(0.378644209744+0.0269385335901j)*(1+2.*(dim-1)/(o+1.)) + ((0.298698606099+0.241657121838j))*dim
            arg[(2, 3, 3)]=(-0.668417781295-0.532000142058j)*x[0]**o + ((-0.814955213535-0.81546947214j))*x[0] + ((-0.774461656829+0.614814020114j))*x[1]**o + ((0.0145664752324-0.572170941405j))*x[1] + ((0.373980884887-0.125695445301j))*x[2]**o + ((0.0169814551473+0.808857184473j))*x[2]
            ref[(2, 3, 3)]=(-1.06889855324-0.0428815672442j)*(1+2.*(dim-1)/(o+1.)) + ((-0.783407283155-0.578783229071j))*dim
            arg[(3, 0, 0)]=(-0.59938081129-0.917963939448j)*x[0]**o + ((0.183582767746+0.436157898881j))*x[0] + ((-0.686837459777-0.141468176498j))*x[1]**o + ((-0.970706968044-0.720741722786j))*x[1] + ((-0.90247768464-0.713027629085j))*x[2]**o + ((-0.602170555132-0.725336796054j))*x[2]
            ref[(3, 0, 0)]=(-2.18869595571-1.77245974503j)*(1+2.*(dim-1)/(o+1.)) + ((-1.38929475543-1.00992061996j))*dim
            arg[(3, 0, 1)]=(-0.79392015205-0.652425508485j)*x[0]**o + ((-0.84309567627-0.528507103066j))*x[0] + ((-0.680531881481+0.161538548278j))*x[1]**o + ((0.616842232935-0.816024751832j))*x[1] + ((-0.408631036596+0.123860491493j))*x[2]**o + ((-0.402420179666-0.414955051672j))*x[2]
            ref[(3, 0, 1)]=(-1.88308307013-0.367026468714j)*(1+2.*(dim-1)/(o+1.)) + ((-0.628673623001-1.75948690657j))*dim
            arg[(3, 0, 2)]=(-0.861978261709+0.433218616259j)*x[0]**o + ((0.813494408757+0.539274346026j))*x[0] + ((0.00295979248168-0.132674922852j))*x[1]**o + ((0.902227939025-0.0965011013926j))*x[1] + ((0.226654530123-0.98180785716j))*x[2]**o + ((0.640120183738+0.901643724362j))*x[2]
            ref[(3, 0, 2)]=(-0.632363939104-0.681264163753j)*(1+2.*(dim-1)/(o+1.)) + ((2.35584253152+1.344416969j))*dim
            arg[(3, 0, 3)]=(0.507833956975+0.581534082815j)*x[0]**o + ((-0.838807879426+0.286979919615j))*x[0] + ((-0.020077309453+0.730969083057j))*x[1]**o + ((-0.443847982121-0.898319681628j))*x[1] + ((-0.765503427138+0.931089527149j))*x[2]**o + ((-0.859052266446-0.299895832562j))*x[2]
            ref[(3, 0, 3)]=(-0.277746779616+2.24359269302j)*(1+2.*(dim-1)/(o+1.)) + ((-2.14170812799-0.911235594575j))*dim
            arg[(3, 1, 0)]=(0.669435458461+0.722578787136j)*x[0]**o + ((-0.734868984472-0.00836036650464j))*x[0] + ((0.226524690148-0.376791168846j))*x[1]**o + ((-0.339526547437-0.392311522681j))*x[1] + ((0.924205573124+0.343393437052j))*x[2]**o + ((-0.187967397252-0.930377767958j))*x[2]
            ref[(3, 1, 0)]=(1.82016572173+0.689181055342j)*(1+2.*(dim-1)/(o+1.)) + ((-1.26236292916-1.33104965714j))*dim
            arg[(3, 1, 1)]=(0.287093789587+0.308659737943j)*x[0]**o + ((0.200186786995+0.80058359566j))*x[0] + ((0.694213689718-0.966948272932j))*x[1]**o + ((0.522330399089-0.090544517313j))*x[1] + ((0.973912310446+0.946387864075j))*x[2]**o + ((0.0959117525119-0.412043645107j))*x[2]
            ref[(3, 1, 1)]=(1.95521978975+0.288099329086j)*(1+2.*(dim-1)/(o+1.)) + ((0.818428938596+0.29799543324j))*dim
            arg[(3, 1, 2)]=(0.453284216795-0.825102308798j)*x[0]**o + ((-0.71444695124-0.730813088099j))*x[0] + ((0.558845900107-0.793649474833j))*x[1]**o + ((0.621851463329-0.697026913028j))*x[1] + ((0.181777542879+0.62867814089j))*x[2]**o + ((0.579549811815+0.467037194771j))*x[2]
            ref[(3, 1, 2)]=(1.19390765978-0.990073642741j)*(1+2.*(dim-1)/(o+1.)) + ((0.486954323904-0.960802806355j))*dim
            arg[(3, 1, 3)]=(0.681906740186+0.407641371983j)*x[0]**o + ((0.582125351751-0.636012034062j))*x[0] + ((0.997872618876+0.791396157389j))*x[1]**o + ((-0.420595659708+0.447061902294j))*x[1] + ((0.707324140642-0.00839725659304j))*x[2]**o + ((0.049853340159-0.281430964203j))*x[2]
            ref[(3, 1, 3)]=(2.3871034997+1.19064027278j)*(1+2.*(dim-1)/(o+1.)) + ((0.211383032202-0.47038109597j))*dim
            arg[(3, 2, 0)]=(0.170997500447-0.919567568085j)*x[0]**o + ((0.638189325512+0.0305827394853j))*x[0] + ((-0.236432616234-0.7257246394j))*x[1]**o + ((-0.237688558065+0.878973167397j))*x[1] + ((0.93386671898-0.972008550485j))*x[2]**o + ((0.154168349055-0.693215956214j))*x[2]
            ref[(3, 2, 0)]=(0.868431603193-2.61730075797j)*(1+2.*(dim-1)/(o+1.)) + ((0.554669116502+0.216339950669j))*dim
            arg[(3, 2, 1)]=(0.142323975215+0.294146958948j)*x[0]**o + ((0.820347789322+0.997131572183j))*x[0] + ((-0.056220955031+0.224892638341j))*x[1]**o + ((0.76062827743-0.0179436755783j))*x[1] + ((-0.809046922137-0.55575381777j))*x[2]**o + ((0.839514055741-0.514674065427j))*x[2]
            ref[(3, 2, 1)]=(-0.722943901953-0.0367142204805j)*(1+2.*(dim-1)/(o+1.)) + ((2.42049012249+0.464513831178j))*dim
            arg[(3, 2, 2)]=(-0.916461025807+0.715236734503j)*x[0]**o + ((-0.282231419431+0.628877035551j))*x[0] + ((-0.0034217865597+0.941492775156j))*x[1]**o + ((-0.0743253715705+0.649500771499j))*x[1] + ((-0.478599132615+0.963316656553j))*x[2]**o + ((-0.0174718854478+0.668647660611j))*x[2]
            ref[(3, 2, 2)]=(-1.39848194498+2.62004616621j)*(1+2.*(dim-1)/(o+1.)) + ((-0.37402867645+1.94702546766j))*dim
            arg[(3, 2, 3)]=(0.240248490164+0.136962600531j)*x[0]**o + ((-0.376941781764-0.837415903407j))*x[0] + ((-0.379059375365-0.408117839837j))*x[1]**o + ((-0.650483590025-0.420775678604j))*x[1] + ((0.903600291996-0.0442663776016j))*x[2]**o + ((-0.239965605119-0.191755652724j))*x[2]
            ref[(3, 2, 3)]=(0.764789406794-0.315421616907j)*(1+2.*(dim-1)/(o+1.)) + ((-1.26739097691-1.44994723474j))*dim
            arg[(3, 3, 0)]=(0.0552239223906+0.905185837357j)*x[0]**o + ((0.32683775242+0.847971328139j))*x[0] + ((0.193304290263-0.430139071214j))*x[1]**o + ((-0.362724245181+0.0139977238244j))*x[1] + ((-0.552444480955+0.957676570473j))*x[2]**o + ((0.942983121958+0.591355004956j))*x[2]
            ref[(3, 3, 0)]=(-0.303916268302+1.43272333662j)*(1+2.*(dim-1)/(o+1.)) + ((0.907096629197+1.45332405692j))*dim
            arg[(3, 3, 1)]=(-0.128464761548-0.494278970778j)*x[0]**o + ((0.56229124965+0.12383813027j))*x[0] + ((0.559895201549-0.882831770935j))*x[1]**o + ((0.019068521535-0.666054959388j))*x[1] + ((-0.259039671526-0.223010818496j))*x[2]**o + ((-0.380634024682-0.623475454104j))*x[2]
            ref[(3, 3, 1)]=(0.172390768475-1.60012156021j)*(1+2.*(dim-1)/(o+1.)) + ((0.200725746503-1.16569228322j))*dim
            arg[(3, 3, 2)]=(0.839677143418+0.183170226572j)*x[0]**o + ((0.579264603794-0.721691770999j))*x[0] + ((-0.769470974891-0.311102504337j))*x[1]**o + ((0.238989236566+0.278057402991j))*x[1] + ((0.590444186528-0.840926169251j))*x[2]**o + ((-0.576751564737-0.0586893059937j))*x[2]
            ref[(3, 3, 2)]=(0.660650355055-0.968858447016j)*(1+2.*(dim-1)/(o+1.)) + ((0.241502275624-0.502323674002j))*dim
            arg[(3, 3, 3)]=(0.896002802701+0.812737611622j)*x[0]**o + ((0.235399677164-0.183131711008j))*x[0] + ((-0.307577351832+0.141778362338j))*x[1]**o + ((-0.438114050267+0.228036186045j))*x[1] + ((0.898849105014-0.384554015356j))*x[2]**o + ((-0.347622142089-0.162811349536j))*x[2]
            ref[(3, 3, 3)]=(1.48727455588+0.569961958604j)*(1+2.*(dim-1)/(o+1.)) + ((-0.550336515193-0.117906874499j))*dim
        res=integrate(arg, where=w_ref)
        self.assertTrue(isinstance(res,numpy.ndarray),'wrong type of result.')
        self.assertEqual(res.shape, (4, 4, 4),'wrong shape of result.')
        self.assertLess(Lsup(res-ref), self.RES_TOL*Lsup(ref),'wrong result')
    #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
    def test_integrateC_onFunctionOnBoundary_fromData_Solution_rank4(self):
        """
        tests integral of rank 4 Data on FunctionOnBoundary

        assumptions: Solution(self.domain) exists
                     self.domain supports integral on FunctionOnBoundary
        """
        o = self.order
        dim = self.domain.getDim()
        w_ref = FunctionOnBoundary(self.domain)
        w = Solution(self.domain)
        x = w.getX()
        arg = Data(0,(4, 4, 4, 4),w)
        ref=numpy.zeros((4, 4, 4, 4),numpy.complex_)
        if dim==2:
            arg[(0, 0, 0, 0)]=(0.509913351026+0.985788759835j)*x[0]**o + ((0.318055963654-0.783439864679j))*x[0] + ((-0.868203154577-0.493275517065j))*x[1]**o + ((0.871295869181-0.611653344589j))*x[1]
            ref[(0, 0, 0, 0)]=(-0.358289803552+0.49251324277j)*(1+2.*(dim-1)/(o+1.)) + ((1.18935183284-1.39509320927j))*dim
            arg[(0, 0, 0, 1)]=(-0.528947231616-0.197001620202j)*x[0]**o + ((-0.0670736869329-0.321172289604j))*x[0] + ((-0.243438745242+0.440265409249j))*x[1]**o + ((0.00284615755234+0.353322006503j))*x[1]
            ref[(0, 0, 0, 1)]=(-0.772385976858+0.243263789047j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0642275293805+0.0321497168981j))*dim
            arg[(0, 0, 0, 2)]=(-0.862193216738-0.841387602406j)*x[0]**o + ((0.203394111758+0.883074369825j))*x[0] + ((-0.130063297012+0.123644403753j))*x[1]**o + ((0.152937928533+0.733860103323j))*x[1]
            ref[(0, 0, 0, 2)]=(-0.99225651375-0.717743198653j)*(1+2.*(dim-1)/(o+1.)) + ((0.356332040291+1.61693447315j))*dim
            arg[(0, 0, 0, 3)]=(-0.506856896426+0.624494448416j)*x[0]**o + ((0.0862429843609+0.362996761814j))*x[0] + ((0.104111937924+0.665233547868j))*x[1]**o + ((0.217192074294+0.728301174108j))*x[1]
            ref[(0, 0, 0, 3)]=(-0.402744958502+1.28972799628j)*(1+2.*(dim-1)/(o+1.)) + ((0.303435058655+1.09129793592j))*dim
            arg[(0, 0, 1, 0)]=(-0.821741304416+0.81603092516j)*x[0]**o + ((0.423673224929+0.353333401236j))*x[0] + ((-0.196622158137-0.0786880410466j))*x[1]**o + ((-0.960745779754-0.593112339302j))*x[1]
            ref[(0, 0, 1, 0)]=(-1.01836346255+0.737342884113j)*(1+2.*(dim-1)/(o+1.)) + ((-0.537072554825-0.239778938067j))*dim
            arg[(0, 0, 1, 1)]=(0.276270503205+0.0691286041427j)*x[0]**o + ((-0.455803032125-0.597753014092j))*x[0] + ((0.0640448090498-0.279346606556j))*x[1]**o + ((0.477799991389+0.70504949676j))*x[1]
            ref[(0, 0, 1, 1)]=(0.340315312255-0.210218002413j)*(1+2.*(dim-1)/(o+1.)) + ((0.0219969592641+0.107296482668j))*dim
            arg[(0, 0, 1, 2)]=(-0.972072221808-0.748349299319j)*x[0]**o + ((0.22702681487+0.842577168391j))*x[0] + ((-0.734060476457+0.072823167276j))*x[1]**o + ((0.29791436563+0.385087009321j))*x[1]
            ref[(0, 0, 1, 2)]=(-1.70613269827-0.675526132043j)*(1+2.*(dim-1)/(o+1.)) + ((0.5249411805+1.22766417771j))*dim
            arg[(0, 0, 1, 3)]=(0.931405962557-0.233845157454j)*x[0]**o + ((0.892211886215+0.369937834455j))*x[0] + ((-0.450167380903+0.0692157393439j))*x[1]**o + ((0.42793222588-0.715329167236j))*x[1]
            ref[(0, 0, 1, 3)]=(0.481238581653-0.16462941811j)*(1+2.*(dim-1)/(o+1.)) + ((1.3201441121-0.345391332781j))*dim
            arg[(0, 0, 2, 0)]=(-0.202176692558+0.949788541606j)*x[0]**o + ((-0.484916133531-0.163923816658j))*x[0] + ((-0.422186345241-0.409310667065j))*x[1]**o + ((-0.942242943037+0.480188610449j))*x[1]
            ref[(0, 0, 2, 0)]=(-0.624363037799+0.540477874541j)*(1+2.*(dim-1)/(o+1.)) + ((-1.42715907657+0.316264793791j))*dim
            arg[(0, 0, 2, 1)]=(-0.530225660715-0.261749872691j)*x[0]**o + ((0.672522252312-0.319087881776j))*x[0] + ((-0.330800135081+0.233467326694j))*x[1]**o + ((0.58384284834-0.598772971753j))*x[1]
            ref[(0, 0, 2, 1)]=(-0.861025795797-0.0282825459965j)*(1+2.*(dim-1)/(o+1.)) + ((1.25636510065-0.917860853528j))*dim
            arg[(0, 0, 2, 2)]=(-0.165028858539+0.571324146216j)*x[0]**o + ((0.604135705693+0.777751894944j))*x[0] + ((-0.686162451558-0.20603237145j))*x[1]**o + ((-0.695378759191+0.0497321364547j))*x[1]
            ref[(0, 0, 2, 2)]=(-0.851191310097+0.365291774766j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0912430534984+0.827484031398j))*dim
            arg[(0, 0, 2, 3)]=(0.461621005567+0.476841006199j)*x[0]**o + ((-0.344643907359+0.613048220661j))*x[0] + ((0.217269980751-0.792153954998j))*x[1]**o + ((0.0550683285334+0.35511749151j))*x[1]
            ref[(0, 0, 2, 3)]=(0.678890986318-0.315312948799j)*(1+2.*(dim-1)/(o+1.)) + ((-0.289575578825+0.968165712172j))*dim
            arg[(0, 0, 3, 0)]=(0.00868900339517+0.51969517301j)*x[0]**o + ((0.247402024804+0.567237608744j))*x[0] + ((-0.842709012643-0.874359932481j))*x[1]**o + ((0.892184270864-0.0221910574945j))*x[1]
            ref[(0, 0, 3, 0)]=(-0.834020009248-0.354664759471j)*(1+2.*(dim-1)/(o+1.)) + ((1.13958629567+0.545046551249j))*dim
            arg[(0, 0, 3, 1)]=(0.734920248617+0.0642721057697j)*x[0]**o + ((0.324818998375+0.138884017594j))*x[0] + ((-0.0895148514738-0.901451245962j))*x[1]**o + ((0.363911898308+0.0391733140513j))*x[1]
            ref[(0, 0, 3, 1)]=(0.645405397143-0.837179140192j)*(1+2.*(dim-1)/(o+1.)) + ((0.688730896683+0.178057331645j))*dim
            arg[(0, 0, 3, 2)]=(0.923725857118-0.955923517889j)*x[0]**o + ((0.793641478685-0.175058238718j))*x[0] + ((-0.829207003698-0.76841391909j))*x[1]**o + ((-0.0510150461351-0.363598073971j))*x[1]
            ref[(0, 0, 3, 2)]=(0.0945188534208-1.72433743698j)*(1+2.*(dim-1)/(o+1.)) + ((0.74262643255-0.538656312689j))*dim
            arg[(0, 0, 3, 3)]=(-0.416632852521-0.0822235648547j)*x[0]**o + ((0.99546848385+0.193871934881j))*x[0] + ((0.392459635427+0.121947253883j))*x[1]**o + ((-0.852688938551+0.644586734586j))*x[1]
            ref[(0, 0, 3, 3)]=(-0.0241732170941+0.039723689028j)*(1+2.*(dim-1)/(o+1.)) + ((0.142779545299+0.838458669467j))*dim
            arg[(0, 1, 0, 0)]=(0.497870283658-0.298312038104j)*x[0]**o + ((-0.841572031387+0.576849357574j))*x[0] + ((-0.0599773419988-0.689461412199j))*x[1]**o + ((0.582487642654-0.0406745600036j))*x[1]
            ref[(0, 1, 0, 0)]=(0.43789294166-0.987773450303j)*(1+2.*(dim-1)/(o+1.)) + ((-0.259084388733+0.536174797571j))*dim
            arg[(0, 1, 0, 1)]=(0.00611581388725+0.0839994165205j)*x[0]**o + ((-0.930844317091+0.28922106628j))*x[0] + ((-0.33408524838-0.0263960548577j))*x[1]**o + ((-0.00237209602438-0.905082986623j))*x[1]
            ref[(0, 1, 0, 1)]=(-0.327969434493+0.0576033616628j)*(1+2.*(dim-1)/(o+1.)) + ((-0.933216413115-0.615861920344j))*dim
            arg[(0, 1, 0, 2)]=(0.834685654593+0.0973142653766j)*x[0]**o + ((0.781303403985-0.597057675843j))*x[0] + ((0.783240553665-0.35807165226j))*x[1]**o + ((-0.252143021163+0.174188790176j))*x[1]
            ref[(0, 1, 0, 2)]=(1.61792620826-0.260757386884j)*(1+2.*(dim-1)/(o+1.)) + ((0.529160382821-0.422868885666j))*dim
            arg[(0, 1, 0, 3)]=(-0.678582381211+0.270596602851j)*x[0]**o + ((-0.934086491984-0.986271453292j))*x[0] + ((0.698124762473-0.197690265273j))*x[1]**o + ((-0.00423646976852-0.504037978519j))*x[1]
            ref[(0, 1, 0, 3)]=(0.019542381262+0.0729063375776j)*(1+2.*(dim-1)/(o+1.)) + ((-0.938322961753-1.49030943181j))*dim
            arg[(0, 1, 1, 0)]=(-0.631738151415+0.587037731869j)*x[0]**o + ((0.678313867202+0.718850945888j))*x[0] + ((-0.423858912273-0.0501384734816j))*x[1]**o + ((0.974128836173-0.337974360223j))*x[1]
            ref[(0, 1, 1, 0)]=(-1.05559706369+0.536899258388j)*(1+2.*(dim-1)/(o+1.)) + ((1.65244270338+0.380876585665j))*dim
            arg[(0, 1, 1, 1)]=(-0.576671737931-0.0340927649922j)*x[0]**o + ((-0.727388907583-0.3308619451j))*x[0] + ((0.54193219484+0.861929525345j))*x[1]**o + ((0.0173944463627+0.333320517049j))*x[1]
            ref[(0, 1, 1, 1)]=(-0.0347395430909+0.827836760353j)*(1+2.*(dim-1)/(o+1.)) + ((-0.709994461221+0.00245857194968j))*dim
            arg[(0, 1, 1, 2)]=(0.439551882109-0.631329742468j)*x[0]**o + ((0.853271497708+0.621176407183j))*x[0] + ((-0.88079351992-0.856746814389j))*x[1]**o + ((0.824862641186+0.946471591325j))*x[1]
            ref[(0, 1, 1, 2)]=(-0.441241637811-1.48807655686j)*(1+2.*(dim-1)/(o+1.)) + ((1.67813413889+1.56764799851j))*dim
            arg[(0, 1, 1, 3)]=(0.730051236971-0.699894757175j)*x[0]**o + ((-0.362436684422+0.144471802854j))*x[0] + ((0.562943622789-0.275158098935j))*x[1]**o + ((0.663624639498+0.260947885532j))*x[1]
            ref[(0, 1, 1, 3)]=(1.29299485976-0.975052856111j)*(1+2.*(dim-1)/(o+1.)) + ((0.301187955076+0.405419688386j))*dim
            arg[(0, 1, 2, 0)]=(-0.474167134985-0.809937919157j)*x[0]**o + ((-0.0616898109387-0.44095982373j))*x[0] + ((0.527077177235-0.537479375985j))*x[1]**o + ((0.866170177725+0.40534290947j))*x[1]
            ref[(0, 1, 2, 0)]=(0.0529100422505-1.34741729514j)*(1+2.*(dim-1)/(o+1.)) + ((0.804480366786-0.0356169142597j))*dim
            arg[(0, 1, 2, 1)]=(-0.640826728017-0.311227117724j)*x[0]**o + ((0.699855635127+0.0438505513751j))*x[0] + ((0.396170300583+0.440644138419j))*x[1]**o + ((0.519230614651+0.168940159287j))*x[1]
            ref[(0, 1, 2, 1)]=(-0.244656427434+0.129417020695j)*(1+2.*(dim-1)/(o+1.)) + ((1.21908624978+0.212790710663j))*dim
            arg[(0, 1, 2, 2)]=(-0.192992952451-0.975391713413j)*x[0]**o + ((-0.504912702561+0.558597826213j))*x[0] + ((-0.504158009368+0.816984466393j))*x[1]**o + ((-0.949343669333-0.348129870204j))*x[1]
            ref[(0, 1, 2, 2)]=(-0.697150961819-0.158407247019j)*(1+2.*(dim-1)/(o+1.)) + ((-1.45425637189+0.21046795601j))*dim
            arg[(0, 1, 2, 3)]=(-0.531176475952+0.778021987276j)*x[0]**o + ((0.200350047708+0.127231549237j))*x[0] + ((0.00762035074805+0.915592633361j))*x[1]**o + ((-0.174126127679+0.301181904337j))*x[1]
            ref[(0, 1, 2, 3)]=(-0.523556125204+1.69361462064j)*(1+2.*(dim-1)/(o+1.)) + ((0.026223920029+0.428413453574j))*dim
            arg[(0, 1, 3, 0)]=(0.853427320537+0.368589259j)*x[0]**o + ((0.15970072215-0.724882117076j))*x[0] + ((0.695835769913-0.572640376943j))*x[1]**o + ((-0.480345063362+0.782907169703j))*x[1]
            ref[(0, 1, 3, 0)]=(1.54926309045-0.204051117943j)*(1+2.*(dim-1)/(o+1.)) + ((-0.320644341212+0.0580250526271j))*dim
            arg[(0, 1, 3, 1)]=(-0.748033259035+0.148261737235j)*x[0]**o + ((-0.644048418191+0.465010449301j))*x[0] + ((-0.415443959304-0.035437372739j))*x[1]**o + ((0.985073441888-0.748290679991j))*x[1]
            ref[(0, 1, 3, 1)]=(-1.16347721834+0.112824364496j)*(1+2.*(dim-1)/(o+1.)) + ((0.341025023698-0.28328023069j))*dim
            arg[(0, 1, 3, 2)]=(-0.446542444985+0.871094969256j)*x[0]**o + ((-0.0600759218161-0.722953768408j))*x[0] + ((-0.622764674335+0.780747748677j))*x[1]**o + ((0.209975056257+0.338006522188j))*x[1]
            ref[(0, 1, 3, 2)]=(-1.06930711932+1.65184271793j)*(1+2.*(dim-1)/(o+1.)) + ((0.149899134441-0.38494724622j))*dim
            arg[(0, 1, 3, 3)]=(0.80378751227-0.673877630785j)*x[0]**o + ((0.92271469934+0.432618640855j))*x[0] + ((0.643882156116+0.607427339654j))*x[1]**o + ((-0.28983391136-0.837465978552j))*x[1]
            ref[(0, 1, 3, 3)]=(1.44766966839-0.0664502911316j)*(1+2.*(dim-1)/(o+1.)) + ((0.63288078798-0.404847337697j))*dim
            arg[(0, 2, 0, 0)]=(0.0318564732619+0.68882126342j)*x[0]**o + ((-0.614306445411-0.771439925891j))*x[0] + ((0.260275235651-0.165314182014j))*x[1]**o + ((0.0676804706256-0.540518688634j))*x[1]
            ref[(0, 2, 0, 0)]=(0.292131708913+0.523507081405j)*(1+2.*(dim-1)/(o+1.)) + ((-0.546625974785-1.31195861453j))*dim
            arg[(0, 2, 0, 1)]=(-0.787634712906+0.978362611128j)*x[0]**o + ((0.0401298096047+0.167228578163j))*x[0] + ((0.464356358933+0.171327194013j))*x[1]**o + ((0.928273245483-0.998078058994j))*x[1]
            ref[(0, 2, 0, 1)]=(-0.323278353972+1.14968980514j)*(1+2.*(dim-1)/(o+1.)) + ((0.968403055088-0.830849480831j))*dim
            arg[(0, 2, 0, 2)]=(0.0561363312705+0.515234265498j)*x[0]**o + ((-0.0900333520868+0.863717219198j))*x[0] + ((0.863746898587+0.0782584934958j))*x[1]**o + ((0.953059703744+0.175585714585j))*x[1]
            ref[(0, 2, 0, 2)]=(0.919883229857+0.593492758994j)*(1+2.*(dim-1)/(o+1.)) + ((0.863026351657+1.03930293378j))*dim
            arg[(0, 2, 0, 3)]=(0.255773399046+0.421760198887j)*x[0]**o + ((0.843536565049-0.940401690092j))*x[0] + ((0.930496813597-0.824233809597j))*x[1]**o + ((-0.323011104586-0.385221549224j))*x[1]
            ref[(0, 2, 0, 3)]=(1.18627021264-0.402473610711j)*(1+2.*(dim-1)/(o+1.)) + ((0.520525460464-1.32562323932j))*dim
            arg[(0, 2, 1, 0)]=(-0.0888115271035+0.0834310671727j)*x[0]**o + ((0.431223115372+0.808348321124j))*x[0] + ((-0.408997825738-0.4944792051j))*x[1]**o + ((-0.277929676636+0.891827866995j))*x[1]
            ref[(0, 2, 1, 0)]=(-0.497809352841-0.411048137928j)*(1+2.*(dim-1)/(o+1.)) + ((0.153293438736+1.70017618812j))*dim
            arg[(0, 2, 1, 1)]=(0.731444396622-0.505443370433j)*x[0]**o + ((-0.838052155852+0.0761739116597j))*x[0] + ((-0.29873902443+0.297216076817j))*x[1]**o + ((0.357925208639+0.126064416365j))*x[1]
            ref[(0, 2, 1, 1)]=(0.432705372192-0.208227293616j)*(1+2.*(dim-1)/(o+1.)) + ((-0.480126947214+0.202238328025j))*dim
            arg[(0, 2, 1, 2)]=(0.933719171202+0.662401198874j)*x[0]**o + ((0.568085558937+0.57598853471j))*x[0] + ((-0.649165331965+0.18132247999j))*x[1]**o + ((0.795043267984+0.562900726908j))*x[1]
            ref[(0, 2, 1, 2)]=(0.284553839236+0.843723678863j)*(1+2.*(dim-1)/(o+1.)) + ((1.36312882692+1.13888926162j))*dim
            arg[(0, 2, 1, 3)]=(0.915592190481-0.968075660287j)*x[0]**o + ((-0.34562738567+0.45353357049j))*x[0] + ((-0.0114879645902+0.334859161009j))*x[1]**o + ((0.732990797716-0.947674178967j))*x[1]
            ref[(0, 2, 1, 3)]=(0.904104225891-0.633216499277j)*(1+2.*(dim-1)/(o+1.)) + ((0.387363412046-0.494140608477j))*dim
            arg[(0, 2, 2, 0)]=(-0.20219377205-0.987623722884j)*x[0]**o + ((-0.832156988371-0.205059129864j))*x[0] + ((-0.661324106493+0.777389468528j))*x[1]**o + ((-0.721558860293-0.950954721584j))*x[1]
            ref[(0, 2, 2, 0)]=(-0.863517878543-0.210234254356j)*(1+2.*(dim-1)/(o+1.)) + ((-1.55371584866-1.15601385145j))*dim
            arg[(0, 2, 2, 1)]=(-0.647757393626-0.839069557793j)*x[0]**o + ((-0.901842448145-0.989961206912j))*x[0] + ((-0.437534086788+0.452499601355j))*x[1]**o + ((0.627250950098+0.462640365228j))*x[1]
            ref[(0, 2, 2, 1)]=(-1.08529148041-0.386569956437j)*(1+2.*(dim-1)/(o+1.)) + ((-0.274591498046-0.527320841684j))*dim
            arg[(0, 2, 2, 2)]=(-0.265306165876+0.745962022773j)*x[0]**o + ((0.175635849497-0.315223811371j))*x[0] + ((-0.778095233129-0.161496230708j))*x[1]**o + ((-0.155997772147+0.587691335695j))*x[1]
            ref[(0, 2, 2, 2)]=(-1.04340139901+0.584465792065j)*(1+2.*(dim-1)/(o+1.)) + ((0.0196380773494+0.272467524324j))*dim
            arg[(0, 2, 2, 3)]=(-0.618489290629+0.12350008556j)*x[0]**o + ((0.837960853489-0.673599295031j))*x[0] + ((0.717183576483+0.200043794019j))*x[1]**o + ((-0.635460287428+0.722065629395j))*x[1]
            ref[(0, 2, 2, 3)]=(0.0986942858542+0.323543879579j)*(1+2.*(dim-1)/(o+1.)) + ((0.202500566061+0.048466334364j))*dim
            arg[(0, 2, 3, 0)]=(0.131111017146-0.614212284107j)*x[0]**o + ((0.150790038005+0.2850066657j))*x[0] + ((0.363286608671-0.632663900683j))*x[1]**o + ((0.469877316568+0.128561458364j))*x[1]
            ref[(0, 2, 3, 0)]=(0.494397625817-1.24687618479j)*(1+2.*(dim-1)/(o+1.)) + ((0.620667354573+0.413568124064j))*dim
            arg[(0, 2, 3, 1)]=(-0.393258278779+0.663622265917j)*x[0]**o + ((-0.317950942643-0.764624616703j))*x[0] + ((-0.392756299491-0.744262138106j))*x[1]**o + ((-0.255994617946-0.762385423491j))*x[1]
            ref[(0, 2, 3, 1)]=(-0.78601457827-0.080639872189j)*(1+2.*(dim-1)/(o+1.)) + ((-0.573945560589-1.52701004019j))*dim
            arg[(0, 2, 3, 2)]=(-0.542995072371-0.806883841463j)*x[0]**o + ((-0.865738551703-0.234989395104j))*x[0] + ((-0.210721697088-0.304049860862j))*x[1]**o + ((0.240694989217-0.872754395954j))*x[1]
            ref[(0, 2, 3, 2)]=(-0.75371676946-1.11093370232j)*(1+2.*(dim-1)/(o+1.)) + ((-0.625043562486-1.10774379106j))*dim
            arg[(0, 2, 3, 3)]=(-0.691705015148+0.0263678749029j)*x[0]**o + ((-0.265610407896+0.880692844287j))*x[0] + ((-0.725231484048+0.700359936795j))*x[1]**o + ((-0.558684157881+0.491670570007j))*x[1]
            ref[(0, 2, 3, 3)]=(-1.4169364992+0.726727811698j)*(1+2.*(dim-1)/(o+1.)) + ((-0.824294565777+1.37236341429j))*dim
            arg[(0, 3, 0, 0)]=(0.642163223672+0.0771813924453j)*x[0]**o + ((-0.0748799405314+0.207882922881j))*x[0] + ((-0.680198363838-0.034844307397j))*x[1]**o + ((-0.750123375049-0.595980343594j))*x[1]
            ref[(0, 3, 0, 0)]=(-0.0380351401659+0.0423370850483j)*(1+2.*(dim-1)/(o+1.)) + ((-0.825003315581-0.388097420712j))*dim
            arg[(0, 3, 0, 1)]=(-0.864388085414+0.817610884897j)*x[0]**o + ((-0.284443735792-0.945480527965j))*x[0] + ((-0.245440174822-0.407529129401j))*x[1]**o + ((-0.325842817688-0.521712717389j))*x[1]
            ref[(0, 3, 0, 1)]=(-1.10982826024+0.410081755496j)*(1+2.*(dim-1)/(o+1.)) + ((-0.61028655348-1.46719324535j))*dim
            arg[(0, 3, 0, 2)]=(-0.54940410279+0.98190566448j)*x[0]**o + ((-0.0092486287394+0.833991875152j))*x[0] + ((-0.198752586796+0.782232076492j))*x[1]**o + ((-0.00242775193695-0.469496295177j))*x[1]
            ref[(0, 3, 0, 2)]=(-0.748156689587+1.76413774097j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0116763806763+0.364495579975j))*dim
            arg[(0, 3, 0, 3)]=(0.482401591015+0.606953537419j)*x[0]**o + ((0.00185897720658+0.39646929418j))*x[0] + ((-0.22524329322+0.0853657760291j))*x[1]**o + ((0.254657162476-0.668031989323j))*x[1]
            ref[(0, 3, 0, 3)]=(0.257158297795+0.692319313449j)*(1+2.*(dim-1)/(o+1.)) + ((0.256516139683-0.271562695143j))*dim
            arg[(0, 3, 1, 0)]=(0.1781365525+0.566665713494j)*x[0]**o + ((-0.750015503196-0.224199985849j))*x[0] + ((0.766780753113+0.799193371244j))*x[1]**o + ((0.619273777162-0.360221294191j))*x[1]
            ref[(0, 3, 1, 0)]=(0.944917305613+1.36585908474j)*(1+2.*(dim-1)/(o+1.)) + ((-0.130741726034-0.584421280039j))*dim
            arg[(0, 3, 1, 1)]=(-0.0406971764268-0.419943041607j)*x[0]**o + ((-0.0891565745597+0.590142569592j))*x[0] + ((0.560318373249-0.943930325109j))*x[1]**o + ((-0.14399390161+0.645859084288j))*x[1]
            ref[(0, 3, 1, 1)]=(0.519621196822-1.36387336672j)*(1+2.*(dim-1)/(o+1.)) + ((-0.23315047617+1.23600165388j))*dim
            arg[(0, 3, 1, 2)]=(-0.268626878444-0.871155062268j)*x[0]**o + ((-0.857276453978+0.113971407208j))*x[0] + ((0.872330911821-0.182339563187j))*x[1]**o + ((-0.784743589374+0.553993770382j))*x[1]
            ref[(0, 3, 1, 2)]=(0.603704033377-1.05349462545j)*(1+2.*(dim-1)/(o+1.)) + ((-1.64202004335+0.66796517759j))*dim
            arg[(0, 3, 1, 3)]=(-0.727024897708+0.445434972421j)*x[0]**o + ((-0.0507883436292+0.516662692429j))*x[0] + ((-0.794437685451+0.287688648979j))*x[1]**o + ((0.352392261307-0.0494399920182j))*x[1]
            ref[(0, 3, 1, 3)]=(-1.52146258316+0.7331236214j)*(1+2.*(dim-1)/(o+1.)) + ((0.301603917678+0.46722270041j))*dim
            arg[(0, 3, 2, 0)]=(0.377870724632+0.351458337992j)*x[0]**o + ((0.519668800371+0.320205340609j))*x[0] + ((0.87601564755+0.0519987696677j))*x[1]**o + ((-0.304254409744+0.936808305737j))*x[1]
            ref[(0, 3, 2, 0)]=(1.25388637218+0.40345710766j)*(1+2.*(dim-1)/(o+1.)) + ((0.215414390627+1.25701364635j))*dim
            arg[(0, 3, 2, 1)]=(-0.962812226122+0.616260541973j)*x[0]**o + ((0.639012624699+0.93091856552j))*x[0] + ((0.189823328929-0.910854484781j))*x[1]**o + ((0.509007662299-0.598540454599j))*x[1]
            ref[(0, 3, 2, 1)]=(-0.772988897193-0.294593942808j)*(1+2.*(dim-1)/(o+1.)) + ((1.148020287+0.33237811092j))*dim
            arg[(0, 3, 2, 2)]=(0.904328296486-0.356439694624j)*x[0]**o + ((-0.49607189995+0.388917755459j))*x[0] + ((-0.339164985359+0.613707504579j))*x[1]**o + ((-0.0366994956102-0.798201610419j))*x[1]
            ref[(0, 3, 2, 2)]=(0.565163311126+0.257267809956j)*(1+2.*(dim-1)/(o+1.)) + ((-0.53277139556-0.409283854961j))*dim
            arg[(0, 3, 2, 3)]=(0.483783832815-0.898926415228j)*x[0]**o + ((0.795710361016+0.109292979052j))*x[0] + ((0.244103695148+0.15566900482j))*x[1]**o + ((0.106560283141-0.556795178605j))*x[1]
            ref[(0, 3, 2, 3)]=(0.727887527962-0.743257410409j)*(1+2.*(dim-1)/(o+1.)) + ((0.902270644157-0.447502199553j))*dim
            arg[(0, 3, 3, 0)]=(0.0703397012342-0.975847952058j)*x[0]**o + ((-0.261091058196-0.321025197459j))*x[0] + ((0.644623278417+0.110029548225j))*x[1]**o + ((-0.536705441516-0.262951830346j))*x[1]
            ref[(0, 3, 3, 0)]=(0.714962979651-0.865818403833j)*(1+2.*(dim-1)/(o+1.)) + ((-0.797796499712-0.583977027804j))*dim
            arg[(0, 3, 3, 1)]=(0.679544468018-0.924704898922j)*x[0]**o + ((-0.473702858206+0.410716813628j))*x[0] + ((-0.686855403847+0.281505342581j))*x[1]**o + ((0.405837653106-0.596825625376j))*x[1]
            ref[(0, 3, 3, 1)]=(-0.0073109358296-0.643199556342j)*(1+2.*(dim-1)/(o+1.)) + ((-0.0678652050996-0.186108811748j))*dim
            arg[(0, 3, 3, 2)]=(-0.770473178891-0.469956648584j)*x[0]**o + ((0.0163372293019-0.302473004263j))*x[0] + ((-0.112563273667+0.887823381464j))*x[1]**o + ((-0.0451403259119-0.691797642762j))*x[1]
            ref[(0, 3, 3, 2)]=(-0.883036452558+0.41786673288j)*(1+2.*(dim-1)/(o+1.)) + ((-0.02880309661-0.994270647026j))*dim
            arg[(0, 3, 3, 3)]=(0.735776391272+0.867895083283j)*x[0]**o + ((-0.86964233819-0.771145834301j))*x[0] + ((-0.961100300533+0.709766845287j))*x[1]**o + ((-0.37128918423+0.226056204981j))*x[1]
            ref[(0, 3, 3, 3)]=(-0.225323909261+1.57766192857j)*(1+2.*(dim-1)/(o+1.)) + ((-1.24093152242-0.54508962932j))*dim
            arg[(1, 0, 0, 0)]=(0.195910515869-0.90540157371j)*x[0]**o + ((0.659734556368+0.492429831697j))*x[0] + ((0.380984094551+0.503873955197j))*x[1]**o + ((-0.541332453681+0.808132326279j))*x[1]
            ref[(1, 0, 0, 0)]=(0.57689461042-0.401527618513j)*(1+2.*(dim-1)/(o+1.)) + ((0.118402102687+1.30056215798j))*dim
            arg[(1, 0, 0, 1)]=(0.970281241554-0.777993953133j)*x[0]**o + ((-0.838645164862+0.955852793522j))*x[0] + ((-0.896131227077-0.81947444609j))*x[1]**o + ((0.610621226342-0.57401844317j))*x[1]
            ref[(1, 0, 0, 1)]=(0.0741500144768-1.59746839922j)*(1+2.*(dim-1)/(o+1.)) + ((-0.228023938521+0.381834350353j))*dim
            arg[(1, 0, 0, 2)]=(-0.783185607415+0.214857919045j)*x[0]**o + ((0.348653522569-0.372084223245j))*x[0] + ((0.707782130757-0.456572486874j))*x[1]**o + ((0.582301718887+0.401065979071j))*x[1]
            ref[(1, 0, 0, 2)]=(-0.0754034766576-0.241714567829j)*(1+2.*(dim-1)/(o+1.)) + ((0.930955241456+0.028981755826j))*dim
            arg[(1, 0, 0, 3)]=(-0.594244449426+0.963146413285j)*x[0]**o + ((-0.950766634017+0.0226429195585j))*x[0] + ((0.671818038425-0.205560676483j))*x[1]**o + ((0.503110555666-0.751670320249j))*x[1]
            ref[(1, 0, 0, 3)]=(0.0775735889989+0.757585736801j)*(1+2.*(dim-1)/(o+1.)) + ((-0.447656078351-0.72902740069j))*dim
            arg[(1, 0, 1, 0)]=(0.609631984267+0.996135994806j)*x[0]**o + ((0.470503375447-0.187619144626j))*x[0] + ((0.81858193243+0.910794632992j))*x[1]**o + ((-0.642918752607-0.373297056404j))*x[1]
            ref[(1, 0, 1, 0)]=(1.4282139167+1.9069306278j)*(1+2.*(dim-1)/(o+1.)) + ((-0.17241537716-0.56091620103j))*dim
            arg[(1, 0, 1, 1)]=(0.750527210593+0.497257564027j)*x[0]**o + ((-0.735980604423-0.101499232405j))*x[0] + ((-0.55890154215-0.0841323179299j))*x[1]**o + ((-0.307695885891+0.00564654562459j))*x[1]
            ref[(1, 0, 1, 1)]=(0.191625668443+0.413125246097j)*(1+2.*(dim-1)/(o+1.)) + ((-1.04367649031-0.0958526867803j))*dim
            arg[(1, 0, 1, 2)]=(-0.212119701067-0.182123997107j)*x[0]**o + ((0.190976529264-0.512442308918j))*x[0] + ((0.977463833507-0.906665113496j))*x[1]**o + ((-0.920982842859-0.0887682219384j))*x[1]
            ref[(1, 0, 1, 2)]=(0.765344132439-1.0887891106j)*(1+2.*(dim-1)/(o+1.)) + ((-0.730006313595-0.601210530856j))*dim
            arg[(1, 0, 1, 3)]=(0.212118546907-0.00424991368218j)*x[0]**o + ((-0.194782134093-0.19166008714j))*x[0] + ((0.0835962949818-0.356182152564j))*x[1]**o + ((-0.519532224963-0.320683630172j))*x[1]
            ref[(1, 0, 1, 3)]=(0.295714841889-0.360432066246j)*(1+2.*(dim-1)/(o+1.)) + ((-0.714314359056-0.512343717312j))*dim
            arg[(1, 0, 2, 0)]=(-0.318107638898-0.897430947618j)*x[0]**o + ((-0.75574569766-0.122335448013j))*x[0] + ((-0.0359223950439+0.104976658238j))*x[1]**o + ((-0.177588863506-0.598250752252j))*x[1]
            ref[(1, 0, 2, 0)]=(-0.354030033942-0.792454289381j)*(1+2.*(dim-1)/(o+1.)) + ((-0.933334561166-0.720586200265j))*dim
            arg[(1, 0, 2, 1)]=(0.940687247638-0.189663662236j)*x[0]**o + ((0.479745194933-0.835417833122j))*x[0] + ((-0.116594653549+0.78558175773j))*x[1]**o + ((0.73595430352-0.627027385324j))*x[1]
            ref[(1, 0, 2, 1)]=(0.82409259409+0.595918095494j)*(1+2.*(dim-1)/(o+1.)) + ((1.21569949845-1.46244521845j))*dim
            arg[(1, 0, 2, 2)]=(-0.980168889787+0.775609559682j)*x[0]**o + ((-0.0650618858823+0.435285386237j))*x[0] + ((0.704711430423-0.390804223537j))*x[1]**o + ((0.870310924219-0.133194950694j))*x[1]
            ref[(1, 0, 2, 2)]=(-0.275457459364+0.384805336145j)*(1+2.*(dim-1)/(o+1.)) + ((0.805249038337+0.302090435542j))*dim
            arg[(1, 0, 2, 3)]=(-0.658635154006-0.940675063311j)*x[0]**o + ((-0.110724224066+0.18135064194j))*x[0] + ((-0.600516666611-0.510188639369j))*x[1]**o + ((-0.081005992912+0.209034488671j))*x[1]
            ref[(1, 0, 2, 3)]=(-1.25915182062-1.45086370268j)*(1+2.*(dim-1)/(o+1.)) + ((-0.191730216978+0.390385130611j))*dim
            arg[(1, 0, 3, 0)]=(0.196030202866-0.90761084622j)*x[0]**o + ((-0.108322329125-0.629727285485j))*x[0] + ((0.28683183199+0.715401001039j))*x[1]**o + ((0.880157460265+0.343673332861j))*x[1]
            ref[(1, 0, 3, 0)]=(0.482862034856-0.192209845181j)*(1+2.*(dim-1)/(o+1.)) + ((0.771835131139-0.286053952623j))*dim
            arg[(1, 0, 3, 1)]=(0.317610039527-0.932093958212j)*x[0]**o + ((-0.999874456675-0.0404833308007j))*x[0] + ((-0.951202463323+0.803288248831j))*x[1]**o + ((0.319202103417-0.720015019743j))*x[1]
            ref[(1, 0, 3, 1)]=(-0.633592423796-0.128805709381j)*(1+2.*(dim-1)/(o+1.)) + ((-0.680672353258-0.760498350544j))*dim
            arg[(1, 0, 3, 2)]=(0.311390230844-0.974873780564j)*x[0]**o + ((0.717450052716-0.172696249102j))*x[0] + ((-0.808236933024+0.653266697008j))*x[1]**o + ((-0.323155964413+0.0457374944237j))*x[1]
            ref[(1, 0, 3, 2)]=(-0.49684670218-0.321607083556j)*(1+2.*(dim-1)/(o+1.)) + ((0.394294088303-0.126958754678j))*dim
            arg[(1, 0, 3, 3)]=(-0.487756581982-0.30421499622j)*x[0]**o + ((-0.764726102165-0.599691567473j))*x[0] + ((-0.759318209541-0.955504785018j))*x[1]**o + ((0.224262989837-0.0738958194127j))*x[1]
            ref[(1, 0, 3, 3)]=(-1.24707479152-1.25971978124j)*(1+2.*(dim-1)/(o+1.)) + ((-0.540463112329-0.673587386885j))*dim
            arg[(1, 1, 0, 0)]=(-0.530014496353+0.918929710694j)*x[0]**o + ((-0.131627441551-0.199385434093j))*x[0] + ((0.696477065046-0.958894050733j))*x[1]**o + ((-0.113924826572+0.933305325152j))*x[1]
            ref[(1, 1, 0, 0)]=(0.166462568693-0.0399643400388j)*(1+2.*(dim-1)/(o+1.)) + ((-0.245552268124+0.733919891059j))*dim
            arg[(1, 1, 0, 1)]=(0.996984814382-0.0639592977261j)*x[0]**o + ((-0.460327677675+0.573673898689j))*x[0] + ((0.259261402297-0.972154638667j))*x[1]**o + ((0.671819806566+0.203530644706j))*x[1]
            ref[(1, 1, 0, 1)]=(1.25624621668-1.03611393639j)*(1+2.*(dim-1)/(o+1.)) + ((0.211492128891+0.777204543395j))*dim
            arg[(1, 1, 0, 2)]=(-0.747430977873+0.982864535749j)*x[0]**o + ((0.0398427195615+0.955170704456j))*x[0] + ((0.550259084093+0.0609488775557j))*x[1]**o + ((-0.622274885129-0.557705634639j))*x[1]
            ref[(1, 1, 0, 2)]=(-0.19717189378+1.0438134133j)*(1+2.*(dim-1)/(o+1.)) + ((-0.582432165567+0.397465069817j))*dim
            arg[(1, 1, 0, 3)]=(-0.278234309553-0.78871668112j)*x[0]**o + ((-0.482558254775-0.50558714369j))*x[0] + ((0.389336800523+0.504832211703j))*x[1]**o + ((-0.555049380179-0.408711516713j))*x[1]
            ref[(1, 1, 0, 3)]=(0.11110249097-0.283884469417j)*(1+2.*(dim-1)/(o+1.)) + ((-1.03760763495-0.914298660403j))*dim
            arg[(1, 1, 1, 0)]=(-0.788351829312+0.107343967674j)*x[0]**o + ((-0.70291613036+0.663154613735j))*x[0] + ((0.593409717407+0.499430448109j))*x[1]**o + ((-0.741379896086+0.664989292872j))*x[1]
            ref[(1, 1, 1, 0)]=(-0.194942111905+0.606774415782j)*(1+2.*(dim-1)/(o+1.)) + ((-1.44429602645+1.32814390661j))*dim
            arg[(1, 1, 1, 1)]=(-0.44147352456+0.554791367471j)*x[0]**o + ((-0.53195470217+0.920923496888j))*x[0] + ((-0.913940314107-0.764584491595j))*x[1]**o + ((-0.745135495019+0.825872712465j))*x[1]
            ref[(1, 1, 1, 1)]=(-1.35541383867-0.209793124124j)*(1+2.*(dim-1)/(o+1.)) + ((-1.27709019719+1.74679620935j))*dim
            arg[(1, 1, 1, 2)]=(-0.090776198581+0.0043239571058j)*x[0]**o + ((-0.205186972652+0.364791460121j))*x[0] + ((0.00897177848328-0.157705552458j))*x[1]**o + ((0.36148770439+0.103473779647j))*x[1]
            ref[(1, 1, 1, 2)]=(-0.0818044200977-0.153381595353j)*(1+2.*(dim-1)/(o+1.)) + ((0.156300731739+0.468265239768j))*dim
            arg[(1, 1, 1, 3)]=(-0.323518728152-0.104541581459j)*x[0]**o + ((0.52650954347-0.517148676888j))*x[0] + ((0.788927866573+0.203384912481j))*x[1]**o + ((-0.191529548544+0.846871157932j))*x[1]
            ref[(1, 1, 1, 3)]=(0.465409138422+0.098843331022j)*(1+2.*(dim-1)/(o+1.)) + ((0.334979994926+0.329722481043j))*dim
            arg[(1, 1, 2, 0)]=(0.760840312926-0.0210578144673j)*x[0]**o + ((-0.791604721161-0.0991453223472j))*x[0] + ((-0.259801274134-0.488765507267j))*x[1]**o + ((-0.463834208597+0.118401098641j))*x[1]
            ref[(1, 1, 2, 0)]=(0.501039038791-0.509823321734j)*(1+2.*(dim-1)/(o+1.)) + ((-1.25543892976+0.0192557762942j))*dim
            arg[(1, 1, 2, 1)]=(-0.914554725934+0.0125025369j)*x[0]**o + ((-0.912130578397+0.307263670021j))*x[0] + ((0.93410526128+0.59614728378j))*x[1]**o + ((-0.913202123371-0.6