/****************************************************************************** 
 *
 * File:        smfplay.c
 * Version:     $Id: smfplay.c,v 1.23 1995/08/06 19:10:07 burgaard Exp $
 *              $Version: 1.1$
 *
 * Purpose:     Standard MIDI file player.
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MPU-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Description: This is a ``little'' example application that utilizes my Roland
 *              MPU-401 Device Driver.
 *
 *              It should give a pretty clear picture of what you can do with
 *              /dev/mpu401data if you ignore all the MIDI file parsing.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "mpu401.h"
#include "mpuioctl.h"

#include "midiconst.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"

/*
 * # of events to transfer to driver per demand.
 *
 * If you think replay gets sloppy sometimes, especially if a lot
 * of notes is being played at once, try make this amount bigger.
 * Recommended range is 50 to 200 events. There's no point in making
 * it bigger than 200 events (except for momentarily vasting kernel-
 * memory! ;-)
 */
#define PACK_SIZE 50

/*** GLOBAL VARIABLES *********************************************************/

static char skipcheck = 0;
static char noplay    = 0;
static char loop      = 0;
static char dorandom  = 0;
static char nogsreset = 0;
static char quiet     = 0;
static char verbose   = 0;

static int  mpudev    = 0;
static int  filec     = 0;

static char isplay   = 0;

static char dostop    = 0; /* SIGNAL handling */
static char docont    = 0; /* SIGNAL handling */

static midi_file        midifile;
static midi_file_timing timing;
static midi_file_data   queues;

static long totaltime = 0; /* in seconds */

/*** MEMORY HANDLING **********************************************************/

static char * my_malloc(long size)
{
  static char *tmp = NULL;

  if ( !(tmp = malloc(size)) )
    {
      fprintf(stderr, "Out of memory!\n");
      exit(-3);
    };

  return tmp;
}

/*** ARGUMENT PARSING *********************************************************/

static void usage(int exitcode)
{
  printf("Standard MIDI File Player version 1.1\n");
  printf("Copyright (c) 1994, 1995 Kim Burgaard\n");
  printf("\n");
  printf("Usage: smfplay [options] filename...\n");
  printf("\n");
  printf("   -snlrqgvh --skipcheck --noplay --loop --random --nogsreset\n");
  printf("   --quit --verbose --help\n");
  printf("\n");
  printf("   -s --skipcheck  Disables initial checking of files\n");
  printf("   -n --noplay     Do not play files - just check them\n");
  printf("   -l --loop       Loop replaying forever\n");
  printf("   -r --random     Play files in random order\n");
  printf("   -g --nogsreset  Do not send GSRESET MIDI SysEx message.\n");
  printf("   -q --quiet      No output to console.\n");
  printf("   -v --verbose    Verbose output to console. Two or three times\n");
  printf("                   produces increased output\n");
  printf("   -h --help       This help text.\n");
  exit(exitcode);
}

inline static char isfilename(char *s)
{
  return !( !s || !strlen(s) || s[0] == '-' );
}

static char isoption(char *s)
{
  return !( !s || !strlen(s) || s[0] != '-' );
}

static char islongoption(char *s)
{
  return !( !s || strlen(s) < 3 || s[0] != '-' || s[1] != '-' );
}

static int parsearg(int argc, char *argv[])
{
  int i, j;
  if (argc < 2) usage(1);

  for (i = 1; i < argc; i++)
    if ( islongoption(argv[i]) )
      {
	if ( !strcmp("--skipcheck", argv[i]) )
	  skipcheck = 1;
	if ( !strcmp("--noplay", argv[i]) )
	  noplay = 1;
	else if ( !strcmp("--verbose", argv[i]) )
	  {
	    if (!quiet)
	      {
		verbose++;
		if (verbose > MIDI_PRINT_ALL)
		  fprintf(stderr, "Repeating %s more than %d times will have no effect.\n",
			  argv[i], MIDI_PRINT_ALL);
	      }
	    else fprintf(stderr, "Ambiguous option %s with -v or --verbose\n", argv[i]);
	  }
	else if ( !strcmp("--quiet", argv[i]) )
	  {
	    quiet = 1;
	    if (verbose) fprintf(stderr, "Ambiguous option %s with -q or --quiet\n", argv[i]);
	    verbose = 0;
	  }
	else if ( !strcmp("--loop", argv[i]) )
	  {
	    if (!noplay) loop = 1;
	    else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
	  }
	else if ( !strcmp("--random", argv[i]) )
	  {
	    if (!noplay) dorandom = 1;
	    else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
	  }
	else if ( !strcmp("--nogsreset", argv[i]) )
	  {
	    if (!noplay) nogsreset = 1;
	    else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
	  }
	else if ( !strcmp("--help", argv[i]) )
	  usage(0);
	else
	  {
	      fprintf(stderr, "Unknown option `%s' :\n\n", argv[i]);
	      fprintf(stderr, "Try `smfplay --help' for more information\n");
	      exit(-1);
	  };
      }
    else if ( isoption(argv[i]) )
      {
	for (j = 1; j < strlen(argv[i]); j++)
	  {
	    switch (argv[i][j])
	      {
	      case 's':
		skipcheck = 1;
		break;
	      case 'n':
		noplay = 1;
		break;
	      case 'v':
		if (!quiet)
		  {
		    verbose++;
		    if (verbose > MIDI_PRINT_ALL)
		      fprintf(stderr, "Repeating %s more than %d times will have no effect.\n",
			      argv[i], MIDI_PRINT_ALL);
		  }
		else fprintf(stderr, "Ambiguous option %s with -v or --verbose\n", argv[i]);
		break;
	      case 'q':
		quiet = 1;
		if (verbose) fprintf(stderr, "Ambiguous option %s with -q or --quiet\n", argv[i]);
		verbose = 0;
		break;
	      case 'l':
		if (!noplay) loop = 1;
		else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
		break;
	      case 'r':
		if (!noplay) dorandom = 1;
		else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
		break;
	      case 'g':
		if (!noplay) nogsreset = 1;
		else fprintf(stderr, "Ambiguous option %s with -n or --noplay\n", argv[i]);
		break;
	      case 'h':
		usage(0);
		break;
	      default:
		fprintf(stderr, "Unknown option `-%c' :\n\n", argv[i][j]);
		fprintf(stderr, "Try `smfplay --help' for more information\n");
		exit(-1);
	      };
	  };
      }
    else
      filec++;

  if (!filec)
    { 
      fprintf(stderr, "Try `smfplay --help' for more information\n");
      exit(-1);
    };

  return 0;
}

/*** MIDI FILE HANDLING *******************************************************/

struct name_e
{
  struct name_e *prev;
  struct name_e *next;
  char *name;
  word timebase;
  long playtime;
};

typedef struct name_e name_e;

struct name_list
{
  name_e *current;
  long count;
};
typedef struct name_list name_list;

struct name_list names_A = { NULL, 0 };
struct name_list names_B = { NULL, 0 };

static void name_list_put(name_list *nl, name_e *nm, int putrandom)
{
  if (!nl || !nm) return;

  if (nl->current)
    {
      if (putrandom)
        {
	  long i = (rand() % (nl->count + 1)) + 1;
	  while (i--) nl->current = nl->current->next;
	};
      nm->prev = nl->current;
      nm->next = nl->current->next;
      nm->next->prev = nm;
      nl->current->next = nm;
      nl->current = nm;
    }
  else
    nl->current = nm->prev = nm->next = nm;

  nl->count++;
}

static name_e * name_list_add(name_list *nl, char *name, float playt, word timebase)
{
  name_e *tmp = NULL;

  if (!nl || !name) return NULL;

  tmp = (name_e *)my_malloc(sizeof(name_e));
  tmp->name = name;
  tmp->playtime = playt;
  tmp->timebase = timebase;
  name_list_put(nl, tmp, dorandom);
  return tmp;
}

static name_e * name_list_rem(name_list *nl, name_e *nm)
{
  if (!nl || !nm) return NULL;
  if (nl->current)
    {
      nm->prev->next = nm->next;
      nm->next->prev = nm->prev;
      if (nm == nl->current) nl->current = nl->current->next;
      nm->prev = nm->next = NULL;
      nl->count--;
    }
  else
    nm = NULL;

  if (!nl->count) nl->current = NULL;
  return nm;
}

static name_e * name_list_get(name_list *nl, int getrandom)
{
  if (!nl || !nl->current) return NULL;
  if (getrandom)
    {
      long i = (rand() % (nl->count + 1)) + 1;
      while (i--) nl->current = nl->current->next;
    };
  return name_list_rem(nl, nl->current);
};

static void name_list_clear(name_list *nl)
{
  if (!nl) return;
  while (nl->count) free(name_list_get(nl, 0));
}

static int mpuaction(void)
{
  midi_event *event = NULL;
  mpu_demand_t demand;
  long starttime = -1;
  long total = 0;
  int zerotime = 0, cur_dt, cur_h, cur_m, cur_s;
  int i = 0;
  char usehour = 0;

  usehour = ( (timing.time/(60*60)) % 60) > 0;

  isplay = 1;
  docont = 0;

  total = queues.voice.count;

  ioctl(mpudev, MPUIOC_RESET, 0);    /* play it safe */
  ioctl(mpudev, MPUIOC_GM_ON, 0);    /* ``Clean up'' the synth. */
  if (!nogsreset) ioctl(mpudev, MPUIOC_GS_RESET, 0); /* Even more if it's a Roland */

  if (!quiet)
    {
      if (midifile.title && midifile.copyright)
        {
          printf("[ %s, %s ] ", midifile.title, midifile.copyright);
          free(midifile.title);
          free(midifile.copyright);
        }
      else if (midifile.copyright)
        {
          printf("[ %s, %s ] ", midifile.name, midifile.copyright);
          free(midifile.copyright);
        }
      else if (midifile.title)
        {
          printf("[ %s ] ", midifile.title);
          free(midifile.title);
        }
      else
	printf("[ %s ] ", midifile.name);
    };

  midifile.title = midifile.copyright = NULL;

  /* usually there is not very many sysex events */
  while ( !docont && (event = midi_queue_get(&queues.sysex)) )
    {
      if (starttime == -1) starttime = event->time;
      else if (starttime > event->time) starttime = event->time;

      if (event->type == MIDI_TYP_SYSEX ||
	  (event->type == MIDI_TYP_META && (event->data[0]&0xff) == MID_META_SET_TEMPO))
	{
	  ioctl(mpudev, MPUIOC_PUT_EVENT, event);
	};
      midi_event_free(event);
    };

  /* preload 2 * PACK_SIZE meta events */
  for (i = 0; !docont && i < 2*PACK_SIZE && queues.meta.count > 0; i++)
    {
      if ( !(event = midi_queue_get(&queues.meta)) ) break;

      if (starttime == -1) starttime = event->time;
      else if (starttime > event->time) starttime = event->time;

      ioctl(mpudev, MPUIOC_PUT_EVENT, event);
      midi_event_free(event);
    };

  /* preload 2 * PACK_SIZE voice events */
  for (i = 0; !docont && i < 2*PACK_SIZE && queues.voice.count > 0; i++)
    {
      if ( !(event = midi_queue_get(&queues.voice)) ) break;

      if (starttime == -1) starttime = event->time;
      else if (starttime > event->time) starttime = event->time;

      ioctl(mpudev, MPUIOC_PUT_EVENT, event);
      midi_event_free(event);
    };

  fflush(NULL);

  if ( !docont )
    {
      ioctl(mpudev, MPUIOC_SET_TIMEBASE, &timing.division);
      ioctl(mpudev, MPUIOC_START_PLAY, &starttime);
      zerotime = time(NULL);
    };

  demand.record = 0;
  demand.message = MPU_BLK_NONE;

  if (!quiet)
    {
      cur_dt = time(NULL) - zerotime;
      cur_s = cur_dt % 60;
      cur_m = (cur_dt/60) % 60;
      cur_h = (cur_dt/(60*60)) % 60;
      i = ((((float)total - (float)queues.voice.count)*100.0)/(float)total);
      if (usehour) printf("%02d:", cur_h);
      printf("%02d:%02d %02d%%", cur_m, cur_s, (i < 100 && i >= 0) ? i : 0);
      fflush(NULL); /* force update */
    };

  while ( !docont && (queues.voice.count || queues.meta.count) )
    {
      demand.voice   = (queues.voice.count > PACK_SIZE) ? PACK_SIZE : queues.voice.count;
      demand.control = (queues.meta.count > PACK_SIZE) ? PACK_SIZE : queues.meta.count;
      demand.sync = 1;

      if (!quiet)
	{
	  cur_dt = time(NULL) - zerotime;
	  cur_s = cur_dt % 60;
	  cur_m = (cur_dt/60) % 60;
	  cur_h = (cur_dt/(60*60)) % 60;
	  if (usehour) printf("\b\b\b");
	  printf("\b\b\b\b\b\b\b\b\b");
	  i = ((((float)total - (float)queues.voice.count)*100.0)/(float)total);
	  if (usehour) printf("%02d:", cur_h);
	  printf("%02d:%02d %02d%%", cur_m, cur_s, (i < 100 && i >= 0) ? i : 0);
	  fflush(NULL); /* force update */
	};

      if (docont) break;

      if (ioctl(mpudev, MPUIOC_BLOCK_DEMAND, &demand)) return -EINVAL;
      if ( (demand.message & MPU_BLK_END) || docont)
	{
	  docont = 1;
	  break;
	};

      if (demand.message & MPU_BLK_VOICE)
        for (i = 0; i < PACK_SIZE && queues.voice.count > 0; i++)
          {
            if ( !(event = midi_queue_get(&queues.voice)) ) break;
            ioctl(mpudev, MPUIOC_PUT_EVENT, event);
            midi_event_free(event);
          };
      if (demand.message & MPU_BLK_CONTROL)
        for (i = 0; i < PACK_SIZE && queues.meta.count > 0; i++)
          {
            if ( !(event = midi_queue_get(&queues.meta)) ) break;
            ioctl(mpudev, MPUIOC_PUT_EVENT, event);
	    midi_event_free(event);
          };
    };

  isplay = 0;

  if (!docont)
    {
      ioctl(mpudev, MPUIOC_BLOCK_PLAY_END, 0);

      if (!quiet)
	{
	  cur_dt = time(NULL) - zerotime;
	  cur_s = cur_dt % 60;
	  cur_m = (cur_dt/60) % 60;
	  cur_h = (cur_dt/(60*60)) % 60;

	  if (usehour) printf("\b\b\b");
	  printf("\b\b\b\b\b\b\b\b\b");
	  if (usehour) printf("%02d:", cur_h);
	  printf("%02d:%02d 100%%\n", cur_m, cur_s);
	  fflush(NULL); /* force update */
	};
      sleep(1); /* Don't cut the last note played abrutly */
    };

  ioctl(mpudev, MPUIOC_STOP_ALL, 0);

  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  return 0;
}

static char buffer[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

static int load(char *nm, char doplay, word timebase)
{
  char *tn = nm;
  int i = 0;

  if ( !(midifile.file = fopen(nm, "rb")) )
    {
      fprintf(stderr, "Could not open `%s': ", nm);
      perror("");
      return -EINVAL;
    };

  if ( (tn = strrchr(nm, '/')) ) nm = ++tn; /* strip leading path */
  midifile.name = nm;

  if (skipcheck)
    {
      i = 0;

      if (
	  fread(&buffer[0], 10, 1, midifile.file) != 1 ||
	  memcmp(&buffer[0], "MThd\0\0\0\006", 8) != 0
	  )
	{
	  fprintf(stderr, "`%s' is not a Standard MIDI File\n", midifile.name);
	  i = -EINVAL;
	}
      else if ( (buffer[8]&0xff) || (buffer[9]&0xff) > 2 )
	{
	  fprintf(stderr, "MIDI File format 2 or higher not supported!\n");
	  i = -EINVAL;
	}
      else if ( ferror(midifile.file) )
	{
	  fprintf(stderr, "Error while reading `%s': ", midifile.name);
	  perror("");
	  i = ferror(midifile.file);
	};
      
      fclose(midifile.file);
      return i;
    };

  midifile.title = NULL;
  midifile.copyright = NULL;
  
  timing.newdivision = timebase;
  timing.time = 0.0;

  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  queues.dovoice = queues.dosysex = queues.dometa = doplay;

  if ( (i = midi_load(&midifile, &timing, &queues, verbose)) == 0 && doplay && !noplay) 
    {
      mpuaction();
    }
  else if ( ferror(midifile.file) )
    {
      fprintf(stderr, "Error while reading `%s': ", midifile.name);
      perror("");
      i = ferror(midifile.file);
    };

  fclose(midifile.file);
  return i;
}

static long validatefiles(int argc, char *argv[])
{
  word timebase = 0;
  char *tm = NULL;
  byte h = 0;
  byte m = 0;
  byte s = 0;
  int i = 0;

  totaltime = 0;

  if (!quiet) printf("Checking files...\n");

  for (i = 1; i < argc; i++)
    {
      if (!isfilename(argv[i])) continue;
      if (load(argv[i], 0, 120) == 0)
	{
	  if (skipcheck) timebase = 120;
	  else if (timing.maxtempo < 179) timebase = 192;
	  else if (timing.maxtempo < 208) timebase = 144;
	  else timebase = 120;

	  name_list_add(&names_A, argv[i], timing.time, timebase);

	  if (skipcheck) continue;
	  timing.time += 1; /* add time for allowing the last notes to fade out */
	  totaltime += timing.time;
	  
	  if (midifile.title)
	    tm = midifile.title;
	  else
	    {
	      if ( (tm = strrchr(argv[i], '/')) ) tm++;
	      else tm = argv[i];
	    };

	  s = timing.time % 60;
	  m = (timing.time/60) % 60;
	  h = (timing.time/(60*60));
	  
	  if (!quiet)
	    {
	      if (h > 0) printf("%02hd:", h);
	      printf("%02hd:%02hd ", m, s);

	      if (midifile.title && midifile.copyright)
		{
		  printf("[ %s, %s ]\n", midifile.title, midifile.copyright);
		  free(midifile.title);
		  free(midifile.copyright);
		}
	      else if (midifile.copyright)
		{
		  printf("[ %s, %s ]\n", midifile.name, midifile.copyright);
		  free(midifile.copyright);
		}
	      else if (midifile.title)
		{
		  printf("[ %s ]\n", midifile.title);
		  free(midifile.title);
		}
	      else
		printf("[ %s ]\n", midifile.name);
	    };

	  midifile.title = midifile.copyright = NULL;

	  midi_queue_flush(&queues.voice);
	  midi_queue_flush(&queues.sysex);
	  midi_queue_flush(&queues.meta);
	}
    };
  
  if (names_A.count > 1 && !skipcheck)
    {
      s = totaltime % 60;
      m = (totaltime/60) % 60;
      h = (totaltime/(60*60));
      
      if (!quiet)
	{
	  if (h > 0) printf("===");
	  printf("=====\n");
	  if (h > 0) printf("%02hd:", h);
	  printf("%02hd:%02hd Total playing time for %ld files.\n\n", m, s, names_A.count);
	};
    };
  
  return names_A.count;
}

static void playfiles(void)
{
  name_list *names_C = &names_A;
  name_e *current = NULL;
  name_e *old = NULL;

  srandom(time(NULL));
 
  if (names_C->count) names_C->current = names_C->current->next;
  else return;

  if ( (mpudev = open("/dev/mpu401data", O_RDWR)) == -1 )
    {
      perror("Could not open `/dev/mpu401'");
      name_list_clear(&names_A);
      name_list_clear(&names_B);
      exit(-3);
    };

  verbose = 0;   /* don't need this anymore! */
  skipcheck = 0; /* don't *want* this anymore! */

  while (names_C->count)
    {
      current = name_list_get(names_C, dorandom);
      if (dorandom && current == old && names_C->count > 1)
	{
	  current = name_list_get(names_C, dorandom);
	  name_list_put(names_C, old, dorandom);
	};
      old = current;

      if ( load(current->name, 1, current->timebase) )
	{
	  free(current);
	  current = 0;
	};

      if (loop)
	{
	  if (names_C == &names_A)
	    {
	      if (current) name_list_put(&names_B, current, dorandom);
	      if (!names_A.count)
		{
		  names_C = &names_B;
		  names_C->current = names_C->current->next;
		};
	    }
	  else
	    {
	      if (current) name_list_put(&names_A, current, dorandom);
	      if (!names_B.count)
		{
		  names_C = &names_A;
		  names_C->current = names_C->current->next;
		};
	    };
	};
    };

  if (!nogsreset) ioctl(mpudev, MPUIOC_GS_RESET, 0);
  close(mpudev);

  name_list_clear(&names_A);
  name_list_clear(&names_B);
}

void forcecont(void)
{
  if (names_A.count || names_B.count)
    {
      docont = 1;
      if (isplay)
	{
	  signal(SIGINT,  (void(*)()) forcecont);
	  if (!quiet)
	    {
	      printf(" skipping to next file...\n");
	      fflush(NULL);
	    };
	}
      else
	{
	  if (!quiet) printf("\n");
	  if (mpudev != -1)
	    {
	      dostop = 1;
	      ioctl(mpudev, MPUIOC_RESET, 0);
	      ioctl(mpudev, MPUIOC_GM_ON, 0);
	      if (!nogsreset) ioctl(mpudev, MPUIOC_GS_RESET, 0);
	      close(mpudev);
	    };
	  exit(1);
	};
      isplay = 0;
    }
  else
    {
      if (!quiet) printf("\n");
      if (mpudev != -1)
	{
	  dostop = 1;
	  ioctl(mpudev, MPUIOC_RESET, 0);
	  ioctl(mpudev, MPUIOC_GM_ON, 0);
	  if (!nogsreset) ioctl(mpudev, MPUIOC_GS_RESET, 0);
	  close(mpudev);
	};
      exit(1);
    };
};

void closedown(void)
{
  if (!quiet) printf("\n");
  if (mpudev != -1)
    {
      dostop = 1;
      ioctl(mpudev, MPUIOC_RESET, 0);
      ioctl(mpudev, MPUIOC_GM_ON, 0);    /* if it's GM compliant it'll shut up */
      if (!nogsreset) ioctl(mpudev, MPUIOC_GS_RESET, 0); /* if it's GS compliant it'll shut even more up */
      close(mpudev);
    };
  exit(1);
}

/*** MAIN *********************************************************************/

int main(int argc, char *argv[])
{
  if ( parsearg(argc, argv) ) return 1;

  mpudev = -1;

  /* Prevent hanging sounds if terminated by fx. Ctrl-C */
  signal(SIGINT,  (void(*)()) forcecont);
  signal(SIGQUIT, (void(*)()) closedown);
  signal(SIGTERM, (void(*)()) closedown);
  signal(SIGABRT, (void(*)()) closedown);

  midi_queue_reset(&queues.voice);
  midi_queue_reset(&queues.sysex);
  midi_queue_reset(&queues.meta);
  
  if (validatefiles(argc, argv) && !noplay) playfiles();
  else if (!noplay) fprintf(stderr, "\nNo files to play.\n");

  return 0;
}

/*** END OF FILE **************************************************************/
