/*
** Xtank
**
** Copyright 1988 by Terry Donahue
**
** display.c
*/

/*
$Author: lidl $
$Id: display.c,v 2.12 1992/01/29 08:37:01 lidl Exp $

$Log: display.c,v $
 * Revision 2.12  1992/01/29  08:37:01  lidl
 * post aaron patches, seems to mostly work now
 *
 * Revision 2.11  1992/01/26  04:59:20  stripes
 * delete this revision (KJL)
 *
 * Revision 2.10  1992/01/06  07:52:49  stripes
 * Changes for teleport
 *
 * Revision 2.9  1991/11/24  17:52:20  lidl
 * well, fixed an itts-bitsy bug in the mapper update bug.  Hopefull,
 * I've really and truely fixed it this time.
 *
 * Revision 2.8  1991/11/08  06:29:38  lidl
 * fixed the long-outstanding bug where the mapper doesn't get updated
 * properly when the vehicle you are observing dies -- an extra radar
 * line would get drawn into the radar display -- no more!
 *
 * Revision 2.7  1991/10/28  13:52:54  lidl
 * removed #ifdefs for NONAMETAGS -- they are now the default
 *
 * Revision 2.6  1991/09/19  02:39:55  lidl
 * NEW_STRING_REDRAW is now STINGY_REDRAW
 *
 * Revision 2.5  1991/09/19  02:28:58  lidl
 * eradicated STINGY_REDRAW code
 *
 * Revision 2.4  1991/09/18  22:09:29  lidl
 * additions of both STINGY_REDRAW and NEW_STINGY_REDRAW
 *
 * Revision 2.3  1991/02/10  13:50:22  rpotter
 * bug fixes, display tweaks, non-restart fixes, header reorg.
 *
 * Revision 2.2  91/01/20  09:57:35  rpotter
 * complete rewrite of vehicle death, other tweaks
 * 
 * Revision 2.1  91/01/17  07:11:13  rpotter
 * lint warnings and a fix to update_vector()
 * 
 * Revision 2.0  91/01/17  02:09:18  rpotter
 * small changes
 * 
 * Revision 1.1  90/12/29  21:02:18  aahz
 * Initial revision
 * 
*/

#include "xtank.h"

#ifdef BATCH_LINES
#define USE_BATCHED_LINES
#endif

#ifdef BATCH_POINTS
#define USE_BATCHED_POINTS
#endif

#include "malloc.h"
#include "screen.h"
#include "graphics.h"
#include "gr.h"
#include "vstructs.h"
#include "message.h"
#include "bullet.h"
#include "terminal.h"
#include "globals.h"


#if !defined(usleep)
#include <sys/time.h>
#define usleep(x) { \
    struct timeval st_delay; \
    st_delay.tv_usec = (x); \
    st_delay.tv_sec = 0; \
    select(0, NULL, NULL, NULL, &st_delay); \
}
#endif defined(usleep)


extern Terminal *term;
extern Weapon_stat weapon_stat[];
extern int team_color[];
extern int team_color_bright[];
extern Settings settings;
extern Map real_map;
extern int frame;


/*
** Displays everything on the current terminal.
*/
display_terminal(status, lastterm)
unsigned int status;
int lastterm;
{
    /* Display all of the windows */
    display_anim(status, lastterm);
    display_cons(status);
    display_map(status);
    display_game(status);
    display_help(status);
    display_msg(status);
    display_status(status);
}

/*
** Displays the vehicles, walls, landmarks, bullets, and explosions in the
** animation window.
*/
display_anim(status, lastterm)
unsigned int status;
int lastterm;
{
    Vehicle *v;
    int i;

    /* Check for being exposed */
    check_expose(ANIM_WIN, status);

    /* If we are turning this window on, clear it first */
    if (status == ON)
    {
	clear_window(ANIM_WIN);

	/* If paused, display the pause message */
	if (settings.game_speed == 0)
	    display_pause_message();
    }
    /* If 3d mode is on, display in 3d */
    if (term->status & TS_3d)
    {
	display_anim_3d(status);
	return;
    }
    /* Display the walls and landmarks in the maze */
    display_maze(status);

    /* Display the vehicles in a manner dependent on their status */
    for (i = 0; i < num_veh_alive; i++) {
	v = live_vehicles[i];
	switch (status)
	{
	  case REDISPLAY:
	    if (tstflag(v->status, VS_is_alive) &&
		    tstflag(v->status, VS_was_alive)) {
		display_vehicle(v, REDISPLAY);
	    } else if (tstflag(v->status, VS_is_alive)) {
		display_vehicle(v, ON);
	    } else if (tstflag(v->status, VS_was_alive)) {
		display_vehicle(v, OFF);
            }
	    break;
	  case ON:
	    if (tstflag(v->status, VS_is_alive))
		display_vehicle(v, ON);
	    break;
	  case OFF:
	    if (tstflag(v->status, VS_was_alive))
		display_vehicle(v, OFF);
	}
    }

    display_bullets(status, lastterm);
    display_explosions(status);
}


#define VEHICLE_NAME_Y 25

/*
** Displays the specified vehicle and its turrets in the animation window.
*/
display_vehicle(v, status)
Vehicle *v;
unsigned int status;
{
    Loc *loc, *old_loc;
    Picture *pic;

    /* Erase the old vehicle picture */
    if (status != ON) {
	old_loc = v->old_loc;
	pic = &v->obj->pic[v->vector.old_rot];
#ifdef STINGY_REDRAW
	loc = v->loc;
	if ( (status != REDISPLAY) ||
		(old_loc->screen_x[term->num] != loc->screen_x[term->num]) ||
		(old_loc->screen_y[term->num] != loc->screen_y[term->num]) ||
		(v->vector.old_rot != v->vector.rot) )
#endif /* STINGY_REDRAW */
	draw_picture(ANIM_WIN, old_loc->screen_x[term->num],
		     old_loc->screen_y[term->num], pic, DRAW_XOR, v->color);

	/* Erase the string showing name and team */
	if (!settings.si.no_nametags)
#ifdef STINGY_REDRAW
	if ( (status != REDISPLAY) ||
		(old_loc->screen_x[term->num] != loc->screen_x[term->num]) ||
		(old_loc->screen_y[term->num] != loc->screen_y[term->num]) )
#endif
	if (should_disp_name())
	    draw_text(ANIM_WIN,
		      old_loc->screen_x[term->num],
		      old_loc->screen_y[term->num] + VEHICLE_NAME_Y,
		      v->disp, S_FONT, DRAW_XOR, v->color);
    }

    /* Draw the new vehicle picture */
    if (status != OFF) {
	loc = v->loc;
	pic = &v->obj->pic[v->vector.rot];
#ifdef STINGY_REDRAW
	old_loc = v->old_loc;
	if ( (status != REDISPLAY) ||
		(old_loc->screen_x[term->num] != loc->screen_x[term->num]) ||
		(old_loc->screen_y[term->num] != loc->screen_y[term->num]) ||
		(v->vector.old_rot != v->vector.rot) )
#endif /* STINGY_REDRAW */
	draw_picture(ANIM_WIN, loc->screen_x[term->num],
		     loc->screen_y[term->num], pic, DRAW_XOR, v->color);

	/* Display a string showing name and team */
	if (!settings.si.no_nametags)
#ifdef STINGY_REDRAW
	if ( (status != REDISPLAY) ||
		(old_loc->screen_x[term->num] != loc->screen_x[term->num]) ||
		(old_loc->screen_y[term->num] != loc->screen_y[term->num]) )
#endif /* STINGY_REDRAW */

	if (should_disp_name()) {
	    draw_text(ANIM_WIN,
		      loc->screen_x[term->num],
		      loc->screen_y[term->num] + VEHICLE_NAME_Y,
		      v->disp, S_FONT, DRAW_XOR, v->color);
	}
    }
    /* Display the vehicle's turrets */
    display_turrets(v, status);
}

/*
** Displays all turrets for the specified vehicle in the animation window.
*/
display_turrets(v, status)
Vehicle *v;
unsigned int status;
{
	Picture *pic;
	Loc *loc, *old_loc;
	Object *obj;
	Coord *tcoord, *old_tcoord;
	Turret *t;
	int i;
#ifdef STINGY_REDRAW
	int mov_rot;
#endif /* STINGY_REDRAW */

	loc = v->loc;
	old_loc = v->old_loc;
#ifdef STINGY_REDRAW
	mov_rot = ( (old_loc->screen_x[term->num] != loc->screen_x[term->num]) ||
		(old_loc->screen_y[term->num] != loc->screen_y[term->num]) ||
		(v->vector.old_rot != v->vector.rot) );
#endif /* STINGY_REDRAW */
	for (i = 0; i < v->num_turrets; i++)
	{
		t = &v->turret[i];
		obj = t->obj;

		/* erase the old turret */
		if (status != ON)
		{
			old_tcoord = &v->obj->picinfo[v->vector.old_rot].turret_coord[i];
			pic = &obj->pic[t->old_rot];
#ifdef STINGY_REDRAW
			if ( (status != REDISPLAY) || mov_rot ||
#ifdef NEW_TURRETS
				(t->old_end.x != t->end.x) || (t->old_end.y != t->end.y) ||
#endif
				(t->old_rot != t->rot) )
#endif /* STINGY_REDRAW */
#ifndef TEST_TURRETS
			draw_picture(ANIM_WIN, old_loc->screen_x[term->num] + old_tcoord->x,
						 old_loc->screen_y[term->num] + old_tcoord->y,
						 pic, DRAW_XOR, v->color);
#else /* TEST_TURRETS */
			draw_line(ANIM_WIN, old_loc->screen_x[term->num] + old_tcoord->x,
						 old_loc->screen_y[term->num] + old_tcoord->y,
			                    old_loc->screen_x[term->num] + old_tcoord->x + t->old_end.x,
					    old_loc->screen_y[term->num] + old_tcoord->y + t->old_end.y,
						 DRAW_XOR, v->color);
#endif /* TEST_TURRETS */
		}
		/* draw the new turret */
		if (status != OFF)
		{
			tcoord = &v->obj->picinfo[v->vector.rot].turret_coord[i];
			pic = &obj->pic[t->rot];
#ifdef STINGY_REDRAW
			if ( (status != REDISPLAY) || mov_rot ||
#ifdef NEW_TURRETS
				(t->old_end.x != t->end.x) || (t->old_end.y != t->end.y) ||
#endif
				(t->old_rot != t->rot) )
#endif /* STINGY_REDRAW */
#ifndef TEST_TURRETS
			draw_picture(ANIM_WIN, loc->screen_x[term->num] + tcoord->x,
						 loc->screen_y[term->num] + tcoord->y,
						 pic, DRAW_XOR, v->color);
#else /* TEST_TURRETS */
			draw_line(ANIM_WIN, loc->screen_x[term->num] + tcoord->x,
						 loc->screen_y[term->num] + tcoord->y,
			                    loc->screen_x[term->num] + tcoord->x + t->end.x,
					    loc->screen_y[term->num] + tcoord->y + t->end.y,
						 DRAW_XOR, v->color);
#endif /* TEST_TURRETS */
		}
	}
}

/*
** If point_bullets is on, all non-disc bullets are drawn as points.
** Otherwise, the bullet bitmaps are used.
*/
display_bullets(status, lastterm)
unsigned int status;
int lastterm;
{
    extern Bset *bset;
    Bullet *b;
    Picture *pic;
    int i;
    int bullet_color;

    for (i = 0; i < bset->number; i++)
    {
	b = bset->list[i];
        pic = &bullet_obj->pic[(int)b->type];

	if (b->type == DISC && b->thrower != -1)
	{
	    bullet_color = b->thrower;
	    if (lastterm)
		b->thrower = -1;
	}
	else if (b->owner != NULL)
	{
	    bullet_color = b->owner->color;
	}
	else
	{
	    bullet_color = WHITE;
	}

	/* Erase the old picture of the bullet */
	if (status != ON)
	{
            if (b->life < weapon_stat[(int)b->type].frames - 1 &&
		b->life != -2)
	    {
		if (settings.point_bullets == TRUE && b->type != DISC)
		{
		    draw_point(ANIM_WIN, b->old_loc->screen_x[term->num],
			       b->old_loc->screen_y[term->num], DRAW_XOR,
			       bullet_color);
		}
		else
		{
		    draw_picture(ANIM_WIN, b->old_loc->screen_x[term->num],
				 b->old_loc->screen_y[term->num],
				 pic, DRAW_XOR,
				 bullet_color);
		}
	    }
	}
	if (b->owner != NULL)
	{
	    bullet_color = b->owner->color;
	}
	else
	{
	    bullet_color = WHITE;
	}

	/* Draw the new picture of the bullet */
	if (status != OFF)
	{
            if (b->life > 0 && b->life < weapon_stat[(int)b->type].frames)
	    {
		if (settings.point_bullets == TRUE && b->type != DISC)
		{
		    draw_point(ANIM_WIN, b->loc->screen_x[term->num],
			       b->loc->screen_y[term->num], DRAW_XOR,
			       bullet_color);
		}
		else
		{
		    draw_picture(ANIM_WIN, b->loc->screen_x[term->num],
				 b->loc->screen_y[term->num],
				 pic, DRAW_XOR,
				 bullet_color);
		}
	    }
	}
    }
#ifdef USE_BATCHED_POINTS
    if (settings.point_bullets) flush_point_batch;
#endif
}

/*
** Displays all the explosions visible in the animation window.
*/
display_explosions(status)
unsigned int status;
{
    extern Eset *eset;
    Exp *e;
    Picture *pic;
    int i;

    for (i = 0; i < eset->number; i++)
    {
	e = eset->list[i];

	/* Compute the address of the old picture */
	pic = &e->obj->pic[e->obj->num_pics - e->life - 1];

	/* Erase the old picture of the explosion */
	if (status != ON)
	    if (e->life < e->obj->num_pics)
		draw_picture(ANIM_WIN, e->old_screen_x[term->num],
			     e->old_screen_y[term->num], pic, DRAW_XOR, e->color);

	/* Draw the new picture of the explosion */
	if (status != OFF)
	    if (e->life > 0)
	    {
		pic++;
		draw_picture(ANIM_WIN, e->screen_x[term->num],
			     e->screen_y[term->num], pic, DRAW_XOR, e->color);
	    }
    }
}

/* Draws a north wall */
#define draw_north_wall(b,x,y) \
  if(b->flags & NORTH_WALL) \
    draw_hor(ANIM_WIN,x,y,BOX_WIDTH,DRAW_XOR, \
	     (b->flags & NORTH_DEST) ? GREY : WHITE)

/* Draws a west wall */
#define draw_west_wall(b,x,y) \
  if(b->flags & WEST_WALL) \
    draw_vert(ANIM_WIN,x,y,BOX_HEIGHT,DRAW_XOR, \
	      (b->flags & WEST_DEST) ? GREY : WHITE)

/* Draws fuel, ammo, armor, and goal in center, outpost wherever it is */
#define draw_type(b,line_x,line_y,fr) \
  switch(b->type) { \
    case FUEL: case AMMO: case ARMOR: case GOAL: case PEACE: case TELEPORT: \
      pic = &landmark_obj[0]->pic[(int)b->type - 1]; \
      draw_picture(ANIM_WIN,line_x+BOX_WIDTH/2,line_y+BOX_HEIGHT/2, \
		   pic,DRAW_XOR,WHITE); \
      break; \
    case OUTPOST: \
      pic = &landmark_obj[0]->pic[(int)b->type - 1]; \
      oc = outpost_coordinate(b,fr); \
      draw_picture(ANIM_WIN,line_x+oc->x,line_y+oc->y,pic,DRAW_XOR, \
		   team_color_bright[b->team]); \
  }

/* Draws team character in the center of the box */
#define draw_team(b,x,y) \
  if(b->team != 0) { \
    buf[0] = team_char[b->team]; \
    draw_text(ANIM_WIN,x+BOX_WIDTH/2,y+BOX_HEIGHT/2-5, \
	      buf,S_FONT,DRAW_XOR,team_color[b->team]); \
  }

/*
** Displays all the walls and landmarks in the animation window.
*/
display_maze(status)
unsigned int status;
{
    extern char team_char[];
    extern Object *landmark_obj[];
    extern Coord *outpost_coordinate();
    Coord *oc;
    Picture *pic;
    Intloc *sloc, *old_sloc;
    int right, bottom, i, j;
    int left_x, old_left_x, top_y, old_top_y;
    int line_x, old_line_x, line_y, old_line_y;
    int ox, oy;
    Box *b, *ob, temp;
    char buf[2];

    buf[1] = '\0';

    sloc = &term->loc;
    old_sloc = &term->old_loc;

    /* Draw just the north walls for the leftmost column of boxes. Draw just
       the west walls for the topmost row of boxes except for upperleftmost
       one, which isn't done at all.  For all the rest, do both the north and
       west walls. */
    switch (status)
    {
      case REDISPLAY:
	left_x = sloc->grid_x * BOX_WIDTH - sloc->x;
	old_left_x = old_sloc->grid_x * BOX_WIDTH - old_sloc->x;

	top_y = sloc->grid_y * BOX_HEIGHT - sloc->y;
	old_top_y = old_sloc->grid_y * BOX_HEIGHT - old_sloc->y;

	line_x = left_x;
	old_line_x = old_left_x;
	for (i = 0; i <= NUM_BOXES; i++)
	{
	    line_y = top_y;
	    old_line_y = old_top_y;
	    for (j = 0; j <= NUM_BOXES; j++)
	    {
		b = &real_map[sloc->grid_x + i][sloc->grid_y + j];
		ox = old_sloc->grid_x + i;
		oy = old_sloc->grid_y + j;
		ob = &real_map[ox][oy];

		/* If the old box has been changed, get the old value */
		if (ob->flags & BOX_CHANGED)
		    if (old_box(&temp, ox, oy))
			ob = &temp;

#ifdef STINGY_REDRAW
		if ( (sloc->x != old_sloc->x) || (sloc->y != old_sloc->y) ) {
#endif /* STINGY_REDRAW */

		/* Redisplay walls */
		if (j) {
		    draw_north_wall(ob, old_line_x, old_line_y);
		    draw_north_wall(b, line_x, line_y);
		}
		if (i) {
		    draw_west_wall(ob, old_line_x, old_line_y);
		    draw_west_wall(b, line_x, line_y);
		}
#ifdef STINGY_REDRAW
		} else {
			if ((b->flags & NORTH_WALL) != (ob->flags & NORTH_WALL)) {

				if (j) {
					draw_north_wall(ob, old_line_x, old_line_y);
					draw_north_wall(b, line_x, line_y);
				}
			}
			if ((b->flags & WEST_WALL) != (ob->flags & WEST_WALL)) {
				if (i) {
					draw_west_wall(ob, old_line_x, old_line_y);
					draw_west_wall(b, line_x, line_y);
				}
			}
		}
#endif
#ifdef STINGY_REDRAW
		if ( (sloc->x != old_sloc->x) || (sloc->y != old_sloc->y) || (ob->type == OUTPOST) ) {
#endif /* STINGY_REDRAW */


		/* Redisplay type */
		draw_type(ob, old_line_x, old_line_y, frame - 1);
		draw_type(b, line_x, line_y, frame);
#ifdef STINGY_REDRAW
		}
#endif
		/* Redisplay team */
#ifdef STINGY_REDRAW
		if ( (sloc->x != old_sloc->x) || (sloc->y != old_sloc->y) || (b->team != ob->team) ) {
#endif /* STINGY_REDRAW */

		draw_team(ob, old_line_x, old_line_y);
		draw_team(b, line_x, line_y);
#ifdef STINGY_REDRAW
		}
#endif

		line_y += BOX_HEIGHT;
		old_line_y += BOX_HEIGHT;
	    }
	    line_x += BOX_WIDTH;
	    old_line_x += BOX_WIDTH;
	}
	break;
      case ON:
	right = sloc->grid_x + NUM_BOXES + 1;
	bottom = sloc->grid_y + NUM_BOXES + 1;

	left_x = sloc->grid_x * BOX_WIDTH - sloc->x;
	top_y = sloc->grid_y * BOX_HEIGHT - sloc->y;

	line_x = left_x;
	for (i = sloc->grid_x; i < right; i++)
	{
	    line_y = top_y;
	    for (j = sloc->grid_y; j < bottom; j++)
	    {
		b = &real_map[i][j];

		/* Draw walls */
		if (j != sloc->grid_y)
		    draw_north_wall(b, line_x, line_y);
		if (i != sloc->grid_x)
		    draw_west_wall(b, line_x, line_y);

		/* Draw type */
		draw_type(b, line_x, line_y, frame)
		    /* Draw team */
		    draw_team(b, line_x, line_y);

		line_y += BOX_HEIGHT;
	    }
	    line_x += BOX_WIDTH;
	}
	break;
    }
#ifdef USE_BATCHED_LINES
    flush_line_batch;
#endif
}

/*
** Displays mapper and radar in the map window.
*/
display_map(status)
    unsigned int status;
{
    Vehicle *v = term->vehicle;
    /* Check for being exposed */
    check_expose(MAP_WIN, status);

    if (status == ON)
	clear_window(MAP_WIN);

    if (v != NULL) {
	unsigned int action;

	switch (status) {
	  case REDISPLAY:
	    action = SP_redisplay;
	    break;
	  case ON:
	    action = SP_draw;
	    break;
	  case OFF:
	    action = SP_erase;
	    break;
	}
	if (tstflag(v->status, VS_is_alive)) {
	    do_special(v, MAPPER, action);
	    do_special(v, RADAR, action);
#ifndef NO_NEW_RADAR
		do_special(v, NEW_RADAR, action);
		do_special(v, TACLINK, action);
#endif /* !NO_NEW_RADAR */
	} else {
	    if (v->death_timer == DEATH_DELAY - 1) {
		do_special(v, RADAR, SP_erase);
	    }
	}
    } else if (term->observer) {
	full_mapper(status);
	full_radar(status);
    }
}

/*
** Displays the console information in the console window.
*/
display_cons(status)
unsigned int status;
{
	unsigned int action;

	/* Check for being exposed */
	check_expose(CONS_WIN, status);

	switch (status)
	{
		case REDISPLAY:
			action = SP_redisplay;
			break;
		case ON:
			clear_window(CONS_WIN);
			action = SP_draw;
			break;
		case OFF:
			action = SP_erase;
			break;
	}

	if (term->vehicle != (Vehicle *) NULL)
		do_special(term->vehicle, CONSOLE, action);
}

/* these two message arrays must have the same size: */
char *help_normal[] = {
	"BASIC FUNCTIONS                                       DISC CONTROL           GAME FUNCTIONS         3D FUNCTIONS            SYNC CONTROL",
	"space  fire weapons (left)     C   toggle console     s   spin <=            Q   quit game          T   toggle 3D view      i   every frame",
	"t      turn turret (middle)    M   toggle mapper      d   spin toggle        P   pause game         W   toggle wide         o   every 2 frames",
	"g      turn tank   (right)     R   toggle radar       f   spin =>            <   slow down game     D   toggle distance     p   every 4 frames",
	"0-9    set forward drive       z   toggle safety      w   throw slow         >   speed up game      E   toggle extend       [   every 8 frames",
	"-      full reverse drive      +   toggle repair      e   throw medium                              L   toggle clipping     ]   every 16 frames",
	"!@#$%^ toggle weapons 1-6      F   toggle teleport    r   throw fast",
	"=      toggle all weapons      c   stop",
	"return send message            v   speed up",
	"                               x   slow down",
};
char *help_battle[] = {
	"0-9     track vehicle",
	"button  move view",
	"space   pause game",
	"w       map battle windows",
	"W       unmap battle windows",
	"<       slow game down",
	">       speed game up",
	"Q       quit",
	"",
	""
};

/*
** Displays helpful information in the help window.
*/
display_help(status)
unsigned int status;
{
    int i, lim;
    char **text;

    /* Check for being exposed */
    check_expose(HELP_WIN, status);

    if (status == ON)
    {
	clear_window(HELP_WIN);

	/* Determine which text to show in the help window */
	if (term->observer) {
	    text = help_battle;
	} else {
	    text = help_normal;
	}

	lim = (term->observer) ? sizeof(help_normal) / sizeof(help_normal[0]) :
		sizeof(help_battle) / sizeof(help_battle[0]);
	for (i = 0; i < lim; i++)
	    display_mesg(HELP_WIN, text[i], i, T_FONT);
    }
}

#ifdef S1024x864
#define VEH_X 1
#define VEH_Y 0
#define VEHICLE_H 53
#define VEHICLE_W 60

#define BULLET_X  180
#define BULLET_Y  0
#define BULLET_H  31

#define LAND_X    440
#define LAND_Y    0
#define LAND_H    32
#define LAND_W    85

#define EXP_X     160
#define EXP_Y     615
#define EXP_H     45
#define EXP_W     180

#define MSG_X     8
#define MSG_Y     2

#define PIC_X   30
#define PIC_Y   50
#define EXP_PIC_Y    -13
#define TEXT_OFFSET  48
#endif

/*
** Displays pictures of all bodies, bullets, explosions, and landmarks.
*/
display_pics()
{
    int exp_view, v_view, max_pics, split;
    int xpos;

    display_mesg2(HELP_WIN, "Hit any key or button to continue",
		  MSG_X, MSG_Y, XL_FONT);

    clear_window(ANIM_WIN);

    /* Put in separator rectangles between the pictures */
    draw_filled_rect(ANIM_WIN, BULLET_X - 1, 0, 3, EXP_Y, DRAW_COPY, WHITE);
    draw_filled_rect(ANIM_WIN, LAND_X - 1, 0, 3, EXP_Y, DRAW_COPY, WHITE);
    draw_filled_rect(ANIM_WIN, 0, EXP_Y - 1, ANIM_WIN_WIDTH, 3, DRAW_COPY,
		     WHITE);

    /* Draw the vehicles in one column */
    draw_text(ANIM_WIN, BULLET_X / 2, VEH_Y + 5, "Vehicles", L_FONT, DRAW_COPY,
	      WHITE);
    draw_objs(vehicle_obj, TRUE, 0, num_vehicle_objs, 0, VEH_X, VEH_Y,
	      VEHICLE_H);

    /* Draw the bullets in a column */
    draw_text(ANIM_WIN, (BULLET_X + LAND_X) / 2, BULLET_Y + 5,
	      "Bullets", L_FONT, DRAW_COPY, WHITE);
    draw_obj(bullet_obj, 0, BULLET_X, BULLET_Y, BULLET_H);

    /* Draw the landmarks in three columns */
    draw_text(ANIM_WIN, (LAND_X + ANIM_WIN_WIDTH) / 2, LAND_Y + 5,
	      "Landmarks", L_FONT, DRAW_COPY, WHITE);
    draw_text(ANIM_WIN, LAND_X + PIC_X + LAND_W, LAND_Y + 38,
	      "Game      Map   Design", M_FONT, DRAW_COPY, WHITE);
    draw_obj(landmark_obj[0], 1, LAND_X, LAND_Y + LAND_H, LAND_H);
    draw_obj(landmark_obj[1], 2, LAND_X + LAND_W, LAND_Y + LAND_H, LAND_H);
    draw_obj(landmark_obj[2], 3, LAND_X + 2 * LAND_W, LAND_Y + 3, LAND_H);

    /* Draw the explosions in 3 columns */
    split = (num_exp_objs + 1) / 3;
    draw_text_left(ANIM_WIN, 15, EXP_Y + 15, "Explosions", L_FONT, DRAW_COPY,
		   WHITE);
    xpos = EXP_X;
    draw_objs(exp_obj, TRUE, 0, split, 0, xpos, EXP_Y + EXP_PIC_Y, EXP_H);
    xpos += EXP_W;
    draw_objs(exp_obj, TRUE, split, 2 * split, 0, xpos, EXP_Y + EXP_PIC_Y,
	      EXP_H);
    xpos += EXP_W;
    draw_objs(exp_obj, TRUE, 2 * split, num_exp_objs, 0, xpos,
	      EXP_Y + EXP_PIC_Y, EXP_H);

    /* Animate the explosions and vehicles until a key or button is pressed */

    max_pics = vehicle_obj[0]->num_pics;	/* hack */
    exp_view = 0;
    v_view = 0;
    while (!scan_input())
    {
	usleep(40000);
	if (++exp_view >= max_pics) {
	    exp_view = 0;

	    if (++v_view >= max_pics)
		v_view = 0;

	    /* rotate vehicles one step */

#if 1
	    draw_filled_rect(ANIM_WIN, VEH_X + PIC_X - VEHICLE_W / 2,
			     VEH_Y + PIC_Y - VEHICLE_H / 2,
			     VEHICLE_W, num_vehicle_objs * VEHICLE_H,
			     DRAW_COPY, BLACK);
	    draw_objs(vehicle_obj, FALSE, 0, num_vehicle_objs, v_view,
		      VEH_X, VEH_Y, VEHICLE_H);
#else
	    /* draw in 2 sets to reduce flicker */
	    draw_filled_rect(ANIM_WIN, VEH_X + PIC_X - VEHICLE_W / 2,
			     VEH_Y + PIC_Y - VEHICLE_H / 2,
			     VEHICLE_W, vheight,
			     DRAW_COPY, BLACK);
	    draw_objs(vehicle_obj, FALSE, 0, num_vehicle_objs / 2, v_view,
		      VEH_X, VEH_Y, VEHICLE_H);
	    draw_filled_rect(ANIM_WIN, VEH_X + PIC_X - VEHICLE_W / 2,
			     VEH_Y + PIC_Y - VEHICLE_H / 2 + vheight,
			     VEHICLE_W, vheight,
			     DRAW_COPY, BLACK);
	    draw_objs(vehicle_obj, FALSE, num_vehicle_objs / 2,
		      num_vehicle_objs, v_view, VEH_X, VEH_Y + vheight,
		      VEHICLE_H);
#endif
	}
	xpos = EXP_X + PIC_X - EXP_H / 2;
	draw_filled_rect(ANIM_WIN, 
			 xpos, EXP_Y + EXP_PIC_Y + PIC_Y - EXP_H / 2,
			 EXP_H, split * EXP_H, DRAW_COPY, BLACK);
	xpos += EXP_W;
	draw_filled_rect(ANIM_WIN, 
			 xpos, EXP_Y + EXP_PIC_Y + PIC_Y - EXP_H / 2,
			 EXP_H, (2*split) * EXP_H, 
			 DRAW_COPY, BLACK);
	xpos += EXP_W;
	draw_filled_rect(ANIM_WIN, 
			 xpos, EXP_Y + EXP_PIC_Y + PIC_Y - EXP_H / 2,
			 EXP_H, (num_exp_objs - 2*split) * EXP_H, 
			 DRAW_COPY, BLACK);

	xpos = EXP_X;
	draw_objs(exp_obj, FALSE, 0, split, exp_view, xpos, EXP_Y + EXP_PIC_Y,
		  EXP_H);
	xpos += EXP_W;
	draw_objs(exp_obj, FALSE, split, 2*split, exp_view, xpos,
		  EXP_Y + EXP_PIC_Y, EXP_H);
	xpos += EXP_W;
	draw_objs(exp_obj, FALSE, 2*split, num_exp_objs, exp_view, xpos,
		  EXP_Y + EXP_PIC_Y, EXP_H);
	sync_output(FALSE);
    }

    clear_window(HELP_WIN);
}

/*
** Draws all the objects in the array in a vertical column, starting
** at the specified location and working downwards in jumps of height.
** The specified view is used for each object, provided it exists.
*/
draw_objs(obj, text, first, last, view, x, y, height)
Object *obj[];
Boolean text;
int first, last, view, x, y, height;
{
    int i;

    for (i = first; i < last; i++)
	if (view < obj[i]->num_pics)
	    draw_picture_string(obj[i], view, (text ? obj[i]->type : ""),
				x + PIC_X,
				y + PIC_Y + height * (i - first), 0);
}

/* Keep names of land mark &c &c types in a global where others can see 'em */
char *box_type_name[NUM_LANDMARK_TYPES];

init_box_names()
{
	int i;

    for (i = 0; i < NUM_LANDMARK_TYPES; i++) box_type_name[i] = "???";
	
    box_type_name[FUEL]      = "fuel";
    box_type_name[AMMO]      = "ammo";
    box_type_name[ARMOR]     = "armor";
    box_type_name[GOAL]      = "goal";
    box_type_name[OUTPOST]   = "outpost";
    box_type_name[PEACE]     = "peace";
    box_type_name[TELEPORT]  = "teleport";
    box_type_name[SCROLL_N]  = "scroll";
    box_type_name[SLIP]      = "slip";
    box_type_name[SLOW]      = "slow";
    box_type_name[START_POS] = "start";
}

/*
** Draws all the views of a given object in a vertical column, starting
** at the specified location and working downwards in jumps of height.
*/
draw_obj(obj, type, x, y, height)
Object *obj;
int type, x, y, height;
{
    extern Weapon_stat weapon_stat[];
    char *str;
    int adj, i;

	init_box_names();

    for (i = 0; i < obj->num_pics; i++)
    {
	adj = 0;
	str = "";
	if (type == 0)
	    str = weapon_stat[i].type;
	else if (type == 2)
	    adj = -4;
	else if (type == 3)
	{
	    /* Skip drawing the empty pixmap for normal landmark */
	    if (i == 0)
		continue;
	    str = box_type_name[i];
	    adj = -13;
	}
	draw_picture_string(obj, i, str, x + PIC_X, y + PIC_Y + height * i,
			    adj);
    }
}


/*
** Draws the specified view of the object with the string written beneath
** at the specified location in the animation window.  The adj parameter
** is added to the picture coordinates but not the text coordinates.
*/
draw_picture_string(obj, view, str, x, y, adj)
Object *obj;
char *str;
int view, x, y, adj;
{
    draw_picture(ANIM_WIN, x + adj, y + adj, &obj->pic[view], DRAW_COPY,
		 WHITE);
    if (str[0] != '\0')
	draw_text_left(ANIM_WIN, x + TEXT_OFFSET, y - font_height(M_FONT) / 2,
		       str, M_FONT, DRAW_COPY, WHITE);
}
