/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file is the global header file, containing definitions that
 * are needed about everywhere and prototypes for public functions.
 */

/* Essential typedefs */

/* This ought to be defined by the header files but I guess it isn't
 * always. */

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS 0
#endif
#ifndef EXIT_FAILURE
#define EXIT_FAILURE 1
#endif

typedef int token;
/* We define the token to be an ordinal type because they are _much_
 * easier to handle (than structures, for example). */

typedef struct token_list {
  token *l; /* Pointer to the actual list of tokens. */
  /* This pointer is always malloc()ated, and realloc()ated when the
   * need arises. */
  int size; /* Size of the list (number of tokens in it). */
  int allsize; /* Allocated size (also in tokens, _not_ bytes). */
} token_list;

typedef int command;
/* Command numbers used to be tokens in version 1.x.x but the complete
 * change in syntax has made it otherwise. We use negative numbers for
 * builtin commands and positive numbers for user-defined macros. */

#define UNREACHABLE (-2)
/* Character value which cannot be returned by getchar().  Used for
 * removing some special characters. */

/* Token values */

/* The actual values are unimportant. However, the special tokens
 * should come after characters (because to test whether a token is
 * special we use tests such as `if (t>=FIRST_SPECIAL)' */

/* Note that token value -1 is reserved for EOF. No other negative
 * value is used. Also, value 256 is reserved (who knows, it may turn
 * out to be useful some day). */

#define FIRST_SPECIAL 256
/* If we had a version which works with unicode we would change that
 * to 65536, I guess. */

/* These are the special tokens. */
#define BEGIN_COMMAND (FIRST_SPECIAL+1) /* < */
#define END_COMMAND (FIRST_SPECIAL+2)   /* > */
#define NEXT_PARAM (FIRST_SPECIAL+3)    /* | */
#define OPEN_QUOTE (FIRST_SPECIAL+4)    /* [ */
#define CLOSE_QUOTE (FIRST_SPECIAL+5)   /* ] */
#define QUOTE_NEXT (FIRST_SPECIAL+6)    /* # */
#define AT_SIGN (FIRST_SPECIAL+7)       /* @ */

/* Command values */

/* Command value 0 is reserved (it is used to mark the end of a
 * dictionary). */

/* These are the command numbers of builtin functions */
/* They must all be negative (and distinct). Apart from that, the
 * actual values are unimportant. */
#define BI_IDENTITY                  (-1001)
#define BI_VOID                      (-1002)
#define BI_OUTPUT                    (-1003)
#define BI_DEFINE                    (-1004)
#define BI_INCLUDE                   (-1005)
#define BI_IT                        (-1006)
#define BI_QUOTE                     (-1007)
#define BI_DQUOTE                    (-1008)
#define BI_INPUTFORM                 (-1009)
#define BI_IF                        (-1010)
#define BI_DEFOF                     (-1011)
#define BI_QDEFOF                    (-1012)
#define BI_QUIT                      (-1501)
#define BI_ERROR                     (-1502)
#define BI_HEAD                      (-2001)
#define BI_TAIL                      (-2002)
#define BI_AHEAD                     (-2011)
#define BI_ATAIL                     (-2012)
#define BI_LEN                       (-2021)
#define BI_PUSH                      (-3001)
#define BI_LAST                      (-3002)
#define BI_POP                       (-3003)
#define BI_POPLAST                   (-3004)
#define BI_DEPTH                     (-3005)
#define BI_PLUS                      (-4001)
#define BI_MINUS                     (-4002)
#define BI_TIMES                     (-4003)
#define BI_DIV                       (-4004)
#define BI_MOD                       (-4005)
#define BI_EQ                        (-4011)
#define BI_NEQ                       (-4012)
#define BI_GT                        (-4013)
#define BI_GE                        (-4014)
#define BI_LT                        (-4015)
#define BI_LE                        (-4016)
#define BI_AND                       (-4021)
#define BI_OR                        (-4022)
#define BI_NOT                       (-4023)
#define BI_BAND                      (-4031)
#define BI_BOR                       (-4032)
#define BI_BNOT                      (-4033)
#define BI_CARTYPE_ORDINARY          (-5001)
#define BI_CARTYPE_ESCAPE_NEXT       (-5002)
#define BI_CARTYPE_ESCAPE_STRING     (-5003)
#define BI_CARTYPE_COMMENT           (-5004)
#define BI_CARTYPE_ACTIVE            (-5005)
#define BI_TRANSLATE                 (-6001)
#define BI_FIND                      (-6002)
#define BI_SUBSTR                    (-6003)

/* Public functions */

#if defined(__GNUC__)
void fatal(char internal,char *fmt, ...)
__attribute__((noreturn,format(printf,2,3)));
#else
void fatal(char internal,char *fmt, ...);
#endif

void push_input_file(char *name);

token identify_command(char *name);

token input_token(void);

void syntax_cartype_ordinary(unsigned char c);

void syntax_cartype_escape_next(unsigned char c);

void syntax_cartype_escape_string(unsigned char c);

void syntax_cartype_comment(unsigned char c);

void syntax_cartype_active(unsigned char c,token_list *tl);

void init_syntax(void);

void output_token(token tk);

void expand_input(void);

void call_macro(token_list *expansion,int argc,token_list *argv);

void call_builtin(token_list *expansion,command cmd,int argc,token_list *argv);

void call_user(token_list *expansion,command cmd,int argc,token_list *argv);

void def_of_user(token_list *expansion,command cmd);

void define_user(token_list *definition,token tk);

/* Some more things I wish I didn't have to worry about. */

#if defined(HAVE_GOOD_ALLOC)
#define FREE free
#define REALLOC realloc
#else
#define FREE safe_free
#define REALLOC safe_realloc
#endif
