#include	"defs.h"
#include	"global.h"

extern struct fontop
    pkop,	/* pkfont.c */
    gfop,	/* gffont.c */
    vfop,	/* virfont.c */
    jvfop,	/* virfont.c */
    tfmop,	/* bifont.c */
    jstfmop,	/* bifont.c */
    jfmop,	/* bifont.c */
    cmpop,	/* compfont.c */
    dcmpop,	/* decompfont.c */
    execop,	/* execfont.c */
    funcop;	/* funcfont.c */
struct fontop *fontops[] = {
    &pkop,
    &gfop,
    &vfop,
    &jvfop,
    &tfmop,
    &jstfmop,
    &jfmop,
    &cmpop,
    &dcmpop,
    &execop,
    &funcop,
    NULL
};

extern struct confop
    fdcop,	/* fontdesc.c */
    fontcop,	/* fontdesc.c */
    defcop,	/* fontdesc.c */
    pdcop,	/* psrast.c */
    ddcop,	/* psrast.c */
    mapcop,	/* psbi.c */
    bicop,	/* psbi.c */
    bfcop,	/* psbi.c */
    bkcop,	/* psbi.c */
    vfccop,	/* vflfont.c */
    vfbcop,	/* vflfont.c */
    resolutioncop, /* fontdesc.c */
    includecop,	/* fontdesc.c */
    includeifcop,	/* fontdesc.c */
    setupcop,	/* fontdesc.c */
    setupifcop,	/* fontdesc.c */
    cconvcop,	/* fontdesc.c */
    substcop,	/* fontdesc.c */
    replfcop;	/* fontdesc.c */
struct confop *confops[] = {
    &fdcop,
    &fontcop,
    &defcop,
    &pdcop,
    &ddcop,
    &mapcop,
    &bicop,
    &bfcop,
    &bkcop,
    &vfccop,
    &vfbcop,
    &resolutioncop,
    &includecop,
    &includeifcop,
    &setupcop,
    &setupifcop,
    &cconvcop,
    &substcop,
    &replfcop,
    NULL
};


/*
 * read_fontdesc
 */
static struct fdin {
    char *name;
    FILE *file;
    int ch;
    int line;
} fdin;
#define	fdname	fdin.name
#define	fdf	fdin.file
#define	fdch	fdin.ch
#define	fdline	fdin.line

static struct libdir {
    char *dir, *lcldir;	/* library directory and local directory */
    char *defpath;	/* default path used when not found */
} libdir = {
    "", "", ""
};
struct libdir_body {
    char dir[PATHLEN], lcldir[PATHLEN];
    char *defpath;
};

static void
setlibdir(nld)
struct libdir_body *nld;
{
    libdir.dir = nld->dir;
    libdir.lcldir = nld->lcldir;
    libdir.defpath = nld->defpath;
}

void
read_fontdesc(fdn, top)
char *fdn;
__BOOLEAN__ top;
{
    char fdfile[PATHLEN];
    FILE *f;
    char field[STRSIZE];
    struct libdir_body newlibdir;
    struct fdin save_fdin;
    struct libdir save_libdir;
    struct confop *findconfop();

    if (searchfile(fdn, fdfile, &newlibdir, top, (char *)NULL)) {
#ifdef DEBUG
	if (Debuguser) {
	    (void)fprintf(stderr, "open fontdesc \"%s\"\n", fdfile);
	    (void)fprintf(stderr, "    in [%s]+[%s]:[%s]\n",
			  newlibdir.dir, newlibdir.lcldir, newlibdir.defpath);
	}
#endif
	f = fopen(fdfile, "r");
    } else
	Fatal("cannot open fontdesc file \"%s\"", fdn);
    save_fdin = fdin;
    save_libdir = libdir;
    fdname = fdfile;
    fdf = f;
    setlibdir(&newlibdir);
    for (fdline = 1; (fdch = getc(fdf)) != EOF; fdline++) {
	if (fdch == '#' || fdch == '\n') {
	    skipline();
	    continue;
	}
	getfield(field);
	(*(findconfop(field)->co_get))();
    }
    (void)fclose(fdf);
    fdin = save_fdin;
    libdir = save_libdir;
}

/* newf <- ld+lcd+f(+ext)
 * newlcd <- dir part of lcd+f
 */
static
findsearchfile(newf, newlcd, ld, lcd, f, ext)
char *newf, *newlcd, *ld, *lcd, *f, *ext;
{
    char *p;

    (void)strcpy(newlcd, lcd);
    (void)strcat(newlcd, f);
    (void)strcpy(newf, ld);
    (void)strcat(newf, newlcd);
    if (access(newf, R_OK) != 0) {
	if (ext != NULL) {
	    (void)strcat(newf, ext);
	    if (access(newf, R_OK) != 0)
		return FALSE;
	} else
	    return FALSE;
    }
    if ((p = strrchr(newlcd, '/')) != NULL)
	*(p+1) = '\0';
    else
	*newlcd = '\0';
    return TRUE;
}

/* newld <- copy of directory in path
 * newf, newlcd <- set by findsearchfile
 */
static
pathsearchfile(nextpath, path, newf, newlcd, newld, lcd, f, ext)
char **nextpath, *path;
char *newf, *newlcd, *newld, *lcd, *f, *ext;
{
    char *p, *d;

    for (; *path != '\0'; ) {
	for (; *path == PATH_SEP; path++)
	    ;
	for (p = path, d = newld; *p != PATH_SEP && *p != '\0'; )
	    *d++ = *p++;
	if (d == newld)
	    continue;
	if (*(d-1) != '/')
	    *d++ = '/';
	*d = '\0';
	if (findsearchfile(newf, newlcd, newld, lcd, f, ext)) {
	    *nextpath = p;
	    return TRUE;
	}
	if (*p == '\0')
	    break;
	path = p+1;
    }
    return FALSE;
}

searchfile(f, newf, nld, top, ext)
char *f, *newf;
struct libdir_body *nld;
__BOOLEAN__ top;
char *ext;
{
    char *dfp, *lcd, *next, *p;

#ifdef DEBUG
    if (Debuguser)
	(void)fprintf(stderr, "search file \"%s\" in [%s]+[%s]\n",
		      f, libdir.dir, libdir.lcldir);
#endif
    if (strncmp(f, "//", 2) == 0) {
	f += 2;
	lcd = "";
	dfp = top ? dvi2lib : libdir.defpath;
    } else if (*f == '/'
#if defined(MSDOS) || defined(WIN32)
	       || (isalpha(*f) && *(f+1) == ':')
#endif
	       ) {
	p = strrchr(f, '/');
#if defined(MSDOS) || defined(WIN32)
	if (p == NULL)
	    p = f+1;
#endif
	(void)strncpy(nld->dir, f, p+1-f);
	nld->dir[p+1-f] = '\0';
	nld->defpath = dvi2lib;
	return findsearchfile(newf, nld->lcldir, nld->dir, "", p+1, ext);
    } else if (top) {
	nld->defpath = dvi2lib;
	if (strncmp(f, "./", 2) == 0 || strncmp(f, "../", 3) == 0) {
	    p = strrchr(f, '/');
	    (void)strncpy(nld->dir, f, p+1-f);
	    nld->dir[p+1-f] = '\0';
	    return findsearchfile(newf, nld->lcldir, nld->dir, "", p+1, ext);
	}
	if (pathsearchfile(&next, dvi2path, newf, nld->lcldir, nld->dir,
			   "", f, ext))
	    return TRUE;
	lcd = "";
	dfp = dvi2lib;
    } else {
	if (findsearchfile(newf, nld->lcldir,
			   libdir.dir, libdir.lcldir, f, ext)) {
	    (void)strcpy(nld->dir, libdir.dir);
	    nld->defpath = libdir.defpath;
	    return TRUE;
	}
	lcd = libdir.lcldir;
	dfp = libdir.defpath;
    }

#ifdef DEBUG
    if (Debuguser)
	(void)fprintf(stderr, "search file \"%s\" in lib [%s]+[%s]\n",
		      f, dfp, lcd);
#endif
    if (pathsearchfile(&next, dfp, newf, nld->lcldir, nld->dir, lcd, f, ext)) {
	nld->defpath = next;
	return TRUE;
    }
    return FALSE;
}

searchfiledir(f, ld, newf)
char *f;
struct libdir_body *ld;
char *newf;
{
    struct libdir_body newlibdir;	/* dummy */
    struct libdir save_libdir;
    int found;

    save_libdir = libdir;
    setlibdir(ld);
    found = searchfile(f, newf, &newlibdir, FALSE, (char *)NULL);
    libdir = save_libdir;
    return found;
}

void
skipline()
{
    for (; fdch != '\n' && fdch != EOF; fdch = getc(fdf))
	;
}

void
skipblanks0()
{
    int c;

    for (; fdch == ' ' || fdch == '\t' || fdch == '\\'; fdch = getc(fdf))
	if (fdch == '\\')
	    if ((c = getc(fdf)) != '\n') {
		(void)ungetc(c, fdf);
		break;
	    }
}

void
skipblanks()
{
    skipblanks0();
    if (fdch == '\n')
	Fatal("fontdesc: %s illegal line %d", fdname, fdline);
}

getfield(char *field)
{
    char *fend;
    int c;

    skipblanks();
    for (fend = field+STRSIZE-1;
	 fdch != ' ' && fdch != '\t' && fdch != '\n' && fdch != EOF;
	 fdch = getc(fdf)) {
	if (fdch == '\\')
	    if ((c = getc(fdf)) != '\n') {
		(void)ungetc(c, fdf);
	    } else {
		fdch = ' ';
		break;
	    }
	if (field < fend)
	    *field++ = fdch;
	else
	    Fatal("fontdesc: field too long (%s line %d)", fdname, fdline);
    }
    *field = '\0';
    return 0;
}

getqfield0(field)
char *field;
{
    char *fend;
    int c;

    if (fdch == FDQUO) {
	*field++ = fdch;
	for (fend = field+STRSIZE-1;
	     (*field++ = fdch = getc(fdf)) != FDQUO && fdch != EOF; ) {
	    if (fdch == '\\')
		*(field-1) = getc(fdf);
	    if (field >= fend)
		Fatal("fontdesc: field too long (%s line %d)", fdname, fdline);
	}
	if (fdch == EOF)
	    Fatal("fontdesc: %c not closed (%s line %d)",
		  FDQUO, fdname, fdline);
	*field = '\0';
	fdch = getc(fdf);
    } else
	getfield(field);
}

getqfield(field)
char *field;
{
    skipblanks();
    getqfield0(field);
}

getoqfield(field)
char *field;
{
    skipblanks0();
    if (fdch != '\n')
	getqfield0(field);
    else
	*field = '\0';
}

getlfield(field)
char *field;
{
    char *fend;
    int c;

    skipblanks();
    for (fend = field+STRSIZE-1; fdch != '\n' && fdch != EOF;
	 fdch = getc(fdf)) {
	if (fdch == '\\')
	    if ((c = getc(fdf)) != '\n') {
		(void)ungetc(c, fdf);
	    } else {
		fdch = ' ';
	    }
	if (field < fend)
	    *field++ = fdch;
	else
	    Fatal("fontdesc: field too long (%s line %d)", fdname, fdline);
    }
    *field = '\0';
}

struct confop skipcop = {
    "",
    skipline
};

struct confop *
findconfop(field)
char *field;
{
    struct confop **co;

    for (co = confops; *co != NULL; co++)
	if (STREQ((*co)->co_name, field))
	    return *co;
    Warning("fontdesc: %s illegal (%s line %d)", field, fdname, fdline);
    return &skipcop;
}

/*
 * configuration operations
 */

void getfontdesc();
struct confop fdcop = {
    "fontdesc",
    getfontdesc
};

void
getfontdesc()
{
    char field_file[PATHLEN];
    char path[PATHLEN];

    getfield(field_file);
    skipline();
    defexpand(path, field_file);
    read_fontdesc(path, FALSE);
}

void
arg_fontdesc(fdn)
char *fdn;
{
    char path[PATHLEN];

    defexpand(path, fdn);
    read_fontdesc(path, TRUE);
}

void getfdfont();
struct confop fontcop = {
    "font",
    getfdfont
};

struct fontdesc {
    struct fontop *fd_op;
    char *fd_spec;	/* specifier */ /* not used */
    int	fd_sub;		/* font substitution */
    struct funcfont *fd_path;	/* prototype path */
    struct fontdesc *fd_next;
};
static struct fontdesc *fontdescs = NULL;
static struct fontdesc **nextfd = &fontdescs;

void
getfdfont()
{
    char field_type[STRSIZE];
    char field_spec[STRSIZE];
    char field_sub[STRSIZE];
    char field_path[STRSIZE];
    struct fontdesc *fd;
    struct fontop *fop;
    struct funcfont *ff;

    getfield(field_type);
    if ((fop = findfontop(field_type)) == NULL) {
	Warning("fontdesc: illegal font type %s (%s line %d)",
		field_type, fdname, fdline);
	skipline();
	return;
    }
    getfield(field_spec);
    getfield(field_sub);
    getlfield(field_path);
    if (!(*fop->fo_init)(field_path, &ff)) {
	Warning("fontdesc: illegal font line (%s line %d)", fdname, fdline);
	return;
    }
    fd = NEW(struct fontdesc, "fontdesc");
    fd->fd_op = fop;
    fd->fd_spec = strsave(field_spec);
    fd->fd_sub = atoi(field_sub);
    fd->fd_path = ff;
    fd->fd_next = NULL;
    *nextfd = fd;
    nextfd = &(fd->fd_next);
}

struct fontop *
findfontop(type)
char *type;
{
    struct fontop **fo;
    
    for (fo = fontops; *fo != NULL; fo++)
	if (STREQ((*fo)->fo_type, type))
	    return *fo;
    return NULL;
}

pathtype_init(proto, ff)
char *proto;
struct funcfont **ff;
{
    char path[PATHLEN];

    defexpand(path, proto);
    *ff = (struct funcfont *)strsave(path);
    return TRUE;
}

void getdef();
struct confop defcop = {
    "define",
    getdef
};

struct definition {
    char *def_name;
    char *def_body;
    struct definition *def_next;
};
static struct definition *definitions = NULL;
static struct definition *arg_defs = NULL;

add_def0(var, val)
char *var, *val;
{
    struct definition *def;

    if (val == NULL)
	val = "";
    def = NEW(struct definition, "definition");
    def->def_name = var;
    def->def_body = val;
    def->def_next = definitions;
    definitions = def;
}

add_def(var, val)
char *var, *val;
{
    char path[PATHLEN];
    struct definition *def;

    def = NEW(struct definition, "definition");
    def->def_name = strsave(var);
    defexpand(path, val);
    def->def_body = strsave(path);
    def->def_next = definitions;
    definitions = def;
}

struct definition *
get_def(var, defs)
char *var;
struct definition *defs;
{
    struct definition *def;

    for (def = defs; def != NULL; def = def->def_next)
	if (STREQ(var, def->def_name))
	    return def;
    return NULL;
}

arg_define(def)
char *def;
{
    char *val;

    for (val = def; *val != '=' && *val != '\0'; val++)
	;
    if (*val == '=')
	*val++ = '\0';
    add_def(def, val);
    arg_defs = definitions;
}

void getdef()
{
    char field_name[STRSIZE];
    char field_body[STRSIZE];
    char path[PATHLEN];
    struct definition *def;

    getfield(field_name);
    getfield(field_body);
    skipline();
    if (get_def(field_name, arg_defs) == NULL)
	add_def(field_name, field_body);
}

char *
getdefbody(var)
char *var;
{
    struct definition *def;

    return (def = get_def(var, definitions)) != NULL ? def->def_body : "";
}

void
defexpand(path, proto)
char *path, *proto;
{
    char *p, *s, *t;
    char *pend;
    char dname[STRSIZE];

    for (p = path, s = proto, pend = path+PATHLEN-1; *s != '\0'; s++)
	if (*s == '$') {
	    if (*++s == '{') {
		for (t = dname, ++s; *s != '}' && *s != '\0'; s++, t++)
		    *t = *s;
		if (*s == '\0')
		    --s;	/* error */
	    } else {
		for (t = dname; isalnum(*s); s++, t++)
		    *t = *s;
		--s;
	    }
	    *t = '\0';
	    t = getdefbody(dname);
	    if (p+strlen(t) >= pend)
		Fatal("font path too long %s", proto);
	    (void)strcpy(p, t);
	    p += strlen(t);
	} else if (p < pend) {
	    *p++ = *s;
	} else
	    Fatal("font path too long %s", proto);
    *p = '\0';
}

void getresolution();
struct confop resolutioncop = {
    "resolution",
    getresolution
};

set_resolution(r)
char *r;
{
    char val[PATHLEN];

    defexpand(val, r);
    resolution = atoi(val);
}

void
getresolution()
{
    char fd_resolution[STRSIZE];

    getfield(fd_resolution);
    skipline();
    set_resolution(fd_resolution);
}

void getinclude();
struct confop includecop = {
    "include",
    getinclude
};
void getincludeif();
struct confop includeifcop = {
    "includeif",
    getincludeif
};
void getsetup();
struct confop setupcop = {
    "setup",
    getsetup
};
void getsetupif();
struct confop setupifcop = {
    "setupif",
    getsetupif
};

#define	DF_ALWAYS	0
struct devfile {
    char *df_file;
    void (*df_op)();
    int df_kind;
    struct devfile *df_next;
};
static struct devfile *includefiles = NULL;
static struct devfile **nextif = &includefiles;
static struct devfile *setupfiles = NULL;
static struct devfile **nextsf = &setupfiles;

void
add_include(f, top)
char *f;
__BOOLEAN__ top;
{
    add_devfile(f, top, DF_ALWAYS, "includefile", &nextif);
}

void
add_includeif(f, top, kind)
char *f;
__BOOLEAN__ top;
int kind;
{
    add_devfile(f, top, kind, "includefile", &nextif);
}

void
add_setup(f, top)
char *f;
__BOOLEAN__ top;
{
    add_devfile(f, top, DF_ALWAYS, "setupfile", &nextsf);
}

void
add_setupif(f, top, kind)
char *f;
__BOOLEAN__ top;
int kind;
{
    add_devfile(f, top, kind, "setupfile", &nextsf);
}

add_devfile(char *f, __BOOLEAN__ top, int kind, char *k, struct devfile ***np)
{
    struct devfile *df;
    char file[PATHLEN];
    char incfile[PATHLEN];
    struct libdir_body newlibdir;	/* dummy */

    df = NEW(struct devfile, k);
    if (*f == FDQUO) {
	*(f+strlen(f)-1) = '\0';
	df->df_file = strsave(f+1);
	df->df_op = dev_copystring;
    } else {
	defexpand(file, f);
	if (searchfile(file, incfile, &newlibdir, top, INCEXT)) {
	    df->df_file = strsave(incfile);
	    df->df_op = dev_copyfile;
	} else
	    Fatal("cannot open file %s", file);
    }
    df->df_kind = kind;
    df->df_next = NULL;
    **np = df;
    *np = &(df->df_next);
    return 0;
}

add_include_spec(f, op)
char *f;
void (*op)();
{
    struct devfile *df;

    df = NEW(struct devfile, "special header");
    df->df_file = strsave(f);
    df->df_op = op;
    df->df_next = NULL;
    *nextif = df;
    nextif = &(df->df_next);
}

add_setup_spec(f, op)
char *f;
void (*op)();
{
    struct devfile *df;

    df = NEW(struct devfile, "special header");
    df->df_file = strsave(f);
    df->df_op = op;
    df->df_next = NULL;
    *nextsf = df;
    nextsf = &(df->df_next);
}

void
getinclude()
{
    getdevfile(add_include);
}

void
getincludeif()
{
    getdevfileif(add_includeif);
}

void
getsetup()
{
    getdevfile(add_setup);
}

void
getsetupif()
{
    getdevfileif(add_setupif);
}

getdevfile(add)
void (*add)();
{
    char field_file[PATHLEN];

    getqfield(field_file);
    skipline();
    (*add)(field_file, FALSE);
    return 0;
}

getdevfileif(add)
void (*add)();
{
    char field_kind[PATHLEN];
    char field_file[PATHLEN];
    int kind;

    getfield(field_kind);
    getqfield(field_file);
    skipline();
    (*add)(field_file, FALSE, dev_devfilekind(field_kind));
    return 0;
}

do_include()
{
    do_devfile(includefiles);
}

do_setup()
{
    do_devfile(setupfiles);
}

do_devfile(struct devfile *df)
{
    for (; df != NULL; df = df->df_next)
	if (df->df_kind == DF_ALWAYS || dev_devfileif(df->df_kind))
	    (*df->df_op)(df->df_file);
    return 0;
}

void getcconv();
struct confop cconvcop = {
    "cconvdir",
    getcconv
};

static struct libdir_body cconvlibdir;

void
getcconv()
{
    char field_mapdir[STRSIZE];

    getfield(field_mapdir);
    skipline();
    strcpy(cconvlibdir.dir, libdir.dir);
    strcpy(cconvlibdir.lcldir, field_mapdir);
    strcat(cconvlibdir.lcldir, "/");
    cconvlibdir.defpath = dvi2lib;
}

findcconvmap(name, file)
char *name, *file;
{
    return searchfiledir(name, &cconvlibdir, file);
}

void getsubst();
struct confop substcop = {
    "subst",
    getsubst
};

struct fontsubst {
    char *fs_font;
    int fs_len;
    int fs_reqmag;
    int fs_submag;
    struct fontsubst *fs_next;
};
static struct fontsubst *fontsubsts = NULL;
static struct fontsubst **nextfs = &fontsubsts;

void
getsubst()
{
    char field_font[STRSIZE];
    char field_reqmag[STRSIZE];
    char field_submag[STRSIZE];
    struct fontsubst *fs;;

    getfield(field_font);
    getfield(field_reqmag);
    getfield(field_submag);
    skipline();
    fs = NEW(struct fontsubst, "fontsubst");
    fs->fs_font = strsave(field_font);
    fs->fs_len = strlen(field_font);
    if (strcmp(field_reqmag, "%R") == 0)
	fs->fs_reqmag = resolution;
    else
	fs->fs_reqmag = atoi(field_reqmag);
    fs->fs_submag = atoi(field_submag);
    fs->fs_next = NULL;
    *nextfs = fs;
    nextfs = &(fs->fs_next);
}

void getreplfont();
struct confop replfcop = {
    "replfont",
    getreplfont
};

struct fontreplace {
    char *fr_replfont;
    char *fr_font;
    int fr_ds;
    int fr_fix;
    struct fontreplace *fr_next;
};
static struct fontreplace *fontreplaces = NULL;
static struct fontreplace **nextfr = &fontreplaces;

void
getreplfont()
{
    char field_replfont[STRSIZE];
    char field_font[STRSIZE];
    char field_par[STRSIZE];
    struct fontreplace *fr;
    char *c, *f;
    __BOOLEAN__ fixs;
    float fx;

    getfield(field_replfont);
    getfield(field_font);
    getfield(field_par);
    skipline();
    fr = NEW(struct fontreplace, "fontreplace");
    fr->fr_replfont = strsave(field_replfont);
    fr->fr_font = strsave(field_font);
    for (c = field_par; *c != '\0' && *c != ','; c++)
	;
    fixs = *c == ',';
    *c = '\0';
    fr->fr_ds = atoi(field_par)<<16;
    fr->fr_fix = -1;
    while (fixs) {
	for (f = ++c; *c != '\0' && *c != '*'; c++)
	    ;
	fixs = *c == '*';
	*c = '\0';
	if (*f == 'f') {
	    if (fr->fr_fix >= 0)
		fr->fr_fix = scale_exact(fr->fr_fix, atoi(f+1));
	    else
		fr->fr_fix = atoi(f+1);
	} else {
	    (void)sscanf(f, "%f", &fx);
	    if (fr->fr_fix >= 0)
		fr->fr_fix = scale_exact(fr->fr_fix, (int)(fx*(float)(1<<20)));
	    else
		fr->fr_fix = (int)(fx*(float)(1<<20));
	}
    }
    fr->fr_next = NULL;
    *nextfr = fr;
    nextfr = &(fr->fr_next);
}

/*
 * initialize and fix some default values
 */
char *mfmodevname = "mode";
char *kpsevname = "kpse";

init_default(def_mfmode, def_kpse)
char *def_mfmode, *def_kpse;
{
    add_def0(mfmodevname, mfmode = def_mfmode);
    add_def0(kpsevname, def_kpse);
}

fix_default(def_resolution, def_mfmode)
int def_resolution;
char *def_mfmode;
{
    char *m;

    if (resolution <= 0)
	resolution = def_resolution;
    m = getdefbody(mfmodevname);
    if (*m != '\0')
	mfmode = m;
    else
	add_def0(mfmodevname, mfmode = def_mfmode);
}


/*
 * replace font
 */
replfont(n, s, rn, rd, rs)
char *n;
int s;
char *rn;
int *rd, *rs;
{
    struct fontreplace *fr;
    char *sb, *se;

    for (fr = fontreplaces; fr != NULL; fr = fr->fr_next)
	if (match_subf(fr->fr_replfont, n, FALSE, &sb, &se)) {
	    subst_subf(rn, fr->fr_font, sb, se);
	    *rd = fr->fr_ds;
	    if (fr->fr_fix < 0)
		*rs = s;
	    else
		*rs = scale_exact(fr->fr_fix, s);
	    return TRUE;
	}
    return FALSE;
}

/*
 * init_fontinfo
 */
void
init_fontinfo(fe)
struct font_entry *fe;
{
    struct accarg acca;
    float rawmagfact, newmagfact;
    struct fontdesc *fd;
    struct fontsubst *fs;
    char *sb, *se;
    int i, next;
    int null_markchar();
    static int advice = FALSE;

    acca.rawmagfact = rawmagfact = newmagfact = dev_fontmag(fe);
    acca.pv_name = fe->n;
    /* We don't treat the case fe->a != 0 */
#ifdef DEBUG
    if (Debuguser)
	(void)fprintf(stderr, "<- Begin to find font file for %s (mag %d)\n",
		      fe->n, MAGSIZE(rawmagfact));
#endif

    acca.acc_mode = ACC_EXACT;
    acca.actmagfact = actfact(acca.rawmagfact);
    for (fd = fontdescs; fd != NULL; fd = fd->fd_next) {
	if ((*(fd->fd_op->fo_access))(fd->fd_path, fe, &acca)) {
#ifdef DEBUG
	    if (Debuguser)
		(void)fprintf(stderr, "-> Font file for %s (mag %d) found\n",
			      fe->n, MAGSIZE(rawmagfact));
#endif
	    (*(fd->fd_op->fo_initfontinfo))(fe);
	    return;
	}
    }

    acca.acc_mode = ACC_SUBST;
    for (fs = fontsubsts; fs != NULL; fs = fs->fs_next)
	if (match_subf(fs->fs_font, fe->n, TRUE, &sb, &se)) {
	    acca.submag = fs->fs_submag;
	    acca.reqmag = fs->fs_reqmag;
	    for (fd = fontdescs; fd != NULL; fd = fd->fd_next) {
		if ((fd->fd_sub&ACC_SUBST) == 0)
		    continue;
		if ((*(fd->fd_op->fo_access))(fd->fd_path, fe, &acca)) {
		    Warning("-> Font file for %s: font mag %d replaced by %d",
			    fe->n, fs->fs_reqmag, fs->fs_submag);
		    (*(fd->fd_op->fo_initfontinfo))(fe);
		    return;
		}
	    }
	}

    acca.acc_mode = ACC_MAGSTEP;
    (void)apprfact(rawmagfact);
    for (i = 0; ; i = next) {
	acca.stepmagfact = newmagfact = mag_table[mag_index+i];
	for (fd = fontdescs; fd != NULL; fd = fd->fd_next) {
	    if ((fd->fd_sub&ACC_MAGSTEP) == 0)
		continue;
	    if ((*(fd->fd_op->fo_access))(fd->fd_path, fe, &acca)) {
		Warning("-> Font file for %s: magnification %d replaced by %d",
			fe->n, MAGSIZE(rawmagfact), MAGSIZE(newmagfact));
		(*(fd->fd_op->fo_initfontinfo))(fe);
		return;
	    }
	}
	if (i >= 0) {
	    if (mag_index+(next = -i-1) < 0)
		if (mag_index+(next = i+1) >= magtabsize)
		    break;
	} else {
	    if (mag_index+(next = -i) >= magtabsize)
		if (mag_index+(next = i-1) < 0)
		    break;
	}
    }

    acca.actmagfact = actfact(acca.rawmagfact);
    for (fd = fontdescs; fd != NULL; fd = fd->fd_next) {
	if ((fd->fd_sub&~ACC_EXIST) == 0)
	    continue;
	acca.acc_mode = fd->fd_sub&~ACC_EXIST;
	if ((*(fd->fd_op->fo_access))(fd->fd_path, fe, &acca)) {
#ifdef DEBUG
	    if (Debuguser)
		(void)fprintf(stderr, "-> Font file for %s (mag %d) found\n",
			      fe->n, MAGSIZE(rawmagfact));
#endif
	    (*(fd->fd_op->fo_initfontinfo))(fe);
	    return;
	}
    }

    /* allow missing FNT files */
    Warning("%sNo font file for %s (mag %d)", Debugoff ? "" : "-> ",
	    fe->n, MAGSIZE(rawmagfact));
    if (Debugoff && !advice) {
	advice = TRUE;
	Warning("(use -d option to know the font file names tried)");
    }
    fe->fnt_markchar = null_markchar;
    (void)sprintf(fe->name, "null:%s:%d", fe->n, MAGSIZE(rawmagfact));
}

/*
 * Turn a prototype path into a fullpath.
 */
void
pave(path, proto, acca)
char *path, *proto;
struct accarg *acca;
{
    char *p, *s, *t;
    char *pend;
    int len;
    char buf[32];
#ifdef MSDOS
    char *l, *b;
#endif

    for (p = path, s = proto, pend = path+PATHLEN-1; *s != '\0'; s++)
	if (*s == '%') {
	    switch (*++s) {
	    case 'f':
	    case 'n':
	    case 's':
#ifdef MSDOS
		len = strlen(t = acca->pv_name);
		if ((l = strrchr(t, '.')) == NULL)
		    l = t+len;
		if (l-t > 8) {
		    if (G_longfontname) {
			(void)strncpy(buf, t, 4);
			(void)strncpy(buf+4, l-4, 4);
		    } else {
			(void)strncpy(buf, t, 8);
		    }
		    b = buf+8;
		} else {
		    (void)strncpy(buf, t, l-t);
		    b = buf+(l-t);
		}
		(void)strncpy(b, l, len-(l-t)+1);
		t = buf;
#else
		t = acca->pv_name;
#endif
		break;
	    case 'd':
	    case 'm':
		(void)sprintf(t = buf, "%d", acca->pv_mag);
		break;
	    case 'M':
		(void)sprintf(t = buf, "%f", acca->actmagfact);
		break;
	    case 'R':
		(void)sprintf(t = buf, "%d", resolution);
		break;
	    case 'F':
		t = acca->pv_fam;
		break;
	    case 'D':
		(void)sprintf(t = buf, "%d", acca->pv_ds);
		break;
	    case 'j':
		t = acca->pv_jsub;
		break;
	    default:
		*(t = buf) = *s;  *(t+1) = '\0';
		break;
	    }
	    if (p + (len = strlen(t)) <= pend) {
		(void)strncpy(p, t, len);
		p += len;
	    } else
		Fatal("font path too long %s", proto);
	} else if (p < pend) {
	    *p++ = *s;
	} else
	    Fatal("font path too long %s", proto);
    *p = '\0';
}

#ifdef KPATHSEA
static char pathk[PATHLEN], pathk0[PATHLEN], base[PATHLEN];

void
pavek(path, pathp, basep, proto, acca)
char *path, **pathp, **basep, *proto;
struct accarg *acca;
{
    char *p, *pthk;

    pave(pathk, proto, acca);
    if (strncmp(pathk, "//", 2) == 0) {
	(void)strcpy(pathk0, dvi2lib);
	(void)strcat(pathk0, "/");
	(void)strcat(pathk0, pathk+2);
	pthk = pathk0;
    } else 
	pthk = pathk;
    (void)strcpy(path, pthk);
    *pathp = pthk;
    for (p = pthk+strlen(pthk)-1; p >= pthk; --p)
	if (*p == '/' || *p == '}')
	    break;
    if (p < pthk) {
	*basep = pthk;
	*pathp = NULL;
    } else if (*++p != '\0') {
	strcpy(base, p);
	*basep = base;
	*p = '\0';
    } else {
	*basep = NULL;
    }
}
#endif
