#include <stdio.h>
#include "gsr.h"

#define DEBUG
#undef DEBUG

char *ylabel="", *xlabel="", *title="", *term=NULL;

/* externs needed for getopt */
extern char *optarg;
extern int optind, opterr;  /* 0 opterr: print err message and return "?" */

/*
   Do bar plots:
      bar [-x xlabel] [-y ylabel] [-t title] [-T term] [file [file2 ...]]
*/

main(argc, argv)
int argc;
char *argv[];
{
   FILE *in, *fopen();
   int c;

/* Check options. */
   while ((c = getopt (argc, argv, "x:y:t:T:")) != EOF) {
      switch (c) {
      case 'x': 
         xlabel = optarg;
      break;
      case 'y': 
         ylabel = optarg;
      break;
      case 't': 
         title = optarg;
      break;
      case 'T': 
         term = optarg;
      break;
      case '?': 
      default:
         usage();
      }
   }
/* Find out what terminal we're on to compute the max number of points. */
   gt_init_terminal();
   if (term != NULL) 
      gt_change_term (term, -1);
   if (GTterm == 0) {
      fprintf (stderr, "Unknown terminal\n");
      exit(1);
   }

   if (argc <= optind) {
      bar_graph(stdin);
      if (isatty(fileno(stdout)) && isatty(fileno(stdin)))
         getchar();
   }
   else {
      for (; optind < argc; ++optind) {
         if ((in = fopen (argv[optind], "r")) == NULL) {
            fprintf (stderr, "Can't open %s\n", argv[optind]);
            usage();
         }
         bar_graph(in);
         if (isatty(fileno(stdout)) && isatty(fileno(stdin)))
            getchar();
      }
   }
   pause();
   gsr_reset_terminal();
}


bar_graph(in)
FILE *in;
/* 
   Routine to make a bar graph out of a list of points.  The list of
   points come from the file named on the command line.
 
   Parameter:

          in:  Pointer to file descriptor containing list of points.
*/
{
#define MAXBUF 255
   struct termentry *t = &GTterm_tbl[GTterm];
   int *pnum, bar_w, x_loc, height, lcnt, max_points;
   int err, i, k, result, bar_cnt=0;
   float *pval, scale, pmax, pmin, dummy;
   char buf[MAXBUF];

   max_points = (t->xmax - 4*t->h_char)/2;

/* Read everything into an array of floats with room for 10 per line. */
   pval = (float *)malloc (10*sizeof(float)*max_points);
   pnum = (int *)malloc (sizeof(int )*max_points);
   pmin = VERYLARGE;
   pmax = -VERYLARGE;
   err = 0;
   for (lcnt=0; fgets (buf, MAXBUF, in) != NULL; ++lcnt) {
      result = sscanf(buf,"%f %f %f %f %f %f %f %f %f %f %f", 
        &pval[lcnt*10], &pval[lcnt*10+1], &pval[lcnt*10+2], &pval[lcnt*10+3], 
        &pval[lcnt*10+4], &pval[lcnt*10+5], &pval[lcnt*10+6], &pval[lcnt*10+7], 
        &pval[lcnt*10+8], &pval[lcnt*10+9], &dummy);
      if (result > 10) {
         err = 1;
         break;
      }
   /* Find the max and min of the overall array. */
      for (k = 0; k < result; ++k) {
         if (pval[lcnt*10+k] > pmax) pmax = pval[lcnt*10+k];
         if (pval[lcnt*10+k] < pmin) pmin = pval[lcnt*10+k];
      }
      pnum[lcnt] = result;  /* number of items on this line. */
      bar_cnt += result;
      if (bar_cnt > max_points) {
         err = 1;
         break;
      }
   }
   if (bar_cnt >= max_points) {
      fprintf (stderr, "Error on input: more than %d data elements\n", bar_cnt);
      exit(1);
   }
   if (err) {
      fprintf (stderr, "Error on input: line %d\n", lcnt);
      exit(1);
   }

/* Now that we know max and min, we can initialize the plotting device. */
   gsr_init (stdout, 1.0, 100.0, pmin < 0.0 ? pmin : 0.0, pmax, 1, 1, 0, 0);

/* Draw the bars represented by the array pval. */
   bar_plot (pval, pnum, lcnt, pmax, pmin);
}


usage()
{
   fprintf (stderr, 
"usage: bar [-x xlabel] [-y ylabel] [-t title] [file [file2 ...]]\n");
   fprintf (stderr, "List of terminals (on standard output):\n");
   gt_list_terms();
   exit(1);
}


bar_plot (pval, pnum, lcnt, pmax, pmin)
float pval[], pmax, pmin;
int pnum[], lcnt;
/*
   Routine to draw the bars represented by values in array pval.

   Parameters:

         pval: array of groups of points to take as bar heights.

         pnum: array of the number of points in the ith bar group.

         lcnt: number of groups (10 at most per group) of bars to draw.

         pcnt: number of points to draw. 

         pmax: largest value in the pval array.

         pmin: smallest value in the pval array.
*/
{
   int i, j, k, bar_cnt, zero, bar_h, bar_w, x_loc;
   struct termentry *t = &GTterm_tbl[GTterm];

/* Compute the width of the bar graph. */
   for (bar_cnt=i=0; i < lcnt; ++i)
      bar_cnt += pnum[i]+1;

   bar_w = (GSRxright - GSRxleft)/bar_cnt;

/* Compute start of first bar. */
   x_loc = (GSRxright - (bar_cnt-1)*bar_w)/2.0 + GSRxleft/2.0 + 0.5;

   zero = gsr_map_y (0.0);

#ifdef DEBUG
   for (i=0; i < lcnt; ++i) {
      fprintf (stderr, "%d: pval[%d] = ", pnum[i], i*10);
      for (k=0; k < pnum[i]; ++k) {
         fprintf (stderr, "%f ", pval[i*10+k]);
      }
      fprintf (stderr,"\n");
   }
   fprintf (stderr, "lcnt = %d\t", lcnt);
   fprintf (stderr, "pmax = %f\t", pmax);
   fprintf (stderr, "pmin = %f\n", pmin);
   fprintf (stderr, "zero = %d\t", zero);
   fprintf (stderr, "bar_cnt = %d\n", bar_cnt);
   fprintf (stderr, "bar_w = %d\n", bar_w);
   fprintf (stderr, "Hit return\n");
   (void )getchar();
#endif

/* Put device into graphics mode. */
   gsr_graphics();

/* Draw a box around the plot. */
   gsr_boundary();

/* Draw tics for the y axis of the plot. */
   gsr_draw_tics (0, 0, 0, 1, 1, 1);

/* Write the ylabel, xlabel, and title to the plot. */
   gsr_ylabel (ylabel);
   gsr_xlabel (xlabel);
   gsr_title  (title);

/* Draw the zero line (we might want a different line type here). */
   if (pmin < 0.0)
      gsr_draw_axis (0.0, 0.0, 2);  /* Just y axis */

/* Put the linetype back to something we expect. */
   gsr_line_point_type (0, 0);

/* Draw lcnt bars of height pval[i] and width bar_w. */
   for (i = 0; i < lcnt; ++i) {
      for (k = 0; k < pnum[i]; ++k) {
         bar_h = gsr_map_y (pval[i*10+k]);

#ifdef DEBUG
         fprintf (stderr, "bar_h = %d, linetype = %d\n", bar_h, k);
#endif
   
      /* Draw the bar's rectangle. */
         (*t->move)(x_loc,zero);
         (*t->vector)(x_loc, bar_h);
         (*t->vector)(x_loc+bar_w, bar_h);
         (*t->vector)(x_loc+bar_w, zero);

      /* Shade in the rectangle just drawn. */
         if (k % 6 < 4) {
            for (j=0; j < bar_w; ++j) {
            /* Terminal drivers define points as unsigned int... */
               (*t->move)(x_loc+j,zero);
               if (j % (k % 4 + 1) == 0)
                  (*t->vector)(x_loc+j,bar_h);
            }
         }
         else {
            int lower, upper;
            if (zero < bar_h) {
               lower = zero;
               upper = bar_h;
            }
            else {
               lower = bar_h;
               upper = zero;
            }
            for (j=lower; j < upper; ++j) {
               if (j % (k % 3 + 1) == 0) {
                  (*t->move)(x_loc,j);
                  (*t->vector)(x_loc+bar_w,j);
               }
            }
         }
         x_loc += bar_w;
      }
      x_loc += bar_w;
   }
/* Back into text mode (to see the plot generated) */
   gsr_text();
}



