;;; footnote-kana.el --- more Japanese footnote support

;; Copyright (C) 1997 by Stephen J Turnbull
;; It is my intent to assign the copyright to the FSF but the papers 
;; haven't been signed yet ;-)

;; Author: Stephen J Turnbull <turnbull@sk.tsukuba.ac.jp>
;; Keywords: mail, news, mule
;; Version: 0.17

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:


;; $BCm0U!'JQ$JF|K\8l$,$"$j$^$9!*=u$1$F$/$@$5$$!#(B

;; $B;2>H?t$(J}$rBX$($k(B
;; $B%$%m%O=g$^$?$O8^==2;=g$GJ?2>L>$^$?$OJR2>L>$N;X?t$rDs6!$$$?$7$^$9!#(B

;; I diffidently suggest that the functions
;; `footnote-kana-circular-index-string' (deprecated) and
;; `footnote-kana-rep-index-string' and the variable
;; `footnote-kana-index-function' be moved to footnote.el, the prefix be
;; changed from `footenote-kana-' to `footnote-', and that
;; footnote-greek.el and so on be modified to use the `rep' function.  The
;; names are probably poorly chosen; I'm not at all wedded to them and
;; didn't think too hard since they were originally only for use in
;; footnote-kana.el.

;;; Change Log:

;;; Code:

(require 'footnote)

(defun Footnote-kana-circular-index-string (n index-string)
  "Index footnotes using characters in order specified by INDEX-STRING.
If the number of footnotes exceeds the length of the string, wrap around."
  (char-to-string (aref index-string (1- (mod n (length index-string))))))

;; Adapted (slightly) from footnote-greek.el.
(defun Footnote-kana-rep-index-string (n index-string)
  "Index footnotes using characters in order specified by INDEX-STRING.
If the number of footnotes exceeds the length of the string, wrap around.
Wrapping around the alphabet implies successive repetitions of characters."
  (let* ((ltr (mod (1- n) (length index-string)))
	 (rep (/ (1- n) (length index-string)))
	 (chr (char-to-string (aref index-string ltr)))
	 rc)
    (while (>= rep 0)
      (setq rc (concat rc chr))
      (setq rep (1- rep)))
    rc))

(defvar footnote-kana-index-function
  'Footnote-kana-rep-index-string
  "Function for indexing into an array of index characters.")

(defconst footnote-katakana-iroha
  "$B%$%m%O%K%[%X%H%A%j%L%k%r%o%+%h%?%l%=%D%M%J%i%`%&%p%N%*%/%d%^%1%U%3%(%F%"%5%-%f%a%_%7%q%R%b%;%9(B"
  "$B%+%?%+%J!"%$%m%O=g(B")

(defconst footnote-katakana-iroha-regexp
  ;; can't use a range of JIS here; that would include "small" kana
  ;; not the same as 50-sounds (I think); iroha includes "old" kana
  ;; same comments apply to the remaining 3 regexps
  (concat "[" footnote-katakana-iroha "]"))

(defun Footnote-iroha-katakana (n)
  "($BF|K\8l(B)$B%$%m%O=g$GJR2>L>$G;2>H$9$k$3$H!#(B

Use Japanese katakana for indexing, in `iroha' order.  This is the
`normal' case because katakana are commonly used to catch the eye (eg,
to indicate emphasis in quoted conversation), and because `iroha' is a
peculiarly Japanese ordering.  It refers to a song whose lyrics use
all of the `50 sounds' (don't bother counting them, there aren't 50
there---don't ask _me_ why).  This song used to be memorized by
schoolchildren, and is still practiced by brush-writing students 
($B=qF;(B).  This is also why the Japanese input method `Canna' used to be
called `Iroha' (and still uses that name for its Unix-domain sockets).

Modern Japanese enjoy using this sytem, but most don't remember past
\"i ro ha\", and so must use either the `50 sounds' order or Western
style Latin characters or Arabic numerals for long lists."
  (funcall footnote-kana-index-function n footnote-katakana-iroha))

(defconst footnote-hiragana-iroha
  ;; this is the `marked' case; katakana are more common in this usage
  "$B$$$m$O$K$[$X$H$A$j$L$k$r$o$+$h$?$l$=$D$M$J$i$`$&$p$N$*$/$d$^$1$U$3$($F$"$5$-$f$a$_$7$q$R$b$;$9(B"
  "$B$R$i$,$J!"%$%m%O=g(B")

(defconst footnote-hiragana-iroha-regexp
  (concat "[" footnote-hiragana-iroha "]"))

(defun Footnote-iroha-hiragana (n)
  "($BF|K\8l(B)$B%$%m%O=g$GJ?2>L>$G;2>H$9$k$3$H!#(B

Use Japanese hiragana for indexing, in `iroha' order."
  (funcall footnote-kana-index-function n footnote-hiragana-iroha))

(defconst footnote-katakana-50-sounds
  "$B%"%$%&%(%*%+%-%/%1%3%5%7%9%;%=%?%A%D%F%H%J%K%L%M%N%O%R%U%X%[%^%_%`%a%b%d%f%h%i%j%k%l%m%o%r%s(B"
  "$B%+%?%+%J!"8^==2;=g(B")

(defconst footnote-katakana-50-sounds-regexp
  (concat "[" footnote-katakana-50-sounds "]"))

(defun Footnote-50-sounds-katakana (n)
  "($BF|K\8l(B)$B8^==2;=g$GJR2>L>$G;2>H$9$k$3$H!#(B

Use Japanese katakana for indexing, in `50 sounds' (natural) order.

This really is the natural order in a linguistic sense; both the sequence
of vowel sounds and that of consonant sounds follow a natural phonetic
ordering corresponding to the position of the tongue in the mouth."
  (funcall footnote-kana-index-function n footnote-katakana-50-sounds))

(defconst footnote-hiragana-50-sounds
  "$B$"$$$&$($*$+$-$/$1$3$5$7$9$;$=$?$A$D$F$H$J$K$L$M$s$*$O$R$U$X$[$^$_$`$a$b$d$f$h$i$j$k$l$m$o$r$s(B"
  "$B$R$i$,$J!"8^==2;=g(B")

(defconst footnote-hiragana-50-sounds-regexp
  (concat "[" footnote-hiragana-50-sounds "]"))

(defun Footnote-50-sounds-hiragana (n)
  "($BF|K\8l(B)$B8^==2;=g$GJ?2>L>$G;2>H$9$k$3$H!#(B

Use Japanese hiragana for indexing, in `50 sounds' order."
  (funcall footnote-kana-index-function n footnote-hiragana-50-sounds))

(setq footnote-style-alist
      (append (list `(iroha
		      Footnote-iroha-katakana
		      ,footnote-katakana-iroha-regexp))
	      footnote-style-alist))

(setq footnote-style-alist
      (append (list `(iroha-hiragana
		      Footnote-iroha-hiragana
		      ,footnote-hiragana-iroha-regexp))
	      footnote-style-alist))

(setq footnote-style-alist
      (append (list `(50-sounds
		      Footnote-50-sounds-katakana
		      ,footnote-katakana-50-sounds-regexp))
	      footnote-style-alist))

(setq footnote-style-alist
      (append (list `(50-sounds-hiragana
		      Footnote-50-sounds-hiragana
		      ,footnote-hiragana-50-sounds-regexp))
	      footnote-style-alist))

(provide 'footnote-kana)

;;; footnote-kana.el ends here
