/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
*/

/*
 *  This file contains all of the graph drawing routines.  It was
 *  seperated from xdraw.c on 2/3/92 by Mark Allender.  This is an
 *  attempt to isolate related code from unrelated code.  The ultimate
 *  goal of this is to remove xdraw.c completely from existence and
 *  replace it with several smaller modules.
*/
#include "config.h"
#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "graph.h"
#include "elision.h"
#include "extern.h"
#include "edge.h"

#define PI 3.14159276

extern int InBox();
extern void draw_arc();

/*
 *  ComputeMidpoint takes a list of nodes (in NodeList form), and computes
 *  the "middle" point that lies in, roughly, the middle of all the
 *  node in the list.  We currently just start at the first node, and
 *  take the midpoint of it, and the next node in the list.  We then
 *  take the midpoint of that midpoint and the next node in the list, etc..
*/
void
ComputeMidpoint(list, x, y)
     NodeList *list;
     coord_t *x, *y;
{
  NodeList *entry;
  coord_t x1, y1;

  entry = list;
  *x = entry->node->x;
  *y = entry->node->y;
  for (entry = entry->next ; entry; entry = entry->next) {
    x1 = entry->node->x;
    y1 = entry->node->y;
    *x = (*x + x1) / 2.0;
    *y = (*y + y1) / 2.0;
  }
}

void
RetTypeInfo (type, flag, color, font)
  char *type;
  short flag;
  unsigned long *color; 
  XFontStruct **font;
{
  struct DisplayTable *typeinfo;
 
  typeinfo = FindType(type, flag);
/*
 * See if we have this type defined.  If not, we will use the default
 * foreground color.  Likewise, we get the font information and revert to
 * the graph->font if it doesn't exist.
*/
  if (typeinfo)
  {
    *color = typeinfo->color;
    *font = typeinfo->font;
  }
  else
  {
    *color = graph->foreground;
    *font = graph->font;
  }
}

void
SetColorFont(view_dpy, virt_dpy, gc, color, font)
  Display *view_dpy, *virt_dpy;
  GC *gc;
  unsigned long color;
  XFontStruct *font;
{
  if (view_dpy) {
    XSetForeground (view_dpy, *gc, color);
    XSetFont(view_dpy, *gc, font->fid);
  }
  if (virt_dpy) {
    XSetForeground (virt_dpy, *gc, color);
    XSetFont(virt_dpy, *gc, font->fid);
  }
}
  
/*
 *  SetDisplayInfo takes a gc, a type, and two display tables as it's params
 *  and tries to find a display table entry for these.  The two tables that
 *  are passed in should be the graph node or edge display table, and the
 *  browser wide node/edge display table depending on what is being checked.
 *  This routine will automatically check the common tables for the browser
 *  and the graph.
*/
void
SetDisplayInfo(view_dpy, virt_dpy, gc, type, flag)
     Display *view_dpy, *virt_dpy;
     GC *gc;
     char *type;
     short flag;
{
  XFontStruct *font;
  struct DisplayTable *typeinfo;
  unsigned long color, foreground;
  Arg args[1];

  RetTypeInfo(type, flag, &color, &font);

/*
 *  Now, set the properties.
*/
  SetColorFont(view_dpy, virt_dpy, gc, color, font);
}

void
DrawSimpleEdge(edge)
  Edge *edge;
{
  int ArrowMask;
  Vertex *node1, *node2, *tmp;
  /* Ok, we know that this edge attaches 2 nodes.  Now the problem is that
   * we always want to draw the arc from one of these nodes to the other
   * of these nodes.  The easiest way to do this is to compare the
   * two node names and always draw from the node that is lexicographically
   * less than the other.
   */
  node1 = edge->edge_nodes->node;
  node2 = edge->edge_nodes->next->node;
  if (strcmp(node1->values->i_name, node2->values->i_name) > 0)
  {
    tmp = node1;
    node1 = node2;
    node2 = tmp;
  }    
  ArrowMask = 0;		/* No arrows yet */
  if (edge->directed)
  {
    if (EdgeConnects(edge, node1, TO_NODE)) ArrowMask |= INARROW1;
    if (EdgeConnects(edge, node2, TO_NODE)) ArrowMask |= INARROW2;
  }
  draw_arc(node1->x, node1->y,
	   MINX(node1), MINY(node1), MAXX(node1), MAXY(node1),
	   node2->x, node2->y,
	   MINX(node2), MINY(node2), MAXX(node2), MAXY(node2),
	   edge->offset, ArrowMask);
}

void
DrawHyperEdge(edge)
  Edge *edge;
{
  int ArrowMask;
  Vertex *node;
  NodeList *nodes;
  coord_t mid_x, mid_y;
  nodes = edge->edge_nodes;
  ComputeMidpoint (nodes, &mid_x, &mid_y);
  while (nodes)
  {
    ArrowMask = 0;
    node = nodes->node;
    if (edge->directed)
    {
      if (EdgeConnects(edge, node, TO_NODE)) ArrowMask |= INARROW1;
      if (EdgeConnects(edge, node, FROM_NODE)) ArrowMask |= OUTARROW1;
    }
    draw_arc(node->x, node->y,
	     MINX(node), MINY(node), MAXX(node), MAXY(node),
	     mid_x, mid_y,
             0.0, 0.0, -1.0, 0.0,
	     edge->offset, ArrowMask);
    nodes = nodes->next;
  }    
}  

void
DrawEdge(edge)
  Edge *edge;
{
  if (EdgeHidden(edge))
      return;
  /* We should never see a count < 2 since RecomputeEdgePos will take care
   * of that case.
   */
  SetDisplayInfo(XtDisplay(graph->viewing_area.widget), NULL,
		 &(graph->graph_gc), edge->values->type,
		 EDGEMASK);
  if (edge->node_count == 2)
      DrawSimpleEdge(edge);
  else
      DrawHyperEdge(edge);
}

void
DrawAllEdges()
{
  Edge *edge;

  for (edge = graph->edges; edge != NULL; edge = edge->next) {
    DrawEdge(edge);
  }
}

/* draw_node_box was written by Doug Bogia on 08/07/91.
 * It is designed to draw a box the size of the node text at the
 * position specified by x and y.
 */
void
draw_node_box (node, gc, font)
  Vertex *node;
  GC gc;
  XFontStruct *font;
{
  XCharStruct size;
  int tmp;
  char *str;
  int xmin, xmax, ymin, ymax;
  XPoint points[5];

  Display *dpy;
  Window win;

  dpy = XtDisplay(graph->viewing_area.widget);
  win = XtWindow(graph->viewing_area.widget);
  str = node->values->labels[node->label_index];
  xmin = MINX(node) * graph->viewing_area.width;
  xmax = MAXX(node) * graph->viewing_area.width;
  ymin = MINY(node) * graph->viewing_area.height;
  ymax = MAXY(node) * graph->viewing_area.height;
  
  points[0].x = xmin;
  points[0].y = ymax;
  points[1].x = xmin;
  points[1].y = ymin;
  points[2].x = xmax;
  points[2].y = ymin;
  points[3].x = xmax;
  points[3].y = ymax;
  points[4].x = xmin;
  points[4].y = ymax;
  XDrawLines(dpy, win, gc, points, 5, CoordModeOrigin);
  /* Don't use XDrawImageString here or you will get funny colors and
   * the XOR doesn't work.
   */
  XDrawString(dpy, win, gc,
	      xmin + 1, ymax - 1 - font->descent,
	      str, strlen(str));
}
  
void
DrawRubberBandBox(gc, xmin, ymin, xlen, ylen)
     GC gc;
     int xmin, ymin, xlen, ylen;
{
  Display *dpy;
  Window win;

  dpy = XtDisplay(graph->virtual_area.widget);
  win = XtWindow(graph->virtual_area.widget);

  xlen--;
  ylen--;
  if (xlen < 0) {
    xlen = -xlen;
    xmin -= xlen;
  }
  if (ylen < 0) {
    ylen = -ylen;
    ymin -= ylen;
  }
  
  XDrawRectangle(dpy, win, gc, xmin, ymin, xlen, ylen);
}

/*
 *  DrawAllNodes draw all the nodes according to the flag
 *  The flag will tell us either to draw the viewing area nodes,
 *  the virtual area nodes, or both.
*/
void
DrawAllNodes(flag)
     int flag;
{
  Vertex *node;
  Display *view_dpy, *virt_dpy;
  Window view_win, virt_win;
  unsigned long color;
  XFontStruct *font;
  int width;
  short x, y, minx, miny, maxx, maxy;
  char *str;
     
  view_dpy = XtDisplay(graph->viewing_area.widget);
  view_win = XtWindow(graph->viewing_area.widget);
  virt_dpy = XtDisplay(graph->virtual_area.widget);
  virt_win = XtWindow(graph->virtual_area.widget);

  if ((flag & (VIRT_NODES | VIEW_NODES))) {
    for (node = graph->v; node != NULL; node = node->next) {
      if (!NodeHidden(node))
      {
/*
 *  Let's extract some neccessary values to avoid redundant computation.
 *  Could use SetDisplayInfo, but we want to know the font.
*/
	RetTypeInfo(node->values->type, NODEMASK, &color, &font);
	SetColorFont(view_dpy, virt_dpy, &(graph->text_gc), color, font);
	str = node->values->labels[node->label_index];
	minx = INTX_VIEWX(MINX(node));
	miny = INTY_VIEWY(MINY(node));
	maxx = INTX_VIEWX(MAXX(node));
	maxy = INTY_VIEWY(MAXY(node));
	
	x = minx + 1;
	y = maxy - 1 - font->descent;
	
/*
 *  Draw the node with possible underline for supernode
*/
	if (flag & VIEW_NODES) 
	{
	  XDrawImageString(view_dpy, view_win, graph->text_gc, x, y, 
			   str, strlen(str));
	  if (NodeSelected(node))
	      XDrawLine(view_dpy, view_win, graph->text_gc,
			minx, miny, minx, maxy);
	  if (NodeSuper(node)) 
	      XDrawLine(view_dpy, view_win, graph->text_gc,
			minx, maxy, maxx, maxy);
	  if (NodeFixed(node))
	      XDrawLine(view_dpy, view_win, graph->text_gc,
			maxx, miny, maxx, maxy);
	}
	if (flag & VIRT_NODES)
	    XDrawPoint(virt_dpy, virt_win, graph->text_gc,
		       (short)(node->pos[0] * graph->virtual_area.width),
		       (short)(node->pos[1] * graph->virtual_area.height));
      }
    }
  }
}


/*
 *  This routine, DrawGraph, will draw all of the viewing are virtual
 *  area as specifed by the flag.  The flag is given as ORed together
 *  values to specify what to clear, and draw.  If clear is TRUE,
 *  the graph areas will be cleared before doing any drawing; otherwise,
 *  the graph will simply be redrawn.  The flag "flag" is passed
 *  to the DrawNodes routine to draw the node with the same set of
 *  flags.
*/
void
DrawGraph(flag, clear)
     int flag, clear;
{
  Display *virt_dpy, *view_dpy;
  Window virt_win, view_win;
  int width, height, xmin, ymin, xlen, ylen;

/*
 *  If we are drawing the viewing area, then get the display
 *  variables, and clear the area.  We can also draw the edges
 *  at this point.
*/
  if (flag & VIEW_NODES) {
    view_dpy = XtDisplay(graph->viewing_area.widget);
    view_win = XtWindow(graph->viewing_area.widget);
    if (clear)
	XClearWindow(view_dpy, view_win);
    DrawAllEdges();
  }
/*
 *  Likewise for the virtual area
*/
  if (flag & VIRT_NODES) {
    virt_dpy = XtDisplay(graph->virtual_area.widget);
    virt_win = XtWindow(graph->virtual_area.widget);
    if (clear)
	XClearWindow(virt_dpy, virt_win);
/*
 *  Get the width and height of virtual area to determine the
 *  box size, and then draw the box.
*/
    width = graph->virtual_area.width;
    height = graph->virtual_area.height;
    xmin = graph->virtx_min * width;
    xlen = (graph->virtx_max * width) - xmin;
    ymin = graph->virty_min * height;
    ylen = (graph->virty_max * height) - ymin;
    DrawRubberBandBox(graph->virt_xor_gc, xmin, ymin, xlen, ylen);
  }
  DrawAllNodes(flag);
  if (flag & VIEW_NODES)
    XFlush(view_dpy);
  if (flag & VIRT_NODES)
    XFlush(virt_dpy);
}

/*
   calc_angle was created by Doug Bogia on Aug. 26, 1991
   Description:
     This is meant to calculate the line angle from x1, y1 to x2, y2
   Entry:
     x1--------------> This is the x location of the first point
     y1--------------> This is the y location of the first point
     x2--------------> This is the x location of the second point
     y2--------------> This is the y location of the second point
     theta-----------> None.
   Exit:
     x1--------------> None.
     y1--------------> None.
     x2--------------> None.
     y2--------------> None.
     theta-----------> This will be the angle (in radians) of the slope of the
       line from x1, y1 to x2, y2
   Returns:
     1 if the x1, y1 and x2, y2 represent the same point
*/
int calc_angle (x1, y1, x2, y2, theta)
  coord_t            x1;
  coord_t            y1;
  coord_t            x2;
  coord_t            y2;
  coord_t *          theta;
{
  /* Start of variables section */
  coord_t            xdiff;	/* This is the difference in x values */
  coord_t            ydiff;	/* This is the difference in the y values */
  /* End of variables section */
  *theta = 0;
  if (x1 == x2 && y1 == y2)
  {
    return 1;
  }
  xdiff = x2 - x1;
  ydiff = y2 - y1;
  if (xdiff == 0.0) 
  {
    /* if xdiff is 0 then the ydiff/xdiff would fail.  Instead we know that
     * the angle is either pi/2 or 3pi/2.  We can figure that by looking
     * at ydiff.
     */
    *theta = (ydiff > 0.0) ? PI / 2.0 : (3.0 * PI) / 2.0;
  }
  else
  {
    *theta = atan(ydiff / xdiff);
    /* atan returns theta from -pi/2 to pi/2.  We want something that runs
     * from 0 to 2pi.  If x1, y1 is to the left of x2, y2, then we know
     * that theta is in the 2nd or 3rd quadrants.  We can just add pi to
     * the value returned and that will shift us from the 4th quadrant to
     * the 2nd or the 1st quadrant to the 3rd, which is what we want.
     */
    if (x2 > x1)
    {
      *theta += PI;
    }
  }    
  return (0);
} /* End of calc_angle */




