/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 *	This file is distributed under license and is confidential
 *
 *	File title and purpose
 *	Author:  Thomas Fruchterman
 *		 John Jozwiak
 *		 Mark Allender (allender@cs.uiuc.edu)
 *               Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */
/*
     Nature -- a force-directed graph-drawing program
     author: Thomas Fruchterman
     copyright 1990 Thomas Fruchterman

     Converted to C,  Jan 1991   Mark Allender
*/
#ifndef GRAPHDRAW_H
#define GRAPHDRAW_H

/* 
   this file shows how haphazard the translation to C++ has been
   there is good reason while Adj is still a struct, 
   and Vertex is equvivalent to a struct, since it has no private
   members but fixing
   such things is really on the back burner

   the advertisemnt says that i am drawing undirected graphs,
   but i actually am drawing directed ones. Giving the
   edge only one way in the input file cause disaters
   I also dont check for multiple edges or self loops

   both the vertex and graph classes are defined here
*/

#include "config.h"
#include <X11/X.h>			/* for mask definitions */
#include <X11/Xlib.h>			/* for GC definitions in graph */
#include <Xm/Xm.h>			/* for Widget definition in graph */

#define TRUE 1
#define FALSE 0

/*
 * If you change the W or H you should also be sure to change
 * LEFT, RIGHT, ... and LEFTSIDE, RIGHTSIDE, ...
 */
#define W 1.0
#define H 1.0

#define LEFT 0.05
#define RIGHT 0.95
#define TOP 0.05
#define BOTTOM 0.95

#define LEFTSIDE 0.0
#define RIGHTSIDE 1.0
#define TOPSIDE 0.0
#define BOTTOMSIDE 1.0

/* Masks for knowing if we want a node, an edge, a region, etc. */
#define NODEMASK 1
#define EDGEMASK 2
#define SWEPT_REGION 4
#define SWEPT_NODES 8
#define SELECTED_NODES 16

/* Masks for knowing the status of the node/edge. */
#define FIXED 1
#define SELECTED 2
#define SUPER 4
#define HIDDEN 8
#define MARKED 16
#define VISITED 32

#define MAX_LABELS 3

#define VIRT_NODES 1
#define VIEW_NODES 2

#define NEW(obj) (obj *)malloc(sizeof(obj))
#define ARRAY_SIZE(a) (sizeof(a) / (sizeof((a)[0])))

#define MINX(node) ((node)->x - (node)->label_x_offset)
#define MAXX(node) ((node)->x + (node)->label_x_offset)
#define MINY(node) ((node)->y - (node)->label_y_offset)
#define MAXY(node) ((node)->y + (node)->label_y_offset)

/* Macros to go from our internal x/y (0 to 1) coords to viewing x/y coords. */
#define INTX_VIEWX(x) (int)((x) * graph->viewing_area.width)
#define INTY_VIEWY(y) (int)((y) * graph->viewing_area.height)
/* Macro to go from view x/y coords to internal x/y coords. */
#define VIEWX_INTX(x) ((coord_t)(x) / (coord_t)graph->viewing_area.width)
#define VIEWY_INTY(y) ((coord_t)(y) / (coord_t)graph->viewing_area.height)
/* Macros to go from our internal x/y (0 to 1) coords to virtual x/y coords. */
#define INTX_VIRTX(x) (int)(((x) * (graph->virtx_max - graph->virtx_min)\
			    + graph->virtx_min) * graph->virtual_area.width)
#define INTY_VIRTY(y) (int)(((y) * (graph->virty_max - graph->virty_min)\
			    + graph->virty_min) * graph->virtual_area.height)

/*
 * Some defines for what to do after we update a graph.
 */
#define REDRAW 1
#define REDETERMINE 2
#define CHANGED_NODES 4
#define CHANGED_EDGES 8

typedef double coord_t;

/*
 *  the link structure is what binds all nodes and edge together.
 *  Look to documentation for further explanation of what the pointers
 *  mean.  However, node that ptr_type controls what type of structure
 *  that the p_adj_node, and p_edge_adj_node point to.  The lower order
 *  bits, if not set, point to another link structure.  If the lower order
 *  bits are set, then it points to either a Vertex * (in the case of
 *  the p_adj_node), or an Edge * (in the case of p_edge_adj_node).
*/
typedef struct link
{
  char ptr_type;
  struct link *n_adj_node, *p_adj_node, *n_edge_adj_node, *p_edge_adj_node;
  struct vertex *tnode, *fnode;
  struct edge *edge;
} Link;

/*
 *  Linked list of internal labels for edges.  These are kept off of
 *  nodes to help in deletion processes.
*/
typedef struct edge_id_list
{
  char *edge_id;
  struct edge_id_list *next;
} EdgeIdList;

/*
 *  A linked list of vertex pointers.
*/
typedef struct node_list
{
  struct vertex *node;
  struct node_list *next;
} NodeList;

/*
 *  A linked list of edge pointers
*/
typedef struct edge_list
{
  struct edge *edge;
  struct edge_list *next;
} EdgeList;

typedef struct keylist
{
  char *key;
  char *value;
  struct keylist *next;
} KeyList;

/*
 *  typedef structure for common values that are associated with both
 *  edges and nodes.
*/
typedef struct commonvals
{
  char *i_name, *type;
  char *labels[MAX_LABELS];	/* 0 for the label, 1 for short, 2 for long */
  unsigned long color;
  short status;			/* This is the status (selected, fixed,...)*/
  KeyList *key_list;
  struct vertex *super_node;
} CommonVals;

/*
 *  the node and edge definitions.
*/
typedef struct vertex
{
   CommonVals *values;
   Link *links;
   struct vertex *next;
   EdgeIdList *edge_id_list;
   coord_t pos[2], newpos[2];
   coord_t x,y;
   short label_index;		/* Holds printing index for this node */
   coord_t label_x_offset;	/* Holds the offset of the text from center */
   coord_t label_y_offset;	/* Holds the offset of the text from center */
} Vertex;

typedef struct edge
{
  CommonVals *values;
  struct edge *next, *super_edge;
  Link *links;
  NodeList *edge_nodes;
  short directed;
  short node_count;
  coord_t offset;
} Edge;

typedef struct {	     /* structure used to declare the different */
  Widget widget;	     /* areas that we will draw in since there are */
  int width, height;	     /* more than one. */
} warea;

typedef struct {
  Vertex *v;
  Edge *edges;
  char *title;
  char *active_type;
  int name_type, have_hyper_edge;
  Widget parent;
  warea viewing_area, virtual_area;
  coord_t virtx_min, virty_min, virtx_max, virty_max; 
  XFontStruct *font;		/* The font for this graph */
  unsigned long foreground, background;	/* Keep these colors of the graph */
  GC xor_gc;			/* GC for moving objects */
  GC text_gc;			/* GC for drawing the text in */
  GC graph_gc;			/* GC for drawing the lines, etc in */
  GC virt_xor_gc;		/* GC for the bounding view area */
  short label_policy;			/* how to label the supernode */
  int elision_policy;			/* how is elision going to be done */
  int unelision_policy;
  struct DisplayTables *display_table;	/* defs for type-color table */
  struct KeyBindings *callback_keys;
  struct FunctionBindings *callback_functions;
  struct SuperNodeList *super_nodes;
} Graph, *Graph_ptr;

/*-----------------------------------------------------------
We must also keep around a table of graphs, so that multiple
graphs can be supported.
*/

struct TableStruct {
  char *id;
  Graph *graph;
  int status;
  struct TableStruct *next;
};

struct TableStruct *GraphList;

/*
 *  We also want to keep around type-color tables.  We will maintain a
 *  type-color table for the browser level that will apply to all graphs.
 *  We will also keep around a table for each individual graph (if the
 *  user so desires).  We will be able to load these tables either through
 *  the message bus, or Xresources.  We will keep around the actual
 *  pixel value.  Only makes sense to avoid redundancy, and doing
 *  computations more than once.
*/

struct DisplayTable {
  short noclobber;
  char *type, *color_name, *font_name;
  unsigned long color;
  XFontStruct *font;
  struct DisplayTable *next;
};

struct DisplayTables {
  struct DisplayTable *common, *nodes, *edges;
};

struct DisplayTables *BrowserDisplays;

#endif



