/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* Main message bus program */
/* $Source: /import/kaplan/kaplan/carroll/cb/mbus/commands/RCS/mbus.c,v $ */

static char rcsid[] = "$Revision: 2.1.1.2 $ $Date: 91/11/15 13:37:59 $ $State: Exp $ $Author: carroll $";

/* ------------------------------------------------------------------------- */
#include "header.h"
#include "cb-defs.h"
#include "struct.h"
#include "getopt.h"
#include "extern.h"

#include <sys/signal.h>
/* ------------------------------------------------------------------------- */
/* Global variables */

struct Client *client;

int n_clients;				/* max files */
int mbus_port = DEF_MBUS_PORT;		/* default port */
int nohup;				/* ignore SIGHUP */

int MBLogLevel = 1;			/* amount of debugging output */

FILE *trace = NULL;			/* trace file */
/* ------------------------------------------------------------------------- */
void
warning(s) char *s;
{
  fprintf(stderr,"%s\n",s);
}

void
bomb(s) char *s;
{
  perror(s);
  exit(1);
}
/* ------------------------------------------------------------------------- */
void
ResetClient(c) struct Client *c;
{
  c->state = CLIENT_DEAD;		/* no client */
  c->loop = 0;				/* no loop back */
  MBfree(c->id);
  c->id = NULL;
  MBfree(c->domains);
  c->domains = NULL;
  MBChunkEmpty(&(c->input));
  MBChunkEmpty(&(c->output));
  MBResetParseState(&(c->parse));
}
/* ------------------------------------------------------------------------- */
void
Cleanup(sig) int sig;
{
  int i;

  if (accept_fd >= 0) close(accept_fd);

  for ( i = 0 ; i < n_clients ; ++i )
    if (CLIENT_ALIVE == client[i].state)
      close(client[i].fd);

  if (NULL != trace) fclose(trace);
  exit(1);
}
/* ------------------------------------------------------------------------- */

void
Initialize()
{
  int i;

#ifdef USE_GETDTABLESIZE
  n_clients = getdtablesize();
#else
#ifdef HPUX
  n_clients = 20;			/* how do you do this under HPUX? */
#else					/* SysV systems. */
  n_clients = ulimit(4,0);
#endif
#endif

  client = (struct Client *) malloc(n_clients * sizeof(struct Client));
  for ( i = 0 ; i < n_clients ; ++i )
    {
      client[i].state = CLIENT_DEAD;
      client[i].id = NULL;
      client[i].domains = NULL;
      MBChunkReset(&(client[i].input));
      MBChunkReset(&(client[i].output));
      MBInitializeParseState(&(client[i].parse));
    }

  InitializeIO();
  InitializeSocket();

  signal(SIGTERM, Cleanup);
  signal(SIGINT, Cleanup);
  if (nohup) signal(SIGHUP, SIG_IGN);
  else signal(SIGHUP,Cleanup);
}
/* ------------------------------------------------------------------------- */
void
PrintUsageMessage(argc,argv)
     int argc;
     char **argv;
{
  printf("Usage: %s -p # -l # -t file -n -N\n",argv[0]);
  printf("-p : Port number for the Mbus\n-l Debugging log level\n-t : Trace file\n-n : Ignore SIGHUP\n-N : Exit on SIGHUP\n");
}
/* ------------------------------------------------------------------------- */
int
DealWithCommandLine(argc,argv)
     int argc;
     char **argv;
{
  int opt;
  int tmp;
  char *s;
  extern char *getenv();
  char useage = 0;

  /* check environment variables */
  s = getenv(ENV_MBUS_PORT);
  if (NULL != s) mbus_port = strtol(s, NULL, 0);

  while ( (opt = getopt(argc,argv,"nNl;p:t:")) != EOF)
    switch (opt)
      {
      case '?' :
	fprintf(stderr,"%c\tBad option\n",opt_err_char);
	useage = 1;
	break;
      case 'l' :
	if (optarg != NULL) MBLogLevel = atoi(optarg);
	else MBLogLevel = 1;
	break;
      case 'n' : nohup = 1; break;
      case 'N' : nohup = 0; break;
      case 'p' :
	if (optarg != NULL) tmp = atoi(optarg);
	if (tmp > 0) mbus_port = tmp;
	break;
      case 't' :
	if (NULL != optarg)
	  {
	    if (!strcmp(optarg,"=")) trace = stdout;
	    else
	      {
		trace = fopen(optarg,"a");
		if (NULL == trace)
		  {
		    char buff[128];
		    if (strlen(optarg) > 100) optarg[100] = 0;
		    sprintf(buff,"Trace file %s",optarg);
		    perror(buff);
		  }
	      }
	  }
	break;
      }

  if (useage)
    {
      PrintUsageMessage(argc,argv);
      exit(1);
    }
  return optind;
}
/* ------------------------------------------------------------------------- */
void
DumpClient(c) struct Client *c;
{
  close(c->fd);
  ResetClient(c);
  if (MBLogLevel > 1) printf("Dumped client %d\n",c - client);
  if (NULL != trace)
    {
      fprintf(trace,"Dumped %d\n",c - client);
      fflush(trace);
    }
}
/* ------------------------------------------------------------------------- */
void
TraceSexp(f, e)
     t_sexp e;
     FILE *f;
{
  if (NULL != f)
    {
      t_sexp s = MBprint(e, NULL);
      fflush(f);
      MBwrite(fileno(f), s);
      MBfree(s);
    }
}
/* ------------------------------------------------------------------------- */
int
ClientAcceptsMessage(c,e)
     struct Client *c;			/* the client */
     struct mb_object *e;		/* The message expression */
{
  int accept = 0;			/* default, don't accept */
  struct mb_object *dl,*dg,*d;		/* domain list, domain group, domain */
  struct mb_object *md;			/* message domain */
  struct mb_object *tag;		/* tag */
  struct mb_object *mtag;		/* message tag */
  struct mb_object *tmp;
  struct mb_object mdomains;

  tmp = MBcar(MBcdr(e));		/* get the domain stuff */
  mdomains.type = MB_CONS;
  if (MBconsp(tmp))
    {
      MBsetcar(&mdomains,MBcar(tmp));
      MBsetcdr(&mdomains,MBcdr(tmp));
    }
  else
    {
      MBsetcar(&mdomains,tmp);
      MBsetcdr(&mdomains,NULL);
    }
  mtag = MBcar(e);			/* and the tag. */

  for ( dl = c->domains ; !accept && MBconsp(dl) ; dl = MBcdr(dl) )
    {
      dg = MBcar(dl);			/* get the next domain group */

      /* check the tag first, if not a match, skip this one */
      if (!MBequal(MBcar(dg), mtag)) continue;

      /* now, loop through the domains of this entry*/
      for ( dg = MBcdr(dg) ; !accept && MBconsp(dg) ; dg = MBcdr(dg) )
	{
	  /* loop through the message domains */
	  for ( md = &mdomains ; !accept && MBconsp(md) ; md = MBcdr(md) )
	    {
	      accept = MBequal(MBcar(dg),MBcar(md));
	    }
	}
    }
  return accept;
}
  
/* ------------------------------------------------------------------------- */
void
ClientAddDomain(c,exp)
     struct Client *c;
     struct mb_object *exp;
{
  struct mb_object *new, *set, *d;
  struct mb_object *safe;		/* for domain conversion */
  struct mb_chunk *chunk;		/* chunk holder */

  exp = MBcdr(exp);			/* skip command tag */

  if (!MBstringp(MBcar(exp))) return;	/* no tag, fail. */
  if (!MBcar(MBcdr(exp))) return;       /* no domain, return */

  new = MBGetCons();			/* this will hold the new tag/domain */
  set = MBGetCons();			/* current list extender */
  MBsetcar(set,new);			/* link them together */

  if (NULL == MBsetcar(new, MBRegexpCompile(MBcar(exp))))
    {
      MBfree(set);
      return;
    }

  /* two cases - either the domains are just the rest of the expression,
   * or they are a list as the next element
   */

  d = MBcdr(exp);			/* skip over tag */
  if (MBconsp(MBcar(d))) d = MBcar(d);

  for ( safe = MBGetName() , chunk = &(safe->object.chunk)
       ; MBconsp(d)
       ; d = MBcdr(d)
       )
    {
      struct mb_object *domain = MBcar(d);
      struct mb_object *re;		/* regexp holder */
      int pc,c;				/* prev char, char */

      if (!(domain->type == MB_NAME || domain->type == MB_STRING)) continue;

      MBChunkPutChar(chunk,'^');	/* anchor  */

      for ( pc = CHUNK_EMPTY
	   ; CHUNK_EMPTY != (c = MBChunkGetChar(&(domain->object.chunk)))
	   ; pc = c
	   )
	{
	  if ('.' == c)
	    {
	      if (CHUNK_EMPTY == MBChunkPeekChar(&(domain->object.chunk)))
		{
		  if ('.' == pc || CHUNK_EMPTY == pc)
		    MBChunkPutBuffer(chunk,"[^.]+(\\.[^.]+)*",15);
		  else
		    MBChunkPutBuffer(chunk,"(\\.[^.]+)*",10);
		}
	      else				/* not last character */
		{
		  if ('.' == pc)
		    MBChunkPutBuffer(chunk,"[^.]+\\.", 7);
		  else if (CHUNK_EMPTY == pc)
		    MBChunkPutBuffer(chunk,"([^.]+\\.)*",10);
		  else
		    MBChunkPutBuffer(chunk,"\\.",2);
		}
	    }
	  else if (MB_REGEXP_IS_META(c))
	    {
	      MBChunkPutChar(chunk,'\\');
	      MBChunkPutChar(chunk,c);
	    }
	  else
	    MBChunkPutChar(chunk,c);
	}
      MBChunkPutChar(chunk,'$');	/* anchor  */

      re = MBRegexpCompile(safe);
      MBChunkEmpty(chunk);
      if (NULL != re)
	{
	  struct mb_object *tmp = MBGetCons();
	  MBsetcar(tmp,re);
	  MBsetcdr(tmp,MBcdr(new));
	  MBsetcdr(new,tmp);
	}
    }

  MBfree(safe);
  MBsetcdr(set, c->domains);
  c->domains = set;
}
/* ------------------------------------------------------------------------- */
void
ClientPing(c,exp)
     struct Client *c;
     struct mb_object *exp;
{
  int i;				/* index */
  struct Client *dst;			/* current client */
  struct mb_object *m;			/* ping message */

  exp = MBcdr(exp);			/* drop command */

  m = MBGetName();
  MBput_Cstring(m,"\n(\"ping\" \"bus\"");

  if (MBconsp(exp))			/* if not consp, no point in looking */
    for ( i = 0 , dst = client ; i < n_clients ; ++i , ++dst )
      {
	if (dst == c || dst->state == CLIENT_DEAD) continue;
	if (ClientAcceptsMessage(dst,exp))
	  {
	    MBput_char(m,' ');
	    MBprint(dst->id, m);
	  }
      }

  MBput_Cstring(m,")\n");
  MBChunkPutChunk(&(c->output), &(m->object.chunk));
  MBfree(m);
}
  
/* ------------------------------------------------------------------------- */
void
SendInfo(c)
     struct Client *c;
{
  int i,count;
  struct mb_object *m;			/* ping message */
  char buff[128];

  m = MBGetName();
  MBput_Cstring(m,"\n(\"info\" \"bus\"\n  (version \"");
  MBput_Cstring(m,rcsid);
  MBput_Cstring(m,"\")\n  ");
  MBput_Cstring(m,MBDumpMemoryUsage());

  for ( count = i = 0 ; i < n_clients ; ++i )
    count += client[i].state == CLIENT_ALIVE;

  sprintf(buff,"\n  (clients %d)",count);
  MBput_Cstring(m,buff);

  MBput_Cstring(m,"\n)\n");
  MBChunkPutChunk(&(c->output), &(m->object.chunk));
  MBfree(m);
}
  
/* ------------------------------------------------------------------------- */
void
SendClientState(c) struct Client *c;
{
  int i,count;
  struct mb_object *m;			/* ping message */
  char buff[128];

  m = MBGetName();
  MBput_Cstring(m,"\n(\"clients\" \"bus\"");

  for ( count = i = 0 ; i < n_clients ; ++i )
    {
      if (CLIENT_ALIVE != client[i].state) continue;
      sprintf(buff,"\n  (client %d ",i);
      MBput_Cstring(m, buff);
      MBprint(client[i].id, m);
      if (client + i == c) MBput_Cstring(m,"\n    :self t");
      sprintf(buff,"\n    :state %s :depth %d :input %d :output %d\n  )",
	      0 <= client[i].parse.state
	      && client[i].parse.state < MB_N_PARSE_STATE_NAMES
	      ? MBparse_state_name[(int)(client[i].parse.state)] : ":bogus",
	      client[i].parse.depth,
	      client[i].input.count, client[i].output.count);
      MBput_Cstring(m,buff);
    }

  MBput_Cstring(m,"\n)\n");
  MBChunkPutChunk(&(c->output), &(m->object.chunk));
  MBfree(m);
}  
/* ------------------------------------------------------------------------- */
void
HandleClose(c,sexp)
     struct Client *c;
     struct mb_object *sexp;
{
  int n;
  char *tmp,buff[32];

  sexp = MB_CDR(sexp); sexp = MB_CAR(sexp);

  if (NULL == sexp)
    {
      if (NULL != trace) fprintf(trace,"Self-termination granted\n");
      DumpClient(c);
    }
  else if (MB_NAMEP(sexp))
    {
      buff[MBbuffer(sexp, buff, 31)] = 0;
      n = strtol(buff, &tmp, 0);
      if (tmp != buff && n >= 0 && n < n_clients
	&& CLIENT_ALIVE == client[n].state)
	{
	  if (NULL != trace)
	    {
	      if (client + n == c)
		fprintf(trace,"Indirect self-termination granted\n");
	      else
		fprintf(trace,"Terminated client %d\n",n);
	    }
	  DumpClient(client + n);
	}
      else if (NULL != trace)
	fprintf(trace,"Invalid close request ignored\n");
    }
  else if (NULL != trace)
    fprintf(trace,"Invalid close request ignored\n");
}
/* ------------------------------------------------------------------------- */
void
HandleClient(c) struct Client *c;
{
  int i;
  struct Client *dst;

  while (MBParseChunk(&(c->input), &(c->parse)))
    {
      struct mb_object *e = MBExtractSexp(&(c->parse));
      struct mb_object *m = MBGetName();

      if (NULL != trace)
	{
	  fprintf(trace,"Message from %d, id = ",c - client);
	  TraceSexp(trace, c->id);
	  putc('\n',trace);
	  TraceSexp(trace, e);
	  putc('\n',trace);
	}
      /* First, let's see if it's a command */
      if (e && MBconsp(e) && MBnamep(MBcar(e)))
	{
	  struct mb_object *comm = MBcar(e);

	  if (NULL != trace)
	    {
	      fprintf(trace,"Looks like a command called ");
	      TraceSexp(trace, comm);
	      fprintf(trace,"\n");
	    }

	  if (!MBcompare_Cstring(comm, "accept")) ClientAddDomain(c,e);
	  else if (!MBcompare_Cstring(comm, "clear"))
	    {
	      MBfree(c->domains);
	      c->domains = NULL;
	    }
	  else if (!MBcompare_Cstring(comm, "id"))
	    {
	      struct mb_object *tmp = MBnth(e, 1);
	      if (MBstringp(tmp))
		{
		  c->id = tmp;
		  MBsetcar(MBcdr(e),NULL);
		}
	    }
	  else if (!MBcompare_Cstring(comm, "ping")) ClientPing(c,e);
	  else if (!MBcompare_Cstring(comm, "close")) HandleClose(c,e);
	  else if (!MBcompare_Cstring(comm, "info")) SendInfo(c);
	  else if (!MBcompare_Cstring(comm, "clients")) SendClientState(c);
	  else if (!MBcompare_Cstring(comm, "loop-back"))
	    {
	      t_sexp arg = MBnth(e, 1);

	      /* loop if the arg is not NULL or "nil" */
	      c->loop = NULL != arg && MBcompare_Cstring(arg, "nil");
	    }
	  else
	    {
	      if (NULL != trace)
		fprintf(trace, "I'm sorry Dave, I can't do that\n");
	    }
	}
      else				/* not in command format */
	{
	  int was_received = 0;

	  if (NULL != trace) fprintf(trace,"Received by");

	  MBput_char(m, '\n');
	  MBprint(e, m);
	  MBput_char(m, '\n');
	  for ( i = 0 , dst = client ; i < n_clients ; ++i , ++dst )
	    {
	      /* Don't send things to clients that aren't there */
	      if (dst->state == CLIENT_DEAD) continue;
	      /* Don't send client's own message back unless it asked for it */
	      if (dst == c && !c->loop) continue;

	      /* Ok, see if client wants the message */
	      if (ClientAcceptsMessage(dst,e))
		{
		  MBChunkPutChunk(&(dst->output), &(m->object.chunk));
		  was_received = 1;
		  if (NULL != trace)
		    {
		      fprintf(trace," %d ", i);
		      TraceSexp(trace, dst->id);
		    }
		}
	    }
	  if (NULL != trace)
	    {
	      if (was_received) putc('\n',trace);
	      else fprintf(trace," nobody!\n");
	    }
	}
      MBfree(m);
      MBfree(e);
      if (NULL != trace) fflush(trace);
    }
}

/* ------------------------------------------------------------------------- */

main(argc,argv)
     int argc;
     char **argv;
{

  DealWithCommandLine(argc,argv);

  Initialize();

  while (1)
    {
      DoIO();
    }
}
