/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *	Modified: Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* widget server area code */
/* $Source: /export/kaplan/stable/sun4.os4.1/cb-2.0/src/data/node105.text,v $ */

static char rcsid[] = "area.c $Revision: 1.0.1.2 $ $Date: 92/05/08 16:29:34 $ $State: Exp $ $Author: CBmgr $";

/* ------------------------------------------------------------------------ */
#include "config.h"
#include "header.h"
#include <Xm/DrawingA.h>
#include <Xm/ScrolledW.h>
/* ------------------------------------------------------------------------ */
typedef struct area_widget_struct
{
  DECLARE_STANDARD_WIDGET_SLOTS;
  Window child;				/* reparented X window */
  char *message;			/* message header */
  Boolean resizing;			/* resizing flag */
  Dimension v_margin, h_margin;		/* vertical and horizontal margins */
  unsigned int child_border;		/* size of child's border */
  Widget w_area;
  Widget w_scrolled;
  Widget hscrollbar, vscrollbar; /* Holders for the scrollbars */
  int hvalue, vvalue;		/* Current horiz or vert. values */
} * t_area_widget;

#define AREA_MIN_WIDTH(a) ((a)->h_margin * 2 + 1)
#define AREA_MIN_HEIGHT(a) ((a)->v_margin * 2 + 1)
/* ------------------------------------------------------------------------ */
extern void
  AreaRemove(), AreaUpdate(), AreaChildDestroyed(),
  AreaRealized(), AreaTransmit(), AreaEditable();

extern t_generic_widget AreaHandle();

extern void UpdateScrollbar(), ScrollbarCallback();
extern Widget XmCreateScrollBar();

extern Atom xa_cb_hscrollbar, xa_cb_vscrollbar;

struct widget_class_struct AreaClassRecord =
{
  WS_AREA, "area",
  AreaHandle, AreaRemove, AreaChildDestroyed, AreaUpdate,
  GenericReply, AreaRealized, AreaTransmit, AreaEditable,
} ;

/* ------------------------------------------------------------------------ */
/* Bogus variables for passing to functions that need to return things */
static Window Bwindow;
static int Bx, By;
static unsigned int Bborder, Bdepth;
/* ------------------------------------------------------------------------ */
void
AreaDropChild(area, send_atom)
  t_area_widget area;
  Atom send_atom;
{
  Display *dpy = XtDisplay(area->w_area);

  if (area->child)
    {
      XUnmapWindow(dpy, area->child);
      XReparentWindow(dpy, area->child,
		      RootWindowOfScreen(XtScreen(area->w_area)), 0, 0);
      SendClientMessage(dpy, area->child, send_atom, 0, 0);

      /* Note that by setting this here, we prevent a resize when the
       * ReparentNotify comes through to AreaSubstructure().
       */
      area->child = 0;
    }
}
/* ------------------------------------------------------------------------ */
/* This is called when a widget resizes the drawing area (probably via
 * a window sash)
 * There are some tricky dependencies between this and the substructurenotify
 * handler, so watch out.
 */
void
AreaResizeCallback(w, area, data)
     Widget w;
     t_area_widget area;
     XmDrawingAreaCallbackStruct *data;
{
  unsigned int width, height;

  if (0 != area->child && ! area->resizing && XtWindow(w))
    {
      /* get the geometry of the drawing area first */
      if (XGetGeometry(XtDisplay(w), XtWindow(w), &Bwindow, &Bx, &By,
		       &width, &height, &Bborder, &Bdepth))
	{
	  /* Adjust for margins */
	  if ((width -= 2 * area->h_margin + 2 * area->child_border) < 1)
	    width = 1;
	  if ((height -= 2 * area->v_margin + 2 * area->child_border) < 1)
	    height = 1;
	  XResizeWindow(XtDisplay(w), area->child, width, height);
	  XMoveWindow(XtDisplay(w), area->child, area->h_margin, area->v_margin);
	  XSync(XtDisplay(w), False);		/* slow down a bit here */
	}
    }
}
/* ------------------------------------------------------------------------ */
/* We get this when something happens with a child window */
void
AreaSubstructure(w, area, event)
     Widget w;
     t_area_widget area;
     XEvent *event;
{
  Arg argl[2];

  switch (event->type)
    {
    case ReparentNotify:
      {
	unsigned int width, height;
	Window child = event->xreparent.window;

	if (event->xreparent.parent != XtWindow(area->w_area))
	  {
	    /* window was reparented somewhere else, remove it */
	    if (child == area->child)
	      {
		area->child = 0;
		XtSetArg(argl[0], XmNwidth, AREA_MIN_WIDTH(area));
		XtSetArg(argl[1], XmNheight, AREA_MIN_HEIGHT(area));
		/* no resizing flag setting here, since we've set the child to 0 */
		XtSetValues(area->w_area, argl, 2);
	      }
	  }
	else if (XGetGeometry(XtDisplay(w), child, &Bwindow, &Bx, &By,
			 &width, &height, &Bborder, &Bdepth))
	  {
	    if (child != area->child)	/* got a replacement */
	      {
		AreaDropChild(area, xa_cb_dropchild);	/* out with the old */
		area->child = child;	/* in with the new */
	      }
	    area->child_border = Bborder;
	    /* margin adjust */
	    width += 2 * area->h_margin + 2 * area->child_border;
	    height += 2 * area->v_margin + 2 * area->child_border;
	    XtSetArg(argl[0], XmNwidth, width);
	    XtSetArg(argl[1], XmNheight, height);
	    area->resizing = True;
	    XtSetValues(area->w_area, argl, 2);
	    area->resizing = False;
	    XMoveWindow(XtDisplay(w), child, area->h_margin, area->v_margin);
	    XMapWindow(XtDisplay(w), child);
	  }
      }
      break;

    case ConfigureNotify:
      if (event->xconfigure.window != area->child) break;
      area->child_border = event->xconfigure.border_width;
      XtSetArg(argl[0], XmNwidth,
	       event->xconfigure.width
	       + 2 * area->h_margin + 2 * area->child_border);
      XtSetArg(argl[1], XmNheight,
	       event->xconfigure.height
	       + 2 * area->v_margin + 2 * area->child_border);
      area->resizing = True;
      XtSetValues(area->w_area, argl, 2);
      area->resizing = False;
      break;

    case DestroyNotify:
      if (event->xdestroywindow.window == area->child)
	{
	  /* the child window is toast, clean up */
	  area->child = 0;
	  XtSetArg(argl[0], XmNwidth, AREA_MIN_WIDTH(area));
	  XtSetArg(argl[1], XmNheight, AREA_MIN_HEIGHT(area));
	  /* No resizing flag set since child is 0 now */
	  XtSetValues(area->w_area, argl, 2);
	  if (area->w_scrolled)
	      XtSetValues(area->w_scrolled, argl, 2);
	}
      break;
    }
}
/* ------------------------------------------------------------------------ */
/* For creation, the :child argument is only used to get a geometry, the
 * child is not actually reparented until an update is sent.
 */
static struct keyword_entry_struct AreaKeywords[] =
{
  { ":width", (void *)-1, KEYWORD_INT },
  { ":height", (void *)-1, KEYWORD_INT },
  { ":child", (void *)0, KEYWORD_INT },
  { ":message", NULL, KEYWORD_COOKED },
  { ":scrollbars", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
};

t_generic_widget
AreaHandle(parent,sexp)
     t_generic_widget parent;
     t_sexp sexp;
{
  t_area_widget area;
  Dimension width, height;
  int n;
  int scrollable;
  t_keyword_flag scrollbars;
  Arg argl[4];

  if (NULL == parent || NULL == sexp) return NULL;

  area = NEW_STRUCT(area_widget_struct);
  area->type = WS_AREA;
  area->class = &AreaClassRecord;
  area->parent = parent;
  area->top = parent->top;
  area->id = Getnth(sexp,1);
  area->resizing = False;
  area->child_border = 0;
  area->hvalue = 0;
  area->vvalue = 0;

  /* sexp should be (area "id" &key ...) */
  MBparse_keywords(MBnthcdr(sexp,2), AreaKeywords, ARRAY_SIZE(AreaKeywords));
  width = (Dimension) AreaKeywords[0].result;
  height = (Dimension) AreaKeywords[1].result;
  area->child = (Window) AreaKeywords[2].result;
  area->message = (char *) AreaKeywords[3].result;
  scrollable = KEYWORD_TRUE == (t_keyword_flag) AreaKeywords[4].result;
  
  if (area->child)
    {
      unsigned int dx, dy, border;
      if (XGetGeometry(XtDisplay(toplevel_widget), area->child,
		       &Bwindow, &Bx, &By, &dx, &dy, &border, &Bdepth))
	{
	  width = (Dimension)dx;
	  height = (Dimension)dy;
	  area->child_border = border;
	}
      else fprintf(stderr, "Bad Child 0x%x\n", area->child);
    }
  area->child = 0;			/* not going to reparent it here */

  if (scrollable)
  {
    area->w_scrolled = XmCreateScrolledWindow(parent->widget, "area",
					      NULL, 0);
  }
  else
      area->w_scrolled = NULL;
  n = 0;
  XtSetArg(argl[n], XmNuserData, (caddr_t)area), ++n;
  area->w_area = XtCreateWidget("area", xmDrawingAreaWidgetClass,
				area->w_scrolled ? area->w_scrolled
				: parent->widget, argl, n);

  XtSetArg(argl[0], XmNmarginHeight, &(area->v_margin));
  XtSetArg(argl[1], XmNmarginWidth, &(area->h_margin));
  XtGetValues(area->w_area, argl, 2);

  n = 0;
  /* if we didn't get numbers from :width, :height, or :child, make them up */
  if (width <= 0) width = 1;
  if (height <= 0) height = 1;
  XtSetArg(argl[n], XmNwidth, (Dimension)(width + 2 * area->v_margin)), ++n;
  XtSetArg(argl[n], XmNheight, (Dimension)(height + 2 * area->h_margin)), ++n;
  XtSetValues(area->w_area, argl, n);

  XtAddCallback(area->w_area, XmNresizeCallback, AreaResizeCallback, area);
  XtAddEventHandler(area->w_area, SubstructureNotifyMask, False,
		    AreaSubstructure, area);

  if (area->w_scrolled)
  {
    /* Create the scroll bars.  They won't show up until we manage them. */
    n = 0;
    XtSetArg(argl[n], XmNorientation, XmHORIZONTAL), ++n;
    area->hscrollbar = XmCreateScrollBar(area->w_scrolled, "hscroll", argl, n);
    XtAddCallback(area->hscrollbar, XmNvalueChangedCallback,
		  ScrollbarCallback, area);
    XtAddCallback(area->hscrollbar, XmNdragCallback,
		  ScrollbarCallback, area);
    area->vscrollbar = XmCreateScrollBar(area->w_scrolled, "vscroll", NULL, 0);
    XtAddCallback(area->vscrollbar, XmNvalueChangedCallback,
		  ScrollbarCallback, area);
    XtAddCallback(area->vscrollbar, XmNdragCallback,
		  ScrollbarCallback, area);
    n = 0;
    XtSetArg(argl[n], XmNhorizontalScrollBar, area->hscrollbar), ++n;
    XtSetArg(argl[n], XmNverticalScrollBar, area->vscrollbar), ++n;
    XtSetArg(argl[n], XmNworkWindow, area->w_area), ++n;
    XtSetValues(area->w_scrolled, argl, n);
    XtManageChild(area->w_area);
    area->widget = area->w_scrolled;
  }
  else
      area->widget = area->w_area;
  
  return (t_generic_widget) area;
}
/* ------------------------------------------------------------------------ */
/* Called by someone else when they want us to remove all of our data
 * structures. Destruction of the widgets is the caller's job.
 */
void
AreaRemove(self) t_area_widget self;
{
  Display *dpy = XtDisplay(self->w_area);

  MBfree(self->id);
  AreaDropChild(self, xa_cb_terminate);
  XtFree(self);
}
/* ------------------------------------------------------------------------ */
/* Called with an update sexp */
/* Accept an update message and update the Area */
static struct keyword_entry_struct AreaUpdateKeys[] =
{
  { ":width", (void*)-1, KEYWORD_INT },
  { ":height", (void*)-1, KEYWORD_INT },
  { ":child", 0, KEYWORD_INT },
  { ":hscrollbar", NULL, KEYWORD_SEXP },
  { ":vscrollbar", NULL, KEYWORD_SEXP },
};

void
AreaUpdate(self,sexp)
     t_area_widget self;
     t_sexp sexp;
{
  t_sexp hscroll = NULL, vscroll = NULL;
  int width, height;
  unsigned int dx, dy;
  Window child;
  int n = 0;
  Arg argl[2];
  Window win = XtWindow(self->w_area);
  Display *dpy = XtDisplay(self->w_area);

  if (!MBequal(MBnth(sexp, 1), self->id)) return;

  MBparse_keywords(MBnthcdr(sexp,2), AreaUpdateKeys, ARRAY_SIZE(AreaUpdateKeys));
  width = (int) AreaUpdateKeys[0].result;
  height = (int) AreaUpdateKeys[1].result;
  child = (Window) AreaUpdateKeys[2].result;
  if (self->w_scrolled)		/* Are we a scrolled widget? */
  {
    hscroll = (t_sexp) AreaUpdateKeys[3].result;
    vscroll = (t_sexp) AreaUpdateKeys[4].result;
  }
  if (!child) child = self->child;

  if (child)				/* either a new one, or we have one */
    {
      if (XGetGeometry(dpy, child, &Bwindow, &Bx, &By,
		       &dx, &dy, &Bborder, &Bdepth))
	{
	  if (width <= 0) width = dx;
	  if (height <= 0) height = dy;
	  self->child_border = Bborder;
	  /* if a width / height was specified, force the child widget
	   * to be that size
	   */
	  XResizeWindow(dpy, child, width, height);
	  XSync(dpy, False);		/* wait for things to happen */
	  /* if we've changed childs */
	  if (child != self->child)	/* a new child to reparent */
	    {
	      AreaDropChild(self, xa_cb_dropchild);
	      XReparentWindow(dpy, child, win, self->h_margin, self->v_margin);
	      /* And you may say to yourself, why isn't self->child set here?
	       * Because...it's all taken care of in AreaSubstructure()!
	       * The reparenting generates a ReparentNotify, and it's best
	       * if we wait until then to update our internal data.
	       */
	    }
	}
      else fprintf(stderr, "Unruly child 0x%x\n", child);
    }

  if (width > 0)
    XtSetArg(argl[n], XmNwidth,
	     width + 2 * self->h_margin + 2 * self->child_border), ++n;
  if (height > 0)
    XtSetArg(argl[n], XmNheight,
	     height + 2 * self->v_margin + 2 * self->child_border), ++n;

  if (n > 0)
    {
      self->resizing = True;
      XtSetValues(self->w_area, argl, n);
      self->resizing = False;
    }
  if (hscroll)
      UpdateScrollbar(self, self->hscrollbar, hscroll);
  if (vscroll)
      UpdateScrollbar(self, self->vscrollbar, vscroll);
    
}
/* ------------------------------------------------------------------------ */
/* Called by a child when it decides to destroy itself for whatever reason.
 * We should update our internal data to remove references to the child.
 * Assume that all of the childs internal and widgets stuff is taken care
 * of by the child.
 */
void
AreaChildDestroyed(self, child)
     t_area_widget self;
     t_generic_widget child;
{
}
/* ------------------------------------------------------------------------ */
/* Called after the widget has been realized, for final fiddling */
void
AreaRealized(self) t_area_widget self;
{
  char *id;
  char buff[64];

  if (NULL != self->message)
    {
      /* Wait as long as feasible to make sure that the window is really
       * there, even if we have the X window ID
       */
      XSync(XtDisplay(self->w_area), False);

      MBtransmit_Cstring(mbus, "(");
      MBtransmit_Cstring(mbus, self->message);
/*      id = MBprint_Cstring(self->id); */
      id = MBprint_Cstring(self->top->id);
      sprintf(buff, " (area-window %s \"0x%x\"))", id,
	      XtWindow(self->w_area));
      MBtransmit_Cstring(mbus, buff);
      XtFree(id);
    }
}
/* ------------------------------------------------------------------------ */
void
AreaEditable(self, flag)
     t_area_widget self;
     int flag;
{
  if (self->child)
    {
      SendClientMessage(XtDisplay(self->w_area), self->child,
			xa_cb_editable, flag ? xa_cb_true : xa_cb_false,
			0);
    }
}
/* ------------------------------------------------------------------------ */
void
AreaTransmit(message, orig_reply, self)
  t_sexp message;
  t_mbus_reply orig_reply;
  t_area_widget self;
{
}
/* ------------------------------------------------------------------------ */
static struct keyword_entry_struct ScrollbarKeywords[] =
{
  { ":delete", (void *)KEYWORD_FALSE, KEYWORD_FLAG },
  { ":minimum", (void *)-1, KEYWORD_INT },
  { ":maximum", (void *)-1, KEYWORD_INT },
  { ":size", (void *)-1, KEYWORD_INT },
  { ":value", (void *)-1, KEYWORD_INT },
};

/*----------------------------------------------------------------------*/
void
UpdateScrollbar(self, scrollbar, message)
  t_area_widget self;
  Widget scrollbar;
  t_sexp message;
{
  int n, min, max, size, value;
  Arg argl[10];
  /* message should be (&key ...) */
  MBparse_keywords(message, ScrollbarKeywords, ARRAY_SIZE(ScrollbarKeywords));
  if (KEYWORD_TRUE == (t_keyword_flag) ScrollbarKeywords[0].result)
      XtUnmanageChild(scrollbar); /* Remove the scroll bars from view */
  else
  {
    min = (int) ScrollbarKeywords[1].result;
    max = (int) ScrollbarKeywords[2].result;
    size = (int) ScrollbarKeywords[3].result;
    value = (int) ScrollbarKeywords[4].result;
    XtManageChild(scrollbar);	/* Make sure the scroll bars are visible */
    n = 0;
    if (min > 0)
	XtSetArg(argl[n], XmNminimum, min), ++n;
    if (max > 0)
	XtSetArg(argl[n], XmNmaximum, max), ++n;
    if (size)
	XtSetArg(argl[n], XmNsliderSize, size), ++n;
    if (value)
    {
      XtSetArg(argl[n], XmNvalue, value), ++n;
      if (scrollbar == self->hscrollbar)
	  self->hvalue = value;
      else
	  self->vvalue = value;
    }	
    if (n > 0)
	XtSetValues(scrollbar, argl, n);
  }
}

void
ScrollbarCallback(w, area, data)
  Widget w;
  t_area_widget area;
  XmScrollBarCallbackStruct *data;
{
  Display *dpy = XtDisplay(area->w_area);
  int n, value, offset;
  Arg argl[10];
  /* WARNING:  Scroll bars don't work all the way.  When you slide it
   * all the way to the bottom of the scrolled area, it doesn't always
   * send the last possible value.  This is a bug in Motif (you can
   * make the motif managed scroll bars do the same thing).
   * Where you grab the scroll bar can have a big affect on how well it
   * functions.
   */
  if (area->child)
  {
    value = data->value;
    if (w == area->hscrollbar)
    {
      offset = value - area->hvalue;
      SendClientMessage(dpy, area->child, xa_cb_hscrollbar,
			(unsigned long)value, (unsigned long)offset);
      area->hvalue = value;
    }
    else
    {
      offset = value - area->vvalue;
      SendClientMessage(dpy, area->child, xa_cb_vscrollbar,
			(unsigned long)value, (unsigned long)offset);
      area->vvalue = value;
    }
  }
}
