/* This file is part of the 
 *
 *	Delta Project  (ConversationBuilder)  
 *	Human-Computer Interaction Laboratory
 *	University of Illinois at Urbana-Champaign
 *	Department of Computer Science
 *	1304 W. Springfield Avenue
 *	Urbana, Illinois 61801
 *	USA
 *
 *	c 1989,1990,1991,1992 Board of Trustees
 *		University of Illinois
 *		All Rights Reserved
 *
 * This code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY. No author or distributor accepts
 * responsibility to anyone for the consequences of using this code
 * or for whether it serves any particular purpose or works at all,
 * unless explicitly stated in a written agreement.
 *
 * Everyone is granted permission to copy, modify and redistribute
 * this code, except that the original author(s) must be given due credit,
 * and this copyright notice must be preserved on all copies.
 *
 *	Author:  Alan Carroll (carroll@cs.uiuc.edu)
 *      Modified:  Doug Bogia (bogia@cs.uiuc.edu)
 *
 *	Project Leader:  Simon Kaplan (kaplan@cs.uiuc.edu)
 *	Direct enquiries to the project leader please.
 */

/* widget server list driver code */
/* $Source: /export/kaplan/stable/sun4.os4.1/cb-2.0/src/data/node114.text,v $ */

static char rcsid[] = "list.c $Revision: 1.0.1.3 $ $Date: 92/05/28 09:21:57 $ $State: Exp $ $Author: CBmgr $";

/* ------------------------------------------------------------------------ */
#include "header.h"
#include <Xm/List.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
/* ------------------------------------------------------------------------ */
extern void
  ListRemove(), ListUpdate(), ListReply(), ListRealized(),
  ListEditable(), ListTransmit();

extern t_generic_widget ListHandle();

struct widget_class_struct ListClassRecord =
{
  WS_LIST, "list",
  ListHandle, ListRemove, NULL, ListUpdate, ListReply, ListRealized,
  ListTransmit, ListEditable,
} ;
/* ------------------------------------------------------------------------ */
typedef struct list_item_struct
{
  XmString label;			/* visible label */
  char *value;				/* selection value of item */
  t_sexp id;				/* update identifier */
  int pos;				/* list position */
  struct list_item_struct *next;
} * t_list_item;

/* a list consists of a frame widget with a form widget, containing a
 * title widget and the actual scrolled list widget.
 * The title widget is managed only if it has been set.
 */
typedef struct list_widget_struct
{
  DECLARE_STANDARD_WIDGET_SLOTS;
  char *key;				/* key for item reply */
  char *tag;				/* tag for item reply */
  char *transmit_key;			/* key for transmit */
  char *select_key;			/* key for list reply */
  t_list_item items;
  Widget w_list;			/* the actual list widget */
  Widget w_frame;
  Widget form_widget;
  Widget title_widget;
  t_sexp empty_action;			/* action when list becomes empty */
  t_sexp fill_action;			/* when list becomes non-empty */
  Boolean auto_visible;			/* reset visible on updates */
} * t_list_widget;

/* ------------------------------------------------------------------------ */
/* Return an array with the item pointers in it.
 * For various reasons (mostly that Motif labels list items starting at 1),
 * the array returned has a count in slot 0, and the items in slots starting
 * at 1.
 */
t_list_item *
ListItemArray(list) t_list_widget list;
{
  t_list_item *the_items;		/* holder for array of items */
  t_list_item item;			/* pointer for stepping through list */
  int count;				/* counter for # of items */
  int n = 0;				/* for double checking */
  Arg arg;

  /* Now, here's a key point - if things are copasetic, then it will turn
   * out that if there are n items, they will be in positions 1..n.
   * If this is not the case, we're hosed.
   */

  /* How many items? */
  XtSetArg(arg, XmNitemCount, &count);
  XtGetValues(list->w_list, &arg, 1);

  if (!count) return NULL;		/* empty list, take off */

  the_items = (t_list_item *)XtMalloc(sizeof(t_list_item) * (count+1));
  the_items[0] = (t_list_item)count;

  /* Put the items in the list, in their positions */

  for (item = list->items ; NULL != item; item = item->next)
    {
      if (1 <= item->pos && item->pos <= count)
	{
	  the_items[item->pos] = item;
	  n += 1;
	}
      else
	{
	  fprintf(stderr, "Bad list position (%d / %d)\n", item->pos, count);
	  XtFree(the_items);
	  the_items = NULL;
	  break;
	}
    }

  if (n != count)
    {
      fprintf(stderr, "Bad list item count (motif=%d, internal=%d)\n",
	      count, n);
      XtFree(the_items);
      the_items = NULL;
    }

  return the_items;			/* remember to free this in caller */
}
/* ------------------------------------------------------------------------ */
void
ListItemCallback(w, list, reason)
     Widget w;
     t_list_widget list;
     XmListCallbackStruct *reason;
{
  int *items;
  int count;
  int pos;
  t_mbus_reply reply;
  t_list_item item;
  Boolean r;

  if (NULL == list->key) return;	/* no key, no reply */
  pos = reason->item_position;

  /* find the selected item */
  for ( item = list->items ; NULL != item ; item = item->next )
    if (item->pos == pos) break;

  if (NULL == item)			/* this shouldn't happen */
    {
      if (MBLogLevel > 0) fprintf(stderr, "Bad item select (%d)\n", pos);
      return;
    }

  reply = NewReply();
  reply->value = strdup(item->value);
  list->class->reply(list, reply);
  SendReply(reply);
  FreeReply(reply);
  XmListDeselectPos(w, pos);
}
/* ------------------------------------------------------------------------ */
void
ListFreeItem(item) t_list_item item;
{
  MBfree(item->id);
  XtFree(item->value);
  XtFree(item->label);
  XtFree(item);
}
/* ------------------------------------------------------------------------ */
void
FreeListItems(items) t_list_item items;
{
  t_list_item next;

  while (NULL != items)
    {
      next = items->next;
      ListFreeItem(items);
      items = next;
    }
}
/* ------------------------------------------------------------------------ */
void
ListSelectItem(list, id)
     t_list_widget list;
     t_sexp id;
{
  t_list_item item;

  if (MB_CHUNKP(id))
    {
      for ( item = list->items ; NULL != item ; item = item->next )
	if (MBequal(id, item->id))
	  {
	    XmListSelectPos(list->w_list, item->pos, True);
	    break;
	  }
    }
}
/* ------------------------------------------------------------------------ */
/* Process selection request.
 * This should be either a keyword, an id, or list of id's.
 */
void
ListDoSelect(list, e)
     t_list_widget list;
     t_sexp e;
{
  t_list_item item;

  /* Check the keyword cases first */
  if (NULL == e)
    ;					/* do nothing */
  else if (!MBcompare_Cstring(e, ":none"))
    XmListDeselectAllItems(list->w_list);
  else if (!MBcompare_Cstring(e, ":all"))
    {
      int count;
      XmStringTable stuff;

      XtVaGetValues(list->w_list,
		    XmNitems, &stuff, XmNitemCount, &count, NULL);
      XtVaSetValues(list->w_list,
		    XmNselectedItems, stuff, XmNselectedItemCount, count,
		    NULL);
    }
  else if (MB_CHUNKP(e))		/* assume single id */
    ListSelectItem(list, e);
  else if (MB_CONSP(e))
    {
      /* Motif Wonderfulness. The only way to not reset the previously
       * selected items as we select additional items is to have the
       * selectionPolicy be MULTIPLE_SELECT, so what we do is save the
       * current value, switch over to MULTIPLE_SELECT, select the items
       * specified, and then flip back.
       */
      unsigned char sp;
      XtVaGetValues(list->w_list, XmNselectionPolicy, &sp, NULL);
      XtVaSetValues(list->w_list, XmNselectionPolicy, XmMULTIPLE_SELECT, NULL);
      for ( ; MB_CONSP(e) ; e = MB_CDR(e) )
	ListSelectItem(list, MB_CAR(e));
      XtVaSetValues(list->w_list, XmNselectionPolicy, sp, NULL);
    }
}
/* ------------------------------------------------------------------------ */
/* Update our list positions. Start is first position to change, and delta
 * is the amount to change all positions past start.
 */
void
ListMovePositions(list, start, delta)
     t_list_widget list;
     int start, delta;
{
  t_list_item item;

  for ( item = list->items ; NULL != item ; item = item->next )
    if (item->pos >= start) item->pos += delta;
}
/* ------------------------------------------------------------------------ */
/* Return the position the item was in, or -1 if no item deleted */
int
ListDeleteItem(list, id)
     t_list_widget list;
     t_sexp id;			/* id of the item to delete */
{
  t_list_item prev, current;
  int pos = -1;

  for ( prev = NULL, current = list->items
       ; NULL != current
       ; prev = current, current = current->next )
    {
      if (MBequal(id, current->id)) /* found it */
	{
	  if (NULL == prev) list->items = current->next;
	  else prev->next = current->next;
	  pos = current->pos;
	  XmListDeletePos(list->w_list, pos);
	  ListMovePositions(list, current->pos, -1);
	  MBfree(current->id);
	  XtFree(current->value); XtFree(current->label);
	  XtFree(current);
	  break;
	}
    }
  return pos;
}
/* ------------------------------------------------------------------------ */
void
ListAdjustVisible(list) t_list_widget list;
{
  Arg arg;
  int count;

  if (list->auto_visible)
    {
      XtSetArg(arg, XmNitemCount, &count);
      XtGetValues(list->w_list, &arg, 1);
      if (count > 0)
	{
	  XtSetArg(arg, XmNvisibleItemCount, count);
	  XtSetValues(list->w_list, &arg, 1);
	}
    }
}
/* ------------------------------------------------------------------------ */
void
ListDoDelete(list, e)
     t_list_widget list;
     t_sexp e;
{
  t_list_item item;
  t_list_item *item_array = NULL;
  int *selected;			/* array of selected positions */
  int count;				/* # selected */
  int i, n;
  Boolean f_free;
  Boolean f_empty = NULL == list->items; /* initially empty? */

  /* Check for keywords first */
  if (NULL == e)
    ;					/* do nothing */
  else if (!MBcompare_Cstring(e, ":self"))
    {
      ListRemove(list);
      return;				/* avoid end of function checks */
    }
  else if (!MBcompare_Cstring(e, ":all"))
    {
      FreeListItems(list->items);
      XmListDeleteAllItems(list->w_list);
      list->items = NULL;
    }
  else if (!MBcompare_Cstring(e, ":selected"))
    {
      /* Delete all selected items */

      f_free = XmListGetSelectedPos(list->w_list, &selected, &count);
      if (f_free && count > 0 && NULL != (item_array = ListItemArray(list)))
	{
	  /* Delete all the selected items */
	  for ( i = count - 1 ; i >= 0 ; --i )
	  {
	    item = item_array[selected[i]];
	    item_array[selected[i]] = NULL;
	    XmListDeletePos(list->w_list, item->pos);
	    ListFreeItem(item);
	  }
	  /* Regenerate the linked list */
	  list->items = NULL;
	  count = (int)(item_array[0]);	/* extract original # of items */
	  for ( i = n = 1 ; i <= count ; ++i )
	    {
	      if (NULL != (item = item_array[i]))
		{
		  item->pos = n++;
		  item->next = list->items;
		  list->items = item;
		}
	    }
	}
      if (NULL != item_array) XtFree(item_array);
      if (f_free) XtFree(selected);
    }
  else
    {
      for ( ; NULL != e ; e = MB_CDR(e) )
	{
	  if (MB_CHUNKP(e)) ListDeleteItem(list, e);
	  else ListDeleteItem(list, MB_CAR(e));
	}
    }

  /* if it wasn't empty, and now it is, do empty action */
  if (! f_empty && NULL == list->items) Act(list, list->empty_action, 0);

}

/* ------------------------------------------------------------------------ */
void
ListAddItem(list, item, pos)
     t_list_widget list;		/* list widget */
     t_list_item item;			/* item to add */
     int pos;				/* location to add at */
{
  int old_pos;
  t_list_item check_item;

  /* Look and see if it's already there. If so, and no position specified,
   * add it back in the same place
   */
  old_pos = ListDeleteItem(list, item->id);
  if (-1 == pos) pos = old_pos >= 0 ? old_pos : 1;

  if (pos > 0)
      /* Make way for this new position. */
      ListMovePositions(list, pos, 1);
  else
  {
    for (pos = 0, check_item = list->items;
	 NULL != check_item;
	 check_item = check_item->next, ++pos)
	;
    pos++;
  }
  XmListAddItem(list->w_list, item->label, pos);
  item->pos = pos;
  item->next = list->items;
  list->items = item;
}
/* ------------------------------------------------------------------------ */
/* Accept an update message and update the list */
static struct keyword_entry_struct liukw[] =
{
  { ":delete", NULL, KEYWORD_SEXP },
  { ":label", NULL, KEYWORD_COOKED },
};

void
ListUpdateItems(list,sexp)
     t_list_widget list;
     t_sexp sexp;
{
  t_sexp s_item, s_delete, s_id;
  t_list_item item;
  char *label;
  Boolean delete_flag;

  s_id = MBnth(sexp, 1);

  /* check these first, because if there's nothing for us, why do the loop? */
  MBparse_keywords(MBnthcdr(sexp,2), liukw, ARRAY_SIZE(liukw));
  s_delete = (t_sexp)liukw[0].result;
  label = (char *)liukw[1].result;

  delete_flag = !MBcompare_Cstring(s_delete, ":self");

  /* Make sure that there's something to do before going through the list */
  if (!delete_flag && NULL == label) return;

  for ( item = list->items ; NULL != item ; item = item->next )
    {
      if (MBequal(s_id, item->id))
	{
	  if (delete_flag) ListDeleteItem(list, s_id);
	  else if (NULL != label)
	    {
	      XtFree(item->label);
	      item->label = XmStringCreate(label,XmSTRING_DEFAULT_CHARSET);
	      XmListDeletePos(list->w_list, item->pos);
	      XmListAddItem(list->w_list, item->label, item->pos);
	    }
	  break;
	}
    }
  XtFree(label);
}
/* ------------------------------------------------------------------------ */
/* Each list item looks like ( "id" "label" value ) */
t_list_item
HandleListItems(sexp)
     t_sexp sexp;
{
  t_list_item base = NULL;
  t_sexp item, s_label, s_id, s_value;

  for ( ; MB_CONSP(sexp) ; sexp = MB_CDR(sexp) )
    {
      item = MB_CAR(sexp);
      /* special check for possible singleton */
      if (!MB_CONSP(item))
	{
	  item = sexp;
	  sexp = NULL;			/* don't parse anything more */
	}
      s_id = MB_CAR(item);
      s_label = MBnth(item,1);
      s_value = MBnth(item,2);

      if (MB_STRINGP(s_label))
	{
	  t_list_item new = NEW_STRUCT(list_item_struct);
	  char *str;

	  new->id = Getnth(item, 0);

	  str = MBCstring(s_label);
	  new->label = XmStringCreate(str,XmSTRING_DEFAULT_CHARSET);
	  XtFree(str);

	  new->value = MBprint_Cstring(s_value);

	  new->next = base;
	  base = new;
	}
    }
  return base;
}
/* ------------------------------------------------------------------------ */
/* We need to form up all the strings and stick them into the list */
int
ManageListItems(list) t_list_widget list;
{
  XmString *strings;
  int count = 0, i;
  t_list_item item;

  for ( item = list->items ; NULL != item ; item = item->next )
    count += 1;

  strings = (XmString *) XtMalloc(sizeof(XmString) * count);

  for ( i = count, item = list->items
       ; NULL != item
       ; item = item->next, --i )
    {
      item->pos = i;
      strings[i-1] = item->label;
    }

  XmListAddItems(list->w_list, strings, count, 0);
  XtFree(strings);
  return count;
}
/* ------------------------------------------------------------------------ */
static struct keyword_entry_struct ListKeywords[] =
{
  { ":title", NULL, KEYWORD_COOKED },
  { ":tag", NULL, KEYWORD_RAW },
  { ":key", NULL, KEYWORD_COOKED },
  { ":visible", NULL, KEYWORD_SEXP },
  { ":transmit-key", NULL, KEYWORD_RAW },
  { ":select-key", NULL, KEYWORD_RAW },
  { ":selection", NULL, KEYWORD_SEXP},
  { ":empty-action", NULL, KEYWORD_GET_SEXP },
  { ":fill-action", NULL, KEYWORD_GET_SEXP },
  { ":scroll-bars", NULL, KEYWORD_SEXP },
};

t_generic_widget
ListHandle(parent,sexp)
     t_generic_widget parent;
     t_sexp sexp;
{
  t_list_widget list;
  int n, visible = -1;
  char *title;
  t_sexp s_selection, s_visible, s_bars;
  Arg argl[10];

  /* check things to see if they're in the right format */
  if (NULL == parent || NULL == sexp) return NULL;

  list = NEW_STRUCT(list_widget_struct);
  list->type = WS_LIST;
  list->class = &ListClassRecord;
  list->parent = parent;
  list->top = parent->top;
  list->id = Getnth(sexp, 1);
  list->items = NULL;
  list->auto_visible = False;

  /* sexp should be (list "id" <items> &key ...) */
  MBparse_keywords(MBnthcdr(sexp,3), ListKeywords, ARRAY_SIZE(ListKeywords));
  n = 0;
  title = (char *)ListKeywords[n++].result;
  list->tag = (char *)ListKeywords[n++].result;
  list->key = (char *)ListKeywords[n++].result;
  s_visible = (t_sexp)ListKeywords[n++].result;
  list->transmit_key = (char *)ListKeywords[n++].result;
  list->select_key = (char *)ListKeywords[n++].result;
  s_selection = (t_sexp)ListKeywords[n++].result;
  list->empty_action = (t_sexp)ListKeywords[n++].result;
  list->fill_action = (t_sexp)ListKeywords[n++].result;
  s_bars = (t_sexp)ListKeywords[n++].result;

  /* Deal with the visible flag */
  if (!MBcompare_Cstring(s_visible, ":auto")) list->auto_visible = True;
  else
    {
      int chars;
      visible = MBint(s_visible, &n);
      if (n <= 0) visible = -1;		/* not a number, blow off */
    }

  /* outer frame */
  XtSetArg(argl[0], XmNuserData, list);
  list->w_frame = XtCreateWidget("frame", xmFrameWidgetClass,
				      parent->widget, argl, 1);
  list->widget = list->w_frame;

  /* form */
  XtSetArg(argl[0], XmNuserData, list);
  if (!list->auto_visible)
    XtSetArg(argl[1], XmNresizePolicy, XmRESIZE_NONE);
  list->form_widget = XtCreateWidget("form", xmFormWidgetClass,
				     list->w_frame, argl, 2);

  /* title */
  list->title_widget = CreateTitle(NULL == title ? "" : title,
				   list->form_widget);
  XtSetArg(argl[0], XmNleftAttachment, XmATTACH_FORM);
  XtSetArg(argl[1], XmNtopAttachment, XmATTACH_FORM);
  XtSetArg(argl[2], XmNrightAttachment, XmATTACH_FORM);
  XtSetArg(argl[3], XmNalignment, XmALIGNMENT_CENTER);
  XtSetValues(list->title_widget, argl, 4);

  n = 0;
  XtSetArg(argl[n], XmNuserData, list), ++n;
  XtSetArg(argl[n], XmNleftAttachment, XmATTACH_FORM), ++n;
  XtSetArg(argl[n], XmNbottomAttachment, XmATTACH_FORM), ++n;
  XtSetArg(argl[n], XmNrightAttachment, XmATTACH_FORM), ++n;

  XtSetArg(argl[n], XmNlistSizePolicy, XmCONSTANT), ++n;

  if (NULL == title)
    XtSetArg(argl[n], XmNtopAttachment, XmATTACH_FORM), ++n;
  else
    XtSetArg(argl[n], XmNtopAttachment, XmATTACH_WIDGET), ++n;
  XtSetArg(argl[n], XmNtopWidget, list->title_widget), ++n;

  if (!MBcompare_Cstring(s_selection, ":single"))
    XtSetArg(argl[n], XmNselectionPolicy, XmSINGLE_SELECT), ++n;
  else if (!MBcompare_Cstring(s_selection, ":browse"))
    XtSetArg(argl[n], XmNselectionPolicy, XmBROWSE_SELECT), ++n;
  else if (!MBcompare_Cstring(s_selection, ":multiple"))
    XtSetArg(argl[n], XmNselectionPolicy, XmMULTIPLE_SELECT), ++n;
  else if (!MBcompare_Cstring(s_selection, ":extended"))
    XtSetArg(argl[n], XmNselectionPolicy, XmEXTENDED_SELECT), ++n;

  list->w_list = XmCreateScrolledList(list->form_widget, "list",
					   argl, n);

  list->items = HandleListItems(MBnth(sexp,2));

  n = ManageListItems(list);
  if (visible > 0)
    {
      XtSetArg(argl[0], XmNvisibleItemCount, visible);
      XtSetValues(list->w_list, argl, 1);
    }

  if (!MBcompare_Cstring(s_bars, ":static"))
    {
      XtSetArg(argl[0], XmNscrollBarDisplayPolicy, XmSTATIC);
      XtSetValues(list->w_list, argl, 1);
    }

  XtAddCallback(list->w_list, XmNdefaultActionCallback,
		ListItemCallback, list);

  if (NULL != title) XtManageChild(list->title_widget);
  XtManageChild(list->w_list);
  XtManageChild(list->form_widget);

  XtFree(title);
  return (t_generic_widget) list;
}

/* ------------------------------------------------------------------------ */
void
ListRemove(self) t_list_widget self;
{
  FreeListItems(self->items);
  MBfree(self->id);
  MBfree(self->empty_action);
  MBfree(self->fill_action);
  XtFree(self->key);
  XtFree(self->tag);
  XtFree(self->transmit_key);
  XtFree(self->select_key);
  XtFree(self);
}
/* ------------------------------------------------------------------------ */
/* Send a reply with a list of the selected items */
void
ListSelectReply(list, e)
     t_list_widget list;		/* the list widget */
     t_sexp e;				/* sexp of reply pieces */
{
  t_mbus_reply reply;
  t_sexp message;
  Boolean f_free;
  t_list_item *item_array;
  int *selected;			/* array of selected positions */
  int count;				/* # selected */

  reply = ParseReplyList(e);

  if (NULL == reply->key)
    {
      if (NULL == list->select_key)
	{
	  FreeReply(reply);
	  return;
	}
      else
	reply->key = strdup(list->select_key);
    }

  message = MBGetString();
  MBput_Cstring(message, " (");
  f_free = XmListGetSelectedPos(list->w_list, &selected, &count);
  if (f_free && count > 0 && NULL != (item_array = ListItemArray(list)))
    {
      while (count > 0)
	{
	  MBput_Cstring(message, " ");
	  MBput_Cstring(message, item_array[selected[--count]]->value);
	}
      XtFree(item_array);
    }

  MBput_Cstring(message, " )");
  if (f_free) XtFree(selected);

  /* Put this in the value field */
  XtFree(reply->value);
  reply->value = MBCstring(message);

  /* get our tag if any */
  if (NULL == reply->tag) reply->tag = MBstrdup(list->tag);
  /* cast up the chain for any other pieces */
  list->parent->class->reply(list->parent, reply);

  SendReply(reply);
  FreeReply(reply);

}
/* ------------------------------------------------------------------------ */
/* Accept an update message and update the list */
static struct keyword_entry_struct ListUpdateKeys[] =
{
  { ":title", NULL, KEYWORD_SEXP },
  { ":add", NULL, KEYWORD_SEXP },
  { ":position", (void *) -1, KEYWORD_INT },
  { ":delete", NULL, KEYWORD_SEXP },
  { ":replace", NULL, KEYWORD_SEXP },
  { ":select", NULL, KEYWORD_SEXP },
  { ":select-reply", NULL, KEYWORD_SEXP },
  { ":id", NULL, KEYWORD_GET_SEXP },
};

void
ListUpdate(self,sexp)
     t_list_widget self;
     t_sexp sexp;
{
  XmString xm_string;
  char *title;
  t_sexp s_add, s_delete, s_replace, s_title, s_select;
  t_sexp s_item, s_select_reply, s_id;
  int pos;				/* position to add */
  int n;
  Arg argl[4];

  if (!MBequal(MBnth(sexp,1), self->id))
    {
      ListUpdateItems(self, sexp);
      return;
    }

  MBparse_keywords(MBnthcdr(sexp,2), ListUpdateKeys,
		ARRAY_SIZE(ListUpdateKeys));
  n = 0;
  s_title = (t_sexp)ListUpdateKeys[n++].result;
  s_add = (t_sexp)ListUpdateKeys[n++].result;
  pos = (int)ListUpdateKeys[n++].result;
  s_delete = (t_sexp)ListUpdateKeys[n++].result;
  s_replace = (t_sexp)ListUpdateKeys[n++].result;
  s_select = (t_sexp)ListUpdateKeys[n++].result;
  s_select_reply = (t_sexp)ListUpdateKeys[n++].result;
  s_id = (t_sexp)ListUpdateKeys[n++].result;

  /* recognize an id, list of id, :none, :all */
  ListDoSelect(self, s_select);

  if (NULL != s_id)
    {
      MBfree(self->id);
      self->id = s_id;
    }

  if (NULL != s_title)			/* something was specified */
    {
      if (NULLP(s_title))		/* remove title */
	{
	  if (XtIsManaged(self->title_widget))
	    {
	      XtSetArg(argl[0], XmNtopAttachment, XmATTACH_FORM);
	      XtSetValues(XtParent(self->w_list), argl, 1);
	      XtUnmanageChild(self->title_widget);
	    }
	}
      else				/* set a new title */
	{
	  char * title = MBCstring(s_title);

	  if (!XtIsManaged(self->title_widget))
	    {
	      XtManageChild(self->title_widget);
	      XtSetArg(argl[0], XmNtopAttachment, XmATTACH_WIDGET);
	      XtSetArg(argl[1], XmNtopWidget, self->title_widget);
	      XtSetValues(XtParent(self->w_list), argl, 2);
	    }
	  SetTitle(self->title_widget, title);
	  XtFree(title);
	}
    }

  if (NULL != s_replace)
    {
     Boolean f_empty = NULL == self->items;
 
      /* this cancels add / delete */
      s_add = s_delete = NULL;

      FreeListItems(self->items);
      XmListDeleteAllItems(self->w_list);
      self->items = HandleListItems(s_replace);
      ManageListItems(self);

     if (!f_empty && NULL == self->items) Act(self, self->empty_action, 0);
     if (f_empty && NULL != self->items) Act(self, self->fill_action, 0);

    }

  ListDoDelete(self, s_delete);

  if (NULL != s_add)
    {
      t_list_item new_items, next;
      Boolean f_empty = NULL == self->items;

      new_items = HandleListItems(s_add);
      while (NULL != new_items)
	{
	  next = new_items->next;
	  ListAddItem(self, new_items, pos);
	  new_items = next;
	}
      if (f_empty && NULL != self->items) Act(self, self->fill_action, 0);
    }

  if (NULL != s_select_reply) ListSelectReply(self, s_select_reply);

  ListAdjustVisible(self);		/* checks the flag */

}
/* ------------------------------------------------------------------------ */
void
ListReply(self, reply)
     t_list_widget self;
     t_mbus_reply reply;
{
  if (NULL == reply->key) reply->key = strdup(self->key);
  if (NULL == reply->tag) reply->tag = strdup(self->tag);
  self->parent->class->reply(self->parent, reply);
}
/* ------------------------------------------------------------------------ */
void
ListRealized(self) t_list_widget self;
{
}
/* ------------------------------------------------------------------------ */
/* We transmit only if our transmit_key is set */
void
ListTransmit(message, orig_reply, self)
  t_sexp message;
  t_mbus_reply orig_reply;
  t_list_widget self;
{
  t_list_item item;

  if (NULL != self->transmit_key)
    {
      MBput_Cstring(message, " ");
      MBput_Cstring(message, self->transmit_key);
      MBput_Cstring(message, "  (");
      for ( item = self->items ; NULL != item ; item = item->next )
	{
	  MBput_Cstring(message, " ");
	  MBput_Cstring(message, item->value);
	}
      MBput_Cstring(message, " )");
    }
}
/* ------------------------------------------------------------------------ */
void
ListEditable(self, flag)
     t_list_widget self;
     int flag;
{
}
/* ------------------------------------------------------------------------ */
