/****************************************************************************
 * construct.c
 * Author Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include  <stdio.h>
#include  <ctype.h>

#include "alisp.h"


/*
   The 'not' structure simply evaluates its one argument, returning T if
   the argument <exp> evaluates to NIL.  It returns NIL otherwise.
*/
NODE *eval_not(exp)
NODE *exp;
{
	NODE *result;

	if (null(eval_ilisp(exp)))
		{
		result = TRUE_NODE;
		}
	else
		result = NIL;
	return(result);
}

/*

   This function evaluates AND structures.  Returns the first element of
   <list> to evaluate as NIL, or the evaluation of the last element of <list>,
   whichever comes first.

*/
NODE *eval_and(list)
NODE *list;
{
	NODE *result;
	int not_finished=1;

	if (null(list))
		{
		handle_error("No arguments to and");
		return(NIL);
		}
	while (not_finished)
		{
		if (null(list))
			not_finished = 0;
		else if ( null( (result =eval_ilisp(top_of(list))) ) )
			not_finished = 0;
		else
			list = rest_of(list);
		}
	return(result);
}

/*
    Like eval_and, only uses or instead of and (i.e, returns the first non-nil
    evaluation).
*/
NODE *eval_or(list)
NODE *list;
{
	NODE *result;
	int not_finished=1;

	if (null(list))
		{
		handle_error("No arguments to or");
		return(NIL);
		}
	while (not_finished)
		{
		if (null(list))
			not_finished = 0;
		else if ( !null( (result = eval_ilisp(top_of(list))) ) )
			not_finished = 0;
		else
			list = rest_of(list);
		}
	return(result);
}

/*
    Evaluates if statements.  <list> should be a 3 element list.  The first
    element should be the test, the second the consequence of the condition
    being true, the last element the default value
*/
NODE *eval_if(list)
NODE *list;
{
	NODE *result;

	if ( list_length(list) != 3 )
		{
		fprintf(stderr,"BAD IF STATEMENT\n");
		print_out(list);
		return(NIL);
		}
	if ( !null( eval_ilisp( nth1(list) ) ) )
		{
		result = eval_ilisp(nth2(list));
		}
	else
		{
		result = eval_ilisp(nth3(list));
		}
	return(result);
}

/*
   PURPOSE: This function preforms the COND structure.
   INPUT: <list> is of the form ( (case1 consequence1) (case2 conseq2) ...)
   OUTPUT: the evaluation of the <conseq-x> corresponding to the first <case-x>
	  that evaulates to a non-nil value
   EXPL: If the car of the list <cases> evaluates to a non-NIL value,
	then the car of the list <conseqs> is evaluated and returned as the
	result, else the function is called recursively on the cdr of the
	<cases> and <conseqs> lists.  It is an error for every element of
	the <cases> list to evaluate to NIL.  At least one case must be valid.
*/
NODE *eval_cond(list)
NODE *list;
{
	NODE *result;
	int not_finished=1;

	result = NIL;	
	while (not_finished)
		{
		if (!consp(list))
			{
			handle_error("COND:No valid case in statement");
			not_finished = 0;
			}
		/* car(car(list)) is the case statement  */
		else if (!null( eval_ilisp( car( car(list) ) ) ))
			{
		/* cdr(car(list)) is the case statement  */
			result = eval_program( cdr ( car(list) ) );
			not_finished = 0;
			}
		else
			list = cdr(list);
		}
	return(result);
}

/*
   PURPOSE:This function evaluates the LET and LET* structures.
   INPUT:<let_varlist> is a list of the form ( (var1 val1) (var2 val2) ... ),
	 where each (var val) element represents one temporary variable
	 Binding to be made.  
   OUTPUT: the evaluation of <body> in the new environment containing the 
	   Bindings of <let_varlist>.
   EXPL: For each Binding <x>, a cons pair of the car(<x>) and the evaluation 
	 of the cdr(<x>) is made and added to the environment.
	 <letStar> denotes whether this is a LET or a LET* statement.  If 
	 <letStar> is not zero, then the cdr(<x>) is evaluated in a environment 
	 that includes the other temporary Bindings made so far. If <letStar>
	 is non-zero, the cdr(<x>) is evaluated in the environment 
	 that existed before this function starting executing. This will prevent
	 any dependancies of variable values on other variable values.  These 
	 evaluations are kept in a temporary binding list.  This list is added
	 onto the front of the real environment once all the temp bindings have
	 been made.  The body of the let statement is then evaluated 
	 in this new environment. The <env> is returned to its previous state 
	 once execution is finished.
*/
NODE *eval_let(let_varlist,body,letStar)
NODE *let_varlist,*body;
int letStar;
{
	NODE *newenv,*walk_varlist,*next_var,*next_val,*new_topenv,
	     *new_def,*result;
	int var_count=0;

	walk_varlist = let_varlist;

/*	Initialize let variables */
	if (letStar)
		{
		while (!null(walk_varlist))
			{
			next_var = car(car(walk_varlist));
			next_val = car(cdr(car(walk_varlist)));
			Bind(next_var,eval_ilisp(next_val));
			var_count++;
			new_topenv = next_var;
			walk_varlist = cdr(walk_varlist);
			}
		}
	else
		{
		newenv = NIL;
		incr_ref(newenv);
		while (!null(walk_varlist))
			{
			next_var = car(car(walk_varlist));
			next_val = car(cdr(car(walk_varlist)));
			new_def = cons(next_var, eval_ilisp(next_val) );
			newenv = cons(new_def,newenv);
			var_count++;
			new_topenv = next_var;
			walk_varlist = cdr(walk_varlist);
			}
		incr_ref(newenv);
		add_Bindlist(newenv);	
		}

/*  Evaluate body in new environment	*/
	result = eval_program(body);
	
/*  variables must be in order 		*/
	remove_Bindings(new_topenv,var_count);
	return(result);
}
	
/*
   PURPOSE:  This function evaluates the DO and DO* structures

   INPUT:
   <do_varlist> is of the form:
   ( (var-1 init-val-1 stepper-val-1)  (var-2 init-val-2 stepper-val-2) ...)
   where each <var-x> is a variable of the DO structure, <init-val-x> is its
   intial value, and <stepper-val-1> is the variable's new value after each
   pass through the loop.

   <endform> is of the form:
   ( test-exp endform1 endform2 ... endform-n)
   where <test-exp> is the loop conditional and <endform1> ... <endform-n> are
   the expressions to be evaluated when <test-exp> evaluates to a non-NIL value.

   <doStar> is 0 if this is a DO statement, and 1 if it is a DO* statement.

   OUTPUT:
   The evaluation of <endform-n> is returned.
   EXPL:
   the <var-x>'s are bound to the evaluation of the <init-val-x>'s in the same
   way as in a LET statement, in the environment dictated by <doStar>,  to form
   the new environment. The car of <endform>, (the <end-test> ) is then 
   evaluated.  If its value is NIL, <body> is evaluated in the new environment,
   the <var-x>'s are bound to the evaluation of the <stepper-val-x>'s, in the
   environment dictated by <doStar>, to form a new new-environment, and 
   <end-test> is evaluated again.  This continues until <end-test> returns a
   non-nil value, in which case the cdr of <endform> 
   (the ( <endform-1> ... <endform-n> ) list) is evaluated in the current
   new environment and returned as output.
   
*/
NODE *eval_do(do_varlist,endform,body,doStar)
NODE *do_varlist,*endform,*body;
int doStar;
{
	NODE *newenv,*newvals, *new_topenv,*walk_varlist,*next_var,
	     *next_def,*next_val,*result,*old;
	int var_count=0;

	walk_varlist = do_varlist;

/*	Initialize do variables */

	newenv = NIL;
	incr_ref(newenv);
	if (doStar)
		{
		while (!null(walk_varlist))
			{
			next_var = car(car(walk_varlist));
			next_val = car(cdr(car(walk_varlist)));
			Bind(next_var,eval_ilisp(next_val));
			var_count++;
			new_topenv = next_var;
			walk_varlist = cdr(walk_varlist);
			}
		}
	else
		{
		while (!null(walk_varlist))
			{
			next_var = car(car(walk_varlist));
			next_val = car(cdr(car(walk_varlist)));
			next_def = cons(next_var, eval_ilisp(next_val));
			newenv = cons(next_def, newenv);
			var_count++;
			new_topenv = next_var;
			walk_varlist = cdr(walk_varlist);
			}
		incr_ref(newenv);
		add_Bindlist(newenv);
		}

/* 	Loop, evaluating the body each time, then update variables	*/
	while (null(eval_ilisp(car(endform))))
		{
		result = eval_program(body);

		walk_varlist = do_varlist;
		if (doStar)
			while (!null(walk_varlist))
				{
				next_var = car(car(walk_varlist));
				old = assoc(next_var,EVAL_ENV);
				next_val = eval_ilisp(
			   	   car(cdr(cdr(car(walk_varlist)))));
				decr_elem(cdr(old));
				rplacd(old,next_val);
				incr_ref(cdr(old));
				walk_varlist = cdr(walk_varlist);
				}
		else
			{
			newvals = NIL;
			while (!null(walk_varlist))
				{
				/* we know the variable are in order in env */
				next_val = eval_ilisp(
			   	   car(cdr(cdr(car(walk_varlist)))));
				newvals = cons(next_val, newvals);
				walk_varlist = cdr(walk_varlist);
				}
			incr_ref(newvals);
			add_vallist(new_topenv,newvals);
			decr_elem(newvals);
			}
		}
	result = eval_program(cdr(endform));

	remove_Bindings(new_topenv,var_count);

	return(result);
}


/*  
   PURPOSE: This function evaluates a lambda expression.  This can be
   either a function defined with defun, or an explicit LAMBDA call.
   INPUT:
   <variables> is of the form: ( var-1 var-2 ... var-n)
   <var-n> may be a key or rest marker, indicating that the following variables
   are key variables of a rest variable

   <arguments> is of the form: ( arg-1 arg-2 ... arg-n) where each <arg-n>
   can be either a value to be bound to <var-x>, or a keyword variable.
   If <arg-x> is a keyword variable, then <arg-x+1> is the intended value of
   that variable.

   OUTPUT:
   the evaluation of <body> in the environment containing the Bindings of
   <variables> to <arguments> , is returned as
   output.
   EXPL: The function loops over the length of <arguments>, setting <arguments>
   to its cdr each pass.  It binds one (normal) argument to one (normal) 
   variable each pass.  If a rest or a key is encountered, it falls out of the
   loop for special handeling
*/
NODE *apply_lambda(variables, body, arguments)
NODE *body, *variables, *arguments;
{

	NODE *arg, *narg, *var,*result, *key_list,*top_newenv,*new_pair,*newval;
	int not_finished=1, var_count=0, key_count=0;

	top_newenv = NIL;
	while ( not_finished )
		{
		arg = top_of(arguments);
		var = top_of(variables);
		if (null(var))
			{
			if (null(arg))
				{
				result = eval_program(body);
				remove_Bindings(top_newenv,var_count);
				return(result);  
				}
			else
				{
				fprintf(stderr,"ERROR, Too many arguments to func\n");
				print_out(arguments);
				return(NIL);
				}
			}
		else if ( !symbolp(var) )
			{
			fprintf(stderr,"Bad variable to lambda expression:");
			print_out(var);
			return(NIL);
			}
	 	else if ( symbol_type(var) == K_KEY )
			not_finished = 0;
	 	else if ( symbol_type(var) == K_REST )
			not_finished = 0;
		else
			{
			Bind(var,arg);
			var_count++;
			top_newenv = var;
			variables = rest_of(variables);
			arguments = rest_of(arguments);
			};
		}
	if ( symbol_type(var) == K_REST )
		{
		variables = rest_of(variables);
		var = top_of(variables);
		if ( !symbolp(var) )
			{
			fprintf(stderr,
				   "Expected a variable following &rest, not:");
			print_out(var);
			return(NIL);
			}
		Bind(var,arguments);
		var_count++;
		top_newenv = var;
	
		variables = rest_of(variables);
		var = top_of(variables);
		};
	if ( !null(variables) && (symbol_type(var) == K_KEY) )
		{
		variables = rest_of(variables);
		var = top_of(variables);
		key_list=NIL;
		while ( !null(variables) )
			{
			if ( !symbolp( car(var) ) )
				{
				fprintf(stderr,"Bad keyword variable:");
				print_out( car(var) );
				fprintf(stderr,"::\n:");
				print_out( variables);
				fprintf(stderr,"::\n:");
				return(NIL);
				}
			key_list = cons( 
				        cons( car(var), eval_ilisp(nth2(var)) ),
				        key_list );
			key_count++;
			variables = rest_of(variables);
			var = top_of(variables);
			}
		incr_ref(key_list);

		while (keywordp(arg))
			{
			arguments = rest_of(arguments);
			newval = top_of(arguments);
			narg = lookup_symbol(symbol_name(arg)+1);
			if ( null(new_pair = assoc(narg,key_list)) )
				{
				fprintf(stderr,"Bad keyword $");
				print_out(narg);
				fprintf(stderr,"$$\n:");
				print_out(key_list);
				return(NIL);
				};
			incr_ref(newval);
			decr_ref(cdr(new_pair));
			rplacd( new_pair, newval );
			arguments = rest_of(arguments);
			arg = top_of(arguments);
			};
		if (!null(arg))
			{
			fprintf(stderr, "::Too many arguments to function\n");
		 	print_out(arguments);
			fprintf(stderr,":\n");
			print_out(arg);
			fprintf(stderr,":\n");
			return(NIL);
			};
		add_Bindlist(key_list);
		};
	result = eval_program(body);
	unlink_Bindlist(key_list,key_count);
	if (key_count > 0)
		decr_elem(key_list); 
	remove_Bindings(top_newenv,var_count);
	return(result);
}
