/****************************************************************************
 * material.c
 * Author Joel Welling
 * Copyright 1991, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module implements material properties 
*/

#include <stdio.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"

/* Notes-
*/

#define DEFAULT_KA 0.8
#define DEFAULT_KD 0.8
#define DEFAULT_KS 0.3
#define DEFAULT_EXP 30.0
#define DEFAULT_REFLECT 0.1
#define DEFAULT_REFRACT 1.0

/* The USERDATA struct referenced by all materials */
static USERDATA materialdef;

static void material_print( data )
char *data;
/* This is the print method for the material */
{
  Material mat;
  Color energy;
  mat= (Material)data;
  energy= material_energy(mat);
  ger_debug("material_print");
  fprintf(stderr,"material: { %f %f %f %f %f %f \n   { %f %f %f %f }}\n",
	  material_ka(mat),
	  mat->kd, mat->ks, mat->exp, mat->reflect, mat->refract,
	  color_red(energy), color_green(energy),
	  color_blue(energy), color_alpha(energy));
}

void material_destroy( data )
char *data;
/* This routine destroys a material instance */
{
  Material thismaterial;

  ger_debug("material_destroy");

  thismaterial= (Material)data;
  ren_free_material( thismaterial->ren_data );
  free( thismaterial );
}

static void init_material()
/* This routine initializes this module and plugs it into Alisp. */
{
  ger_debug("init_material");
  materialdef.type= "material";
  materialdef.destroy= material_destroy;
  materialdef.print= material_print;
  materialdef.setf_method= (void (*)())0;
}

NODE *material_create( arglist )
NODE *arglist;
/* This routine creates a new material instance */
{
  Material thismaterial;
  NODE *thisnode;
  static int initialized= 0;

  ger_debug("material_create:");

  if (!initialized) {
    init_material();
    initialized= 1;
  }

  if ( !(thismaterial= (Material)malloc(sizeof(struct Material_struct))) )
    ger_fatal("material: create: unable to allocate %d bytes!",
	      sizeof(struct Material_struct));

  /* Decode the NODE * here.  It is a list containing the data
   * entries in order.
   */
  if (length_list(arglist) < 7) {
    ger_error("material_create: need at least 7 arguments!");
    return( NIL );
  }

  if ( floatp(safe_car(arglist)) )
    thismaterial->ka= getflonum(safe_car(arglist));
  else {
    ger_error("material_create: argument for ka not of type float!");
    thismaterial->ka= DEFAULT_KA;
  }
  arglist= safe_cdr(arglist);

  if ( floatp(safe_car(arglist)) )
    thismaterial->kd= getflonum(safe_car(arglist));
  else {
    ger_error("material_create: argument for kd not of type float!");
    thismaterial->kd= DEFAULT_KD;
  }
  arglist= safe_cdr(arglist);

  if ( floatp(safe_car(arglist)) )
    thismaterial->ks= getflonum(safe_car(arglist));
  else {
    ger_error("material_create: argument for ks not of type float!");
    thismaterial->ks= DEFAULT_KS;
  }
  arglist= safe_cdr(arglist);

  if ( floatp(safe_car(arglist)) || fixp(safe_car(arglist)) )
    thismaterial->exp= get_number(safe_car(arglist));
  else {
    ger_error("material_create: argument for exp not of type float!");
    thismaterial->exp= DEFAULT_EXP;
  }
  arglist= safe_cdr(arglist);

  if ( floatp(safe_car(arglist)) )
    thismaterial->reflect= getflonum(safe_car(arglist));
  else {
    ger_error("material_create: argument for reflect not of type float!");
    thismaterial->reflect= DEFAULT_REFLECT;
  }
  arglist= safe_cdr(arglist);

  if ( floatp(safe_car(arglist)) )
    thismaterial->refract= getflonum(safe_car(arglist));
  else {
    ger_error("material_create: argument for refract not of type float!");
    thismaterial->refract= DEFAULT_REFRACT;
  }
  arglist= safe_cdr(arglist);

  /* The energy could wind up NIL or some other object if there is
   * no valid color in the appropriate slot.
   */
  thismaterial->energy= (Color)safe_car(arglist);
  grab(thismaterial->energy);

  /* call renderer material definition function */
  thismaterial->ren_data= ren_def_material( thismaterial );
  /* Build the node */
  thisnode= new_node(N_USERDATA);
  thisnode->val.userdef.data= (char *)thismaterial;
  thisnode->val.userdef.methods= &materialdef;
  return( thisnode );
}
