/****************************************************************************
 * motif_mi.c
 * Authors Brian Welcker, Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module joins with the mouse user interface module to provide a point-
and-click interface for workstations under Silicon Graphics GL.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#ifndef VMS
#include <unistd.h>
#include <sys/param.h>
#else
#define R_OK 4  /* needed for access() */
#define F_OK 0  /* needed for access() */
#endif

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/MainW.h>
#include <Xm/DialogS.h>
#include <Xm/Separator.h>
#include <Xm/RowColumnP.h>
#include <Xm/PushB.h>
#include <Xm/DrawingA.h>
#include <Xm/FileSB.h>
#include <Xm/Frame.h>
#include <Xm/CascadeB.h>
#include <Xm/MessageB.h>
#include <Xm/Form.h>
#include <Xm/Scale.h>
#include <Xm/Text.h>
#include <Xm/Label.h>

#include "ge_error.h"
#include "mi.h"
#include "motif_mi.h"
#include "flyby.h"

#ifdef RENDERER_IS_GL

#ifndef _IBMR2
#include <X11/Xirisw/GlxMDraw.h>

/* This extern is a backdoor to allow the renderer to set the configuration
 * of the widget into which it will draw.
 */
extern GLXconfig *ren_gl_glx_config;

#else /* ifndef _IBMR2 */
#include "/usr/lpp/GL/utilities/inc/Glib.h"
#include <gl/gl.h>
#include <gl/device.h>

/* This routine is provided by the renderer to initialize the window
 * into which it will draw.  ren_gl_aix_ready is zero before ren_gl_aix_init
 * is called, and nonzero after.
 */
extern void ren_gl_aix_init(Widget w,caddr_t, caddr_t );
extern int ren_gl_aix_ready;
#endif /* ifndef _IBMR2 */

#endif /* RENDERER_IS_GL */

/* Forward Declarations */
static void show_file_box();
static void handle_load();
static void close_box();
static void destroy_callback();
static void handle_distance();
static void button_press();
static void button_release();
static void redraw_callback();
static void play_anim_callback();
static void save_view_callback();
static void handle_save_view();
static void close_help_callback();

static Display *display;
static Cursor trans_cursor, rotate_cursor, arrow_cursor, wait_cursor;
static MousePosition mouse_up, mouse_down;

#ifdef RENDERER_IS_GL
static Window window_id;
static Window top_window_id;
static Display *display_ptr;
#else
extern Window window_id;
extern Window top_window_id;
extern Display *display_ptr;
#endif

/* Help dialog text */
static char help_text[]= "The initial view of the model is displayed in this \
window. To rotate the model, press the left mouse button at the original \
position. When the cursor changes, drag the mouse to the final position \
and release. It's like rolling a ball in which the model is embedded. To \
slide the model, similarly drag across the model with the middle mouse \
button. \n\n\
The slider at the bottom of the window allows you to change the distance from \
the camera to the object. The object can be moved farther or closer by \
sliding the box to the right or left, respectively. \
The display window may be resized or moved at any time.\n\n\
In the 'File' menu, the 'Open' button allows you to open a new P3D \
file for viewing.  The 'Next' button moves to the next frame in that \
file;  at the end of the file it wraps back around to the beginning. \
The 'Play' button causes frames to be displayed one after another;  you \
can stop this by clicking in the main window or on the slider. The \
'Save View' button allows you to save a P3D file which shows the models \
in the current file from the view visible in the main window.  The 'Exit' \
button exits the application.\n\n\
In the 'View' menu, the 'Camera Info' button causes information from the \
current view to be displayed in a dialog, where it can be edited. \
The 'Save Trace' button is a toggle; when it is set all viewpoints are \
saved in a file named path_trace.dat in the current directory.\n\n\
The features in the 'FlyBy' menu are used to produce flybys of the model \
in the main window.  The idea is that a number of viewpoints are saved as \
keyframes, and then a smooth path is drawn through these points to produce \
a flight around the model.  The 'Settings' button produces a dialog which \
controls the splining process, and the 'New Path' button throws out any \
previously defined keyframes to create a new path.  The 'Add Keyframe' \
button adds the current viewpoint as a keyframe for the current path.  \
The 'Preview' button allows the flyby to be previewed in the main window; \
either the entire flyby or just the keyframes can be previewed.  The \
'Save Flyby' button causes the flyby to be saved as a P3D file, for later \
rendering with any P3D renderer.\
";

/* The following flag is true if the P3D file is being played as
 * an animation.
 */
static int play_animation= 0;

/* The following flag is used to hold the knowledge that a redraw is
 * needed until the event queue is empty.
 */
static int needs_redraw= 0;

/* The following flag is true if the 'wait' cursor has been defined.
 * (Note that the renderer may also define it, without setting this
 * flag).
 */
static int wait_cursor_up= 0;

/* This widget shows the viewing distance- global so it's easy to update. */
static Widget label_widget;

Widget mil_motif_file_box( parent, title, filter, ok_callback )
Widget parent;
char *title;
char *filter;
void (*ok_callback)();
/* Called to open a file selection box */
{
  Widget file_box;
  Widget button;
  Arg args[MAX_ARGS];
  int n;
  XmString title_string= NULL, filter_string= NULL;

  ger_debug("mil_motif_file_box:");

  filter_string= XmStringCreate(filter, XmSTRING_DEFAULT_CHARSET);
  title_string= XmStringCreate(title, XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNdirMask, filter_string); n++;
  XtSetArg(args[n], XmNdialogTitle, title_string); n++;
  file_box = XmCreateFileSelectionDialog(parent, title, args, n);
  button= XmFileSelectionBoxGetChild( file_box, XmDIALOG_HELP_BUTTON);
  if (button) XtUnmanageChild(button);
  XtAddCallback(file_box, XmNokCallback, ok_callback, NULL);
  XtAddCallback(file_box, XmNcancelCallback, close_box, NULL);
  if (filter_string) XmStringFree(filter_string);
  if (title_string) XmStringFree(filter_string);
  return(file_box);
}

static void play_anim_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* Called when 'Play' is selected from the File menu */
{
  ger_debug("play_anim_callback:");

  play_animation= !play_animation;
}

static void show_file_box(w, client_data, call_data)
/* Called when 'Open' is selected from the File menu. */
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  static Widget file_box= NULL;

  ger_debug("show_file_box:");

  if (!file_box) file_box= mil_motif_file_box(w, "P3D Load", "*.p3d",
					      handle_load);

  XtManageChild(file_box);
}

static void save_view_callback(w, client_data, call_data)
/* Called when 'Save View' is selected from the File menu. */
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  static Widget file_box= NULL;

  ger_debug("save_view_callback:");

  if (!file_box) file_box= mil_motif_file_box(w, "SaveView", "*_view*.p3d",
					      handle_save_view);

  XtManageChild(file_box);
}

static void close_box(box, client_data, call_data)
/* Called when message box needs to be removed. */
Widget box;
caddr_t client_data;
caddr_t call_data;
{
  Widget shell = XtParent(box);

  ger_debug("close_box:");
  
  XtUnmanageChild(box);
}

static int real_filename(fname)
/* This function check to see if the given path name is a complete file name */
char *fname;
{
  char *last;

  ger_debug("real_filename:");

  while (*fname) last= fname++;
  if ( *last == '/' ) return(0);
  else return(1);
}

static int readable_filename(fname)
char *fname;
/* This function checks to see if the given path points to a real 
 * readable file.
 */
{
  if (access(fname,R_OK)) return(0);
  else return(1);
}

static void destroy_callback(w, target_of_destruction, call_data)
Widget w;
Widget target_of_destruction;
caddr_t call_data;
/* This destroys the calling widget */
{
  ger_debug("destroy_callback:");

  XtDestroyWidget(target_of_destruction);
}

void mil_motif_alert_box(shell,string)
/* Called to open an alert box with the given message */
char *string;
Widget shell;
{
  Widget alert_box;
  Widget button;
  Arg args[MAX_ARGS];
  int n;
  XmString title_string= NULL;
  XmString message_string= NULL;

  ger_debug("mil_motif_alert_box: string is <%s>",string);

  title_string= XmStringCreate("Alert!", XmSTRING_DEFAULT_CHARSET);
  message_string= XmStringCreateLtoR(string, XmSTRING_DEFAULT_CHARSET);
  
  n = 0;
  XtSetArg(args[n], XmNdialogTitle, title_string); n++;
  XtSetArg(args[n], XmNmessageString, message_string); n++;
  XtSetArg(args[n], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL);n++;
  XtSetArg(args[n], XmNnoResize, TRUE); n++;
  alert_box = XmCreateInformationDialog(shell, "alertbox", args, n);
  button = XmMessageBoxGetChild(alert_box, XmDIALOG_CANCEL_BUTTON);
  XtUnmanageChild(button);
  button = XmMessageBoxGetChild(alert_box, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(button);
  button = XmMessageBoxGetChild(alert_box, XmDIALOG_OK_BUTTON);
  XtAddCallback(button, XmNactivateCallback, destroy_callback, alert_box);

  if (title_string) XmStringFree(title_string);
  if (message_string) XmStringFree(message_string);
		      
  XtManageChild(alert_box);
}

void mil_motif_warning_box(parent,string,ok_callback,cancel_callback,
			   client_data)
Widget parent;
char *string;
void (*ok_callback)();
void (*cancel_callback)();
caddr_t client_data;
{
  XmString text= NULL;
  Widget dialog;
  Widget button;
  Arg args[MAX_ARGS];
  int n;
  
  ger_debug("mil_motif_warning_box:");

  text= XmStringCreateLtoR(string, XmSTRING_DEFAULT_CHARSET);
  n= 0;
  XtSetArg(args[n], XmNmessageString, text); n++;
  XtSetArg(args[n], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); n++;
  XtSetArg(args[n], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL);n++;
  dialog= XmCreateWarningDialog(parent, "warning", args, n);

  button = XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON);
  XtUnmanageChild(button);
  button = XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON);
  XtAddCallback(button, XmNactivateCallback, cancel_callback, client_data);
  XtAddCallback(button, XmNactivateCallback, destroy_callback, dialog);
  button = XmMessageBoxGetChild(dialog, XmDIALOG_OK_BUTTON);
  XtAddCallback(button, XmNactivateCallback, ok_callback, client_data);
  XtAddCallback(button, XmNactivateCallback, destroy_callback, dialog);
  
  if (text) XmStringFree(text);
  XtManageChild(dialog);
}

static void handle_load(file_box, client_data, call_data)
/* Called when 'Open' is selected in the file box. */
Widget file_box;
caddr_t client_data;
XmFileSelectionBoxCallbackStruct *call_data;
{
  Widget shell = XtParent(file_box);
  char *fname;
  XEvent event;

  ger_debug("handle_load");

  if ( !XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET,
			&fname) ) return;
  if ( real_filename(fname) && readable_filename(fname) ) {
    XtUnmanageChild(file_box);
    miu_load(fname); /* takes place on exit from routine */
    /* Run through events queued by unmanaging the dialog */
    while (XtPending()) {
      XtNextEvent( &event );
      XtDispatchEvent(&event);
    }
  }
  else mil_motif_alert_box(shell,
			   "The file selected is invalid or is not readable!");
  (void)free(fname);

  /* Put the wait cursor up while load is in progress */
  if (!wait_cursor_up) {
    XDefineCursor(display, window_id, wait_cursor);
    wait_cursor_up= 1;
    XFlush(display); 
  };
}

static int save_view_filename_ok(fname)
char *fname;
/* This routine checks for a filename which is valid for writing. */
{
  char *runner, *last;

  ger_debug("save_view_filename_ok:");

  runner= fname;
  while (*runner) last= runner++;
  if ( *last == '/' ) return(0);

  return(1);
}

static void view_save_overwrite_callback(w, data, call_data)
Widget w;
Warning_data *data;
caddr_t call_data;
/* Called when 'OK' is pressed in the confirm warning of the file box */
{
  ger_debug("view_save_overwrite_callback:");
  
  /* Pop down the file selection dialog */
  XtUnmanageChild( data->to_be_closed );

  fb_save_view(data->fname);
  (void)free(data->fname);
  (void)free(data);
}

static void view_save_cancel_callback(w, data, call_data)
Widget w;
Warning_data *data;
caddr_t call_data;
/* Called when 'cancel' is pressed in the confirm warning of the file box */
{
  ger_debug("view_save_cancel_callback:");
  
  (void)free(data->fname);
  (void)free(data);
}

static void handle_save_view(file_box, client_data, call_data)
/* Called when 'OK' is selected in the Save View file box. */
Widget file_box;
caddr_t client_data;
XmFileSelectionBoxCallbackStruct *call_data;
{
  Widget shell = XtParent(file_box);
  char *fname;
  Warning_data *data;
  char buf[256];

  ger_debug("handle_save_view");

  if ( !XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET,
			&fname) ) return;
  if ( save_view_filename_ok(fname) ) {
    if (access(fname,F_OK)) { /* file doesn't exist */
      XtUnmanageChild(file_box);
      fb_save_view(fname);
      (void)free(fname);
    }
    else {
      if ( !(data= (Warning_data *)malloc(sizeof(Warning_data))) )
	ger_fatal("handle_save_view: unable to allocate %d bytes!",
		  sizeof(Warning_data));
      data->fname= fname;
      data->to_be_closed= file_box;
      sprintf(buf,"WARNING:\n overwrite the file %s?",fname);
      mil_motif_warning_box(file_box, buf,
			    view_save_overwrite_callback,
			    view_save_cancel_callback,
			    data);
    }

  }
  else mil_motif_alert_box(shell,
			   "The file selected is invalid or is not writable!");
  (void)free(fname);
}

static void close_help_callback(w, dialog, call_data)
Widget w;
Widget dialog;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("close_help_callback:");

  XtUnmanageChild(dialog);
}

Widget mil_motif_help_box(w,title,text)
Widget w;
char *title, *text;
{
  Widget help_box;
  Widget dialogshell;
  Widget separator, text_widget, action_area, ok_button;
  XmString title_string;
  Pixel fg, bg;
  Pixmap pixmap;
  Arg args[MAX_ARGS];
  int n;

  ger_debug("mil_motif_help_box:");

  /* Create the main dialogshell and form dialog */
  n = 0;
  XtSetArg(args[n], XmNdeleteResponse, XmDESTROY); n++;
  XtSetArg(args[n], XmNtitle, title); n++;
  dialogshell= XmCreateDialogShell(w,"help_dialog",args,n);
  n = 0;
  help_box= XmCreateForm(dialogshell,"form",args,n);
  XtVaGetValues(help_box,
		XmNforeground, &fg,
		XmNbackground, &bg,
		NULL);
  
  /* Action area */
  n= 0;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNfractionBase, 7); n++;
  action_area= XmCreateForm(help_box,"action_area",args,n);
  
  /* OK button */
  title_string= XmStringCreate("OK", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'O'); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNleftPosition, 3); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 4); n++;
  ok_button= XmCreatePushButton(action_area,"Close",args,n);
  XtAddCallback(ok_button, XmNactivateCallback, close_help_callback, help_box);
  if (title_string) XmStringFree(title_string);
  XtManageChild(ok_button);
  
  XtManageChild(action_area);
  
  /* Separator */
  n= 0;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNbottomWidget, action_area); n++;
  separator= XmCreateSeparator(help_box,"separator",args,n);
  XtManageChild(separator);
  
  /* Create the scrolled field */
  n = 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNbottomWidget, separator); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
  XtSetArg(args[n], XmNeditable, False); n++;
  XtSetArg(args[n], XmNcursorPositionVisible, False); n++;
  XtSetArg(args[n], XmNvalue, text); n++;
  XtSetArg(args[n], XmNwordWrap, True); n++;
  XtSetArg(args[n], XmNscrollHorizontal, False); n++;
  XtSetArg(args[n], XmNcolumns, 67); n++;
  XtSetArg(args[n], XmNrows, 12); n++;
  text_widget= XmCreateScrolledText(help_box, "text", args, n);
  XtManageChild(text_widget);
  
  /* Set default button and return the help_box */
  XtVaSetValues(help_box, XmNdefaultButton, ok_button, NULL);

  return(help_box);
}

static void show_help_box(w, client_data, call_data)
/* Called when help button is pushed */
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  static Widget help_box= (Widget)0;
  char message[BUFSIZ];

  ger_debug("show_help_box:");

  /* Generate help message */
  if (!help_box)
    help_box= mil_motif_help_box(w,"P3D Help",help_text);
		      
  XtManageChild(help_box);
}

static void handle_distance(w, client_data, call_data)
/* Called when distance slider changes values */
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  int value;
  float scale;
  Arg args[MAX_ARGS];
  Widget label;
  int n;
  char dist_string[64];
  XmString dist_mstring= NULL;

  ger_debug("handle_distance:");

  play_animation= 0; /* stop play if it's in progress */

  label = (Widget)client_data;
  XmScaleGetValue(w, &value);
  scale = (float)value / 100;
  miu_approach_model(scale);
  XmScaleSetValue(w, 100);
  sprintf(dist_string, "Distance: %.2f", miu_viewing_distance());
  dist_mstring= XmStringCreate(dist_string, XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, dist_mstring); n++;
  XtSetValues(label, args, n);
  if (dist_mstring) XmStringFree(dist_mstring);
}

static void button_press(w, client_data, event)
Widget w;
caddr_t client_data;
XEvent *event;
{
  XWindowAttributes window_attributes;

  ger_debug("button_press:");

  play_animation= 0; /* stop play if it's in progress */

  if ( XGetWindowAttributes( display, window_id, &window_attributes ) ) {
    mouse_down.maxx = window_attributes.width;
    mouse_down.maxy = window_attributes.height;
    if (event->xbutton.button == Button1) {
      mouse_down.x = event->xbutton.x;
      mouse_down.y = window_attributes.height - event->xbutton.y;
      XDefineCursor(display, window_id, rotate_cursor);
      XFlush(display); 
    } 
    else {
      mouse_down.x = event->xbutton.x;
      mouse_down.y = window_attributes.height - event->xbutton.y;
      XDefineCursor(display, window_id, trans_cursor);
      XFlush(display); 
    } 
  }
  else ger_error("button_press: unable to get window size");
}

static void button_release(w, client_data, event)
/* Called when mouse button is released. Either a rotate or
translate is called, depending on which button is released. */
Widget w;
caddr_t client_data;
XEvent *event;
{
  XWindowAttributes window_attributes;

  ger_debug("button_release:");

  play_animation= 0; /* stop play if it's in progress */

  if ( XGetWindowAttributes( display, window_id, &window_attributes ) ) {
    mouse_up.maxx = window_attributes.width;
    mouse_up.maxy = window_attributes.height;
    if (event->xbutton.button == Button1) {
      mouse_up.x = event->xbutton.x;
      mouse_up.y = window_attributes.height - event->xbutton.y;
      XUndefineCursor(display, window_id);
      miu_rotate_model(mouse_down, mouse_up);
    } else {
      mouse_up.x = event->xbutton.x; 
      mouse_up.y = window_attributes.height - event->xbutton.y;
      XUndefineCursor(display, window_id);
      miu_translate_model(mouse_down, mouse_up);
    }
  }
  else ger_error("button_release: unable to get window size");
}

static void redraw_callback(w, client_data, event)
/* Called when a redraw is needed, for example in an expose event.
 * Rather than redraw immediately, we set a flag requesting the
 * redraw;  if that flag is set at the time of exit from mil_event
 * and if the X event queue is empty we do the redraw.  This saves
 * having multiple redraws for things like window resizes.
 */
Widget w;
caddr_t client_data;
XEvent *event;
{
  ger_debug("redraw_callback:");
#ifdef _IBMR2
#ifdef RENDERER_IS_GL
  if (ren_gl_aix_ready) {
    GlWinsetWidget(w);
    reshapeviewport();
    }
#endif
#endif
  needs_redraw= 1;
}

#ifdef RENDERER_IS_GL
static void gl_resize_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  ger_debug("gl_resize_callback:");
#ifdef _IBMR2
  if (ren_gl_aix_ready) reshapeviewport();
#else /* ifdef _IBMR2 */
  reshapeviewport();
#endif /* _IBMR2 */
  needs_redraw= 1;
}
#endif /* ifdef RENDERER_IS_GL */

void mil_setup(id_string, argc, argv )
char *id_string;
int argc;
char *argv[];
/* Initializes this user interface module. Called only once. */
{
  int captive= 0;
  Widget toplevel;
  Widget main_window;
  Widget panel_frame;
  Widget button;
  Widget canvas;
  Widget menu_bar;
  Widget menu_pane;
  Widget frame;
  Widget form;
  Widget scale;
  XmString title_string;
  XmString dist_mstring= NULL;
  Atom atomcolormapwindows;

  Arg    args[MAX_ARGS];
  int    i, n;

  static int initialized = 0;

  ger_debug("mil_setup");

  if (!initialized) { 
    initialized = 1;

    /* Check the command line for the -captive switch */
    for (i= 1; i<(argc-1); i++ ) 
      if ( !(strcmp(argv[i],"-captive")) ) captive= 1;

    toplevel = XtInitialize(argv[0], "MP3d", NULL, 0, &argc, argv);
    display = XtDisplay(toplevel);
    /* Create main window */
    n = 0;
    main_window = XmCreateMainWindow(toplevel, "main", args, n);
    XtManageChild(main_window);

    /* Create Drawing Canvas */
    n = 0;
#ifdef RENDERER_IS_GL
#ifdef _IBMR2
    canvas= XtCreateWidget("canvas", glibWidgetClass, main_window, args, n);
    XtAddCallback(canvas, XglNgconfigCallback, ren_gl_aix_init, NULL);
    XtAddCallback(canvas, XglNresizeCallback, gl_resize_callback, NULL);
    XtAddCallback(canvas, XglNexposeCallback, redraw_callback, NULL);
#else /* ifdef _IBMR2 */
    XtSetArg(args[n], GlxNglxConfig, ren_gl_glx_config); n++;
    canvas= GlxCreateMDraw(main_window, "canvas", args, n);
    XtAddCallback(canvas, GlxNresizeCallback, gl_resize_callback, NULL);
#endif /* ifdef _IBMR2 */
#else /* ifdef RENDERER_IS_GL */
    canvas = XmCreateDrawingArea(main_window, "canvas", args, n);
#endif /* ifdef RENDERER_IS_GL */
    XtAddCallback(canvas, XmNexposeCallback, redraw_callback, NULL);
    XtAddEventHandler(canvas, ButtonPressMask, FALSE, button_press, NULL);
    XtAddEventHandler(canvas, ButtonReleaseMask, FALSE, button_release, NULL);
    XtManageChild(canvas);

    /* Create menu_bar in main_window */
    n = 0;
    menu_bar = XmCreateMenuBar(main_window, "menu_bar", args, n);
    XtManageChild(menu_bar);

    n = 0;
    menu_pane = XmCreatePulldownMenu(menu_bar, "menu_pane", args, n);

    /* Create File Menu */
    title_string= XmStringCreate("File", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNsubMenuId, menu_pane); n++;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'F'); n++;
    button = XmCreateCascadeButton(menu_bar, "file", args, n);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Create Open button, dimming it if the captive switch is set */
    title_string= XmStringCreate("Open", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'O'); n++;
    button = XmCreatePushButton(menu_pane, "open", args, n);
    XtAddCallback(button, XmNactivateCallback, show_file_box, NULL);
    if (captive) XtSetSensitive(button, False);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Create Next button */
    title_string= XmStringCreate("Next", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'N'); n++;
    button = XmCreatePushButton(menu_pane, "next", args, n);
    XtAddCallback(button, XmNactivateCallback, miu_done, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Create Play button */
    title_string= XmStringCreate("Play", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'P'); n++;
    button = XmCreatePushButton(menu_pane, "play", args, n);
    XtAddCallback(button, XmNactivateCallback, play_anim_callback, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Create Save View button */
    title_string= XmStringCreate("Save View", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'S'); n++;
    button = XmCreatePushButton(menu_pane, "save_view", args, n);
    XtAddCallback(button, XmNactivateCallback, save_view_callback, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Create Exit button */
    title_string= XmStringCreate("Exit", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'E'); n++;
    button = XmCreatePushButton(menu_pane, "exit", args, n);
    XtAddCallback(button, XmNactivateCallback, miu_exit, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Add the view menu */
    mil_motif_view_setup(menu_bar);

    /* Add the flyby menu */
    mil_motif_flyby_setup(menu_bar);

    /* Create Help button */
    title_string= XmStringCreate("Help", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'H'); n++;
    button = XmCreateCascadeButton(menu_bar, "help", args, n);
    XtAddCallback(button, XmNactivateCallback, show_help_box, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(button);

    /* Tell menu bar about help button */
    n = 0;
    XtSetArg(args[n], XmNmenuHelpWidget, (XtArgVal)button); n++;
    XtSetValues(menu_bar, args, n);

    /* Create frame for scale */
    n = 0;
    XtSetArg(args[n], XmNmarginWidth, 5); n++;
    XtSetArg(args[n], XmNmarginHeight, 5); n++;
    frame = XmCreateFrame(main_window, "frame", args, n);
    XtManageChild(frame);

    /* Create form for scale */
    n = 0;
    form = XmCreateForm(frame, "form", args, n);
    XtManageChild(form);

    /* Create distance label */
    n = 0;
    dist_mstring= XmStringCreate("Distance:     ", XmSTRING_DEFAULT_CHARSET);
    XtSetArg(args[n], XmNlabelString, dist_mstring); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
    label_widget = XmCreateLabel(form, "label", args, n);
    if (dist_mstring) XmStringFree(dist_mstring);
    XtManageChild(label_widget);

    /* Create distance scale */
    n = 0;
    XtSetArg(args[n], XmNmaximum, 150); n++;
    XtSetArg(args[n], XmNminimum, 50); n++;
    XtSetArg(args[n], XmNvalue, 100); n++;
    XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_WIDGET); n++;
    XtSetArg(args[n], XmNleftWidget, label_widget); n++;
    XtSetArg(args[n], XmNrightOffset, 10); n++;
    scale = XmCreateScale(form, "Distance", args, n);
    XtAddCallback(scale, XmNvalueChangedCallback, handle_distance, 
		  label_widget);
    XtManageChild(scale);

    XmMainWindowSetAreas(main_window, menu_bar, NULL,
	  		 frame, NULL, canvas);

    trans_cursor = XCreateFontCursor(display, XC_fleur);
    rotate_cursor = XCreateFontCursor(display, XC_exchange);
    wait_cursor = XCreateFontCursor(display, XC_watch);

    XtRealizeWidget(toplevel);
    window_id = XtWindow(canvas); 

    /* A couple of globals needed by the xws device driver */
    top_window_id= window_id; /* to put color map in right place */
    display_ptr= display;

    /* Set top window property so that the window manager will install
     * the canvas' colormap, in case the device driver sets one.
     */
    atomcolormapwindows= XInternAtom(display, "WM_COLORMAP_WINDOWS", False);
    XChangeProperty( display, top_window_id, atomcolormapwindows, 
		    XA_WINDOW, 32, PropModeAppend, 
		    (unsigned char *)&top_window_id, 1 );

  }

  /* Put the wait cursor up while load is in progress */
  if (!wait_cursor_up) {
    XDefineCursor(display, window_id, wait_cursor);
    wait_cursor_up= 1;
  };
  XFlush(display); 
}

void mil_update()
/* This routine updates the controls */
{
  int n;
  Arg args[MAX_ARGS];
  char dist_string[64];
  XmString dist_mstring= NULL;

  ger_debug("mil_update");

  /* Update the viewing distance label */
  sprintf(dist_string, "Distance: %.2f", miu_viewing_distance());
  dist_mstring= XmStringCreate(dist_string, XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, dist_mstring); n++;
  XtSetValues(label_widget, args, n);
  if (dist_mstring) XmStringFree(dist_mstring);
}

void mil_event()
/* This routine carries out one pass through the event loop */
{
  XEvent event;

  ger_debug("mil_event");
  
  if (wait_cursor_up) {
      XUndefineCursor(display, window_id);
      wait_cursor_up= 0;
    }
  XtNextEvent( &event );
  XtDispatchEvent(&event);
  if ( !(XtPending()) ) {
    if (needs_redraw) {
      miu_redraw();
      needs_redraw= 0;
    }
    else if (play_animation) miu_done(); /* advance to next model in file */
  }
}

void mil_shutdown()
/* This routine shuts down this user interface module */
{
  ger_debug("mil_shutdown");
}
