/****************************************************************************
 *                                                                          *
 *	painter_util.c                                                      *
 *      Copyright 1989, Pittsburgh Supercomputing Center                    *
 *                      All Rights Reserved                                 *
 *			Author Chris Nuuja                                  *
 *                                                                          *
 ****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include "painter_vars.h"
#include "painter.h"



/*		GLOBAL VARIABLES USED FOR BUFFERS		*/

Hash_cell **hashtable;   /*  hash table to match primitive idnum to objects */

ren_colortype *ColorBuffer,*DColorBuffer; /* holds colors.D is for disposable */
float	       *CoordBuffer,*DCoordBuffer;
int CoordBuffIndex,MaxCoordIndex,DCoordIndex,MaxDCoordIndex;



/*				FUNCTION  DECLARATION			*/


/* Calculate polygon normal */
void calc_normal(polygon,trans,result)
int polygon;
float *trans;
ren_vectortype *result;
{
  int x_index, y_index, z_index;
  float v1x, v1y, v1z, v2x, v2y, v2z, nx, ny, nz;
  register float *rtrans= trans;

  if (PolyBuffer[polygon].numcoords < 3) {
    fprintf(stderr,"ERROR, calc_normal passed a polygon with %d coords\n",
	    PolyBuffer[polygon].numcoords);
    result->x= result->y= result->z= 0.0;
  }

  x_index= PolyBuffer[polygon].x_index;
  y_index= PolyBuffer[polygon].y_index;
  z_index= PolyBuffer[polygon].z_index;

  /* Find edge vector components in model coordinate system */
  v1x= CoordBuffer[x_index+1] - CoordBuffer[x_index];
  v2x= CoordBuffer[x_index+2] - CoordBuffer[x_index+1];
  v1y= CoordBuffer[y_index+1] - CoordBuffer[y_index];
  v2y= CoordBuffer[y_index+2] - CoordBuffer[y_index+1];
  v1z= CoordBuffer[z_index+1] - CoordBuffer[z_index];
  v2z= CoordBuffer[z_index+2] - CoordBuffer[z_index+1];

  /* Find normal in model coordinate system */
  nx= v1y*v2z - v2y*v1z;
  ny= v1z*v2x - v2z*v1x;
  nz= v1x*v2y - v2x*v1y;

  /* Transform normal to world coordinates */
  result->x = rtrans[0]*nx + rtrans[4]*ny + rtrans[8]*nz;
  result->y = rtrans[1]*nx + rtrans[5]*ny + rtrans[9]*nz;
  result->z = rtrans[2]*nx + rtrans[6]*ny + rtrans[10]*nz;
}

float vector_length(vector)
register ren_vectortype *vector;
{
	float result;
	
	result = (float) sqrt ( ((vector->x * vector->x) + 
				 (vector->y * vector->y) +
			         (vector->z * vector->z)) );
	if (result < 0.0)
		result = 0.0 - result;

	return(result);
}

float dotproduct(v1,v2)
register ren_vectortype *v1,*v2;
{
	float result;

	result = (v1->x * v2->x) + (v1->y * v2->y) + (v1->z * v2->z);
	return(result);
}


void append3dMatrices(oldMatrix,newMatrix)
float *oldMatrix,*newMatrix;
{
	int row,column,i;
	float temp[16];

	for (i=0;i<16;i++)	
		{
		temp[i]=oldMatrix[i];
		oldMatrix[i]=0;
		}
	for (row = 0;row<4;row++)
		for (column= 0;column<4;column++)
			for (i=0;i<4;i++)
				oldMatrix[(4*row)+column] += temp[(4*row)+i]*
				   newMatrix[(4*i)+column];
	

}

ren_vectortype *vector_matrix_mult3d(vector,matrix)
ren_vectortype *vector;
float *matrix;
{
	ren_vectortype *new_vector;

	new_vector = makevector_rec(1.0,1.0,1.0);
	new_vector->x = matrix[0]*vector->x + matrix[4]*vector->y + matrix[8]*vector->z;
	new_vector->y = matrix[1]*vector->x + matrix[5]*vector->y + matrix[9]*vector->z;	
	new_vector->z = matrix[2]*vector->x + matrix[6]*vector->y + matrix[10]*vector->z;	

	return(new_vector);
}

/*
   Global Data Used: NONE
   Expl: given a normal, <oldVector>, and a transformation matrix, <aTrans>
	 this routine returns the vector transformed by that matrix.
   NOTE: not used
*/
ren_vectortype *trans_norm(aTrans,oldVector)
float *aTrans;
ren_vectortype *oldVector;
{
	float x,y,z;
	ren_vectortype *newVector;

	x = oldVector->x;
	y = oldVector->y;
	z = oldVector->z;

	newVector = (ren_vectortype *) malloc(sizeof(ren_vectortype));
	newVector->x = aTrans[0]*x + aTrans[4]*y + aTrans[8]*z;
	newVector->y = aTrans[1]*x + aTrans[5]*y + aTrans[9]*z;
	newVector->z = aTrans[2]*x + aTrans[6]*y + aTrans[10]*z;
	return(newVector);
}
/*
   Global Data Used:NONE
   Expl: debug printing routine 
*/
void print_poly(poly)
ren_polytype *poly;
{
/*
	int i;

	printf("rendering poly with %d coords",poly->numcoords);
	for (i=0;i<poly->numcoords;i++)
		printf("C:(%f %f %f)",poly->xcoords[i],
					   poly->ycoords[i],
					   poly->zcoords[i]);
	printf("Color:(%f %f %f)",poly->color.r,poly->color.g,
	   poly->color.b);
	printf("Normal:(%f %f %f)",poly->normal.x,poly->normal.y,
	   poly->normal.z);
*/
}

/*
   Global Data Used: NONE
   Expl:debuf printing routine
*/
void print_matrix(matrix)
float *matrix;
{
	int i,j;
	
	printf("Matrix::\n");
	for (i=0;i<4;i++)
		{
		printf(" ( ");
		for (j=0;j<4;j++)
			printf(" %f ",matrix[4*i+j]);
		printf(" )\n ");
		}
	printf("\n");	
}

int new_CoordIndex(numcoords)
int numcoords;
{
	int result;

	if (CoordBuffIndex+numcoords >= MaxCoordIndex)
		{
		MaxCoordIndex = 2 * MaxCoordIndex;
		CoordBuffer = (float *)
			realloc(CoordBuffer,MaxCoordIndex*sizeof(float));
		if (!CoordBuffer)
			{
			printf("ERROR, Could not reallocate %d Coordinates\n",
				MaxCoordIndex);
			exit(1);
			}
		return(new_CoordIndex(numcoords));
		}
	else
		{
		result = CoordBuffIndex;
		CoordBuffIndex += numcoords;
		return(result);
		}
}

int new_DepthCoords(numcoords)
int numcoords;
{
	int result;

	if (DCoordIndex+numcoords >= MaxDCoordIndex)
		{
		MaxDCoordIndex = 2 * MaxDCoordIndex;
		DCoordBuffer = (float *)
			realloc(DCoordBuffer,MaxDCoordIndex*sizeof(float));
		if (!DCoordBuffer)
			{
			printf("ERROR, Could not reallocate %d Coordinates\n",
				numcoords);
			exit(1);
			}
		return(new_DepthCoords(numcoords));
		}
	else
		{
		result = DCoordIndex;
		DCoordIndex += numcoords;
		return(result);
		}
}

int new_ColorIndex()
{
	if (ColorCount >= MaxColorCount)
		{
		MaxColorCount = 2 * MaxColorCount;
		ColorBuffer  = (ren_colortype *)
		      realloc(ColorBuffer,MaxColorCount*sizeof(ren_colortype));
		if (!ColorBuffer)
			{
			printf("ERROR, Could not reallocate %d Colors\n",
				MaxColorCount);
			exit(1);
			};
		return(new_ColorIndex());
		}
	else
		return(ColorCount++);
}

int new_DepthColor()
{
	if (DColorCount >= MaxDColorCount)
		{
		MaxDColorCount = 2 * MaxDColorCount;
		DColorBuffer  = (ren_colortype *)
		      realloc(DColorBuffer,
			      MaxDColorCount*sizeof(ren_colortype));
		if (!DColorBuffer)
			{
			printf("ERROR, Could not reallocate %d Colors\n",
				MaxColorCount);
			exit(1);
			};
		return(new_DepthColor());
		}
	else
		return(DColorCount++);
}

int new_ObjectIndex()
{
	if (ObjectCount+1 == MaxObjects)
		{
		MaxObjects = 2 * MaxObjects;
		ObjectBuffer = (ren_objecttype *)
			realloc(ObjectBuffer,MaxObjects*sizeof(ren_objecttype));
		if (!ObjectBuffer)
			{
			printf("ERROR, Could not reallocate %d Objects\n",
				MaxObjects);
			exit(1);
			};
		return(new_ObjectIndex());
		}
	else
		return(ObjectCount++);
}

int new_PolyIndex()
{
	if (PolyCount+1 == MaxPolyCount)
		{
		MaxPolyCount = 2 * MaxPolyCount;
		PolyBuffer = (ren_polytype *)
			realloc(PolyBuffer,MaxPolyCount*sizeof(ren_polytype));
		if (!PolyBuffer)
			{
			printf("ERROR, Could not reallocate %d Polygons\n",
				MaxPolyCount);
			exit(1);
			};
		return(new_PolyIndex());
		}
	else
		return(PolyCount++);
}

int new_DepthPoly()
{
	if (DepthPolyCount+1 == MaxDepthPoly)
		{
		MaxDepthPoly = 2 * MaxDepthPoly;
		DPolyBuffer = (ren_polytype *)
			realloc(DPolyBuffer,MaxDepthPoly*sizeof(ren_polytype));
		if (!DPolyBuffer)
			{
			printf("ERROR, Could not reallocate %d Polygons\n",
				MaxDepthPoly);
			exit(1);
			};
		DepthBuffer = (depthtable_rec *)
		       realloc( DepthBuffer,
				MaxDepthPoly*sizeof(depthtable_rec) );
		return(new_DepthPoly());
		}
	else
		return(DepthPolyCount++);
}

int new_PolyIndicies(size)
{
	int result;

	if (PolyCount+size >= MaxPolyCount)
		{
		MaxPolyCount = 2 * MaxPolyCount;
		PolyBuffer = (ren_polytype *)
			realloc(PolyBuffer,MaxPolyCount*sizeof(ren_polytype));
		if (!PolyBuffer)
			{
			printf("ERROR, Could not reallocate %d Polys\n",
				MaxPolyCount);
			exit(1);
			};
		return(new_PolyIndicies(size));
		}
	else
		{
		result = PolyCount;
		PolyCount += size;
		return(result);
		}
}

int new_LightIndex()
{
	if (LightCount+1 >= MaxLightCount)
		{
		MaxLightCount = 2 * MaxLightCount;
		LightPosBuffer = (ren_pointtype *)
			realloc(LightPosBuffer,MaxLightCount*sizeof(ren_pointtype));
		if (!LightPosBuffer)
			{
			printf("ERROR, Could not reallocate %d Light Positions\n",
				MaxLightCount);
			exit(1);
			}
		LightColorBuffer = (ren_colortype *)
			realloc(LightColorBuffer,MaxLightCount*sizeof(ren_colortype));
		if (!LightColorBuffer)
			{
			printf("ERROR, Could not reallocate %d Light Colors\n",
				MaxLightCount);
			exit(1);
			}
		return(new_LightIndex());
		}
	else
		return(LightCount++);
}


/*
   Global Data Used:NONE
   Expl:   this allocates <numcoords> of floats for <poly>'s x_index y_index 
	   and z_index fields
*/
void fillpoly_rec(poly,numcoords,type)
int poly;
int numcoords;
primtype type;
{
	ren_polytype *newpoly;

	newpoly = PolyBuffer+poly;
	newpoly->x_index = new_CoordIndex(numcoords);
	newpoly->y_index = new_CoordIndex(numcoords);
	newpoly->z_index = new_CoordIndex(numcoords);

	newpoly->numcoords = numcoords; 
	newpoly->type = type;
}

void fillDpoly_rec(poly,numcoords,type)
int poly;
int numcoords;
primtype type;
{
	ren_polytype *newpoly;

	newpoly = DPolyBuffer+poly;
	newpoly->x_index = new_DepthCoords(numcoords);
	newpoly->y_index = new_DepthCoords(numcoords);
	newpoly->z_index = new_DepthCoords(numcoords);

	newpoly->numcoords = numcoords; 
	newpoly->type = type;
}

/*
   Global Data Used:NONE
   Expl: returns a pointer to a poly_rec which has <numcoords> of floats 
	 allocated to it's x_index, y_index, and z_index fields.
*/
ren_polytype *makepoly_rec(numcoords,type)
int numcoords;
primtype type;
{
	ren_polytype *poly;
	int polyindex;

	ger_debug("make_polyrec :numcoords %d",numcoords);
	/* Must call new_PolyIndex() BEFORE adding result to PolyBuffer  */
	/* because PolyBuffer might change in the call to new_PolyIndex() */
	polyindex = new_PolyIndex();
	poly = PolyBuffer + polyindex;
	poly->x_index = new_CoordIndex(numcoords);
	poly->y_index = new_CoordIndex(numcoords);
	poly->z_index = new_CoordIndex(numcoords);

	poly->numcoords=numcoords; 
	poly->type = type;
	return(poly);
}

/*
   Global Data Used:NONE
   Expl:  returns a pointer to a vector record with the input values.
*/
ren_vectortype *makevector_rec(x,y,z)
float x,y,z;
{
	ren_vectortype *vector;

	ger_debug("makevector_rec (%f %f %f)",x,y,z);
	vector = (ren_vectortype *) malloc(sizeof(ren_vectortype));
	vector->x = x;
	vector->y = y;
	vector->z = z;
	return(vector);
}

/*
   Global Data Used:NONE
   Expl:  returns a pointer to a point record.  You might as well make an
	  instance of ren_vectortype.
*/
ren_pointtype *makepoint_rec(x,y,z)
float x,y,z;
{
	ren_pointtype *point;

	ger_debug("makepoint_rec (%f %f %f)",x,y,z);
	point = (ren_pointtype *) malloc(sizeof(ren_pointtype));
	point->x = x;
	point->y = y;
	point->z = z;
	return(point);
}

/*
   Global Data Used:NONE
   Expl:  this returns a pointer to a vector of unit length pointing from 
	  <pnt1> to <pnt2>
*/
ren_vectortype *make_directionvector(pnt1,pnt2)
ren_pointtype *pnt1,*pnt2;
{
	ren_vectortype *vector;
	float length;

	ger_debug("make_directionvector (%f %f %f)",
		   pnt1->x,pnt1->y,pnt1->z);
	vector = (ren_vectortype *) malloc(sizeof(ren_vectortype));
	vector->x = pnt1->x - pnt2->x;
	vector->y = pnt1->y - pnt2->y;
	vector->z = pnt1->z - pnt2->z;
	length = vector_length(vector);
	if (length == 0.0)
		length = HUGE;
	vector->x /= length;
	vector->y /= length;
	vector->z /= length;
	
	return(vector);
}

/*
   Global Data Used:NONE
   Expl:  returns a pointer to a color record with the input values.
*/
int makeDcolor_rec( r,g,b,a )
float r,g,b,a;
{
	int new_color;

	ger_debug("make_color (%f %f %f %f)",r,g,b,a);
	new_color = new_DepthColor();
	DColorBuffer[new_color].r = r;
	DColorBuffer[new_color].g = g;
	DColorBuffer[new_color].b = b;
	DColorBuffer[new_color].a = a;
	return(new_color);
}

int makecolor_rec( r,g,b,a )
float r,g,b,a;
{
	int new_color;

	ger_debug("make_color (%f %f %f %f)",r,g,b,a);
	new_color = new_ColorIndex();
	ColorBuffer[new_color].r = r;
	ColorBuffer[new_color].g = g;
	ColorBuffer[new_color].b = b;
	ColorBuffer[new_color].a = a;
	return(new_color);
}
/*
   Global Data Used: NONE
   Expl:  frees all the things pointed to by an instance of ren_polyrec;
*/
void free_polyrec(poly)
ren_polytype *poly;
{
	/*
	free(poly->x_index);
	free(poly->y_index);
	free(poly->z_index);
	if (poly->color)
		free(poly->color);
	if (poly->normal)
		free(poly->normal);
	*/
}


/********** Hash table implementation follows **********************/
/* What the int is being hashed to, and failure return code.
   failed lookups return (Hash_result)BAD_HASH_LOOKUP 
   multiple defines of same value return (Hash_result)BAD_HASH_ADD
   */


Hash_result hash_lkup(val)
int val;
/* Hash table lookup */
{
  int id;
  Hash_cell *thiscell;

  id= abs( val % HASHSIZE );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val== val ) return(thiscell->result);
  return( (Hash_result) BAD_HASH_LOOKUP );
}

Hash_result hash_add( val, result )
int val;
Hash_result result;
/* Add a value-result pair to hash table */
{
  int id;
  Hash_cell *thiscell;

  if ( hash_lkup(val) != (Hash_result)BAD_HASH_LOOKUP )
    return( (Hash_result)BAD_HASH_ADD );

  id= abs( val % HASHSIZE );
  if (!(thiscell= (Hash_cell *)malloc( sizeof(Hash_cell) )))
	ger_fatal("hash_add: unable to allocate %d bytes!\n",
		sizeof(Hash_cell));
  thiscell->val= val;
  thiscell->result= result;
  thiscell->next= hashtable[id];
  if (hashtable[id]) hashtable[id]->last= thiscell;
  thiscell->last= NULL;
  hashtable[id]= thiscell;

  return( result );
}

void hash_free(val)
int val;
/* Free a value-result pair from the hash table.  Does nothing if the
 * value is not found.
 */
{
  int id;
  Hash_cell *thiscell;

  id= abs( val % HASHSIZE );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val==val ) {
      if (thiscell->next) thiscell->next->last= thiscell->last;
      if (thiscell->last) thiscell->last->next= thiscell->next;
      else hashtable[id]= thiscell->next;
      free( (char *)thiscell );
    }
}
/****************End of hash table implementation*******************/
