/*
 * raytrace.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: raytrace.c,v 1.1 1991/01/04 18:35:14 welling Exp $
 *
 * $Log: raytrace.c,v $
 * Revision 1.1  1991/01/04  18:35:14  welling
 * Initial revision
 *
 * Revision 3.0.1.6  90/02/12  13:29:09  craig
 * patch4: Changes to reflect new variable names in linda code.
 * 
 * Revision 3.0.1.5  89/12/07  22:55:25  craig
 * patch2: Renamed utime and stime to avoid name clashes.
 * 
 * Revision 3.0.1.4  89/12/02  16:41:46  craig
 * patch2: Added ReportFreq, reporting of total CPU & split times.
 * 
 * Revision 3.0.1.3  89/12/02  14:42:37  craig
 * patch2: Added call to focus_blur_ray() to support depth of field.
 * 
 * Revision 3.0.1.2  89/11/16  20:35:30  craig
 * patch1: ShadeRay is now called on background rays.
 * 
 * Revision 3.0.1.1  89/11/16  18:27:38  craig
 * patch1: Workers now report statistics to supervisor.
 * patch1: Linda syntax is more up-to-date.
 * 
 * Revision 3.0  89/10/27  02:06:02  craig
 * Baseline for first official release.
 * 
 */

/*
 * This module could use some work.  In particular, a better antialiasing
 * scheme would be nice.
 */

#include <math.h>
#include <stdio.h>
#include "typedefs.h"
#include "constants.h"
#include "funcdefs.h"
#include "raytrace.h"

ray_Color *pixel_buf[2], background;	/* Point buffer, background color */
ray_Color *out_buf;			/* Output pixel buffer */
int	pixel_div = UNSET;		/* max times to subdivide pixel */
int	JitSamples = UNSET;		/* sqrt of # jittered samples/pixel */
double	JitterWeight;			/* 1. / (Total Samples Taken) */
int	Jittered;			/* Use distributed raytracing */
int	*SampleNumbers;
int	SampleNumber;
int	StartLine = UNSET;
int	ReportFreq;			/* Frequency of status report */
double	RedContrast = UNSET, GreenContrast = UNSET, BlueContrast = UNSET;
double	SampleSpacing;

extern	int Xres, Yres;
extern	ray_Vector eyep, firstray;
ray_Ray	TopRay;				/* Top-level ray. */

raytrace()
{
	/*
	 * The top-level ray TopRay always has as its origin the
	 * eye position and as its medium NULL, indicating that it
	 * is passing through a medium with index of refraction
	 * equal to DefIndex.
	 */
	TopRay.pos = eyep;
	TopRay.media = (ray_SurfaceList *)0;
	TopRay.shadow = FALSE;

	out_buf = (ray_Color *)Malloc(Xres * sizeof(ray_Color));

	if (Jittered)
		distributed_trace();
	else
		adaptive_trace();
}

/*
 * Raytrace an image using "distributed" raytracing.
 */
distributed_trace()
{
	register int y;
	extern unsigned long EyeRays;
	double usertime, systime, lasttime;

	switch (JitSamples) {
		case 1:
			SampleNumbers = OneSample;
			break;
		case 2:
			SampleNumbers = TwoSamples;
			break;
		case 3:
			SampleNumbers = ThreeSamples;
			break;
		case 4:
			SampleNumbers = FourSamples;
			break;
		case 5:
			SampleNumbers = FiveSamples;
			break;
		default:
			fprintf(stderr,"Sorry, %d rays/pixel not supported.\n",
				JitSamples*JitSamples);
			exit(2);
	}

	JitterWeight= 1. / (JitSamples * JitSamples);
	SampleSpacing = 1. / JitSamples;

	/*
	 * Trace each scanline, writing results to output file.
	 */
	lasttime = 0;
	for (y = StartLine; y >= 0; y--) {
		trace_jit_line(y, out_buf);
		outline(out_buf);
		if (y % ReportFreq == 0) {
			fprintf(stderr,"Finished line %d (%ld rays",y,
							EyeRays);
			fprintf(stderr,")\n");
			fflush(stderr);
		}
	}
}

trace_jit_line(line, buf)
int line;
ray_Color *buf;
{
	register int x;
	for (x = 0; x < Xres; x++)
		trace_jit_pixel(x, line, buf++);
}

trace_jit_pixel(xp, yp, color)
int xp, yp;
ray_Color *color;
{
	ray_Color tmp;
	double x, y, xpos, ypos;
	int i, j, index;

	ypos = (double)yp - 0.5;
	color->r = color->g = color->b = 0.;
	index = 0;
	for (i = 0; i < JitSamples; i++, ypos += SampleSpacing) {
		xpos = (double)xp - 0.5;
		for (j = 0; j < JitSamples; j++, xpos += SampleSpacing) {
			x = xpos + nrand() * SampleSpacing;
			y = ypos + nrand() * SampleSpacing;
			SampleNumber = SampleNumbers[index++];
			trace_point(x, y, &tmp);
			color->r += tmp.r;
			color->g += tmp.g;
			color->b += tmp.b;
		}
	}
	ScaleColor(JitterWeight, *color, color);
}

/*
 * Raytrace an image using adaptive supersampling to perform antialising.
 */
adaptive_trace()
{
	register int line;
	extern unsigned long EyeRays;
	double usertime, systime, lasttime;

	/*
	 * In the adaptive supersampling case, Jitter, JitterWeight,
	 * and SampleSpacing refer are used in sampling extended
	 * light sources.  JitterWeight has a -4 in the denominator
	 * due to the fact that we don't sample the corners of extended
	 * sources when performing adaptive supersampling.
	 */
	JitterWeight = 1. / (JitSamples * JitSamples - 4);
	SampleSpacing = 1. / JitSamples;

	pixel_buf[0] = (ray_Color *)Malloc(sizeof(ray_Color) *
				(unsigned)(Xres + 1));
	pixel_buf[1] = (ray_Color *)Malloc(sizeof(ray_Color) *
				(unsigned)(Xres + 1));
	/*
	 * Minimum pixel square size.
	 */
	Minsquare = 1./pow(2.,(double)pixel_div);
	/*
	 * At any time, there can be a maximum of 3 * pixel_div + 1
	 * pixel squares on the stack.
	 */
	SquareStack = (pixel_square *)Malloc((unsigned)(3 * pixel_div + 1) *
				sizeof(pixel_square));
	/*
	 * A pixel is treated as a square through whose corners rays
	 * are traced.  If the color values at the corners are
	 * "too different" (as measured by pixel_ok()), the square is
	 * divided into four squares (tracing 5 additional rays)
	 * and the process is repeated on the four new squares.
	 * The color assigned to a square is the average of the 4 corners.
	 * Note that this means that adjacent super-sampled pixels
	 * cause the same ray to be traced more than once.
	 * This scheme is adequate but far from wonderful.
	 */
	line = StartLine + 1;
	trace_line(line, &pixel_buf[line & 01][0]);
	lasttime = 0;
	/*
	 * Work bottom-to-top, as that's the way Utah-raster wants to
	 * operate.
	 */
	for(line--;line >= 0;line--) {
		trace_line(line, &pixel_buf[line & 01][0]);
		subdivide_line(line, pixel_buf[line & 01],
				     pixel_buf[(line+1) & 01],
				     out_buf);
		outline(out_buf);
		if(line % ReportFreq == 0) {
			fprintf(stderr,"Finished line %d (%ld rays",line,
								EyeRays);
			fprintf(stderr,")\n");
			fflush(stderr);
		}
	}
}

/*
 * Trace a line of sample points along "line".
 */
trace_line(line, buf)
int line;
ray_Color *buf;
{
	register int x;
	/*
	 * We need to trace Xres + 1 rays.
	 */
	for(x = 0; x <= Xres;x++)
		trace_point((double)x, (double)line, buf + x);
}

/*
 * Given the two arrays of sample points which define the upper and
 * lower edges of all the pixel squares in line "y," push each
 * square in turn on the pixelsquare stack and determine a color value
 * for the pixel by calling subdivide_square().
 */
subdivide_line(y, upper, lower, buf)
int y;
ray_Color *upper, *lower, *buf;
{
	register int x;

	/*
	 * Upper is the array of
	 * sample values which define the "upper" part (corners) of this
	 * row, while lower are the "lower" corners.  For the
	 * next (lower) row, the current "upper" becomes "lower".
	 */
	for(x = 0; x < Xres;x++) {
		SquareStack[0].x = (float)x;
		SquareStack[0].y = (float)y;
		SquareStack[0].size = 1.0;
		SquareStack[0].ul = upper[x];
		SquareStack[0].ur = upper[x+1];
		SquareStack[0].ll = lower[x];
		SquareStack[0].lr = lower[x+1];
		subdivide_square(&buf[x]);
	}
}

/*
 * Bleck, but it saves us a function call and keeps the code much cleaner.
 */
#define push_square(u,v,s,a,b,c,d) {\
			Stackp->x = u; \
			Stackp->y = v; \
			Stackp->size = s; \
			Stackp->ul = a; \
			Stackp->ur = b; \
			Stackp->ll = c; \
			Stackp->lr = d; \
			Stackp++;}

/*
 * Subdivide a pixel square.
 */
subdivide_square(color)
ray_Color *color;
{
	register pixel_square *Stackp;
	register float x, y;
	float size, halfsize;
	double avfact, xdelta, ydelta;
	ray_Color ul, ur, ll, lr, u, d, l, r, c;
	extern unsigned long SuperSampled;

	color->r = color->g = color->b = 0.;
	Stackp = SquareStack + 1;

	do {
		Stackp--;
		size = Stackp->size;
		ul = Stackp->ul;
		ur = Stackp->ur;
		ll = Stackp->ll;
		lr = Stackp->lr;

		if(size <= Minsquare || pixel_ok(&ul,&ur,&ll,&lr)) {
			/*
			 * The square is either the smallest allowed, or
			 * the four corners of the square are similar.
			 * Average the four corners (weighted by the
			 * size of the square) to get this square's
			 * contribution to the whole pixel's color.
			 */
			avfact = (size * size) * 0.25;
			color->r += (ul.r + ur.r + ll.r + lr.r) * avfact;
			color->g += (ul.g + ur.g + ll.g + lr.g) * avfact;
			color->b += (ul.b + ur.b + ll.b + lr.b) * avfact;
			continue;
		}
		/*
		 * Subdivide into four squares -- trace 5 additional
		 * rays and push the appropriate squares on the pixelsquare
		 * stack.
		 */
		x = Stackp->x;
		y = Stackp->y;
		halfsize = size * 0.5;
		xdelta = (double)(x + halfsize);
		ydelta = (double)(y + halfsize);
		trace_point(xdelta, (double)y, &u);
		trace_point((double)x, ydelta, &l);
		trace_point(xdelta, ydelta, &c);
		trace_point((double)(x + size),ydelta, &r);
		trace_point(xdelta, (double)(y + size), &d);
		if(size == 1.)
			SuperSampled++;
		push_square(x, y, halfsize, ul, u, l, c);
		push_square((float)xdelta, y, halfsize, u, ur, c, r);
		push_square(x, (float)ydelta, halfsize, l, c, ll, d);
		push_square((float)xdelta, (float)ydelta, halfsize,
					c, r, d, lr);
	} while (Stackp != SquareStack);
}

/*
 * Trace a ray through x, y on the screen, placing the result in "color."
 */
trace_point(x, y, color)
double x, y;
ray_Color *color;
{
	double dist;
	ray_HitInfo hitinfo;
	extern ray_Vector scrnx, scrny;
	extern double aperture;
	extern unsigned long EyeRays;
	extern double TraceRay();

	/*
	 * Calculate ray direction.
	 */
	EyeRays++;
	TopRay.dir.x = firstray.x + x*scrnx.x - y*scrny.x;
	TopRay.dir.y = firstray.y + x*scrnx.y - y*scrny.y;
	TopRay.dir.z = firstray.z + x*scrnx.z - y*scrny.z;

	(void)normalize(&TopRay.dir);

	if (aperture > 0.0) {
		/*
		 * If the aperture is open, adjust the initial ray
		 * to account for depth of field.  
		 */
		focus_blur_ray(&TopRay);
	}

	/*
	 * Do the actual ray trace.
	 */
	dist = TraceRay((ray_Primitive *)NULL, &TopRay, &hitinfo);
	ShadeRay(&hitinfo, &TopRay, dist, &background, color, 1.0);
}

/*
 * Return TRUE if this pixel is okay and doesn't need to be supersampled,
 * FALSE otherwise.
 */
pixel_ok(w,x,y,z)
ray_Color *w, *x, *y, *z;
{
	double rmax, rmin, gmax, gmin, bmax, bmin;
	double rsum, gsum, bsum;

	/*
	 * Find min & max R, G, & B.
	 */
	rmax = max(w->r, max(x->r, max(y->r, z->r)));
	rmin = min(w->r, min(x->r, min(y->r, z->r)));
	gmax = max(w->g, max(x->g, max(y->g, z->g)));
	gmin = min(w->g, min(x->g, min(y->g, z->g)));
	bmax = max(w->b, max(x->b, max(y->b, z->b)));
	bmin = min(w->b, min(x->b, min(y->b, z->b)));

	/*
	 * Contrast is defined as (Max - Min) / (Max + Min) for each
	 * of RG&B.  If any of these values is greater than the maximum
	 * allowed, we have to supersample.
	 */
	rsum = rmax + rmin;
	gsum = gmax + gmin;
	bsum = bmax + bmin;
	if ((rsum == 0. || (rmax - rmin) / rsum < RedContrast) &&
	    (gsum == 0. || (bmax - bmin) / gsum < BlueContrast) &&
	    (bsum == 0. || (gmax - gmin) / bsum < GreenContrast))
		return TRUE;

	return FALSE;
}

