/******************************************************************************
 * rendermn_ren.c
 * Author Quan Le
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

/*
  This renderer renders images using the RenderMan interface.
*/

#include <stdio.h>
#include <math.h>
#include <ri.h>
# include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "assist.h"

#define DEFAULT_X_RES 512
#define DEFAULT_Y_RES 512
#define DEFAULT_FRAME_ID 0
#define DEFAULT_SHADING_RATE 1.0
#define BINARY_FORMAT "binary"
#define ASCII_FORMAT "ascii"
#define DEFAULT_FORMAT BINARY_FORMAT

/* Notes:
   -Current method of LH-RH conversion: scale(-1,1,1) right after beginframe
   -Why do some animations produce 'bad matrix' errors?
   -A nicer way to handle text would be good.
   -Add some hints to control antialiasing parameters?
   -This renderer is dependent on the calling sequence of tui.c
*/

/* Symbols defined by setup and used in parsing attribute lists */
static Symbol depth_cue_symbol, color_symbol, backcull_symbol, 
  lighting_model_symbol, text_height_symbol, material_symbol,
  line_diameter_symbol, point_diameter_symbol;

static Symbol x_res_symbol, y_res_symbol, text_font_symbol;

static Symbol text_stroke_width_symbol, text_thickness_symbol,
  shading_rate_symbol, binary_rib_symbol, 
  initial_frame_symbol;

/* An enumerated type and storage to indicate whether a traversal is
 * for rendering or lighting.
 */
enum traversal_type { RENDER, LIGHTING, INVALID };
static enum traversal_type current_traversal= INVALID;

/* Some variables controlling output file generation */
static char *filename;
static char *rib_format[1];
static int frame_id = DEFAULT_FRAME_ID;
static float shading_rate = DEFAULT_SHADING_RATE;
static int x_res = DEFAULT_X_RES;
static int y_res = DEFAULT_Y_RES;

/* Corners and color of the background rectangle */
#define BACKGROUND_SIZE_MARGIN 1.1
#define BACKGROUND_CLIP_MARGIN 0.99999
static RtPoint back_rect[4];
static RtColor backgroundclr;

/*
  	get_renderman_color()

	This routine returns a renderman color data type from
	the argument clr.
*/
static RtFloat *get_renderman_color(clr)
Color clr;
{
  RtFloat *result;

  result = (RtFloat *) malloc(3*sizeof(RtFloat));
  if(!result)
    ger_fatal("get_renderman_color: unbable to allocate 3 RtFloats!");
  
  result[0]= (RtFloat) (color_red(clr));
  result[1]= (RtFloat) (color_green(clr));
  result[2]= (RtFloat) (color_blue(clr));

  return result;
}	

/*
  	get_renderman_opacity()

	This routine returns a renderman color alpha type given
	the argument clr.
*/
static RtFloat *get_renderman_opacity(clr)
Color clr;
{
  RtFloat *result;

  result = (RtFloat *) malloc(3*sizeof(RtFloat));
  if(!result)
    ger_fatal("get_renderman_opacity: unbable to allocate 3 RtFloats!");
  
  result[0]=result[1]=result[2]= (RtFloat) color_alpha(clr);
  return result;
}	

/*
  	get_vtx_list()

	This routine returns an array of RenderMan points given
	the vlist (list of lisp vertices) and count (the number of 
	vertices).
*/
static RtFloat *get_vtx_list( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
{
  Vertex vertex;
  RtFloat *vertexlist, *vtx_copy;

  ger_debug("get_vtx_list:");
  vertexlist = (RtFloat *) malloc(3*vertexcount*sizeof(RtFloat));

  if (!vertexlist)
    ger_fatal( "get_loc_vertex: unable to allocate %d RtFloat!",3*vertexcount);
 
  vtx_copy = vertexlist;
  
  while (!null(vlist))
  {
    vertex = first_vertex(vlist);
    vlist= rest_of_vertices( vlist );
    *vtx_copy++= (RtFloat)vertex_x(vertex);
    *vtx_copy++= (RtFloat)vertex_y(vertex);
    *vtx_copy++= (RtFloat)vertex_z(vertex);
  }
 
  return vertexlist;
}

/*
  	get_vtx_normals()

	This routine returns an array of RenderMan normal data types
	from each vertex in the argument vlist.
*/
static RtFloat *get_vtx_normals(vlist, count)
Vertex_list vlist;
int count;
{
  Vertex norm;
  RtFloat *normals, *ncopy;
  int i;

  ger_debug("get_vtx_normals");

  if ( !(normals= (RtFloat *)malloc(3*count*sizeof(RtPoint))))
    ger_fatal( "get_vtx_normals: unable to allocate %d RtFloat!", 3*count);

  ncopy = normals;
  
  while (!null(vlist))
  {	
    if (!(norm = vertex_normal(first_vertex(vlist))))
      ger_error("get_vtx_normals: vertex missing normal!");
    vlist = rest_of_vertices(vlist);
    *ncopy++ = (RtFloat) vector_x(norm);
    *ncopy++ = (RtFloat) vector_y(norm);
    *ncopy++ = (RtFloat) vector_z(norm);
  }

  return normals;
}

/* 
  	get_vtx_colors()

	This routine returns an array of RenderMan color data type from
	from each vertex in the argument list.
*/
static RtFloat *get_vtx_colors(vlist, count)
Vertex_list vlist;
int count;
{
  Color clr;
  RtFloat *colors, *cclr;
  int i;

  ger_debug("get_vtx_colors:");

  colors= (RtFloat *) malloc(3*count*sizeof(RtFloat));

  if (!colors)
    ger_fatal( "get_color: unable to allocate %d RtFloats!", 3*count);

  cclr = colors;

  while(!null(vlist))
  {	
    if (!(clr = vertex_color(first_vertex(vlist))))
      ger_error("get_color: vertex missing color!");
    vlist = rest_of_vertices(vlist);
    *cclr++ = (RtFloat) color_red(clr);
    *cclr++ = (RtFloat) color_green(clr);
    *cclr++ = (RtFloat) color_blue(clr);
  }

  return colors;
}

/* 
  	get_vtx_opacity()

	This routine returns an array of RenderMan color data type from
	from each vertex in the argument list.
*/
static RtFloat *get_vtx_opacity(vlist, count)
Vertex_list vlist;
int count;
{
  Color clr;
  RtFloat *opacity, *copacity, alpha;
  int i;

  ger_debug("get_vtx_opacity");

  opacity = (RtFloat *) malloc(3*count*sizeof(RtFloat));

  if (!opacity)
    ger_fatal( "get_opacity: unable to allocate %d RtFloats!", 3*count);

  copacity = opacity;

  while(!null(vlist))
  {	
    if (!(clr = vertex_color(first_vertex(vlist))))
      ger_error("get_color: vertex missing color!");
    vlist = rest_of_vertices(vlist);
    alpha = (RtFloat) color_alpha(clr);
    *copacity++ = alpha;
    *copacity++ = alpha;
    *copacity++ = alpha;
  }

  return opacity;
}

static void add_trans(thistrans)
Transformation thistrans;
{
  RtMatrix *transmatrix;

  ger_debug("add_trans:");

  transmatrix = (RtMatrix *) transpose(array2d_to_c(thistrans));
  RiConcatTransform(*transmatrix);
  free ((char *) transmatrix);
}

static void add_attr(thisattrlist)
Attribute_list thisattrlist;
{
  RtFloat *color, *opacity, ks, kd, ka, roughness;
  Pair thispair;
  int id;
  Material material;

  ger_debug("add_attr:");

  /* push entire attributelist just in case 'text-height attribute 
   * is defined
   */
  ast_push_attributes(thisattrlist);

  /* check for color attribute */
  if (!null(thispair = symbol_attr(color_symbol, thisattrlist)))
  {
    color = (RtFloat *) get_renderman_color(pair_color(thispair));
    opacity = (RtFloat *) get_renderman_opacity(pair_color(thispair));
    RiColor(color);
    RiOpacity(opacity);
    free((char *) color);
    free((char *) opacity);
  }

  /* check for backcull attribute */
  if (!null(thispair = symbol_attr(backcull_symbol, thisattrlist)))
  {
    if (pair_boolean(thispair))
      RiSides(1); /* backface culling switched on */
    else
      RiSides(2); /* backface culling switched off */
  }

  /* check for 'material atribute */
  if (!null(thispair = symbol_attr(material_symbol, thisattrlist)))
  {
    material = pair_material(thispair);
    ka = material_ka(material);
    ks = material_ks(material);
    kd = material_kd(material);
    roughness = 1/(material_exp(material));
    RiSurface("plastic","Ka",&ka,"Kd",&kd,"Ks",&ks,"roughness",&roughness,
	      RI_NULL);
  }
}

/* This routine does nothing, since the renderer has no internal rep of
 * the gob. 
 */
void ren_dump(thisgob)
Gob thisgob;
{
  ger_debug("ren_dump: doing nothing.");
}

/*
	ren_sphere()
  
	This routine will construct a RenderMan sphere of radius 1
	which is centered at the origin.
*/
void ren_sphere()
{
  ger_debug("ren_sphere:");
  if (current_traversal != RENDER) return;
  RiSphere(1.0, -1.0, 1.0, 360.0, RI_NULL);
}

/*
  	ren_cylinder()

	This routine constructs a Renderman cylinder object of radius
	1 and is aligned with the z axis.  The ends of the cylinder are
	at z=0.0 and z=1.0.
*/
void ren_cylinder()
{
  ger_debug("ren_cylinder:");
  if (current_traversal != RENDER) return;
  RiCylinder(1.0, 0.0, 1.0, 360.0, RI_NULL);
  RiDisk(1.0, 1.0, 360.0, RI_NULL); /* cover the top of cylinder */
  RiDisk(0.0, 1.0, 360.0, RI_NULL);  /* cover the bottom of cylinder */
}

/*
	ren_torus()

	This routine constructs a RenderMan torus object from the
	major and minor radii passed as the arguments.  The torus
	is drawn so that the hole lies along the z axis.
*/
void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
{
  ger_debug("ren_torus:");
  if (current_traversal != RENDER) return;
  if (smallradius > bigradius)
    ger_error("ren_torus: minor radius is larger than major radius.");  
  RiTorus(bigradius, smallradius, 0.0, 360.0, 360.0, RI_NULL);
}

/*
  	ren_text()

	This routine draws a text primitive object.
*/
void ren_text(txtpoint, uvec, vvec, txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
{
  ger_debug("ren_text: <%s>",txtstring);
  if (current_traversal != RENDER) return;
  RiAttributeBegin();
  ast_text(txtpoint, uvec, vvec, txtstring, add_trans, ren_polyline);
  RiAttributeEnd();
}

/*
	ren_polyline()

	This routine constructs a polyline primitive object from the vertex 
	list passed it.  The parameter count is the number of vertices in the
	vlist.  The object constructed is a single polyline,  which maybe 
	concave but should not be self-intersecting.  The attributes (color
	and normal) at all vertices are assumed to be the same as those of the 
	first vertex.
*/
void ren_polyline(vlist, count)
Vertex_list vlist;
int count;
{
  RtFloat *colors = (RtFloat *) NULL;
  RtFloat color;
  RtFloat x, y, z, diameter;
  double len;
  RtPoint direction;
  Vertex from, to;
  static float zvect[] = {0.0, 0.0, 1.0};
  RtMatrix *RotMat;

  ger_debug("ren_polyline:");

  if (current_traversal != RENDER) return;

  if (count < 2)
  {
    ger_error("ren_polyline: not enough vertices!");
    return;
  }

  diameter = ast_float_attribute(line_diameter_symbol);

  while(!null(rest_of_vertices(vlist)))
  {
    from = first_vertex(vlist);
    vlist = rest_of_vertices(vlist);
    to = first_vertex(vlist);

    x = (RtFloat) point_x(from);
    y = (RtFloat) point_y(from);
    z = (RtFloat) point_z(from);

    direction[0] = (RtFloat) (point_x(to) - x);
    direction[1] = (RtFloat) (point_y(to) - y);
    direction[2] = (RtFloat) (point_z(to) - z);
    
    len = sqrt((double) (direction[0]*direction[0] + direction[1]*direction[1] 
	       + direction[2]*direction[2]));

    RiAttributeBegin();
    RiTranslate(x,y,z);
    RotMat = (RtMatrix *) make_aligning_rotation(direction[0],direction[1],
						 direction[2], .0, .0, 1.0);
    RiConcatTransform(*RotMat);
    RiScale(diameter, diameter, diameter);
    RiSphere(1.0, -1.0, 1.0, 360.0, RI_NULL);
    RiCylinder(1.0, 0.0, (RtFloat)(len/diameter), 360.0, RI_NULL);
    RiAttributeEnd();	
  }

  /* Add last sphere, to close the last cylinder */
  RiAttributeBegin();
  x = (RtFloat) point_x(to);
  y = (RtFloat) point_y(to);
  z = (RtFloat) point_z(to);
  RiTranslate(x,y,z);
  RiConcatTransform(*RotMat);
  RiScale(diameter, diameter, diameter);
  RiSphere(1.0, -1.0, 1.0, 360.0, RI_NULL);
  RiAttributeEnd();	

}

/* 
  	ren_polymarker()

  	This routine constructs a RenderMan polymarker from the vertex list
	passed it.  The parameter count is the number of vertices in the vlist.
	The object constructed is a single polymarker.   The attributes (color
	and normal) at all vertives are assumed to be the same as those of the
	first vertex.
*/
void ren_polymarker(vlist, count)
Vertex_list vlist;
int count;
{
  RtFloat *vertices, diameter;
  Vertex thisvertex;
  RtFloat *colors = (RtFloat *)NULL, *opacity; 
  RtColor color, alpha;
  short i;

  ger_debug("ren_polymarker:");

  if (current_traversal != RENDER) return;

  vertices = get_vtx_list(vlist, count);

  /* Check if the polymarker has color attributes */
  if (!(null(vertex_color(first_vertex(vlist)))))
  {
    colors = get_vtx_colors(vlist, count);
    opacity = get_vtx_opacity(vlist, count);
  }

  diameter = ast_float_attribute(point_diameter_symbol);

  if (!colors)
    for (i=0; i < 3*count; i += 3)
    {
      RiAttributeBegin();
      RiTranslate(vertices[i], vertices[i+1], vertices[i+2]);
      RiScale(diameter, diameter, diameter);
      RiSphere(1.0, -1.0, 1.0, 360.0, RI_NULL);
      RiAttributeEnd();
    }
  else
    for (i=0; i < 3*count; i += 3)
    {
      RiAttributeBegin();
      color[0] = colors[i];
      color[1] = colors[i+1];
      color[2] = colors[i+2];
      alpha[0] = opacity[i];
      alpha[1] = opacity[i+1];
      alpha[2] = opacity[i+2];
      RiColor(color);
      RiOpacity(alpha);
      RiTranslate(vertices[i], vertices[i+1], vertices[i+2]);
      RiScale(diameter, diameter, diameter);
      RiSphere(1.0, -1.0, 1.0, 360.0, RI_NULL);
      RiAttributeEnd();
    }

  free ((char *) vertices);
  if (colors != NULL)
  {
    free ((char *) colors);
    free ((char *) opacity);
  }
}

/*
  	ren_polygon()

	This routine renders a closed planar convex polygon whose vertices
	can have normal and color attributes.
*/
void ren_polygon(vlist, count)
Vertex_list vlist;
int count;
{
  RtFloat *vertices;
  Vertex thisvertex;
  RtFloat *normals = (RtFloat *)NULL;
  RtFloat *colors = (RtFloat *)NULL, *opacity;

  ger_debug("ren_polygon:");

  if (current_traversal != RENDER) return;

  if (count < 3)
    ger_error("ren_polygon: there are less than 3 vertices.");
  
  vertices = get_vtx_list(vlist, count);

  thisvertex= first_vertex(vlist);

  if (!(null(vertex_color(thisvertex))))
  {
    colors = get_vtx_colors(vlist, count);
    opacity = get_vtx_opacity(vlist, count);
  }

  if (!null(vertex_normal(thisvertex)))
    normals = get_vtx_normals(vlist, count);

  if (!normals && !colors)
    RiPolygon(count,RI_P, (RtPointer) vertices, RI_NULL);
  else if (!colors)
    RiPolygon(count, RI_P, (RtPointer) vertices, RI_N, 
	      (RtPointer) normals, RI_NULL);
  else if (!normals)
    RiPolygon(count, RI_P, (RtPointer) vertices, RI_CS, 
	      (RtPointer) colors, RI_OS, (RtPointer) opacity, RI_NULL);
  else     
    RiPolygon(count, RI_P, (RtPointer) vertices, RI_N, 
	      (RtPointer) normals, RI_CS, (RtPointer) colors,
	      RI_OS, (RtPointer) opacity, RI_NULL);

  free ((char *) vertices);
  if (colors != NULL)
  {
    free ((char *) colors);
    free ((char *) opacity);
  }
  if (normals != NULL)
    free ((char *) normals);
}

/*
  	ren_triangle()

	This routine constructs a triangle mesh object from the args 
	(all vertices) in the list passed it.  The object constructed is a 
	triangle strip, with each adjacent triple of vertices specifying one
	triangle.  The attributes (color and normal) at all vertices are 
	assumed to be the same as those of the first vertex.
*/
void ren_triangle(vlist, count)
Vertex_list vlist;
int count;
{
  RtFloat *vertices;
  Vertex thisvertex;
  RtFloat *normals = (RtFloat *)NULL;
  RtFloat *colors = (RtFloat *)NULL, *opacity;
  RtInt npolys; 
  RtInt *nverts; 
  RtInt *verts, *verts_copy;
  int i;

  ger_debug("ren_triangle:");

  if (current_traversal != RENDER) return;

  vertices = get_vtx_list(vlist, count);

  thisvertex= first_vertex(vlist);

  if (!(null(vertex_color(thisvertex))))
  {
    colors = get_vtx_colors(vlist, count);
    opacity = get_vtx_opacity(vlist, count);
  }

  /* if there are no normals, add them */
  if (null(vertex_normal( thisvertex )) ) ast_triangle_normals(vlist, count);
  normals = get_vtx_normals(vlist, count);

  /* the number of triangles */
  npolys = (RtInt) (count - 2); 

  /* allocate and fill in nverts with 3 (for number of vertices in triangle)*/
  nverts = (RtInt *) malloc((int) npolys*sizeof(RtInt));
  if (!nverts)
    ger_fatal("ren_triangle: unable to allocate %d RtInts!", npolys);
  for (i=0; i < npolys; i++)
    nverts[i] = (RtInt) 3;
  
  /* Allocate and fill the triangle index array */
  verts = (RtInt *) malloc(3*(int)npolys*sizeof(RtInt));
  if (!verts)
    ger_fatal("ren_triangle: unable to allocate %d RtInts!", npolys);  
  verts_copy = verts;
  for(i=0; i < npolys; i++)
    if (!(i % 2))
    {
      *verts_copy++ = (RtInt) i;
      *verts_copy++ = (RtInt) i+1;
      *verts_copy++ = (RtInt) i+2;
    }
    else 
    {
      *verts_copy++ = (RtInt) i+1;
      *verts_copy++ = (RtInt) i;
      *verts_copy++ = (RtInt) i+2;      
    }

  if (!colors)
    RiPointsPolygons(npolys,nverts,verts,RI_P, (RtPointer) vertices, RI_N, 
		    (RtPointer) normals, RI_NULL);
  else     
    RiPointsPolygons(npolys,nverts,verts,RI_P, (RtPointer) vertices, RI_N, 
		    (RtPointer) normals, RI_CS, (RtPointer) colors,
		     RI_OS,(RtPointer) opacity, RI_NULL);

  free ((char *) vertices);
  free ((char *) verts);
  free ((char *) nverts);
  free ((char *) normals);
  if (colors != NULL)
  {
    free ((char *) colors);
    free ((char *) opacity);
  }
}

/*
  	This routine causes RenderMan to construct a general polygonal
	mesh object from the args passed it.  The attributes (color and 
	normal) at all vertices	are assumed to be the same as those of the 
	first vertex.
*/
void ren_mesh(vlist, vcount, flist, fcount)
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
{
  RtFloat *vertices;
  Facet_list flist_copy;
  Vertex_list facet, vlist_copy;
  RtFloat *normals = (RtFloat *)NULL;
  RtFloat *colors = (RtFloat *)NULL, *opacity;
  RtInt *nverts; /* index table for the number of vertices in each poly. */
  RtInt *verts; /* contains indices into the vertices array */
  RtInt *verts_copy; 
  int i, verts_size = 0;

  ger_debug("ren_mesh:");

  if (current_traversal != RENDER) return;

  vertices = get_vtx_list(vlist, vcount);

  /* Fill in normals if there aren't any, and extract them. */
  if (null(vertex_normal(first_vertex(vlist))))
    ast_mesh_normals( vlist,vcount, flist, fcount );
  normals = get_vtx_normals(vlist, vcount);

  if (!(null(vertex_color(first_vertex(vlist)))))
  {
    colors = get_vtx_colors(vlist, vcount);
    opacity = get_vtx_opacity(vlist, vcount);
  }

   /* Traverse the vertex list, assigning an index to each vertex. */
  vlist_copy= vlist;
  for (i = 0; i < vcount; i++) 
  {
    (void)set_vertex_index(first_vertex(vlist_copy), i);
    vlist_copy= rest_of_vertices(vlist_copy);
  }

  /* nverts has fcount number of facets */
  nverts = (RtInt *) malloc(fcount*sizeof(RtInt));
  if (!nverts)
    ger_fatal("ren_mesh: unable to allocate %d RtInts for nverts!", fcount);

  /* 
   * fill in nverts table with the number of vertices for each polygon 
   * and counting the number of indices in the verts table as we go along.
   */
  flist_copy = flist;
  for (i=0; i < fcount; i++)
  {
    nverts[i] = (RtInt) length_list(first_facet(flist_copy));
    verts_size += nverts[i];    
    flist_copy = rest_of_facets(flist_copy);
  }

  /* Allocate and fill the vertex indices table */
  verts = (RtInt *) malloc(verts_size*sizeof(RtInt));
  if (!verts)
    ger_fatal("ren_mesh: unable to allocate %d RtInts for verts!", verts_size);
  verts_copy = verts;
  flist_copy = flist;
  for(i=0; i < fcount; i++)
  {
    facet = first_facet(flist_copy);
    while (!null(facet))
    {
      *verts_copy++ = (RtInt) vertex_index(first_vertex(facet));
      facet = rest_of_vertices(facet);
    }
    flist_copy = rest_of_facets(flist_copy);
  }
 
  if (!normals && !colors)
    RiPointsPolygons(fcount,nverts,verts,"P",(RtPointer) vertices,RI_NULL);
  else if (!colors)
    RiPointsPolygons(fcount,nverts,verts,RI_P, (RtPointer) vertices, RI_N, 
		    (RtPointer) normals, RI_NULL);
  else if (!normals)
    RiPointsPolygons(fcount,nverts,verts,RI_P, (RtPointer) vertices, RI_CS, 
		     (RtPointer) colors, RI_OS, (RtPointer) opacity, RI_NULL);
  else     
    RiPointsPolygons(fcount,nverts,verts,RI_P, (RtPointer) vertices, RI_N, 
		    (RtPointer) normals, RI_CS, (RtPointer) colors,
		     RI_OS, (RtPointer) opacity, RI_NULL);

  /* free memory */
  free((char *) vertices);
  free((char *) nverts);
  free((char *) verts); 
  if (colors != NULL)
  {
    free ((char *) colors);
    free ((char *) opacity);
  }
  if (normals != NULL)
    free ((char *) normals);
}

/*
	ren_bezier()

	This routine creates a Bezier Patch object from the args (16 vertices)
	in the list passed it.  The object constructed is a 4x4 bicubic Bezier
	patch.  The attributes (color and normal) at all vertices are assumed 
	to be the same as those of the first vertex.
*/
void ren_bezier(vlist, count)
Vertex_list vlist;
int count;
{
  RtFloat *vertices;
  Vertex thisvertex;
  RtFloat *normals = (RtFloat *)NULL;
  RtFloat *colors = (RtFloat *)NULL, *opacity;

  ger_debug("ren_bezier:");

  if (current_traversal != RENDER) return;

  vertices = get_vtx_list(vlist, count);

  thisvertex= first_vertex(vlist);

  if (!(null(vertex_color(thisvertex))))
  {
    colors = get_vtx_colors(vlist, count);
    opacity = get_vtx_opacity(vlist, count);
  } 

  if (!null(vertex_normal(thisvertex)))
    normals = get_vtx_normals(vlist, count);

  if (!normals && !colors)
    RiPatch(RI_BICUBIC, RI_P, (RtPointer) vertices, RI_NULL);
  else if (!colors)
    RiPatch(RI_BICUBIC, RI_P, (RtPointer) vertices, RI_N, 
		     (RtPointer) normals, RI_NULL);
  else if (!normals)
    RiPatch(RI_BICUBIC, RI_P, (RtPointer) vertices, RI_CS, 
		     (RtPointer) colors, RI_OS,(RtPointer)opacity, RI_NULL);
  else     
    RiPatch(RI_BICUBIC, RI_P, (RtPointer) vertices, RI_N, 
	    (RtPointer) normals, RI_CS, (RtPointer) colors,
	    RI_OS, (RtPointer) opacity, RI_NULL);

  /* free memory */
  free (vertices);
  if (colors != NULL)
  {
    free ((char *) colors);
    free ((char *) opacity);
  }
  if (normals != NULL)
    free ((char *) normals);
}

/*
  	ren_gob():

	This routine sees all gobs as they are defined.  Since traversal is
	done from the lisp-side DAG, ren_gob does nothing.
*/
void ren_gob(current_gob, trans, attr, primitive, children)
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
{
  ger_debug("ren_gob: doing nothing with gob %d",current_gob);
}

/*
  	This routine constructs a RenderMan light object,  first adding a
	color if necessary.
*/
void ren_light(location, lightcolor)
Point location;
Color lightcolor;
{
  float pos[4];
  float *worldpos;
  RtPoint lightpos;
  RtPoint atpos;
  RtFloat *color;

  ger_debug("ren_light:");

  if (current_traversal != LIGHTING) return;

  /* set the location, including trans. to world coords */
  pos[0]= point_x( location );
  pos[1]= point_y( location );
  pos[2]= point_z( location );
  pos[3]= 0.0;
  worldpos= matrix_vector_c( ast_get_matrix(), pos );
  lightpos[0]= (RtFloat)worldpos[0];
  lightpos[1]= (RtFloat)worldpos[1];
  lightpos[2]= (RtFloat)worldpos[2];
  free( worldpos );
  atpos[0]= atpos[1]= atpos[2]= 0.0; /* aim light at origin */

  /* set the color */
  color =  get_renderman_color(lightcolor);
  
  RiLightSource(RI_DISTANTLIGHT,RI_LIGHTCOLOR,color,RI_FROM,
		lightpos, RI_TO, atpos, RI_NULL);
  free(color);
}

/*
	ren_ambient()

	This routine constructs a RenderMan ambient light object,  
	first adding a color if necessary.  
*/
void ren_ambient(lightcolor)
Color lightcolor;
{
  RtFloat *color;

  ger_debug("ren_ambient:");

  if (current_traversal != LIGHTING) return;

  /* set the color */
  color = get_renderman_color(lightcolor);
  
  RiLightSource(RI_AMBIENTLIGHT,RI_LIGHTCOLOR,color,RI_NULL);
}

static void calc_ortho_screen( width, height, lookfrom, lookat, 
			      fov, x_res, y_res )
float *width, *height;
Point lookfrom, lookat;
float fov;
int x_res, y_res;
/* This routine calculates equivalent orthographic projection screen
 * dimensions for a given fovea and view distance.
 */
{
    float dx, dy, dz, range, min_dim;

    ger_debug("calc_ortho_screen:");

    dx= point_x(lookfrom) - point_x(lookat);
    dy= point_y(lookfrom) - point_y(lookat);
    dz= point_z(lookfrom) - point_z(lookat);
    range= sqrt( dx*dx + dy*dy + dz*dz );

    if (fov==0.0) {
      fov= 0.5; /* to avoid overflow */
      ger_error("calc_ortho_screen: fovea of 0.0 illegal; using 0.5 degrees.");
    }
    min_dim= 2.0*range*tan( DegtoRad*fov/2.0 );
    if (x_res<y_res) {
      *width= min_dim;
      *height= ((float)y_res/(float)x_res) * min_dim;
    }
    else {
      *height= min_dim;
      *width= ((float)x_res/(float)y_res) * min_dim;
    }
}

/* FrameCamera(): give physical parameters for a camera.  
   Parameters:
    fov       :	  the "camera's" foveal angle
    framewidth:  the width of the image in pixels
    frameheight: the height of the image in pixels
*/
static void FrameCamera(fov, framewidth, frameheight, lookfrom, lookat)
float fov;
int framewidth, frameheight;
Point lookfrom, lookat;
{
    RtFloat normwidth, normheight, fovea;
    float aspect_ratio;

    ger_debug("FrameCamera:");

    /* Fovea < 1.0 is taken to be an orthographic projection */
    if (fov > 1.0) {
      fovea= fov;
      RiProjection("perspective",RI_FOV,(RtPointer)&fovea,RI_NULL);  
    } else {
      float width, height;
      calc_ortho_screen( &width, &height, lookfrom, lookat, fov, 
			framewidth, frameheight );
      RiProjection("orthographic", RI_NULL);
      RiScreenWindow( (RtFloat)(-width/2.0), (RtFloat)(width/2.0),
		     (RtFloat)(-height/2.0), (RtFloat)(height/2.0) );
    }
}


static float *get_normal_component( v1x, v1y, v1z, v2x, v2y, v2z )
float v1x, v1y, v1z, v2x, v2y, v2z;
/* This routine returns a 4-vector holding the perpendicular component
 * of v1 with respect to v2, or (0, 1, 0, 1) if the two vectors are
 * parallel.
 */
{
  float *perpvec;
  float dot, normsqr;

  ger_debug("get_normal_component:");

  if ( !(perpvec= (float *)malloc( 4*sizeof(float) ) ) )
    ger_fatal("get_normal_component: unable to allocate 4 floats!\n");

  dot= v1x*v2x + v1y*v2y + v1z*v2z;
  normsqr= v2x*v2x + v2y*v2y + v2z*v2z;

  perpvec[0]= v1x - v2x*dot/normsqr;
  perpvec[1]= v1y - v2y*dot/normsqr;
  perpvec[2]= v1z - v2z*dot/normsqr;
  perpvec[3]= 1.0;

  if ( (perpvec[0]==0.0) && (perpvec[1]==0.0) && (perpvec[2]==0.0) ) {
    ger_error("get_normal_component: up and viewing vectors align; using Y.");
    perpvec[1]= 1.0;
    }

  return( perpvec );
}

/*
  	set_background()

	this function sets the background color by placing a rectangle
	of yon distance away from the loofrom point.  This rectangle is
	later rendered in ren_render.
*/
static void set_background(fov, lookfrom, lookat, lookup, yon)
float fov, yon;
Point lookfrom, lookat;
{
  float x, y, dx, dy, dz, upx, upz, upy, norm, dot, min_dim, distance, theta;
  float centerx, centery, centerz, xvecx, xvecy, xvecz;

  /* Get the normalized up vector */
  upx = vector_x(lookup); upy = vector_y(lookup); upz = vector_z(lookup);
  norm = sqrt(upx*upx + upy*upy + upz*upz);
  if (norm>0.0) {
    upx = upx/norm; 
    upy = upy/norm; 
    upz = upz/norm;
  }
  else ger_error("set_background: null up vector!");

  /* Get the normalized view vector */
  dx= point_x(lookat) - point_x(lookfrom);
  dy= point_y(lookat) - point_y(lookfrom);
  dz= point_z(lookat) - point_z(lookfrom);
  norm= sqrt( dx*dx + dy*dy + dz*dz );
  if (norm>0.0) {
    dx=dx/norm; 
    dy=dy/norm;
    dz=dz/norm;
  }
  else ger_error("set_background: lookat and lookfrom are the same!");

  /* Replace the up vector with its normalized perpendicular component */
  dot= upx*dx + upy*dy + upz*dz;
  upx= upx - dot*dx;
  upy= upy - dot*dy;
  upz= upz - dot*dz;
  norm = sqrt(upx*upx + upy*upy + upz*upz);
  if (norm>0.0) {
    upx = upx/norm; 
    upy = upy/norm; 
    upz = upz/norm;
  }
  else ger_error("set_background: up parallel to line of sight!");

  /* X direction on screen is cross product of view and up vectors */
  xvecx= dy*upz - dz*upy;
  xvecy= dz*upx - dx*upz;
  xvecz= dx*upy - dy*upx;

  /* get center of rectangle */
  centerx = -BACKGROUND_CLIP_MARGIN*yon*dx + point_x(lookfrom);
  centery = -BACKGROUND_CLIP_MARGIN*yon*dy + point_y(lookfrom);
  centerz = -BACKGROUND_CLIP_MARGIN*yon*dz + point_z(lookfrom);

  /* determine height(y) width(x) */
  if (fov==0.0) {
    fov= 0.5; /* to avoid overflow */
    ger_error("set_background: fovea of 0.0 illegal; using 0.5 degrees.");
  }
  min_dim= -BACKGROUND_SIZE_MARGIN*yon*tan( DegtoRad*fov/2.0 );
  if (x_res<y_res) {
    x = min_dim;
    y = ((float)y_res/(float)x_res) * min_dim;
  }
  else {
    y = min_dim;
    x = ((float)x_res/(float)y_res) * min_dim;
  }

  /* upper left corner */
  back_rect[0][0] = centerx + upx*y - xvecx*x;
  back_rect[0][1] = centery + upy*y - xvecy*x;
  back_rect[0][2] = centerz + upz*y - xvecz*x;
  /* upper right corner */
  back_rect[1][0] = centerx + upx*y + xvecx*x;
  back_rect[1][1] = centery + upy*y + xvecy*x;
  back_rect[1][2] = centerz + upz*y + xvecz*x;
  /* bottom right corner */
  back_rect[2][0] = centerx - upx*y + xvecx*x;
  back_rect[2][1] = centery - upy*y + xvecy*x;
  back_rect[2][2] = centerz - upz*y + xvecz*x;
  /* bottom left corner */
  back_rect[3][0] = centerx - upx*y - xvecx*x;
  back_rect[3][1] = centery - upy*y - xvecy*x;
  back_rect[3][2] = centerz - upz*y - xvecz*x;
}

/*
  	ren_camera():

	This routine sets up the current camera.  The alignment of the 
	y-axis with the lookup vector is defined after the camera placement
	since transformations in RenderMan are done backwards.
*/
void ren_camera(lookat, lookfrom, lookup, fov, hither, yon, background)
Point lookat, lookfrom;
Color background;
Vector lookup;
float fov, hither, yon;

{
  float *trans, *rot, *urot, *temp, *cameramatrix, *upvec, *rup;
  float fx, fy, fz, dx, dy, dz, upx, upy, upz;
  long xsize, ysize;

  ger_debug("ren_camera");

  RiFrameBegin(frame_id);

  /* Extract components */
  fx= point_x(lookfrom);
  fy= point_y(lookfrom);
  fz= point_z(lookfrom);
  dx= point_x(lookat) - point_x(lookfrom);
  dy= point_y(lookat) - point_y(lookfrom);
  dz= point_z(lookat) - point_z(lookfrom);
  upx= vector_x(lookup);
  upy= vector_y(lookup);
  upz= vector_z(lookup);

  /* Stick in an inverting scale transformation */
  RiScale(-1.0,1.0,1.0);

  /* Set clipping and perspective projection */
  RiClipping((RtFloat)(-hither), (RtFloat)(-yon));
  FrameCamera(fov, x_res, y_res, lookfrom, lookat);

  /* Construct matrix to implement camera position */
  /* Translate the lookfrom point to the origin */
  trans= make_translate_c( -fx, -fy, -fz );

  /* Rotate the lookat point onto the +z axis */
  rot= make_aligning_rotation( dx, dy, dz, 0.0, 0.0, 1.0 );

  /* Align the up vector with the y axis */
  upvec= get_normal_component( upx, upy, upz, dx, dy, dz ); 
  rup= matrix_vector_c( rot, upvec );
  urot= make_aligning_rotation(rup[0], rup[1], rup[2], 0.0, 1.0, 0.0);

  /* Construct the matrix */
  temp= matrix_mult_c( rot, trans );
  cameramatrix= matrix_mult_c( urot, temp );

  /* Install the camera transformation */
  RiConcatTransform( *(RtMatrix *)transpose(cameramatrix) );

 /* set backgound color */
  backgroundclr[0]= (RtFloat) color_red( background );
  backgroundclr[1]= (RtFloat) color_green( background );
  backgroundclr[2]= (RtFloat) color_blue( background );
  set_background(fov, lookfrom, lookat, lookup, yon+(RI_EPSILON));

  /* clean up */
  free( (char *)trans );
  free( (char *)rot );
  free( (char *)urot );
  free( (char *)temp );
  free( (char *)upvec );
  free( (char *)rup );
  free( (char *)cameramatrix );

}

/*
  	fast_render():

	This routine renders a gob, by handling its transformation and
	attributes and either triggering execution of its primitive or
	rendering its children.  It is used to traverse eometry gobs.
*/
static fast_render(thisgob)
Gob thisgob;
{
  Child_list kidlist;
  RtFloat *color;
  RtFloat tempfloat;

  ger_debug("fast_render: rendering gob %d",gob_idnum(thisgob));

  RiAttributeBegin();

  if(!null(gob_trans(thisgob)))
    add_trans(gob_trans(thisgob));

  if(!null(gob_attr(thisgob)))
    add_attr(gob_attr(thisgob));

  /* render the primitive or render the children */
  if(!null(gob_primitive(thisgob)))
    eval_function(primitive_op(gob_primitive(thisgob)));
  else
  {
    kidlist = gob_children(thisgob);
    while (!null(kidlist))
    {
      fast_render(first_child(kidlist));
      kidlist = rest_of_children(kidlist);
    }
  }

  if(!null(gob_attr(thisgob)))
    ast_pop_attributes(gob_attr(thisgob));  

  RiAttributeEnd();
}

/*
  	This routine initiates rendering, by handling the default attribute
	list and transformation and rendering of the gob.
*/
void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
{
  static char curr_filename[30];
  static RtInt nverts[1] = {4};
  static RtInt verts[4] = {0, 1, 2, 3};

  ger_debug("ren_render:");

  if (strcmp("-", filename) != 0)
    sprintf(curr_filename, "%s_%04d.tiff", filename, frame_id++);
  else
    sprintf(curr_filename, "frame_%04d.tiff", frame_id++);

  RiDisplay(curr_filename, RI_FILE, RI_RGB, RI_NULL);


  RiWorldBegin();

  /* display the background color */
  RiAttributeBegin();
  RiSurface(RI_CONSTANT, RI_NULL);
  RiColor(backgroundclr);
  RiPointsPolygons(1, nverts, verts, RI_P, (RtPointer) back_rect, RI_NULL);
  RiAttributeEnd();

  RiAttributeBegin();

  RiShadingRate(shading_rate);
  RiShadingInterpolation(RI_SMOOTH);

  /* handle top-level transformation and attributes */
  if (!null(thistrans))
    add_trans(thistrans);

  if (!null(thisattrlist))
    add_attr(thisattrlist);

  current_traversal = RENDER;
  fast_render(thisgob);

  if(!null(thisattrlist))
    ast_pop_attributes(thisattrlist);  

  RiAttributeEnd();
  RiWorldEnd();

  RiFrameEnd();
}

static fast_traverse(thisgob)
Gob thisgob;
/* This routine traverses a gob DAG, by handling its transformation 
 * and attributes and either triggering execution of its primitive 
 * or rendering its children.  It is used to traverse the lighting DAG.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  float *transmatrix;
  Primitive newprim;
  Child_list kidlist;

  thisid= gob_idnum( thisgob );
  ger_debug("fast-traverse: traversing object given by gob %d", thisid);

  /* If there is a transformation, add it to the transformation stack */
  if ( !null( newtrans=gob_trans(thisgob) ) ) {
    transmatrix= array2d_to_c( newtrans );
    ast_push_matrix(transmatrix);
  }
  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );

  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) {
    eval_function( primitive_op(newprim) );
  }
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_traverse( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );
    }
  }

  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );
  if ( !null(newtrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
  }
}

/*
  	This routine traverses a gob,  looking for light sources
	and defining them as they are found.
*/
void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
{
  int thisid;
  float *transmatrix;

  thisid = gob_idnum(thisgob);
  ger_debug("ren_traverselights: traversing gob %d in search of lights\n",
	    thisid);

  current_traversal = LIGHTING;
  
  /* If there is a transformation, add it to the transformation stack */
  if ( !null( thistrans ) ) {
    transmatrix= array2d_to_c( thistrans );
    ast_push_matrix(transmatrix);
  }
  
  /* Check for and if necessary handle new attributes */
  if ( !null( thisattrlist ) ) ast_push_attributes( thisattrlist );
  
  fast_traverse(thisgob);
  
  /* Pop attributes and transformations, and clean up. */
  if ( !null(thisattrlist) ) ast_pop_attributes( thisattrlist );
  if ( !null(thistrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
  }
  current_traversal = RENDER;
}

/*
  	ren_free()

  	This routine frees memory associated with the given gob.
	Since traversal takes place on the lisp side,  there is no
	memory associated with the gobs in the renderer,  so this does nothing.
*/
void ren_free(thisgob)
Gob thisgob;
{
  ger_debug("ren_free: doing nothing");
}

static void parse_hints(hints)
Attribute_list hints;
/* This routine handles hint list parsing */
{
  Pair thispair;
  char format[10];

  ger_debug("parse_hints:");

  if ( !null( thispair=symbol_attr(initial_frame_symbol,hints) ) ) 
    frame_id = pair_int(thispair);
  if ( !null( thispair=symbol_attr(x_res_symbol,hints) ) ) {
    x_res= pair_int(thispair);
    if (x_res<1) x_res= 1;
  }
  if ( !null( thispair=symbol_attr(y_res_symbol,hints) ) ) {
    y_res= pair_int(thispair);
    if (y_res<1) x_res= 1;
  }
  if ( !null( thispair=symbol_attr(shading_rate_symbol,hints) ) ) 
    shading_rate = pair_int(thispair);

  /* parse rib format */
  if (!null(thispair = symbol_attr(binary_rib_symbol, hints)))
  {
    if (pair_boolean(thispair))
      sprintf(format, "%s", BINARY_FORMAT);
    else
      sprintf(format, "%s", ASCII_FORMAT);
  }	
  else
      sprintf(format, "%s", DEFAULT_FORMAT);
  rib_format[0] = format;
}

void ren_setup(renderer, device, open_device, outfile, hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
{
  static int initialized = 0; /* holds initialization state */
  static char rib_filename[30];

  ger_debug("ren_setup:");

  filename = outfile;

  ast_init_matrix();

  if (!initialized)
  {
    /* create needed symbols*/
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    initial_frame_symbol= create_symbol("initial-frame-id");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");
    shading_rate_symbol =  create_symbol("shading-rate");
    binary_rib_symbol = create_symbol("binary-rib");

    parse_hints(hints);
    initialized = 1;
  }
  else ger_error("ren_setup: called twice, this call ignored");

  RiOption("rib", "format", (RtPointer) rib_format, RI_NULL); 

  /* check if output rib is to stdout */
  if (strcmp("-", outfile) != 0)
  {
    sprintf(rib_filename,"%s.rib",outfile);
    RiBegin(rib_filename);
  }
  else 
    RiBegin(RI_NULL);

  RiFormat((RtInt) x_res, (RtInt) y_res, -1.0);
}

/*
	ren_reset()

	This routine resets the renderer.
*/
void ren_reset(hard)
int hard;
{

  ger_debug("ren_reset: hard= %d",hard);

  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    initial_frame_symbol= create_symbol("initial_frame_id");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");    
    shading_rate_symbol =  create_symbol("shading-rate");
    binary_rib_symbol = create_symbol("binary-rib");
  }
}

void ren_shutdown()
{
  ger_debug("ren_shutdown:");
  RiEnd();
}

char *ren_def_material(material)
Material material;
{
  ger_debug("ren_def_material: doing nothing");
}

void ren_free_material(thismat)
Material thismat;
{
  ger_debug("ren_free_material: doing nothing");
}

