/****************************************************************************
 * dialog.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides dialog box support under Silicon Graphics GL.
*/

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <gl.h>
#include <device.h>
#include "knobs.h"

#define TEXT_STEP_Y 20
#define MAX_INPUT_LENGTH 128
#define DEL_KEY 8
#define RET_KEY 13

static int char_width;

static void erase_character(x, y)
int x, y;
/* This routine overwrites a character with the background color */
{
  KnobColor(0);
  y -= TEXT_STEP_Y/4; /* shift down a bit */
  rectf( (Coord)x, (Coord)y, 
         (Coord)(x+char_width), (Coord)(y+TEXT_STEP_Y) );
  KnobColor(3);
}

static void draw_text(start_x, start_y, step_y, textstr)
int start_x, start_y, step_y;
char *textstr;
/* This routine draws text, with returns at line ends. */
{
  int current_x, current_y;
  char string[2];

  current_x= start_x;
  current_y= start_y;
  string[1]= '\0';

  while (*textstr) {
    if ( *textstr=='\n' ) {
      current_x= start_x;
      current_y= current_y - step_y;
    }
    else {
      cmov2( (Coord)current_x, (Coord)current_y );
      string[0]= *textstr;
      charstr( string );
      current_x += char_width;
    }
    textstr++;
  }
}

static ControlType *create_simple_button(name, x, y, xsize, ysize)
char *name;
int x, y, xsize, ysize;
/* This routine creates a simple button. */
{
  ControlType *thisbutton;

  if ( !(thisbutton= (ControlType *)malloc(sizeof(ControlType)) ) ) {
    fprintf(stderr,"Unable to allocate %d bytes for a control button!\n",
            sizeof(ControlType));
    exit(2);
  }
  thisbutton->type= BUTTON;
  thisbutton->style= 0;
  strncpy(thisbutton->name, name, 127); thisbutton->name[127]= '\0';
  thisbutton->x= x;
  thisbutton->y= y;
  thisbutton->x_size= xsize;
  thisbutton->y_size= ysize;
  thisbutton->auto_flush= 0;
  thisbutton->next_control= NULL;
  thisbutton->sub_control= NULL;
  thisbutton->count= 0;
  thisbutton->current= 0;
  thisbutton->pos= 0;
  thisbutton->flag= 0;
  thisbutton->new_string= NULL;
  thisbutton->old_string= NULL;
  thisbutton->max= 0.0;
  thisbutton->min= 0.0;
  thisbutton->new= 0.0;
  thisbutton->old= 0.0;
  thisbutton->scale= 0.0;
  thisbutton->incr= 0.0;
  thisbutton->base= 0.0;
  thisbutton->notick= 0;
  
  return( thisbutton );
}

static int prompt_x( thisdialog )
ControlType *thisdialog;
/* This routine returns an appropriate x coordinate for the prompt line. */
{
  return( thisdialog->x + 30 );
}

static int prompt_y( thisdialog )
ControlType *thisdialog;
/* This routine returns an appropriate x coordinate for the prompt line. */
{
  return( thisdialog->y + ((2*thisdialog->y_size)/3) );
}

static int answer_x( thisdialog )
ControlType *thisdialog;
/* This routine returns an appropriate x coordinate for the prompt line. */
{
  return( thisdialog->x + 30 ) + char_width;
}

static int answer_y( thisdialog )
ControlType *thisdialog;
/* This routine returns an appropriate x coordinate for the prompt line. */
{
  return( thisdialog->y + (thisdialog->y_size/2) );
}

int dialog_init(thisdialog)
ControlType *thisdialog;
/* This routine initializes the given dialog box. */
{
  ControlType *ok_button, *cancel_button;

  /* Are we the right thing? */
  if (thisdialog->type != DIALOG) return(0);

  /* Set globals */
  if (!char_width) char_width= strwidth("M"); 

  /* Allocate and fill the OK and Cancel buttons */
  thisdialog->sub_control= ok_button= create_simple_button(
    "OK", thisdialog->x+10, thisdialog->y+10, 0, 0 );
  ok_button->next_control= create_simple_button(
    "Cancel",thisdialog->x+70, thisdialog->y+10, 0, 0 );

  /* Create input buffer, and set initial values */
  if ( !(thisdialog->new_string=(char *)malloc(MAX_INPUT_LENGTH)) ) {
    fprintf(stderr,"init_dialog: unable to allocate %d byte string buffer!\n",
            MAX_INPUT_LENGTH);
    exit(2);
  }
  if (thisdialog->old_string)
    strncpy(thisdialog->new_string,thisdialog->old_string,MAX_INPUT_LENGTH);
  else thisdialog->new_string[0]= '\0';
  thisdialog->pos= strlen(thisdialog->new_string);
  thisdialog->current= answer_x(thisdialog) + thisdialog->pos*char_width;

  return(1);
}

int dialog_service(thisdialog, dev, val)
Device dev;
short val;
ControlType *thisdialog;
/*
This routine handles input to the dialog box.  It returns 1 if the
dialog box has completed its interaction, and zero otherwise.  If the
option was accepted (for example via the OK button) thisdialog->flag
is set to 1; otherwise it is set to 0.
*/
{
  ControlType *ok_button, *cancel_button;
  static char string[2]= "*"; /* automatically null terminated */
  int ok_flag= 0, cancel_flag= 0;

  /* Are we the right thing? */
  if (thisdialog->type != DIALOG) return(0);

  ok_button= thisdialog->sub_control;
  cancel_button= ok_button->next_control;

  if ( (mouse_x>=thisdialog->x) && (mouse_y>=thisdialog->y)
      && (mouse_x<=(thisdialog->x+thisdialog->x_size))
      && (mouse_y<=(thisdialog->y+thisdialog->y_size)) ) {

    switch (dev) {

    case KEYBD:
      if ( (char)val == DEL_KEY ) {
        /* delete a character */
        if (thisdialog->pos > 0) {
          thisdialog->pos--;
          thisdialog->current -= char_width;
          erase_character( thisdialog->current, answer_y(thisdialog) );
        }
      } 
      else if ( (char)val == RET_KEY ) {
        /* counts as accepting the current string */
        ok_flag= 1;
      } 
      else if ( thisdialog->pos < MAX_INPUT_LENGTH-1 ) {
        /* add keystroke to the growing input buffer */
	thisdialog->new_string[thisdialog->pos++]= (char)val;
	cmov2( (Coord)thisdialog->current, (Coord)answer_y(thisdialog) );
	string[0]= (char)val;
	charstr( string );
	thisdialog->current += char_width; /* move right 1 space */
      } else ringbell();
      break;

    case LEFTMOUSE:
      ok_flag= button_service(ok_button);
      cancel_flag= button_service(cancel_button);
    }
    
  } else if (mouse_state || (dev=KEYBD)) ringbell(); 
                     /* Mouse must be in box for events to be valid */

  thisdialog->new_string[thisdialog->pos]= '\0';

  if (ok_flag) thisdialog->flag= 1;
  else thisdialog->flag= 0;

  if (ok_flag || cancel_flag) return(1);
  else return(0);
}

int dialog_draw(thisdialog)
ControlType *thisdialog;
/* This routine draws the dialog box, and associated text and buttons. */
{
  ControlType *ok_button, *cancel_button;
  static char prompt[]= ">";

  /* Are we the right thing? */
  if (thisdialog->type != DIALOG) return(0);

  ok_button= thisdialog->sub_control;
  cancel_button= ok_button->next_control;

  KnobPanel( thisdialog->x, thisdialog->y, 
	    thisdialog->x_size, thisdialog->y_size );
  button_draw( ok_button, 1 );
  button_draw( cancel_button, 1 );
  draw_text( prompt_x(thisdialog), prompt_y(thisdialog), TEXT_STEP_Y,
	    thisdialog->name );
  draw_text( answer_x(thisdialog)-char_width, answer_y(thisdialog),
            TEXT_STEP_Y, prompt);
  draw_text( answer_x(thisdialog), answer_y(thisdialog), TEXT_STEP_Y,
	    thisdialog->new_string);

  return(1);
}
