/* idct.c, inverse fast discrete cosine transform                           */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

/**********************************************************/
/* inverse two dimensional DCT, Chen-Wang algorithm       */
/* (cf. IEEE ASSP-32, pp. 803-816, Aug. 1984)             */
/* 32-bit integer arithmetic (8 bit coefficients)         */
/* 11 mults, 29 adds per DCT                              */
/*                                      sE, 18.8.91       */
/**********************************************************/
/* coefficients extended to 12 bit for IEEE1180-1990      */
/* compliance                           sE,  2.1.94       */
/**********************************************************/

/* this code assumes >> to be a two's-complement arithmetic */
/* right shift: (-2)>>1 == -1 , (-3)>>1 == -2               */

#include "config.h"

#define W1 5681 /* 4096*sqrt(2)*cos(1*pi/16) */
#define W2 5352 /* 4096*sqrt(2)*cos(2*pi/16) */
#define W3 4816 /* 4096*sqrt(2)*cos(3*pi/16) */
#define W5 3218 /* 4096*sqrt(2)*cos(5*pi/16) */
#define W6 2217 /* 4096*sqrt(2)*cos(6*pi/16) */
#define W7 1130 /* 4096*sqrt(2)*cos(7*pi/16) */

/* global declarations */
void init_idct _ANSI_ARGS_((void));
void idct _ANSI_ARGS_((short *block));

/* private data */
static long temp[64];     /* intermediate storage for 1D transform */
static short iclip[1024]; /* clipping table */
static short *iclp;

/* private prototypes */
static void idctrow _ANSI_ARGS_((short *src, long *dst));
static void idctcol _ANSI_ARGS_((long *src, short *dst));

/* row (horizontal) IDCT
 *
 *           7                       pi         1
 * dst[k] = sum c[l] * src[l] * cos( -- * ( k + - ) * l )
 *          l=0                      8          2
 *
 * where: c[0]    = 256*sqrt(0.5)
 *        c[1..7] = 256
 */

static void idctrow(src,dst)
short *src;
long *dst;
{
  long x0, x1, x2, x3, x4, x5, x6, x7, x8;

  /* shortcut */
  if (!src[0] && !src[1] && !src[2] && !src[3] &&
      !src[4] && !src[5] && !src[6] && !src[7])
  {
    dst[0]=dst[1]=dst[2]=dst[3]=dst[4]=dst[5]=dst[6]=dst[7]=0;
    return;
  }

  /* first stage */
  x0 = src[0];
  x1 = src[4];
  x2 = src[6];
  x3 = src[2];
  x4 = src[1];
  x5 = src[7];
  x6 = src[5];
  x7 = src[3];
  x8 = W7*(x4+x5);
  x4 = x8 + (W1-W7)*x4;
  x5 = x8 - (W1+W7)*x5;
  x8 = W3*(x6+x7);
  x6 = x8 - (W3-W5)*x6;
  x7 = x8 - (W3+W5)*x7;

  /* second stage */
  x8 = ((x0+x1)<<12) + 16; /* +16 for proper rounding in the fourth stage */
  x0 = ((x0-x1)<<12) + 16;
  x1 = W6*(x3+x2);
  x2 = x1 - (W2+W6)*x2;
  x3 = x1 + (W2-W6)*x3;
  x1 = x4 + x6;
  x4 -= x6;
  x6 = x5 + x7;
  x5 -= x7;

  /* third stage */
  x7 = x8 + x3;
  x8 -= x3;
  x3 = x0 + x2;
  x0 -= x2;
  x2 = (181*(x4+x5)+128)>>8;
  x4 = (181*(x4-x5)+128)>>8;

  /* fourth stage */
  dst[0] = (x7+x1)>>5;
  dst[1] = (x3+x2)>>5;
  dst[2] = (x0+x4)>>5;
  dst[3] = (x8+x6)>>5;
  dst[4] = (x8-x6)>>5;
  dst[5] = (x0-x4)>>5;
  dst[6] = (x3-x2)>>5;
  dst[7] = (x7-x1)>>5;
}

/* column (vertical) IDCT
 *
 *             7                         pi         1
 * dst[8*k] = sum c[l] * src[8*l] * cos( -- * ( k + - ) * l )
 *            l=0                        8          2
 *
 * where: c[0]    = 1/1024*sqrt(0.5)
 *        c[1..7] = 1/1024
 */
static void idctcol(src,dst)
long *src;
short *dst;
{
  long x0, x1, x2, x3, x4, x5, x6, x7, x8;

  /* first stage */
  x0 = src[8*0];
  x1 = src[8*4];
  x2 = src[8*6];
  x3 = src[8*2];
  x4 = src[8*1];
  x5 = src[8*7];
  x6 = src[8*5];
  x7 = src[8*3];
  x8 = W7*(x4+x5) + 2048;
  x4 = (x8+(W1-W7)*x4)>>12;
  x5 = (x8-(W1+W7)*x5)>>12;
  x8 = W3*(x6+x7) + 2048;
  x6 = (x8-(W3-W5)*x6)>>12;
  x7 = (x8-(W3+W5)*x7)>>12;

  /* second stage */
  x8 = x0 + x1;
  x0 -= x1;
  x1 = W6*(x3+x2) + 2048;
  x2 = (x1-(W2+W6)*x2)>>12;
  x3 = (x1+(W2-W6)*x3)>>12;
  x1 = x4 + x6;
  x4 -= x6;
  x6 = x5 + x7;
  x5 -= x7;

  /* third stage */
  x7 = x8 + x3 + 512;
  x8 += -x3 + 512;
  x3 = x0 + x2 + 512;
  x0 += -x2 + 512;
  x2 = (181*(x4+x5)+128)>>8;
  x4 = (181*(x4-x5)+128)>>8;

  /* fourth stage */
  dst[8*0] = iclp[(x7+x1)>>10];
  dst[8*1] = iclp[(x3+x2)>>10];
  dst[8*2] = iclp[(x0+x4)>>10];
  dst[8*3] = iclp[(x8+x6)>>10];
  dst[8*4] = iclp[(x8-x6)>>10];
  dst[8*5] = iclp[(x0-x4)>>10];
  dst[8*6] = iclp[(x3-x2)>>10];
  dst[8*7] = iclp[(x7-x1)>>10];
}

/* two dimensional inverse discrete cosine transform */
void idct(block)
short *block;
{
  int i;

  for (i=0; i<8; i++)
    idctrow(block+8*i, temp+8*i);

  for (i=0; i<8; i++)
    idctcol(temp+i, block+i);
}

void init_idct()
{
  int i;

  iclp = iclip+512;
  for (i= -512; i<512; i++)
    iclp[i] = (i<-256) ? -256 : ((i>255) ? 255 : i);
}
