/*
 * Fake sendto and recvfrom calls, which take advantage of a udprelay
 * process running on a firewall.
 *
 * TODO:
 *  - Maybe insure that received packets really do come from the firewall?
 *  - Add Rrecv (easy) and Rsend (hard, requires Rconnect hook and
 *	static table of ports mapped to remote end addresses.)
 *
 * Copyright 1993 Tom Fitzgerald <fitz@wang.com>; permission to use, modify
 * and distribute this source is granted provided only that this notice is
 * retained intact.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pwd.h>
#include <errno.h>
#include "udprelay.h"

#define Debug		0
#define InitBufSize	2048

#if Debug
int			debug = 0;
#define	DBPRINTF(X)	printf X
#else
#define	DBPRINTF
#endif


static SockIn	proxysin;

static char	*bufptr = NULL;
static int	bufsize = 0;
static char	username [12];

extern char	*getenv ();
extern char	*malloc ();

void UDPrelayinit ()
{
	char		*ns;
	struct hostent	*hp;
	struct servent	*sp;
	struct passwd	*pwd;

	DBPRINTF (("Initializing\n"));
	ns = getenv ("UDP_RELAY_HOST");
	if (ns == NULL)
		ns = getenv ("SOCKS_HOST");
	if (ns == NULL)
		ns = RELAYHOST;
	/*
	 * Build socket pointing to the proxy host, suitable for sendto
	 */
	proxysin.sin_family = AF_INET;

	if ((hp = gethostbyname (ns)) == NULL)
		proxysin.sin_addr.s_addr = inet_addr (ns);
	else
		bcopy (hp->h_addr_list[0], &proxysin.sin_addr, hp->h_length);

	if (proxysin.sin_addr.s_addr == INADDR_ANY) {
		fprintf (stderr, "Lookup on proxy host failed\n");
		exit (1);
	}
	if ((sp = getservbyname ("udprelay", "udp")) != NULL) {
		proxysin.sin_port = sp->s_port;
	}
	else {
#ifdef RELAYPORT
		proxysin.sin_port = htons (RELAYPORT);
#else
		fprintf (stderr, "Unknown service: udprelay/udp\n");
		exit (1);
#endif
	}
	DBPRINTF (("Using proxy %s port %d\n", inet_ntoa (proxysin.sin_addr),
	  ntohs (proxysin.sin_port)));
	bufptr = malloc (InitBufSize);
	if (bufptr == NULL) {
		fprintf (stderr, "UdpRelayInit: No memory\n");
		exit (1);
	}
	bufsize = InitBufSize;

	if ((pwd = getpwuid (getuid ())) == NULL)
		sprintf (username, "uid=%d", getuid ());
	else {
		strncpy (username, pwd->pw_name, sizeof (username) - 1);
		username [sizeof (username) - 1] = 0;
	}
	DBPRINTF (("User ID %s\n", username));
}


static int makebuf (len)
	int		len;
{
	char		*newbuf;

	len *= 2;
	newbuf = malloc (len);
	if (newbuf == NULL)
		return 0;
	if (bufptr)
		free (bufptr);
	bufptr = newbuf;
	bufsize = len;
	return (1);
}


int Rsendto (s, msg, len, flags, to, tolen)
	int		s;
	char		*msg;
	int		len, flags;
	struct sockaddr_in	*to;
	int		tolen;		/* (ignored) */
{
	UdpEncaps	*wrapper;
	char		*data;
	int		val;

	DBPRINTF (("Rsendto (%d, -, %d, 0x%x, { %s / %d }, %d)\n", s, len,
	  flags, inet_ntoa (to->sin_addr), ntohs (to->sin_port), tolen));
	if (len + sizeof (UdpEncaps) > bufsize) {
		if (bufsize == 0)
			UDPrelayinit ();
		if (!makebuf (len + sizeof (UdpEncaps))) {
			errno = EFAULT;
			return -1;
		}
	}
	wrapper = (UdpEncaps *) bufptr;
	wrapper->sin = *to;
	wrapper->proxyport = INADDR_ANY;
	wrapper->version = UdpRelayVersion;
	wrapper->contents = UdpRelayData;
	wrapper->twoway = 1;
	wrapper->padding1 = 0;
	strcpy (wrapper->username, username);
	memcpy (bufptr + sizeof (UdpEncaps), msg, len);
	val = sendto (s, bufptr, len + sizeof (UdpEncaps), flags, &proxysin,
	  sizeof (proxysin));
	DBPRINTF (("sendto returned %d\n", val));
	return (val - sizeof (UdpEncaps));
}


int Rrecvfrom (s, buf, len, flags, from, fromlen)
	int		s;
	char		*buf;
	int		len, flags;
	struct sockaddr_in	*from;
	int		*fromlen;
{
	int		val, sinlen;
	UdpEncaps	*wrapper;

	DBPRINTF (("Rrecvfrom (%d, -, %d, 0x%x, -, &%d\n", s, len, flags,
	  fromlen));
	if (len + sizeof (UdpEncaps) > bufsize) {
		if (bufsize == 0) {
			fprintf (stderr, "Not initialized\n");
			exit (1);
		}
		if (!makebuf (len + sizeof (UdpEncaps))) {
			errno = EFAULT;
			return -1;
		}
	}
	sinlen = 0;
	if ((val = recvfrom (s, bufptr, bufsize, flags, NULL, &sinlen)) <= 0) {
		DBPRINTF (("recvfrom err, errno=%s\n", errno));
		return val;
	}
	wrapper = (UdpEncaps *) bufptr;
	DBPRINTF (("Rrecvfrom addr %s port %d\n",
	  inet_ntoa (wrapper->sin.sin_addr), ntohs (wrapper->sin.sin_port)));
	if (val <= sizeof (UdpEncaps)
	  || (from != NULL && *fromlen < sizeof (*from))
	  || wrapper->version != UdpRelayVersion
	  || wrapper->contents != UdpRelayData) {
		/* !!! do something */
		DBPRINTF (("Bad encapsulation or fromlen\n"));
		return -1;
	}
	if (from != NULL) {
		*from = wrapper->sin;
		from->sin_family = AF_INET;	/* Insurance */
	}
	val -= sizeof (UdpEncaps);
	memcpy (buf, bufptr + sizeof (UdpEncaps), val);
	return val;
}

