/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  SSL Engine Internals
**  _________________________________________________________________
*/

/*
 *  Connect Handler:
 *  Connect SSL to the accepted socket
 *
 *  Usually we would need an Apache API hook which is triggered right after
 *  the socket is accepted for handling a new request. But Apache 1.3 doesn't
 *  provide such a hook, so we have to patch http_main.c and call this
 *  function directly.
 */
int SSL_HOOK_SetupConnection(conn_rec *conn)
{
    server_rec *srvr = conn->server;
    BUFF *fb = conn->client;
    SSLSrvConfigRec *pConfig = mySrvConfig(srvr);
    char *cipher;
    int keysize, seckeysize;

    /*
     * Immediately stop processing if SSL is disabled for
     * this connection
     */
    if (!pConfig->bEnabled) {
        fb->ssl = NULL;
        return TRUE;
    }

    /*
     * Remember the connection information for
     * later access inside callback functions
     */
    ssl_log(srvr, SSL_LOG_INFO, "Connection to child %d established (server %s:%d)",
            conn->child_num, srvr->server_hostname, srvr->port);
    ssl_ModConfig->rCtx.pConn = conn;

    /*
     * Create a new SSL connection with the configured server SSL context and
     * attach this to the socket. Additionally we register this attachment
     * so we can detach later.
     */
    fb->ssl = __SSLeay SSL_new(pConfig->pSSLCtx);
    __SSLeay SSL_set_fd(fb->ssl, fb->fd);
    ap_register_cleanup(conn->pool, (void *)conn, 
                        SSL_HOOK_CloseConnection, SSL_HOOK_CloseConnection);

    /*
     * Now verify the SSL connection
     */
    conn->client->szClientX509 = NULL;
    conn->client->nVerifyError = X509_V_OK;

    if (!ssl_int_SetCertStuff(conn))
        return FALSE;

    while (!__SSLeay SSL_is_init_finished(fb->ssl)) {

        if (__SSLeay SSL_accept(fb->ssl) <= 0) {
            ssl_log(srvr, SSL_LOG_ERROR|SSL_ADD_SSLEAY, "SSL socket acception failed");
            fb->flags |= B_EOF|B_EOUT;
            return FALSE;
        }

        if (conn->client->nVerifyError != X509_V_OK) {
            ssl_log(srvr, SSL_LOG_ERROR|SSL_ADD_SSLEAY, "SSL verification failed");
            fb->flags |= B_EOF | B_EOUT;
            return FALSE;
        }

        if (pConfig->nVerifyClient != VERIFY_NONE) {
            char *s;
            X509 *xs;
            if ((xs = __SSLeay SSL_get_peer_certificate(fb->ssl)) != NULL) {
                s = __SSLeay X509_NAME_oneline(X509_get_subject_name(xs), NULL, 0);
                conn->client->szClientX509 = ap_pstrdup(conn->pool, s);
                free(s);
            }
        }

        if (   pConfig->nVerifyClient == VERIFY_REQUIRE
            && !conn->client->szClientX509             ) {
            ssl_log(srvr, SSL_LOG_ERROR|SSL_ADD_SSLEAY, "No SSL Client Certificate");
            return FALSE;
        }

        cipher = __SSLeay SSL_get_cipher(conn->client->ssl);
        ssl_int_DetermineCipherSizes(cipher, &keysize, &seckeysize);
        ssl_log(srvr, SSL_LOG_INFO, "Client IP: %s, Protocol: %s, Cipher: %s (%d/%d bit)", 
                conn->remote_ip, ssl_int_GetVersion(conn->client->ssl),
                cipher, keysize, seckeysize);
    }

    /* 
     * This should be safe.... so I'll use it -- ben
     */
    __SSLeay SSL_set_read_ahead(fb->ssl, 1);

    return TRUE;
}

/*
 *  Close the SSL part of the socket connection
 */
void SSL_HOOK_CloseConnection(void *_conn)
{
    conn_rec *conn = _conn;
    SSL *ssl = conn->client->ssl;

    if (ssl != NULL) {
        __SSLeay SSL_set_shutdown(ssl, SSL_SENT_SHUTDOWN|SSL_RECEIVED_SHUTDOWN);
        __SSLeay SSL_free(ssl);
    }
    ssl_log(conn->server, SSL_LOG_INFO, "Connection to child %d closed (server %s:%d)",
            conn->child_num, conn->server->server_hostname, conn->server->port);

    ssl_ModConfig->rCtx.pConn = NULL;
    ssl_ModConfig->rCtx.pServ = NULL;
    ssl_ModConfig->rCtx.pPool = NULL;

    return;
}

/*
 *  Auth Handler:
 *  Fake a Basic authentication from the X509 client certificate.
 *
 *  This must be run fairly early on to prevent a real authentication from
 *  occuring, in particular it must be run before anything else that
 *  authenticates a user.  This means that the Module statement for this
 *  module should be LAST in the Configuration file.
 */
int ssl_hook_Auth(request_rec *r)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);
    char b1[MAX_STRING_LEN], b2[MAX_STRING_LEN];

    /*
     * We decline operation in various situations..
     */
    if (OPTIONAL_SSL)
        return DECLINED;
    if (!pConfig->bFakeBasicAuth)
        return DECLINED;
    if (r->connection->user)
        return DECLINED;
    if (!r->connection->client->szClientX509)
        return DECLINED;
    
    /*
     * Fake a password - which one would be immaterial, as, it seems, an empty
     * password in the users file would match ALL incoming passwords, if only
     * we were using the standard crypt library routine. Unfortunately, SSLeay
     * "fixes" a "bug" in crypt and thus prevents blank passwords from
     * working.  (IMHO what they really fix is a bug in the users of the code
     * - failing to program correctly for shadow passwords).  We need,
     * therefore, to provide a password. This password can be matched by
     * adding the string "xxj31ZMTZzkVA" as the password in the user file.
     * This is just the crypted variant of the word "password" ;-)
     */
    ap_snprintf(b1, sizeof(b1), "%s:password", 
                r->connection->client->szClientX509);
    ssl_util_uuencode(b2, b1);
    ap_snprintf(b1, sizeof(b1), "Basic %s", b2);
    ap_table_set(r->headers_in, "Authorization", b1);

    return DECLINED;
}

/*
 *  Access Handler
 */
int ssl_hook_Access(request_rec *r)
{
    char *cipher;
    SSLDirConfigRec *rec  = myDirConfig(r);
    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);

    /*
     * Support for SSLRequire directive
     */
    if (rec->bSSLRequired && !r->connection->client->ssl) {
        ap_log_reason("SSL connection required", r->filename, r);
        return FORBIDDEN;
    }

    /* 
     * Check to see if SSL protocol is on 
     */
    if (!pConfig->bEnabled || OPTIONAL_SSL)
        return DECLINED;

    /*
     * Check for banned SSL chipers 
     */
    cipher = __SSLeay SSL_get_cipher(r->connection->client->ssl);
    if (ap_table_get(rec->tbBannedCiphers, cipher)) {
        char *buf;
        buf = ap_pstrcat(r->pool, "SSL Cipher ", cipher, " is forbidden", NULL);
        ap_log_reason(buf, r->filename, r);
        return FORBIDDEN;
    }

    /*
     * If cipher is one of the required ones or no
     * required ciphers are configured, it's ok.
     */
    if (ap_table_get(rec->tbRequiredCiphers, cipher))
        return OK;
    if (ap_is_empty_table(rec->tbRequiredCiphers))
        return OK;
    else {
        char *buf;
        buf = ap_pstrcat(r->pool, "SSL Cipher ", cipher, 
                         " is not on the permitted list", NULL);
        ap_log_reason(buf, r->filename, r);
        return FORBIDDEN;
    }
}

/*
 *   Fixup Handler
 */
int ssl_hook_Fixup(request_rec *r)
{
    table *e = r->subprocess_env;
    int keysize;
    int secretkeysize;
    char buf[MAX_STRING_LEN];
    char *cipher;
    X509 *xs;

    SSLSrvConfigRec *pConfig = mySrvConfig(r->server);

    /* 
     * Check to see if SSL is on
     */
    if (!pConfig->bEnabled || OPTIONAL_SSL)
        return DECLINED;

    /*
     * Annotate the SSI/CGI environment with SSL information
     */

    cipher = __SSLeay SSL_get_cipher(r->connection->client->ssl);
    ssl_int_DetermineCipherSizes(cipher, &keysize, &secretkeysize);

    ap_table_set(e, "HTTPS", "on");
    ap_table_set(e, "HTTPS_CIPHER", cipher);
    ap_table_set(e, "SSL_CIPHER", cipher);

    ap_table_set(e, "SSL_PROTOCOL_VERSION",
                 __SSLeay SSL_get_version(r->connection->client->ssl));

    ap_table_set(e, "SSL_SSLEAY_VERSION", 
                 __SSLeay SSLeay_version(SSLEAY_VERSION));

    ap_snprintf(buf, sizeof(buf), "%d", keysize);
    ap_table_set(e, "HTTPS_KEYSIZE", buf);
    ap_snprintf(buf, sizeof(buf), "%d", secretkeysize);
    ap_table_set(e, "HTTPS_SECRETKEYSIZE", buf);

    if (r->connection->client->szClientX509) {
        ssl_int_ExpandCert(e, "SSL_CLIENT_", r->connection->client->szClientX509);
        xs = __SSLeay SSL_get_peer_certificate(r->connection->client->ssl);
        ssl_int_ExpandCert(e, "SSL_CLIENT_I", 
                           __SSLeay X509_NAME_oneline(X509_get_issuer_name(xs), 
                                                      NULL, 0));
    }

    xs = __SSLeay SSL_get_certificate(r->connection->client->ssl);
    ssl_int_ExpandCert(e, "SSL_SERVER_", 
                       __SSLeay X509_NAME_oneline(__SSLeay X509_get_subject_name(xs),
                                                  NULL, 0));
    ssl_int_ExpandCert(e, "SSL_SERVER_I",
                       __SSLeay X509_NAME_oneline(__SSLeay X509_get_issuer_name(xs),
                                                  NULL, 0));
    return DECLINED;
}

/*  _________________________________________________________________
**
**  SSLeay Callback Functions
**  _________________________________________________________________
*/

/*
 * Just a stub in case SSLeay again wants a pass phrase
 */
int ssl_callback_PassPhrase(char *buf, int bufsize, int dummy)
{
    ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_ERROR,
            "Ops, an unexpected request from SSLeay for a phrase phrase occured.");
    ssl_log(ssl_ModConfig->rCtx.pServ, SSL_LOG_INFO,
            "You should perform a hard server restart because the keyfiles seems to have changed.");
    return (-1);
}

/* 
 * Handle out the already generated RSA key...
 */
RSA *ssl_callback_TmpRSA(SSL *pSSL, int nExport)
{
    return ssl_ModConfig->pRSATmpKey;
}

int ssl_callback_SSLVerify(int ok, X509_STORE_CTX *ctx)
{
    X509 *xs  = __SSLeay X509_STORE_CTX_get_current_cert(ctx);
    int depth = __SSLeay X509_STORE_CTX_get_error_depth(ctx);
    int error = __SSLeay X509_STORE_CTX_get_error(ctx);

    SSLSrvConfigRec *pConfig = mySrvConfig(ssl_ModConfig->rCtx.pConn->server);

    /* 
     * Some verification debugging...
     */
    {
        char *s;
        if ((s = __SSLeay X509_NAME_oneline(
                 __SSLeay X509_get_subject_name(xs), NULL, 0)) == NULL) {
            ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Cannot get subject information for debugging purposes");
            return (0);
        }
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_INFO,
                "Certificate Verify: depth: %d, subject: %s\n", depth, s);
        free(s);
    }

    /*
     * ...
     */
    if (   error == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY 
        || error == X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) {
        char *s;

        if (pConfig->nVerifyClient == VERIFY_OPTIONAL_NO_CA) {
            ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG,
                    "No issuer, returning OK");
            return TRUE;
        }
        s = __SSLeay X509_NAME_oneline(__SSLeay X509_get_issuer_name(xs), NULL, 0);
        if (s == NULL) {
            ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Verify error: no issuer name");
            return(0);
        }
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, "SSL Issuer: %s", s);
        free(s);
    }
    if (!ok) {
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                "Verify error:num=%d:%s", error,
               __SSLeay X509_verify_cert_error_string(error));
        ssl_ModConfig->rCtx.pConn->client->szClientX509 = NULL;
    }
    if (depth >= pConfig->nVerifyDepth) {
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_ERROR, "Verify depth exceeded");
        ok = 0;
    }
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_DEBUG, 
            "SSL Verify return: %d", ok);
    return (ok);
}

/*
 *  This callback function is executed by SSLeay whenever a new SSL_SESSION is
 *  added to the internal SSLeay session cache. We use this hook to spread the
 *  SSL_SESSION also to the inter-process disk-cache to make share it with our
 *  other Apache pre-forked server processes.
 */
int ssl_callback_NewSessionCacheEntry(SSL *pSSL, SSL_SESSION *pNew)
{
    SSLSrvConfigRec *pConfig;
    long t = 0;

    /*
     * Set the timeout also for the internal SSLeay cache, because this way
     * our inter-process cache is consulted only when it's really necessary.
     */
    pConfig = mySrvConfig(ssl_ModConfig->rCtx.pConn->server);
    t = SSL_get_time(pNew) + pConfig->nSessionCacheTimeout;
    SSL_set_timeout(pNew, t);

    /*
     * Store the SSL_SESSION in the inter-process cache with the
     * same expire time, so it expires automatically there, too.
     */
    ssl_scache_store(pNew, t);

    /*
     * Log this cache operation
     */
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_INFO, 
            "Inter-Process Session Cache: request=SET id=%s timeout=%ds (session caching)",
            ssl_scache_id2sz(pNew->session_id, pNew->session_id_length),
            time(NULL)-t);

    /* 
     * return 0 which means to SSLeay that the pNew is still
     * valid and was not freed by us with SSL_SESSION_free().
     */
    return 0;
}

/*
 *  This callback function is executed by SSLeay whenever a
 *  SSL_SESSION is looked up in the internal SSLeay cache and it
 *  was not found. We use this to lookup the SSL_SESSION in the
 *  inter-process disk-cache where it was perhaps stored by one
 *  of our other Apache pre-forked server processes.
 */
SSL_SESSION *ssl_callback_GetSessionCacheEntry(
    SSL *pSSL, unsigned char *id, int idlen, int *pCopy)
{
    SSL_SESSION *pSession;

    /*
     * Try to retrieve the SSL_SESSION from the inter-process cache
     */
    pSession = ssl_scache_retrieve(id, idlen);

    /*
     * Log this cache operation
     */
    if (pSession != NULL)
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_INFO, 
                "Inter-Process Session Cache: request=GET status=FOUND id=%s (session reuse)",
                ssl_scache_id2sz(id, idlen));
    else
        ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_INFO, 
                "Inter-Process Session Cache: request=GET status=MISSED id=%s (session renewal)",
                ssl_scache_id2sz(id, idlen));

    /*
     * Return NULL or the retrieved SSL_SESSION. But indicate (by
     * setting pCopy to 0) that the reference count on the
     * SSL_SESSION should not be incremented by the SSL library,
     * because we will no longer hold a reference to it ourself.
     */
    *pCopy = 0;
    return pSession;
}

/*
 *  This callback function is executed by SSLeay whenever a
 *  SSL_SESSION is removed from the the internal SSLeay cache.
 *  We use this to remove the SSL_SESSION in the inter-process
 *  disk-cache, too.
 */
void ssl_callback_DelSessionCacheEntry(
    SSL_CTX *ctx, SSL_SESSION *pSession)
{
    /*
     * Remove the SSL_SESSION from the inter-process cache
     */
    ssl_scache_remove(pSession);

    /*
     * Log this cache operation
     */
    ssl_log(ssl_ModConfig->rCtx.pConn->server, SSL_LOG_INFO, 
            "Inter-Process Session Cache: request=REM status=OK id=%s (session dead)",
            ssl_scache_id2sz(pSession->session_id, pSession->session_id_length));

    return;
}

/*  _________________________________________________________________
**
**  SSL Internals
**  _________________________________________________________________
*/

/* 
 * Expand a X509_oneline entry into it's base components and register
 * them as environment variables. Needed if you want to pass certificate
 * information to CGI's. The naming convention SHOULD be fairly compatible
 * with CGI's written for stronghold's certificate info - Q 
 */
void ssl_int_ExpandCert(table *pEnv, char *szPrefix, char *szCert)
{
    char var[HUGE_STRING_LEN];
    char buf[HUGE_STRING_LEN];
    char *s;

    ap_snprintf(var, sizeof(var), "%sDN", szPrefix);
    ap_table_set(pEnv, var, szCert);

    /* FIXME - strtok() and strcspn() may cause problems on some systems - Q */
    ap_cpystrn(buf, szCert, sizeof(buf));
    for (s = strtok(buf, "/"); s != NULL; s = strtok(NULL, "/")) {
        int n = strcspn(s, "=");
        s[n] = NUL;
        ssl_util_strupper(s);
        ap_snprintf(var, sizeof(var), "%s%s", szPrefix, s);
        ap_table_set(pEnv, var, s + n + 1);
    }
}

BOOL ssl_int_SetCertStuff(conn_rec *conn)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(conn->server);
    SSL *con = conn->client->ssl;
    char *cert_file = pConfig->szCertificateFile;

    /* PEM_set_getkey_callback(SSLKeyCallback); */
    if (cert_file != NULL) {
        if (__SSLeay SSL_use_certificate(con, pConfig->px509Certificate) <= 0) {
            ssl_log(conn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Unable to set certificate information for server");
            return FALSE;
        }
        if (__SSLeay SSL_use_RSAPrivateKey(con, pConfig->prsaKey) <= 0) {
            ssl_log(conn->server, SSL_LOG_ERROR|SSL_ADD_SSLEAY,
                    "Unable to set private key information for server");
            return FALSE;
        }
    }
    return TRUE;
}

char *ssl_int_GetVersion(SSL *ssl)
{
    char *result;

#if SSLEAY_VERSION_NUMBER >= 0x0900
    switch(ssl->session->ssl_version) {
        case TLS1_VERSION:
            result = "TLS1";
            break;
        case SSL3_VERSION:
            result = "SSL3";
            break;
        case SSL2_VERSION:
            result = "SSL2";
            break;
        default:
            result = "SSL?";
    }
#else
    {
        static char v[5];
        ap_cpystrn(v, "SSL", sizeof(v));
        v[3] = '0' + ssl->session->ssl_version;
        v[4] = '\0';
        result = v;
    }
#endif
    return result;
}

static struct {
    char *szName;
    int nKeySize;
    int nSecretKeySize;
} aCipherspecs[] = {
#if SSLEAY_VERSION_NUMBER >= 0x0900
    { SSL3_TXT_RSA_IDEA_128_SHA          /*IDEA-CBC-SHA*/,           128, 128 },
#endif
    { SSL3_TXT_RSA_NULL_MD5              /*NULL-MD5*/,                 0,   0 },
    { SSL3_TXT_RSA_NULL_SHA              /*NULL-SHA*/,                 0,   0 },
    { SSL3_TXT_RSA_RC4_40_MD5            /*EXP-RC4-MD5*/,            128,  40 },
    { SSL3_TXT_RSA_RC4_128_MD5           /*RC4-MD5*/,                128, 128 },
    { SSL3_TXT_RSA_RC4_128_SHA           /*RC4-SHA*/,                128, 128 },
    { SSL3_TXT_RSA_RC2_40_MD5            /*EXP-RC2-CBC-MD5*/,        128,  40 },
    { SSL3_TXT_RSA_IDEA_128_SHA          /*IDEA-CBC-MD5*/,           128, 128 },
    { SSL3_TXT_RSA_DES_40_CBC_SHA        /*EXP-DES-CBC-SHA*/,         56,  40 },
    { SSL3_TXT_RSA_DES_64_CBC_SHA        /*DES-CBC-SHA*/ ,            56,  56 },
    { SSL3_TXT_RSA_DES_192_CBC3_SHA      /*DES-CBC3-SHA*/ ,          168, 168 },
    { SSL3_TXT_DH_DSS_DES_40_CBC_SHA     /*EXP-DH-DSS-DES-CBC-SHA*/,  56,  40 },
    { SSL3_TXT_DH_DSS_DES_64_CBC_SHA     /*DH-DSS-DES-CBC-SHA*/,      56,  56 },
    { SSL3_TXT_DH_DSS_DES_192_CBC3_SHA   /*DH-DSS-DES-CBC3-SHA*/,    168, 168 },
    { SSL3_TXT_DH_RSA_DES_40_CBC_SHA     /*EXP-DH-RSA-DES-CBC-SHA*/,  56,  40 },
    { SSL3_TXT_DH_RSA_DES_64_CBC_SHA     /*DH-RSA-DES-CBC-SHA*/,      56,  56 },
    { SSL3_TXT_DH_RSA_DES_192_CBC3_SHA   /*DH-RSA-DES-CBC3-SHA*/,    168, 168 },
    { SSL3_TXT_EDH_DSS_DES_40_CBC_SHA    /*EXP-EDH-DSS-DES-CBC-SHA*/, 56,  40 },
    { SSL3_TXT_EDH_DSS_DES_64_CBC_SHA    /*EDH-DSS-DES-CBC-SHA*/,     56,  56 },
    { SSL3_TXT_EDH_DSS_DES_192_CBC3_SHA  /*EDH-DSS-DES-CBC3-SHA*/,   168, 168 },
    { SSL3_TXT_EDH_RSA_DES_40_CBC_SHA    /*EXP-EDH-RSA-DES-CBC*/,     56,  40 },
    { SSL3_TXT_EDH_RSA_DES_64_CBC_SHA    /*EDH-RSA-DES-CBC-SHA*/,     56,  56 },
    { SSL3_TXT_EDH_RSA_DES_192_CBC3_SHA  /*EDH-RSA-DES-CBC3-SHA*/,   168, 168 },
    { SSL3_TXT_ADH_RC4_40_MD5            /*EXP-ADH-RC4-MD5*/,        128,  40 },
    { SSL3_TXT_ADH_RC4_128_MD5           /*ADH-RC4-MD5*/,            128, 128 },
    { SSL3_TXT_ADH_DES_40_CBC_SHA        /*EXP-ADH-DES-CBC-SHA*/,    128,  40 },
    { SSL3_TXT_ADH_DES_64_CBC_SHA        /*ADH-DES-CBC-SHA*/,         56,  56 },
    { SSL3_TXT_ADH_DES_192_CBC_SHA       /*ADH-DES-CBC3-SHA*/,       168, 168 },
    { SSL3_TXT_FZA_DMS_NULL_SHA          /*FZA-NULL-SHA*/,             0,   0 },
    { SSL3_TXT_FZA_DMS_FZA_SHA           /*FZA-FZA-CBC-SHA*/,         -1,  -1 },
    { SSL3_TXT_FZA_DMS_RC4_SHA           /*FZA-RC4-SHA*/,            128, 128 },
    { SSL2_TXT_DES_64_CFB64_WITH_MD5_1   /*DES-CFB-M1*/,              56,  56 },
    { SSL2_TXT_RC2_128_CBC_WITH_MD5      /*RC2-CBC-MD5*/,            128, 128 },
    { SSL2_TXT_DES_64_CBC_WITH_MD5       /*DES-CBC-MD5*/,             56,  56 },
    { SSL2_TXT_DES_192_EDE3_CBC_WITH_MD5 /*DES-CBC3-MD5*/,           168, 168 },
    { SSL2_TXT_RC4_64_WITH_MD5           /*RC4-64-MD5*/,              64,  64 },
    { SSL2_TXT_NULL                      /*NULL*/,                     0,   0 },
    { NULL,                                                            0,   0 }
};

void ssl_int_DetermineCipherSizes(char *cipher, int *keysize, int *secretkeysize)
{
    int n;

    *keysize       = 0;
    *secretkeysize = 0;
    for (n = 0; aCipherspecs[n].szName; ++n) {
        if (strEQ(cipher, aCipherspecs[n].szName)) {
            *keysize       = aCipherspecs[n].nKeySize;
            *secretkeysize = aCipherspecs[n].nSecretKeySize;
            break;
        }
    }
    return;
}

