/* ====================================================================
 * Copyright (c) 1998 Ralf S. Engelschall. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * 4. The names "mod_ssl" must not be used to endorse or promote
 *    products derived from this software without prior written
 *    permission. For written permission, please contact
 *    rse@engelschall.com.
 *
 * 5. Products derived from this software may not be called "mod_ssl"
 *    nor may "mod_ssl" appear in their names without prior
 *    written permission of Ralf S. Engelschall.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by 
 *     Ralf S. Engelschall <rse@engelschall.com> for use in the
 *     mod_ssl project (http://www.engelschall.com/sw/mod_ssl/)."
 *
 * THIS SOFTWARE IS PROVIDED BY RALF S. ENGELSCHALL ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RALF S. ENGELSCHALL OR
 * HIS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 */


#include "mod_ssl.h"


/*  _________________________________________________________________
**
**  Support for Global Configuration
**  _________________________________________________________________
*/

void ssl_hook_AddModule(module *m)
{
    if (m == &ssl_module) {
        /*
         * Announce us for the configuration files
         */
        ap_add_config_define("MOD_SSL");

        /*
         * Link ourself into the Apache kernel
         */
        ssl_var_register();
        ssl_ext_register();
        ssl_io_register();
    }
    return;
}

void ssl_config_global_create(void)
{
    pool *pPool;
    SSLModConfigRec *mc;

    mc = ap_ctx_get(ap_global_ctx, "mod_ssl::ssl_ModConfig");
    if (mc == NULL) {
        /* 
         * allocate an own subpool which survives server restarts
         */
        pPool = ap_make_sub_pool(NULL);
        mc = (SSLModConfigRec *)ap_palloc(pPool, sizeof(SSLModConfigRec));
        mc->pPool = pPool;

        /*
         * initialize per-module configuration
         */
        mc->nInitCount             = 0;
        mc->pRSATmpKey             = NULL;
        mc->nSessionCacheMode      = SSL_SCMODE_UNSET;
        mc->szSessionCacheDataFile = NULL;
        mc->nMutexMode             = SSL_MUTEXMODE_UNSET;
        mc->szMutexFile            = NULL;
        mc->nMutexFD               = -1;
        mc->nMutexSEMID            = -1;

        mc->tPrivateKey            = ssl_ds_table_make(pPool, sizeof(RSA *));
        mc->tPublicCert            = ssl_ds_table_make(pPool, sizeof(X509 *));

        mc->rCtx.pConn             = NULL;
        mc->rCtx.pServ             = NULL;
        mc->rCtx.pPool             = NULL;

        /*
         * And push it into Apache's global context
         */
        ap_ctx_set(ap_global_ctx, "mod_ssl::ssl_ModConfig", mc);
    }

    /*
     * And make sure our access variable points to it...
     */
    if (ssl_ModConfig == NULL)
        ssl_ModConfig = mc;
    return;
}


/*  _________________________________________________________________
**
**  Configuration handling
**  _________________________________________________________________
*/

/*
 *  Create per-server SSL configuration
 */
void *ssl_config_server_create(pool *p, server_rec *s)
{
    SSLSrvConfigRec *rec;
    
    ssl_config_global_create();
    
    rec = ap_palloc(p, sizeof(SSLSrvConfigRec));
    rec->bEnabled               = UNSET;
    rec->szCertificateFile      = CFGDEF_CERTIFCATE_FILE;
    rec->szKeyFile              = NULL;
    rec->szCACertificatePath    = NULL;
    rec->szCACertificateFile    = NULL;
    rec->szCACertificateReqFile = NULL;
    rec->szLogFile              = NULL;
    rec->szCipherSuite          = NULL;
    rec->nLogLevel              = SSL_LOG_NONE;
    rec->nVerifyDepth           = 0;
    rec->nVerifyClient          = VERIFY_UNSET;
    rec->bFakeBasicAuth         = UNSET;
    rec->nSessionCacheTimeout   = CFGDEF_SESSION_CACHE_TIMEOUT;
    rec->nPassPhraseDialogType  = SSL_PPTYPE_UNSET;
    rec->szPassPhraseDialogPath = NULL;

    rec->fileLogFile            = NULL;
    rec->px509Certificate       = NULL;
    rec->prsaKey                = NULL;
    rec->pSSLCtx                = NULL;

    return rec;
}

/*
 *  Merge per-server SSL configurations
 */
void *ssl_config_server_merge(pool *p, void *basev, void *addv)
{
    SSLSrvConfigRec *base = (SSLSrvConfigRec *)basev;
    SSLSrvConfigRec *add  = (SSLSrvConfigRec *)addv;
    SSLSrvConfigRec *new  = (SSLSrvConfigRec *)ap_palloc(p, sizeof(SSLSrvConfigRec));

    cfgMergeBool(bEnabled);
    cfgMergeString(szCertificateFile);
    cfgMergeString(szKeyFile);
    cfgMergeString(szCACertificatePath);
    cfgMergeString(szCACertificateFile);
    cfgMergeString(szCACertificateReqFile);
    cfgMergeString(szLogFile);
    cfgMergeString(szCipherSuite);
    cfgMerge(nLogLevel, SSL_LOG_NONE);
    cfgMergeInt(nVerifyDepth);
    cfgMerge(nVerifyClient, VERIFY_UNSET);
    cfgMergeBool(bFakeBasicAuth);
    cfgMergeInt(nSessionCacheTimeout);
    cfgMerge(nPassPhraseDialogType, SSL_PPTYPE_UNSET);
    cfgMergeString(szPassPhraseDialogPath);

    cfgMerge(fileLogFile, NULL);
    cfgMerge(px509Certificate, NULL);
    cfgMerge(prsaKey, NULL);
    cfgMerge(pSSLCtx, NULL);

    return new;
}

/*
 *  Create per-directory SSL configuration
 */
void *ssl_config_perdir_create(pool *p, char *dummy)
{
    SSLDirConfigRec *rec = ap_palloc(p, sizeof(SSLDirConfigRec));

    rec->bSSLRequired      = FALSE;
    rec->aRequirement      = ap_make_array(p, 4, sizeof(SSLRequirement));

    return rec;
}

/*
 *  Merge per-directory SSL configurations
 */
void *ssl_config_perdir_merge(pool *p, void *basev, void *addv)
{
    SSLDirConfigRec *base = (SSLDirConfigRec *)basev;
    SSLDirConfigRec *add  = (SSLDirConfigRec *)addv;
    SSLDirConfigRec *new  = (SSLDirConfigRec *)ap_palloc(p, 
                                               sizeof(SSLDirConfigRec));

    cfgMerge(bSSLRequired, FALSE);
    cfgMergeArray(aRequirement);

    return new;
}

/*
 * Directive Rewriting
 */

char *ssl_hook_RewriteCommand(cmd_parms *cmd, const char *cmd_line)
{
    return ssl_compat_directive(cmd->server, cmd->pool, cmd_line);
}

/*
 *  Configuration functions for particular directives
 */

const char *ssl_cmd_SSLMutex(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    if (cmd->server->is_virtual)
        return "SSLMutex: Not allowed inside <VirtualHost> - "
               "Only configurable for main server";
    if (strcEQ(arg, "none")) {
        ssl_ModConfig->nMutexMode  = SSL_MUTEXMODE_NONE;
    }
    else if (strlen(arg) > 5 && strcEQn(arg, "file:", 5)) {
        ssl_ModConfig->nMutexMode  = SSL_MUTEXMODE_FILE;
        ssl_ModConfig->szMutexFile = ap_server_root_relative(cmd->pool, arg+5);
    }
    else if (strcEQ(arg, "sem")) {
#ifdef SSL_CAN_USE_SEM
        ssl_ModConfig->nMutexMode  = SSL_MUTEXMODE_SEM;
#else
        return "SSLMutex: Semaphores not available on this platform";
#endif
    }
    else
        return "SSLMutex: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLPassPhraseDialog(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);

    if (cmd->server->is_virtual)
        return "SSLPassPhraseDialog: Not allowed inside <VirtualHost> - "
               "Only configurable for main server";

    if (strcEQ(arg, "builtin")) {
        pConfig->nPassPhraseDialogType  = SSL_PPTYPE_BUILTIN;
        pConfig->szPassPhraseDialogPath = NULL;
    }
    else if (strlen(arg) > 2 && arg[0] == '/') {
        pConfig->nPassPhraseDialogType  = SSL_PPTYPE_FILTER;
        pConfig->szPassPhraseDialogPath = ap_server_root_relative(cmd->pool, arg);
        if (!ssl_util_path_check(SSL_PCM_EXISTS, pConfig->szPassPhraseDialogPath))
            return "SSLPassPhraseDialig: file not exists";
    }
    else 
        return "SSLPassPhraseDialog: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLEngine(
    cmd_parms *cmd, char *struct_ptr, int flag)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);

    pConfig->bEnabled = (flag ? TRUE : FALSE);
    return NULL;
}

const char *ssl_cmd_SSLCipherSuite(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szCipherSuite = arg;
    return NULL;
}

const char *ssl_cmd_SSLCertificateFile(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szCertificateFile = arg;
    if (!ssl_util_path_check(SSL_PCM_EXISTS|SSL_PCM_ISREG|SSL_PCM_ISNONZERO, arg))
        return "SSLCertificateFile: certificate file not existing or empty";
    return NULL;
}

const char *ssl_cmd_SSLCertificateKeyFile(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szKeyFile = arg;
    if (!ssl_util_path_check(SSL_PCM_EXISTS|SSL_PCM_ISREG|SSL_PCM_ISNONZERO, arg))
        return "SSLCertificateKeyFile: file not exists or empty";
    return NULL;
}

const char *ssl_cmd_SSLCACertificatePath(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szCACertificatePath = arg;
    if (!ssl_util_path_check(SSL_PCM_EXISTS|SSL_PCM_ISDIR, arg))
        return "SSLCACertificatePath: directory not exists";
    return NULL;
}

const char *ssl_cmd_SSLCACertificateFile(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szCACertificateFile = arg;
    if (!ssl_util_path_check(SSL_PCM_EXISTS|SSL_PCM_ISREG|SSL_PCM_ISNONZERO, arg))
        return "SSLCACertificateFile: file not exists or empty";
    return NULL;
}

const char *ssl_cmd_SSLCACertificateReqFile(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szCACertificateReqFile = arg;
    if (!ssl_util_path_check(SSL_PCM_EXISTS|SSL_PCM_ISREG|SSL_PCM_ISNONZERO, arg))
        return "SSLCACertificateReqFile: file not exists or empty";
    return NULL;
}

const char *ssl_cmd_SSLVerifyClient(
    cmd_parms *cmd, char *struct_ptr, char *level)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);

    if (strEQ(level, "0") || strcEQ(level, "none"))
        pConfig->nVerifyClient = VERIFY_NONE;
    else if (strEQ(level, "1") || strcEQ(level, "optional"))
        pConfig->nVerifyClient = VERIFY_OPTIONAL;
    else if (strEQ(level, "2") || strcEQ(level, "require"))
        pConfig->nVerifyClient = VERIFY_REQUIRE;
    else if (strEQ(level, "3") || strcEQ(level, "optional_no_ca"))
        pConfig->nVerifyClient = VERIFY_OPTIONAL_NO_CA;
    else
        return "SSLVerifyClient: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLVerifyDepth(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->nVerifyDepth = atoi(arg);
    if (pConfig->nVerifyDepth < 0)
        return "SSLVerifyDepth: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLSessionCache(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    if (cmd->server->is_virtual)
        return "SSLSessionCache: Not allowed inside <VirtualHost> - "
               "Has to be configured in main server only";
    if (strcEQ(arg, "none")) {
        ssl_ModConfig->nSessionCacheMode      = SSL_SCMODE_NONE;
        ssl_ModConfig->szSessionCacheDataFile = NULL;
    }
    else if (strlen(arg) > 4 && strcEQn(arg, "dbm:", 4)) {
        ssl_ModConfig->nSessionCacheMode      = SSL_SCMODE_DBM;
        ssl_ModConfig->szSessionCacheDataFile = ap_server_root_relative(cmd->pool, arg+4);
    }
    else
        return "SSLSessionCache: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLSessionCacheTimeout(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->nSessionCacheTimeout = atoi(arg);
    if (pConfig->nSessionCacheTimeout < 0)
        return "SSLSessionCacheTimeout: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLFakeBasicAuth(
    cmd_parms *cmd, char *struct_ptr)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->bFakeBasicAuth = TRUE;
    return NULL;
}

const char *ssl_cmd_SSLLog(
    cmd_parms *cmd, char *struct_ptr, char *arg)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);
    pConfig->szLogFile = arg;
    return NULL;
}

const char *ssl_cmd_SSLLogLevel(
    cmd_parms *cmd, char *struct_ptr, char *level)
{
    SSLSrvConfigRec *pConfig = mySrvConfig(cmd->server);

    if (strcEQ(level, "none"))
        pConfig->nLogLevel = SSL_LOG_NONE;
    else if (strcEQ(level, "error"))
        pConfig->nLogLevel = SSL_LOG_ERROR;
    else if (strcEQ(level, "warn"))
        pConfig->nLogLevel = SSL_LOG_WARN;
    else if (strcEQ(level, "info"))
        pConfig->nLogLevel = SSL_LOG_INFO;
    else if (strcEQ(level, "debug"))
        pConfig->nLogLevel = SSL_LOG_DEBUG;
    else
        return "SSLLogLevel: Invalid argument";
    return NULL;
}

const char *ssl_cmd_SSLRequireSSL(
    cmd_parms *cmd, SSLDirConfigRec *rec, char *cipher)
{
    rec->bSSLRequired = TRUE;
    return NULL;
}

const char *ssl_cmd_SSLRequire(
    cmd_parms *cmd, SSLDirConfigRec *rec, char *cpExpr)
{
    ssl_expr *mpExpr;
    SSLRequirement *pReqRec;
    
    if ((mpExpr = ssl_expr_comp(cmd->pool, cpExpr)) == NULL)
        return ap_pstrcat(cmd->pool, "SSLRequire: ", ssl_expr_get_error(), NULL);
    pReqRec = ap_push_array(rec->aRequirement);
    pReqRec->cpExpr = ap_pstrdup(cmd->pool, cpExpr);
    pReqRec->mpExpr = mpExpr;
    return NULL;
}

