/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include <string.h>
#include <stdlib.h>

#include "pgplib.h"
#include "pgp_m.h"

/* SSLeay */
#include <rsa.h>			/* includes bn.h */
#include <idea.h>



/*
 * We find the key, and add the UserID, unless it is already present;  
 * both the length and the characters themselves have to match (case
 * sensitive).
 * We add the UserID to the UserID database with the KeyID as
 * data.
 * We add the UserID to the reversUID database with the keyID as key
 * and UID as data.
 */

int
add_user_id(u_quad_t	key_id,
	    PGPuserid_t	*user_id)
{
    
    u_char	len;
    u_char 	*rec;
    int		uid_len;
    int 	found;
    int 	sofar, buffer_len;
    int		found_pubkey;
    /* scratch */
    int		i;
    u_char	*p, *q;
    
    i = get_pub_key_record(key_id, &rec, &buffer_len);
    if ( i != 0 ) {
	fprintf(stderr, "Can't add UserID (unknown key)");
	return(i);
    }

    /* We parse the buffer, skipping things we don't care about,
     * looking for a matching UserID.
     * if we don't find a match, we add this at the end.
     */
    found = 0;
    found_pubkey = 0;
    sofar = 0;
    while ((sofar < buffer_len) && (found == 0))
    {
	switch ( (rec[sofar] & CTB_TYPE_MASK) >> 2) {
	default:
	    i = packet_length(rec+sofar);
	    sofar += i;
	    break;
	case CTB_CERT_PUBKEY_TYPE:
	    /* Just to be sure */
	    if ( found_pubkey == 1 ) {
		fprintf(stderr, "Found two keys at ID=%#.16llx\n", key_id);
		return(-1);
	    }
	    else
		found_pubkey = 1;
	    i = packet_length(rec+sofar);
	    sofar += i;
	    break;

	case CTB_USERID_TYPE:
	    q = rec+sofar;
	    q++;			/* Skip the CTB */
	    len = *q;
	    i = strlen(user_id->name);
	    if ( i == len ) {
		q++;
		i = strncmp(q, user_id->name, len);
		if ( i == 0 ) {
		    found = 1;
		    break;
		}
	    }
	    i = packet_length(rec + sofar);
	    sofar += i;
	    break;
	}
    } /* While */

    if ( found == 0 ) {

	/* If we get here, we have a new UserID.  Let's add it. */
	uid_len = uid2buf(user_id, &p);
	if ( uid_len == 0 )
	    return(-1);
	q = realloc(rec, buffer_len + uid_len);
	if ( q == NULL ) {
	    fprintf(stderr, "Out of memory\n");
	    free(p);
	    free(rec);
	    return(-1);
	}
	rec = q;
	memcpy(rec + buffer_len, p, uid_len);
	free(p);
	i = store_pub_key_record(key_id, REPLACE, rec, 
				 buffer_len + uid_len);
	if ( i != 0 ) {
	    fprintf(stderr, "Couldn't store key with UID\n");
	    return(-1);
	}
    }
    free(rec);

    /* Then, store the string in the string_db with the KeyID as data.
     * this ensures fast lookup.  We try even if FOUND==TRUE, since we
     * might update the database. 
     */
    i = store_user_id(user_id, key_id);
    if ( i == 1 && found != 1)
	fprintf(stderr, "WARNING: Two identical UserID's exists\n");
    return(i);
}


/*
 * Adding a signature to a key, verifying that it isn't already
 * there.
 * You may add a signature for a non-existant UserID since I don't
 * know if it is incorrect.
 * If we added, return 0.  No adding returns 1, error gives -1.
 */

int
add_sig_to_keyid(u_quad_t	key_id,
		 PGPsig_t 	*sig)
{

    u_char 	*buf;
    int 	found;
    int		buffer_len, sofar, used;
    res_t	what;
    u_char	*sig_buf;
    int		sig_len;
    PGPsig_t	*tmpsig;

    /* scratch */
    u_char	*p;
    int 	i, j;

    i = get_pub_key_record(key_id, &buf, &buffer_len);
    if ( i != 0 ) {
	fprintf(stderr, "Couldn't add sig to key\n");
	return(i);
    }

    sofar = 0;
    found = 0;

    while ( sofar < buffer_len && found == 0 ) {
	i = get_keyentry_from_buffer( buf + sofar, buffer_len - sofar, &what, 
				      &tmpsig, NULL, NULL, NULL, &used);
	if ( i == 1 || i == -1 )
	    return(-1);

	sofar += used;

	if ( what != SIG ) {
	    continue;
	}
	/* We must verify that the signature at hand is different from
	 * the one already on the key.  We first check if it is made
	 * in the same date, then if it is made with the same key
	 * (long comparison is faster than quad).  If both matches,
	 * we dig out the actual signatures and compare them.
	 */
	assert(tmpsig != NULL );

	if ( tmpsig->timestamp == sig->timestamp ) {
	    if ( tmpsig->key_id == sig->key_id ) {
		j = BN_cmp(tmpsig->I, sig->I);
		if ( j == 0 ) {
		    /* A copy */
		    free_signature(tmpsig);
		    found = 1;
		    /* The need for this break was found by means
		     * of dbmalloc---the best tool there is.
		     */
		    break;
		}
	    }
	}
	/* Nope, not this one */
	free_signature(tmpsig);
	continue;
    }

    if ( found == 1 ) {
	/* It was already there */
	free(buf);
	return(1);
    }

    /* This signature wasn't on the key.  Let us add it. */
    sig_len = signature2buf(sig, &sig_buf);
    if ( sig_len < 1 ) {
	free(buf);
	return(-1);
    }
    p = realloc(buf, buffer_len + sig_len);
    if ( p == NULL ) {
	fprintf(stderr, "Out of memory\n");
	free(buf);
	return(-1);
    }
    buf = p;
    memcpy(buf+buffer_len, sig_buf, sig_len);
    free(sig_buf);
    j = store_pub_key_record(key_id, REPLACE, buf, buffer_len+sig_len);
    free(buf);
    if ( j != 0 ) {
	if ( j == 1 )	
	    fprintf(stderr, "REPLACE failed\n");
	else
	    fprintf(stderr, "Error from store_pub_key_record\n");
	return(-1);
    }
    return(0);
}

PGPpubkey_t *
copy_pubkey(PGPpubkey_t *key)
{
    PGPpubkey_t *tmp;

    tmp = malloc(sizeof(PGPpubkey_t));
    if ( tmp == NULL ) 
	return(NULL);
    /* We copy everyting, then make new BIGNUMs */
    memcpy(tmp, key, sizeof(PGPpubkey_t));
    tmp->N = BN_dup(key->N);
    if ( tmp->N == NULL )
	goto bad;
    tmp->E = BN_dup(key->E);
    if (tmp->E == NULL )
	goto bad;

    return(tmp);
    
bad:
    BN_free(tmp->N);
    BN_free(tmp->E);
    free(tmp);
    return(NULL);
}

void
free_pubkey(PGPpubkey_t	*key)
{
    if ( key == NULL )
	return;
    if ( key->N != NULL )
	BN_free(key->N);

    if ( key->E != NULL )
	BN_free(key->E);
    return;
}

/* 
 * In order to fetch just a public key, we fetch everything
 * stored on this keyID,  Then copy out the key proper and
 * free'ing the rest.
 * Return -1 on error, 1 on no-such-key and 0 on OK.
 */

int
fetch_only_pubkey(u_quad_t 	key_id,
		  PGPpubkey_t 	**key)
{
    keyent_t	**recp;
    PGPpubkey_t	*tmpkey;
    int		index;
    /* scratch */
    int 	i;

    i = get_keycontents(key_id, &recp);
    if ( i == -1 || i == 1 )
	return(i);
    assert(i == 0);
    
    tmpkey = NULL;
    for ( index = 0; recp[index] != NULL; index++) {
	switch (recp[index]->what) {
	case SIG:
	case UID:
	    continue;
	case PUBKEY:
	    if ( tmpkey != NULL )
		fprintf(stderr, "Two keys (%#.16llx and %#.16llx) on record %#.16llx\n", 
			BN_get_keyid(tmpkey->N), 
			BN_get_keyid(recp[index]->u.key->N),
			key_id);
	    tmpkey = copy_pubkey(recp[index]->u.key);
	    if ( tmpkey == NULL ) {
		free_keyrecords(recp);
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    continue;
	default:
	    assert(0);
	    /*NOTREACHED*/
	}
    }
    free_keyrecords(recp);
    *key = tmpkey;

    return(0);
}


/*
 * Get a public key, and parse out the different packages that 
 * we find there.  Return a pointer to an array of pointers to 
 * structures.  Everything is malloc'ed and must be freed by the 
 * caller; call free_keyentry (see below).
 * A return of 1 means no such key, -1 error, 0=OK.
 */


int
get_keycontents(u_quad_t	key_id,
		keyent_t	***keyent)
{
    int		buf_len, used;
    u_char	*keybuf;
    keyent_t	**arr;
    int		ent;
    int		offset;
    res_t	what;
    PGPsig_t	*sig;
    PGPpubkey_t	*key;
    PGPuserid_t	*uid;
    int		ret;
    
    int		i;
    char	*p;

    i = get_pub_key_record(key_id, &keybuf, &buf_len);
    if ( i != 0 )
	return(1);

    /* Ensure our array is NULL terminated */
    arr = calloc(1, sizeof(keyent_t **));
    if ( arr == NULL ) {
	fprintf(stderr, "Out of memory\n");
	free(keybuf);
	return(-1);
    }

    ent = 0;
    offset = 0;
    used = 0;
    
    while ( offset < buf_len ) {
	ret = get_keyentry_from_buffer(keybuf+offset, 
				       buf_len-offset, 
				       &what, &sig, &key, 
				       &uid, NULL, &used);
	if ( ret != 0 ) {
	    /* free memory */
	    return(-1);
	}

	offset += used;

	p = realloc(arr, (ent+2) * sizeof(keyent_t *));
	if ( p == NULL ) {
	    free(arr);
	    fprintf(stderr, "Out of memory\n");
	    return(-1);
	}
	arr = (keyent_t **)p;
	arr[ent+1] = NULL;
	
	arr[ent] = malloc(sizeof(keyent_t));
	if ( arr[ent] == NULL ) {
	    fprintf(stderr, "out of memory\n");
	    return(-1);
	}
	arr[ent]->what = what;

	switch (what) {
	default:
	case NONE:
	    assert(0);
	case PUBKEY:
	    arr[ent]->u.key = key;
	    break;
	case SIG:
	    arr[ent]->u.sig = sig;
	    break;
	case UID:
	    arr[ent]->u.uid = uid;
	    break;
	}
	ent += 1;
    }
    
    *keyent = arr;
    return(0);
}

void
free_keyrecords(keyent_t **records)
{
    int		index;

    if ( records == NULL || records[0] == NULL )
	return;
    for(index=0; records[index] != NULL; index += 1) {
	switch (records[index]->what) {
	case PUBKEY :
	    free_pubkey(records[index]->u.key);
	    free(records[index]->u.key);
	    break;
	case SIG:
	    free_signature(records[index]->u.sig);
	    free(records[index]->u.sig);
	    break;
	case UID:
	    free_uid(records[index]->u.uid);
	    free(records[index]->u.uid);
	    break;
	default: 
	    assert(0);
	    /*NOTREACHED*/
	}
    }
    free(records);

    return;
}
	    
    

int
print_keyrecords(u_char	*buffer, 
		 int 	buf_len, 
		 FILE 	*fd)
{
    u_char	*packet;
    int		p_len;
    int		how_much;
    PGPpubkey_t	*key;
    PGPsig_t	*sig;
    PGPuserid_t	*uid;
    int		have_key;
    u_char	CTB;
    /* scratch */
    int		i;

    packet = NULL;
    how_much = 0;

    while ( how_much < buf_len ) {

	free(packet);
	packet = NULL;

	i = find_packet_in_buffer(buffer+how_much, buf_len-how_much, 
				  &packet, &p_len);
	switch( i ) {
	case -1:
	    free(packet);
	    fprintf(stderr, "Some error in print_keyrecords\n");
	    return(-1);
	case 1:
	    free(packet);
	    if ( buf_len != how_much)
		fprintf(stderr, "Syntax error in buffer\n");
	    return(1);
	case 0:
	    /* We found a package.  We consumed j bytes */
	    how_much += p_len;
	    break;
	default:
	    assert(0);
	}
	
	switch ( (packet[0] & CTB_TYPE_MASK) >> 2 ) {
	case CTB_COMMENT_TYPE:
	case CTB_CKE_TYPE:
	case CTB_COMPRESSED_TYPE:
	    fprintf(stderr, "CTB = %x\n", packet[0]);
	    assert(0);
	    /*NOTREACHED*/
	case CTB_CERT_SECKEY_TYPE: {
	    u_short	us;
	    
	    memcpy(&us, packet+11, 2);
	    us = ntohs(us);
	    fprintf(fd, "\tSecret key (length=%d)!! \n", us);
	    continue;
	}
 	case CTB_KEYCTRL_TYPE:
	    fprintf(fd, "\tKey Control Packet\n");
	    continue;
	case CTB_CERT_PUBKEY_TYPE:
	    have_key = 1;
	    key = malloc(sizeof(PGPpubkey_t));
	    if ( key == NULL ) {
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    i = buf2pubkey(packet, p_len, key);
	    fprintf(fd, "\tPublic Key: %#.16llx\n", BN_get_keyid(key->N));
	    free_pubkey(key);
	    free(key);
	    continue;
 	case CTB_USERID_TYPE:
	    uid = malloc(sizeof(PGPuserid_t));
	    if ( uid == NULL ) {
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    i = buf2userid(packet, p_len, uid);
	    fprintf(fd, "\tUserID: %s\n", uid->name);
	    free_uid(uid);
	    free(uid);
	    continue;

	case CTB_SKE_TYPE: {
	    /* a local variable */
	    PGPuserid_t	**uidp;

	    sig = malloc(sizeof(PGPsig_t));
	    if ( sig == NULL ) {
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    i = buf2signature(packet, p_len, sig);
	    fprintf(fd, "\tSignature with KeyID: %#.16llx\n", sig->key_id);
	    i = find_uids_with_key(sig->key_id, &uidp);
	    if ( i == 1 || i == -1)
		fprintf(fd, "\t\tUnknown\n");
	    else {
		for(i = 0; uidp[i] != NULL; i++) {
		    fprintf(fd, "\t\t\t%s\n", uidp[i]->name);
		    free_uid(uidp[i]);
		}
		free(uidp);
	    }
	    free_signature(sig);
	    free(sig);
	    continue;
	}
	default:
	    fprintf(stderr, "Unknown CTB=%d\n", CTB);
	    assert(0);
	    /*NOTREACHED*/
	}
    } /* while */	
    return(0);
}


/* 
 * Return the size of the buffer or 0
 */

int
pubkey2buf(PGPpubkey_t 	*key, 
	   u_char 	**bufptr)
{
    u_char	CTB;
    u_char 	*buf;
    short 	hs;
    unsigned long hl;
    int		howfar;
    int 	i;
    

    CTB = CTB_CERT_PUBKEY;

    buf = calloc(1, key->length);
    if ( buf == NULL ) {
	fprintf(stderr, "Out of memory \n");
	return(-1);
    }
    howfar = 0;

    /* CTB */
    buf[howfar] = CTB;
    howfar += 1;

    /* length */
    hs = htons( (u_short) (key->length - 1 - ctb_llength(CTB)));
    memcpy(buf+howfar, &hs, 2);
    assert ( ctb_llength(CTB) == 2);
    howfar += 2;

    /* version */
    buf[howfar] = key->version;
    howfar += 1;

    /* timestamp */
    hl = htonl(key->created);
    memcpy(buf+howfar, &hl, SIZEOF_TIMESTAMP);
    howfar += SIZEOF_TIMESTAMP;

    /* Validity */
    hs = htons( (u_short)key->valid);	
    memcpy(buf+howfar, &hs, 2);
    howfar += 2;

    /* algorithm */
    buf[howfar] = key->algorithm;
    howfar += 1;

    /* prefix to N */
    hs = htons( BN_num_bits(key->N));
    memcpy(buf+howfar, &hs, 2);
    howfar += 2;

    /* N */
    i = BN_bn2bin(key->N, buf+howfar);
    if ( i == 0 ) {
	free(buf);
	return(0);
    }
    howfar += i;

    /* prefix to E */
    hs = htons( BN_num_bits(key->E));
    memcpy(buf+howfar, &hs, 2);
    howfar += 2;

    /* E */
    i = BN_bn2bin(key->E, buf+howfar);
    if ( i == 0 ) {
	free(buf);
	return(0);
    }
    howfar += i;

    if ( key->length != howfar) {
	fprintf(stderr, "Hmmmm, length doesn't match\n");
	free(buf);
	return(0);
    }
    *bufptr = buf;
    return(howfar);
}
	   

/*
 * Take a buffer and convert the thing into a struct.  It must be
 * assumed that this indeed is the binary representation of a 
 * key.  The only thing we can check is the CTB and look for a
 * reasonable size of N.
 */

int
buf2pubkey(u_char	*buf, 
	   int 		len, 
	   PGPpubkey_t  *key)   
{
    int		sofar;
    /* scratch */
    u_long 	l;
    u_short	us;
    int		i;
    
    if ( buf == NULL || len < 3 || key == NULL) {
	return(-1);
    }

    sofar = 0;
    memset(key, 0, sizeof(PGPpubkey_t));

    if ( ! is_ctb_type(buf[sofar], CTB_CERT_PUBKEY_TYPE )) {
	fprintf(stderr, "Buffer doesn't hold a public key\n");
	return(-1);
    }
    sofar++;

    /* length of the _entire_ packet */
    key->length = packet_length(buf);
    if ( key->length < len ) {
	fprintf(stderr, "Buffer corrupted\n");
	free(key);
	return(-1);
    }
	
    sofar += ctb_llength(*buf);

    /*version */
    memcpy( &key->version, buf+sofar, 1);
    sofar++;

    /* time */
    l = 0;
    memcpy( &l, buf+sofar, SIZEOF_TIMESTAMP);
    key->created = ntohl(l);
    sofar += SIZEOF_TIMESTAMP;

    /* validity */
    us = 0;
    memcpy( &us, buf+sofar, 2);
    key->valid = ntohs(us);
    sofar += 2;

    /* Read algotithm */
    memcpy( &key->algorithm, buf+sofar, 1);
    if (key->algorithm != RSA_ALGORITHM_BYTE )
	goto bad;
    sofar++;

    i = mpi2bignum(buf+sofar, len-sofar, &key->N);
    if ( i == 0 )
	goto bad;
    sofar += i;
    i = mpi2bignum(buf+sofar, len-sofar, &key->E);
    if (i == 0 )
	goto bad;

    return(0);

    /* We only get here if something is wrong.  Notice that free(NULL)
     * is not an error.
     */
bad:
    BN_free(key->N);
    BN_free(key->E);
    return(-1);
}


/*
 * Routines to manipulate secret keys.
 */

void
free_seckey(PGPseckey_t	*key)
{
    if ( key == NULL)
	return;
    if ( key->N != NULL )
	BN_free(key->N);
    if ( key->E != NULL )
	BN_free(key->E);
    if ( key->D != NULL )
	BN_clear_free(key->D);
    if ( key->P != NULL )
	BN_clear_free(key->P);
    if ( key->Q != NULL )
	BN_clear_free(key->Q);
    if ( key->U != NULL )
	BN_clear_free(key->U);
    memset(key, 0, sizeof(PGPseckey_t));
}

/*
 * Turn a buffer into a secret key. 
 * We must assume that the buffer contains a key, but try 
 * to do verification as we go.
 * Return -1 on error, 0 if all is well.
 */


int
buf2seckey(u_char	*buf,
	   int		buflen,
	   PGPseckey_t	*key)
{
    int		sofar;
    u_long	ul;
    u_short	us;
    /* scratch */
    int		i;

    sofar = 0;
    
    if ( ! is_ctb_type(buf[sofar], CTB_CERT_SECKEY_TYPE )) {
	fprintf(stderr, "Buffer doesn't hold a secret key\n");
	return(-1);
    }
    sofar++;

    /* length of the _entire_ packet */
    key->length = packet_length(buf);
    if ( key->length < buflen ) {
	fprintf(stderr, "Buffer corrupted\n");
	return(-1);
    }
    sofar += ctb_llength(*buf);

    /* version*/
    key->version = buf[sofar];
    assert ( key->version == VERSION_BYTE_NEW || key->version == VERSION_BYTE_OLD);
    sofar += 1;
    /* timestamp */
    memcpy(&ul, buf+sofar, 4);
    key->created = ntohl(ul);
    sofar += 4;
    /* Validity */
    memcpy(&us, buf+sofar, 2);
    key->valid = ntohs(us);
    sofar += 2;
    /* RSA algorithm byte */
    key->pk_alg = buf[sofar];
    assert( key->pk_alg == RSA_ALGORITHM_BYTE);
    sofar += 1;
    /* Public N */
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->N);
    if ( i == 0 )
	goto bad;
    sofar += i;
    /* Public E */
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->E);
    if ( i == 0 )
	goto bad;
    sofar += i;
    /* Type of chipher used */
    key->pk_alg = buf[sofar];
    sofar += 1;
    /* Eight bytes of iv */
    memcpy(key->iv, buf+sofar, 8);
    sofar += 8;
    
    /* Then the four private numbers */
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->D);
    if ( i == 0 )
	goto bad;
    sofar += i;
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->P);
    if ( i == 0 )
	goto bad;
    sofar += i;    
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->Q);
    if ( i == 0 )
	goto bad;
    sofar += i;
    i = mpi2bignum(buf+sofar, buflen-sofar, &key->U);
    if ( i == 0 )
	goto bad;

    /* And, at the end, the checksum */
    if ( sofar+2 > buflen )
	goto bad;
    memcpy(&key->cksum, buf+sofar, 2);

    return(0);

bad:
    free_seckey(key);
    return(1);
}

void
print_seckey(int	verbose,
	     FILE 	*fd,
	     PGPseckey_t *key)
{
    u_quad_t	q;

    fprintf(fd, "Length	 \t%d\n", (int)key->length);
    fprintf(fd, "Version \t%d\n", (int)key->version);
    fprintf(fd, "Created \t%ld\n", key->created);
    fprintf(fd, "Valid	\t%d\n", key->valid);
    if ( verbose ) {
	fprintf(fd, "N		");
	BN_print_fp(fd, key->N);
	fprintf(fd, "\nE \t\t");
	BN_print_fp(fd, key->E);
	fprintf(fd, "\n");
    }
    else {
	q = BN_get_keyid(key->N);
	fprintf(fd, "KeyID \t\t%#.16llx\n", q);
    }
    fprintf(fd, "Alg \t\t%s\n", key->pk_alg ==  RSA_ALGORITHM_BYTE ?
	"RSA\n" : "Unknown\n");
}

/*
 * In order to actually use a secret key, the secret parts must
 * be decrypted.  The key is now armed, reeady for use, and with 
 * all your secrets exposed.  Use it with some care.
 */

int
decrypt_seckey(PGPseckey_t 	*key, 
	       u_char 		*digest)
{
    BIGNUM	*tmp, *P1, *Q1, *D1, *U1;
    u_char	*buf;
    IDEA_KEY_SCHEDULE ks;
    int		len1, len2;
    u_char	iv[8];
    int		useiv;			/* Keep track of IV usage */
    /* scratch */
    int		i;

    idea_set_encrypt_key(digest, &ks);
    memcpy(iv, key->iv, 8);
    useiv = 0;
    
    /* Do the number D */
    len1 = BN_num_bytes(key->D);
    buf = malloc(len1);
    len2 = BN_bn2bin(key->D, buf);
    assert(len1 == len2);
    idea_cfb64_encrypt(buf, buf, len1, &ks, iv, &useiv, IDEA_DECRYPT);
    tmp = NULL;
    D1 = BN_bin2bn(buf, len1, NULL);
    memset(buf, 0, len1);

    /* The number P */
    len1 = BN_num_bytes(key->P);
    buf = malloc(len1);
    len2 = BN_bn2bin(key->P, buf);
    assert(len1 == len2);
    idea_cfb64_encrypt(buf, buf, len1, &ks, iv, &useiv, IDEA_DECRYPT);
    tmp = NULL;
    P1 = BN_bin2bn(buf, len1, NULL);
    memset(buf, 0, len1);

    /* The number Q */
    len1 = BN_num_bytes(key->Q);
    buf = malloc(len1);
    len2 = BN_bn2bin(key->Q, buf);
    assert(len1 == len2);
    idea_cfb64_encrypt(buf, buf, len1, &ks, iv, &useiv, IDEA_DECRYPT);
    tmp = NULL;
    Q1 = BN_bin2bn(buf, len1, NULL);
    memset(buf, 0, len1);

    /*
     * There are two ways for us to check whether the password was correct
     * or not: Either verify that P*Q = N, or using the PGP checksum
     * provided with the key.   The latter is calculated over the MPI's 
     * (that is, on the numbers, after decryption, but including the
     * prefixed bitcount).  Since we store numbers as BIGNUMs we use the
     * former method.
     * As usual, I solicit your comments.
     */

    tmp = BN_new();
    i = BN_mul(tmp, P1, Q1);
    if ( i == 0 ) {
	fprintf(stderr, "Error multiplying P and Q\n");
	BN_free(tmp);
	/* One of them might be in order; clear just in case */
	BN_clear_free(P1);
	BN_clear_free(Q1);
	BN_clear_free(D1);
        return(-1);
    }
    if ( BN_cmp(tmp, key->N) != 0 ) {
	fprintf(stderr, "Incorrect password - secret key not armed\n");
        BN_free(tmp);
	/* One of them might be in order; clear just in case*/
	BN_clear_free(D1);
	BN_clear_free(P1);
	BN_clear_free(Q1);
	return(1);
    }
    BN_free(tmp);

    /* The number U */
    len1 = BN_num_bytes(key->U);
    buf = malloc(len1);
    len2 = BN_bn2bin(key->U, buf);
    assert(len1 == len2);
    idea_cfb64_encrypt(buf, buf, len1, &ks, iv, &useiv, IDEA_DECRYPT);
    U1 = BN_bin2bn(buf, len1, NULL);
    memset(buf, 0, len1);

    BN_clear_free(key->D);
    key->D = D1;
    BN_clear_free(key->P);
    key->P = P1;
    BN_clear_free(key->Q);
    key->Q = Q1;
    BN_clear_free(key->U);
    key->U = U1;

    return(0);
}

/* Remove the secrets stored in the key. */

void
clear_seckey(PGPseckey_t	*key)
{
    BN_clear(key->D);
    BN_clear(key->P);
    BN_clear(key->Q);
    BN_clear(key->U);
}
