//	WINEXEJN.CPP
//
//	Execute another application and wait until it completes.
//
//	Adapted by John Navas from code made available by Microsoft.

#define STRICT
#include <windows.h>
#include <shellapi.h>
#include <ctype.h>
#include <direct.h>
#include <stdlib.h>

#include "winexejn.h"

////////////////////////////////////////////////////////////////////////////////

// used to search for window of executed program
typedef struct _ENUMINFO {
	HINSTANCE hInstance;        // Supplied on input.
	HWND hWnd;                  // Filled in by enum function.
} ENUMINFO, FAR *LPENUMINFO;

// Define a structure used to hold info about an app we
// start with ExecApp so that this info can be used again
// later to test if the app is still running
typedef struct _EXECAPPINFO {
    HINSTANCE hInstance;            // The instance value
    HWND hWnd;                      // The main window handle
    HTASK hTask;                    // The task handle
} EXECAPPINFO, *PEXECAPPINFO;

////////////////////////////////////////////////////////////////////////////////

// search for the window of an executed program
BOOL
CALLBACK
EnumWndProc(
HWND hWnd,
LPARAM lParam
)
{
    HINSTANCE hInstance;
    LPENUMINFO lpInfo;

    lpInfo = (LPENUMINFO) lParam;

    hInstance = (HINSTANCE) GetWindowWord(hWnd, GWW_HINSTANCE);

    if (hInstance == lpInfo->hInstance) {
        lpInfo->hWnd = hWnd;
        return FALSE;
    }

    return TRUE;
}

// Try running an app by calling WinExec.  If it works, return some
// info about the app in the struct pointed to by pInfo.
// It returns RAN_OK, NOT_FOUND or SOME_ERROR.
static
int
TryWinExec(
const char* pszPath,					// pathname of program to execute
const char* pszParams,					// command line parameters
const char* pszCWD,						// working directory
UINT fiCmdShow,							// window state
PEXECAPPINFO pInfo						// return app info here
)
{
    HINSTANCE uiResult;
    ENUMINFO EnumInfo;					// used to enumerate windows

	// execute program
   	uiResult = ShellExecute(NULL, NULL, pszPath, pszParams, pszCWD, fiCmdShow);

    // check results of WinExec
	switch ((UINT) uiResult) {
	case 2:
	case 3:
		return NOT_FOUND;
	default:
	    if (((UINT) uiResult) <= 32)
    	    return SOME_ERROR;
	}

    // The app is running (or at least it was) so return some info about it

    pInfo->hInstance = uiResult;

    // Find the window handle of the instance by enumerating all
    // the main windows until we find one with the correct
    // instance value

    EnumInfo.hInstance = pInfo->hInstance;
    EnumInfo.hWnd = NULL;

	// I don't have to use MakeProcInstance because this goes in a DLL
    EnumWindows((WNDENUMPROC) EnumWndProc, (LPARAM)(LPENUMINFO)&EnumInfo);

    if (! EnumInfo.hWnd) {
        // Didn't find a matching window so the task probably
        // isn't running any more.  Maybe it did what it had to
        // and went away already.
        pInfo->hWnd = NULL;
        pInfo->hTask = NULL;
    }
    else {
        // We have the window handle. Now get the task.
        pInfo->hWnd = EnumInfo.hWnd;
        pInfo->hTask = GetWindowTask(pInfo->hWnd);
    }

    return RAN_OK;
}

////////////////////////////////////////////////////////////////////////////////

// check to see if an app is running
static
BOOL									// TRUE if app is running
IsAppRunning(PEXECAPPINFO pInfo)
{
	// make sure window and task are still valid
    if (! IsWindow(pInfo->hWnd) || ! IsTask(pInfo->hTask))
        return FALSE;
    // check instance against window
    if ((HINSTANCE) GetWindowWord(pInfo->hWnd, GWW_HINSTANCE) != pInfo->hInstance)
        return FALSE;
    // check task against window
    if (GetWindowTask(pInfo->hWnd) != pInfo->hTask)
        return FALSE;
    // app is running!
    return TRUE;
}

////////////////////////////////////////////////////////////////////////////////

int
WinExeJN(
const char* pszPath,					// pathname of program to execute
const char* pszParams,					// command line parameters
const char* pszCWD,						// working directory
UINT fiCmdShow							// window state
)
{
	EXECAPPINFO Info;					// return app info here
	int i;
	HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));	// hourglass

	// do the execution
	i = TryWinExec(pszPath,				// pathname of program to execute
			pszParams,					// command line parameters
			pszCWD,						// working directory
			fiCmdShow,					// window state
			&Info);						// return app info here

	// wait for app to finish
	if (RAN_OK == i)
		while (IsAppRunning(&Info)) {	// use PeekMessage to permit multitasking
			MSG msg;

			if (GetMessage(&msg, NULL, 0, 0)) {

				TranslateMessage(&msg);
				DispatchMessage(&msg);

				if (WM_MOUSEMOVE == msg.message)
					SetCursor(LoadCursor(NULL, IDC_WAIT));	// redisplay hourglass
			}
		}

	SetCursor(hcurSave);				// remove hourglass

	return i;
}

////////////////////////////////////////////////////////////////////////////////

//	WINEXEJN.CPP
