/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 * sun.c -- Sun Architecture-Specific Functions
 */

/* REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/09/12  B. Dorsey          Added skeleton functions
 * 93/09/23  B. Dorsey		Added IncomingPort() and UngetPort()
 * 93/10/13  B. Dorsey		Filled in all skeleton functions
 * 93/10/13  B. Dorsey		Implemented ptt() and quit()
 * 93/10/13  B. Dorsey		Added argument to InitPort to allow
 *				port speed selection.
 * 93/10/20  B. Dorsey		Created AudioFlow()
 * 93/11/01  B. Dorsey		Added OS specific initialization and
 *				cleanup functions SysInit() and SysDone()
 * 94/10/17  B. Dorsey		Added timeout capability to ReadPort() and
 *				fixed a long-standing bug in ReadPort()
 */

#if defined(sun)
#define	AUDIO_DEVICE	"/dev/audio"
#include <stdio.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <sys/filio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stropts.h>
#ifdef SOLARIS
#include <sys/resource.h>
#include <multimedia/audio_device.h>
#include <multimedia/audio_encode.h>
#include <multimedia/audio_errno.h>
#else
#include <multimedia/audio_hdr.h>
#include <multimedia/ulaw2linear.h>
#include <multimedia/libaudio.h>
#include <multimedia/audio_device.h>
#include <sun/audioio.h>
#endif
#elif defined(linux)
#define	AUDIO_DEVICE	"/dev/dsp"
#include <fcntl.h>
#include <termios.h>
#include <linux/soundcard.h>
#endif

#include "machine.h"

#define	IOBUFSZ		256	/* size of i/o buffer */
enum flow {
    RECEIVE, TRANSMIT
};				/* mode for Audioflow() */
int             fdtabsize;	/* size of file descriptor table */

/* return pathname corresponding to port designated by arg */
char           *
GetPort(char *arg)
{
    return arg;
}

static int      s_port;
static int      head, tail;
static UINT8    iobuf[IOBUFSZ + 4];
static int      bauds[] = {
    0, 50, 75, 110, 134, 150, 200, 300, 600,
    1200, 1800, 2400, 4800, 9600, 19200, 38400, -1
};

/* initialize the serial port specified by arg */
int
InitPort(char *arg, UINT16 speed)
{
    int             brate;
    struct termios  tbuf;

    /* computer baud rate argument */
    for (brate = 0; bauds[brate] != -1; brate++)
	if (speed == bauds[brate])
	    break;
    if (bauds[brate] < 0)
	return -1;

    head = tail = 4;
    s_port = open(arg, O_RDWR | O_NOCTTY);
    if (s_port < 0) {
	return -1;
    }
    ioctl(s_port, TCGETS, (char *) &tbuf);
    tbuf.c_cflag &= (CREAD | HUPCL | CLOCAL | CRTSCTS);
    tbuf.c_cflag |= brate | CS8 | CRTSCTS;	/* CRTSCTS? */
    tbuf.c_iflag = 0;
    tbuf.c_oflag = 0;
    tbuf.c_lflag = 0;
    tbuf.c_cc[VMIN] = 1;
    tbuf.c_cc[VTIME] = 0;
    ioctl(s_port, TCSETSF, (char *) &tbuf);

    return 0;
}

/* see if there is data waiting on the previously initialized serial port */
int
IncomingPort(void)
{
    long            n = 0;

    ioctl(s_port, FIONREAD, &n);
    n += tail - head;

    return (int) n;
}

/*
 * Read a byte of data from the previously initialized serial port.
 * If a byte is not currently available, block until it is.  If more
 * bytes are available, buffer them in an internal i/o buffer until
 * they are needed.
 */
int
ReadPort(int time_out)
{
    int             c;
    struct timeval  timeout;
    fd_set          r_fdset;

    if (head < tail)
	c = iobuf[head++];
    else {
	FD_ZERO(&r_fdset);
	FD_SET(s_port, &r_fdset);
	timeout.tv_sec = time_out;
	timeout.tv_usec = 0;
	head = 4;
	if (select(fdtabsize, &r_fdset, (fd_set *) 0,
		   (fd_set *) 0, &timeout) <= 0) {
	    tail = 4;
	    c = -1;
	}
	else {
	    if ((tail = read(s_port, iobuf + 4, IOBUFSZ)) <= 0) {
		tail = 4;
		c = -1;
	    }
	    else {
		tail += 4;
		c = iobuf[head++];
	    }
	}
    }

    return c;
}

/* Push a byte of data back onto input stream */
int
UngetPort(UINT8 c)
{
    if (head > 0) {
	iobuf[--head] = c;
    }
    else {
	return -1;
    }

    return 0;
}

/* write n bytes of data from buf to the previously initialized serial port */
int
WritePort(UINT8 * buf, int n)
{
    return write(s_port, buf, n);
}

static int      a_port;
static int      fr_size;

/*
 * initialize the audio device to the specifications called for by the coder
 * designated by arg
 */
int
InitAudio(int sample_rate, int frame_size)
{
#if defined(sun)
    audio_info_t    info;

    /*
     * The Sun audio device only runs at a 8KHZ sampling rate.
     */
    if (sample_rate != 8000)
        return -1;

    if ((a_port = open(AUDIO_DEVICE, O_RDWR)) < 0)
        return -1;

    /*
     * because of a bug in the SunOS 4.1.3 audio driver, it is necessary to
     * pause and resume recording before the driver will actually begin
     * recording (when opened in O_RDWR mode).
     */
    AUDIO_INITINFO(&info);
    info.record.pause = 1;
    ioctl(a_port, AUDIO_SETINFO, &info);
    ioctl(a_port, I_FLUSH, FLUSHR);
#elif defined(linux)
    /* LXX -- need to add code to set the sampling rate */
    a_port = open(AUDIO_DEVICE, O_RDONLY);
#endif
    fr_size = frame_size;	/* save frame size */

    return 0;
}

/*
 * read n bytes of data into buf from the previously initialized audio device
 */
int
ReadAudio(UINT8 * buf, int n)
{
    if (read(a_port, buf, n * fr_size) == n * fr_size)
        return 0;
    else
    	return -1;
}

/* write n bytes of data from buf to the previously initialized audio device */
int
WriteAudio(UINT8 * buf, int n)
{
    return write(a_port, buf, n * fr_size);
}

/*
 * The variable mode is set to indicate whether we are in transmit mode
 * (recording from the sound device) or in receive mode (playing to the sound
 * device).  Configure the sound hardware appropriately for the mode we're
 * in.
 */
int
AudioFlow(int mode)
{
#if defined(sun)
    audio_info_t    info;

    AUDIO_INITINFO(&info);

    if (mode == RECEIVE) {
	info.play.pause = 0;
	info.record.pause = 1;
    }
    else {
	info.play.pause = 1;
	info.record.pause = 0;
    }
    ioctl(a_port, AUDIO_SETINFO, &info);

    /*
     * if we're entering RECEIVE mode, flush any remaining input if we're
     * entering TRANSMIT mode, wait for output to finish
     */
    if (info.record.pause)
	ioctl(a_port, I_FLUSH, FLUSHR);

    /*
     * else ioctl(a_port, AUDIO_DRAIN, (void *) 0);
     */
#elif defined(linux)
    if (mode == RECEIVE) {
	close(a_port);
	a_port = open(AUDIO_DEVICE, O_WRONLY);
    }
    else {
	close(a_port);
	a_port = open(AUDIO_DEVICE, O_RDONLY);
    }
#endif
    return 0;
}

/*
 * Block until previously queued audio data has been played.
 */
void
DrainAudio(void)
{
    audio_drain(a_port, 0);
}

/*
 * Close the previously initialized audio device
 */
void
CloseAudio(void)
{
    close(a_port);
}

/*
 * Return true if underflow has occurred while outputing to audio device
 */
int
AudioUnderflow(void)
{
    unsigned int    flag;

    if (audio_get_play_error(a_port, &flag) != AUDIO_SUCCESS) {
        fprintf(stderr, "\naudio_get_play_error() failed\n");
        exit(1);
    }
    if (flag) {
        flag = 0;
        audio_set_play_error(a_port, &flag);
        flag = 1;
    }

    return flag;
}

/*
 * Return true if overflow has occurred while reading from audio device
 */
int
AudioOverflow(void)
{
    unsigned int    flag;

    if (audio_get_record_error(a_port, &flag) != AUDIO_SUCCESS) {
        fprintf(stderr, "\naudio_get_record_error() failed\n");
        exit(1);
    }
    if (flag) {
        flag = 0;
        audio_set_record_error(a_port, &flag);
        flag = 1;
    }

    return flag;
}

/*
 * Play the logon voice file.
 */
void
VoiceLogon(char *voice_fname)
{
    int             i, fd;
    UINT8           audio_buf[fr_size];

    if ((fd = open(voice_fname, O_RDONLY)) != -1) {
    	while (read(fd, audio_buf, fr_size) > 0) {
    	    for (i=0; i<fr_size; i++)
    	    	audio_buf[i] = audio_c2u(audio_buf[i] - 128);
	    WriteAudio(audio_buf, 1);
	}
	close(fd);
    }
    DrainAudio();
}

/*
 * convert raw audio data pointed to by the variable data to samples in
 * standard form (floating point values in the range -1<x<1) in memory
 * pointed to by the variable samples.
 */
int
Raw2Std(UINT8 * data, INT16 * samples, int len)
{
    int             i;

    for (i = 0; i < len; i++)
#if defined(sun)
	samples[i] = audio_u2s(data[i]);
#elif defined(linux)
    samples[i] = ((INT16) (data[i] ^ 0x80)) << 8;
#endif
    return 0;
}

/*
 * convert audio data in standard form to raw audio data suitable for output
 * to the audio device.
 */
int
Std2Raw(INT16 * samples, UINT8 * data, int len)
{
    int             i;

    for (i = 0; i < len; i++)
#if defined(sun)
	data[i] = audio_s2u(samples[i]);
#elif defined(linux)
    data[i] = ((UINT8) (samples[i] >> 8)) ^ 0x80;
#endif

    return 0;
}

static          qflag;

/* detect ptt change.  return true if ptt has been pressed, false otherwise */
int
ptt(void)
{
    long            n;
    char            buf[256];

    /* see if there are any characters waiting to be read */
    ioctl(0, FIONREAD, &n);

    if (n > 0) {
	read(0, buf, n);
	if (buf[0] == 'q' || buf[0] == 'Q')
	    qflag = 1;
	else
	    return 1;
    }

    return 0;
}

/* detect user abort.  return true if user desires to abort, false otherwise */
int
quit(void)
{
    return qflag;
}

/* O.S. specific system initialization */
int
SysInit(void)
{
    /* store max number of file descriptors for later use */
#ifdef SOLARIS
    struct rlimit rlp;

    getrlimit(RLIMIT_NOFILE, &rlp);
    fdtabsize = rlp.rlim_cur;
#else
    fdtabsize = getdtablesize();
#endif

    return 0;
}

/* Return time in seconds */
long
Clock(void)
{
    return time((time_t *) 0);
}

/* "real time" millisecond timer */
INT32
Timer(int init)
{
    static long sec, usec;
    struct timeval tp;

    if (init) {
        gettimeofday(&tp, (struct timezone *) 0);
        sec = tp.tv_sec;
        usec = tp.tv_usec;
    }

    gettimeofday(&tp, (struct timezone *) 0);
    return (tp.tv_sec - sec) * 1000 + (tp.tv_usec - usec) / 1000;
}

/* Get passphrase from user without echo */
int
GetPassPhrase(char *pw, int maxlen, char *msg)
{
    int tty_fd;
    FILE *tty;
    struct termios mytty, origtty;

    /* make sure we're talking to a terminal */
    if ((tty = fopen("/dev/tty", "r")) == NULL)
        return -1;
    tty_fd = fileno(tty);

    /* prompt user */
    fputs(msg, stdout);
    fflush(stdout);

    /* turn of echoing */
    tcdrain(tty_fd);
    tcgetattr(tty_fd, &origtty);
    mytty = origtty;
    mytty.c_lflag &= ~ECHO;
    tcsetattr(tty_fd, TCSANOW, &mytty);

    /* get passphrase */
    if (fgets(pw, maxlen, tty) == NULL)
        return -1;
    pw[strlen(pw)-1] = '\0';

    /* restore terminal characteristics */
    tcdrain(tty_fd);
    tcsetattr(tty_fd, TCSANOW, &origtty);
    fclose(tty);
    fputs("\n", stdout);

    return 0;
}
