// === Starter Application 1.0.0 === //


/* ======================================================

FUNCTION:  CreateCreditEvent()

INPUT:     batch - batch object associated with the current batch

      purchaseid - id of the CAPTURED transaction that is to be credited.

RETURN:    a Payevent object associated with the credit transaction.

DESCRIPTION:  Checks to make sure the current batch is in the OPENED status,
	since credits may only take place when the batch is OPENED.  The 
	function then creates a new Payevent object and initializes some
	of it's properties (see code below).  After checking to be sure
	that the "purchaseid" parameter points to a valid CAPTURED transaction
	in the database which is to be credited, a new record is
	created in the LP_PURCHASE table for this credit transaction
	and it is given an initial status of CREDITING.

STATE CHANGES:  Initializes the state of the credit transaction to:
	      CREDITING.


SPECIAL NOTES:  The actual credit is not performed in this function.
	(see "Credit()" in this page).


====================================================== */

var slipID = 0;
function CreateCreditEvent(batch, purchaseid, merchantReference)
{

   var nextPurchaseID       = GetNextPurchaseID();

   database.beginTransaction();

   //
   //  Make sure the batch we were passed is valid and status is OPENED.
   //
   batchCursor = database.cursor("select ID from LP_BATCH where ID = " + batch.ID + " AND status='OPENED'");
     
   if (!batchCursor.next())
   {
		
      batchCursor.close();
      database.rollbackTransaction();
      PrintError("No OPENED batch found.", "No batch with ID " + batch.ID +
                 " and status OPENED was found in the database.  Either the batch " + 
		 " is currently SETTLING or there is no batch with a matching ID in the database.  " + 
		"No captures or credits may take place while a batch is SETTLING.");
   }
   batchCursor.close();

  //
  // Create a payevent object 
  //
  payevent                 = new PayEvent(merchantReference);
  
  //
  // Save payevent information in database
  //
  database.beginTransaction();

  // Set eventID required property

  eventID = GetNextEventID();
  payevent.eventID = eventID;

  // Make sure the CAPTURED transaction you are trying to credit exists
  cursor = database.cursor("select ID, CARDHOLDERNAME, AMOUNT, SLIPID " +
		" from LP_PURCHASE where id = " + purchaseid);

  if(cursor.next())
     error = database.execute("insert into LP_PURCHASE values (" +
             nextPurchaseID + ",'" +
             payevent.merchantReference + "','" +
	     cursor.cardHolderName + "','" +
	     project.currency + "'," +
             cursor.amount + 
             ", null, null, null, " + 
	     eventID + 
             ", null," +
	     cursor.slipID + "," +
	     batch.ID + ", 'CREDITING')"); 
  else
  {	/* Otherwise, the id of the CAPTURED transaction we were passed 
	was invalid */

   	database.rollbackTransaction();
	cursor.close()
	PrintError("Invalid transaction ID", "Invalid CAPTURED purchase with id " + purchaseid + " selected for CREDITING.  Credit aborted.");
  }
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table." +
			"Error code " + error + " returned from the database.");

  cursor.close();
  database.commitTransaction();

  //
  // ID is not a built-in property of the payevent object.
  // It is created here to remember the purchase ID.
  //
  payevent.ID              = nextPurchaseID;

  return (payevent);

}  // END CreateCreditEvent()


/* ======================================================

FUNCTION:  CreateManualCreditEvent()

INPUT:     batch - batch object associated with the current batch.
	   merchantReference - transaction reference number created
	   by the merchant.
	   creditSlipID - slipID used when inserting the transaction record.
	   creditCardHolderName - name of the card holder.
	   credtAmount - dollar amount of the credit transaction.

RETURN:    payevent object, including the purchase ID of the credit transaction.

DESCRIPTION:  Checks to make sure the current batch is in the OPENED status,
	since credits may only take place when the batch is OPENED.  The 
	function then creates a new Payevent object and initializes some of
	it's properties (see code below).  A new record is created in the 
	LP_PURCHASE table for this credit transaction and it is given an 
	initial status of CREDITING.

STATE CHANGES:  Initializes the state of the credit transaction to:
	      CREDITING.

====================================================== */

var slipID = 0;
function CreateManualCreditEvent(batch, merchantReference, creditSlipID, creditCardHolderName, credtAmount)
{

   var nextPurchaseID       = GetNextPurchaseID();
   database.beginTransaction();

   //
   //  Make sure the batch we were passed is valid and status is OPENED.
   //
   batchCursor = database.cursor("select ID from LP_BATCH where ID = " + batch.ID + " AND status='OPENED'");
     
   if (!batchCursor.next())
   {

      batchCursor.close();
      database.rollbackTransaction();
      PrintError("No OPENED batch found.", "No batch with ID " + batch.ID +
                 " and status OPENED was found in the database.  Either the batch " + 
		 " is currently SETTLING or there is no batch with a matching ID in the database.  " + 
		"No captures or credits may take place while a batch is SETTLING.");
   }
   batchCursor.close();

  //
  // Create a payevent object 
  //
  payevent                 = new PayEvent(merchantReference);

  //
  // Save payevent information in database
  //
  database.beginTransaction();

  // Set eventID required property
  eventID = GetNextEventID();
  payevent.eventID = eventID;

     error = database.execute("insert into LP_PURCHASE values (" +
             nextPurchaseID + ",'" +
             payevent.merchantReference + "','" +
	     creditCardHolderName + "','" +
	     project.currency + "'," +
             credtAmount + 
             ", null, null, null, " + 
	     eventID + 
             ", null," +
	     creditSlipID + "," +
	     batch.ID + ", 'CREDITING')"); 

	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not insert into LP_PURCHASE table." +
			"Error code " + error + " returned from the database.");

  database.commitTransaction();

  //
  // ID is not a built-in property of the payevent object.
  // It is created here to remember the purchase ID.
  //
  payevent.ID              = nextPurchaseID;

  return (payevent.ID);

}  // END CreateManualCreditEvent()



/* ======================================================

FUNCTION:  SaveCreditEvent()

INPUT:     payevent - a payevent object associated with the completed
			credit.
	       slip - a slip object associated with the purchase
			that was credited.

RETURN:    none.

DESCRIPTION:  This function is to be called AFTER the actual credit
	has been performed via communication with the acquirer across
	the Internet.  It updates the record in the database for
	this transaction with information relevant to the transaction
	(e.g. eventID, etc.), it's status/state is changed to "CREDITED"
	and the changes are committed to the database.  

STATE CHANGES:  Changes the status/state of the credit transaction in 
	the database from "CREDITING" to "CREDITED"

SPECIAL NOTES:  The actual credit is not performed in this function.
	(see "Credit()" in this page).


====================================================== */

function SaveCreditEvent(payevent, slip)
{

// Assumption:
//  1 slip per purchase
//  1 terminal/merchant pair specified in system.ini 
//     If this is changed in the middle, things will go wrong.

  database.beginTransaction();

  cursor = database.cursor("select * from LP_PURCHASE where ID = " + payevent.ID + " AND status = 'CREDITING'", true);

  if (cursor.next())
  {
     cursor.authCode = "n/a";
     cursor.amount = payevent.amount;
     cursor.paySvcData = "n/a"
     cursor.avsResp = "n/a"
     cursor.eventID = GetNextEventID();
     cursor.eventTime = payevent.eventTime;
     cursor.merchantReference = payevent.merchantReference;
//     var curDate = new Date();
//     cursor.eventTime = curDate;
     cursor.status = "CREDITED";
     

     if ((error = cursor.updateRow("LP_PURCHASE")))
     {
        cursor.close();
        database.rollbackTransaction();
	if ((error == 5) || (error == 7))
		PrintError("Database Error.", "Could not update LP_PURCHASE table.  " +
			"Error code " + error + ": " + database.majorErrorMessage());
	else if (error)
		PrintError("Database Error.", "Could not update LP_PURCHASE table." +
			"Error code " + error + " returned from the database.");
     }
  }
  else
  {
     cursor.close();
     database.rollbackTransaction();
     PrintError("Transaction not found.", "No credit transaction with ID " +
                nextPurchaseID +
                " and status CREDITING in database.");
  }
  cursor.close();

  database.commitTransaction();

} // END SaveCreditEvent



/* ======================================================

FUNCTION:  Credit()

INPUT:     batch - the batch object associated with the current batch.
	   (all other parameters are simply passed through to the 
	   "DoCredit()" function which is called from within this function).

RETURN:    true if the batch is in the OPENED status and the credit
	may take place, false otherwise.

DESCRIPTION:  This function must be called BEFORE DoCredit() is called.
	If this function returns TRUE, the DoCredit() may take place.  
	Otherwise, the Credit may not take place because the batch is
        currently in the "SETTLING" state, and no credits or captures 
	may take place during this time.

====================================================== */

function Credit(purchaseid, batch, merchant, terminal, processor, merchantReference)
{

   var returnval = false;

   //
   //  Make sure the batch we were passed is valid and status is OPENED.
   //
   batchCursor = database.cursor("select ID from LP_BATCH where ID = " + batch.ID + " AND status='OPENED'");
     
   if (!batchCursor.next())
   {
	// The ID was bad or the batch is in the SETTLING state, so the credit
        // cannot take place.

      batchCursor.close();
      database.rollbackTransaction();
      PrintError("No OPENED batch found.", "No batch with ID " + batch.ID +
                 " and status OPENED was found in the database.  Either the batch " + 
		 " is currently SETTLING or there is no batch with a matching ID in the database.  " + 
		"No captures or credits may take place while a batch is SETTLING.");

   }
   else
   {
	batchCursor.close();
      
	// CALL to DoCredit()
      returnval = DoCredit(purchaseid, batch, merchant, terminal, processor, merchantReference);
  
	// returns TRUE if credit was successful
   }

   return returnval;

} // END Credit()



/* ======================================================

FUNCTION:  DoCredit()

CALLED BY:  Credit()

INPUT:	   purchaseid - the id of the credit transaction which is
	currently in the "CREDITING" state.

           batch - the batch object associated with the current batch.

	   merchant,
	   terminal,
	   processor - objects required for the credit transaction
			to take place (previously created)	   

RETURN:    true if the credit is successful, false otherwise.

DESCRIPTION:  Performs the actual credit operation across the Internet.
	This function will attempt to credit the transaction indicated
	by the "purchaseid" parameter.

====================================================== */


function DoCredit(purchaseid, batch, merchant, terminal, processor, merchantReference)
{
   var returnval = false;
   var newEventFlag = false;	

   database.beginTransaction();

   // Select the transaction record indicated by purchaseid and make
   // sure it is a "CAPTURED" or "CREDITING" transaction that we are 
   // crediting back, or that it is a retry of a transaction that is 
   // CREDITING.
   //
   purchaseCursor = database.cursor("select * from LP_PURCHASE where (id = " + purchaseid + ") AND ((status = 'CAPTURED') OR (status='CREDITING'))");

      if (purchaseCursor.next())
      {
         eventID = GetNextEventID();

         slipID = purchaseCursor.slipID;

         //
         // re-construct slip object
         //
         slipCursor = database.cursor("select * from LP_SLIP where ID = " + slipID);
         if (slipCursor.next())
         {
            asciiDER = slipCursor.slipder; 
            slip = new Slip(asciiDER);
            if (slip.bad())
            {
               slipCursor.close();
               purchaseCursor.close();
               database.rollbackTransaction();
               PrintFmtError("Failed to construct slip object from DER.",
                             slip.getStatusMessage());
            }
         } 
         else
         {
            slipCursor.close();
            purchaseCursor.close();
            database.rollbackTransaction();
            PrintError("Slip not found.", "No slip with ID " + slipID + " found in database.  Cannot perform credit.");
         }
         slipCursor.close();
   	 

         //
         // construct payevent object
         //

	if (merchantReference == null)
		merchantReference = slip.merchantReference;

	// Create a new pay event and initialize the status to CREDITING

	 if (purchaseCursor.status == "CAPTURED")
	 {
         	 payevent = CreateCreditEvent(batch, purchaseCursor.id, merchantReference);
	 }
	 else  // otherwise it's status must be "CREDITING" so it is a retry
	 {
		payevent = new PayEvent(merchantReference);	
		payevent.eventID = purchaseCursor.eventID;
		var nextPurchaseID = GetNextPurchaseID();
		payevent.ID = purchaseCursor.id;
	 }
      	 payevent.amount = purchaseCursor.amount;
	 purchaseCursor.close();

         //
         // Perform the credit
         //
         if (processor.credit(terminal, merchant, payevent, slip, batch))
         {
	    SaveCreditEvent(payevent, slip);
         }
         else
         {
            database.rollbackTransaction();
            PrintFmtError("Failed to credit.", processor.getStatusMessage());
         }
      
      database.commitTransaction();
      returnval = true;

   } else
     {	
   	// Otherwise, the purchase id was bad or we couldn't create	
	// the event in the database
	purchaseCursor.close();
	database.rollbackTransaction();

	PrintError("Transaction not found.", "Could not find CAPTURED or CREDITING transaction with ID " + purchaseid + " in database.");
	
     }  // END outer if-else

     return returnval;

} // END DoCredit()

