/*
 * Copyright (c) 1992, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software donated to Berkeley by
 * Jan-Simon Pendry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)cryptfs.h    8.3 (Berkeley) 8/20/94
 *
 * $Id: cryptfs.h,v 1.1.1.1 1998/11/05 21:06:27 ezk Exp $
 */

#include <blowfish.h>

struct cryptfs_args {
  char *target;			/* Target of loopback  */
};

struct cryptfs_mount {
  struct mount *cryptfsm_vfs;
  struct vnode *cryptfsm_rootvp;	/* Reference to root cryptfs_node */
};

/* EZK ADDED */
// #define DO_FILENAMES
// #define FIST_ENCODE

/* for ioctl */
#define FIST_IOCTL_GET_DEBUG_VALUE	_IOR('F', 1,  int)
#define FIST_IOCTL_SET_DEBUG_VALUE	_IOW('F', 2,  int)
#define FIST_IOCTL_GET_NUM_VMA		_IOW('F', 3,  int)
#define FIST_IOCTL_SET_KEY		_IOW('F', 10, char [16])

#ifdef KERNEL
/*
 * A cache of vnode references
 * (This is a private structure per vnode->v_data)
 */
struct cryptfs_node {
  LIST_ENTRY(cryptfs_node) cryptfs_hash; /* Hash list */
  struct vnode *cryptfs_lowervp; /* VREFed once */
  struct vnode *cryptfs_vnode;	/* Back pointer */
};

extern int cryptfs_init __P((struct vfsconf * vfsp));
extern int cryptfs_node_create __P((struct mount *mp, struct vnode * target, struct vnode ** vpp));

/*
 * Flags for crypt_{en,de}code_filename
 * DO_DOTS means the special entries . and .. should be encoded (for symlink)
 * SKIP_DOTS means they should be preserved intact
 */
#define DO_DOTS   0
#define SKIP_DOTS 1

#define MAXNAMLEN	255	/* as per POSIX */

#define	MOUNT_TO_CRYPTFS_MOUNT(mp) ((struct cryptfs_mount *)((mp)->mnt_data))
#define	VP_TO_CRYPTFS(vp) ((struct cryptfs_node *)(vp)->v_data)
#define	CRYPTFS_TO_VP(xp) ((xp)->cryptfs_vnode)
#ifdef CRYPTFS_DIAGNOSTIC
extern struct vnode *cryptfs_checkvp __P((struct vnode * vp, char *fil, int lno));
# define CRYPTFS_VP_TO_LOWERVP(vp) cryptfs_checkvp((vp), __FILE__, __LINE__)
#else
# define CRYPTFS_VP_TO_LOWERVP(vp) (VP_TO_CRYPTFS(vp)->cryptfs_lowervp)
#endif /* CRYPTFS_DIAGNOSTIC */

/* simpler macros for using with file name encryption */
#ifdef FIST_ENCODE
#define CNP_VARS	cn_t *thiscnp=NULL, *lowercnp=NULL
#define CNP_BEFORE(V) \
	do { \
	  thiscnp = ap->a_cnp; /* save original */ \
	  lowercnp = cryptfs_new_cnp((V), thiscnp); \
	  ap->a_cnp = lowercnp;		/* prepare for bypass */ \
	} while (0)
#define CNP_AFTER(V) \
	do { \
	  if ((thiscnp->cn_flags & PARAMASK) != (lowercnp->cn_flags & PARAMASK)) \
	    panic("%s: FLAGS CHANGED this:0x%x lower:0x%x", \
		  __FUNCTION__, thiscnp->cn_flags, lowercnp->cn_flags); \
	  cryptfs_update_cnp((V), &lowercnp, thiscnp, error); \
	  ap->a_cnp = thiscnp;	/* update for caller */ \
	} while (0)
#else /* not FIST_ENCODE */
#define CNP_VARS
#define CNP_BEFORE(x)
#define CNP_AFTER(x)
#endif /* not FIST_ENCODE */

extern int cryptfs_bypass __P((struct vop_generic_args * ap));
extern vop_t **cryptfs_vnodeop_p;

/* XXX: EZK hacky stuff I have to do rather than change system headers */
typedef struct vnode vnode_t;
typedef struct vattr vattr_t;
typedef struct uio uio_t;
typedef struct iovec iovec_t;
typedef struct ucred cred_t;
typedef struct mount vfs_t;
typedef struct componentname cn_t;

#define	MIN(a,b)		(((a)<(b))?(a):(b))
#define	MAX(a,b)		(((a)>(b))?(a):(b))
#define VT_CRYPTFS		VT_NULL
#define MOUNT_CRYPTFS		MOUNT_NULL

#define kmem_alloc(size)	malloc((size), M_TEMP, M_WAITOK)
#define kmem_free(addr)		free((addr), M_TEMP)

#ifdef DO_FILENAMES
extern void fist_crypt_encodefilename(vfs_t *vfsp, char *name, char **uuencoded_name, int *uuencoded_length, int skip_dots, int use_namei_zone, cred_t *cr);
extern int fist_crypt_decodefilename(vfs_t *vfsp, char *name, int length, char **decrypted_name, int *decrypted_length, int skip_dots, cred_t *cr);
#endif /* DO_FILENAMES */

extern BF_KEY *fist_get_userpass(const vfs_t * this_vfsp, cred_t * cr);
extern int crypt_encode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp, cred_t *cr);
extern int crypt_decode_block(char *func, int line, caddr_t base, int len, const vnode_t *vp, cred_t *cr);
extern void *kmem_zalloc(unsigned long size);
extern int fist_uiomove(caddr_t cp, int n, enum uio_rw rwflag, struct uio *uio);
extern void crypt_verify_lower_object(vnode_t *vp, char *fxn);
extern BF_KEY fixed_key;

extern cn_t *cryptfs_new_cnp(const vnode_t *thisvp, const cn_t *cnp);
extern void cryptfs_update_cnp(const vnode_t *thisvp, cn_t **lowercnpp, cn_t *thiscnp, int error);
extern char *fist_cn_flags(u_long flags);

#ifdef FIST_DEBUG
extern void fist_dprint_internal(int level, char *str, ...);
extern void fist_print_vnode(char *, vnode_t *);
extern void fist_print_vfs(char *, vfs_t *);
/* extern void fist_print_uap(char *, struct mounta *); */
extern void fist_print_uio(const char *, uio_t *);
extern void fist_print_uios(const char *, uio_t *);
extern void fist_print_page(const char *fxn_name, vm_page_t pp);
extern void fist_print_cn(const char *fxn_name, cn_t *);
/* extern void fist_print_pathname(char *fxn_name, struct pathname *pnp); */
extern int fist_get_debug_value(void);
extern int fist_set_debug_value(int val);
extern int fist_isdeadcode(void *addr);

# define print_location() \
	fist_dprint(2, "%s %s:%d\n", __FUNCTION__, __FILE__, __LINE__)

# define fist_dprint(level, str, args...) \
	fist_dprint_internal(level, str, ## args)
# define ASSERT(e)	((e) ? (void)0 : panic("ASSERTION \"%s\" FAILED at %s:%d", #e, __FILE__, __LINE__))

#else /* not FIST_DEBUG */

# define fist_dprint(level, format, args...)
# define print_location()
# define fist_print_vnode(a, b)
# define fist_print_vfs(a, b)
# define fist_print_uap(a, b)
# define fist_print_page(a, b)
# define fist_print_cn(a, b)
# define ASSERT(e)

#endif /* not FIST_DEBUG */

#endif /* KERNEL */
