/*
 *------------------------------------------------------------------
 *
 * $Source: /afs/net.mit.edu/project/krb4/src/admin/RCS/make_srvtab.c,v $
 * $Revision: 1.2 $
 * $Date: 92/10/23 15:37:39 $
 * $State: Exp $
 * $Author: tytso $
 * $Locker:  $
 *
 * Copyright 1989 by the Massachusetts Institute of Technology. 
 *
 * For copying and distribution information, please see the file
 * <mit-copyright.h>. 

 * $Log:	make_srvtab.c,v $
 * Revision 1.2  92/10/23  15:37:39  tytso
 * Updated calls to use new-style function names.
 * 
 * Added return after printing error message.
 * 
 * Revision 1.1  92/07/21  01:08:39  tytso
 * Initial revision
 * 
 * Revision 1.3  89/10/10  15:12:21  jon
 * verify master key (oops), use master key version when setting principal
 * info, and clean up usage messages
 * 
 * Revision 1.2  89/10/09  23:19:30  jon
 * long comment and several fixes.  make -s bash default services
 * 
 * Revision 1.1  89/10/09  22:24:50  jon
 * Initial revision
 * 
 *------------------------------------------------------------------
 */

/*  This program makes it easy to add new service keys and extract srvtabs
    It does the most useful thing (for the Athena Kerberos adminstrators) by
    default and allows some customization.

    Basically you indicate a list of services (defaults to rcmd and rvdsrv)
    you're interested and at least one host.  make_srvtab will add the
    appropriate principals to the database. You can either bash existing
    prinicipals (-bash_old_keys, the default) or preserve old keys
    (-no_bash_old_keys) or be queried for each case (-query, default is
    -no_query).

    The make_srvtab will generate srvtabs for each host specified.  This will
    not be done using wildcards as ext_srvtab does, but by explicitly 
    retrieving each service on each host.  It is much quicker (if you have 
    a database of any size).
    
    -- Jon Rochlis, MIT Network Services, Octoboer 1989
*/

#ifndef lint
static char *rcsid_gen_srvtab_c = "$Header: /afs/net.mit.edu/project/krb4/src/admin/RCS/make_srvtab.c,v 1.2 92/10/23 15:37:39 tytso Exp $";
#endif	lint

#include <krb.h>
#include <krb_db.h>

#include <stdio.h>
#include <sys/file.h>

static char *WHOAMI = "make_srvtab";

#define TRUE 1
#define FALSE 0

/* for add_principal, from kdb_edit */
enum ap_op {
    NULL_KEY,			/* setup null keys */
    MASTER_KEY,                 /* use master key as new key */
    RANDOM_KEY,			/* choose a random key */
};

int master_key_version;
C_Block master_key;
Key_schedule master_key_schedule;

char realm[REALM_SZ];

main (argc, argv)
     int argc;
     char **argv;
{
  int reading_service, ask_for_master_key, n_services, n_hosts;
  int bash_old_keys, query, h, s, saw_service_arg;

#define MAX_SERVICES 16
#define MAX_HOSTS 128

  char *service_names[MAX_SERVICES], *host_names[MAX_HOSTS];

  bzero(realm, sizeof(realm));
  ask_for_master_key = TRUE;
  reading_service = FALSE;
  bash_old_keys = TRUE;
  saw_service_arg = FALSE;
  query = FALSE;
 
  n_services = 2;
  n_hosts = 0;

  service_names[0] = "rcmd";
  service_names[1] = "rvdsrv";  

  while (--argc) {
    argv++;
    if (strcmp (*argv, "-s") == 0) {
      /* this is the only way not to include rcmd and rvdsrv or to
	 add other services */
      if (!saw_service_arg) {
	saw_service_arg = TRUE;
	n_services = 0;
      }
      reading_service = TRUE;
      continue;
    }
    if (strcmp (*argv, "-h") == 0) {
      reading_service = FALSE;
      continue;
    }
    if (strcmp (*argv, "-r") == 0) {
      reading_service = FALSE;
      argc--; argv++;
      if (!argc) {
	fprintf (stderr, "%s: -r given but no realm.\n", WHOAMI);
	exit (2);
      }
      strcpy (realm, *argv);
      continue;
    }
    if (strcmp (*argv, "-n") == 0) {
      reading_service = FALSE;
      ask_for_master_key = FALSE;
      continue;
    }
    if (strcmp (*argv, "-bash_old_keys") == 0) {
      reading_service = FALSE;
      bash_old_keys = TRUE;
      continue;
    }
    if (strcmp (*argv, "-no_bash_old_keys") == 0) {
      reading_service = FALSE;
      bash_old_keys = FALSE;
      continue;
    }
    if (strcmp (*argv, "-query") == 0) {
      reading_service = FALSE;
      query = TRUE;
      continue;
    }
    if (strcmp (*argv, "-no_query") == 0) {
      reading_service = FALSE;
      query = FALSE;
      continue;
    }
    if (reading_service) {
      if (n_services == MAX_SERVICES) {
	fprintf (stderr, "%s, Max number (%i) of services exceeded, need to recompile.\n",
		 WHOAMI, MAX_SERVICES);
	exit(1);
      }
      service_names[n_services++] = *argv;
    }
    /* otherwise default to reading a host name */
    else { 
      if (*argv[0] == '-') {
      usage:
	fprintf (stderr, "Usage: %s [-h] host1 ... hostN [-s service1 ... serviceN] [-n] [-bash_old_keys] [-no_bash_old_keys] [-query] [-no_query] [-r realm]\nDefault is %s -bash_old_keys -no_query -s rcmd rvdsrv -h\n", WHOAMI, WHOAMI);
	exit (3);
      }
      if (n_hosts == MAX_HOSTS) {
	fprintf (stderr, "%s, Max number (%i) of services exceeded, need to recompile.\n",
		 WHOAMI, MAX_HOSTS);
	exit(1);
      }
      host_names[n_hosts++] = *argv;
    }
  }

  if ((n_services <= 0) || (n_hosts <= 0))
    goto usage;
  
  if (!realm[0])
    if (krb_get_lrealm(realm, 1) != KSUCCESS) {
      fprintf(stderr, "%s: couldn't get local realm\n", WHOAMI);
      exit(1);
    }

  kerb_init();

  if (kdb_get_master_key (ask_for_master_key, 
			  master_key, master_key_schedule) != 0) {
    fprintf (stderr, "Couldn't read master key.\n");
    exit (-1);
  }

  if ((master_key_version = kdb_verify_master_key(master_key,
						  master_key_schedule,
						  stdout)) < 0)
    exit (-1);
  
  des_init_random_number_generator(master_key);

  for (h = 0; h < n_hosts; h++) {
    for (s = 0; s < n_services; s++)
      add_principal (service_names[s], host_names[h],
		     RANDOM_KEY, bash_old_keys, query);
    make_srvtab (service_names, n_services, host_names[h]);
   }

  /* play it safe */
  bzero (master_key, sizeof (C_Block));
  bzero (master_key_schedule, sizeof (Key_schedule));

  exit(0);
}

/* the following stolen (and slightly modified) from admin/kdb_init.c ...
   perhaps it should be in a library */

/* use a return code to indicate success or failure.  check the return */
/* values of the routines called by this routine. */

add_principal(name, instance, aap_op, bash_old_keys, query)
     char   *name, *instance;
     enum ap_op aap_op;
     int bash_old_keys;
     int query;
{
    Principal principal;
    char    temp[8];
    char    datestring[50];
    char    pw_str[255];
    void    read_pw_string();
    void    string_to_key();
    void    des_new_random_key();
    struct tm *tm, *localtime();
    C_Block new_key;
    int n, more;

    bzero(&principal, sizeof(principal));
    strncpy(principal.name, name, ANAME_SZ);
    strncpy(principal.instance, instance, INST_SZ);

    n = 0;
    n = kerb_get_principal (name, instance, &principal, 1, &more);

    if (n) {
      if (query) {
	/* ask what we should do */
	fprintf (stdout, "%s.%s already exists.  Change key (y/n) ? ", name, instance);
	if (!gets (temp)) exit(0);
	if ((temp[0] != 'y') && (temp[0] != 'Y')) return;
      }
      else
	/* don't ask */
	if (!bash_old_keys) return;
    }

    /* kerb_get_principal will bash this if name.instance didn't exist */
    strncpy(principal.name, name, ANAME_SZ);
    strncpy(principal.instance, instance, INST_SZ);

    switch (aap_op) {
    case NULL_KEY:
	principal.key_low = 0;
	principal.key_high = 0;
	break;
    case RANDOM_KEY:
#ifdef NOENCRYPTION
	bzero(new_key, sizeof(C_Block));
	new_key[0] = 127;
#else
	des_new_random_key(new_key);
#endif
	kdb_encrypt_key (new_key, new_key, master_key, master_key_schedule,
			 ENCRYPT);
	bcopy(new_key, &principal.key_low, 4);
	bcopy(((long *) new_key) + 1, &principal.key_high, 4);
	break;
    case MASTER_KEY:
	bcopy (master_key, new_key, sizeof (C_Block));
	kdb_encrypt_key (new_key, new_key, master_key, master_key_schedule,
			 ENCRYPT);
	bcopy(new_key, &principal.key_low, 4);
	bcopy(((long *) new_key) + 1, &principal.key_high, 4);
	break;
    }

    if (n == 0) {
      /* for new principals */
      principal.exp_date = 946702799;	/* Happy new century */
      strncpy(principal.exp_date_txt, "12/31/99", DATE_SZ);

      principal.attributes = 0;
      principal.max_life = 255;

      principal.key_version = 1;
    }
    else {
      /* for existing principals */
      principal.key_version++;
    }

    principal.kdc_key_ver = master_key_version;

    tm = localtime(&principal.mod_date);
    principal.mod_date = time(0);
    principal.old = 0;
    strncpy(principal.mod_name, "make_srvtab", ANAME_SZ);
    strncpy(principal.mod_instance, "", INST_SZ);

    kerb_db_put_principal(&principal, 1);
    
    /* let's play it safe */
    bzero (new_key, sizeof (C_Block));
    bzero (&principal.key_low, 4);
    bzero (&principal.key_high, 4);

    fprintf (stderr, "%s %s.%s\n", (n == 0) ? "Added" : "Modified", 
	     name, instance);

    return;
}

make_srvtab (services, n_services, host)
     char *services[];
     int n_services;
     char *host;
{
  char fname[1024];
  int more, n;
  Principal p;
  FILE *fout;
  C_Block s_key;

  (void) umask(077);
  
  sprintf(fname, "%s-new-srvtab", host);
  if ((fout = fopen(fname, "w")) == NULL) {
    fprintf(stderr, "Couldn't create file '%s'.\n", fname);
    return;
  }
  printf("\tMaking srvtab for '%s'....\n", fname);
  while (n_services--) {
    n = kerb_get_principal(services[n_services], host, &p, 1, &more);
    if (more)
	    fprintf(stderr, "\tMore than one %s.%s found...\n", services[n_services], host);
    fwrite(p.name, strlen(p.name) + 1, 1, fout);
    fwrite(p.instance, strlen(p.instance) + 1, 1, fout);
    fwrite(realm, strlen(realm) + 1, 1, fout);
    fwrite(&p.key_version,
	   sizeof(p.key_version), 1, fout);
    bcopy(&p.key_low, s_key, sizeof(long));
    bcopy(&p.key_high, s_key + sizeof(long), sizeof(long));
    kdb_encrypt_key (s_key, s_key, 
		     master_key, master_key_schedule, DES_DECRYPT);
    fwrite(s_key, sizeof s_key, 1, fout);
  }
  /* don't leave anything around (even on the stack) */
  bzero (s_key, sizeof (s_key));
  bzero (&p, sizeof (p));
  fclose(fout);
}
