/* movemail foo bar -- move file foo to file bar,
   locking file foo the way /bin/mail respects.
   Copyright (C) 1986 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Important notice: defining MAIL_USE_FLOCK *will cause loss of mail*
   if you do it on a system that does not normally use flock as its way of
   interlocking access to inbox files.  The setting of MAIL_USE_FLOCK
   *must agree* with the system's own conventions.
   It is not a choice that is up to you.

   So, if your system uses lock files rather than flock, then the only way
   you can get proper operation is to enable movemail to write lockfiles there.
   This means you must either give that directory access modes
   that permit everyone to write lockfiles in it, or you must make movemail
   a setuid or setgid program.  */

/*
 * Modified January, 1986 by Michael R. Gretzinger (Project Athena)
 *
 * Added POP (Post Office Protocol) service.  When compiled -DPOP
 * movemail will accept input filename arguments of the form
 * "po:username".  This will cause movemail to open a connection to
 * a pop server running on $MAILHOST (environment variable).  Movemail
 * must be setuid to root in order to work with POP.
 * 
 * New module: popmail.c
 * Modified routines:
 *	main - added code within #ifdef MAIL_USE_POP; added setuid (getuid ())
 *		after POP code. 
 * New routines in movemail.c:
 *	get_errmsg - return pointer to system error message
 *
 * Modified November, 1990 by Jonathan I. Kamens (Project Athena)
 *
 * Added KPOP (Kerberized POP) service to POP code.  If KERBEROS is
 * defined, then:
 *
 * 1. The "kpop" service is used instead of the "pop" service.
 * 2. Kerberos authorization data is sent to the server upon start-up.
 * 3. Instead of sending USER and RPOP, USER and PASS are sent, both
 *    containing the username of the user retrieving mail.
 *
 * Added HESIOD support.  If HESIOD is defined, then an attempt will
 * be made to look up the user's mailhost in the hesiod nameserver
 * database if the MAILHOST environment variable is not set.
 * 
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <string.h>
#define NO_SHORTNAMES   /* Tell config not to load remap.h */
/* #include "../src/config.h" */

#ifdef KERBEROS
#ifdef KRB5
/* these need to be here to declare the functions which are used by
   non-kerberos specific code */
#include <krb5/krb5.h>
#include <krb5/ext-proto.h>
#endif
#endif

#ifdef USG
#include <fcntl.h>
#include <unistd.h>
#ifndef F_OK
#define F_OK 0
#define X_OK 1
#define W_OK 2
#define R_OK 4
#endif
#endif /* USG */

#ifdef XENIX
#include <sys/locking.h>
#endif

#ifdef MAIL_USE_MMDF
extern int lk_open (), lk_close ();
#endif

/* Cancel substitutions made by config.h for Emacs.  */
#undef open
#undef read
#undef write
#undef close

#ifdef __STDC__
#define use_proto
#define ARGS(args) args
#else
#define ARGS(args) ()
#endif

/*
 * Forward declarations.
 */
char *concat ARGS ((char *, char *, char *));
int getline ARGS ((char *, int, FILE *));
int putline ARGS ((char *, char *, FILE *));
int multiline ARGS ((char *, int, FILE *));
void error ARGS ((char *, char *, char *));
void fatal ARGS ((char *, char *));
void pfatal_with_name ARGS ((char *));
void pfatal_and_delete ARGS ((char *));
int popmail ARGS ((char *, char *));
int pop_init ARGS ((char *));
int pop_command ();		/* unchecked */
int pop_stat ARGS ((int *, int *));
int pop_retr ARGS ((int, void (*)(char *, FILE *), FILE *));
void mbx_delimit_begin ARGS ((FILE *));
void mbx_write ARGS ((char *, FILE *));
void mbx_delimit_end ARGS ((FILE *));

int fsync ARGS ((int));
int fchown ();			/* unchecked */
char *mktemp ARGS ((char *));

/* Nonzero means this is name of a lock file to delete on fatal error.  */
char *delete_lockname;

/* Like malloc but get fatal error if memory is exhausted.  */

char *
#ifdef use_proto
xmalloc (int size)
#else
xmalloc (size)
     int size;
#endif
{
  char *result = (char*) malloc (size);
  if (!result)
    fatal ("virtual memory exhausted", 0);
  return result;
}

int
#ifdef use_proto
main (int argc, char **argv)
#else
main (argc, argv)
     int argc;
     char **argv;
#endif
{
  char *inname, *outname;
  int indesc, outdesc;
  char buf[1024];
  int nread;

#ifndef MAIL_USE_FLOCK
  struct stat st;
  long now;
  int tem;
  char *lockname, *p;
  char *tempname;
  int desc;
#endif /* not MAIL_USE_FLOCK */

  delete_lockname = 0;

  if (argc < 3)
    fatal ("two arguments required", "");

  inname = argv[1];
  outname = argv[2];

#ifdef MAIL_USE_MMDF
  mmdf_init (argv[0]);
#endif

  /* Check access to output file.  */
  if (access (outname, F_OK) == 0 && access (outname, W_OK) != 0)
    pfatal_with_name (outname);

  /* Also check that outname's directory is writeable to the real uid.  */
  {
    char *buf = (char *) xmalloc (strlen (outname) + 1);
    char *p;
    strcpy (buf, outname);
    p = buf + strlen (buf);
    while (p > buf && p[-1] != '/')
      *--p = 0;
    if (p == buf)
      *p++ = '.';
    if (access (buf, W_OK) != 0)
      pfatal_with_name (buf);
    free (buf);
  }

#ifdef MAIL_USE_POP
  if (strncmp (inname, "po:", 3) == 0)
    {
      int status; char *user;

      user = (char *) strrchr (inname, ':') + 1;
      status = popmail (user, outname);
      exit (status);
    }

  setuid (getuid ());
#endif /* MAIL_USE_POP */

  /* Check access to input file.  */
  if (access (inname, R_OK | W_OK) != 0)
    pfatal_with_name (inname);

#ifndef MAIL_USE_MMDF
#ifndef MAIL_USE_FLOCK
  /* Use a lock file named /usr/spool/mail/$USER.lock:
     If it exists, the mail file is locked.  */
  /* Note: this locking mechanism is *required* by the mailer
     (on systems which use it) to prevent loss of mail.

     On systems that use a lock file, extracting the mail without locking
     WILL occasionally cause loss of mail due to timing errors!

     So, if creation of the lock file fails
     due to access permission on /usr/spool/mail,
     you simply MUST change the permission
     and/or make movemail a setgid program
     so it can create lock files properly.

     You might also wish to verify that your system is one
     which uses lock files for this purpose.  Some systems use other methods.

     If your system uses the `flock' system call for mail locking,
     define MAIL_USE_FLOCK in config.h or the s-*.h file
     and recompile movemail.  If the s- file for your system
     should define MAIL_USE_FLOCK but does not, send a bug report
     to bug-gnu-emacs@prep.ai.mit.edu so we can fix it.  */

  lockname = concat (inname, ".lock", "");
  tempname = (char *) xmalloc (strlen (inname) + 20);
  strcpy (tempname, inname);
  p = tempname + strlen (tempname);
  while (p != tempname && p[-1] != '/')
    p--;
  *p = 0;
  strcpy (p, "EXXXXXX");
  mktemp (tempname);
  unlink (tempname);

  while (1)
    {
      /* Create the lock file, but not under the lock file name.  */
      /* Give up if cannot do that.  */
      desc = open (tempname, O_WRONLY | O_CREAT, 0666);
      if (desc < 0)
        pfatal_with_name ("lock file--see source file etc/movemail.c");
      close (desc);

      tem = link (tempname, lockname);
      unlink (tempname);
      if (tem >= 0)
	break;
      sleep (1);

      /* If lock file is a minute old, unlock it.  */
      if (stat (lockname, &st) >= 0)
	{
	  now = time (0);
	  if (st.st_ctime < now - 60)
	    unlink (lockname);
	}
    }

  delete_lockname = lockname;
#endif /* not MAIL_USE_FLOCK */

#ifdef MAIL_USE_FLOCK
  indesc = open (inname, O_RDWR);
#else /* if not MAIL_USE_FLOCK */
  indesc = open (inname, O_RDONLY);
#endif /* not MAIL_USE_FLOCK */
#else /* MAIL_USE_MMDF */
  indesc = lk_open (inname, O_RDONLY, 0, 0, 10);
#endif /* MAIL_USE_MMDF */

  if (indesc < 0)
    pfatal_with_name (inname);

#if defined (BSD) || defined (XENIX)
  /* In case movemail is setuid to root, make sure the user can
     read the output file.  */
  /* This is desirable for all systems
     but I don't want to assume all have the umask system call */
  umask (umask (0) & 0333);
#endif /* BSD or Xenix */
  outdesc = open (outname, O_WRONLY | O_CREAT | O_EXCL, 0666);
  if (outdesc < 0)
    pfatal_with_name (outname);
#ifdef MAIL_USE_FLOCK
#ifdef XENIX
  if (locking (indesc, LK_RLCK, 0L) < 0) pfatal_with_name (inname);
#else
  if (flock (indesc, LOCK_EX) < 0) pfatal_with_name (inname);
#endif
#endif /* MAIL_USE_FLOCK */

  while (1)
    {
      nread = read (indesc, buf, sizeof buf);
      if (nread != write (outdesc, buf, nread))
	{
	  int saved_errno = errno;
	  unlink (outname);
	  errno = saved_errno;
	  pfatal_with_name (outname);
	}
      if (nread < sizeof buf)
	break;
    }

#ifdef BSD
  if (fsync (outdesc) < 0)
    pfatal_and_delete (outname);
#endif

  /* Check to make sure no errors before we zap the inbox.  */
  if (close (outdesc) != 0)
    pfatal_and_delete (outname);

#ifdef MAIL_USE_FLOCK
#if defined (STRIDE) || defined (XENIX)
  /* Stride, xenix have file locking, but no ftruncate.  This mess will do. */
  close (open (inname, O_CREAT | O_TRUNC | O_RDWR, 0666));
#else
  ftruncate (indesc, 0L);
#endif /* STRIDE or XENIX */
#endif /* MAIL_USE_FLOCK */

#ifdef MAIL_USE_MMDF
  lk_close (indesc, 0, 0, 0);
#else
  close (indesc);
#endif

#ifndef MAIL_USE_FLOCK
  /* Delete the input file; if we can't, at least get rid of its contents.  */
  if (unlink (inname) < 0)
    if (errno != ENOENT)
      creat (inname, 0666);
#ifndef MAIL_USE_MMDF
  unlink (lockname);
#endif /* not MAIL_USE_MMDF */
#endif /* not MAIL_USE_FLOCK */
  exit (0);
}

/* Print error message and exit.  */

void
#ifdef use_proto
fatal (char *s1, char *s2)
#else
fatal (s1, s2)
     char *s1, s2;
#endif
{
  if (delete_lockname)
    unlink (delete_lockname);
  error (s1, s2, "");
  exit (1);
}

/* Print error message.  `s1' is printf control string, `s2' is arg for it. */

void
#ifdef use_proto
error (char *s1, char *s2, char *s3)
#else
error (s1, s2, s3)
     char *s1, *s2, *s3;
#endif
{
  printf ("movemail: ");
  printf (s1, s2, s3);
  printf ("\n");
}

void
#ifdef use_proto
pfatal_with_name (char *name)
#else
pfatal_with_name (name)
     char *name;
#endif
{
  extern int errno, sys_nerr;
  extern char *sys_errlist[];
  char *s;

  if (errno < sys_nerr)
    s = concat ("", sys_errlist[errno], " for %s");
  else
    s = "cannot open %s";
  fatal (s, name);
}

void
#ifdef use_proto
pfatal_and_delete (char *name)
#else
pfatal_and_delete (name)
     char *name;
#endif
{
  extern int errno, sys_nerr;
  extern char *sys_errlist[];
  char *s;

  if (errno < sys_nerr)
    s = concat ("", sys_errlist[errno], " for %s");
  else
    s = "cannot open %s";

  unlink (name);
  fatal (s, name);
}

/* Return a newly-allocated string whose contents concatenate those of s1, s2, s3.  */

char *
#ifdef use_proto
concat (char *s1, char *s2, char *s3)
#else
concat (s1, s2, s3)
     char *s1, *s2, *s3;
#endif
{
  int len1 = strlen (s1), len2 = strlen (s2), len3 = strlen (s3);
  char *result = (char *) xmalloc (len1 + len2 + len3 + 1);

  strcpy (result, s1);
  strcpy (result + len1, s2);
  strcpy (result + len1 + len2, s3);
  *(result + len1 + len2 + len3) = 0;

  return result;
}

/* This is the guts of the interface to the Post Office Protocol.  */

#ifdef MAIL_USE_POP

#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pwd.h>

#ifdef KERBEROS
#define       POP_SERVICE     "pop"
#ifdef KRB4
#ifdef KRB5
 #error can only use one of KRB4 or KRB5
#endif
#include <krb.h>
#include <des.h>
#endif /* KRB4 */
#ifdef KRB5
#include <krb5/los-proto.h>
#include <com_err.h>
#include <ctype.h>
#endif /* KRB5 */
#endif /* KERBEROS */
#ifdef HESIOD
#include <hesiod.h>
#endif

#ifdef USG
#include <fcntl.h>
/* Cancel substitutions made by config.h for Emacs.  */
#undef open
#undef read
#undef write
#undef close
#endif /* USG */

#define NOTOK (-1)
#define OK 0
#define DONE 1

char *progname;
FILE *sfi;
FILE *sfo;
char Errmsg[80];

static int debug = 0;

char *get_errmsg ();
char *getenv ();

int
#ifdef use_proto
popmail (char *user, char *outfile)
#else
popmail (user, outfile)
     char *user, *outfile;
#endif
{
  char *host;
  int nmsgs, nbytes;
  char response[128];
  register int i;
  int mbfi;
  FILE *mbf;
  struct passwd *pw = (struct passwd *) getpwuid (getuid ());
  if (pw == NULL)
    fatal ("cannot determine user name", "");

  host = getenv ("MAILHOST");
#ifdef HESIOD
  if (host == NULL)
    {
      p = hes_getmailhost(user);
      if (p != NULL && strcmp(p->po_type, "POP") == 0)
	host = p->po_host;
      else
	fatal("no POP server listed in Hesiod");
    }
#endif /* HESIOD */

  if (host == NULL)
    {
      fatal ("no MAILHOST defined", "");
    }

  if (pop_init (host) == NOTOK)
    {
      fatal (Errmsg, "");
    }

  if (getline (response, sizeof response, sfi) != OK)
    {
      fatal (response, "");
    }

#ifdef KERBEROS
  if (pop_command ("USER %s", user) == NOTOK
      || pop_command ("PASS %s", pw->pw_name) == NOTOK)
#else
  if (pop_command ("USER %s", user) == NOTOK
      || pop_command ("RPOP %s", pw->pw_name) == NOTOK)
#endif
    {
      pop_command ("QUIT");
      fatal (Errmsg, "");
    }

  if (pop_stat (&nmsgs, &nbytes) == NOTOK)
    {
      pop_command ("QUIT");
      fatal (Errmsg, "");
    }

  if (!nmsgs)
    {
      pop_command ("QUIT");
      return 0;
    }

  mbfi = open (outfile, O_WRONLY | O_CREAT | O_EXCL, 0666);
  if (mbfi < 0)
    {
      pop_command ("QUIT");
      pfatal_and_delete (outfile);
    }
  fchown (mbfi, getuid (), -1);

  if ((mbf = fdopen (mbfi, "w")) == NULL)
    {
      pop_command ("QUIT");
      pfatal_and_delete (outfile);
    }

  for (i = 1; i <= nmsgs; i++)
    {
      mbx_delimit_begin (mbf);
      if (pop_retr (i, mbx_write, mbf) != OK)
	{
	  pop_command ("QUIT");
	  close (mbfi);
	  unlink (outfile);
	  fatal (Errmsg, "");
	}
      mbx_delimit_end (mbf);
      fflush (mbf);
    }

  if (fsync (mbfi) < 0)
    {
      pop_command ("QUIT");
      pfatal_and_delete (outfile);
    }

  if (close (mbfi) == -1)
    {
      pop_command ("QUIT");
      pfatal_and_delete (outfile);
    }

  for (i = 1; i <= nmsgs; i++)
    {
      if (pop_command ("DELE %d", i) == NOTOK)
	{
	  /* Better to ignore this failure.  */
	}
    }

  pop_command ("QUIT");
  return (0);
}

int
#ifdef use_proto
pop_init (char *host)
#else
pop_init (host)
     char *host;
#endif
{
  register struct hostent *hp;
  register struct servent *sp;
  int lport = IPPORT_RESERVED - 1;
  struct sockaddr_in sin;
  register int s;
#ifdef KERBEROS
#ifdef KRB4
  KTEXT ticket;
  MSG_DAT msg_data;
  CREDENTIALS cred;
  Key_schedule schedule;
  int rem;
#endif /* KRB4 */
#ifdef KRB5
  krb5_error_code retval;
  krb5_ccache ccdef;
  krb5_principal client, server;
  krb5_error *err_ret;
  register char *cp;
#endif /* KRB5 */
#endif /* KERBEROS */

  hp = gethostbyname (host);
  if (hp == NULL)
    {
      sprintf (Errmsg, "MAILHOST unknown: %s", host);
      return NOTOK;
    }

#ifdef KERBEROS
  sp = getservbyname ("kpop", "tcp");
#else
  sp = getservbyname ("pop", "tcp");
#endif
  if (sp == 0)
    {
      strcpy (Errmsg, "tcp/(k)pop: unknown service");
      return NOTOK;
    }

  sin.sin_family = hp->h_addrtype;
  bcopy (hp->h_addr, (char *)&sin.sin_addr, hp->h_length);
  sin.sin_port = sp->s_port;
#ifdef KERBEROS
  s = socket (AF_INET, SOCK_STREAM, 0);
#else
  s = rresvport (&lport);
#endif
  if (s < 0)
    {
      sprintf (Errmsg, "error creating socket: %s", get_errmsg ());
      return NOTOK;
    }

  if (connect (s, (char *)&sin, sizeof sin) < 0)
    {
      sprintf (Errmsg, "error during connect: %s", get_errmsg ());
      close (s);
      return NOTOK;
    }

#ifdef KERBEROS
#ifdef KRB4
  ticket = (KTEXT) xmalloc(sizeof(KTEXT_ST));
  rem = krb_sendauth(0L, s, ticket, POP_SERVICE, hp->h_name,
                     (char *) krb_realmofhost(hp->h_name),
                     (unsigned long)0, &msg_data, &cred, schedule,
                     (struct sockaddr_in *)0,
                     (struct sockaddr_in *)0,
                     "KPOPV0.1");
  if (rem != KSUCCESS) {
    sprintf(Errmsg, "kerberos error: %s", krb_err_txt[rem]);
    close(s);
    return(NOTOK);
  }
#endif /* KRB4 */
#ifdef KRB5
  krb5_init_ets();
  
  if (retval = krb5_cc_default(&ccdef)) {
  krb5error:
    sprintf(Errmsg, "krb5 error: %s", error_message(retval));
    close(s);
    return(NOTOK);
  }
  if (retval = krb5_cc_get_principal(ccdef, &client)) {
    goto krb5error;
  }
  
  /* lower-case to get name for "instance" part of service name */
  for (cp = hp->h_name; *cp; cp++)
    if (isupper(*cp))
      *cp = tolower(*cp);
  
  if (retval = krb5_sname_to_principal(hp->h_name, POP_SERVICE,
                                       FALSE, /* FALSE means don't
                                                 canonicalize hostname
                                                 (we already have...) */
                                       &server)) {
    goto krb5error;
  }
  
  retval = krb5_sendauth((krb5_pointer) &s, "KPOPV1.0", client, server,
                         AP_OPTS_MUTUAL_REQUIRED,
                         0,           /* no checksum */
                         0,           /* no creds, use ccache instead */
                         ccdef,
                         0,           /* don't need seq # */
                         0,           /* don't need a subsession key */
                         &err_ret,
                         0);          /* don't need reply */
  krb5_free_principal(server);
  if (retval) {
    if (err_ret && err_ret->text.length) {
      sprintf(Errmsg, "krb5 error: %s [server says '%*s'] ",
	      error_message(retval),
	      err_ret->text.length,
	      err_ret->text.data);
      krb5_free_error(err_ret);
    } else
      sprintf(Errmsg, "krb5 error: %s", error_message(retval));
    close(s);
    return(NOTOK);
  }
#endif /* KRB5 */
#endif /* KERBEROS */

  sfi = fdopen (s, "r");
  sfo = fdopen (s, "w");
  if (sfi == NULL || sfo == NULL)
    {
      sprintf (Errmsg, "error in fdopen: %s", get_errmsg ());
      close (s);
      return NOTOK;
    }

  return OK;
}

int
pop_command (fmt, a, b, c, d)
     char *fmt;
     int a, b, c, d;
{
  char buf[128];

  sprintf (buf, fmt, a, b, c, d);

  if (debug) fprintf (stderr, "---> %s\n", buf);
  if (putline (buf, Errmsg, sfo) == NOTOK) return NOTOK;

  if (getline (buf, sizeof buf, sfi) != OK)
    {
      strcpy (Errmsg, buf);
      return NOTOK;
    }

  if (debug)
    fprintf (stderr, "<--- %s\n", buf);
  if (*buf != '+')
    {
      strcpy (Errmsg, buf);
      return NOTOK;
    }
  else
    {
      return OK;
    }
}

    
int
#ifdef use_proto
pop_stat (int *nmsgs, int *nbytes)
#else
pop_stat (nmsgs, nbytes)
     int *nmsgs, *nbytes;
#endif
{
  char buf[128];

  if (debug)
    fprintf (stderr, "---> STAT\n");
  if (putline ("STAT", Errmsg, sfo) == NOTOK)
    return NOTOK;

  if (getline (buf, sizeof buf, sfi) != OK)
    {
      strcpy (Errmsg, buf);
      return NOTOK;
    }

  if (debug) fprintf (stderr, "<--- %s\n", buf);
  if (*buf != '+')
    {
      strcpy (Errmsg, buf);
      return NOTOK;
    }
  else
    {
      sscanf (buf, "+OK %d %d", nmsgs, nbytes);
      return OK;
    }
}

int
#ifdef use_proto
pop_retr (int msgno, void (*action)(), FILE *arg)
#else
pop_retr (msgno, action, arg)
     int msgno;
     int (*action)();
     FILE *arg;
#endif
{
  char buf[1024];

  sprintf (buf, "RETR %d", msgno);
  if (debug) fprintf (stderr, "%s\n", buf);
  if (putline (buf, Errmsg, sfo) == NOTOK) return NOTOK;

  if (getline (buf, sizeof buf, sfi) != OK)
    {
      strcpy (Errmsg, buf);
      return NOTOK;
    }

  while (1)
    {
      switch (multiline (buf, sizeof buf, sfi))
	{
	case OK:
	  (*action)(buf, arg);
	  break;
	case DONE:
	  return OK;
	case NOTOK:
	  strcpy (Errmsg, buf);
	  return NOTOK;
	}
    }
}

int
#ifdef use_proto
getline (char *buf, int n, FILE *f)
#else
getline (buf, n, f)
     char *buf;
     int n;
     FILE *f;
#endif
{
  register char *p;
  int c;

  p = buf;
  while (--n > 0 && (c = fgetc (f)) != EOF)
    if ((*p++ = c) == '\n') break;

  if (ferror (f))
    {
      strcpy (buf, "error on connection");
      return NOTOK;
    }

  if (c == EOF && p == buf)
    {
      strcpy (buf, "connection closed by foreign host");
      return DONE;
    }

  *p = NULL;
  if (*--p == '\n') *p = NULL;
  if (*--p == '\r') *p = NULL;
  return OK;
}

int
#ifdef use_proto
multiline (char *buf, int n, FILE *f)
#else
multiline (buf, n, f)
     char *buf;
     int n;
     FILE *f;
#endif
{
  if (getline (buf, n, f) != OK)
    return NOTOK;
  if (*buf == '.')
    {
      if (*(buf+1) == NULL)
	return DONE;
      else
	strcpy (buf, buf+1);
    }
  return OK;
}

char *
get_errmsg ()
{
  extern int errno, sys_nerr;
  extern char *sys_errlist[];
  char *s;

  if (errno < sys_nerr)
    s = sys_errlist[errno];
  else
    s = "unknown error";
  return (s);
}

int
putline (buf, err, f)
     char *buf;
     char *err;
     FILE *f;
{
  fprintf (f, "%s\r\n", buf);
  fflush (f);
  if (ferror (f))
    {
      strcpy (err, "lost connection");
      return NOTOK;
    }
  return OK;
}

void
#ifdef use_proto
mbx_write (char *line, FILE *mbf)
#else
mbx_write (line, mbf)
     char *line;
     FILE *mbf;
#endif
{
  fputs (line, mbf);
  fputc (0x0a, mbf);
}

void
#ifdef use_proto
mbx_delimit_begin (FILE *mbf)
#else
mbx_delimit_begin (mbf)
     FILE *mbf;
#endif
{
#ifdef BABYL
  fputs ("\f\n0, unseen,,\n", mbf);
#else
  char *truename = "received_at";
  time_t date;
  time(&date);
  fprintf(mbf, "From %s %s", truename, ctime(&date));
#endif
}

void
#ifdef use_proto
mbx_delimit_end (FILE *mbf)
#else
mbx_delimit_end (mbf)
     FILE *mbf;
#endif
{
#ifdef BABYL
  putc ('\037', mbf);
#endif
}

#endif /* MAIL_USE_POP */
