/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_acl.c	1.17 96/10/04 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include <skip_types.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_es.h>
#include <skip_if.h>
#include <skip_crypt.h>
#include <skip_ah.h>


/*
 * Imported symbols...
 */
extern skip_es_t	*skip_es_ifs;
extern MUTEX_T		skip_es_lock;

/*
 * Local functions prototypes
 */
static boolean_t	skip_acl_find(skip_es_t *, struct in_addr *,
				skip_es_hash_t **, skip_es_hash_t ***);
static int 		skip_es_netadd(skip_es_t *, skip_es_hash_t *,
							skip_es_hash_t **);


/* skip_es_find_if()
 *
 * Given an interface name, find the corresponding interface structure
 *
 * Returns: 0 on success, errno otherwise
 */
skip_es_t *
skip_es_find_if(char *if_name)
{
	register skip_es_t	*ifp;

	for (ifp = skip_es_ifs; ifp; ifp = ifp->next) {
		if (SKIP_IFCMP(ifp->if_name, if_name) == 0) {
			break;
		}
	}
	return (ifp);
}

/* skip_get_if_stats()
 *
 *  Get SKIP interface statistics
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_get_if_stats(skip_ctrl_stats_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;
	register ioctl_if_stat_t	*if_statp;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		rc = ENODEV;
		MUTEX_EXIT(&skip_es_lock);
		return (rc);
	}

	if_statp = (ioctl_if_stat_t *) &(reqp->if_stats);

	BCOPY((caddr_t) &ifp->stats,
		(caddr_t) if_statp, sizeof(ioctl_if_stat_t));

	MUTEX_EXIT(&skip_es_lock);
	return (rc);

}

/* skip_es_set_mode()
 *
 * Turn on/off access control
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_set_mode(SKIP_NOTIFYQ_T wq, skip_es_mode_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return(ENODEV);
	}

	switch (reqp->if_mode) {

	case SkipInteractive:
		if (ifp->notifier) {
			MUTEX_EXIT(&skip_es_lock);
			return (EEXIST);
		}
		SKIP_SET_NOTIFIER(ifp->notifier, wq);
		break;

	case SkipAccessControlOn:
	case SkipAccessControlOff:
		ifp->if_mode = reqp->if_mode;
		break;

	default:
		rc = EINVAL;
	}
	MUTEX_EXIT(&skip_es_lock);
	return (rc);
}

/* skip_es_get_mode()
 *
 * Test if access control is on or off
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_get_mode(skip_es_mode_t *reqp)
{
	register skip_es_t		*ifp;
	register int			rc = 0;

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		rc = ENODEV;
	} else { 
		reqp->if_mode = ifp->if_mode;
	}

	MUTEX_EXIT(&skip_es_lock);
	return (rc);
}

/* skip_es_params_ok()
 *
 * validate parameters to an ACL "add" operation
 *
 * Returns: 0 if OK, errno otherwise
 */
static int
skip_es_params_ok(skip_param_t *params)
{
	if ((params->ip_addr.s_addr & params->mask.s_addr) !=
						params->ip_addr.s_addr) {
		/*
		 * address and mask are incompatible
		 */
		return (EFAULT);
	}

	if (params->flags & SKIP_NOMADIC) {
		if (!SKIP_IS_HOSTACL(params)) {
			/*
			 * invalid mask for nomadic
			 */
			return (EFAULT);
		}
	}

	if (params->comp_alg) {
		return (EPROTONOSUPPORT);
	}

	if (!IPADDRMATCH(&params->ip_addr, &params->tunnel_addr)) {
		if (params->version == SKIP_NONE) {
			/*
			 * can't have a tunnel address in cleartext
			 */
			return (EADDRNOTAVAIL);
		}
		if (params->ip_mode & SKIP_TRS_ON) {
			/*
			 * can't have a tunnel address in transport mode
			 */
			return (EADDRNOTAVAIL);
		}
	}

	/*
	 * XXX Check if we need to use default Sender Master Key ID XXX
	 */
	if (params->s_nsid && (params->s_mkeyid.len == 0) && 
		(params->version == SKIP_V1 || params->version == SKIP_V2)) {

		params->s_mkeyid.len = skip_get_default_keyid(params->s_nsid,
					(unsigned char *) params->s_mkeyid.buf);
	}

	/*
	 * Check sender Master Key ID (Local) validity
	 */
	if (skip_check_keyid(params->s_nsid, &params->s_mkeyid, B_FALSE)) {
		/*
		 * Bad sender Master Key Id
		 */
		return (EINVAL);
	}

	/*
	 * Check receiver Master Key ID (Remote) validity
	 */
	if (skip_check_keyid(params->r_nsid, &params->r_mkeyid, B_FALSE)) {
		/*
		 * Bad receiver Master Key Id
		 */
		return (EINVAL);
	}
	/*
	 * Add the receiver master key default for NSID 0 XXX
	 */
	if ((params->r_nsid == 0) &&
		((params->kp_alg || params->mac_alg) || 
		(params->version == SKIP_V2) ||
		(params->version == SKIP_RAW))) {

		BUFTOKEY(&(params->ip_addr),
			sizeof(params->ip_addr), params->r_mkeyid);
	}

	/*
	 * Protect against bad configuration from the lib.
	 */
	switch (params->version) {
	case SKIP_NONE:
		if (params->kij_alg ||
			params->kp_alg ||
			params->mac_alg ||
			params->comp_alg) {
			/*
			 * cleartext, all algs must be zero
			 */
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_V1:
		if (!params->kij_alg || 
			!params->kp_alg ||
			params->mac_alg || 
			params->comp_alg) {
			/*
			 * V1, kij/kp algs required
			 */
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_V2:
		if (((params->kp_alg || params->mac_alg) && !params->kij_alg) ||
			params->comp_alg) {
			/*
			 * V2, kp/mac algs require kij_alg
			 */
			return (EPROTONOSUPPORT);
		}
		if ((!params->kp_alg && !params->mac_alg) && 
					(params->ip_mode == SKIP_TRS_ON)) {
			/*
			 * Transport mode requires kp/mac algs
			 */
			return (EPROTONOSUPPORT);
		}
		break;

	case SKIP_RAW:
		if ((!params->kp_alg && !params->mac_alg) || 
			params->comp_alg) {
			/*
			 * raw mode, kp and/or mac algs required
			 */
			return (EPROTONOSUPPORT);
		}
		break;

	default:
		return (EPROTONOSUPPORT);
	}
	return (0);
}

/* skip_es_hosthash()
 *
 * Compute a host hash for the address
 *
 * Returns: an index (0 .. SKIP_ES_HASH_TABLESZ-1)
 */
static unsigned int
skip_es_hosthash(struct in_addr *addr)
{
	return (((unsigned char *) addr)[3]);
}

/* skip_es_nethash()
 *
 * Compute a net hash for the address and mask
 *
 * Returns: an index (0 .. SKIP_ES_HASH_TABLESZ-1)
 */
static unsigned int
skip_es_nethash(struct in_addr addr, unsigned long mask)
{
	unsigned char 	*h = (unsigned char *) &addr.s_addr;
	unsigned int	hash;

	addr.s_addr &= mask;

	hash = h[0] ^ h[1] ^ h[2];

	return (hash);
}

/* skip_es_acl_list()
 *
 * Return list of current skip access control list entries
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_acl_list(skip_es_req_t *reqp, int *nbytesp)
{
	register unsigned int		nentries, maxentries, h;
	register skip_es_t		*ifp;
	register skip_param_t		*params;
	register skip_es_hash_t		*hp;

	/*
	 * check to see if attached buffer is valid
	 */
	(*nbytesp) -= SKIPSZ;

	if ((*nbytesp % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	maxentries = *nbytesp / sizeof (skip_param_t);

	if (maxentries == 0) {
		return (0);
	}

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	params = (skip_param_t *) &reqp->params;

	*nbytesp = SKIPSZ;

	for (nentries = 0, h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
		/*
		 * add host access control entries
		 */
		for (hp = ifp->hosthash[h]; hp; hp = hp->next) {
			if (nentries == maxentries) {
				break;
			} else {
				nentries++;
				(*nbytesp) += sizeof(skip_param_t);
			}
			PARAMSCOPY(&hp->params, params);

			/*
			 * the kernel internally initialises r_mkeyid even
			 * for NSID 0.
			 * This is confusing for user-space tools so zap it.
			 */
			if (params->r_nsid == 0) {
				KEYVARZERO(params->r_mkeyid);
			}
			params++;
		}
	}
	for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
		/*
		 * add network access control entries
		 */
		for (hp = ifp->nethash[h]; hp; hp = hp->next) {
			if (nentries == maxentries) {
				break;
			} else {
				nentries++;
				(*nbytesp) += sizeof(skip_param_t);
			}
			PARAMSCOPY(&hp->params, params);
			if (params->r_nsid == 0) {
				params->r_mkeyid.len = 0;
			}
			params++;
		}
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_acl_get()
 *
 * Return an ACL entry by using its params
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_acl_get(skip_es_req_t *reqp, int nbytes)
{
	register unsigned int		nentries;
	register skip_es_t		*ifp;
	register skip_param_t		*params;
	skip_es_hash_t			*entry, **acl;

	/*
	 * check to see if valid
	 */
	nbytes -= SKIPSZ;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nentries = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	params = (skip_param_t *) &reqp->params;

	for (; nentries; nentries--, params++) {

		if (params->flags & SKIP_NOMADIC) {
			/*
			 * look up by key id
			 */
			if (!skip_es_find_keyid(ifp,
						params->version,
						params->r_nsid,
						&params->r_mkeyid,
						&entry,
						&acl)) {
				/*
				 * does not exist...
				 */
				MUTEX_EXIT(&skip_es_lock);
				return (ENOENT);
			}
			/*
			 * the kernel internally initialises r_mkeyid even for
			 * NSID 0. This is confusing for user-space tools so
			 * zap it.
			 */
			PARAMSCOPY(&entry->params, params);
			if (params->r_nsid == 0) {
				KEYVARZERO(params->r_mkeyid);
			}
		} else {
			/*
			 * look up by IP address
			 */
			if (!skip_acl_find(ifp, &params->ip_addr, &entry,
									&acl)) {
				/*
				 * does not exist...
				 */
				MUTEX_EXIT(&skip_es_lock);
				return (ENOENT);
			}
			PARAMSCOPY(&entry->params, params);
			if (params->r_nsid == 0) {
				KEYVARZERO(params->r_mkeyid);
			}
		}
	}
	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_rehash()
 *
 * Reposition entries in the network hash table based on new hash criteria
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_es_rehash(skip_es_t *ifp)
{
	skip_es_hash_t	*hp, *nexthp, **newacl;
	int		i, aclsize;

	aclsize = sizeof(skip_es_hash_t *) * SKIP_ES_HASH_TABLESZ;

	newacl = SYSMEM_ALLOC(aclsize);
	if (newacl == NULL) {
		return (ENOMEM);
	}

	BZERO(newacl, aclsize);

	for (i = 0; i < SKIP_ES_HASH_TABLESZ; i++) {
		for (hp = ifp->nethash[i]; hp; hp = nexthp) {
			nexthp = hp->next;
			hp->next = hp->prev = NULL;
			(void) skip_es_netadd(ifp, hp, newacl);
		}
	}

	BCOPY(newacl, ifp->nethash, aclsize);
	SYSMEM_FREE(newacl, aclsize);
	return (0);
}

/* skip_es_netadd()
 *
 * Add a network to the interface's net hash table
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_es_netadd(skip_es_t *ifp, skip_es_hash_t *net, skip_es_hash_t **acl)
{
	register unsigned int	h;
	register skip_es_hash_t	*hp;
	register int		rc;

	/*
	 * check if new mask is different than the current one
	 */
	if ((net->params.mask.s_addr & ifp->netmask) != ifp->netmask) {
		/*
		 * we only use bits common in all net masks, so we must
		 * adjust the hash table accordingly
		 */
		ifp->netmask = net->params.mask.s_addr & ifp->netmask;

		if ((rc = skip_es_rehash(ifp))) {
			return (rc);
		}
	}

	h = skip_es_nethash(net->params.ip_addr, ifp->netmask);

	hp = acl[h];

	while (hp) {
		/*
		 * check collision entries
		 */
		if (IPADDRMATCH(&net->params.ip_addr, &hp->params.ip_addr) &&
			IPADDRMATCH(&net->params.mask, &hp->params.mask)) {
			return (EEXIST);
		}
		hp = hp->next;
	}

	if (acl[h]) {
		net->next = acl[h];
		acl[h]->prev = net;
	}
	acl[h] = net;
	return(0);
}

/* skip_es_hostadd()
 *
 * Add a host to the interface's host hash table
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_es_hostadd(skip_es_t *ifp, skip_es_hash_t *host)
{
	unsigned int		h;
	skip_es_hash_t		*hp, *entry, **acl;

	/*
	 * check to see if the key already exists - must be unique
	 * for nomadic entries to be accepted.
	 */
	if ((host->params.flags & SKIP_NOMADIC) &&
				skip_es_find_keyid(ifp,
						host->params.version,
						host->params.r_nsid,
						&host->params.r_mkeyid,
						&entry,
						&acl)) {
		/*
		 * an entry with this key id exists
		 */
		return (EEXIST);
	}

	h = skip_es_hosthash(&host->params.ip_addr);
	hp = ifp->hosthash[h];

	if (!SKIP_IS_UNKNOWNIP(host->params.ip_addr)) {
		/*
		 * IP addresses must be unique (except for unknown ones)
		 */
		while (hp) {
			/*
			 * check collision entries
			 */
			if (IPADDRMATCH(&host->params.ip_addr,
							&hp->params.ip_addr)) {
				return (EEXIST);
			}
			hp = hp->next;
		}
	}

	if (ifp->hosthash[h]) {
		host->next = ifp->hosthash[h];
		(ifp->hosthash[h])->prev = host;
	}
	ifp->hosthash[h] = host;
	return (0);
}

/* skip_es_acl_add()
 *
 * Add a new entry to the skip access control list
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_acl_add(skip_es_req_t *reqp, int nbytes)
{
	register int			nentries, rc;
	register skip_es_t		*ifp;
	register skip_param_t		*params;
	skip_es_hash_t			*hp;

	/*
	 * check to see if valid
	 */
	nbytes -= SKIPSZ;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nentries = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	params = (skip_param_t *) &reqp->params;

	for (; nentries; nentries--, params++) {

		/*
		 * sanity check for this new ACL entry
		 */
		rc = skip_es_params_ok(params);
		if (rc) {
			MUTEX_EXIT(&skip_es_lock);
			return (rc);
		}

		hp = (skip_es_hash_t *) SYSMEM_ALLOC(sizeof (*hp));
		if (hp == NULL) {
			MUTEX_EXIT(&skip_es_lock);
			return (ENOMEM);
		}

		hp->next = hp->prev = NULL;

		/*
		 * Fill in ACL entry with parameters
		 */
		PARAMSCOPY(params, &hp->params);

		/*
		 * Now add this entry into the ACL
		 */
		if (SKIP_IS_HOSTACL(params)) {
			rc = skip_es_hostadd(ifp, hp);
		}

		if (SKIP_IS_NETACL(params)) {
			rc = skip_es_netadd(ifp, hp, ifp->nethash);
		}

		if (rc) {
			SYSMEM_FREE(hp, sizeof (*hp));
			MUTEX_EXIT(&skip_es_lock);
			return (rc);
		}
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_es_acl_del()
 *
 * Remove an access control entry
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_acl_del(skip_es_req_t *reqp, int nbytes)
{
	register skip_es_t		*ifp;
	register skip_param_t		*params;
	unsigned int			nentries, h = 0;
	skip_es_hash_t			*hp, **acl;

	/*
	 * check to see if valid
	 */
	nbytes -= SKIPSZ;

	if ((nbytes % sizeof (skip_param_t)) != 0) {
		/*
		 * bad parameter list
		 */
		return (EINVAL);
	}
	nentries = nbytes / sizeof (skip_param_t);

	/*
	 * grab the lock on the interface table
	 */
	MUTEX_ENTER(&skip_es_lock);

	ifp = skip_es_find_if(reqp->if_name);

	if (!ifp) {
		/*
		 * interface not found...
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENODEV);
	}

	params = (skip_param_t *) &reqp->params;

	for (; nentries; nentries--, params++) {

		if (params->flags & SKIP_NOMADIC) {
			/*
			 * search by key id
			 */
			if (!skip_es_find_keyid(ifp,
					params->version,
					params->r_nsid,
					&params->r_mkeyid,
					&hp,
					&acl)) {
				/*
				 * does not exist...
				 */
				MUTEX_EXIT(&skip_es_lock);
				return (ENOENT);
			}
			h = skip_es_hosthash(&hp->params.ip_addr);
		} else {
			/*
			 * search by ip address
			 */
			if (SKIP_IS_HOSTACL(params)) {
				h = skip_es_hosthash(&params->ip_addr);
				acl = ifp->hosthash;
			}
			if (SKIP_IS_NETACL(params)) {
 				h = skip_es_nethash(params->ip_addr,
								ifp->netmask);
				acl = ifp->nethash;
			}

			hp = acl[h];
			while (hp) {
				if (IPADDRMATCH(&params->ip_addr,
					&hp->params.ip_addr) &&
					IPADDRMATCH(&params->mask,
					&hp->params.mask)) {
					break;
				}
				hp = hp->next;
			}
			if (!hp) {
				/*
				 * does not exist...
				 */
				MUTEX_EXIT(&skip_es_lock);
				return (ENOENT);
			}
		}
		/*
		 * remove this element from the hash table
		 */
		if (hp->next != NULL) {
			hp->next->prev = hp->prev;
		}
		if (hp->prev != NULL) {
			hp->prev->next = hp->next;
		} else {
			acl[h] = hp->next;
		}
		SYSMEM_FREE(hp, sizeof (*hp));
	}

	/*
	 * and release the interface table
	 */
	MUTEX_EXIT(&skip_es_lock);
	return (0);
}

/* skip_acl_find()
 *
 * Look up a host in the access control list by IP address.
 *
 * Returns: B_TRUE if the host is in list, B_FALSE otherwise
 */
static boolean_t
skip_acl_find(skip_es_t *ifp,
		struct in_addr *addr,
		skip_es_hash_t **entry,
		skip_es_hash_t ***acl)
{
	register unsigned int		h;
	register skip_es_hash_t		*hp, *best = NULL;
	static struct in_addr		def; /* = 0 = INADDR_ANY */

	/*
	 * try to find a host entry
	 */
	(*acl) = ifp->hosthash;
	h = skip_es_hosthash(addr);
	hp = ifp->hosthash[h];

	while (hp) {
		if (IPADDRMATCH(addr, &hp->params.ip_addr)) {
			/*
			 * found entry we were looking for...
			 */
			goto found;
		}
		hp = hp->next;
	}

	/*
	 * try to find a network entry
	 */
	(*acl) = ifp->nethash;
	h = skip_es_nethash(*addr, ifp->netmask);
	hp = ifp->nethash[h];

	while (hp) {
		if (SKIP_NETMATCH(addr, hp->params)) {
			/*
			 * found a match, but is it the best?
			 */
			if (hp->params.ip_addr.s_addr == addr->s_addr) {
				/*
				 * exact match
				 */
				goto found;
			} 
			if ((best == NULL)  ||
				(ntohl(hp->params.mask.s_addr) >
					ntohl(best->params.mask.s_addr))) {
				/*
				 * a better match
				 */
				best = hp;
			}
		}
		hp = hp->next;
	}
	hp = best;
	if (hp) {
		goto found;
	}

	/*
	 * use the default acl if it exists
	 */

	(*acl) = ifp->hosthash;
	h = skip_es_hosthash(&def);
	hp = ifp->hosthash[h];

	while (hp) {
		if (IPADDRMATCH(&def, &hp->params.ip_addr)) {
			/*
			 * found entry we were looking for...
			 */
			goto found;
		}
		hp = hp->next;
	}

	/*
	 * out of luck...
	 */
	return (B_FALSE);

found:
	(*entry) = hp;
	return (B_TRUE);
}

/* skip_es_find_keyid()
 *
 * Returns a hash entry for a host in the access control list by key id.
 *
 * Returns: B_TRUE if the host is in list, B_FALSE otherwise
 */
boolean_t
skip_es_find_keyid(skip_es_t *ifp,
		unsigned char version,
		unsigned char nsid,
		skip_key_var_t *keyid,
		skip_es_hash_t **entry,
		skip_es_hash_t ***acl)
{
	register unsigned int		i, h;
	register skip_es_hash_t		*hp;

	SKIP_DEBUG2("skip_es_find_keyid  looking for nsid %d version %d ",
							nsid, version);
        SKIP_DEBUG("key ID [");
        for (i = 0; i < 4; i++ ) {
                SKIP_DEBUG1("0x%x ",
                        (unsigned char) keyid->buf[i]);

        }

        SKIP_DEBUG("]\n");

	/*
	 * examine host entries
	 */
	for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
		hp = ifp->hosthash[h];
		(*acl) = ifp->hosthash;
		while (hp) {
			if ((hp->params.version == version) &&
				(hp->params.r_nsid == nsid) &&
				KEYVAREQUAL(hp->params.r_mkeyid, *keyid)) {
				/*
				 * found entry we were looking for...
				 */
				goto found;
			}
			hp = hp->next;
		}
	}

	/*
	 * examine network entries
	 */
	for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
		hp = ifp->nethash[h];
		(*acl) = ifp->nethash;
		while (hp) {
			if ((hp->params.version == version) &&
				(hp->params.r_nsid == nsid) &&
				KEYVAREQUAL(hp->params.r_mkeyid, *keyid)) {
				goto found;
			}
			hp = hp->next;
		}
	}
	/*
	 * out of luck...
	 */
	SKIP_DEBUG("skip_es_find_keyid: not found\n");
	return (B_FALSE);

found:
	SKIP_DEBUG("skip_es_find_keyid: found\n");
	(*entry) = hp;
	return (B_TRUE);
}

/* skip_es_change_addr()
 *
 * Change the IP address of a host specified by its parameters
 *
 * Returns: 0 on success, errno otherwise
 */
int
skip_es_change_addr(skip_es_t *ifp, skip_param_t *params, struct in_addr *addr)
{
	register unsigned int		h;
	int				rc;
	skip_es_hash_t			*hp, **acl;
	struct in_addr			old_addr;

	if (!(params->flags & SKIP_NOMADIC)) {
		return (EINVAL);
	}

	if (!SKIP_IS_HOSTACL(params)) {
		return (EINVAL);
	}

	MUTEX_ENTER(&skip_es_lock);

	if (!skip_es_find_keyid(ifp,
				params->version,
				params->r_nsid,
				&params->r_mkeyid,
				&hp,
				&acl)) {
		/*
		 * couldn't find entry
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (ENOENT);
	}

	if (!(hp->params.flags & SKIP_NOMADIC)) {
		/*
		 * already exists
		 */
		MUTEX_EXIT(&skip_es_lock);
		return (EEXIST);
	}

	SKIP_DEBUG2("skip_es_change_addr: changing address from %x to %x\n",
		(int) hp->params.ip_addr.s_addr, (int) addr->s_addr);

	/*
	 * remove this element from its current position
	 */
	if (hp->next != NULL) {
		hp->next->prev = hp->prev;
	}
	if (hp->prev != NULL) {
		hp->prev->next = hp->next;
	} else {
		h = skip_es_hosthash(&hp->params.ip_addr);
		acl[h] = hp->next;
	}
	hp->next = hp->prev = NULL;

	IPADDRCOPY(&hp->params.ip_addr, &old_addr);

	/*
	 * change IP address
	 */
	IPADDRSET(addr, &hp->params);
	
	/*
	 * and reinsert...
	 */
	rc = skip_es_hostadd(ifp, hp);

	if (rc) {
		/*
		 * add failed, reestablish the old ACL entry...
		 */
		IPADDRSET(&old_addr, &hp->params);
		h = skip_es_hosthash(&hp->params.ip_addr);
		hp->next = hp->prev = NULL;
		if (acl[h]) {
			hp->next = acl[h];
			(acl[h])->prev = hp;
		}
		acl[h] = hp;
	}
	MUTEX_EXIT(&skip_es_lock);

	SKIP_DEBUG1("skip_es_change_addr: rc = %d\n", rc);
	return (rc);
}

/* skip_es_ok()
 *
 * Return parameters for a host, searching by IP address.  Even if the host
 * was not found, params will contain at least the IP address/mask requested.
 *
 * Returns: B_TRUE if the host is in ACL, B_FALSE otherwise
 */
boolean_t
skip_es_ok(skip_es_t *ifp,
		struct in_addr *addr,
		skip_param_t *params,
		struct ip *iphdr,
		int size)
{
	register boolean_t	ok = B_FALSE;
	skip_es_hash_t		*hp, **acl;

	if (skip_acl_find(ifp, addr, &hp, &acl)) {
		/*
		 * found an ACL entry
		 */
		PARAMSCOPY(&hp->params, params);

		if (params->version != SKIP_NONE) {
			/*
			 * SKIP remote - check for possible candidate for bypass
			 */
			if ((iphdr->ip_p == IPPROTO_UDP) &&
						skip_es_bypass(iphdr, size)) {
				/*
				 * bypass, allow in the clear
				 */
				BZERO(params, sizeof(skip_param_t));
				ifp->stats.skip_if_bypasses++;
			}
		}
		ok = B_TRUE;
	} else {

		/*
		 * Unknown remote. Check for possible candidate for bypass
		 */
		if (iphdr->ip_p == IPPROTO_UDP && skip_es_bypass(iphdr, size)) {
			/*
			 * suitable for bypass, allow in the clear
			 */
			BZERO(params, sizeof(skip_param_t));
			ifp->stats.skip_if_bypasses++;
			ok = B_TRUE;
		}
	}

	/*
	 * update the returned parameters with real address info
	 */
	if (IPADDRMATCH(&params->ip_addr, &params->tunnel_addr)) {
		/*
		 * for wildcards with unconfigured tunnel addresses,
		 * use real destination address
		 */
		IPADDRCOPY(addr, &params->tunnel_addr);
	}
	IPADDRSET(addr, params);
	return (ok);
}
