/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_es.c	1.18 96/07/08 Sun Microsystems"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_conf.h>
#include <skip_keymgrmsgs.h>
#include <skip_key.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_es.h>
#include <skip_if.h>
#include <skip_crypt.h>

/*
 * global SKIP interface information
 */
char			skip_module_name[] 	= "skip v1.18";
int			skip_busy 		= 0;
int			skip_key_debug 		= 0;
const int		skip_key_tick		= SKIP_KEY_TICK;
int			skip_alignment		= 4; /* 32 bit */
static unsigned int	inetsw_size		= 0;

static struct protosw	*inetsw_default;
extern struct protosw	inetsw[];

skip_es_t		*skip_es_ifs		= NULL;
static int		skip_inited 		= 0;
static unsigned short	skip_pktid;
static skip_softc_t	skip_softc[SKIP_MAX_OPENS];

/*
 * statistics
 */
static unsigned long	skip_bufallocfail	= 0;
extern ioctl_key_stat_t	skip_keys_stats;
MUTEX_T			skip_es_lock;


/*
 * crypto modules available...
 */
#ifdef HAVE_SIMPLECRYPT
extern skip_cryptor_t	simplecrypt_module_v1;
extern skip_cryptor_t	simplecrypt_module;
#endif
#ifdef HAVE_DES_CBC
extern skip_cryptor_t	des_cbc_module_v1;
extern skip_cryptor_t	des_cbc_module;
#endif
#ifdef HAVE_DES_EDE_K3
extern skip_cryptor_t	des_ede_k3_module;
#endif
#ifdef HAVE_SAFER_128SK_CBC
extern skip_cryptor_t	safer_128sk_cbc_module;
#endif
#ifdef HAVE_IDEA_CBC
extern skip_cryptor_t	idea_cbc_module;
#endif

/*
 * Local functions prototypes
 */
static int		skip_ifoutput(struct ifnet *, struct mbuf *,
				struct sockaddr *, struct rtentry *rtp);
static int		skip_ifinput(struct mbuf *, int);
static void		skip_inittimers();
static void		skip_uninittimers();
static void		skip_timer();
static int		skip_add_interface(char *);
static int		skip_del_interface(char *);

static void	     skip_encrypt_done(void *, struct mbuf *, struct mbuf *,
				void *, skip_arg_t *);
static void	     skip_decrypt_done(void *, struct mbuf *, struct mbuf *,
				void *, skip_arg_t *);

static struct mbuf *	skip_bufalloc(int);
static struct mbuf *	skip_bufalign(struct mbuf *, int);
static void		skip_notify(skip_es_t *, struct ip *, skip_param_t *,
				int);
static skip_es_t 	*skip_es_find_ifnet(struct ifnet *);
static void		skip_ipsum(struct ip *);
/*
 * From Crypt/MAC system...
 */
extern int		skip_es_bypass_init();
extern void		skip_es_bypass_uninit();
extern void		skip_key_initstore();
extern void		skip_key_uninitstore();
extern void		skip_key_initcryptors();
extern void		skip_key_uninitcryptors();
extern void		skip_mac_init();
extern void		skip_mac_uninit();
extern int		skip_fmt_kmgr(union skip_messages *, skip_keycb_t *);

extern struct cdevsw skipdevsw;

extern struct domain inetdomain;

/* skip_init()
 *
 * Install SKIP in system
 *
 * Returns: 0 on success, errno otherwise.
 */
int
skip_init()
{
	register int		s, rc;
	register struct protosw	*pr;

	/*
	 *  one-off initialisations
	 */
	if (!skip_inited) {

		skip_inited = 1;
 
		/*
		 * Initialize the bypass channel for certificate discovery
		 */
		rc = skip_es_bypass_init();
		if (rc) { 
			skip_inited = 0; 
			return (rc); 
		} 
 
		skip_key_initstore();

		/*
		 * Initialize the crypto system
		 */
		skip_key_initcryptors();

		/*
		 * SunScreen Mode cryptors
		 */
#ifdef HAVE_SIMPLECRYPT
		skip_install_cryptmod(&simplecrypt_module_v1,
						SKIP_SIMPLECRYPT, SKIP_V1);
#endif
#ifdef HAVE_DES_CBC
		skip_install_cryptmod(&des_cbc_module_v1,
						SKIP_DES_CBC, SKIP_V1);
#endif
		/*
		 * IPSP Mode cryptors
		 */
#ifdef HAVE_SIMPLECRYPT
		skip_install_cryptmod(&simplecrypt_module,
					SKIP_CRYPT_SIMPLECRYPT, SKIP_V2);
#endif
#ifdef HAVE_DES_CBC
		skip_install_cryptmod(&des_cbc_module,
					SKIP_CRYPT_DES_CBC, SKIP_V2);
#endif
#ifdef HAVE_DES_EDE_K3
		skip_install_cryptmod(&des_ede_k3_module,
					SKIP_CRYPT_DES_EDE_K3, SKIP_V2);
#endif
#ifdef HAVE_SAFER_128SK_CBC
		skip_install_cryptmod(&safer_128sk_cbc_module,
					SKIP_CRYPT_SAFER_128SK_CBC, SKIP_V2);
#endif
#ifdef HAVE_IDEA_CBC
		skip_install_cryptmod(&idea_cbc_module,
					SKIP_CRYPT_IDEA_CBC, SKIP_V2);
#endif

		bzero((caddr_t)&skip_softc, sizeof(skip_softc_t)*
				SKIP_MAX_OPENS);
		/*
		 * Initialize the MAC system
		 */
		skip_mac_init();

		skip_inittimers();
		
		skip_pktid = htons(lbolt);
		s = splimp();

		/*
		 * save current inetsw
		 */
		inetsw_size = (inetdomain.dom_protoswNPROTOSW -
			inetdomain.dom_protosw) * sizeof(struct protosw);
	
		inetsw_default = SYSMEM_ALLOC(inetsw_size);
		if (inetsw_default== NULL) {
			return (ENOMEM);
		}
		
		bcopy ((caddr_t) inetsw, (caddr_t) inetsw_default, inetsw_size);

		/*
		 * redirect protocol input routines to skip
		 */

		for (pr = inetdomain.dom_protosw;
			pr < inetdomain.dom_protoswNPROTOSW; pr++) {

			pr->pr_input = skip_ifinput;
		}
		splx(s);
	}
	return (0);
}
 
/* skip_uninit()
 *
 * Uninstall SKIP in system
 *
 * Returns: 0 on success, errno otherwise.
 */
int
skip_uninit()
{
	register int		s;
	if (skip_busy || skip_keys_stats.skip_encrypt_keys_active 
			|| skip_keys_stats.skip_decrypt_keys_active) {
		return (EBUSY);
	}

	if (skip_inited) {
		/*
		 * restore inetsw
		 */
		s = splimp();
		bcopy((caddr_t) inetsw_default, (caddr_t) inetsw, inetsw_size);
		splx(s);
		SYSMEM_FREE(inetsw_default, inetsw_size);
		inetsw_size = 0;

		skip_uninittimers();
		skip_mac_uninit();
		skip_key_uninitcryptors();
		skip_key_uninitstore();
		skip_es_bypass_uninit();
		skip_inited = 0;
	}
	return (0);
}
 
/*
 * SKIP File Interface routines
 */

/* skip_ifopen()
 *
 * Common open routine for communication with user space.
 *
 * Returns: 0 on success, errno otherwise
 */
/*ARGSUSED*/
int
skip_ifopen(dev, flags, devtype, p)
	dev_t		dev;	/* maj/min device number */
	int		flags;	/* flags */
	int 		devtype;
	struct 		proc *p;
{
	skip_softc_t	*sp;

#if 0
	SKIP_DEBUG2("skip_ifopen(): DEVICE major=%d, flags = %d\n",
						major(dev), flags);
#endif

	if (suser(p->p_ucred, &p->p_acflag )) {
		return (EPERM);
	}
	if (minor(dev) >= SKIP_MAX_OPENS) {
		return (ENXIO);
	}

	sp = &skip_softc[minor(dev)];

	if (sp->sp_flags & SKIP_BUSY) {
		return (EBUSY);
	}
	sp->sp_flags |= SKIP_BUSY;

	if (flags & FNDELAY) {
		sp->sp_flags |= SKIP_NDELAY;
	}
	bzero((caddr_t) &sp->q, sizeof (struct ifqueue));
	sp->q.ifq_maxlen = SKIP_KEY_MAX_PKTS;

	skip_busy++;
	return (0);
}

/* skip_ifclose()
 *
 * Common close routine for communication with user space.
 *
 * Returns: 0
 */
int
skip_ifclose(dev, flags, devtype, p)
	dev_t		dev;	/* maj/min device number */
	int		flags;	/* flags */
	int 		devtype;
	struct		proc *p;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register struct mbuf	*m;
	register skip_es_t	*skip_if;
	int			s;


	s = splimp();
	for (;;) {
		IF_DEQUEUE(&sp->q, m); 
		if (m) {
			m_freem(m);
		} else {
			break;
		}
	}
	splx(s);

	if (minor(dev) == SKIP_KEYMGR_MINOR) {
		/*
		 * Flush the local Master Key IDs
		 */
		skip_local_keyid_uninit();
	}

	/*
	 * if this pseudo-device was used to notify events on an interface
	 * then remove it
	 */
	for (skip_if = skip_es_ifs; skip_if; skip_if = skip_if->next) {
		if (skip_if->notifier == minor(dev)) {
			skip_if->notifier = 0;
		}
	}

	bzero((caddr_t) sp, sizeof (skip_softc_t));

	skip_busy--;
	return (0);
}

/* skip_ifwakeup()
 *
 * Wakeup a sleeping user-process
 *
 * Returns: none.
 */
static void
skip_ifwakeup(sp)
	skip_softc_t	*sp;
{
	if (sp->sp_si.si_pid) {
		selwakeup(&sp->sp_si);
		sp->sp_flags &= ~SKIP_RCOLL;
		sp->sp_si.si_pid = 0;
	}
	wakeup((caddr_t) sp);

}

/* skip_ifselect()
 *
 * pseudo-device select function.
 *
 * Returns: 0 if no data available, 1 otherwise
 */
int
skip_ifselect(dev, rw, p)
	dev_t		dev;
	int		rw;
	struct proc 	*p;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register int		s;

	if (rw == FWRITE) {
		return (1);
	}
	s = splimp();
	if (sp->q.ifq_len > 0) {
		splx(s);
		return (1);
	}

	selrecord(p, &sp->sp_si);

	splx(s);
	return(0);
}

/* skip_ifread()
 *
 * pseudo-device read function.
 *
 * Returns: 0
 */
int
skip_ifread(dev, uio, ioflag)
	dev_t		dev;
	struct uio	*uio;
	int ioflag;
{
	register skip_softc_t	*sp = &skip_softc[minor(dev)];
	register struct mbuf	*m;
	register int		s, rc, len;

	if (minor(dev) >= SKIP_MAX_OPENS) {
		return (ENXIO);
	}

	/*
	 * read request from the user space process
	 */
	s = splimp();
	for (;;) {
		IF_DEQUEUE(&sp->q, m); 

		if (m == NULL) {
			/*
			 * no messages at the moment
			 */
			if (sp->sp_flags & SKIP_NDELAY) {
				splx(s);
				return (EWOULDBLOCK);
			}
			rc = tsleep((caddr_t)sp, SKIP_PRI | PCATCH, NULL, 0);
			if (rc)  {
				splx(s);
				return rc;
			}
		} else {
			break;
		}
	}
	splx(s);

	len = m->m_len;

	rc = uiomove(mtod(m, caddr_t), len, uio);

	m_freem(m);
	return (0);
}

/* skip_ifwrite()
 *
 * pseudo-device write function.
 *
 * Returns: 0
 */
int
skip_ifwrite(dev, uio, devtype)
	dev_t		dev;
	struct uio	*uio;
	int 		devtype;
{
	register struct mbuf	*m;
	register int		rc;

	if (minor(dev) != SKIP_KEYMGR_MINOR) {
		return (ENXIO);
	}

	/*
	 * process the message from user-space
	 */
	m = skip_bufalloc(uio->uio_resid);
	if (m == NULL) {
		skip_bufallocfail++;
		return (ENOBUFS);
	}

	m->m_len = uio->uio_resid;

	rc = uiomove(mtod(m, caddr_t), uio->uio_resid, uio);
	if (rc) {
		m_freem(m);
		return (rc);
	}
	skip_key_fromkm(mtod(m, union skip_messages *), m->m_len);
	m_freem(m);
	return (0);
}

/* skip_notify()
 *
 * Tell someone about network events
 *
 * Returns: none
 */
static void
skip_notify(skip_if, ip, params, what)
	skip_es_t	*skip_if;
	struct ip	*ip;
	skip_param_t	*params;
	int		what;
{
	register skip_softc_t		*sp;
	register int			s;
	register struct mbuf		*m;
	register skip_es_notification_t	*np;

	if (skip_if->notifier == 0) {
		/*
		 * no one is listening
		 */
		return;
	}

	if (params && (params->flags & SKIP_DONTLOG)) {
		/*
		 * ignore events from this host
		 */
		return;
	}
	sp = &skip_softc[skip_if->notifier];

	s = splimp();
	if (IF_QFULL(&sp->q)) {
		SKIP_DEBUG("skip_notify: too many requests\n");
		IF_DROP(&sp->q);
		splx(s);
		return;
	}
	splx(s);

	/*
	 * create an mbuf for the message
	 */
	m = skip_bufalloc(sizeof (union skip_messages) + MAXVARSZ);
	if (m == NULL) {
		SKIP_DEBUG("skip_notify: mbuf alloc failed\n");
		skip_bufallocfail++;
		return;
	}

	np = mtod(m, skip_es_notification_t *);

	m->m_len = sizeof (skip_es_notification_t);

	bzero((caddr_t) np, sizeof(*np));

	bcopy(skip_if->if_name, np->if_name, sizeof(skip_if->if_name));
	np->what = what;
	bcopy((caddr_t) ip, (caddr_t) &np->iphdr, sizeof(*ip));

	if (params) {
		PARAMSCOPY(params, &np->params);
		/*
		 * the kernel internally initialises r_mkeyid even for NSID 0
		 * This is confusing for user-space tools so zap it.
		 */
		if (np->params.r_nsid == 0) {
			np->params.r_mkeyid.len = 0;
		}
	}

	s = splimp();
	IF_ENQUEUE(&sp->q, m);
	splx(s);
	skip_ifwakeup(sp);
}

/* skip_key_tellkm()
 *
 * send a message to the key manager daemon
 *
 * Returns: none
 */
void
skip_key_tellkm(msg, cp)
	enum skip_msgs	msg;
	skip_keycb_t	*cp;
{
	register skip_softc_t		*sp = &skip_softc[SKIP_KEYMGR_MINOR];
	register struct mbuf		*m;
	union skip_messages		*sm;
	register int			s;

	if ((sp->sp_flags & SKIP_BUSY) == 0) {
		SKIP_DEBUG("skip_key_tellkm: no key manager\n");
		return;
	}

	s = splimp();
	if (IF_QFULL(&sp->q)) {
		SKIP_DEBUG("skip_key_tellkm: too many requests\n");
		IF_DROP(&sp->q);
		splx(s);
		return;
	}
	splx(s);

	/*
	 * create an mbuf for the message
	 */
	m = skip_bufalloc(sizeof (union skip_messages) + MAXVARSZ);
	if (m == NULL) {
		SKIP_DEBUG("skip_key_tellkm: mbuf alloc failed\n");
		skip_bufallocfail++;
		return;
	}

	sm = mtod(m, union skip_messages *);

	sm->msgtype   = msg;

	/*
	 * Now, format the key manager message
	 */
	m->m_len = skip_fmt_kmgr(sm, cp);

	s = splimp();
	IF_ENQUEUE(&sp->q, m);
	splx(s);
	skip_keys_stats.skip_keymgr_requests++;
	skip_ifwakeup(sp);
}

/*
 * SKIP Buffer management routines
 */

/* skip_bufalloc()
 *
 * Allocate a contiguous buffer of at least the specified size.
 *
 * The returned mbuf's length is marked as zero but the storage is available.
 *
 * Returns: mbuf on success, NULL otherwise.
 */
static struct mbuf *
skip_bufalloc(len)
	int		len;
{
	register struct mbuf	*m;

	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL) { 
		return (NULL);
	}
	MCLGET(m, M_DONTWAIT);
	if ((m->m_flags & M_EXT) == 0)  { 
		m_free(m);
		return (NULL);
	}
	m->m_len = 0;
	return (m);
}

/* skip_bufextend()
 * 
 * test if space exists at the end of the buffer to extend it
 */
static int
skip_bufextend(m, pad)
struct mbuf	*m;
int		pad;
{
	int freespace;

	return(0);  /* XXX We don't do buf extends in BSD */

	if (m->m_flags & M_EXT) {
		freespace = (m->m_ext.ext_buf + m->m_ext.ext_size) -
						(m->m_data + m->m_len);
	} else {
		freespace = mtod(m, caddr_t) + MLEN - m->m_data - m->m_len;
	}
	return (freespace >= pad);
}

/* skip_bufalign()
 *
 * concatenate and align buffers for the bulk data cryptor
 *
 * An optional extend can be specified
 *
 */
static struct mbuf *
skip_bufalign(m, pad)
	struct mbuf	*m;
	int		pad;
{
	register struct mbuf	*m0;
	register int		len;

	/*
	 * test for already concatenated and aligned
	 */
	if ((m->m_next == NULL) && ALIGNED(mtod(m, caddr_t)) &&
						skip_bufextend(m, pad)) {
		return (m);
	}

	/*
	 * work out total size of mbuf
	 */
	for (len = 0, m0 = m; m0; m0 = m0->m_next) {
		len += m0->m_len;
	}

	/*
	 * allocate new contiguous and aligned mbuf
	 */
	m0 = skip_bufalloc(len + pad);
	if (m0 == NULL) {
		m_freem(m);
		return (NULL);
	}

	/*
	 * copy data to contiguous and aligned buffer
	 */
	m_copydata(m, 0, len, mtod(m0, caddr_t));
	m0->m_pkthdr.len = m0->m_len = len;
	m_freem(m);
	return (m0);
}

/*
 * SKIP Timer management routines
 */

/* skip_inittimers()
 *
 * start timing operations
 *
 * Returns: None
 */
static void
skip_inittimers()
{
	timeout(skip_timer, NULL, skip_key_tick * hz);
}
 
/* skip_uninittimers()
 *
 * stop timing operations
 *
 * Returns: None
 */
static void
skip_uninittimers()
{
	untimeout(skip_timer, NULL);
}
 
/* skip_timer()
 *
 * scan the current key list, trying to find those which have
 * aged or exceeded their transmission quota
 *
 * Returns: none
 */
/*ARGSUSED*/
static void
skip_timer(arg)
	caddr_t		arg;
{
	/*
	 * run through the key store
	 */
	skip_key_iterate(skip_key_check, NULL);
	timeout(skip_timer, NULL, skip_key_tick * hz);
}

#ifdef notdef
/*
 * mbuf debug
 */
static int
pr_mbuf(struct mbuf *m, skip_es_t *skip_if)
{
	for (; m; m = m->m_next) {

		if (SKIP_DECRYPTED(m)) {
			printf("from skip: ");
		}
		printf("m = %p m_len = %d m_off = 0x%lu",
			m, m->m_len, m->m_data);
		if (m->m_next) {
			printf(" -> \n");
		} else {
			printf("\n\n");
		}
	}
	return (0);
}
#endif

void
skip_dump_buf(char *what, unsigned char *p, int len) 
{
	int idx;

	printf("%s : [", what);
	for (idx = 0; idx < len; idx++) {
		printf("%x ", *p++);
	}
	printf("].\n");
}

void
skip_dump_ip(struct ip *ip)
{

	int idx;
	unsigned char *p = (unsigned char *)ip;

	printf("IP Header: [");

	for (idx = 0; idx < sizeof(struct ip); idx++) {
		printf("%x ", *p++);
	}
	printf("].\n");
	printf("ip_tos = %x, ip_len = %d, ip_id = %d, ip_off = %d\n",
			ip->ip_tos, ntohs(ip->ip_len), ntohs(ip->ip_id), 
			ntohs(ip->ip_off));
	printf("ip_ttl = %x, ip_p = %x, ip_sum = %d\n",
			ip->ip_ttl, ip->ip_p, ntohs(ip->ip_sum));

}

/*
 * SKIP Ioctl and Interface management routines
 */

/* skip_es_find_ifnet()
 *
 * Given an ifnet pointer, find the corresponding skip interface structure
 *
 * Returns: 0 on success, errno otherwise
 */
static skip_es_t *
skip_es_find_ifnet(ifp)
	struct ifnet		*ifp;
{
	register skip_es_t	*skip_if;

	for (skip_if = skip_es_ifs; skip_if; skip_if = skip_if->next) {
		if (skip_if->ifp == ifp) {
			break;
		}
	}
	return (skip_if);
}

/* skip_add_interface()
 *
 * Add a new skip interface
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_add_interface(if_name)
	char 		*if_name;
{
	register skip_es_t	*skip_if;
	register struct ifnet	*ifp;
	register void		*sd;
	register int		s;

	/*
	 * find specified network interface in system
	 */
	if ((ifp = SKIP_IFUNIT(if_name)) == NULL) {
		return (ENODEV);
	}

	if (ifp->if_mtu < SKIP_MIN_MTU) {
		printf("skip: %s: interface mtu of %ld is too small\n", 
						if_name, ifp->if_mtu);
		return (EINVAL);
	}

	skip_if = skip_es_find_if(if_name);

	if (skip_if) {
		/*
		 * already exists... 
		 */
		return (EEXIST);
	}

	skip_if = SYSMEM_ALLOC(sizeof(skip_es_t));

	if (skip_if == NULL) {
		/*
		 * no more space... 
		 */
		return (ENOSPC);
	}

	BZERO(skip_if, sizeof(skip_es_t));

	SKIP_DEBUG1("adding skip to %s\n", if_name);

	/*
	 * declare this interface as a SKIP client
	 */
	sd = skip_open(skip_if, skip_encrypt_done, skip_decrypt_done);

	if (sd == NULL) {
		SYSMEM_FREE(skip_if, sizeof (skip_es_t));
		return (ENOSPC);
	}

	/*
	 * create the hash tables for the access control information
	 */
	skip_if->hosthash = SYSMEM_ALLOC(sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ);
	if (skip_if->hosthash == NULL) {
		SYSMEM_FREE(skip_if, sizeof (skip_es_t));
		skip_close(sd);
		return (ENOSPC);
	}
	BZERO(skip_if->hosthash, sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ);

	skip_if->nethash = SYSMEM_ALLOC(sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ);

	if (skip_if->nethash == NULL) {
		SYSMEM_FREE(skip_if->hosthash, sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ);
		SYSMEM_FREE(skip_if, sizeof (skip_es_t));
		skip_close(sd);
		return (ENOSPC);
	}
	BZERO(skip_if->nethash, sizeof (skip_es_hash_t *) *
					SKIP_ES_HASH_TABLESZ);


	skip_if->sd		= sd;
	skip_if->ifp		= ifp;
	skip_if->if_mode	= SkipAccessControlOn;
	skip_if->netmask	= SKIP_HOSTMASK;

	strncpy(skip_if->if_name, if_name, sizeof (skip_if->if_name));

	/*
	 * add to list
	 */
	s = splimp();
	skip_busy++;
	skip_if->next = skip_es_ifs;
	skip_es_ifs = skip_if;

	/*
	 * and redirect network interface traffic to SKIP
	 */
	skip_if->if_output	= ifp->if_output;
	ifp->if_output		= skip_ifoutput;

	/*
	 * drop interface mtu to allow for SKIP header
	 */
	ifp->if_mtu 		-= SKIP_HDR_SZ;

	splx(s);

	return (0);
}

/* skip_del_interface()
 *
 * Remove a skip interface
 *
 * Returns: 0 on success, errno otherwise
 */
static int
skip_del_interface(if_name)
	char 		*if_name;
{
	register skip_es_t		*skip_if, **prevskip_if;
	register skip_es_hash_t		*hp, *nhp;
	register int			h;
	register int			s;
	
	s = splimp();

	/*
	 * remove the interface from the list
	 */
	for (prevskip_if = &skip_es_ifs; (skip_if = *prevskip_if);
						prevskip_if = &skip_if->next) {
		if (strcmp(skip_if->if_name, if_name) == 0) {
			break;
		}
	}

	if (skip_if != NULL) {
		*prevskip_if = skip_if->next;
	}

	if (skip_if == NULL) {
		splx(s);
		return (ENXIO);
	}

	SKIP_DEBUG1("removing skip from %s\n", if_name);

	skip_close(skip_if->sd);

	/*
	 * restore normal network interface output
	 */
	skip_if->ifp->if_output = skip_if->if_output;

	/*
	 * restore original mtu
	 */
	skip_if->ifp->if_mtu 	+= SKIP_HDR_SZ;
	skip_busy--;

	splx(s);

	if (skip_if->hosthash) {
		for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
			for (hp = skip_if->hosthash[h]; hp; hp = nhp) {
				nhp = hp->next;
				SYSMEM_FREE(hp, sizeof (*hp));
			}
		}
		SYSMEM_FREE(skip_if->hosthash, sizeof (skip_es_hash_t *) *
						SKIP_ES_HASH_TABLESZ);
	}

	if (skip_if->nethash) {
		for (h = 0; h < SKIP_ES_HASH_TABLESZ; h++) {
			for (hp = skip_if->nethash[h]; hp; hp = nhp) {
				nhp = hp->next;
				SYSMEM_FREE(hp, sizeof (*hp));
			}
		}
		SYSMEM_FREE(skip_if->nethash, sizeof (skip_es_hash_t *) *
						SKIP_ES_HASH_TABLESZ);
	}
	SYSMEM_FREE(skip_if, sizeof(skip_es_t));
	return (0);
}


/* skip_ifioctl()
 *
 * handle ioctls from user space
 */
int
skip_ifioctl(dev, cmd, data, fflag, p)
	dev_t		dev;
	int		cmd;
	caddr_t		data;
	int		fflag;
	struct		proc *p;
{
	register skip_es_req_t		*reqp;
	register caddr_t		iobuf;
	register int			rc = EINVAL;
	int				len;
	skip_io_t			*iocp = (skip_io_t *) data;

	
	if (iocp->ic_len < SKIPSZ || iocp->ic_len > SKIP_MAX_IOSZ) {
		SKIP_DEBUG("skip_ifioctl: invalid IOCTL size\n");
		return (EINVAL);
	}

	iobuf = SYSMEM_ALLOC(iocp->ic_len);
	if (iobuf == NULL) {
		return (ENOMEM);
	}
	len = iocp->ic_len;

	rc = copyin(iocp->ic_dp, (caddr_t) iobuf, iocp->ic_len);
	if (rc) {
		SYSMEM_FREE(iobuf, iocp->ic_len);
		return (rc);
	}

	reqp = (skip_es_req_t *) iobuf;

	switch (iocp->ic_cmd) {

	case SKIP_ES_ADD_IF:

		if (iocp->ic_len != sizeof(reqp->if_name)) {
			 goto out;
		}
		rc = skip_add_interface(reqp->if_name);
		break;

	case SKIP_ES_DEL_IF:

		if (iocp->ic_len != sizeof(reqp->if_name)) {
			 goto out;
		}
		rc = skip_del_interface(reqp->if_name);
		break;

	case SKIP_ES_ACL_ADD:

		rc = skip_es_acl_add(reqp, iocp->ic_len);
		break;

	case SKIP_ES_ACL_DEL:

		rc = skip_es_acl_del(reqp, iocp->ic_len);
		break;

	case SKIP_ES_ACL_GET:

		rc = skip_es_acl_get(reqp, iocp->ic_len);

		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);		}
		break;

	case SKIP_ES_SET_MODE:

		if (iocp->ic_len != sizeof(skip_es_mode_t)) {
			goto out;
		}
		rc = skip_es_set_mode(minor(dev),(skip_es_mode_t *)reqp);
		break;

	case SKIP_ES_GET_MODE:

		if (iocp->ic_len != sizeof(skip_es_mode_t)) {
			goto out;
		}
		rc = skip_es_get_mode((skip_es_mode_t *)reqp);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp,
					sizeof(skip_es_mode_t));
		}
		break;

	case SKIP_ES_ACL_LIST:

		rc = skip_es_acl_list(reqp, &iocp->ic_len);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_ES_GET_IF_STATS:

		if (iocp->ic_len != sizeof(skip_ctrl_stats_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_if_stats((skip_ctrl_stats_t *)iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_GET_KEY_STATS:

		if (iocp->ic_len != sizeof(ioctl_key_stat_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_key_stats((ioctl_key_stat_t *) iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_GET_HDR_STATS:

		if (iocp->ic_len != sizeof(ioctl_hdr_stat_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_hdr_stats((ioctl_hdr_stat_t *) iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_GET_CRYPTORS:

		if (iocp->ic_len < sizeof(ioctl_alg_list_t)) {
			rc = EINVAL;
			goto out;
		}

		skip_get_kp_alg_list((ioctl_alg_list_t *) iobuf);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_KEY_GET_KIJ_ALGS:

		if (iocp->ic_len < sizeof(ioctl_alg_list_t)) {
			rc = EINVAL;
			goto out;
		}

		skip_get_kij_alg_list((ioctl_alg_list_t *) iobuf);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_KEY_GET_CPT_STATS:

		if (iocp->ic_len < sizeof(ioctl_crypt_stat_t)) {
			rc = EINVAL;
			goto out;
		}

		skip_cryptor_stats((ioctl_crypt_stat_t *) iobuf);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_KEY_GET_MAC_STATS:

		if (iocp->ic_len < sizeof(ioctl_mac_stat_t)) {
			rc = EINVAL;
			goto out;
		}

		skip_mac_stats((ioctl_mac_stat_t *) iobuf);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	case SKIP_KEY_GET_VAR:

		if (iocp->ic_len < sizeof(ioctl_skip_sym_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_var((ioctl_skip_sym_t *) iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_SET_VAR:

		if (iocp->ic_len < sizeof(ioctl_skip_sym_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_set_var((ioctl_skip_sym_t *) iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_STATUS:

		if (iocp->ic_len < sizeof(ioctl_key_status_t)) {
			rc = EINVAL;
			goto out;
		}

		rc = skip_get_key_status((ioctl_key_status_t *) iobuf);
		if (rc == 0) {
			rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		}
		break;

	case SKIP_KEY_LIST:
 
		if (iocp->ic_len < sizeof(ioctl_keyid_t)) {
			rc = EINVAL;
			goto out;
		}
 
		skip_get_keyid_list((ioctl_keyid_t *) reqp);
		rc = copyout((caddr_t) reqp, iocp->ic_dp, iocp->ic_len);
		break;

	default:
		break;
	}

out:
	SYSMEM_FREE(iobuf, len);
	return (rc);
}
/*
 * IP Packet Processing routines
 */

/* skip_ipsum()
 *
 * compute checksum of an IP packet
 */
static void
skip_ipsum(struct ip *hptr)
{
	unsigned short		*sump;
	unsigned int		sum;
	
	sump = (unsigned short *) hptr;
	hptr->ip_sum = 0;

	/* compute new checksum */
	sum = sump[0] + sump[1] + sump[2] + sump[3] + sump[4] +
		sump[6] + sump[7] + sump[8] + sump[9];
	sum = (sum & 0xFFFF) + (sum >> 16);
	sum = ~(sum + (sum >> 16)) & 0xFFFF;
	if (sum == 0xFFFF) {
		sum = 0;
	}

	hptr->ip_sum = (unsigned short)sum; 
}

/* skip_ifinput()
 *
 * Receive a packet from ipintr(), apply access control and possibly
 * decrypt it if required.
 *
 * Returns: 0 on success, errno otherwise.
 */
int
skip_ifinput(m, hlen)
	struct mbuf		*m;
	int hlen;
{

	struct ip		*ip;
	struct ifnet		*ifp;
	skip_es_t		*skip_if;
	int			iphlen, hdrlen = 0;
	struct mbuf		*decryptbuf = NULL;
	extern u_char		ip_protox[];
	skip_param_t		params;
	skip_hdr_t		skip_hdr;
	skip_es_hash_t          *entry, **acl;


	ifp = m->m_pkthdr.rcvif;

	bzero((caddr_t) &params, sizeof(skip_param_t));
	/* cleartext by default */
	bzero((caddr_t) &skip_hdr, sizeof(skip_hdr_t));
	
	ip = mtod(m, struct ip *);

	if (ip == NULL) {
		panic("skip_ifinput:  mtod returned NULL!");
	}
	
	if (!SKIP_IF(ifp)) {
		/*
		 * SKIP not present on this interface, pass the packet
		 */
		goto pass;
	}

	skip_if = skip_es_find_ifnet(ifp);
	if (skip_if == NULL) {
		panic("skip: failed to find interface in skip_ifinput()\n");
	}

	/*
	 * check for tag which indicates the packet has already been
	 * through SKIP
	 */

	if (SKIP_DECRYPTED(m)) {
		goto pass;
	}

	skip_if->stats.skip_if_ipkts++;

	SKIP_DEBUG("^");

	if (skip_if->if_mode == SkipAccessControlOff) {
		goto pass;
	}

	/*
	 * check the IP protocol field
	 */
	iphlen = IPHLEN(ip);

	switch (ip->ip_p) {

	case SKIP_PROTO_V1:
	case SKIP_IPSP:
		/*
		 * SKIP packet - decode the header to verify the
		 * algorithm information
		 */
		m = skip_bufalign(m, 0);
		if (m == NULL) {
			skip_if->stats.skip_if_drops++;
			return (ENOBUFS);
		}
		ip = mtod(m, struct ip *);

		if (ip->ip_p == SKIP_PROTO_V1) {
			/*
			 * We are in SunScreen mode...
			 */
			hdrlen = skip_hdr_v1_decode(
					(unsigned char *) ip + iphlen,
				 	(unsigned char *) ip + m->m_len,
				 	&skip_hdr	
					);
		} else {
			/*
			 * We are in IPSP mode...
			 */
			hdrlen = skip_hdr_ipsp_decode(
					(unsigned char *) ip + iphlen,
				 	(unsigned char *) ip + m->m_len,
				 	&skip_hdr	
					);
		}

		if (hdrlen < 0) {
			/*
			 * bad SKIP header
			 */
			skip_if->stats.skip_if_drops++;
			m_freem(m);
			SKIP_DEBUG("skip_ifinput: bad header length\n");
			return (ENOPROTOOPT);
		}
		if (skip_hdr.params.r_nsid == SKIP_NSID_NONE) {
			/*
			* Keep the source IP address as the
			* the Receiver Master Key ID.
			*/
			BUFTOKEY(&ip->ip_src, sizeof(ip->ip_src),
						skip_hdr.params.r_mkeyid);
		}

		break;

	case SKIP_NEXT_ESP:
	case SKIP_NEXT_AH:
		/*
		 * "raw" ESP/AH mode packet 
		 */
		m = skip_bufalign(m, 0);
		if (m == NULL) {
			skip_if->stats.skip_if_drops++;
			return (ENOBUFS);
		}
		ip = mtod(m, struct ip *);

		skip_if->stats.skip_if_raw_in++;

		/*
		 * Manually fill in the "params" for ACL lookup..
		 */
		skip_hdr.params.version = SKIP_RAW;
		skip_hdr.params.payload_type = ip->ip_p;
		break;

	case IPPROTO_UDP:
		/*
		 * make sure that the UDP header is present for any
		 * potential bypass test.
		 */
		if (((ip->ip_off & IP_OFFSET) == 0) &&
			m->m_len < (iphlen + sizeof (struct udphdr))) {

			m = m_pullup(m, iphlen + sizeof (struct udphdr));
			if (m == NULL) {
				skip_if->stats.skip_if_drops++;
				return (ENOBUFS);
			}
			ip = mtod(m, struct ip *);
		}

	}
	IPADDRSET(&ip->ip_src, &skip_hdr.params);
	IPADDRCOPY(&ip->ip_src, &skip_hdr.params.tunnel_addr);

	/*
	 * check if we've heard of this host
	 */
	MUTEX_ENTER(&skip_es_lock);

	/* Common works in network order, so we switch the bytes back */
	ip->ip_len = htons(ip->ip_len);
	ip->ip_id = htons(ip->ip_id);
	ip->ip_off = htons(ip->ip_off);
	
	if (!skip_es_ok(skip_if,
			&ip->ip_src,
			&params,
			ip,
			m->m_len
			)) {

		/*
		 * host not found, try to find a nomadic entry...
		 */
		if ((skip_hdr.params.version == SKIP_NONE) ||
		       !skip_es_find_keyid(skip_if,
			       skip_hdr.params.version,
			       skip_hdr.params.r_nsid,
			       &skip_hdr.params.r_mkeyid,
			       &entry,
			       &acl)) {

			/*
			 * host is not configured - tell someone about it.
			 */
			skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipUnknownSrc
				);
			MUTEX_EXIT(&skip_es_lock);
			skip_if->stats.skip_if_drops++;
			m_freem(m);
			return (EACCES);
		}
		PARAMSCOPY(&entry->params, &params);

		/*
		 * mark entry as nomadic (new address)
		 */
		skip_hdr.params.flags |= SKIP_NOMADIC;
	}

	if (params.flags & SKIP_EXCL) {
		/*
		 * host is excluded
		 */
		skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipExcludedSrc
				);
		MUTEX_EXIT(&skip_es_lock);
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		return (EACCES);
	}
		
       switch (skip_hdr.params.version) {
 
	case SKIP_V1:
		/*
		* We have to use the Local Key ID from the ACL entry
		*/
		KEYVARSET(skip_hdr.params.s_mkeyid, params.s_mkeyid);
		skip_hdr.params.s_nsid = params.s_nsid;
		break;
 
	case SKIP_RAW:
		/*
		* For "raw" ESP/AH mode, ACLMATCH will always be OK.
		*/
		params.payload_type = skip_hdr.params.payload_type;
		PARAMSCOPY(&params, &skip_hdr.params);
		break;
	}


	if (!ACLMATCH(&params, &(skip_hdr.params))) {
		/*
		 * The remote is not using the correct parameters
		 */
		SKIP_DIFFS("skip_ifinput", &params, &(skip_hdr.params));

		/*
		 * ACL mismatch, nomadic changed address?
		 */
		if ((skip_hdr.params.version == SKIP_NONE) ||
			!skip_es_find_keyid(skip_if,
				skip_hdr.params.version,
				skip_hdr.params.r_nsid,
				&skip_hdr.params.r_mkeyid,
				&entry,
				&acl)) {
 
			skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipBadParams
				);
			skip_if->stats.skip_if_drops++;
			m_freem(m);
			MUTEX_EXIT(&skip_es_lock);
			return (EACCES);
		}

		/*
		 * recheck the params against nomadic entry
		 */
		if (!ACLMATCH(&entry->params, &(skip_hdr.params))) {
			skip_notify(skip_if,
				ip,
				&skip_hdr.params,
				SkipBadParams
				);
			skip_if->stats.skip_if_drops++;
			m_freem(m);
			MUTEX_EXIT(&skip_es_lock);
			return (EACCES);
		}
		/*
		 * mark entry as nomadic (changed address)
		 */
		skip_hdr.params.flags |= SKIP_NOMADIC;
	}

	MUTEX_EXIT(&skip_es_lock);

	if (!skip_hdr.params.kp_alg && !skip_hdr.params.mac_alg &&
			!skip_hdr.params.comp_alg) {

		if (!hdrlen) {
			/*
			* host allowed to communicate in the clear
			*/
			goto flippass;
		}

	}
	if (skip_hdr.params.comp_alg) {
		/*
		 * Compressed packet...drop it!
		 */
		skip_if->stats.skip_if_drops++;
		SKIP_DEBUG("skip_ifinput: received IPSP compressed!\n");
		m_freem(m);
                return (EPROTO);
        }
 
	if (skip_hdr.params.s_nsid == SKIP_NSID_NONE) {
		/*
		 * Keep the dst IP address as the the Sender Master Key ID.
		 * This is required by the key manager (slotting).
		 */
		BUFTOKEY(&ip->ip_dst, sizeof(ip->ip_dst),
						 skip_hdr.params.s_mkeyid);
	}
	if (skip_hdr.params.kp_alg) {
		/*
	 	 * The packet was encrypted.
		 * Allocate a buffer to decrypt it.
		 */
		decryptbuf = skip_bufalloc(m->m_len + iphlen); 
		if (decryptbuf == NULL) {
			m_freem(m);
			skip_if->stats.skip_if_drops++;
			skip_bufallocfail++;
			return(ENOBUFS);
		}
		/*
		 * save space for the IP header
		 */
		decryptbuf->m_data += iphlen;
	
                SKIP_DEBUG2("skip_ifinput: decryptbuf m_len=%d m_data=%d\n",
                                        decryptbuf->m_len, decryptbuf->m_data);

	} 


	/* This sets the len to be the length of the original packet
	*  before ipintr removed the header */

	ip->ip_len = htons(m->m_len);

	/*
	 * get the packet authenticated and/or decrypted
	 */
	skip_decrypt(skip_if->sd, m, decryptbuf, 
				&skip_hdr, (iphlen + hdrlen), NULL);
	skip_if->stats.skip_if_decrypts++;
	return (0);

flippass:
	/*
	 * Flip bytes back to host order the way BSD expects 
	 */
	ip->ip_len = ntohs(ip->ip_len);
	ip->ip_id = ntohs(ip->ip_id);
	ip->ip_off = ntohs(ip->ip_off);
pass:
	m->m_flags &= ~ M_EOR;
	(*inetsw_default[ip_protox[ip->ip_p]].pr_input)(m, hlen);
	return (0);
}


/* skip_ifoutput()
 *
 * Receive a packet from IP, apply access control & optionally encrypt
 * and/or authenticate the packet before transmission.
 *
 * Returns: 0
 */
int
skip_ifoutput(ifp, m, dst, rtp)
	struct ifnet		*ifp;
	struct mbuf		*m;
	struct sockaddr		*dst;
	struct rtentry 		*rtp;
{
	register skip_es_t		*skip_if;
	register struct ip		*ip, *newip;
	register int			iphlen;
	register struct mbuf		*encryptbuf;
	skip_param_t			params;
	
	bzero((caddr_t) &params, sizeof(skip_param_t));

	skip_if = skip_es_find_ifnet(ifp);
	if (skip_if == NULL) {
		panic("skip: failed to find interface in skip_ifoutput()\n");
	}

	if (dst->sa_family != AF_INET) {
		return (*skip_if->if_output)(ifp, m, dst, rtp);
	}

	SKIP_DEBUG(".");

	if (skip_if->if_mode == SkipAccessControlOff) {
		return (*skip_if->if_output)(ifp, m, dst, rtp);
	}
		
	skip_if->stats.skip_if_opkts++;

	/*
	 * check if destination requires encryption
	 */
	ip = mtod(m, struct ip *);
	iphlen = IPHLEN(ip);


	if (ip->ip_p == IPPROTO_UDP) {
		/*
		 * make sure that the UDP header is present for any
		 * potential bypass test.
		 */
		if (((ip->ip_off & IP_OFFSET) == 0) &&
			m->m_len < (iphlen + sizeof (struct udphdr))) {

			m = m_pullup(m, iphlen + sizeof (struct udphdr));
			if (m == NULL) {
				skip_if->stats.skip_if_drops++;
				return (0);
			}
			ip = mtod(m, struct ip *);
		}
	}

	MUTEX_ENTER(&skip_es_lock);

	if (!skip_es_ok(skip_if, &ip->ip_dst, &params, ip, m->m_len)) {

		/*
		 * never heard of this host
		 */
		IPADDRCOPY(&ip->ip_dst, &params.tunnel_addr);
		skip_notify(skip_if,
				ip,
				&params,
				SkipUnknownDst
				);
		MUTEX_EXIT(&skip_es_lock);
		m_freem(m);
		skip_if->stats.skip_if_drops++;
		return (EACCES);
	}

	if (params.flags & SKIP_EXCL) {
		/*
		 * host is excluded
		 */
		skip_notify(skip_if,
				ip,
				&params,
				SkipExcludedDst
				);
		MUTEX_EXIT(&skip_es_lock);
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		return (EACCES);
	}

	MUTEX_EXIT(&skip_es_lock);

	/*
	 * Check if it is a non SKIP host 
	 */
	if (params.version == SKIP_NONE) {
		/*
		 * configured to send in the clear
		 */
		return (*skip_if->if_output)(ifp, m, dst, rtp);
	}

	/*
	 * Check if we  are in SunScreen mode..
	 */
	if (params.version == SKIP_V1) {
		/*
		 * packet is to be encrypted - make sure source is aligned and
		 * contiguous.
		 */
		m = skip_bufalign(m, SKIP_MAX_PAD);
		if (m == NULL) {
			skip_if->stats.skip_if_drops++;
			return (ENOBUFS);
		}
		ip = mtod(m, struct ip *);
	
		/*
		 * prepare a buffer for the encryptor
		 */
		encryptbuf = skip_bufalloc(m->m_len + SKIP_HDR_SZ);
		if (encryptbuf == NULL) {
			m_freem(m);
			skip_bufallocfail++;
			return (ENOBUFS);
		}
	
		/*
		 * insert new IP header in encryptbuf
		 */
		newip = mtod(encryptbuf, struct ip *);
	
		bcopy((caddr_t) ip, (caddr_t) newip, sizeof(struct ip));
	
		newip->ip_p = SKIP_PROTO_V1;
	
		/*
		 * Mark the SKIP packet as unfragmented, even though the payload
		 * may have been a fragment.
		 */
		newip->ip_off = 0;
	
		/*
		 * insert a new packet id
		 */
		newip->ip_id = htons(skip_pktid++);
	
		/*
 		 * insert tunnel address
		 */
                IPADDRCOPY(&params.tunnel_addr, &newip->ip_dst);

		encryptbuf->m_len += sizeof (struct ip);

		/*
		 * Keep the Source IP address as the Sender Master Key  ID.
		 * This is required by the key manager (slotting).
		 */
		if (params.s_mkeyid.len == 0) {
			BUFTOKEY(&ip->ip_src, sizeof(ip->ip_src), 
						params.s_mkeyid);
		}
		IPADDRSET(&ip->ip_dst, &params);

		/*
		 * handle bundle to encryptor
		 */
		rtp->rt_refcnt++;	/* bump up the reference count */

		/*
		 * save the route pointer in unused field
		 */
		m->m_pkthdr.rcvif = (struct ifnet *) rtp;

		skip_encrypt(skip_if->sd,	/* SKIP client handle */
				m,		/* plaintext packet */
				encryptbuf,	/* buffer for ciphertext */
				&params,	/* SKIP parameters       */
				dst);

		skip_if->stats.skip_if_encrypts++;
		return (0);
	}

	/*
	 * From now, IPSP mode...
	 */
	if (params.comp_alg) {
		/*
		 * XXX - The packet should be compressed - drop it!
		 */
		m_freem(m);
		SKIP_DEBUG1(
			"skip_ifoutput: should compress ESP, alg=%d !\n",
			params.comp_alg);
		skip_if->stats.skip_if_drops++;
		return (ENOPROTOOPT);
	}


	/*
	 * packet is to be encrypted or authenticated - make sure 
	 * source is aligned and  contiguous.
	 */
	m = skip_bufalign(m, SKIP_MAX_PAD);
	if (m == NULL) {
		skip_if->stats.skip_if_drops++;
		return (ENOBUFS);
	}
	ip = mtod(m, struct ip *);

#ifdef notdef
	printf("ifoutput: len=%d\n", BUFLEN(m));
	skip_dump_buf("ifoutput (IPSP mode)", BUFSTART(m), BUFLEN(m));
#endif

	/*
	 * prepare a buffer for the encryptor
	 */
	encryptbuf = skip_bufalloc(m->m_len + SKIP_HDR_SZ);
	if (encryptbuf == NULL) {
		m_freem(m);
		skip_bufallocfail++;
		return (ENOBUFS);
	}

	/*
	 * insert new IP header in encryptbuf
	 */
	newip = mtod(encryptbuf, struct ip *);
	bcopy((caddr_t) ip, (caddr_t) newip, sizeof(struct ip));

	if (params.version == SKIP_V2) {
		/*
		 * SKIP IPSP mode...
		 */
		newip->ip_p = SKIP_IPSP;
	} else {
		/*
		 * "raw" IPSP mode...
		 */
		if (params.mac_alg) {
			newip->ip_p = SKIP_NEXT_AH;
		} else {
			if (params.kp_alg) {
				newip->ip_p = SKIP_NEXT_ESP;
			} else {
				newip->ip_p = IPPROTO_ENCAP;
			}
		}
		skip_if->stats.skip_if_raw_out++;

	}
	encryptbuf->m_len += sizeof (struct ip);

	/*
	 * insert tunnel address
	 */
	if (params.flags & SKIP_NOMADIC) {
		if (!SKIP_IS_UNKNOWNIP(params.tunnel_addr)) {
			/*
			 * insert tunnel address as source if supplied
			 */
			IPADDRCOPY(&params.tunnel_addr, &newip->ip_src);
		}
	} else {
		/*
		 * insert tunnel address as destination
		 */
		IPADDRCOPY(&params.tunnel_addr, &newip->ip_dst);
	}

	if (params.s_nsid == 0) {
		/*
		 * Keep the Source IP address as the Sender Master Key  ID.
		 * This is required by the key manager (slotting).
		 */
		BUFTOKEY(&ip->ip_src, sizeof(ip->ip_src), params.s_mkeyid);
	}


	if (params.ip_mode != SKIP_TRS_ON) {
		/*
	 	 * SKIP tunneling mode...
		 * Mark the SKIP packet as unfragmented,
		 * even though the payload may have been a fragment.
		 */
		newip->ip_off = 0;
	
		/*
		 * insert a new packet id
		 */
		newip->ip_id = htons(skip_pktid++);
	}

	IPADDRSET(&ip->ip_dst, &params);

#ifdef notdef
	skip_dump_buf("ifoutput (original)", BUFSTART(m), BUFLEN(m));
#endif
	rtp->rt_refcnt++;		/* bump up the reference count on the 						   route */

	/*
	 * save the route pointer in unused field
	 */
	m->m_pkthdr.rcvif = (struct ifnet *) rtp;

	/*
	 * handle bundle to encryptor
	 */
	
	skip_encrypt(skip_if->sd,	/* SKIP client handle */
			m,		/* plaintext packet */
			encryptbuf,	/* buffer for ciphertext */
			&params,	/* SKIP parameters */
			dst); 

	skip_if->stats.skip_if_encrypts++;
	return (0);
}

/* skip_decrypt_done()
 *
 * receive an authenticated/decrypted packet from SKIP and send to IP
 *
 * Returns: none
 */
static void
skip_decrypt_done(client_data, original, m, argv, res)
	void		*client_data;
	struct mbuf	*original;
	struct mbuf	*m;
	void 		*argv;
	skip_arg_t 	*res;
{
	register skip_es_t	*skip_if = client_data;
	register skip_param_t   *params = &res->params;
	register struct ip      *ip = mtod(original, struct ip *);
	int                     rc, s, iphlen;
	struct mbuf             *outbuf, *new_hdr;

	SKIP_PRINT("skip_decrypt_done", params);

	if (res->rc) {
		goto bad;
	}

	/*
	 * if this is a nomadic host, check for updated IP address
	 */
	if (params->flags & SKIP_NOMADIC) {
		/*
		 * change to new address
		 */
		rc = skip_es_change_addr(skip_if, params, &ip->ip_src);
		if (rc) {
			SKIP_DEBUG1("skip_decrypt_done: skip_es_change_addr "
							"rc = %d\n", rc);
			goto bad;
		}
	}

	/*
	 * if no encryption - we do everything in the original buffer
	 */
	outbuf = (res->modes & SKIP_CRYPT_ON) ? m : original;

	if (res->proto != IPPROTO_ENCAP) {
		/*
		 * transport mode, need to copy original IP header
		 */
		iphlen = IPHLEN(ip);
		if (outbuf == original) {
			outbuf->m_data += res->offset;
			outbuf->m_len -= res->offset;
		}
		/*
		 * back up enough to insert IP header
		 */
		outbuf->m_data -= iphlen;
		outbuf->m_len += iphlen;

		/*
		 * XXX overlapping buffers ?
		 */
		bcopy((char *) ip, mtod(outbuf, caddr_t), iphlen);

		/*
		 *  Set the protocol and length fields...
		 */
		ip = mtod(outbuf, struct ip *);
		ip->ip_p = res->proto;
		ip->ip_len = htons(BUFLEN(outbuf));

		/*
		 * Now, produce a new IP check sum...
		 */
		skip_ipsum(ip);
	} else {
		/*
		 * tunnel mode
		 */	
		if (outbuf == original) {
			outbuf->m_data += res->offset;
			outbuf->m_len -= res->offset;
		}
	}
	if (outbuf == m) {
		m_freem(original);
	}

	ip = mtod(outbuf, struct ip *);
	iphlen = IPHLEN(ip);

	if (outbuf->m_len < iphlen) {
		/*
		 * short packet?
		 */
		skip_if->stats.skip_if_drops++;
		m_freem(outbuf);
		return;
	}

#ifdef notdef
	printf("skip_decrypt_done: OK mode=%x, proto=%x, offset=%x\n",
					res->modes, res->proto, res->offset);
#endif

	/*
	 * send the decrypted packet on to IP
	 */
	outbuf->m_pkthdr.rcvif  = skip_if->ifp;
	outbuf->m_pkthdr.len = outbuf->m_len;
	
	/*
	 * tag the start of the header buffer so SKIP can recognise it
	 */
	outbuf->m_flags |= M_EOR | M_PKTHDR;
    
	s = splimp();
	if (IF_QFULL(&ipintrq)) {
		IF_DROP(&ipintrq);
		m_freem(outbuf);
		skip_if->stats.skip_if_drops++;
		(void) splx(s);
		return; 
	}	       
	if (ipintrq.ifq_len == 0) {
		schednetisr(NETISR_IP); 
	}
	IF_ENQUEUE(&ipintrq, outbuf);
	(void) splx(s); 
	return;

bad:
	skip_if->stats.skip_if_drops++;
	skip_notify(skip_if, ip, NULL, SkipAuthFailed);
	m_freem(original);
	if (m) {
		m_freem(m);
	}
}

/* skip_encrypt_done()
 *
 * complete an encrypted packet from SKIP and send down to the network
 *
 * Returns: none
 */
static void
skip_encrypt_done(client_data, original, m, arg, res)
	void	    *client_data; 
	struct mbuf     *original;
	struct mbuf     *m; 
	void	    *arg;
	skip_arg_t      *res;
{
	skip_es_t		*skip_if = client_data;
	struct sockaddr		*dst = (struct sockaddr *)arg;

	unsigned short		len;
	struct ip		*hptr;
	struct rtentry		*rtp;
	
	/*
	 * recover the route pointer
	 */
	rtp = (struct rtentry *) original->m_pkthdr.rcvif;

	m_freem(original);

	if (res->rc) {
		skip_if->stats.skip_if_drops++;
		m_freem(m);
		RTFREE(rtp);
		return;
	}

	/*
	 * compute checksum of encrypted packet
	 */
	len = m->m_len;
	hptr = mtod(m, struct ip *);
	hptr->ip_len = htons(len); /* total length */

	skip_ipsum(hptr);

	(*skip_if->if_output)(skip_if->ifp, m, dst, rtp);
	RTFREE(rtp);
}
