/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skiplog.c	1.6 96/08/06"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_conf.h>
#include <skip_types.h>
#include <skip_proto.h>
#include <skip_ioctl.h>
#include <skip_acl.h>
#include <skip_lib.h>

#define	MAXMSGS			10
#define	STRSZ			256
static const int		inaddrsz = 4;
static struct in_addr		myaddr;
static char			*ifname, hostname[STRSZ];
static char			*versions[]  = {
					"cleartext",
					"SKIP (version 1)",
					"SKIP V2",
					"RAW ESP/AH"
				};
/*
 * logging routines
 */
static void
add_msg(char *msgs[], char *msg)
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] == NULL) {
			msgs[i] = strdup(msg);
			break;
		}
	}
}

static void
log_msgs(char *msgs[])
{
	register int	i;
	char		t[STRSZ];

	skip_time_stamp(t);

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i]) {
			if (i == 0) {
				printf("%s : ", t);
			} else {
				printf("\t\t");
			}
			puts(msgs[i]);
		}
	}
}

static void
destroy_msgs(char *msgs[])
{
	register int	i;

	for (i = 0; i < MAXMSGS; i++) {
		if (msgs[i] != NULL) {
			free(msgs[i]);
			msgs[i] = NULL;
		}
	}
}

/* is_this_system()
 *
 */
static int
is_this_system(struct in_addr *addr)
{
	return (addr->s_addr == myaddr.s_addr);
}

/* get_src_dst()
 *
 * Get address information
 */
static void
get_src_dst(struct ip *ip, char src[], char dst[])
{
	if (is_this_system(&ip->ip_src)) {
		strcpy(src, "this system");
	} else {
		(void) skip_addr_to_host(&ip->ip_src, src);
	}

	if (is_this_system(&ip->ip_dst)) {
		strcpy(dst, "this system");
	} else {
		(void) skip_addr_to_host(&ip->ip_dst, dst);
	}
}

/*
 * handle incoming packet from an unknown destination
 */
static void
unknown_src(skip_es_notification_t *skipmsgp)
{
	boolean_t		secure, is_bad = B_FALSE;
	skip_param_t		*params = &skipmsgp->params;
	struct ip		*ip;
	char			*kij_algname, *kp_algname, *versionstr;
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ], src[STRSZ], dst[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;

	get_src_dst(ip, src, dst);

	switch (params->version) {
	case SKIP_NONE:
		versionstr = "in the clear";
		break;

	case SKIP_V1:
	case SKIP_V2:
	case SKIP_RAW:
		sprintf(s, "with %s using\n", versions[params->version]);
		versionstr = s;
		break;
	default:
		sprintf(s, "with bad version #%d\n", params->version);
		versionstr = s;
	}

	secure = (params->version != SKIP_NONE);

	sprintf(msg, "%s (%s) is trying to connect to %s",
					src, inet_ntoa(ip->ip_src), dst);
	add_msg(msgs, msg);

	if (secure) {
		/*
		 * check we could support this host
		 */
		kij_algname = skip_kij_alg_to_name(params->kij_alg,
							params->version);
		kp_algname = skip_kp_alg_to_name(params->kp_alg,
							params->version);

		if (!skip_supported_kij_alg(params->kij_alg, params->version)) {
			sprintf(msg, "and unsupported key encryption "
				"algorithm #%d (%s)",
				params->kij_alg, kij_algname);

			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if (!skip_supported_kp_alg(params->kp_alg, params->version)) {
			sprintf(msg, "and unsupported traffic encryption "
				"algorithm #%d (%s)",
				params->kp_alg, kp_algname);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}
		if ((int) params->r_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported sender NSID #%d",
				params->r_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if ((int) params->s_nsid >= skip_max_nsids) {
			sprintf(msg, "and unsupported receiver NSID #%d ",
				params->s_nsid);
			add_msg(msgs, msg);
			is_bad = B_TRUE;
		}

		if (is_bad) {
			log_msgs(msgs);
			destroy_msgs(msgs);
			return;
		}

		/*
		 * build algorithm/key information messages
		 */
		sprintf(msg, "%s (key encryption) %s (traffic encryption)",
						kij_algname, kp_algname); 
		add_msg(msgs, msg);

		(void) skip_keyid_to_s(&params->r_mkeyid, params->r_nsid, s);
		sprintf(msg, "- its key id is %s %s",
				skip_nsids[params->r_nsid].name,
				s);
		add_msg(msgs, msg);

		sprintf(msg, "- local key id specified as %s",
				skip_nsids[params->s_nsid].name);
		add_msg(msgs, msg);
	}
	log_msgs(msgs);
	destroy_msgs(msgs);
}

/*
 * handle incoming packet which doesn't match ACL
 */
static void
bad_params(skip_es_notification_t *skipmsgp)
{
	skip_param_t		*in = &skipmsgp->params;
	skip_param_t		acl_params, *ref = &acl_params;
	struct ip		*ip;
	char			in_str[STRSZ], ref_str[STRSZ];
	char			*msgs[MAXMSGS] = { NULL };
	char			msg[STRSZ], s[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;

	(void) skip_addr_to_host(&ip->ip_src, s);

	ref->ip_addr = ip->ip_src;

	if (skip_acl_get(ifname, ref) != 0) {
		sprintf(msg, "Failed to get ACL entry for host %s (%s) - %s",
			s, inet_ntoa(ip->ip_src), skip_errmsg);
		add_msg(msgs, msg);
		log_msgs(msgs);
		destroy_msgs(msgs);
		return;
	}

	sprintf(msg, "%s (%s) is trying to connect with",
					s, inet_ntoa(ip->ip_src));

	add_msg(msgs, msg);

	/*
	 * check SKIP version
	 */
	if (in->version != ref->version) {
		if ((int) in->version <= SKIP_RAW) {
			strcpy(in_str, versions[in->version]);
		} else {
			sprintf(s, "<bad SKIP version %d>", in->version);
		}

		if ((int) ref->version <= SKIP_RAW) {
			strcpy(ref_str, versions[ref->version]);
		} else {
			sprintf(ref_str, "<bad SKIP version %d>", in->version);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
		log_msgs(msgs);
		destroy_msgs(msgs);
		return;
	}

	/*
	 * check remote NSID
	 */
	if (in->r_nsid != ref->r_nsid) {
		if ((int) in->r_nsid >= skip_max_nsids) {
			sprintf(in_str, "remote NSID #%d "
				"<unknown>", in->r_nsid);
		} else {
			sprintf(in_str, "remote NSID #%d " "(%s)",
				in->r_nsid, skip_nsids[in->r_nsid].name);
		}
		if ((int) ref->r_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->r_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->r_nsid, skip_nsids[ref->r_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	} else {
		/*
		 * remote NSID is correct, check key value
		 */
		if (memcmp(in->r_mkeyid.buf, ref->r_mkeyid.buf,
							in->r_mkeyid.len)) {
			/*
			 * Key values do not match.  Check the key size is
			 * valid.
			 */
			if (skip_keyid_to_s(&in->r_mkeyid, in->r_nsid, in_str)
									< 0) {
				sprintf(in_str, "<bad key of length %d>",
						in->r_mkeyid.len);
			}
			if (skip_keyid_to_s(&ref->r_mkeyid, ref->r_nsid,
								ref_str) < 0) {
				sprintf(ref_str, "<bad key of length %d>",
						ref->r_mkeyid.len);
			}
			sprintf(msg, "with remote key %s", in_str);
			add_msg(msgs, msg);
			sprintf(msg, "instead of %s", ref_str);
			add_msg(msgs, msg);
		}
	}

	/*
	 * check local NSID
	 */
	if (in->s_nsid != ref->s_nsid) {
		if ((int) in->s_nsid >= skip_max_nsids) {
			sprintf(in_str, "local NSID #%d "
				"<unknown>", in->s_nsid);
		} else {
			sprintf(in_str, "local NSID #%d (%s)",
				in->s_nsid, skip_nsids[in->s_nsid].name);
		}
		if ((int) ref->s_nsid >= skip_max_nsids) {
			sprintf(ref_str, "#%d <unknown>",
				ref->s_nsid);
		} else {
			sprintf(ref_str, "#%d (%s)",
				ref->s_nsid, skip_nsids[ref->s_nsid].name);
		}
		sprintf(msg, "%s instead of %s", in_str, ref_str);
		add_msg(msgs, msg);
	}

	/*
	 * check Kij algs
	 */
	if (in->kij_alg != ref->kij_alg) {
		sprintf(msg, "key encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kij_alg,
			skip_kij_alg_to_name(in->kij_alg, in->version),
			ref->kij_alg,
			skip_kij_alg_to_name(ref->kij_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check Kp algs
	 */
	if (in->kp_alg != ref->kp_alg) {
		sprintf(msg, "traffic encryption algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->kp_alg,
			skip_kp_alg_to_name(in->kp_alg, in->version),
			ref->kp_alg,
			skip_kp_alg_to_name(ref->kp_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check MAC algs
	 */
	if (in->mac_alg != ref->mac_alg) {
		sprintf(msg, "authentication algorithm #%d (%s) "
				"instead of #%d (%s)",
			in->mac_alg,
			skip_mac_alg_to_name(in->mac_alg, in->version),
			ref->mac_alg,
			skip_mac_alg_to_name(ref->mac_alg, ref->version));
		add_msg(msgs, msg);
	}
	/*
	 * check compression algs
	 */
	if (in->comp_alg != ref->comp_alg) {
		sprintf(msg, "compression algorithm #%d "
				"instead of #%d",
			in->comp_alg,
			ref->comp_alg);
		add_msg(msgs, msg);
	}

	log_msgs(msgs);
	destroy_msgs(msgs);
}

/*
 * handle outgoing packet to an unknown destination
 */
static void
unknown_dst(skip_es_notification_t	*skipmsgp)
{
	struct ip		*ip;
	char			*msgs[MAXMSGS] = { NULL }, msg[STRSZ];
	char			src[STRSZ], dst[STRSZ];

	ip = (struct ip *) &skipmsgp->iphdr;
	get_src_dst(ip, src, dst);
	sprintf(msg, "%s is trying to connect to %s (%s)",
					src, dst, inet_ntoa(ip->ip_dst));
	add_msg(msgs, msg);
	log_msgs(msgs);
	destroy_msgs(msgs);
}


static void
skip_msg(int skip_fd)
{
	int			rc;
	skip_es_notification_t	skipmsg;
	struct ip		*ip;
	char			msg[STRSZ], *msgs[MAXMSGS] = { NULL },
				src[STRSZ], dst[STRSZ];

	rc = skip_getmsg(skip_fd, (void *) &skipmsg, sizeof(skipmsg));

	if (rc < 0) {
		perror("skip_get_message failed");
		return;
	}

	ip = (struct ip *) &skipmsg.iphdr;

	get_src_dst(ip, src, dst);

	switch (skipmsg.what) {

	case SkipUnknownSrc:
		unknown_src(&skipmsg);
		break;

	case SkipUnknownDst:
		unknown_dst(&skipmsg);
		break;

	case SkipBadParams:
		bad_params(&skipmsg);
		break;

	case SkipAuthFailed:
		sprintf(msg, "Warning: %s (%s) failed authentication",
				src, inet_ntoa(ip->ip_src));
		add_msg(msgs, msg);
		sprintf(msg, "Its certificate/key id may be invalid");
		add_msg(msgs, msg);
		log_msgs(msgs);
		break;

	case SkipExcludedSrc:
		sprintf(msg, "Excluded host %s (%s) trying to connect "
				"to %s", src, inet_ntoa(ip->ip_src), dst);
		add_msg(msgs, msg);
		log_msgs(msgs);
		break;

	case SkipExcludedDst:
		sprintf(msg, "%s is trying to connect to "
				" excluded host %s (%s)",
				src, dst, inet_ntoa(ip->ip_dst));
		add_msg(msgs, msg);
		log_msgs(msgs);
		break;

	default:
		sprintf(msg, "unknown message <%d> from skip kernel",
						skipmsg.what);
		add_msg(msgs, msg);
		log_msgs(msgs);
	}
	destroy_msgs(msgs);
}

void
main(int argc, char *argv[])
{
	char			msg[STRSZ], *msgs[MAXMSGS] = { NULL };
	int			fds;
	fd_set			in;
	skip_fd_t		skip_fd;

	if (argc > 2) {
		fprintf(stderr, "usage: %s [ interface ]\n", argv[0]);
		exit(1);
	}

	if (argc == 2) {
		ifname = argv[1];
	} else {
		ifname = skip_default_if();
	}
	(void) skip_hostname(hostname, STRSZ);
	(void) skip_host_to_addr(hostname, &myaddr);

	skip_fd = skip_set_mode(ifname, SkipInteractive);
	if (skip_fd < 0) {
		if (errno == EEXIST) {
			fprintf(stderr, "%s: mode already in use\n", argv[0]);
		} else {
			fprintf(stderr, "%s: error - %s\n", argv[0],
								skip_errmsg);
		}
		exit (1);
	}

	if (skip_var_init(argv[0], ifname)) {
		fprintf(stderr, "%s.\n", skip_errmsg);
	}

	sprintf(msg, "starting logging for interface %s", ifname);

	add_msg(msgs, msg);
	log_msgs(msgs);

	for (;;) {
		FD_ZERO(&in);
		FD_SET(skip_fd, &in);
		fds = select(32, &in, NULL, NULL, NULL);
		if (fds < 0) {
			perror("select");
		} else {
			skip_msg(skip_fd);
		}
	}
	/*NOTREACHED*/
}
