/*
This code is copyright (C) 1998 Robert O'Callahan.
This code is free for non-commercial use. Modification in all forms is permitted.
This license continues to apply to any modified versions. This license text must be
reproduced and distributed with any modified versions.
As a matter of courtesy I (Robert O'Callahan) would like to be informed of
any potentially useful modifications.
*/

#define WINDOWS

#include "ttssh.h"
#include "util.h"

#include "rand.h"
#include "bn.h"
#include "md5.h"
#include "err.h"

static void no_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
}

static void c3DES_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  Cipher3DESState FAR * encryptstate = &pvar->crypt_state.enc.c3DES;

  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
  	encryptstate->k1, &encryptstate->ivec1, DES_ENCRYPT);
  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  encryptstate->k2, &encryptstate->ivec2, DES_DECRYPT);
  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  encryptstate->k3, &encryptstate->ivec3, DES_ENCRYPT);
}

static void c3DES_decrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  Cipher3DESState FAR * decryptstate = &pvar->crypt_state.dec.c3DES;

  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  decryptstate->k3, &decryptstate->ivec3, DES_DECRYPT);
  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  decryptstate->k2, &decryptstate->ivec2, DES_ENCRYPT);
  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  decryptstate->k1, &decryptstate->ivec1, DES_DECRYPT);
}

static void cDES_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherDESState FAR * encryptstate = &pvar->crypt_state.enc.cDES;

  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
  	encryptstate->k, &encryptstate->ivec, DES_ENCRYPT);
}

static void cDES_decrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherDESState FAR * decryptstate = &pvar->crypt_state.dec.cDES;

  des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	  decryptstate->k, &decryptstate->ivec, DES_DECRYPT);
}

static void cIDEA_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherIDEAState FAR * encryptstate = &pvar->crypt_state.enc.cIDEA;
  int num = 0;

  idea_cfb64_encrypt(buf, buf, bytes, &encryptstate->k, encryptstate->ivec,
  	&num, IDEA_ENCRYPT);
}

static void cIDEA_decrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherIDEAState FAR * decryptstate = &pvar->crypt_state.dec.cIDEA;
  int num = 0;

  idea_cfb64_encrypt(buf, buf, bytes, &decryptstate->k, decryptstate->ivec,
  	&num, IDEA_DECRYPT);
}

static void flip_endianness(unsigned char FAR * cbuf, int bytes) {
  uint32 FAR * buf = (uint32 FAR *)cbuf;
  int count = bytes/4;

  while (count > 0) {
    uint32 w = *buf;

    *buf = ((w << 24) & 0xFF000000) | ((w << 8) & 0x00FF0000)
      | ((w >> 8) & 0x0000FF00) | ((w >> 24) & 0x000000FF);
    count--;
    buf++;
  }
}

static void cBlowfish_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherBlowfishState FAR * encryptstate = &pvar->crypt_state.enc.cBlowfish;

  flip_endianness(buf, bytes);
  BF_cbc_encrypt(buf, buf, bytes, &encryptstate->k, encryptstate->ivec,
    BF_ENCRYPT);
  flip_endianness(buf, bytes);
}

static void cBlowfish_decrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherBlowfishState FAR * decryptstate = &pvar->crypt_state.dec.cBlowfish;

  flip_endianness(buf, bytes);
  BF_cbc_encrypt(buf, buf, bytes, &decryptstate->k, decryptstate->ivec,
    BF_DECRYPT);
  flip_endianness(buf, bytes);
}

static void cRC4_encrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherRC4State FAR * encryptstate = &pvar->crypt_state.enc.cRC4;
  int num = 0;

  RC4(&encryptstate->k, bytes, buf, buf);
}

static void cRC4_decrypt(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  CipherRC4State FAR * decryptstate = &pvar->crypt_state.dec.cRC4;
  int num = 0;

  RC4(&decryptstate->k, bytes, buf, buf);
}

void CRYPT_set_random_data(struct _TInstVar FAR * pvar, unsigned char FAR * buf, int bytes) {
  if (pvar->crypt_state.cipher != SSH_CIPHER_NONE) {
    RAND_bytes(buf, bytes);
  } else {
    memset(buf, 0, bytes);
  }
}

void CRYPT_initialize_random_numbers(struct _TInstVar FAR * pvar) {
  RAND_screen();
}

static BIGNUM FAR * get_bignum(unsigned char FAR * bytes) {
  int bits = get_ushort16_MSBfirst(bytes);

  return BN_bin2bn(bytes + 2, (bits + 7)/8, NULL);
}

static RSA FAR * make_key(struct _TInstVar FAR * pvar,
                          int bits, unsigned char FAR * exp, unsigned char FAR * mod) {
  RSA FAR * key = RSA_new();

  if (key != NULL) {
    key->e = get_bignum(exp);
    key->n = get_bignum(mod);
  }

  if (key == NULL || key->e == NULL || key->n == NULL) {
    notify_fatal_error(pvar, "Error setting up RSA keys");

    if (key != NULL) {
      if (key->e != NULL) {
        BN_free(key->e);
      }
      if (key->n != NULL) {
        BN_free(key->n);
      }
      RSA_free(key);
    }

    return NULL;
  } else {
    return key;
  }
}

int CRYPT_set_cookie(struct _TInstVar FAR * pvar, unsigned char FAR * cookie) {
  memcpy(pvar->crypt_state.cookie, cookie, 8);
  return 1;
}

int CRYPT_set_server_key(struct _TInstVar FAR * pvar,
                         int bits, unsigned char FAR * exp, unsigned char FAR * mod) {
  pvar->crypt_state.server_key = make_key(pvar, bits, exp, mod);

  return pvar->crypt_state.server_key != NULL;
}

int CRYPT_set_host_key(struct _TInstVar FAR * pvar,
                       int bits, unsigned char FAR * exp, unsigned char FAR * mod) {
  pvar->crypt_state.host_key = make_key(pvar, bits, exp, mod);

  return pvar->crypt_state.host_key != NULL;
}

int CRYPT_set_supported_ciphers(struct _TInstVar FAR * pvar, int ciphers) {
  ciphers &= (1 << SSH_CIPHER_IDEA) | (1 << SSH_CIPHER_DES)
           | (1 << SSH_CIPHER_3DES) | (1 << SSH_CIPHER_RC4)
           | (1 << SSH_CIPHER_BLOWFISH);

  pvar->crypt_state.supported_ciphers = ciphers;

  if (ciphers == 0) {
    notify_fatal_error(pvar, "The server does not support any of my ciphers.\n"
      "A secure connection cannot be made.");
    return 0;
  } else {
    return 1;
  }
}

int CRYPT_choose_cipher(struct _TInstVar FAR * pvar) {
  int i;

  for (i = 0; pvar->session_settings.CipherOrder[i] != 0; i++) {
    int cipher = pvar->session_settings.CipherOrder[i] - '0';
    
    if (cipher == SSH_CIPHER_NONE) {
      break;
    } else if ((pvar->crypt_state.supported_ciphers & (1 << cipher)) != 0) {
      pvar->crypt_state.cipher = cipher;
      return cipher;
    }
  }

  notify_fatal_error(pvar,
    "All the ciphers that this program and the server both understand have been disabled.\n"
    "To communicate with this server, you will have to enable some more ciphers\n"
    "in the TTSSH Setup dialog box when you run Teraterm again.\n"
    "This connection will now close.");
  return SSH_CIPHER_NONE;
}

int CRYPT_get_encrypted_session_key_len(struct _TInstVar FAR * pvar) {
  int server_key_bits = BN_num_bits(pvar->crypt_state.server_key->n);
  int host_key_bits = BN_num_bits(pvar->crypt_state.host_key->n);
  int server_key_bytes = (server_key_bits + 7)/8;
  int host_key_bytes = (host_key_bits + 7)/8;

  if (server_key_bits < host_key_bits) {
    return host_key_bytes;
  } else {
    return server_key_bytes;
  }
}

int CRYPT_choose_session_key(struct _TInstVar FAR * pvar, unsigned char FAR * encrypted_key_buf) {
  int server_key_bits = BN_num_bits(pvar->crypt_state.server_key->n);
  int host_key_bits = BN_num_bits(pvar->crypt_state.host_key->n);
  int server_key_bytes = (server_key_bits + 7)/8;
  int host_key_bytes = (host_key_bits + 7)/8;
  int encrypted_key_bytes;
  int bit_delta;

  if (server_key_bits < host_key_bits) {
    encrypted_key_bytes = host_key_bytes;
    bit_delta = host_key_bits - server_key_bits;
  } else {
    encrypted_key_bytes = server_key_bytes;
    bit_delta = server_key_bits - host_key_bits;
  }

  if (bit_delta < 128 || server_key_bits < 512 || host_key_bits < 512) {
    notify_fatal_error(pvar, "Server RSA keys are too weak. A secure connection cannot be established.");
    return 0;
  } else {
    /* following Goldberg's code, I'm using MD5(servkey->n || hostkey->n || cookie)
       for the session ID, rather than the one specified in the RFC */
    int session_buf_len = server_key_bytes + host_key_bytes + 8;
    char FAR * session_buf = (char FAR *)malloc(session_buf_len);
    char session_id[16];
    int i;

    BN_bn2bin(pvar->crypt_state.host_key->n, session_buf);
    BN_bn2bin(pvar->crypt_state.server_key->n, session_buf + host_key_bytes);
    memcpy(session_buf + server_key_bytes + host_key_bytes, pvar->crypt_state.cookie, 8);
    MD5(session_buf, session_buf_len, session_id);

    free(session_buf);

    RAND_bytes(pvar->crypt_state.session_key, SESSION_KEY_LENGTH);

    memcpy(encrypted_key_buf + encrypted_key_bytes - SESSION_KEY_LENGTH,
      pvar->crypt_state.session_key, SESSION_KEY_LENGTH);
    for (i = 0; i < sizeof(session_id); i++) {
      encrypted_key_buf[encrypted_key_bytes - SESSION_KEY_LENGTH + i]
        ^= session_id[i];
    }

    if (host_key_bits > server_key_bits) {
	    if (RSA_public_encrypt(SESSION_KEY_LENGTH,
        encrypted_key_buf + encrypted_key_bytes - SESSION_KEY_LENGTH,
        encrypted_key_buf + encrypted_key_bytes - server_key_bytes,
	      pvar->crypt_state.server_key, RSA_PKCS1_PADDING) < 0) return 0;

	    if (RSA_public_encrypt(server_key_bytes,
        encrypted_key_buf + encrypted_key_bytes - server_key_bytes,
        encrypted_key_buf,
	      pvar->crypt_state.host_key, RSA_PKCS1_PADDING) < 0) return 0;
    } else {
	    if (RSA_public_encrypt(SESSION_KEY_LENGTH,
        encrypted_key_buf + encrypted_key_bytes - SESSION_KEY_LENGTH,
        encrypted_key_buf + encrypted_key_bytes - host_key_bytes,
	      pvar->crypt_state.host_key, RSA_PKCS1_PADDING) < 0) return 0;

	    if (RSA_public_encrypt(host_key_bytes,
        encrypted_key_buf + encrypted_key_bytes - host_key_bytes,
        encrypted_key_buf,
	      pvar->crypt_state.server_key, RSA_PKCS1_PADDING) < 0) return 0;
    }
  }

  return 1;
}

int CRYPT_generate_RSA_challenge_response(struct _TInstVar FAR * pvar, unsigned char FAR * challenge,
  int challenge_len, unsigned char FAR * response) {
  int server_key_bits = BN_num_bits(pvar->crypt_state.server_key->n);
  int host_key_bits = BN_num_bits(pvar->crypt_state.host_key->n);
  int server_key_bytes = (server_key_bits + 7)/8;
  int host_key_bytes = (host_key_bits + 7)/8;
  int session_buf_len = server_key_bytes + host_key_bytes + 8;
  char FAR * session_buf = (char FAR *)malloc(session_buf_len);
  char decrypted_challenge[48];
  int decrypted_challenge_len;

  decrypted_challenge_len =
    RSA_private_decrypt(challenge_len, challenge, challenge,
      AUTH_get_cur_cred(pvar)->RSA_key, RSA_PKCS1_PADDING);
  if (decrypted_challenge_len < 0) {
    free(session_buf);
    return 0;
  }
  if (decrypted_challenge_len >= RSA_CHALLENGE_LENGTH) {
    memcpy(decrypted_challenge, challenge + decrypted_challenge_len - RSA_CHALLENGE_LENGTH,
      RSA_CHALLENGE_LENGTH);
  } else {
    memset(decrypted_challenge, 0, RSA_CHALLENGE_LENGTH - decrypted_challenge_len);
    memcpy(decrypted_challenge + RSA_CHALLENGE_LENGTH - decrypted_challenge_len,
      challenge, decrypted_challenge_len);
  }

  BN_bn2bin(pvar->crypt_state.host_key->n, session_buf);
  BN_bn2bin(pvar->crypt_state.server_key->n, session_buf + host_key_bytes);
  memcpy(session_buf + server_key_bytes + host_key_bytes, pvar->crypt_state.cookie, 8);
  MD5(session_buf, session_buf_len, decrypted_challenge + 32);
  
  free(session_buf);

  MD5(decrypted_challenge, 48, response);
  
  return 1;
}

static void c3DES_init(char FAR * session_key, Cipher3DESState FAR * state) {
  des_set_key((des_cblock FAR *)session_key, state->k1);
  des_set_key((des_cblock FAR *)(session_key + 8), state->k2);
  des_set_key((des_cblock FAR *)(session_key + 16), state->k3);
  memset(state->ivec1, 0, 8);
  memset(state->ivec2, 0, 8);
  memset(state->ivec3, 0, 8);
}

static void cDES_init(char FAR * session_key, CipherDESState FAR * state) {
  des_set_key((des_cblock FAR *)session_key, state->k);
  memset(state->ivec, 0, 8);
}

static void cIDEA_init(char FAR * session_key, CipherIDEAState FAR * state) {
  idea_set_encrypt_key(session_key, &state->k);
  memset(state->ivec, 0, 8);
}

static void cBlowfish_init(char FAR * session_key, CipherBlowfishState FAR * state) {
  BF_set_key(&state->k, 32, session_key);
  memset(state->ivec, 0, 8);
}

int CRYPT_start_encryption(struct _TInstVar FAR * pvar) {
  switch (pvar->crypt_state.cipher) {
  case SSH_CIPHER_3DES: {
    c3DES_init(pvar->crypt_state.session_key, &pvar->crypt_state.enc.c3DES);
    c3DES_init(pvar->crypt_state.session_key, &pvar->crypt_state.dec.c3DES);

    pvar->crypt_state.encrypt = c3DES_encrypt;
    pvar->crypt_state.decrypt = c3DES_decrypt;

    return 1;
  }
 
  case SSH_CIPHER_IDEA: {
    cIDEA_init(pvar->crypt_state.session_key, &pvar->crypt_state.enc.cIDEA);
    cIDEA_init(pvar->crypt_state.session_key, &pvar->crypt_state.dec.cIDEA);

    pvar->crypt_state.encrypt = cIDEA_encrypt;
    pvar->crypt_state.decrypt = cIDEA_decrypt;

    return 1;
  }

  case SSH_CIPHER_DES: {
    cDES_init(pvar->crypt_state.session_key, &pvar->crypt_state.enc.cDES);
    cDES_init(pvar->crypt_state.session_key, &pvar->crypt_state.dec.cDES);

    pvar->crypt_state.encrypt = cDES_encrypt;
    pvar->crypt_state.decrypt = cDES_decrypt;

    return 1;
  }

  case SSH_CIPHER_RC4: {
    RC4_set_key(&pvar->crypt_state.dec.cRC4.k, 16, pvar->crypt_state.session_key);
    RC4_set_key(&pvar->crypt_state.enc.cRC4.k, 16, pvar->crypt_state.session_key + 16);

    pvar->crypt_state.encrypt = cRC4_encrypt;
    pvar->crypt_state.decrypt = cRC4_decrypt;

    return 1;
  }

  case SSH_CIPHER_BLOWFISH: {
    cBlowfish_init(pvar->crypt_state.session_key, &pvar->crypt_state.dec.cBlowfish);
    cBlowfish_init(pvar->crypt_state.session_key, &pvar->crypt_state.enc.cBlowfish);

    pvar->crypt_state.encrypt = cBlowfish_encrypt;
    pvar->crypt_state.decrypt = cBlowfish_decrypt;

    return 1;
  }

  default:
    notify_fatal_error(pvar, "no cipher selected"); /* should never get here! */
    return 0;
  }
}

void CRYPT_init(struct _TInstVar FAR * pvar) {
  pvar->crypt_state.encrypt = no_encrypt;
  pvar->crypt_state.decrypt = no_encrypt;
  pvar->crypt_state.cipher = SSH_CIPHER_NONE;
  pvar->crypt_state.server_key = NULL;
  pvar->crypt_state.host_key = NULL;
}

void CRYPT_get_cipher_info(struct _TInstVar FAR * pvar, char FAR * dest, int len) {
  switch (pvar->crypt_state.cipher) {
  case SSH_CIPHER_NONE:
    strncpy(dest, "None", len);
    return;
  case SSH_CIPHER_3DES:
    strncpy(dest, "3DES (168 key bits)", len);
    return;
  case SSH_CIPHER_DES:
    strncpy(dest, "DES (56 key bits)", len);
    return;
  case SSH_CIPHER_IDEA:
    strncpy(dest, "IDEA (128 key bits)", len);
    return;
  case SSH_CIPHER_RC4:
    strncpy(dest, "RC4 (128 key bits)", len);
    return;
  case SSH_CIPHER_BLOWFISH:
    strncpy(dest, "Blowfish (256 key bits)", len);
    return;
  default:
    strncpy(dest, "Unknown", len);
  }
}

void CRYPT_get_server_key_info(struct _TInstVar FAR * pvar, char FAR * dest, int len) {
  if (pvar->crypt_state.server_key == NULL || pvar->crypt_state.host_key == NULL) {
    strncpy(dest, "None", len);
  } else {
    _snprintf(dest, len, "%d-bit server key, %d-bit host key",
      BN_num_bits(pvar->crypt_state.server_key->n),
      BN_num_bits(pvar->crypt_state.host_key->n));
  }
}

void CRYPT_end(struct _TInstVar FAR * pvar) {
  if (pvar->crypt_state.server_key != NULL) {
    RSA_free(pvar->crypt_state.server_key);
  }
  if (pvar->crypt_state.host_key != NULL) {
    RSA_free(pvar->crypt_state.host_key);
  }
}

int CRYPT_passphrase_decrypt(int cipher, char FAR * passphrase, char FAR * buf,
  int bytes) {
  unsigned char passphrase_key[16];
  
  MD5(passphrase, strlen(passphrase), passphrase_key);

  switch (cipher) {
  case SSH_CIPHER_3DES: {
    Cipher3DESState state;

    des_set_key((des_cblock FAR *)passphrase_key, state.k1);
    des_set_key((des_cblock FAR *)(passphrase_key + 8), state.k2);
    des_set_key((des_cblock FAR *)passphrase_key, state.k3);
    memset(state.ivec1, 0, 8);
    memset(state.ivec2, 0, 8);
    memset(state.ivec3, 0, 8);
    des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
      state.k3, &state.ivec3, DES_DECRYPT);
    des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
      state.k2, &state.ivec2, DES_ENCRYPT);
    des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
      state.k1, &state.ivec1, DES_DECRYPT);
    break;
  }
 
  case SSH_CIPHER_IDEA: {
    CipherIDEAState state;
    int num = 0;

    cIDEA_init(passphrase_key, &state);
    idea_cfb64_encrypt(buf, buf, bytes, &state.k, state.ivec,
  	  &num, IDEA_DECRYPT);
    break;
  }

  case SSH_CIPHER_DES: {
    CipherDESState state;

    cDES_init(passphrase_key, &state);
    des_ncbc_encrypt((des_cblock FAR *)buf, (des_cblock FAR *)buf, bytes,
	    state.k, &state.ivec, DES_DECRYPT);
    break;
  }

  case SSH_CIPHER_RC4: {
    CipherRC4State state;
    int num = 0;

    RC4_set_key(&state.k, 16, passphrase_key);
    RC4(&state.k, bytes, buf, buf);
    break;
  }

  case SSH_CIPHER_BLOWFISH: {
    CipherBlowfishState state;

    BF_set_key(&state.k, 16, passphrase_key);
    memset(state.ivec, 0, 8);
    flip_endianness(buf, bytes);
    BF_cbc_encrypt(buf, buf, bytes, &state.k, state.ivec, BF_DECRYPT);
    flip_endianness(buf, bytes);
    break;
  }

  case SSH_CIPHER_NONE:
    break;

  default:
    memset(passphrase_key, 0, sizeof(passphrase_key));
    return 0;
  }

  memset(passphrase_key, 0, sizeof(passphrase_key));
  return 1;
}
