
/* $Source$
 * $Author$
 *
 * Copyright 1987 by the Massachusetts Institute of Technology.
 *
 * For copying and distribution information,
 * please see the file <mit-copyright.h>.
 *
 * This file contains tools for manipulating the tickets file.
 *
 * Description of ticket file format:
 *
 */

#include	<mit-copyright.h>

#ifndef	lint
static char rcsid_module_c[] =
"$Header$";
#endif	lint
#include	<stdio.h>
#include	<errno.h>
#include	<sys/types.h> 
#include	<sys/stat.h>
#include	<sys/file.h>
#include	<krb.h>

#define	R_TKT_FIL	0
#define	W_TKT_FIL	1
#define TOO_BIG		-1

/* actually defined elsewhere; delete after installing */
#define	TFILE		"tkt"

#define	TKT_FIL_OK	0
#define NO_TKT_FIL	1
#define TKT_FIL_ACC	2
#define	TKT_FIL_LCK	3
#define	TKT_FIL_FMT	4
#define	TKT_FIL_INI	5

char *tf_err_txt[] = {
	"OK.",
	"No ticket file.",
	"Can't access ticket file.",
	"Can't lock ticket file; try later.",
	"Bad ticket file format.",
	"Tried to read ticket file before init."
};

extern errno;

static fd;

/* tf_init should be called before the other ticket file
 * routines.  It takes the name of the ticket file to use,
 * and a read/write flag as arguments.
 *
 * It tries to open the ticket file, checks the mode and if
 * everything is okay, locks the file.  If it's opened for
 * reading, the lock is shared.  If it's opened for writing,
 * the lock is exclusive.
 *
 * Returns KSUCCESS if all went well, otherwise one of the
 * following:
 *
 * NO_TKT_FIL	- file wasn't there
 * TKT_FIL_ACC	- file was in wrong mode, etc.
 * TKT_FIL_LCK	- couldn't lock the file (retry is advisable)
 */

tf_init(tf_name, rw)
char *tf_name;
{
	int i, me, wflag;
	struct stat stat_buf;

	switch (rw) {
	case R_TKT_FIL:
		wflag = 0;
		break;
	case W_TKT_FIL:
		wflag = 1;
		break;
	default:
		fprintf(stderr, "tf_init: illegal parameter\n");
			exit (-1);
	}
	if (lstat(tf_name, &stat_buf) < 0) {
		perror("lstat");
		switch(errno) {
			case ENOENT:
				return NO_TKT_FIL;
			default:
				return TKT_FIL_ACC;
		}
	}
	me = getuid();
	if (stat_buf.st_uid != me && me != 0 || !(stat_buf.st_mode &
	    S_IFREG))
		return TKT_FIL_ACC;
	if ((fd = open(tf_name, wflag ? O_APPEND : O_RDONLY)) < 0)
		return TKT_FIL_ACC;
	if (flock(fd, (wflag ? LOCK_EX : LOCK_SH) | LOCK_NB) < 0)
		return TKT_FIL_LCK;
	return KSUCCESS;
}

/* This routine reads the principal's name from a ticket file.
 * It should only be called after tf_init has been called.  The
 * principal's name is filled into the p parameter.  If all goes
 * well, KSUCCESS is returned.  If tf_init wasn't called, TKT_FIL_INI
 * is returned.  If the name was null, or EOF was encountered, or the
 * name was longer than ANAME_SZ, TKT_FIL_FMT is returned.
 */

tf_get_pname(p)
char *p;
{
	if (!fd) {
		fprintf(stderr, "tf_get_pname called before tf_init.\n");
		return TKT_FIL_INI;
	}
	if (tf_gets(fd, p, ANAME_SZ) < 2)	/* can't be just a null */
		return TKT_FIL_FMT;
	return KSUCCESS;
}

/* This routine reads the principal's instance from a ticket file.
 * It should only be called after tf_init and tf_get_pname have been
 * called.  The instance is filled into the inst parameter.  If all goes
 * well, KSUCCESS is returned.  If tf_init wasn't called, TKT_FIL_INI
 * is returned.  If EOF was encountered, or the instance was longer than
 * ANAME_SZ, TKT_FIL_FMT is returned.  Note that the instance may be null.
 */

tf_get_pinst(inst)
char *inst;
{
	if (!fd) {
		fprintf(stderr, "tf_get_pinst called before tf_init.\n");
		return TKT_FIL_INI;
	}
	if (tf_gets(fd, inst, INST_SZ) < 1)
		return TKT_FIL_FMT;
	return KSUCCESS;
}

/* This routine reads a CREDENTIALS record from a ticket file and
 * fills in the given structure.  It should only be called after
 * tf_init, tf_get_pname, and tf_get_pinst have been called.
 * If all goes well, KSUCCESS is returned.  Possible error codes
 * are:
 *
 * TKT_FIL_INI	- tf_init wasn't called first
 * TKT_FIL_FMT	- bad format
 * EOF		- end of file encountered
 */

tf_get_cred(c)
CREDENTIALS *c;
{
	KTEXT ticket = &c->ticket_st;	/* pointer to ticket */
	int ret;

	if (!fd) {
		fprintf(stderr, "tf_get_cred called before tf_init.\n");
		return TKT_FIL_INI;
	}
	if ((ret = tf_gets(fd, c->service, SNAME_SZ)) < 2)
		switch (ret) {
		case TOO_BIG:
		case 1:			/* can't be just a null */
			return TKT_FIL_FMT;
		case 0:
			return EOF;
		}
	if ((ret =tf_gets(fd, c->instance, INST_SZ)) < 1)
		switch(ret) {
		case TOO_BIG:
			return TKT_FIL_FMT;
		case 0:
			return EOF;
		}
	if ((ret = tf_gets(fd, c->realm, REALM_SZ)) < 2)
		switch (ret) {
		case TOO_BIG:
		case 1:			/* can't be just a null */
			return TKT_FIL_FMT;
		case 0:
			return EOF;
		}
	if (
	read(fd, c->session, KEY_SZ) < 1 ||
	read(fd, &(c->lifetime), sizeof(c->lifetime)) < 1 ||
	read(fd, &(c->kvno), sizeof(c->kvno)) < 1 ||
	read(fd, &(ticket->length), sizeof(ticket->length)) < 1 ||
	/* don't try to read a silly amount into ticket->dat */
	ticket->length > MAX_KTXT_LEN ||
	read(fd, ticket->dat, ticket->length) < 1 ||
	read(fd, &(c->issue_date), sizeof(c->issue_date)) < 1
	) {
		fprintf(stderr, "tf_get_cred: bad ticket file format\n");
		return TKT_FIL_FMT;
	}
	return KSUCCESS;
}
	
/* tf_gets takes a file descriptor, a string and a count.
 * It reads from the file until either it has read
 * count characters, or until it reads a null byte.
 * When finished, what has been read exists in s.
 *
 * n		the number of bytes read (including null terminator)
 *		is returned when all goes well.
 *
 * 0		is returned on end of file or read error
 *
 * TOO_BIG	is returned if count characters are read
 *		and no null is encountered.  This is an
 *		indication that the ticket file is seriously
 *		ill.
 */

tf_gets(fd, s, n)
register fd;
register char *s;
{
	register count;

	for (count = n - 1; read(fd, s, 1) > 0 && count; --count)
		if (*s++ == '\0')
			return (n - count);
	if (count)
		return 0;
	if (*s)
		return TOO_BIG;
	return (n - count);
}

#ifdef TF_DEBUG
main()
{
	int k_errno;
	char pname[ANAME_SZ], pinst[INST_SZ];
	CREDENTIALS c;

	k_errno = tf_init(TFILE, R_TKT_FIL);
	fprintf(stderr, "tf_init: %s\n", tf_err_txt[k_errno]);

	k_errno = tf_get_pname(pname);
	fprintf(stderr, "tf_get_pname: %s\n", tf_err_txt[k_errno]);

	k_errno = tf_get_pinst(pinst);
	fprintf(stderr, "tf_get_pinst: %s\n", tf_err_txt[k_errno]);

	while(!(k_errno = tf_get_cred(&c)))
		fprintf(stderr, "got %s\n", c.service);
}
#endif TF_DEBUG
